/*============================================================================
*	This example show how to use the VAPI utilities APIs in conjunction
*	with the Comcerto header files.
*
*	Few configurations are prepared before creating any connections.
*	Then a connection is created and configured with one of the prepared configuration
*============================================================================*/

#include <vapi.h>
#include <gtl.h>
#include <msp.h>

#include "comcerto-ethernet-interface-types.h"
#include "comcerto-ip-device-level-types.h"


/* This function prepares some configurations to be used later on, "on demand".*/
int create_configuration(void)
{
	VSTATUS status;

	/* defining 3 different configurations */
	void *config_30ms; /* this message will contain VCEOPT (30ms frame size) + DTMFOPT*/ 
	void *config_20ms; /* this message will contain VCEOPT (20ms frame size) + DTMFOPT*/ 
	void *config_10ms; /* this message will contain VCEOPT (10ms frame size) + DTMFOPT + ECHOCAN OPT*/ 

	/* the protype are defined in comcerto_headers/comcerto-voip-types.h*/
	struct _VOIP_VCEOPT my_vce_options, *this_voice_opt = &my_vce_options;
	struct _VOIP_DTMFOPT my_dtmf_options, *this_dtmf_opt = &my_dtmf_options;
	struct _VOIP_ECHOCAN my_ec_options, *this_ec_opt = &my_ec_options;

	/* Now initialize the the VOICE, DTMF, ECH CANCELLER options with default values*/
	this_voice_opt->param_4.bits.compander = VOIP_VCEOPT_COMPANDER_DEFAULT;
	this_voice_opt->param_4.bits.cng = VOIP_VCEOPT_CNG_DEFAULT;
	this_voice_opt->param_4.bits.g_723_dc_removal =VOIP_VCEOPT_G_723_DC_REMOVAL_DEFAULT;
	this_voice_opt->param_4.bits.packet_generation = VOIP_VCEOPT_PACKET_GENERATION_DISABLE;
	this_voice_opt->param_4.bits.g_723_rate = VOIP_VCEOPT_G_723_RATE_DEFAULT;
	this_voice_opt->param_4.bits.adaptive_post_filter = VOIP_VCEOPT_ADAPTIVE_POST_FILTER_DEFAULT;
	this_voice_opt->param_4.bits.vadtype = VOIP_VCEOPT_VADTYPE_DEFAULT;
	this_voice_opt->param_4.bits.packet_interval = 20;

	this_voice_opt->param_5.bits.redundancy = VOIP_VCEOPT_REDUNDANCY_DEFAULT;
	this_voice_opt->param_5.bits.amr_mode = VOIP_VCEOPT_AMR_MODE_DEFAULT;
	this_voice_opt->param_5.bits.packing = VOIP_VCEOPT_PACKING_DEFAULT;
	this_voice_opt->param_5.bits.udptl = VOIP_VCEOPT_UDPTL_DEFAULT;
	this_voice_opt->param_5.bits.pkt_recept = VOIP_VCEOPT_PKT_RECEPT_DEFAULT;

	this_voice_opt->vad_tune = VOIP_VCEOPT_VAD_TUNE_DEFAULT;
	
	this_voice_opt->param_7.bits.cng_model = VOIP_VCEOPT_CNG_MODEL_DEFAULT;
	this_voice_opt->param_7.bits.g711_model = VOIP_VCEOPT_G711_MODEL_DEFAULT;

	this_voice_opt->param_8.bits.plc_cfg = VOIP_VCEOPT_PLC_CFG_DEFAULT;
	this_voice_opt->param_8.bits.bits_per_sample = VOIP_VCEOPT_BITS_PER_SAMPLE_DEFAULT;
	this_voice_opt->param_8.bits.crc_for_amr = VOIP_VCEOPT_CRC_FOR_AMR_DEFAULT;
	this_voice_opt->param_8.bits.g729eg_rate = VOIP_VCEOPT_G729EG_RATE_DEFAULT;
	this_voice_opt->param_8.bits.cdma_rate = VOIP_VCEOPT_CDMA_RATE_DEFAULT;
	this_voice_opt->param_8.bits.amr_wb_rate = VOIP_VCEOPT_AMR_WB_RATE_DEFAULT;
	this_voice_opt->param_8.bits.amr_wb_crc = VOIP_VCEOPT_AMR_WB_CRC_DEFAULT;
	this_voice_opt->param_8.bits.transcode_mode = VOIP_VCEOPT_TRANSCODE_MODE_DEFAULT;
	this_voice_opt->param_8.bits.amr_frame_format = VOIP_VCEOPT_AMR_FRAME_FORMAT_DEFAULT;

	this_dtmf_opt->param_4.bits.dtmf_voice = VOIP_DTMFOPT_DTMF_VOICE_DEFAULT;
	this_dtmf_opt->param_4.bits.dtmf_rtp = VOIP_DTMFOPT_DTMF_RTP_ENABLE;
	this_dtmf_opt->param_4.bits.redundancy = VOIP_DTMFOPT_REDUNDANCY_DEFAULT;
	this_dtmf_opt->param_4.bits.removal = VOIP_DTMFOPT_REMOVAL_DEFAULT;
	this_dtmf_opt->param_4.bits.regeneration = VOIP_DTMFOPT_REGENERATION_DISABLE;
	this_dtmf_opt->param_4.bits.event_report = VOIP_DTMFOPT_EVENT_REPORT_ENABLE;
	this_dtmf_opt->param_4.bits.rfc_packet_format = VOIP_DTMFOPT_RFC_PACKET_FORMAT_DEFAULT;
	this_dtmf_opt->param_4.bits.silence_removal = VOIP_DTMFOPT_SILENCE_REMOVAL_DEFAULT;
	this_dtmf_opt->param_4.bits.ssrc = VOIP_DTMFOPT_SSRC_DEFAULT;
	this_dtmf_opt->dtmf_pt = (VOIP_PTMNG_RTP_DTMF_DEFAULT << 8) | VOIP_PTMNG_REDUNDANT_DTMF_DEFAULT;

	this_ec_opt->param_4.bits.heclen = VOIP_ECHOCAN_HECLEN_DEFAULT;
	this_ec_opt->param_4.bits.nlptune_lo = VOIP_ECHOCAN_NLPTUNE_LO_DEFAULT;
	this_ec_opt->param_4.bits.cng = VOIP_ECHOCAN_CNG_DEFAULT;
	this_ec_opt->param_4.bits.nlptune_hi = VOIP_ECHOCAN_NLPTUNE_HI_DEFAULT;
	this_ec_opt->param_4.bits.nlp = VOIP_ECHOCAN_NLP_DEFAULT;
	this_ec_opt->param_4.bits.hecfrz = VOIP_ECHOCAN_HECFRZ_DEFAULT;
	this_ec_opt->param_4.bits.ecinit = VOIP_ECHOCAN_ECINIT_INIT;
	this_ec_opt->param_4.bits.ec_h_reset = VOIP_ECHOCAN_EC_H_RESET_DEFAULT;
	this_ec_opt->param_4.bits.ec_dc_removal = VOIP_ECHOCAN_EC_DC_REMOVAL_DEFAULT;
	this_ec_opt->param_4.bits.ecenb = VOIP_ECHOCAN_ECENB_DEFAULT;


	/*as we want to construct messages containing VOICE, DTMF, ECHOCAN options 
	make sure the allocated message has an enough size*/
	/* In any case the message construction utilities function checks if the message doesn't overflow*/ 

	/* allocate a message to prepare a multi command buffer */
	config_30ms = VAPI_AllocateMessage(100);
	/* allocate a message to prepare a multi command buffer */
	config_20ms = VAPI_AllocateMessage(100);
	/* allocate a message to prepare a multi command buffer */
	config_10ms = VAPI_AllocateMessage(100);

	if ((config_30ms == NULL) || (config_20ms == NULL) || (config_10ms == NULL))
	{
		status = -1;
		goto err;
	}

	/* We want to set VCEOPT for 20ms frame size (default value)*/
	status = VAPI_SetMessageFromBuffer(config_20ms,		/*message to be filled*/
					CMD_CLASS_CONF_CHANNEL, 
					CMD_TYPE_CONF_CHANGE,
					FC_VOIP_VCEOPT, 
					sizeof(struct _VOIP_VCEOPT)/2, /* number of parameters = fifo size of this command divided by 2*/
					(U16 *)this_voice_opt); /* Voice option to be added to the config_30ms message*/
	if (status != SUCCESS)
		goto err;

	/* We want to set VCEOPT for 30ms frame size*/
	this_voice_opt->param_4.bits.packet_interval = 30;
	status = VAPI_SetMessageFromBuffer(config_30ms,		/*message to be filled*/
					CMD_CLASS_CONF_CHANNEL, 
					CMD_TYPE_CONF_CHANGE,
					FC_VOIP_VCEOPT, 
					sizeof(struct _VOIP_VCEOPT)/2, /* number of parameters = fifo size of this command divided by 2*/
					(U16 *)this_voice_opt); /* Voice option to be added to the config_30ms message*/
	if (status != SUCCESS)
		goto err;

	/* We want to set VCEOPT for 10ms frame size*/
	this_voice_opt->param_4.bits.packet_interval = 10;
	status = VAPI_SetMessageFromBuffer(config_10ms,		/*message to be filled*/
					CMD_CLASS_CONF_CHANNEL, 
					CMD_TYPE_CONF_CHANGE,
					FC_VOIP_VCEOPT, 
					sizeof(struct _VOIP_VCEOPT)/2, /* number of parameters = fifo size of this command divided by 2*/
					(U16 *)this_voice_opt); /* Voice option to be added to the config_30ms message*/
	if (status != SUCCESS)
		goto err;

	/* we have 3 message containing voice options with different frame sizes*/

	/* We want to add DTMF OPT to config_30ms config*/
	status = VAPI_SetMessageFromBuffer(config_30ms,		/*message to be filled*/
					CMD_CLASS_CONF_CHANNEL, 
					CMD_TYPE_CONF_CHANGE,
					FC_VOIP_VCEOPT, 
					sizeof(struct _VOIP_DTMFOPT)/2, /* number of parameters = fifo size of this command divided by 2*/
					(U16 *)this_dtmf_opt); /* DTMF options to be added to the config_30ms message*/
	if (status != SUCCESS)
		goto err;

	/* We want to add DTMF OPT to config_20ms config*/
	status = VAPI_SetMessageFromBuffer(config_20ms,		/*message to be filled*/
					CMD_CLASS_CONF_CHANNEL, 
					CMD_TYPE_CONF_CHANGE,
					FC_VOIP_VCEOPT, 
					sizeof(struct _VOIP_DTMFOPT)/2, /* number of parameters = fifo size of this command divided by 2*/
					(U16 *)this_dtmf_opt); /* DTMF options to be added to the config_20ms message*/
	if (status != SUCCESS)
		goto err;

	/* We want to add DTMF OPT to config_10ms config*/
	status = VAPI_SetMessageFromBuffer(config_10ms,		/*message to be filled*/
					CMD_CLASS_CONF_CHANNEL, 
					CMD_TYPE_CONF_CHANGE,
					FC_VOIP_VCEOPT, 
					sizeof(struct _VOIP_DTMFOPT)/2, /* number of parameters = fifo size of this command divided by 2*/
					(U16 *)this_dtmf_opt); /* DTMF options to be added to the config_10ms message*/
	if (status != SUCCESS)
		goto err;

	/* We want to add DTMF OPT to config_10ms config*/
	status = VAPI_SetMessageFromBuffer(config_10ms,		/*message to be filled*/
					CMD_CLASS_CONF_CHANNEL, 
					CMD_TYPE_CONF_CHANGE,
					FC_VOIP_VCEOPT, 
					sizeof(struct _VOIP_ECHOCAN)/2, /* number of parameters = fifo size of this command divided by 2*/
					(U16 *)this_ec_opt); /* Echo Can options to be added to the config_10ms message*/
	if (status != SUCCESS)
		goto err;

err:
	return status;
}


/* This function configure an existing endpoint with one of the configuration prepared 
* by create_configuration function.*/
VSTATUS customize_endpoint(U16 connection_id, SMsg *config)
{
	VSTATUS status;

	/* the max message fifo lenght is device firmware & control interface dependant
	it is 256 bytes if the fimrware release is below 07 (i.e v5.07)
	else it is 256 bytes for PCI control intarface, 1500 bytes for other interfaces (ETH, POS)*/ 
	U8 device_response[DEFAULT_FIFO_MAX_SIZE];
	U32 response_size = DEFAULT_FIFO_MAX_SIZE;

	/* first, disable the endpoint */
	status = VAPI_SetConnectionState(connection_id, eInactive, NULL);
	if (status != SUCCESS)
		goto err;

	/* configure the endpoint with the new configuration */
	status = VAPI_SendConnectionMessage(connection_id, config, NULL, device_response, &response_size);

	/* At this point the endoint is disable.
	The application will need to enable it (if required) with :
	-VAPI_SetConnectionState(connection_id, eTdmActive, NULL); 
		to get the tone detection/generation active 
	or 
	- VAPI_SetConnectionState(connection_id, eActive, NULL);
		to get the tone detection/generation active + RTP traffic*/
err:
	return status;
}


/* This function creates a voice channel on the specified device_id, set it up as g.711 codec and other 
* default configuration (Done by VAPI_CreateConnection).
* Then the configuration is modified by applying the customized configuration.
* The endpoint is not active (no tone detectiopn/generation neither RTP traffic.*/

VSTATUS create_endpoint(int device_id, U16 connection_id, SMsg *configuration)
{
	VSTATUS status;

	status = VAPI_CreateConnection(
			device_id, 
			connection_id,
			eVOIP,
			connection_id,	/* use the connection_id as the timeslot*/
			A_LAW,		/* lets create it in A-ALAW system_law,	coding law alaw/ulaw*/
			NULL,
			NULL);

	if (status == SUCCESS)
	/* Now apply the configuration to this endpoint */
		status = customize_endpoint(connection_id, configuration);

	return status;
}

