/*! \file vapi_migrate.c  */
/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <argp.h>

#include <vapi/msp.h>
#include <vapi/msp.h>
#include <vapi/vapi.h>
#include <vapi/gtl.h>


#define DEVICE_CONTROL_MAC		{0x00,0x11,0x22,0x33,0x44,0x55}
#define HOST_CONTROL_INTERFACE		"eth0"
#define HOST_CONTROL_MAC		{0x00,0x02,0xb3,0x9c,0xd4,0xe9} /*00:02:B3:9C:D4:E9*/
#define MAX_STRING 256
#define MAX_FIFO_SIZE 256

static char firmware[MAX_STRING];
unsigned char *firmware_buf;
int firmware_size;
unsigned char host_mac_addr[6] = {0x00,0x02,0xb3,0x9c,0xd4,0xe9};
unsigned char device_mac_addr[6] = {0x00, 0x11, 0x22, 0x33, 0x66, 0x77};

SIpAddrInfo DevIpAddr;
STdmSetupParams tdm_parameters;
IN SIpParams conn_ip_params;
SFaxConfigOpts fax_configuration;
unsigned char announce_buffer_in[1024 * 1024];
unsigned char announce_buffer_out[1024 * 1024];
int coredump = 0;

#ifdef  VAPI_RELEASE /*exist from VAPI 2.4*/
#if VAPI_RELEASE >= (VAPI_VERSION(2, 9, 0)) /* CidInfo, SRecData, SPlayData exist from VAPI 2.9*/
	CidInfo stCidInfo;
	SRecData stRecData;
	SPlayData stPlayData;
#if VAPI_RELEASE >= (VAPI_VERSION(2, 10, 0)) /*SConvertData exists from VAPI 2.10*/
	SConvertData stConvertData;
#endif

#endif
#endif

/* Caller ID clip message*/
U8 clip_msg[24] = {
	0x80, 0x01, 0x08, 0x31, 0x31, 0x30, 0x38, 0x32,
	0x30, 0x30, 0x30, 0x02, 0x07, 0x39, 0x34, 0x34,
	0x32, 0x38, 0x32, 0x30, 0x08, 0x01, 0x4F
};

U8 complex_tone[] = {
         /*CADENCE_ENGINE, length = 0x18 (24) no padding required*/
                        0x18, 0x00, 0x00, 0x02, 0x6f, 0x80, 0x00, 0x00, /*CC, CT, FC*/
                        0x01, 0x00, 0x00, 0x40, 0x32, 0x00, 0x00, 0x10,
                        0x02, 0x00 ,0xc8, 0x00, 0x2c, 0x01, 0x00, 0x20,
        /* PROGRAM_TONE_ENGINE, length =  0x2e (46) 2 bytes of padding required*/
                        0x2e, 0x00, 0x00, 0x02, 0x6e, 0x80, 0x00, 0x00,
                        0x02, 0x00, 0x6c, 0x07, 0xd3, 0x00, 0x00, 0x00,
                        0x00, 0x00 ,0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00 ,0x00, 0x00, 0x78, 0x05, 0xfb, 0x00,
                        0x00, 0x00 ,0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00 ,0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* VOPENA_TONEGEN , length = 0x0a (10) 2 bytes of padding required*/
                        0x0a, 0x00, 0x00, 0x02, 0x00, 0x80, 0x00, 0x00,
                        0x0d, 0x00 ,0x00, 0x00
};


/*! structure with configuration parameters, structure type is defined in gtl.h */
SCSMEUsrData gtl_device_configuration;

/*! This structure holds default values for the device configuration, user can
 * overwrite them somehow in set_device_configuration() function using his own
 * config. scheme.
 */
SCSMEUsrData default_device_configuration = {
	NULL,				/**< pointer to next device */
	eCSM_ITF,			/**< control interface to use */
	0,				/**< device ID */
	0,				/**< Version infor (not used) */
	eSLAVE,				/**< Slave / Master mode */
#ifdef  VAPI_RELEASE /*exist from VAPI 2.4*/
#if VAPI_RELEASE >= (VAPI_VERSION(2, 4, 0))
	DEV_TYPE_M827XX,
#else
	DEV_TYPE_MIRO,			/**< Device type */
#endif
#else
	DEV_TYPE_MIRO,			/**< Device type */
#endif
	True,				/**< Default or custom max channels */
	0,				/**< Max Channels if above flag is custom */
	DEVICE_CONTROL_MAC,		/**< MAC address for control over csmencaps */
	HOST_CONTROL_MAC,		/**< MAC address of the host interface controling the device */
	(unsigned char *)HOST_CONTROL_INTERFACE,	/**< host interface used to control the device */
	1				/**< csme ack required */
};

const char *argp_program_version = "vapi_migrate 0.1";
const char *argp_program_bug_address = "philippe.vivarelli@mindspeed.com";
const char doc[] = "vapi_migrate - program to run all VAPI APIs.";

const struct argp_option options[] = {
	{"firmware", 'f', "FIRMWARE_FILE", 0, "firmware code filename (using VAPI_BootDevice)"},
	{NULL}
};

/*=================================================================================*/
/*!
 *	\brief This function reads the firmware (.axf or .elf) and store it to a buffer.
 */
/*=================================================================================*/
static int comcerto_read_firmware_file(char *filename, unsigned char **buf, int *size)
{
	FILE *fp;

	fp = fopen(filename, "r");

	if (fp == NULL)
	{
		perror(filename);
		goto err0;
	}

	/* Figure out how big the size of the file and allocate that much space in memory */
	fseek(fp, 0, SEEK_END);
	*size = ftell(fp);

	*buf = (unsigned char *)malloc(*size);
	if (*buf == NULL)
	{
		printf("can't allocate memory\n");
		goto err1;
	}

	fseek(fp, 0, SEEK_SET);

	if (fread(*buf, sizeof(unsigned char), *size, fp) != *size)
	{
		printf( "error reading '%s'\n", filename);
		goto err2;
	}

	fclose(fp);

	return 0;

      err2:
	free(*buf);

      err1:
	fclose(fp);

      err0:
	return -1;
}

/*=================================================================================*/
/*!
 *	\brief This function fills the global structure with device config parameters
 */
/*=================================================================================*/
void set_device_configuration(U16 device_id)
{
	U8 *mac;

	memcpy(&gtl_device_configuration, &default_device_configuration, sizeof(SCSMEUsrData));

	/* user may put here code to overwrite default config parameters, for example
	 * read them from config file
	 */

	gtl_device_configuration.uiDevId = device_id;
	gtl_device_configuration.pucEthDevName = (unsigned char *) HOST_CONTROL_INTERFACE;

	/* this code is executed before VAPI_Init, so we have to use printf's */

	printf("CONTROL_INTERFACE_TYPE  = %d\n", gtl_device_configuration.usControlInterface);
	printf("DEVICE_ID    = %d\n", gtl_device_configuration.uiDevId);
	printf("DEVICE_MODE  = %d\n", gtl_device_configuration.eDevMode);
	printf("DEVICE_TYPE  = %d\n", gtl_device_configuration.ucDevType);
	printf("USE_DEFAULT_MAX_CHANNEL = %d\n", gtl_device_configuration.bUseDefaultMaxChnls);
	printf("CUSTOM_MAX_CHANNEL      = %d\n", gtl_device_configuration.usMaxChannels);
	printf("ACK_REQUIRED = %d\n", gtl_device_configuration.ucIsAckReqd);

	mac = gtl_device_configuration.aucDevMac;
	printf("DEVICE MAC = %02x:%02x:%02x:%02x:%02x:%02x\n", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

	mac = gtl_device_configuration.aucHostMac;
	printf("HOST   MAC = %02x:%02x:%02x:%02x:%02x:%02x\n", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

	printf("HOST_CONTROL_INTERFACE  = %s\n\n", gtl_device_configuration.pucEthDevName);
}

/* dummy function used for VAPI_RegisterEventCallback*/
void comcerto_indication_handler(EEventCode eEventCode, void *pvData)
{
}

/* dummy function used for VVAPI_RegisterReset*/
VSTATUS comcerto_reset_handler(void *reset_data)
{
	return 0;
}

error_t parser(int key, char *arg, struct argp_state *state)
{
	switch (key)
	{

	case 'f':
		strncpy(firmware, arg, MAX_STRING);
		break;

	default:
		return ARGP_ERR_UNKNOWN;
	}

	return 0;
}

/* parser option structure */
static struct argp argp = { options, parser, 0, doc, };

/***************************************************************************
 * main
 ***************************************************************************/
/*! \brief Application entry.

 *  \section Description
*/

int main(int argc, char *argv[])
{
	int result;
	int device_id = 0;
	SMsg MyMsg;
	U8 OutBuff[MAX_FIFO_SIZE];
	U32 OutBuffLen = MAX_FIFO_SIZE;
	MyMsg.pusFifo = (U16 *)&complex_tone;
	MyMsg.uiFifoSize = sizeof(complex_tone);

	argp_parse(&argp, argc, argv, 0, 0, NULL);

	result = comcerto_read_firmware_file((char *)firmware, &firmware_buf, &firmware_size);
	if (result)
		goto err0;

	set_device_configuration(device_id);

	result = VAPI_Init(&gtl_device_configuration);
	if (result != SUCCESS)
	{
		fprintf(stderr, "VAPI_Init: failed, result = %d\n", result);
		goto err0;
	}

	/* disable debug output*/
	VAPI_SetDebugLevel(2, 2);

	/* Device APIs */
	result = VAPI_OpenDevice(device_id, NULL);
	if (result != SUCCESS)
	{
		fprintf(stderr, "VAPI_OpenDevice: failed, result = %d\n", result);
		goto err1;
	}

	result = VAPI_RegisterEventCallback(device_id, EVENT_LEVEL_GENERIC, comcerto_indication_handler);

#ifdef  VAPI_RELEASE /*exist from VAPI 2.4*/
#if (VAPI_RELEASE >= (VAPI_VERSION(2, 7, 0))) /* new API from 2.7*/
	VAPI_RegisterReset(device_id, comcerto_reset_handler, NULL);
#endif
#endif

	printf("firmware: '%s', size: %d\n", firmware, firmware_size);

	if (firmware_size > 0 && firmware_buf != NULL)
	{
		/* the device_id comes from the -d option */
		result = VAPI_AssignBootMAC(device_id, NULL);
		if (result != SUCCESS)
		{
			printf("VAPI_AssignBootMAC failed, status: %d", result);
			goto err1;
		}
	}


#ifdef  VAPI_RELEASE /*exist from VAPI 2.4*/

#if (VAPI_RELEASE >= (VAPI_VERSION(2, 6, 0))) /* new param to VAPI_BootDeviceFile from 2.6*/
	result = VAPI_BootDeviceFile(device_id, firmware, NULL, NULL);
#elif (VAPI_RELEASE >= (VAPI_VERSION(2, 5, 0))) /* VAPI_BootDeviceFile exists from 2.5*/
	result = VAPI_BootDeviceFile(device_id, firmware, NULL);
#else 
	result = VAPI_BootDevice(device_id, firmware_buf, firmware_size, NULL);
#endif

#else /* if VAPI_RELEASE macro does not exist (prior to 2.4)*/
	result = VAPI_BootDevice(device_id, firmware_buf, firmware_size, NULL);
#endif

	result = VAPI_SetEthMac(device_id, CMD_LEVEL_DEVICE, device_mac_addr, host_mac_addr, NULL);

	DevIpAddr.ucNumOfSrcIpAddr = 1;
	DevIpAddr.bIsMultipleMode = 0;
	DevIpAddr.auiDevIPAddress[0] = 0xc0a82081; /*192.168.32.177*/
	DevIpAddr.ucChkSumVerification = 0;
	DevIpAddr.ucEncapsulation = 0;
	result = VAPI_SetDeviceIPAddr(device_id, &DevIpAddr, NULL);

	/* zero all structure, */
	memset(&tdm_parameters, 0, sizeof(tdm_parameters));
	tdm_parameters.usMode = 1;
	tdm_parameters.usNoOfBus = 1;
	tdm_parameters.astTdmBusParam[0].ucTxOrder = 1;
	tdm_parameters.astTdmBusParam[0].ucTxClkEdge = 1;
	tdm_parameters.astTdmBusParam[0].ucFrameEdge = 1;
	tdm_parameters.astTdmBusParam[0].usNumOfTS = 128;
	tdm_parameters.astTdmBusParam[0].ucInvertedFrmSig = 1;
	tdm_parameters.astTdmBusParam[0].ucBitOrderRcv = 1;
	tdm_parameters.astTdmBusParam[0].ucBusId = 0;

	result = VAPI_SetTDMParams(device_id, &tdm_parameters, NULL);

#ifdef  VAPI_RELEASE /*exist from VAPI 2.4*/
#if VAPI_RELEASE >= (VAPI_VERSION(2, 4, 0)) /* VAPI_InitDevice exists from 2.4*/
	result = VAPI_InitDevice(device_id, VAPI_DEV_OPMODE_DEFAULT, VAPI_DEV_PROF_DEFAULT, NULL, NULL);
#endif
#endif

	/* Connection APIs */
	result = VAPI_CreateConnection(device_id, 0, 2, 0, 0, NULL, NULL);

	result = VAPI_CreateConnection(device_id, 1, 2, 1, 0, NULL, NULL);

	result = VAPI_StartTone(0, 0, eDIALTONE, eUSA, NULL);

	result = VAPI_StopTone(0, 0, NULL);

	result =  VAPI_PassThru(1, CMD_LEVEL_CONN, &MyMsg, NULL, OutBuff, &OutBuffLen);

	result = VAPI_StopTone(1, 0, NULL);

	result = VAPI_EnableCIDDetection(1, eUSA, 0, NULL);

#ifdef  VAPI_RELEASE /*exist from VAPI 2.4*/
#if VAPI_RELEASE >= (VAPI_VERSION(2, 9, 0)) /*VAPI_StartCallerId uses CidInfo struct from VAPI 2.9*/
	stCidInfo.MsgLen = sizeof(clip_msg);
	stCidInfo.pucMsgStr = clip_msg;
	stCidInfo.bIsOnHook = False;
	stCidInfo.eCountryCode = eUSA;

	result = VAPI_StartCallerId(0, &stCidInfo, NULL);
#else /*if VAPI release before VAPI 2.9*/
	result = VAPI_StartCallerId(0, clip_msg, 0, eUSA, NULL);
#endif
#else /*if VAPI_RELEASE macro does not exist (prior to 2.4)*/
	result = VAPI_StartCallerId(0, clip_msg, 0, eUSA, NULL);
#endif

	result = VAPI_StopCallerId(0, NULL);

#ifdef  VAPI_RELEASE /*exists from VAPI 2.4*/
#if VAPI_RELEASE <= (VAPI_VERSION(2, 5, 0)) /* APIs removed from 2.5.x*/
	result = VAPI_SetConnDestIpUdp(0, DevIpAddr.auiDevIPAddress[0], 30000, NULL);

	result = VAPI_SetConnSrcUdp(0, 30002, 0, NULL);

	result = VAPI_SetConnDestIpUdp(1, DevIpAddr.auiDevIPAddress[0], 30002, NULL);

	result = VAPI_SetConnSrcUdp(1, 30000, 0, NULL);

	conn_ip_params.uiDestIpAddr = DevIpAddr.auiDevIPAddress[0];
	conn_ip_params.usDestUdpPort = 30000;
	conn_ip_params.uiSrcIpAddr = DevIpAddr.auiDevIPAddress[0];
	conn_ip_params.usSrcUdpPort = 30002;
#endif
#else   /* if VAPI_RELEASE macro does not exist (prior to 2.4) */
	result = VAPI_SetConnDestIpUdp(0, DevIpAddr.auiDevIPAddress[0], 30000, NULL);

	result = VAPI_SetConnSrcUdp(0, 30002, 0, NULL);

	result = VAPI_SetConnDestIpUdp(1, DevIpAddr.auiDevIPAddress[0], 30002, NULL);

	result = VAPI_SetConnSrcUdp(1, 30000, 0, NULL);

	conn_ip_params.uiDestIpAddr = DevIpAddr.auiDevIPAddress[0];
	conn_ip_params.usDestUdpPort = 30000;
	conn_ip_params.uiSrcIpAddr = DevIpAddr.auiDevIPAddress[0];
	conn_ip_params.usSrcUdpPort = 30002;
#endif

	result = VAPI_SetConnIpParams(0, &conn_ip_params, NULL);

	conn_ip_params.uiDestIpAddr = DevIpAddr.auiDevIPAddress[0];
	conn_ip_params.usDestUdpPort = 30002;
	conn_ip_params.uiSrcIpAddr = DevIpAddr.auiDevIPAddress[0];
	conn_ip_params.usSrcUdpPort = 30000;
	result = VAPI_SetConnIpParams(1, &conn_ip_params, NULL);

	result = VAPI_SetCodecType(0, eG711_ULAW_PCM, NULL);
	result = VAPI_SetCodecType(1, eG711_ULAW_PCM, NULL);

	result = VAPI_SetPacketInterval(0, 20, NULL);
	result = VAPI_SetPacketInterval(1, 20, NULL);

	result = VAPI_SetPayloadType(0, eG726_40_audio, 0x40, eBoth, NULL);
	result = VAPI_SetPayloadType(1, eG726_40_audio, 0x40, eBoth, NULL);

	result = VAPI_SetRtpSsrcHeader(0, 0xffeeddcc, NULL);
	result = VAPI_SetRtpSsrcHeader(1, 0xffeeddcc, NULL);

	result = VAPI_EchoCancellorReset(0, NULL);
	result = VAPI_EchoCancellorReset(1, NULL);

	result = VAPI_EnableConnection(0, NULL);
	result = VAPI_EnableConnection(1, NULL);

	result = VAPI_Loopback(0, 1, eINTER_CHNL_POST_ENCAPS, NULL);

	result = VAPI_SendNteEvent(0, 0x20000bb8, 0, 0, 0, 0, NULL);

	fax_configuration.bUseExistingOpts = False;
	fax_configuration.ucFaxTxLevel = 9;
	fax_configuration.bEnableFaxAutoSwitch = False;
	fax_configuration.usSwitchoverEventMask = 0;
	fax_configuration.stFaxOpts.ucT38PktLossConcealment = 1;
	fax_configuration.stFaxOpts.ucECMDisable = 1;
	fax_configuration.stFaxOpts.ucFaxDataRedundancyCnt = 3;
	fax_configuration.stFaxOpts.ucT30RedundancyCnt = 7;
	fax_configuration.stFaxOpts.ucFaxConnSpeedLimit = 6;
	fax_configuration.stFaxOpts.ucErrRcvyMethod = 0;
	fax_configuration.stFaxOpts.ucTCFProcedure = 0;
	fax_configuration.stFaxOpts.ucNumFEC = 0;
	fax_configuration.stFaxOpts.ucNumIFPs = 4;

	result = VAPI_ConfigureT38Options(0, &fax_configuration, NULL);
	result = VAPI_ConfigureT38Options(1, &fax_configuration, NULL);

	result = VAPI_SwitchToT38(0, NULL, NULL);
	result = VAPI_SwitchToT38(1, NULL, NULL);

	result = VAPI_DisableConnection(0, NULL);
	result = VAPI_DisableConnection(1, NULL);

	result = VAPI_DestroyConnection(0, NULL);
	result = VAPI_DestroyConnection(1, NULL);

	/* conferencing APIs*/
	result = VAPI_CreateConference(device_id, 0, NULL, 0, NULL, NULL);

	result = VAPI_CreateTdmSideParticipant(0, 0 ,0 , 0, NULL, NULL);

	result = VAPI_CreateIpSideParticipant(0, 1 , NULL, NULL);

	result = VAPI_EnableParticipant(0, NULL);

	result = VAPI_MuteParticipant(0, 0, NULL);

	result = VAPI_PutParticipantOnHold(0, eSIMPLEHOLD, 0, NULL);

	result = VAPI_DisableParticipant(0, NULL);

	result = VAPI_DestroyParticipant(0, NULL);

	result = VAPI_DestroyConference(0, NULL);

	/* Announcement APIs*/
	result = VAPI_CreateConnection(device_id, 0, 2, 0, 0, NULL, NULL);

#ifdef  VAPI_RELEASE /*exist from VAPI 2.4*/
#if VAPI_RELEASE >= (VAPI_VERSION(2, 9, 0)) /*SRecData, SPlayData exist from VAPI 2.9*/
	stRecData.eCodingScheme = 0;
	stRecData.eMediaHost = 0;
	stRecData.eSourceType = 0;
	stRecData.pfnFrameHndlr = NULL;
	stRecData.pucBuffer  =announce_buffer_in;
	stRecData.uiBufferLength = sizeof(announce_buffer_in);
	stRecData.uiHostSpeechDataFormat = eHedFormat;
	stRecData.uiTimeOut = 2;

	stPlayData.eCodingScheme = 0;
	stPlayData.eDestinationType = 0;
	stPlayData.eMediaHost = 0;

#if VAPI_RELEASE >= (VAPI_VERSION(2, 10, 0)) /*enumeration ePlaybackMode exists from VAPI 2.10*/	
	stPlayData.ePlaybackMode = ePlaybackStandard;
#else
	stPlayData.uiExtended = False;
#endif

	stPlayData.pucBuffer = announce_buffer_in;
	stPlayData.uiBufferLength = sizeof(announce_buffer_in);
	stPlayData.uiHostSpeechDataFormat = eHedFormat;
	stPlayData.uiTimeOut = 2;

#if VAPI_RELEASE >= (VAPI_VERSION(2, 10, 0)) /*struct SConvertData exists from VAPI 2.10*/
	stConvertData.pucOutBuffer = announce_buffer_out;
	stConvertData.pucSrcBuffer = announce_buffer_in;
	stConvertData.ucCurrentCoding = 0;
	stConvertData.ucNewCoding = 8;
	stConvertData.uiOutBufferLength = sizeof(announce_buffer_in);
	stConvertData.uiSrcBufferLength = sizeof(announce_buffer_out);
#endif
	
	result = VAPI_StartRecord(0, &stRecData, NULL);

	result = VAPI_StopRecord(0, eAS_STOP, NULL);

	result = VAPI_StartPlayback(0, &stPlayData, NULL);

#if VAPI_RELEASE >= (VAPI_VERSION(2, 10, 0)) /*VAPI_PlaybackSetRate exists from VAPI 2.10*/
	result = VAPI_PlaybackSetRate(0, ePlaybackRate_2_1, NULL);
#endif
	result = VAPI_StopPlayback(0, eAS_STOP, NULL);

#if VAPI_RELEASE >= (VAPI_VERSION(2, 10, 0))
	result = VAPI_ConvertHostSpeechData(0, &stConvertData, NULL);
#endif

#else /*if VAPI release before VAPI 2.9*/
	result = VAPI_StartRecording(0, 0, 0, 0, 2, announce_buffer_in, sizeof(announce_buffer_in), NULL, NULL);

	result = VAPI_StopRecording(0, NULL);

	result = VAPI_StartPlaying(0, 0, 0, 0, 2, announce_buffer_in, sizeof(announce_buffer_in), NULL);

	result = VAPI_StopPlaying(0, NULL);

	result = VAPI_ConvertHostSpeechData(0, announce_buffer_in, sizeof(announce_buffer_in), 0, 
						announce_buffer_out, sizeof(announce_buffer_out), 8, NULL);	
#endif
#else /* if VAPI_RELEASE macro does not exist (prior to 2.4)*/
	result = VAPI_StartRecording(0, 0, 0, 0, 2, announce_buffer_in, sizeof(announce_buffer_in), NULL, NULL);

	result = VAPI_StopRecording(0, NULL);

	result = VAPI_StartPlaying(0, 0, 0, 0, 2, announce_buffer_in, sizeof(announce_buffer_in), NULL);

	result = VAPI_StopPlaying(0, NULL);

	result = VAPI_ConvertHostSpeechData(0, announce_buffer_in, sizeof(announce_buffer_in), 0, 
						announce_buffer_out, sizeof(announce_buffer_out), 8, NULL);	
#endif
	result = VAPI_DestroyConnection(0, NULL);

#ifdef  VAPI_RELEASE /*exists from VAPI 2.4*/
#if VAPI_RELEASE >= (VAPI_VERSION(2, 5, 0))  /* new APIs from VAPI 2.5*/
	result = VAPI_CreateConnection(device_id, 0, 2, 0, 0, NULL, NULL);
	result = VAPI_CreateConnection(device_id, 1, 2, 1, 0, NULL, NULL);
	
	VAPI_TranscodingSession(0, 1, 1, NULL);

	VAPI_RecoverConnection(device_id, 0, 2, 0, 0, NULL, NULL);

	result = VAPI_DestroyConnection(0, NULL);
	result = VAPI_DestroyConnection(1, NULL);
#endif
#endif

#ifdef  VAPI_RELEASE /*exists from VAPI 2.4*/
#if VAPI_RELEASE >= (VAPI_VERSION(2, 8, 0)) /* new APIs from VAPI 2.8*/
	result = VAPI_SetDeviceVlan(device_id, 0x0101, NULL);

	result = VAPI_CreateConnection(device_id, 0, 2, 0, 0, NULL, NULL);
	result = VAPI_SetConnVlan(0, 0x0102, NULL);

	void *message;
	message = VAPI_AllocateMessage(MAX_FIFO_SIZE);

	VAPI_InitMessage(message);

	result = VAPI_SetMessage(message , CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE,
						FC_VOIP_VCEOPT, 5, /* 5 parameters */
						UT_CPU2LE16(0x1431), UT_CPU2LE16(0x7C00),
						UT_CPU2LE16(0x0004), UT_CPU2LE16(0X0000),
						UT_CPU2LE16(0x0000));

	result = VAPI_SetMessageFromBuffer(message, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, UT_CPU2LE16(0x806f), 8, (U16 *)&complex_tone[8]);

	result = VAPI_SendConnectionMessage(0, message, NULL, OutBuff, &OutBuffLen);

	VAPI_InitMessage(message);

	result = VAPI_SetMessage(message , CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY, FC_SPU_FEATURES_CONTROL, 0);

	result = VAPI_SendDeviceMessage(device_id, message, NULL, OutBuff, &OutBuffLen);

	VAPI_FreeMessage(message);

	result = VAPI_DestroyConnection(0, NULL);

	result = VAPI_CreateConnection(device_id, 0, eCSEoPSN, 0, U_LAW, NULL, NULL);
	result = VAPI_CreateConnection(device_id, 1, eCSEoPSN, 0, U_LAW, NULL, NULL);

        result = VAPI_CESoPSN_Mapper(device_id, NULL, eChannelMap, 0, 1, 2, 0, 1);

#endif
#endif
	/* not executed, just compiled*/
	if (coredump == 1)
	{
		U32 ulOffset = 0;
		Boolean fLastRead;

#ifdef  VAPI_RELEASE /*exists from VAPI 2.4*/
#if (VAPI_RELEASE >= (VAPI_VERSION(2, 6, 0)))      /* new param from 2.6*/
		result = VAPI_GetCoreDump(device_id, 2, firmware_buf, 100, &firmware_size, ulOffset, &fLastRead, NULL, NULL);
#elif (VAPI_RELEASE >= (VAPI_VERSION(2, 5, 0)))   /* support async mode*/
		result = VAPI_GetCoreDump(device_id, 2, firmware_buf, 100, &firmware_size, ulOffset, &fLastRead, NULL);
#else
		result = VAPI_GetCoreDump(device_id, 2, firmware_buf, 100, &firmware_size, ulOffset, &fLastRead);
#endif
#else
		result = VAPI_GetCoreDump(device_id, 2, firmware_buf, 100, &firmware_size, ulOffset, &fLastRead);
#endif
	}
	
	VAPI_CloseDevice(device_id, ePURGE);
err1:
	VAPI_Close();
err0:
	return FAILURE;
}
