/*=================================================================================
	This function showas how to set the Secure RTP options for a connection.
	SRTP enabled with CIPHER SRTP OPTIONS:
			- HMAC SHA1 
			- AES Counter mode
			- Full Spec key derivation
			  Symmetric configuration
			- Lifetime disabled
			- Encryption key size 128 bits
			- Master key size 128 bits
			- Master key identifier disabled
			- No key derivation rate
			- Master Salt key size 128 bits
			- Session salt key size 112 bits
			- No Authentication prefix
			- Authentication tag size 10 bytes
			- Authentication session key size 40

		CIPHER_SRTP_KEY_DERIVATION:
			- Symmetric configuration
			- Disable SRTP master re-keying lifetime
			- Disable SRTCP master re-keying lifetime
			- From/to index
			- Master key 128bits
			- Master salt key (128 bits used out of 256)

=================================================================================*/
#include <vapi.h>
#include <gtl.h>
#include <msp.h>
#include <vapi/comcerto-encryption-types.h>

VSTATUS set_srtp(U32 conn_id, U8 key_size, U8 key)
{
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];
	VSTATUS result;
	char *key_array;
	int master_key_size;
	int num_of_param = 0;

	struct _CIPHER_SRTP_OPT srtp_option = {
	.param_4.bits.from_to = CIPHER_SRTP_OPT_FROM_TO_DISABLE,
		.param_4.bits.authentication = CIPHER_SRTP_OPT_AUTHENTICATION_HMAC_SHA1,
		.param_4.bits.encryption = CIPHER_SRTP_OPT_ENCRYPTION_DEFAULT,
		.param_4.bits.key_material = CIPHER_SRTP_OPT_KEY_MATERIAL_DEFAULT,
		.param_4.bits.direction = CIPHER_SRTP_OPT_DIRECTION_SYMMETRIC,
		.param_4.bits.layer = CIPHER_SRTP_OPT_LAYER_SRTP,
		.param_4.bits.byteorder = CIPHER_SRTP_OPT_BYTEORDER_DEFAULT,
		.param_4.bits.srtp = CIPHER_SRTP_OPT_SRTP_ENABLE,
	
		.param_5.bits.key_derivation = CIPHER_SRTP_OPT_KEY_DERIVATION_DEFAULT,
		.param_5.bits.mki_size = 0,
		.param_5.bits.master_key_size = key_size,
		.param_5.bits.encrypt_key_size = CIPHER_SRTP_OPT_ENCRYPT_KEY_SIZE_DEFAULT,
	
		.param_6.bits.master_salt_key_size = 128,
	
		.param_7.bits.salt_key_size = CIPHER_SRTP_OPT_SALT_KEY_SIZE_DEFAULT,
	
		.param_8.bits.authentication_key_size = CIPHER_SRTP_OPT_AUTHENTICATION_KEY_SIZE_DEFAULT,
		.param_8.bits.authentication_tag_size = CIPHER_SRTP_OPT_AUTHENTICATION_TAG_SIZE_DEFAULT,
		.param_8.bits.authentication_prefix_size = CIPHER_SRTP_OPT_AUTHENTICATION_PREFIX_SIZE_DEFAULT,
	};

	struct _CIPHER_SRTP_KEY_DERIVATION srtp_key_derivation = {
		.param_4.bits.direction = CIPHER_SRTP_KEY_DERIVATION_DIRECTION_SYMMETRIC,
		.srtp_max_lifetime_32 = 0,
		.srtp_max_lifetime_16 = 0,
		.srtp_max_lifetime_0 = 0,
		.srtcp_max_lifetime = 0, /* Disable SRTP master re-keying lifetime*/
		.from_index_32 = 0,
		.from_index_16 = 0,
		.from_index_0 = 0,
		.to_index_32 = 0xffff,
		.to_index_16 = 0xffff,
		.to_index_0 = 0xffff,
	};

	num_of_param = 12;

	switch (srtp_option.param_5.bits.master_key_size)
	{
		case CIPHER_SRTP_OPT_MASTER_KEY_SIZE_BITS_128:
			master_key_size = 16;
			num_of_param += 16/2;
		break;

		case CIPHER_SRTP_OPT_MASTER_KEY_SIZE_BITS_192:
			master_key_size = 24;
			num_of_param += 24/2;
		break;

		case CIPHER_SRTP_OPT_MASTER_KEY_SIZE_BITS_256:
			master_key_size = 32;
			num_of_param += 32/2;
		break;
		default:
			printf("wrong master key size\n");
			return -1;
	}

	/*set a master key*/
	key_array = (char *)&srtp_key_derivation.data[0];
	memset(key_array, key, master_key_size);

	/*set a salt key*/
	key_array = (char *)&srtp_key_derivation.data[master_key_size];
	memset(key_array, key + 1, CIPHER_SRTP_OPT_SALT_KEY_SIZE_DEFAULT/8);
	num_of_param += 32/2; /*salt key padded to 256 bits*/

	/* allocate a message to build commands */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return VAPI_ERR_NOMEM;

	result = VAPI_SetMessageFromBuffer(message, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_CIPHER_SRTP_OPT, 5, (U16*)&srtp_option);

	if(result != SUCCESS)
		goto out_free;

	result = VAPI_SetMessageFromBuffer(message, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_CIPHER_SRTP_KEY_DERIVATION,
							num_of_param,
							(U16*)&srtp_key_derivation);

	if(result != SUCCESS)
		goto out_free;

	/* send the new CIPHER SRTP option  to the connection */
	result = VAPI_SendConnectionMessage(conn_id, (SMsg *)message, NULL, device_response, &response_len);

out_free:
	VAPI_FreeMessage(message);

	return result;
}
