/*============================================================================\n
*	This example show how to use the VAPI utilities APIs in conjunction
*	with the Comcerto header files to run a query / change connection setting.
*============================================================================*/

#include <vapi.h>
#include <gtl.h>
#include <msp.h>

/*=================================================================================
 *	This function send a configuration query to the MSP
 *	and store the answer in the device_response buffer
=================================================================================*/
VSTATUS connection_query(U32 conn_id, U16 function_code, void *message, U8 *device_response, U32 *response_len)
{
	VSTATUS result;

	/* reinitialize the message fields (remove a potential command from the message)*/
	VAPI_InitMessage(message);

	/* query the current voice option config*/
	result = VAPI_SetMessage(message, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_QUERY, function_code, 0);
	if(result != SUCCESS)
		return result;

	/* send the query, the response is stored in device_response*/
	result = VAPI_SendConnectionMessage(conn_id, (SMsg *)message, NULL, device_response, response_len);

	return result;
}


/*=================================================================================
 *	This function set the VAD of a connection
=================================================================================*/
VSTATUS connection_set_vad(U32 conn_id, U8 vad_option)
{
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];
	struct _VOIP_VCEOPT *this_vce_options;
	VSTATUS result;

	/* allocate a message to build commands */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return VAPI_ERR_NOMEM;

	/* send a VCE_OPT query to the connection (quesry response stored in device_response buffer*/ 
	result = connection_query(conn_id, FC_VOIP_VCEOPT, message, device_response, &response_len);
	if(result != SUCCESS)
		goto out_free;

	/* Point to the beginning of the parameters */
	/* The VAD bits are in the param 4 of VCEOPT command*/
	this_vce_options = (struct _VOIP_VCEOPT*) &device_response[sizeof(struct comcerto_api_hdr)];

	/* This is only required for BIG ENDIAN host*/
	this_vce_options->param_4.word = UT_CPU2LE16(this_vce_options->param_4.word);

	/* Check if the VAD option is not out of range*/
	if(vad_option > VOIP_VCEOPT_VADTYPE_ENABLE_PT13)
	{
		result = VAPI_ERR_UNSUPP_FEATURE;
		goto out_free;
	}

	/* Set the new VAD value*/
	this_vce_options->param_4.bits.vadtype = vad_option;

	/* This is only required for BIG ENDIAN host*/
	this_vce_options->param_4.word = UT_CPU2LE16(this_vce_options->param_4.word);

	/* re-use the same message structure (remove the query command)*/
	VAPI_InitMessage(message);

	/* build the command to change the voice VAD options*/
	result = VAPI_SetMessageFromBuffer(message, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, 
			FC_VOIP_VCEOPT, 5, (U16 *)this_vce_options);

	if(result != SUCCESS)
		goto out_free;

	/* send the new VCEOPT to the connection */
	result = VAPI_SendConnectionMessage(conn_id, (SMsg *)message, NULL, device_response, &response_len);

out_free:
	VAPI_FreeMessage(message);

	return result;
}


#define ALIGN_MESSAGE_LENGTH(size) (((size) + 3) & ~0x3)
/*=================================================================================
*	This function send a multi configuration query to the MSP
*	according the passed function codes.
*	For each function code in prints it, the len of the API, the number of parameters and the parameters values.
*	The usage is:
*	get_multi_config(conn_id, 4, FC_VOIP_ECHOCAN, FC_VOIP_VCEOPT, FC_VOIP_DTMFOPT, FC_CIPHER_SRTP_OPT);
=================================================================================*/

VSTATUS get_multi_config(U32 conn_id, U16 num_of_fc, ...)
{
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];
	VSTATUS result;
	U8 api_len = 0;				/* length of a single API */
	U8 current_len = 0;			/* offset in the device_response */
	U16 *current_param = NULL;		/* Pointer to param in a single API response */
	struct comcerto_api_hdr *this_api;	/* Pointer to api header structure */
	int i;
	va_list fc_list;

	/* allocate a message to build commands */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return VAPI_ERR_NOMEM;

	va_start (fc_list, num_of_fc );           /* Initializing arguments to store all values after num_of_fc */
	for (i = 0; i < num_of_fc; i ++)
	{
		/* Build the multi query command */
		result = VAPI_SetMessage(message, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_QUERY, va_arg(fc_list, int), 0);
		if(result != SUCCESS)
		{
			va_end (fc_list); 
			return result;
		}
	}

	/* send the multi query, the response is stored in device_response*/
	result = VAPI_SendConnectionMessage(conn_id, (SMsg *)message, NULL, device_response, &response_len);

	/* get the size of full reponse */
	current_len = response_len;
	/* Point to first API response */
	this_api = (struct comcerto_api_hdr*) device_response;
	/* Point to first API parameters (parameters start after the API header) */
	current_param = (U16 *)(this_api + 1);

	/* Parse all the response buffer */
	while(current_len)
	{
		printf("API Function Code 0x%03x (len %d), number of parameters %d\n", this_api->func_code, 
											this_api->length,
											(this_api->length - sizeof(struct comcerto_api_hdr))/2);

		/*Print all the parameters for this API config*/
		for (i = 0; i < (this_api->length - sizeof(struct comcerto_api_hdr))/2; i++)
		{
			printf("Param %d = 0x%04x\n", i, *current_param++);
		}

		api_len = this_api->length;
		/* Take in account padding if any*/
		api_len = ALIGN_MESSAGE_LENGTH(api_len);
		current_len -= api_len;

		/* Now point to the next API single query response*/
		this_api = (struct comcerto_api_hdr*) ((U8*)this_api + api_len);
		/* Now point to the parameters of the next API single query response*/
		current_param = (U16 *)(this_api + 1);
	}

	return SUCCESS;
}