/*=================================================================================
Local Memory Playback
This is a VAPI based code implementation of a function which can be used to play
announcement using the Local Memory Playback feature of the MSP.
=================================================================================*/
#include <stdio.h>
#include <vapi.h>
#include <msp.h>
#include <ut.h>

/*=================================================================================*/
/*!
*	\brief
*	This function plays a buffer to a connection.
*	conn_id 	Connection ID to play the buffer to
*	fileid  	File ID to play from
*	side 		Side to play TDM=0, IP=1 or BOTH=2.
*	time_to_play	Number of time to play the buffer
*	completion_ind	Set the completion indication (see CRM)
*/
/*=================================================================================*/
VSTATUS start_playback_buffer(unsigned int conn_id,
				unsigned int fileid,
				unsigned int side,
				unsigned int time_to_play,
				unsigned int completion_ind)
{
	VSTATUS result;

	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/* allocate a message to query the current VCEOPT */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;

	/*If not both sides, play to the required side*/
	if (side != 2)
		result = VAPI_SetMessage(message , CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE,
					FC_PLAYBACK_START, 4, side, fileid, time_to_play, completion_ind);
	else	/*If both sides*/
	{
		/*Play to TDM*/
		result = VAPI_SetMessage(message , CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE,
					FC_PLAYBACK_START, 4, 0, fileid, time_to_play, completion_ind);
		/*Play to IP*/
		result |= VAPI_SetMessage(message , CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE,
					FC_PLAYBACK_START, 4, 1, fileid, time_to_play, completion_ind);
	}
	if(result != SUCCESS)
		goto out;

	result = VAPI_SendConnectionMessage(conn_id, (SMsg *)message, NULL, device_response, &response_len);

out:
	VAPI_FreeMessage (message);
	return result;
}


/*=================================================================================*/
/*!
*	\brief
*	This function stops to play a buffer to a connection.
*	conn_id 	Connection ID to stop to play.
*	side 		Stop TDM side=0, Stop IP side=1 or Stop BOTH=2.
*	time_to_stop	0=immediately, 1=at the end of the file
*/
/*=================================================================================*/
VSTATUS stop_playback_buffer(unsigned int conn_id,
				unsigned int side,
				unsigned int time_to_stop)
{
	VSTATUS result;

	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/* allocate a message to query the current VCEOPT */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;

	/*If not both sides, stop the required side*/
	if (side != 2)
		result = VAPI_SetMessage(message , CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE,
					FC_PLAYBACK_STOP, 2, side, time_to_stop);
	else
	{
		/*Stop TDM */
		result = VAPI_SetMessage(message , CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE,
					FC_PLAYBACK_STOP, 0, side, time_to_stop);

		/*Stop IP */
		result |= VAPI_SetMessage(message , CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE,
					FC_PLAYBACK_STOP, 1, side, time_to_stop);
	}

	if(result != SUCCESS)
		goto out;

	result = VAPI_SendConnectionMessage(conn_id, (SMsg *)message, NULL, device_response, &response_len);

out:
	VAPI_FreeMessage (message);
	return result;
}

/************************************************************************************************
THIS FUNCTION ***MUST BE CALLED*** JUST AFTER VAPI_OpenDevice()
This is required if extended MSP memory is needed.
Basically if more than 256K of memory is required for local memory playback 
*************************************************************************************************/
/*=================================================================================*/
/*!
*	\brief	
*	This function extends the MSP to more than 16MB.
*/
/*=================================================================================*/
VSTATUS extend_memory_setup(unsigned int dev_id, unsigned int ext_mem_size, unsigned int cachable_heap_size)
{
	VSTATUS result;

	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/* allocate a message to query the current VCEOPT */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;

	result = VAPI_SetMessage(message , CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE,
					FC_EXT_MEM_SETUP,
					2, /* 2 parameters */
					ext_mem_size, cachable_heap_size);

	if(result != SUCCESS)
		goto out;

	result = VAPI_SendDeviceMessage(dev_id, (SMsg *)message, NULL, device_response, &response_len);

out:
	VAPI_FreeMessage (message);
	return result;
}

/*=================================================================================*/
/*!
*	\brief
*	This function create, clear or delete a memory playback buffer.
*/
/*=================================================================================*/
VSTATUS set_playback_buffer(unsigned int dev_id,
				Boolean action,
				unsigned int file_number,
				unsigned int buffer_size)
{
	VSTATUS result;

	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/* allocate a message to query the current VCEOPT */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;

	result = VAPI_SetMessage(message , CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE,
					FC_PLAYBACK_SET_BUFFER,
					4, /* 4 parameters */
					action,
					file_number,
					buffer_size,
					buffer_size >> 16);

	if(result != SUCCESS)
		goto out;

	result = VAPI_SendDeviceMessage(dev_id, (SMsg *)message, NULL, device_response, &response_len);

out:
	VAPI_FreeMessage (message);
	return result;
}


/*=================================================================================*/
/*!
*	\brief
*	This function load a file buffer to MSP playback buffer.
*	dev_id	device to load the buffer to
*	file_id ID of the file to load
*	pt payload of the data from the file,
*	file_size,
*	char *file pointer to a buffer containing the data
*/
/*=================================================================================*/
VSTATUS set_playback_file(unsigned int dev_id,
			unsigned int file_id,
			char pt,
			unsigned int file_size,
			unsigned char *file)
{

	VSTATUS result;
	unsigned int byte_to_send;
	unsigned int offset = 0;
	unsigned len = 0;

	void *message;
	U16 ext_sync_eof[MAX_SYNCEOF_SIZE];

	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/*Build the PLAYBACK_SET_FILE and sent it to the MSP*/
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;

	result = VAPI_SetMessage(message , CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE,
					FC_PLAYBACK_SET_FILE,
					4, /* 4 parameters */
					file_id,
					pt,
					file_size,
					file_size >> 16);

	if(result != SUCCESS)
		goto out;

	result = VAPI_SendDeviceMessage(dev_id, (SMsg *)message, NULL, device_response, &response_len);

	/*Now build as many Extended SYNCEOF messages required to load the full file to the MSP buffer*/
	VAPI_InitMessage(message);
	response_len = DEFAULT_FIFO_MAX_SIZE;

	byte_to_send = file_size;

	while(byte_to_send > 0)
	{
		UT_MemSet(ext_sync_eof, 0 , MAX_SYNCEOF_SIZE);
		VAPI_InitMessage(message);
		response_len = DEFAULT_FIFO_MAX_SIZE;

		if(byte_to_send < MAX_SYNCEOF_DATA)
			len = byte_to_send ;
		else
			len = MAX_SYNCEOF_DATA;

		ext_sync_eof[0] = PACKET_TYPE_LOAD_PLAYBACK;
		ext_sync_eof[2] = file_id;
		ext_sync_eof[4] = offset;
		ext_sync_eof[5] = offset >> 16;
		UT_MemCopy(&ext_sync_eof[6], &file[offset], len);

		result = VAPI_SetMessageFromBuffer(message , CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE,
						FC_SYNCEOF,
						(len + 12)/2, ext_sync_eof);

		result = VAPI_SendDeviceMessage(dev_id, (SMsg *)message, NULL, device_response, &response_len);

		byte_to_send -= len;
		offset += len;
	}

out:
	VAPI_FreeMessage (message);

	return result;
}


/**functions call example.
These functions can be called during the device initialization i.e after VAPI_DeviceInit()
This has to be once at the init of the system */

/**This buffer must contains PCM data accordingly the set_playback_file payload type used*/
unsigned char myfile_pattern_data[100];

VSTATUS init_playback(int device_id)
{
	VSTATUS status;
	/*With 16M of SDRAM for the MSP only 256k can be reserved for playback buffer*/
	status = set_playback_buffer(device_id, 0, 1, 1024*256);

	status = set_playback_file(device_id, 1, VOIP_VOPENA_RTP_PT_PCMA,
					sizeof(myfile_pattern_data),
					(unsigned char *)myfile_pattern_data);
	return status;
}

/**This function plays the file which has been loaded with the ID 1 to the memory playback buffer */
VSTATUS play_myfile_tone_pattern(unsigned int conn_id)
{
	VSTATUS status;
	/*Play file ID 1 to TDM forever*/
	status = start_playback_buffer(conn_id, 1, 0, 0xff, 0);

	return status;
}
