/*=================================================================================
 IPV6 example
This is a VAPI based code implementation of 2 functions which can be used to configure a device 
and a conention for IP6 operations.
=================================================================================*/

#include <arpa/inet.h>		/* for inet_pton function */
#include <linux/if_ether.h>	/* for ETH_P_IPV6 ETH_P_IPV6 */

#include <vapi.h>
#include <msp.h>

#include <comcerto-ipv6-types.h>
#include <comcerto-ethernet-interface-types.h>

#define DEVICE_IPV4 "192.168.32.222"
#define DEVICE_IPV6 "fec0:0:0:1f::3"


/*=================================================================================*/
/*
*	This function buils a multicommand to set a MSP channel in IPv6 mode and the IPv6 header.
*	The IPv6 source and destination are provided in IPv6 format string such as "fec0:0:0:1f::3".
*	We make the assumption that a endpoint structure contains some information such as UDP ports.
*/
/*=================================================================================*/
int set_ipv6_udp_parameters(int connid, char *ipv6_src_addr, char *ipv6_dst_addr, unsigned short src_udp, unsigned short dst_udp)
{
	int result;
	struct _SET_IPV6_HDR_CHANNEL ipv6_hdr; /* this struct is defined in comcerto-ipv6-types.h */

	void *config_message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/* allocate a message */
	config_message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (config_message == NULL)
		return -1;

	/* add the command channel mode to the message (IPV6)*/
	result = VAPI_SetMessage(config_message, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, 
			FC_SET_CHANNEL_IP_MODE, 1, (U16 *) 0x0001);

	if(result != SUCCESS)
		goto err_free;

	memset(&ipv6_hdr, 0, sizeof(struct _SET_IPV6_HDR_CHANNEL));

	ipv6_hdr.param_4.bits.serviceid = SET_IPV6_HDR_CHANNEL_SERVICEID_DEFAULTPKT;

	ipv6_hdr.param_5.word = 0x0060;	/*Traffic class bit 0:3, version, flow label bits 0:3, traffic class bit 4:7*/
	ipv6_hdr.param_6.word = 0x0000;	/*flow label bits 4:11 & 12:19*/

	ipv6_hdr.payload_len = 0x0000;	/*len (overwritten) */

	ipv6_hdr.param_8.bits.next_header_id = 0x11;
	ipv6_hdr.param_8.bits.hop_limit = 0x80;

	/* Set the IPv6 source Address (place it at the 1st U16 of the IPv6 src addr in the ipv6_hdr struct */
	inet_pton(AF_INET6, ipv6_src_addr, &ipv6_hdr.ipv6_src_15_0);
	/* Set the IPv6 destination Address (place it at the 1st U16 of the IPv6 dst addr in the ipv6_hdr struct */
	inet_pton(AF_INET6, ipv6_dst_addr, &ipv6_hdr.ipv6_dst_15_0);

	/* set UDP ports */
	ipv6_hdr.uh_sport = htons(src_udp);
	ipv6_hdr.uh_dport = htons(dst_udp);

	ipv6_hdr.uh_ulen = 0x0000;
	ipv6_hdr.uh_sum = 0x0000;

	/* Add the IPv6 header to the command */
	result = VAPI_SetMessageFromBuffer(config_message, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, 
			FC_SET_IP_HDR_CHANNEL, sizeof(struct _SET_IPV6_HDR_CHANNEL)/2, (U16 *)&ipv6_hdr);

	if(result != SUCCESS)
		goto err_free;

	/* send the command to the device, the response is stored in device_response */
	result = VAPI_SendConnectionMessage(connid, (SMsg *)config_message, NULL, device_response, &response_len);

err_free:
	VAPI_FreeMessage (config_message);

	return result;
}

/*=================================================================================*/
/*
*	This function configures the Comcerto device to operates in IPv4 and IPv6 mode.
*	The IPv4 address is provided in IPv4 format string such as "192.168.32.222".
*	The IPv6 address is provided in IPv6 format string such as "fec0:0:0:1f::3".
*/
/*=================================================================================*/
int network_parameters_initialisation(int device_id, unsigned char *device_mac, unsigned char *default_dest_mac)
{
	int result;
	SIpAddrInfo ip_info;
	U16 ipv6_addr[8];
	void *config_message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];
	struct _SET_ETH_HDR eth_hdr;

	/* allocate a message */
	config_message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (config_message == NULL)
		return -1;

	/* configure the ethernet header for IPv4 operation */ 
	result = VAPI_SetEthMac(device_id, CMD_LEVEL_DEVICE, device_mac, default_dest_mac, NULL);
	if(result != SUCCESS)
		goto err_free;

	/* prepare the ethernet header for IPv6 operation */ 
	memset(&eth_hdr, 0x00, sizeof(struct _SET_ETH_HDR));
	eth_hdr.param_4.bits.protocol = SET_ETH_HDR_PROTOCOL_IP_V6;
	eth_hdr.param_4.bits.action = SET_ETH_HDR_ACTION_REGISTER;
	memcpy(&(eth_hdr.assignedmac), device_mac, 6);
	memcpy(&(eth_hdr.hostmac), default_dest_mac, 6);
	eth_hdr.type = htons(ETH_P_IPV6);

	/* Set the ETH_HDR for IPv6 */
	result = VAPI_SetMessageFromBuffer(config_message, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, 
			FC_SET_ETH_HDR, 8, (U16 *)&eth_hdr); /*8 parameters to send*/

	if(result != SUCCESS)
		goto err_free;

	/* send the ETH_HDR for IPv6 command */
	result = VAPI_SendDeviceMessage(device_id, (SMsg *)config_message, NULL, device_response, &response_len);
	if(result != SUCCESS)
		goto err_free;

	/* set IPv4*/
	ip_info.ucNumOfSrcIpAddr = 1;
	ip_info.bIsMultipleMode = 0;
	ip_info.ucEncapsulation = 2;
	ip_info.ucChkSumVerification = 1;

	/* Set the IPv4 device Address*/
	inet_pton(AF_INET, DEVICE_IPV4, ip_info.auiDevIPAddress);

	/* configure the IPv4 address.
	this command also configures the IP_SERVICE_CONFIG, ARP_SERVICE_CONFIG,ICMP_SERVICE_CONFIG*/ 
	result = VAPI_SetDeviceIPAddr(device_id, &ip_info, NULL);
	if (result != SUCCESS)
		goto err_free;

	/* reuse the config_message */
	VAPI_InitMessage (config_message);

	/* Set the IPv6 device Address*/
	inet_pton(AF_INET6, DEVICE_IPV6, ipv6_addr);

	result = VAPI_SetMessageFromBuffer(config_message, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_IP_ADDRESS_V6, 8, ipv6_addr);
	if(result != SUCCESS)
		goto err_free;

	/* send IPv6 address command*/
	result = VAPI_SendDeviceMessage(device_id, (SMsg *)config_message, NULL, device_response, &response_len);

err_free:
	VAPI_FreeMessage (config_message);

	return result;
}

