/*=================================================================================
Diagnostic example
This is a VAPI based code implementation of a function which can be used to configure the diagnostic
The structure redirect_info is filled with the required parameters
Then a call to the funcion will set the redirection parameters then the diagnostic command 
=================================================================================*/
/*Function usage:

	// function call example 
	static U8 default_dest_mac[6] = {0x00, 0x11, 0xD8, 0xE7, 0xA5, 0x52};

	unsigned int device_src_ip = htonl(0xc0a820b1); // 192.168.32.177

	redirect_info diag_redirect;

	// prepare the diag redirect over CSME 
	diag_redirect.type = FC_REDIRECT_CSM;
	// destination MAC address to redirect to
	memcpy(&diag_redirect.dest_mac, default_dest_mac, 6);

	// conn_id & action are not relevant for redirection
	status = configure_diag_redirect(&diag_redirect);

	// enable TDM diag for connection 0
	status = set_diagnostics(0, CMD_CLASS_OPEN_DIAG, CMD_TYPE_DIAG_MON_LIVE_CTRL, 0x0e00, 1);
*/

#include <stdio.h>
#include "vapi.h"
#include "msp.h"
#include "comcerto-ud-types.h"


/* Diagnostics redirection structure */
typedef struct
{
	unsigned char type;			/**< over CSME or IP/UDP*/
	unsigned char src_mac[6];	/**< source MAC address (must be the same as the one set to the device)*/
	unsigned char dst_mac[6];	/**< destination MAC address to redirect to*/
	unsigned int src_ip;		/**< source IP address (must bethe same as the one set to the device)*/
	unsigned int dst_ip;		/**< destination IP address to redirect to*/
	unsigned short src_udp;		/**< Source UDP port used in case of IP/UDP redirection type*/	
	unsigned short dst_udp;		/**< Destination UDP port used in case of IP/UDP redirection type*/
} redirect_info;

int configure_diag_redirect(redirect_info *redirect, int device_id)
{
	VSTATUS result = SUCCESS;
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];
	struct _REDIRECT_CSM_UDP diag_cmd; /*defined in comcerto-ud-types.h */

	if(redirect != NULL)
		return -1;

	/* allocate a message to query the current options */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;

	memset(&diag_cmd, 0, sizeof(struct _REDIRECT_CSM_UDP));

	diag_cmd.param_4.bits.protocol = REDIRECT_CSM_PROTOCOL_OPENDIAGNOSTICS;
	memcpy(diag_cmd.dst_mac, redirect->dst_mac, 6);
	memcpy(diag_cmd.src_mac, redirect->src_mac, 6);

	if(redirect->type == FC_REDIRECT_CSM)
	{
		diag_cmd.packet_type = htons(REDIRECT_CSM_TYPE_DEFAULT); /* csme protocol*/
	
		/* build the command to set the diag redirection*/
		result = VAPI_SetMessageFromBuffer(message, 
						CMD_CLASS_OPEN_DIAG,
						CMD_TYPE_DIAG_CONFIG, 
						FC_REDIRECT_CSM,
						8,		/*8 words in the command*/
						(U16 *)&diag_cmd);
		if(result != SUCCESS)
			goto err;
	}
	else /* assume redirect over IP/UDP*/
	{
		diag_cmd.packet_type = htons(REDIRECT_CSM_UDP_PACKET_TYPE_DEFAULT); /* IP protocol*/
	
		diag_cmd.param_12.bits.ip_hl = REDIRECT_CSM_UDP_IP_HL_DEFAULT;
		diag_cmd.param_12.bits.ip_v = REDIRECT_CSM_UDP_IP_V_DEFAULT;
		diag_cmd.param_12.bits.ip_tos = REDIRECT_CSM_UDP_IP_TOS_DEFAULT;
	
		diag_cmd.ip_len = REDIRECT_CSM_UDP_IP_LEN_DEFAULT;
		diag_cmd.ip_id = REDIRECT_CSM_UDP_IP_ID_DEFAULT;
		diag_cmd.ip_fragment = REDIRECT_CSM_UDP_IP_FRAGMENT_DEFAULT;
		diag_cmd.param_16.word = htons(0x8011); /* UDP , TTL*/
	
		diag_cmd.ip_checksum = REDIRECT_CSM_UDP_IP_CHECKSUM_DEFAULT;
	
		diag_cmd.ip_src = redirect->src_ip;
		diag_cmd.ip_dst = redirect->dst_ip;
		diag_cmd.udp_sport = redirect->src_udp;
		diag_cmd.udp_dport = redirect->dst_udp;
	
		diag_cmd.udp_len = UT_CPU2LE16(REDIRECT_CSM_UDP_UDP_LEN_DEFAULT);
		diag_cmd.udp_checksum = UT_CPU2LE16(REDIRECT_CSM_UDP_UDP_CHECKSUM_DEFAULT);
	
		/* build the command to set the diag redirection*/
		result = VAPI_SetMessageFromBuffer(message, CMD_CLASS_OPEN_DIAG, CMD_TYPE_DIAG_CONFIG, 
			FC_REDIRECT_CSM_UDP, sizeof(struct _REDIRECT_CSM_UDP)/2, (U16 *)&diag_cmd);
	
		if(result != SUCCESS)
			goto err;
	}

	result = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, device_response, &response_len);
	if(result != SUCCESS)
		goto err;

err:
	VAPI_FreeMessage(message);
	return result ;
}


int set_diagnostics(unsigned int id, unsigned int level,   /* if 0xFFFF device else connection*/
			unsigned char command_class, 
			unsigned char command_type, 
			unsigned short function_code, 
			unsigned char action) /* 1 = enable, 0 = disable */
{
	VSTATUS result = SUCCESS;
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/* allocate a message to query the current options */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;

	/* build the command to set the diag config to enable or disable */
	result = VAPI_SetMessage(message, command_class, command_type, function_code, 1, action);
	if(result != SUCCESS)
		goto err;

	if(level == 0xFFFF)
		result = VAPI_SendDeviceMessage(id, (SMsg *)message, NULL, 
		device_response, &response_len);
	else
		result = VAPI_SendConnectionMessage(id, (SMsg *)message, NULL, 
		device_response, &response_len);

err:
	VAPI_FreeMessage(message);
	return result;
}






