/*! \file vapi_pots_utility.c
 @defgroup Utilities vapi pots to pots application example utilities
 *  @{
 */
 
/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include <string.h>
#include <vapi/msp.h>
#include <vapi/vapi.h>
#include <vapi/gtl.h>

#include "vapi_pots_type.h"
#include "vapi_pots_var.h"
#include "vapi_pots_func.h"
#include "readcfg.h"

#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>

extern u_int16_t device_id;
extern u_int16_t endpoint_id;
extern u_int16_t mode;

/* default values used later in default_device_configuration */

/* device control MAC used at boot stage
 *
 * On Slave configuration this address is set by the MAAS_ASSIGN command (VAPi_AssignBootMAC)
 * On Master configuration this address must be the same as the on set by the bootloader (u-boot, ARMboot),
 * the default value is 00:11:22:33:44:55
 */
/*00:03:C0:A8:20:28*/
#define DEVICE_CONTROL_MAC		{0x00,0x11,0x22,0x33,0x44,0x55}

/* host control MAC
 *
 * On slave device this MAC address is the MAC address on the network interface
 * used to control the device. For example if a PC is used to control the device,
 * it is the MAC address of the network card
 */
#define HOST_CONTROL_INTERFACE		"eth1"
#define HOST_CONTROL_MAC		{0x00,0x1A,0x1B,0x1C,0x1D,0x1E}

/* This structure holds default values for the device configuration, user can
 * overwrite them somehow in set_device_configuration() function using his own
 * config. scheme.
 */
static SCSMEUsrData default_device_configuration = {
	NULL,			/* pointer to next device */
	eCSM_ITF,		/* control interface to use */
	0,			/* device ID */
	0,			/* Version infor (not used) */
	eMASTER,		/* Slave / Master mode */
	DEV_TYPE_M825XX,	/* Device type */
	True,			/* Default or custom max channels */
	0,			/* Max Channels if above flag is custom */
	DEVICE_CONTROL_MAC,	/* MAC address for control over csmencaps */
	HOST_CONTROL_MAC,	/* MAC address of the host interface controling the device */
	(char *)HOST_CONTROL_INTERFACE,	/* host interface used to control the device */
	1			/* csme ack required */
};

/*=================================================================================*/
/*! \brief
 *	
 *	This function fills the global structure with device config parameters
*/
void set_device_configuration(U16 device_id)
{
	U8 *mac;

	memcpy(&gtl_device_configuration, &default_device_configuration, sizeof(SCSMEUsrData));

	/* user may put here code to overwrite default config parameters, for example
	 * read them from config file
	 */

	gtl_device_configuration.uiDevId = device_id;
	gtl_device_configuration.pucEthDevName = HOST_CONTROL_INTERFACE;

	/* this code is executed before VAPI_Init, so we have to use printf's */

	printf("CONTROL_INTERFACE_TYPE  = %d\n", gtl_device_configuration.usControlInterface);
	printf("DEVICE_ID    = %d\n", gtl_device_configuration.uiDevId);
	printf("DEVICE_MODE  = %d\n", gtl_device_configuration.eDevMode);
	printf("DEVICE_TYPE  = %d\n", gtl_device_configuration.ucDevType);
	printf("USE_DEFAULT_MAX_CHANNEL = %d\n", gtl_device_configuration.bUseDefaultMaxChnls);
	printf("CUSTOM_MAX_CHANNEL      = %d\n", gtl_device_configuration.usMaxChannels);
	printf("ACK_REQUIRED = %d\n", gtl_device_configuration.ucIsAckReqd);

	mac = gtl_device_configuration.aucDevMac;
	printf("DEVICE MAC = %02x:%02x:%02x:%02x:%02x:%02x\n", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

	mac = gtl_device_configuration.aucHostMac;
	printf("HOST   MAC = %02x:%02x:%02x:%02x:%02x:%02x\n", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

	printf("HOST_CONTROL_INTERFACE  = %s\n\n", gtl_device_configuration.pucEthDevName);
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function change the default VAPI configuration.\n
 *
*/
VSTATUS set_default_config(IN OUT SVAPIConfig * default_config)
{
	struct _VOIP_DTMFOPT *my_dtmf_options = &(default_config->pstVoIPChnlParams->stDtmfOpt);

	/* change the default timeout to 10S */
	default_config->uiMspRespTimeout = 10;	/* MSP response timeout value 10 seconds */

	/* change the default VAPI DTMF options */
	my_dtmf_options->param_4.bits.event_report = VOIP_DTMFOPT_EVENT_REPORT_DEFAULT;
	my_dtmf_options->param_4.bits.regeneration = VOIP_DTMFOPT_REGENERATION_DEFAULT;
	my_dtmf_options->param_4.bits.removal = VOIP_DTMFOPT_REMOVAL_DEFAULT;
	my_dtmf_options->param_4.bits.redundancy = VOIP_DTMFOPT_REDUNDANCY_DEFAULT;
	my_dtmf_options->param_4.bits.dtmf_rtp = VOIP_DTMFOPT_DTMF_RTP_DEFAULT;
	my_dtmf_options->param_4.bits.dtmf_voice = VOIP_DTMFOPT_DTMF_VOICE_DEFAULT;

	return SUCCESS;
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function sets an event for the specified endpoint
*/
void set_endpoint_event(int event_type, int index)
{
	endpoints[index].event = event_type;
	PDEBUG (DEBUG_FUNCTION, "Endpoint %d set event %d (state %d)", index,endpoints[index].event, endpoints[index].state);
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function sets an endpoint in the specified state
*/
void set_endpoint_state(int state, int index)
{
	endpoints[index].state = state;
	PDEBUG (DEBUG_FUNCTION, "Endpoint %d, set state %d (event %d)", index,endpoints[index].state, endpoints[index].event);
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function sets an endpoint in the specified side
*/
void set_endpoint_side(int side, int index)
{

	endpoints[index].side = side;
	PDEBUG (DEBUG_FUNCTION, "Endpoint %d, set side %s", index, endpoints[index].side == IP?"IP":"RTP");
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function sets an endpoint in the specified peer_destroy state - for destroy connection by 
 *	peer connection. It's needed in case if channel was created by peer side
*/
void set_endpoint_peer_destroy( int on_off, int index)
{
	endpoints[index].peer_destroy = on_off;
	PDEBUG (DEBUG_FUNCTION, "Endpoint %d, peer_destroy %s", index, endpoints[index].peer_destroy == ENABLE?"enabled":"diasabled");
}

	/*=================================================================================*/
/*! \brief
 *	
 *	This function sets an endpoint in the specified peer_destroy state - for destroy connection by 
 *	peer connection. It's needed in case if channel was created by peer side
*/
void set_endpoint_voice_mail_mode(int voice_mail_mode, int index)
{
	endpoints[index].voice_mail_mode = voice_mail_mode;

	if (voice_mail_mode > 0)
		VOICE_MAIL_MODE = 1;
	else
		VOICE_MAIL_MODE = 0;
	
	PDEBUG (DEBUG_FUNCTION, "Endpoint %d, voice_mail_mode: %s", index, endpoints[index].voice_mail_mode == DISABLE?"disabled": endpoints[index].voice_mail_mode == SERVER?"server":"caller");
}

/*=================================================================================
This function handle the endpoint timed connection.
=================================================================================*/
void timed_connection (void *timer_data)
{
	struct _ENDPOINT_DESC *this_endpoint = (struct _ENDPOINT_DESC*) timer_data;
	
	PDEBUG(DEBUG_INIT, "Endpoint %d, time up occured", this_endpoint->index);
	UT_TimerStop(this_endpoint->this_timer);
	
	set_endpoint_event(ENDPOINT_EVENT_TIME_UP, this_endpoint->index);
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function creates a VoIP endpoint in VAPI ASYNC mode using the endpoint ID as the timeslot number.
*/
void create_endpoint(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request;

	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_INIT,"create_endpoint:%d type %d timeslot %d (request id = 0x%04x)", endpoint_id, eVOIP, endpoint_id, request.uiReqId);

	result = VAPI_CreateConnection(device_id,
				endpoint_id,
				eVOIP,          /* type 2*/
				endpoint_id,    /* use the endpoint id as timeslot */
				U_LAW,
				&request,
				NULL);

}

/*=================================================================================*/
/*! \brief
 *	
 *	This function destroy a previously created endpoint in VAPI ASYNC mode.
*/
void destroy_endpoint(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request;

	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_INIT,"destroy_endpoint:%d  (request id = 0x%04x)", endpoint_id, request.uiReqId);

	result = VAPI_DestroyConnection(endpoint_id, &request);
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function generates a dialtone to the specified endpoint.
*/
void dialtone_generation(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request;

	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_INIT,"dial tone generation:%d (request id = 0x%04x)", endpoint_id, request.uiReqId);

	result = VAPI_PlayTone(endpoint_id, eDIALTONE, eDirToTDM, NULL, 0, &request);
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function generates a busytone to the specified endpoint.
*/
void busytone_generation(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request;

	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_INIT,"busy tone generation:%d (request id = 0x%04x)", endpoint_id, request.uiReqId);

	result = VAPI_PlayTone(endpoint_id, eBUSYTONE, eDirToTDM, NULL, 0, &request);
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function generates a busytone to the specified endpoint.
*/
void ringbacktone_generation(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request;

	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_INIT,"ring back tone generation:%d (request id = 0x%04x)", endpoint_id, request.uiReqId);

	result = VAPI_PlayTone(endpoint_id, eRINGBACKTONE, eDirToTDM, NULL, 0, &request);
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function stops a tone currently generated on the specified endpoint.
*/
void stop_tone_generation(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request;

	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_INIT,"stop tone generation:%d (request id = 0x%04x)", endpoint_id, request.uiReqId);

	result = VAPI_StopTone(endpoint_id, 0, 0, &request);
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function starts recording on endpoint in VAPI ASYNC mode using the endpoint ID as the timeslot number.
*/
void start_recording(int index, int endpoint_state)
{
	int result;
	int endpoint_id = endpoints[index].index;	
	SRequest request;
	SRecData stRecData;

	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	stRecData.eMediaHost = eCSMENCAPS;
	stRecData.eCodingScheme = eCS_G711_ULAW;
	stRecData.uiTimeOut = 10;
	stRecData.pucBuffer = media_buffer;
	stRecData.uiBufferLength = media_buffer_length;
	stRecData.pfnFrameHndlr = NULL;
	stRecData.uiHostSpeechDataFormat = speech_format;

	if (endpoints[index].side == IP)
		stRecData.eSourceType = eRTP;
	else
		stRecData.eSourceType = ePCM;

	PDEBUG(DEBUG_INIT, "start recording on endpoint: %d media_host %d coding_scheme %d source_type %d timeout %d speech_format %d  (request id = 0x%04x)", 
		endpoint_id, stRecData.eMediaHost, stRecData.eCodingScheme, stRecData.eSourceType, stRecData.uiTimeOut, stRecData.uiHostSpeechDataFormat, request.uiReqId);

	result = VAPI_StartRecord(endpoint_id, 
				&stRecData,
				&request);
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function stops recording on endpoint in VAPI SYNC mode using the endpoint ID.
*/
void stop_recording(int endpoint_id)
{
	PDEBUG(DEBUG_INIT, "stop recording on endpoint: %d ", endpoint_id);
	
	int result;
	result = VAPI_StopRecord(endpoint_id, stop_type, NULL);
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function starts playing on endpoint in VAPI ASYNC mode using the endpoint ID as the timeslot number.
*/
void start_playing(int index,int endpoint_state)
{
	int result;
	int endpoint_id = endpoints[index].index;
	SRequest request;
	SPlayData stPlayData;

	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	stPlayData.eMediaHost = eCSMENCAPS;
	stPlayData.eCodingScheme = eCS_G711_ULAW;
	stPlayData.ePlaybackMode = playback_mode;
	stPlayData.uiTimeOut = 10;
	stPlayData.pucBuffer = media_buffer;
	stPlayData.uiBufferLength = media_buffer_length;
	stPlayData.uiHostSpeechDataFormat = speech_format;

	if (endpoints[index].side == IP)
		stPlayData.eDestinationType = eRTP;
	else
		stPlayData.eDestinationType = ePCM;	

	PDEBUG(DEBUG_INIT, "start playing on endpoint: %d media_host %d coding_scheme %d destination_type %d playback_mode %d timeout %d  (request id = 0x%04x)", 
		endpoint_id, stPlayData.eMediaHost, stPlayData.eCodingScheme, stPlayData.eDestinationType, stPlayData.ePlaybackMode, stPlayData.uiTimeOut, request.uiReqId);

	result = VAPI_StartPlayback(endpoint_id, 
				&stPlayData,
				&request);
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function stops playing on endpoint in VAPI SYNC mode using the endpoint ID.
*/
void stop_playing(int endpoint_id)
{
	PDEBUG(DEBUG_INIT, "stop playing on endpoint: %d ", endpoint_id);
	
	int result;
	result = VAPI_StopPlayback(endpoint_id, stop_type, NULL);
}

/*=================================================================================*/
/*! \brief
*	
*	This function initializes the device network parameters:
*	- ETH header
*	- IP address.
*	The configuration is comming from the /usr/local/etc/vapi_pots.conf file
*/
int network_parameters_initialisation(void)
{
	struct _CFG *cfg_info;
	SIpAddrInfo ip_info;
	
	cfg_info = cfg_read ("/usr/local/etc/vapi_pots.conf", 0);

	if (!cfg_info)
	{
		PDEBUG(DEBUG_ERROR,"Config read: Can't Open config file device_configuration");
		return -1;
	}

	PDEBUG(DEBUG_INIT,"Init IPv4 device address");
	/* read the IPV4 address if  */
	READ_IP (cfg_info, "DEVICE_NETWORK_CONFIG", "DEVICE_IP_ADDRESS", le_device_ip_address, "192.168.1.2");
	printf("device ");
	print_ipv4_addr(&le_device_ip_address,1);
	be_device_ip_address = htonl(le_device_ip_address);

	PDEBUG(DEBUG_INIT,"Init device MAC address");
	READ_MAC (cfg_info,"DEVICE_NETWORK_CONFIG", "DEVICE_MAC_ADDRESS",&be_device_mac_addr[0],&def_mac_addr[0]);
	READ_LE_MAC (cfg_info,"DEVICE_NETWORK_CONFIG", "DEVICE_MAC_ADDRESS",&le_device_mac_addr[0],&def_mac_addr[0]);
	printf("device ");
	print_le_mac_addr(&be_device_mac_addr[0], 1);
	print_le_mac_addr(&le_device_mac_addr[0], 1);


	PDEBUG(DEBUG_INIT,"Init host MAC address");
	READ_MAC (cfg_info,"DEVICE_NETWORK_CONFIG", "HOST_MAC_ADDRESS",&be_host_mac_addr[0],&def_mac_addr[0]);
	READ_LE_MAC (cfg_info,"DEVICE_NETWORK_CONFIG", "HOST_MAC_ADDRESS",&le_host_mac_addr[0],&def_mac_addr[0]);
	printf("host ");
	print_le_mac_addr(&be_host_mac_addr[0], 1);
	print_le_mac_addr(&le_host_mac_addr[0], 1);

	VAPI_SetEthMac(device_id, CMD_LEVEL_DEVICE, (U8 *)&le_device_mac_addr[0], (U8 *)&le_host_mac_addr[0], NULL);

	ip_info.ucNumOfSrcIpAddr = 1;
	ip_info.bIsMultipleMode = 0;
	ip_info.ucEncapsulation = 2;
	ip_info.ucChkSumVerification = 1;

	memcpy(&ip_info.auiDevIPAddress[0], &be_device_ip_address, sizeof(u_int32_t));

	VAPI_SetDeviceIPAddr(device_id, &ip_info, NULL);

	cfg_clean (cfg_info);

	return 0;
}

/*=================================================================================*/
/*! \brief
*	
*	This function initializes the TDM buses:
*	The configuration is comming from the /usr/local/etc/vapi_pots.conf file
*/
int tdm_parameters_initialisation(void)
{
	struct _CFG *cfg_info;
	STdmSetupParams tdm_parameters;
	int result;

	/* open the configuration file*/
	cfg_info = cfg_read ("/usr/local/etc/vapi_pots.conf", 0);

	if (!cfg_info)
	{
		PDEBUG(DEBUG_ERROR,"Config read: Can't Open config file device_configuration");
		return -1;
	}

	READ_INT (cfg_info, "DEVICE_TDM_CONFIG", "TDM_BUS_MODE", tdm_parameters.usMode, 1);

	READ_INT (cfg_info, "DEVICE_TDM_CONFIG", "TDM_BUS_TO_ENABLE", tdm_parameters.usNoOfBus, 1);
 
	READ_INT (cfg_info, "DEVICE_TDM_CONFIG", "BUS_ID", tdm_parameters.astTdmBusParam[0].ucBusId, 0);
	READ_INT (cfg_info, "DEVICE_TDM_CONFIG", "TX_ORDER", tdm_parameters.astTdmBusParam[0].ucTxOrder, 0);
	READ_INT (cfg_info, "DEVICE_TDM_CONFIG", "RX_POLARITY", tdm_parameters.astTdmBusParam[0].ucRxPolarity, 0);
	READ_INT (cfg_info, "DEVICE_TDM_CONFIG", "TX_POLARITY", tdm_parameters.astTdmBusParam[0].ucTxPolarity, 0);
	READ_INT (cfg_info, "DEVICE_TDM_CONFIG", "RX_CLOCK_MODE", tdm_parameters.astTdmBusParam[0].ucRxClkMode, 0);
	READ_INT (cfg_info, "DEVICE_TDM_CONFIG", "TX_CLOCK_MODE", tdm_parameters.astTdmBusParam[0].ucTxClkMode, 0);
	READ_INT (cfg_info, "DEVICE_TDM_CONFIG", "RX_CLOCK_EDGE", tdm_parameters.astTdmBusParam[0].ucRxClkEdge, 0);
	READ_INT (cfg_info, "DEVICE_TDM_CONFIG", "TX_CLOCK_EDGE", tdm_parameters.astTdmBusParam[0].ucTxClkEdge, 1);
	READ_INT (cfg_info, "DEVICE_TDM_CONFIG", "FRAME_EDGE", tdm_parameters.astTdmBusParam[0].ucFrameEdge, 1);
	READ_INT (cfg_info, "DEVICE_TDM_CONFIG", "NUMBER_OF_TS", tdm_parameters.astTdmBusParam[0].usNumOfTS, 0x20);
	READ_INT (cfg_info, "DEVICE_TDM_CONFIG", "INVERTED_FRAME_SIGNAL", tdm_parameters.astTdmBusParam[0].ucInvertedFrmSig, 1);
	READ_INT (cfg_info, "DEVICE_TDM_CONFIG", "BIT_ORDER_RCV", tdm_parameters.astTdmBusParam[0].ucBitOrderRcv, 1);
	READ_INT (cfg_info, "DEVICE_TDM_CONFIG", "NUM_TDM_CLK_CYCLES_DELAY_TX", tdm_parameters.astTdmBusParam[0].usNumTdmClkCyclesDelayTx, 0);
	READ_INT (cfg_info, "DEVICE_TDM_CONFIG", "NUM_TDM_CLK_CYCLES_DELAY_RX", tdm_parameters.astTdmBusParam[0].usNumTdmClkCyclesDelayRx, 0);

	/* uses the same parameters for 4 buses*/
	memcpy(&tdm_parameters.astTdmBusParam[1],&tdm_parameters.astTdmBusParam[0],sizeof(STdmBusParams));
	memcpy(&tdm_parameters.astTdmBusParam[2],&tdm_parameters.astTdmBusParam[0],sizeof(STdmBusParams));
	memcpy(&tdm_parameters.astTdmBusParam[3],&tdm_parameters.astTdmBusParam[0],sizeof(STdmBusParams));

	result = VAPI_SetTDMParams(device_id, &tdm_parameters, NULL);

	/* Not really an issue ignore it this happens if we try to initialise the TDM several time */
	if (result == -CNF_ERR_TDM_CONFIG_PARMS_FAILED_SELECT_MULT_INSTANCE)
		result = SUCCESS;

	return result;
}

/*=================================================================================*/
/*! \brief
*	
*	This function gets the loopback mode from the configuration file:
*	The configuration is comming from the /usr/local/etc/vapi_pots.conf file
*/
int get_loopback_mode(void)
{
	struct _CFG *cfg_info;
	int loopback_mode;
	
	cfg_info = cfg_read ("/usr/local/etc/vapi_pots.conf", 0);

	if (!cfg_info)
	{
		PDEBUG(DEBUG_ERROR,"Config read: Can't Open config file device_configuration");
		return -1;
	}

	READ_INT (cfg_info, "CONNECTION_MNGT", "LOOPBACK_MODE", loopback_mode, 0);

	cfg_clean (cfg_info);

	return loopback_mode;
}

/*=================================================================================*/
/*! \brief
*	
*	This function sets the IP/UDP header of a endpoint. \n
*/
void set_ip_udp_parameters(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request;
	SIpParams ip_parameters;
	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_INIT,"set ip parameters:%d (request id = 0x%04x)", endpoint_id, request.uiReqId);

	/* same ip address for surce and estination */
	ip_parameters.uiDestIpAddr = be_device_ip_address;
	ip_parameters.uiSrcIpAddr = be_device_ip_address;
	ip_parameters.ucIPServiceId = htons(0);

	/* cross the UDP ports */
	if (request.uiReqId == ENDPOINT_STATE_WAIT_IP_PARAMETERS_ORIGINATE_ACK)
	{
		ip_parameters.usSrcUdpPort = htons(DEFAULT_UDP_PORT);
		ip_parameters.usDestUdpPort = htons(DEFAULT_UDP_PORT + 2);
	}
	else if (request.uiReqId == ENDPOINT_STATE_WAIT_IP_PARAMETERS_TERMINATE_ACK)
	{
		ip_parameters.usSrcUdpPort = htons(DEFAULT_UDP_PORT + 2);
		ip_parameters.usDestUdpPort = htons(DEFAULT_UDP_PORT);
	}
	else
		PDEBUG(DEBUG_INIT,"Wrong state for endpoint %d (request id = 0x%04x)", endpoint_id, request.uiReqId);

/* FIXME: we should here check the result of the VAPI_SetConnIpParams function */
	result = VAPI_SetConnIpParams(endpoint_id, &ip_parameters, &request);
}

/*=================================================================================*/
/*! \brief
*	
*	This function enables the RTP traffic of an endpoint. \n
*/
void start_voip_connection(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request;
	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_INIT,"Start connection on endpoint:%d (request id = 0x%04x)", endpoint_id, request.uiReqId);

/* FIXME: we should here check the result of the VAPI_EnableConnection function */
	result = VAPI_EnableConnection(endpoint_id, &request);

}

/*=================================================================================*/
/*! \brief
*	
*	This function switches an endpoint in T38 mode. \n
*/
void switch_voip_connection_to_t38(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request;
	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;


	PDEBUG(DEBUG_INIT,"Switch connection to T38 on endpoint:%d (request id = 0x%04x)", endpoint_id, request.uiReqId);

/* FIXME: we should here check the result of the VAPI_SwitchToT38 function */
	result = VAPI_SwitchToT38(endpoint_id, NULL, &request);

}

/*=================================================================================*/
/*! \brief
*	
*	This function handles the onhook event in several states of an endpoints.
*/
void onhook_handler(int index)
{
	int endpoint_id = endpoints[index].index;
	int peer_endpoint_id = endpoints[index].peer_index;

	if (!endpoints[index].peer_destroy)
	{
		/* switch to the state WAIT_DESTROY_CONNECTION_ACK to wait the result of destroy_conenction*/
		set_endpoint_state(ENDPOINT_STATE_WAIT_DESTROY_CONNECTION_ACK, endpoint_id);
		set_endpoint_event(ENDPOINT_EVENT_NONE, endpoint_id);
		destroy_endpoint(endpoint_id, ENDPOINT_STATE_WAIT_DESTROY_CONNECTION_ACK);
	}

	if (endpoints[endpoints[index].peer_index].peer_destroy)
	{
		/* switch to the state WAIT_DESTROY_CONNECTION_ACK to wait the result of destroy_conenction*/
		set_endpoint_state(ENDPOINT_STATE_WAIT_DESTROY_CONNECTION_ACK, peer_endpoint_id);
		set_endpoint_event(ENDPOINT_EVENT_NONE, peer_endpoint_id);
		destroy_endpoint(peer_endpoint_id, ENDPOINT_STATE_WAIT_DESTROY_CONNECTION_ACK);
	}
}

int write_file (void)
{
	PDEBUG (DEBUG_INIT,"call write_file");

	DIR *dp;
	FILE *fp;	
      	struct dirent *ep;
	char filename[STR_SIZE];
	char file_names [200][STR_SIZE];	
	U8 *buf = media_buffer;
	int size = media_buffer_length;
	int i, j;
	int edge = 0;
	int last_found = 0;
	
	dp = opendir ("./");
	if (dp != NULL)
	{
		while (1)
		{
			ep = readdir (dp);
			if (ep)
			{
				edge++;
				memcpy(file_names[edge], ep->d_name, STR_SIZE);
			}
			else
				break;
		}

		(void) closedir (dp);
	}
	else
		perror ("Couldn't open the directory");	

	for (j = 1; j < edge; j ++)
	{
		snprintf(filename, STR_SIZE, "media_file_%d.rec", j);	
		for (i = 1; i < edge; i++)
		{
			if (!strncmp(filename, file_names[i], STR_SIZE))
				last_found = j;
		}			
	}

	snprintf(filename, STR_SIZE, "media_file_%d.rec", last_found  + 1);
	PDEBUG (DEBUG_FUNCTION, "filename =%s", filename);

	if (buf == NULL)
	{
		PDEBUG (DEBUG_ERROR, "NULL pointer is passed for memory to write");
		return FAILURE;
	}

	fp = fopen (filename, "w");
	if (fp == NULL)
	{
		PDEBUG (DEBUG_ERROR,  "write_file fopen FAILURE");
		perror (filename);
		return FAILURE;
	}

	if (fwrite (buf, sizeof (U8), size, fp) != size)
	{
		PDEBUG (DEBUG_ERROR, "error writing file(%s)", filename);
		fclose (fp);
		return FAILURE;
	}

	fclose (fp);
	return SUCCESS;
}


int read_file (void)
{
	PDEBUG (DEBUG_INIT,  "call read_file");
	
	DIR *dp;
	FILE *fp;	
      	struct dirent *ep;
	char filename[STR_SIZE];
	char file_names [200][STR_SIZE];
	U8 **buf = &media_buffer;
	int *size = &media_buffer_length;
	int i, j;
	int edge = 0;
	int last_found = 1;

	dp = opendir ("./");
	if (dp != NULL)
	{
		while (1)
		{
			ep = readdir (dp);
			if (ep)
			{
				edge++;
				memcpy(file_names[edge], ep->d_name, STR_SIZE);
			}
			else
				break;			
		}

		(void) closedir (dp);
	}
	else
		perror ("Couldn't open the directory");	

	for (j = edge; j > 1; j --)
	{
		snprintf(filename, STR_SIZE, "media_file_%d.rec", j);	
		for (i = edge; i > 1; i--)
		{
			if (!strncmp(filename, file_names[i], STR_SIZE))
				last_found = j;
		}

		if (last_found > 1)
			break;
	}

	snprintf(filename, STR_SIZE, "media_file_%d.rec", last_found);
	PDEBUG (DEBUG_FUNCTION, "filename =%s", filename);

	fp = fopen (filename, "r");
	if (fp == NULL)
	{
		PDEBUG (DEBUG_ERROR, "read_file fopen FAILURE");
		perror (filename);
		goto set_to_zero;
	}

	/* Figure out how big the size of the file and allocate that much space in memory */
	fseek (fp, 0, SEEK_END);
	*size = ftell (fp);

	*buf = (unsigned char *) __UT_malloc(*size, "read_file", 0);
	if (*buf == NULL)
	{
		PDEBUG (DEBUG_ERROR, "error allocating memory");
		fclose (fp);
		goto set_to_zero;
	}

	fseek (fp, 0, SEEK_SET);

	if (fread (*buf, sizeof (unsigned char), *size, fp) != *size)
	{
		PDEBUG (DEBUG_ERROR, "error reading file(%s)", filename);
		__UT_free(*buf, "read_file", 0);
		fclose (fp);
		goto set_to_zero;
	}

	fclose (fp);
	
	return SUCCESS;

set_to_zero:
	*buf = NULL;
	*size = 0;
	return FAILURE;
}

