/*! \file vapi_pots_main.c
 @defgroup Handlers vapi pots to pots application example event and state machine handlers
 *  @{
 */
/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

/*! \mainpage VAPI based pots to pots application using the VAPI and LEGERITY (SLIC control) libraries services.

	This example is a simple pots to pots multi threads application intended to be used on M82xxx EVM running in Master mode.
	All the VAPI function calls are performed in asynchronous mode.

	<b>Building/installing this application under Linux:</b>\n
	It requires VAPI to be built with this options :\n
		- make GTL=CSME install

	The Makefile provided has to be edited to configured the FSROOT to match the target file system installation.\n
	To build the application just enter make.
	Running make install installs the vapi_pots executable to the FSROOT/usr/local/bin directory.\n
	Running make install_conf installs the vapi_pots.conf configuration file to the FSROOT/usr/local/etc directory.\n

	<b>Configuring the application under Linux:</b>\n
	The configuration of the TDM parameters, MAC and IP addresses are modifiable in the /usr/local/etc/vapi_pots.conf
	This parameters have to be modified prior the application is started.

	<b>Connection parameters:</b>\n
	Some connection parameters from the vapi_pots.conf can be set while the application is running.
	For example CODEC can be set to G711 (4) for a first call.
	The for a second call it can be changed G729 (8) for a second call.

	<b>Starting the pots to pots application under Linux:</b>\n
	The steps to run the application are:
	- load the csmencaps driver:
			<pre>modprobe csmencaps</pre>
	- if the application runs on a M823xx device run the board initialization application.
			<pre>c300_init</pre>
	- load the legerity driver:
			<pre>modprobe legerity</pre>
	- start the pots application, The -dn and -Dn (1 to 4) options can be added to display more or less debug information.
			<pre>vapi_pots</pre>

	<b>Display of the process:</b>\n
	<pre>
	
	comcerto:~# modprobe csmencaps
	NET: Registered protocol family 27

	comcerto:~# lsmod
	Module                  Size  Used by
	csmencaps              19240  0

	comcerto:~# c300_init
	Network Timming Generator successfully configured
	TSI successfully configured

	comcerto:~#modprobe legerity
	legerity spi1.6: detected SLIC type 5 on chip select 6
	legerity spi1.7: detected SLIC type 5 on chip select 7
	legerity: loaded version 2.01.0

	comcerto:~# lsmod
	Module                  Size  Used by
	legerity               97288  0
	csmencaps              19240  0

	comcerto:~# vapi_pots

	*************************************************************
	VAPI Library Release 2.xx.x, API Version x.x
	*************************************************************
	VAPI Initilized sucessfully
	VAPI Library Release 2.10.0-cvs, API Version 8.0
	Entered VAPI_SetDebugLevel
	vapi_pots_main.c: 845: VAPI_OpenDevice: ok
	vapi_pots_main.c: 849: VAPI_InitDevice: ok
	vapi_pots_main.c: 852: TDM parameters initialisation: ok
	vapi_pots_main.c: 855: Network initialisation: ok
	Signaling SLIC event handler thread started
	Endpoint state machine thread started
	</pre>


	<b>Only the 2 pots line labelled JSCLIO-1 are currently working \n
	The phone number of the line 1 is 1, the phone number of line 2 is 2
	When a phone goes offhook a dialtone is played to the phone_id
	when a digit is dialed, the dialtone is stoped a Rinback tone is played to the phone.
	The other phone rings and when it goes off hook a VoIP connection is done between the 2 phones </b>\n
*/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifndef _VXWORKS_
#include <getopt.h>
#include <vapi/msp.h>
#include <vapi/vapi.h>
#include <vapi/gtl.h>
#include <sys/time.h>
#else
#include <taskLib.h>
#include <sys/ioctl.h>
#include <msp.h>
#include <vapi.h>
#include <gtl.h>
#endif

#include "vapi_pots_type.h"
#include "vapi_pots_var.h"
#include "vapi_pots_func.h"
#include "readcfg.h"

/* call control interface */
#include <legerity_lib.h>

#define check_status(title, error_label)					\
	PDEBUG(DEBUG_ERROR, title ": %s", result == SUCCESS?"ok":"failed");	\
	if (result != SUCCESS)							\
		goto error_label

u_int16_t endpoint_id;
u_int16_t originate_endpoint_id;
u_int16_t terminate_endpoint_id;
u_int32_t le_device_ip_address;
u_int32_t be_device_ip_address;
u_int16_t be_device_mac_addr[3];
u_int16_t le_device_mac_addr[3];
u_int16_t be_host_mac_addr[3];
u_int16_t le_host_mac_addr[3];
u_int16_t def_mac_addr[3];
SCSMEUsrData gtl_device_configuration;
struct _ENDPOINT_DESC endpoints [MAX_ENDPOINTS];
int legerity_device_file;

/*default values*/
u_int16_t device_id = 0;
u_int16_t VOICE_MAIL_MODE = DISABLE;
U8* media_buffer = NULL;
int media_buffer_length = 1024*64;
int vapi_debug_level = ALL_INFO;
int app_debug_level = ALL_INFO;
int playback_mode = ePlaybackStandard;
int  stop_type = eAS_STOP;
int speech_format = eHedFormat;

pthread_t signaling_event;
pthread_t state_machine;

#define PROGRAM_VERSION  "vapi_pots 0.1"

#if !defined(_VXWORKS_)
static void show_help(void)
{
	printf(
		"vapi_pots - program to make a pots to pots call on Comcerto EVM embeded boardsn"
		"Usage: %s [OPTION]...\n"
		"  -d, --vapi_debug_level=VAPI_DEBUG_LEVEL\n"
		"			Vapi Debug Level\n\n"
		"  -D, --app_debug_level=APP_DEBUG_LEVEL\n"
		"			App Debug Level\n\n"
		"  -p, --playback_mode=PLAYBACK_MODE\n"
		"			playback mode (default %d)\n\n"
		"  -s, --stop_type=STOP_TYPE\n"
		"			stop announcement type (default %d)\n\n"
		"  -h, --speech_format=SPEECH_FORMAT\n"
		"			speech data format (default %d)\n\n"
		"      --help		Show this help and exit\n"
		"      --version	Show version and exit\n",
		"vapi_pots",
		playback_mode,
		stop_type,
		speech_format
	);
}

static void show_version(void)
{
	printf("vapi pots application, %s\n", PROGRAM_VERSION);
}

static struct option const long_options[] =
{
	{ "vapi_debug_level",	required_argument, NULL, 'd' },
	{ "app_debug_level",	required_argument, NULL, 'D' },
	{ "playback_mode",	required_argument, NULL, 'p' },
	{ "stop_type",		required_argument, NULL, 's' },
	{ "speech_format",	required_argument, NULL, 'h' },

	{ "help",		no_argument, NULL, CHAR_MIN - 1 },
	{ "version",		no_argument, NULL, CHAR_MIN - 2 },

	{ NULL, 0, NULL, 0},
};

#else
const char *argp_program_version = PROGRAM_VERSION;
const char *argp_program_bug_address = "alexander.perekhodko@mindspeed.com";
const char doc[] = "vapi_pots - program to make a pots to pots call on Comcerto EVM embeded boards";

const struct argp_option options[] = {
	{"vapi_debug_level", 'd', "VAPI_DEBUG_LEVEL", 0, "Vapi Debug Level"},
	{"app_debug_level", 'D', "APP_DEBUG_LEVEL", 0, "App Debug Level"},
	{"playback_mode", 'p', "PLAYBACK_MODE", 0, "playback mode"},
	{"stop_type", 's', "STOP_TYPE", 0, "stop announcement type"},
	{"speech_format", 'h', "SPEECH_FORMAT", 0, "speech data format"},
	{0}
};
#endif

/*=================================================================================*/
/*! \brief
*	This function handles the response to VAPI commands sent in ASYNC mode. \n
*	It sets event for the endpoint accordingly the endpoint state. The state is retrieved through the request ID.\n
*/
void comcerto_response_handler(U32 connection_id,
			U8 command_level,
			S32 result,
			void *response_data,
			U32 data_length,
			U32 request_id)
{

	if ( result != SUCCESS)
	{
		PDEBUG(DEBUG_INIT,"Error on response received on endpoint id %d, request id = 0x%04x", connection_id, request_id);
		return;
	}
	else
	{
		PDEBUG(DEBUG_INIT,"response received on endpoint id %d, request id = 0x%04x",connection_id, request_id);
	}

	switch (request_id)
	{
		case ENDPOINT_STATE_WAIT_CREATE_CONNECTION_ORIGINATE_ACK:
		case ENDPOINT_STATE_WAIT_CREATE_CONNECTION_TERMINATE_ACK:
			PDEBUG(DEBUG_INIT,"VoIP channel %d created,",connection_id );
			/* The VoIP channel has been succesfully created, post the event */
			set_endpoint_event (ENDPOINT_EVENT_VOIP_CHANNEL_CREATED, connection_id);
			break;
			
		case ENDPOINT_STATE_WAIT_DESTROY_CONNECTION_ACK:
			PDEBUG(DEBUG_INIT,"VoIP channel %d destroyed,",connection_id );
			/* The VoIP channel has been succesfully created, post the event */
			set_endpoint_event (ENDPOINT_EVENT_VOIP_CHANNEL_DESTROYED, connection_id);
			break;

		case ENDPOINT_STATE_WAIT_IP_PARAMETERS_ORIGINATE_ACK:
			PDEBUG(DEBUG_INIT,"VoIP channel %d IP parameters set,",connection_id );
			/* The VoIP channel IP parameters have been succesfully set, post the event */
			set_endpoint_event (ENDPOINT_EVENT_IP_PARAMETERS_SET, connection_id);
			break;

		case ENDPOINT_STATE_WAIT_IP_PARAMETERS_TERMINATE_ACK:
			PDEBUG(DEBUG_INIT,"VoIP channel %d IP parameters set,",connection_id );
			/* The VoIP channel IP parameters have been succesfully set, post the event */
			set_endpoint_event (ENDPOINT_EVENT_IP_PARAMETERS_SET, connection_id);
			break;

		case ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_ORIGINATE_ACK:
			PDEBUG(DEBUG_INIT,"VoIP channel %d enabled,",connection_id );
			/* The VoIP channel has been succesfully enabled, post the event */
			set_endpoint_event (ENDPOINT_EVENT_CONNECTION_ENABLED, connection_id);
			break;

		case ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_TERMINATE_ACK:
			PDEBUG(DEBUG_INIT,"VoIP channel %d enabled,",connection_id );
			/* The VoIP channel has been succesfully enabled, post the event */
			set_endpoint_event (ENDPOINT_EVENT_CONNECTION_ENABLED, connection_id);
			break;

		case ENDPOINT_STATE_WAIT_T38_SWITCH_ACK:
			PDEBUG(DEBUG_INIT,"VoIP channel %d switched to T38,",connection_id );
			/* The VoIP channel has been succesfully switched to T38, post the event */
			set_endpoint_event (ENDPOINT_EVENT_T38_SWITCH_ACK, connection_id);
			break;

		case ENDPOINT_STATE_WAIT_T38_IP_PARAMETERS_ACK:
			PDEBUG(DEBUG_INIT,"FoIP channel %d IP parameters set,",connection_id );
			/* The IP params of the FoIP channel have been succesfully set, post the event */
			set_endpoint_event (ENDPOINT_EVENT_T38_IP_PARAMETERS_SET, connection_id);
			break;

		case ENDPOINT_STATE_WAIT_ENABLE_T38_CONNECTION_ACK:
			PDEBUG(DEBUG_INIT,"FoIP channel %d enabled,",connection_id );
			/* The FoIP channel has been succesfully enabled, post the event */
			set_endpoint_event (ENDPOINT_EVENT_CONNECTION_ENABLED, connection_id);
			break;

		case ENDPOINT_STATE_WAIT_START_RECORDING_ACK:
			PDEBUG(DEBUG_INIT,"recording completed on %d",connection_id );
			/* The VAPI_StartRecording has been succesfully passed, post the event */
			set_endpoint_event (ENDPOINT_EVENT_START_RECORDING_COMPLETED, connection_id);
			break;

		case ENDPOINT_STATE_WAIT_START_PLAYING_ACK:
			PDEBUG(DEBUG_INIT,"playing completed on %d,",connection_id );
			/* The VAPI_StartPlaying has been succesfully passed, post the event */
			set_endpoint_event (ENDPOINT_EVENT_START_PLAYING_COMPLETED, connection_id);
			break;			

	}
}

/*=================================================================================*/
/*! \brief
*	
*	This function handles the indication comminf from VAPI. \n
*	It is registered to VAPI using the VAPI_RegisterEventCallback() API.\n
*	It sets event V21 detected to operates a T38 switch in case of V21 flag detection.\n
*/
void comcerto_indication_handler(EEventCode eEventCode, void *pvData)
{
	SToneDetectEventParams *tone_detected;
	SRemoteDetectEventParams *remote_tone_detected;
	SSsrcChangeEventParams *ssrc_changed;
	SUndefinedEventParams *this_event;
	
	/* FIXME: The parser below need to be improved */
	switch(eEventCode)
	{
		case eVAPI_TONE_DETECT_EVENT:
			tone_detected = (SToneDetectEventParams *)pvData;

			PDEBUG(DEBUG_INIT,"Tone %d detected on Connection ID %d",
					tone_detected->usDetectedTone, tone_detected->ConId);

			switch (tone_detected->usDetectedTone)
			{
			case 0x22:
				printf("======== FAX V.21 detected =========\n");
				set_endpoint_event(ENDPOINT_EVENT_V21FLAG_DETECTED, tone_detected->ConId);
				break;

			case 1:
				/* if the DTMF tone 1 is detected on endpoint 1 proceed with the call*/
				if(( endpoints[endpoints[tone_detected->ConId].peer_index].state == ENDPOINT_STATE_IDLE) 
					&& (tone_detected->ConId == 1))
				{
					set_endpoint_event(ENDPOINT_EVENT_DIALED_CALL_REMOTE, tone_detected->ConId);
				}
				else
				{
					set_endpoint_event(ENDPOINT_EVENT_BUSY, tone_detected->ConId);
				}

				break;

			case 2:
				/* if the DTMF tone 2 is detected on endpoint 0 proceed with the call*/
				if(( endpoints[endpoints[tone_detected->ConId].peer_index].state == ENDPOINT_STATE_IDLE) 
					&& (tone_detected->ConId == 0))
				{
					set_endpoint_event(ENDPOINT_EVENT_DIALED_CALL_REMOTE, tone_detected->ConId);
				}
				else
				{
					set_endpoint_event(ENDPOINT_EVENT_BUSY, tone_detected->ConId);
				}
				break;

			case 0x0A:
				/* start/stop recordiing*/
					set_endpoint_event(ENDPOINT_EVENT_DIALED_RECORDING , tone_detected->ConId);
				break;

			case 0x0B:
				/* start/stop playing*/
					set_endpoint_event(ENDPOINT_EVENT_DIALED_PLAYING , tone_detected->ConId);
				break;				

			case 0xFF: /* end of tone no action*/
				break;

			default:
				/* else send busy tone */
					set_endpoint_event(ENDPOINT_EVENT_BUSY, tone_detected->ConId);
                        	break;
			}

			break;

		case eVAPI_SSRC_CHANGE_EVENT:
			ssrc_changed = (SSsrcChangeEventParams *)pvData;
			PDEBUG(DEBUG_INIT,"SSRC changed 0x%08x on Connection ID %d",
					ssrc_changed->uNewSSRC, ssrc_changed->ConId);

			break;

		case eVAPI_SPI_EVENT:
		case eVAPI_TONE_GEN_CMPLT_EVENT:
		case eVAPI_PT_CHANGE_EVENT:
		case eVAPI_SSRC_VIOLATION_EVENT:
		case eVAPI_NTE_TRANSMIT_COMPLETE_EVENT:
		case eVAPI_NTE_RECVD_EVENT:
		case eVAPI_CALLER_ID_CMPLT_EVENT:
		case eVAPI_G711_CONCURRENT_DECODER_EVENT:
		case eVAPI_PASSTHRU_CONCURRENT_DECODER_EVENT:
		case eVAPI_CALLER_ID_DETECTED_EVENT:
		case eVAPI_FAX_SWITCH_CMPLT_EVENT:
		case eVAPI_ALERT_IND:

			break;
			
		case eVAPI_REMOTE_DETECT_EVENT:
			remote_tone_detected = (SRemoteDetectEventParams *)pvData;
			
			PDEBUG(DEBUG_INIT,"Remote tone %d detected on Connection ID %d", (remote_tone_detected->ucDetectedEvent - 1), remote_tone_detected->ConId);

			switch (remote_tone_detected->ucDetectedEvent - 1)
			{
			case 0x0A:
				/* start/stop recordiing*/
					set_endpoint_event(ENDPOINT_EVENT_DIALED_RECORDING , remote_tone_detected->ConId);
				break;

			case 0x0B:
				/* start/stop playing*/
					set_endpoint_event(ENDPOINT_EVENT_DIALED_PLAYING , remote_tone_detected->ConId);
				break;				

			case 0xFF: /* end of tone no action*/
				break;

			default:
				/* else send busy tone */
					set_endpoint_event(ENDPOINT_EVENT_BUSY, remote_tone_detected->ConId);
                        	break;
			}

			break;
			
		case eVAPI_UNDEFINED_EVENT:
			PDEBUG(DEBUG_INIT,"event %d received",eEventCode);
			this_event = (SUndefinedEventParams *) pvData;

			if(this_event->ucCmdType == CMD_TYPE_INDICATION)
				PDEBUG(DEBUG_INIT,"indication 0x%04x received",this_event->usFnCode);
		
			break;

		default:
			break;
	}

	return;
}

/*=================================================================================*/
/*! \brief
*	
*	This function handle the states of the endpoints.
*	It is started from a thread.
*	All endpoints are initialized in to the IDLE state, then the states change accordingly the events \n
*	set by the comcerto_response_handler, comcerto_indication_handler, or the signaling_event_handler threads\n.
*/
void * state_machine_handler(void *none)
{
	int i;
	struct legerity_io_ring_desc ring;
	struct legerity_io_disconnect_desc disconnect;	
	int time_out_remote = 2;	/* 3 sec */
	int time_out_prompt_message = 10; /* sec */
	int loopback = 0; /* by default no loopback*/

	/* We should have here a signal handler to interrupt the while loop
	For now loop forever, CTRL-C to be used to stop. */
	PDEBUG (DEBUG_INIT, "State machine started");
	while(1)
	{
		/* Parse all the endpoints in the global endpoints array*/
		for (i = 0; i < MAX_ENDPOINTS; i ++)

		{
			/* If the state is IDLE we should receive only OFFHOOK or RING events*/
			switch(endpoints[i].state)
			{
				case ENDPOINT_STATE_IDLE:
					switch(endpoints[i].event)
					{
						/* If the OFFHOOK event was posted*/
						case ENDPOINT_EVENT_OFF_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d off hook", endpoints[i].index);
							/* Mark this endpoint in OFFHOOK state*/

							set_endpoint_state(ENDPOINT_STATE_OFF_HOOK, endpoints[i].index);
							/* Post the appropriate event to create a channel */
							set_endpoint_event(ENDPOINT_EVENT_CREATE_VOIP_CHANNEL, endpoints[i].index);

							break;

						/* If the RINGING event was posted*/
						case ENDPOINT_EVENT_RINGING:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d ringing", endpoints[i].index);							
							set_endpoint_state(ENDPOINT_STATE_RINGING, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_START_TIMER, endpoints[i].index);
							break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);

					}
				break;
				
				/* This endpoint is now OFFHOOK we need to create a VoIP channel to handle the call*/
				case ENDPOINT_STATE_OFF_HOOK:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);

							break;

						case ENDPOINT_EVENT_CREATE_VOIP_CHANNEL:
							/* This endpoint is now offhook we want to create a VoIP channel
							for this endpoint */
							/* so we switch to a state to wait for the Comcerto response to create connection */
							set_endpoint_state(ENDPOINT_STATE_WAIT_CREATE_CONNECTION_ORIGINATE_ACK, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);

							create_endpoint(endpoints[i].index, ENDPOINT_STATE_WAIT_CREATE_CONNECTION_ORIGINATE_ACK);
							break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);

					}
				break;

				/* The phone is RINGING. If the phone goes OFFHOOK, an event OFF_HOOK is posted by the signalling thread
				Run timer. If time is up set ENDPOINT_EVENT_REMOTE_TIME_UP event*/
				case ENDPOINT_STATE_RINGING:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							UT_TimerStop(endpoints[i].this_timer);
							onhook_handler(endpoints[i].index);
							
						break;

						case ENDPOINT_EVENT_OFF_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d off hook", endpoints[i].index);
							/* This endpoint is now offhook we want to create a VoIP channel
							for this endpoint */
							/* so we switch to a state to wait for the Comcerto response to create connection */
							UT_TimerStop(endpoints[i].this_timer);
							stop_tone_generation(endpoints[i].peer_index, ENDPOINT_STATE_NONE);
							set_endpoint_state(ENDPOINT_STATE_WAIT_CREATE_CONNECTION_TERMINATE_ACK, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);

							create_endpoint(endpoints[i].index, ENDPOINT_STATE_WAIT_CREATE_CONNECTION_TERMINATE_ACK);

						break;

						case ENDPOINT_EVENT_START_TIMER:
							/* start timer */
							UT_TimerStart(endpoints[i].this_timer, time_out_remote, timed_connection, (void *)(&endpoints[i]));
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							break;

						case ENDPOINT_EVENT_TIME_UP:
							/* Time up occured */
							stop_tone_generation(endpoints[i].peer_index, ENDPOINT_STATE_NONE);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_REMOTE_TIME_UP, endpoints[i].peer_index);
							break;
						
						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;

				/* The phone went onhook and the destroy connection command has been sent.
				once the response will be received the state will be ONHOOK
				set enpoint side to TDM side, disable peer_destroy and voice mail mode*/
				case ENDPOINT_STATE_WAIT_DESTROY_CONNECTION_ACK:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_VOIP_CHANNEL_DESTROYED:
							set_endpoint_state(ENDPOINT_STATE_ON_HOOK, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							set_endpoint_side (TDM, endpoints[i].index);
							set_endpoint_peer_destroy (DISABLE, endpoints[i].index);
							set_endpoint_voice_mail_mode(DISABLE, endpoints[i].peer_index);

						break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;

				/* This endpoint is OFFHOOK, the request to create a VoIP channel has been posted. We are waiting for the response
				in case of originate side the next action is to call the remote side*/
				case ENDPOINT_STATE_WAIT_CREATE_CONNECTION_ORIGINATE_ACK:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);
							
						break;

						case ENDPOINT_EVENT_VOIP_CHANNEL_CREATED:
							set_endpoint_state(ENDPOINT_STATE_WAIT_FOR_DIALING, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							dialtone_generation(endpoints[i].index, ENDPOINT_STATE_NONE);

						break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;

				/* This endpoint is OFFHOOK, generating dialtone on right digit dialed
				 the next action is
				 a). to call the remote side 
				 b). recording 
				 c). playing*/
				case ENDPOINT_STATE_WAIT_FOR_DIALING:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);

						break;

						case ENDPOINT_EVENT_DIALED_CALL_REMOTE:
							stop_tone_generation(endpoints[i].index, ENDPOINT_STATE_NONE);
							set_endpoint_state(ENDPOINT_STATE_CALL_REMOTE, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_MAKE_OUTGOING_CALL, endpoints[i].index);

						break;

						case ENDPOINT_EVENT_DIALED_RECORDING:
							stop_tone_generation(endpoints[i].index, ENDPOINT_STATE_NONE);
							set_endpoint_state(ENDPOINT_STATE_ANNOUNCEMENT, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_START_RECORDING, endpoints[i].index);

						break;

						case ENDPOINT_EVENT_DIALED_PLAYING:
							stop_tone_generation(endpoints[i].index, ENDPOINT_STATE_NONE);
							set_endpoint_state(ENDPOINT_STATE_ANNOUNCEMENT, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_START_PLAYING, endpoints[i].index);

						break;						

						case ENDPOINT_EVENT_BUSY:
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							busytone_generation(endpoints[i].index, ENDPOINT_STATE_NONE);

						break;
 
						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;

				/* This endpoint is in ANNOUNCEMENT state. Here there is being handling announcement features*/
				case ENDPOINT_STATE_ANNOUNCEMENT:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);
						break;

						case ENDPOINT_EVENT_START_RECORDING:
							/*On this endpoint is being processed recording*/
							set_endpoint_state(ENDPOINT_STATE_WAIT_START_RECORDING_ACK, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);

							media_buffer = (U8*) __UT_malloc( media_buffer_length, "start_recording", 0);
							memset(media_buffer, 0, media_buffer_length);	
							start_recording(endpoints[i].index, ENDPOINT_STATE_WAIT_START_RECORDING_ACK);							

						break;

						case ENDPOINT_EVENT_START_PLAYING:
							/*On this endpoint is being processed playing*/
							set_endpoint_state(ENDPOINT_STATE_WAIT_START_PLAYING_ACK, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);

							read_file ();
							start_playing(endpoints[i].index, ENDPOINT_STATE_WAIT_START_PLAYING_ACK);							

						break;
						
						case ENDPOINT_EVENT_PLAY_GREETING_MESSAGE:
							PDEBUG(DEBUG_INIT, "Endpoint %d: play greeting message", endpoints[i].index);							
							/* paly greeting message (last recorded media_file_%d.rec) by VAPI_StartPlaying in sync mode */
							set_endpoint_state(ENDPOINT_STATE_WAIT_FOR_DIALED_COMMAND, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_START_TIMER, endpoints[i].index);							

						break;	

						case ENDPOINT_EVENT_PLAY_PROMPT_MESSAGE:
							PDEBUG(DEBUG_INIT, "Endpoint %d: play prompt message", endpoints[i].index);							
							/* paly prompt message (prompt_message.rec) by VAPI_StartPlaying in sync mode */
							set_endpoint_state(ENDPOINT_STATE_WAIT_FOR_DIALED_COMMAND, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_START_TIMER, endpoints[i].index);							

						break;							
 
						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;

				/* This endpoint is being recorded. Waiting for completeng*/
				case ENDPOINT_STATE_WAIT_START_RECORDING_ACK:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);
						break;

						case ENDPOINT_EVENT_START_RECORDING_COMPLETED:
							/* received response from VAPI_StartRecording. Write recorded media buffer to file, free buffer*/
							PDEBUG(DEBUG_INIT, "save media buffer to file recorded on endpoint: %d", endpoints[i].index);

							write_file();
							__UT_free(media_buffer, "case ENDPOINT_EVENT_START_RECORDING_COMPLETED", 0);
							set_endpoint_state(ENDPOINT_STATE_ANNOUNCEMENT, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_PLAY_PROMPT_MESSAGE, endpoints[i].index);

						break;

						case ENDPOINT_EVENT_DIALED_RECORDING:
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							stop_recording(endpoints[i].index);
							
						break;							

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;				

				/* This endpoint is being played. Waiting for completeng*/
				case ENDPOINT_STATE_WAIT_START_PLAYING_ACK:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);

						break;

						case ENDPOINT_EVENT_START_PLAYING_COMPLETED:
							/* received response from VAPI_StartRecording. free media buffer*/
							__UT_free(media_buffer, "case ENDPOINT_EVENT_START_PLAYING_COMPLETED", 0);
							set_endpoint_state(ENDPOINT_STATE_ANNOUNCEMENT, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_PLAY_PROMPT_MESSAGE, endpoints[i].index);

						break;

						case ENDPOINT_EVENT_DIALED_PLAYING:
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);							
							stop_playing (endpoints[i].index);

						break;						

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;	

				/* Run timer, by having excceded witch play prompt message. 
				Wait on this endpoint right dialed digit for the next action 
				a). recording 
				b). playing*/
				case ENDPOINT_STATE_WAIT_FOR_DIALED_COMMAND:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							UT_TimerStop(endpoints[i].this_timer);
							onhook_handler(endpoints[i].index);

						break;

						case ENDPOINT_EVENT_DIALED_RECORDING:
							UT_TimerStop(endpoints[i].this_timer);
							set_endpoint_state(ENDPOINT_STATE_ANNOUNCEMENT, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_START_RECORDING, endpoints[i].index);

						break;

						case ENDPOINT_EVENT_DIALED_PLAYING:
							UT_TimerStop(endpoints[i].this_timer);
							set_endpoint_state(ENDPOINT_STATE_ANNOUNCEMENT, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_START_PLAYING, endpoints[i].index);

						break;						

						case ENDPOINT_EVENT_START_TIMER:
							/* start timer */
							UT_TimerStart(endpoints[i].this_timer, time_out_prompt_message, timed_connection, (void *)(&endpoints[i]));
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							break;

						case ENDPOINT_EVENT_TIME_UP:
							/* time up occured */
							set_endpoint_state(ENDPOINT_STATE_ANNOUNCEMENT, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_PLAY_PROMPT_MESSAGE, endpoints[i].index);
							break;
 
						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;


				/* This endpoint has a VoIP channel associated. An outgoing call to the destination endpoint is requested through the  Legerity*/
				case ENDPOINT_STATE_CALL_REMOTE:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);

						break;

						case ENDPOINT_EVENT_MAKE_OUTGOING_CALL:
							/* we force an outgoing call to the peer POTS phone */

							printf(" Calling Peer Endpoint %d\n", endpoints[i].peer_index);
							ring.line = endpoints[i].peer_index;
							legerity_ring(legerity_device_file, &ring);

							ringbacktone_generation(endpoints[i].index, ENDPOINT_STATE_NONE);

							/* Set the event of the destinqtion endpoint to RINGING.
							If the destination endpoint is in IDLE state, it will switch to RINGING State
							FIXME: If not in a idle state we should here generate a busy tone to the originqe endpoint */
							set_endpoint_event(ENDPOINT_EVENT_RINGING, endpoints[i].peer_index);

							set_endpoint_state(ENDPOINT_STATE_REMOTE_CALLED, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);

						break;


						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}

				break;

				/* This endpoint is OFFHOOK, the VoIP channel is created and the remote endpoint is ringing
				We keep this state until the destination endpoint goes offhook or this endpoint goes onhook 
				If time up posted on remote side move to ato answer state*/
				case ENDPOINT_STATE_REMOTE_CALLED:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);
							/* this endpoint went onhook while the remote is ringing*/
							/* So we stop the call on the remote side and set back the endpoint in idlle state*/
							disconnect.line = endpoints[i].peer_index;
							legerity_disconnect(legerity_device_file, &disconnect);

							set_endpoint_state(ENDPOINT_STATE_IDLE, endpoints[i].peer_index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].peer_index);

						break;

						case ENDPOINT_EVENT_REMOTE_TIME_UP:
							/* proceed auto answer mode*/
							set_endpoint_state(ENDPOINT_STATE_AUTO_ANSWER, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_START_AUTO_ANSWER, endpoints[i].index);
							break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;

				/* Stop calling on remote side. Set for both sides appropriate voice mail mode - server/caller, 
				create terminate endpoint */
				case ENDPOINT_STATE_AUTO_ANSWER:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);

							set_endpoint_state(ENDPOINT_STATE_IDLE, endpoints[i].peer_index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].peer_index);

						break;

						case ENDPOINT_EVENT_START_AUTO_ANSWER:
							/* Stop the call on the remote side*/
							disconnect.line = endpoints[i].peer_index;
							legerity_disconnect(legerity_device_file, &disconnect);						

							set_endpoint_voice_mail_mode(SERVER, endpoints[i].peer_index);
							set_endpoint_voice_mail_mode(CALLER, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_CREATE_TERMINATE_VOIP_CHANNEL, endpoints[i].index);

						break;

						case ENDPOINT_EVENT_CREATE_TERMINATE_VOIP_CHANNEL:
							/* create VoIP channel on terminate endpoint */
							set_endpoint_state(ENDPOINT_STATE_WAIT_CREATE_CONNECTION_TERMINATE_ACK, endpoints[i].peer_index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].peer_index);

							create_endpoint(endpoints[i].peer_index, ENDPOINT_STATE_WAIT_CREATE_CONNECTION_TERMINATE_ACK);

							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;

				/* The request to create a VoIP channel has been posted for this enpoint. We are waiting for the response
				in case of terminate side the next action is to set the IP parameters of both sides */
				case ENDPOINT_STATE_WAIT_CREATE_CONNECTION_TERMINATE_ACK:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);

						break;

						case ENDPOINT_EVENT_VOIP_CHANNEL_CREATED:
							/* if request has been posted by another enpoint, set peer_index to indicate that 
							this enpoint can be destroyed only by caused enpoint.
							Even if this enpoint goes off-hook enpoint won't be destroyed*/
							if (endpoints[endpoints[i].peer_index].state == ENDPOINT_STATE_AUTO_ANSWER)
								set_endpoint_peer_destroy(ENABLE, endpoints[i].index);
							
							/* The VoIP channel is created we set now the IP parameters on both sides*/
							set_endpoint_state(ENDPOINT_STATE_WAIT_IP_PARAMETERS_TERMINATE_ACK, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							set_ip_udp_parameters(endpoints[i].index, ENDPOINT_STATE_WAIT_IP_PARAMETERS_TERMINATE_ACK);

							set_endpoint_state(ENDPOINT_STATE_WAIT_IP_PARAMETERS_ORIGINATE_ACK, endpoints[i].peer_index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].peer_index);
							set_ip_udp_parameters(endpoints[i].peer_index, ENDPOINT_STATE_WAIT_IP_PARAMETERS_ORIGINATE_ACK);

						break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;

				/* This endpoint has a VoIP channel associated, the Command to set the IP params been posted. 
				We are waiting for the response.
				In case of originate side the next action is to start the packect processing */
				case ENDPOINT_STATE_WAIT_IP_PARAMETERS_ORIGINATE_ACK:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_SLIC_FUNC, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);

						break;

						case ENDPOINT_EVENT_IP_PARAMETERS_SET:
							set_endpoint_state(ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_ORIGINATE_ACK, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							start_voip_connection(endpoints[i].index, ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_ORIGINATE_ACK);

						break;


						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;

				/* This endpoint has a VoIP channel associated, the Command to set the IP params been posted. 

				We are waiting for the response.
				In case of terminate side the next action is to stqrt the packect processing */
				case ENDPOINT_STATE_WAIT_IP_PARAMETERS_TERMINATE_ACK:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);

						break;

						case ENDPOINT_EVENT_IP_PARAMETERS_SET:
							set_endpoint_state(ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_TERMINATE_ACK, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							start_voip_connection(endpoints[i].index, ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_TERMINATE_ACK);

						break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;

				/* This endpoint has a VoIP channel started, the VoIP conenction is now running.
				Move to state ready for talking between both sides or to voice mail mode*/
				case ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_ORIGINATE_ACK:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);

						break;

						case ENDPOINT_EVENT_CONNECTION_ENABLED:
							if (VOICE_MAIL_MODE)
							{
								set_endpoint_state(ENDPOINT_STATE_VOICE_MAIL_CALLER, endpoints[i].index);
								set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
								PDEBUG(DEBUG_INIT, "=====Endpoint %d: event %d in state voice mail caller (%d)", endpoints[i].index, endpoints[i].event, endpoints[i].state);
							}
							else
							{
								set_endpoint_state(ENDPOINT_STATE_READY, endpoints[i].index);
								set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
								PDEBUG(DEBUG_INIT, "=====Endpoint %d: event %d in state ready (%d)", endpoints[i].index, endpoints[i].event, endpoints[i].state);
							}
							/* set side of recording/palying source/destination type*/
							set_endpoint_side (IP, endpoints[i].index);

						break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:

							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;

				/* This endpoint has a VoIP channel started, the VoIP conenction is now running.*/
				case ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_TERMINATE_ACK:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);

						break;

						case ENDPOINT_EVENT_CONNECTION_ENABLED:
							if (VOICE_MAIL_MODE)
							{
								set_endpoint_state(ENDPOINT_STATE_VOICE_MAIL_SERVER, endpoints[i].index);
								set_endpoint_event(ENDPOINT_EVENT_START_VOICE_MAIL_SERVER, endpoints[i].index);
								PDEBUG(DEBUG_INIT, "=====Endpoint %d: event %d in state voice mail server (%d)", endpoints[i].index, endpoints[i].event, endpoints[i].state);
								
							}
							else
							{
								set_endpoint_state(ENDPOINT_STATE_READY, endpoints[i].index);
								set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
								PDEBUG(DEBUG_INIT, "=====Endpoint %d: event %d in state ready (%d)", endpoints[i].index, endpoints[i].event, endpoints[i].state);
							}

							/* set side of recording/palying source/destination type*/
							set_endpoint_side (IP, endpoints[i].index);

							/* get loopback function from the config file */
							loopback = get_loopback_mode();

							/* perform the loop back in ready state*/	
							if (loopback != 0)
							{
								VAPI_Loopback (endpoints[i].peer_index, endpoints[i].index, loopback, NULL);
							}						

						break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;

				/* This endpoint is voice mail caller. Keep this state until this goes on-hook */
				case ENDPOINT_STATE_VOICE_MAIL_CALLER:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);

						break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;

				/* This endpoint is voice mail server.*/
				case ENDPOINT_STATE_VOICE_MAIL_SERVER:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);

						break;

						case ENDPOINT_EVENT_START_VOICE_MAIL_SERVER:
							set_endpoint_state(ENDPOINT_STATE_ANNOUNCEMENT, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_PLAY_GREETING_MESSAGE, endpoints[i].index);							

						break;

						case ENDPOINT_EVENT_NONE:
							break;
						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;				

				/* This endpoint is connected.*/
				case ENDPOINT_STATE_READY:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);

						break;

						case ENDPOINT_EVENT_V21FLAG_DETECTED:
							PDEBUG(DEBUG_INIT, "Endpoint %d: event V21 flags detected (%d) in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
							set_endpoint_state(ENDPOINT_STATE_WAIT_T38_SWITCH_ACK, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							switch_voip_connection_to_t38(endpoints[i].index, ENDPOINT_STATE_WAIT_T38_SWITCH_ACK);
							switch_voip_connection_to_t38(endpoints[i].peer_index, ENDPOINT_STATE_WAIT_T38_SWITCH_ACK);
							break;

						case ENDPOINT_EVENT_NONE:
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							
							break;
						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;

				/* This endpoint is waiting for T38 switch to be performed.*/
				case ENDPOINT_STATE_WAIT_T38_SWITCH_ACK:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);
						break;

						/* the T38 switch has corecly been performed
						now we need to send again IP/UDP  */
						case ENDPOINT_EVENT_T38_SWITCH_ACK:
							set_endpoint_state(ENDPOINT_STATE_WAIT_IP_PARAMETERS_TERMINATE_ACK, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							set_ip_udp_parameters(endpoints[i].index, ENDPOINT_STATE_WAIT_IP_PARAMETERS_TERMINATE_ACK);

							set_endpoint_state(ENDPOINT_STATE_WAIT_IP_PARAMETERS_ORIGINATE_ACK, endpoints[i].peer_index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].peer_index);
							set_ip_udp_parameters(endpoints[i].peer_index, ENDPOINT_STATE_WAIT_IP_PARAMETERS_ORIGINATE_ACK);
							break;

						case ENDPOINT_EVENT_NONE:
							break;
						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;

				/* This endpoint has a FoIP channel associated, the Command to set the IP params been posted.

				We are waiting for the response.
				In case of terminate side the next action is to start the UDPTL packect processing */
				case ENDPOINT_STATE_WAIT_T38_IP_PARAMETERS_ACK:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);

						break;

						case ENDPOINT_EVENT_T38_IP_PARAMETERS_SET:
							set_endpoint_state(ENDPOINT_STATE_WAIT_ENABLE_T38_CONNECTION_ACK, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							start_voip_connection(endpoints[i].index, ENDPOINT_STATE_WAIT_ENABLE_T38_CONNECTION_ACK);

						break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;
				/* This endpoint has a FoIP channel with IP params configured
				we now need to start UDPTL traffic*/
				case ENDPOINT_STATE_WAIT_ENABLE_T38_CONNECTION_ACK:
					switch(endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							onhook_handler(endpoints[i].index);

						break;

						case ENDPOINT_EVENT_CONNECTION_ENABLED:
							set_endpoint_state(ENDPOINT_STATE_READY, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							PDEBUG(DEBUG_INIT, "Endpoint %d: event %d in state ready T38 (%d)", endpoints[i].index, endpoints[i].event, endpoints[i].state);

						break;

						case ENDPOINT_EVENT_NONE:
							break;
						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
					}
				break;
			}
		}
		usleep(100);
	}
}



/*=================================================================================*/
/*! \brief
*	
	This function handles the events reported by the signaling thread (SLIC events) such as offhook, onhokk, ring.
	The timeslot reported with the event is used as the endpoint index in the global endpoints array
*/
void * signaling_event_handler(void * none)
{
	int res;
	struct legerity_io_event_desc legerity_event;
	
	/* we should have here a state machine waiting for events */
	PDEBUG (DEBUG_INIT, "Signaling event handler started");
	while(1)
	{
		res = legerity_read_event(legerity_device_file, &legerity_event);

		if (res != 0)
		{
			goto  sleep;
		}

		switch (legerity_event.type)
		{
			case 0 /*EVENT_ONHOOK*/:
				PDEBUG(DEBUG_SLIC_FUNC, "Legerity event = EVENT_ONHOOK ts = %d", legerity_event.timeslot);
				set_endpoint_event(ENDPOINT_EVENT_ON_HOOK, legerity_event.timeslot);
				break;		

			case 1 /*EVENT_OFFHOOK*/:
				PDEBUG(DEBUG_SLIC_FUNC, "Legerity event = EVENT_OFFHOOK ts = %d", legerity_event.timeslot);
				/* use the timeslot as the endpoint index */
				endpoints[legerity_event.timeslot].timeslot = legerity_event.timeslot;

				/* post the OFFHOOK event for this endpoint */
				set_endpoint_event(ENDPOINT_EVENT_OFF_HOOK, legerity_event.timeslot);
				break;

			default:
				PDEBUG(DEBUG_ERROR, "Unexpected slic event = %d", legerity_event.type);
				break;

		}

sleep:
		usleep(100);
	}
}

#if !defined(_VXWORKS_)
static int
get_int(const char *s, int *value)
{
	int i;
	char *endptr = NULL;

	if (!s || !*s)
		goto err;

	i = strtol(s, &endptr, 0);
	if (!i && s == endptr)
		goto err;

	if (value)
		*value = i;

	return 0;

err:
	return -1;
}

/*=================================================================================*/
/*!	\brief command line option parser
 *	\section Description
 *	This function is the command line option parser.\n
 */	
/*=================================================================================*/
static int get_opt(int argc, char *argv[])
{

	int c;
	int status = SUCCESS;
	
	while ((c = getopt_long(argc, argv, "d:D:p:s:h", long_options, NULL)) != - 1) {
		switch (c) {
		case 'd':
			get_int(optarg, (int *) &vapi_debug_level);
			break;

		case 'D':
			get_int(optarg, (int *) &app_debug_level);
			break;

		case 'p':
			get_int(optarg, (int *) &playback_mode);

			if (1 == playback_mode)
				playback_mode = ePlaybackExt;
			else
				playback_mode = ePlaybackStandard;

			break;

		case 's':
			get_int(optarg, (int *) &stop_type);

			if (1 == stop_type)
				stop_type= eAS_STOP;
			else
				stop_type = eDRAIN;

			break;

		case 'h':
			get_int(optarg, (int *) &speech_format);

			if (1 == speech_format)
				speech_format = eNoHedFormat;
			else
				speech_format = eHedFormat;

			break;

		case CHAR_MIN - 1:
			show_help();
			exit(0);
			break;

		case CHAR_MIN - 2:
			show_version();
			exit(0);
			break;
			
		default:
			status = -1;
		}

	}

	return status;
}

#else
/*=================================================================================*/
/*! \brief
*	
*	This function parses the command line options to set the VAPI and application debug levels
*/
static error_t parser (int key, char *arg, struct argp_state *state)
{

	switch (key)
	{
	case 'd':

		vapi_debug_level = strtoul(arg, NULL, 0);
		break;

	case 'D':

		app_debug_level = strtoul(arg, NULL, 0);
		break;

	case 'p':
		playback_mode = strtoul(arg, NULL, 0);

		if (1 == playback_mode)
			playback_mode = ePlaybackExt;
		else
			playback_mode = ePlaybackStandard;

		break;

	case 's':
		stop_type = strtoul(arg, NULL, 0);

		if (1 == stop_type)
			stop_type= eAS_STOP;
		else
			stop_type = eDRAIN;

		break;

	case 'h':
		speech_format = strtoul(arg, NULL, 0);

		if (1 == speech_format)
			speech_format = eNoHedFormat;
		else
			speech_format = eHedFormat;

		break;		

	default:
		return ARGP_ERR_UNKNOWN;
	}
	return 0;
}

static struct argp argp = { options, parser, 0, doc, };
#endif

/***************************************************************************
 * main
 ***************************************************************************/
/*! \brief
*	
	This is the main function of the application.\n
	It does:\n
	 - initialise an endpoints array
	 - then parse the command line arguments (VAPI debug level)
	 - then it:
		- Initialise the setup configuration using parameters from the configration file (Device type, Control interface, Slave/Master mode , etc)
		- Print the VAPI verson (VAPI_GetVersion)
		- Initialise VAPI (VAPI_Init)
		- Open the Device (VAPI_OpenDevice)
		- Initialise the TDM buses using default parameters (VAPI_SetTDMParams).
		- Using VAPI_InitDevice set the SPU feature and enable the Multicommand mode.
		- Set the Network Parameters (MAC, IP addresses - VAPI_SetEthMac, VAPI_SetDeviceIPAddr).
	- then it starts the signaling and state machine threads.
*/
int main(int argc, char *argv[])
{
	int result;
	int i;

	/* these are used in the LoopbackInit scenario */
	originate_endpoint_id = endpoint_id;
	terminate_endpoint_id = endpoint_id + 1;

	/* Initialise the endpoint info array */
	memset(&endpoints, 0, sizeof(endpoints));

	for (i = 0 ; i < MAX_ENDPOINTS; i++)
	{
		endpoints[i].index = i;
		/* the RTP port must be even */
		endpoints[i].rtp_port = DEFAULT_UDP_PORT + i*2;
		/* the RTCP port must be odd */
		endpoints[i].rtcp_port = DEFAULT_UDP_PORT + i;

		endpoints[i].this_timer = UT_TimerCreate();
		set_endpoint_side (TDM, endpoints[i].index);
	}

	endpoints[0].peer_index = 1;
	endpoints[1].peer_index = 0;

	/* The VAPI default parameters are not convenient for us
	 * we overwrite some of them. This must be done before VAPI_Init()
	 */
	pfnUsrQueryVapiDefaultsEnumerate = set_default_config;

#if !defined(_VXWORKS_)
	/* returns value < 0 on error, == 0 if all's Ok */
	result = get_opt(argc, argv);

	if (result < 0)
		return -1;
#else
	/* returns value < 0 on error, > 1 on help or version request, == 0 if all's Ok */
	result = argp_parse(&argp, args);

	if (result > 0)
		return 0;
	if (result < 0)
		return -1;
#endif	

	/* Setting device parameters - this also must be done before VAPI_Init() */
	set_device_configuration(device_id);

	/*Initialise the VAPI*/
	result = VAPI_Init(&gtl_device_configuration);

	if(result == SUCCESS)
	{
		PDEBUG (DEBUG_ERROR, "VAPI_Init SUCCESS\n");
	}
	else
	{
		PDEBUG (DEBUG_ERROR, "VAPI_Init Failed\n");
		goto err0;
	}


	printf("%s\n", VAPI_GetVersion());

	VAPI_SetDebugLevel(vapi_debug_level, app_debug_level);
	
	result = VAPI_OpenDevice(device_id,NULL);
	check_status("VAPI_OpenDevice", err1);

	result = VAPI_InitDevice(device_id, VAPI_DEV_OPMODE_DEFAULT, VAPI_DEV_PROF_SPU_FSK_IP_DETECTION, NULL, NULL);
	PDEBUG (DEBUG_INIT, "result = %d", result );	
	check_status("VAPI_InitDevice", err2);

	result = tdm_parameters_initialisation();
	check_status("TDM parameters initialisation", err2);

	result = network_parameters_initialisation();
	check_status("Network initialisation", err2);

	/* First try to open the access to the legerity driver (/dev/legerity)*/
	legerity_device_file = legerity_open("/dev/legerity");

	if (legerity_device_file < 0)
	{
		PDEBUG (DEBUG_ERROR,"Can't open Legerity device");
		goto err2;
	}

	/* Now create a thread to handle the events comming from the Signaling side (SLIC) */
	result = pthread_create(&signaling_event, NULL, signaling_event_handler, NULL);
	if (result != 0)
	{
		PDEBUG (DEBUG_ERROR,"Can't create Signaling event handler thread");
		goto err2;
	}

	/* Now create a thread to handle the states */
	result = pthread_create(&state_machine, NULL, state_machine_handler, NULL);

	if (result != 0)
	{
		PDEBUG (DEBUG_ERROR,"Can't create State machine thread");
		goto err2;
	}

	PDEBUG (DEBUG_INIT,"Register callback comcerto_indication_handler");
	VAPI_RegisterEventCallback(device_id, EVENT_LEVEL_GENERIC, comcerto_indication_handler);

	pthread_join(state_machine, NULL);
	pthread_join(signaling_event, NULL);


	return 0;

err2:
	legerity_close(legerity_device_file);
	VAPI_CloseDevice(device_id, ePURGE);

err1:
	VAPI_Close();

err0:
	return FAILURE;
}

/** @} */ /*pots to pots example*/
