/*! \readcfg.c  */
/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <malloc.h>
#include <ctype.h>


#include "readcfg.h"

//#define _DEBUG_READ_CFG_ 1
#undef _DEBUG_READ_CFG_

int cfg_parse_line(int *line_n, char *line, struct _CFG **conf, struct _CFG **clast, struct _KEY **klast)
{
	struct _CFG *cnode;
	struct _KEY *knode;
	char *temp, *temp2, *temp3;

	(*line_n)++;
	temp = line + 1;

	if (*temp == '#')
		return 0;

	temp = (char *) strchr (temp, '#');
	if (temp)
		*temp = 0;

	temp = line + 1;
	while ((*temp == ' ') || (*temp == '\t'))
		temp++;

	temp2 = temp + strlen (temp) - 1;
	while ((*temp2 == ' ') || (*temp2 == '\t') || (*temp2 == '\n'))
		*temp2-- = 0;

	if (!*temp)	/* ignore blanks */
		return 0;

	if (*temp == '[')
	{
		temp++;

		temp2 = (char *) strchr (temp, ']');
		if (!temp2)
		{
			fprintf (stderr, "Config line #%i missing ], %s\n", *line_n, line);
			return -1;
		}

		*temp2-- = 0;
		if (*(temp2 + 2))
		{
			fprintf (stderr, "Junk after ] in line #%i, %s\n", *line_n, line);
			return -1;
		}

		while ((*temp2 == ' ') || (*temp2 == '\t'))
			*temp2-- = 0;

		if (!*temp)
		{
			fprintf (stderr, "Missing APP on line #%i, %s\n", *line_n, line);
			return -1;
		}

		for (cnode = *conf; cnode; cnode = cnode->next)
			if (strcasecmp (cnode->app, temp) == 0)
			{
				fprintf (stderr, "Duplicate APP values in line #%i, %s\n", *line_n, line);
				return -1;
			}

		cnode = (struct _CFG *) malloc (sizeof (struct _CFG));
		memset (cnode, 0, sizeof (struct _CFG));

		cnode->app = (char *) malloc (strlen (temp) + 1);
		strcpy (cnode->app, temp);

#ifdef _DEBUG_READ_CFG_
		fprintf (stderr, "%s\n", temp);
#endif
		if (!*clast)
			*conf = *clast = cnode;
		else
		{
			(*clast)->next = cnode;
			*clast = cnode;
		}

		*klast = knode = NULL;
	}
	else if ((temp2 = (char *) strchr (temp, '=')) != NULL)
	{
		if (!(*clast))
		{
			fprintf (stderr, "No APP value specified by line #%i, %s\n", *line_n, line);
			return -1;
		}

		temp3 = temp2 - 1;
		while ((*temp3 == ' ') || (*temp3 == '\t'))
			*temp3-- = 0;

		*temp2++ = 0;
		while ((*temp2 == ' ') || (*temp2 == '\t'))
			temp2++;

		if (!*temp)
		{
			fprintf (stderr, "Missing KEY on line #%i, %s\n", *line_n, line);
			return -1;
		}

		for (knode = (*clast)->key; knode; knode = knode->next)
			if (strcasecmp (knode->name, temp) == 0)
			{
				fprintf (stderr, "Duplicate KEY on line #%i, %s\n", *line_n, line);
				return -1;
			}

		knode = (struct _KEY *) malloc (sizeof (struct _KEY));
		memset (knode, 0, sizeof (struct _KEY));

		knode->name = (char *) malloc (strlen (temp) + 1);
		strcpy (knode->name, temp);
#ifdef _DEBUG_READ_CFG_
		fprintf (stderr, "%s ", temp);
#endif
		if (!*temp2)
		{
			fprintf (stderr, "Missing VALUE on line #%i, %s\n", *line_n, line);
			knode->value = (char *) malloc (strlen ("0") + 1);
			sprintf (knode->value, "0");
		}
		else
		{
			knode->value = (char *) malloc (strlen (temp2) + 1);
			strcpy (knode->value, temp2);
#ifdef _DEBUG_READ_CFG_
			fprintf (stderr, "%s\n", knode->value);
#endif
		}		

		if (!*klast)
			(*clast)->key = *klast = knode;
		else
		{
			(*klast)->next = knode;
			*klast = knode;
		}
	}
	else
	{
		fprintf (stderr, "Invalid config line #%i, %s\n", *line_n, line);
//		return -1;
	}

	return 0;
}

/* Reads the config file and constructs a tree */
struct _CFG *cfg_read (const char *filename, int print_on)
{
	FILE *cf;
	struct _CFG *conf, *clast;
	struct _KEY *klast;
	char line[MAX_LINE_SIZE];
	int line_n;

	conf = clast = NULL;
	klast = NULL;

	cf = fopen (filename, "r");
	if (!cf)
	{
		if (print_on == 1)
  		{
    			fprintf (stderr, "Can't find config file \"%s\"\n", filename);
   		}

		return (NULL);
   	}

	line_n = 0;
	line[0] = '*';		/* this stops the space searchers from going too far */

	while (fgets (&line[1], MAX_LINE_SIZE - 1, cf) != NULL)
	{
		if (cfg_parse_line(&line_n, line, &conf, &clast, &klast) < 0)
		{
			conf = NULL;
			break;
		}
	}

	if (!conf)
		fprintf (stderr, "Invalid config file %s.\n", filename);

	fclose (cf);

	return (conf);
}

/* Reads the config file and constructs a tree */
struct _CFG *cfg_read_mem (char *app, char *data)
{
	struct _CFG *conf, *clast;
	struct _KEY *klast;
	char *data_copy;
	char *str_token;
	char line[MAX_LINE_SIZE];
	int line_n;

	conf = clast = NULL;
	klast = NULL;

	line_n = 0;
	line[0] = '*';		/* this stops the space searchers from going too far */

	/* workaround, add the section name here */
	snprintf(line + 1, MAX_LINE_SIZE - 1, "[%s]", app);

	if (cfg_parse_line(&line_n, line, &conf, &clast, &klast) < 0)
	{
		conf = NULL;
		goto out;
	}

	data_copy = strdup(data);

	str_token = strtok(data_copy, "\n");

	while (str_token != NULL)
	{
		strncpy(line + 1, str_token, MAX_LINE_SIZE - 1);

		if (cfg_parse_line(&line_n, line, &conf, &clast, &klast) < 0)
		{
			conf = NULL;
			break;
		}

		str_token = strtok(NULL, "\n");
	}

	free(data_copy);

out:
	if (!conf)
		fprintf (stderr, "Invalid config record\n");

	return (conf);
}

struct _KEY *cfg_get_app (struct _CFG *cnode, const char *app)
{
	while (cnode)
	{

#ifdef _DEBUG_READ_CFG_
		fprintf (stderr, "cnode: %s %s\n", cnode->app, app);
#endif
		
		if (!strcmp (app, cnode->app))
			return cnode->key;

		cnode = cnode->next;
	}

	return NULL;
}

char *cfg_get_val_with_knode (struct _KEY *knode, const char *key_name)
{
	while (knode)
	{
#ifdef _DEBUG_READ_CFG_
		fprintf (stderr, "knode: %s %s\n", knode->name, key_name);
#endif

		if (!strcmp (key_name, knode->name))
			return knode->value;

		knode = knode->next;
	}

	return NULL;
}

char *cfg_get_val(struct _CFG *cfg, const char *app, const char *key_name)
{
	struct _KEY *knode;

	knode = cfg_get_app(cfg, app);
	if (knode == NULL)
		return NULL;

	return cfg_get_val_with_knode(knode, key_name);
}

int cfg_get_val_int (struct _KEY *knode, const char *key_name, int min_val, int max_val)
{
	char *value;
	int val;

	value = cfg_get_val_with_knode(knode, key_name);
	if (value != NULL)
	{
		val = strtol (value, NULL, 0);

		if ((val < min_val || val > max_val) && val != 0)
		{
			fprintf (stderr,
				 "%s = %d out of range [%d; %d] => using min value %d\n",
				 key_name, val, min_val, max_val, min_val);
			return min_val;
		}
		else
			return val;
	}

	fprintf (stderr, "Missing %s => use min value: %d\n", key_name, min_val);

	return min_val;
}

int cfg_get_val_ip (struct _KEY *knode, const char *key_name, const char *default_ip)
{
	const char *value;
	int val;
	char *ip_ascii;
	char *str_token;
	int i;

	value = cfg_get_val_with_knode(knode, key_name);
	if (value == NULL)
	{
		fprintf (stderr, "Missing %s => use default value: %s\n", key_name, default_ip);
		value = default_ip;
	}

	ip_ascii = strdup(value);
	str_token = strtok(ip_ascii, ".");
	val = 0;
	for (i = 3; i >= 0; i--)
	{
		val |= (strtol (str_token, NULL, 10) & 0xff) << ( 8 * i);
		str_token = strtok(NULL, ".");
	}

	free (ip_ascii);

	return val;
}

void cfg_get_val_mac (struct _KEY *knode, const char *key_name, u_int16_t *mac_addr,u_int16_t *default_mac_addr)
{
	char *value;
	char *mac_ascii;
	char *str_token;
	int i;

	value = cfg_get_val_with_knode(knode, key_name);
	if (value == NULL)
	{
		fprintf (stderr, "Missing %s => use default value\n", key_name);
		mac_addr[0] = default_mac_addr[0];
		mac_addr[1] = default_mac_addr[1];
		mac_addr[2] = default_mac_addr[2];
	}
	else
	{
		mac_ascii = strdup(value);
		str_token = strtok(mac_ascii, ":");

		/* read it in LE form */
		for (i = 2; i >= 0; i--)
		{
			mac_addr[i] = 0;
			mac_addr[i] |= (strtol (str_token, NULL, 16) & 0xff) << 8;
			str_token = strtok(NULL, ":");

			mac_addr[i] |= (strtol (str_token, NULL, 16) & 0xff);
			str_token = strtok(NULL, ":");
		}
	}
}

void cfg_get_val_le_mac (struct _KEY *knode, const char *key_name, u_int16_t *mac_addr,u_int16_t *default_mac_addr)
{
	char *value;
	char *mac_ascii;
	char *str_token;
	int i;

	value = cfg_get_val_with_knode(knode, key_name);
	if (value == NULL)
	{
		fprintf (stderr, "Missing %s => use default value\n", key_name);
		mac_addr[0] = default_mac_addr[0];
		mac_addr[1] = default_mac_addr[1];
		mac_addr[2] = default_mac_addr[2];
	}
	else
	{
		mac_ascii = strdup(value);
		str_token = strtok(mac_ascii, ":");

		/* read it in E form */
		for (i = 0; i < 3 ; i++)
		{
			mac_addr[i] = 0;
			mac_addr[i] |= (strtol (str_token, NULL, 16) & 0xff) << 8;
			str_token = strtok(NULL, ":");

			mac_addr[i] |= (strtol (str_token, NULL, 16) & 0xff);
			str_token = strtok(NULL, ":");
			mac_addr[i] = htons(mac_addr[i]);
		}
	}
}

void cfg_print(struct _CFG *conf)
{
	struct _CFG *cnode;
        struct _KEY *knode;

	while (conf)
	{
		printf ("[%s]\n", conf->app);
		while(conf->key)
		{
			knode = conf->key;

			printf ("%s=%s\n", knode->name, knode->value);

			conf->key = knode->next;
		}

		printf("\n");

		cnode = conf;
		conf = conf->next;
        }
}


// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ CFGFILE Library
//  cfg_clean
//
//  Purpose :	
//  Input   :	
//  Output  :	
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
void cfg_clean(struct _CFG *conf)
{
	struct _CFG *cnode;
	struct _KEY *knode;

	while (conf)
	{
		while(conf->key)
		{
			knode = conf->key;
			conf->key = knode->next;
			free(knode->value);
			free(knode->name);
			free(knode);
		}

		cnode = conf;
		conf = conf->next;
		free(cnode->app);
		free(cnode);
	}
}

/**
 * print_ipv4_addr -
 *
 *
 */
void print_ipv4_addr(u_int32_t *ip_addr, int crlf)
{
	struct in_addr be_ip_addr;

	be_ip_addr.s_addr = htonl(*ip_addr);
	
	printf ("ipv4_addr(%s)", inet_ntoa(be_ip_addr));

	if (crlf)
		printf(" \n");
}

/**
 * print_be_mac_addr -
 *
 *
 */
void print_be_mac_addr(u_int16_t *mac_addr, int crlf)
{
	u_int8_t *this_byte;
	this_byte = (u_int8_t *)mac_addr;

	printf ("mac_addr(%02x:%02x:%02x:%02x:%02x:%02x)", this_byte[0], this_byte[1], this_byte[2], this_byte[3], this_byte[4], this_byte[5]);

	if (crlf)
		printf("\n");
}

void print_le_mac_addr(u_int16_t *mac_addr, int crlf)
{
	u_int8_t *this_byte;
	this_byte = (u_int8_t *)mac_addr;

	printf ("mac_addr(%02x:%02x:%02x:%02x:%02x:%02x)", this_byte[5], this_byte[4], this_byte[3], this_byte[2], this_byte[1], this_byte[0]);

	if (crlf)
		printf("\n");
}
/**
 * print_be_ipv6_addr -
 *
 *
 */
void print_be_ipv6_addr(struct in6_addr *be_ipv6_addr, int crlf)
{
	char ip_string[40];

	inet_ntop(AF_INET6, be_ipv6_addr, ip_string, sizeof(ip_string));

	printf ("ipv6_addr(%s)", ip_string);

	if (crlf)
		printf("\n");
}
