/*! \file readcfg.h
	\brief This file contains the helper macros to read the configuration file (-c option)
*/

/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#ifndef _READCFG_H
#define _READCFG_H

#include "vxcompat.h"

#if !defined(_VXWORKS_)
#include <arpa/inet.h>
#else
#include <inetLib.h>
#endif

#define MAX_LINE_SIZE 2048

#if !defined(_VXWORKS_)
#ifndef _NETINET_IP6_H
struct ip6_hdr
{
	union
	{
		struct ip6_hdrctl
		{
			U32	ip6_un1_flow;	/* 4 bits version, 8 bits TC,
						 * 20 bits flow-ID
						 */
			U16	ip6_un1_plen;	/* payload length */
			U8	ip6_un1_nxt;	/* next header */
			U8	ip6_un1_hlim;	/* hop limit */
		} ip6_un1;

		U8	ip6_un2_vfc;		/* 4 bits version, top 4 bits tclass */
	} ip6_ctlun;

	struct in6_addr ip6_src;	/* source address */
	struct in6_addr ip6_dst;	/* destination address */
};
#endif
#endif

#define		READ_INT(this_config, this_section, key_name, parameter, default)	\
		do {									\
			char * value;							\
											\
			value = cfg_get_val( this_config, this_section, key_name);	\
											\
			if (value)							\
			{								\
				parameter = strtol(value, NULL, 0);			\
			}								\
			else								\
			{								\
				parameter = default;					\
			}								\
		} while (0)								\

#define		READ_INT_WITH_RC(this_config, this_section, key_name, parameter, default, rc)	\
		do {									\
			char * value;							\
											\
			value = cfg_get_val( this_config, this_section, key_name);	\
											\
			if (value)							\
			{								\
				parameter = strtol(value, NULL, 0);			\
				rc = 1;							\
			}								\
			else								\
			{								\
				parameter = default;					\
				rc = 0;							\
			}								\
		} while (0)								\

#define         READ_STR(this_config, this_section, key_name, this_string, default)	\
		do {									\
			char * value;							\
											\
                        value = cfg_get_val(this_config, this_section, key_name);	\
			if (value)							\
			{								\
				strcpy(this_string, value);				\
			}								\
			else								\
			{								\
				strcpy(this_string, default);				\
			}								\
		} while (0)

#define		READ_STRN(this_config, this_section, key_name, this_string, size, default)	\
		do {									\
			char * value;							\
											\
			memset(this_string, 0, size);					\
											\
			value = cfg_get_val(this_config, this_section, key_name);	\
			if (value)							\
			{								\
				strncpy(this_string, value, size - 1);			\
			}								\
			else								\
			{								\
				strncpy(this_string, default, size - 1);		\
			}								\
		} while (0)

#define		READ_IP(this_config, this_section, key_name, parameter, default)	\
		do {									\
			struct _KEY *knode;						\
			knode = cfg_get_app(this_config, this_section);			\
			cfg_get_val_ip(knode, key_name, parameter, default);		\
		} while (0)

#define		READ_MAC(this_config, this_section, key_name, parameter, default )	\
		do {									\
			struct _KEY *knode;						\
			knode = cfg_get_app(this_config, this_section);			\
			cfg_get_val_mac(knode, key_name, parameter, default);		\
		} while (0)


struct _KEY
{
	char *name;
	char *value;
	struct _KEY *next;
};

struct _CFG
{
	char *app;
	struct _KEY *key;
	struct _CFG *next;
};

struct _CFG	*cfg_read(const char *filename, int print_on);
struct _CFG	*cfg_read_mem(char *app, char *data);
void		cfg_clean(struct _CFG *cfg);
void		cfg_print(struct _CFG *cfg);
void		cfg_export_to_db(struct _CFG *cfg, char *base_file);
struct _KEY	*cfg_get_app(struct _CFG *cfg, const char *app);
char		*cfg_get_val_with_key(struct _KEY *knode, const char *key_name);
char		*cfg_get_val(struct _CFG *cfg, const char *app, const char *key_name);
int		cfg_get_val_int(struct _KEY *knode, const char *key_name, int min_val, int max_val);
void		cfg_get_val_ip(struct _KEY *knode, const char *key_name, U8 *ip_addr, const char *default_ip);
void		cfg_get_val_mac(struct _KEY *knode, const char *key_name, U8 *mac_addr, U8 *default_mac_addr);

void		print_ip_addr(const char *prefix, U8 *ip_addr);
void		print_mac_addr(const char *prefix, U8 *mac_addr);

#endif
