/*! \file demo_utility.c
    \brief VAPI demo utility functions

 @defgroup Utilities vapi pots to pots application example utilities
 *  @{
 */
 
/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include <string.h>
#ifndef _VXWORKS_
#include <vapi/msp.h>
#include <vapi/vapi.h>
#include <vapi/gtl.h>
#else
#include <msp.h>
#include <vapi.h>
#include <gtl.h>
#endif

#include "demo.h"
#include "readcfg.h"

#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>

/* call control interface */
#include <legerity_lib.h>

/* These are global variables intended to be modified from the vxworks shell */
/*_DDK_ is defined when vapi_demo is build together with the vxWorks kernel + VAPI + legerity */ 
#if defined (_DDK_) 
	int EC_MODE = 0;
	int LOOPBACK_MODE = 5;
	unsigned short NB_CODEC = 4;
	unsigned short WB_CODEC = 4;
	unsigned short PACKET_SIZE = 20;
	U8 DEV_IP_ADDR[4] = {192, 168, 32, 200};
	U8 DEV_MAC_ADDR[6] = {0x00, 0xaa, 0xbb, 0xcc, 0xdd, 0xee};
	U8 HOST_MAC_ADDR[6] = {0x00, 0x11, 0xd8,  0xe7, 0xa5, 0x2d};
	char DIAG_STRING[50] = "NONE";
#endif

extern char config_file[FILENAME_MAX];
extern int is_wb_slic;
/* default values used later in default_device_configuration */

/* device control MAC used at boot stage
 *
 * On Slave configuration this address is set by the MAAS_ASSIGN command (VAPi_AssignBootMAC)
 * On Master configuration this address must be the same as the on set by the bootloader (u-boot, ARMboot),
 * the default value is 00:11:22:33:44:55
 */
/*00:11:22:33:44:55*/
#define DEVICE_CONTROL_MAC		{0x00,0x11,0x22,0x33,0x44,0x55}

/* host control MAC
 *
 * On slave device this MAC address is the MAC address on the network interface
 * used to control the device. For example if a PC is used to control the device,
 * it is the MAC address of the network card
 */
#define HOST_CONTROL_INTERFACE		"eth1"
#define HOST_CONTROL_MAC		{0x00,0x1A,0x1B,0x1C,0x1D,0x1E}


/* This structure holds default values for the device configuration, user can
 * overwrite them somehow in set_device_configuration() function using his own
 * config. scheme.
 */
static SCSMEUsrData default_device_configuration = {
	NULL,			/* pointer to next device */
	eCSM_ITF,		/* control interface to use */
	0,			/* device ID */
	0,			/* Version infor (not used) */
	eMASTER,		/* Slave / Master mode */
	DEV_TYPE_M825XX,	/* Device type */
	True,			/* Default or custom max channels */
	0,			/* Max Channels if above flag is custom */
	DEVICE_CONTROL_MAC,	/* MAC address for control over csmencaps */
	HOST_CONTROL_MAC,	/* MAC address of the host interface controling the device */
	(char *)HOST_CONTROL_INTERFACE,	/* host interface used to control the device */
	1			/* csme ack required */
};

/*=================================================================================*/
/*! \brief
 *	
 *	This function fills the global structure with device config parameters
*/
void set_device_configuration(SCSMEUsrData *device_configuration)
{
	U8 *mac;

	memcpy(device_configuration, &default_device_configuration, sizeof(SCSMEUsrData));

	/* user may put here code to overwrite default config parameters, for example
	 * read them from config file
	 */

	device_configuration->uiDevId = 0;
	device_configuration->pucEthDevName = HOST_CONTROL_INTERFACE;

	/* this code is executed before VAPI_Init, so we have to use printf's */

	PDEBUG(DEBUG_INIT, "CONTROL_INTERFACE_TYPE  = %d\n", device_configuration->usControlInterface);
	PDEBUG(DEBUG_INIT, "DEVICE_ID    = %d\n", device_configuration->uiDevId);
	PDEBUG(DEBUG_INIT, "DEVICE_MODE  = %d\n", device_configuration->eDevMode);
	PDEBUG(DEBUG_INIT, "DEVICE_TYPE  = %d\n", device_configuration->ucDevType);
	PDEBUG(DEBUG_INIT, "USE_DEFAULT_MAX_CHANNEL = %d\n", device_configuration->bUseDefaultMaxChnls);
	PDEBUG(DEBUG_INIT, "CUSTOM_MAX_CHANNEL      = %d\n", device_configuration->usMaxChannels);
	PDEBUG(DEBUG_INIT, "ACK_REQUIRED = %d\n", device_configuration->ucIsAckReqd);

	mac = device_configuration->aucDevMac;
	PDEBUG(DEBUG_INIT, "DEVICE MAC = %02x:%02x:%02x:%02x:%02x:%02x\n", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

	mac = device_configuration->aucHostMac;
	PDEBUG(DEBUG_INIT, "HOST   MAC = %02x:%02x:%02x:%02x:%02x:%02x\n", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

	PDEBUG(DEBUG_INIT, "HOST_CONTROL_INTERFACE  = %s\n\n", device_configuration->pucEthDevName);
}

/*=================================================================================*/
/*! \brief
*	
*	This function changes the default VAPI configuration.
*/
VSTATUS set_default_config(IN OUT SVAPIConfig * default_config)
{
	/* change the default timeout to 10S */
	default_config->uiMspRespTimeout = 10;	/* MSP response timeout value 10 seconds */

	return SUCCESS;
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function sets an endpoint in a error state
*/

void set_endpoint_error(VSTATUS status, int index)
{
	set_endpoint_state(ENDPOINT_STATE_NONE, index);
	set_endpoint_event(ENDPOINT_EVENT_NONE, index);
	PDEBUG(DEBUG_ERROR, "Endpoint %d: error %d", index, status);
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function sets an event for the specified endpoint
*/
void set_endpoint_event(int event_type, int index)
{
	endpoints[index].event = event_type;
	PDEBUG(DEBUG_FUNCTION, "Endpoint %d: set event %d (state %d)", index,endpoints[index].event, endpoints[index].state);
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function sets an endpoint in the specified state
*/
void set_endpoint_state(int state, int index)
{
	endpoints[index].state = state;
	PDEBUG(DEBUG_FUNCTION, "Endpoint %d: set state %d (event %d)", index,endpoints[index].state, endpoints[index].event);
}

/*=================================================================================*/
/*! \brief
*	
*	This function creates a VoIP endpoint using VAPI_CreateConnection
*	If the state = -1 the request is performed is SYNC mode else ASYNC mode.
*	The endpoint ID is used as the timeslot number.
*	It also sets the codec and the packet size of the channel.
*/
VSTATUS create_endpoint(const char *config_file, int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request, *this_request = NULL;
	struct _ENDPOINT_CFG_DESC endpoint_cfg;
	
	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_FUNCTION, "Endpoint %d: create connection type %d, timeslot %d(request id = 0x%04x)", endpoint_id, eVOIP, endpoint_id,request.uiReqId);

	if (endpoint_state != ENDPOINT_STATE_SYNC)
		this_request = &request;

	if (channel_mode == eNarrowBand)
	{
		result = VAPI_AllocateConnection(0,
					endpoint_id,
					eVOIP,          /* type 2*/
					eNarrowBand,
					1,
					endpoints[endpoint_id].timeslot,
					this_request,
					NULL);
	}
	else
	{
		result = VAPI_AllocateConnection(0,
					endpoint_id,
					eVOIP,          /* type 2*/
					eWideBand,
					4,
					endpoints[endpoint_id].timeslot,
					this_request,
					NULL);
	}		

	if (result != SUCCESS)
		goto out;
		
	/* Configure a payload type for G722 */
	result = VAPI_SetPayloadType(endpoint_id, eG722, 0x66, eBoth, NULL);

	/* get endpoint config from config file*/
	result = get_endpoint_cfg(config_file, &endpoint_cfg);

	/* if the codec is supported switch both endpoints */

	if ((endpoint_cfg.nb_codec == eG711_ULAW_ID) ||
		(endpoint_cfg.nb_codec == eG711_ALAW_ID) ||
		(endpoint_cfg.nb_codec == eG723_1_ID) ||
		(endpoint_cfg.nb_codec == eG728_Audio_ID) ||
		(endpoint_cfg.nb_codec == eG729_A_ID) ||
		(endpoint_cfg.nb_codec == eG722_ID) ||
		(endpoint_cfg.nb_codec == eG726_32_ID))
	{
		result = VAPI_SetCodecType(endpoints[endpoint_id].index, endpoint_cfg.nb_codec, NULL);
		printf("Endpoint %d running codec %d\n", endpoints[endpoint_id].index, endpoint_cfg.nb_codec);
	}
		
	result = VAPI_SetPacketInterval(endpoint_id, endpoint_cfg.packet_size, NULL);

	result = VAPI_SetConnectionState(endpoint_id, eTdmActive, NULL);

out:
	return result;
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function destroy a previously created endpoint in VAPI ASYNC mode.
*/
VSTATUS destroy_endpoint(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request, *this_request = NULL;

	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_FUNCTION, "Endpoint %d: destroy connection (request id = 0x%04x)", endpoint_id, request.uiReqId);

	if (endpoint_state != ENDPOINT_STATE_SYNC)
		this_request = &request;

	result = VAPI_DestroyConnection(endpoint_id, this_request);
	return result;
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function generates a dialtone to the specified endpoint.
*/
VSTATUS dialtone_generation(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request, *this_request = NULL;

	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_FUNCTION, "Endpoint %d: dial tone generation (request id = 0x%04x)", endpoint_id, request.uiReqId);

	if (endpoint_state != ENDPOINT_STATE_SYNC)
		this_request = &request;

	result = VAPI_PlayTone(endpoint_id, eDIALTONE, eDirToTDM, NULL, 0, this_request);

	return result;
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function generates a busytone to the specified endpoint.
*/
VSTATUS busytone_generation(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request, *this_request = NULL;

	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_FUNCTION, "Endpoint %d: busy tone generation (request id = 0x%04x)", endpoint_id, request.uiReqId);

	if (endpoint_state != ENDPOINT_STATE_SYNC)
		this_request = &request;

	result = VAPI_PlayTone(endpoint_id, eBUSYTONE, eDirToTDM, NULL, 0, this_request);
	return result;
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function generates a busytone to the specified endpoint.
*/
VSTATUS ringbacktone_generation(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request, *this_request = NULL;

	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_FUNCTION, "Endpoint %d: ring back tone generation (request id = 0x%04x)", endpoint_id, request.uiReqId);

	if (endpoint_state != ENDPOINT_STATE_SYNC)
		this_request = &request;

	result = VAPI_PlayTone(endpoint_id, eRINGBACKTONE, eDirToTDM, NULL, 0, this_request);
	return result;
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function generates a waiting tone to the specified endpoint.
*/
VSTATUS specialtone_generation(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request, *this_request = NULL;

	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_FUNCTION, "Endpoint %d: special tone generation (request id = 0x%04x)", endpoint_id, request.uiReqId);

	if (endpoint_state != ENDPOINT_STATE_SYNC)
		this_request = &request;

	result = VAPI_PlayTone(endpoint_id, eWAITINGTONE, eDirToTDM, NULL, 0, this_request);
	return result;
}

/*=================================================================================*/
/*! \brief
 *	
 *	This function stops a tone currently generated on the specified endpoint.
*/
VSTATUS stop_tone_generation(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request, *this_request = NULL;

	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_FUNCTION, "Endpoint %d: stop tone generation (request id = 0x%04x)", endpoint_id, request.uiReqId);

	if (endpoint_state != ENDPOINT_STATE_SYNC)
		this_request = &request;

	result = VAPI_StopTone(endpoint_id, 0, 0, this_request);
	return result;
}

/*=================================================================================*/
/*! \brief
*	
*	This function initializes the device network parameters:
*	- ETH header
*	- IP address.
*/
int network_parameters_initialisation(const char *config_file)
{
	SIpAddrInfo ip_info;
	int result = SUCCESS;
	
#if defined (_DDK_) && defined (_VXWORKS_)
	memcpy (&device_ip_address,  &DEV_IP_ADDR, sizeof(DEV_IP_ADDR));
	memcpy (&device_mac_addr,  &DEV_MAC_ADDR, sizeof(DEV_MAC_ADDR));
	memcpy (&host_mac_addr,  &HOST_MAC_ADDR, sizeof(HOST_MAC_ADDR));
#else

	struct _CFG *cfg_info;
	cfg_info = cfg_read(config_file, 0);
	if (!cfg_info)
		return -1;

	PDEBUG(DEBUG_INIT,"Init IPv4 device address");
	READ_IP(cfg_info, "DEVICE_NETWORK_CONFIG", "DEVICE_IP_ADDRESS", device_ip_address, "192.168.1.2");

	PDEBUG(DEBUG_INIT,"Init device MAC address");
	READ_MAC(cfg_info,"DEVICE_NETWORK_CONFIG", "DEVICE_MAC_ADDRESS", &device_mac_addr[0], &def_mac_addr[0]);


	PDEBUG(DEBUG_INIT,"Init host MAC address");
	READ_MAC(cfg_info,"DEVICE_NETWORK_CONFIG", "HOST_MAC_ADDRESS", &host_mac_addr[0], &def_mac_addr[0]);

	result = VAPI_SetEthMac(0, CMD_LEVEL_DEVICE, (U8 *)&device_mac_addr[0], (U8 *)&host_mac_addr[0], NULL);

	cfg_clean(cfg_info);
#endif

	ip_info.ucNumOfSrcIpAddr = 1;
	ip_info.bIsMultipleMode = 0;
	ip_info.ucEncapsulation = 2;
	ip_info.ucChkSumVerification = 1;

	ip_info.auiDevIPAddress[0] = (device_ip_address[3] << 24) | (device_ip_address[2] << 16) | (device_ip_address[1] << 8) | device_ip_address[0];

	result |= VAPI_SetDeviceIPAddr(0, &ip_info, NULL);


	return result;
}

/*=================================================================================*/
/*! \brief
*	
*	This enable/disable choosen diagnostics.
*/
int set_diagnostics(unsigned int id,   /* if 0xFFFF device else connection*/
			unsigned char command_class, 
			unsigned char command_type, 
			unsigned short function_code, 
			unsigned char action) /* 1 = enable, 0 = disable */
{
	VSTATUS result = SUCCESS;
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/* allocate a message to query the current options */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;

	/* build the command to set the diag config to enable or disable */
	result = VAPI_SetMessage(message, command_class, command_type, function_code, 1, action);
	if(result != SUCCESS)
		goto err;

	if(id == 0xFFFF)
		result = VAPI_SendDeviceMessage(0, (SMsg *)message, NULL, 
		device_response, &response_len);
	else
		result = VAPI_SendConnectionMessage(id, (SMsg *)message, NULL, 
		device_response, &response_len);

err:
	VAPI_FreeMessage(message);
	return result;
}

/*=================================================================================*/
/*! \brief
*	
*	This function initializes the TDM buses
*/
int tdm_parameters_initialisation(const char *config_file)
{
	int i;
	STdmSetupParams tdm_parameters;
	int result;

#if defined (_DDK_) && defined (_VXWORKS_)

	tdm_parameters.usMode = 2;
	tdm_parameters.usNoOfBus =  2;
	tdm_parameters.astTdmBusParam[0].ucBusId = 0;
	tdm_parameters.astTdmBusParam[0].ucTxOrder = 1;
	tdm_parameters.astTdmBusParam[0].ucRxPolarity = 0;
	tdm_parameters.astTdmBusParam[0].ucTxPolarity = 0;
	tdm_parameters.astTdmBusParam[0].ucRxClkMode = 0;
	tdm_parameters.astTdmBusParam[0].ucTxClkMode = 0;
	tdm_parameters.astTdmBusParam[0].ucRxClkEdge = 0;
	tdm_parameters.astTdmBusParam[0].ucTxClkEdge = 1;
	tdm_parameters.astTdmBusParam[0].ucFrameEdge = 0;
	tdm_parameters.astTdmBusParam[0].usNumOfTS =  128;
	tdm_parameters.astTdmBusParam[0].ucInvertedFrmSig = 0;
	tdm_parameters.astTdmBusParam[0].ucBitOrderRcv = 1;
	tdm_parameters.astTdmBusParam[0].usNumTdmClkCyclesDelayTx = 0;
	tdm_parameters.astTdmBusParam[0].usNumTdmClkCyclesDelayRx = 0;
#else

	struct _CFG *cfg_info;

	/* open the configuration file*/
	cfg_info = cfg_read(config_file, 0);
	if (!cfg_info)
		return -1;

	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "TDM_BUS_MODE", tdm_parameters.usMode, 1);

	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "TDM_BUS_TO_ENABLE", tdm_parameters.usNoOfBus, 1);
 
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "BUS_ID", tdm_parameters.astTdmBusParam[0].ucBusId, 0);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "TX_ORDER", tdm_parameters.astTdmBusParam[0].ucTxOrder, 0);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "RX_POLARITY", tdm_parameters.astTdmBusParam[0].ucRxPolarity, 0);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "TX_POLARITY", tdm_parameters.astTdmBusParam[0].ucTxPolarity, 0);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "RX_CLOCK_MODE", tdm_parameters.astTdmBusParam[0].ucRxClkMode, 0);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "TX_CLOCK_MODE", tdm_parameters.astTdmBusParam[0].ucTxClkMode, 0);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "RX_CLOCK_EDGE", tdm_parameters.astTdmBusParam[0].ucRxClkEdge, 0);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "TX_CLOCK_EDGE", tdm_parameters.astTdmBusParam[0].ucTxClkEdge, 1);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "FRAME_EDGE", tdm_parameters.astTdmBusParam[0].ucFrameEdge, 1);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "NUMBER_OF_TS", tdm_parameters.astTdmBusParam[0].usNumOfTS, 0x20);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "INVERTED_FRAME_SIGNAL", tdm_parameters.astTdmBusParam[0].ucInvertedFrmSig, 1);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "BIT_ORDER_RCV", tdm_parameters.astTdmBusParam[0].ucBitOrderRcv, 1);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "NUM_TDM_CLK_CYCLES_DELAY_TX", tdm_parameters.astTdmBusParam[0].usNumTdmClkCyclesDelayTx, 0);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "NUM_TDM_CLK_CYCLES_DELAY_RX", tdm_parameters.astTdmBusParam[0].usNumTdmClkCyclesDelayRx, 0);

#endif

	/* uses the same parameters for all buses*/
	for (i = 1; i < tdm_parameters.usNoOfBus; i++)
		memcpy(&tdm_parameters.astTdmBusParam[i], &tdm_parameters.astTdmBusParam[0], sizeof(STdmBusParams));

	/* FIXME: we should check the result */
	result = VAPI_SetTDMParams(0, &tdm_parameters, NULL);

	/*Not really an issue ignore it this happens if we try to initialise the TDM several time*/
	if (result == -CNF_ERR_TDM_CONFIG_PARMS_FAILED_SELECT_MULT_INSTANCE)
		result = SUCCESS; 

	return result;
}
/*=================================================================================*/
/*! \brief
*	
*	This iunction initializes the diagnostics (if any)
*/
int diagnostics_initialisation(const char *config_file)
{
	char temp_string[50];
	char *str_token;
	U16 function_code;
	int result = 0;

#if defined (_DDK_) && defined (_VXWORKS_)
		memcpy(temp_string, DIAG_STRING, sizeof(temp_string));
#else

	struct _CFG *cfg_info;

	/* open the configuration file*/
	cfg_info = cfg_read(config_file, 0);
	if (!cfg_info)
		return -1;

	/* read the diag string from the condif file*/
	if (get_tdm_diag_mode(config_file, temp_string, sizeof(temp_string)) != 0)
		return -1;

#endif

	if (strncmp(temp_string, "NONE", 4) == 0)
	{
		printf("No diag activated\n");
		result = 0;
	}
	else
	{
		str_token = (char *)strtok(temp_string, ",");
		while (str_token != NULL)
		{
			function_code = strtol(str_token, NULL, 16);
			printf("Set diag 0x%04x \n", function_code);
			result |= set_diagnostics(0xffff, CMD_CLASS_OPEN_DIAG, CMD_TYPE_DIAG_MON_LIVE_CTRL, function_code, 1);
			str_token = (char *)strtok(NULL, ",");
		}
	}
	
	return result;

}
/*=================================================================================*/
/*! \brief
*	
*	This function gets the loopback mode from the configuration file:
*	The configuration is comming from the /usr/local/etc/demo.conf file
*/
int get_loopback_mode(const char *config_file)
{
	int loopback_mode;

#if defined (_DDK_) && defined (_VXWORKS_)

	loopback_mode = LOOPBACK_MODE;

#else

	struct _CFG *cfg_info;
	cfg_info = cfg_read(config_file, 0);
	if (!cfg_info)
		return -1;

	READ_INT(cfg_info, "CONNECTION_MNGT", "LOOPBACK_MODE", loopback_mode, 0);

	cfg_clean(cfg_info);

#endif

	return loopback_mode;
}

/*=================================================================================*/
/*! \brief
*	
*	This function gets the endpoint configuration to use from the configuration file.
*/
int get_endpoint_cfg(const char *config_file, struct _ENDPOINT_CFG_DESC *endpoint_cfg)
{
#if defined (_DDK_) && defined (_VXWORKS_)

	endpoint_cfg->nb_codec = NB_CODEC;
	endpoint_cfg->wb_codec = WB_CODEC;
	endpoint_cfg->packet_size = PACKET_SIZE;

#else
	struct _CFG *cfg_info;
	cfg_info = cfg_read(config_file, 0);
	if (!cfg_info)
		return -1;

	READ_INT(cfg_info, "CONNECTION_MNGT", "NB_CODEC", endpoint_cfg->nb_codec, 0);
	READ_INT(cfg_info, "CONNECTION_MNGT", "WB_CODEC", endpoint_cfg->wb_codec, 0);
	READ_INT(cfg_info, "CONNECTION_MNGT", "PACKET_SIZE", endpoint_cfg->packet_size, 0);

	cfg_clean(cfg_info);

#endif

	return 0;
}


/*=================================================================================*/
/*! \brief
*	
*	This function gets the echo canceller mode to use from the configuration file.
*/
int get_ec_mode(const char *config_file)
{
	int ec_mode;

#if defined (_DDK_) && defined (_VXWORKS_)
	ec_mode  = EC_MODE;
#else
	struct _CFG *cfg_info;
	cfg_info = cfg_read (config_file, 0);
	if (!cfg_info)
		return -1;

	READ_INT(cfg_info, "DEVICE_MNGT", "EC_MODE", ec_mode, 0);

	cfg_clean(cfg_info);

#endif
	return ec_mode;
}

/*=================================================================================*/
/*! \brief
*	
*	This function gets the TDM diagnostics mode from the configuration file:
*	The configuration is comming from the /usr/local/etc/demo.conf file
*/
int get_tdm_diag_mode(const char *config_file, char *string, int size)
{
#if !defined (_DDK_) && !defined (_VXWORKS_)
	struct _CFG *cfg_info;

	cfg_info = cfg_read(config_file, 0);
	if (!cfg_info)
		return -1;

	READ_STRN(cfg_info, "DIAGNOSTICS_MNGT", "TDM_DIAG_MODE", string, size, "NONE");

	cfg_clean(cfg_info);
#endif

	return 0;

}
/*=================================================================================*/
/*! \brief
*	
*	This function sets the IP/UDP header of a endpoint using VAPI_SetConnIpParams() in sync mode. \n
*/
VSTATUS set_ip_udp_parameters_sync(int endpoint_id, U8 *ip_src_addr, U8 *ip_dst_addr, unsigned short udp_src, unsigned short udp_dst)
{
	int result;
	SIpParams ip_parameters;

	PDEBUG(DEBUG_FUNCTION, "Endpoint %d: set ip parameters sync mode", endpoint_id);
	PDEBUG(DEBUG_FUNCTION, "set endpoint %d udp src %d dst %d\n", endpoint_id, udp_src, udp_dst);
	/* make sure all fields are set to 0, so ucIPServiceId field is set to RTP */
	memset(&ip_parameters, 0, sizeof(SIpParams));

	ip_parameters.uiSrcIpAddr = (ip_src_addr[3] << 24) | (ip_src_addr[2] << 16) | (ip_src_addr[1] << 8) | ip_src_addr[0];
	ip_parameters.uiDestIpAddr = (ip_dst_addr[3] << 24) | (ip_dst_addr[2] << 16) | (ip_dst_addr[1] << 8) | ip_dst_addr[0];

	ip_parameters.usSrcUdpPort = udp_src;
	ip_parameters.usDestUdpPort = udp_dst;

	/*RTP defined in comcerto_ip_device_level_api.h */
	ip_parameters.ucIPServiceId = SET_IP_HDR_CHANNEL_SERVICEID_DEFAULTPKT;

	result = VAPI_SetConnIpParams(endpoint_id, &ip_parameters, NULL);
	result = VAPI_SetRtpSsrcHeader (endpoint_id, udp_src | (udp_dst << 16), NULL);


	return result;
}


/*=================================================================================*/
/*! \brief
*	
*	This function enables the RTP traffic of an endpoint. \n
*/
VSTATUS start_voip_connection(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request, *this_request = NULL;
	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_FUNCTION, "Endpoint %d: start connection (request id = 0x%04x)", endpoint_id, request.uiReqId);

	if (endpoint_state != ENDPOINT_STATE_SYNC)
		this_request = &request;

	result = VAPI_SetConnectionState(endpoint_id, eActive, this_request);

	return result;
}

/*=================================================================================*/
/*! \brief
*	
*	This function switches an endpoint in T38 mode. \n
*/
VSTATUS switch_voip_connection_to_t38(int endpoint_id, int endpoint_state)
{
	int result;
	SRequest request, *this_request = NULL;
	request.uiReqId = endpoint_state;
	request.pfnIoCompCallback = &comcerto_response_handler;

	PDEBUG(DEBUG_FUNCTION, "Endpoint %d: switch connection to T38 (request id = 0x%04x)", endpoint_id, request.uiReqId);

	if (endpoint_state != ENDPOINT_STATE_SYNC)
		this_request = &request;

	result = VAPI_SetPayloadType(endpoint_id, eIFP, 0x62, eBoth, this_request);

	result |= VAPI_SwitchToT38(endpoint_id, NULL, this_request);
	result |= VAPI_SetConnectionState(endpoint_id, eActive, this_request);

	return result;
}

/*=================================================================================*/
/*! \brief
*	
*	This function handles the onhook event in several states of an endpoints.
*/
void onhook_handler(int index, int endpoint_state)
{
	int peer_endpoint;

	/* destroy the connection*/
	/* if for any reason it failed we do not check the status and reset some variables anyway*/
	destroy_endpoint(index, endpoint_state);

	/* the RTP port must be even */
	endpoints[index].rtp_port = htons(DEFAULT_UDP_PORT + index*2);
	/* the RTCP port must be odd */
	endpoints[index].rtcp_port = htons(DEFAULT_UDP_PORT + index*2 + 1);

	peer_endpoint = endpoints[index].peer_index;
	/*make sure this endpoint is no longer a waiting endpoint of a peer endpoint*/
	if(peer_endpoint != -1)
		endpoints[peer_endpoint].call_waiting_index = -1;

	memset(endpoints[index].phonenumber, 0, MAX_PHONE_SIZE);
	endpoints[index].phonenumber_index = 0;
	endpoints[index].peer_index = -1;
	endpoints[index].call_waiting_index = -1;
}

/*=================================================================================*/
/*! \brief
*	
*	This function sets the 2 lines of a SLIC in wideband mode.*/
int slic_set_wide_band(int fd, int line1, int line2)
{
	int status;

	printf ("Switching to WB mode\n");
	status = slic_write_regs(fd, line1, CODEC_REG_WRT, 1, 0x80);
	if (status >= 0)
		printf ("SUCCESS to set linear coding on %d line\n", line1);

	status |= slic_write_regs(fd, line2, CODEC_REG_WRT, 1, 0x80);
	if (status >= 0)
		printf ("SUCCESS to set linear coding on %d line\n", line2);

	/* set appropriate SLIC mode*/
	status |= slic_write_regs(fd,
			line1, /*line 1 or 2 doesn't matter: global param*/
			OPERATING_MODE_REGISTER, 1, 0x03 | VP880_WBAND_MODE_MASK);
	return status;
}

/*=================================================================================*/
/*! \brief
*	
*	This function sets the 2 lines of a SLIC in narrow band mode.*/
int slic_set_narrow_band(int fd, int line1, int line2)
{
	int status;

	printf ("Switching to NB mode\n");
	status = slic_write_regs(fd, line1, CODEC_REG_WRT, 1, 0x40);
	if (status >= 0)
		printf ("SUCCESS to set ulaw coding on %d line\n", line1);

	status |= slic_write_regs(fd, line2, CODEC_REG_WRT, 1, 0x40);
	if (status >= 0)
		printf ("SUCCESS to set ulaw coding on %d line\n", line2);

	/* set appropriate SLIC mode*/
	status |= slic_write_regs(fd,
			line1, /*line 1 or 2 doesn't matter: global param*/
			OPERATING_MODE_REGISTER, 1, 1, 0x03 & ~VP880_WBAND_MODE_MASK);
	return status;
}

/*=================================================================================*/
/*! \brief
*	
*	This function returns the slic device type and version.*/
unsigned short slic_get_version(int fd)
{
	unsigned short slic_version = 0;
	int status = slic_read_regs(fd, 1, REVISION_PRODUCT_CODE, 2, (unsigned char *)&slic_version);

	if (status < 0)
		printf ("FAIL slic device version\n");

	return slic_version;

}


/*=================================================================================*/
/*! \brief
*	
*	This function stop the current loopback (if any) */
int stop_loopback(int index)
{
	VSTATUS status = SUCCESS;
	int loopback_mode;

	/*Get the loopback mode set in the config file*/
	loopback_mode = get_loopback_mode(config_file);

	/* make sure the option from the config file is OK*/
	if (loopback_mode == eINTER_CHNL_POST_ENCAPS)
		status = VAPI_Loopback(endpoints[index].index, endpoints[index].peer_index, eREMOVE_INTER_CHNL_POSTENCAPS_LOOPBACK , NULL);
	else if (loopback_mode == eTHC)
		status = VAPI_Loopback(endpoints[index].index, endpoints[index].peer_index, eREMOVE_THC_LOOPBACK, NULL);
	/*else do nothing*/
	return status;
}

/*=================================================================================*/
/*! \brief
*	
*	This function handle the DTMF service
*	Depending the detected DTMF it take action accordingly */
void handle_dtmf_service(int i)
{
	U16 temp_index;
	int peer_endpoint;
	VSTATUS status;

	printf ("Endpoint %d DTMF service %d\n", endpoints[i].index, endpoints[i].dtmf_service);

	switch (endpoints[i].dtmf_service)
	{
	/*CALL WAITING SCENARIO*/
	case 1:
	/*Switch back to the original endpoint when DTMF 1*/
	case 2:
	/*Switch to the new endpoint when DTMF 2*/

		/*Do not perform any action if there is no endpoint in call waiting*/
		if (endpoints[i].call_waiting_index != -1)
		{
			PDEBUG(DEBUG_FUNCTION, "Remap to original caller index %d\n", endpoints[i].call_waiting_index);
			PDEBUG(DEBUG_FUNCTION, "Stop voice on endpoint %d \n", endpoints[i].peer_index);
			status = VAPI_SetConnectionState(endpoints[i].peer_index, eTdmActive, NULL);
	
			/*stop loopback (if any)*/
			status = stop_loopback(endpoints[i].peer_index);
	
			/*place the current peer in call waiting*/
			set_endpoint_state(ENDPOINT_STATE_CALL_WAITING, endpoints[i].peer_index);
			PDEBUG(DEBUG_FUNCTION, "Endpoint %d in ENDPOINT_STATE_CALL_WAITING\n",endpoints[i].peer_index); 
			/*Save the current peer endpoint index*/
			temp_index = endpoints[i].peer_index;
			/*The new peer endpoint is the one which was in call waiting */
			endpoints[i].peer_index = endpoints[i].call_waiting_index;
			/*The alod peer endpoint is now the call waiting endpoint*/
			endpoints[i].call_waiting_index = temp_index;
		
			peer_endpoint = endpoints[i].peer_index;
			/*get the new peer out of call waiting state*/
			set_endpoint_event(ENDPOINT_EVENT_CALL_ACCEPTED, endpoints[i].peer_index);
			PDEBUG(DEBUG_FUNCTION, "Endpoint %d set out of ENDPOINT_STATE_CALL_WAITING\n",endpoints[i].peer_index); 

			/*Change the IP header with the new peer endpoint UDP port*/
			status = set_ip_udp_parameters_sync(endpoints[i].index,
					endpoints[i].ip_src_addr,
					endpoints[peer_endpoint].ip_src_addr,
					endpoints[i].rtp_port,
					endpoints[peer_endpoint].rtp_port);

			/*Start the RTP traffic*/
			status = start_voip_connection(endpoints[i].index, ENDPOINT_STATE_SYNC);
		}

		set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);

		break;

	/*Set endpoints in narrowband mode when DTMF 8*/
	case 8:
		if (is_wb_slic == True)
		{
			/* narrow band mode*/
			channel_mode = eNarrowBand;
			set_endpoint_event(ENDPOINT_EVENT_SWITCH_TO_NB, endpoints[i].index);
			set_endpoint_state(ENDPOINT_STATE_SWITCH_MODE, endpoints[i].index);
		}
		else
		{
			set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
		}
		
		break;

	/*Set endpoints in Wideband mode when DTMF 9*/
	case 9:
		if (is_wb_slic == True)
		{
			/* wide band mode*/
			channel_mode = eWideBand;
			set_endpoint_event(ENDPOINT_EVENT_SWITCH_TO_WB, endpoints[i].index);
			set_endpoint_state(ENDPOINT_STATE_SWITCH_MODE, endpoints[i].index);
		}
		else
		{
			set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
		}
		
		break;

	default:
		set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
		break;
	}
}

/*=================================================================================
 *	This function get the device MR version
=================================================================================*/
int get_mr(U32 device_id)
{

#if defined (_DDK_) && defined (_VXWORKS_)
	VSTATUS result;
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/* allocate a message to query the current options */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;

	result = VAPI_SetMessage(message, CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY, FC_ARM_FW_BRANCH, 0);

	/* send the command, the response is stored in device_response*/
	if(result == SUCCESS)
	{
		result = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, device_response, &response_len);

		if(result == SUCCESS)
		{
			PDEBUG(DEBUG_FUNCTION, "FC_GET_ARM_CODE_VERSION: response %u bytes: ", response_len);
			result = atoi(&device_response[17]);
		}
	}

	VAPI_FreeMessage(message);

	return result;

#else	
	/*if not in a DDK and VxWorks config lets return 100 to make sure we allow wide band (see check in demo_main.c)*/
	return 100;
#endif

}


#if defined (_DDK_) && defined (_VXWORKS_)
/*The functions below allows to modify variable from the vxworks shell*/
void set_device_ip(U8 *  ip_string)
{
	char *ip_ascii;
	char *str_token;
	int i;

	ip_ascii = strdup(ip_string);
	str_token = strtok(ip_ascii, ".");
	for (i = 0; i < 4; i++)
	{
		DEV_IP_ADDR[i] = strtol(str_token, NULL, 10);
		str_token = strtok(NULL, ".");
	}

	free(ip_ascii);
}

void set_device_mac(U8 *  mac_string)
{
	char *mac_ascii;
	char *str_token;
	int i;

	mac_ascii = strdup(mac_string);
	str_token = strtok(mac_ascii, ":");
	for (i = 0; i < 6; i++)
	{
		DEV_MAC_ADDR[i] = strtol(str_token, NULL, 10);
		str_token = strtok(NULL, ":");
	}

	free(mac_ascii);
}

void set_host_mac(U8 *  mac_string)
{
	char *mac_ascii;
	char *str_token;
	int i;

	mac_ascii = strdup(mac_string);
	str_token = strtok(mac_ascii, ":");
	for (i = 0; i < 6; i++)
	{
		HOST_MAC_ADDR[i] = strtol(str_token, NULL, 10);
		str_token = strtok(NULL, ":");
	}

	free(mac_ascii);
}

void set_diagnostic(U8 *  diag_string)
{
	strncpy(DIAG_STRING, diag_string, sizeof(DIAG_STRING));
}

#endif
/** @} */ /*pots to pots example utilities*/
