/*! \file demo_main.c
    \brief VAPI demo main file

 @defgroup VAPI demo event and state machine handlers
 *  @{
 */
/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

/*! \mainpage Demo application using the VAPI (MSP control) and Legerity (SLIC control) libraries services.

	This example is multi threaded VoIP application intended to be used on M82xxx EVM running in Master mode.
	- pots-to-pots local calls - different types of loopback, voice is not routed outside the board
	- pots-to-pots remote calls - voice routed outside the board, phones may be connected to the same or different boards

	All the VAPI function calls are performed in synchronous mode.

	<hr><H1>Linux instructions</H1><hr>
	\ref vxworks_instruction "(for vxworks instruction please click here)"\n
	
	<b>VAPI demo components</b>\n
	VAPI demo is composed by a set of binaries, drivers, scripts and configuration files:
	- csmencaps.ko: csmencaps Linux kernel driver to control the MSP
	- legerity.ko: legerity Linux kernel driver to control the Zarlink LExxx devices.
	- c300_init: application to perform basic initialization of the MSP and TSI devices.
	- vapi_demo: application handling the call.
	- demo.conf: configuration file of the vapi_demo application.
	- start_demo: a script to start the demo (execute all the necessary commands to load the drivers, init the board, start the application).
	- conf.dump: script to display the current vapi_demo configuration.
	- conf.set: script to change the vapi_demo configuration.
  
	<b>Building/installing this application</b>\n
	The VAPI demo application requires the csmencaps driver, VAPI library and include files to be installed.\n
	For more details on the VAPI library build and install, please refer to the VAPI Programming Guide (82xxx-SWG-007-xxx)\n
	The installed files are:
		- csmencaps.ko (in /lib/modules/2.6.22.19-4.03.0-c300evm/extra)
		- libgtlcommon.a, libgtlcsme.a and libvapi.a (in /usr/local/lib)
		- vapi.h, gtl.h, msp.h, ut.h (and others) (in /usr/local/include/vapi)

	It requires the legerity SLIC driver, library and include files to be installed too.\n
	These installed files are:
		- legerity.ko (in /lib/modules/2.6.22.19-4.03.0-c300evm/extra)
		- liblegerity.a (in /usr/local/lib)
		- legerity.h and legerity_lib.h (in /usr/local/include)


	The delivered Makefile has to be edited to configure the FSROOT to match the target file system installation.\n
	To build the application just enter make.\n

	Running make install installs:
		- the vapi_demo executable file to the FSROOT/usr/local/bin directory.
		- the start_demo, conf.dump, conf.set scripts to the FSROOT/usr/local/bin directory. 
		- the demo.conf configuration file to the FSROOT/usr/local/etc directory.
	
	<b>Post install instructions </b>\n
	Once the vapi_demp package, it is required to create a device file to get read/write operations working for the legerity devices.
	Enter the following commands to create it:
	<pre>
	comcerto:~# mknod /dev/legerity c 119 0
	comcerto:~# chmod 0666 /dev/legerity
	</pre>

	<b>Configuring the C300 EVM to start the file system from the NAND flash</b>\n
	Once a filesystem is flashed in the NAND flash (please see uboot User Guide for details), the uboot may need to be configured
	to run the file system from the NAND.\n
	Make sure the:
	- mtdboot variable is set to root=/dev/mtdblock7 rw rootfstype=jffs2 <pre>mtdboot=setenv bootargs root=/dev/mtdblock7 rw rootfstype=jffs2</pre>
	- setbootargs variable is set to mtdboot addtty addip addmtd <pre>setbootargs=run mtdboot addtty addip addmtd</pre>

	<b>Configuring the application</b>\n
	
	The configuration of the TDM parameters, MAC and IP addresses are modifiable in the /usr/local/etc/demo.conf.\n
	This parameters have to be modified prior the application is started.\n
	The user can also defines its own configuration file and use the -c option to instruct the application
	to use an alternative file instead of the default one (/usr/local/etc/demo.conf file).\n
	
	- Device parameters
	There are 3 main sections in the demo.conf file to configure devices parameters:
		- [DEVICE_TDM_CONFIG] for TDM parameters (should not be changed)
		- [DEVICE_NETWORK_CONFIG] to configure the MAC and IP addresses.
			- DEVICE_IP_ADDRESS =     <- this must be set to the same eth0 IP address shown by ifconfig
			- DEVICE_MAC_ADDRESS =    <- this must be set to the same eth0 MAC address shown by ifconfig
			- HOST_MAC_ADDRESS =      <- this must be set to the MAC address of an equipement which is IP forward capable (i.e PC)
		- [DEVICE_MNGT]	to configure the echo canceler type to use (Standard EC or Dual filter EC)
	
	- Connection parameters
	The section [CONNECTION_MNGT]) allows the user to set few connection level parameters such as:
		- NB_CODEC = 48    <- Codec index to use for Narrow Band operation 
		- WB_CODEC = 48    <- Codec index to use for Wide Band operation
		- PACKET_SIZE = 10     <- Packet interval 10 ms

	- Diagnostics parameters
	The section [DIAGNOSTICS_MNGT] allows the user to enable the TDM diagnostics.\n
	Few examples:
		- TDM_DIAG_MODE = NONE no diagnostics enabled 
		- TDM_DIAG_MODE = 0x0e00, 0x0e01 to enabled TDM RX/TX diags 
		- TDM_DIAG_MODE = 0x1011, 0x1012 to enabled SPU in/out diags 
		- TDM_DIAG_MODE = 0x0e00, 0x0e01, 0x1011, 0x1012 to enabled both TDM RX/TX & SPU in/out diags

	<b>Changing the configuration</b>\n
	The /usr/local/etc/demo.conf file (or an alternative file i.e my_cfg.conf) can be modified to change to parameter of the connection (i.e PACKET_SIZE) \n
	This requires to edit the configuration file with an editor such as vi. \n
	However it could be convenient to use the conf.dump and conf.set utilities to display the configuration and change it. \n
	- conf.dump displays the parameters of /usr/local/etc/demo.conf
	- conf.dump my_cfg.conf display the parameters of my_cfg.conf
	<pre>
	comcerto:~# conf.dump 
	TDM_BUS_MODE = 2
	TDM_BUS_TO_ENABLE = 2
	BUS_ID = 0x0
	TX_ORDER = 0x1
	RX_POLARITY = 0x0
	TX_POLARITY = 0x0
	RX_CLOCK_MODE = 0x0
	TX_CLOCK_MODE = 0x0
	RX_CLOCK_EDGE = 0x0 
	TX_CLOCK_EDGE = 0x1
	FRAME_EDGE = 0x0
	NUMBER_OF_TS = 128
	INVERTED_FRAME_SIGNAL = 0x0
	BIT_ORDER_RCV = 0x01
	NUM_TDM_CLK_CYCLES_DELAY_TX = 0x0
	NUM_TDM_CLK_CYCLES_DELAY_RX = 0x0
	DEVICE_IP_ADDRESS = 192.168.35.220
	DEVICE_MAC_ADDRESS = 00:AA:BB:CC:DD:EE
	HOST_MAC_ADDRESS = 00:d0:b7:0b:c5:6d
	LOOPBACK_MODE = 5
	NB_CODEC = 48
	WB_CODEC = 48
	PACKET_SIZE = 20
	EC_MODE = 1
	TDM_DIAG_MODE = NONE
	comcerto:~#
	</pre>

	- conf.set PACKET_SIZE 10 sets PACKET_SIZE = 10 in the /usr/local/etc/demo.conf file
	- conf.set TDM_DIAG_MODE 0x01e0 my_cfg.conf set TDM_DIAG_MODE = 0x01e0 in the my_cfg.conf file

	<pre>
	comcerto:~# conf.set PACKET_SIZE 10
	Changing PACKET_SIZE = 20 to PACKET_SIZE = 10 in /usr/local/etc/demo.conf
	comcerto:~#

	comcerto:~# conf.set TDM_DIAG_MODE 0x0e00,0x0e01,0x1011,0x1012 demo2.conf 
	Changing TDM_DIAG_MODE = 0x01e0 to TDM_DIAG_MODE = 0x0e00,0x0e01,0x1011,0x1012 in demo2.conf
	comcerto:~#
	</pre>
	In case of file corruption or erroneous deletion of the /usr/local/etc/demo.conf file, there is a backup file called demo.conf.bak in /usr/local/etc. 

	<b>Using the application</b>\n
 	The pots to pots application contains vapi_demo binary.\n
	It's located to the /usr/local/bin directory.\n
	The vapi_demo can take several command line options. Please run vapi_demo --help to list the available options:
	<pre>
	comcerto:~# vapi_demo --help
	Usage: vapi_demo [OPTION...]
	vapi_demo - program to make a pots to pots call on Comcerto EVM embeded boards

	-c, --config_file=CONFIGURATION_FILE
                             Name of the configuration file
	-d, --vapi_debug_level=VAPI_DEBUG_LEVEL
                             Vapi Debug Level
	-D, --app_debug_level=APP_DEBUG_LEVEL
                             App Debug Level
	-?, --help                 Give this help list
	    --usage                Give a short usage message
	-V, --version              Print program version

	</pre>

	<b>Starting the pots to pots application under Linux:</b>\n
	The steps to run the application are:
	- load the csmencaps driver:
			<pre>modprobe csmencaps</pre>
	- if the application runs on a M823xx device run the board initialization application.
			<pre>c300_init</pre>
	- load the legerity driver:
			<pre>modprobe legerity</pre>
	- start the pots application, The -dn and -Dn (1 to 4) options can be added to display more or less debug information.
			<pre>vapi_demo</pre>

	<b>Display of the process</b>\n
	<pre>

	comcerto:~# modprobe csmencaps
	NET: Registered protocol family 27

	comcerto:~# lsmod
	Module                  Size  Used by
	csmencaps              19240  0

	comcerto:~# c300_init
	Network Timming Generator successfully configured
	TSI successfully configured

	comcerto:~# modprobe legerity
	legerity spi1.6: detected SLIC type 5 on chip select 6
	legerity spi1.7: detected SLIC type 5 on chip select 7
	legerity: loaded version 2.01.0

	comcerto:~# lsmod
	Module                  Size  Used by
	legerity               97288  0
	csmencaps              19240  0

	comcerto:~# vapi_demo

	*************************************************************
	VAPI Library Release 2.xx.x, API Version x.x
	*************************************************************
	VAPI Initilized sucessfully
	VAPI Library Release 2.10.0-cvs, API Version 8.0
	Entered VAPI_SetDebugLevel
	...
	Signaling SLIC event handler thread started
	Endpoint state machine thread started
	</pre>


	<b>The phone number of the line 1 is 1, the phone number of line 2 is 2, etc ... </b>\n
	- When a phone goes offhook a dialtone is generated to the phone.\n
	- When a digit is dialed, the dialtone is stopped and replaced ringback tone.\n
	- The other phone rings and when it goes off hook a VoIP connection is done between the two phones.\n
	
	<b>Wide band mode operation</b>\n
	The wide band mode operation is only supported if :
	- the EVM board SLIC device are wide band mode capable.
	- the MSP firmware is wide band mode capable.\n
	To really take benefits of the wide band feature, the phones must be wide band mode capable too.\n
	In this case a user menu is available from the console to be able to switch between wide and narrow band mode.\n
	- By pressing 8/9 buttons on originated phone choose SLIC mode:\n
		8 - NB mode\n
		9 - WB mode\n
	While switching to particular SLIC mode the codec type is being changed as well:\n 
		NB_CODEC while NB SLIC mode\n
		WB_CODEC while WB SLIC mode\n
	The codes types are configurable in the demo.conf file.\n

	- Press whenever 8/9 buttons to switch between WB/NB modes.\n
	- By default the call between the 2 pots is done using the inter channel loopback.\n
	To change the loopback to TDM Hairpin enter: conf.set LOOPBACK_MODE 7.\n
	To change to no loopback enter: conf.set LOOPBACK_MODE 0 (requires IP forwarding enabled on an external machine).\n


	<b>Reseting the setup</b>\n
	In some cases it could be required to reset the MSP, restart it and restart the application.\n
	This requires to have a copy of the MSP firmware file on the file system (i.e m823app.elf).\n
	To perform the setup reset proceed as follow: 
	- CTRL-C to stop the application
	- modprobe -r legerity		<-- unload the legerity driver
	- modprobe -r csmencaps		<-- unload the csmencaps driver
	- msp-control -i m823app.elf	<-- reset the MSP, relaod the firmware and restart the MSP 
	- modprobe csmencaps		<-- load the csmencaps driver
	- c300_init			<-- basic init of the setup (NTG, TSI)
	- modprobe legerity		<-- load the legerity driver
	- vapi_demo -c my_config.conf	<-- restart the application with the "my_config.conf" configuration.

	<b>The installed start_demo script can be used to :</b>
	- start the demo application 
	<pre>comcerto:~# start_demo</pre> 
	- reset the MSP and restart the application 
	<pre>comcerto:~# start_demo m823_app_master.elf</pre> 

\anchor vxworks_instruction
	<hr><H1>VxWorks instructions</H1><hr>

	This section describe how to use the vapi_demo application from a single vxWorks kernel image already installed into the NOR flash of the Comcerto 300 EVM.\n

	The instructions to rebuild the CSP image and to reload new MSP and CSP images to the NOR flash are \ref building_image "here".

	<b>VAPI demo components</b>\n
	VAPI demo is embeded in a single vxWorks image containing:
	- comcerto VxWorks BSP
	- csmencaps: low level driver to control the MSP
	- legerity: low level driver to control the Zarlink LExxx devices.
	- c300_init: shell function to perform basic initialization of the MSP and TSI devices.
	- vapi_demo: shell function handling the call.


	<b>Booting the system</b>\n
	From the console enter:\n
	<pre> Comcerto-300 > run bootvxworksDKT</pre>

	Once the images are started the VxWorks prompt is displayed to the console as follow:\n
	<pre>
	## No elf image at address 0xb8660000
	## Downloading image at b8660000 ...
	code_offset=0x80
	code_base=100
	data_offset=178
	code_size=f8
	data_base=1f8
	data_size=0
	zeroinit_base=1f8
	prog_entry=100

	AIFHEADER:
	BL_DecompressCode=e1a00000
	BL_SelfRelocCode=e1a00000
	BL_DbgInitZeroInit=eb00000c
	EntryPointOffset=0
	ProgramExitInstr=ef000011
	ImageReadOnlySize=f8
	ImageReadWriteSize=0
	ImageDebugSize=1560
	ImageZeroInitSize=0
	ImageDebugType=15601
	ImageBase=100
	WorkSpace=0
	AddressMode=20
	DataBase=0
	FirstFatOffset=16d8
	Reserved2=1b2b68
	DebugInitInstr=e1a00000
	ZeroInitCode[0]=e04ec00f
	data section size 0
	Loading .text @ 0x01002000 (1202896 bytes)
	Loading .data @ 0x01127ad0 (158576 bytes)
	Clearing .bss @ 0x0114e640 (87840 bytes)
	Copying ARM1 startup code from 07e003e0, start address 01002000
	Attached TCP/IP interface to eth unit 0


	Attaching network interface lo0... done.
	Unable to add route to 192.168.0.0; errno = 0xffffffff.

	Adding 4989 symbols for standalone.


	 ]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]
	 ]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]
	 ]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]
	      ]]]]]]]]]]]  ]]]]     ]]]]]]]]]]       ]]              ]]]]         (R)
	 ]     ]]]]]]]]]  ]]]]]]     ]]]]]]]]       ]]               ]]]]
	 ]]     ]]]]]]]  ]]]]]]]]     ]]]]]] ]     ]]                ]]]]
	 ]]]     ]]]]] ]    ]]]  ]     ]]]] ]]]   ]]]]]]]]]  ]]]] ]] ]]]]  ]]   ]]]]]
	 ]]]]     ]]]  ]]    ]  ]]]     ]] ]]]]] ]]]]]]   ]] ]]]]]]] ]]]] ]]   ]]]]
	 ]]]]]     ]  ]]]]     ]]]]]      ]]]]]]]] ]]]]   ]] ]]]]    ]]]]]]]    ]]]]
	 ]]]]]]      ]]]]]     ]]]]]]    ]  ]]]]]  ]]]]   ]] ]]]]    ]]]]]]]]    ]]]]
	 ]]]]]]]    ]]]]]  ]    ]]]]]]  ]    ]]]   ]]]]   ]] ]]]]    ]]]] ]]]]    ]]]]
	 ]]]]]]]]  ]]]]]  ]]]    ]]]]]]]      ]     ]]]]]]]  ]]]]    ]]]]  ]]]] ]]]]]
	 ]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]
	 ]]]]]]]]]]]]]]]]]]]]]]]]]]]]]       Development System
	 ]]]]]]]]]]]]]]]]]]]]]]]]]]]]
	 ]]]]]]]]]]]]]]]]]]]]]]]]]]]       VxWorks version 5.5.1
	 ]]]]]]]]]]]]]]]]]]]]]]]]]]       KERNEL: WIND version 2.6
	 ]]]]]]]]]]]]]]]]]]]]]]]]]       Copyright Wind River Systems, Inc., 1984-2003

	                               CPU: Comcerto 300 (x570).  Processor #1.
	                              Memory Size: 0x7fef000.  BSP version 2.03/0.
	                             WDB Comm Type: WDB_COMM_END
	                            WDB: Ready.

	->
	</pre>
	
	<b>Configuring the application</b>\n
	
	The configuration of the MAC and IP addresses are modifiable using these commands from the vxworks shell:
	- set_device_ip("192.168.32.220"). This is the IP source address used by the MSP for the RTP traffic (default 192.168.32.220)\n
		It must match the one displayed by the vxWorks ifShow "eth0" command.\n
		
	- set_device_mac("00:AA:BB:CC:DD:EE"). This is the MAC source address used by the MSP for the RTP traffic (default 00:AA:BB:CC:DD:EE).\n
		It must match the one displayed by the vxWorks ifShow "eth0" command.\n

	- set_host_mac("00:11:D8:E7:A5:2D"). This is the MAC of the host intended to be the IP/RTP forwarder. (default 00:11:D8:E7:A5:2D)\n
		By default the application operates in loopback mode, so this IP value is not relevant.
		
	- set_diagnostic("0x0e00, 0x0e01"). 
		- NONE no diagnostics enabled (default)
		- 0x0e00, 0x0e01 to enabled TDM RX/TX diags 
		- 0x1011, 0x1012 to enabled SPU in/out diags 
		- 0x0e00, 0x0e01, 0x1011, 0x1012 to enabled both TDM RX/TX & SPU in/out diags

	Some other parameters are configurable from the VxWorks shell:
	<pre>
	EC_MODE=0 Sparse EC (default)
	EC_MODE=1 Dual Filder EC
	
	LOOPBACK_MODE 0 = no loopback (require IP forwarding enabled on an external machine)
	LOOPBACK_MODE = 5 Internal loopback eINTER_CHNL_POST_ENCAPS   (default)
	LOOPBACK_MODE = 7 TDM hairpin eTHC

	NB_CODEC  to use for Narrow Band operation
	NB_CODEC = 4: g711 uLaw (default) 
	NB_CODEC = 5: g711 ALaw
	NB_CODEC = 6: g723
	NB_CODEC = 7: g728 audio
	NB_CODEC = 8: g729 A
	NB_CODEC = 48: g722

	WB_CODEC  to use for Narrow Band operation
	WB_CODEC = 4: g711 uLaw
	WB_CODEC = 5: g711 ALaw
	WB_CODEC = 6: g723
	WB_CODEC = 7: g728 audio
	WB_CODEC = 8: g729 A
	WB_CODEC = 48: g722  (default)

	PACKET_SIZE = 20 <- Packet interval 20 ms
	</pre>

	
	<b>Using the application</b>\n
	The application consists of one  vapi_demo binary.

	<pre>
	-> c300_init
	Network Timing Generator successfully configured
	TSI successfully configured
	value = 0 = 0x0

	-> taskSpawn("vapi_demo", 100, 0, 64*1024, vapi_demo,0,0,0,0)
	value = 133256808 = 0x7f15668
	-> Using config: /usr/local/etc/demo.conf
	0x7f15668 (vapi_demo): /legerity0: detected SLIC type 5 on chip select 6
	SLIC Product code : 0xbf
	SLIC Revision code : 0x04
	VAPI Library Release 2.12.1--candidate1, API Version 8.0
	0x7f15668 (vapi_demo): ../vapi-2.12//app_examples/vapi_demo/demo_main.c: 1264: VAPI_OpenDevice: ok
	0x7f15668 (vapi_demo): ../vapi-2.12//app_examples/vapi_demo/demo_main.c: 1273: VAPI_InitDevice: ok
	0x7f15668 (vapi_demo): ../vapi-2.12//app_examples/vapi_demo/demo_main.c: 1276: TDM parameters initialisation: ok
	0x7f15668 (vapi_demo): ../vapi-2.12//app_examples/vapi_demo/demo_main.c: 1279: Network initialisation: ok
	No diag activated
	0x7f15668 (vapi_demo): ../vapi-2.12//app_examples/vapi_demo/demo_main.c: 1282: Diagnostics initialisation: ok
	Signaling SLIC event handler thrEndpoint state machine thread stead started
	</pre>


	<b>The phone number of the line 1 is 1, the phone number of line 2 is 2, etc ... </b>\n
	- When a phone goes offhook a dialtone is generated to the phone.\n
	- When a digit is dialed, the dialtone is stopped and replaced ringback tone.\n
	- The other phone rings and when it goes off hook a VoIP connection is done between the two phones.\n
	
	<b>Wide band mode operation</b>\n
	The wide band mode operation is only supported if :
	- the EVM board SLIC device are wide band mode capable.
	- the MSP firmware is wide band mode capable.\n
	To really take benefits of the wide band feature, the phones must be wide band mode capable too.\n
	In this case a user menu is available from the console to be able to switch between wide and narrow band mode.\n
	- By pressing 8/9 buttons on originated phone choose SLIC mode:\n
		8 - NB mode\n
		9 - WB mode\n
	While switching to particular SLIC mode the codec type is being changed as well:\n 
		NB_CODEC while NB SLIC mode\n
		WB_CODEC while WB SLIC mode\n
	The codecs types are configurable through the NB_CODEC or WB_CODEC variables (i.e NB_CODEC=8 for g.729a).\n

	- Press whenever 8/9 buttons to switch between WB/NB modes.\n
	- By default the call between the 2 pots is done using the inter channel loopback.\n
	To change the loopback to TDM Hairpin enter: LOOPBACK_MODE=7.\n
	To change to no loopback enter: LOOPBACK_MODE=0 (requires IP forwarding enabled on an external machine).\n

\anchor building_image 

	<b>Updating the C300 EVM to start the MSP & CSP images from the NOR flash</b>\n
	The 6Mbytes vxWorks image is bootable using the u-boot. \n
	However the size implies some specific u-boot variables.
	
	To define the CSP NOR address
	<pre>setenv cspnoraddr B8060000</pre>
	
	To define the MSP NOR address
	<pre>setenv mspnoraddr B8660000</pre>

	To create a script to upload the CSP image to the NOR Flash
	<pre>setenv updatecspNORDKT 'tftp 1000000 ${tftpdir}/${cspname}; erase 2:3-50; cp.b ${fileaddr} ${cspnoraddr} ${filesize}'</pre>

	To create a script to upload the MSP image to the NOR Flash
	<pre>setenv updatemspNORDKT 'tftp 1000000 ${tftpdir}/${mspname}; erase 2:51-82; cp.b ${fileaddr} ${mspnoraddr} ${filesize}'</pre>

	To create a script to start the Comcerto MSP & CSP from the images located in the NOR flash
	<pre>setenv bootvxworksDKT 'run setbootargs; loadmsp ${mspnoraddr} 400000; cp.b ${cspnoraddr} 2000000 600000; bootcomcerto 2000000'</pre>

	<b>Updating the MSP image from a tftp server.</b> \n
	The u-boot must have some other regular variable correctly set to proceed with a MSP & CSP image update (tftpdir, mspname, cspname, ipaddr. etc..). \n
	Please check the u-boot user guide for more details.

	<b>To update the MSP image enter:</b>
	<pre>
	Comcerto-300 > run updatemspNORDKT
	PHY 1000Mbit FD
	Using comcerto_gemac0 device
	TFTP from server 192.168.32.40; our IP address is 192.168.32.200
	Filename './m823app.elf'.
	Load address: 0x1000000
	Loading: #################################################################
	         #################################################################
	         #################################################################
	         #################################################################
	         #################################################################
	         #################################
	done
	Bytes transferred = 1829460 (1bea54 hex)
	Erase Flash Sectors 51-82 in Bank # 2
	................................ done
	Copy to Flash... done
	Comcerto-300 >
	</pre>
	
	<b>To update the CSP image enter:</b>
	<pre>
	Comcerto-300 >  run updatecspNORDKT
	PHY 1000Mbit FD
	Using comcerto_gemac0 device
	TFTP from server 192.168.32.40; our IP address is 192.168.32.200
	Filename './/vxWorks'.
	Load address: 0x1000000
	Loading: #################################################################
	         #################################################################
	         #################################################################
	         #################################################################
	         #################################################################
	         #################################################################
	         #################################################################
	         #################################################################
	         #################################################################
	         #################################################################
	         #################################################################
	         #################################################################
	         #################################################################
	         #################################################################
	         #################################################################
	         #################################################################
	         #################################################################
	         ###########
	done
	Bytes transferred = 5711861 (5727f5 hex)
	Erase Flash Sectors 3-50 in Bank # 2
	................................................ done
	Copy to Flash... done
	Comcerto-300 >
	</pre>
*/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/* call control interface */
#include <legerity_lib.h>

#ifndef _VXWORKS_
#include <vapi/msp.h>
#include <vapi/vapi.h>
#include <vapi/gtl.h>
#include <sys/time.h>
#include <getopt.h>
#else
#include <taskLib.h>
#include <sys/ioctl.h>
#include <msp.h>
#include <vapi.h>
#include <gtl.h>
#endif

#include "demo.h"
#include "readcfg.h"

#define check_status(title, error_label)					\
	PDEBUG(DEBUG_ERROR, title ": %s", result == SUCCESS?"ok":"failed");	\
	if (result != SUCCESS)							\
		goto error_label

static SThread *tid_signalling;
static SThread *tid_state_machine;
char config_file[FILENAME_MAX];


/* =================================*/
/* ========= GLOBAL VARIABLES ======*/
/* =================================*/

/* structure with configuration parameters, structure type is defined in gtl.h */
SCSMEUsrData gtl_device_configuration;
int vapi_debug_level;
int app_debug_level;

struct _ENDPOINT_DESC endpoints[MAX_ENDPOINTS];

U8 device_ip_address[4];
U8 device_mac_addr[6];
U8 host_mac_addr[6];
U8 channel_mode;

/* set the default mac address to 0 */
U8 def_mac_addr[6];
int legerity_device_file;
int loopback_mode;

int is_wb_slic;

#define DEMO_VERSION "1.1.3"

#ifndef _VXWORKS_
static void show_help(void)
{
	printf(
		"vapi demo - program to make a pots to pots call on Comcerto EVM embeded boards\n"
		"Usage: %s [OPTION]...\n"
		"  -c, --config_file=CONFIGURATION_FILE\n"
		"			Name of the configuration file\n\n"
		"  -d, --vapi_debug_level=VAPI_DEBUG_LEVEL\n"
		"			Vapi Debug Level\n\n"
		"  -D, --app_debug_level=APP_DEBUG_LEVEL\n"
		"			App Debug Level\n\n"
		"      --help		Show this help and exit\n"
		"      --version	Show version and exit\n",
		"vapi_demo"
	);
}

static void show_version(void)
{
	printf("vapi demo test application, %s\n", DEMO_VERSION);
}

static struct option const long_options[] =
{
	{ "config_file",	required_argument, NULL, 'c' },
	{ "vapi_debug_level",	required_argument, NULL, 'd' },
	{ "app_debug_level",	required_argument, NULL, 'D' },

	{ "help",		no_argument, NULL, CHAR_MIN - 1 },
	{ "version",		no_argument, NULL, CHAR_MIN - 2 },

	{ NULL, 0, NULL, 0},
};

#else

const char *argp_program_version = DEMO_VERSION;
const char *argp_program_bug_address = "alexander.perekhodko@mindspeed.com";
const char doc[] = "vapi_demo - program to make a pots to pots call on Comcerto EVM embeded boards";

const struct argp_option options[] = {
	{"config_file", 'c', "CONFIGURATION_FILE", 0, "Name of the configuration file"},
	{"vapi_debug_level", 'd', "VAPI_DEBUG_LEVEL", 0, "Vapi Debug Level"},
	{"app_debug_level", 'D', "APP_DEBUG_LEVEL", 0, "App Debug Level"},
	{0}
};
#endif

/*=================================================================================*/
/*! \brief
*	This function is used to retrieve the endpoint number which has its phone_id matching the passed number.
*/
int lookup_endpoint_from_number (unsigned long number)
{
	int i;
	int rc = -1;

	for (i = 0; i < MAX_ENDPOINTS; i ++)
	{
		if (endpoints[i].phone_id == number)
		{
			rc = i;
			printf("endpoint %d phone id %d\n", i, endpoints[i].phone_id);
			break;
		}
	}
	printf("lookup_endpoint_from_number: found %d\n", rc);
	return rc;
}

/*=================================================================================*/
/*! \brief
*	This function handles the response to VAPI commands sent in ASYNC mode. \n
*	It sets event for the endpoint accordingly the endpoint state. The state is retrieved through the request ID.\n
*/
void comcerto_response_handler(U32 connection_id,
			U8 command_level,
			S32 result,
			void *response_data,
			U32 data_length,
			U32 request_id)
{

	if ( result != SUCCESS)
	{
		PDEBUG(DEBUG_ERROR,"Error on response received on endpoint id %d, request id = 0x%04x", connection_id, request_id);
		return;
	}
	else
	{
		PDEBUG(DEBUG_INIT,"Response received on endpoint id %d, request id = 0x%04x",connection_id, request_id);
	}

}

/*=================================================================================*/
/*! \brief
*	
*	This function handles the indication comminf from VAPI. \n
*	It is registered to VAPI using the VAPI_RegisterEventCallback() API.\n
*	It handles DTMF tones detection, and some Comcerto device indications .\n
*	It sets event V21 detected to operates a T38 switch in case of V21 flag detection.\n
*/
void comcerto_indication_handler(EEventCode eEventCode, void *pvData)
{
	SToneDetectEventParams *tone_detected;
	SRemoteDetectEventParams *remote_tone_detected;
	SSsrcChangeEventParams *ssrc_changed;
	SUndefinedEventParams *this_event;

	/* FIXME: The parser below need to be improved */
	switch (eEventCode)
	{
		case eVAPI_TONE_DETECT_EVENT:
			tone_detected = (SToneDetectEventParams *)pvData;

			PDEBUG(DEBUG_INIT,"Tone %d detected on Connection ID %d",
					tone_detected->usDetectedTone, tone_detected->ConId);

			if(tone_detected->usDetectedTone != 0xff)
				printf("Tone %d detected\n", tone_detected->usDetectedTone);

			switch (tone_detected->usDetectedTone)
			{
			case 0x22:
				printf("======== FAX V.21 detected =========\n");
				set_endpoint_event(ENDPOINT_EVENT_V21FLAG_DETECTED, tone_detected->ConId);
				break;

			case 0:
			case 1:
			case 2:
			case 3:
			case 4:
			case 5:
			case 6:
			case 7:
			case 8:
			case 9:
			case 0x0A:
			case 0x0B:
				endpoints[tone_detected->ConId].dtmf_service = tone_detected->usDetectedTone;
				set_endpoint_event(ENDPOINT_EVENT_DTMF_SERVICE, tone_detected->ConId);
				break;

			case 0xFF: /* end of tone*/
/*				if (endpoints[tone_detected->ConId].event == ENDPOINT_EVENT_WAIT_UNPRESS)
					set_endpoint_event(ENDPOINT_EVENT_UNPRESSED, tone_detected->ConId);
*/				break;

			default:
				/* else send busy tone */
					set_endpoint_event(ENDPOINT_EVENT_BUSY, tone_detected->ConId);
                        	break;
			}

			break;

		case eVAPI_SSRC_CHANGE_EVENT:
			ssrc_changed = (SSsrcChangeEventParams *)pvData;
			PDEBUG(DEBUG_INIT,"SSRC changed 0x%08x on Connection ID %d",
					ssrc_changed->uNewSSRC, ssrc_changed->ConId);

			break;

		case eVAPI_SPI_EVENT:
		case eVAPI_TONE_GEN_CMPLT_EVENT:
		case eVAPI_PT_CHANGE_EVENT:
		case eVAPI_SSRC_VIOLATION_EVENT:
		case eVAPI_NTE_TRANSMIT_COMPLETE_EVENT:
		case eVAPI_NTE_RECVD_EVENT:
		case eVAPI_CALLER_ID_CMPLT_EVENT:
		case eVAPI_G711_CONCURRENT_DECODER_EVENT:
		case eVAPI_PASSTHRU_CONCURRENT_DECODER_EVENT:
		case eVAPI_CALLER_ID_DETECTED_EVENT:
		case eVAPI_FAX_SWITCH_CMPLT_EVENT:
		case eVAPI_ALERT_IND:

			break;
			
		case eVAPI_REMOTE_DETECT_EVENT:
			remote_tone_detected = (SRemoteDetectEventParams *)pvData;
			
			PDEBUG(DEBUG_INIT,"Remote tone %d detected on Connection ID %d", (remote_tone_detected->ucDetectedEvent - 1), remote_tone_detected->ConId);

			switch (remote_tone_detected->ucDetectedEvent - 1)
			{			
			case 0x0A:
				/* future use start/stop recordiing*/
				break;

			case 0x0B:
				/* future use  start/stop playing*/
				break;				

			case 0xFF: /* end of tone no action*/
				break;

			default:
				/* else send busy tone */
					set_endpoint_event(ENDPOINT_EVENT_BUSY, remote_tone_detected->ConId);
                        	break;
			}

			break;
			
		case eVAPI_UNDEFINED_EVENT:
			PDEBUG(DEBUG_INIT,"event %d received",eEventCode);
			this_event = (SUndefinedEventParams *) pvData;

			if (this_event->ucCmdType == CMD_TYPE_INDICATION)
				PDEBUG(DEBUG_INIT,"indication 0x%04x received",this_event->usFnCode);
		
			break;

		default:
			break;
	}

	return;
}


/*=================================================================================*/
/*! \brief
*	
*	This function handle the states of the endpoints.
*	It is started from a thread.
*	All endpoints are initialized in to the IDLE state, then the states change accordingly the events \n
*	comcerto_indication_handler, or the signaling_event_handler threads \n.
*	The calls to the VAPI_xxx API are performed in SYNC mode
*/
void *state_machine_handler(void *none)
{
	int i;
	struct legerity_io_ring_desc ring;
	struct legerity_io_disconnect_desc disconnect;
	int peer_endpoint;
	unsigned long number;
	VSTATUS status = SUCCESS;
	U8 pots[] = {EC_CH1, EC_CH2, EC_CH3, EC_CH4};
	int index;

	/* We should have here a signal handler to interrupt the while loop
	For now loop forever, CTRL-C to be used to stop. */
	printf("Endpoint state machine thread started\n");
	while (1)
	{
		/* Parse all the endpoints in the global endpoints array*/
		for (i = 0; i < MAX_ENDPOINTS; i++)
		{
			/* If the state is IDLE we should receive only OFFHOOK or RING events*/
			switch (endpoints[i].state)
			{
				case ENDPOINT_STATE_IDLE:
					switch (endpoints[i].event)
					{
						/* If the OFFHOOK event was posted*/
						case ENDPOINT_EVENT_OFF_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d off hook", endpoints[i].index);

							/* so Create the connection and generate a dial tone */
							status = create_endpoint(config_file, endpoints[i].index, ENDPOINT_STATE_SYNC);
							if (status != SUCCESS)
								set_endpoint_error(status, endpoints[i].index);
							else
							{
								status = dialtone_generation(endpoints[i].index, ENDPOINT_STATE_SYNC);
								if (status != SUCCESS)
									set_endpoint_error(status, endpoints[i].index);
							}
							set_endpoint_state(ENDPOINT_STATE_WAIT_FOR_DIALING, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);

							break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Unexpected event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);

					}
					break;

				/* The phone is RINGING. If the phone goes OFFHOOK, an event OFF_HOOK is posted by the signalling thread.
				 * Run timer. If time is up set ENDPOINT_EVENT_REMOTE_TIME_UP event */
				case ENDPOINT_STATE_RINGING:
					switch (endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:

							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							set_endpoint_state(ENDPOINT_STATE_IDLE, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							onhook_handler(endpoints[i].index, ENDPOINT_STATE_SYNC);
							break;

						case ENDPOINT_EVENT_OFF_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d off hook", endpoints[i].index);
							/* This endpoint is now offhook we want to create a VoIP channel for this endpoint 
							 * so we switch to a state to wait for the Comcerto response to create connection
							 * Create the connection and generate a dial tone */
							status = create_endpoint(config_file, endpoints[i].index, ENDPOINT_STATE_SYNC);
							if (status != SUCCESS)
								set_endpoint_error(status, endpoints[i].index);
							/* proceed with the IP setting*/
							else
							{
								peer_endpoint = endpoints[i].peer_index;

								if (peer_endpoint < 0)
								{
									set_endpoint_error(status, endpoints[i].index);
									PDEBUG(DEBUG_ERROR, "Endpoint %d invalid peer endpoint = %d\n", endpoints[i].index, peer_endpoint);
								}
								else
								{
									status = set_ip_udp_parameters_sync(endpoints[i].index,
										endpoints[i].ip_src_addr,
										endpoints[peer_endpoint].ip_src_addr,
										endpoints[i].rtp_port,
										endpoints[peer_endpoint].rtp_port);

									if (status != SUCCESS)
									{
										set_endpoint_error(status, endpoints[i].index);
										break;
									}

									status = start_voip_connection(endpoints[i].index, ENDPOINT_STATE_SYNC);

									if (status != SUCCESS)
									{
										set_endpoint_error(status, endpoints[i].index);
										break;
									}

									/* indicate to the peer endpoint that it can start the RTP traffic */
									set_endpoint_event(ENDPOINT_EVENT_PEER_OFFHOOK, peer_endpoint);
									/* this endpoint is now ready */
									set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
									set_endpoint_state(ENDPOINT_STATE_READY, endpoints[i].index);
								}
							}
							break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Unexpected event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					}
					break;


				case ENDPOINT_STATE_WAIT_FOR_DIALING:
					switch (endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							set_endpoint_state(ENDPOINT_STATE_IDLE, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							onhook_handler(endpoints[i].index, ENDPOINT_STATE_SYNC);

							break;

						case ENDPOINT_EVENT_DTMF_SERVICE:
							/* store the detected digit into the endpoint structure*/
							index = endpoints[i].phonenumber_index;
		
							endpoints[i].phonenumber[index] = endpoints[i].dtmf_service + 0x30; /* store the ASCII value of the digit*/
							endpoints[i].phonenumber_index++;
							index++;
	
							if (index == (MAX_PHONE_SIZE -1))
								set_endpoint_event(ENDPOINT_EVENT_DIALED_CALL_REMOTE, endpoints[i].index);
							break;

						/* While playing dial tone, the comcerto_indication_handler() has detected the number to dial*/
						case ENDPOINT_EVENT_DIALED_CALL_REMOTE:
							/*So stop the dial tone and dial out the peer endpoint*/
							stop_tone_generation(endpoints[i].index, ENDPOINT_STATE_SYNC);

							/* figure out which endpoint has the dialed nunber */
							number = strtoul(endpoints[i].phonenumber, NULL, 0);

							memset(endpoints[i].phonenumber, 0, MAX_PHONE_SIZE);
							endpoints[i].phonenumber_index = 0;							

							peer_endpoint = lookup_endpoint_from_number(number);

							/* if no endpoint found play busy tone*/
							if (peer_endpoint < 0)
							{
								set_endpoint_event(ENDPOINT_EVENT_BUSY, endpoints[i].index);
								printf("Peer endpoint %ld number not found\n", number);
							}

							else if (((endpoints[peer_endpoint].state != ENDPOINT_STATE_IDLE) &&
								(endpoints[peer_endpoint].state != ENDPOINT_STATE_READY)) || 
								((endpoints[peer_endpoint].state == ENDPOINT_STATE_READY) && 
								(endpoints[peer_endpoint].call_waiting_index == -1)))

							{
								set_endpoint_event(ENDPOINT_EVENT_BUSY, endpoints[i].index);
								printf("Peer endpoint %d busy\n", peer_endpoint);
							}
							else
							{
								printf("POTS (%d) dialing to peer POTS (%d) \n", endpoints[i].index, peer_endpoint);

								endpoints[i].peer_index = peer_endpoint;

								/* if the dialed phone is on a voice call
								send it a tone to indicate aonther call is comming*/
								if (endpoints[peer_endpoint].state == ENDPOINT_STATE_READY)
								{
									status = specialtone_generation(endpoints[i].peer_index, ENDPOINT_STATE_SYNC);
									/*save index of the calling pots for further use*/
									endpoints[peer_endpoint].call_waiting_index = endpoints[i].index;
									/*use regular ringback on dialing pots*/
									status = ringbacktone_generation(endpoints[i].index, ENDPOINT_STATE_SYNC);
									set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
									set_endpoint_state(ENDPOINT_STATE_CALL_WAITING, endpoints[i].index);
								}
								else
								{
									endpoints[peer_endpoint].peer_index = i;
									ring.line = endpoints[i].peer_index;
									legerity_ring(legerity_device_file, &ring);
									status = ringbacktone_generation(endpoints[i].index, ENDPOINT_STATE_SYNC);

									if (status != SUCCESS)
										set_endpoint_error(status, endpoints[i].index);
									else
									{
										set_endpoint_state(ENDPOINT_STATE_REMOTE_CALLED, endpoints[i].index);
										set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
										/* change the peer endpoint state to RINGING*/
										set_endpoint_state(ENDPOINT_STATE_RINGING, endpoints[i].peer_index);
										set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].peer_index);
									}
								}
							}
							break;

						case ENDPOINT_EVENT_BUSY:
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							busytone_generation(endpoints[i].index, ENDPOINT_STATE_SYNC);

							break;
 
						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Unexpected event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					}
					break;

				/* This endpoint is OFFHOOK, the VoIP channel is created and the remote endpoint is ringing
				We keep this state until the destination endpoint goes offhook or this endpoint goes onhook 
				If time up posted on remote side move to ato answer state*/
				case ENDPOINT_STATE_REMOTE_CALLED:
					switch (endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							set_endpoint_state(ENDPOINT_STATE_IDLE, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							set_endpoint_state(ENDPOINT_STATE_IDLE, endpoints[i].peer_index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].peer_index);

							/* We stop the call on the remote side and set back the endpoint in idlle state*/
							disconnect.line = endpoints[i].peer_index;
							legerity_disconnect(legerity_device_file, &disconnect);

							/* this endpoint went onhook while the remote is ringing*/
							onhook_handler(endpoints[i].index, ENDPOINT_STATE_SYNC);
							break;

						case ENDPOINT_EVENT_PEER_OFFHOOK:
							/*So stop the dial tone and dial out the peer endpoint*/
							status = stop_tone_generation(endpoints[i].index, ENDPOINT_STATE_SYNC);
							if (status != SUCCESS)
							{
								set_endpoint_error(status, endpoints[i].index);
								break;
							}

							peer_endpoint = endpoints[i].peer_index;
							status = set_ip_udp_parameters_sync(endpoints[i].index,
									endpoints[i].ip_src_addr,
									endpoints[peer_endpoint].ip_src_addr,
									endpoints[i].rtp_port,
									endpoints[peer_endpoint].rtp_port);

							if (status != SUCCESS)
							{
								set_endpoint_error(status, endpoints[i].index);
								break;
							}

							status = start_voip_connection(endpoints[i].index, ENDPOINT_STATE_SYNC);

							if (status != SUCCESS)
							{
								set_endpoint_error(status, endpoints[i].index);
								break;
							}

							printf ("Demo tool works in %s mode\n", channel_mode == eNarrowBand?"NB":"WB");
							set_endpoint_state(ENDPOINT_STATE_READY, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_READY, endpoints[i].index);
							break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Unexpected event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					}
					break;

				/* This endpoint is connected.*/
				case ENDPOINT_STATE_READY:
					switch (endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							set_endpoint_state(ENDPOINT_STATE_IDLE, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							peer_endpoint = endpoints[i].peer_index;
							onhook_handler(endpoints[i].index, ENDPOINT_STATE_SYNC);

							/*destory peer endpoint, then genarate busytone*/
							if(peer_endpoint != -1)
							{
								set_endpoint_event(ENDPOINT_EVENT_BUSY, peer_endpoint);
								onhook_handler(peer_endpoint, ENDPOINT_STATE_SYNC);
							}
						break;

						case ENDPOINT_EVENT_V21FLAG_DETECTED:
							PDEBUG(DEBUG_INIT, "Endpoint %d: event V21 flags detected (%d) in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							status = switch_voip_connection_to_t38(endpoints[i].index, ENDPOINT_STATE_SYNC);
							status = switch_voip_connection_to_t38(endpoints[i].peer_index, ENDPOINT_STATE_SYNC);
							break;

						case ENDPOINT_EVENT_READY:
							/*make sure we have 2 endpoints */
							/*Check if we want to use loopback or not*/
							if (endpoints[i].peer_index != -1)
							{
								/*Get the loopback mode set in the config file*/
								loopback_mode = get_loopback_mode(config_file);

								/* make sure the option from the config file is OK*/
								if ((loopback_mode == eINTER_CHNL_POST_ENCAPS) || (loopback_mode == eTHC))
									status = VAPI_Loopback(endpoints[i].index, endpoints[i].peer_index, loopback_mode, NULL);
								{
									if(status == SUCCESS)
									{
										if((loopback_mode == eINTER_CHNL_POST_ENCAPS))
											printf("Endpoint %d & %d in inter channel loopback mode\n", endpoints[i].index, endpoints[i].peer_index);
										if((loopback_mode == eTHC))
											printf("Endpoint %d & %d in TDM Hairpin loopback mode\n", endpoints[i].index, endpoints[i].peer_index);
									}
								}
							}

							if (is_wb_slic == True)
								printf ("To choose SLIC mode press button:\n8 - NB mode\n9 - WB mode\n");

							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
						break;

						case ENDPOINT_EVENT_DTMF_SERVICE:
							handle_dtmf_service(i);
							break;

						case ENDPOINT_EVENT_BUSY:
							status = create_endpoint(config_file, endpoints[i].index, ENDPOINT_STATE_SYNC);

							if (status != SUCCESS)
								PDEBUG(DEBUG_ERROR, "Endpoint %d: error %d", endpoints[i].index, status);
							else
								busytone_generation(endpoints[i].index, ENDPOINT_STATE_SYNC);

							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);

							break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: unexpected event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					}
				break;

				/*Case when switch from NB to WB or WB to NB mode*/
				case ENDPOINT_STATE_SWITCH_MODE:
					switch (endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							set_endpoint_state(ENDPOINT_STATE_IDLE, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							onhook_handler(endpoints[i].index, ENDPOINT_STATE_SYNC);

						break;

						case ENDPOINT_EVENT_SWITCH_TO_NB:

							status = slic_set_narrow_band(legerity_device_file,
									pots[endpoints[i].index],
									pots[endpoints[i].peer_index]);

							if (status >= 0)
							{
								printf ("SUCCESS to switch SLIC to NBAND mode\n");
								set_endpoint_event(ENDPOINT_EVENT_REINITIALIZE_ENDPS, endpoints[i].index);
							}
							else
								set_endpoint_error(status, endpoints[i].index);

							break;

						case ENDPOINT_EVENT_SWITCH_TO_WB:

							if (is_wb_slic == False)
							{
								printf ("Current SLIC doesn't support switching WB/NB mode\n");
								set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
									break;
							}

							status = slic_set_wide_band(legerity_device_file,
								pots[endpoints[i].index],
								pots[endpoints[i].peer_index]);

							if (status >= 0)
							{
								printf ("SUCCESS to switch SLIC to WBAND mode\n");
								set_endpoint_event(ENDPOINT_EVENT_REINITIALIZE_ENDPS, endpoints[i].index);
							}
							else
								set_endpoint_error(status, endpoints[i].index);

							break;

						case ENDPOINT_EVENT_REINITIALIZE_ENDPS:

							peer_endpoint = endpoints[i].peer_index;
							destroy_endpoint(endpoints[i].index, ENDPOINT_STATE_SYNC);

							/* Create the connection */
							status = create_endpoint(config_file, endpoints[i].index, ENDPOINT_STATE_SYNC);

							if (status != SUCCESS)
							{
								set_endpoint_error(status, endpoints[i].index);
								break;
							}

							status = set_ip_udp_parameters_sync(endpoints[i].index,
									endpoints[i].ip_src_addr,
									endpoints[peer_endpoint].ip_src_addr,
									endpoints[i].rtp_port,
									endpoints[peer_endpoint].rtp_port);

							if (status != SUCCESS)
							{
								set_endpoint_error(status, endpoints[i].index);
								break;
							}

							status = start_voip_connection(endpoints[i].index, ENDPOINT_STATE_SYNC);

							if (status != SUCCESS)
							{
								set_endpoint_error(status, endpoints[i].index);
								break;
							}

							destroy_endpoint(peer_endpoint, ENDPOINT_STATE_SYNC);

							/* Create the peer connection */
							status = create_endpoint(config_file, peer_endpoint, ENDPOINT_STATE_SYNC);
							if (status != SUCCESS)
							{
								set_endpoint_error(status, peer_endpoint);
								break;
							}

							status = set_ip_udp_parameters_sync(endpoints[i].index,
									endpoints[i].ip_src_addr,
									endpoints[peer_endpoint].ip_src_addr,
									endpoints[i].rtp_port,
									endpoints[peer_endpoint].rtp_port);

							if (status != SUCCESS)
							{
								set_endpoint_error(status, peer_endpoint);
								break;
							}

							status = start_voip_connection(peer_endpoint, ENDPOINT_STATE_SYNC);

							if (status != SUCCESS)
							{
								set_endpoint_error(status, peer_endpoint);
								break;
							}

							endpoints[i].peer_index = peer_endpoint;
							endpoints[peer_endpoint].peer_index = endpoints[i].index;

							printf ("Demo tool works in %s mode\n", channel_mode == eNarrowBand?"NB":"WB");

							set_endpoint_state(ENDPOINT_STATE_READY, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_READY, endpoints[i].index);

							set_endpoint_state(ENDPOINT_STATE_READY, peer_endpoint);
							set_endpoint_event(ENDPOINT_EVENT_NONE, peer_endpoint);

							break;

						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Unexpected event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					}
				break;

				/*Endpoint waiting for call to be accepted*/
				case ENDPOINT_STATE_CALL_WAITING:
					switch (endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							/*Stop waiting tone on peer endpoint */
							stop_tone_generation(endpoints[i].peer_index, ENDPOINT_STATE_SYNC);
							/*put back the peer endpoint in ready state*/
							set_endpoint_state(ENDPOINT_STATE_READY, endpoints[i].peer_index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].peer_index);

							set_endpoint_state(ENDPOINT_STATE_IDLE, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							onhook_handler(endpoints[i].index, ENDPOINT_STATE_SYNC);
							break;

						/* Waiting for call accepted by remote*/
						case ENDPOINT_EVENT_CALL_ACCEPTED:
							/* Stop ring back */
							stop_tone_generation(endpoints[i].index, ENDPOINT_STATE_SYNC);

							peer_endpoint = endpoints[i].peer_index;
							if (peer_endpoint < 0)
							{
								set_endpoint_error(status, endpoints[i].index);
								PDEBUG(DEBUG_ERROR, "Endpoint %d invalid peer endpoint = %d\n", endpoints[i].index, peer_endpoint);
							}
							else
							{
								/* Enable connection with IP/UDP */
								status = set_ip_udp_parameters_sync(endpoints[i].index,
										endpoints[i].ip_src_addr,
										endpoints[peer_endpoint].ip_src_addr,
										endpoints[i].rtp_port,
										endpoints[peer_endpoint].rtp_port);

								if (status != SUCCESS)
								{
									set_endpoint_error(status, endpoints[i].index);
									break;
								}

								status = start_voip_connection(endpoints[i].index, ENDPOINT_STATE_SYNC);

								if (status != SUCCESS)
								{
									set_endpoint_error(status, endpoints[i].index);
									break;
								}

								/* this endpoint is now ready */
								set_endpoint_state(ENDPOINT_STATE_READY, endpoints[i].index);
								set_endpoint_event(ENDPOINT_EVENT_READY, endpoints[i].index);
							}

							break;
	
						case ENDPOINT_EVENT_NONE:
							break;

						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Unexpected event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					}
				break;

				case ENDPOINT_STATE_NONE:
					switch (endpoints[i].event)
					{
						case ENDPOINT_EVENT_ON_HOOK:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
							set_endpoint_state(ENDPOINT_STATE_IDLE, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							onhook_handler(endpoints[i].index, ENDPOINT_STATE_SYNC);
							break;

						case ENDPOINT_EVENT_NONE:
							break;
						default:
							PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Unexpected event %d in state %d", endpoints[i].index, endpoints[i].event, endpoints[i].state);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					}
				break;
			}
		}
#ifndef _VXWORKS_
		usleep(100);
#else
		taskDelay(1);
#endif
	}
}

/*=================================================================================*/
/*! \brief
*	
	This function handles the events reported by the signaling thread (SLIC events) such as offhook, onhokk, ring.
	The timeslot reported with the event is used as the endpoint index in the global endpoints array
*/
void *signaling_event_handler(void * none)
{
	struct legerity_io_event_desc event;
	int rc;

	/* we should have here a state machine waiting for events */
	printf ("Signaling SLIC event handler thread started\n");
	while (1)
	{
		rc = legerity_read_event(legerity_device_file, &event);

		if (rc != 0)
		{
			goto  sleep;
		}

		switch (event.type)
		{
			case 0 /*EVENT_ONHOOK*/:
				PDEBUG(DEBUG_SLIC_FUNC, "SLIC event = ONHOOK device = %d, line = %d", event.device, event.line);
				/* post the OFFHOOK event for this endpoint */
				set_endpoint_event(ENDPOINT_EVENT_ON_HOOK, event.device*2 + event.line);
				break;

			case 1 /*EVENT_OFFHOOK*/:
				PDEBUG(DEBUG_SLIC_FUNC, "SLIC event = OFFHOOK device = %d, line = %d", event.device, event.line);
				/* post the OFFHOOK event for this endpoint */
				set_endpoint_event(ENDPOINT_EVENT_OFF_HOOK, event.device*2 + event.line);
				break;

			default:
				PDEBUG(DEBUG_ERROR, "Unexpected SLIC event = %d", event.type);
				break;

		}

sleep:
#ifndef _VXWORKS_
		usleep(100);
#else
		taskDelay(1);
#endif
	}
}

#ifndef _VXWORKS_
static int
get_int(const char *s, int *value)
{
	int i;
	char *endptr = NULL;

	if (!s || !*s)
		goto err;

	i = strtol(s, &endptr, 0);
	if (!i && s == endptr)
		goto err;

	if (value)
		*value = i;

	return 0;

err:
	return -1;
}

/*=================================================================================*/
/*!	\brief command line option parser
 *	\section Description
 *	This function is the command line option parser.\n
 */	
/*=================================================================================*/
static int get_opt(int argc, char *argv[])
{

	int c;
	int status = SUCCESS;

	while ((c = getopt_long(argc, argv, "c:d:D:", long_options, NULL)) != - 1) {
		switch (c) {
		case 'c':
			strncpy(config_file, optarg, sizeof(config_file));
			break;

		case 'd':
			get_int(optarg, (int *) &vapi_debug_level);
			break;

		case 'D':
			get_int(optarg, (int *) &app_debug_level);
			break;

		case CHAR_MIN - 1:
			show_help();
			exit(0);
   			break;

		case CHAR_MIN - 2:
    			show_version();
			exit(0);
    			break;
			
		default:
			status = -1;
		}

	}

	return status;
}

#else
/*=================================================================================*/
/*! \brief
*	
*	This function parses the command line options to set the VAPI and application debug levels
*/
static error_t parser (int key, char *arg, struct argp_state *state)
{

	switch (key)
	{
	case 'c':
		strncpy(config_file, arg, sizeof(config_file));
		break;

	case 'd':

		vapi_debug_level = strtoul(arg, NULL, 0);
		break;

	case 'D':

		app_debug_level = strtoul(arg, NULL, 0);
		break;

	default:
		return ARGP_ERR_UNKNOWN;
	}
	return 0;
}

static struct argp argp = { options, parser, 0, doc, };
#endif

/***************************************************************************
 * main
 ***************************************************************************/
/*! \brief
*	
	This is the main function of the application.\n
	It does:\n
	 - initialise an endpoints array
	 - then parse the command line arguments (VAPI debug level)
	 - then it:
		- Initialise the setup configuration using parameters from the configration file (Device type, Control interface, Slave/Master mode , etc)
		- Print the VAPI verson (VAPI_GetVersion)
		- Initialise VAPI (VAPI_Init)
		- Open the Device (VAPI_OpenDevice)
		- Initialise the TDM buses using default parameters (VAPI_SetTDMParams)
		- Using VAPI_InitDevice set the SPU feature and enable the Multicommand mode
		- Set the Network Parameters (VAPI_SetEthMac VAPI_SetDeviceIPAddr)
		- Set the diagnostics if enabled.
	 - then it starts the signaling and state machine threads.
*/
#ifndef _VXWORKS_
int main(int argc, char *argv[])
#else
int vapi_demo(char *args)
#endif
{
	int result;
	int i;
	U16 slic_version;

	channel_mode = eNarrowBand;
	vapi_debug_level = NO_DEBUG;
	app_debug_level = NO_DEBUG;

	/* try to set default config file */
	strcpy(config_file, "demo.conf");	/* start from current dir */
	result = open(config_file, O_RDONLY, 0);
	if (result < 0)
	{
		strcpy(config_file, "/etc/demo.conf");
		result = open(config_file, O_RDONLY, 0);
		if (result < 0)
			strcpy(config_file, "/usr/local/etc/demo.conf");	/* this one is the last, don't even bother to open */
	}

	if (result >= 0)
		close(result);

#ifndef _VXWORKS_
	/* returns value < 0 on error, == 0 if all's Ok */
	result = get_opt(argc, argv);

	if (result < 0)
		return -1;
#else
	if (args) {
		/* returns value < 0 on error, > 1 on help or version request, == 0 if all's Ok */
		result = argp_parse(&argp, args);
		if (result > 0)
			return 0;
		if (result < 0)
			return -1;
	}

	taskPriorityGet(taskIdSelf(), &i);
	taskPrioritySet(taskIdSelf(), i+1);
#endif

	printf("Using config: %s\n", config_file);

	/* Initialise the endpoint info array */
	memset(&endpoints, 0, sizeof(endpoints));

	for (i = 0 ; i < MAX_ENDPOINTS; i++)
	{
		/* this is the phone number for this endpoint */
		endpoints[i].phone_id = i + 1;
		endpoints[i].index = i;
		endpoints[i].peer_index = -1; /* means no peer for now */

		/* the RTP port must be even */
		endpoints[i].rtp_port = htons(DEFAULT_UDP_PORT + i*2);
		/* the RTCP port must be odd */
		endpoints[i].rtcp_port = htons(DEFAULT_UDP_PORT + i*2 + 1);

		/*timeslot assignment*/
		endpoints[i].timeslot[0] = 2*i;
		endpoints[i].timeslot[1] = 2*i + 1;
		endpoints[i].timeslot[2] = 2*i + 64;
		endpoints[i].timeslot[3] = 2*i + 65;
		endpoints[i].dtmf_service = -1;
		endpoints[i].call_waiting_index = -1;
	}

	/* The VAPI default parameters are not convenient for us
	 * we overwrite some of them. This must be done before VAPI_Init()
	 */
	pfnUsrQueryVapiDefaultsEnumerate = set_default_config;

	/* Setting device parameters - this also must be done before VAPI_Init() */
	set_device_configuration(&gtl_device_configuration);

	/* First try to open the access to the legerity driver (/dev/legerity) */
#ifndef _VXWORKS_
	legerity_device_file = legerity_open("/dev/legerity");
#else
	result = legerity_init("/legerity0", "/spi1", 4*1000*1000, 6, 5, 1);
	if (result < 0)
	{
		PDEBUG (DEBUG_ERROR,"Can't init Legerity device");
		goto err0;
	}
	legerity_device_file = legerity_open("/legerity0");
#endif

	if (legerity_device_file < 0)
	{
		PDEBUG (DEBUG_ERROR,"Can't open Legerity device");
		goto err0;
	}

	result = VAPI_Init(&gtl_device_configuration);
	if(result == SUCCESS)
	{
		PDEBUG (DEBUG_INIT, "VAPI_Init SUCCESS\n");
	}
	else
	{
		PDEBUG (DEBUG_ERROR, "VAPI_Init Failed\n");
		goto err0;
	}

	/* get to know wheather current SLIC type supports WB mode*/
	slic_version = slic_get_version(legerity_device_file);

	printf("SLIC Product code : 0x%02x\n", (slic_version & 0xff00) >> 8);
	printf("SLIC Revision code : 0x%02x\n", (slic_version & 0x00ff));

	if (((slic_version & 0xff00) >> 8) == VP880_DEV_PCN_88276)
		is_wb_slic = True;
	else
		is_wb_slic = False;

	/*Get the MSP MR  number */
	result = get_mr(0);
	/*if < to MR 22 we do not support WB*/
	if (result < 21)
		is_wb_slic = False;

	VAPI_SetDebugLevel(vapi_debug_level, app_debug_level);
	
	result = VAPI_OpenDevice(0, NULL);
	check_status("VAPI_OpenDevice", err1);


	if (get_ec_mode(config_file) == 1)	/* Dual Filter EC if requested */
		result = VAPI_InitDevice(0, VAPI_DEV_OPMODE_DEFAULT, VAPI_DEV_PROF_SPU_DFECAN_ENABLE, NULL, NULL);
	else					/* else use standard EC */
		result = VAPI_InitDevice(0, VAPI_DEV_OPMODE_DEFAULT, VAPI_DEV_PROF_SPU_FSK_IP_DETECTION, NULL, NULL);

	PDEBUG (DEBUG_INIT, "result = %d", result);
	check_status("VAPI_InitDevice", err2);

	result = tdm_parameters_initialisation(config_file);
	check_status("TDM parameters initialisation", err2);

	result = network_parameters_initialisation(config_file);
	check_status("Network initialisation", err2);

	result = diagnostics_initialisation(config_file);
	check_status("Diagnostics initialisation", err2);

	for (i = 0 ; i < MAX_ENDPOINTS; i++)
			memcpy(&endpoints[i].ip_src_addr, &device_ip_address, sizeof(device_ip_address));

	/* Now create a thread to handle the events comming from the Signaling side (SLIC) */
	tid_signalling = UT_ThreadCreate("tSignalling", signaling_event_handler, NULL);
	if (tid_signalling == NULL)
	{
		PDEBUG(DEBUG_ERROR, "Can't create Signaling event handler thread");
		goto err3;
	}

	/* Now create a thread to handle the states */
	tid_state_machine = UT_ThreadCreate("tStateMachine", state_machine_handler, NULL);
	if (tid_state_machine == NULL)
	{
		PDEBUG(DEBUG_ERROR, "Can't create State machine thread");
		goto err4;
	}

	PDEBUG(DEBUG_INIT,"Register callback comcerto_indication_handler");
	VAPI_RegisterEventCallback(0, EVENT_LEVEL_GENERIC, comcerto_indication_handler);

	UT_ThreadJoin(tid_state_machine);
	UT_ThreadJoin(tid_signalling);

	return 0;

err4:
	UT_ThreadCancel(tid_signalling);

err3:
	legerity_close(legerity_device_file);

err2:
	VAPI_CloseDevice(0, ePURGE);

err1:
	VAPI_Close();

err0:
	return FAILURE;
}

/** @} */ /*pots to pots example*/
