/*! \file vapi_example.c  */
/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

/*! \mainpage This document describes vapi_coredump application which uses the VAPI library services to perform a coredump of a Comcerto device in Slave     mode.

	This application gives the idea of how to proceed with a coredump using VAPI on a slave configuration
	device configuration.

	<b>Some notes about configuring this application:</b>\n
	Before compiling the application, the user has to correctly configure the MAC and IP addresses accordingly
	the network configuration.
	The variables to configured are in vapi_app_utility.c:
	- DEVICE_CONTROL_MAC
	- HOST_CONTROL_INTERFACE
	- HOST_CONTROL_MAC
	
	<b>Some notes about using this application under the Linux:</b>\n
	It requires VAPI to be built with this options :
		- make GTL=CSME install

	The Makefile provided has to be edited to configured:
		- Set the native or cross compilation options

	the example usage is:
	./vapi_coredump -f firmware name -c coredum_file_name  -d device_id -t scenario_id -s buff_size 

	buff_size: 
		size of block which is used to get sdram. Default 1 M
	scenario_id: 
		1 Before device is open.
		2 The device is in normal running state(voip is active).
		3 The device is crashed already, we can issue an force_alert before doing.

	i.e ./vapi_coredump -f firm.axf -c test -d 3 -t 2

	<b>Some notes about using this application under the VxWorks:</b>\n
	ToBe described
*/

#include "vxcompat.h"

#if defined(_VXWORKS_)
#include <msp.h>
#include <taskLib.h>
#include <ioLib.h>
#include <gtl.h>
#else
#include <getopt.h>
#include <vapi/msp.h>
#include <vapi/gtl.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "vapi_app_type.h"
#include "vapi_app_func.h"

#define TCP_PORT 4000

extern SItfDesc astItfList[];

/*! structure with configuration parameters, structure type is defined in gtl.h */
SCSMEUsrData gtl_device_configuration;

static int device_id;
static int buf_size = 1024;
static int scenario_type = SCENARIO_VOIP_IS_ACTIVE;
int dumping;

/* this holds the filename entered by the user*/
static char coredump_file[256];
/* this holds the firmware filename entered by the user*/
static char firmware[256];
/* this holds the filename entered by the user with the memory suffix (i.e filename_sdram) */
static char coredump_filename[256];

/*Server name or IP address*/
static char *server_name = NULL;

#define PROGRAM_VERSION "vapi_coredump_app 0.3"

#if !defined(_VXWORKS_)
static void show_help(void)
{
	printf(
		"vapi_coredump_app - program to run coredump of a device.\n"
		"Usage: %s [OPTION]...\n"
		"  -d, --device_id=DEVICE_ID\n"
		"			ID of the Device to use (default %d)\n\n"
		"  -c, --coredump_filename=COREDUMP_FILE\n"
		"			coredump code filename (this option assumes slave mode)\n\n"
		"  -s, --buf_size=SIZE_OF_USED_BUFFER\n"
		"			size of block to store SDRAM\n\n"		
		"  -f, --firmware=FIRMWARE_FILE\n"
		"			firmware code filename (this option assumes slave mode)\n\n"
		"  -t, --scenario_type=SCENARIO_TYPE\n"
		"			which scenario of coredump to use  (default %d)\n\n"
		"  -i, --ip_address=IP_ADDRESSn"
		"			IP address (or name) of the coredump TCP server\n\n"
		"      --help		Show this help and exit\n"
		"      --version	Show version and exit\n",
		"vapi_coredump_app",
		device_id,
		scenario_type
	);
}

static void show_version(void)
{
	printf("vapi example application, %s\n", PROGRAM_VERSION);
}

static struct option const long_options[] =
{
	{ "device_id",		required_argument, NULL, 'd' },
	{ "coredump_filename",	required_argument, NULL, 'c' },
	{ "buf_size",		required_argument, NULL, 's' },
	{ "firmware",		required_argument, NULL, 'f' },
	{ "scenario_type",	required_argument, NULL, 't' },
	{ "ip_address",		required_argument, NULL, 'i' },

	{ "help",		no_argument, NULL, CHAR_MIN - 1 },
	{ "version",		no_argument, NULL, CHAR_MIN - 2 },

	{ NULL, 0, NULL, 0},
};

#else
const char *argp_program_version = PROGRAM_VERSION;
const char *argp_program_bug_address = "alexander.perekhodko@mindspeed.com";
const char doc[] = "vapi_coredump_app - program to run coredump of a device.";

const struct argp_option options[] = {
	{"device_id", 'd', "DEVICE_ID", 0, "ID of the Device to use (default 0)"},
	{"coredump_filename", 'c', "COREDUMP_FILE", 0, "coredump code filename (this option assumes slave mode)"},
	{"buf_size",'s',"SIZE_OF_USED_BUFFER",0,"size of block to store SDRAM"},
	{"firmware", 'f', "FIRMWARE_FILE", 0, "firmware code filename (this option assumes slave mode)"},
	{"scenario_type",'t',"SCENARIO_TYPE", 0, "which scenario of coredump to use  (default 2)"}, 
	{"ip_address",'i',"IP_ADDRESS", 0, "IP address (or name) of the coredump TCP server"},
	{0}
};
#endif

/*=================================================================================*/
/*!
 *	\section Description
 *	
 *	This function handles responses comming from the Comcerto device.\n
 *	This function is registered as call back when a function is called in asynchronous mode.\n
 *	
 *	\section Inputs-Outputs
 *		See VAPI documentation
 *	\section Execution
 *	This function is called in a thread created by VAPI.\n
 *	When it is called the request_id is set to the current state of endpoint.\n
 *	This function set an event accordingly the state retrieved thru the request_id.\n
 */
/*=================================================================================*/
void comcerto_response_handler(U32 connection_id,
			       U8 command_level, S32 result, void *response_data, U32 data_length, U32 request_id)
{
	info("response received request id = 0x%04x", request_id);

	if (request_id == 0x5555)
	{
		dumping = False;
		info("Dump finished");
	}
}

#if !defined(_VXWORKS_)
static int
get_int(const char *s, int *value)
{
	int i;
	char *endptr = NULL;

	if (!s || !*s)
		goto err;

	i = strtol(s, &endptr, 0);
	if (!i && s == endptr)
		goto err;

	if (value)
		*value = i;

	return 0;

err:
	return -1;
}

/*=================================================================================*/
/*!	\brief command line option parser
 *	\section Description
 *	This function is the command line option parser.\n
 */	
/*=================================================================================*/
static int get_opt(int argc, char *argv[])
{

	int c;
	int status = SUCCESS;
	
	while ((c = getopt_long(argc, argv, "d:c:f:s:t:i:", long_options, NULL)) != - 1) {
		switch (c) {
		case 'd':
			get_int(optarg, (int *) &device_id);
			break;
		case 'c':
			strncpy(coredump_file, optarg, sizeof(coredump_file) - 1);
			break;
		case 'f':
			strncpy(firmware, optarg, sizeof(firmware) - 1);
			break;
		case 's':
			get_int(optarg, (int *) &buf_size);
			break;
		case 't':
			get_int(optarg, (int *) &scenario_type);
			break;
		case 'i':
			server_name = optarg;
			break;

		case CHAR_MIN - 1:
			show_help();
			exit(0);
   			break;

		case CHAR_MIN - 2:
    			show_version();
			exit(0);
    			break;
			
		default:
			status = -1;
		}

	}

	return status;
}

#else
static error_t parser(int key, char *arg, struct argp_state *state)
{
	switch (key)
	{
	case 'd':
		device_id = strtoul(arg, NULL, 0);
		break;
	case 'c':
		strncpy(coredump_file, arg, sizeof(coredump_file) - 1);
		break;
	case 'f':
		strncpy(firmware, arg, sizeof(firmware) - 1);
		break;
	case 's':
		buf_size = strtoul(arg, NULL, 0);
		break;
	case 't':
		scenario_type = strtoul(arg, NULL, 0);
		break;
	case 'i':
		server_name = arg;
		break;
	default:
		return ARGP_ERR_UNKNOWN;
	}

	return 0;
}
static struct argp argp = { options, parser, 0, doc, };
#endif

#define check_status(title, error_label)						\
	PDEBUG(DEBUG_ERROR, title ": %s", result == SUCCESS?"ok":"failed");	\
	if (result != SUCCESS)							\
		goto error_label

#if !defined(_VXWORKS_)

/* under the Linux we close VAPI here, in atexit() handler */

static void close_vapi_on_exit()
{
	int result = VAPI_Close();

	fprintf(stderr, "VAPI_Close: %s\n", result == SUCCESS ? "ok" : "failed");
}
#endif

#define DEFAULT_BLOK_SIZE (1024 * 1024 *1) /* 1M  */
#define DEFAULT_MIN_BLOK_SIZE (1024 * 1) /* 8K  */

/* this function is registered in the astItfList[0].pstAdapterFns->pfnAdapterReset pointer */
static VSTATUS manual_reset(void *pvData)
{
	info("manual_reset: manual reset the board then enter a char ");
	getchar();
	return SUCCESS;
}

/***************************************************************************
 * main
 ***************************************************************************/
/*! \brief Application entry.

 *  \section Description
	This is the main function of the application.\n
	It does:\n
	- Initialise the setup configuration using parameters from the configration file (Device type, Control interface, Slave/Master mode , etc)
	- Print the VAPI verson (VAPI_GetVersion)
	- Initialise VAPI (VAPI_Init)
	- Open the Device (VAPI_OpenDevice) 
	- Using VAPI_InitDevice set the recover mode.
	- Send a Force Alert command
	- Perform the coredump of 3 memories.  
*/

#if !defined(_VXWORKS_)
int main(int argc, char *argv[])
#else
int vapi_example(char *args)
#endif
{
#if defined(_VXWORKS_)
	extern int consoleFd;
#endif
	int result, i = 0;
	U8* core_buffer;
	U32 ulOffset = 0;	/* Offset in the specified memory to get dump*/
	U32 ulDataSize = 0;		/* Number of bytes read */
	U32 nbytes = 0;		/* Number of bytes written */
	Boolean fLastRead = 0;
	int file_desc = -1;
	SRequest my_request;
	int sdram_size = 0;
	int iram_size = 0;
	int eram_size = 0;
	int aram_size = 0;
	int tcp_socket_id  = -1;

	/* Setting device parameters - this also must be done before VAPI_Init() */
	set_device_configuration(device_id);

	/* Coredump could be taken from devices in Slave mode only 
	 * M828xx device works in Master mode only */
	if (gtl_device_configuration.ucDevType == DEV_TYPE_M821XX
			|| gtl_device_configuration.ucDevType == DEV_TYPE_M83XXX
			|| gtl_device_configuration.ucDevType == DEV_TYPE_M828XX 
			|| gtl_device_configuration.eDevMode == eMASTER)
	{
		fprintf(stderr, "Couldn't take coredump from device in Master mode\n");
		goto err0;
	}
	
	/* keeping this call early, since we using VAPI logging facility */
	result = VAPI_Init(&gtl_device_configuration);
	if (result != SUCCESS)
	{
		/* VAPI init failed - we can't use VAPI logging */
		fprintf(stderr, "VAPI_Init: failed, result = %d\n", result);
		goto err0;
	}

#if !defined(_VXWORKS_)
	if (argc < 2)
	{
		PDEBUG(DEBUG_ERROR, "Please enter at least 1 argument");
		goto err1;
	}

	/* returns value < 0 on error, == 0 if all's Ok */
	result = get_opt(argc, argv);

	if (result < 0)
		goto err1;

	atexit(close_vapi_on_exit);
#else
	if (!args || !*args)
	{
		PDEBUG(DEBUG_ERROR, "Please enter at least 1 argument");
		goto err1;
	}

	/* Set standard input  to console */ 
	ioGlobalStdSet(STD_IN, consoleFd); 
	ioGlobalStdSet(STD_OUT, consoleFd); 
	ioGlobalStdSet(STD_ERR, consoleFd); 
	ioctl(consoleFd, FIOSETOPTIONS, OPT_TERMINAL & ~OPT_MON_TRAP); 

	/* Making VAPI Coredump utility task priority lower then VAPI internal threads priority*/
	taskPriorityGet(taskIdSelf(), &i);
	taskPrioritySet(taskIdSelf(), i+1);

	/* returns value < 0 on error, > 1 on help or version request, == 0 if all's Ok */
	result = argp_parse(&argp, args);

	if (result > 0)
		goto ok;
	if (result < 0)
		goto err1;
#endif

	/* we expect user to input buffer size in kilobytes */
	buf_size *= 1024;
	if (buf_size == 0)
	{
		buf_size = DEFAULT_BLOK_SIZE;
	}
	else
	{
		if (buf_size < DEFAULT_MIN_BLOK_SIZE)
		{
			PDEBUG(DEBUG_ERROR, "Please specify buffer size 8K or larger \n");
			goto err1;
		}
	}
	info("buf_size=%d bytes",buf_size);

	if (server_name != NULL)
	{
		tcp_socket_id = tcp_open(server_name, TCP_PORT);

		if (tcp_socket_id == -1)
		{
			PDEBUG(DEBUG_ERROR, "Can't connect to coredump IP server \n");
			goto err1;
		}
	}

	info("%s", VAPI_GetVersion());

	result = VAPI_OpenDevice(device_id, NULL);
	check_status("VAPI_OpenDevice", err1);

	if (scenario_type != SCENARIO_DEVICE_ALREADY_RESET)
	{

		#ifdef SECOND_DEV
			result = VAPI_OpenDevice(device_id+1, NULL);
			check_status("VAPI_OpenDevice", err1);
			result = VAPI_InitDevice(device_id+1, VAPI_DEV_OPMODE_RECOVER, VAPI_DEV_PROF_DEFAULT, NULL, NULL);

			display_firmware_version(device_id+1);
		#endif

		if (*firmware)
		{
			PDEBUG(DEBUG_INIT, "Booting device...");
			result = boot_device(device_id, firmware);
			check_status("VAPI_BootDevice", err1);
		}

		display_firmware_version(device_id);

		result = tdm_parameters_initialisation(device_id);
		check_status("TDM parameters initialisation", err1);

		result = VAPI_InitDevice(device_id, VAPI_DEV_OPMODE_DEFAULT, VAPI_DEV_PROF_DEFAULT, NULL, NULL);
		check_status("VAPI_InitDevice", err1);

		result = network_parameters_initialisation(device_id);
		PDEBUG(DEBUG_ERROR, "Network parameters initialisation %s", result == SUCCESS ? "ok" : "failed");

		info("The device has been initialized using VAPI in blocking mode (Synchronous)\n");

		info("VAPI_CreateConnection(%d) SYNC;\n" ,0);
		result = VAPI_CreateConnection(device_id, 0, eVOIP, 0, U_LAW, NULL, NULL);

		info("VAPI_CreateConnection(%d) SYNC;\n" ,1);
		result = VAPI_CreateConnection(device_id, 1, eVOIP, 1, U_LAW, NULL, NULL);

		if (scenario_type == SCENARIO_DEVICE_IS_CRASHED)
		{
			info("Force alert to crash device (%d) \n", device_id);
			result = send_force_alert(device_id);
			if (result != SUCCESS)
				PDEBUG(DEBUG_ERROR, "Endpoint %d >>>>>>>>>>> error %d", device_id, result);
			else 
				info("device (%d) is now crashed \n", device_id);
		}
	}

	info("device type = %d, device id = %d", gtl_device_configuration.ucDevType, device_id);
	switch (gtl_device_configuration.ucDevType)
	{
	case DEV_TYPE_M823XX:
		sdram_size = M823XX_SDRAM_SIZE;
		eram_size = M823XX_ERAM_SIZE;
		iram_size = 0;
		aram_size = M823XX_ARAM_SIZE;
		break;
	case DEV_TYPE_M825XX: 
		/* for M82505, M82510, M82520, M82524, M82530*/
		sdram_size = M825XX_SDRAM_SIZE;
	 	iram_size = M825XX_IRAM_SIZE;
		eram_size = M825XX_ERAM_SIZE;
		aram_size = 0;

		/* for M82501, M82506, M82511, M82514, M82515
		sdram_size = M825XX_2_SDRAM_SIZE;
	 	iram_size = M825XX_2_IRAM_SIZE;
		eram_size = M825XX_2_ERAM_SIZE;
		aram_size = M825XX_2_ARAM_SIZE;*/
		break;
	case DEV_TYPE_M826XX:
		sdram_size = M826XX_SDRAM_SIZE;
		eram_size = M826XX_ERAM_SIZE;
		iram_size = M826XX_IRAM_SIZE;
		aram_size = 0;
		break;
	case DEV_TYPE_M827XX:
		sdram_size = M827XX_SDRAM_SIZE;
	 	iram_size = M827XX_IRAM_SIZE;
		eram_size = M827XX_ERAM_SIZE;
		aram_size = M827XX_ARAM_SIZE;
		break;
	case DEV_TYPE_M829XX:
		sdram_size = M829XX_SDRAM_SIZE;
	 	iram_size = M829XX_IRAM_SIZE;
		eram_size = M829XX_ERAM_SIZE;
		aram_size = M829XX_ARAM_SIZE;
		break;
	case DEV_TYPE_M823XX_2:
		sdram_size = M823XX_2_SDRAM_SIZE;
		eram_size = M823XX_2_ERAM_SIZE;
		iram_size = 0;
		aram_size = M823XX_2_ARAM_SIZE;
		break;
	default: 
		info("Device was not specefied \n");
		goto err1;
	}

	/* Register user's reset function to call before core dump */
	VAPI_RegisterReset(0, manual_reset, NULL);

	info("VAPI is ready to proceed with the core dump\n");

	my_request.uiReqId = 0x5555;
	my_request.pfnIoCompCallback = &comcerto_response_handler;
	if (iram_size != 0)
	{
		core_buffer = (U8 *) UT_AllocMem(iram_size);    /* Buffer to hold data */
		if (core_buffer == NULL)
		{
			PDEBUG(DEBUG_INIT, "Can't allocate coredump buffer for IRAM");
			return FAILURE;
		}

		/* Open coredump file iram */
		strcpy(coredump_filename, coredump_file);
		strcat(coredump_filename, "_iram.bin");
		file_desc = open (coredump_filename, O_CREAT|O_WRONLY, 0644);
		if (file_desc < 0)
		{
			info("Can't open corefile %s", coredump_filename );
			return FAILURE;
		}
	
		/* flag to indicate the dump is in process */
		dumping = True;
		info("Starting device(%d) coredump procedure for IRAM(idx1) (Async mode) to file %s",device_id, coredump_filename);

/*! \cond */
#if VAPI_RELEASE >= (VAPI_VERSION(2, 6, 0))
		result = VAPI_GetCoreDump(device_id, IRAM_INDEX, core_buffer, iram_size, &ulDataSize, ulOffset, &fLastRead, NULL, &my_request);
#else
		result = VAPI_GetCoreDump(device_id, IRAM_INDEX, core_buffer, iram_size, &ulDataSize, ulOffset, &fLastRead, &my_request);
#endif /* VAPI_RELEASE >= (VAPI_VERSION(2, 6, 0)) */
/*! \endcond */

		if (result != VAPI_ERR_PENDING)
		{
			info("Coredump device(%d) Failed\n",device_id);
			result = FAILURE;
			goto err;
		}

		
		while (dumping == True)
		{
#ifdef SECOND_DEV
			display_firmware_version(1);
#else
			info("waiting for device(%d) dump completion\n",device_id);
#endif
			/* There is no need to wait if dumping is finished during log print */
			if (dumping == True) 
				usleep(200);
		}

		info("device %d dumped: Offset %d, mem %d\n",device_id, ulOffset, ulDataSize);
		write(file_desc, core_buffer, ulDataSize);
		close(file_desc);
		result = SUCCESS;
		UT_FreeMem(core_buffer);
	}

	/* Open coredump file for sdram*/
	strcpy(coredump_filename, coredump_file);
	strcat(coredump_filename, "_sdram.bin");

	if (server_name != NULL)
	{
		result = cdoip_open_file(tcp_socket_id, coredump_filename);
		if (result != 0)
		{
			info("Can't open corefile %s on coredump IP server", coredump_filename );
			return FAILURE;
		}
	}
	else
	{
		file_desc = open (coredump_filename, O_CREAT|O_WRONLY, 0644);
		if (file_desc < 0)
		{
			info("Can't open corefile %s", coredump_filename );
			return FAILURE;
		}
	}

	info("buf_size=%d",buf_size);
	core_buffer = (U8 *) UT_AllocMem(buf_size);    /* Buffer to hold data */
	if (core_buffer == NULL)
	{
		PDEBUG(DEBUG_INIT, "Can't allocate  buffer for sdram %d", device_id );
		return FAILURE;
	}
	
	fLastRead = 0;
	ulOffset = 0;
	ulDataSize = 0;

	info("Starting device(%d) coredump procedure for SDRAM(idx0) (Async mode) to file %s",device_id, coredump_filename);
	for (i = 0; !fLastRead; i++)
	{
		fLastRead = 0;
		ulDataSize = 0;
		/* flag to indicate the dump is in process */
		dumping = True;

/*! \cond */
#if VAPI_RELEASE >= (VAPI_VERSION(2, 6, 0))
		result = VAPI_GetCoreDump(device_id, SDRAM_INDEX, core_buffer, buf_size, &ulDataSize, ulOffset, &fLastRead, NULL, &my_request);
#else
		result = VAPI_GetCoreDump(device_id, SDRAM_INDEX, core_buffer, buf_size, &ulDataSize, ulOffset, &fLastRead, &my_request);
#endif /* VAPI_RELEASE >= (VAPI_VERSION(2, 6, 0)) */
/*! \endcond */

		if (result != VAPI_ERR_PENDING)
		{
			info("Coredump device(%d) Failed\n",device_id);
			result = FAILURE;
			goto err;
		}
		else
		{	
			while (dumping == True)
			{
#ifdef SECOND_DEV
				display_firmware_version(1);
#else
				info("waiting for device(%d) dump completion\ndumping == %d",device_id, dumping);
#endif
			/* There is no need to wait if dumping is finished during log print */
				if (dumping == True) 
					usleep(200);
			}

			if (server_name != NULL)
			{
				result = cdoip_write_file(tcp_socket_id, ulDataSize, (char *)core_buffer);
				if (result == 0)
					info("Write %d bytes to coredump IP server (total %d)\n", ulDataSize, ulOffset);
				else
					info("Error to write bytes to coredump IP server\n");
			}
			else
			{
				nbytes = write(file_desc, core_buffer, ulDataSize);
				info("device(%d) dumped: Offset %d, mem %d; %d bytes written\n last_read == %d\n",device_id, ulOffset, ulDataSize, nbytes, fLastRead);				
			}
		}
		ulOffset += ulDataSize;
	}
	if (server_name != NULL)
	{
		result = cdoip_close_file(tcp_socket_id, coredump_filename);	
	}
	else
	{
		close(file_desc);
		result = SUCCESS;
	}

	UT_FreeMem(core_buffer);

	core_buffer = (U8 *) UT_AllocMem(eram_size);    /* Buffer to hold data */
	if (core_buffer == NULL)
	{
		PDEBUG(DEBUG_INIT, "Can't allocate  buffer for ERAM %d", device_id );
		return FAILURE;
	}
	/* Open coredump file for eram */
	strcpy(coredump_filename, coredump_file);
	strcat(coredump_filename, "_eram.bin");
	file_desc = open (coredump_filename, O_CREAT|O_WRONLY, 0644);
	if (file_desc < 0)
	{
		info("Can't open corefile %s", coredump_filename );
		return FAILURE;
	}
	
	ulOffset = 0;
	fLastRead = 0;
	ulDataSize = 0;

	info("Starting device(%d) coredump procedure for ERAM(idx2) (Sync mode) to file %s",device_id, coredump_filename);

/*! \cond */
#if VAPI_RELEASE >= (VAPI_VERSION(2, 6, 0))
	result = VAPI_GetCoreDump(device_id, ERAM_INDEX, core_buffer, eram_size, &ulDataSize, ulOffset, &fLastRead, NULL, NULL);
#else
	result = VAPI_GetCoreDump(device_id, ERAM_INDEX, core_buffer, eram_size, &ulDataSize, ulOffset, &fLastRead, NULL);
#endif /* VAPI_RELEASE >= (VAPI_VERSION(2, 6, 0)) */
/*! \endcond */
	
	if (result != SUCCESS)
	{
		info("Coredump device(%d) Failed\n",device_id);
		result = FAILURE;
		goto err;
	}
	else
	{
		info("device(%d) dumped: Offset %d, mem %d\n",device_id, ulOffset, ulDataSize);
		write(file_desc, core_buffer, ulDataSize);
		close(file_desc);
		result = SUCCESS;
	}
	close(file_desc);
	result = SUCCESS;

	if (aram_size != 0)
	{
		/* Open coredump file for aram */
		UT_FreeMem(core_buffer);
		core_buffer = (U8 *) UT_AllocMem(aram_size);    /* Buffer to hold data */
		if (core_buffer == NULL)
		{
			PDEBUG(DEBUG_INIT, "Can't allocate  buffer for ARAM %d", device_id );
			return FAILURE;
		}
		strcpy(coredump_filename, coredump_file);
		strcat(coredump_filename, "_aram.bin");
		file_desc = open (coredump_filename, O_CREAT|O_WRONLY, 0644);
		if (file_desc < 0)
		{
			info("Can't open corefile %s", coredump_filename );
			return FAILURE;
		}
	
		ulOffset = 0;
		fLastRead = 0;
		ulDataSize = 0;

		info("Starting device(%d) coredump procedure for ARAM(idx2) (Sync mode) to file %s",device_id, coredump_filename);

/*! \cond */
#if VAPI_RELEASE >= (VAPI_VERSION(2, 6, 0))
		result = VAPI_GetCoreDump(device_id, ARAM_INDEX, core_buffer, aram_size, &ulDataSize, ulOffset, &fLastRead, NULL ,NULL);
#else
		result = VAPI_GetCoreDump(device_id, ARAM_INDEX, core_buffer, aram_size, &ulDataSize, ulOffset, &fLastRead, NULL);
#endif /* VAPI_RELEASE >= (VAPI_VERSION(2, 6, 0)) */
/*! \endcond */
	
		if (result != SUCCESS)
		{
			info("Coredump device(%d) Failed\n",device_id);
			result = FAILURE;
			goto err;
		}
		else
		{
			info("device(%d) dumped: Offset %d, mem %d\n",device_id, ulOffset, ulDataSize);
			info("%d \n",fLastRead);
			write(file_desc, core_buffer, ulDataSize);
			close(file_desc);
			result = SUCCESS;
			UT_MemSet(core_buffer, eram_size, 0);
		}
	}

err:
	UT_FreeMem(core_buffer);
	close(file_desc);

	result = VAPI_CloseDevice(device_id, ePURGE);
		check_status("VAPI_CloseDevice", err1);

#if defined(_VXWORKS_)
/* under VxWorks we close VAPI here, under Linux we close it in atexit() handler */
      ok:
	result = VAPI_Close();
	fprintf(stderr, "VAPI_Close: %s\n", result == SUCCESS ? "ok" : "failed");
#endif

	return 0;

      err1:
	VAPI_Close();

      err0:
	return FAILURE;
}
