/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

/* coredump over IP test client  application
 * gcc -Wall cdoip_client.c -o cdoip_client
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include "cdoip.h"

/* test buffer, filled with 0, 1, 2, 3, etc.. */
static char cd_buffer[1024 * 1024];

static int handle_cdoip_response(int socket_id, struct cdoip_header *this_cdoip_frame)
{
	int rc = SUCCESS;

	/* receive the answer from the server*/
	rc = recv(socket_id,(char *) this_cdoip_frame, MAXDATASIZE + CDOIP_HDR_SIZE, 0);

	switch(this_cdoip_frame->type)
	{
		case RESP:
			/* print the result received from the server*/
			fprintf(stdout,"Received response from server: status(%d)\n",
					this_cdoip_frame->buffer[0]);
					rc = this_cdoip_frame->buffer[0];
			break;

		default:
			printf("Unknown command type %d\n", this_cdoip_frame->type);
			rc = FAILURE;
	}

	return rc;
}

static int cdoip_open_file(int socket_id, char *filename)
{
	int status = 0;
	struct cdoip_header coredump_data;

	memset(&coredump_data, 0, sizeof(struct cdoip_header));

	coredump_data.type = OPEN;
	coredump_data.size = strlen(filename);
	memcpy(&coredump_data.buffer, filename, coredump_data.size);

	fprintf(stdout,"Send OPEN file %s (%d)\n", filename, coredump_data.size);
	send(socket_id,(char *) &coredump_data, coredump_data.size + CDOIP_HDR_SIZE, 0);

	status = handle_cdoip_response(socket_id, &coredump_data);
	if (status != 0)
		printf("Can't open coredump file %s on server\n", filename);

	return status;
}

static int cdoip_close_file(int socket_id, char *filename)
{
	int status = 0;
	struct cdoip_header coredump_data;

	memset(&coredump_data, 0, sizeof(struct cdoip_header));

	coredump_data.type = CLOSE;
	coredump_data.size = strlen(filename);
	memcpy(&coredump_data.buffer, filename, coredump_data.size);

	fprintf(stdout,"Send CLOSE file\n");
	send(socket_id,(char *) &coredump_data, coredump_data.size + CDOIP_HDR_SIZE, 0);

	status = handle_cdoip_response(socket_id, &coredump_data);
	if (status != 0)
		printf("Can't close coredump file %s on server\n", filename);

	return status;
}

static int cdoip_write_file(int socket_id, int cd_size, char *cd_buffer)
{
	int status = 0;
	struct cdoip_header coredump_data;
	memset(&coredump_data, 0, sizeof(struct cdoip_header));

	if (cd_size > MAXDATASIZE)
	{
		printf("Coredump data to big %d (max %d)\n", cd_size, MAXDATASIZE);
		return -1;
	}

	coredump_data.type = WRITE;
	coredump_data.size = cd_size;
	memcpy(&coredump_data.buffer, cd_buffer, cd_size);
	send(socket_id,(char *) &coredump_data, coredump_data.size + CDOIP_HDR_SIZE, 0);

	status = handle_cdoip_response(socket_id, &coredump_data);

	if (status != 0)
		printf("Can't write to coredump file on server\n");

	return status;
}

static int tcp_open(char *server_name, short server_port)
{
	int sock;
	int status;
	struct hostent *he = NULL;
	struct sockaddr_in sock_addr_def;
	struct cdoip_header coredump_data;

	sock = socket(AF_INET,SOCK_STREAM,0);
	if (sock == -1)
	{
		perror("socket() failed");
		return -1;
	}

	he = gethostbyname(server_name);
	if (he == NULL)
	{
		fprintf(stderr, "gethostbyname() failed\n");
		return -1;
	}

	fprintf(stdout,"Server official name: \"%s\"\n", he->h_name);

	sock_addr_def.sin_family = AF_INET;
	memcpy(&sock_addr_def.sin_addr.s_addr, he->h_addr, he->h_length);
	sock_addr_def.sin_port = htons(server_port);

	status = connect(sock, (struct sockaddr *) &sock_addr_def, sizeof(struct sockaddr_in));
	if (status == -1)
	{
		perror("connect() failed");
		return status;
	} else
		printf("Connected to %s:%d\n", server_name, server_port);

	memset(&coredump_data, 0, sizeof(struct cdoip_header));

	/* receive the answer from the server */
	status = recv(sock,(char *) &coredump_data.buffer, MAXDATASIZE, 0);

	/* print the result received from the server*/
	printf("Data received: %d, result received from server: %s\n", status, coredump_data.buffer);

	return sock;
}

int main(int argc, char **argv) 
{
	char *server_name;
	short server_port = TCP_PORT;
	int sock;
	int result;
	char filename[] = "coredump";
	int i;

	/* fill the test buffer with 0, 1, 2, 3, etc.. */
	for (i = 0; i < 1024; i ++)
		memset(&cd_buffer[i * 1024], i, 1024);

	/* command line parsing */
	if (argc > 4 || argc < 2)
	{
		fprintf(stderr, "Usage: %s <server name> [server port] [filename]\n", argv[0]);
		exit(1);
	}

	if (argc > 3)	/* user specified filename */
		strncpy(filename, argv[3], sizeof(filename));
	if (argc > 2)	/* user specified port number */
		server_port = atoi(argv[2]);
	if (argc > 1)	/* user specified only server use default port and filename */
		server_name = argv[1];

	printf("Server: %s:%d, file: %s\n", server_name, server_port, filename);

	sock = tcp_open(server_name, server_port);
	if (sock == -1)
		exit(1);

	fprintf(stdout,"Opening file %s \n", filename);
	result = cdoip_open_file(sock, filename);
	if (result != 0) {
		fprintf(stderr, "Can't open coredump file\n");	
		exit(1);
	}

	/* send the test buffer to the cdoip_server by frame of 1024 bytes */
	for (i = 0; i < 1024; i++) {
		printf("Write %d bytes (total %d) to file %s\n", MAXDATASIZE, i * 1024, filename);	
		result = cdoip_write_file(sock, MAXDATASIZE, &cd_buffer[i * 1024]);
		if (result != 0) {
			fprintf(stderr, "Can't write to coredump file %d data\n", i);
			cdoip_close_file(sock, filename);
			exit(1);
		}
	}

	fprintf(stdout,"Closing file %s\n", filename);
	cdoip_close_file(sock, filename);

	return 0;
}
