/*! \file vapi_example.c  */
/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

/*! \mainpage This document describes vapi_exapp application which uses the VAPI library services.

	This application gives the idea of how to use VAPI with user-supplied
	device configuration.

	<b>Some notes about configuring this application:</b>\n
	Before compiling the application, the user has to correctly configure the MAC and IP addresses accordingly
	the network configuration.
	The variables to configured are in vapi_app_utility.c:
	- DEVICE_CONTROL_MAC
	- HOST_CONTROL_INTERFACE
	- HOST_CONTROL_MAC
	- device_mac
	- default_dest_mac
	- device_ip
	
	<b>Some notes about using this application under the Linux:</b>\n
	It requires VAPI to be built with this options :
		- make GTL=CSME install

	The Makefile provided has to be edited to configured:
		- Set the native or cross compilation options

	the example usage is:
	./vapi_exapp -f firmware_filename.axf -d device_itf -e number_of_endpoints

	i.e ./vapi_exapp -f /home/v5_07_cand1_26416.axf -d 0 -e 12

	<b>Some notes about using this application under the VxWorks:</b>\n
	This application is shipped with a sample Tornado "vapi_example" project,
	added to vapilib workspace in VxProj directory.

	This project links together all VAPI and example code. As a result you'll
	get a module that has everything included. Load it and run vapi_example()
	function, the best way is to spawn it as a task:
	-> ld <vapi_example.out
	-> taskSpawn("vapi_example", 100, 0, 64*1024, vapi_example, "-f /home/v5_07_cand1_26416.axf -d 0 -e 12", 0, 0, 0)
*/

#include "vxcompat.h"

#if defined(_VXWORKS_)
#include <msp.h>
#include <taskLib.h>
#else
#include <getopt.h>
#include <vapi/msp.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "vapi_app_type.h"
#include "vapi_app_func.h"

static int device_id = 0;
static int device_itf = eCSM_ITF;
static int number_of_endpoints;

static char firmware[256];

/* Some hard-coded configuration parameters are placed here, TDM parameter are
 * in tdm_parameters_initialisation() function.
 */

struct _ENDPOINT_DESC endpoints[MAX_ENDPOINTS];

/*! structure with configuration parameters, structure type is defined in gtl.h */
SDeviceHeader gtl_device_configuration;

#define PROGRAM_VERSION  "vapi_simple_app 0.1"

#if !defined(_VXWORKS_)
static void show_help(void)
{
	printf(
		"vapi_ex_app - program to run vapi device init and 2 connections.\n"
		"Usage: %s [OPTION]...\n"
		"  -e, --num_of_endpoints=NUMBER_OF_ENDPOINTS\n"
		"			Number of endpoints to create (default %d)\n\n"
		"  -d, --device_itf=DEVICE_ITF\n"
		"			device itf control type (default %d)\n\n"
		"  -f, --firmware=FIRMWARE_FILE\n"
		"			firmware code filename (this option assumes slave mode)\n\n"
		"      --help		Show this help and exit\n"
		"      --version	Show version and exit\n",
		"vapi_demo",
		number_of_endpoints,
		device_itf
	);
}

static void show_version(void)
{
	printf("vapi example application, %s\n", PROGRAM_VERSION);
}

static struct option const long_options[] =
{
	{ "num_of_endpoints",	required_argument, NULL, 'c' },
	{ "device_itf",		required_argument, NULL, 'd' },
	{ "firmware",		required_argument, NULL, 'f' },

	{ "help",		no_argument, NULL, CHAR_MIN - 1 },
	{ "version",		no_argument, NULL, CHAR_MIN - 2 },

	{ NULL, 0, NULL, 0},
};

#else
const char *argp_program_version = PROGRAM_VERSION;
const char *argp_program_bug_address = "alexander.perekhodko@mindspeed.com";
const char doc[] = "vapi_ex_app - program to run vapi device init and 2 connections.";

const struct argp_option options[] = {
	{"num_of_endpoints", 'e', "NUMBER_OF_ENDPOINTS", 0, "Number of endpoints to create (default = 2)"},
	{"device_itf", 'd', "DEVICE_ITF", 0, "Device control itf (default CSME)"},
	{"firmware", 'f', "FIRMWARE_FILE", 0, "firmware code filename (this option assumes slave mode)"},
	{0}
};
#endif

static void set_endpoint_event(int event_type, int index)
{
	endpoints[index].event = event_type;
	PDEBUG(DEBUG_INIT, "Endpoint %d set event %d (state %d)", index, endpoints[index].event,
	       endpoints[index].state);
}

static void set_endpoint_state(int state, int index)
{
	endpoints[index].state = state;
	PDEBUG(DEBUG_INIT, "Endpoint %d, set state %d (event %d)", index, endpoints[index].state,
	       endpoints[index].event);
}

/*=================================================================================
This function handle the onhook event in several states the endpoints.
=================================================================================*/

static void onhook_handler(int endpoint_id)
{
	/* switch to the state WAIT_DESTROY_CONNECTION_ACK to wait the result of destroy_conenction */
	set_endpoint_state(ENDPOINT_STATE_WAIT_DESTROY_CONNECTION_ACK, endpoint_id);
	set_endpoint_event(ENDPOINT_EVENT_NONE, endpoint_id);
	destroy_endpoint(endpoint_id, ASYNC, ENDPOINT_STATE_WAIT_DESTROY_CONNECTION_ACK);
}

/*=================================================================================*/
/*!
 *	\section Description
 *	
 *	This function handles responses comming from the Comcerto device.\n
 *	This function is registered as call back when a function is called in asynchronous mode.\n
 *	
 *	\section Inputs-Outputs
 *		See VAPI documentation
 *	\section Execution
 *	This function is called in a thread created by VAPI.\n
 *	When it is called the request_id is set to the current state of endpoint.\n
 *	This function set an event accordingly the state retrieved thru the request_id.\n
 */
/*=================================================================================*/
void comcerto_response_handler(U32 connection_id,
			       U8 command_level, S32 result, void *response_data, U32 data_length, U32 request_id)
{
	PDEBUG(DEBUG_INIT, "response received on endpoint id %d, request id = 0x%04x", connection_id, request_id);

	switch (request_id)
	{
	case ENDPOINT_STATE_WAIT_CREATE_CONNECTION_ORIGINATE_ACK:
	case ENDPOINT_STATE_WAIT_CREATE_CONNECTION_TERMINATE_ACK:
		PDEBUG(DEBUG_INIT, "VoIP Endpoint %d created,", connection_id);
		/* The VoIP Endpoint has been succesfully created, post the event */
		set_endpoint_event(ENDPOINT_EVENT_VOIP_CHANNEL_CREATED, connection_id);
		break;

	case ENDPOINT_STATE_WAIT_DESTROY_CONNECTION_ACK:
		PDEBUG(DEBUG_INIT, "VoIP Endpoint %d destroyed,", connection_id);
		/* The VoIP Endpoint has been succesfully created, post the event */
		set_endpoint_event(ENDPOINT_EVENT_VOIP_CHANNEL_DESTROYED, connection_id);
		break;

	case ENDPOINT_STATE_WAIT_IP_PARAMETERS_ORIGINATE_ACK:
		PDEBUG(DEBUG_INIT, "VoIP Endpoint %d IP parameters set,", connection_id);
		/* The VoIP Endpoint IP parameters have been succesfully set, post the event */
		set_endpoint_event(ENDPOINT_EVENT_IP_PARAMETERS_SET, connection_id);
		break;

	case ENDPOINT_STATE_WAIT_IP_PARAMETERS_TERMINATE_ACK:
		PDEBUG(DEBUG_INIT, "VoIP Endpoint %d IP parameters set,", connection_id);
		/* The VoIP Endpoint IP parameters have been succesfully set, post the event */
		set_endpoint_event(ENDPOINT_EVENT_IP_PARAMETERS_SET, connection_id);
		break;

	case ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_ORIGINATE_ACK:
		PDEBUG(DEBUG_INIT, "VoIP Endpoint %d enabled,", connection_id);
		/* The VoIP Endpoint has been succesfully enabled, post the event */
		set_endpoint_event(ENDPOINT_EVENT_CONNECTION_ENABLED, connection_id);
		break;

	case ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_TERMINATE_ACK:
		PDEBUG(DEBUG_INIT, "VoIP Endpoint %d enabled,", connection_id);
		/* The VoIP Endpoint has been succesfully enabled, post the event */
		set_endpoint_event(ENDPOINT_EVENT_CONNECTION_ENABLED, connection_id);
		break;

	}
}

/*=================================================================================*/
/*!
 *	\section Description
 *	
 *	This function handles indications from the Comcerto device.\n
 *	This function is registered as call back for event.\n
 *	
 *	\section Inputs-Outputs
 *		See VAPI documentation.\n
 *	\section Execution
 *	This function is called by VAPI every time a event occurs on the Comcerto device.\n
 *	It just prints:.\n
 *		- the tone number if the event reported is  eVAPI_TONE_DETECT_EVENT. 
 *		- the new SSRC  if the event reported is  eVAPI_SSRC_CHANGE_EVENT. 
 *	In a real system it should set some endpoint events such as CED detected to proceed a T38 switch. 
*/
/*=================================================================================*/
void comcerto_indication_handler(EEventCode eEventCode, void *pvData)
{
	SToneDetectEventParams *tone_detected;
	SSsrcChangeEventParams *ssrc_changed;
	SUndefinedEventParams *this_event;

	/* FIXME: The parser below need to be improved */

	switch (eEventCode)
	{
	case eVAPI_TONE_DETECT_EVENT:
		tone_detected = (SToneDetectEventParams *) pvData;

		PDEBUG(DEBUG_INIT, "Tone %d detected on Connection ID %d",
		       tone_detected->usDetectedTone, tone_detected->ConId);

		break;

	case eVAPI_SSRC_CHANGE_EVENT:
		ssrc_changed = (SSsrcChangeEventParams *) pvData;
		PDEBUG(DEBUG_INIT, "SSRC changed 0x%08x on Connection ID %d",
		       ssrc_changed->uNewSSRC, ssrc_changed->ConId);

	case eVAPI_SPI_EVENT:
	case eVAPI_TONE_GEN_CMPLT_EVENT:
	case eVAPI_PT_CHANGE_EVENT:
	case eVAPI_SSRC_VIOLATION_EVENT:
	case eVAPI_NTE_TRANSMIT_COMPLETE_EVENT:
	case eVAPI_NTE_RECVD_EVENT:
	case eVAPI_CALLER_ID_CMPLT_EVENT:
	case eVAPI_G711_CONCURRENT_DECODER_EVENT:
	case eVAPI_PASSTHRU_CONCURRENT_DECODER_EVENT:
	case eVAPI_CALLER_ID_DETECTED_EVENT:
	case eVAPI_REMOTE_DETECT_EVENT:
	case eVAPI_FAX_SWITCH_CMPLT_EVENT:
	case eVAPI_ALERT_IND:
		break;

	case eVAPI_UNDEFINED_EVENT:
		PDEBUG(DEBUG_INIT, "event %d received", eEventCode);
		this_event = (SUndefinedEventParams *) pvData;

		if (this_event->ucCmdType == CMD_TYPE_INDICATION)
			PDEBUG(DEBUG_INIT, "indication 0x%04x received", this_event->usFnCode);

		break;
	default:
		break;
	}

	return;
}

/*=================================================================================*/
/*!
 *	\section Description
 *	This function handles the states of the endpoints.\n
 *	
 *	\section Inputs-Outputs
 *		None
 *	
 *	\section Execution
 *	A infinite while loop is running to check the state of the endpoints array.\n
 *	For each state a limited number of event are handled.\n
 *	The events are set by the comcerto_response_handler or the comcerto_indication_handler.
 *	If the event is handled in the current state, an action is taken accordingly for the current endpoint.
 *	ELse the endpoint keep the same state.
*/
/*=================================================================================*/
void *state_machine_handler()
{
	int i;

	/* We should have here a signal handler to interrupt the while loop
	 * For now loop forever, CTRL-C to be used to stop.
	 */
	PDEBUG(DEBUG_INIT, "State machine started");

	while (1)
	{
		/* Parse all the endpoints in the global endpoints array */
		for (i = 0; i < number_of_endpoints; i++)
		{
			/* If the state is IDLE we should receive only OFFHOOK or RING events */
			switch (endpoints[i].state)
			{
			case ENDPOINT_STATE_IDLE:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			case ENDPOINT_STATE_WAIT_CREATE_CONNECTION_ORIGINATE_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_VOIP_CHANNEL_CREATED:

					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			case ENDPOINT_STATE_WAIT_CREATE_CONNECTION_TERMINATE_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_VOIP_CHANNEL_CREATED:
					/* The VoIP channel is created we set now the IP parameters on both sides */

					set_endpoint_state(ENDPOINT_STATE_WAIT_IP_PARAMETERS_TERMINATE_ACK,
							   endpoints[i].index);
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					set_ip_udp_parameters(device_ip, endpoints[i].index, ASYNC,
							      ENDPOINT_STATE_WAIT_IP_PARAMETERS_TERMINATE_ACK);

					set_endpoint_state(ENDPOINT_STATE_WAIT_IP_PARAMETERS_ORIGINATE_ACK,
							   endpoints[i].peer_index);
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].peer_index);
					set_ip_udp_parameters(device_ip, endpoints[i].peer_index, ASYNC,
							      ENDPOINT_STATE_WAIT_IP_PARAMETERS_ORIGINATE_ACK);

					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			/* This endpoint has a VoIP channel associated, the Command to set the IP params been posted. 
			 * We are waiting for the response.
			 * In case of originate side the next action is to stqrt the packect processing
			 */
			case ENDPOINT_STATE_WAIT_IP_PARAMETERS_ORIGINATE_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_STI_FUNC, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_IP_PARAMETERS_SET:
					set_endpoint_state(ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_ORIGINATE_ACK,
							   endpoints[i].index);
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					start_voip_endpoint(endpoints[i].index, ASYNC,
							    ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_ORIGINATE_ACK);

					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}

				break;

			/* This endpoint has a VoIP channel associated, the Command to set the IP params been posted. 
			 * We are waiting for the response.
			 * In case of terminate side the next action is to stqrt the packect processing
			 */
			case ENDPOINT_STATE_WAIT_IP_PARAMETERS_TERMINATE_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_IP_PARAMETERS_SET:
					set_endpoint_state(ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_TERMINATE_ACK,
							   endpoints[i].index);
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					start_voip_endpoint(endpoints[i].index, ASYNC,
							    ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_TERMINATE_ACK);

					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			/* This endpoint has a VoIP channel started, the VoIP conenction is now running. */
			case ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_ORIGINATE_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_CONNECTION_ENABLED:
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);

					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			/* This endpoint has a VoIP channel started, the VoIP conenction is now running. */
			case ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_TERMINATE_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_CONNECTION_ENABLED:
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;
			}
		}

#if !defined(_VXWORKS_)
		usleep(100);
#else
		taskDelay(1);
#endif
	}
}

#if !defined(_VXWORKS_)
static int
get_int(const char *s, int *value)
{
	int i;
	char *endptr = NULL;

	if (!s || !*s)
		goto err;

	i = strtol(s, &endptr, 0);
	if (!i && s == endptr)
		goto err;

	if (value)
		*value = i;

	return 0;

err:
	return -1;
}

/*=================================================================================*/
/*!	\brief command line option parser
 *	\section Description
 *	This function is the command line option parser.\n
 */	
/*=================================================================================*/
static int get_opt(int argc, char *argv[])
{
	int c;
	int status = SUCCESS;
	
	while ((c = getopt_long(argc, argv, "d:e:f:", long_options, NULL)) != -1)
	{
		switch (c) {
		case 'd':
			get_int(optarg, (int *) &device_itf);
			break;

		case 'e':
			get_int(optarg, (int *) &number_of_endpoints);

			if (number_of_endpoints > MAX_ENDPOINTS)
				number_of_endpoints = MAX_ENDPOINTS;
			break;

		case 'f':
			strncpy(firmware, optarg, sizeof(firmware));
			break;

		case CHAR_MIN - 1:
			show_help();
			exit(0);
			break;

		case CHAR_MIN - 2:
			show_version();
			exit(0);
			break;

		default:
			status = -1;
		}

	}

	return status;
}

#else
static error_t parser(int key, char *arg, struct argp_state *state)
{
	switch (key)
	{
	case 'd':
		device_itf = strtoul(arg, NULL, 0);
		break;

	case 'e':
		number_of_endpoints = strtoul(arg, NULL, 0);

		if (number_of_endpoints > MAX_ENDPOINTS)
			number_of_endpoints = MAX_ENDPOINTS;
		break;

	case 'f':
		strncpy(firmware, arg, sizeof(firmware) - 1);
		break;

	default:
		return ARGP_ERR_UNKNOWN;
	}

	return 0;
}
static struct argp argp = { options, parser, 0, doc, };
#endif

#define check_status(title, error_label)						\
	PDEBUG(DEBUG_ERROR, title ": %s", result == SUCCESS?"ok":"failed");	\
	if (result != SUCCESS)							\
		goto error_label

#if !defined(_VXWORKS_)

/* under the Linux we close VAPI here, in atexit() handler */

static void close_vapi_on_exit()
{
	int result = VAPI_Close();

	fprintf(stderr, "VAPI_Close: %s\n", result == SUCCESS ? "ok" : "failed");
}
#endif

/***************************************************************************
 * main
 ***************************************************************************/
/*! \brief Application entry.

 *  \section Description
	This is the main function of the application.\n
	It does:\n
	 - initialise an endpoints array
	 - then parse the command line arguments (device ID, firmware filename)
	 - then it:
		- Initialise the setup configuration using parameters from the configration file (Device type, Control interface, Slave/Master mode , etc)
		- Print the VAPI verson (VAPI_GetVersion)
		- Initialise VAPI (VAPI_Init)
		- Open the Device (VAPI_OpenDevice) 
		- Boot the Device (VAPI_AssignBootMAC, VAPI_BootDevice) if in Slave mode (if no firmware file name is provided it assumes master mode).
		- Initialise the TDM buses using default parameters (VAPI_SetTDMParams).
		- Using VAPI_InitDevice set the SPU feature and enable the Multicommand mode.
		- Set the Network Parameters (MAC, IP addresses - VAPI_SetEthMac, VAPI_SetDeviceIPAddr).
		- Display device informations (ARM Version, DSP version, Device type) using the VAPI_Passthru command.

	All the device level VAPI APIs are called in Synchronous mode.
	The connection level VAPI APIs are called in Asynchronous mode.
	- then it prompts the user to start the state machine handler, which will create, initialise and start endpoints.  
*/

#if !defined(_VXWORKS_)
int main(int argc, char *argv[])
#else
int vapi_example(char *args)
#endif
{
	int result;
	int i;

#if !defined(_VXWORKS_)
	if (argc < 2)
	{
		PDEBUG(DEBUG_ERROR, "Please enter at least 1 argument");
		goto err0;
	}

	/* returns value < 0 on error, == 0 if all's Ok */
	result = get_opt(argc, argv);

	if (result < 0)
		goto err1;

	atexit(close_vapi_on_exit);
#else
	if (!args || !*args)
	{
		PDEBUG(DEBUG_ERROR, "Please enter at least 1 argument");
		goto err1;
	}

	/* returns value < 0 on error, > 1 on help or version request, == 0 if all's Ok */
	result = argp_parse(&argp, args);

	if (result > 0)
		goto ok;
	if (result < 0)
		goto err1;
#endif	

	/* The VAPI default parameters are not convenient for us
	 * we overwrite some of them. This must be done before VAPI_Init()
	 */
	pfnUsrQueryVapiDefaultsEnumerate = set_default_config;

	/* Setting device parameters - this also must be done before VAPI_Init() */
	set_device_configuration(device_id, device_itf);

	/* keeping this call early, since we using VAPI logging facility */
	result = VAPI_Init(&gtl_device_configuration);
	if (result != SUCCESS)
	{
		/* VAPI init failed - we can't use VAPI logging */
		fprintf(stderr, "VAPI_Init: failed, result = %d\n", result);
		goto err0;
	}

	/* set global variables default values */
	number_of_endpoints = 2;
	device_id = 0;

	/* Initialise the endpoint info array */
	memset(&endpoints, 0, sizeof(endpoints));

	for (i = 0; i < number_of_endpoints; i++)
	{
		endpoints[i].index = i;
		/* timeslot used by create_endpoint */
		endpoints[i].timeslot = i;

		/* give a different port number to each endpoints in the array
		 * the RTP port must be even
		 */
		endpoints[i].rtp_port = DEFAULT_UDP_PORT + i * 2;
		/* the RTCP port must be odd */
		endpoints[i].rtcp_port = DEFAULT_UDP_PORT + i;
		PDEBUG(DEBUG_FUNCTION, "endpoints[%d].rtp_port %d", i, endpoints[i].rtp_port);

		/* Cross the endpoints 2 by 2
		 * (endpoint 0 -> endpoint 1)
		 * (endpoint 1 -> endpoint 0)
		 */
		if (i % 2)
		{
			endpoints[i].peer_index = i - 1;
			endpoints[i - 1].peer_index = i;
		}
	}

	info("%s", VAPI_GetVersion());

	result = VAPI_OpenDevice(device_id, NULL);
	check_status("VAPI_OpenDevice", err1);

	if (*firmware)
	{
		PDEBUG(DEBUG_INIT, "Booting device...");
		result = boot_device(device_id, firmware);
		check_status("VAPI_BootDevice", err2);
	}

	result = tdm_parameters_initialisation(device_id);
	check_status("TDM parameters initialisation", err2);

	result = VAPI_InitDevice(device_id, VAPI_DEV_OPMODE_DEFAULT, VAPI_DEV_PROF_DEFAULT, NULL, NULL);
	check_status("VAPI_InitDevice", err2);

	result = network_parameters_initialisation(device_id);
	PDEBUG(DEBUG_ERROR, "Network parameters initialisation %s", result == SUCCESS ? "ok" : "failed");

	info("The device has been initialized using VAPI in blocking mode (Synchronous)");

#if !defined(_VXWORKS_)
	info("Enter a key to create %d endpoints in Asynchonous mode", number_of_endpoints);
	info("The %d endpoints will be created, configured with IP headers, then enabled", number_of_endpoints);
	getchar();
#else
	info("The %d endpoints will be created, configured with IP headers, then enabled", number_of_endpoints);
#endif

	for (i = 0; i < number_of_endpoints; i++)
	{
		/* odds endpoints are terminate side, even endpoints are originate side */
		if (i % 2)
		{
			set_endpoint_state(ENDPOINT_STATE_WAIT_CREATE_CONNECTION_TERMINATE_ACK, i);
			result =
			    create_voip_endpoint(device_id, i, ASYNC,
						 ENDPOINT_STATE_WAIT_CREATE_CONNECTION_TERMINATE_ACK);
		}
		else
		{
			set_endpoint_state(ENDPOINT_STATE_WAIT_CREATE_CONNECTION_ORIGINATE_ACK, i);
			result =
			    create_voip_endpoint(device_id, i, ASYNC,
						 ENDPOINT_STATE_WAIT_CREATE_CONNECTION_ORIGINATE_ACK);
		}
	}

	state_machine_handler();

	result = VAPI_CloseDevice(device_id, ePURGE);
	check_status("VAPI_CloseDevice", err1);

#if defined(_VXWORKS_)
/* under VxWorks we close VAPI here, under Linux we close it in atexit() handler */
      ok:
	result = VAPI_Close();
	fprintf(stderr, "VAPI_Close: %s\n", result == SUCCESS ? "ok" : "failed");
#endif

	return 0;

      err2:
	VAPI_CloseDevice(device_id, ePURGE);

      err1:
	VAPI_Close();

      err0:
	return FAILURE;
}
