/*! \file vapi_app_utility.c */
/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include "vxcompat.h"

#if defined(_VXWORKS_)
#include <msp.h>
#else
#include <vapi/msp.h>
#endif

#include <string.h>

#include "vapi_app_type.h"
#include "vapi_app_func.h"

static U16 max_timeslots;

/* these function codes are missing in msp.h */
#define FC_GET_ARM_CODE_VERSION 	UT_CPU2LE16(0x0021)
#define FC_GET_SPU_CODE_VERSION 	UT_CPU2LE16(0x0022)
#define FC_GET_DEVICE_TYPE 		UT_CPU2LE16(0x0031)

/* default values used later in default_csme_device_configuration */

/*! device control MAC used at boot stage
 *
 * On Slave configuration this address is set by the MAAS_ASSIGN command (VAPi_AssignBootMAC)
 * On Master configuration this address must be the same as the on set by the bootloader (u-boot, ARMboot),
 * the default value is 00:11:22:33:44:55
 */
#define DEVICE_CONTROL_MAC		{0x00,0x11,0x22,0x33,0x44,0x55}

/*! host control interface
 *
 * This defines the host network interface to use to control the device over ethernet (CSME protocol)
 */
#define HOST_CONTROL_INTERFACE		"eth0"

/* This defines the host pci interface to use to control the device over PCI */
#define PCI_HOST_CONTROL_INTERFACE	"/dev/comcerto0"

/*! device MAC for RTP traffic
 *
 * This MAC address will be used in the IP/UDP/RTP (UDPTL) frame as the source MAC address.
 * This MAC address must be unique.
 * On master configuration this MAC address must be the same as the one defined for eth0
 */
U8 device_mac[6] = { 0x00, 0x50, 0xFC, 0xC3, 0xBD, 0xE2 };

/*! default destination MAC address for the RTP/UDPTL traffic
 *
 * This can be the MAC adrress of an equipment on the subnet having IP forward capability
 * (PC, Router, etc).
 * This destination MAC is global for the device and all channnels, unless
 * it is overwritten at channel level by the SET_ETH_HDR_CHAN (VAPI_SetEthMac channel level call)
 */
U8 default_dest_mac[6] = { 0x00, 0x11, 0xD8, 0xE7, 0xA5, 0x52 };

/*! IP address used as the IP source for the RTP/UDPTL traffic
 *
 * This IP address must be unique
 * On master configuration this IP address must be the same as the one defined
 * for eth0
 */
U8 device_ip[4] = { 192, 168, 0, 5 };

/*! This structure holds default values for the device configuration, user can
 * overwrite them somehow in set_device_configuration() function using his own
 * config. scheme.
 */
SCSMEUsrData default_csme_device_configuration = {
	NULL,				/**< pointer to next device */
	eCSM_ITF,			/**< control interface to use */
	0,				/**< device ID */
	0,				/**< Version infor (not used) */
	eSLAVE,				/**< Slave / Master mode */
	DEV_TYPE_M823XX,		/**< Device type */
	True,				/**< Default or custom max channels */
	0,				/**< Max Channels if above flag is custom */
	DEVICE_CONTROL_MAC,		/**< MAC address for control over csmencaps */
	{0x00,0x00,0x00,0x00,0x00,0x00},/**< MAC address of the host interface controling the device */
	(char *)HOST_CONTROL_INTERFACE,	/**< host interface used to control the device */
	1				/**< csme ack required */
};

SPCIUsrData default_pci_device_configuration = {
	.pvNext = NULL,			/**< pointer to next device */
	.usControlInterface = ePCI_ITF,	/**< control interface to use */
	.uiDevId = 0,			/**< device ID */
	.uiVersion = 0,			/**< Version infor (not used) */
	.eDevMode = eSLAVE,		/**< Slave / Master mode */
	.ucDevType = DEV_TYPE_M823XX,	/**< Device type */
	.bUseDefaultMaxChnls = True,	/**< Default or custom max channels */
	.usMaxChannels = 0,		/**< Max Channels if above flag is custom */
	.pucPCIDevName = (char *)PCI_HOST_CONTROL_INTERFACE,	/**< host interface used to control the device */
};

static void print_memory_dump(int type, void *ptr, unsigned int len_in_bytes, int radix);
static void print_ip_addr(const char *prefix, U8 * ip_addr);

static int m823xx_refclk = 10;
static int m823xxv2_amba_clk = 137;

/*=================================================================================*/
/*!
 *	\brief This function fills the global structure with device config parameters
 */
/*=================================================================================*/
void set_device_configuration(U16 device_id, int device_ctrl_itf)
{
	U8 *mac;

	if (device_ctrl_itf == eCSM_ITF)
	{
		memcpy(&gtl_device_configuration, &default_csme_device_configuration, sizeof(SCSMEUsrData));
	}
	else if (device_ctrl_itf == ePCI_ITF)
	{
		memcpy(&gtl_device_configuration, &default_pci_device_configuration, sizeof(SPCIUsrData));
	}
	else
	{
		printf("set_device_configuration: Interface control type %d not supported", device_ctrl_itf);
		return;
	}

	/* user may put here code to overwrite default config parameters, for example
	 * read them from config file */
	gtl_device_configuration.uiDevId = device_id;

	/* this code is executed before VAPI_Init, so we have to use printf's */
	printf("CONTROL_INTERFACE_TYPE  = %d\n", gtl_device_configuration.usControlInterface);
	printf("DEVICE_ID    = %d\n", gtl_device_configuration.uiDevId);
	printf("DEVICE_MODE  = %d\n", gtl_device_configuration.eDevMode);
	printf("DEVICE_TYPE  = %d\n", gtl_device_configuration.ucDevType);
	printf("USE_DEFAULT_MAX_CHANNEL = %d\n", gtl_device_configuration.bUseDefaultMaxChnls);
	printf("CUSTOM_MAX_CHANNEL      = %d\n", gtl_device_configuration.usMaxChannels);

	if (device_ctrl_itf == eCSM_ITF)
	{
		printf("ACK_REQUIRED = %d\n", ((SCSMEUsrData *) &gtl_device_configuration)->ucIsAckReqd);

		UT_GetInterfaceMac(((SCSMEUsrData *) &gtl_device_configuration)->pucEthDevName, 
					(char *)((SCSMEUsrData *) &gtl_device_configuration)->aucHostMac);

		mac = ((SCSMEUsrData *) &gtl_device_configuration)->aucDevMac;
		printf("DEVICE MAC = %02x:%02x:%02x:%02x:%02x:%02x\n", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

		mac = ((SCSMEUsrData *) &gtl_device_configuration)->aucHostMac;
		printf("HOST   MAC = %02x:%02x:%02x:%02x:%02x:%02x\n", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

		 ((SCSMEUsrData *) &gtl_device_configuration)->pucEthDevName = HOST_CONTROL_INTERFACE;
		printf("HOST_CONTROL_INTERFACE  = %s\n\n", ((SCSMEUsrData *) &gtl_device_configuration)->pucEthDevName);
	}
	else
	{
		printf("HOST_CONTROL_INTERFACE  = %s\n\n", ((SPCIUsrData *) &gtl_device_configuration)->pucPCIDevName);
	}
}

/*=================================================================================*/
/*!
 * \brief This function change the default VAPI configuration.
 */
/*=================================================================================*/
VSTATUS set_default_config(IN OUT SVAPIConfig * default_config)
{
	struct _VOIP_DTMFOPT *my_dtmf_options = &(default_config->pstVoIPChnlParams->stDtmfOpt);
	/* change the default timeout to 10S */
	default_config->uiMspRespTimeout = 10; /* MSP response timeout value 10 seconds */

	/* change the default VAPi DTMF options */
	my_dtmf_options->param_4.bits.event_report = VOIP_DTMFOPT_EVENT_REPORT_DEFAULT;
	my_dtmf_options->param_4.bits.regeneration = VOIP_DTMFOPT_REGENERATION_DEFAULT;
	my_dtmf_options->param_4.bits.removal = VOIP_DTMFOPT_REMOVAL_DEFAULT;
	my_dtmf_options->param_4.bits.redundancy = VOIP_DTMFOPT_REDUNDANCY_DEFAULT;
	my_dtmf_options->param_4.bits.dtmf_rtp = VOIP_DTMFOPT_DTMF_RTP_DEFAULT;
	my_dtmf_options->param_4.bits.dtmf_voice = VOIP_DTMFOPT_DTMF_VOICE_DEFAULT;

	return SUCCESS;
}

/*=================================================================================*/
/*!
 *	\brief This function reads the firmware (.axf or .elf) and store it to a buffer.
 */
/*=================================================================================*/
static int comcerto_read_firmware_file(char *filename, unsigned char **buf, int *size)
{
	FILE *fp;

	PDEBUG(DEBUG_FUNCTION, "trying to load '%s'", filename);
	fp = fopen(filename, "r");

	if (fp == NULL)
	{
		perror(filename);
		goto err0;
	}

	/* Figure out how big the size of the file and allocate that much space in memory */
	fseek(fp, 0, SEEK_END);
	*size = ftell(fp);

	*buf = (unsigned char *)malloc(*size);
	if (*buf == NULL)
	{
		PDEBUG(DEBUG_ERROR, "can't allocate memory");
		goto err1;
	}

	fseek(fp, 0, SEEK_SET);

	if (fread(*buf, sizeof(unsigned char), *size, fp) != *size)
	{
		PDEBUG(DEBUG_ERROR, "error reading '%s'", filename);
		goto err2;
	}

	fclose(fp);

	return 0;

      err2:
	free(*buf);

      err1:
	fclose(fp);

      err0:
	return -1;
}

/*=================================================================================*/
/*!
 *	\brief This function boots the device by calling:
 *		- comcerto_read_firmware_file to read the firmware file
 *		- VAPI_AssignBootMAC
 *		- VAPI_BootDevice
 */
/*=================================================================================*/
int boot_device(U16 device_id, const char *firmware)
{
	int result;
	unsigned char *firmware_buf = NULL;
	int firmware_size;

	SHwCfg hw_cfg, *hw_cfg_param = NULL;


	memset(&hw_cfg, 0, sizeof(hw_cfg));

	/* specify refclk for the C300 EVM */
	if ((gtl_device_configuration.ucDevType == DEV_TYPE_M823XX) ||
		(gtl_device_configuration.ucDevType == DEV_TYPE_M823XX_2))
	{
		hw_cfg.usDeviceRefclk = m823xx_refclk;
		hw_cfg.usHwParamsMask = VAPI_HWCFG_REFCLK_MASK;

		hw_cfg_param = &hw_cfg;
		PDEBUG(DEBUG_ERROR, "C300 EVM REFCLK: %dMHz", m823xx_refclk);
	}

	/* specify AMBA clk for the C300V2 EVM */
	if (gtl_device_configuration.ucDevType == DEV_TYPE_M823XX_2)
	{
		hw_cfg.usDeviceAmbaClkM823V2 = m823xxv2_amba_clk;
		hw_cfg.usHwParamsMask |= VAPI_HWCFG_M823V2_HIGH_SPEED_MASK;

		hw_cfg_param = &hw_cfg;
		PDEBUG(DEBUG_ERROR, "C300V2 EVM AMBA CLK: %dMHz", m823xxv2_amba_clk);
	}

	/* firmware filename comes from the -f option */
	result = comcerto_read_firmware_file((char *)firmware, &firmware_buf, &firmware_size);
	if (result)
		goto err0;

	info("firmware: '%s', size: %d", firmware, firmware_size);

	/* Send MAAS_ASSIGN only for CSME*/
	if (gtl_device_configuration.usControlInterface == eCSM_ITF)
	{
		/* the device_id comes from the -d option */
		result = VAPI_AssignBootMAC(device_id, NULL);

		if (result != SUCCESS)
		{
			PDEBUG(DEBUG_ERROR, "VAPI_AssignBootMAC failed, status: %d", result);
			goto err1;
		}
	}

	if (firmware_size > 0 && firmware_buf != NULL)
	{	
/*! \cond */
#if VAPI_RELEASE >= (VAPI_VERSION(2, 6, 0))
		result = VAPI_BootDevice(device_id, firmware_buf, firmware_size, hw_cfg_param, NULL);
#else
		result = VAPI_BootDevice(device_id, firmware_buf, firmware_size, NULL);
#endif
/*! \endcond */
		if (result != SUCCESS)
		{
			PDEBUG(DEBUG_ERROR, "VAPI_BootDevice failed, status: %d", result);
			goto err1;
		}
	}

	if (firmware_buf)
		free(firmware_buf);

	return 0;

      err1:
	if (firmware_buf)
		free(firmware_buf);
      err0:
	return -1;
}

/*=================================================================================*/
/*!
 *	\brief This function initializes the TDM buses.
 *	\note the 4 TDM buses have the same configurations.
*/
/*=================================================================================*/
int tdm_parameters_initialisation(U16 device_id)
{
	STdmSetupParams tdm_parameters;
	int i, result;

	/* zero all structure, to save some lines of initialization code */
	memset(&tdm_parameters, 0, sizeof(tdm_parameters));

	if (gtl_device_configuration.ucDevType == DEV_TYPE_M823XX)
	{
		tdm_parameters.usMode = 2; /* two-bus mode */
		tdm_parameters.usNoOfBus = 2;
	}
	else
	{
		tdm_parameters.usMode = 4; /* four-bus mode */
		tdm_parameters.usNoOfBus = 4;
	}

	if (tdm_parameters.usNoOfBus == 0 || tdm_parameters.usNoOfBus > MAX_TDM_BUS)
	{
		PDEBUG(DEBUG_ERROR, "Error in number of TDM buses to enable, valid range: 1..%u", MAX_TDM_BUS);
		return -1;
	}

	tdm_parameters.astTdmBusParam[0].ucTxOrder = 1;
	tdm_parameters.astTdmBusParam[0].ucTxClkEdge = 1;
	tdm_parameters.astTdmBusParam[0].ucFrameEdge = 1;
	tdm_parameters.astTdmBusParam[0].usNumOfTS = 128;
	tdm_parameters.astTdmBusParam[0].ucInvertedFrmSig = 1;
	tdm_parameters.astTdmBusParam[0].ucBitOrderRcv = 1;

	/* propagate the same parameters for the rest of buses */
	for (i = 1; i < MAX_TDM_BUS; i++)
	{
		memcpy(&tdm_parameters.astTdmBusParam[i], &tdm_parameters.astTdmBusParam[0], sizeof(STdmBusParams));

		/* NOTE: fix bus id, if you provide non-zero ::ucBusId for the first bus
		 * in the code above, you probably will need to manually set up ids
		 */
		tdm_parameters.astTdmBusParam[i].ucBusId = i;
	}

	/* counting the max. number of timeslots over the all buses */
	for (i = 0; i < tdm_parameters.usNoOfBus; i++)
		max_timeslots += tdm_parameters.astTdmBusParam[i].usNumOfTS;

	result = VAPI_SetTDMParams(device_id, &tdm_parameters, NULL);
	if (result != SUCCESS)
		PDEBUG(DEBUG_ERROR, "VAPI_SetTDMParams failed, status: %d", result);

	return result;
}

/*=================================================================================*/
/*!
 *	\brief
 *	This function set the NTG (Network Timming Generator) of the M823xx device
 */
/*=================================================================================*/
VSTATUS set_ntg_config(U16 device_id)
{
	VSTATUS result;
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/* allocate a message to query the current options */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;

	result = VAPI_SetMessage(message, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_NTG_CONFIG, 2, 0x0001, 0x0006);
	if(result != SUCCESS)
		goto err;

	/* send the command, the response is stored in device_response*/
	result = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, device_response, &response_len);

err:
	VAPI_FreeMessage(message);
	return result;
}

/*=================================================================================*/
/*!
 *	\brief This function initializes the network parameters of the device by calling:
 *		- VAPI_SetEthMac
 *		- VAPI_SetDeviceIPAddr
 */
/*=================================================================================*/
int network_parameters_initialisation(U16 device_id)
{
	int result;
	SIpAddrInfo ip_info;

	print_ip_addr("DEVICE IPv4 = ", device_ip);
	print_mac_addr("DEVICE MAC  = ", device_mac);

	result = VAPI_SetEthMac(device_id, CMD_LEVEL_DEVICE, device_mac, default_dest_mac, NULL);
	if (result != SUCCESS)
		PDEBUG(DEBUG_ERROR, "VAPI_SetEthMac failed, status: %d", result);

	if (result == SUCCESS)
	{
		ip_info.ucNumOfSrcIpAddr = 1;
		ip_info.bIsMultipleMode = 0;
		ip_info.ucEncapsulation = 2;
		ip_info.ucChkSumVerification = 1;

		memcpy(ip_info.auiDevIPAddress, device_ip, 4);

		result = VAPI_SetDeviceIPAddr(device_id, &ip_info, NULL);
		if (result != SUCCESS)
			PDEBUG(DEBUG_ERROR, "VAPI_SetDeviceIPAddr failed, status: %d", result);
	}

	/*Configure the Network Timming Generator for the M823XX
	This is mandatory on the Mindspeed EVM*/
	if (gtl_device_configuration.ucDevType == DEV_TYPE_M823XX)
	{
		result = set_ntg_config(device_id);

		if (result != SUCCESS)
			PDEBUG(DEBUG_ERROR, "set_ntg_config failed, status: %d", result);
	}

	return result;
}

/*=================================================================================*/
/*!
 *	\brief This function set the IP/UDP header for the VoIP channel by calling VAPI_SetConnIpParams.
 */
/*=================================================================================*/
int set_ip_udp_parameters(U8 * device_ip, int endpoint_id, int mode, int endpoint_state)
{
	int result;
	SRequest request;
	SIpParams ip_parameters;

	memset(&ip_parameters, 0, sizeof(SIpParams));
	request.uiReqId = endpoint_state;

	if (mode == ASYNC)
		request.pfnIoCompCallback = &comcerto_response_handler;
	else
		request.pfnIoCompCallback = NULL;

	PDEBUG(DEBUG_INIT, "set ip parameters:%d (request id = 0x%04x)", endpoint_id, request.uiReqId);

	/* same ip address for source and destination */
	memcpy((void *)&ip_parameters.uiDestIpAddr, device_ip, 4);
	memcpy((void *)&ip_parameters.uiSrcIpAddr, device_ip, 4);

	/* cross the UDP ports */
	ip_parameters.usSrcUdpPort = htons(endpoints[endpoint_id].rtp_port);
	ip_parameters.usDestUdpPort = htons(endpoints[endpoints[endpoint_id].peer_index].rtp_port);
	PDEBUG(DEBUG_FUNCTION, "endpoints[%d].src rtp_port %d", endpoint_id, ip_parameters.usSrcUdpPort);
	PDEBUG(DEBUG_FUNCTION, "endpoints[%d].dst rtp_port %d", endpoint_id, ip_parameters.usDestUdpPort);

	/*RTP defined in comcerto_ip_device_level_api.h */
	ip_parameters.ucIPServiceId = SET_IP_HDR_CHANNEL_SERVICEID_DEFAULTPKT;

	result = VAPI_SetConnIpParams(endpoint_id, &ip_parameters, &request);

	return result;
}

/*=================================================================================*/
/*!
 *	\brief This function starts the RTP traffic by calling VAPI_EnableConnection.
 */
/*=================================================================================*/
int start_voip_endpoint(int endpoint_id, int mode, int endpoint_state)
{
	int result;
	SRequest request;

	request.uiReqId = endpoint_state;

	if (mode == ASYNC)
		request.pfnIoCompCallback = &comcerto_response_handler;
	else
		request.pfnIoCompCallback = NULL;

	PDEBUG(DEBUG_INIT, "Start connection on endpoint:%d (request id = 0x%04x)", endpoint_id, request.uiReqId);

	result = VAPI_EnableConnection(endpoint_id, &request);

	return result;
}

/*=================================================================================*/
/*!
 *	\brief This function creates a endpoint by calling VAPI_CreateConnection
 *	The coding law is forced to ulaw, the timeslot used comes from the endpoints[endpoint_id].timeslot variable \n
 *	initialised in the main function.
 */
int create_voip_endpoint(U16 device_id, int endpoint_id, int mode, int endpoint_state)
{
	int result;
	SRequest request;

	request.uiReqId = endpoint_state;

	if (mode == ASYNC)
		request.pfnIoCompCallback = &comcerto_response_handler;
	else
		request.pfnIoCompCallback = NULL;

	PDEBUG(DEBUG_INIT, "create VoIP channel: %d type %d timeslot %d (request id = 0x%04x)", endpoint_id, eVOIP,
	       endpoint_id, request.uiReqId);

	if (endpoints[endpoint_id].timeslot > max_timeslots)
	{
		PDEBUG(DEBUG_INIT, "Timeslot %d out of range", endpoints[endpoint_id].timeslot);
		return -1;
	}

	result = VAPI_CreateConnection(device_id, endpoint_id,	/* use the endpoint_id as connection_id */
				       eVOIP,	/* type 2 */
				       endpoints[endpoint_id].timeslot, U_LAW, &request, NULL);

	return result;
}

/*=================================================================================*/
/*!
 *	\brief This function destroys a endpoint by calling VAPI_DestroyConnection
 */
/*=================================================================================*/
int destroy_endpoint(int endpoint_id, int mode, int endpoint_state)
{
	int result;
	SRequest request;

	request.uiReqId = endpoint_state;

	if (mode == ASYNC)
		request.pfnIoCompCallback = &comcerto_response_handler;
	else
		request.pfnIoCompCallback = NULL;

	PDEBUG(DEBUG_INIT, "destroy VoIP channel:%d  (request id = 0x%04x)", endpoint_id, request.uiReqId);

	result = VAPI_DestroyConnection(endpoint_id, &request);

	return result;
}

/*=================================================================================*/
/*!
 *	\brief This function displays the :
 *		- Fimware version
 *		- DSP version
 *		- device type
 */
/*=================================================================================*/
void display_firmware_version(U16 device_id)
{
	VSTATUS result;
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/* allocate a message to query the current options */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return;
	result = VAPI_SetMessage(message, CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY, FC_SUPVSR_GET_ARM_CODE_VERSION, 0);

	/* send the command, the response is stored in device_response*/
	if(result == SUCCESS)
	{
		result = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, device_response, &response_len);

		if(result == SUCCESS)
		{
			PDEBUG(DEBUG_FUNCTION, "FC_GET_ARM_CODE_VERSION: response %u bytes: ", response_len);
			print_memory_dump(DEBUG_FUNCTION, device_response, response_len, 8);
		}
	}

	VAPI_InitMessage(message);

	result = VAPI_SetMessage(message, CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY, FC_SUPVSR_GET_SPU_CODE_VERSION, 0);

	/* send the command, the response is stored in device_response*/
	if(result == SUCCESS)
	{
		/*re-initialize the len to MAX (it has been changed by the previous command*/
		response_len = DEFAULT_FIFO_MAX_SIZE;
		result = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, device_response, &response_len);

		if(result == SUCCESS)
		{
			PDEBUG(DEBUG_FUNCTION, "FC_GET_SPU_CODE_VERSION: response %u bytes: ", response_len);
			print_memory_dump(DEBUG_FUNCTION, device_response, response_len, 8);
		}
	}


	VAPI_InitMessage(message);

	result = VAPI_SetMessage(message, CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY, FC_SUPVSR_GET_DEVICE_TYPE, 0);

	/* send the command, the response is stored in device_response*/
	if(result == SUCCESS)
	{
		/*re-initialize the len to MAX (it has been changed by the previous command*/
		response_len = DEFAULT_FIFO_MAX_SIZE;
		result = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, device_response, &response_len);

		if(result == SUCCESS)
		{
			PDEBUG(DEBUG_FUNCTION, "FC_GET_DEVICE_TYPE: response %u bytes: ", response_len);
			print_memory_dump(DEBUG_FUNCTION, device_response, response_len, 8);
		}
	}

	VAPI_FreeMessage(message);
}

static void print_memory_dump(int type, void *ptr, unsigned int len, int radix)
{
	int i, buflen = len * 3;
	char *buf, *s;

	if (!type)
		return;

	if (radix == 32)
		len >>= 2, buflen = len * 9;
	else if (radix == 16)
		len >>= 1, buflen = len * 5;
	else if (radix != 8)
	{
		PDEBUG(DEBUG_ERROR, "invalid radix: %d", radix);
		return;
	}

	if (!len)
		return;

	buf = malloc(buflen + 1);

	if (!buf)
	{
		PDEBUG(DEBUG_ERROR, "memory allocation failed!");
		return;
	}

	s = buf;
	for (s = buf, i = 0; i < len; i++)
	{
		if (radix == 32)
			s += sprintf(s, "%08x", ((unsigned int *)ptr)[i]);
		else if (radix == 16)
			s += sprintf(s, "%04x", ((unsigned short *)ptr)[i]);
		else
			s += sprintf(s, "%02x", ((unsigned char *)ptr)[i]);

		s += sprintf(s, "%c", (i < len - 1) ? ' ' : '\n');
	}

	info("%s", buf);	/* we can't pass pointer to string */
	free(buf);
}

static void print_ip_addr(const char *prefix, U8 * ip_addr)
{
	info("%s%u.%u.%u.%u", prefix ? prefix : "", ip_addr[0], ip_addr[1], ip_addr[2], ip_addr[3]);
}

void print_mac_addr(const char *prefix, U8 * mac_addr)
{
	info("%s%02x:%02x:%02x:%02x:%02x:%02x",
	     prefix ? prefix : "", mac_addr[0], mac_addr[1], mac_addr[2], mac_addr[3], mac_addr[4], mac_addr[5]);
}
