/*!
*	\file tdmif.c
*
*	\defgroup tdmif tdmif (TDM Interface Module)
*
*	TDM Interface Module (tdmif) including all C-functions and declarations
*	(defines, types, variables)
*
*	\attention
*	Copyright © 2004-2010 Mindspeed Technologies, Inc.\n
*	Mindspeed Confidential.\n
*	All rights reserved.\n
*	This file is a component of the Mindspeed® VAPI software ("VAPI") and is
*	distributed under the Mindspeed Software License Agreement (the "Agreement").\n
*	Before using this file, you must agree to be bound by the the terms and 
*	conditions of the Agreement.
*
*	@{
*/

#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <string.h>

/* VAPI headers */
#include <msp.h>
#include <vapi.h>
#include <gtl.h>
#include <comcerto-api-defs.h>
#include <comcerto-ud-types.h>

/* Zarlink headers */
#include <legerity_lib.h>

/* mtalk header */
#include <mtalk.h>


/*!
*	This function allocates and initializes a tdmif_cfg instance with 
*	hardcoded and/or user defined values.
*	\param *cfg_file Configuration filename (string).
*	\retval NULL on failure
*	\retval STdmSetupParams* on success, pointer to the allocated and initialized tdmif_cfg structure
*/
STdmSetupParams *get_tdmif_cfg(const char *cfg_file)
{
	struct _CFG *cfg_info;
	STdmSetupParams *tdmif_cfg;
	int i;

	tdmif_cfg = (STdmSetupParams *)malloc(sizeof(STdmSetupParams));
	exit_if_null(tdmif_cfg, err, "get_tdmif_cfg() - mem allocation fail");

	/* open the configuration file*/
	cfg_info = cfg_read(cfg_file, 0);
	exit_if_null(cfg_info, err, "get_tdmif_cfg() - mem allocation fail");

	READ_INT(cfg_info, "TDMIF", "TDM_BUS_MODE", tdmif_cfg->usMode, 2);

	READ_INT(cfg_info, "TDMIF", "TDM_BUS_TO_ENABLE", tdmif_cfg->usNoOfBus, 2);
 
	READ_INT(cfg_info, "TDMIF", "BUS_ID", tdmif_cfg->astTdmBusParam[0].ucBusId, 0);
	READ_INT(cfg_info, "TDMIF", "TX_ORDER", tdmif_cfg->astTdmBusParam[0].ucTxOrder, 1);
	READ_INT(cfg_info, "TDMIF", "RX_POLARITY", tdmif_cfg->astTdmBusParam[0].ucRxPolarity, 0);
	READ_INT(cfg_info, "TDMIF", "TX_POLARITY", tdmif_cfg->astTdmBusParam[0].ucTxPolarity, 0);
	READ_INT(cfg_info, "TDMIF", "RX_CLOCK_MODE", tdmif_cfg->astTdmBusParam[0].ucRxClkMode, 0);
	READ_INT(cfg_info, "TDMIF", "TX_CLOCK_MODE", tdmif_cfg->astTdmBusParam[0].ucTxClkMode, 0);
	READ_INT(cfg_info, "TDMIF", "RX_CLOCK_EDGE", tdmif_cfg->astTdmBusParam[0].ucRxClkEdge, 0);
	READ_INT(cfg_info, "TDMIF", "TX_CLOCK_EDGE", tdmif_cfg->astTdmBusParam[0].ucTxClkEdge, 1);
	READ_INT(cfg_info, "TDMIF", "FRAME_EDGE", tdmif_cfg->astTdmBusParam[0].ucFrameEdge, 0);
	READ_INT(cfg_info, "TDMIF", "NUMBER_OF_TS", tdmif_cfg->astTdmBusParam[0].usNumOfTS, 128);
	READ_INT(cfg_info, "TDMIF", "INVERTED_FRAME_SIGNAL", tdmif_cfg->astTdmBusParam[0].ucInvertedFrmSig, 0);
	READ_INT(cfg_info, "TDMIF", "BIT_ORDER_RCV", tdmif_cfg->astTdmBusParam[0].ucBitOrderRcv, 1);
	READ_INT(cfg_info, "TDMIF", "NUM_TDM_CLK_CYCLES_DELAY_TX", tdmif_cfg->astTdmBusParam[0].usNumTdmClkCyclesDelayTx, 0);
	READ_INT(cfg_info, "TDMIF", "NUM_TDM_CLK_CYCLES_DELAY_RX", tdmif_cfg->astTdmBusParam[0].usNumTdmClkCyclesDelayRx, 0);

	cfg_clean(cfg_info);

	/* uses the same parameters for all busses */
	for (i = 1; i < tdmif_cfg->usNoOfBus; i++)
		memcpy(&tdmif_cfg->astTdmBusParam[i], &tdmif_cfg->astTdmBusParam[0], sizeof(STdmBusParams));

	return tdmif_cfg;

err:
	return NULL;
}


/*!
*	This function displays the values of tdmif parameters (tdmif_cfg).
*	\param *tdmif_cfg Pointer to the tdmif configuration parameter structure.
*	\retval None
*/
void display_tdmif_cfg(STdmSetupParams *tdmif_cfg)
{
	int i;

	if (!tdmif_cfg)
		return;

	PDEBUG(DBG_L2, "\n-------------------- tdmif_cfg");

	PDEBUG(DBG_L2, " - usMode (TDM bus mode)     %d", tdmif_cfg->usMode);
	PDEBUG(DBG_L2, "- usNoOfBus                 %d", tdmif_cfg->usNoOfBus);

	for (i = 0; i < tdmif_cfg->usNoOfBus; i++)
	{
		PDEBUG(DBG_L2, "------- TDM  Bus %d", i);
		PDEBUG(DBG_L2, "- ucTxOrder                 %d", tdmif_cfg->astTdmBusParam[i].ucTxOrder);
		PDEBUG(DBG_L2, "- ucRxPolarity              %d", tdmif_cfg->astTdmBusParam[i].ucRxPolarity);
		PDEBUG(DBG_L2, "- ucTxPolarity              %d", tdmif_cfg->astTdmBusParam[i].ucTxPolarity);
		PDEBUG(DBG_L2, "- ucRxClkMode               %d", tdmif_cfg->astTdmBusParam[i].ucRxClkMode);
		PDEBUG(DBG_L2, "- ucTxClkMode               %d", tdmif_cfg->astTdmBusParam[i].ucTxClkMode);
		PDEBUG(DBG_L2, "- ucRxClkEdge               %d", tdmif_cfg->astTdmBusParam[i].ucRxClkEdge);
		PDEBUG(DBG_L2, "- ucTxClkEdge               %d", tdmif_cfg->astTdmBusParam[i].ucTxClkEdge);
		PDEBUG(DBG_L2, "- ucFrameEdge               %d", tdmif_cfg->astTdmBusParam[i].ucFrameEdge);
		PDEBUG(DBG_L2, "- usNumOfTS                 %d", tdmif_cfg->astTdmBusParam[i].usNumOfTS);
		PDEBUG(DBG_L2, "- ucInvertedFrmSig          %d", tdmif_cfg->astTdmBusParam[i].ucInvertedFrmSig);
		PDEBUG(DBG_L2, "- ucBitOrderRcv             %d", tdmif_cfg->astTdmBusParam[i].ucBitOrderRcv);
		PDEBUG(DBG_L2, "- usNumTdmClkCyclesDelayTx  %d", tdmif_cfg->astTdmBusParam[i].usNumTdmClkCyclesDelayTx);
		PDEBUG(DBG_L2, "- usNumTdmClkCyclesDelayRx  %d", tdmif_cfg->astTdmBusParam[i].usNumTdmClkCyclesDelayRx);
	}

	PDEBUG(DBG_L2, "--------------------\n");
}


/*!
*	This function displays the values of tdmif parameters (tdmif_cfg).
*	\param *tdmif_cfg Pointer to the tdmif configuration parameter structure.
*	\retval 0 if success
*	\retval <0 if failure
*       \see VAPI_SetTDMParams
*/
int tdmif_init(STdmSetupParams *tdmif_cfg)
{
	int rc = 0; /* return code */
	int i;

	/* uses the same parameters for all buses*/
	for (i = 1; i < tdmif_cfg->usNoOfBus; i++)
		memcpy(&tdmif_cfg->astTdmBusParam[i], &tdmif_cfg->astTdmBusParam[0], sizeof(STdmBusParams));


	/* FIXME: we should check the result */
	rc = VAPI_SetTDMParams(0, tdmif_cfg, NULL);

	/* Not really an issue ignore it this happens if we try to initialise the TDM several time */
	if (rc == -CNF_ERR_TDM_CONFIG_PARMS_FAILED_SELECT_MULT_INSTANCE)
		rc = SUCCESS; 

	return rc;
}


/*!	@} */
