/*!
*	\file slic.c
*
*	\defgroup slic slic Module
*
*	slic Module including all C-functions and declarations
*	(defines, types, variables)
*
*	\attention
*	Copyright © 2004-2010 Mindspeed Technologies, Inc.\n
*	Mindspeed Confidential.\n
*	All rights reserved.\n
*	This file is a component of the Mindspeed® VAPI software ("VAPI") and is
*	distributed under the Mindspeed Software License Agreement (the "Agreement").\n
*	Before using this file, you must agree to be bound by the the terms and 
*	conditions of the Agreement.
*
*	@{
*/

#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <string.h>

/* VAPI headers */
#include <msp.h>
#include <vapi.h>
#include <gtl.h>
#include <comcerto-api-defs.h>
#include <comcerto-ud-types.h>

/* Zarlink headers */
#include <legerity_lib.h>

/* mtalk header */
#include <mtalk.h>


/*=================================================================================*/
/*! \brief
*	
	This function handles the events reported by the signaling thread (SLIC events) such as offhook, onhokk, ring.
	The timeslot reported with the event is used as the endpoint index in the global endpoints array
*/
void * pots_event_handler(void * none)
{
	struct legerity_io_event_desc event;
	endpt_t *endpt;
	int rc;

	/* we should have here a state machine waiting for events */
	PDEBUG(DBG_L1, "Signaling SLIC event handler thread started");
	while (1)
	{
		rc = legerity_read_event(slic.dev_file, &event);

		if (rc != 0)
			goto sleep;

		switch (event.type)
		{
		case EVENT_ON_HOOK:
		case EVENT_OFF_HOOK:
			PDEBUG(DBG_L3, "Slic event = %d device = %d, line = %d", 
					event.type, event.device, event.line);

			endpt = &endpt_pool[event.device*2 + event.line];

			endpt_raise_event(endpt, event.type); /* post the event to the associated endpoint */
			break;

		default:
			PDEBUG(DBG_ERR, "Unexpected SLIC event = %d", event.type);
			break;
		}

sleep:
		usleep(100);
	}
}

/*=================================================================================*/
/*! \brief
*	
*	This function sets the 2 lines of a SLIC in wideband mode.*/
int slic_set_wide_band(int fd, int line1, int line2)
{
	int rc;

	PDEBUG(DBG_L1, "Switching to WB mode");
	rc = slic_write_regs(fd, line1, CODEC_REG_WRT, 1, 0x80);
	if (rc >= 0)
		PDEBUG(DBG_L1, "SUCCESS to set linear coding on %d line", line1);

	rc |= slic_write_regs(fd, line2, CODEC_REG_WRT, 1, 0x80);
	if (rc >= 0)
		PDEBUG(DBG_L1, "SUCCESS to set linear coding on %d line", line2);

	/* set appropriate SLIC mode*/
	rc |= slic_write_regs(fd,
			line1, /*line 1 or 2 doesn't matter: global param*/
			OPERATING_MODE_REGISTER, 1, 0x03 | VP880_WBAND_MODE_MASK);
	return rc;
}

/*=================================================================================*/
/*! \brief
*	
*	This function sets the 2 lines of a SLIC in narrow band mode.*/
int slic_set_narrow_band(int fd, int line1, int line2)
{
	int rc; /* return code */

	PDEBUG(DBG_L1, "Switching to NB mode");
	rc = slic_write_regs(fd, line1, CODEC_REG_WRT, 1, 0x40);
	if (rc >= 0)
		PDEBUG(DBG_L1, "SUCCESS to set ulaw coding on %d line", line1);

	rc |= slic_write_regs(fd, line2, CODEC_REG_WRT, 1, 0x40);
	if (rc >= 0)
		PDEBUG(DBG_L1, "SUCCESS to set ulaw coding on %d line", line2);

	/* set appropriate SLIC mode*/
	rc |= slic_write_regs(fd,
			line1, /*line 1 or 2 doesn't matter: global param*/
			OPERATING_MODE_REGISTER, 1, 1, 0x03 & ~VP880_WBAND_MODE_MASK);
	return rc;
}

/*=================================================================================*/
/*! \brief
*	This function is rings the POTS phone line associated to the specified endpt.
*/
void ring_pots(int pots_line_id)
{
	struct legerity_io_ring_desc ring;

	ring.line = pots_line_id;
	legerity_ring(slic.dev_file, &ring);
}

void stop_ring_pots(int pots_line_id)
{
	struct legerity_io_disconnect_desc disconnect;

	disconnect.line = pots_line_id;
	legerity_disconnect(slic.dev_file, &disconnect);
}

/*=================================================================================*/
/*! \brief
*	
*	This function returns the slic device type and version.*/
unsigned short get_slic_version(int fd)
{
	unsigned short slic_version = 0;
	int rc; /* return code */

	rc = slic_read_regs(fd, 1, REVISION_PRODUCT_CODE, 2, (unsigned char *) &slic_version);

	if (rc < 0)
	{
		PDEBUG(DBG_ERR, "!ERR! Fail to get SLIC device version");
	}
	else
	{
		PDEBUG(DBG_L1, "SLIC Product code : 0x%02x", (slic_version & 0xff00) >> 8);
		PDEBUG(DBG_L1, "SLIC Revision code : 0x%02x", (slic_version & 0x00ff));
	}

	return slic_version;
}

/*=================================================================================*/
/*! \brief
*	
*	This function returns the slic device type and version.*/
unsigned short is_slic_wideband(unsigned short slic_version)
{
	unsigned short is_wideband = False;

	is_wideband = False;

	if (((slic_version & 0xff00) >> 8) == VP880_DEV_PCN_88276)
		is_wideband = True;

	return is_wideband;
}


/*=================================================================================*/
/*! \brief
*	
*	This function returns the slic device type and version.*/
int slic_init(slic_t *slic)
{
	int rc = 0; /* return code */

	/* First try to open the access to the legerity driver (/dev/legerity) */
	slic->dev_file = legerity_open("/dev/legerity");

	if (slic->dev_file < 0)
	{
		PDEBUG(DBG_ERR, "!ERR! slic_init");
		goto exit;
	}

	/* get to know if current SLIC type supports WB mode */
	slic->version = get_slic_version(slic->dev_file);
	if (slic->version != 0)
		slic->is_wideband = is_slic_wideband(slic->version);

exit:
	return rc;
}


/*!	@} */
