/*!
*	\file sipua.h 
*
*	\attention
*	Copyright © 2004-2010 Mindspeed Technologies, Inc.\n
*	Mindspeed Confidential.\n
*	All rights reserved.\n
*	This file is a component of the Mindspeed® VAPI software ("VAPI") and is
*	distributed under the Mindspeed Software License Agreement (the "Agreement").\n
*	Before using this file, you must agree to be bound by the the terms and 
*	conditions of the Agreement.
*
*	@{
*/

#ifndef __SIPUA_H
#define __SIPUA_H


#include <pjsua-lib/pjsua.h>
#include <pjsua-lib/pjsua_internal.h>


/* ==============================
	Constants
   ============================== */


/*! Maximum size in bytes of the SIP URI string */
#define	SIP_URI_MAX_SIZE   50

/*! Maximin size in bytes of the SIP User Id string */
#define	SIP_USER_ID_MAX_SIZE   10

/*! Imported PJSIP define */
#define NO_LIMIT   (int)0x7FFFFFFF

/*! Codec priority (0 means codec is disabled) */
#define CODEC_PRIORITY_G726_16 0
#define CODEC_PRIORITY_G726_24 0
#define CODEC_PRIORITY_G726_32 0
#define CODEC_PRIORITY_G726_40 0
#define CODEC_PRIORITY_G729_8000   0
#define CODEC_PRIORITY_G723_8000   0
#define CODEC_PRIORITY_SPEEX_8000   0
#define CODEC_PRIORITY_SPEEX_16000   0
#define CODEC_PRIORITY_SPEEX_32000   0
#define CODEC_PRIORITY_ILBC_8000 0
#define CODEC_PRIORITY_AMR   0
#define CODEC_PRIORITY_AMR_WB   0
#define CODEC_PRIORITY_G722_8000 100
#define CODEC_PRIORITY_G722_16000   101
#define CODEC_PRIORITY_G7221_16000 201
#define CODEC_PRIORITY_G7221_24000 202
#define CODEC_PRIORITY_G7221_32000 203
#define CODEC_PRIORITY_PCMA_8000   300
#define CODEC_PRIORITY_PCMU_8000   301


/* ==============================
	Types
   ============================== */


/*! SIP phone descriptor (describing SIP phones used for SIP testing) */
typedef struct sip_phone {
	struct sip_session *sip_session; 
		/*! Pointer to the SIP session */
	unsigned char ip[4];
		/*! IP address stored as an arry of 4 numbers */
	long user_id;
		/*! SIP phone User Id which is also used as SIP Phone number */
	char uri[SIP_URI_MAX_SIZE];
		/*! SIP phone URI (Uniform User Id) */
} sip_phone_t;


/*! SIP server descriptor */
typedef struct sip_server {
	char registration;
		/*! > 0 : The SIP User Agent regsiters to the SIP Server */
		/*! <=0 : The SIP User Agent doesn't register to the SIP Server */
	char domain[50];
		/*! SIP domain name */
	unsigned int udp;
		/*! SIP server UDP port, which is usually 5060 */
	unsigned char ip[4];
		/*! SIP server IP address, defined as an array of 4 numbers */
	char uri[SIP_URI_MAX_SIZE];
		/*! SIP server Uniform Id */
} sip_server_t;


/*! SIP session descriptor */
typedef struct sip_session {
	struct endpt *endpt;
		/*! Pointer to the endpoint involved in this SIP session */
	struct sip_phone *sip_phone;
		/*! Pointer to the SIP phone involved in this SIP session */
	pjsua_acc_config *acc_cfg;
		/*! Pointer to the SIP account associated to this SIP session */
	int id;
		/*! SIP session unique identifier */
	unsigned long user_id;
		/*! SIP user identifier associated to this SIP session */
	pjsua_call_id call_id;
		/*! PJSIP caller id */
	unsigned char dst_ip[4];
		/*! Destination IP address */
	unsigned int dst_rtp;
		/*! Destination RTP port */
	unsigned int dst_rtcp;
		/*! Destination RTCP port */
} sip_session_t;


/*! SIPUA configuration descriptor */
typedef struct sipua_cfg {
	unsigned int first_user_id;
	unsigned int nb_sip_tdm_acc;
	unsigned int nb_sip_ip_acc;

	int codec_priority_g722_8000;
	int codec_priority_g722_16000;
	int codec_priority_g7221_16000;
	int codec_priority_g7221_24000;
	int codec_priority_g7221_32000;
	int codec_priority_g723_8000;
	int codec_priority_g726_16;
	int codec_priority_g726_24;
	int codec_priority_g726_32;
	int codec_priority_g726_40;
	int codec_priority_g728_8000;
	int codec_priority_g729_8000;
	int codec_priority_amr;
	int codec_priority_amr_wb;
	int codec_priority_gsm;
	int codec_priority_ilbc;
	int codec_priority_pcmu;
	int codec_priority_pcma;
	int codec_priority_speex_8000;
	int codec_priority_speex_16000;
	int codec_priority_speex_32000;

	sip_server_t sip_server;

	int max_sip_phones;
	sip_phone_t sip_phone[PJSUA_MAX_ACC];
} sipua_cfg_t;


/*! SIPUA application data */
typedef struct sipua
{
	pjsua_config cfg;
	pjsua_logging_config log_cfg;
	pjsua_media_config media_cfg;
	pj_bool_t no_refersub;
	pj_bool_t ipv6;
	pj_bool_t enable_qos;
	pj_bool_t no_tcp;
	pj_bool_t no_udp;
	pj_bool_t use_tls;
	pjsua_transport_config udp_cfg;
	pjsua_transport_config rtp_cfg;
	pjsip_redirect_op redir_op;

	/* buddies */
	unsigned buddy_cnt;
	pjsua_buddy_config buddy_cfg[PJSUA_MAX_BUDDIES];

/*	struct call_data call_data[PJSUA_MAX_CALLS]; */

	pj_pool_t *pool;

	/* Compatibility with older pjsua */
	unsigned codec_cnt;
	pj_str_t codec_arg[32];
	unsigned codec_dis_cnt;
	pj_str_t codec_dis[32];
	pj_bool_t null_audio;
	unsigned wav_count;
	pj_str_t wav_files[32];
	unsigned tone_count;
	pjmedia_tone_desc tones[32];
	pjsua_conf_port_id tone_slots[32];
	pjsua_player_id wav_id;
	pjsua_conf_port_id wav_port;
	pj_bool_t auto_play;
	pj_bool_t auto_play_hangup;
	pj_timer_entry auto_hangup_timer;
	pj_bool_t auto_loop;
	pj_bool_t auto_conf;
	pj_str_t rec_file;
	pj_bool_t auto_rec;
	pjsua_recorder_id rec_id;
	pjsua_conf_port_id rec_port;
	unsigned auto_answer;
	unsigned duration;

#ifdef STEREO_DEMO
	pjmedia_snd_port *snd;
	pjmedia_port *sc, *sc_ch1;
	pjsua_conf_port_id sc_ch1_slot;
#endif

	float mic_level, speaker_level;
	int capture_dev, playback_dev;
	unsigned capture_lat, playback_lat;

	pj_bool_t no_tones;
	int ringback_slot;
	int ringback_cnt;
	pjmedia_port *ringback_port;
	int ring_slot;
	int ring_cnt;
	pjmedia_port *ring_port;

	char src_uri[40];
/*	char dst_uri[40]; */

	pjsip_endpoint *sip_endpt;
	pjmedia_endpt *med_endpt;

	/* sip accounts */
	unsigned acc_cnt; 
	pjsua_acc_config acc_cfg[PJSUA_MAX_ACC];

	/* Connections */
	sip_session_t sip_session[MAX_ENDPT_ID];
} sipua_t;


/* ==============================
	Function prototypes
   ============================== */

sipua_cfg_t *get_sipua_cfg(const char *cfg_file);
void display_sipua_cfg(sipua_cfg_t *sipua_cfg);

extern void convert_ip_addr(char *ip_addr_ascii, unsigned char *ip_addr);
extern unsigned long get_user_id_from_uri(pj_str_t * uri);
extern void get_ip_from_uri(pj_str_t *uri, unsigned char *ip);
extern sip_session_t *get_sip_session_from_user_id(long sip_user_id);
extern sip_session_t *get_sip_session_from_call_id(pjsua_call_id call_id);
extern sip_phone_t *get_sip_phone_from_user_id(long sip_user_id);
extern sip_session_t *check_sip_session_is_idle(sip_session_t *sip_session);
extern void display_sip_call_info(pjsua_call_id call_id, pjsua_call_info *ci);
extern void display_sip_media_info(void);
void update_sdp_session_local_info(sip_session_t *sip_session, const pjmedia_sdp_session *sdp_session);
void update_sdp_session_remote_info(sip_session_t *sip_session, const pjmedia_sdp_session *sdp_session);

extern void get_local_sdp_media_info(sip_session_t *sip_session, struct endpt_voip_profile *voip_profile);

extern int call_sip_phone(sip_phone_t *sip_phone, pjsua_call_id *call_id, int acc_id);
extern int ack_sip_invite(sip_session_t *sip_session);

extern void sip_session_reset(sip_session_t *sip_session);
extern void sip_session_init(int id, sip_session_t *sip_session, endpt_t *endpt, sip_phone_t *sip_phone, pjsua_acc_config *acc_cfg);
extern void sip_phone_init(sip_phone_t *sip_phone);

extern int sip_session_hangup(sip_session_t *sip_session);
extern void sip_error_exit(const char *title, pj_status_t status);
extern int sipua_init(sipua_t *sipua);
extern void sipua_set_codec_priority(char *codec_label, int codec_priority);
extern void sipua_set_all_codec_priorities(void);
extern void sipua_create(sipua_t *sipua);


/* ==============================
	Variables
   ============================== */


/* PJSUA application instance. */
struct pjsua_data pjsua_var;

pj_status_t sip_status;

pjmedia_sdp_rtpmap rtpmap;

sipua_cfg_t *sipua_cfg;

sipua_t sipua;


#endif /* __SIPUA_H */

/*! @} */
