/*!
*	\file sipua.c
*
*	@defgroup sipua sipua (SIP User Agent) Module
*
*	sipua Module, including all C-functions and declarations
*	(defines, types, variables)
*
*	\attention
*	Copyright © 2004-2010 Mindspeed Technologies, Inc.\n
*	Mindspeed Confidential.\n
*	All rights reserved.\n
*	This file is a component of the Mindspeed® VAPI software ("VAPI") and is
*	distributed under the Mindspeed Software License Agreement (the "Agreement").\n
*	Before using this file, you must agree to be bound by the the terms and 
*	conditions of the Agreement.
*
*	@{
*/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <sys/time.h>
#include <getopt.h>

/* VAPI headers */
#include <msp.h>
#include <vapi.h>
#include <gtl.h>

/* Zarlink headers */
#include <legerity_lib.h>

/* mtalk header */
#include <mtalk.h>

#define THIS_FILE   "SIPUA"


/*!
*	This function reads allocates and initializes a sipua_cfg instance with hardcoded and/or user defined values.
*	\param *cfg_file Configuration filename (string).
*	\retval NULL on failure
*	\retval sipua_cfg_t* on success, pointer to the allocated and initialized sipua_cfg structure
*/
sipua_cfg_t *get_sipua_cfg(const char *cfg_file)
{
	struct _CFG *cfg_info;
	sipua_cfg_t *sipua_cfg;
	char sip_phone_list[200];
	char *str_token;

	sipua_cfg = (sipua_cfg_t *)malloc(sizeof(sipua_cfg_t));
	exit_if_null(sipua_cfg, err, "get_sipua_cfg() - mem allocation fail");

	cfg_info = cfg_read(cfg_file, 0);
	exit_if_null(cfg_info, err, "get_sipua_cfg() - mem allocation fail");

	READ_INT(cfg_info, "SIPUA", "NB_SIP_TDM_ACC", sipua_cfg->nb_sip_tdm_acc, NB_SIP_TDM_ACC);
	READ_INT(cfg_info, "SIPUA", "NB_SIP_IP_ACC", sipua_cfg->nb_sip_ip_acc, NB_SIP_IP_ACC);
	READ_INT(cfg_info, "SIPUA", "FIRST_SIP_USER_ID", sipua_cfg->first_user_id, FIRST_SIP_USER_ID);

	READ_IP(cfg_info,  "SIPUA", "SIP_SERVER_IP", sipua_cfg->sip_server.ip, SIP_SERVER_IP);
	READ_INT(cfg_info, "SIPUA", "SIP_SERVER_UDP", sipua_cfg->sip_server.udp, SIP_SERVER_UDP);
	READ_STR(cfg_info, "SIPUA", "SIP_SERVER_DOMAIN", sipua_cfg->sip_server.domain, SIP_SERVER_DOMAIN);
	READ_INT(cfg_info, "SIPUA", "SIP_SERVER_REGISTRATION", sipua_cfg->sip_server.registration, SIP_SERVER_REGISTRATION);

	READ_STR(cfg_info, "SIPUA", "LIST_SIP_PHONES", (char *) sip_phone_list, "200@192.1.1.1");

	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_G722_8000", sipua_cfg->codec_priority_g722_8000, 0);
	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_G722_16000", sipua_cfg->codec_priority_g722_16000, 0);

	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_G7221_16000", sipua_cfg->codec_priority_g7221_16000, 0);
	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_G7221_24000", sipua_cfg->codec_priority_g7221_24000, 0);
	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_G7221_32000", sipua_cfg->codec_priority_g7221_32000, 0);

	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_G723_8000", sipua_cfg->codec_priority_g723_8000, 0);

	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_G726_16", sipua_cfg->codec_priority_g726_16, 0);
	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_G726_24", sipua_cfg->codec_priority_g726_24, 0);
	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_G726_32", sipua_cfg->codec_priority_g726_32, 0);
	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_G726_40", sipua_cfg->codec_priority_g726_40, 0);

	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_G728_8000", sipua_cfg->codec_priority_g728_8000, 0);

	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_G729_8000", sipua_cfg->codec_priority_g729_8000, 0);

	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_AMR", sipua_cfg->codec_priority_amr, 0);
	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_AMR_WB", sipua_cfg->codec_priority_amr_wb, 0);

	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_GSM", sipua_cfg->codec_priority_gsm, 0);

	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_ILBC", sipua_cfg->codec_priority_ilbc, 0);

	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_PCMA", sipua_cfg->codec_priority_pcma, 0);
	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_PCMU", sipua_cfg->codec_priority_pcmu, 0);

	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_SPEEX_8000", sipua_cfg->codec_priority_speex_8000, 0);
	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_SPEEX_16000", sipua_cfg->codec_priority_speex_16000, 0);
	READ_INT(cfg_info, "SIPUA", "SIP_CODEC_PRIORITY_SPEEX_32000", sipua_cfg->codec_priority_speex_32000, 0);

	sipua_cfg->max_sip_phones = 0;
	str_token = (char *)strtok(sip_phone_list, ",");

	while ((str_token != NULL) && (sipua_cfg->max_sip_phones < PJSUA_MAX_ACC))
	{
		memset(&sipua_cfg->sip_phone[sipua_cfg->max_sip_phones], 0, sizeof(sip_phone_t));

		strcpy(sipua_cfg->sip_phone[sipua_cfg->max_sip_phones].uri, str_token);

		sipua_cfg->max_sip_phones++;

		str_token = (char *)strtok(NULL, ",");
	}

	cfg_clean(cfg_info);

	sprintf(sipua_cfg->sip_server.uri, "sip:%d.%d.%d.%d", 
		sipua_cfg->sip_server.ip[0], sipua_cfg->sip_server.ip[1], sipua_cfg->sip_server.ip[2], sipua_cfg->sip_server.ip[3]);

	if ((sipua_cfg->nb_sip_tdm_acc + sipua_cfg->nb_sip_ip_acc) > PJSUA_MAX_ACC);
	{
		sipua_cfg->nb_sip_tdm_acc = NB_SIP_TDM_ACC;
		sipua_cfg->nb_sip_ip_acc = NB_SIP_IP_ACC;
	}

	return sipua_cfg;

err:
	return NULL;
}


/*!
*	This function displays the values of sipua parameters (sipua_cfg).
*	\param *sipua_cfg Pointer to the sipua configuration parameter structure.
*	\retval None
*/
void display_sipua_cfg(sipua_cfg_t *sipua_cfg)
{
	int i;

	if (!sipua_cfg)
		return;

	PDEBUG(DBG_L2, "\n-------------------- sipua_cfg");
	PDEBUG(DBG_L2, "SIP Client: Nb SIP/TDM ACC %d, Nb SIP/IP ACC %d, First SIP User Id %d", 
			sipua_cfg->nb_sip_tdm_acc, sipua_cfg->nb_sip_ip_acc, sipua_cfg->first_user_id);

	PDEBUG(DBG_L2, "SIP Server: IP %d.%d.%d.%d, UDP port %d, Domain %s Registration %d", 
			sipua_cfg->sip_server.ip[0], sipua_cfg->sip_server.ip[1], sipua_cfg->sip_server.ip[2], 
			sipua_cfg->sip_server.ip[3], sipua_cfg->sip_server.udp, 
			sipua_cfg->sip_server.domain, sipua_cfg->sip_server.registration);

	PDEBUG(DBG_L2, "SIP Phone - Predefined URIs:");

	for (i = 0; i < sipua_cfg->max_sip_phones; i++)
	{
		PDEBUG(DBG_L2, "    - URI #%d = %s", i, sipua_cfg->sip_phone[i].uri);
	}

	PDEBUG(DBG_L2, "Codec Priorities:");
	PDEBUG(DBG_L2, "     %3d G722_8000", sipua_cfg->codec_priority_g722_8000);
	PDEBUG(DBG_L2, "     %3d G722_16000", sipua_cfg->codec_priority_g722_16000);
	PDEBUG(DBG_L2, "     %3d G7221_16000", sipua_cfg->codec_priority_g7221_16000);
	PDEBUG(DBG_L2, "     %3d G7221_24000", sipua_cfg->codec_priority_g7221_24000);
	PDEBUG(DBG_L2, "     %3d G7221_32000", sipua_cfg->codec_priority_g7221_32000);
	PDEBUG(DBG_L2, "     %3d G723_8000", sipua_cfg->codec_priority_g723_8000);
	PDEBUG(DBG_L2, "     %3d G726_16", sipua_cfg->codec_priority_g726_16);
	PDEBUG(DBG_L2, "     %3d G726_24", sipua_cfg->codec_priority_g726_24);
	PDEBUG(DBG_L2, "     %3d G726_32", sipua_cfg->codec_priority_g726_32);
	PDEBUG(DBG_L2, "     %3d G726_40", sipua_cfg->codec_priority_g726_40);
	PDEBUG(DBG_L2, "     %3d G728_8000", sipua_cfg->codec_priority_g728_8000);
	PDEBUG(DBG_L2, "     %3d G729_8000", sipua_cfg->codec_priority_g729_8000);
	PDEBUG(DBG_L2, "     %3d AMR", sipua_cfg->codec_priority_amr);
	PDEBUG(DBG_L2, "     %3d AMR_WB", sipua_cfg->codec_priority_amr_wb);
	PDEBUG(DBG_L2, "     %3d GSM", sipua_cfg->codec_priority_gsm);
	PDEBUG(DBG_L2, "     %3d ILBC", sipua_cfg->codec_priority_ilbc);
	PDEBUG(DBG_L2, "     %3d PCMA", sipua_cfg->codec_priority_pcmu);
	PDEBUG(DBG_L2, "     %3d PCMU", sipua_cfg->codec_priority_pcma);
	PDEBUG(DBG_L2, "     %3d SPEEX_8000", sipua_cfg->codec_priority_speex_8000);
	PDEBUG(DBG_L2, "     %3d SPEEX_16000", sipua_cfg->codec_priority_speex_16000);
	PDEBUG(DBG_L2, "     %3d SPEEX_32000", sipua_cfg->codec_priority_speex_32000);

	PDEBUG(DBG_L2, "--------------------\n ");
}



/*!
*	This function converts an IP address from an ASCII format to an array of numbers.
*	\param ip_addr_ascii IP address in ASCII format.
*	\param ip_addr IP address array.
*	\retval None
*/
void convert_ip_addr(char *ip_addr_ascii, unsigned char *ip_addr)
{
	char ip_ascii[50];
	char *str_token;
	int i;

	strcpy(ip_ascii, ip_addr_ascii);

	str_token = strtok(ip_ascii, ".");
	for (i = 0; i < 4; i++)
	{
		ip_addr[i] = (unsigned char) strtoul(str_token, NULL, 10);
		str_token = strtok(NULL, ".");
	}
}


/*!
*	This function extracts the User Id from a given URI.
*	\param uri SIP URI
*	\retval sip_user_id SIP User Id extracted from the SIP URI
*/
unsigned long get_user_id_from_uri(pj_str_t * uri)
{
	unsigned long sip_user_id;
	char *str_token;

	PDEBUG(DBG_L3, "get_user_id_from_uri(): URI %s", uri->ptr);

	str_token = strtok(uri->ptr, ":"); /* skip "<sip:"  */
	str_token = strtok(NULL, "@");

	sip_user_id = (U32) strtoul(str_token, NULL, 10);

	PDEBUG(DBG_L3, "get_user_id_from_uri(): SIP user Id %ld", sip_user_id);
	return sip_user_id;
}


/*!
*	This function extracts the IP address from a given URI.
*	\param uri SIP URI
*	\param ip IP address
*	\retval None
*/
void get_ip_from_uri(pj_str_t * uri, unsigned char *ip)
{
	char *str_token;
	int i;

	str_token = strtok(uri->ptr, "@"); /* skip "sip:"  */
	str_token = strtok(NULL, ".");

	for (i = 0; i < 4; i++)
	{
		ip[i] = (unsigned char) strtoul(str_token, NULL, 10);
		str_token = strtok(NULL, ".");
	}	
}


/*!
*	This function returns the pointer of the SIP session associated to a given SIP User Id (if any).
*	\param sip_user_id SIP user Id
*	\retval NULL on failure.
*	\retval sip_session_t* on success, pointer of the found SIP session on success
*/
sip_session_t *get_sip_session_from_user_id(long sip_user_id)
{
	sip_session_t *sip_session = NULL;
	int i;

	for (i = 0; (i < MAX_ENDPT_ID) && (sip_session == NULL); i++)
	{
		if (sipua.sip_session[i].user_id == sip_user_id)
			sip_session = &sipua.sip_session[i];
	}
 
	if (sip_session != NULL)
		PDEBUG(DBG_L3, "get_sip_session_from_user_id(): (SIP Acc %d, user Id %ld)", sip_session->id, sip_session->user_id);
	else
 		PDEBUG(DBG_L3, "get_sip_session_from_user_id(): none found");

	return sip_session;
}


/*!
*	This function returns the pointer of the SIP session associated to a Caller Id (if any).
*	\param call_id Caller Id
*	\retval NULL on failure
*	\retval sip_session_t* on success, pointer to the found SIP session
*/
sip_session_t *get_sip_session_from_call_id(pjsua_call_id call_id)
{
	sip_session_t *sip_session = NULL;
	int i;

	for (i = 0; (i < PJSUA_MAX_ACC) && (sip_session == NULL); i++)
	{
		if (sipua.sip_session[i].call_id == call_id)
			sip_session = &sipua.sip_session[i];
	}
 
	if (sip_session != NULL)
		PDEBUG(DBG_L3, "get_sip_session_from_call_id(): (SIP Acc %d, call Id %d)", sip_session->id, sip_session->call_id);
	else
 		PDEBUG(DBG_L3, "get_sip_session_from_call_id(): none found");

	return sip_session;
}



/*!
*	This function returns the pointer of the SIP phone associated to a SIP User Id (if any).
*	\param sip_user_id SIP User Id
*	\retval NULL on failure.
*	\retval sip_phone_t* on success, pointer to the found SIP phone instance
*/
sip_phone_t *get_sip_phone_from_user_id(long sip_user_id)
{
	sip_phone_t *sip_phone = NULL;
	int i;
	
	if (sip_user_id > 0)
	{
		for (i = 0; (i < sipua_cfg->max_sip_phones) && (sip_phone == NULL) ; i++)
		{
			if (sipua_cfg->sip_phone[i].user_id == sip_user_id)
				sip_phone = &sipua_cfg->sip_phone[i];
		}
	}

	if (sip_phone != NULL)
		PDEBUG(DBG_L3, "get_sip_phone_from_user_id(): (User Id %ld, URI %s)", sip_phone->user_id, sip_phone->uri);
	else
 		PDEBUG(DBG_L3, "get_sip_phone_from_user_id(): none found");

	return sip_phone;
}


/*!
*	This function tries to find an available or idle (not connected) SIP phone (if any).
*	\param *sipua Pointer to the sipua instance
*	\retval NULL on failure.
*	\retval sip_phone_t* Pointer to the found and idle SIP phone
*/
sip_phone_t * get_free_sip_phone(sipua_t *sipua)
{
	int i;
	sip_phone_t *sip_phone = NULL;

	for (i = 0; (i < sipua_cfg->max_sip_phones) && (sip_phone == NULL); i++)
	{
		if (sipua_cfg->sip_phone[i].sip_session == NULL)
			sip_phone = &sipua_cfg->sip_phone[i];
	}

	return sip_phone;
}


/*!
*	This function indicates whether the specified SIP account id was registered to teh SIP Server or not.
*	\param *sipua Pointer to the sipua instance
*	\param pjsua_acc_id PJSUA account Id.
*	\param acc_id Account Id.
*	\retval -1 on failure (SIP account not registered)
*	\retval 0 on success (SIP account already registered to the SIP server)
*/
int is_sip_registered(sipua_t *sipua, int pjsua_acc_id, int acc_id)
{
	if (pjsua_acc_id >= PJSUA_MAX_ACC)
		return -1;

	if (acc_id < MAX_POTS_ENDPT_ID)
	{
		if (acc_id >= sipua_cfg->nb_sip_tdm_acc)
			return -1;
	}
	else if (acc_id - MAX_POTS_ENDPT_ID >= sipua_cfg->nb_sip_ip_acc)
	{
		return -1;
	}

	return 0;
}


/*!
*	This function indicates whether the specified SIP session is idle (not connected) or not.
*	\param *sip_session Pointer to a given SIP session instance
*	\retval NULL on failure (if the specified SIP session exists and is not idle)
*	\retval sip_session_t* on success (if the specified SIP session is idle)
*/
sip_session_t *check_sip_session_is_idle(sip_session_t *sip_session)
{
	endpt_t *endpt;

	if (sip_session != NULL)
	{
		endpt = sip_session->endpt;

		/* check SIP account availability for this call */
		if (endpt != NULL)
		{
			/* associated endpt must be in idle state to take the call */
			if ((endpt->state == STATE_IDLE) && (endpt->peer_endpt == NULL))
				return sip_session;
		}
	}

	return NULL;
}


/*!
*	This function displays the SIP Call Information provided by the PJSUA stack.
*	\param call_id Caller Id
*	\param *ci Pointer to the PJSUA call information
*	\retval None
*/
void display_sip_call_info(pjsua_call_id call_id, pjsua_call_info *ci)
{
	PDEBUG(DBG_L2, " ------------- SIP Call Info ----------- ");
	PDEBUG(DBG_L2, " Call Id         %d",   (int) call_id);
	PDEBUG(DBG_L2, " Local URI       %.*s", (int) ci->local_info.slen, ci->local_info.ptr);
	PDEBUG(DBG_L2, " Local contact   %.*s", (int) ci->local_contact.slen, ci->local_contact.ptr);
	PDEBUG(DBG_L2, " Remote URI      %.*s", (int) ci->remote_info.slen, ci->remote_info.ptr);
	PDEBUG(DBG_L2, " remote URI      %.*s", (int) ci->remote_contact.slen, ci->remote_contact.ptr);
	PDEBUG(DBG_L2, " Call-ID string  %.*s", (int) ci->call_id.slen, ci->call_id.ptr);
	PDEBUG(DBG_L2, " Conf slot       %d",   (int) ci->conf_slot);
	PDEBUG(DBG_L2, " ----------------------------------------");
}


/*!
*	This function displays the SIP Media Information provided by the PJSUA stack.
*	\retval None
*/
void display_sip_media_info(void)
{
	PDEBUG(DBG_L2, " ------------- SIP Media Info ----------- ");
	/* Specify audio frame ptime. The value here will affect the samples per frame of both the
	   sound device and the conference bridge. Specifying lower ptime will normally reduce the latency.
	   Default value: PJSUA_DEFAULT_AUDIO_FRAME_PTIME
	*/
	PDEBUG(DBG_L2, " audio_frame_ptime %d", (int) pjsua_var.media_cfg.audio_frame_ptime);
	PDEBUG(DBG_L2, " quality           %d", (int) pjsua_var.media_cfg.quality);
	PDEBUG(DBG_L2, " ptime             %d", (int) pjsua_var.media_cfg.ptime);
	PDEBUG(DBG_L2, " no_vad            %d", (int) pjsua_var.media_cfg.no_vad);
	PDEBUG(DBG_L2, " ec_options        %d", (int) pjsua_var.media_cfg.ec_options);

	PDEBUG(DBG_L2, " ec_tail_len     %d", (int) pjsua_var.media_cfg.ec_tail_len);

	/* Jitter buffer initial prefetch delay in msec. The value must be between jb_min_pre and jb_max_pre below.
	   Default: -1 (to use default stream settings, currently 150 msec)
	*/
	PDEBUG(DBG_L2, " jb_init     %d", (int) pjsua_var.media_cfg.jb_init);
   
	/* Jitter buffer minimum prefetch delay in msec.
	   Default: -1 (to use default stream settings, currently 60 msec)
	*/
	PDEBUG(DBG_L2, " jb_min_pre     %d", (int) pjsua_var.media_cfg.jb_min_pre);

	/* Jitter buffer maximum prefetch delay in msec.
	 * Default: -1 (to use default stream settings, currently 240 msec)
	*/
	PDEBUG(DBG_L2, " jb_max_pre     %d", (int) pjsua_var.media_cfg.jb_max_pre);

	/* Set maximum delay that can be accomodated by the jitter buffer msec.
           Default: -1 (to use default stream settings, currently 360 msec)
	*/
	PDEBUG(DBG_L2, " jb_max          %d", (int) pjsua_var.media_cfg.jb_max);
	PDEBUG(DBG_L2, " -----------------------------------------");
}

#if 0

/** 
 * Standard RTP static payload types, as defined by RFC 3551. 
 * The header file <pjmedia-codec/types.h> also declares dynamic payload
 * type numbers that are used by PJMEDIA when advertising the capability
 * for example in SDP message.
 */
enum pjmedia_rtp_pt
{
    PJMEDIA_RTP_PT_PCMU = 0,	    /**< audio PCMU			    */
    PJMEDIA_RTP_PT_G726_32 = 2,    /**< audio G726-32			    */
    PJMEDIA_RTP_PT_GSM  = 3,	    /**< audio GSM			    */
    PJMEDIA_RTP_PT_G723 = 4,	    /**< audio G723			    */
    PJMEDIA_RTP_PT_DVI4_8K = 5,	    /**< audio DVI4 8KHz		    */
    PJMEDIA_RTP_PT_DVI4_16K = 6,    /**< audio DVI4 16Khz		    */
    PJMEDIA_RTP_PT_LPC = 7,	    /**< audio LPC			    */
    PJMEDIA_RTP_PT_PCMA = 8,	    /**< audio PCMA			    */
    PJMEDIA_RTP_PT_G722 = 9,	    /**< audio G722			    */
    PJMEDIA_RTP_PT_L16_2 = 10,	    /**< audio 16bit linear 44.1KHz stereo  */
    PJMEDIA_RTP_PT_L16_1 = 11,	    /**< audio 16bit linear 44.1KHz mono    */
    PJMEDIA_RTP_PT_QCELP = 12,	    /**< audio QCELP			    */
    PJMEDIA_RTP_PT_CN = 13,	    /**< audio Comfort Noise		    */
    PJMEDIA_RTP_PT_MPA = 14,	    /**< audio MPEG1/MPEG2 elemetr. streams */
    PJMEDIA_RTP_PT_G728 = 15,	    /**< audio G728			    */
    PJMEDIA_RTP_PT_DVI4_11K = 16,   /**< audio DVI4 11.025KHz mono	    */
    PJMEDIA_RTP_PT_DVI4_22K = 17,   /**< audio DVI4 22.050KHz mono	    */
    PJMEDIA_RTP_PT_G729 = 18,	    /**< audio G729			    */

    PJMEDIA_RTP_PT_CELB = 25,	    /**< video/comb Cell-B by Sun (RFC2029) */
    PJMEDIA_RTP_PT_JPEG = 26,	    /**< video JPEG			    */
    PJMEDIA_RTP_PT_NV = 28,	    /**< video NV  by nv program by Xerox   */
    PJMEDIA_RTP_PT_H261 = 31,	    /**< video H261			    */
    PJMEDIA_RTP_PT_MPV = 32,	    /**< video MPEG1 or MPEG2 elementary    */
    PJMEDIA_RTP_PT_MP2T = 33,	    /**< video MPEG2 transport		    */
    PJMEDIA_RTP_PT_H263 = 34,	    /**< video H263			    */

    PJMEDIA_RTP_PT_DYNAMIC = 96     /**< start of dynamic RTP payload	    */

};


/** 
 * Identification used to search for codec factory that supports specific 
 * codec specification. 
 */
typedef struct pjmedia_codec_info
{
    pjmedia_type    type;	    /**< Media type.			*/
    unsigned	    pt;		    /**< Payload type (can be dynamic). */
    pj_str_t	    encoding_name;  /**< Encoding name.			*/
    unsigned	    clock_rate;	    /**< Sampling rate.			*/
    unsigned	    channel_cnt;    /**< Channel count.			*/
} pjmedia_codec_info;

#define PJMEDIA_CODEC_MAX_FMTP_CNT  8

/** 
 * Structure of codec specific parameters which contains name=value pairs.
 * The codec specific parameters are to be used with SDP according to 
 * the standards (e.g: RFC 3555).
 */
typedef struct pjmedia_codec_fmtp
{
    pj_uint8_t	    cnt;
    struct param {
	pj_str_t    name;
	pj_str_t    val;
    } param [PJMEDIA_CODEC_MAX_FMTP_CNT];
} pjmedia_codec_fmtp;



enum
{
    /* PJMEDIA_RTP_PT_TELEPHONE_EVENTS is declared in
     * <pjmedia/config.h>
     */
#if PJMEDIA_RTP_PT_TELEPHONE_EVENTS
    PJMEDIA_RTP_PT_START = PJMEDIA_RTP_PT_TELEPHONE_EVENTS,
#else
    PJMEDIA_RTP_PT_START = 102,
#endif

    PJMEDIA_RTP_PT_SPEEX_NB,			/**< Speex narrowband/8KHz  */
    PJMEDIA_RTP_PT_SPEEX_WB,			/**< Speex wideband/16KHz   */
    PJMEDIA_RTP_PT_SPEEX_UWB,			/**< Speex 32KHz	    */
    PJMEDIA_RTP_PT_L16_8KHZ_MONO,		/**< L16 @ 8KHz, mono	    */
    PJMEDIA_RTP_PT_L16_8KHZ_STEREO,		/**< L16 @ 8KHz, stereo     */
    //PJMEDIA_RTP_PT_L16_11KHZ_MONO,		/**< L16 @ 11KHz, mono	    */
    //PJMEDIA_RTP_PT_L16_11KHZ_STEREO,		/**< L16 @ 11KHz, stereo    */
    PJMEDIA_RTP_PT_L16_16KHZ_MONO,		/**< L16 @ 16KHz, mono	    */
    PJMEDIA_RTP_PT_L16_16KHZ_STEREO,		/**< L16 @ 16KHz, stereo    */
    //PJMEDIA_RTP_PT_L16_22KHZ_MONO,		/**< L16 @ 22KHz, mono	    */
    //PJMEDIA_RTP_PT_L16_22KHZ_STEREO,		/**< L16 @ 22KHz, stereo    */
    PJMEDIA_RTP_PT_L16_32KHZ_MONO,		/**< L16 @ 32KHz, mono	    */
    PJMEDIA_RTP_PT_L16_32KHZ_STEREO,		/**< L16 @ 32KHz, stereo    */
    PJMEDIA_RTP_PT_L16_48KHZ_MONO,		/**< L16 @ 48KHz, mono	    */
    PJMEDIA_RTP_PT_L16_48KHZ_STEREO,		/**< L16 @ 48KHz, stereo    */
    PJMEDIA_RTP_PT_ILBC,			/**< iLBC (13.3/15.2Kbps)   */
    PJMEDIA_RTP_PT_AMR,				/**< AMR (4.75 - 12.2Kbps)  */
    PJMEDIA_RTP_PT_AMRWB,			/**< AMRWB (6.6 - 23.85Kbps)*/
    PJMEDIA_RTP_PT_AMRWBE,			/**< AMRWBE		    */
    PJMEDIA_RTP_PT_G726_16,			/**< G726 @ 16Kbps	    */
    PJMEDIA_RTP_PT_G726_24,			/**< G726 @ 24Kbps	    */
    /* PJMEDIA_RTP_PT_G726_32,*/		/**< G726 @ 32Kbps, static? */
    PJMEDIA_RTP_PT_G726_40,			/**< G726 @ 40Kbps	    */
    PJMEDIA_RTP_PT_G722_1_16,			/**< G722.1 (16Kbps)	    */
    PJMEDIA_RTP_PT_G722_1_24,			/**< G722.1 (24Kbps)	    */
    PJMEDIA_RTP_PT_G722_1_32,			/**< G722.1 (32Kbps)	    */
    PJMEDIA_RTP_PT_G7221C_24,			/**< G722.1 Annex C (24Kbps)*/
    PJMEDIA_RTP_PT_G7221C_32,			/**< G722.1 Annex C (32Kbps)*/
    PJMEDIA_RTP_PT_G7221C_48,			/**< G722.1 Annex C (48Kbps)*/
    PJMEDIA_RTP_PT_G7221_RSV1,			/**< G722.1 reserve	    */
    PJMEDIA_RTP_PT_G7221_RSV2,			/**< G722.1 reserve	    */
};

#endif


/*!
*	This function displays the local information (provided by PJSUA) for a given Session Description Protocol (SDP) session.
*	\param *sdp_session Pointer to a given PJSUA SDP session
*	\retval None
*/
void display_sdp_session_local_info(const pjmedia_sdp_session *sdp_session)
{
	int i;

	for (i = 0; i < sdp_session->media[0]->attr_count; i++)
		PDEBUG(DBG_L3, "Attr %d = %.*s / %.*s", i, 
				(int) sdp_session->media[0]->attr[i]->name.slen, 
				sdp_session->media[0]->attr[i]->name.ptr, 
				(int) sdp_session->media[0]->attr[i]->value.slen, 
				sdp_session->media[0]->attr[i]->value.ptr);
}


/*!
*	This function displays the local information (provided by PJSUA) for a given SIP and SDP sessions.
*	\param *sip_session Pointer to a given SIP session
*	\param *sdp_session Pointer to a given PJSUA SDP session.
*	\retval None
*/
void update_sdp_session_local_info(sip_session_t *sip_session, const pjmedia_sdp_session *sdp_session)
{
	sip_session->endpt->rtp = htons(sdp_session->media[0]->desc.port);
	sip_session->endpt->rtcp = htons(sdp_session->media[0]->desc.port + 1);

	PDEBUG(DBG_L1, " ----------- SDP Local Info --------------------");
/*	PDEBUG(1, "local  IP addr  %.*s", (int) sdp_session->origin.addr.slen,  sdp_session->origin.addr.ptr); */
	PDEBUG(DBG_L1, " IP addr     %.*s", (int) sdp_session->conn->addr.slen, sdp_session->conn->addr.ptr);
	PDEBUG(DBG_L1, " UDP port    %d",   (int) sdp_session->media[0]->desc.port);
	PDEBUG(DBG_L1, " Media Type  %.*s", (int) sdp_session->media[0]->desc.media.slen, sdp_session->media[0]->desc.media.ptr);
	PDEBUG(DBG_L1, " -----------------------------------------");

	{
		int i;

		for (i = 0; i < sdp_session->media[0]->attr_count; i++)
			PDEBUG(DBG_L1, "Attr %d = %.*s / %.*s", i, 
					(int) sdp_session->media[0]->attr[i]->name.slen, 
					sdp_session->media[0]->attr[i]->name.ptr, 
					(int) sdp_session->media[0]->attr[i]->value.slen, 
					sdp_session->media[0]->attr[i]->value.ptr);
	}


/*
PJ_DECL(pj_status_t) pjmedia_sdp_attr_get_rtpmap(const pjmedia_sdp_attr *attr,
						 pjmedia_sdp_rtpmap *rtpmap);
*/
	pjmedia_sdp_attr_get_rtpmap(sdp_session->media[0]->attr[1], &rtpmap);
	PDEBUG(DBG_L1, "PT %.*s - Codec %.*s - Clock Rate %d - Param %.*s",
		(int) rtpmap.pt.slen, rtpmap.pt.ptr,
		(int) rtpmap.enc_name.slen, rtpmap.enc_name.ptr,
		rtpmap.clock_rate,
		(int) rtpmap.param.slen, rtpmap.param.ptr);
#if 0
    pj_str_t		pt;	    /**< Payload type.	    */
    pj_str_t		enc_name;   /**< Encoding name.	    */
    unsigned		clock_rate; /**< Clock rate.	    */
    pj_str_t		param;	    /**< Parameter.	    */
#endif
}



/*!
*	This function displays the remote information (provided by PJSUA) for a given SIP and SDP sessions.
*	\param *sip_session Pointer to a given SIP session instance
*	\param *sdp_session Pointer to a given PJSUA SDP session instance
*	\retval None
*/
void update_sdp_session_remote_info(sip_session_t *sip_session, const pjmedia_sdp_session *sdp_session)
{
	convert_ip_addr(sdp_session->conn->addr.ptr, &sip_session->dst_ip[0]);
	
	sip_session->dst_rtp = htons(sdp_session->media[0]->desc.port);
	sip_session->dst_rtcp = htons(sdp_session->media[0]->desc.port + 1);

	PDEBUG(DBG_L1, " ----------- SDP Remote Info --------------------");
/*	PDEBUG(1, "local  IP addr  %.*s", (int) sdp_session->origin.addr.slen,  sdp_session->origin.addr.ptr); */
	PDEBUG(DBG_L1, " IP addr     %.*s", (int) sdp_session->conn->addr.slen, sdp_session->conn->addr.ptr);
	PDEBUG(DBG_L1, " UDP port    %d",   (int) sdp_session->media[0]->desc.port);
	PDEBUG(DBG_L1, " Media Type  %.*s", (int) sdp_session->media[0]->desc.media.slen, sdp_session->media[0]->desc.media.ptr);

	{
		int i;

		for (i = 0; i < sdp_session->media[0]->attr_count; i++)
			PDEBUG(DBG_L1, "Attr %d = %.*s / %.*s", i, 
					(int) sdp_session->media[0]->attr[i]->name.slen, 
					sdp_session->media[0]->attr[i]->name.ptr, 
					(int) sdp_session->media[0]->attr[i]->value.slen, 
					sdp_session->media[0]->attr[i]->value.ptr);
	}
}





#if 0
/*
a=rtpmap:9 G722/16000
a=rtpmap:8 PCMA/8000
a=rtpmap:0 PCMU/8000
a=rtpmap:18 G729/8000
a=rtpmap:2 G726-32/8000
a=rtpmap:4 G723/8000
a=ptime:20
*/

	switch (payload_type)
	{
	case PJMEDIA_RTP_PT_PCMU: codec = eG711_ULAW_PCM; break;
	case PJMEDIA_RTP_PT_PCMA: codec = eG711_ALAW_PCM; break;		
	case PJMEDIA_RTP_PT_G722: codec = eG722; break;
	default: codec = -1; break;
	}

typedef enum {
        eGeneric_Comfort_Noise = 0,     /**< 0: Comfort noise */
        eRedundant_Scheme_for_DTMF = 1, /**< 1: DTMF redundant scheme*/
        eDTMF_RTP = 2,                  /**< 2: DTMF in RTP*/
        eIFP = 3,                       /**< 3: Internet Fax Protocol*/
        eG711_ULAW_PCM = 4,             /**< 4: g711 uLaw*/
        eG711_ALAW_PCM = 5,             /**< 5: g711 ALaw*/
        eG723_1 = 6,                    /**< 6: g723*/
        eG728_Audio = 7,                /**< 7: g728 audio*/
        eG729_A = 8,                    /**< 8: g729 A*/
        eG726_32 = 9,                   /**< 9: g726 32*/
        eReserved1 = 10,                /**< 10: Not Used*/
        eGSM_FR_Audio = 11,             /**< 11: GSM FR*/
        eClear_Channel = 12,            /**< 12: Clear Channel*/
        ePassthrough_ULAW = 13,         /**< 13: PassThrough uLaw*/
        ePassthrough_ALAW = 14,         /**< 14: PassThrough uAaw*/
        eG726_16_audio = 15,            /**< 15: g726 16*/
        eG726_24_audio = 16,            /**< 16: g726 24*/
        eG726_40_audio = 17,            /**< 17: g726 40*/
        eGSM_EFR = 18,                  /**< 18: GSM EFR*/
        eAMR = 19,                      /**< 19: AMR*/
        eEVRC = 20,                     /**< 20: EVRC*/
        eCNF_SEC = 21,                  /**< 21: Secondary CNF*/
        eSMV = 22,                      /**< 22: SMV*/
        eQCELP_13K = 23,                /**< 23: QCELP 13K*/
        eQCELP_8K = 24,                 /**< 24: QCELP 8K*/
        eCRBT_ADPCM = 25,               /**< 25: CRBT ADPCM*/
        eG729_EG = 26,                  /**< 26: g729 EG*/
        eFEC = 27,                      /**< 27: Forward Error Correction*/
        eI_LBC_15K = 28,                /**< 28: I-LBC 15K*/
        eI_LBC_13K = 29,                /**< 29: I-LBC 13K*/
        eRFC2833_EVENT = 30,            /**< 30: RFC2833 Event*/
        eGSM_AMR_BWE = 31,              /**< 31: GSM_AMR_BWE*/
        e4GV = 32,                      /**< 32: 4GV*/
        eRFC2833_TONE = 33,             /**< 33: RFC2833 Tone*/
        eAMR_WB = 34,                   /**< 34: AMR_WB*/
        eG729_MSPD = 35,                /**< 35: G.729 MSPD*/
        eCESoPSN = 36,                  /**< 36: CESoPSN*/
        eGSM_HR = 37,                   /**< 37: eGSM_HR*/
        eCSD_BS30T_64k = 38,            /**< 38: CSD_BS30T_64k*/
        eCSD_BS30T_32k = 39,            /**< 39: CSD_BS30T_32k*/
        eCSD_BS30T_33k6 = 40,           /**< 40: CSD_BS30T_33k6*/
        eCSD_BS30T_28k8 = 41,           /**< 41: CSD_BS30T_28k8*/
        eCSD_BS20NT_57k6 = 42,          /**< 42: CSD_BS20NT_57k6*/
        eCSD_BS20NT_28k8 = 43,          /**< 43: CSD_BS20NT_28k8*/
        eCSD_BS20NT_14k4 = 44,          /**< 44: CSD_BS20NT_14k4*/
        eG729_1 = 45,                   /**< 45: G.729.1*/
        eAMR_WB_ba = 46,                /**< 46: AMR WB(G.722.2)[byte aligned]*/
        eCCISoIP = 47,                  /**< 47: CCISoIP*/
        eG722 = 48,                     /**< 48: G.722:*/
	}
#endif


/*!
*	This function queries PJSIP to get some of the negotiated SDP patameter values.
*	\param *sip_session Pointer to a given SIP session.
*	\param *voip_profile Pointer to the negotiated VoIP information.
*	\retval None
*	\see pjmedia_sdp_neg_get_active_local
*	\see pjmedia_sdp_attr_get_rtpmap
*/
void get_local_sdp_media_info(sip_session_t *sip_session, endpt_voip_profile_t *voip_profile)
{
	const pjmedia_sdp_session *sdp_session;
	pjmedia_sdp_rtpmap rtpmap;
	pjsua_call *call;
	int i;

	voip_profile->codec = -1;
	voip_profile->payload_type = -1;
	voip_profile->ptime = -1;

	call = &pjsua_var.calls[sip_session->call_id];

	sip_status = pjmedia_sdp_neg_get_active_local(call->inv->neg, &sdp_session);
	if (sip_status != PJ_SUCCESS)
		sip_error_exit("Error in pjmedia_sdp_neg_get_active_remote()", sip_status);

	for (i = 0; i < sdp_session->media[0]->attr_count; i++)
	{
		PDEBUG(DBG_L3, "Attr %d => Name <%.*s> / Value <%.*s>", i,
			(int) sdp_session->media[0]->attr[i]->name.slen, sdp_session->media[0]->attr[i]->name.ptr, 
			(int) sdp_session->media[0]->attr[i]->value.slen, sdp_session->media[0]->attr[i]->value.ptr);

		if (strncmp(sdp_session->media[0]->attr[i]->name.ptr, "rtpmap", 6) == 0)
		{
			pjmedia_sdp_attr_get_rtpmap(sdp_session->media[0]->attr[i], &rtpmap);
			PDEBUG(DBG_L1, "get_local_sdp_media_info() - Negotiated PT %.*s - Codec %.*s - Clock Rate %d - Param %.*s",
				(int) rtpmap.pt.slen, rtpmap.pt.ptr,
				(int) rtpmap.enc_name.slen, rtpmap.enc_name.ptr,
				rtpmap.clock_rate,
				(int) rtpmap.param.slen, rtpmap.param.ptr);

			voip_profile->payload_type = strtol(rtpmap.pt.ptr, NULL, 10);

/* ECodecIndex */
			if (strncmp(rtpmap.enc_name.ptr, "PCMU", 4) == 0)
				voip_profile->codec = (int) eG711_ULAW_PCM;
			else if (strncmp(rtpmap.enc_name.ptr, "PCMA", 4) == 0)
 				voip_profile->codec = (int) eG711_ALAW_PCM;
			else if (strncmp(rtpmap.enc_name.ptr, "G722", 4) == 0)
				voip_profile->codec = (int) eG722;
			else if (strncmp(rtpmap.enc_name.ptr, "G729", 4) == 0)
				voip_profile->codec = (int) eG729_A;
			else if (strncmp(rtpmap.enc_name.ptr, "G726-32", 7) == 0)
				voip_profile->codec = (int) eG726_32;
			else if (strncmp(rtpmap.enc_name.ptr, "G723", 4) == 0)
				voip_profile->codec = (int) eG723_1;

			break;
		}
	}

	for (i = 0; i < sdp_session->media[0]->attr_count; i++)
	{
		PDEBUG(DBG_L3, "Attr %d => Name <%.*s> / Value <%.*s>", i,
			(int) sdp_session->media[0]->attr[i]->name.slen, sdp_session->media[0]->attr[i]->name.ptr, 
			(int) sdp_session->media[0]->attr[i]->value.slen, sdp_session->media[0]->attr[i]->value.ptr);


		if (strncmp(sdp_session->media[0]->attr[i]->name.ptr, "ptime", 5) == 0)
		{
			voip_profile->ptime = strtol(sdp_session->media[0]->attr[i]->value.ptr, NULL, 10);
			PDEBUG(DBG_L1, "get_local_sdp_media_info() - Negotiated Packet time = %d", voip_profile->ptime);

			break;
		}
	}

	PDEBUG(DBG_L1, "get_local_sdp_media_info() payload_type = %d  - codec %d  -  ptime = %d", 
		voip_profile->payload_type, voip_profile->codec, voip_profile->ptime);
}


/*!
*	Callback function called by PJSIP when receiving an INVITE message (incoming call).
*	\param *acc_id Pointer to a given SIP account Id.
*	\param call_id Caller Id.
*	\param *rdata Pointer to the receive data.
*	\retval None
*/
static void on_incoming_call(pjsua_acc_id acc_id, pjsua_call_id call_id, pjsip_rx_data *rdata)
{
	pjsua_call_info ci;
	long sip_user_id;

	sip_session_t *calling_sip_session;
	sip_session_t *called_sip_session;
	sip_phone_t *sip_phone;

	PJ_UNUSED_ARG(acc_id);
	PJ_UNUSED_ARG(rdata);

	/* get call info, starting by SIP call id */
	pjsua_call_get_info(call_id, &ci);

	PJ_LOG(3,(THIS_FILE, "Incoming call from %.*s!!", (int)ci.remote_info.slen, ci.remote_info.ptr));
	display_sip_call_info(call_id, &ci);

	/* check called SIP acc */

	/* extract the SIP user id from the local URI */
	sip_user_id = (long) get_user_id_from_uri(&ci.local_info);
	if (sip_user_id == -1)
		return;

	/* get the SIP acc associated to this incoming SIP call, if any! */
	called_sip_session = get_sip_session_from_user_id(sip_user_id);

	/* check the SIP account is idle hence available for this SIP incoming call */
	called_sip_session = check_sip_session_is_idle(called_sip_session);
	if (called_sip_session == NULL)
		return;

	if (called_sip_session->endpt->id < MAX_POTS_ENDPT_ID) /* SIP to POTS call */
	{
		called_sip_session->endpt->call = CALL_SIP2POTS;
		called_sip_session->call_id = call_id;
		endpt_raise_event(called_sip_session->endpt, EVENT_SIP2POTS_CALL);
	}
	else /* SIP to SIP call */
	{
		/* check calling SIP acc (should be map to a remote SIP phone to call) */

		/* the SIP user id associated to this SIP incoming call is extracted from the remote SIP info */
		sip_user_id = (long) get_user_id_from_uri(&ci.remote_info);

		sip_phone = get_sip_phone_from_user_id(sip_user_id);
		if (sip_phone == NULL)
			return;

		/* check the SIP account is idle hence available for this SIP incoming call */
		calling_sip_session = check_sip_session_is_idle(sip_phone->sip_session);
		if (calling_sip_session == NULL)
			return;

		/* map calling party to called party */
		calling_sip_session->endpt->peer_endpt = called_sip_session->endpt;
		called_sip_session->endpt->peer_endpt = calling_sip_session->endpt;

		calling_sip_session->call_id = call_id;

		called_sip_session->endpt->call = CALL_VOIP2SIP;
		calling_sip_session->endpt->call = CALL_SIP2VOIP;

		endpt_raise_event(called_sip_session->endpt, EVENT_VOIP2SIP_CALL);
		endpt_raise_event(calling_sip_session->endpt, EVENT_SIP2VOIP_CALL);
	}

	PDEBUG(DBG_L3, "on_incoming_call() - Endpoint %d", called_sip_session->endpt->id);
}


/*!
*	Callback function called by the PJSIP library when call's state has changed.
*	\param call_id Caller Id.
*	\param *e Pointer to PJSIP event.
*	\retval None
*	\see pjsua_call_get_info
*	\see pjmedia_sdp_neg_get_active_local
*	\see pjmedia_sdp_neg_get_active_remote
*/
static void on_call_state(pjsua_call_id call_id, pjsip_event *e)
{
	pjsua_call *call;
	pjsua_call_info ci;
	const pjmedia_sdp_session *local_sdp;
	const pjmedia_sdp_session *remote_sdp;
	sip_session_t *sip_session;

	pjsua_call_get_info(call_id, &ci);
/*	PJ_LOG(3,(THIS_FILE, "Call %d state=%.*s", call_id, (int)ci.state_text.slen, ci.state_text.ptr));*/


	switch(ci.state)
	{
	case PJSIP_INV_STATE_NULL:		/* Before INVITE is sent or received */
		break;

	case PJSIP_INV_STATE_CALLING:		/* After INVITE is sent */
		PJ_LOG(3,(THIS_FILE, "Call %d state=%.*s - PJSIP_INV_STATE_CALLING", call_id, (int)ci.state_text.slen, ci.state_text.ptr));
		break;

	case PJSIP_INV_STATE_INCOMING:		/* After INVITE is received */
		PJ_LOG(3,(THIS_FILE, "Call %d state=%.*s - PJSIP_INV_STATE_INCOMING", call_id, (int)ci.state_text.slen, ci.state_text.ptr));
		break;

	case PJSIP_INV_STATE_EARLY:		/* After response with To tag */
		PJ_LOG(3,(THIS_FILE, "Call %d state=%.*s - PJSIP_INV_STATE_EARLY", call_id, (int)ci.state_text.slen, ci.state_text.ptr));
		break;

	case PJSIP_INV_STATE_CONNECTING:	/* After 2xx is sent/received */
		PJ_LOG(3,(THIS_FILE, "Call %d state=%.*s - PJSIP_INV_STATE_CONNECTING", call_id, (int)ci.state_text.slen, ci.state_text.ptr));
		break;

	case PJSIP_INV_STATE_CONFIRMED:		/* After ACK is sent/received */
		PJ_LOG(3,(THIS_FILE, "Call %d state=%.*s - PJSIP_INV_STATE_CONFIRMED", call_id, (int)ci.state_text.slen, ci.state_text.ptr));

		sip_session = get_sip_session_from_call_id(call_id);
		if (sip_session == NULL)
			return;

		call = &pjsua_var.calls[call_id];

		/* Get local and remote SDP */
		sip_status = pjmedia_sdp_neg_get_active_local(call->inv->neg, &local_sdp);
		if (sip_status != PJ_SUCCESS)
			sip_error_exit("Error in pjmedia_sdp_neg_get_active_local()", sip_status);

		sip_status = pjmedia_sdp_neg_get_active_remote(call->inv->neg, &remote_sdp);
		if (sip_status != PJ_SUCCESS)
			sip_error_exit("Error in pjmedia_sdp_neg_get_active_remote()", sip_status);

		update_sdp_session_local_info(sip_session, local_sdp);
		update_sdp_session_remote_info(sip_session, remote_sdp);


/*
	PDEBUG(1, " IP addr     %.*s", (int) sdp_session->conn->addr.slen, sdp_session->conn->addr.ptr);
	PDEBUG(1, " UDP port    %d",   (int) sdp_session->media[0]->desc.port);
	PDEBUG(1, " Media Type  %.*s", (int) sdp_session->media[0]->desc.media.slen, sdp_session->media[0]->desc.media.ptr);
*/

/*
		display_sip_call_info(call_id, &ci);
		display_sip_call_info(call_id, &ci);
		display_sdp_session_info(local_sdp);
		display_sdp_session_info(remote_sdp);
*/

		PDEBUG(DBG_L1, "\n\nendpt %d\n   src IP %d.%d.%d.%d\n   src UDP %d\n   dst IP %d.%d.%d.%d\n   dst UDP %d\n\n", 
			sip_session->endpt->id, sip_session->endpt->ip[0], sip_session->endpt->ip[1], 
			sip_session->endpt->ip[2], sip_session->endpt->ip[3],
 			sip_session->endpt->rtp,
			sip_session->dst_ip[0], sip_session->dst_ip[1], sip_session->dst_ip[2], 
			sip_session->dst_ip[3], sip_session->dst_rtp);

		endpt_raise_event(sip_session->endpt, EVENT_SIP_CONFIRMED);
		break;

	case PJSIP_INV_STATE_DISCONNECTED:	/* Session is terminated */
		PJ_LOG(3,(THIS_FILE, "Call %d state=%.*s - PJSIP_INV_STATE_DISCONNECTED", call_id, (int)ci.state_text.slen, ci.state_text.ptr));

		display_sip_call_info(call_id, &ci);

		sip_session = get_sip_session_from_call_id(call_id);
		if (sip_session == NULL)
			return;

		endpt_raise_event(sip_session->endpt, EVENT_SIP_DISCONNECTED);
		break;

	default:
		break;
	}
}

/*!
*	Callback function called by the PJSIP library when call's media state has changed.
*	\param call_id Caller Id
*	\retval None
*	\see pjsua_call_get_info
*	\see pjsua_conf_connect
*/
static void on_call_media_state(pjsua_call_id call_id)
{
	pjsua_call_info ci;

	pjsua_call_get_info(call_id, &ci);

	if (ci.media_status == PJSUA_CALL_MEDIA_ACTIVE) {
		// When media is active, connect call to sound device.
		pjsua_conf_connect(ci.conf_slot, 0);
		pjsua_conf_connect(0, ci.conf_slot);
	}
#if 0
	switch (ci.media_status) 
	{
    	/* Call currently has no media */
	case PJSUA_CALL_MEDIA_NONE:
		printf("............. MEDIA_NONE\n");
		break;

	/* The media is active */
	case PJSUA_CALL_MEDIA_ACTIVE:
		printf("............. MEDIA_ACTIVE\n");
		// When media is active, connect call to sound device.
		pjsua_conf_connect(ci.conf_slot, 0);
		pjsua_conf_connect(0, ci.conf_slot);
		break;

	/* The media is currently put on hold by local endpt */
	case PJSUA_CALL_MEDIA_LOCAL_HOLD:
		printf("............. MEDIA_LOCAL_HOLD\n");
		break;

	/* The media is currently put on hold by remote endpt */
	case PJSUA_CALL_MEDIA_REMOTE_HOLD:
		printf("............. MEDIA_REMOTE_HOLD\n");
		break;

	/* The media has reported error (e.g. ICE negotiation) */
	case PJSUA_CALL_MEDIA_ERROR:
		printf("............. MEDIA_ERROR\n");
		break;
	}
#endif
}


/*!
*	This function initializes all SIP parameters to default values.
*	\param *sipua Pointer to the sipua instance.
*	\retval None
*	\see pjsua_config_default
*	\see pjsua_logging_config_default
*	\see pjsua_media_config_default
*	\see pjsua_transport_config_default
*	\see pjsua_acc_config_default
*	\see pjsua_buddy_config_default
*/
static void sip_default_config(sipua_t *sipua)
{
	char tmp[80];
	unsigned i;

	pjsua_config_default(&sipua->cfg);

	pj_ansi_sprintf(tmp, "PJSUA v%s/%s", pj_get_version(), PJ_OS_NAME);
/*	pj_strdup2_with_null(sip_config.pool, &cfg->cfg.user_agent, tmp); */

	pjsua_logging_config_default(&sipua->log_cfg);
	pjsua_media_config_default(&sipua->media_cfg);
	pjsua_transport_config_default(&sipua->udp_cfg);
	sipua->udp_cfg.port = sipua_cfg->sip_server.udp;	/* expected to be 5060 for SIP */
	pjsua_transport_config_default(&sipua->rtp_cfg);
	sipua->rtp_cfg.port = DEFAULT_UDP_PORT;

/*
	cfg->redir_op = PJSIP_REDIRECT_ACCEPT;
	cfg->duration = NO_LIMIT;
	cfg->wav_id = PJSUA_INVALID_ID;
	cfg->rec_id = PJSUA_INVALID_ID;
	cfg->wav_port = PJSUA_INVALID_ID;
	cfg->rec_port = PJSUA_INVALID_ID;
	cfg->mic_level = cfg->speaker_level = 1.0;
	cfg->capture_dev = PJSUA_INVALID_ID;
	cfg->playback_dev = PJSUA_INVALID_ID;
	cfg->capture_lat = PJMEDIA_SND_DEFAULT_REC_LATENCY;
	cfg->playback_lat = PJMEDIA_SND_DEFAULT_PLAY_LATENCY;
	cfg->ringback_slot = PJSUA_INVALID_ID;
	cfg->ring_slot = PJSUA_INVALID_ID;
*/

	/* Register to SIP server by creating SIP account. */
	for (i=0; i<PJ_ARRAY_SIZE(sipua->acc_cfg); ++i)
		pjsua_acc_config_default(&sipua->acc_cfg[i]);

	for (i=0; i<PJ_ARRAY_SIZE(sipua->buddy_cfg); ++i)
		pjsua_buddy_config_default(&sipua->buddy_cfg[i]);
}


/*!
*	This function initializes the specified SIP phone instance.
*	\param *sip_phone Pointer to a given SIP phone instance.
*	\retval None
*/
void sip_phone_init(sip_phone_t *sip_phone)
{
	pj_str_t uri;
	char sip_uri[SIP_URI_MAX_SIZE];

	if (sip_phone == NULL)
		return;

	strcpy(sip_uri, sip_phone->uri);
	uri = pj_str(sip_uri);
	sip_phone->user_id = get_user_id_from_uri(&uri);

	strcpy(sip_uri, sip_phone->uri);
	uri = pj_str(sip_uri);
	get_ip_from_uri(&uri, &sip_phone->ip[0]);

	PDEBUG(DBG_L3, "sip_phone_init %d\n", sipua_cfg->max_sip_phones);
}


/*!
*	This function resets the specified SIP session instance.
*	\param *sip_session Pointer to a given SIP session instance.
*	\retval None
*/
void sip_session_reset(sip_session_t *sip_session)
{
	sip_session->dst_rtp = -1;
	sip_session->dst_rtcp = -1;
	sip_session->call_id = -1;
}

/*!
*	This function initializes the specified SIP session instance.
*	\param id SIP session Id
*	\param *sip_session Pointer to a given SIP session instance.
*	\param *endpt Pointer to a given endpt instance.
*	\param *sip_phone Pointer to a given SIP phone instance.
*	\param *acc_cfg Pointer to a given SIP account configuration structure.
*	\retval None
*/
void sip_session_init(int id, sip_session_t *sip_session, endpt_t *endpt, sip_phone_t *sip_phone, pjsua_acc_config *acc_cfg)
{
	memset(sip_session, 0, sizeof(sip_session_t));

	sip_session->id = id;

	/* IP address is the Comcerto device source IP */
	sip_session->user_id = sipua_cfg->first_user_id + endpt->phone_id;

	sip_session_reset(sip_session);

	sip_session->acc_cfg = acc_cfg;	

	/* map to endpt (basically sip_session[i] is mapped to IP endpoints (skip POTS ones) */
	endpt->sip_session = sip_session;
	sip_session->endpt = endpt;

	if (sip_phone != NULL)
	{
		/* no associated sip phone so far */
		sip_session->sip_phone = sip_phone;
		sip_phone->sip_session = sip_session;
	}
}


/*!
*	This function initiates an outgoing SIP call to the specified SIP phone.
*	\param *sip_phone Pointer to a given SIP phone instance.
*	\param *call_id Pointer to a given Caller Id information.
*	\param acc_id SIP account Id.
*	\retval 0 on success.
*	\retval <0 on failure.
*	\see pjsua_call_make_call
*/
int call_sip_phone(sip_phone_t *sip_phone, pjsua_call_id *call_id, int acc_id)
{
	char sip_uri[SIP_URI_MAX_SIZE];
	pj_str_t uri;
	int rc = 0; /* return code */

	exit_if_null(sip_phone, err, "call_sip_phone");

	strcpy(sip_uri, sip_phone->uri);

	uri = pj_str(sip_uri);

	rc = pjsua_call_make_call(acc_id, &uri, 0, NULL, NULL, call_id);
	exit_on_err(rc, err, "call_sip_phone: error making call");
						
	PDEBUG(DBG_L1, "call SIP phone %.*s (call id %d)", (int) uri.slen, uri.ptr, *call_id);

	return rc;

err:
	sip_error_exit("Error making call", -1);
	return rc;
}


/*!
*	This function acknowledges an incoming INVITE to proceed with the incoming call associated to the specified SIP session.
*	\param *sip_session Pointer to a given SIP session instance.
*	\retval 0 on success.
*	\retval <0 on failure.
*	\see pjmedia_sdp_neg_set_prefer_remote_codec_order
*	\see pjmedia_sdp_neg_get_neg_local
*	\see pjmedia_sdp_neg_get_neg_remote
*	\see pjsua_call_answer
*/
int ack_sip_invite(sip_session_t *sip_session)
{
	int rc = 0; /* return code */
	pjsua_call *call;
	const pjmedia_sdp_session *local_sdp;
	const pjmedia_sdp_session *remote_sdp;

	call = &pjsua_var.calls[sip_session->call_id];

	pjmedia_sdp_neg_set_prefer_remote_codec_order(call->inv->neg, 1); /* 0 = local, 1 = remote */

	/* Get local and remote SDP */
	sip_status = pjmedia_sdp_neg_get_neg_local(call->inv->neg, &local_sdp);
	if (sip_status != PJ_SUCCESS)
		sip_error_exit("========================================================== Error in pjmedia_sdp_neg_get_neg_local()", sip_status);

	display_sdp_session_local_info(local_sdp);

	sip_status = pjmedia_sdp_neg_get_neg_remote(call->inv->neg, &remote_sdp);
	if (sip_status != PJ_SUCCESS)
		sip_error_exit("========================================================== Error in pjmedia_sdp_neg_get_neg_remote()", sip_status);

	display_sdp_session_local_info(remote_sdp);


	/* automatically answer incoming SIP call with 200/OK */
	rc = pjsua_call_answer(sip_session->call_id, 200, NULL, NULL);

	return rc;
}


/*!
*	This function hangs up the SIP call associated to the specified SIP session.
*	\param *sip_session Pointer to a given SIP session instance.
*	\retval 0 on success.
*	\retval <0 on failure.
*	\see pjsua_call_hangup
*/
int sip_session_hangup(sip_session_t *sip_session)
{
	int rc = 0; /* return code */

	if (pjsua_var.calls[sip_session->call_id].inv)
	{
		PDEBUG(DBG_L1, "--> SIP Disconnect (pjsua_call_hangup) SIP Call id %d", sip_session->call_id);
		pjsua_call_hangup(sip_session->call_id, 0, NULL, NULL);

		sip_session_reset(sip_session);
	}

	return rc;
}


/*!
*	This function is the called whenever a major SIP error shows up.\n
*	\param *title Error message to be displayed (string).
*	\param status PJSIP status.
*	\retval None
*	\see pjsua_perror
*	\see pjsua_destroy
*/
void sip_error_exit(const char *title, pj_status_t status)
{
    pjsua_perror(THIS_FILE, title, status);
    pjsua_destroy();
    exit(1);
}


/*!
*	This function initializes the SIPUA instance.
*	\param *sipua Pointer to the SIPUA instance.
*	\retval 0 on success.
*	\retval <0 on failure.
*/
int sipua_init(sipua_t *sipua)
{
	int i;
	endpt_t * endpt;
	sip_session_t *sip_session;
	sip_phone_t *sip_phone;
	pjsua_acc_config *pjsua_acc;

	int pjsua_acc_id = 0;
	int status;

	for (i = 0; i < MAX_ENDPT_ID; i++)
	{
		sip_session = &sipua->sip_session[i];
		endpt = &endpt_pool[i];
	
		sip_phone =NULL;

		if (i >= MAX_POTS_ENDPT_ID)
		{
			sip_phone = get_free_sip_phone(sipua);
			if (sip_phone != NULL)
				sip_phone_init(sip_phone);
		}

		pjsua_acc = NULL;

		status = is_sip_registered(sipua, pjsua_acc_id, i);
		if (status >= 0)
		{
			pjsua_acc = &sipua->acc_cfg[pjsua_acc_id];
			pjsua_acc_id++;
		}

		sip_session_init(i, sip_session, endpt, sip_phone, pjsua_acc);
	}

	return 0;
}


/*!
*	This function initializes a given SIP account and registers it to the SIP server if required.
*	\param *acc_cfg Pointer to a given SIP account.
*	\param user_id Pointer to the SIP User Id value (associated to the SIP account).
*	\param *uri Pointer to the SIP URI (associated to the SIP account.
*	\retval None
*	\see pjsua_acc_add
*/
void sip_acc_create_and_register(pjsua_acc_config *acc_cfg, char *user_id, char *uri)
{
	/* Initialize the pjsua part */
	acc_cfg->cred_count = 1;
	acc_cfg->id = pj_str(uri);
	acc_cfg->cred_info[0].realm = pj_str(sipua_cfg->sip_server.domain);
	acc_cfg->cred_info[0].scheme = pj_str("digest");
	acc_cfg->cred_info[0].username = pj_str(user_id);
	acc_cfg->cred_info[0].data_type = PJSIP_CRED_DATA_PLAIN_PASSWD;
	acc_cfg->cred_info[0].data = pj_str(SIP_PASSWD);

	/* Register to SIP Server only if user required */
	if (sipua_cfg->sip_server.registration > 0)
		acc_cfg->reg_uri = pj_str(sipua_cfg->sip_server.uri);


	sip_status = pjsua_acc_add(acc_cfg, PJ_TRUE, NULL);
	if (sip_status != PJ_SUCCESS)
    		sip_error_exit("Error adding account", sip_status);

	if (sipua_cfg->sip_server.registration)
		PDEBUG(1, "Registering to SIP Server\n");
}


/*!
*	This function displays the list of the PJSIP supported and allowed codecs.
*	\param *sipua Pointer to a the sipua instance.
*	\retval None
*	\see pjsua_enum_codecs
*/
void sipua_display_codecs(sipua_t *sipua)
{
	pjsua_codec_info codec[32];
	unsigned uCount = 32;
	unsigned k;

	memset(codec, 0, sizeof(codec));

	if (pjsua_enum_codecs(codec, &uCount) == PJ_SUCCESS)
	{
		PDEBUG(DBG_L3, "List of codecs:");
		for (k=0; k<uCount; ++k) {
			PDEBUG(DBG_L3, " [%d] - priority %d\tcodec %.*s\n", k, codec[k].priority, (int)codec[k].codec_id.slen, codec[k].codec_id.ptr);
		}
	}
}


/*!
*	This function sets the priority for a given PJSIP codec.
*	\param *codec_label String defining a given codec.
*	\param codec_priority Integer defining the codec priority (a higher value means a higher priority).
*	\retval None
*	\see pjsua_codec_set_priority
*/
void sipua_set_codec_priority(char *codec_label, int codec_priority)
{
	pj_str_t tmp;

	if (codec_priority > 0)
	{
		PDEBUG(DBG_L1, "sipua_set_codec_priority (codec %s priority %d)", codec_label, codec_priority);
		pjsua_codec_set_priority(pj_cstr(&tmp, codec_label), codec_priority);
	}
	else
		pjsua_codec_set_priority(pj_cstr(&tmp, codec_label), PJMEDIA_CODEC_PRIO_DISABLED);
}

/*!
*	This function sets the priority for all supported codecs based on user requirements.
*	\retval None
*/
void sipua_set_all_codec_priorities(void)
{
	sipua_set_codec_priority("g722/8000", sipua_cfg->codec_priority_g722_8000);
	sipua_set_codec_priority("g722/16000", sipua_cfg->codec_priority_g722_16000);

	sipua_set_codec_priority("g7221/16000", sipua_cfg->codec_priority_g7221_16000);
	sipua_set_codec_priority("g7221/24000", sipua_cfg->codec_priority_g7221_24000);
	sipua_set_codec_priority("g7221/32000", sipua_cfg->codec_priority_g7221_32000);

	sipua_set_codec_priority("g723/8000", sipua_cfg->codec_priority_g723_8000);

	sipua_set_codec_priority("g726-16", sipua_cfg->codec_priority_g726_16);
	sipua_set_codec_priority("g726-24", sipua_cfg->codec_priority_g726_24);
	sipua_set_codec_priority("g726-32", sipua_cfg->codec_priority_g726_32);
	sipua_set_codec_priority("g726-40", sipua_cfg->codec_priority_g726_40);

	sipua_set_codec_priority("g728/8000", sipua_cfg->codec_priority_g728_8000);

	sipua_set_codec_priority("g729/8000", sipua_cfg->codec_priority_g729_8000);

	sipua_set_codec_priority("amr", sipua_cfg->codec_priority_amr);
	sipua_set_codec_priority("amr-wb", sipua_cfg->codec_priority_amr_wb);

	sipua_set_codec_priority("gsm", sipua_cfg->codec_priority_gsm);

	sipua_set_codec_priority("ilbc", sipua_cfg->codec_priority_ilbc);

	sipua_set_codec_priority("pcma", sipua_cfg->codec_priority_pcma);
	sipua_set_codec_priority("pcmu", sipua_cfg->codec_priority_pcmu);

	sipua_set_codec_priority("speex/8000", sipua_cfg->codec_priority_speex_8000);
	sipua_set_codec_priority("speex/16000", sipua_cfg->codec_priority_speex_16000);
	sipua_set_codec_priority("speex/32000", sipua_cfg->codec_priority_speex_32000);


/*
	pjsua_codec_set_priority(pj_cstr(&tmp, "g722/8000"), sipua_cfg->codec_priority_g722_8000);
	pjsua_codec_set_priority(pj_cstr(&tmp, "g722/16000"), sipua_cfg->codec_priority_g722_16000);

	pjsua_codec_set_priority(pj_cstr(&tmp, "g7221/16000"), sipua_cfg->codec_priority_g7221_16000);
	pjsua_codec_set_priority(pj_cstr(&tmp, "g7221/24000"), sipua_cfg->codec_priority_g7221_24000);
	pjsua_codec_set_priority(pj_cstr(&tmp, "g7221/32000"), sipua_cfg->codec_priority_g7221_32000);

	pjsua_codec_set_priority(pj_cstr(&tmp, "g723/8000"), sipua_cfg->codec_priority_g723_8000);

	pjsua_codec_set_priority(pj_cstr(&tmp, "g726-16"), sipua_cfg->codec_priority_g726_16);
	pjsua_codec_set_priority(pj_cstr(&tmp, "g726-24"), sipua_cfg->codec_priority_g726_24);
	pjsua_codec_set_priority(pj_cstr(&tmp, "g726-32"), sipua_cfg->codec_priority_g726_32);
	pjsua_codec_set_priority(pj_cstr(&tmp, "g726-40"), sipua_cfg->codec_priority_g726_40);

	pjsua_codec_set_priority(pj_cstr(&tmp, "g728/8000"), sipua_cfg->codec_priority_g728_8000);

	pjsua_codec_set_priority(pj_cstr(&tmp, "g729/8000"), sipua_cfg->codec_priority_g729_8000);

	pjsua_codec_set_priority(pj_cstr(&tmp, "amr"), sipua_cfg->codec_priority_amr);
	pjsua_codec_set_priority(pj_cstr(&tmp, "amr-wb"), sipua_cfg->codec_priority_amr_wb);

	pjsua_codec_set_priority(pj_cstr(&tmp, "gsm"), sipua_cfg->codec_priority_gsm);

	pjsua_codec_set_priority(pj_cstr(&tmp, "ilbc"), sipua_cfg->codec_priority_ilbc);

	pjsua_codec_set_priority(pj_cstr(&tmp, "pcma"), sipua_cfg->codec_priority_pcma);
	pjsua_codec_set_priority(pj_cstr(&tmp, "pcmu"), sipua_cfg->codec_priority_pcmu);

	pjsua_codec_set_priority(pj_cstr(&tmp, "speex/8000"), sipua_cfg->codec_priority_speex_8000);
	pjsua_codec_set_priority(pj_cstr(&tmp, "speex/16000"), sipua_cfg->codec_priority_speex_16000);
	pjsua_codec_set_priority(pj_cstr(&tmp, "speex/32000"), sipua_cfg->codec_priority_speex_32000);
*/
}


/*!
*	This function intializes the PJSIP stack.
*	\param *sipua Pointer to a the sipua instance.
*	\retval None
*	\see pjsua_create
*	\see pjsua_init
*	\see pj_thread_is_registered
*	\see pjsua_transport_create
*	\see pjsua_media_transports_create
*	\see pjsua_buddy_add
*	\see pjsua_codec_set_priority
*	\see pjsua_start
*/
void sipua_create(sipua_t *sipua)
{
	int i;
	int acc_id;

	/* Create pjsua first! */
	sip_status = pjsua_create();
	if (sip_status != PJ_SUCCESS)
 		sip_error_exit("Error in pjsua_create()", sip_status);

	/* Apply our default SIP configuration */
	sip_default_config(sipua);

	/* Initialize application callbacks */
	sipua->cfg.cb.on_incoming_call = &on_incoming_call;
	sipua->cfg.cb.on_call_media_state = &on_call_media_state;
	sipua->cfg.cb.on_call_state = &on_call_state;


	/* Initialize pjsua */
	sip_status = pjsua_init(&sipua->cfg, &sipua->log_cfg, &sipua->media_cfg);
	if (sip_status != PJ_SUCCESS)
		sip_error_exit("Error in pjsua_init()", sip_status);
/*	
	{
		pjsua_codec_info codec_info[PJMEDIA_CODEC_MGR_MAX_CODECS];
		int i, max_codec;

		pjsua_enum_codecs(codec_info, &max_codec);
		printf("Max Codecs = %d  --------------\n", max_codec);
		for (i=0; i<max_codec; i++)
			printf("--------- Priority %d\tCodec %.*s\n", 
				codec_info[i].priority, (int)codec_info[i].codec_id.slen, codec_info[i].codec_id.ptr);
	}
*/
	if (!pj_thread_is_registered())
		PDEBUG(DBG_ERR, "!ERR! Thread not registered to PJLIB");

	sip_status = pjsua_transport_create(PJSIP_TRANSPORT_UDP, &sipua->udp_cfg, NULL);
	if (sip_status != PJ_SUCCESS)
		sip_error_exit("Error creating transport", sip_status);

	sip_status = pjsua_media_transports_create(&sipua->rtp_cfg);
	if (sip_status != PJ_SUCCESS)
		sip_error_exit("Error creating media (RTP) transport", sip_status);

	/* Add buddies */
	for (i=0; i<sipua->buddy_cnt; ++i) {
		sip_status = pjsua_buddy_add(&sipua->buddy_cfg[i], NULL);
		if (sip_status != PJ_SUCCESS)
			sip_error_exit("Error adding buddies", sip_status);
    	}

	/* Set codec priority */
	sipua_set_all_codec_priorities();

	/* Display all supported codecs information */
	sipua_display_codecs(sipua);

	/* Initialization is done, now start pjsua */
	sip_status = pjsua_start();
	if (sip_status != PJ_SUCCESS)
		sip_error_exit("Error starting pjsua", sip_status);

	/* Add PJSIP accounts and register to the SIP server if required */
	PDEBUG(DBG_L1, "Add MAX SIP accounts: %d", PJSUA_MAX_ACC);

	acc_id = 0;

	for (i = 0; i < MAX_ENDPT_ID; i++)
	{
		char uri[SIP_URI_MAX_SIZE];
		char user_id[SIP_USER_ID_MAX_SIZE];

		sprintf(user_id, "%ld", sipua->sip_session[i].user_id);

		sprintf(uri, "sip:%ld@%d.%d.%d.%d", sipua->sip_session[i].user_id, sipua_cfg->sip_server.ip[0],
			sipua_cfg->sip_server.ip[1], sipua_cfg->sip_server.ip[2], sipua_cfg->sip_server.ip[3]);

		if (i < MAX_POTS_ENDPT_ID)
		{
			if (i >= sipua_cfg->nb_sip_tdm_acc)
				continue;
		}
		else
		{
			if (i - MAX_POTS_ENDPT_ID >= sipua_cfg->nb_sip_ip_acc)
				continue;
		}

		sip_acc_create_and_register(&sipua->acc_cfg[acc_id], user_id, uri);
		acc_id++; 
	}
}

/*!	@} */
