/*!
*	\file netif.c
*
*	\defgroup netif netif Module
*
*	netif Module including all C-functions and declarations
*	(defines, types, variables)
*
*	\attention
*	Copyright © 2004-2010 Mindspeed Technologies, Inc.\n
*	Mindspeed Confidential.\n
*	All rights reserved.\n
*	This file is a component of the Mindspeed® VAPI software ("VAPI") and is
*	distributed under the Mindspeed Software License Agreement (the "Agreement").\n
*	Before using this file, you must agree to be bound by the the terms and 
*	conditions of the Agreement.
*
*	@{
*/

#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <string.h>

/* VAPI headers */
#include <msp.h>
#include <vapi.h>
#include <gtl.h>
#include <comcerto-api-defs.h>
#include <comcerto-ud-types.h>

/* Zarlink headers */
#include <legerity_lib.h>

/* mtalk header */
#include <mtalk.h>

/*!
*	This function allocates and initializes a netif_cfg instance with 
*	hardcoded and/or user defined values.
*	\param *cfg_file Configuration filename (string).
*	\retval NULL on failure
*	\retval *netif_cfg_t on success, pointer to the allocated and initialized netif_cfg structure
*/
netif_cfg_t *get_netif_cfg(const char *cfg_file)
{
	struct _CFG *cfg_info;
	netif_cfg_t *netif_cfg;

	netif_cfg = (netif_cfg_t *)malloc(sizeof(netif_cfg_t));
	exit_if_null(netif_cfg, err, "get_netif_cfg() - mem allocation fail");

	cfg_info = cfg_read(cfg_file, 0);
	exit_if_null(cfg_info, err, "get_netif_cfg() - mem allocation fail");

	READ_IP(cfg_info, "NETIF", "DEVICE_IP", netif_cfg->device_ip, DEVICE_IP);
	READ_MAC(cfg_info,"NETIF", "DEVICE_MAC", netif_cfg->device_mac, (U8 *) DEVICE_MAC);
	READ_MAC(cfg_info,"NETIF", "HOST_MAC", netif_cfg->host_mac, (U8 *) HOST_MAC);

	cfg_clean(cfg_info);

	return netif_cfg;

err:
	return NULL;
}


/*!
*	This function displays the values of netif parameters (netif_cfg).
*	\retval *netif_cfg_t Pointer to the netif configuration parameter structure.
*	\retval None
*/
void display_netif_cfg(netif_cfg_t *netif_cfg)
{
	if (!netif_cfg)
		return;

	PDEBUG(DBG_L2, "\n-------------------- netif_cfg");
	PDEBUG(DBG_L2,"- device_ip   %d.%d.%d.%d", 
		netif_cfg->device_ip[0], netif_cfg->device_ip[1], netif_cfg->device_ip[2], netif_cfg->device_ip[3]);

	PDEBUG(DBG_L2,"- device_mac  %02x:%02x:%02x:%02x:%02x:%02x",
		netif_cfg->device_mac[0], netif_cfg->device_mac[1], netif_cfg->device_mac[2],
		netif_cfg->device_mac[3], netif_cfg->device_mac[4], netif_cfg->device_mac[5]);

	PDEBUG(DBG_L2,"- host_mac    %02x:%02x:%02x:%02x:%02x:%02x",
		netif_cfg->host_mac[0], netif_cfg->host_mac[1], netif_cfg->host_mac[2],
		netif_cfg->host_mac[3], netif_cfg->host_mac[4], netif_cfg->host_mac[5]);
	PDEBUG(DBG_L2, "--------------------\n");
}


/*!
*	This function initializes the netif module.
*	\param netif_cfg Pointer to the netif module configuration (structure).
*	\retval None
*	\sa VAPI_SetEthMac
*	\sa VAPI_SetDeviceIPAddr
*/
int netif_init(netif_cfg_t *netif_cfg)
{
	SIpAddrInfo ip_info;
	int rc; /* return code */

	PDEBUG(DBG_L3, "netif_init(netif_cfg 0x%lx)", (unsigned long) netif_cfg);

	rc = VAPI_SetEthMac(0, CMD_LEVEL_DEVICE, (U8 *)&netif_cfg->device_mac[0], (U8 *)&netif_cfg->host_mac[0], NULL);
	exit_on_err(rc, exit, "netif_init -> VAPI_SetEthMac");

	ip_info.ucNumOfSrcIpAddr = 1;
	ip_info.bIsMultipleMode = 0;
	ip_info.ucEncapsulation = 2;
	ip_info.ucChkSumVerification = 1;

	ip_info.auiDevIPAddress[0] = (netif_cfg->device_ip[3] << 24) | (netif_cfg->device_ip[2] << 16) | 
					(netif_cfg->device_ip[1] << 8) | netif_cfg->device_ip[0];

	rc |= VAPI_SetDeviceIPAddr(0, &ip_info, NULL);
	exit_on_err(rc, exit, "netif_init -> VAPI_SetDeviceIPAddr");
exit:
	return rc;
}


/*!
*	This function sets the IP/UDP header of a endpt using VAPI_SetConnIpParams() in sync mode.
*	\param conn_id Connection Id.
*	\param src_ip Source IP address (string).
*	\param dst_ip Destination IP address (string).
*	\param src_rtp Source RTP port.
*	\param dst_rtp Destination TP port.
*	\retval 0 if success
*	\retval <0 on failure.
*	\sa VAPI_SetConnIpParams
*	\sa VAPI_SetRtpSsrcHeader
*/
int netif_modify(int conn_id, unsigned char *src_ip, unsigned char *dst_ip, int src_rtp, int dst_rtp)
{
	SIpParams ip_parameters;
	int rc;

	PDEBUG(DBG_L3, "netif_modify(ConnId %d): src (%d.%d.%d.%d, %d) -> dst (%d.%d.%d.%d, %d)", 
			conn_id, src_ip[0], src_ip[1], src_ip[2], src_ip[3], src_rtp,
				 dst_ip[0], dst_ip[1], dst_ip[2], dst_ip[3], dst_rtp);

	/* make sure all fields are set to 0, so ucIPServiceId field is set to RTP */
	memset(&ip_parameters, 0, sizeof(SIpParams));

	ip_parameters.uiSrcIpAddr = (src_ip[3] << 24) | (src_ip[2] << 16) | (src_ip[1] << 8) | src_ip[0];
	ip_parameters.usSrcUdpPort = src_rtp;

	ip_parameters.uiDestIpAddr = (dst_ip[3] << 24) | (dst_ip[2] << 16) | (dst_ip[1] << 8) | dst_ip[0];
	ip_parameters.usDestUdpPort = dst_rtp;

	/* RTP defined in comcerto_ip_device_level_api.h */
	ip_parameters.ucIPServiceId = SET_IP_HDR_CHANNEL_SERVICEID_DEFAULTPKT;

	rc = VAPI_SetConnIpParams(conn_id, &ip_parameters, NULL);
	exit_on_err(rc, exit, "netif_modify -> VAPI_SetConnIpParams");

	rc = VAPI_SetRtpSsrcHeader (conn_id, src_rtp | (dst_rtp << 16), NULL);
	exit_on_err(rc, exit, "netif_modify -> VAPI_SetRtpSsrcHeader");

exit:
	return rc;
}


/*!	@} */
