/*!
*	\file mtalk.c
*
*	@defgroup mtalk mtalk Module
*
*	Mtalk Module including all C-functions and declarations
*	(defines, types, variables)
*
*	\attention
*	Copyright © 2004-2010 Mindspeed Technologies, Inc.\n
*	Mindspeed Confidential.\n
*	All rights reserved.\n
*	This file is a component of the Mindspeed® VAPI software ("VAPI") and is
*	distributed under the Mindspeed Software License Agreement (the "Agreement").\n
*	Before using this file, you must agree to be bound by the the terms and 
*	conditions of the Agreement.
*
*	@{
*/

/*! \mainpage MTALK Developer's Guide 

	Mtalk is a Mindspeed VoiP application handling: 
	- Mindspeed Media Control through the Mindspeed Voice API (VAPI) library services.
	- POTS call control through the Zarlink VP880 SLIC control libraries services.
	- SIP/SDP call through the PJSIP libraries services.

	This example is multi threaded VoIP application intended to be used on M82xxx EVM running in Master mode.
	- sip-to-sip calls
	- sip-to-pots calls
	- pots to sip calls
	
	All the VAPI function calls are performed in synchronous mode.

	<b>VAPI demo components</b>\n
	VAPI demo is composed by a set of binaries, drivers, scripts and configuration files:

	\li MSP Media Stream Processor
	\li CSP Control Stream Processor
	\li SIP Session Information Protocol
	\li SDP Session Description Protocol
*/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <sys/time.h>
#include <getopt.h>

/* VAPI headers */
#include <msp.h>
#include <vapi.h>
#include <gtl.h>

/* Zarlink headers */
#include <legerity_lib.h>

/* mtalk header */
#include <mtalk.h>


#define check_status(title, error_label)					\
	PDEBUG(DEBUG_ERROR, title ": %s", result == SUCCESS?"ok":"failed");	\
	if (result != SUCCESS)							\
		goto error_label

static void show_help(void)
{
	printf(
		"vapi demo - program to make a pots to pots call on Comcerto EVM embeded boards\n"
		"Usage: %s [OPTION]...\n"
		"  -c, --config_file=CONFIGURATION_FILE\n"
		"			Name of the configuration file\n\n"
		"  -d, --vapi_debug_level=VAPI_DEBUG_LEVEL\n"
		"			Vapi Debug Level\n\n"
		"  -D, --app_debug_level=APP_DEBUG_LEVEL\n"
		"			App Debug Level\n\n"
		"      --help		Show this help and exit\n"
		"      --version	Show version and exit\n",
		"vapi_demo"
	);
}

static void show_version(void)
{
	printf("mtalk demo app, %s\n", MTALK_VERSION);
}

static struct option const long_options[] =
{
	{ "config_file",	required_argument, NULL, 'c' },
	{ "vapi_debug_level",	required_argument, NULL, 'd' },
	{ "app_debug_level",	required_argument, NULL, 'D' },

	{ "help",		no_argument, NULL, CHAR_MIN - 1 },
	{ "version",		no_argument, NULL, CHAR_MIN - 2 },

	{ NULL, 0, NULL, 0},
};

/* vapi variables */
SCSMEUsrData gtl_device_configuration; /* structure with config parameters, defined in gtl.h */
int vapi_debug_level;
int app_debug_level;


/* thread variables */
static SThread *tid_signalling;
static SThread *tid_state_machine;


/* config variables */
STdmSetupParams *tdmif_cfg;
netif_cfg_t *netif_cfg;
debug_cfg_t *debug_cfg;
media_cfg_t *media_cfg;

sipua_cfg_t *sipua_cfg;

U8 channel_mode;

/* slic */
slic_t slic;

char cfg_file[FILENAME_MAX];


/* default values used later in default_device_configuration */
/* This structure holds default values for the device configuration, user can
 * overwrite them somehow in set_device_configuration() function using his own
 * config. scheme.
 */
static SCSMEUsrData default_device_configuration = {
	NULL,			/* pointer to next device */
	eCSM_ITF,		/* control interface to use */
	0,			/* device ID */
	0,			/* Version infor (not used) */
	eMASTER,		/* Slave / Master mode */
	DEV_TYPE_M823XX,	/* Device type */
	True,			/* Default or custom max channels */
	0,			/* Max Channels if above flag is custom */
	DEVICE_CTRL_MAC,	/* MAC address for control over csmencaps */
	HOST_CTRL_MAC,	/* MAC address of the host interface controling the device */
	(char *)HOST_CTRL_INTERFACE,	/* host interface used to control the device */
	1			/* csme ack required */
};



/*!
*	This function handles the response to VAPI commands sent in ASYNC mode.
*	It sets event for the endpt accordingly the endpt state. The state is retrieved through the request ID.
*	\param connection_id Connection Id
*	\param command_level
*	\param result
*	\param *response_data
*	\param data_length
*	\param request_id
*	\retval None
*/
void comcerto_response_handler(U32 connection_id, U8 command_level, S32 result,	void *response_data, U32 data_length, U32 request_id)
{

	if (result != SUCCESS)
	{
		PDEBUG(DBG_ERR,"Error on response received on endpt id %d, request id = 0x%04x", connection_id, request_id);
	}
	else
	{
		PDEBUG(DBG_L3,"Response received on endpt id %d, request id = 0x%04x",connection_id, request_id);
	}
}


/*!
*	This function handles the indication comminf from VAPI. \n
*	It is registered to VAPI using the VAPI_RegisterEventCallback() API.\n
*	It handles DTMF tones detection, and some Comcerto device indications .\n
*	It sets event V21 detected to operates a T38 switch in case of V21 flag detection.\n
*	\param eEventCode VAPI event Id
*	\param *pvData
*	\retval None
*/
void comcerto_indication_handler(EEventCode eEventCode, void *pvData)
{
	SToneDetectEventParams *tone_detected;
	SRemoteDetectEventParams *remote_tone_detected;
	SSsrcChangeEventParams *ssrc_changed;
	SUndefinedEventParams *this_event;
	endpt_t *endpt;

	/* FIXME: The parser below need to be improved */
	switch(eEventCode)
	{
		case eVAPI_TONE_DETECT_EVENT:
			tone_detected = (SToneDetectEventParams *)pvData;

			endpt = &endpt_pool[tone_detected->ConId];

			PDEBUG(DBG_L3,"Tone %d detected on Connection ID %d",
					tone_detected->usDetectedTone, tone_detected->ConId);

			if(tone_detected->usDetectedTone != 0xff)
				PDEBUG(DBG_L3, "Tone %d detected", tone_detected->usDetectedTone);

			switch (tone_detected->usDetectedTone)
			{
			case 0x22:
				PDEBUG(DBG_L5, "======== FAX V.21 detected =========");
				endpt_raise_event(endpt, EVENT_V21FLAG_DETECTED);
				break;

			case 0:
			case 1:
			case 2:
			case 3:
			case 4:
			case 5:
			case 6:
			case 7:
			case 8:
			case 9:
			case 0x0A:
			case 0x0B:
				if (tone_detected->usDetectedTone == 0x0B) 
					PDEBUG(DBG_L3, "ENTERED #");

				endpt->dtmf = tone_detected->usDetectedTone;
				endpt_raise_event(endpt, EVENT_DTMF_DETECTED);
				break;

			case 0xFF: /* end of tone*/
/*				if (endpts[tone_detected->ConId].event == ENDPOINT_EVENT_WAIT_UNPRESS)
					set_endpt_event(ENDPOINT_EVENT_UNPRESSED, tone_detected->ConId);
*/				break;

			default:
				/* else send busy tone */
				endpt_raise_event(endpt, EVENT_BUSY);
                        	break;
			}

			break;

		case eVAPI_SSRC_CHANGE_EVENT:
			ssrc_changed = (SSsrcChangeEventParams *)pvData;
			PDEBUG(DBG_L3, "SSRC changed 0x%08x on Connection ID %d",
					ssrc_changed->uNewSSRC, ssrc_changed->ConId);

			break;

		case eVAPI_SPI_EVENT:
		case eVAPI_TONE_GEN_CMPLT_EVENT:
		case eVAPI_PT_CHANGE_EVENT:
		case eVAPI_SSRC_VIOLATION_EVENT:
		case eVAPI_NTE_TRANSMIT_COMPLETE_EVENT:
		case eVAPI_NTE_RECVD_EVENT:
		case eVAPI_CALLER_ID_CMPLT_EVENT:
		case eVAPI_G711_CONCURRENT_DECODER_EVENT:
		case eVAPI_PASSTHRU_CONCURRENT_DECODER_EVENT:
		case eVAPI_CALLER_ID_DETECTED_EVENT:
		case eVAPI_FAX_SWITCH_CMPLT_EVENT:
		case eVAPI_ALERT_IND:

			break;
			
		case eVAPI_REMOTE_DETECT_EVENT:
			remote_tone_detected = (SRemoteDetectEventParams *)pvData;

			PDEBUG(DBG_L3, "Remote tone %d detected on Connection ID %d", (remote_tone_detected->ucDetectedEvent - 1), remote_tone_detected->ConId);

			endpt = &endpt_pool[remote_tone_detected->ConId];

			switch (remote_tone_detected->ucDetectedEvent - 1)
			{			
			case 0x0A:
				/* future use start/stop recording*/
				break;

			case 0x0B:
				/* future use  start/stop playing*/
				break;				

			case 0xFF: /* end of tone no action*/
				break;

			default:
				/* else send busy tone */
				endpt_raise_event(endpt, EVENT_BUSY);
                        	break;
			}

			break;
			
		case eVAPI_UNDEFINED_EVENT:
			PDEBUG(DBG_L3, "event %d received",eEventCode);
			this_event = (SUndefinedEventParams *) pvData;

			if(this_event->ucCmdType == CMD_TYPE_INDICATION)
				PDEBUG(DBG_L3, "indication 0x%04x received",this_event->usFnCode);
		
			break;

		default:
			break;
	}

	return;
}


/*!
*	This function fills the global structure with device config parameters
*	\param device_id Comcerto device Id
*	\retval >0 on success
*	\retval <0 on failure
*	\see VAPI_AllocateMessage
*	\see VAPI_SetMessage
*	\see VAPI_SendDeviceMessage
*	\see VAPI_FreeMessage
*/
int get_mr(U32 device_id)
{
	/*if not in a DDK and VxWorks config lets return 100 to make sure we allow wide band (see check in demo_main.c)*/
	return 100;
}


/*!
*	This function fills the global structure with device config parameters
*	\param *device_configuration Pointer to the device configuration instance
*	\retval None
*/
void set_device_configuration(SCSMEUsrData *device_configuration)
{
	U8 *mac;

	memcpy(device_configuration, &default_device_configuration, sizeof(SCSMEUsrData));

	/* user may put here code to overwrite default config parameters, for example
	 * read them from config file */

	device_configuration->uiDevId = 0;
	device_configuration->pucEthDevName = HOST_CTRL_INTERFACE;

	/* this code is executed before VAPI_Init, so we have to use printf's */

	PDEBUG(DBG_L1, "CTRL_INTERFACE_TYPE  = %d", device_configuration->usControlInterface);
	PDEBUG(DBG_L1, "DEVICE_ID            = %d", device_configuration->uiDevId);
	PDEBUG(DBG_L1, "DEVICE_MODE          = %d", device_configuration->eDevMode);
	PDEBUG(DBG_L1, "DEVICE_TYPE          = %d", device_configuration->ucDevType);
	PDEBUG(DBG_L1, "USE_DEFAULT_MAX_CHAN = %d", device_configuration->bUseDefaultMaxChnls);
	PDEBUG(DBG_L1, "CUSTOM_MAX_CHANNEL   = %d", device_configuration->usMaxChannels);
	PDEBUG(DBG_L1, "ACK_REQUIRED         = %d", device_configuration->ucIsAckReqd);

	mac = device_configuration->aucDevMac;
	PDEBUG(DBG_L1, "DEVICE MAC           = %02x:%02x:%02x:%02x:%02x:%02x", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

	mac = device_configuration->aucHostMac;
	PDEBUG(DBG_L1, "HOST MAC             = %02x:%02x:%02x:%02x:%02x:%02x", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
	PDEBUG(DBG_L1, "HOST CTRL INTERFACE  = %s\n", device_configuration->pucEthDevName);
}


/*!
*	This function changes the default VAPI configuration.
*	\param *default_config
*	\retval 0 on success (no failure)
*/
int set_default_config(IN OUT SVAPIConfig *default_config)
{
	/* change the default timeout to 10S */
	default_config->uiMspRespTimeout = 10;	/* MSP response timeout value 10 seconds */

	return SUCCESS;
}

/*!
*	This function 
*	\param *s
*	\param *value
*	\retval 0 on success
*	\retval -1 on failure
*/
static int get_int(const char *s, int *value)
{
	int i;
	char *endptr = NULL;

	if (!s || !*s)
		goto err;

	i = strtol(s, &endptr, 0);
	if (!i && s == endptr)
		goto err;

	if (value)
		*value = i;

	return 0;

err:
	return -1;
}


/*!
*	This function is the command line option parser.
*	\param argc
*	\param *argv
*	\retval 0 on success
*	\retval -1 on failure
*/	
static int get_opt(int argc, char *argv[])
{
	int c;
	int rc = 0; /* return code */

	while ((c = getopt_long(argc, argv, "c:d:D:", long_options, NULL)) != - 1)
	{
		switch (c)
		{
		case 'c':
			strncpy(cfg_file, optarg, sizeof(cfg_file));
			break;

		case 'd':
			get_int(optarg, (int *) &vapi_debug_level);
			break;

		case 'D':
			get_int(optarg, (int *) &app_debug_level);
			break;

		case CHAR_MIN - 1:
			show_help();
			exit(0);
   			break;

		case CHAR_MIN - 2:
    			show_version();
			exit(0);
    			break;
			
		default:
			rc = -1;
		}
	}

	return rc;
}

/*!
*	This function initializes all endpoints
*	\retval None
*/
void endpt_init_all(void)
{
	unsigned int i;

	for (i = 0 ; i < MAX_ENDPT_ID; i++)
	{
		endpt_init_once(i, &endpt_pool[i]);
		endpt_reset(&endpt_pool[i]);
	}
}

/*!
*	This function displays the pool of endpoints
*	\retval None
*/
void endpt_pool_display(void)
{
	unsigned int i;
	endpt_t * endpt;
	unsigned int j = 0;

	printf("\n---------------------------------    ----    -----------    ---------------------------------\n");
	printf("endpt                                POTS    SIP            SIP Phone                         \n");
	printf("Id  IP address       RTP    RTCP     Tel.    User-Id  UA    User-Id  URI                      \n");
	printf("--  ---------------  -----  -----    ----    -------  --    -------  ------------------------ \n");

	for (i = 0; i < MAX_ENDPT_ID; i++)
	{
		endpt = &endpt_pool[i];

		printf("%-2d  %03d.%03d.%03d.%03d  %-5d  %-5d    %-4d    ",
				endpt->id, endpt->ip[0], endpt->ip[1], endpt->ip[2], endpt->ip[3], 
				endpt->rtp, endpt->rtcp, endpt->phone_id);

		if (endpt->sip_session != NULL)
		{
			printf("%-7ld  ", endpt->sip_session->user_id);

			if (endpt->sip_session->acc_cfg != NULL)
			{
				printf ("%-2d", j);
				j++;
			}
			else
				printf("  ");

			if (endpt->sip_session->sip_phone != NULL)
				printf("    %-7ld  %-s",
					endpt->sip_session->sip_phone->user_id, endpt->sip_session->sip_phone->uri);

		}

		printf("\n");
	}

	printf("---------------------------------    ----    -----------    ---------------------------------\n\n");
}


/*!
*	This function parses the command line options to set the VAPI and application debug levels
*	\param *filename Configuration filename (string)
*	\retval 0 on succes
*	\retval -1 on failure
*/
int check_cfg_file(char *filename)
{
	int rc = 0; /* return code */

	/* try to set default config file */
	strcpy(cfg_file, filename); /* start from current dir */
	rc = open(filename, O_RDONLY, 0);

	if (rc < 0) /* fail with previous filename, try a 2nd one */
	{
		sprintf(cfg_file, "/etc/%s", filename);
		rc = open(cfg_file, O_RDONLY, 0);
	}

	if (rc < 0) /* fail with previous filenames, try a 3rd one */
	{
		sprintf(cfg_file, "/usr/local/etc/%s", filename);
		rc = open(cfg_file, O_RDONLY, 0);
	}

	if (rc < 0)
		return -1;

	close(rc);
	return 0;
}



/*!
*	This is the main function of the application.
*	 - initialize an endpts array
*	 - then parse the command line arguments (VAPI debug level)
*	 - then it:
*		- Initialise the setup configuration using parameters from the configration file 
*			(Device type, Control interface, Slave/Master mode , etc)
*		- Print the VAPI verson (VAPI_GetVersion)
*		- Initialise VAPI (VAPI_Init)
*		- Open the Device (VAPI_OpenDevice)
*		- Initialise the TDM buses using default parameters (VAPI_SetTDMParams)
*		- Using VAPI_InitDevice set the SPU feature and enable the Multicommand mode
*		- Set the Network Parameters (VAPI_SetEthMac VAPI_SetDeviceIPAddr)
*		- Set the diagnostics if enabled.
*	 - then it starts the signaling and state machine threads.
*	\param argc
*	\param *argv
*	\retval 0 on success
*	\retval -1 on failure
*	\see VAPI_Init
*	\see VAPI_SetDebugLevel
*	\see VAPI_OpenDevice
*	\see VAPI_InitDevice
*	\see VAPI_CloseDevice
*	\see VAPI OpenDevice error
*	\see VAPI_Close
*/
int main(int argc, char *argv[])
{
	int rc; /* return code */

	channel_mode = eNarrowBand;
	vapi_debug_level = NO_DEBUG;
	app_debug_level = NO_DEBUG;

	rc = check_cfg_file(MTALK_CFG_FILE);
	exit_on_err(rc, err0, "check_cfg_file");

	/* returns value < 0 on error, == 0 if all's Ok */
	rc = get_opt(argc, argv);
	exit_on_err(rc, err0, "get_opt");

	rc = slic_init(&slic);
	exit_on_err(rc, err0, "slic_init");

	tdmif_cfg = get_tdmif_cfg(cfg_file);
	exit_if_null(tdmif_cfg, err0, "get_tdmif_cfg");
	display_tdmif_cfg(tdmif_cfg);

	netif_cfg = get_netif_cfg(cfg_file);
	exit_if_null(netif_cfg, err0, "get_netif_cfg");
	display_netif_cfg(netif_cfg);

	media_cfg = get_media_cfg(cfg_file);
	exit_if_null(media_cfg, err0, "get_media_cfg");
	display_media_cfg(media_cfg);

	sipua_cfg = get_sipua_cfg(cfg_file);
	exit_if_null(sipua_cfg, err0, "get_sipua_cfg");
	display_sipua_cfg(sipua_cfg);

	debug_cfg = get_debug_cfg(cfg_file);
	exit_if_null(debug_cfg, err0, "get_debug_cfg");
	display_debug_cfg(debug_cfg);

	/* The VAPI default parameters are not convenient for us
	 * we overwrite some of them. This must be done before VAPI_Init() */
	pfnUsrQueryVapiDefaultsEnumerate = set_default_config;

	/* Setting device parameters - this also must be done before VAPI_Init() */
	set_device_configuration(&gtl_device_configuration);

	rc = VAPI_Init(&gtl_device_configuration);
	exit_on_err(rc, err0, "VAPI_Init");

	/* if < to MR 22 we do not support wideband */
	mr_version = get_mr(0); /* get the MSP MR version */
	if ((mr_version >= 21) && (slic.is_wideband == True))
	{
		is_wideband_slic = True;
		PDEBUG(DBG_L1, "PASS .......... Wide-Band SLIC Init");
	}
	else
	{
		is_wideband_slic = False;
		PDEBUG(DBG_L1, "PASS .......... Narrow-Band SLIC Init");
	}

	rc = VAPI_SetDebugLevel(vapi_debug_level, app_debug_level);
	exit_on_err(rc, err1, "VAPI_SetDebugLevel");

	 rc = VAPI_OpenDevice(0, NULL);
	exit_on_err(rc, err1, "VAPI_OpenDevice");

	rc = VAPI_InitDevice(0, VAPI_DEV_OPMODE_DEFAULT, media_cfg->spu_ec_type, NULL, NULL);
	exit_on_err(rc, err2, "VAPI_InitDevice");

	rc = tdmif_init(tdmif_cfg); /* Setup TDM params */
	exit_on_err(rc, err2, "tdmif_init");

	rc = netif_init(netif_cfg);
	exit_on_err(rc, err2, "netif_init");

	rc = debug_init(debug_cfg);
	exit_on_err(rc, err2, "debug_init");

	/* init all endpoints */
	endpt_init_all();

	rc = sipua_init(&sipua);
	exit_on_err(rc, err2, "sipua_init");

	/* Now create a thread to handle the events comming from the Signaling side (SLIC) */
	tid_signalling = UT_ThreadCreate("tSignalling", pots_event_handler, NULL);
	exit_if_null(tid_signalling, err3, "Create tSignalling thread");

	/* Now create a thread to handle the states */
	tid_state_machine = UT_ThreadCreate("tStateMachine", endpt_state_machine, NULL);
	exit_if_null(tid_signalling, err4, "Create tStateMachine thread");
	
	rc = VAPI_RegisterEventCallback(0, EVENT_LEVEL_GENERIC, comcerto_indication_handler);
	exit_on_err(rc, err4, "VAPI_RegisterEventCallback()");

	/* display status for all endpoints in the pool */
	endpt_pool_display();


	UT_ThreadJoin(tid_state_machine);
	UT_ThreadJoin(tid_signalling);

	return 0;


err4:
	UT_ThreadCancel(tid_signalling);

err3:
	legerity_close(slic.dev_file);

err2: /* VAPI InitDevice and other init errors */
	VAPI_CloseDevice(0, ePURGE);

err1: /* VAPI OpenDevice error */
	VAPI_Close();

err0: /* VAPI Init error */

	display_tdmif_cfg(tdmif_cfg);
	display_netif_cfg(netif_cfg);
	display_media_cfg(media_cfg);
	display_debug_cfg(debug_cfg);
	display_sipua_cfg(sipua_cfg);

	return FAILURE;
}

/*!	@} */
