/*!
*	\file media.c
*
*	\defgroup media media (Media Module)
*
*	Media Module including all C-functions and declarations
*	(defines, types, variables)
*
*	\attention
*	Copyright © 2004-2010 Mindspeed Technologies, Inc.\n
*	Mindspeed Confidential.\n
*	All rights reserved.\n
*	This file is a component of the Mindspeed® VAPI software ("VAPI") and is
*	distributed under the Mindspeed Software License Agreement (the "Agreement").\n
*	Before using this file, you must agree to be bound by the the terms and 
*	conditions of the Agreement.
*
*	@{
*/

#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <string.h>

/* VAPI headers */
#include <msp.h>
#include <vapi.h>
#include <gtl.h>
#include <comcerto-api-defs.h>
#include <comcerto-ud-types.h>

/* Zarlink headers */
#include <legerity_lib.h>

/* mtalk header */
#include <mtalk.h>


/*!
*	This function allocates and initializes a media_cfg instance with hardcoded and/or user defined values.
*	\param *cfg_file Configuration filename (string).
*	\retval NULL on failure
*	\retval *media_cfg Pointer to the allocated and initialized media configuration structure
*/
media_cfg_t *get_media_cfg(const char *cfg_file)
{
	struct _CFG *cfg_info;
	media_cfg_t *media_cfg;

	media_cfg = (media_cfg_t *)malloc(sizeof(media_cfg_t));
	exit_if_null(media_cfg, err, "get_media_cfg() - mem allocation fail");

	cfg_info = cfg_read(cfg_file, 0);
	exit_if_null(cfg_info, err, "get_media_cfg() - mem allocation fail");

	READ_INT(cfg_info, "MEDIA", "SPU_EC_TYPE", media_cfg->spu_ec_type, VAPI_DEV_PROF_SPU_DFECAN_ENABLE);
	READ_INT(cfg_info, "MEDIA", "LOOPBACK", media_cfg->loopback_mode, 0);
	READ_INT(cfg_info, "MEDIA", "CODEC", media_cfg->codec, eG711_ULAW_ID);
	READ_INT(cfg_info, "MEDIA", "PACKET_SIZE", media_cfg->packet_size, 20);

	cfg_clean(cfg_info);

	return media_cfg;

err:
	return NULL;
}


/*!
*	This function displays the values of media user parameters of the media module.
*	\param *media_cfg Pointer to media module parameter structure.
*	\retval None
*/
void display_media_cfg(media_cfg_t *media_cfg)
{
	if (!media_cfg)
		return;

	PDEBUG(DBG_L2, "-------------------- media_cfg");
	PDEBUG(DBG_L2, "- spu_ec_type = %d", media_cfg->spu_ec_type);
	PDEBUG(DBG_L2, "- loopback_mode = %d", media_cfg->loopback_mode);
	PDEBUG(DBG_L2, "- codec = %d", media_cfg->codec);
	PDEBUG(DBG_L2, "- packet_size = %d", media_cfg->packet_size);
	PDEBUG(DBG_L2, "--------------------");
}


/*!
*	This function creates the MSP resources associated to a given endpoint.
*	The endpoint ID is also used as timeslot number.
*	This function also sets the packet size and codec.
*	\param *endpt Pointer to endpt.
*	\param vapi_mode Specifies whether VAPI sync or asyn mode is applied.
*	\retval 0 if success.
*	\retval <0 if failure
*       \see VAPI_AllocateConnection
*	\see VAPI_SetPayloadType
*	\see VAPI_SetCodecType
*	\see VAPI_SetPacketInterval
*	\see VAPI_SetConnectionState
*/
int endpt_create_voip(endpt_t *endpt, int vapi_mode)
{
	SRequest request;
	SRequest *this_request = NULL;
	int nb_params;
	int rc = 0;

	PDEBUG(DBG_L3, "endpt_create_voip (endpt %d, mode %d), type %d, timeslot %d", endpt->id, vapi_mode, eVOIP, endpt->timeslot[0]);

	if (vapi_mode != VAPI_SYNC_MODE)
	{
		request.uiReqId = vapi_mode;
		request.pfnIoCompCallback = &comcerto_response_handler;
		this_request = &request;
		PDEBUG(DBG_L3, "endpt_create_voip (request id 0x%04x)", request.uiReqId);
	}

	nb_params = 1; /* only 1 timeslot is usually specified */
	if ((endpt->call == CALL_POTS2POTS) && (channel_mode != eNarrowBand))
		nb_params = 4; /* except for POTS to POTS wideband where 4 timeslots are required */

	rc = VAPI_AllocateConnection(0, endpt->id, eVOIP, channel_mode, nb_params, endpt->timeslot, this_request, NULL);
	exit_on_err(rc, exit, "VAPI_AllocateConnection");

#if 0
/** 
 * Standard RTP static payload types, as defined by RFC 3551. 
 * The header file <pjmedia-codec/types.h> also declares dynamic payload
 * type numbers that are used by PJMEDIA when advertising the capability
 * for example in SDP message.
 */
enum pjmedia_rtp_pt
{
    PJMEDIA_RTP_PT_PCMU = 0,	    /**< audio PCMU			    */
    PJMEDIA_RTP_PT_G726_32 = 2,    /**< audio G726-32			    */
    PJMEDIA_RTP_PT_GSM  = 3,	    /**< audio GSM			    */
    PJMEDIA_RTP_PT_G723 = 4,	    /**< audio G723			    */
    PJMEDIA_RTP_PT_DVI4_8K = 5,	    /**< audio DVI4 8KHz		    */
    PJMEDIA_RTP_PT_DVI4_16K = 6,    /**< audio DVI4 16Khz		    */
    PJMEDIA_RTP_PT_LPC = 7,	    /**< audio LPC			    */
    PJMEDIA_RTP_PT_PCMA = 8,	    /**< audio PCMA			    */
    PJMEDIA_RTP_PT_G722 = 9,	    /**< audio G722			    */
    PJMEDIA_RTP_PT_L16_2 = 10,	    /**< audio 16bit linear 44.1KHz stereo  */
    PJMEDIA_RTP_PT_L16_1 = 11,	    /**< audio 16bit linear 44.1KHz mono    */
    PJMEDIA_RTP_PT_QCELP = 12,	    /**< audio QCELP			    */
    PJMEDIA_RTP_PT_CN = 13,	    /**< audio Comfort Noise		    */
    PJMEDIA_RTP_PT_MPA = 14,	    /**< audio MPEG1/MPEG2 elemetr. streams */
    PJMEDIA_RTP_PT_G728 = 15,	    /**< audio G728			    */
    PJMEDIA_RTP_PT_DVI4_11K = 16,   /**< audio DVI4 11.025KHz mono	    */
    PJMEDIA_RTP_PT_DVI4_22K = 17,   /**< audio DVI4 22.050KHz mono	    */
    PJMEDIA_RTP_PT_G729 = 18,	    /**< audio G729			    */

    PJMEDIA_RTP_PT_CELB = 25,	    /**< video/comb Cell-B by Sun (RFC2029) */
    PJMEDIA_RTP_PT_JPEG = 26,	    /**< video JPEG			    */
    PJMEDIA_RTP_PT_NV = 28,	    /**< video NV  by nv program by Xerox   */
    PJMEDIA_RTP_PT_H261 = 31,	    /**< video H261			    */
    PJMEDIA_RTP_PT_MPV = 32,	    /**< video MPEG1 or MPEG2 elementary    */
    PJMEDIA_RTP_PT_MP2T = 33,	    /**< video MPEG2 transport		    */
    PJMEDIA_RTP_PT_H263 = 34,	    /**< video H263			    */

    PJMEDIA_RTP_PT_DYNAMIC = 96     /**< start of dynamic RTP payload	    */
};

enum
{
    /* PJMEDIA_RTP_PT_TELEPHONE_EVENTS is declared in
     * <pjmedia/config.h>
     */
#if PJMEDIA_RTP_PT_TELEPHONE_EVENTS
    PJMEDIA_RTP_PT_START = PJMEDIA_RTP_PT_TELEPHONE_EVENTS,
#else
    PJMEDIA_RTP_PT_START = 102,
#endif

    PJMEDIA_RTP_PT_SPEEX_NB,			/**< Speex narrowband/8KHz  */
    PJMEDIA_RTP_PT_SPEEX_WB,			/**< Speex wideband/16KHz   */
    PJMEDIA_RTP_PT_SPEEX_UWB,			/**< Speex 32KHz	    */
    PJMEDIA_RTP_PT_L16_8KHZ_MONO,		/**< L16 @ 8KHz, mono	    */
    PJMEDIA_RTP_PT_L16_8KHZ_STEREO,		/**< L16 @ 8KHz, stereo     */
    //PJMEDIA_RTP_PT_L16_11KHZ_MONO,		/**< L16 @ 11KHz, mono	    */
    //PJMEDIA_RTP_PT_L16_11KHZ_STEREO,		/**< L16 @ 11KHz, stereo    */
    PJMEDIA_RTP_PT_L16_16KHZ_MONO,		/**< L16 @ 16KHz, mono	    */
    PJMEDIA_RTP_PT_L16_16KHZ_STEREO,		/**< L16 @ 16KHz, stereo    */
    //PJMEDIA_RTP_PT_L16_22KHZ_MONO,		/**< L16 @ 22KHz, mono	    */
    //PJMEDIA_RTP_PT_L16_22KHZ_STEREO,		/**< L16 @ 22KHz, stereo    */
    PJMEDIA_RTP_PT_L16_32KHZ_MONO,		/**< L16 @ 32KHz, mono	    */
    PJMEDIA_RTP_PT_L16_32KHZ_STEREO,		/**< L16 @ 32KHz, stereo    */
    PJMEDIA_RTP_PT_L16_48KHZ_MONO,		/**< L16 @ 48KHz, mono	    */
    PJMEDIA_RTP_PT_L16_48KHZ_STEREO,		/**< L16 @ 48KHz, stereo    */
    PJMEDIA_RTP_PT_ILBC,			/**< iLBC (13.3/15.2Kbps)   */
    PJMEDIA_RTP_PT_AMR,				/**< AMR (4.75 - 12.2Kbps)  */
    PJMEDIA_RTP_PT_AMRWB,			/**< AMRWB (6.6 - 23.85Kbps)*/
    PJMEDIA_RTP_PT_AMRWBE,			/**< AMRWBE		    */
    PJMEDIA_RTP_PT_G726_16,			/**< G726 @ 16Kbps	    */
    PJMEDIA_RTP_PT_G726_24,			/**< G726 @ 24Kbps	    */
    /* PJMEDIA_RTP_PT_G726_32,*/		/**< G726 @ 32Kbps, static? */
    PJMEDIA_RTP_PT_G726_40,			/**< G726 @ 40Kbps	    */
    PJMEDIA_RTP_PT_G722_1_16,			/**< G722.1 (16Kbps)	    */
    PJMEDIA_RTP_PT_G722_1_24,			/**< G722.1 (24Kbps)	    */
    PJMEDIA_RTP_PT_G722_1_32,			/**< G722.1 (32Kbps)	    */
    PJMEDIA_RTP_PT_G7221C_24,			/**< G722.1 Annex C (24Kbps)*/
    PJMEDIA_RTP_PT_G7221C_32,			/**< G722.1 Annex C (32Kbps)*/
    PJMEDIA_RTP_PT_G7221C_48,			/**< G722.1 Annex C (48Kbps)*/
    PJMEDIA_RTP_PT_G7221_RSV1,			/**< G722.1 reserve	    */
    PJMEDIA_RTP_PT_G7221_RSV2,			/**< G722.1 reserve	    */
};

	/* Configure a payload type for G722 */
	rc = VAPI_SetPayloadType(endpt->id, eG722, PJMEDIA_RTP_PT_G722, eBoth, NULL);

	exit_on_err(rc, exit, "VAPI_SetPayloadType");

	switch (endpt->call)
	{
	case CALL_SIP2VOIP:
		codec = media_cfg->sip2voip_codec;
		break;

	case CALL_VOIP2SIP:
		codec = media_cfg->voip2sip_codec;
		break;

	default:
		codec = media_cfg->codec;
		break;
	}

	rc = VAPI_SetCodecType(endpt->id, codec, NULL);
	exit_on_err(rc, exit, "VAPI_SetCodecType");
#endif

	rc = VAPI_SetCodecType(endpt->id, media_cfg->codec, NULL);
	exit_on_err(rc, exit, "VAPI_SetCodecType");

	rc = VAPI_SetPacketInterval(endpt->id, media_cfg->packet_size, NULL);
	exit_on_err(rc, exit, "VAPI_SetPacketInterval");

	if ((endpt->call == CALL_POTS2POTS) || (endpt->call == CALL_SIP2POTS) || (endpt->call == CALL_POTS2SIP))
	{
		rc = VAPI_SetConnectionState(endpt->id, eTdmActive, NULL);
		exit_on_err(rc, exit, "VAPI_SetConnectionState");
	}

	PDEBUG(DBG_L3, "endpt %d running codec %d\n", endpt->id, media_cfg->codec);

exit:
	return rc;
}


/*!
*	This function deletes the MSP resources associated to a given endpoint.
*	\param *endpt Pointer to endpt.
*	\param vapi_mode Specifies whether VAPI sync or asyn mode is applied.
*	\retval 0 if success.
*	\retval <0 if failure
*       \see VAPI_DestroyConnection
*/
int endpt_delete_voip(endpt_t *endpt, int vapi_mode)
{
	SRequest request;
	SRequest *this_request = NULL;
	int rc = 0;

	PDEBUG(DBG_L3, "endpt_delete_voip (endpt %d)", endpt->id);

	if (vapi_mode != VAPI_SYNC_MODE)
	{
		request.uiReqId = vapi_mode;
		request.pfnIoCompCallback = &comcerto_response_handler;
		this_request = &request;
		PDEBUG(DBG_L3, "endpt_delete_voip (request id 0x%04x)", request.uiReqId);
	}
	
	rc = VAPI_DestroyConnection(endpt->id, this_request);
	exit_on_err(rc, exit, "VAPI_DestroyConnection");
exit:
	return rc;
}


/*!
*	This function instructs a given endpoint to change the codec and packetization interval.
*	\param *endpt Pointer to endpt.
*	\retval 0 if success.
*	\retval <0 if failure
*       \see VAPI_SetPayloadType
*	\see VAPI_SetPacketInterval
*	\see VAPI_SetCodecType
*/
int endpt_modify_voip(endpt_t *endpt)
{
	int rc = 0;

	PDEBUG(DBG_L1, "\nendpt_modify_voip() [ codec %d, PT %d, ptime %d ]\n",
		endpt->voip_profile.codec, endpt->voip_profile.payload_type, endpt->voip_profile.ptime);

	if ((endpt->voip_profile.payload_type < 0) && (endpt->voip_profile.codec < 0))
		exit_on_err(rc, exit, "endpt_modify_voip");

	rc = VAPI_SetPayloadType(endpt->id, endpt->voip_profile.codec, endpt->voip_profile.payload_type, eBoth, NULL);
	exit_on_err(rc, exit, "VAPI_SetPayloadType");
	
	if (endpt->voip_profile.ptime > 0)
	{
		rc = VAPI_SetPacketInterval(0, endpt->voip_profile.ptime, NULL);
		exit_on_err(rc, exit, "VAPI_SetPacketInterval");
	}

	rc = VAPI_SetCodecType(endpt->id, endpt->voip_profile.codec, NULL);
	exit_on_err(rc, exit, "VAPI_SetCodecType");

exit:
	return rc;
}


/*!
*	This function activate instructs a given endpoint to start VoIP traffic.
*	\param *endpt Pointer to endpt.
*	\param vapi_mode Specifies whether VAPI sync or asyn mode is applied.
*	\retval 0 if success.
*	\retval <0 if failure
*       \see VAPI_SetConnectionState
*/
int endpt_start_voip(endpt_t *endpt, int vapi_mode)
{
	SRequest request;
	SRequest *this_request = NULL;
	int rc = 0; /* return code */

	PDEBUG(DBG_L3, "endpt_start_voip (endpt %d)", endpt->id);

	if (vapi_mode != VAPI_SYNC_MODE)
	{
		request.uiReqId = vapi_mode;
		request.pfnIoCompCallback = &comcerto_response_handler;
		this_request = &request;
		PDEBUG(DBG_L3, "endpt_start_voip (request id 0x%04x)", request.uiReqId);
	}

	rc =  VAPI_SetConnectionState(endpt->id, eActive, this_request);
	exit_on_err(rc, exit, "VAPI_SetConnectionState");

exit:
	return rc;
}


/*!
*	This function instructs a given endpoint to start transcoding with its peer endpoint.
*	\param *endpt Pointer to endpt.
*	\retval 0 if success.
*	\retval <0 if failure
*       \see VAPI_TranscodingSession
*/
int endpt_start_transcoding(endpt_t *endpt)
{
	int rc = 0; /* return code */

	if (endpt->call == CALL_VOIP2SIP) /* POTS to SIP or SIP to POTS call */
	{
		/* Check if peer SIP is already connected, if so, start transcoding session */
		if (endpt->peer_endpt->state == STATE_READY)
		{
			STranscodingOption transcoding_option;

    			/* Enable Virtual transcoding session between connection 0 & 1 */ 
			transcoding_option.bAction = 1 ; /* Enable transcoding*/
			transcoding_option.ucDSPMode = 0 ; /* Enable DSP bypass mode*/
			transcoding_option.ucVirtualTranscoding = 0 ; /* Enable virtual transcoding*/

			PDEBUG(DBG_L1, "<<<<<<<<<<<<<<< Start Transcoding endpt %d <-> endpt %d >>>>>>>>>>>>>>>", endpt->id, endpt->peer_endpt->id);
			rc = VAPI_TranscodingSession(endpt->id, endpt->peer_endpt->id, &transcoding_option, NULL);
			exit_on_err(rc, exit, "VAPI_TranscodingSession");
		}
	}

exit:
	return rc;
}


/*!
*	This function instructs a given endpoint to switch to T.38.
*	\param *endpt Pointer to endpt.
*	\param vapi_mode Specifies whether VAPI sync or asyn mode is applied.
*	\retval 0 if success.
*	\retval <0 if failure
*       \see VAPI_SwitchToT38
*	\see VAPI_SetConnectionState
*/
int endpt_switch_to_t38(endpt_t *endpt, int vapi_mode)
{
	SRequest request;
	SRequest *this_request = NULL;
	int rc = 0; /* return code */

	PDEBUG(DBG_L3, "endpt_switch_to_t38 (endp %d)", endpt->id);

	if (vapi_mode != VAPI_SYNC_MODE)
	{
		request.uiReqId = vapi_mode;
		request.pfnIoCompCallback = &comcerto_response_handler;
		this_request = &request;
		PDEBUG(DBG_L3, "endpt_switch_to_t38 (request id 0x%04x)", request.uiReqId);
	}

	rc = VAPI_SwitchToT38(endpt->id, NULL, this_request);
	exit_on_err(rc, exit, "VAPI_SwitchToT38");

	rc = VAPI_SetConnectionState(endpt->id, eActive, this_request);
	exit_on_err(rc, exit, "VAPI_SetConnectionState");

exit:
	return rc;
}


/*!
*	This function instructs a given endpoint to hangup.
*	\param *endpt Pointer to endpt.
*	\param vapi_mode Specifies whether VAPI sync or asyn mode is applied.
*	\retval 0 if success.
*	\retval <0 if failure
*       \see VAPI_TranscodingSession
*/
int endpt_hangup(endpt_t *endpt, int vapi_mode)
{
	int rc = 0; /* return code */

	switch (endpt->call)
	{
	case CALL_SIP2VOIP:
	case CALL_VOIP2SIP:

		/* stop transcoding once! hence check it's not already done with the peer endpt */
		if (endpt->peer_endpt->peer_endpt == endpt)
		{
			STranscodingOption transcoding_option;

    			/* Enable Virtual transcoding session between connection 0 & 1 */ 
			transcoding_option.bAction = 0 ; /* Disable transcoding*/
			transcoding_option.ucDSPMode = 0 ; /* Enable DSP bypass mode*/
			transcoding_option.ucVirtualTranscoding = 0 ; /* Enable virtual transcoding*/

			PDEBUG(DBG_L4, "--> Stop Transcoding\n");
			rc = VAPI_TranscodingSession(endpt->id, endpt->peer_endpt->id, &transcoding_option, NULL);
			exit_on_err(rc, exit, "VAPI_TranscodingSession");
		}
		break;

	case CALL_POTS2SIP:
	case CALL_SIP2POTS:

		sip_session_hangup(endpt->sip_session);
		break;
	}


exit:
	return rc;
}


/*!
*	This function instructs a given endpoint to play the specified tone.
*	\param *endpt Pointer to endpt.
*	\param tone_id Identifier of the tone to be generated.
*	\param vapi_mode Specifies whether VAPI sync or asyn mode is applied.
*	\retval 0 if success.
*	\retval <0 if failure
*       \see VAPI_PlayTone
*/
int endpt_play_tone(endpt_t *endpt, EToneId tone_id, int vapi_mode)
{
	SRequest request;
	SRequest *this_request = NULL;
	int rc = 0;

	PDEBUG(DBG_L3, "endpt_play_tone (%d, toneI_id %d, mode %d)", endpt->id, tone_id, vapi_mode);

	switch (tone_id)
	{
	case eDIALTONE:
	case eBUSYTONE:
	case eRINGBACKTONE:
	case eWAITINGTONE:

		if (vapi_mode != VAPI_SYNC_MODE)
		{
			request.uiReqId = vapi_mode;
			request.pfnIoCompCallback = &comcerto_response_handler;
			this_request = &request;
			PDEBUG(DBG_L3, "endpt_play_tone (request id 0x%04x)", request.uiReqId);
		}

		rc = VAPI_PlayTone(endpt->id, tone_id, eDirToTDM, NULL, 0, this_request);
		exit_on_err(rc, exit, "endpt_play_tone > VAPI_PlayTone");
		break;

	default:
		break;
	}

exit:
	return rc;
}


/*!
*	This function stops tone generation for a given endpoint.
*	\param *endpt Pointer to endpt.
*	\param vapi_mode Specifies whether VAPI sync or asyn mode is applied.
*	\retval 0 if success.
*	\retval <0 if fail
*	\see VAPI_StopTone
*/
int endpt_stop_tone(endpt_t *endpt, int vapi_mode)
{
	SRequest request;
	SRequest *this_request = NULL;
	int rc = 0; /* return code */

	PDEBUG(DBG_L3, "endpt_stop_tone (endpt %d)", endpt->id);

	if (vapi_mode != VAPI_SYNC_MODE)
	{
		request.uiReqId = vapi_mode;
		request.pfnIoCompCallback = &comcerto_response_handler;
		this_request = &request;
		PDEBUG(DBG_L3, "endpt_stop_tone (request id 0x%04x)", request.uiReqId);
	}

	rc = VAPI_StopTone(endpt->id, 0, 0, this_request);
	exit_on_err(rc, exit, "VAPI_StopTone");

exit:
	return rc;
}


/*!
*	This function starts VoIP loopback for a given endpoint.
*	\param *endpt Pointer to endpt.
*	\retval 0 if success.
*	\retval <0 if fail
*	\see VAPI_Loopback
*/
int endpt_start_loopback(endpt_t *endpt)
{
	int rc = 0; /* return code */

	if (media_cfg->loopback_mode == 0)
		goto exit;

	/* make sure we have 2 endpt */
	exit_if_null(endpt->peer_endpt, err, "endpt_start_loopback > missing peer endpt");

	rc = VAPI_Loopback(endpt->id, endpt->peer_endpt->id, media_cfg->loopback_mode, NULL);
	exit_on_err(rc, exit, "VAPI_Loopback");

	if (media_cfg->loopback_mode == eINTER_CHNL_POST_ENCAPS)
		PDEBUG(DBG_L3, "Endpoint %d & %d in inter channel loopback mode", endpt->id, endpt->peer_endpt->id);

	if (media_cfg->loopback_mode == eTHC)
		PDEBUG(DBG_L3, "Endpoint %d & %d in TDM Hairpin loopback mode", endpt->id, endpt->peer_endpt->id);

	if (is_wideband_slic == True)
		PDEBUG(DBG_L3, "To choose SLIC mode press button:\n8 - NB mode\n9 - WB mode");

exit:
	return rc;

err:
	return -1;
}



/*!
*	This function stops VoIP loopback for a given endpoint.
*	\param *endpt Pointer to endpt for which loopback is stopped.
*	\retval 0 if success.
*	\retval <0 if fail
*	\see VAPI_Loopback
*/
int endpt_stop_loopback(endpt_t *endpt)
{
	int rc = 0; /* return code */

	/* make sure the option from the config file is OK*/
	if (media_cfg->loopback_mode == eINTER_CHNL_POST_ENCAPS)
		rc = VAPI_Loopback(endpt->id, endpt->peer_endpt->id, eREMOVE_INTER_CHNL_POSTENCAPS_LOOPBACK , NULL);
	else if (media_cfg->loopback_mode == eTHC)
		rc = VAPI_Loopback(endpt->id, endpt->peer_endpt->id, eREMOVE_THC_LOOPBACK, NULL);
	/*else do nothing*/

	exit_on_err(rc, exit, "VAPI_Loopback");
exit:
	return rc;

}


/*!
*	This function "restarts"  a given endpt <(meaning delete it, re-create it, 
*	setup VoIP and activate it.
*	\param *endpt Pointer to endpt for which loopback is stopped.
*	\retval 0 if success.
*	\retval <0 if fail
*/
int endpt_restart(endpt_t *endpt)
{
	int rc = 0; /* return code */
		
	PDEBUG(DBG_L3, "Endpoint %d DTMF service %d", endpt->id, endpt->dtmf);

	rc = endpt_delete_voip(endpt, VAPI_SYNC_MODE);
	exit_on_err(rc, exit, "endpt_delete_voip");
	rc = endpt_create_voip(endpt, VAPI_SYNC_MODE); /* allocate MSP resources */
	exit_on_err(rc, exit, "endpt_create_voip");
	rc = netif_modify(endpt->id, &endpt->ip[0], &endpt->peer_endpt->ip[0], endpt->rtp, endpt->peer_endpt->rtp);
	exit_on_err(rc, exit, "netif_modify");
	rc = endpt_start_voip(endpt, VAPI_SYNC_MODE);
	exit_on_err(rc, exit, "endpt_start_voip");

exit:
	return rc;
}


/*!
*	
*	This function switches a given endpt from wideband to narrowband or vice-versas
*	\param *endpt Pointer to endpt for which loopback is stopped.
*	\param mode narrowband or wideband
*	\retval 0 if success.
*	\retval <0 if fail
*/
int endpt_band_switch(endpt_t *endpt, int mode)
{
	int rc = 0; /* return code */
		
	if (channel_mode == mode)
		goto exit;

	channel_mode = mode;

	if (channel_mode == eNarrowBand)
	{
		rc = slic_set_narrow_band(slic.dev_file, endpt->id+1, endpt->peer_endpt->id+1);
		exit_on_err(rc, exit, "slic_set_narrow_band");
	}
	else if (is_wideband_slic == True)
	{
		rc = slic_set_wide_band(slic.dev_file, endpt->id+1, endpt->peer_endpt->id+1);
		exit_on_err(rc, exit, "slic_set_wide_band");
	}

	rc = endpt_restart(endpt);
	exit_on_err(rc, exit, "endpt_reinit");
	rc = endpt_restart(endpt->peer_endpt);	
	exit_on_err(rc, exit, "endpt_reinit");
	rc = endpt_start_loopback(endpt);	
	exit_on_err(rc, exit, "endpt_start_loopback");

exit:
	return rc;
}



#if 0
	switch (endpt->dtmf_service)
	{
	/*CALL WAITING SCENARIO*/
	case 1:
	/*Switch back to the original endpt when DTMF 1*/
	case 2:
	/*Switch to the new endpt when DTMF 2*/

		/*Do not perform any action if there is no endpt in call waiting*/
		if (endpt->call_waiting_index != -1)
		{
			PDEBUG(DEBUG_FUNCTION, "Remap to original caller index %d\n", endpt->call_waiting_index);
			PDEBUG(DEBUG_FUNCTION, "Stop voice on endpt %d \n", peer_endpt->id);
			peer_endpt->vstatus = VAPI_SetConnectionState(peer_endpt->id, eTdmActive, NULL);
	
			/*stop loopback (if any)*/
			status = stop_loopback(peer_endpt->id);
	
			/*place the current peer in call waiting*/
			set_endpt_state(ENDPOINT_STATE_CALL_WAITING, peer_endpt->id);
			PDEBUG(DEBUG_FUNCTION, "Endpoint %d in ENDPOINT_STATE_CALL_WAITING\n",peer_endpt->id); 

			/*Save the current peer endpt index*/
			temp_index = endpts[i].peer_index;
			/*The new peer endpt is the one which was in call waiting */
			endpts[i].peer_index = endpts[i].call_waiting_index;
			/*The alod peer endpt is now the call waiting endpt*/
			endpts[i].call_waiting_index = temp_index;
		
			peer_endpt = endpts[i].peer_index;
			/*get the new peer out of call waiting state*/
			set_endpt_event(ENDPOINT_EVENT_CALL_ACCEPTED, endpts[i].peer_index);
			PDEBUG(DEBUG_FUNCTION, "Endpoint %d set out of ENDPOINT_STATE_CALL_WAITING\n",endpts[i].peer_index); 

			/*Change the IP header with the new peer endpt UDP port*/
			status = set_ip_udp_parameters_sync(endpts[i].index,
					endpts[i].ip_src_addr,
					endpts[peer_endpt].ip_src_addr,
					endpts[i].rtp_port,
					endpts[peer_endpt].rtp_port);

			/*Start the RTP traffic*/
			status = start_voip_connection(endpts[i].index, ENDPOINT_STATE_SYNC);
		}

		set_endpt_event(ENDPOINT_EVENT_NONE, endpts[i].index);

		break;

	/*Set endpts in narrowband mode when DTMF 8*/
	case 8:
		if (is_wb_slic == True)
		{
			/* narrow band mode*/
			channel_mode = eNarrowBand;
			set_endpt_event(ENDPOINT_EVENT_SWITCH_TO_NB, endpts[i].index);
			set_endpt_state(ENDPOINT_STATE_SWITCH_MODE, endpts[i].index);
		}
		else
		{
			set_endpt_event(ENDPOINT_EVENT_NONE, endpts[i].index);
		}
		
		break;

	/*Set endpts in Wideband mode when DTMF 9*/
	case 9:
		if (is_wb_slic == True)
		{
			/* wide band mode*/
			channel_mode = eWideBand;
			set_endpt_event(ENDPOINT_EVENT_SWITCH_TO_WB, endpts[i].index);
			set_endpt_state(ENDPOINT_STATE_SWITCH_MODE, endpts[i].index);
		}
		else
		{
			set_endpt_event(ENDPOINT_EVENT_NONE, endpts[i].index);
		}
		
		break;

	default:
		set_endpt_event(ENDPOINT_EVENT_NONE, endpts[i].index);
		break;
	}
#endif


/*!	@} */
