/*!
*	\file endpt.h
*
*	\attention
*	Copyright © 2004-2010 Mindspeed Technologies, Inc.\n
*	Mindspeed Confidential.\n
*	All rights reserved.\n
*	This file is a component of the Mindspeed® VAPI software ("VAPI") and is
*	distributed under the Mindspeed Software License Agreement (the "Agreement").\n
*	Before using this file, you must agree to be bound by the the terms and 
*	conditions of the Agreement.
*
*	@{
*/

#ifndef __ENDPT_H
#define __ENDPT_H


/* ==============================
	Constants
   ============================== */


/*! endpt dialstring length max */
#define MAX_DIALSTRING_SIZE   20


/*! endpt states */
typedef enum
{
	STATE_IDLE = 0,
		/*! idle state, endpt is not connected and not associated to any MSP resource */
	STATE_READY,
		/*! ready state, endpt is online or connected, associated to a MSP resource generating VoIP packets */
	STATE_RINGING,
		/*! ringing state, the endpt is asssociated to a SLIC POTS ringing */
	STATE_DIALING,
		/*! dialing state, endpt is associated to a MSP resource detecting DTMFs */
	STATE_CALLING,
		/*! calling state, endpt is associated to a MSP resource and waiting for the remote end to go offhook */
	STATE_CALL_WAITING
		/*! call waiting state */
} endpt_state_t;


/*! endpt events */
typedef enum
{
	EVENT_ON_HOOK = 0,
		/*! on hook event (shared slic and endpt event) */
	EVENT_OFF_HOOK,
		/*! off hook event (shared slic and endpt event) */
	EVENT_DIALING,
		/*! dialing event, endpt starts playing dial tone and detecting DTMF  */
	EVENT_RINGING,
		/*! ringing event, the pots mapped to the endpt rings */
	EVENT_CALLED_POTS_READY,
		/*! called pots ready event */
	EVENT_V21FLAG_DETECTED,
		/*! v21 flag detected event */
	EVENT_BUSY,
		/*! busy event, the endpt is not ready to take a call */
	EVENT_NB_WB_SWITCH,
		/*! narrowband/wideband switch event */
	EVENT_SWITCH_TO_NB,
		/*! narrowband switch event */
	EVENT_SWITCH_TO_WB,
		/*! wideband switch event */
	EVENT_READY,
		/*! ready event */
	EVENT_REINITIALIZE_ENDPS,
		/*! reinitialize endpt event */
	EVENT_DTMF_DETECTED,
		/*! dtmf detected event */
	EVENT_POTS2POTS_CALL,
		/*! POTS-to-POTS call event */
	EVENT_POTS2SIP_CALL,
		/*! POTS-to-SIP call event */
	EVENT_SIP2POTS_CALL,
		/*! SIP-to-POTS call event */
	EVENT_SIP2VOIP_CALL,
		/*! SIP-to-VoIP call event, call initiated by an IP Phone to another IP Phone through 2 VoIP channels running transcoding */
	EVENT_VOIP2SIP_CALL,
		/*! SIP-to-VoIP call event, call initiated by an IP Phone to another IP Phone through 2 VoIP channels running transcoding */
	EVENT_SIP_CONFIRMED,
		/*! SIP confirmed event, meaning that the remote IP Phone is ready to take the call */
	EVENT_SIP_DISCONNECTED
		/*! SIP disconnected event, meaning that the remote IP Phone hanged up the call */
} endpt_event_t;


/*! endpt call types */
typedef enum
{
		/*! POTS-to-POTS call, i.e. call between 2 phones connected to the onboard SLIC */
	CALL_POTS2POTS = 0,
		/*! POTS-to-SIP call, i.e. call initiated by a phone connected to an onboard POTS calling a SIP phone */
	CALL_POTS2SIP,
		/*! SIP-to-POTS call, i.e. call initiated by a SIP phone calling a phone connected to an onboard POTS */
	CALL_SIP2POTS,
		/*! SIP-to-POTS call, i.e. call initiated by a SIP phone calling another phone through 2 Comcerto VoIP channels in transcoding */
	CALL_SIP2VOIP,
		/*! SIP-to-VoIP call, i.e. call initiated by a Comcerto VoIP channel running transcoding calling a SIP phone */
	CALL_VOIP2SIP
} endpt_calltype_t;


/*! DTMF reserved to switch to the SIP dial plan */
typedef enum
{
	DTMF_SWITCH_DIAL_PLAN_SIP = 0,
	DTMF_SWITCH_DIAL_PLAN_POTS = 9
} dtmf_switch_dial_plan_t;


/*! dial plan */
typedef enum
{
	DIAL_PLAN_POTS = 1,
	DIAL_PLAN_SIP
} dial_plan_t;

#if 0

#define STATE_NONE   0
#define STATE_IDLE   1
#define STATE_READY   2
#define STATE_RINGING   3
#define STATE_DIALING   4
#define STATE_CALLING   5
#define STATE_CALL_WAITING   6
#define MAX_STATES STATE_CALL_WAITING


/*! shared endpt & slic events */
#define EVENT_ON_HOOK   0
#define EVENT_OFF_HOOK   1

/*! endpt events */
#define EVENT_DIALING   2
#define EVENT_RINGING   3
#define EVENT_CALLED_POTS_READY   4
#define EVENT_V21FLAG_DETECTED   5
#define EVENT_BUSY   6
#define EVENT_NB_WB_SWITCH   7
#define EVENT_SWITCH_TO_NB   8
#define EVENT_SWITCH_TO_WB   0
#define EVENT_READY   10
#define EVENT_REINITIALIZE_ENDPS   11
#define EVENT_DTMF_DETECTED   12
#define	EVENT_POTS2POTS_CALL   13
#define	EVENT_POTS2SIP_CALL   14
#define	EVENT_SIP2POTS_CALL   15
#define	EVENT_SIP2VOIP_CALL   16
#define	EVENT_VOIP2SIP_CALL   17
#define EVENT_SIP_CONFIRMED   18
#define	EVENT_SIP_DISCONNECTED   19


/*! endpt call types */
#define CALL_POTS2POTS   0
#define CALL_POTS2SIP   1
#define	CALL_SIP2POTS   2	
#define CALL_SIP2VOIP   3
#define	CALL_VOIP2SIP   4

/*! DTMF reserved to switch to the SIP dial plan */
#define DTMF_SWITCH_DIAL_PLAN_SIP   0
#define DTMF_SWITCH_DIAL_PLAN_POTS   9

/*! dial plan */
#define DIAL_PLAN_POTS   1
#define DIAL_PLAN_SIP   2

#endif
/* ==============================
	Types
   ============================== */

/*! state descriptor */
typedef struct state
{
	int id;
		/*! state identifier */
	char str[40];
		/*! state description */
} state_t;


/*! event descriptor */
typedef struct event
{
	int id;
		/*! endpt event identifier */
	char str[40];
		/*! endpt event description */
} event_t;


/*! endpt call descriptor */
typedef struct call
{
	int id;
		/*! endpt call unique identifier */
	char str[40];
		/*! endpt call description */
} call_t;


/*! endpt voip_profile negotiated for a given call */
typedef struct endpt_voip_profile {
	int codec;
		/*! codec value */
	int payload_type;
		/*! payload type value */
	int ptime;
		/*! packetization time in ms */
} endpt_voip_profile_t;


/*! endpt descriptor */
typedef struct endpt
{
	unsigned int id;
		/*! endpt unique identifier */

	struct endpt *peer_endpt;
		/*! Pointer to the peer endpt */

	struct endpt *call_waiting_endpt;
		/*! Pointer to the call waiting endpt */

	int state;
		/*! current endpt state */
	int event;
		/*! current event (if any) raised for endpt */
	int call;
		/*! call type SIP->POTS, SIP->VoiP, POTS->POTS, etc */

	int dial_plan;
		/*! 0:SIP dialing scheme, -1:POTS dialing scheme */
	char dialstring[MAX_DIALSTRING_SIZE];
		/*! DTMF dialstring */
	U8 dialstring_len;
		/*! number of DTMF digits in the dialstring */
	U16 dtmf;
		/*! DTMF detected and reported by the MSP */

	U32 phone_id;
		/*! POTS phone id used as POTS phone number */
	U16 timeslot[4];
		/*! timeslot number */

	U8 ip[4];
		/*! endpt source IP address */
	U16 rtp;
		/*! endpt source RTP port */
	U16 rtcp;
		/*! endpt source RTCP port */

	endpt_voip_profile_t voip_profile;
		/*! media profile to be applied to the endpt */

	struct sip_session *sip_session;
		/*! pointer to the SIP session assciated to the endpt */
} endpt_t;


/* ==============================
	Function Prototypes
   ============================== */

void *get_called_party_from_dialstring(int *calltype, char *dialstring, int dialstring_len, int dial_plan);
void endpt_reset(endpt_t *endpt);
void endpt_init_once(int endpt_id, endpt_t *endpt);
void endpt_raise_event(endpt_t *endpt, int event);
void endpt_switch_state(endpt_t *endpt, int state);
void *endpt_state_machine(void *none);


/* ==============================
	Variables
   ============================== */


endpt_t endpt_pool[MAX_ENDPT_ID];


#endif /* __ENDPT_H */

/*! @} */