/*!
*	\file endpt.c
*
*	\defgroup endpt endpt Module
*
*	Endpt Module including all C-functions and declarations
*	(defines, types, variables)
*
*	\attention
*	Copyright © 2004-2010 Mindspeed Technologies, Inc.\n
*	Mindspeed Confidential.\n
*	All rights reserved.\n
*	This file is a component of the Mindspeed® VAPI software ("VAPI") and is
*	distributed under the Mindspeed Software License Agreement (the "Agreement").\n
*	Before using this file, you must agree to be bound by the the terms and 
*	conditions of the Agreement.
*
*	@{
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/* call control interface */
#include <legerity_lib.h>

#include <sys/time.h>
#include <getopt.h>

/* VAPI headers */
#include <msp.h>
#include <vapi.h>
#include <gtl.h>

/* mtalk header */
#include <mtalk.h>



#define check_status(title, error_label)					\
	PDEBUG(DEBUG_ERROR, title ": %s", result == SUCCESS?"ok":"failed");	\
	if (result != SUCCESS)							\
		goto error_label

state_t pool_state[] =
{
	{ STATE_IDLE, "IDLE" },
	{ STATE_READY, "READY" },
	{ STATE_RINGING, "RINGING" },
	{ STATE_DIALING, "DIALING" },
	{ STATE_CALLING, "CALLING" },
	{ STATE_CALL_WAITING, "CALL WAITING" }
};

event_t pool_event[] = 
{
	{ EVENT_ON_HOOK, "ON_HOOK" },
	{ EVENT_OFF_HOOK, "OFF_HOOK" },
	{ EVENT_DIALING, "DIALING" },
	{ EVENT_RINGING, "RINGING" },
	{ EVENT_CALLED_POTS_READY, "EVENT_CALLED_POTS_READY" },
	{ EVENT_V21FLAG_DETECTED, "EVENT_V21FLAG_DETECTED" },
	{ EVENT_BUSY, "EVENT_BUSY" },
	{ EVENT_NB_WB_SWITCH, "EVENT_NB_WB_SWITCH" },
	{ EVENT_SWITCH_TO_WB, "EVENT_SWITCH_TO_WB" },
	{ EVENT_SWITCH_TO_WB, "EVENT_SWITCH_TO_WB" },
	{ EVENT_READY, "READY" },
	{ EVENT_REINITIALIZE_ENDPS, "SIP TO EVENT_REINITIALIZE_ENDPS" },
	{ EVENT_DTMF_DETECTED, "EVENT_DTMF_DETECTED" },
	{ EVENT_POTS2POTS_CALL, "POTS to POTS CALL" },
	{ EVENT_POTS2SIP_CALL, "POTS to SIP CALL" },
	{ EVENT_SIP2POTS_CALL, "SIP to POTS CALL" },
	{ EVENT_SIP2VOIP_CALL, "SIP TO VOIP (TRANSCODING)" },
	{ EVENT_VOIP2SIP_CALL, "VOIP (TRANSCODING) TO SIP" },
	{ EVENT_SIP_CONFIRMED, "SIP CONFIRMED" },
	{ EVENT_SIP_DISCONNECTED, "SIP DISCONNECTED" }
};


call_t pool_call[] = 
{
	{ CALL_POTS2POTS, "POTS to POTS" },
	{ CALL_POTS2SIP, "POTS to SIP" },
	{ CALL_SIP2POTS, "SIP to POTS" },
	{ CALL_SIP2VOIP, "SIP to VOIP (transcoding)" },
	{ CALL_VOIP2SIP, "VOIP (transcoding) to SIP" }
};


/*!
*	This function returns the pointer to the called party when a call is initiated
*	from a POTS phone. The following 2 dial plans are considered:
*	- POTS dial plan:
*		- For a POTS to POTS call, the function returns the pointer to the endpt 
*		whose Phone Id matches the dialstring number.
*		- For a POTS to SIP call, the function returns the pointer to the SIP phone
*		whose associated endpt Phone Id matches the dialstring.
*	- SIP dial plan:
*		- For a POTS to POTS call, the function returns the pointer to the endpt 
*		whose associated SIP session User Id matches the dialstring number.
*		- For a POTS to SIP call, the function returns the pointer to the SIP phone
*		whose associated SIP session User Id matches the dialstring.
*
*	\param calltype Pointer to the calltype value (POTS2POTS or POTS2SIP in this case).
*	\param dialstring String that contains the sequence of dialed DTMF.
*	\param dialstring_len DTMF string length.
*	\param dial_plan Pointer to the dial plan value.
*	\retval NULL on failure
*	\retval void* on success, pointer to endpt in case of POTS to POTS call
*	\retval void* on success, pointer to SIP phone instance in case of POTS to SIP call
*/
void *get_called_party_from_dialstring(int *calltype, char *dialstring, int dialstring_len, int dial_plan)
{
	void * ptr = NULL;
	endpt_t *endpt;
	unsigned long dialed_number;
	int i;

	dialed_number = (U32) strtoul(dialstring, NULL, 10);

	if ((dial_plan == DIAL_PLAN_SIP) && (dialstring_len != 4))
		goto exit;

	if ((dial_plan == DIAL_PLAN_POTS) && (dialstring_len > 1))
		goto exit;

	for (i = 0; (i < MAX_ENDPT_ID) && (ptr == NULL); i++)
	{
		endpt = &endpt_pool[i];

		if ((dial_plan == DIAL_PLAN_POTS) && (endpt->phone_id != dialed_number))
			continue;
		if ((dial_plan == DIAL_PLAN_SIP) && (endpt->sip_session->user_id != dialed_number))
			continue;

		if (i < MAX_POTS_ENDPT_ID) /* POTS to POTS call */
		{
			ptr = (void *) endpt;
			*calltype = CALL_POTS2POTS;
			PDEBUG(DBG_L2, "get_called_party_from_dialstring() -> found sip POTS endpt %d, PhoneId %d",
				endpt->id, endpt->phone_id);
		}
		else /* POTS to SIP call */
		{
			sip_phone_t *sip_phone = endpt->sip_session->sip_phone;
			if (!sip_phone)
				continue;

			ptr = (void *) sip_phone;
			*calltype = CALL_POTS2SIP;
			PDEBUG(DBG_L2, "get_called_party_from_dialstring() -> found sip phone user id %ld, URI %s",
					sip_phone->user_id, sip_phone->uri);
		}
	}

exit:
	return ptr;
}


/*!
*	This function performs a "soft reset" of a given endpt instance by modifying
*	some of the endpt structure parameters to mark the endpt as idle.
*	\param *endpt Pointer to the endpt instance to be "soft reset".
*	\retval None
*/
void endpt_reset(endpt_t *endpt)
{
	endpt->peer_endpt = NULL; /* means no peer for now */
	endpt->call_waiting_endpt = NULL; /* means no call waiting for now */

	endpt->state = STATE_IDLE;		
	endpt->event = -1;
	endpt->call = CALL_POTS2POTS;

	endpt->dial_plan = DIAL_PLAN_POTS;
	memset(endpt->dialstring, 0, sizeof(endpt->dialstring));
	endpt->dialstring_len = 0;
	endpt->dtmf = -1;

	memcpy(&endpt->ip, &netif_cfg->device_ip, sizeof(netif_cfg->device_ip));
	endpt->rtp = htons(DEFAULT_UDP_PORT + 2*endpt->id); /* RTCP port must be even */
	endpt->rtcp = htons(DEFAULT_UDP_PORT + 2*endpt->id + 1); /* RTCP port must be odd */

	endpt->voip_profile.codec = -1;
	endpt->voip_profile.payload_type = -1;
	endpt->voip_profile.ptime = -1;
}


/*!
*	For a given endpt instance, this function initializes all the parameters which have 
*	to be setup once for ever at the initialization time.
*	\param endpt_id Unique Id of the endpt instance to be initialized.
*	\param *endpt Pointer to the endpt instance to be initialized.
*	\retval None
*/
void endpt_init_once(int endpt_id, endpt_t *endpt)
{
	memset(endpt, 0, sizeof(endpt_t));

	endpt->id = endpt_id;

	/* this is the phone number for this endpt */
	endpt->phone_id = endpt_id + 1;

	if (endpt_id < MAX_POTS_ENDPT_ID)
	{
		/* timeslot assignment */
		endpt->timeslot[0] = 2*endpt_id;
		endpt->timeslot[1] = 2*endpt_id + 1;
		endpt->timeslot[2] = 2*endpt_id + 64;
		endpt->timeslot[3] = 2*endpt_id + 65;
	}
	else
	{
		/*timeslot assignment*/
		endpt->timeslot[0] = 0xFFFF;
		endpt->timeslot[1] = 0xFFFF;
		endpt->timeslot[2] = 0xFFFF;
		endpt->timeslot[3] = 0xFFFF;
	}

	endpt->sip_session = NULL;
}


/*!
*	This function raises the specified event for a given endpt instance.
*	\param *endpt Pointer to the endpt.
*	\param event Event Id.
*	\retval None
*/
void endpt_raise_event(endpt_t *endpt, int event)
{
	PDEBUG(DBG_L3, "<<< event %s (endpt %d, state %s)", pool_event[event].str, 
				endpt->id, pool_state[endpt->state].str);
	endpt->event = event;
}


/*!
*	This function places a given endpt into the specified state.
*	\param *endpt Pointer to the endpt.
*	\param state State Id.
*	\retval None
*/
void endpt_switch_state(endpt_t *endpt, int state)
{
	PDEBUG(DBG_L3, "--- endpt %d: %s -> %s", 
			endpt->id, pool_state[endpt->state].str, pool_state[state].str);

	endpt->state = state;
}


/*!
*	This function is the state machine handler for an endpt in the idle state.
*	In the idle state, an endpt is not connected and ready to take any call.
*	\param *endpt Pointer to the endpt.
*	\param event Event Id.
*	\retval 0 if the endpt event is successfully processed
*	\retval -1 in case of failure
*/
int endpt_statehdlr_idle(endpt_t *endpt, int event)
{
	int rc = 0;

	/* If the state is IDLE we should receive only OFFHOOK or RING events*/
	switch (event)
	{
	case EVENT_OFF_HOOK: /* endpt associated to a POTS phone going offhook */
		rc = endpt_create_voip(endpt, VAPI_SYNC_MODE); /* allocate MSP resources */
		exit_on_err(rc, exit, "endpt_create_voip");
		rc = endpt_play_tone(endpt, eDIALTONE, VAPI_SYNC_MODE); /* play dialtone until user starts dialing */
		exit_on_err(rc, exit, "endpt_play_tone");
		endpt_switch_state(endpt, STATE_DIALING);
		break;

	case EVENT_SIP2POTS_CALL: /* endpt associated to a SIP phone calling a POTS phone */
		rc = endpt_create_voip(endpt, VAPI_SYNC_MODE); /* allocate MSP resources */
		exit_on_err(rc, exit, "endpt_create_voip"); 
		ring_pots(endpt->id); /* ring the POTS phone */
		endpt_switch_state(endpt, STATE_RINGING);
		break;

	case EVENT_SIP2VOIP_CALL: /* endpt associated to a SIP phone calling a Comcerto IP channel */
		rc = endpt_create_voip(endpt, VAPI_SYNC_MODE); /* allocate MSP resources */
		exit_on_err(rc, exit, "endpt_create_voip");
		ack_sip_invite(endpt->sip_session);
		endpt_switch_state(endpt, STATE_RINGING);

	case EVENT_VOIP2SIP_CALL: /* endpt associated to a SIP phone callied by a Comcerto IP channel */
		rc = endpt_create_voip(endpt, VAPI_SYNC_MODE); /* allocate MSP resources */
		exit_on_err(rc, exit, "endpt_create_voip");
		rc = call_sip_phone(endpt->sip_session->sip_phone, &endpt->sip_session->call_id, endpt->sip_session->id);
		endpt_switch_state(endpt, STATE_CALLING);
		break;

	case EVENT_SIP_DISCONNECTED: /* endpt already hangup just ignore event */
		break;

	default: 
		PDEBUG(DBG_ERR, "!ERR! Endpoint %d: Unexpected event %s in state %s",
				endpt->id, pool_event[event].str, pool_state[endpt->state].str);
		break;
	}
exit:
	return rc;
}


/*!
*	This function is the state machine handler for an endpt in the ringing state.
*	In the ringing state, an endpt is being called by another POTS or SIP party.
*	This state handler applies for both POTS or SIP calls.
*	\param *endpt Pointer to endpt
*	\param event Event identifier
*	\retval 0 if the endpt event is successfully processed
*	\retval -1 in case of failure
*/
int endpt_statehdlr_ringing(endpt_t *endpt, int event)
{
/*	struct legerity_io_disconnect_desc disconnect; */
	int rc = 0; /* return code */

	switch (event) /* If the state is IDLE we should receive only OFFHOOK or RING events*/
	{
	/* stop ringing on pots and move to on hook procedure*/
	case EVENT_SIP_DISCONNECTED:
		stop_ring_pots(endpt->id);

	/* The phone is RINGING. If the phone goes OFFHOOK, an event OFF_HOOK is posted by the signalling thread
	Run timer. If time is up set ENDPOINT_EVENT_REMOTE_TIME_UP event*/
	case EVENT_ON_HOOK:
		/* we stop the call on the remote side and set back the endpoint in idle state */
/* to be UNCOMMENTED		
		disconnect.line = endpt->peer_endpt->id;
		legerity_disconnect(slic->dev_file, &disconnect);
*/
		rc = endpt_hangup(endpt, VAPI_SYNC_MODE);
		exit_on_err(rc, exit, "endpt_hangup");
		rc = endpt_delete_voip(endpt, VAPI_SYNC_MODE);
		exit_on_err(rc, exit, "endpt_delete_voip");
		endpt_reset(endpt);
		break;

	/* POTS to POTS call event only */
	case EVENT_OFF_HOOK:
		if (endpt->call == CALL_SIP2POTS)
		{
			ack_sip_invite(endpt->sip_session); /* acknowledge the received SIP invite */
		}

		if (endpt->call == CALL_POTS2POTS)
		{
			rc = netif_modify(endpt->id, &endpt->ip[0], &endpt->peer_endpt->ip[0], endpt->rtp, endpt->peer_endpt->rtp);
			exit_on_err(rc, exit, "netif_modify");
			rc = endpt_start_voip(endpt, VAPI_SYNC_MODE);
			exit_on_err(rc, exit, "endpt_start_voip");
			endpt_switch_state(endpt, STATE_READY); /* this endpt is now ready */

			endpt_raise_event(endpt->peer_endpt, EVENT_CALLED_POTS_READY); /* notify calling party */
		}
		break;

	/* SIP to POTS call and SIP to SIP transcoding incoming call event */
	case EVENT_SIP_CONFIRMED:
		rc = netif_modify(endpt->id, &endpt->ip[0], &endpt->sip_session->dst_ip[0], endpt->rtp, endpt->sip_session->dst_rtp);
		exit_on_err(rc, exit, "netif_modify");
		get_local_sdp_media_info(endpt->sip_session, &endpt->voip_profile);
		rc = endpt_modify_voip(endpt);
		exit_on_err(rc, exit, "endpt_modify_voip");
		rc = endpt_start_voip(endpt, VAPI_SYNC_MODE);
		exit_on_err(rc, exit, "endpt_start_voip");
		endpt_switch_state(endpt, STATE_READY); /* this endpt is now ready */
		break;

	default:
		PDEBUG(DBG_ERR, "!ERR! Endpoint %d: Unexpected event %s in state %s",
				endpt->id, pool_event[event].str, pool_state[endpt->state].str);
		break;
	}				

exit:
	return rc;
}


/*!
*	This function is the state machine handler for an endpt in the dialing state.
*	In the dialing state, an endpt is dialing DTMF from a POTS phone.
*	This state handler applies only for calling POTS phones.
*	\param *endpt Pointer to endpt
*	\param event Event identifier
*	\retval 0 if the endpt event is successfully processed
*	\retval -1 in case of failure
*/
int endpt_statehdlr_dialing(endpt_t *endpt, int event)
{
	endpt_t *called_endpt;
	int calltype;
	int block_pots_to_pots = 1;
	void *called_party;
	int rc = 0; /* return code */

	/* If the state is IDLE we should receive only OFFHOOK or RING events*/
	switch (event)
	{
	/* The phone is RINGING. If the phone goes OFFHOOK, an event OFF_HOOK is posted by the signalling thread
	Run timer. If time is up set ENDPOINT_EVENT_REMOTE_TIME_UP event*/
	case EVENT_ON_HOOK:
		rc = endpt_hangup(endpt, VAPI_SYNC_MODE);
		exit_on_err(rc, exit, "endpt_hangup");
		rc = endpt_delete_voip(endpt, VAPI_SYNC_MODE);
		exit_on_err(rc, exit, "endpt_delete_voip");
		endpt_reset(endpt);
		break;

	case EVENT_DTMF_DETECTED:
		/* store the detected digit (ASCII value) into the endpt structure */
		PDEBUG(DBG_L2, "Adding DTMF digits to endpt %d", endpt->id);

		if (endpt->dialstring_len == 0)
		{
			if (endpt->dtmf == DTMF_SWITCH_DIAL_PLAN_POTS)
			{
				endpt->dial_plan = DIAL_PLAN_POTS;
				PDEBUG(DBG_L1, "Apply POTS Dial Plan");
				endpt_pool_display();
				break;
			}

			if (endpt->dtmf == DTMF_SWITCH_DIAL_PLAN_SIP)
			{
				endpt->dial_plan = DIAL_PLAN_SIP;
				PDEBUG(DBG_L1, "Apply SIP Dial Plan");
				endpt_pool_display();
				break;
			}
		}
		
		endpt->dialstring[endpt->dialstring_len] = endpt->dtmf + 0x30; 
		endpt->dialstring_len++;

		/* stop the dial tone at the first DTMF digit dialed */
		if (endpt->dialstring_len == 1)
		{
			rc = endpt_stop_tone(endpt, VAPI_SYNC_MODE);
			exit_on_err(rc, exit, "endpt_stop_tone");
		}

		/* if no endpt found then continue collecting dtmf digits */
		called_party = get_called_party_from_dialstring(&calltype, endpt->dialstring, endpt->dialstring_len, endpt->dial_plan);
		if (called_party == NULL)
			break;

		/* found endpt with the dial string... */
		/* reset phonenumber */
		memset(endpt->dialstring, 0, sizeof(endpt->dialstring));
		endpt->dialstring_len = 0;

		/* POTS to POTS calls blocked*/
		if (calltype == CALL_POTS2POTS && block_pots_to_pots == 1)
			break;

		if (calltype == CALL_POTS2POTS)
		{
			called_endpt = (endpt_t *) called_party;

			/* play busy tone to originate POTS endpt if peer endpt is not idle */
			if (called_endpt->state != STATE_IDLE)
			{
				rc = endpt_play_tone(endpt, eBUSYTONE, VAPI_SYNC_MODE);
				exit_on_err(rc, exit, "endpt_play_tone");
			}
			else
			{
				called_endpt->call = calltype;

				/* allocate MSP resource */
				rc = endpt_create_voip(called_endpt, VAPI_SYNC_MODE);
				exit_on_err(rc, exit, "endpt_create_voip");

				/* map called/calling parties */
				endpt->peer_endpt = called_endpt;
				called_endpt->peer_endpt = endpt;

				/* ring the called POTS phone */
				ring_pots(called_endpt->id); /* ring the POTS phone */
				endpt_switch_state(called_endpt, STATE_RINGING);

				rc = endpt_play_tone(endpt, eRINGBACKTONE, VAPI_SYNC_MODE);
				exit_on_err(rc, exit, "endpt_play_tone");
				endpt_switch_state(endpt, STATE_CALLING);
			}
		}
		else /* POTS to SIP call */
		{
			sip_phone_t *called_sip_phone = (sip_phone_t *) called_party;
			endpt->call = calltype;
			rc = call_sip_phone(called_sip_phone, &endpt->sip_session->call_id, endpt->sip_session->id);

			rc = endpt_play_tone(endpt, eRINGBACKTONE, VAPI_SYNC_MODE);
			exit_on_err(rc, exit, "endpt_play_tone");
			endpt_switch_state(endpt, STATE_CALLING);
		}

		break;

	default:
		PDEBUG(DBG_ERR, "!ERR! Endpoint %d: Unexpected event %s in state %s",
				endpt->id, pool_event[event].str, pool_state[endpt->state].str);
		break;
	}				

exit:
	return rc;
}


/*!
*	This function is the state machine handler for an endpt in the calling state.
*	In the calling state, an endpt waits for the called party (SIP or POTS) to accept the call.
*	This state handler applies to both POTS and SIP calls.
*	\param *endpt Pointer to endpt
*	\param event Event identifier
*	\retval 0 if the endpt event is successfully processed
*	\retval -1 in case of failure
*/
int endpt_statehdlr_calling(endpt_t *endpt, int event)
{
	int rc = 0; /* return code */

	switch (event) /* If the state is IDLE we should receive only OFFHOOK or RING events*/
	{
	case EVENT_ON_HOOK:
		rc = endpt_hangup(endpt, VAPI_SYNC_MODE);
		exit_on_err(rc, exit, "endpt_hangup");
		rc = endpt_delete_voip(endpt, VAPI_SYNC_MODE);
		exit_on_err(rc, exit, "endpt_delete_voip");
		endpt_reset(endpt);
		break;

	case EVENT_CALLED_POTS_READY: /* POTS to POTS call */
		rc = endpt_stop_tone(endpt, VAPI_SYNC_MODE); /* stop the ringback tone */
		exit_on_err(rc, exit, "endpt_stop_tone");
		rc = netif_modify(endpt->id, &endpt->ip[0], &endpt->peer_endpt->ip[0], endpt->rtp, endpt->peer_endpt->rtp);
		exit_on_err(rc, exit, "netif_modify");
		rc = endpt_start_voip(endpt, VAPI_SYNC_MODE);
		exit_on_err(rc, exit, "endpt_start_voip");
		rc = endpt_start_loopback(endpt);
		exit_on_err(rc, exit, "endpt_start_loopback");

		endpt_switch_state(endpt, STATE_READY); /* this endpt is now ready */
		break;

	case EVENT_SIP_CONFIRMED: /* SIP to VOIP or POTS to SIP call */
		if (endpt->call == CALL_POTS2SIP)
		{
			rc = endpt_stop_tone(endpt, VAPI_SYNC_MODE); /* stop the ringback tone */
			exit_on_err(rc, exit, "endpt_stop_tone");
		}

		rc = netif_modify(endpt->id, &endpt->ip[0], &endpt->sip_session->dst_ip[0], endpt->rtp, endpt->sip_session->dst_rtp);
		exit_on_err(rc, exit, "netif_modify");
		get_local_sdp_media_info(endpt->sip_session, &endpt->voip_profile);
		rc = endpt_modify_voip(endpt);
		exit_on_err(rc, exit, "endpt_modify_voip");
		rc = endpt_start_voip(endpt, VAPI_SYNC_MODE);
		exit_on_err(rc, exit, "endpt_start_voip");
		rc = endpt_start_transcoding(endpt);
		exit_on_err(rc, exit, "endpt_start_transcoding");

		endpt_switch_state(endpt, STATE_READY); /* this endpt is now ready */
		break;

	default:
		PDEBUG(DBG_ERR, "!ERR! Endpoint %d: Unexpected event %s in state %s",
				endpt->id, pool_event[event].str, pool_state[endpt->state].str);
		break;
	}				

exit:
	return rc;
}


/*!
*	This function is the state machine handler for an endpt in the ready state.
*	In the ready state, an endpt is online and connected with a SIP or POTS party.
*	This state handler applies to both POTS and SIP calls.
*	\param *endpt Pointer to endpt
*	\param event Event identifier
*	\retval 0 if the endpt event is successfully processed
*	\retval -1 in case of failure
*/
int endpt_statehdlr_ready(endpt_t *endpt, int event)
{
	int rc = 0; /* return code */
	int band_switch_block = 1;
	endpt_t *peer_endpt;
	sip_session_t *peer_sip_session;

	switch (event) /* This endpt is connected.*/
	{
	case EVENT_ON_HOOK:
		rc = endpt_hangup(endpt, VAPI_SYNC_MODE);
		exit_on_err(rc, exit, "endpt_hangup");
		rc = endpt_delete_voip(endpt, VAPI_SYNC_MODE);
		exit_on_err(rc, exit, "endpt_delete_voip");
		sip_session_reset(endpt->sip_session);
		endpt_reset(endpt);
		break;

	case EVENT_SIP_DISCONNECTED:
		if ((endpt->call == CALL_POTS2SIP) || (endpt->call == CALL_SIP2POTS))
		{
			rc = endpt_play_tone(endpt, eBUSYTONE, VAPI_SYNC_MODE); /* play busy tone to notify peer disconnected */
			exit_on_err(rc, exit, "endpt_play_tone");
		}
		else if ((endpt->call == CALL_SIP2VOIP) || (endpt->call == CALL_VOIP2SIP))
		{
			peer_endpt = endpt->peer_endpt;
			peer_sip_session = peer_endpt->sip_session;

			rc = endpt_hangup(endpt, VAPI_SYNC_MODE);
			exit_on_err(rc, exit, "endpt_hangup");
			rc = endpt_delete_voip(endpt, VAPI_SYNC_MODE);
			exit_on_err(rc, exit, "endpt_delete_voip");
			sip_session_reset(endpt->sip_session);
			endpt_reset(endpt);
			
			if (peer_endpt != NULL)
			{
				peer_sip_session = peer_endpt->sip_session;

				rc = endpt_hangup(peer_endpt, VAPI_SYNC_MODE);
				exit_on_err(rc, exit, "endpt_hangup");
				rc = endpt_delete_voip(peer_endpt, VAPI_SYNC_MODE);
				exit_on_err(rc, exit, "endpt_delete_voip");
				endpt_reset(peer_endpt);

				sip_session_hangup(peer_sip_session);
			}
		}
		break;

	case EVENT_V21FLAG_DETECTED:
		PDEBUG(DBG_L2, "Endpoint %d: event V21 flags detected (%d) in state %d", endpt->id, event, endpt->state);
		rc = endpt_switch_to_t38(endpt, VAPI_SYNC_MODE);
		exit_on_err(rc, exit, "endpt_switch_to_t38");
		rc = endpt_switch_to_t38(endpt, VAPI_SYNC_MODE);
		exit_on_err(rc, exit, "endpt_switch_to_t38");
		break;

	case EVENT_DTMF_DETECTED: /* dtmf service */
		if (endpt->dtmf == 8 && band_switch_block != 1) /* switch to narrowband mode with DTMF 8 */
			rc = endpt_band_switch(endpt, eNarrowBand);
		
		if (endpt->dtmf == 9 && band_switch_block != 1) /* switch to widewband mode with DTMF 9 */
			rc = endpt_band_switch(endpt, eWideBand);

		exit_on_err(rc, exit, "endpt NB/WB switch");
		break;

	default:
		PDEBUG(DBG_ERR, "!ERR! Endpoint %d: Unexpected event %s in state %s",
				endpt->id, pool_event[event].str, pool_state[endpt->state].str);
		break;
	}				

exit:
	return rc;
}


/*!
*	This is the endpt state machine which calls the appropriate endpt state 
*	handler for endpts for which a new event was raised.
*	\param *none 
*	\retval None
*/
void *endpt_state_machine(void *none)
{
	endpt_t *endpt;
	int event;
	int i;
	int rc = 0; /* return code */

	sipua_create(&sipua);

	/* We should have here a signal handler to interrupt the while loop
	For now loop forever, CTRL-C to be used to stop. */
	PDEBUG(DBG_L1, "Endpoint state machine thread started");
	while(1)
	{
		/* Parse all the endpt_pool in the global endpt_pool array*/
		for (i = 0; i < MAX_ENDPT_ID; i ++)
		{
			endpt = &endpt_pool[i];

			if (endpt->event < 0) /* check next endpt if no event for current one */
				continue;

			PDEBUG(DBG_L2, "[ -- state_machine: endpt %d, state %s, call %s / process event %s -- ]", 
				endpt->id, pool_state[endpt->state].str,
				pool_call[endpt->call].str, pool_event[endpt->event].str);

			event = endpt->event; /* save event to be processed now */
			endpt->event = -1; /* clear endpoint event as it's processed now */

			switch (endpt->state) /* process event according to current state */
			{
			case STATE_IDLE:
				rc = endpt_statehdlr_idle(endpt, event);
				break;

			case STATE_RINGING:
				rc = endpt_statehdlr_ringing(endpt, event);
				break;

			case STATE_DIALING:
				rc = endpt_statehdlr_dialing(endpt, event);
				break;

			case STATE_CALLING:
				rc = endpt_statehdlr_calling(endpt, event);
				break;

			case STATE_READY:
				rc = endpt_statehdlr_ready(endpt, event);
				break;

			default:
				break;
			}
		}

		usleep(100);
	}
}


/*!	@} */