/*!
*	\file debug.c
*
*	\defgroup debug debug Module
*
*	debug Module C-functions
*
*	\attention
*	Copyright © 2004-2010 Mindspeed Technologies, Inc.\n
*	Mindspeed Confidential.\n
*	All rights reserved.\n
*	This file is a component of the Mindspeed® VAPI software ("VAPI") and is
*	distributed under the Mindspeed Software License Agreement (the "Agreement").\n
*	Before using this file, you must agree to be bound by the the terms and 
*	conditions of the Agreement.
*
*	@{
*/

#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <string.h>

/* VAPI headers */
#include <msp.h>
#include <vapi.h>
#include <gtl.h>
#include <comcerto-api-defs.h>
#include <comcerto-ud-types.h>

/* Zarlink headers */
#include <legerity_lib.h>

/* mtalk header */
#include <mtalk.h>

/*!
*	This function allocates and initializes a debug_cfg instance with hardcoded and/or user defined values.
*	\param *cfg_file Configuration filename (string).
*	\retval NULL on failure
*	\retval debug_cfg_t* on success, pointer to the allocated and initialized debug_cfg structure
*/
debug_cfg_t *get_debug_cfg(const char *cfg_file)
{
	struct _CFG *cfg_info;
	debug_cfg_t *debug_cfg;
	unsigned char src_ip[4]; /* diag source IP */
	unsigned char dst_ip[4]; /* diag destination IP */
	char diag_fc_list[100]; /* list of MSP diag function codes (string) */
	char *str_token;

	debug_cfg = (debug_cfg_t *)malloc(sizeof(debug_cfg_t));
	exit_if_null(debug_cfg, err, "get_debug_cfg() - mem allocation fail");

	cfg_info = cfg_read(cfg_file, 0);
	exit_if_null(cfg_info, err, "get_debug_cfg() - mem allocation fail");

	READ_STR(cfg_info, "DEBUG", "DIAG_FUNCTION_CODES", diag_fc_list, "");
	READ_INT(cfg_info, "DEBUG", "DIAG_REDIRECT_MODE", debug_cfg->diag.redirect.mode, 0);

	READ_MAC(cfg_info,"NETIF", "DEVICE_MAC", &debug_cfg->diag.redirect.src_mac[0], (U8 *) DEVICE_MAC);
	READ_MAC(cfg_info,"DEBUG", "DIAG_REDIRECT_DST_MAC", &debug_cfg->diag.redirect.dst_mac[0], (U8 *) HOST_MAC);

	READ_IP(cfg_info, "NETIF", "DEVICE_IP", src_ip, DEVICE_IP);
	READ_IP(cfg_info, "DEBUG", "DIAG_REDIRECT_DST_IP", dst_ip, "192.168.32.3");

	READ_INT(cfg_info, "DEBUG", "DIAG_REDIRECT_SRC_UDP", debug_cfg->diag.redirect.src_udp, DIAG_REDIRECT_SRC_UDP);
	READ_INT(cfg_info, "DEBUG", "DIAG_REDIRECT_DST_UDP", debug_cfg->diag.redirect.dst_udp, DIAG_REDIRECT_DST_UDP);

	cfg_clean(cfg_info);


	debug_cfg->diag.redirect.src_ip = (src_ip[3] << 24) | (src_ip[2] << 16) | (src_ip[1] << 8) | src_ip[0];
	debug_cfg->diag.redirect.dst_ip = (dst_ip[3] << 24) | (dst_ip[2] << 16) | (dst_ip[1] << 8) | dst_ip[0];

	debug_cfg->diag.nb_fc = 0;
	str_token = (char *)strtok(diag_fc_list, ",");
	while (str_token != NULL)
	{
		debug_cfg->diag.fc[debug_cfg->diag.nb_fc] = strtol(str_token, NULL, 16);
		PDEBUG(DBG_L1, "Set diag 0x%04x", debug_cfg->diag.fc[debug_cfg->diag.nb_fc]);

		str_token = (char *)strtok(NULL, ",");
		debug_cfg->diag.nb_fc++;
	}

	return debug_cfg;

err:
	return NULL;
}


/*!
*	This function displays the values of debug parameters (debug_cfg).
*	\param *debug_cfg Pointer to the debug configuration instance
*	\retval None
*/
void display_debug_cfg(debug_cfg_t *debug_cfg)
{
	int i;

	if (!debug_cfg)
		return;

	PDEBUG(DBG_L2, "\n-------------------- debug_cfg");

	PDEBUG(DBG_L2, "- nb_diag_fc   %d", debug_cfg->diag.nb_fc);

	for (i = 0; i < debug_cfg->diag.nb_fc; i++)
		PDEBUG(DBG_L2, " diag_fc[%d]   %d", i, debug_cfg->diag.fc[i]);

	PDEBUG(DBG_L2, "- diag.redirect.mode   %d", debug_cfg->diag.redirect.mode);

	PDEBUG(DBG_L2, "- diag.redirect.src_mac  %02x:%02x:%02x:%02x:%02x:%02x",
		debug_cfg->diag.redirect.src_mac[0], debug_cfg->diag.redirect.src_mac[1], 
		debug_cfg->diag.redirect.src_mac[2], debug_cfg->diag.redirect.src_mac[3], 
		debug_cfg->diag.redirect.src_mac[4], debug_cfg->diag.redirect.src_mac[5]);

	PDEBUG(DBG_L2, "- diag.redirect.dst_mac  %02x:%02x:%02x:%02x:%02x:%02x",
		debug_cfg->diag.redirect.dst_mac[0], debug_cfg->diag.redirect.dst_mac[1], 
		debug_cfg->diag.redirect.dst_mac[2], debug_cfg->diag.redirect.dst_mac[3], 
		debug_cfg->diag.redirect.dst_mac[4], debug_cfg->diag.redirect.dst_mac[5]);

	PDEBUG(DBG_L2, "- diag.redirect.src_ip  %02x", debug_cfg->diag.redirect.src_ip);
	PDEBUG(DBG_L2, "- diag.redirect.dst_ip  %02x", debug_cfg->diag.redirect.dst_ip);  
	PDEBUG(DBG_L2, "- diag.redirect.src_udp  %d", debug_cfg->diag.redirect.src_udp);
	PDEBUG(DBG_L2, "- diag.redirect.dst_udp  %d", debug_cfg->diag.redirect.dst_udp);
	PDEBUG(DBG_L2, "--------------------\n");
}


/*!	
*	This function redirects the diagnostics output to the specified host.
*	\param *diag_redirect
*	\param device_id
*	\retval 0 on success
*	\retval <0 on failure
*	\see VAPI_AllocateMessage
*	\see VAPI_SetMessageFromBuffer
*	\see VAPI_SendDeviceMessage
*	\see VAPI_FreeMessage
*/
int diag_redirect(diag_redirect_t *diag_redirect, int device_id)
{
        int rc = 0; /* return code */
        void *message;
        U32 response_len = DEFAULT_FIFO_MAX_SIZE;
        U8 device_response [DEFAULT_FIFO_MAX_SIZE];
        struct _REDIRECT_CSM_UDP diag_cmd; /*defined in comcerto-ud-types.h */

        if(diag_redirect == NULL)
                return -1;

        /* allocate a message to query the current options */
        message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
        if (message == NULL)
                return -1;

        memset(&diag_cmd, 0, sizeof(struct _REDIRECT_CSM_UDP));

        diag_cmd.param_4.bits.protocol = REDIRECT_CSM_PROTOCOL_OPENDIAGNOSTICS;
        memcpy(diag_cmd.dst_mac, diag_redirect->dst_mac, 6);
        memcpy(diag_cmd.src_mac, diag_redirect->src_mac, 6);

        if(diag_redirect->mode == FC_REDIRECT_CSM)
        {
                diag_cmd.packet_type = htons(REDIRECT_CSM_TYPE_DEFAULT); /* csme protocol*/

                /* build the command to set the diag redirection*/
                rc = VAPI_SetMessageFromBuffer(message, 
                                                CMD_CLASS_OPEN_DIAG,
                                                CMD_TYPE_DIAG_CONFIG, 
                                                FC_REDIRECT_CSM,
                                                8,              /*8 words in the command*/
                                                (U16 *)&diag_cmd);

		exit_on_err(rc, err, "VAPI_SetMessageFromBuffer");
        }
        else /* assume redirect over IP/UDP*/
        {
                diag_cmd.packet_type = htons(REDIRECT_CSM_UDP_PACKET_TYPE_DEFAULT); /* IP protocol*/

                diag_cmd.param_12.bits.ip_hl = REDIRECT_CSM_UDP_IP_HL_DEFAULT;
                diag_cmd.param_12.bits.ip_v = REDIRECT_CSM_UDP_IP_V_DEFAULT;
                diag_cmd.param_12.bits.ip_tos = REDIRECT_CSM_UDP_IP_TOS_DEFAULT;

                diag_cmd.ip_len = REDIRECT_CSM_UDP_IP_LEN_DEFAULT;
                diag_cmd.ip_id = REDIRECT_CSM_UDP_IP_ID_DEFAULT;
                diag_cmd.ip_fragment = REDIRECT_CSM_UDP_IP_FRAGMENT_DEFAULT;
                diag_cmd.param_16.word = htons(0x8011); /* UDP , TTL*/

                diag_cmd.ip_checksum = REDIRECT_CSM_UDP_IP_CHECKSUM_DEFAULT;

                diag_cmd.ip_src = diag_redirect->src_ip;
                diag_cmd.ip_dst = diag_redirect->dst_ip;
                diag_cmd.udp_sport = diag_redirect->src_udp;
                diag_cmd.udp_dport = diag_redirect->dst_udp;

		PDEBUG(DBG_L2, "Redirect Diags over UDP : src IP 0x%x, src UDP %d dst IP 0x%x, dst UDP %d", 
					diag_redirect->src_ip, diag_redirect->src_udp,
					diag_redirect->dst_ip, diag_redirect->dst_udp);

                diag_cmd.udp_len = UT_CPU2LE16(REDIRECT_CSM_UDP_UDP_LEN_DEFAULT);
                diag_cmd.udp_checksum = UT_CPU2LE16(REDIRECT_CSM_UDP_UDP_CHECKSUM_DEFAULT);

                /* build the command to set the diag redirection*/
                rc = VAPI_SetMessageFromBuffer(message, CMD_CLASS_OPEN_DIAG, CMD_TYPE_DIAG_CONFIG, 
                        FC_REDIRECT_CSM_UDP, sizeof(struct _REDIRECT_CSM_UDP)/2, (U16 *)&diag_cmd);
		exit_on_err(rc, err, "VAPI_SetMessageFromBuffer");
        }

        rc = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, device_response, &response_len);
	exit_on_err(rc, err, "VAPI_SendDeviceMessage");

err:
        VAPI_FreeMessage(message);
        return rc ;
}



/*!	
*	This function enables/disables the sprecified diagnostics.
*	\param id
*	\param command_class
*	\param command_type
*	\param function_code
*	\param action
*	\retval 0 on success
*	\retval <0 on failure
*	\see VAPI_AllocateMessage
*	\see VAPI_SetMessage
*	\see VAPI_SendDeviceMessage
*	\see VAPI_SendConnectionMessage
*	\see VAPI_FreeMessage
*/
int diag_activate(unsigned int id,   /* if 0xFFFF device else connection*/
			unsigned char command_class, 
			unsigned char command_type, 
			unsigned short function_code, 
			unsigned char action) /* 1 = enable, 0 = disable */
{
	int rc = 0; /* return code */
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/* allocate a message to query the current options */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;

	/* build the command to set the diag config to enable or disable */
	rc = VAPI_SetMessage(message, command_class, command_type, function_code, 1, action);
	exit_on_err(rc, err,"VAPI_SetMessage");

	if(id == 0xFFFF)
		rc = VAPI_SendDeviceMessage(0, (SMsg *)message, NULL, 
				device_response, &response_len);
	else
		rc = VAPI_SendConnectionMessage(id, (SMsg *)message, NULL, 
				device_response, &response_len);

err:
	VAPI_FreeMessage(message);
	return rc;
}


/*!
*	This function parses the command line options to set the VAPI and application debug levels.
*	\param *diag Pointer to the diagnostics instance.
*	\retval 0 on success
*	\retval <0 on failure
*/
int diag_init(diag_t *diag)
{
	int rc = 0; /* return code */
	int i;

	if (diag->redirect.mode == 1)
		rc = diag_redirect(&diag->redirect, 0);
	
	exit_on_err(rc, exit, "diag_init");

	for (i = 0; i < diag->nb_fc; i++)
		rc |= diag_activate(0xffff, CMD_CLASS_OPEN_DIAG, CMD_TYPE_DIAG_MON_LIVE_CTRL, diag->fc[i], 1);

	exit_on_err(rc, exit, "diag_init");

exit:
	return rc;
}



/*!	
*	This function parses the command line options to set the VAPI and application debug levels
*	\param *debug_cfg Pointer to the debug configuration instance
*	\retval 0 on success
*	\retval -1 on failure
*/
int debug_init(debug_cfg_t *debug_cfg)
{
	int rc = 0; /* return code */

	rc = diag_init(&debug_cfg->diag);

	return rc;
}


/*!	@} */
