/*
 * Copyright (C) 2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed (R) libtsi and is distributed under
 * the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and
 * conditions of the Agreement.
 *
 */

#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/mman.h>

#include "libtsi.h"

int tsi_connect(void *base,
		char instream, char outstream,
		unsigned char inch, unsigned char outch,
		char instep, char outstep, char count)
{
	int in_type, out_type, in_stream, out_stream;
	int i;
	volatile unsigned short *ibrr, *obrr;
	unsigned short connection_bits;

	if (!base)
		return -EFAULT;

	in_type = instream & TSI_PORT_BACKPLANE ? 1 : 0;
	out_type = outstream & TSI_PORT_BACKPLANE ? 1 : 0;
	in_stream = instream & ~TSI_PORT_BACKPLANE;
	out_stream = outstream & ~TSI_PORT_BACKPLANE;
	if (in_stream < 0 || in_stream > 31)
		goto err;
	if (out_stream < 0 || out_stream > 31)
		goto err;

	if (count > 1 && (inch + instep*count >= 256 || outch + outstep*count >= 256))
		goto err;

	/* get correct addresses for selected input and output stream bit rate registers */
	ibrr = in_type ? &TSI_BIBRR(base, 0) : &TSI_LIBRR(base, 0);
	obrr = out_type ? &TSI_BOBRR(base, 0) : &TSI_LOBRR(base, 0);
	/* connection memory word - high bit selects opposite input if zero, we also set LE bit */
	connection_bits = (in_type ^ out_type) ? 0x2000 : 0xA000;

	/* 8MHz bit rate */
	ibrr[in_stream] = 2;
	obrr[out_stream] = 2;

	/* FPW    0   frame pulse width 122ns
	 * MODE32 0   32MHz mode off
	 * C8IPOL 1   8MHz input clock polarity
	 * COPOL  0   output clock is the same polarity as input
	 * MBP    0   memory block programming disabled
	 * OSB    1   output stand by, outputs enabled
	 * MS   xxx   memory select - equal to out_type
	 */
	TSI_CR(base) = 0x0048 | out_type;

	for (i = 0; i < count; i++) {
		((volatile unsigned short *)base)[0x4000 | (out_stream << 9) | outch] =
			connection_bits | (in_stream << 8) | inch;

		inch += instep;
		outch += outstep;
	}

	return 0;

err:
	return -EINVAL;
}

int tsi_pattern(void *base,
		char stream, unsigned char ch,
		unsigned char pattern,
		char step, char count)
{
	int out_type, out_stream;
	int i;
	volatile unsigned short *obrr;

	if (!base)
		return -EFAULT;

	out_type = stream & TSI_PORT_BACKPLANE ? 1 : 0;
	out_stream = stream & ~TSI_PORT_BACKPLANE;
	if (out_stream < 0 || out_stream >= 32)
		goto err;

	if (count > 1 && ch + step*count >= 256)
		goto err;

	/* get correct addresses for selected output stream bit rate registers */
	obrr = out_type ? &TSI_BOBRR(base, 0) : &TSI_LOBRR(base, 0);

	/* 8MHz bit rate */
	obrr[out_stream] = 2;

	/* FPW    0   frame pulse width 122ns
	 * MODE32 0   32MHz mode off
	 * C8IPOL 1   8MHz input clock polarity
	 * COPOL  0   output clock is the same polarity as input
	 * MBP    0   memory block programming disabled
	 * OSB    1   output stand by, outputs enabled
	 * MS   xxx   memory select - equal to out_type
	 */
	TSI_CR(base) = 0x0048 | out_type;

	for (i = 0; i < count; i++) {
		/* connection memory word - message mode, pattern, we also set LE bit */
		((volatile unsigned short *)base)[0x4000 | (out_stream << 9) | ch] =
			0x6000 | pattern;

		ch += step;
	}

	return 0;

err:
	return -1;
}

#if defined(_WRS_VXWORKS_5_X) || defined(_WRS_VXWORKS_MAJOR)

/* always ok for VxWorks */

int tsi_map(unsigned long phys, void **base)
{
	return 0;
}

int tsi_unmap(void *base)
{
	return 0;
}

#else

int tsi_map(unsigned long phys, void **base)
{
	int fd, res = 0;

	if (!base) {
		res = -EINVAL;
		goto err;
	}

	*base = NULL;

	fd = open("/dev/mem", O_RDWR);
	if (fd < 0) {
		res = -errno;
		goto err;
	}

	*base = mmap(NULL, 64*1024, PROT_READ|PROT_WRITE, MAP_SHARED, fd, phys);

	close(fd);

err:
	return res;
}

int tsi_unmap(void *base)
{
	if (!base)
		return -EINVAL;

	return munmap(base, 64*1024);
}

#endif
