/*! \file c300_init.c  */
/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

/*! This document describes the C300 EVM demo board application (c300_init).

	This application initialises the Network Timming Generator (NTG) and the TSI device of the EVM board.
	device configuration.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifndef _VXWORKS_
#include <vapi/msp.h>
#include <vapi/vapi.h>
#include <vapi/gtl.h>
#else
#include <msp.h>
#include <vapi.h>
#include <gtl.h>
#endif

#include "libtsi.h"


#define DEVICE_CONTROL_MAC		{0x00,0x11,0x22,0x33,0x44,0x55}
#define HOST_CONTROL_INTERFACE		"eth1"
#define HOST_CONTROL_MAC		{0x00,0x1A,0x1B,0x1C,0x1D,0x1E}

#define FC_NTG_CONFIG 0x043F

#ifndef _VXWORKS_

/*! structure with configuration parameters, structure type is defined in gtl.h */
static SCSMEUsrData gtl_device_configuration;

/*! This structure holds default values for the device configuration, user can
 * overwrite them somehow in set_device_configuration() function using his own
 * config. scheme.
 */
SCSMEUsrData default_device_configuration = {
	NULL,				/**< pointer to next device */
	eCSM_ITF,			/**< control interface to use */
	0,				/**< device ID */
	0,				/**< Version infor (not used) */
	eMASTER,			/**< Slave / Master mode */
	DEV_TYPE_M823XX,		/**< Device type */
	True,				/**< Default or custom max channels */
	0,				/**< Max Channels if above flag is custom */
	DEVICE_CONTROL_MAC,		/**< MAC address for control over csmencaps */
	HOST_CONTROL_MAC,		/**< MAC address of the host interface controling the device */
	(char *)HOST_CONTROL_INTERFACE,	/**< host interface used to control the device */
	1				/**< csme ack required */
};

/*=================================================================================*/
/*!
 *	\brief This function fills the global structure with device config parameters
 */
/*=================================================================================*/
void set_device_configuration()
{
	U8 *mac;

	memcpy(&gtl_device_configuration, &default_device_configuration, sizeof(SCSMEUsrData));

	/* user may put here code to overwrite default config parameters, for example
	 * read them from config file
	 */

	gtl_device_configuration.uiDevId = 0;
	gtl_device_configuration.pucEthDevName = HOST_CONTROL_INTERFACE;

	/* this code is executed before VAPI_Init, so we have to use printf's */

	printf("CONTROL_INTERFACE_TYPE  = %d\n", gtl_device_configuration.usControlInterface);
	printf("DEVICE_ID    = %d\n", gtl_device_configuration.uiDevId);
	printf("DEVICE_MODE  = %d\n", gtl_device_configuration.eDevMode);
	printf("DEVICE_TYPE  = %d\n", gtl_device_configuration.ucDevType);
	printf("USE_DEFAULT_MAX_CHANNEL = %d\n", gtl_device_configuration.bUseDefaultMaxChnls);
	printf("CUSTOM_MAX_CHANNEL      = %d\n", gtl_device_configuration.usMaxChannels);
	printf("ACK_REQUIRED = %d\n", gtl_device_configuration.ucIsAckReqd);

	mac = gtl_device_configuration.aucDevMac;
	printf("DEVICE MAC = %02x:%02x:%02x:%02x:%02x:%02x\n", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

	mac = gtl_device_configuration.aucHostMac;
	printf("HOST   MAC = %02x:%02x:%02x:%02x:%02x:%02x\n", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

	printf("HOST_CONTROL_INTERFACE  = %s\n\n", gtl_device_configuration.pucEthDevName);
}

#else

extern SCSMEUsrData	gtl_device_configuration;
extern void		set_device_configuration(SCSMEUsrData *device_configuration);

#endif


/*=================================================================================*/
/*!
 *	\brief
 *	This function set the NTG (Network Timming Generator) of the M823xx device
 */
/*=================================================================================*/
VSTATUS set_ntg_config()
{
	VSTATUS result;
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/* allocate a message to query the current options */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;

	result = VAPI_SetMessage(message, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_NTG_CONFIG, 2, 0x0001, 0x0006);
	if(result != SUCCESS)
		goto err;

	/* send the command, the response is stored in device_response*/
	result = VAPI_SendDeviceMessage(0, (SMsg *)message, NULL, device_response, &response_len);

err:
	VAPI_FreeMessage(message);
	return result;
}

/*=================================================================================*/
/*!
 *	\brief
 *	This function initializes the TSI device of the C300 EVM
 */
/*=================================================================================*/
VSTATUS set_init_tsi_config()
{
	void *base;

	if (tsi_map(0x11400000, &base))
		goto err;

	/*
	 * local stream 0 - TDM0
	 * local streams 6,7 - SLIC0, SLIC1
	 *
	 * cross-connect time-slots 0..3, 64..67 for 0 <-> 7
	 * cross-connect time-slots 4..7, 68..71 for 0 <-> 6
	 */
	tsi_connect(base, 0,7, 0,0, 1,1, 4);
	tsi_connect(base, 7,0, 0,0, 1,1, 4);

	tsi_connect(base, 0,6, 4,4, 1,1, 4);
	tsi_connect(base, 6,0, 4,4, 1,1, 4);

	tsi_connect(base, 0,7, 64,64, 1,1, 4);
	tsi_connect(base, 7,0, 64,64, 1,1, 4);

	tsi_connect(base, 0,6, 68,68, 1,1, 4);
	tsi_connect(base, 6,0, 68,68, 1,1, 4);

	tsi_unmap(base);

	return SUCCESS;

err:
	fprintf(stderr, "tsi init: failed\n");

	return -1;
}


/***************************************************************************
 * main
 ***************************************************************************/
/*! \brief Application entry.

 *  \section Description
*/

#if !defined(_VXWORKS_)
int main(void)
#else
int c300_init(void)
#endif
{
	int status;

	set_device_configuration(&gtl_device_configuration);

	status = VAPI_Init(&gtl_device_configuration);
	if (status != SUCCESS)
	{
		fprintf(stderr, "VAPI_Init: failed, result = %d\n", status);
		goto err0;
	}

	/* disable debug output */
	VAPI_SetDebugLevel(0, 0);

	status = VAPI_OpenDevice(0, NULL);
	if (status != SUCCESS)
	{
		fprintf(stderr, "VAPI_OpenDevice: failed, result = %d\n", status);
		goto err1;
	}

	status = set_ntg_config(0);
	if (status == SUCCESS)
		printf("Network Timing Generator successfully configured\n");
	else
		printf("Network Timing Generator init failed\n");

	status = set_init_tsi_config(0);
	if (status == SUCCESS)
		printf("TSI successfully configured\n");
	else
		printf("TSI init failed\n");

	VAPI_CloseDevice(0, ePURGE);

err1:
	VAPI_Close();
err0:
	return status;
}
