/*! \file ut_port.c 
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include <vxWorks.h>
#include <etherLib.h>
#include <fioLib.h>
#include <logLib.h>
#include <signal.h>
#include <sigevent.h>
#include <sysLib.h>
#include <taskLib.h>
#include <time.h>

#if defined(VXWORKS_6_6)
#include <ipnet_h.h>
#include <ipnet_netif.h>
#elif defined(VXWORKS_6_4)
#include <net/if_arp.h>
#else
#include <ifLib.h>
#endif

#include "ut.h"
#include "ut_port.h"


static FILE *fpLogout = NULL;/**< File Handler for logging*/


static U32 uCurrLogIdx_g = 0;
static char astrLogs[MAX_LOG_ENTRIES][BUFFER_SIZE];
static SMutex logMutex;


struct vsbuffer {
	char *str;
	int size;
	int offset;
};


static STATUS outRoutine(char *buffer, int nchars, int outarg)
{
	struct vsbuffer *pvsb = (struct vsbuffer *)outarg;

	if (pvsb->size - pvsb->offset > nchars)
	{
		memcpy(pvsb->str + pvsb->offset, buffer, nchars);
		pvsb->offset += nchars;
	}
	else
	{
		memcpy(pvsb->str + pvsb->offset, buffer, pvsb->size - pvsb->offset);
		pvsb->offset = pvsb->size - 1;
	}

	return OK;
}

/*This function returns number written characters without '\0'  */
int UT_Vsnprintf(char *str, int size, const char *format, va_list ap)
{
	struct vsbuffer vsb;
	vsb.size = size;
	vsb.str = str;
	vsb.offset = 0;

	fioFormatV(format, ap, outRoutine, (int)&vsb);
	str[vsb.offset] = 0;
	return vsb.offset;
}

int UT_Snprintf(char *str, int size, const char *format, ...)
{
	va_list list;
	int ret;
	UT_va_start(list, format);
	ret = UT_Vsnprintf(str, size, format, list);
	UT_va_end(list);

	return ret;
}

/*Interface Function Definitions*/

/***************************************************************************
 * UT_GetTimeOfDay: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function gets the time of the day.\n
 *  \return
 *   SUCCESS    In case of success\n    
 *   FAILURE    In case of failure\n
 *
 *  \param  pstTime Time struct in which the time of the day is returned.
 */
INT UT_GetTimeOfDay(OUT STimeSpec * psTime)
{
	/* Return clock_gettime return value (0 or -1) */
	if (UT_clock_gettime(CLOCK_REALTIME, psTime) == ERROR)
		return -1;

	return 0;
}

/***************************************************************************
 * UT_ThreadCreate: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function creates the thread.\n
 *
 *  \return
 *   Pointer to the created thread.\n
 *   NULL in case the pthread creation failed.\n
 *
 *  \param pfnFunc  Pointer to the thread entry function
 *  \param pvArg    Pointer to the Args to be passed to above function
 */
SThread *UT_ThreadCreate(IN char *pTName, IN PFNThreadEntry pfnFunc, IN void *pvArg)
{
	SThread *pstTh;
	int iPriority;
	int iStackSz = UT_DEF_STACK_SZ;
	int iTaskOpts = UT_DEF_TASK_OPT;

	pstTh = (SThread *) UT_AllocMem(sizeof(SThread));
	if (NULL == pstTh)
		goto err0;

	UT_SET_DEF_PRIO(iPriority);

	pstTh->iTaskId = UT_taskSpawn(pTName, iPriority, iTaskOpts, iStackSz,
			(FUNCPTR) pfnFunc, (int)pvArg, 0, 0, 0, 0, 0, 0, 0, 0, 0);
	if (pstTh->iTaskId == ERROR)
	{
		UT_ErrorLog(UT, "UT_ThreadCreate: can't spawn task\n");
		goto err1;
	}

	return pstTh;

err1:
	UT_FreeMem(pstTh);

err0:
	return NULL;
}

/***************************************************************************
 * UT_ThreadDetach: The function does the following things -
 ***************************************************************************/
/*!
 *  This function exists only for compatibility reasons under VxWorks.\n
 *  \return
 *   Always return zero to indicate success.\n
 */
INT UT_ThreadDetach(SThread * pstThread)
{
	return 0;
}

/***************************************************************************
 * UT_ThreadCancel: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function will cancel thread, specify by input parameter thread.\n
 *  \return
 *   Non-zero in case the cancellation failed.\n
 */
INT UT_ThreadCancel(SThread * pstThread)
{
	return UT_taskDelete(pstThread->iTaskId);
}

/***************************************************************************
 * UT_ThreadTestCancel: The function does the following things -
 ***************************************************************************/
/*!
 *  This function exists only for compatibility reasons under VxWorks.\n
 *  \return
 *   None.\n
 */
void UT_ThreadTestCancel()
{
}

/***************************************************************************
 * UT_ThreadSetCancelParams: The function does the following things -
 ***************************************************************************/
/*!
 *  This function exists only for compatibility reasons under VxWorks.\n
 *  \return
 *   Always return zero to indicate success.\n
 */
INT UT_ThreadSetCancelParams(Boolean iEnableCancel, Boolean iEnableAsync)
{
	return 0;
}

/***************************************************************************
 * UT_ThreadJoin: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function will join thread, specify by input parameter pstThread.\n
 *  \return
 *   0 in case of SUCCESS.\n
 *   -1 in case the pthread join failed.\n
 */
INT UT_ThreadJoin(SThread * pstThread)
{
	return 0;
}

/***************************************************************************
 * UT_SemInit: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function initializes the semaphore\n
 *  2.Returns the pointer of newly created semaphore\n
 *  \return
 *   NULL in case semaphore creation failed.\n
 *   Pointer to the semaphore in  case of success.\n
 *
 *  \param uiValue  Value to be passed to initialize the semaphore
 */
SSem *UT_SemInit(IN UINT uiValue)
{
	SSem *pstSem;

	pstSem = (SSem *) UT_AllocMem(sizeof(SSem));
	if (NULL == pstSem)
	{
		return NULL;
	}

	*pstSem = semCCreate(SEM_Q_PRIORITY, uiValue);
	if (NULL == *pstSem)
	{
		UT_FreeMem(pstSem);
		return NULL;
	}

	return pstSem;
}

/***************************************************************************
 * UT_SemWait: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function waits for the specified amount of time with semTake.\n
 *  
 *  \return
 *   The value returned by semTake in VxWorks.\n
 *
 *  \param  pstSem      Pointer to the semaphore to wait upon.
 *  \param  uiTimeout   Time out value. 
 */
INT UT_SemWait(IN SSem * pstSem, IN UINT uiTimeout)
{
	int iVxTimeout;

	if (uiTimeout != INFINITE_TIMEOUT)
	{
		long msec = min(uiTimeout, 2000000);
		unsigned long ms_per_tick = 1000 / UT_sysClkRateGet();
		unsigned long wait_ticks;

		ms_per_tick = ms_per_tick ? ms_per_tick : 1;
		wait_ticks = max(1, (msec / ms_per_tick));

		iVxTimeout = wait_ticks;
	}
	else
	{
		iVxTimeout = WAIT_FOREVER;
	}

	if (UT_semTake(*pstSem, iVxTimeout) == ERROR)
		return -1;

	return 0;
}

/***************************************************************************
 * UT_SemPost: The function does the following things -
 ***************************************************************************/
/*!
 *  1.It releases the semaphore passed to it using semGive.\n
 *
 *  \return
 *   Value returned by semGive.\n
 *  \param  pstSem  Pointer to the semaphore to be posted.
 */
INT UT_SemPost(IN SSem * pstSem)
{
	if (UT_semGive(*pstSem) == ERROR)
		return -1;

	return 0;
}

/***************************************************************************
 * UT_SemDestroy: The function does the following things -
 ***************************************************************************/
/*!
 *  1.It destroys the semaphore passed to it.\n
 *
 *  \return
 *   Value returned by semDelete.\n
 *
 *  \param  pstSem  Pointer to the semaphore   
 */
INT UT_SemDestroy(IN SSem * pstSem)
{
	if (UT_semDelete(*pstSem) == ERROR)
		return -1;

	UT_FreeMem(pstSem);

	return 0;
}

/***************************************************************************
 * UT_MutexInit: The function does the following things -
 ***************************************************************************/
/*!
 *  1.It creates and initilize SMutex.\n
 *
 *  \return
 *   pointer to SMutex.\n
 */
INT UT_MutexInit(SMutex * pstMutex)
{
	*pstMutex = UT_semMCreate(SEM_Q_PRIORITY | SEM_DELETE_SAFE);
	if (*pstMutex == NULL)
		return -1;

	return 0;
}

/***************************************************************************
 * UT_MutexDestroy: The function does the following things -
 ***************************************************************************/
/*!
 *  1.It  destroys the mutex passed to it.\n
 *
 *  \return
 *    void \n
 */
INT UT_MutexDestroy(SMutex * pstMutex)
{
	if (UT_semTake(*pstMutex, WAIT_FOREVER) == ERROR)
		return -1;

	if (UT_semDelete(*pstMutex) == ERROR)
		return -1;

	return 0;
}

/***************************************************************************
 * UT_MutexLock: The function does the following things -
 ***************************************************************************/
/*!
 *  1.It Locks Mutex passed to it \n
 *
 *  \return
 *   return Code. \n
 */
INT UT_MutexLock(SMutex * pstMutex)
{
	if (UT_semTake(*pstMutex, WAIT_FOREVER) == ERROR)
		return -1;

	return 0;
}

/***************************************************************************
 * UT_MutexUnLock: The function does the following things -
 ***************************************************************************/
/*!
 *  1.It UnLocks Mutex passed to it \n
 *
 *  \return
 *   return Code. \n
 */
INT UT_MutexUnLock(SMutex * pstMutex)
{
	if (UT_semGive(*pstMutex) == ERROR)
		return -1;

	return 0;
}

/***************************************************************************
 * UT_InitLog: The function does the following things -
 ***************************************************************************/
/*!
 *  1.Opens a file for logging in write mode.\n
 *  2.If the file could not be opened then log message are displayed on\n
 *    console.\n
 *  \return
 *  SUCCESS  if file is opened in write mode for logging.\n
 *  FAILURE  if couldn't be opened.\n
 */
static STATUS UT_InitLog(char *pLog)
{
	STATUS iStatus = OK;

	if (pLog)
	{
		fpLogout = UT_fopen(pLog, "w");

		if (fpLogout == NULL)
			goto err0;

		if (UT_logFdAdd((int)fpLogout) == ERROR)
			goto err1;

	}

	{
		int i;
		uCurrLogIdx_g = 0;
		for (i = 0; i < MAX_LOG_ENTRIES; i++)
		{
			astrLogs[i][0] = 0;
		}
	}

	return iStatus;

      err1:
	if (pLog)
		UT_fclose(fpLogout);

	fpLogout = NULL;

      err0:
	return FAILURE;
}

/***************************************************************************
 * UT_CloseLog: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function closes the file opened for logging.\n
 *
 *  Assumptions\n
 *  None\n
 *
 *  \return
 *   SUCCESS    if the log file is successfully closed.\n
 *   FAILURE    if the log file could not be opened.\n
 */
STATUS UT_CloseLog()
{
	if (fpLogout)
	{
		UT_logFdDelete((int)fpLogout);

		UT_fclose(fpLogout);
	}

	return SUCCESS;
}

/***************************************************************************
 * UT_VErrorLog:
 ***************************************************************************/
void UT_VErrorLog(IN U8 ucModule, IN char *pstrFormat, va_list ap)
{
	U32 uLogIdx;
	char *pstrLog;

	UT_MutexLock(&logMutex);
	uLogIdx = uCurrLogIdx_g++;
	if (uCurrLogIdx_g >= MAX_LOG_ENTRIES)
		uCurrLogIdx_g = 0;
	UT_MutexUnLock(&logMutex);

	pstrLog = astrLogs[uLogIdx];
	UT_Vsnprintf(pstrLog, BUFFER_SIZE, pstrFormat, ap);
	UT_logMsg("%s", (int)astrLogs[uLogIdx], 0, 0, 0, 0, 0);

	return;
}

INT UT_OSWInit()
{
        int ret;
	SThread *TimerThread;
	int prio;
	
        if ((ret = UT_InitLog(NULL)) == SUCCESS)
        {
                if ((TimerThread = UT_TimerModuleInit(NAME_UT_TIMER)) == NULL)
                {
                	UT_ErrorLog(UT, "UT_TimerModuleInit: failed \n");
			ret = FAILURE;
                }
                else
                {
                	UT_ThreadPriorityAdjust(TimerThread,  prio,  PRIO_UT_TIMER);
         		ret = SUCCESS;
       	}
        }

        return ret;
}

void UT_OSWClose()
{
	UT_TimerShutDown();
	UT_CloseLog();
	UT_MutexDestroy(&logMutex);
}

/***************************************************************************
 * UT_MsgQCreate_FIFO: The function does the following things -
 ***************************************************************************/
/*!
 *  Create a message queue and return the queue ID.
 *
 *  \param  maxMess     Maximum number of messages in the queue
 *  \param  maxLen      Maximum length of a message
 */
MSG_Q_ID UT_MsgQCreate_FIFO(int maxMess, int maxLen)
{
	MSG_Q_ID queuenum;

	queuenum = msgQCreate(maxMess, maxLen, MSG_Q_FIFO);

	if (NULL == queuenum)
		UT_ErrorLog(UT, "Error creating message queue\n");

	return queuenum;
}

/***************************************************************************
 * UT_MsgQDelete: The function does the following things -
 ***************************************************************************/
/*!
 *  Delete a message queue and return the queue ID.
 *
 *  \param  msgQId      Message queue ID
 */
STATUS UT_MsgQDelete(MSG_Q_ID msgQId)
{
	return msgQDelete(msgQId);
}

/***************************************************************************
 * UT_MsgQSend: The function does the following things -
 ***************************************************************************/
/*!
 *  Send a message through a message queue
 *
 *  \param  msgQID      Message queue
 *  \param  buffer      Buffer containing message
 *  \param  nbytes      length of message
 *  \param  timeout     Timeout in ticks
 *  \param  priority    Priority of message
 */
STATUS UT_MsgQSend(MSG_Q_ID msgQId,	/* message queue on which to send */
		   char *buffer,	/* message to send */
		   UINT nBytes,	/* length of message */
		   int timeout,	/* ticks to wait */
		   int priority)	/* MSG_PRI_NORMAL or MSG_PRI_URGENT */
{
	STATUS status;

	status = msgQSend(msgQId, buffer, nBytes, timeout, priority);

	if (status != OK)
		UT_ErrorLog(UT, "Error sending to message queue\n");

	return status;

}

/***************************************************************************
 * UT_MsgQReceive: The function does the following things -
 ***************************************************************************/
/*!
 *  Create a message queue and return the queue ID.
 *
 *  \param   msgQId     Message queue ID
 *  \param   buffer     Buffer to receive message
 *  \param   maxNBytes  Length of buffer
 *  \param   timeout    Ticks to wait
 */
int UT_MsgQReceive(MSG_Q_ID msgQId,	/* message queue from which to receive */
		   char *buffer,	/* buffer to receive message */
		   UINT maxNBytes,	/* length of buffer */
		   int timeout)	/* ticks to wait */
{
	int count;

	count = msgQReceive(msgQId, buffer, maxNBytes, timeout);

	return count;
}

/***************************************************************************
 * UT_GetInterfaceMac: The function does the following things -
 ***************************************************************************/
/*!
 *  Retrieve the Ethernet MAC address for the specified interface
 *
 *  \param   pcEthName     		Network interface name
 *  \param   pcHostMacAddr         Associated Ethernet MAC address formatted in an array of 6 chars
 */
int UT_GetInterfaceMac(IN char *pcEthName, char *pcHostMacAddr)
{
	UT_Log(GTL, INFO, "UT_GetInterfaceMac: ENTRY\n");

	if ((pcEthName != NULL) &&  (pcHostMacAddr != NULL))
	{
#ifdef VXWORKS_6_6
		Ipnet_netif *ifp = ipnet_if_nametonetif(ipnet_user_vr(), pcEthName);
#else
		struct ifnet *ifp = ifunit(pcEthName);
#endif

		if (ifp != NULL)
		{
#ifdef VXWORKS_6_6
			UT_MemCopy(pcHostMacAddr, ifp->ipcom.link_addr, 6);
#else
			UT_MemCopy(pcHostMacAddr, ((struct arpcom *)ifp)->ac_enaddr, 6);
#endif
		}
		else
		{
			UT_ErrorLog(GTL, "UT_GetInterfaceMac: failed to get '%s' info\n", pcEthName);
			return FAILURE;
		}
	}
	else
	{
		UT_ErrorLog(GTL, "UT_GetInterfaceMac: invalid parameters passed\n");
		return FAILURE;
	}

	UT_Log(GTL, INFO, "UT_GetInterfaceMac: EXIT\n");

	return SUCCESS;
}
