/*! \file af_csmencaps.c
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include <vxWorks.h>
#include <ctype.h>
#include <end.h>
#include <taskLib.h>

#if defined(VXWORKS_6_6)
#include <ipnet_h.h>
#include <ipnet_netif.h>
#elif defined(VXWORKS_6_4)
#include <net/if_arp.h>
#else
#include <ifLib.h>
#endif

#include "if_csmencaps.h"
#include "csmencaps.h"

#define SUPVSR_CH	0xffff
#define BOOT_CH		0x0000


#define ENET_HDR_LEN		(sizeof(ENET_HDR))			/* 14 bytes */
#define CSME_HDR_LEN		(sizeof(struct csme_hdr))		/* 6 bytes */
#define API_HDR_BOOT_LEN	(sizeof(struct api_hdr_boot))		/* 6 bytes */
#define API_HDR_LEN		(sizeof(struct api_hdr))		/* 8 bytes */

#define mac_addr_equal(a, b)	(UT_MemCmp(a, b, ETH_ALEN) == 0)

#define csme_target_stats_inc(target, field)			\
	{							\
		UT_semTake((target)->stats.lock, WAIT_FOREVER);	\
		(target)->stats.field++;			\
		UT_semGive((target)->stats.lock);		\
	}

#define get_unaligned_le32(ptr)							\
	(									\
		(U32) (  ((U8*)(ptr))[0]        | (((U8*)(ptr))[1] <<  8) |	\
			(((U8*)(ptr))[2] << 16) | (((U8*)(ptr))[3] << 24) )	\
	)									\


#define print_mac_addr(lev, a)	UT_Log(GTL, (lev), "%02x:%02x:%02x:%02x:%02x:%02x\n",(a)[0],(a)[1],(a)[2],(a)[3],(a)[4],(a)[5])

/* netpool parameters */
#define CSME_CL_OVERHEAD	4
#define CSME_CL_SIZE		1520			/* ROUND_UP(1514) + CSME_CL_OVERHEAD */
#define CSME_CL_NUM		(CSME_MAX_CHANNELS * CSME_MAX_DEVICES * 2)
#define CSME_CBLK_NUM		CSME_CL_NUM
#define CSME_MBLK_NUM		(CSME_CL_NUM * 4)

#define TIMEOUT_MS		200			/* timeout in ms */
#define TIMEOUT_TICKS		((UT_sysClkRateGet() * TIMEOUT_MS) / 1000)
#define RETRIES			(5000 / TIMEOUT_MS)	/* retry for 5 seconds */

#define BOOT_MSGQ_SIZE		16
#define CHANNEL_MSGQ_SIZE	64
#define SUPVSR_MSGQ_SIZE	(CSME_MAX_CHANNELS * 2)
#define TX_MSGQ_SIZE		(CSME_MAX_CHANNELS * CSME_MAX_DEVICES)
#define RX_MSGQ_SIZE		(CSME_MAX_CHANNELS * CSME_MAX_DEVICES)
#define CSME_RX_MSGQ_SIZE	(CSME_MAX_CHANNELS * CSME_MAX_DEVICES)
#define TIMER_MSGQ_SIZE		(CSME_MAX_DEVICES * CSME_MAX_CHANNELS)

static unsigned char bcastaddr[6] = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };


static struct {
	LIST		csme_target_list;
	SEM_ID		csme_lock;

	NET_POOL_ID	pool;
	char		*pool_blocks_area;
	char		*pool_clusters_area;

	LIST		csme_iface_list;

	MSG_Q_ID	rx_queue;

	MSG_Q_ID	tx_queue;
	SThread		*tx_thread;

	MSG_Q_ID	csme_rx_queue;
	SThread		*csme_rx_thread;

	MSG_Q_ID	timer_queue;
	SThread		*timer_thread;
} csmencaps;

static struct csme_target *target_find(unsigned char *scsme_devmac);
static void __target_remove(struct csme_target *target);
static void channel_reset(struct csme_channel *channel);

static void target_change_mac(struct csme_target *target, unsigned char *new_mac)
{
	/* make sure that another target with the same mac doesn't already exist */
	if (target_find(new_mac) == NULL) {
		UT_semTake(target->lock, WAIT_FOREVER);
		/* set the new MAC address for the target  */
		UT_MemCopy(target->macaddr, new_mac, ETH_ALEN);
		UT_semGive(target->lock);
	} else {
		UT_Log(GTL, ERROR, "target_change_mac: target(%#lx) already exists with MAC: ", (unsigned long) target);
		print_mac_addr(ERROR, new_mac);
	}
}

/**
 * check_ready_received - checks if a received broadcast frame could be a READY boot indication
 * sent by Comcerto device. If yes it adds a new target structure in the target list.
 *
 */
static int check_ready_received(M_BLK_ID pMblk)
{
	ENET_HDR *eth_hdr;
	struct csme_target *target;
	struct api_hdr_boot *api_hdr_boot;

	/* check if the frame is big enough to contain at leat the csme & api boot headers */
	if (pMblk->mBlkHdr.mLen < ENET_HDR_LEN + CSME_HDR_LEN + API_HDR_BOOT_LEN) {
		UT_ErrorLog(GTL, "check_ready_received: ERROR: mblk len(%d) too short\n", pMblk->mBlkHdr.mLen);
		goto out;
	}

	api_hdr_boot = (struct api_hdr_boot *) (pMblk->mBlkHdr.mData + ENET_HDR_LEN + CSME_HDR_LEN);

	/* if it is not a READY Command drop the packet */
	if ((api_hdr_boot->cmd_type != CMD_TYPE_READY) ||
		(api_hdr_boot->cmd_class != CMD_CLASS_ETH_BOOT_LDR))
		goto out;

	/* check if this target exists for this particular device */
	eth_hdr = (ENET_HDR *) pMblk->mBlkHdr.mData;

	/* check if this target exists for this particular device */
	target = target_find((unsigned char *) eth_hdr->src);

	/* if not check if a target able to handle any READY packet exists */
	if (!target) {
		target = target_find(bcastaddr);
		/* if we do not have a such target drop the frame */
		if (!target)
			goto out;

		target_change_mac(target, (unsigned char *) eth_hdr->src);
	}

	UT_Log(GTL, INFO, "check_ready_received: ready frame received from:");
	print_mac_addr(INFO, target->macaddr);

	return 0;

out:
	return -1;
}


static void csme_ntoh(M_BLK_ID mblk)
{
	struct csme_hdr *csme_hdr = (struct csme_hdr *) (mblk->mBlkHdr.mData + sizeof(ENET_HDR));
	struct api_hdr *api_hdr;
	U8 *next_marker;

	if (csme_hdr->opcode == UT_Htons(CSME_OPCODE_CONTROL) && csme_hdr->endianess) {
		next_marker =  (U8*)(mblk->mBlkHdr.mData + sizeof(ENET_HDR)+sizeof(struct csme_hdr));
		while ((get_unaligned_le32((U32*)next_marker)) && ((next_marker + API_HDR_LEN) <= ((U8*)(mblk->mBlkHdr.mData + mblk->mBlkHdr.mLen))) ) {
			api_hdr = (struct api_hdr *) (next_marker);
			/* avoid infinite loop if message is corrupted */
			if (api_hdr->length < API_HDR_LEN)
				break;
			next_marker += (api_hdr->length + 3) & ~0x3;
			if (next_marker >= (U8*)(mblk->mBlkHdr.mData + mblk->mBlkHdr.mLen))
				break;
		}
		mblk->mBlkHdr.mLen = (next_marker - (U8*)mblk->mBlkHdr.mData);
	} 
}


static struct csme_iface *csme_iface_new(const char *ifname)
{
	int i, len;
	char *ifindex_start;
	struct csme_iface *iface;

	len = UT_StrLen(ifname);

	if (sizeof(iface->ifname)-1 < len) {
		UT_ErrorLog(GTL, "csme_iface_new: interface name '%s' is too long\n", ifname);
		goto err0;
	}

	/* parse interface name - separate index from base name */
	for (i = len-1; i > 0; i--)
		if (!UT_IsDigit((int)ifname[i]))
			break;
	if ((len < 2) || (i < 1) || (i == len-1 && !UT_IsDigit((int)ifname[i]))) {
		UT_ErrorLog(GTL, "csme_iface_new: invalid interface name\n", ifname);
		goto err0;
	}
	ifindex_start = (char*)ifname + i + 1;

	iface = (struct csme_iface*)UT_calloc(sizeof(struct csme_iface), 1);
	if (!iface) {
		UT_ErrorLog(GTL, "csme_iface_new: can't allocate memory\n");
		goto err0;
	}

	{
#ifdef VXWORKS_6_6
		Ipnet_netif *ifinfo = ipnet_if_nametonetif(ipnet_user_vr(), ifname);
#else
		struct ifnet *ifinfo = UT_ifunit((char*)ifname);
#endif
		if (!ifinfo) {
			UT_ErrorLog(GTL, "csme_iface_new: can't get info on '%s'\n", ifname);
			goto err0;
		}
#ifdef VXWORKS_6_6
		UT_MemCopy(iface->ifaddr, ifinfo->ipcom.link_addr, ETH_ALEN);
#else
		UT_MemCopy(iface->ifaddr, ((struct arpcom *)ifinfo)->ac_enaddr, ETH_ALEN);
#endif
	}

	iface->lock = UT_semMCreate(SEM_Q_PRIORITY | SEM_INVERSION_SAFE);
	if (!iface->lock) {
		UT_ErrorLog(GTL, "csme_iface_new: can't create mutex\n");
		goto err1;
	}

	UT_StrnCpy(iface->ifname, ifname, sizeof(iface->ifname)-1);
	UT_MemCopy(iface->ifbasename, ifname, ifindex_start-ifname);
	iface->ifbasename[ifindex_start-ifname+1] = 0;
	iface->ifindex = UT_atoi(ifindex_start);

	return iface;

err1:
	UT_free(iface);

err0:
	return NULL;
}


static void csme_iface_free(struct csme_iface *iface)
{
	UT_semDelete(iface->lock);
	UT_free(iface);
}


static struct csme_iface *csme_iface_find(const char *ifname)
{
	struct csme_iface *iface;

	UT_semTake(csmencaps.csme_lock, WAIT_FOREVER);

	iface = (struct csme_iface *) UT_lstFirst(&csmencaps.csme_iface_list);

	while (iface != NULL) {
		if (UT_StrCmp(ifname, iface->ifname) == 0)
			goto match;
		iface = (struct csme_iface *) UT_lstNext((NODE *) iface);
	}

	UT_semGive(csmencaps.csme_lock);

	return NULL;

match:
	UT_semGive(csmencaps.csme_lock);

	return iface;
}


/**
 * __csme_iface_add - add interface structure to the list
 *
 * You must call this function holding both the csmencaps.csme_lock and iface->lock locks
 *
 */
static struct csme_iface *__csme_iface_add(const char *ifname)
{
	struct csme_iface *iface = csme_iface_new(ifname);

	if (iface != NULL)
		UT_lstAdd(&csmencaps.csme_iface_list, (NODE *) iface);

	return iface;
}

/**
 * __csme_iface_remove - remove the given interface from list, free all targets on this interface
 *
 * You must call this function holding both the csmencaps.csme_lock and iface->lock locks
 *
 */
static void __csme_iface_remove(struct csme_iface *iface)
{
	struct csme_target *target, *next;
	
	/* free up all the target structure listed */
	target = (struct csme_target *) UT_lstFirst(&csmencaps.csme_target_list);
	while (target) {
		next = (struct csme_target *) UT_lstNext((NODE *) target);

		if (target->iface == iface)
			__target_remove(target);

		target = next;
	}

	UT_lstDelete(&csmencaps.csme_iface_list, (NODE *) iface);

	csme_iface_free(iface);
}


static void csme_mblk_queue_purge(MSG_Q_ID queue)
{
	M_BLK_ID mblk;

	while (UT_msgQReceive(queue, (char*)&mblk, sizeof(unsigned long), NO_WAIT) == sizeof(unsigned long))
		UT_netMblkClChainFree(mblk);
}


static void csme_mblk_queue_free(MSG_Q_ID queue)
{
	csme_mblk_queue_purge(queue);
	UT_msgQDelete(queue);
}

/**
 * dump_csme_hdr -
 *
 */
void dump_csme_hdr(struct csme_hdr *hdr)
{
	UT_Log(GTL, INFO, "op code:    %#x\n", UT_Ntohs(hdr->opcode));
	UT_Log(GTL, INFO, "seq number: %#x\n", hdr->seq_number);
	UT_Log(GTL, INFO, "endianess:  %#x\n", hdr->endianess);
	UT_Log(GTL, INFO, "c/r:        %#x\n", hdr->cr);
	UT_Log(GTL, INFO, "ack sup:    %#x\n", hdr->ack_sup);
	UT_Log(GTL, INFO, "reserved:   %#x\n", hdr->reserved);
	UT_Log(GTL, INFO, "ch number:  %#x\n", UT_Ntohs(hdr->channel_nb));
}

/**
 * dump_api_hdr -
 *
 */
void dump_api_hdr(struct api_hdr *hdr)
{
	UT_Log(GTL, INFO, "index:    %#x\n", hdr->index);
	UT_Log(GTL, INFO, "length:   %#x\n", hdr->length);
	UT_Log(GTL, INFO, "cmdclass: %#x\n", hdr->cmd_class);
	UT_Log(GTL, INFO, "cmdtype:  %#x\n", hdr->cmd_type);
	UT_Log(GTL, INFO, "funccode: %#x\n", hdr->func_code);
}


static void csmencaps_print_target_stats(struct csme_target *target, int clear_stats)
{
	UT_semTake(target->stats.lock, WAIT_FOREVER);

	if (clear_stats <= 1) {
		UT_Log(GTL, ALWYS, "target %p, MAC %02x.%02x.%02x.%02x.%02x.%02x\n",
			target,
			target->macaddr[0], target->macaddr[1], target->macaddr[2],
			target->macaddr[3], target->macaddr[4], target->macaddr[5]);
	
		UT_Log(GTL, ALWYS, "\ttx_msg:\t\t\t\t%u\n", target->stats.tx_msg);
		UT_Log(GTL, ALWYS, "\ttx_msg_total_retransmitted:\t%u\n", target->stats.tx_msg_total_retransmitted);
		UT_Log(GTL, ALWYS, "\ttx_msg_max_retransmissions:\t%u\n", target->stats.tx_msg_max_retransmissions);
		UT_Log(GTL, ALWYS, "\ttx_msg_retransmitted:\t\t%u\n", target->stats.tx_msg_retransmitted);
		UT_Log(GTL, ALWYS, "\ttx_msg_err:\t\t\t%u\n", target->stats.tx_msg_err);

		UT_Log(GTL, ALWYS, "\trx_ack:\t\t\t\t%u\n", target->stats.rx_ack);
		UT_Log(GTL, ALWYS, "\trx_ack_wrongseq:\t\t%u\n", target->stats.rx_ack_wrongseq);
		UT_Log(GTL, ALWYS, "\trx_ack_err:\t\t\t%u\n", target->stats.rx_ack_err);

		UT_Log(GTL, ALWYS, "\trx_msg:\t\t\t\t%u\n", target->stats.rx_msg);
		UT_Log(GTL, ALWYS, "\trx_msg_repeated:\t\t%u\n", target->stats.rx_msg_repeated);
		UT_Log(GTL, ALWYS, "\trx_msg_err:\t\t\t%u\n", target->stats.rx_msg_err);

		UT_Log(GTL, ALWYS, "\ttx_ack:\t\t\t\t%u\n", target->stats.tx_ack);
		UT_Log(GTL, ALWYS, "\ttx_ack_err:\t\t\t%u\n\n", target->stats.tx_ack_err);
	}

	if (clear_stats)
		UT_MemSet(&target->stats, 0, sizeof(target->stats));

	UT_semGive(target->stats.lock);
}

/**
 * csmencaps_print_target_mac_stats - prints statistics for target with given MAC
 *
 * ARGUMENT:
 *    mac:		pointer to the MAC address of target
 *
 *    clear_stats:	flag to control the stats cleaning-up, pass != 0 value
 *			to clear the stats and > 1 to supress output
 *
 *
 * RETURN:
 *    non-zero value if NULL passed or target isn't found
 */
int csmencaps_print_target_mac_stats(unsigned char *mac_addr, int clear_stats)
{
	struct csme_target *target;

	if (!mac_addr)
		return -1;

	target = target_find(mac_addr);
	if (!target) {
		UT_Log(GTL, ALWYS, "target is not found\n");
		return -1;
	}

	csmencaps_print_target_stats(target, clear_stats);

	return 0;
}

/**
 * csmencaps_print_stats - prints statistics for all targets
 *
 * ARGUMENT:
 *    clear_stats:	flag to control the stats cleaning-up, pass != 0 value
 *			to clear the stats and > 1 to supress output
 */
void csmencaps_print_stats(int clear_stats)
{
	struct csme_target *target;

	UT_semTake(csmencaps.csme_lock, WAIT_FOREVER);

	target = (struct csme_target *) UT_lstFirst(&csmencaps.csme_target_list);
	while (target != NULL) {
		csmencaps_print_target_stats(target, clear_stats);
		target = (struct csme_target *) UT_lstNext((NODE *) target);
	}

	UT_semGive(csmencaps.csme_lock);
}


static void timer_timeout(int arg)
{
	struct csme_channel *channel = (struct csme_channel *) arg;

	UT_msgQSend(csmencaps.timer_queue, (char *) &channel, sizeof(unsigned long), NO_WAIT, MSG_PRI_NORMAL);
}

/**
 * channel_add - allocate a new channel and add channel to the target channel table
 *
 * Must be called from user context only
 *
 */
static struct csme_channel *channel_add(struct csme_target *target, unsigned short channelid)
{
	struct csme_channel *channel;

	if (channelid >= CSME_MAX_CHANNELS) {
		UT_ErrorLog(GTL, "csme channel_add: ch(%u, %p) out of range\n", channelid, target);
		goto err0;
	}
	
	channel = (struct csme_channel*)UT_calloc(sizeof(struct csme_channel), 1);
	if (!channel) {
		UT_ErrorLog(GTL, "csme channel_add: malloc() failed ch(%u, %p)\n", channelid, target);
		goto err0;
	}
	
	channel->id = channelid;

	UT_MemSet(channel->tx_seq_number, 0, sizeof(channel->tx_seq_number));
	UT_MemSet(channel->last_rx_seq_number, 0xff, sizeof(channel->last_rx_seq_number));

	channel->tx_queue = UT_msgQCreate(CHANNEL_MSGQ_SIZE, sizeof(unsigned long), MSG_Q_FIFO);
	if (!channel->tx_queue) {
		UT_ErrorLog(GTL, "csme channel_add: can't create TX queue, ch(%u, %p)\n", channelid, target);
		goto err1;
	}

	channel->timer = UT_wdCreate();
	if (!channel->timer) {
		UT_ErrorLog(GTL, "csme channel_add: can't create watchdog, ch(%u, %p)\n", channelid, target);
		goto err2;
	}

	channel->target = target;

	channel->lock = UT_semMCreate(SEM_Q_PRIORITY | SEM_INVERSION_SAFE);
	if (!channel->lock) {
		UT_ErrorLog(GTL, "csme channel_add: can't create mutex, ch(%u, %p)\n", channelid, target);
		goto err3;
	}

	UT_semTake(target->lock, WAIT_FOREVER);

	target->channel[channel->id] = channel;

	UT_semGive(target->lock);

	return channel;

err3:
	UT_wdDelete(channel->timer);

err2:
	UT_msgQDelete(channel->tx_queue);

err1:
	UT_free(channel);

err0:
	return NULL;
}

/**
 * channel_reset - reset the resources occupied by a channel
 *
 */
static void __channel_reset(struct csme_channel *channel)
{
	UT_wdCancel(channel->timer);

	csme_mblk_queue_purge(channel->tx_queue);

	UT_MemSet(channel->tx_seq_number, 0, sizeof(channel->tx_seq_number));
	UT_MemSet(channel->last_rx_seq_number, 0xff, sizeof(channel->last_rx_seq_number));
}

static void channel_reset(struct csme_channel *channel)
{
	UT_semTake(channel->lock, WAIT_FOREVER);

	__channel_reset(channel);
	
	UT_semGive(channel->lock);
}

/**
 * channel_remove - remove channel from table and free it
 *
 * This function must be called holding the target->lock for writing
 *
 */
static void channel_remove(struct csme_target *target, struct csme_channel *channel)
{
	UT_semTake(channel->lock, WAIT_FOREVER);
	
	__channel_reset(channel);

	target->channel[channel->id] = NULL;

	UT_wdDelete(channel->timer);

	UT_msgQDelete(channel->tx_queue);	/* all blocks were freed in __channel_reset() */

	UT_semDelete(channel->lock);
	
	UT_free(channel);
}

/**
 * channel_find - find a channel inside a target's channel list
 *
 */
static struct csme_channel *channel_find(struct csme_target *target, unsigned short channelid)
{
	struct csme_channel *channel = NULL;

	if (channelid >= CSME_MAX_CHANNELS) {
		UT_ErrorLog(GTL, "channel_find: ch(%u, %p) out of range\n", channelid, target);
		goto out;
	}

	UT_semTake(target->lock, WAIT_FOREVER);

	channel = target->channel[channelid];

	UT_semGive(target->lock);

out:
	return channel;
}

/**
 * target_add - allocate new target and add it to the target list
 *
 * Must be called from user context only
 *
 */
static struct csme_target *target_add(unsigned char *scsme_devmac, const char *ifname, void *scsme_user)
{
	struct csme_iface *iface;
	struct csme_target *target;

	iface = csme_iface_find(ifname);
	if (!iface) {
		UT_ErrorLog(GTL, "target_add: can't find interface '%s'\n", ifname);
		goto err;
	}

	target = (struct csme_target*)UT_calloc(sizeof(struct csme_target), 1);
	if (!target) {
		UT_ErrorLog(GTL, "csme: malloc() failed for target\n");
		goto err;
	}

	target->lock = UT_semMCreate(SEM_Q_PRIORITY | SEM_INVERSION_SAFE);
	target->stats.lock = UT_semMCreate(SEM_Q_PRIORITY | SEM_INVERSION_SAFE);

	target->supvsr_channel.target = target;
	target->supvsr_channel.id = SUPVSR_CH;

	UT_MemSet(target->supvsr_channel.tx_seq_number, 0, sizeof(target->supvsr_channel.tx_seq_number));
	UT_MemSet(target->supvsr_channel.last_rx_seq_number, 0xff, sizeof(target->supvsr_channel.last_rx_seq_number));

	target->supvsr_channel.tx_queue = UT_msgQCreate(SUPVSR_MSGQ_SIZE, sizeof(unsigned long), MSG_Q_FIFO);

	target->supvsr_channel.timer = UT_wdCreate();

	target->supvsr_channel.lock = UT_semMCreate(SEM_Q_PRIORITY | SEM_INVERSION_SAFE);


	target->boot_channel.target = target;
	target->boot_channel.id = BOOT_CH;

	UT_MemSet(target->boot_channel.tx_seq_number, 0, sizeof(target->boot_channel.tx_seq_number));
	UT_MemSet(target->boot_channel.last_rx_seq_number, 0xff, sizeof(target->boot_channel.last_rx_seq_number));

	target->boot_channel.tx_queue = UT_msgQCreate(BOOT_MSGQ_SIZE, sizeof(unsigned long), MSG_Q_FIFO);

	target->boot_channel.timer = UT_wdCreate();

	target->boot_channel.lock = UT_semMCreate(SEM_Q_PRIORITY | SEM_INVERSION_SAFE);

	UT_MemCopy(target->macaddr, scsme_devmac, ETH_ALEN);
	target->iface = iface;
	target->scsme_user = scsme_user;

	UT_semTake(csmencaps.csme_lock, WAIT_FOREVER);
	UT_lstAdd(&csmencaps.csme_target_list, (NODE *) target);
	UT_semGive(csmencaps.csme_lock);

	return target;

err:
	return NULL;
}

/**
 * __target_remove - remove target from list and free it
 *
 * You must call this function holding the csmencaps.csme_lock
 *
 */
static void __target_remove(struct csme_target *target)
{
	struct csme_channel *channel;
	int i;

	UT_semTake(target->stats.lock, WAIT_FOREVER);
	UT_semTake(target->lock, WAIT_FOREVER);

	/* free up any initialized channel structure */
	for (i = 0; i < CSME_MAX_CHANNELS; i++) {
		channel = target->channel[i];
		if (channel)
			channel_remove(target, channel);
	}
	
	UT_semDelete(target->stats.lock);
	
	channel_reset(&target->boot_channel);
	UT_semTake(target->boot_channel.lock, WAIT_FOREVER);
	UT_semDelete(target->boot_channel.lock);

	UT_wdDelete(target->boot_channel.timer);
	UT_msgQDelete(target->boot_channel.tx_queue);

	channel_reset(&target->supvsr_channel);
	UT_semTake(target->supvsr_channel.lock, WAIT_FOREVER);
	UT_semDelete(target->supvsr_channel.lock);

	UT_wdDelete(target->supvsr_channel.timer);
	UT_msgQDelete(target->supvsr_channel.tx_queue);

	UT_lstDelete(&csmencaps.csme_target_list, (NODE *) target);

	UT_semDelete(target->lock);

	UT_free(target);
}

/**
 * target_find - find a target in the target list based on dest MAC address
 *
 */
static struct csme_target *target_find(unsigned char *scsme_devmac)
{
	struct csme_target *target;

	UT_semTake(csmencaps.csme_lock, WAIT_FOREVER);

	target = (struct csme_target *) UT_lstFirst(&csmencaps.csme_target_list);

	while (target != NULL) {
		if (mac_addr_equal(target->macaddr, scsme_devmac))
			goto match;
		target = (struct csme_target *) UT_lstNext((NODE *) target);
	}

	UT_semGive(csmencaps.csme_lock);

	return NULL;

match:
	UT_semGive(csmencaps.csme_lock);

	return target;
}

/**
 * target_reset -
 *
 */
static void target_reset(struct csme_target *target)
{
	int i;

	UT_semTake(target->lock, WAIT_FOREVER);
	
	channel_reset(&target->boot_channel);
	channel_reset(&target->supvsr_channel);

	for (i = 0; i < CSME_MAX_CHANNELS; i++)
		if (target->channel[i])
			channel_reset(target->channel[i]);

	UT_semGive(target->lock);
}

/**
 * csme_tx_msg_finish -
 *
 */
static int csme_tx_msg_finish(struct csme_iface *iface, M_BLK_ID mblk)
{
	STATUS status;
	
	if (iface->mux && !iface->down) {
		status = UT_muxSend(iface->mux, mblk);
		if (END_ERR_BLOCK == status) {
			UT_ErrorLog(GTL, "csme_tx_msg_finish: can't send mblk, muxSend() failed with status END_ERR_BLOCK\n");
			UT_netMblkClChainFree(mblk);
		}
		
		return status;
	}

	UT_ErrorLog(GTL, "csme_tx_msg_finish: can't send mblk, iface %s is down or not registered\n", iface->ifname);
	return -1;
}

/**
 * csme_tx_ack -
 *
 */
static int csme_tx_ack(struct csme_target *target, M_BLK_ID pMblk)
{
	ENET_HDR *neth_hdr;
	struct csme_hdr *csme_hdr = (struct csme_hdr *) (pMblk->mBlkHdr.mData + sizeof(ENET_HDR));
	struct csme_hdr *ncsme_hdr;

	csme_target_stats_inc(target, tx_ack);

	pMblk = UT_netTupleGet(csmencaps.pool,
			    sizeof(ENET_HDR) + sizeof(struct csme_hdr),
			    M_DONTWAIT, MT_DATA, FALSE);
	if (pMblk == NULL) {
		UT_ErrorLog(GTL, "csme_tx_ack: can't allocate mblk\n");
		goto err;
	}

	/* fill the pMblk struct */	
	pMblk->mBlkHdr.mLen = sizeof(ENET_HDR) + sizeof(struct csme_hdr);
	pMblk->mBlkHdr.mFlags |= M_PKTHDR;

	ncsme_hdr = (struct csme_hdr *) (pMblk->mBlkHdr.mData + sizeof(ENET_HDR));
	ncsme_hdr->channel_nb = csme_hdr->channel_nb;
	ncsme_hdr->seq_number = csme_hdr->seq_number;
	ncsme_hdr->ack_sup = 1;
	ncsme_hdr->cr = 1;
	ncsme_hdr->reserved = 0;
	ncsme_hdr->endianess = csme_hdr->endianess;
	ncsme_hdr->opcode = csme_hdr->opcode;

	/* add ethernet header */
 	neth_hdr = (ENET_HDR *) pMblk->mBlkHdr.mData;
	UT_MemCopy(neth_hdr->dst, target->macaddr, ETH_ALEN);
	UT_MemCopy(neth_hdr->src, target->iface->ifaddr, ETH_ALEN);
	neth_hdr->type = UT_Htons(ETH_P_CSME);

	return csme_tx_msg_finish(target->iface, pMblk);

err:
	return -1;
}

/**
 * csme_tx_msg_next -
 *
 */
static void csme_tx_msg_next(struct csme_channel *channel)
{
	M_BLK_ID mblk;
	struct csme_target *target = channel->target;
	
	/* process the next command in the queue */
	/* start timer if required */

	UT_semTake(channel->lock, WAIT_FOREVER);

	/* check if queue is empty */
	if (UT_msgQReceive(channel->tx_queue, (char *) &mblk, sizeof(unsigned long), NO_WAIT) != sizeof(unsigned long)) {
		UT_semGive(channel->lock);
		goto out;
	}

	if (!((struct csme_hdr *)(mblk->mBlkHdr.mData + sizeof(ENET_HDR)))->ack_sup) {
		M_BLK_ID mblk_copy;

		/* If packet wants an ACK - put it back to the queue since we must
		 * retransmit on timeout. We know the queue has a room for the packet.
		 */
		UT_msgQSend(channel->tx_queue, (char *) &mblk, sizeof(unsigned long), NO_WAIT, MSG_PRI_URGENT);

		/* start timer, it must be started before the netMblkChainDup call */
		channel->retries = RETRIES;
		UT_wdStart(channel->timer, TIMEOUT_TICKS, (FUNCPTR) timer_timeout, (int) channel);

		/* duplicate chain and pass it to MUX via csme_tx_msg_finish below */
		mblk_copy = UT_netMblkChainDup(mblk->pClBlk->pNetPool, mblk, 0, M_COPYALL, M_DONTWAIT);

		UT_semGive(channel->lock);

		if (mblk_copy == NULL) {
			/* If we failed to do this, just go out - original message left
			 * in queue and timeout handler will kick the TX process soon.
			 */
			UT_ErrorLog(GTL, "csme_tx_msg_next: failed to allocate net pool resources\n");
			goto out;
		}

		mblk = mblk_copy;	/* don't forget to swap values, otherwise we have both leak and incorrect free */
	} else {
		UT_semGive(channel->lock);
	}

	/* now send the command/message to the MUX */
	if (csme_tx_msg_finish(target->iface, mblk) < 0) {
		UT_ErrorLog(GTL, "csme_tx_msg_next: csme_tx_msg_finish() failed ch(%u, %p)\n", channel->id, target);
	}

out:
	return;
}

/**
 * handle_channel_timeout -
 *
 */
static void handle_channel_timeout(struct csme_channel *channel)
{
	struct csme_target *target = channel->target;
	M_BLK_ID mblk, mblk_copy;

	UT_semTake(channel->lock, WAIT_FOREVER);

	/* check if queue is empty */
	if (UT_msgQReceive(channel->tx_queue, (char *) &mblk, sizeof(unsigned long), NO_WAIT) != sizeof(unsigned long)) {
		UT_semGive(channel->lock);
		goto out;
	}

	/* if we couldn't process message after all retries, drop it and try to continue */
	if (channel->retries == 0) {
		UT_semGive(channel->lock);

		UT_ErrorLog(GTL, "handle_channel_timeout: ch(%u, %p) final retransmission\n", channel->id, target);

		csme_target_stats_inc(target, tx_msg_max_retransmissions);

		UT_netMblkClChainFree(mblk);

		csme_tx_msg_next(channel);

		goto out;
	}

	if (channel->retries == RETRIES)
		csme_target_stats_inc(target, tx_msg_retransmitted);

	csme_target_stats_inc(target, tx_msg_total_retransmitted);
	
	channel->retries--;
	UT_wdStart(channel->timer, TIMEOUT_TICKS, (FUNCPTR) timer_timeout, (int) channel);

	/* put the message back to queue */
	UT_msgQSend(channel->tx_queue, (char *) &mblk, sizeof(unsigned long), NO_WAIT, MSG_PRI_URGENT);

	UT_semGive(channel->lock);

	mblk_copy = UT_netMblkChainDup(mblk->pClBlk->pNetPool, mblk, 0, M_COPYALL, M_DONTWAIT);
	if (mblk_copy == NULL) {
		/* we failed to reference chain, inform about this and go out */
		UT_ErrorLog(GTL, "handle_channel_timeout: ch(%u, %p) failed to allocate net pool resources\n", channel->id, target);
		goto out;
	}

	csme_tx_msg_finish(target->iface, mblk_copy);

out:
	return;
}

/**
 * csme_rx_msg -
 *
 */
static int csme_rx_msg(M_BLK_ID pMblk, struct csme_target *target, struct csme_channel *channel)
{
	struct csme_hdr *hdr = (struct csme_hdr *) (pMblk->mBlkHdr.mData + sizeof(ENET_HDR));
	U16 opcode = UT_Ntohs(hdr->opcode);

	csme_target_stats_inc(target, rx_msg);

	UT_semTake(channel->lock, WAIT_FOREVER);

	if (hdr->seq_number == channel->last_rx_seq_number[opcode]) {
		/* repeated command */
		UT_semGive(channel->lock);

		csme_target_stats_inc(target, rx_msg_repeated);

		/* transmit ack if requested but don't forward command to upper layer */
		if (!hdr->ack_sup)
			csme_tx_ack(target, pMblk);

		goto err;
	}

	/* save the last command sequence number */
	channel->last_rx_seq_number[opcode] = hdr->seq_number;

	UT_semGive(channel->lock);

	/* it is new message/response */
	if ((hdr->channel_nb == UT_Htons(SUPVSR_CH)) && (opcode == CSME_OPCODE_CONTROL)) {
		struct api_hdr *api_hdr;

		/* track channel/participant create to reset csme sequence number */

		api_hdr = (struct api_hdr *)(pMblk->mBlkHdr.mData + sizeof(ENET_HDR) + sizeof(struct csme_hdr));

		if ((CMD_CLASS_CONFIGURATION_DEVICE == api_hdr->cmd_class)
		    && (CMD_TYPE_CONFIGURATION_RESPONSE == api_hdr->cmd_type)
		    && ((UT_CPU2LE16(FCODE_SUPV_CREATE_CHANNEL) == api_hdr->func_code) || (UT_CPU2LE16(FCODE_CONF_CREATE_PARTICIPANT) == api_hdr->func_code))
		    && !((U16 *)api_hdr)[4]) {
			channel = channel_find(target, UT_LE2CPU16(((U16 *)api_hdr)[5]));
			if (channel)
				channel_reset(channel);
		}
	}

	/* transmit ack if requested */
	if (!hdr->ack_sup)
		csme_tx_ack(target, pMblk);

	return TRUE;

err:
	return FALSE;
}

/**
 * csme_rx_ack -
 *
 * ACK received, now handle it
 *
 */
static int csme_rx_ack(M_BLK_ID pMblk, struct csme_target *target, struct csme_channel *channel)
{
	struct csme_hdr *ack_hdr = (struct csme_hdr *) (pMblk->mBlkHdr.mData + ENET_HDR_LEN);
	struct csme_hdr *hdr;

	csme_target_stats_inc(target, rx_ack);

	UT_semTake(channel->lock, WAIT_FOREVER);

	if (UT_msgQReceive(channel->tx_queue, (char *) &pMblk, sizeof(unsigned long), NO_WAIT) != sizeof(unsigned long)) {
		csme_target_stats_inc(target, rx_ack_wrongseq);
		goto err0;
	}

	hdr = (struct csme_hdr *) (pMblk->mBlkHdr.mData + ENET_HDR_LEN);

	if (hdr->opcode != ack_hdr->opcode) {
		csme_target_stats_inc(target, rx_ack_err);
		goto err1;
	}

	if (hdr->seq_number != ack_hdr->seq_number) {
		csme_target_stats_inc(target, rx_ack_wrongseq);
		goto err1;
	}

	/* stop the timer */
	UT_wdCancel(channel->timer);

	UT_semGive(channel->lock);

	UT_netMblkClChainFree(pMblk);

	csme_tx_msg_next(channel);

	return TRUE;

err1:
	UT_msgQSend(channel->tx_queue, (char *) &pMblk, sizeof(unsigned long), NO_WAIT, MSG_PRI_URGENT);

err0:
	UT_semGive(channel->lock);

	return FALSE;
}

/**
 * csme_rx -
 *
 */
static int csme_rx(M_BLK_ID pMblk)
{
	ENET_HDR *eth_hdr;
	struct csme_hdr *csme_hdr;
	struct csme_target *target;
	struct csme_channel *channel;
	int err;

	eth_hdr = (ENET_HDR *) (pMblk->mBlkHdr.mData);
	target = target_find((unsigned char *) eth_hdr->src);
	if (!target) {
		UT_ErrorLog(GTL, "csme_rx: target_find() failed\n");
		goto drop;
	}

	csme_hdr = (struct csme_hdr *) (pMblk->mBlkHdr.mData + sizeof(ENET_HDR));

	if (!csme_hdr->endianess) {
		if (csme_hdr->channel_nb != UT_Htons(BOOT_CH)) {
			UT_ErrorLog(GTL, "csme_rx: boot message in ch(%u, %p)\n", UT_Ntohs(csme_hdr->channel_nb), target);
			goto drop;
		}
		channel = &target->boot_channel;		
	} else {
		if (csme_hdr->channel_nb == UT_Htons(SUPVSR_CH)) {
			channel = &target->supvsr_channel;
		} else {
			channel = channel_find(target, UT_Ntohs(csme_hdr->channel_nb));
			if (!channel) {
				UT_ErrorLog(GTL, "csme_rx: channel_find() failed, ch(%u, %p)\n", UT_Ntohs(csme_hdr->channel_nb), target);
				goto drop;
			}
		}
	}

	if (csme_hdr->cr) {
		/* it is an ACK packet, handle it and drop it */
		if (csme_rx_ack(pMblk, target, channel) == FALSE) {
			goto drop;
		}
		/*
		 * for bootload message there is no separate ACK. CMD_ACK acts as both low level
		 * ACK and response. So it must be handed over to user application. On the other
		 * hand for non-bootloader messages, there is a separate low level ACK which
		 * should not be sent to application layer.
		 */
		if (csme_hdr->endianess) {
			/*
			 * Non bootload message - so don't give it to user
			 */
			goto drop;
		}

	} else {
		err = 0;
		switch (UT_Ntohs(csme_hdr->opcode)) {
		case CSME_OPCODE_CONTROL:
		case CSME_OPCODE_RESERVED:
		case CSME_OPCODE_UNIFIED_DIAGNOSTICS:
		case CSME_OPCODE_REMOTE_MEDIA:
			if (csme_rx_msg(pMblk, target, channel) == FALSE)
				goto drop;
			
			break;

		case CSME_OPCODE_NOOP:
			goto drop;
			break;

		default:
			/* FIXME give response for an unknown opcode */
			goto drop;
			break;
		}
	}

	return TRUE;

drop:
	/* packet is dropped if it should not be queued for GTL layer */
	return FALSE;
}

/**
 * csme_tx - handles csme message transmission
 *
 * Must be called with pMblk filled with a message
 *
 */
static int csme_tx(M_BLK_ID pMblk)
{
	struct sockaddr_csme *saddr;
	struct api_hdr_boot *api_hdr_boot;
	struct csme_hdr *csme_hdr;
	struct csme_target *target;
	struct csme_channel *channel;
	unsigned char *maas_mac = NULL;
	int err = 0;
	ENET_HDR *eth_hdr;
	U16 opcode;

	saddr = (struct sockaddr_csme*)(pMblk->mBlkHdr.mData + ROUND_UP(pMblk->mBlkHdr.mLen + 4, 4));
	opcode = UT_Ntohs(saddr->scsme_opcode);

	switch (opcode) {
	case CSME_OPCODE_CONTROL:
		/* make sure we can read cmd_class and cmd_type */
		if (pMblk->mBlkHdr.mLen < ENET_HDR_LEN + CSME_HDR_LEN + API_HDR_BOOT_LEN) {
			UT_ErrorLog(GTL, "csme_tx: ERROR: pMblk len(%d) too short\n", pMblk->mBlkHdr.mLen);
			err = ERROR;
			goto err_free;
		}

		api_hdr_boot = (struct api_hdr_boot *) (pMblk->mBlkHdr.mData + ENET_HDR_LEN + CSME_HDR_LEN);

		if (!(saddr->scsme_flags & 0x1) && (api_hdr_boot->cmd_class == CMD_CLASS_ETH_BOOT_LDR)
		    && (api_hdr_boot->cmd_type == CMD_TYPE_MASS_ASSIGN)) {
			if (mac_addr_equal(saddr->scsme_devmac, bcastaddr)) {
				if (pMblk->mBlkHdr.mLen < ENET_HDR_LEN + CSME_HDR_LEN + API_HDR_BOOT_LEN + ETH_ALEN*2) {
					UT_ErrorLog(GTL, "csme_tx: ERROR: pMblk len(%d) too short for the MAAS_ASSIGN\n", pMblk->mBlkHdr.mLen);
					err = ERROR;
					goto err_free;
				}

				/* get a pointer to assigned target MAC from packet */
				maas_mac = (unsigned char*)(pMblk->mBlkHdr.mData + ENET_HDR_LEN + API_HDR_BOOT_LEN + CSME_HDR_LEN);

				/* OK now change the mac of the target with the mac to be assigned
				 * to be able to receive the ack
				 */
				target = target_find(bcastaddr);
				if (target) {
					target_change_mac(target, maas_mac);
				}
			} else {
				UT_MemCopy(maas_mac, pMblk->mBlkHdr.mData + 6, ETH_ALEN);
				/* OK now change the mac of the target with the mac to be assigned
				 * to be able to receive the ack
				 */
				target = target_find(saddr->scsme_devmac);
				if (target)
					target_change_mac(target, maas_mac);
			}

			target = target_find(maas_mac);
			if (!target) {
				target = target_add(maas_mac, saddr->scsme_ifname, saddr->scsme_user);
				if (!target) {
					UT_ErrorLog(GTL, "csme_tx: target_add() failed\n");
					err = ERROR;
					goto err_free;
				}
			} else {
				target_reset(target);
			}
		} else {
			target = target_find(saddr->scsme_devmac);

			if (!target) {
				target = target_add(saddr->scsme_devmac, saddr->scsme_ifname, saddr->scsme_user);
				if (!target) {
					UT_ErrorLog(GTL, "csme_tx: target_add() failed\n");
					err = ERROR;
					goto err_free;
				}
			}
		}

		csme_target_stats_inc(target, tx_msg);

		if (!(saddr->scsme_flags & 0x1)) {
			channel = &target->boot_channel;
			if (saddr->scsme_channelid != UT_Htons(BOOT_CH)) {
				UT_ErrorLog(GTL, "csme_tx: invalid ch(%u, %p) for boot message\n", UT_Ntohs(saddr->scsme_channelid), target);
				csme_target_stats_inc(target, tx_msg_err);
				err = ERROR;
				goto err_free;
			}
		} else if (saddr->scsme_channelid == UT_Htons(SUPVSR_CH)) {
			channel = &target->supvsr_channel;
		} else {
			channel = channel_find(target, UT_Ntohs(saddr->scsme_channelid));
			if (!channel) {
				channel = channel_add(target, UT_Ntohs(saddr->scsme_channelid));
				if (!channel) {
					UT_ErrorLog(GTL, "csme_tx: channel_add(%u, %p) failed\n", UT_Ntohs(saddr->scsme_channelid), target);
					csme_target_stats_inc(target, tx_msg_err);
					err = ERROR;
					goto err_free;
				}
			}
		}

		/* add message terminator (4 bytes set to 0) */
		UT_MemSet (pMblk->mBlkHdr.mData + pMblk->mBlkHdr.mLen, 0, 4);
		pMblk->mBlkHdr.mLen += 4;
		pMblk->pClBlk->clSize += 4;
		break;

	case CSME_OPCODE_NOOP:
	case CSME_OPCODE_RESERVED:
	case CSME_OPCODE_UNIFIED_DIAGNOSTICS:
	case CSME_OPCODE_REMOTE_MEDIA:
		target = target_find(saddr->scsme_devmac);	/* find dest target matching with saddr->scsme_devmac */

		if (!target) {
			target = target_add(saddr->scsme_devmac, saddr->scsme_ifname, saddr->scsme_user);
			if (!target) {
				UT_ErrorLog(GTL, "csme_tx: target_add() failed\n");
				err = ERROR;
				goto err_free;
			}
		}

		csme_target_stats_inc(target, tx_msg);

		if (saddr->scsme_channelid == UT_Htons(SUPVSR_CH)) {
			channel = &target->supvsr_channel;
		} else {
			channel = channel_find(target, UT_Ntohs(saddr->scsme_channelid));
			if (!channel) {
				channel = channel_add(target, UT_Ntohs(saddr->scsme_channelid));
				if (!channel) {
					UT_ErrorLog(GTL, "csme_tx: channel_add(%u, %p) failed\n", UT_Ntohs(saddr->scsme_channelid), target);
					csme_target_stats_inc(target, tx_msg_err);
					err = ERROR;
					goto err_free;
				}
			}
		}

		break;

	default:
		err = ERROR;
		UT_ErrorLog(GTL, "csme_tx: invalid opcode(%#x), ch(%u)\n", UT_Ntohs(saddr->scsme_opcode), UT_Ntohs(saddr->scsme_channelid));
		goto err_free;
		break;
	}

	UT_semTake(channel->lock, WAIT_FOREVER);

	/* fill the csme header */
	csme_hdr = (struct csme_hdr *) (pMblk->mBlkHdr.mData + sizeof(ENET_HDR));
	csme_hdr->channel_nb = UT_Htons(channel->id);
	csme_hdr->ack_sup = target->ack_suppression;
	csme_hdr->cr = 0;
	csme_hdr->reserved = 0;
	csme_hdr->endianess = !(channel == &target->boot_channel);
	csme_hdr->opcode = saddr->scsme_opcode;
	csme_hdr->seq_number = channel->tx_seq_number[opcode]++;
	channel->tx_seq_number[opcode] &= 0xF;

	/* fill the ethernet header */
	eth_hdr = (ENET_HDR *) pMblk->mBlkHdr.mData;
	UT_MemCopy(eth_hdr->dst, saddr->scsme_devmac, ETH_ALEN);
	UT_MemCopy(eth_hdr->src, target->iface->ifaddr, ETH_ALEN);
	eth_hdr->type = htons(ETH_P_CSME);

	if (UT_msgQNumMsgs(channel->tx_queue) == 0) {
		/* the queue is empty - we can send */
		if (!csme_hdr->ack_sup) {
			M_BLK_ID pMblkCopy;

			/* If we require an ack then we must keep a copy for
			 * future retransmissions and start the retransmission timer.
			 */

			UT_msgQSend(channel->tx_queue, (char *) &pMblk, sizeof(unsigned long), NO_WAIT, MSG_PRI_URGENT);

			/* the transmission requires a timeout for future retransmissions */
			channel->retries = RETRIES;
			UT_wdStart(channel->timer, TIMEOUT_TICKS, (FUNCPTR) timer_timeout, (int) channel);

			pMblkCopy = UT_netMblkChainDup(pMblk->pClBlk->pNetPool, pMblk, 0, M_COPYALL, M_DONTWAIT);
			if (pMblkCopy == NULL) {
				UT_msgQReceive(channel->tx_queue, (char *) &pMblk, sizeof(unsigned long), NO_WAIT);
				UT_semGive(channel->lock);

				UT_ErrorLog(GTL, "csme_tx: failed to allocate net pool resources, packet lost\n");

				goto err_free;
			}

			UT_semGive(channel->lock);

			pMblk = pMblkCopy;	/* swap values */
		} else {
			UT_semGive(channel->lock);
		}
	} else {
		/* queue isn't empty - try to queue packet, if we fail - cry and go out */

		if (UT_msgQSend(channel->tx_queue, (char *) &pMblk, sizeof(unsigned long), NO_WAIT, MSG_PRI_NORMAL) != OK) {
			UT_ErrorLog(GTL, "csme_tx: failed to queue packet ch(%u, %p)\n", channel->id, target);
		}

		UT_semGive(channel->lock);

		goto err;
	}

	return csme_tx_msg_finish(target->iface, pMblk);

err_free:
	UT_netMblkClChainFree(pMblk);

err:
	return err;
}

/**
 * csmencaps_timerthread - thread for handling transmission timeouts
 *
 */
static void csmencaps_timerthread(void *dummy)
{
	struct csme_channel *channel;
	
	while (UT_msgQReceive(csmencaps.timer_queue, (char *) &channel, sizeof(unsigned long), WAIT_FOREVER) != ERROR) {
		handle_channel_timeout(channel);
	}
}

/**
 * csmencaps_txthread - polls the Tx queue for messages to be sent out
 *
 */
static void csmencaps_txthread(void *dummy)
{
	M_BLK_ID pMblk;

	while (UT_msgQReceive(csmencaps.tx_queue, (char *) &pMblk, sizeof(unsigned long), WAIT_FOREVER) != ERROR) {
		csme_tx(pMblk);
	}
}

/**
 * csme_rx_thread -
 *
 */
static void csme_rx_thread(void *dummy)
{
	int result;
	M_BLK_ID pMblk;

	while (1) {
		result = UT_msgQReceive(csmencaps.csme_rx_queue, (char*) &pMblk, sizeof(unsigned long), WAIT_FOREVER);
		if (result != sizeof(unsigned long)) {
			UT_ErrorLog(GTL, "csmencaps_rx_thread: failed to read from queue\n");
			continue;
		}

		/* handle packet and send ack if required */
		if (csme_rx(pMblk)) {
			/* try to add message to queue */
			if (UT_msgQSend(csmencaps.rx_queue, (char *) &pMblk, sizeof(unsigned long), NO_WAIT, MSG_PRI_NORMAL) == OK) {
				continue;
			}
		}

		UT_netMblkClChainFree(pMblk);
	}
}

/**
 * csmencaps_rcv - called by MUX for incoming packets
 *
 */
static int csmencaps_rcv(void *pCookie, long type, M_BLK_ID pMblk, LL_HDR_INFO *pLinkHdr, void *pSpare)
{
	ENET_HDR *eth_hdr;

	/* first of all, make sure it is a CSMENCAPS packet */
	if (pLinkHdr->pktType != ETH_P_CSME) 
		goto give;

	eth_hdr = (ENET_HDR *) (pMblk->mBlkHdr.mData);
	if (mac_addr_equal(eth_hdr->dst, bcastaddr))
		if (check_ready_received(pMblk))
			return TRUE;

	if (pMblk->mBlkHdr.mLen < (sizeof(ENET_HDR) + sizeof(struct csme_hdr)))
		goto give;

	/* the packet is for us, grab it */
	if (UT_msgQSend(csmencaps.csme_rx_queue, (char *) &pMblk, sizeof(unsigned long), NO_WAIT, MSG_PRI_NORMAL) == ERROR) {
		UT_netMblkClChainFree(pMblk);
	}

	/* even though we could not queue the packet, we should still take the packet */
	return TRUE;

give:
	/* we are not able to handle the packet, let other NPT try to handle it */
	return FALSE;
}

/**
 * generate_sockaddr -
 *
 */
static void generate_sockaddr(M_BLK_ID pMblk, struct sockaddr_csme *saddr)
{
	ENET_HDR *eth_hdr = (ENET_HDR *) pMblk->mBlkHdr.mData;
	struct csme_hdr *csme_hdr = (struct csme_hdr *) (pMblk->mBlkHdr.mData + sizeof(ENET_HDR));
	struct csme_target *target;

	/* form user socket address */
	UT_MemCopy(saddr->scsme_devmac, eth_hdr->src, ETH_ALEN);

	saddr->scsme_opcode = csme_hdr->opcode;
	saddr->scsme_channelid = csme_hdr->channel_nb;
	saddr->scsme_flags = csme_hdr->endianess;

	target = target_find(saddr->scsme_devmac);
	saddr->scsme_user = target->scsme_user;
}


/**
 * csmencaps_recvmsg - pull a packet from our receive queue and hand it to the user.
 *
 * If necessary we block.
 *
 */
int csmencaps_recvmsg(void *buffer, unsigned int *len, struct sockaddr_csme *saddr)
{
	M_BLK_ID pMblk;

	UT_msgQReceive(csmencaps.rx_queue, (char *) &pMblk, sizeof(unsigned long), WAIT_FOREVER);

	csme_ntoh(pMblk);
	*len = pMblk->mBlkHdr.mLen - (sizeof(struct csme_hdr) + sizeof(ENET_HDR));
	UT_MemCopy(buffer, pMblk->mBlkHdr.mData + sizeof(struct csme_hdr) + sizeof(ENET_HDR), *len);

	generate_sockaddr(pMblk, saddr);

	UT_netMblkClChainFree(pMblk);

	return OK;
}

/**
 * csmencaps_sendmsg - send the csmencaps message
 *
 */
int csmencaps_sendmsg(void *buffer, unsigned int len, struct sockaddr_csme *saddr)
{
	M_BLK_ID mblk;

	/* get and verify the address */
	if (saddr == NULL) {
		UT_ErrorLog(GTL, "csmencaps_sendmsp: saddr is NULL\n");
		goto err0;
	}

	if (len == 0 || len > 1500) {
		UT_ErrorLog(GTL, "csmencaps_sendmsp: invalid len (%u)\n", len);
		goto err0;
	}

	mblk = UT_netTupleGet(csmencaps.pool,
			   len + sizeof(struct csme_hdr) + sizeof(ENET_HDR),
			   M_DONTWAIT, MT_DATA, FALSE);
	if (mblk == NULL) {
		UT_ErrorLog(GTL, "csmencaps_sendmsg: failed to allocate mblk\n");
		goto err0;
	}

	/* copy the data from buffer to cluster */
	UT_MemCopy(mblk->mBlkHdr.mData + sizeof(struct csme_hdr) + sizeof(ENET_HDR), buffer, len);

	/* overhead for ethernet header and csmencaps header */
	len += sizeof(struct csme_hdr) + sizeof(ENET_HDR);
	
	/* add the address structure after the data copied from buffer
	 *
	 * NOTE: ROUND_UP() magic is here because csme_tx adds 4 zero bytes
	 *	 after the boot message and we keep the structure aligned
	 */
	UT_MemCopy(mblk->mBlkHdr.mData + ROUND_UP(len + 4, 4), (char *) saddr, sizeof(struct sockaddr_csme));

	/* fill the pMblk struct */	
	mblk->mBlkHdr.mLen = len;
	mblk->mBlkHdr.mFlags |= M_PKTHDR;

	/* finally add the mBlk to the tx queue */
	/* try to add message to queue
	 * if not successful, drop message
	 */
	if (UT_msgQSend(csmencaps.tx_queue, (char *) &mblk, sizeof(unsigned long), NO_WAIT, MSG_PRI_NORMAL) == ERROR) {
		UT_ErrorLog(GTL, "csmencaps_sendmsg: failed to add message to Tx queue\n");
		goto err1;
	}

	return 0;

err1:
	UT_netMblkClFree(mblk);	
err0:
	return -1;
}

/**
 * csmencaps_release - free up any memory allocated for the network buffer pool
 *
 */
static int csmencaps_release()
{
	if (csmencaps.pool) {
		UT_netPoolDelete(csmencaps.pool);
		UT_free(csmencaps.pool_blocks_area);
		UT_free(csmencaps.pool_clusters_area);
		UT_free(csmencaps.pool);
	}

	return 0;
}


static STATUS csmencaps_mux_shutdown(void *pCookie, void *pSpare)
{
	struct csme_iface *iface = pSpare;

	UT_ErrorLog(GTL, "MUX shutdown, interface '%s'\n", iface->ifname);

	UT_semTake(csmencaps.csme_lock, WAIT_FOREVER);

 	UT_semTake(iface->lock, WAIT_FOREVER);

 	if (UT_muxUnbind(iface->mux, MUX_PROTO_SNARF, csmencaps_rcv) != OK) 
        	 UT_ErrorLog(GTL, "csmencaps_mux_shutdown: can't unbind from '%s'\n", iface->ifname);

	__csme_iface_remove(iface);

	UT_semGive(csmencaps.csme_lock);

	return OK;
}


static STATUS csmencaps_mux_restart(void *pCookie, void *pSpare)
{
 	UT_Log(GTL, WARN, "MUX restart, interface '%s'\n", ((struct csme_iface *)pSpare)->ifname);

	return OK;
}


static void csmencaps_mux_error(END_OBJ *pEnd, END_ERR *pError, void *pSpare)
{
	struct csme_iface *iface = pSpare;
	int level = WARN;

	switch (pError->errCode)
	{
	case END_ERR_INFO:
	case END_ERR_FLAGS:
		level = INFO;
		break;

	case END_ERR_DOWN:
	case END_ERR_UP:
		UT_semTake(iface->lock, WAIT_FOREVER);
		iface->down = (pError->errCode == END_ERR_DOWN);
		UT_semGive(iface->lock);
	}

	if (pError->pMesg)
		UT_Log(GTL, level, "MUX %s 0x%08lx on interface '%s': %s\n", (level == INFO?"info":"error"), pError->errCode, iface->ifname, pError->pMesg);
	else
		UT_Log(GTL, level, "MUX %s 0x%08lx on interface '%s'\n", (level == INFO?"info":"error"), pError->errCode, iface->ifname);
}

/**
 * csmencaps_bind - bind network protocol to END driver
 *
 */
int csmencaps_bind(const char *ifname)
{
	struct csme_iface *iface;

	if (!ifname || !*ifname) {
		UT_ErrorLog(GTL, "csmencaps_bind: invalid interface name\n");
		goto err0;
	}

	iface = csme_iface_find(ifname);

	UT_semTake(csmencaps.csme_lock, WAIT_FOREVER);

	if (iface) {
		UT_semTake(iface->lock, WAIT_FOREVER);
		goto ok;
	}

	iface = __csme_iface_add(ifname);
	if (!iface) {
		UT_ErrorLog(GTL, "csmencaps_bind: can't add interface '%s'\n", ifname);
		goto err1;
	}

	/* bind to the interface */
	UT_semTake(iface->lock, WAIT_FOREVER);

	iface->mux = UT_muxBind(iface->ifbasename, iface->ifindex,
				csmencaps_rcv, csmencaps_mux_shutdown, csmencaps_mux_restart, csmencaps_mux_error,
				MUX_PROTO_SNARF, CSME_PROTO_NAME, iface);
	if (iface->mux == NULL) {
		UT_ErrorLog(GTL, "csmencaps_bind: can't bind to '%s' (iface: '%s', unit: %d)\n",
				ifname, iface->ifbasename, iface->ifindex);
		goto err2;
	}

	iface->down = 0;

ok:
	/* keeping the counter of all opened devices on this interface */
	iface->refcount++;

	UT_semGive(iface->lock);
	UT_semGive(csmencaps.csme_lock);

	return 0;

err2:
	__csme_iface_remove(iface);	/* iface->lock will be freed inside */
	
err1:
	UT_semGive(csmencaps.csme_lock);

err0:
	return -1;
}

/**
 * csmencaps_unbind - unbind network protocol to END driver
 *
 */
int csmencaps_unbind(const char *ifname)
{
	struct csme_iface *iface;

	if (!ifname || !*ifname) {
		UT_ErrorLog(GTL, "csmencaps_unbind: invalid interface name\n");
		goto err0;
	}

	iface = csme_iface_find(ifname);
	if (!iface) {
		UT_ErrorLog(GTL, "csmencaps_unbind: not bound to '%s'\n", ifname);
		goto err0;
	}

	/* it's possible to have many devices opened on the same interface
	 * unbind from MUX only in case when all devices are closed
	 */
	UT_semTake(csmencaps.csme_lock, WAIT_FOREVER);

	UT_semTake(iface->lock, WAIT_FOREVER);

	if (--iface->refcount > 0) {
		UT_semGive(iface->lock);
		goto ok;
	}

	if (UT_muxUnbind(iface->mux, MUX_PROTO_SNARF, csmencaps_rcv) != OK)
		UT_ErrorLog(GTL, "csmencaps_unbind: can't unbind from '%s'\n", ifname);

	__csme_iface_remove(iface);

ok:
	UT_semGive(csmencaps.csme_lock);

	return 0;

err0:
	return -1;
}

/**
 * csmencaps_setsockopt - set socket options
 *
 */
int csmencaps_setsockopt(int level, int optname, char *optval, int optlen)
{
	struct csme_ackopt *ackopt;
	struct csme_target *target;
	struct csme_iface *iface;
	struct csme_resetopt *resetopt;
	struct csme_usersetopt *usersetopt;

	if (level != SOL_CSME)
		return ERROR;

	switch (optname) {
	case CSME_ACK_ENABLE:
		if (optlen != sizeof(struct csme_ackopt))
			return ERROR;

		ackopt = (struct csme_ackopt *) optval;
		if ((ackopt->ack_suppression&~1) != 0)
			return ERROR;

		iface = csme_iface_find(ackopt->scsme_ifname);
		if (!iface)
			return ERROR;

		target = target_find(ackopt->scsme_devmac);	/* find target matching with scsme_devmac */
		if (!target) {
			target = target_add(ackopt->scsme_devmac, iface->ifname, NULL);
			if (!target)
				return ERROR;
		}

		target->ack_suppression = ackopt->ack_suppression;
		break;

	case CSME_TARGET_RESET:
		if (optlen != sizeof(struct csme_resetopt))
			return ERROR;

		resetopt = (struct csme_resetopt *) optval;
	
		target = target_find(resetopt->scsme_devmac);

		/* if no target there, nothing to reset
		 * else reassign the original address (broadcast or dev hardmac)
		 */
		if (target) {
			target_change_mac(target, resetopt->scsme_newmac);
			target_reset(target);
		}

		break;

	case CSME_USER_SET:
		if (optlen != sizeof(struct csme_usersetopt))
			return ERROR;

		usersetopt = (struct csme_usersetopt *) optval;

		target = target_find(usersetopt->scsme_devmac);

		if (target) {
			target->scsme_user = usersetopt->csme_user;
		}

		break;

	default:
		return ERROR;
	}

	return 0;
}

/**
 * csmencaps_getsockopt - get socket options
 *
 */
int csmencaps_getsockopt(int level, int optname, char *optval, int *optlen)
{
	struct csme_ackopt *ackopt;
	struct csme_target *target;

	if (level != SOL_CSME)
		return ERROR;

	switch (optname) {
	case CSME_ACK_ENABLE:
		ackopt = (struct csme_ackopt *) optval;
		target = target_find(ackopt->scsme_devmac);	/* find target matching with saddr->scsme_devmac */
		if (!target)
			return ERROR;

		ackopt->ack_suppression = target->ack_suppression;
		*optlen = sizeof(struct csme_ackopt);
		break;

	default:
		return ERROR;
	}

	return OK;
}


int csmencaps_create(void)
{
	CL_DESC	 cl_desc[1] = { {CSME_CL_SIZE, CSME_CL_NUM, NULL, 0} };
	M_CL_CONFIG mcl_config;

	mcl_config.mBlkNum = CSME_MBLK_NUM;
	mcl_config.clBlkNum = CSME_CBLK_NUM;
	mcl_config.memSize = mcl_config.mBlkNum * (M_BLK_SZ + sizeof (long)) +
				mcl_config.clBlkNum * (CL_BLK_SZ + sizeof(long));
	mcl_config.memArea = (char *) UT_malloc(mcl_config.memSize);
	if (mcl_config.memArea == NULL) {
		UT_ErrorLog(GTL, "csmencaps_create: ERROR: can't allocate memory\n");
		goto err0;
	}

	csmencaps.pool_blocks_area = mcl_config.memArea;

	/* allocate memory for clusters */
	cl_desc[0].clSize = ROUND_UP(cl_desc[0].clSize + CSME_CL_OVERHEAD, 4);
	/* taking in account that sockaddr_csme struct resides
	 * after the rounded by 4 frame in offset of 4 bytes
	 */
	cl_desc[0].clSize = ROUND_UP(cl_desc[0].clSize + 4 + sizeof(struct sockaddr_csme), 4);
	cl_desc[0].memSize = cl_desc[0].clNum * (cl_desc[0].clSize + CSME_CL_OVERHEAD);
	cl_desc[0].memArea = (char *) UT_malloc(cl_desc[0].memSize);
	if (cl_desc[0].memArea == NULL) {
		UT_ErrorLog(GTL, "csmencaps_create: ERROR: can't allocate memory for clusters\n");
		goto err1;
	}

	csmencaps.pool_clusters_area = cl_desc[0].memArea;

	/* initialize the network buffer library */
	if (UT_netBufLibInit()) {
		UT_ErrorLog(GTL, "csmencaps_create: ERROR: can't initialize netBufLib\n");
		goto err2;
	}

	/* allocate memory for the net pool */
	csmencaps.pool = (NET_POOL_ID) UT_malloc(sizeof(NET_POOL));
	if (csmencaps.pool == NULL) {
		UT_ErrorLog(GTL, "csmencaps_create: ERROR: can't allocate memory\n");
		goto err2;
	}

	/* initialize the network pool */

	if (UT_netPoolInit(csmencaps.pool, &mcl_config, &cl_desc[0], NELEMENTS(cl_desc), NULL)) {
		UT_ErrorLog(GTL, "csmencaps_create: ERROR: can't set up mem pool, %d\n", errno);
		goto err3;
	}

	return OK;

err3:
	UT_free(csmencaps.pool);
	csmencaps.pool = NULL;

err2:
	UT_free(csmencaps.pool_clusters_area);

err1:
	UT_free(csmencaps.pool_blocks_area);

err0:
	return ERROR;
}


static int csme_start_tx()
{
	int prio;

	csmencaps.tx_queue = UT_msgQCreate(TX_MSGQ_SIZE, sizeof(unsigned long), MSG_Q_FIFO);
	if (csmencaps.tx_queue == NULL) {
		UT_ErrorLog(GTL, "csme_start_tx: can't create TX queue\n");
		goto err0;
	}

	csmencaps.tx_thread = UT_ThreadCreate(NAME_GTL_TX, (PFNThreadEntry) csmencaps_txthread, NULL);
	if (csmencaps.tx_thread == NULL) {
		UT_ErrorLog(GTL, "csme_start_tx: can't create TX thread\n");
		goto err1;
	}
	UT_ThreadPriorityAdjust(csmencaps.tx_thread,  prio,  PRIO_GTL_TX);

	return 0;

err1:
	UT_msgQDelete(csmencaps.tx_queue);

err0:
	return -1;
}


static void csme_stop_tx()
{
	UT_ThreadCancel(csmencaps.tx_thread);
	UT_ThreadJoin(csmencaps.tx_thread);
	UT_FreeMem(csmencaps.tx_thread);
	csmencaps.tx_thread = NULL;

	csme_mblk_queue_free(csmencaps.tx_queue);
}


static int csme_start_rx()
{
	int prio;

	csmencaps.rx_queue = UT_msgQCreate(RX_MSGQ_SIZE, sizeof(unsigned long), MSG_Q_FIFO);
	if (csmencaps.rx_queue == NULL) {
		UT_ErrorLog(GTL, "csme_start_rx: can't create RX queue\n");
		goto err0;
	}

	csmencaps.csme_rx_queue = UT_msgQCreate(CSME_RX_MSGQ_SIZE, sizeof(unsigned long), MSG_Q_FIFO);
	if (csmencaps.csme_rx_queue == NULL) {
		UT_ErrorLog(GTL, "csme_start_rx: can't create csme RX queue\n");
		goto err1;
	}

	csmencaps.csme_rx_thread = UT_ThreadCreate(NAME_CSME_RX, (PFNThreadEntry) csme_rx_thread, NULL);
	if (csmencaps.csme_rx_thread == NULL) {
		UT_ErrorLog(GTL, "csme_start_rx: can't create RX thread\n");
		goto err2;
	}

	UT_ThreadPriorityAdjust(csmencaps.csme_rx_thread,  prio,  PRIO_CSME_RX);

	return 0;

err2:
	UT_msgQDelete(csmencaps.csme_rx_queue);

err1:
	UT_msgQDelete(csmencaps.rx_queue);

err0:
	return -1;
}


static void csme_stop_rx()
{
	UT_ThreadCancel(csmencaps.csme_rx_thread);
	UT_ThreadJoin(csmencaps.csme_rx_thread);
	UT_FreeMem(csmencaps.csme_rx_thread);
	csmencaps.csme_rx_thread = NULL;

	csme_mblk_queue_free(csmencaps.csme_rx_queue);
}


static int csme_start_timer()
{
	int prio;

	csmencaps.timer_queue = UT_msgQCreate(TIMER_MSGQ_SIZE, sizeof(unsigned long), MSG_Q_FIFO);
	if (csmencaps.timer_queue == NULL) {
		UT_ErrorLog(GTL, "csme_start_timer: can't create timer queue\n");
		goto err0;
	}
	csmencaps.timer_thread = UT_ThreadCreate(NAME_GTL_TIMER, (PFNThreadEntry) csmencaps_timerthread, NULL);
	if (csmencaps.timer_thread == NULL) {
		UT_ErrorLog(GTL, "csme_start_timer: can't create timer thread\n");
		goto err1;
	}

	UT_ThreadPriorityAdjust(csmencaps.timer_thread,  prio,  PRIO_GTL_TIMER);

	return 0;

err1:
	UT_msgQDelete(csmencaps.timer_queue);

err0:
	return -1;
}


static void csme_stop_timer()
{
	UT_ThreadCancel(csmencaps.timer_thread);
	UT_ThreadJoin(csmencaps.timer_thread);
	UT_FreeMem(csmencaps.timer_thread);
	csmencaps.timer_thread = NULL;

	UT_msgQDelete(csmencaps.timer_queue);
}


void csmencaps_exit(void)
{
	struct csme_iface *iface, *next;

	UT_semTake(csmencaps.csme_lock, WAIT_FOREVER);

	csme_stop_rx();

	csme_stop_tx();

	csme_stop_timer();

	/* free up all interfaces (targets will be freed in __csme_iface_remove() */
	iface = (struct csme_iface *) UT_lstFirst(&csmencaps.csme_iface_list);

	while (iface != NULL) {
		UT_semTake(iface->lock, WAIT_FOREVER);

		next = (struct csme_iface *) UT_lstNext((NODE *) iface);

		if (UT_muxUnbind(iface->mux, MUX_PROTO_SNARF, csmencaps_rcv) != OK)
			UT_ErrorLog(GTL, "csmencaps_unbind: can't unbind from '%s'\n", iface->ifname);
		__csme_iface_remove(iface);

		iface = next;
	}

	UT_lstFree(&csmencaps.csme_iface_list);

	UT_lstFree(&csmencaps.csme_target_list);

	UT_semDelete(csmencaps.csme_lock);
	csmencaps.csme_lock = NULL;

	csmencaps_release();

	return;
}


int csmencaps_init(void)
{
	if (csmencaps.csme_lock)
		return 0;

	/* initialize target list and associated semaphore */
	csmencaps.csme_lock = UT_semMCreate(SEM_Q_PRIORITY | SEM_INVERSION_SAFE);
	if (csmencaps.csme_lock == NULL) {
		UT_ErrorLog(GTL, "csmencaps_init: can't create mutex\n");
		goto err0;
	}

	UT_lstInit(&csmencaps.csme_target_list);
	UT_lstInit(&csmencaps.csme_iface_list);

	if (csme_start_rx()) {
		UT_ErrorLog(GTL, "csmencaps_init: failed to allocate RX resources\n");
		goto err1;
	}

	if (csme_start_tx()) {
		UT_ErrorLog(GTL, "csmencaps_init: failed to allocate TX resources\n");
		goto err2;
	}

	if (csme_start_timer()) {
		UT_ErrorLog(GTL, "csmencaps_init: failed to allocate timer resources\n");
		goto err3;
	}

	/* allocate the networking structures and clusters for MUX transmissions */
	if (csmencaps_create()) {
		UT_ErrorLog(GTL, "csmencaps_init: csmencaps_create() failed\n");
		goto err4;
	}

	return 0;

err4:
	csme_stop_timer();

err3:
	csme_stop_tx();

err2:
	csme_stop_rx();

err1:
	UT_semTake(csmencaps.csme_lock, WAIT_FOREVER);
	UT_semDelete(csmencaps.csme_lock);

err0:
	UT_lstFree(&csmencaps.csme_iface_list);
	UT_lstFree(&csmencaps.csme_target_list);

	return -1;
}
