/*! \file ut_port.c 
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */


#define _XOPEN_SOURCE 600
#include <errno.h>
#include <semaphore.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <linux/if.h>
#include <netinet/in.h>
#include <linux/sockios.h>
#include <linux/if_ether.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>

#include "ut.h"
#include "ut_port.h"

/* Global Variables */
static FILE *fpLog = NULL; /**<File Handler for logging*/

/*Function Definitions*/

/***************************************************************************
 * UT_GetTimeOfDay: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function gets the time of the day.\n
 *  \return
 *   SUCCESS    
 *   FAILURE    
 */
INT UT_GetTimeOfDay(OUT STimeSpec * psTime)
{
	struct timeval stTimeVal;

	if (gettimeofday(&stTimeVal, NULL) < 0)
		return -1;

	psTime->tv_sec = stTimeVal.tv_sec;
	psTime->tv_nsec = stTimeVal.tv_usec * 1000;

	return 0;
}

/***************************************************************************
 * UT_ThreadCreate: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function creates the thread in accordance with the args passed.\n
 *  \return
 *   Pointer to the created thread.\n
 *   NULL in case the pthread creation failed.\n
 */
SThread *UT_ThreadCreate(char *pTName, PFNThreadEntry pfnFunc, void *pvArg)
{
	SThread *pstTh;

	pstTh = (SThread *) UT_AllocMem(sizeof(SThread));
	if (NULL == pstTh)
		goto err0;

	if (pthread_create(pstTh, NULL, pfnFunc, pvArg) < 0)
		goto err1;

	return pstTh;

      err1:
	UT_FreeMem(pstTh);
      err0:
	return NULL;
}

/***************************************************************************
 * UT_ThreadDetach: The function does the following things -
 ***************************************************************************/
/*!
 *  This function puts the thread into detached state.
 *  \return
 *   Non-zero in case the operation failed.\n
 */
INT UT_ThreadDetach(SThread * pstThread)
{
	return pthread_detach(*pstThread);
}

/***************************************************************************
 * UT_ThreadCancel: The function does the following things -
 ***************************************************************************/
/*!
 *  This function sends cancellation request to the specified thread.
 *  \return
 *   Non-zero in case the cancellation failed.\n
 */
INT UT_ThreadCancel(SThread * pstThread)
{
	return pthread_cancel(*pstThread);
}

/***************************************************************************
 * UT_ThreadTestCancel: The function does the following things -
 ***************************************************************************/
/*!
 *  This function tests for pending cancellation request for the current thread.\n
 *  \return
 *   None.\n
 */
void UT_ThreadTestCancel()
{
	pthread_testcancel();
}

/***************************************************************************
 * UT_ThreadSetCancelParams: The function does the following things -
 ***************************************************************************/
/*!
 *  This function sets both cancellation state and type. Non-zero iEnableCancel
 *  means PTHREAD_CANCEL_ENABLE, non-zero iEnableAsync means
 *  PTHREAD_CANCEL_ASYNCHRONOUS. Changes affect caller thread.
 *  \return
 *   Non-zero in case any operation failed.\n
 */
INT UT_ThreadSetCancelParams(Boolean iEnableCancel, Boolean iEnableAsync)
{
	INT result;

	result = pthread_setcancelstate(iEnableCancel ? PTHREAD_CANCEL_ENABLE : PTHREAD_CANCEL_DISABLE, NULL);
	if (result != 0)
		return result;

	return pthread_setcanceltype(iEnableAsync ? PTHREAD_CANCEL_ASYNCHRONOUS : PTHREAD_CANCEL_DEFERRED, NULL);
}

/***************************************************************************
 * UT_ThreadJoin: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function will join thread, specify by input parameter pstThread.\n
 *  \return
 *   0 in case of SUCCESS.\n
 */
INT UT_ThreadJoin(SThread * pstThread)
{
	return pthread_join(*pstThread, NULL);
}

/***************************************************************************
 * UT_SemInit: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function initializes the semaphore\n
 *  2.Returns the pointer of newly created semaphore\n
 *  \return
 *   NULL in case semaphore creation failed.\n
 *   Pointer to the semaphore in  case of success.\n
 */
SSem *UT_SemInit(IN UINT uiValue)
{
	SSem *pstSem = NULL;

	pstSem = (SSem *) UT_AllocMem(sizeof(SSem));
	if (NULL == pstSem)
	{
		return NULL;
	}

	if (sem_init(pstSem, 0, uiValue) < 0)
	{
		UT_FreeMem(pstSem);
		return NULL;
	}

	return pstSem;
}

/***************************************************************************
 * UT_SemWait: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function waits for the specified amount of time with sem_timedwait.\n
 *  
 *  \return
 *   the value returned by sem_timedwait.\n
 */
INT UT_SemWait(IN SSem * pstSem, IN UINT uiTimeout)
{
	int iSec, iMsec;
	struct timespec stTs;
	struct timeval stTv;
	int iRc;

	if (uiTimeout != INFINITE_TIMEOUT)
	{
		iSec = uiTimeout / 1000;
		iMsec = (uiTimeout % 1000);

		gettimeofday(&stTv, NULL);

		iMsec += (stTv.tv_usec) / 1000;

		stTs.tv_sec = stTv.tv_sec + iSec + (iMsec / 1000);
		stTs.tv_nsec = (iMsec % 1000) * 1000000;

		iRc = sem_timedwait(pstSem, &stTs);
	}
	else
	{
		iRc = sem_wait(pstSem);
	}

	return iRc;
}

/***************************************************************************
 * UT_SemPost: The function does the following things -
 ***************************************************************************/
/*!
 *  1.It releases the semaphore passed to it using semGive.\n
 *
 *  \return
 *   Value returned by semGive.\n
 */
INT UT_SemPost(IN SSem * pstSem)
{
	return sem_post(pstSem);
}

/***************************************************************************
 * UT_SemDestroy: The function does the following things -
 ***************************************************************************/
/*!
 *  1.It destroys the semaphore passed to it.\n
 *
 *  \return
 *   Value returned by semDelete.\n
 */
INT UT_SemDestroy(IN SSem * pstSem)
{
	int iRc;

	iRc = sem_destroy(pstSem);
	UT_FreeMem(pstSem);

	return iRc;
}

/***************************************************************************
 * UT_MutexInit: The function does the following things -
 ***************************************************************************/
/*!
 *  1.It creates and initilize SMutex.\n
 *
 *  \return
 *   pointer to SMutex.\n
 */
INT UT_MutexInit(IN SMutex * pstMutex)
{
	return pthread_mutex_init(pstMutex, NULL);
}

/***************************************************************************
 * UT_MutexDestroy: The function does the following things -
 ***************************************************************************/
/*!
 *  1.It  destroys the mutex passed to it.\n
 *
 *  \return
 *    void \n
 */
INT UT_MutexDestroy(IN SMutex * pstMutex)
{
	return pthread_mutex_destroy(pstMutex);
}

/***************************************************************************
 * UT_MutexLock: The function does the following things -
 ***************************************************************************/
/*!
 *  1.It Locks Mutex passed to it \n
 *
 *  \return
 *   return Code. \n
 */
INT UT_MutexLock(IN SMutex * pstMutex)
{
	return pthread_mutex_lock(pstMutex);
}

/***************************************************************************
 * UT_MutexUnLock: The function does the following things -
 ***************************************************************************/
/*!
 *  1.It UnLocks Mutex passed to it \n
 *
 *  \return
 *   return Code. \n
 */
INT UT_MutexUnLock(IN SMutex * pstMutex)
{
	return pthread_mutex_unlock(pstMutex);
}

/***************************************************************************
 * UT_InitLog: The function does the following things -
 ***************************************************************************/
/*!
 *  1.Opens a file for logging in write mode.\n
 *  2.If the file could not be opened then log message are displayed on\n
 *    console.\n
 *  \return
 *  SUCCESS  if file is opened in write mode for logging.\n
 *  FAILURE  if couldn't be opened.\n
 */
static int UT_InitLog(void)
{
#ifdef DBG_MODE_FILE
	fpLog = fopen("vapi.log", "w");
	return fpLog == NULL ? -1 : 0;
#else
	return 0;
#endif
}

/***************************************************************************
 * UT_CloseLog: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function closes the file opened for logging.\n
 *
 *  Assumptions\n
 *  None\n
 *
 *  \return
 *   SUCCESS    if the log file is successfully closed.\n
 *   FAILURE    if the log file could not be opened.\n
 */
static int UT_CloseLog(void)
{
#ifdef DBG_MODE_FILE
	fclose(fpLog);
#endif

	return 0;
}

/***************************************************************************
 * UT_VErrorLog: The function does the following things -
 ***************************************************************************/
void UT_VErrorLog(IN U8 ucModule, IN char *pstrFormat, va_list ap)
{
	if (NULL != fpLog)
	{
		vfprintf(fpLog, pstrFormat, ap);
		fflush(fpLog);
	}
	else
	{
		vprintf(pstrFormat, ap);
	}
}

INT UT_OSWInit()
{
	if (UT_TimerModuleInit(NULL) == NULL)
	{
		UT_ErrorLog(UT,	"UT_TimerModuleInit: failed \n");
		return	FAILURE;
	}
	else
		return	UT_InitLog();
}

void UT_OSWClose()
{
	UT_TimerShutDown();
	UT_CloseLog();
}

MSG_Q_ID UT_MsgQCreate_FIFO(int maxMess, int maxLen)
{
	static const char *fnName = "UT_MsgQCreate_FIFO:";
	int *ipc_fd = (int *)UT_AllocMem(sizeof(int) * 2);

	if (ipc_fd == NULL)
	{
		UT_ErrorLog(UT, "%s No Memory Avilable\n", fnName);
		return NULL;
	}

	if (pipe(ipc_fd) != 0)
	{
		UT_ErrorLog(UT, "%s pipe open failed, errno %d: %s\n", fnName, errno, strerror(errno));
		UT_FreeMem(ipc_fd);
		return NULL;
	}

	return ipc_fd;
}

STATUS UT_MsgQDelete(MSG_Q_ID msgQId)
{
	static const char *fnName = "UT_MsgQDelete:";
	int retVal = SUCCESS;

	if (close(msgQId[0]))
	{
		UT_ErrorLog(UT, "%s pipe close 0 failed, errno %d: %s\n", fnName, errno, strerror(errno));
		retVal = FAILURE;
	}

	if (close(msgQId[1]))
	{
		UT_ErrorLog(UT, "%s pipe close 1 failed, errno %d: %s\n", fnName, errno, strerror(errno));
		retVal = FAILURE;
	}

	UT_FreeMem(msgQId);

	return retVal;
}

STATUS UT_MsgQSend(MSG_Q_ID msgQId, char *buffer, U32 nBytes, int timeout, int priority)
{
	static const char *fnName = "UT_MsgQSend:";
	int written;
	int retVal = SUCCESS;

	if (timeout != 0)
	{
		UT_ErrorLog(UT, "%s timeout %d ignored\n", fnName, timeout);
	}

	if (priority != NO_WAIT)
	{
		UT_ErrorLog(UT, "%s priority %d ignored\n", fnName, priority);
	}

	if ((written = write(msgQId[1], buffer, nBytes)) != nBytes)
	{
		UT_ErrorLog(UT, "%s Request for %d bytes, only wrote %d\n", fnName, nBytes, written);
		retVal = FAILURE;
	}

	return SUCCESS;
}

int UT_MsgQReceive(MSG_Q_ID msgQId, char *buffer, U32 maxNBytes, int timeout)
{
	static const char *fnName = "UT_MsgQReceive:";
	fd_set fdRead;
	struct timeval timer;
	int status, len;

	if (timeout != WAIT_FOREVER)
	{
		// Assume 100 ticks per second
		timer.tv_sec = timeout / 100;
		timer.tv_usec = (timeout % 100) * 10000;
		FD_ZERO(&fdRead);
		FD_SET(msgQId[0], &fdRead);

		status = select(msgQId[0] + 1, &fdRead, NULL, NULL, &timer);
		if (status == -1)
		{
			UT_ErrorLog(UT, "%s Error in select call, errno %d: %s\n", fnName, errno, strerror(errno));
			return FAILURE;
		}

		if (status == 0)
		{
			return 0;
		}
	}

	len = read(msgQId[0], buffer, maxNBytes);

	return len;
}

int UT_GetInterfaceMac(IN char *pcEthName, char *pcHostMacAddr)
{
	int fd;
	struct ifreq ifreq;

	UT_Log(GTL, INFO, "UT_GetInterfaceMac: ENTRY\n");
	if (NULL == pcEthName || NULL == pcHostMacAddr)
	{
		UT_ErrorLog(GTL, "UT_GetInterfaceMac: EthName/pcHostMacAddr passed to the function is NULL\n");
		return FAILURE;
	}

	fd = socket(PF_INET, SOCK_DGRAM, 0);
	if (fd == -1)
	{
		UT_ErrorLog(GTL, "UT_GetInterfaceMac: Socket: %s, errno = %d\n", strerror(errno), errno);
		return FAILURE;
	}

	strcpy(ifreq.ifr_name, pcEthName);
	if (ioctl(fd, SIOCGIFHWADDR, (int)&ifreq) < 0)
	{
		UT_ErrorLog(GTL, "UT_GetInterfaceMac: IOCTL: %s, errno = %d\n", strerror(errno), errno);
		close(fd);
		return FAILURE;
	}

	UT_MemCopy(pcHostMacAddr, ifreq.ifr_hwaddr.sa_data, ETH_ALEN);
	close(fd);

	UT_Log(GTL, INFO, "UT_GetInterfaceMac:EXIT\n");
	return SUCCESS;
}
