/*! \file tpci.c  
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include "ut.h"
#include "gtl.h"
#include "msp.h"
#include "tpci.h"
#include "tmap_devinfo.h"

#include <poll.h>
#include <unistd.h>
#include <sys/ioctl.h> 
#include <sys/fcntl.h>
#include <sys/select.h>

#include "comcerto_pci/comcerto_pci.h"

#define MAX_PCI_DEVICE	8

static SSem *pTPCIMutex = NULL;

/**Global array of PCI devices (8 max)*/
static struct pci_device
{
	int	fd;
	int	dev_id;
	SThread	*reader_tid;
} devices[MAX_PCI_DEVICE];

static GTL_READ_CB gtl_read_callback = NULL;

static void *TPCI_RxThread(void *dummy);

/************************** fd_add *************************************/
/*!
 *	This function adds the file descriptor corresponding to a Comcerto PCI driver file \n
 *	to the global device array.Then creates a Rx thread for the Comcerto PCI device. 
 *
 *	\return Device index in the device array \n
 *              On failure, it returns -1 \n
 *	\param  fd Device file descriptor (got by TPCI_Open)
 *	\param  dev_id Device ID provided by VAPI layer
 */
/***************************************************************************/
static int fd_add(int fd, int dev_id)
{
	int i;

	for (i = 0; i < MAX_PCI_DEVICE; i++)
		if (devices[i].fd == -1)
		{
			devices[i].reader_tid = UT_ThreadCreate(NULL, TPCI_RxThread, (void*) fd);
			if (devices[i].reader_tid != NULL)
			{
				devices[i].fd = fd;
				devices[i].dev_id = dev_id;
				return i;
			}
		}

	return -1;
}

/************************** fd_reset *************************************/
/*!
 *	This function removes the file descriptor corresponding to a Comcerto PCI driver file \n
 *	from the global device array.\n
 *	Then destroy the thread for the Comcerto PCI device. 
 *
 *	\return 0 on success \n
 *              -1 On failure \n
 *	\param  fd Device file descriptor (got by TPCI_Open)
 */
/***************************************************************************/
static int fd_reset(int fd)
{
	int i;

	for (i = 0; i < MAX_PCI_DEVICE; i++)
		if (devices[i].fd == fd)
		{
			if (devices[i].reader_tid)
			{
				UT_ThreadCancel(devices->reader_tid);
				devices[i].reader_tid = NULL;
			}
			devices[i].fd = -1;
			devices[i].dev_id = -1;
			return 0;
		}

	return -1;
}

/************************** fd_lookup_device_id *************************************/
/*!
 *	This function returns the VAPI device ID corresponding to a Comcerto PCI driver file descriptor \n
 *
 *	\return Device ID \n
 *              -1 On failure \n
 *	\param  fd Device file descriptor (got by TPCI_Open)
 */
/***************************************************************************/
static int fd_lookup_device_id(int fd)
{
	int i;

	for (i = 0; i < MAX_PCI_DEVICE; i++)
		if (devices[i].fd == fd)
			return devices[i].dev_id;

	return -1;
}

/************************** TPCI_Open *************************************/
/*!
 *    This function activates the device. Performs h/w initialization \n
 *	  and  enables interupts. Before call to this function, \n
 *    GTL Mapper will not be able to send and receive any message to and \n
 *    from MSP.\n
 *    1. Opens the device.\n
 *    2. Initializes the device (Interrupts are enabled)
 *    3. Add the device ID and fd mapping in FD List \n
 *
 *    \return   On successful completion, it returns SUCCESS\n
 *              On failure, it returns GTL_TPCI_ERR_NOT_INITIALIZED\n
 *				or GTL_TPCI_ERR_DEVLVL \n	
 *    \param    deviceid -> device id of the device.\n
 *    \param    pvDevAddrInfo -> device Addressing Info.\n
 *    \param    ppvDevAdapterInfo -> OUT parameter for returning adapter \n
 *              specific device info.\n
 *    \param    eDevMode -> Ignored 
 */
/***************************************************************************/
static S32 TPCI_Open(S32 dev_id, void *pvDevAddrInfo, void **pvAdapterSpecInfo, EDevMode eDevMode)
{
	SGtlPciItf *pstPciItf = (SGtlPciItf *) pvDevAddrInfo;
	S32 iRetCode = SUCCESS;
	int fd = -1;

	UT_Log(GTL, INFO, "TPCI_Open: ENTRY for device %s\n", pstPciItf->pciDevName);

	fd = open((char *)pstPciItf->pciDevName, O_RDWR, 0);
	if (fd < 0)
	{
		iRetCode = GTL_TPCI_ERR_NOT_INITIALIZED;
		goto err;
	}

	/* Adapter specific device identifier */
	*pvAdapterSpecInfo = UT_AllocMem(sizeof(int));
	*(int *)*pvAdapterSpecInfo = fd;

	UT_SemWait(pTPCIMutex, MAX_TIMEOUT);

	iRetCode = fd_add(fd, dev_id);
	if (iRetCode < 0)
	{
		UT_SemPost(pTPCIMutex);
		iRetCode = GTL_TPCI_ERR_NOT_INITIALIZED;
		goto err1;
	}

	iRetCode = SUCCESS;

	UT_SemPost(pTPCIMutex);

	UT_Log(GTL, INFO, "TPCI_Open: EXIT\n");

	return SUCCESS;

err1:
	UT_SemPost(pTPCIMutex);

err:
	UT_ErrorLog(GTL, " TPCI_Open : Error while opening device %s\n",
		(char *)pstPciItf->pciDevName);
	return iRetCode;
}


/************************** TPCI_Close *************************************/
/*!
 *    This function de-activates the device.\n
 *    1. Pus the device in RESET.\n
 *    2. Closes the device.\n
 *
 *    \return   On successful completion, it returns SUCCESS \n
 *              On failure, it returns GTL_TPCI_ERR_DEVLVL or GTL_TPCI_ERR_FATAL \n
 *    \param    pvDevAddrInfo -> device Addressing Info.\n
 */
/***************************************************************************/
static S32 TPCI_Close(void *pvDevAdapterInfo)
{

	int fd = *(int *)pvDevAdapterInfo;
	S32 iRetCode = SUCCESS;

	UT_Log(GTL, INFO, "TPCI_Close: ENTRY\n");

	UT_FreeMem(pvDevAdapterInfo);

	iRetCode = close(fd);
	if (iRetCode != 0) /* Ret code of driver when dev closed successfully */
	{
		UT_ErrorLog(GTL, " TPCI_Close : Unable to close device, error %d\n", iRetCode);
		iRetCode = GTL_TPCI_ERR_DEVLVL;
		goto out;
	}

	iRetCode = SUCCESS;

	UT_SemWait(pTPCIMutex, MAX_TIMEOUT);
	iRetCode = fd_reset(fd);
	UT_SemPost(pTPCIMutex);

out:
	UT_Log(GTL, INFO, "TPCI_Close: EXIT\n");
	return iRetCode;
}


/************************** TPCI_Reset *************************************/
/*!
 *    1. This function resets the PCI device and does h/w initialization. \n
 *				Interrupts are enabled. \n
 *
 *    \return   On successful completion, it returns SUCCESS\n
 *		On failure, it returns GTL_TPCI_ERR_DEVSTATE or GTL_TPCI_ERR_DEVLVL \n
 *    \param    pvDevAddrInfo -> device Addressing Info.\n
 */
/***************************************************************************/
static S32 TPCI_Reset(void *pvDevAddrInfo, void *pvAdapterSpecInfo)
{
	int fd = *(int *)pvAdapterSpecInfo;
	S32 iRetCode = SUCCESS;

	UT_Log(GTL, INFO, "TPCI_Reset: ENTRY\n");
	UT_Log(GTL, INFO, " TPCI_Reset : fd = %d\n", fd);

	iRetCode = ioctl(fd, CIOC_RESET, 0);
	if (iRetCode < 0)
	{
		UT_ErrorLog(GTL, " TPCI_Reset : Unable to reset device, error %d\n", iRetCode);
		iRetCode = GTL_TPCI_ERR_DEVLVL;
	}

	UT_Log(GTL, INFO, "TPCI_Reset: EXIT\n");
	return iRetCode;
}

/************************** TPCI_Write *************************************/
/*!
 *    This function writes data to the device.\n
 *    1. Check the message type(Bootload/Control)\n
 *	  2. Form the DCM level message \n	
 *    3. Send the message to the device.\n
 *
 *    \return   On successful completion, it returns SUCCESS\n
 *              On failure, it returns GTL_TPCI_ERR_NULL_POINTER_PASSED \n
 *		or GTL_TPCI_ERR_UNKNOWN \n
 *    \param1   pvDevAddrInfo -> device Addressing Info.\n
 *    \param2   pstMsg -> message to be sent to the device.\n
 */
/***************************************************************************/
static S32 TPCI_Write(IN void *pvAdapterSpecInfo, IN gtl_msg_t *pstMsg)
{
	int fd = *((int *)pvAdapterSpecInfo);
	S32 iRetCode = SUCCESS;
	U16 pci_msg[(MAX_FIFO_PCI_SIZE + sizeof(gtl_mailbox_t))/2];	/* PCI driver message the first 8 bytes are the mailbox */
	int i;

	UT_Log(GTL, INFO, "TPCI_Write: ENTRY\n");

	if (pstMsg == NULL)
	{
		UT_ErrorLog(GTL, " TPCI_Write : GTL Message is NULL \n");
		iRetCode = GTL_TPCI_ERR_NULL_POINTER_PASSED;
		goto out;
	}

	if (pstMsg->type & BOOT)
	{
		/* fill the mailbox with values comming from VAPI */
		pci_msg[3] = pstMsg->mailbox.reg3;
		pci_msg[2] = pstMsg->mailbox.reg2;
		pci_msg[1] = pstMsg->mailbox.reg1;
		pci_msg[0] = pstMsg->mailbox.reg0;
	}
	else
	{
		/* fill the mailbox with appropriated parameters */
		pci_msg[3] = MAILBOX_BMR_CMD;
		pci_msg[2] = pstMsg->fifo_size;
		pci_msg[1] = pstMsg->channel;
		pci_msg[0] = 0;
	}
	
	/* fill the fifo with values comming from VAPI (if any) */
	if (pstMsg->fifo_size)
	{
		for (i = 0; i < pstMsg->fifo_size/2; i++)
		{
			pci_msg[i + 4] = UT_LE2CPU16(*((U16 *) pstMsg->fifo + i));
		}
	}
	
	/* write the message to the PCI driver */
	iRetCode = write(fd, pci_msg, sizeof(gtl_mailbox_t) + pstMsg->fifo_size);
	if (iRetCode != sizeof(gtl_mailbox_t) + pstMsg->fifo_size)
	{
		iRetCode = GTL_TPCI_ERR_DEVLVL;
		UT_ErrorLog(GTL, " TPCI_Write: write failed - driver returned %d while %d was expected\n",
			iRetCode, sizeof(gtl_mailbox_t) + pstMsg->fifo_size);
	}
	else
		iRetCode = SUCCESS;

out:
	UT_Log(GTL, INFO, "TPCI_Write: EXIT status %d\n", iRetCode);
	return iRetCode; 
}

/************************** TPCI_Shutdown *************************************/
/*!
 *    This function performs the shutdown of PCI adapter.\n
 *    1. Closes all the unclosed devices \n
 *    2. Shuts down the DCM module.\n
 *    \return   On successful completion, it returns SUCCESS\n
 *              On failure, it returns GTL_TPCI_ERR_SHUTDOWN\n
 */
/***************************************************************************/
static S32 TPCI_Shutdown(void)
{
	int i, *arg;
	S32 iRetCode = SUCCESS;
	
	UT_Log(GTL, INFO, "TPCI_Shutdown : ENTRY\n");
	
	if (pTPCIMutex == NULL)
	{
		iRetCode = GTL_TPCI_ERR_INITIALIZATION;
		UT_Log(GTL, INFO, "TPCI_Shutdown : Error pTPCIMutex = NULL\n");
		return iRetCode;
	}
	else
		UT_SemWait(pTPCIMutex, MAX_TIMEOUT);

	for (i = 0; i < MAX_PCI_DEVICE; i++)
		if (devices[i].fd >= 0)
		{
			arg = UT_AllocMem(sizeof(*arg));
			*arg = devices[i].fd;
			if (TPCI_Close((void *)arg) != SUCCESS)
				iRetCode = GTL_TPCI_ERR_DEVLVL;
		}

	UT_SemDestroy(pTPCIMutex);
	pTPCIMutex = NULL;

	UT_Log(GTL, INFO, "TPCI_Shutdown : EXIT\n");
	return iRetCode;
}

/************************** TPCI_RcvHandler *************************************/
/*!
 *    This function handles the data rreceived from the Comcerto device \n
 *
 *    \param   file descriptor of the PCI device.
 */
/***************************************************************************/
static void TPCI_RcvHandler(int fd)
{
	int retCode = SUCCESS;
	gtl_msg_t *pstMsg = NULL;
	U16 pci_msg[(MAX_FIFO_PCI_SIZE + sizeof(gtl_mailbox_t))/2];
	int vapi_device;
	int i;

	UT_Log(GTL,INFO,"TPCI_RcvHandler : ENTRY\n");

	pstMsg = UT_Calloc(1, sizeof(gtl_msg_t));
	if (pstMsg == NULL)
	{
		UT_ErrorLog(GTL, "TPCI_RcvHandler: memory allocation failed\n");
		goto out;
	}

	pstMsg->fifo = NULL;

	retCode = read(fd, (void*)pci_msg, sizeof(pci_msg));

	if (retCode < 0)
	{
		UT_ErrorLog(GTL, "TPCI_RcvHandler: read error %d\n", retCode);
		goto err; 
	}

	if (retCode == 0)
	{
		UT_Log(GTL,INFO,"TPCI_RcvHandler: no data\n");
		goto out; 
	}

	pstMsg->fifo_size = retCode - sizeof(gtl_mailbox_t);

	if (pstMsg->fifo_size != 0)
	{
		pstMsg->fifo = UT_AllocMem(pstMsg->fifo_size);
		if (pstMsg->fifo == NULL)
		{
			UT_ErrorLog(GTL, "TPCI_RcvHandler: memory allocation failed\n");
			goto err;
		}
	}

	/*The device reports BRM_READY after it has been reset.
	The ack has been performed by the driver so just exit*/
	if ((pci_msg[3] >> 8) == PCI_BRM_READY)
	{
		UT_Log(GTL,INFO,"TPCI_RcvHandler: BRM_READY\n");
		goto err;
	}
	/*If it is a command response or indication from the MSP  */
	else if (pci_msg[3] == MAILBOX_BRM_RESP)
	{
		pstMsg->channel = *(U16*) &pci_msg[1];
		if (pstMsg->fifo_size != *(U16*) &pci_msg[2])
		{
			UT_ErrorLog(GTL, "TPCI_RcvHandler: message FIFO size mismatch (%d %d)\n", pstMsg->fifo_size, *(U16*) &pci_msg[2]);
			goto err;
		}

		if (pstMsg->fifo)
		{
			for (i = 0; i < pstMsg->fifo_size/2; i++)
			{
				*((U16 *) pstMsg->fifo + i) = UT_CPU2LE16(pci_msg[i + 4]);
			}
		}

		pstMsg->type = PCI;
		UT_Log(GTL,INFO,"TPCI_RcvHandler: received %d bytes from device\n",pstMsg->fifo_size); 
	}
	/*If it is a boot command response or error  */
	else if (((pci_msg[3] >> 8) == CMD_TYPE_BRM_CMD_ACK) ||
			((pci_msg[3] >> 8) == CMD_TYPE_BRM_ERRIND))
	{
		/* it is a Boot ack or error command, set the msg as PCI and boot flags */
		pstMsg->mailbox.reg0 = pci_msg[0];
		pstMsg->mailbox.reg1 = pci_msg[1];
		pstMsg->mailbox.reg2 = pci_msg[2];
		pstMsg->mailbox.reg3 = pci_msg[3];

		pstMsg->type = BOOT | PCI;

		UT_Log(GTL,INFO,"TPCI_RcvHandler: mailbox.reg3 = 0x%04x, mailbox.reg2 = 0x%04x,mailbox.reg1 = 0x%04x,mailbox.reg0 = 0x%04x\n", 
				pstMsg->mailbox.reg3, pstMsg->mailbox.reg2, pstMsg->mailbox.reg1, pstMsg->mailbox.reg0);
	}
	else
	{
		UT_ErrorLog(GTL, "TPCI_RcvHandler : unhandled message from the MSP\n");
		goto err;
	}

	vapi_device = fd_lookup_device_id(fd);
	if (vapi_device == -1)
	{
		UT_Log(GTL,INFO,"TPCI_RcvHandler : Can't get VAPI device ID for %d\n", fd);
		goto err;
	}

	UT_SemWait(pTPCIMutex, MAX_TIMEOUT);
	UT_Log(GTL,INFO,"TPCI_RcvHandler : Calling callback func for dev %d\n", vapi_device);
	gtl_read_callback(vapi_device, pstMsg);
	UT_SemPost(pTPCIMutex);
	
out:
	return;

err:
	if (pstMsg)
	{
		if (pstMsg->fifo != NULL)
			UT_FreeMem(pstMsg->fifo);
		UT_FreeMem(pstMsg);
	}

	return;
}

/************************** TPCI_RxThread *************************************/
/*!
 *    This function polls the PCI driver for data \n
 *
 *    \param   file descriptor of the PCI device.
 */
/***************************************************************************/
static void *TPCI_RxThread(void *dummy)
{
	int fd = (int) dummy;

	UT_Log(GTL,INFO,"TPCI_RxThread : ENTRY, fd = %d\n", fd);

	while (True)
		TPCI_RcvHandler(fd);

	return NULL;
}

/************************** TPCI_Init *************************************/
/*!
 *    This function should be called by GTL Mapper at start up \n
 *    before issuing any other command to initialize internal data. \n
 *
 *    \return   On successful completion, it returns SUCCESS\n
 *              On failure, it returns GTL_TPCI_ERR_NULL_POINTER_PASSED\n
 *		or GTL_TPCI_ERR_ALREADY_INITIALIZED\n 
 *		or GTL_TPCI_ERR_INITIALIZATION.\n
 *    \param    pfnReadCb -> pointer to the client callback function. \n
 *              On reception of a message from MSP, GTL invokes this\n
 *              callback function to deliver the message to the\n
 *              GTL client.\n
 *    \param    pstAdapterFns -> OUT param to be filled up with adapter funcptrs\n
 */
/***************************************************************************/
S32 TPCI_Init(IN GTL_READ_CB pfnReadCb, OUT STAdapterFns *pstAdapterFns, 
		SGtlDeviceInfo *pstDeviceList, U32 uiNumDevices)
{
	int i;
	S32 iRetCode = SUCCESS;

	UT_Log (GTL, INFO, "TPCI_Init: ENTRY\n");

	if (pTPCIMutex != NULL)
	{
		UT_Log(GTL, INFO, " TPCI_Init : TPCI initialized.....\n");
		iRetCode = GTL_TPCI_ERR_ALREADY_INITIALIZED;
		goto err0;
	}

	if (pfnReadCb == NULL || pstAdapterFns == NULL)
	{
		UT_ErrorLog(GTL, " TPCI_Init : invalid parameters passed.\n");
		iRetCode = GTL_TPCI_ERR_NULL_POINTER_PASSED;
		goto err0;
	}

	pTPCIMutex = UT_SemInit(1);
	if (pTPCIMutex == NULL)
	{
		iRetCode = GTL_TPCI_ERR_INITIALIZATION;
		UT_Log(GTL, INFO, "TPCI_Init : Error during Semaphore(pTPCIMutex) create\n");
		goto err0;
	}

	UT_SemWait(pTPCIMutex, MAX_TIMEOUT);

	/* set all device file descriptors and ids to -1 */
	for (i = 0; i < MAX_PCI_DEVICE; i++)
	{
		devices[i].fd = -1;
		devices[i].reader_tid = NULL;
	}

	gtl_read_callback = pfnReadCb;
	pstAdapterFns->pfnAdapterOpen = TPCI_Open;
	pstAdapterFns->pfnAdapterClose = TPCI_Close;
	pstAdapterFns->pfnAdapterReset = TPCI_Reset;
	pstAdapterFns->pfnAdapterWrite = TPCI_Write;
	pstAdapterFns->pfnAdapterShutdown = TPCI_Shutdown;

	UT_SemPost(pTPCIMutex);

	UT_Log(GTL, INFO, "TPCI_Init: EXIT\n");

	return SUCCESS;

err0:
	UT_Log(GTL, INFO, "TPCI_Init: EXIT\n");
	return iRetCode;
}
