/*
 * comcerto_pci2eth.c
 * PCI-to-Ehernet bridge. Virtual ethernet interface implementation.
 *
 * Copyright (C) 2010 Mindspeed Technologies
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <linux/kernel.h>
#include <linux/etherdevice.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/version.h>

#include "comcerto_pci.h"
#include "comcerto_pci2eth.h"


#define RX_IND_TIMEOUT_MS	1		/* no wait before rx notification */
#define RX_IND_PACKETS		92		/* notify after 1 packet is ready */

#define	FRAME_BUF_SIZE		1664

#define TX_FRAMES_MAX		((128-4) / 3)	/* max number of tx packets per command FIFO == 41 */
#define TX_WAKE_TIMEOUT_MS	10		/* tx queue stop timeout on underruns */

#define RX_FRAMES_MAX		(128-4-1)	/* max number of rx packets per command FIFO == 123 */

#define RX_BUFFERS		RX_FRAMES_MAX
#define TX_BUFFERS		(TX_FRAMES_MAX * 2)

#define RX_BUFFERS_SIZE		((RX_BUFFERS*FRAME_BUF_SIZE + PAGE_SIZE - 1) & PAGE_MASK)
#define TX_BUFFERS_SIZE		((TX_BUFFERS*FRAME_BUF_SIZE + PAGE_SIZE - 1) & PAGE_MASK)
#define DMA_BUFFER_SIZE		(RX_BUFFERS_SIZE + TX_BUFFERS_SIZE)

struct pci2eth {
	struct pci_dev		*pci_dev;
	struct net_device	*net_dev;

	void			*buf_virt;
	dma_addr_t		buf_phys;

	void			*tx_buf;
	dma_addr_t		tx_buf_phys;
	u16			tx_mail[2][4];
	u16			tx_fifo[2][4 + TX_FRAMES_MAX*3];
	u8			tx_fifo_active;		/* active tx_fifo index */
	u8			tx_sent;		/* != when command delivery was scheduled */
	spinlock_t		tx_spin;

	void			*rx_buf;
	dma_addr_t		rx_buf_phys;
	u16			rx_mail[4];
	u16			rx_fifo[5];
	u16			rx_available;
	u16			rx_sizes[RX_BUFFERS];	/* saved packet sizes, corresponding to the same entries in ::rx_buf */

	struct net_device_stats	stats;

	struct work_struct	work_tx, work_rx, work_rx_setup;
	struct workqueue_struct	*work_queue;

	u8			rx_setup_state;
	u32			rx_setup_offs;

	u8			tx_started;
	u8			mac_addr[6];
	u8			mac_addr_set;
};


static int iface_open(struct net_device *dev)
{
	netif_start_queue(dev);

	return 0;
}

static int iface_stop(struct net_device *dev)
{
	netif_stop_queue(dev);

	return 0;
}

static int iface_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct pci2eth *pci2eth = netdev_priv(dev);
	u32 addr, offs;
	u16 *fifo, size;
	ulong flags;

	dev_dbg(&pci2eth->net_dev->dev, "xmit: %u bytes\n", skb->len);

	spin_lock_irqsave(&pci2eth->tx_spin, flags);

	fifo = &pci2eth->tx_fifo[pci2eth->tx_fifo_active][0];

	size = fifo[0] & 0xFF;	/* we hold packets counter here until message is going to be delivered */
	if (size >= TX_FRAMES_MAX) {
		pci2eth->stats.tx_fifo_errors++;

		netif_stop_queue(dev);

		spin_unlock_irqrestore(&pci2eth->tx_spin, flags);

		return NETDEV_TX_BUSY;
	}

	offs = (pci2eth->tx_fifo_active*TX_FRAMES_MAX + size)*FRAME_BUF_SIZE;
	memcpy(pci2eth->tx_buf + offs, skb->data, skb->len);

	/* update control packet FIFO - address/size and packets counter */
	addr = pci2eth->tx_buf_phys + offs;

	fifo[4 + size*3    ] = addr;
	fifo[4 + size*3 + 1] = addr >> 16;
	fifo[4 + size*3 + 2] = skb->len | 0x8000;

	fifo[0] = (fifo[0] & 0xFF00) | (size + 1);

	if (!pci2eth->tx_started) {
		pci2eth->tx_started = 1;
		queue_work(pci2eth->work_queue, &pci2eth->work_tx);
	}

	spin_unlock_irqrestore(&pci2eth->tx_spin, flags);

	dev_kfree_skb(skb);

	dev->trans_start = jiffies;

	return NETDEV_TX_OK;
}

static struct net_device_stats *iface_get_stats(struct net_device *dev)
{
	struct pci2eth *pci2eth = netdev_priv(dev);

	return &pci2eth->stats;
}

static void pci2eth_rx_setup(struct work_struct *work)
{
	struct pci2eth *pci2eth = container_of(work, struct pci2eth, work_rx_setup);
	u16 fifo[128] = {
		0x0000,					/* 0: index & length */
		0x0600,					/* 1: class & type */
		FC_ETHPCI_SET_RX_BUF_PARAMS,		/* 2: ETHPCI_SET_RX_BUF_PARAMS - define rx parameters */
		0x0000,					/* 3: reserved */
		(RX_IND_TIMEOUT_MS<<8) | RX_IND_PACKETS,/* 4: timeout & number of packets */
		(RX_BUFFERS*FRAME_BUF_SIZE + PAGE_SIZE-1)/PAGE_SIZE,		/* 5: total buffer size, pages */
		FRAME_BUF_SIZE,				/* 6: max packet size */
		PAGE_SIZE/32,				/* 7: page size, in 32-bytes units */
	};
	u16 mail[4] = {
		0x0000,		/* 0: reserved */
		0xFFFF,		/* 1: channel */
		0x0000,		/* 2: length */
		0x0020,		/* 3: PCI-to-Ethernet bridge data packet */
	};
	int i, count;

	pci2eth->rx_setup_state++;

	if (pci2eth->rx_setup_state == 1) {
		comcerto_pci_sendmsg(pci2eth->pci_dev, mail, fifo, 8*2);

		pci2eth->rx_setup_offs = 0;

		return;
	}

	fifo[2] = FC_ETHPCI_SET_RX_BUF_MAP;

	fifo[4] = pci2eth->rx_setup_offs;
	fifo[5] = pci2eth->rx_setup_offs >> 16;

	count = (RX_BUFFERS*FRAME_BUF_SIZE + PAGE_SIZE-1)/PAGE_SIZE - pci2eth->rx_setup_offs/PAGE_SIZE;
	if (count <= 0) {
		pci2eth->rx_setup_offs = 0;
		netif_carrier_on(pci2eth->net_dev);
		netif_start_queue(pci2eth->net_dev);
		dev_info(&pci2eth->net_dev->dev, "link is up\n");
		return;
	}

	if (count > (128-6)/3)
		count = (128-6)/3;

	for (i = 0; i < count; pci2eth->rx_setup_offs += PAGE_SIZE, i++) {
		fifo[6 + i*3] = pci2eth->rx_buf_phys + pci2eth->rx_setup_offs;
		fifo[7 + i*3] = (pci2eth->rx_buf_phys + pci2eth->rx_setup_offs) >> 16;
		fifo[8 + i*3] = PAGE_SIZE/32;
	}

	comcerto_pci_sendmsg(pci2eth->pci_dev, mail, fifo, (6 + count*3)*2);
}

static void pci2eth_rx(struct work_struct *work)
{
	struct pci2eth *pci2eth = container_of(work, struct pci2eth, work_rx);

	comcerto_pci_sendmsg(pci2eth->pci_dev, pci2eth->rx_mail, pci2eth->rx_fifo, sizeof(pci2eth->rx_fifo));
}

static void pci2eth_rx_ind(struct pci2eth *pci2eth, u16 *ack_fifo, int len)
{
	int i;

	len = (len - 10) / 2;	/* number of pending rx packets */
	if (len < 1) {
		dev_err(&pci2eth->net_dev->dev, "invalid indication, number of frames %d\n", len);
		BUG();
	}

	if (pci2eth->rx_available) {
		dev_err(&pci2eth->net_dev->dev, "invalid indication, no ack received\n");
		BUG();
	}

	pci2eth->rx_available = len;

	for (i = 0; i < len; i++)
		pci2eth->rx_sizes[i] = ack_fifo[4 + 1 + i];	/* save packet size to be used after packet is transferred to host */

	queue_work(pci2eth->work_queue, &pci2eth->work_rx);
}

static void pci2eth_rx_ack(struct pci2eth *pci2eth, u16 *ack_fifo)
{
	int i, size;
	void *buf;
	struct sk_buff *skb;

	for (i = 0; i < pci2eth->rx_available; i++) {
		buf = pci2eth->rx_buf + i*FRAME_BUF_SIZE;
		size = pci2eth->rx_sizes[i];
		if (size < 14 || size > 1536) {
			dev_err(&pci2eth->pci_dev->dev, "invalid size %d, skipping packet %d of %d\n", size, i, pci2eth->rx_available);
			continue;
		}
		
		skb = dev_alloc_skb(size + 2);
		if (skb) {
			skb_reserve(skb, 2);

			memcpy(skb_put(skb, size), buf, size);

			skb->dev = pci2eth->net_dev;
			skb->protocol = eth_type_trans(skb, skb->dev);
			if (netif_rx(skb) == NET_RX_DROP) {
				pci2eth->stats.rx_dropped++;
			} else {
				pci2eth->stats.rx_packets++;
				pci2eth->stats.rx_bytes += size;
				skb->dev->last_rx = jiffies;
			}
		} else {
			if (net_ratelimit())
				dev_err(&pci2eth->pci_dev->dev, "low memory - packet dropped\n");

			pci2eth->stats.rx_dropped++;
		}
	}

	pci2eth->rx_available = 0;
}

static void pci2eth_tx(struct work_struct *work)
{
	struct pci2eth *pci2eth = container_of(work, struct pci2eth, work_tx);
	u16 *mail, *fifo, len;
	ulong flags;

	spin_lock_irqsave(&pci2eth->tx_spin, flags);

	mail = &pci2eth->tx_mail[pci2eth->tx_fifo_active][0];
	fifo = &pci2eth->tx_fifo[pci2eth->tx_fifo_active][0];

	/* note - we don't set the actual length into FIFO since it will be done by comcerto_pci_sendmsg() anyway */
	len = fifo[0] & 0xFF;
	if (len)
		pci2eth->tx_fifo_active ^= 1;

	spin_unlock_irqrestore(&pci2eth->tx_spin, flags);

	if (len)
		comcerto_pci_sendmsg(pci2eth->pci_dev, mail, fifo, (4 + len*3)*2);
}

static void pci2eth_tx_ack(struct pci2eth *pci2eth, u16 *ack_fifo)
{
	u16 *fifo, passive = pci2eth->tx_fifo_active ^ 1;
	int i, req, res, packets, bytes;

	fifo = &pci2eth->tx_fifo[passive][0];	/* original FIFO - holds packet sizes and requested packets counter */
	res = ((ack_fifo[0] & 0xFF) - 8) / 2;	/* number of packets that were sent */
	req = ((fifo[0] & 0xFF) - 8) / 6;	/* number of packets that were requested to send */
	fifo[0] &= 0xFF00;			/* clean up fifo entry */

	/* check for errors and count good packets & bytes */
	for (i = 0, packets = 0, bytes = 0; i < res; i++)
		if (ack_fifo[4 + i] == 0) {
			bytes += fifo[4 + 2 + i*3] & 0x7FF;
			packets++;
		}

	/* update tx statistics */
	pci2eth->stats.tx_packets += packets;
	pci2eth->stats.tx_bytes += bytes;

	pci2eth->stats.tx_errors += res - packets;	/* count all packets that don't have good status */
	pci2eth->stats.tx_fifo_errors += res - packets;	/* count them as fifo errors to differ from drops */

	pci2eth->stats.tx_errors += req - res;		/* MSP stops on first error - the rest is always dropped */
	pci2eth->stats.tx_dropped += req - res;

	spin_lock(&pci2eth->tx_spin);

	if (netif_carrier_ok(pci2eth->net_dev) && netif_queue_stopped(pci2eth->net_dev)) {
		netif_wake_queue(pci2eth->net_dev);
	}

	fifo = &pci2eth->tx_fifo[pci2eth->tx_fifo_active][0];	/* active FIFO - holds packets sent when we waited for ack */
	if (fifo[0] & 0xFF)
		queue_work(pci2eth->work_queue, &pci2eth->work_tx);
	else
		pci2eth->tx_started = 0;

	spin_unlock(&pci2eth->tx_spin);
}

void *pci2eth_open(struct pci_dev *pci_dev)
{
	struct net_device *net_dev;
	struct pci2eth *pci2eth;
	int i;

	if (!pci_dev)
		BUG();

	net_dev = alloc_etherdev(sizeof(struct pci2eth));
	BUG_ON(!net_dev);

	SET_NETDEV_DEV(net_dev, &pci_dev->dev);

	pci2eth = netdev_priv(net_dev);
	pci2eth->buf_virt = pci_alloc_consistent(pci_dev, DMA_BUFFER_SIZE, &pci2eth->buf_phys);
	if (!pci2eth->buf_virt) {
		free_netdev(net_dev);
		dev_err(&pci_dev->dev, "failed to allocate DMA buffer\n");
		return NULL;
	}

	pci2eth->rx_buf = pci2eth->buf_virt;
	pci2eth->rx_buf_phys = pci2eth->buf_phys;
	pci2eth->tx_buf = pci2eth->rx_buf + RX_BUFFERS_SIZE;
	pci2eth->tx_buf_phys = pci2eth->rx_buf_phys + RX_BUFFERS_SIZE;
	for (i = 0; i < 2; i++) {
		pci2eth->tx_mail[i][1] = 0xFFFF;			/* channel */
		pci2eth->tx_mail[i][3] = 0x0020;			/* PCI-to-Ethernet bridge command */

		pci2eth->tx_fifo[i][1] = 0x0910;			/* class & type */
		pci2eth->tx_fifo[i][2] = FC_ETHPCI_XFER_TX_PACKET;	/* function code */
	}

	pci2eth->rx_mail[1] = 0xFFFF;
	pci2eth->rx_mail[3] = 0x0020;

	pci2eth->rx_fifo[1] = 0x0911;
	pci2eth->rx_fifo[2] = FC_ETHPCI_XFER_RX_PACKET;
	pci2eth->rx_fifo[4] = 0;					/* buffer index, always zero */

	spin_lock_init(&pci2eth->tx_spin);

	pci2eth->pci_dev = pci_dev;
	pci2eth->net_dev = net_dev;
	pci2eth->work_queue = create_singlethread_workqueue("pci2eth");
	BUG_ON(!pci2eth->work_queue);

	INIT_WORK(&pci2eth->work_tx, pci2eth_tx);
	INIT_WORK(&pci2eth->work_rx, pci2eth_rx);
	INIT_WORK(&pci2eth->work_rx_setup, pci2eth_rx_setup);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,29)
	net_dev->open = iface_open;
	net_dev->stop = iface_stop;
	net_dev->hard_start_xmit = iface_start_xmit;
	net_dev->get_stats = iface_get_stats;
#else
	{
		static const struct net_device_ops net_dev_ops = {
			.ndo_open = iface_open,
			.ndo_stop = iface_stop,
			.ndo_start_xmit = iface_start_xmit,
			.ndo_get_stats = iface_get_stats,
		};

		net_dev->netdev_ops = &net_dev_ops;
	}
#endif
	net_dev->irq = pci_dev->irq;

	net_dev->flags &= ~IFF_MULTICAST;

	netif_carrier_off(net_dev);
	netif_stop_queue(net_dev);
	register_netdev(net_dev);

	return net_dev;
}

int pci2eth_close(void *arg)
{
	struct net_device *net_dev = arg;
	struct pci2eth *pci2eth;

	if (!arg)
		BUG();

	pci2eth = netdev_priv(net_dev);

	iface_stop(net_dev);

	unregister_netdev(net_dev);

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21)
	cancel_work_sync(&pci2eth->work_tx);
	cancel_work_sync(&pci2eth->work_rx);
	cancel_work_sync(&pci2eth->work_rx_setup);
#endif
	destroy_workqueue(pci2eth->work_queue);

	pci_free_consistent(pci2eth->pci_dev, DMA_BUFFER_SIZE, pci2eth->buf_virt, pci2eth->buf_phys);

	free_netdev(net_dev);

	return 0;
}

static void pci2eth_stop(struct pci2eth *pci2eth)
{
	netif_carrier_off(pci2eth->net_dev);
	iface_stop(pci2eth->net_dev);

	pci2eth->rx_setup_state = 0;
	pci2eth->tx_started = 0;

	dev_info(&pci2eth->net_dev->dev, "link is down\n");
}

static void pci2eth_prepare_mac_addr(struct pci2eth *pci2eth, u16 *fifo)
{
	if (fifo[4] != 0x0001)	/* register Ethernet IP protocol header */
		return;

	memcpy(pci2eth->mac_addr, &fifo[8], 6);
	if (!pci2eth->mac_addr_set)
		pci2eth->mac_addr_set = 1;
}

static void pci2eth_set_mac_addr(struct pci2eth *pci2eth, u16 *fifo)
{
	struct net_device *net_dev = pci2eth->net_dev;

	if (fifo[4] != 0x0000)	/* whether header was set ok */
		return;

	if (pci2eth->mac_addr_set) {
		dev_info(&net_dev->dev, "setting MAC address: %02X.%02X.%02X.%02X.%02X.%02X\n",
			pci2eth->mac_addr[0], pci2eth->mac_addr[1], pci2eth->mac_addr[2],
			pci2eth->mac_addr[3], pci2eth->mac_addr[4], pci2eth->mac_addr[5]);
		memcpy(net_dev->dev_addr, pci2eth->mac_addr, ETH_ALEN);

		pci2eth->rx_setup_state = 0;

		if (pci2eth->mac_addr_set == 1)
			pci2eth->mac_addr_set = 2;
		else
			pci2eth->rx_setup_state = 1;

		queue_work(pci2eth->work_queue, &pci2eth->work_rx_setup);
	}
}

void pci2eth_control(void *arg, u16 *fifo, int fifo_len)
{
	struct net_device *net_dev = arg;
	struct pci2eth *pci2eth;

	if (!arg)
		BUG();

	if (fifo_len < 8) {
		dev_err(&net_dev->dev, "skipping malformed control message, fifo len: %d\n", fifo_len);
		return;
	}

	pci2eth = netdev_priv(net_dev);

	dev_dbg(&pci2eth->pci_dev->dev, "control, CT: %04x, FC %04x, len: %u\n", fifo[1], fifo[2], fifo_len);

	switch (fifo[2]) {
	case FC_ETHPCI_RX_PACKET_IND:
		pci2eth_rx_ind(pci2eth, fifo, fifo[0]&255);
		break;

	case FC_ETHPCI_XFER_TX_PACKET:
		pci2eth_tx_ack(pci2eth, fifo);
		break;

	case FC_ETHPCI_XFER_RX_PACKET:
		pci2eth_rx_ack(pci2eth, fifo);
		break;

	case FC_ETHPCI_SET_RX_BUF_PARAMS:
	case FC_ETHPCI_SET_RX_BUF_MAP:
		queue_work(pci2eth->work_queue, &pci2eth->work_rx_setup);
		break;

	case FC_SET_ETH_HDR:
		if (fifo[1] == CT_DEVICE_CONFIG_CHANGE)
			pci2eth_prepare_mac_addr(pci2eth, fifo);

		if (fifo[1] == CT_DEVICE_CONFIG_RESPONSE)
			pci2eth_set_mac_addr(pci2eth, fifo);
		break;

	case 0:
		if (fifo[1] == CT_SUPVSR_READY || fifo[1] == CT_ALERT) {
			pci2eth->rx_setup_state = 0;
			pci2eth->mac_addr_set = 0;
			if (netif_carrier_ok(pci2eth->net_dev))
				pci2eth_stop(pci2eth);
		}
		break;

	default:
		goto unknown;
	}

	return;

unknown:
	dev_err(&net_dev->dev, "skipping control message: len: %d, ct %04X, fc %04X\n", fifo_len, fifo[1], fifo[2]);
}
