/*
 * comcerto_pci.c
 * Comcerto PCI control driver.
 *
 * Copyright (C) 2010 Mindspeed Technologies
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/cdev.h>
#include <linux/kfifo.h>
#include <linux/poll.h>
#include <linux/uaccess.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/version.h>

#include "comcerto_pci.h"
#ifdef CONFIG_COMCERTO_PCI2ETH_BRIDGE
#include "comcerto_pci2eth.h"
#endif


MODULE_AUTHOR("Mindspeed Technologies");
MODULE_VERSION(COMCERTO_PCI_VERSION);
MODULE_DESCRIPTION("Comcerto M82xxx PCI device driver");
MODULE_LICENSE("GPL");


#define DRV_NAME		"comcerto_pci"

#define DEV_COUNT		8

#define COMCERTO_RX_QUEUE_MAX   336
#define COMCERTO_TX_QUEUE_MAX   336

struct comcerto_queue_msg {
	unsigned short fifo_len;
	u16 mailbox[4];
};

#define COMCERTO_RX_QUEUE_SIZE  (COMCERTO_RX_QUEUE_MAX * (sizeof(struct comcerto_queue_msg) + 256))
#define COMCERTO_TX_QUEUE_SIZE  (COMCERTO_TX_QUEUE_MAX * (sizeof(struct comcerto_queue_msg) + 256))

#define COMCERTO_TRACE_RMAGIC	0xFEF1F0F6
#define COMCERTO_TRACE_TMAGIC	0xFEF1F0F7

struct comcerto_trace {
	unsigned long magic;
	unsigned long timestamp;
	struct comcerto_queue_msg msg;
};

#define COMCERTO_TRACE_SIZE	PAGE_SIZE

#define IS_BRM_READY(m3, m2, m1, m0)	\
	(((m3) & 0xFF00) == 0x1200 && (m2) == 0x4321 && (m1) == 0x8765 && (m0) == 0xCBA9)

struct comcerto_dev {
	void *iomembase;
	void *plxmembase;
	struct pci_dev *pci_dev;
	struct cdev cdev;

	struct semaphore lock;

	spinlock_t rx_queue_lock;
	struct kfifo *rx_queue;
	wait_queue_head_t rx_proc_list;

	spinlock_t tx_queue_lock;
	struct kfifo *tx_queue;
	wait_queue_head_t tx_proc_list;
	
	spinlock_t mailbox_lock;

	unsigned char trace[COMCERTO_TRACE_SIZE];
	unsigned char *trace_out;

#ifdef CONFIG_COMCERTO_PCI2ETH_BRIDGE
	void *pci2eth;
	u8 tx_locked;			/* tx path lock, disable using mailboxes & FIFO during packets transfer */
#endif

	u8 irq_requested;
};


static struct class *comcerto_class;
static dev_t comcerto_pci_first_dev;
static dev_t comcerto_pci_dev;


static inline u16 comcerto_read_reg(struct comcerto_dev *dev, unsigned offset);
static inline void comcerto_write_reg(struct comcerto_dev *dev, u16 val, unsigned offset);
static int comcerto_reset(struct comcerto_dev *dev);
static void comcerto_start_tx(struct comcerto_dev *dev);
static irqreturn_t comcerto_irq_handler(int irq, void *dev_id);


static struct pci_device_id comcerto_ids[] = {
	{ PCI_VENDOR_ID_MINDSPEED, PCI_DEVICE_ID_MINDSPEED_M825XX,
		PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },
	{ PCI_VENDOR_ID_CONEXANT, PCI_DEVICE_ID_CONEXANT_M826XX,
		PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },
	{ PCI_VENDOR_ID_MINDSPEED, PCI_DEVICE_ID_MINDSPEED_M827XX,
		PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },
	{ PCI_VENDOR_ID_MINDSPEED, PCI_DEVICE_ID_MINDSPEED_M829XX,
		PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },
	{ PCI_VENDOR_ID_MINDSPEED, PCI_DEVICE_ID_MINDSPEED_M823V1,
		PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },
	{ PCI_VENDOR_ID_PLX, PCI_DEVICE_ID_PLX_9030,
		PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },
	{ PCI_VENDOR_ID_MINDSPEED, PCI_DEVICE_ID_MINDSPEED_M823V2,
		PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },
	{ 0, }
};
MODULE_DEVICE_TABLE(pci, comcerto_ids);


static void comcerto_stop(struct comcerto_dev *dev)
{
	ulong flags;

	local_irq_save(flags);

	comcerto_reset(dev);

	comcerto_write_reg(dev, 0, HOST_FIFO_CONTROL);		/* disable interrupts and FIFO operation */
	comcerto_write_reg(dev, 0xFF, HOST_FIFO_INTERRUPT_ACK);	/* ack everything that may be left */

	local_irq_restore(flags);

	if (dev->irq_requested) {
		free_irq(dev->pci_dev->irq, dev);
		dev->irq_requested = 0;
	}

	kfifo_reset(dev->rx_queue);
	kfifo_reset(dev->tx_queue);
}

static int comcerto_open(struct inode *inode, struct file *file)
{
	struct comcerto_dev *dev = container_of(inode->i_cdev, struct comcerto_dev, cdev);

	if (file->f_flags & O_NONBLOCK) {
		if (down_trylock(&dev->lock))
			return -EBUSY;
	} else {
		if (down_interruptible(&dev->lock))
			return -ERESTARTSYS;
	}

	if (!dev->irq_requested) {
		/* Request the IRQ */
		if (request_irq(dev->pci_dev->irq, comcerto_irq_handler, IRQF_SHARED, DRV_NAME, dev)) {
			dev_err(&dev->pci_dev->dev, "failed to request IRQ%d\n", dev->pci_dev->irq);
			up(&dev->lock);
			return -EBUSY;
		}

		/* Enable the RXM3 interrupt and FIFO operation */
		comcerto_write_reg(dev, RXM3IE | FIFO_EN, HOST_FIFO_CONTROL);

		dev->irq_requested = 1;
	}

	file->private_data = dev;

	return 0;
}

static int comcerto_release(struct inode *inode, struct file *file)
{
	struct comcerto_dev *dev = container_of(inode->i_cdev, struct comcerto_dev, cdev);

#ifndef CONFIG_COMCERTO_PCI2ETH_BRIDGE
	comcerto_stop(dev);
#endif

	up(&dev->lock);

	return 0;
}

static int comcerto_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	struct comcerto_dev *dev = file->private_data;
	struct pci_dev *pci_dev = dev->pci_dev;
	int ret = 0;
	struct comcerto_info info;
	u8 revision;

	switch(cmd)
	{
	case CIOC_RESET:
		ret = comcerto_reset(dev);
		break;
	
	case CIOC_GET_INFO:
		info.vendor = pci_dev->vendor;
		info.device = pci_dev->device;
		pci_read_config_byte(pci_dev, PCI_REVISION_ID, &revision);
		info.revision = revision;
		if (sizeof(info) != _IOC_SIZE(cmd)) {
			ret = -EINVAL;
			goto out;
		}

		if (copy_to_user((struct comcerto_info *) arg, &info, sizeof(info)))
			ret = -EFAULT;
		break;

	default:
		ret = -EINVAL;
	}

out:
	return ret;
}

static ssize_t comcerto_read(struct file *file, char __user *buf, size_t count, loff_t *pos)
{
	struct comcerto_dev *dev = file->private_data;
	struct comcerto_queue_msg msg;
	unsigned char fifo[256];
	ssize_t ret;
	ulong flags;

	/* We must read at least the mailbox */
	if (count < sizeof(msg.mailbox))
		return -EINVAL;

	/* Wait for the lower half to supply us with rx_queue messages */
	ret = wait_event_interruptible(dev->rx_proc_list, kfifo_len(dev->rx_queue) != 0);
	if (ret)
		goto out;

	spin_lock_irqsave(&dev->rx_queue_lock, flags);

	/* Get the message from the queue */
	__kfifo_get(dev->rx_queue, (unsigned char *) &msg, sizeof(msg));

	if (msg.fifo_len > sizeof(fifo)) {
		dev_err(&dev->pci_dev->dev, "RX message FIFO size (%d bytes) > expected maximum (%d bytes)\n", msg.fifo_len, sizeof(fifo));
		spin_unlock_irqrestore(&dev->rx_queue_lock, flags);
		ret = -EIO;
		goto out;
	}

	/* Check whether we have enough space for FIFO data */
	if (count < sizeof(msg.mailbox) + msg.fifo_len) {
		spin_unlock_irqrestore(&dev->rx_queue_lock, flags);
		ret = -EINVAL;
		goto out;
	}

	if (msg.fifo_len > 0)
		__kfifo_get(dev->rx_queue, fifo, msg.fifo_len);

	spin_unlock_irqrestore(&dev->rx_queue_lock, flags);

	/* Return the mailbox */
	if (copy_to_user(buf, &msg.mailbox, sizeof(msg.mailbox))) {
		ret = -EFAULT;
		goto out;
	}
	ret = sizeof(msg.mailbox);

	/* Return the FIFO data, if there was one */
	if (msg.fifo_len != 0) {
		ret += msg.fifo_len;

		if (copy_to_user(buf + sizeof(msg.mailbox), fifo, msg.fifo_len)) {
			ret = -EFAULT;
			goto out;
		}
	}

out:
	return ret;
}

static int comcerto_queue(struct comcerto_dev *dev, struct comcerto_queue_msg *msg, u16 *fifo)
{
	int err;
	u32 control;
	ulong flags;
	
	/* Wait for room in tx_queue */
	err = wait_event_interruptible(dev->tx_proc_list,
		((kfifo_len(dev->tx_queue) + sizeof(*msg) + msg->fifo_len) <= COMCERTO_TX_QUEUE_SIZE));
	if (err)
		return err;

	/* Lock tx_queue before adding message & FIFO data in two steps */
	spin_lock_irqsave(&dev->tx_queue_lock, flags);

	/* Insert msg into tx_queue */
	__kfifo_put(dev->tx_queue, (void*)msg, sizeof(*msg));

	/* Copy FIFO data into tx_queue */
	if (msg->fifo_len > 0)
		__kfifo_put(dev->tx_queue, (void*)fifo, msg->fifo_len);

	spin_unlock_irqrestore(&dev->tx_queue_lock, flags);

	/* Hold mailbox lock to protect device registers */
	spin_lock_irqsave(&dev->mailbox_lock, flags);

	/* Start output if necessary */
	control = comcerto_read_reg(dev, HOST_FIFO_CONTROL);
	if ((control & TXM3IE) == 0)
		comcerto_start_tx(dev);

	spin_unlock_irqrestore(&dev->mailbox_lock, flags);

	return 0;
}

int comcerto_pci_sendmsg(struct pci_dev *pci_dev, u16 *mail, u16 *fifo, int fifo_len)
{
	struct comcerto_dev *dev = pci_get_drvdata(pci_dev);
	struct comcerto_queue_msg msg;

	/* The FIFO write may be at most 256 bytes */
	if (fifo_len > 256) {
		dev_err(&dev->pci_dev->dev, "can't handle > 256 bytes of FIFO payload\n");
		return -EINVAL;
	}

	fifo[0] = (fifo[0] & 0xFF00) | fifo_len;

	/* 32-bit padding (required) */
	fifo_len = (fifo_len + 3) & ~3;

	mail[2] = fifo_len;

	memcpy(&msg.mailbox, mail, sizeof(msg.mailbox));
	msg.fifo_len = fifo_len;

	return comcerto_queue(dev, &msg, fifo);
}

static ssize_t comcerto_write(struct file *file, const char __user *buf, size_t count, loff_t *ppos)
{
	struct comcerto_dev *dev = file->private_data;
	struct comcerto_queue_msg msg;
	unsigned char fifo[256];
	ssize_t ret = count, err;

	/* We must write at least the mailbox */
	if (count < sizeof(msg.mailbox))
		return -EINVAL;

	/* Copy the mailbox */
	if (copy_from_user(&msg.mailbox, buf, sizeof(msg.mailbox)))
		return -EFAULT;

	buf += sizeof(msg.mailbox);
	count -= sizeof(msg.mailbox);

	/* The FIFO write may be at most 256 bytes */
	if (count > 256) {
		dev_err(&dev->pci_dev->dev, "can't handle > 256 bytes of FIFO payload\n");
		return -EINVAL;
	}

	/* Copy FIFO data to kernel space */
	if (count > 0) {
		if (copy_from_user(fifo, buf, count)) {
			return -EFAULT;
		}
	}

	msg.fifo_len = count;

	err = comcerto_queue(dev, &msg, (u16*)fifo);
	if (err)
		return err;

	return ret;
}

static unsigned int comcerto_poll(struct file *file, poll_table *wait)
{
	struct comcerto_dev *dev = file->private_data;
	unsigned int mask = 0;

	poll_wait(file, &dev->tx_proc_list, wait);
	poll_wait(file, &dev->rx_proc_list, wait);

	/* Could we read now? */
	if (kfifo_len(dev->rx_queue) != 0)
		mask |= POLLIN | POLLRDNORM;

	/* Could we write now? */
	if (kfifo_len(dev->tx_queue) <= COMCERTO_TX_QUEUE_SIZE - sizeof(struct comcerto_queue_msg))
		mask |= POLLOUT | POLLWRNORM;

	return mask;
}

struct file_operations comcerto_fops = {
	.owner = THIS_MODULE,
	.open = comcerto_open,
	.release = comcerto_release,
	.read = comcerto_read,
	.write = comcerto_write,
	.ioctl = comcerto_ioctl,
	.poll = comcerto_poll,
};

/* Trace buffer storage */
static void comcerto_trace_store(struct comcerto_dev *dev, unsigned long magic, struct comcerto_queue_msg *msgp,  unsigned char *fifop)
{
	size_t room;
	struct comcerto_trace trace;
	unsigned short fifo_len = msgp->fifo_len;

	/* Limit storage of FIFO data to 256 bytes */
	if (fifo_len > 256)
		fifo_len = 256;

	trace.magic = magic;
	trace.timestamp = jiffies;
	memcpy(&trace.msg, msgp, sizeof(trace.msg));

	/* Write the trace header to the circular buffer */
	room = &dev->trace[COMCERTO_TRACE_SIZE] - dev->trace_out;
	if (room > sizeof(trace)) {
		memcpy(dev->trace_out, &trace, sizeof(trace));
		dev->trace_out += sizeof(trace);
	} else {
		memcpy(dev->trace_out, &trace, room);
		dev->trace_out = dev->trace;
		memcpy(dev->trace_out, (unsigned char *)&trace + room, sizeof(trace) - room);
		dev->trace_out += sizeof(trace) - room;
	}

	/* Write the FIFO data to the circular buffer */
	room = &dev->trace[COMCERTO_TRACE_SIZE] - dev->trace_out;
	if (room > fifo_len) {
		memcpy(dev->trace_out, fifop, fifo_len);
		dev->trace_out += fifo_len;
	} else {
		memcpy(dev->trace_out, fifop, room);
		dev->trace_out = dev->trace;
		memcpy(dev->trace_out, fifop + room, fifo_len - room);
		dev->trace_out += fifo_len - room;
	}
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,23)
static ssize_t comcerto_trace_read(struct kobject *kobj, char *buffer, loff_t pos, size_t size)
#else
static ssize_t comcerto_trace_read(struct kobject *kobj, struct bin_attribute *attr, char *buffer, loff_t pos, size_t size)
#endif
{
	struct device *device = container_of(kobj, struct device, kobj);
	struct comcerto_dev *dev = dev_get_drvdata(device);

	if (pos > COMCERTO_TRACE_SIZE)
		return 0;

	if (pos + size > COMCERTO_TRACE_SIZE)
		size = COMCERTO_TRACE_SIZE - pos;

	memcpy(buffer, dev->trace + pos, size);

	return size;
}

struct bin_attribute comcerto_trace_attr = {
	.attr = {
	.name = "trace",
		.mode = S_IRUGO,
		.owner = THIS_MODULE,
	},
	.size = COMCERTO_TRACE_SIZE,
	.read = comcerto_trace_read,
};

static inline u16 comcerto_read_reg(struct comcerto_dev *dev, unsigned offset)
{
	u16 val;

	if (dev->pci_dev->device == PCI_DEVICE_ID_PLX_9030) {
		unsigned long reg = (u32) dev->iomembase + offset*2;

		val = readl((void*) reg) & 0xFF;
		val |= (readl((void*) (reg + 4)) & 0xFF) << 8;
	} else
		val = readl((void*)((u32) dev->iomembase + offset));

	return val;
}

static inline void comcerto_write_reg(struct comcerto_dev *dev, u16 val, unsigned offset)
{
	if (dev->pci_dev->device == PCI_DEVICE_ID_PLX_9030) {
		unsigned long reg = (u32)dev->iomembase + offset*2;

		writel(val & 0xFF, (void*) reg);
		writel((val >> 8) & 0xFF, (void*) (reg + 4));
	} else
		writel(val, (void*)((u32)dev->iomembase + offset));
}

static inline void comcerto_read_fifo(const struct comcerto_dev *dev, unsigned char *fifo, size_t fifo_len)
{
	if (dev->pci_dev->device == PCI_DEVICE_ID_MINDSPEED_M823V2) {
		size_t len = fifo_len/4;
		u32 *fifo32 = (void*) fifo, u;

		while (len--) {
			u = readl(dev->iomembase + HOST_RX_FIFO_DATA);
#ifdef __BIG_ENDIAN
			u = (u >> 16) | (u << 16);	/* swap 16-bit words for BE platforms, to store words in correct order */
#endif
			*fifo32++ = u;
		}
	} else if (dev->pci_dev->device != PCI_DEVICE_ID_PLX_9030) {
		size_t len = fifo_len/2;
		u16 *fifo16 = (void*) fifo;

		while (len--)
			*fifo16++ = readw(dev->iomembase + HOST_RX_FIFO_DATA);
	} else {
		while (fifo_len--)
			*fifo++ = readl(dev->iomembase + HOST_RX_FIFO_DATA*4);
	}
}

static inline void comcerto_write_fifo(const struct comcerto_dev *dev, const unsigned char *fifo, size_t fifo_len)
{
	if (dev->pci_dev->device == PCI_DEVICE_ID_MINDSPEED_M823V2) {
		size_t len = fifo_len/4;
		u32 *fifo32 = (void*) fifo, u;

		while (len--) {
			u = *fifo32++;
#ifdef __BIG_ENDIAN
			u = (u >> 16) | (u << 16);	/* swap 16-bit words for BE platforms, to pass words in correct order */
#endif
			writel(u, dev->iomembase + HOST_TX_FIFO_DATA);
		}
	} else if (dev->pci_dev->device != PCI_DEVICE_ID_PLX_9030) {
		size_t len = fifo_len/2;
		u16 *fifo16 = (void*) fifo;

		while (len--)
			writew(*fifo16++, dev->iomembase + HOST_TX_FIFO_DATA);
	} else {
		while (fifo_len--)
			writel(*fifo++, dev->iomembase + HOST_TX_FIFO_DATA*4);
	}
}

static ssize_t comcerto_host_registers_show(struct device *device, struct device_attribute *attr, char *buf)
{
	struct comcerto_dev *dev = dev_get_drvdata(device);
	unsigned long flags;
	ssize_t ret;

	/* Get exclusive access to FIFO and mailbox */
	spin_lock_irqsave(&dev->mailbox_lock, flags);

	ret = snprintf(buf, PAGE_SIZE,
			"FIFO_CONTROL		= %04x\n"
			"FIFO_STATUS		= %04x\n"
			"TX_FIFO_SIZE		= %04x\n"
			"TX_FIFO_HIGH_THRESHOLD	= %04x\n"
			"TX_FIFO_LOW_THRESHOLD	= %04x\n"
			"RX_FIFO_SIZE		= %04x\n"
			"RX_FIFO_HIGH_THRESHOLD	= %04x\n"
			"RX_FIFO_LOW_THRESHOLD	= %04x\n",
			comcerto_read_reg(dev, HOST_FIFO_CONTROL),
			comcerto_read_reg(dev, HOST_FIFO_STATUS),
			comcerto_read_reg(dev, HOST_TX_FIFO_SIZE),
			comcerto_read_reg(dev, HOST_TX_FIFO_HIGH_THRESHOLD),
			comcerto_read_reg(dev, HOST_TX_FIFO_LOW_THRESHOLD),
			comcerto_read_reg(dev, HOST_RX_FIFO_SIZE),
			comcerto_read_reg(dev, HOST_RX_FIFO_HIGH_THRESHOLD),
			comcerto_read_reg(dev, HOST_RX_FIFO_LOW_THRESHOLD));

	spin_unlock_irqrestore(&dev->mailbox_lock, flags);

	return ret;
}

struct device_attribute comcerto_host_registers_attr = {
	.attr = {
		.name = "host_registers",
		.mode = S_IRUGO,
		.owner = THIS_MODULE,
	},
	.show = comcerto_host_registers_show,
};

static void comcerto_read_mailbox(struct comcerto_dev *dev, struct comcerto_queue_msg *msg)
{
	msg->mailbox[0] = (u16) comcerto_read_reg(dev, RXMAIL0);
	msg->mailbox[1] = (u16) comcerto_read_reg(dev, RXMAIL1);
	msg->mailbox[2] = (u16) comcerto_read_reg(dev, RXMAIL2);
	msg->mailbox[3] = (u16) comcerto_read_reg(dev, RXMAIL3);
}

static void comcerto_write_mailbox(struct comcerto_dev *dev, struct comcerto_queue_msg *msg)
{
	comcerto_write_reg(dev, msg->mailbox[0], TXMAIL0);
	comcerto_write_reg(dev, msg->mailbox[1], TXMAIL1);
	comcerto_write_reg(dev, msg->mailbox[2], TXMAIL2);
	comcerto_write_reg(dev, msg->mailbox[3], TXMAIL3);
}

static int comcerto_reset(struct comcerto_dev *dev)
{
	struct pci_dev *pci_dev = dev->pci_dev;
	struct comcerto_queue_msg msg;
	unsigned long flags;
	int ret = 0;

	spin_lock_irqsave(&dev->mailbox_lock, flags);	/* get exclusive access to FIFO and mailbox */

#ifdef CONFIG_COMCERTO_PCI2ETH_BRIDGE
	/* need to stop interface, fool the driver by sending false alert */
	if (dev->pci2eth) {
		static u16 fifo[4] = { 0, CT_ALERT, 0, 0 };
		pci2eth_control(dev->pci2eth, fifo, sizeof(fifo));
	}
#endif
	comcerto_write_reg(dev, 0, HOST_FIFO_CONTROL);
	comcerto_write_reg(dev, 0xFF, HOST_FIFO_INTERRUPT_ACK);

	if (dev->pci_dev->device == PCI_DEVICE_ID_MINDSPEED_M829XX) {
		/* M829xx FIFO reset is broken, so we need to manually clear the FIFOs.*/

		/* Disable interrupts */
		comcerto_write_reg(dev, 0, HOST_FIFO_CONTROL);

		/* Acknowledge all interrupts */
		comcerto_write_reg(dev, 0x2FF, HOST_FIFO_INTERRUPT_ACK);

		/* Flush out the input and output queues */
		kfifo_reset(dev->rx_queue);
		kfifo_reset(dev->tx_queue);

		/* Perform reset */
		comcerto_write_reg(dev, HSRESET, HOST_FIFO_INTERRUPT_ACK);
		comcerto_write_reg(dev, 0, HOST_FIFO_INTERRUPT_ACK);

		/* Clean fifos of possible stale data */
		comcerto_write_reg(dev, RXM3IE | FIFO_EN, HOST_FIFO_CONTROL);

		/* Clean the RX FIFO */
		readb(dev->iomembase + HOST_RX_FIFO_DATA);
		while (comcerto_read_reg(dev, HOST_RX_FIFO_SIZE))
			readb(dev->iomembase + HOST_RX_FIFO_DATA);

		/* Clean the TX FIFO */
		writeb(0, dev->iomembase + HOST_TX_FIFO_DATA);
		while (comcerto_read_reg(dev, HOST_TX_FIFO_SIZE))
			writeb(0, dev->iomembase + HOST_TX_FIFO_DATA);

		/* Delay until the device is ready */
		mdelay(50);
	} else {
		/* Disable interrupts */
		comcerto_write_reg(dev, 0, HOST_FIFO_CONTROL);

		/* Flush out the input and output queues */
		kfifo_reset(dev->rx_queue);
		kfifo_reset(dev->tx_queue);

		if (dev->pci_dev->device == PCI_DEVICE_ID_PLX_9030) {
			int val = readl(dev->plxmembase + PLX_CNTRL_REG);

			/* remove read ahead bit in order to flush PCI fifo */
			writel(val & 0xFFFEFFFF, dev->plxmembase + PLX_CNTRL_REG);

			/* Reset */
			writel(val | 0x40000000, dev->plxmembase + PLX_CNTRL_REG);

			/* Delay for 1ms */
			mdelay(1);

			/* Remove reset and restore read ahead bit */
			writel(val & 0xBFFFFFFF, dev->plxmembase + PLX_CNTRL_REG);

			val = readl(dev->plxmembase + PLX_CNTRL_REG);
		} else {
			/* Reset the Comcerto */
			comcerto_write_reg(dev, HSRESET, HOST_FIFO_INTERRUPT_ACK);
		}

		/* Delay until the device is ready */
		mdelay(50);

		/* Enable the mailbox read interrupt */
		comcerto_write_reg(dev, RXM3IE | FIFO_EN, HOST_FIFO_CONTROL);
	}

	/* Read the post-reset mailbox */
	msg.fifo_len = 0;
	comcerto_read_mailbox(dev, &msg);

	/* If it contains BRM_READY, go ahead and read it now rather
	 * than waiting for an interrupt, because (unfortunately) it
	 * might not ever come.
	 */
	if (IS_BRM_READY(msg.mailbox[3], msg.mailbox[2], msg.mailbox[1], msg.mailbox[0]))  {
		dev_notice(&pci_dev->dev, "comcerto%d ready after reset\n", MINOR(dev->cdev.dev));
		comcerto_write_reg(dev, RXM3IAK, HOST_FIFO_INTERRUPT_ACK);

		comcerto_trace_store(dev, COMCERTO_TRACE_RMAGIC, &msg, NULL);

		dev_dbg(&pci_dev->dev, "comcerto%d got ready after reset, "
			"m3=%04x m2=%04x m1=%04x m0=%04x\n",
			MINOR(dev->cdev.dev),
			msg.mailbox[3], msg.mailbox[2],
			msg.mailbox[1], msg.mailbox[0]);

		/* Ack the BRM_READY */
		memset(&msg, 0, sizeof(msg));
		msg.fifo_len = 0;
		msg.mailbox[3] = 0x1300; 

		kfifo_put(dev->tx_queue, (unsigned char *)&msg, sizeof(msg));
		comcerto_start_tx(dev);
	} else {
		dev_warn(&pci_dev->dev, "comcerto%d is not ready after reset, "
			"m3=%04x m2=%04x m1=%04x m0=%04x\n",
			MINOR(dev->cdev.dev),
			msg.mailbox[3], msg.mailbox[2],
			msg.mailbox[1], msg.mailbox[0]);
		ret = -1;
	}

	spin_unlock_irqrestore(&dev->mailbox_lock, flags);

	return ret;
}

static void dump_control(struct comcerto_dev *dev, struct comcerto_queue_msg *msg, u8 *fifo, u8 tx)
{
	dev_err(&dev->pci_dev->dev, "%s mail %04x,%04x,%04x,%04x\n",
		tx?"TX":"RX",
		msg->mailbox[0], msg->mailbox[1], msg->mailbox[2], msg->mailbox[3]);
	if (msg->fifo_len) {
		u16 *fifo16 = (void*)fifo;
		int i;

		dev_err(&dev->pci_dev->dev, "%s fifo %04x,%04x,%04x,%04x (%u/%u bytes total)\n",
			tx?"TX":"RX",
			fifo16[0], fifo16[1], fifo16[2], fifo16[3], fifo16[0]&255, msg->mailbox[2]);
		for (i=4; i<msg->mailbox[2]/2; i++)
			printk("%04x%c",fifo16[i],((i-4)&7)==7?'\n':' ');
		if (((i-4)&7) != 7)
			printk("\n");
 	}
}

static void comcerto_rx(struct comcerto_dev *dev)
{
	struct comcerto_queue_msg msg;
	u8 fifo[256];
	int queue_full;

	/* Read the RX FIFO if it's nonempty */
	msg.fifo_len = comcerto_read_reg(dev, HOST_RX_FIFO_SIZE);
	if (msg.fifo_len > sizeof(fifo))
		msg.fifo_len = sizeof(fifo);
	if (msg.fifo_len != 0)
		comcerto_read_fifo(dev, fifo, msg.fifo_len);

	comcerto_read_mailbox(dev, &msg);
	dev_dbg(&dev->pci_dev->dev, "comcerto_rx (%d) m3=%04x m2=%04x m1=%04x m0=%04x\n",
		msg.fifo_len, msg.mailbox[3], msg.mailbox[2], msg.mailbox[1], msg.mailbox[0]);

	if (msg.fifo_len > 16 && ((u16*)fifo)[1] == CT_ALERT)
		dump_control(dev, &msg, fifo, 0);

#ifdef CONFIG_COMCERTO_PCI2ETH_BRIDGE
	/* SVREADY && Alert checks to notify interface that firmware can/can't accept commands */
	if (dev->pci2eth && msg.fifo_len >= 8) {
		u16 *fifo16 = (u16*) fifo;

		if (fifo16[2] == FC_SET_ETH_HDR && fifo16[1] == CT_DEVICE_CONFIG_RESPONSE)
			pci2eth_control(dev->pci2eth, (u16*)fifo, msg.fifo_len);

		if (fifo16[2] == 0 && (fifo16[1] == CT_SUPVSR_READY || fifo16[1] == CT_ALERT))
			pci2eth_control(dev->pci2eth, (u16*)fifo, msg.fifo_len);
	}

	if (dev->pci2eth && msg.mailbox[3] == 0x20) {
		if (dev->tx_locked) {
			u16 *fifo16 = (u16*) fifo;

			/* Unlock TX path if RX or TX packets are transferred */
			if (fifo16[2] == FC_ETHPCI_XFER_TX_PACKET || fifo16[2] == FC_ETHPCI_XFER_RX_PACKET) {
				u32 control;

				dev->tx_locked--;
				comcerto_write_reg(dev, RXM3IE | TXM3IE | FIFO_EN, HOST_FIFO_CONTROL);

				control = comcerto_read_reg(dev, HOST_FIFO_CONTROL);
				if ((control & TXM3IE) == 0)
					comcerto_start_tx(dev);
			}
		}

		pci2eth_control(dev->pci2eth, (u16*)fifo, msg.fifo_len);

		goto out;
	}
#endif

	/* Lock rx queue spin to protect queue - we may write to queue in two steps which must be seen as atomic op */
	spin_lock_irq(&dev->rx_queue_lock);

	queue_full = __kfifo_len(dev->rx_queue) > (COMCERTO_RX_QUEUE_SIZE - (sizeof(msg) + sizeof(fifo)));
	if (queue_full) {
		if (printk_ratelimit())
			dev_notice(&dev->pci_dev->dev, "rx queue overflow, message dropped\n");
		spin_unlock_irq(&dev->rx_queue_lock);
		return;
	}

	/* Pass message and FIFO data to the upper half */
	__kfifo_put(dev->rx_queue, (unsigned char *)&msg, sizeof(msg));
	if (msg.fifo_len != 0)
		__kfifo_put(dev->rx_queue, fifo, msg.fifo_len);

	spin_unlock_irq(&dev->rx_queue_lock);

	wake_up_interruptible(&dev->rx_proc_list);

#ifdef CONFIG_COMCERTO_PCI2ETH_BRIDGE
out:
#endif
	/* Store in the trace buffer */
	comcerto_trace_store(dev, COMCERTO_TRACE_RMAGIC, &msg, fifo);
}

static void comcerto_start_tx(struct comcerto_dev *dev)
{
	struct comcerto_queue_msg msg;
	unsigned char fifo[256];
	ulong flags;

#ifdef CONFIG_COMCERTO_PCI2ETH_BRIDGE
	if (dev->tx_locked)
		return;
#endif

	spin_lock_irqsave(&dev->tx_queue_lock, flags);

	if (__kfifo_len(dev->tx_queue) == 0) {
		/* Nothing to do: disable TXM3I interrupt and quit */
		comcerto_write_reg(dev, RXM3IE | FIFO_EN, HOST_FIFO_CONTROL);
		spin_unlock_irqrestore(&dev->tx_queue_lock, flags);
		return;
	}

	__kfifo_get(dev->tx_queue, (unsigned char *)&msg, sizeof(msg));
	if (msg.fifo_len)
		__kfifo_get(dev->tx_queue, fifo, msg.fifo_len);

	spin_unlock_irqrestore(&dev->tx_queue_lock, flags);

	if (msg.fifo_len > sizeof(fifo))
		BUG();

	if (msg.fifo_len)
		comcerto_write_fifo(dev, fifo, msg.fifo_len);

	comcerto_trace_store(dev, COMCERTO_TRACE_TMAGIC, &msg, fifo);

	comcerto_write_reg(dev, TXCMPLT, HOST_FIFO_INTERRUPT_ACK);

	/* Enable TXM3IE */
	comcerto_write_reg(dev, RXM3IE | TXM3IE | FIFO_EN, HOST_FIFO_CONTROL);

	dev_dbg(&dev->pci_dev->dev, "comcerto_start_tx cmd = %04x, data = %04x,%04x,%04x\n",
		msg.mailbox[3], msg.mailbox[2], msg.mailbox[1], msg.mailbox[0]);

#ifdef CONFIG_COMCERTO_PCI2ETH_BRIDGE
	if (msg.mailbox[3] == 0x20) {
		u16 *fifo16 = (u16*) fifo;

		/* Lock TX path if RX or TX packets are transferred */
		if (fifo16[2] == FC_ETHPCI_XFER_TX_PACKET || fifo16[2] == FC_ETHPCI_XFER_RX_PACKET) {
			dev->tx_locked++;
			comcerto_write_reg(dev, RXM3IE | FIFO_EN, HOST_FIFO_CONTROL);
		}
	}

	if (msg.mailbox[3] == 0x03) {
		u16 *fifo16 = (u16*) fifo;

		if (fifo16[2] == FC_SET_ETH_HDR && fifo16[1] == CT_DEVICE_CONFIG_CHANGE)
			pci2eth_control(dev->pci2eth, (u16*)fifo16, msg.fifo_len);
	}
#endif

	comcerto_write_mailbox(dev, &msg);

	wake_up_interruptible(&dev->tx_proc_list);
}

static irqreturn_t comcerto_irq_handler(int irq, void *dev_id)
{
	struct comcerto_dev *dev = dev_id;
	unsigned int status;

	spin_lock_irq(&dev->mailbox_lock);

	dev_dbg(&dev->pci_dev->dev, "comcerto_irq_handler entry\n");

	status = comcerto_read_reg(dev, HOST_FIFO_STATUS);
	status &= comcerto_read_reg(dev, HOST_FIFO_CONTROL);

	dev_dbg(&dev->pci_dev->dev, "comcerto_irq_handler status = 0x%x\n", status);
	if (status == 0) {
		spin_unlock_irq(&dev->mailbox_lock);
		dev_dbg(&dev->pci_dev->dev, "comcerto_irq_handler - spurious IRQ\n");
		return IRQ_NONE;
	}

	if (status & RXM3I) {
		comcerto_rx(dev);
		comcerto_write_reg(dev, RXM3IAK, HOST_FIFO_INTERRUPT_ACK);
	}

	if (status & TXM3I) {
		comcerto_write_reg(dev, TXM3IAK, HOST_FIFO_INTERRUPT_ACK);
		comcerto_start_tx(dev);
	}

	spin_unlock_irq(&dev->mailbox_lock);

	return IRQ_HANDLED;
}

static int comcerto_probe(struct pci_dev *pci_dev, const struct pci_device_id *id)
{
	int error = 0;
	struct comcerto_dev *dev;
	struct device *device;
	int iobase, iolen;
	int plxbase, plxlen = 0;
	dev_t devid;

	error = pci_enable_device(pci_dev);
	if (error) {
		dev_err(&pci_dev->dev, "failed to enable device\n");
		return error;
	}

	pci_set_master(pci_dev);
	pci_write_config_word(pci_dev, PCI_COMMAND, 7);
	pci_write_config_word(pci_dev, PCI_LATENCY_TIMER, 32);

	error = pci_request_region(pci_dev, 0, DRV_NAME);
	if (error) {
		dev_err(&pci_dev->dev, "failed to request region\n");
		return error;
	}

	dev = kzalloc(sizeof(*dev), GFP_KERNEL);
	if (dev == NULL) {
		error = -ENOMEM;
		goto err0;
	}

	pci_set_drvdata(pci_dev, dev);

	dev->pci_dev = pci_dev;

	dev->iomembase = dev->plxmembase = 0;

	if (pci_dev->device == PCI_DEVICE_ID_PLX_9030) {
		plxbase = pci_resource_start(pci_dev, 0);
		plxlen = pci_resource_len(pci_dev, 0);

		dev->plxmembase = ioremap_nocache(plxbase, plxlen);
		if (!dev->plxmembase) {
			dev_err(&pci_dev->dev, "failed to remap PLX memory\n");
			goto err0;
		}

		/* Chip select and base address set
		 * NRAD Wait States = 3 (4 wait states)
		 * NRDD Wait States = 0 (no wait states)
		 * NWAD Wait States = 3 (4 wait states)
		 * NWDD Wait States = 0 (no wait states)
		 * 32-bit access
		 */
		writel(0x008180C0, dev->plxmembase + 0x28);
		writel(0x01, dev->plxmembase + 0x3C);	/* enable CS0 and map local address space at 0 */
		writel(0x41, dev->plxmembase + 0x4C);	/* enable LINTi1 and PCI interrupt */

		iobase = pci_resource_start(pci_dev, 2);
		iolen = pci_resource_len(pci_dev, 2);
	} else {
		iobase = pci_resource_start(pci_dev, 0);
		iolen = pci_resource_len(pci_dev, 0);
	}

	dev->iomembase = ioremap_nocache(iobase, iolen);
	if (!dev->iomembase) {
		dev_err(&pci_dev->dev, "failed to remap Comcerto memory\n");
		goto err1;
	}

	/* Disable interrupts from the Comcerto before the IRQ is enabled */
	comcerto_write_reg(dev, 0, HOST_FIFO_CONTROL);

	/* Initialize the per-dev semaphore and the device open count */
	init_MUTEX(&dev->lock);

	/* Allocate the per-dev tx and rx queues, their spinlocks,
	 * and the tx and rx wait queues.
	 */
	spin_lock_init(&dev->rx_queue_lock);
	dev->rx_queue = kfifo_alloc(COMCERTO_RX_QUEUE_SIZE, GFP_KERNEL, &dev->rx_queue_lock);
	if (IS_ERR(dev->rx_queue)) {
		error = PTR_ERR(dev->rx_queue);
		goto err2;
	}

	spin_lock_init(&dev->tx_queue_lock);
	dev->tx_queue = kfifo_alloc(COMCERTO_TX_QUEUE_SIZE, GFP_KERNEL, &dev->tx_queue_lock);
	if (IS_ERR(dev->tx_queue)) {
		error = PTR_ERR(dev->tx_queue);
		goto err3;
	}

	spin_lock_init(&dev->mailbox_lock);

	init_waitqueue_head(&dev->rx_proc_list);
	init_waitqueue_head(&dev->tx_proc_list);

	dev->trace_out = dev->trace;

	/* Set up the corresponding cdev; this must be done last,
	 * since everything in dev needs to be ready for use by this
	 * point.
	 */
	cdev_init(&dev->cdev, &comcerto_fops);
	dev->cdev.owner = THIS_MODULE;
	dev->cdev.ops = &comcerto_fops;
	devid = comcerto_pci_dev++;
	error = cdev_add(&dev->cdev, devid, 1);
	if (error)
		goto err4;

	/* Register a class device */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,27)	
	device = device_create(comcerto_class, &pci_dev->dev, devid, "comcerto%d", MINOR(devid));
#else
	device = device_create(comcerto_class, &pci_dev->dev, devid, dev, "comcerto%d", MINOR(devid));
#endif
	if (IS_ERR(device)) {
		error = PTR_ERR(device);
		goto err5;
	}
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,27)	
	dev_set_drvdata(device, dev);
#endif

	/* Create the attribute files */
	error = device_create_bin_file(device, &comcerto_trace_attr);
	if (error)
		goto err6;

	error = device_create_file(device, &comcerto_host_registers_attr);
	if (error)
		goto err7;

	dev_info(&pci_dev->dev, "probed model %#x\n", pci_dev->device);
	dev_info(&pci_dev->dev, "registered as major %d, minor %d\n", MAJOR(devid), MINOR(devid));

#ifdef CONFIG_COMCERTO_PCI2ETH_BRIDGE
	if (pci_dev->device != PCI_DEVICE_ID_PLX_9030) {
		dev->pci2eth = pci2eth_open(pci_dev);

		if (!dev->pci2eth)
			goto err7;
	}
#endif

	return 0;

	/* Undo all that we've done: */
err7:
	device_remove_bin_file(device, &comcerto_trace_attr);
err6:
	device_destroy(comcerto_class, devid);
err5:
	cdev_del(&dev->cdev);
err4:
	kfifo_free(dev->tx_queue);
err3:
	kfifo_free(dev->rx_queue);
err2:
	iounmap(dev->iomembase);
err1:
	if (dev->plxmembase)
		iounmap(dev->plxmembase);
	kfree(dev);
err0:
	pci_disable_device(pci_dev);
	pci_release_region(pci_dev, 0);
	return error;
}

static void comcerto_remove(struct pci_dev *pci_dev)
{
	struct comcerto_dev *dev = pci_get_drvdata(pci_dev);

#ifdef CONFIG_COMCERTO_PCI2ETH_BRIDGE
	if (dev->pci2eth) {
		pci2eth_close(dev->pci2eth);

		comcerto_stop(dev);
	}
#endif

	dev_info(&pci_dev->dev, "removing device\n");

	device_destroy(comcerto_class, dev->cdev.dev);
	
	cdev_del(&dev->cdev);				/* deregister the cdev (this must be done first) */

	kfifo_free(dev->tx_queue);
	kfifo_free(dev->rx_queue);

	comcerto_write_reg(dev, 0, HOST_FIFO_CONTROL);	/* disable interrupts from the Comcerto */

	iounmap((void*) dev->iomembase);

	if (dev->plxmembase)
		iounmap((void*) dev->plxmembase);

	kfree(dev);

	pci_disable_device(pci_dev);

	pci_release_region(pci_dev, 0);
}

static struct pci_driver pci_driver = {
	.name = DRV_NAME,
	.id_table = comcerto_ids,
	.probe = comcerto_probe,
	.remove = comcerto_remove,
};

static int __init comcerto_pci_init(void)
{
	int error;

	printk(KERN_INFO "comcerto_pci: version %s\n", COMCERTO_PCI_VERSION);

	error = alloc_chrdev_region(&comcerto_pci_first_dev, 0, DEV_COUNT, "comcerto");
	comcerto_pci_dev = comcerto_pci_first_dev;
	if (error)
		return error;

	comcerto_class = class_create(THIS_MODULE, "comcerto");
	if (IS_ERR(comcerto_class))
		return PTR_ERR(comcerto_class);

	return pci_register_driver(&pci_driver);
}

static void __exit comcerto_pci_exit(void)
{
	pci_unregister_driver(&pci_driver);
	class_destroy(comcerto_class);
	unregister_chrdev_region(comcerto_pci_first_dev, DEV_COUNT);
}

module_init(comcerto_pci_init);
module_exit(comcerto_pci_exit);
