/*! \file teth.c  
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */


#include "ut.h"
#include "gtl.h"
#include "teth.h"
#include "tmap_devinfo.h"

#include <net/if.h>

#define GTL_CSME_STATE_SHUTDOWN		1
#define GTL_CSME_STATE_INITIALIZING	2
#define GTL_CSME_STATE_INITIALIZED	3
#define GTL_CSME_STATE_SHUTTING_DOWN	4
unsigned char bcastaddr[6] = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };

struct gtl_csme {
	int state;
	int iSockFd;
	SThread *pstCbThread;
	GTL_READ_CB gtl_read_cb;
} gtl_csme = {
	.state = GTL_CSME_STATE_SHUTDOWN,
};

struct SCsmeSpecInfo {
	struct sockaddr_csme	stSockAddr;
	DEVID			DevId;
};


/************************** TETH_Write *************************************/

/*!
 *    This function writes data to the device.\n
 *    1. Check the message type(Data/Control)\n
 *    2. Send the message to CSME library.\n
 *    WARNING: The buffer length send to the csmencaps layer is increased by 4 to add the multi command ending padding
 *    This implies that the pstMsg->fifo buffer must be correctly allocated and filled (last 4 bytes set to 0)
 *    These 4 bytes must be not taken in account in the pstMsg->fifo_size
 *
 *    \return   On successful completion, it returns SUCCESS \n
 *              On failure, it returns ERROR \n
 *    \param    pvDevAddrInfo -> device Addressing Info.\n
 *    \param    pstMsg -> Message to write.\n
 */
/***************************************************************************/
static int TETH_Write(struct SCsmeSpecInfo *pstAdapterSpecInfo, gtl_msg_t *pstMsg)
{
	struct sockaddr_csme *pstSockAddr = &pstAdapterSpecInfo->stSockAddr;
	struct sockaddr_csme stSockAddr;
	int Status = SUCCESS;

	UT_Log(GTL, INFO, "TETH_Write: ENTRY\n");

	if ((pstMsg == NULL) || (pstMsg->fifo == NULL)) {
		UT_ErrorLog(GTL, "TETH_Write: gtl message is NULL\n");
		Status = GTL_TETH_ERR_NULL_POINTER_PASSED;
		goto out;
	}

	UT_MemCopy(&stSockAddr, pstSockAddr, sizeof(struct sockaddr_csme));

	if (!(pstMsg->type & BOOT)) {
		GTL_PrintCsmeMsg(pstMsg);

		stSockAddr.scsme_flags = 0x1;
	} else {
		if (((U8 *) pstMsg->fifo)[3] == CMD_TYPE_BRM_MAAS_ASSIGN) {
			/* update the sockaddr_csme with the assigned mac address*/
			UT_MemCopy(pstSockAddr->scsme_devmac, pstMsg->fifo + 6, 6);
		}

		stSockAddr.scsme_flags = 0x0;
	}

	stSockAddr.scsme_channelid = UT_CPU2BE16(pstMsg->channel);

	UT_Log(GTL, INFO, "TETH_Write: calling sendto\n");

	/*Add 4 bytes to the len to make sure to add the multi command ending padding bytes */
	if (sendto(gtl_csme.iSockFd, pstMsg->fifo, pstMsg->fifo_size + 4, MSG_DONTWAIT, (struct sockaddr *)&stSockAddr, sizeof (stSockAddr)) < 0) {
		UT_ErrorLog(GTL, "TETH_Write: sendto failed\n");
		Status = GTL_ERR_UNKNOWN;
		goto out;
	}

out:
	UT_Log(GTL, INFO, "TETH_Write: EXIT\n");

	return Status;
}

/************************** TETH_RcvHandler ******************************/

/*!
 *    This function calls the function registered as gtl_read_cb.
 *
 *    \return   \n
 *    \param    Sock Fd
 *
 ***************************************************************************/ 
static void TETH_RcvHandler(int iSockFd)
{
	gtl_msg_t *pstMsg;
	struct sockaddr_csme stCsmSockAddr;
	struct SCsmeSpecInfo *pstInfo;
	socklen_t iSockLen;
	int iLen;

	pstMsg = UT_Calloc(1, sizeof(gtl_msg_t) + 1500);
	if (pstMsg == NULL) {
		UT_ErrorLog(GTL, "TETH_RcvHandler: memory allocation failed\n");
		goto out;
	}

	/* Set Fifo */
	pstMsg->fifo = (char *) (pstMsg + 1);

	iSockLen = sizeof(stCsmSockAddr);
	iLen = recvfrom(iSockFd, pstMsg->fifo, 1500, 0, (struct sockaddr *)&stCsmSockAddr, &iSockLen);

	if (iLen <= 0) {
		UT_ErrorLog(GTL, "TETH_RcvHandler: recvfrom failed\n");
		goto free;
	}

	pstInfo = (void*) ((long) stCsmSockAddr.scsme_user);

	/* is it a READY boot message ?*/
	if ((((U8 *) pstMsg->fifo)[2] == CMD_CLASS_ETH_BOOT_MSG) && 
		(((U8 *) pstMsg->fifo)[3] == CMD_TYPE_BRM_READY))
	{
		/* if yes check if we have a target to handle it */
		UT_MemCopy(pstInfo->stSockAddr.scsme_devmac, stCsmSockAddr.scsme_devmac, 6);
	}

	/*Initialize rest of pstMsg */
	pstMsg->fifo_size = iLen;
	pstMsg->channel = UT_BE2CPU16(stCsmSockAddr.scsme_channelid);

	switch (stCsmSockAddr.scsme_opcode)
	{
	case UT_CPU2BE16(CSME_OPCODE_CONTROL):
		pstMsg->type = CSME;
		if ((stCsmSockAddr.scsme_flags & 0x1) == 0)
		{
			pstMsg->type |= BOOT;
			GTL_PrintCsmeMsg(pstMsg);
		}
		break;

	/* these type of messages are not upported yet */
	case UT_CPU2BE16(CSME_OPCODE_RESERVED):
	case UT_CPU2BE16(CSME_OPCODE_UNIFIED_DIAGNOSTICS):
	case UT_CPU2BE16(CSME_OPCODE_REMOTE_MEDIA):
		goto free;

	default:
		UT_ErrorLog(GTL, "TETH_RcvHandler: Invalid Message from MSP\n");
		goto free;
	}

	UT_Log(GTL, INFO, "TETH_RcvHandler: Rcv Message:\n");

	gtl_csme.gtl_read_cb(pstInfo->DevId, pstMsg);	/*Call the Callback function */

free:
	UT_FreeMem(pstMsg);

out:
	return;
}
  
/************************** TETH_RcvThread ******************************/

/*!
 *    This function calls TETH_RcvHandler to fetch data in infinite loop.
 *
 *    \return   \n
 *
 ***************************************************************************/
static void *TETH_RcvThread(void *dummy)
{
	UT_Log(GTL, INFO, "TETH_RcvThread: ENTRY\n");

	while (True)
		TETH_RcvHandler(gtl_csme.iSockFd);

	return NULL;
}

/************************** TETH_Open *************************************/
/*!
 *    This function activates the device. Before call to this function, \n
 *    GTL Mapper will not be able to send and receive any message to and \n
 *    from MSP.\n
 *    1. Change the state of device.\n
 *    2. Create the target structure and calls CSME_addTarget to add device in \n
 *       target list of CSME library\n
 *
 *    \return   On successful completion, it returns SUCCESS\n
 *              On failure, it returns ERROR\n
 *    \param    deviceid -> device id of the device.\n
 *    \param    pvDevAddrInfo -> device Addressing Info.\n
 *    \param    ppvDevAdapterSpecInfo -> OUT parameter for returning adapter \n
 *              specific device info.\n
 */
/***************************************************************************/

static int TETH_Open(S32 deviceid, void *pvDevAddrInfo, void **pvAdapterSpecInfo, EDevMode eDevMode)
{
	SGtlCsmeItf *pstCsmeItf = (SGtlCsmeItf *) pvDevAddrInfo;
	struct SCsmeSpecInfo *pstInfo;
	struct sockaddr_csme *pstSockAddr;
	struct csme_ackopt stAckOpt;
	struct csme_usersetopt stUserSet;
	int Status;
	U8 ucOpenMac[6];
	FILE *br_file;
	char host_mac[18];

	UT_Log(GTL, INFO, "TETH_Open: ENTRY\n");


	if (eDevMode == eMASTER)
	{
		br_file = fopen("/proc/net/ved/bridge", "r");
		if (br_file)
		{
			if (fgets(host_mac, 18, br_file))
			{
				UT_ErrorLog(GTL, "TETH_Open: Bridge mode is set up on host mac [%s]\n", host_mac);
				Status = GTL_TETH_ERR_UNKNOWN;
				goto err0;
			}

			fclose(br_file);
		}
	}

	if (gtl_csme.state != GTL_CSME_STATE_INITIALIZED) {
		UT_ErrorLog(GTL, "TETH_Open: CSME not started yet\n");
		Status = GTL_TETH_ERR_NOT_INITIALIZED;
		goto err0;
	}
	
	pstInfo = UT_Calloc(1, sizeof(struct SCsmeSpecInfo));
	if (pstInfo == NULL) {
		UT_ErrorLog(GTL, "TETH_Open: memory allocation failed\n");
		Status = GTL_TETH_ERR_NOMEM;
		goto err0;
	}

	pstInfo->DevId = deviceid;
	pstInfo->stSockAddr.scsme_user = (long) pstInfo;
	pstInfo->stSockAddr.scsme_family = AF_CSME;
	pstInfo->stSockAddr.scsme_ifindex = if_nametoindex(pstCsmeItf->pcEthName);
	pstSockAddr = &pstInfo->stSockAddr;

	/* In master mode we open the target with the configured MAC
	it won't be modified by a MAAS ASSIGN*/
	if (eDevMode == eMASTER)
	{
		UT_MemCopy(ucOpenMac, pstCsmeItf->devMac, 6);
	}
	else
	{
		/* always open a broacast target */
		if (pstCsmeItf->ucEthBootMode == BOOT_MODE_LEGACY) {
			UT_MemCopy(ucOpenMac, pstCsmeItf->devMac, 6);
		} else {
			/* For the M823XX alternate boot mode the mac address could be
			broadcast or unicast, used the mac set by the user*/
			UT_MemCopy(ucOpenMac, pstCsmeItf->devHardMac, 6);
		}
	}

	UT_MemCopy(pstSockAddr->scsme_devmac, ucOpenMac, 6);

	pstSockAddr->scsme_opcode = UT_CPU2BE16(CSME_OPCODE_CONTROL);

	UT_MemCopy(stAckOpt.scsme_devmac, ucOpenMac, 6);
	stAckOpt.scsme_ifindex = pstSockAddr->scsme_ifindex;
	stAckOpt.ack_suppression = pstCsmeItf->ackRequired ? 0 : 1;

	if (setsockopt(gtl_csme.iSockFd, SOL_CSME, CSME_ACK_ENABLE, &stAckOpt, sizeof(stAckOpt)) < 0) {
		UT_ErrorLog(GTL, "TETH_Open: setsockopt failed\n");
		Status = GTL_TETH_ERR_UNKNOWN;
		goto err1;
	}

	UT_MemCopy(stUserSet.scsme_devmac, stAckOpt.scsme_devmac, 6);
	stUserSet.csme_user = pstInfo->stSockAddr.scsme_user;

	if (setsockopt(gtl_csme.iSockFd, SOL_CSME, CSME_USER_SET, &stUserSet, sizeof(stUserSet)) < 0) {
		UT_ErrorLog(GTL, "TETH_Open: setsockopt failed\n");
		Status = GTL_TETH_ERR_UNKNOWN;
		goto err1;
	}

	*pvAdapterSpecInfo = pstInfo;

	UT_Log(GTL, INFO, "TETH_Open: EXIT\n");

	return SUCCESS;

err1:
	UT_FreeMem(pstInfo);
	
err0:
	UT_Log(GTL, INFO, "TETH_Open: EXIT\n");

	return Status;
}

/************************** TETH_Close *************************************/

/*!
 *    This function de-activates the device.\n
 *    1. Change the state of device IDLE.\n
 *    2. calls CSME_killTarget to remove device in\n
 *       target list of CSME library.\n
 *
 *    \return   On successful completion, it returns SUCCESS\n
 *              On failure, it returns ERROR\n
 *    \param    pstAdapterSpecInfo ->  Adapter Specific Info.\n
 */
/***************************************************************************/
static int TETH_Close(struct SCsmeSpecInfo *pstAdapterSpecInfo)
{
	int Status = SUCCESS;

	UT_Log(GTL, INFO, "TETH_Close: ENTRY\n");

	if (gtl_csme.state != GTL_CSME_STATE_INITIALIZED) {
		UT_ErrorLog(GTL, "TETH_Close: CSME not open\n");
		Status = GTL_TETH_ERR_INITIALIZATION;
		goto out;
	}

	/* free sock addr, allocated by TETH_Open */
	UT_FreeMem(pstAdapterSpecInfo);

out:
	UT_Log(GTL, INFO, "TETH_Close: EXIT\n");

	return Status;
}

/************************** TETH_Reset *************************************/

/*!
 *    1. This function is empty.\n
 *
 *    \return   SUCCESS\n
 *    \param    pstAdapterSpecInfo -> Adapter Specific Info.\n
 */
/***************************************************************************/
static int TETH_Reset(void *pvDevAddrInfo, struct SCsmeSpecInfo *pstAdapterSpecInfo)
{
	struct sockaddr_csme *pstSockAddr = &pstAdapterSpecInfo->stSockAddr;
	SGtlCsmeItf *pstCsmeItf = (SGtlCsmeItf *) pvDevAddrInfo;
	struct csme_resetopt stResetOpt;
	int Status = SUCCESS;

	UT_Log(GTL, INFO, "TETH_Reset: ENTRY\n");

	/* we want to reset the previously assigned mac*/
	UT_MemCopy(stResetOpt.scsme_devmac, pstSockAddr->scsme_devmac, 6);

	if (pstCsmeItf->ucEthBootMode == BOOT_MODE_LEGACY) {
		/* and use broacast address for next boot init (maas assign, query ready or ready)*/
		UT_MemCopy(pstSockAddr->scsme_devmac, bcastaddr, 6);
		/* we want to set the original mac to broadcast*/
		UT_MemCopy(stResetOpt.scsme_newmac, bcastaddr, 6);
	} else {
		/* we want to set the original mac to user configured mac*/
		UT_MemCopy(pstSockAddr->scsme_devmac, pstCsmeItf->devHardMac, 6);
		UT_MemCopy(stResetOpt.scsme_newmac, pstCsmeItf->devHardMac, 6);
	}

	if (setsockopt(gtl_csme.iSockFd, SOL_CSME, CSME_TARGET_RESET, &stResetOpt, sizeof(stResetOpt)) < 0) {
		UT_ErrorLog(GTL, "TETH_Reset: reset failed\n");
		Status = GTL_TETH_ERR_UNKNOWN;
		goto out;
	}

out:
	UT_Log(GTL, INFO, "TETH_Reset: EXIT\n");

	return Status;
}


/************************** TETH_Shutdown *************************************/

/*!
 *    This function performs the shutdown facility of .\n
 *    1. Closes the CSME socket.\n
 *    2. Stops the receive thread. \n
 *    \return   On successful completion, it returns SUCCESS\n
 *              On failure, it returns GTL_TETH_ERR_INITIALIZATION\n
 */
/***************************************************************************/
int TETH_Shutdown(void)
{
	int Status = SUCCESS;

	UT_Log(GTL, INFO, "TETH_Shutdown: ENTRY\n");
  
	if (gtl_csme.state != GTL_CSME_STATE_INITIALIZED) {
		UT_ErrorLog(GTL, "TETH_Shutdown: CSME not closed\n");
		Status = GTL_TETH_ERR_INITIALIZATION;
		goto out;
	}

	gtl_csme.state = GTL_CSME_STATE_SHUTTING_DOWN;

	UT_ThreadCancel(gtl_csme.pstCbThread);
	UT_ThreadJoin(gtl_csme.pstCbThread);
	UT_FreeMem(gtl_csme.pstCbThread);

	close(gtl_csme.iSockFd);

	gtl_csme.state = GTL_CSME_STATE_SHUTDOWN;

	UT_Log(GTL, DEBUG, "TETH_Shutdown: CSME shutdown successful\n");

out:
	UT_Log(GTL, INFO, "TETH_Shutdown: EXIT\n");

	return Status;
}
  
/************************** TETH_Init *************************************/
/*!
 *    This function should be called by GTL Mapper at start up \n
 *    before issuing any other command. It performs the following. fns\n 
 *    1. Initialize the CSME library.\n
 *    
 *    \return   On successful completion, it returns SUCCESS\n
 *              On failure, it returns GTL_ERR_ALREADY_INITIALIZED\n
 *                          or GTL_ERR_NULL_POINTER_PASSED\n 
 *							or GTL_ERR_START.\n
 *    \param    pfnReadCb -> pointer to the client callback function. \n
 *              On reception of a message from MSP, GTL invokes this\n
 *              callback function to deliver the message to the\n
 *              GTL client.\n
 *    \param    pstAdapterFns->OUT param to be filled up with adapter funcptrs\n
 */
/***************************************************************************/

int TETH_Init(IN GTL_READ_CB pfnReadCb, OUT STAdapterFns *pstAdapterFns, SGtlDeviceInfo *pstDevList, U32 uiNumDevices)
{
	int Status = SUCCESS;

	UT_Log(GTL, INFO, "TETH_Init: ENTRY\n");

	if (gtl_csme.state != GTL_CSME_STATE_SHUTDOWN) {
		UT_ErrorLog(GTL, "TETH_Init: CSME already started\n");
		Status = GTL_TETH_ERR_ALREADY_INITIALIZED;
		goto err0;
	}

	gtl_csme.state = GTL_CSME_STATE_INITIALIZING;

	if ((pfnReadCb == NULL) || (pstAdapterFns == NULL)) {
		UT_ErrorLog(GTL, "TETH_Init: GTL callback is NULL\n");
		Status = GTL_TETH_ERR_NULL_POINTER_PASSED;
		goto err1;
	}

	gtl_csme.iSockFd = socket(AF_CSME, SOCK_DGRAM, UT_CPU2BE16(ETH_P_CSME));
	if (gtl_csme.iSockFd < 0) {
		UT_ErrorLog(GTL, "TETH_Init: CSME socket creation failed\n");
		Status = GTL_TETH_ERR_INITIALIZATION;
		goto err1;
	}

	gtl_csme.gtl_read_cb = pfnReadCb;
	pstAdapterFns->pfnAdapterOpen = TETH_Open;
	pstAdapterFns->pfnAdapterClose = (int(*)(void*)) TETH_Close;
	pstAdapterFns->pfnAdapterReset = (int(*)(void*, void*)) TETH_Reset;
	pstAdapterFns->pfnAdapterWrite = (int(*)(void*, gtl_msg_t*)) TETH_Write;
	pstAdapterFns->pfnAdapterShutdown = TETH_Shutdown;

	if ((gtl_csme.pstCbThread = UT_ThreadCreate("gtl_recv", TETH_RcvThread, NULL)) == NULL) {
		UT_ErrorLog(GTL, "TETH_Init: thread creation failed\n");
		Status = GTL_TETH_ERR_INITIALIZATION;
		goto err2;
	}

	gtl_csme.state = GTL_CSME_STATE_INITIALIZED;

	UT_Log(GTL, INFO, "TETH_Init: CSME started\n");

	UT_Log(GTL, INFO, "TETH_Init: EXIT\n");

	return SUCCESS;

err2:
	close(gtl_csme.iSockFd);

err1:
	gtl_csme.state = GTL_CSME_STATE_SHUTDOWN;

err0:
	UT_Log(GTL, INFO, "TETH_Init: EXIT\n");

	return Status;
}
