/*! \file video_itf.c
 @defgroup video_level Video related API functions
 *  @{
 */

/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include "ut.h"
#include "vapi.h"
#include "dmgr.h"
#include "vcore.h"
#include "vcore_voip.h"
#include "appitf.h"
#include "cfg.h"
#include "msp.h"


/***************************************************************************
* VAPI_SetMediaSubLayer: The function does the following things -
***************************************************************************/
/*!
*	\n \b Description: \n
	This API call is to address what media sub-layer subsequent API calls are applied to.\n
*	This API doesn't send any command to the device. It just set an internal variable.\n
*	To set video specific parameters, it is required to issue this API before the VAPI_SetConnIpParams(), VAPI_SetPayloadType(), VAPI_SetCodecType().\n
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection ID.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eMediaSubLayer</td>
*		<td style="vertical-align: top;">Connection media sub-layer to be set.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b usage
*	\include set_media_sub_layer.c
*
*	\n \b Commands:
*	List of Comcerto commands sent:
*	\li NONE
*/
VSTATUS VAPI_SetMediaSubLayer(IN CONNID ConnId, IN EMediaSubLayer eMediaSubLayer, IN SRequest * pstRequest)
{
	VSTATUS Status;
	SChnl* pstChnl = NULL;

	UT_Log(APPITF, INFO, "Entered VAPI_SetMediaSubLayer: dev(%u)\n", ConnId);
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_SetMediaSubLayer", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	pstChnl->MediaSubLayer = eMediaSubLayer;
	UT_Log(APPITF, INFO, "VAPI_SetMediaSubLayer: Exiting status(%d), dev(%u)\n", Status, ConnId);

out:
	/* if ASYNC mode call the callback function*/
	if ((pstRequest != NULL) && (pstRequest->pfnIoCompCallback != NULL))
	{
		pstRequest->pfnIoCompCallback(ConnId, CMD_LEVEL_CONN, Status, NULL, 0, pstRequest->uiReqId);
		Status = VAPI_ERR_PENDING;
	}

	return Status;
}

/***************************************************************************
 * VAPI_SendH245Message: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This API call is to transmit ITU-T H.245 protocol messages over established H.223 transport.\n
*
*	Note:
*	Comcerto reports all correct H.223 control channel data to host using IMSoTDM_RxH245 indication.\n
*	For this purpose user must register indication handler by VAPI_RegisterEventCallback().\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which H245 message to be sent.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">uiH245MsgLength</td>
*		<td style="vertical-align: top;">H.245 message length.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstPlayData</td>
*		<td style="vertical-align: top;">Pointer to H.245 message buffer.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include send_h245_message.c
*
*	\n \b Commands:
*	- IMSoTDM_TxH245
*/
VSTATUS VAPI_SendH245Message(IN CONNID ConnId, IN U16 uiH245MsgLength, IN U8 *ptrH245Msg, IN SRequest *pstRequest)

{
	SVapiReq *pstVapiReq = NULL;
	SChnl *pstChnl = NULL;
	VSTATUS Status;
	Boolean bIsSync;
	SH245Data *pstUserData;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_SendH245Message", &pstChnl);

	if (Status != SUCCESS)
		goto err_out;

	UT_Log(APPITF, INFO, "Entered VAPI_SendH245Message: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_SEND_H245_MESSAGE;

	pstVapiReq = VCORE_AllocateRequest(sizeof(SH245Data ));

	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}

	pstUserData = pstVapiReq->pvUserData;

	pstUserData->uiH245MsgLength = uiH245MsgLength;
	pstUserData->ptrH245Msg = ptrH245Msg;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_SendH245Msg, ConnId);

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_SendH245MessageVAPI_SendH245Message: Exiting status(%d), conn(%u)\n", Status, ConnId);

/* no error exit */
	return Status;

      err_out:
	return Status;
}

/** @} */ /*video_level*/
