/*! \file vcore.c 
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include "ut.h"
#include "vapi.h"
#include "dmgr.h"
#include "vcore.h"
#include "msp.h"
#include "vcore_voip.h"
#include "appitf.h"

extern Boolean bVapiInitialized_g;

/*Global Definitions*/
extern SVAPIConfig stVAPIConfig_g;

/** Pointer that stores the callback pointer required for delivering events*/
PFNEventCallback pfnEventCallback_g;
extern VAPI_Statistics vapi_stat;

extern SAVLTree *pstConnTree_g;
extern SAVLTree *pstConfTree_g;
/*Interface Function Definitions*/

/****************************************************************************
 * VCORE_Init : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      - Currently it doesn't do anything
 *
 *  - Assumptions 
 *      - None
 *
 *  \return None 
 */
VSTATUS VCORE_Init()
{
	pfnEventCallback_g = NULL;
	return SUCCESS;
}


/****************************************************************************
 * VCORE_ProcessRequest : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      - This function can be called in following scenarios.
 *          -#  When a new request is made by the user application 
 *          -#  A request at an intermediate state needs was added on another
 *              channel while execution.
 *      - When a new request is made
 *          -#  Add the request to the tail of request list of the specified 
 *              channel.
 *          -#  Checks,  whether chnl is free (Check the state of the conn.)
 *          -#  If the channel is free, call the corresponding Request-FSM 
 *              Handler
 *          -#  If channel is not free return immediately
 *
 *  - Assumptions:
 *      -#  List manipulation functions provided by UT are thread-safe
 *      -#  When a pending request has to be launched, the channel is assumed
 *          to be free.
 *      -#  Before calling this function Request state and 
 *
 *  \return   
 *  The status returned by this call is not the status of the actual VAPI
 *  request execution. That status will be obtainable by SVapiReq::Status. This
 *  Status inside the SVapiReq will be applicable only when returned Status by 
 *  this call is SUCCESS.\n
 *  Error codes returned by this call are:\n
 *     SUCCESS \n
 *     VAPI_ERR_INVALID_PARAM\n
 *     VAPI_ERR_NOMEM\n
 *     VAPI_ERR_UNDEFINED\n
 *     VAPI_ERR_PENDING\n
 *
 *     negative of MSP error codes ( when MSP returns error )\n
 *
 *  \param pstChnl Channel on which request hass to be sent
 *  \param pstReq  VAPI Request that has to be sent. If this is NULL
 *                      then sends the first request in channel's request list.
 */
VSTATUS VCORE_ProcessRequest(IN SChnl * pstChnl, IN SVapiReq * pstReq)
{
	VSTATUS Status;
	Boolean bBlock;
	CONNID ConnId;

	ConnId = pstChnl->ConnId;
	UT_Log(VCORE, INFO, "VCORE_ProcessRequest: Entering conn(%u) channel(%u) req(0x%x) reqid(%u) pfnReqHdlr(0x%x)\n",
		ConnId, pstChnl->usMSPChnlId, pstReq, pstReq->uiReqId, pstReq->pfnReqFSMHdlr);

	/*Check whether channel is busy */
	UT_MutexLock(&pstChnl->stReqList.stSem);

	/* Increment the vapi_frames_queued counter only on user call (not on response nor event) */
	if (pstReq->ucThreadContext == USER_CONTEXT)
	{
		UT_MutexLock(&vapi_stat.mutex_stat);
		vapi_stat.vapi_frames_queued++;
		UT_MutexUnLock(&vapi_stat.mutex_stat);
	}

	pstReq->pstChnl = pstChnl;

	/* We want to set the bBlock to false if we are called with no callback function from user (sync mode)
	or if we are call from internal thread  */
	if ((pstReq->bIsSync == False) || (pstReq->ucThreadContext == VAPI_CONTEXT))
	{
		bBlock = False;
	}
	else
	{
		bBlock = True;
		pstReq->pstSemReqComp = UT_SemInit(0);
		if (pstReq->pstSemReqComp == NULL)
		{
			UT_ErrorLog(APPITF, "VCORE_ProcessRequest: Can't initialize semaphore\n");
			Status = VAPI_ERR_NOMEM;
			UT_MutexUnLock(&pstChnl->stReqList.stSem);
			goto out;
		}
		else
			UT_Log(VCORE, INFO, "VCORE_ProcessRequest: compsem(0x%X)\n", pstReq->pstSemReqComp);
	}

	if (__UT_IsListEmpty(&pstChnl->stReqList))
	{
		/* Go To the tail of the list of request on the given channel 
		   and add the request */
		__UT_AddToTail(&pstChnl->stReqList, &pstReq->stNode);

		pstReq->ucInProgress = True;

		UT_MutexUnLock(&pstChnl->stReqList.stSem);

		/*If channlel is not busy Call Req-FSM handler. */
		UT_Log(VCORE, INFO, "VCORE_ProcessRequest: Launching reqid(%u) state(%u)\n", pstReq->uiReqId, pstReq->usReqState);

		/* proceed with the state mahine on this request */
		Status = pstReq->pfnReqFSMHdlr(pstChnl, pstReq, FSM_EVENT_START, NULL);
	}
	else
	{
		/* Go To the tail of the list of request on the given channel 
		   and add the request */
		__UT_AddToTail(&pstChnl->stReqList, &pstReq->stNode);

		/* In async mode this request has the VAPI_CONTEXT marker from now for all its life */
		if (pstReq->bIsSync == False)
			pstReq->ucThreadContext = VAPI_CONTEXT;

		UT_MutexUnLock(&pstChnl->stReqList.stSem);

		Status = SUCCESS;
	}


	/*If user has not supplied call back function i.e it is a blocking call
	   then wait for the completion of the request */
	if (bBlock)
	{
		/*Wait for request completion. This semaphore is signalled by 
		   request FSM handler, when it obtains final response */
		UT_Log(VCORE, INFO, "VCORE_ProcessRequest: Blocking on req(0x%x) conn(%u) channel(%u)\n", pstReq, ConnId,
		       pstChnl->usMSPChnlId);
		UT_Log(VCORE, INFO, "VCORE_ProcessRequest: Before wait compsem(0x%X) reqid(%d)\n",
		       pstReq->pstSemReqComp, pstReq->uiReqId);

		/* Wait here until we get wakeup by the request completion */
		/* this can occur if the request completion finished correctly,
		or if the erreoneous device response (NACK) or timeout
		In any case the value set here must be > to CFG_MSP_RESP_TIMEOUT defined in cfg.h*/ 
		UT_SemWait(pstReq->pstSemReqComp, INFINITE_TIMEOUT);

		UT_Log(VCORE, INFO, "VCORE_ProcessRequest: Request completed on req(0x%x) conn(%u) reqid(%d)\n", pstReq, ConnId, pstReq->uiReqId);

		/* retrieve the status set from the GTL thread */ 
		Status = pstReq->Status;

		UT_SemDestroy(pstReq->pstSemReqComp);
	}
	else
	{
		if (Status == SUCCESS)
		{
			Status = VAPI_ERR_PENDING;
		}
	}

out:
	UT_Log(VCORE, INFO, "VCORE_ProcessRequest: Exiting status(%d)\n", Status);

	return Status;
}

/****************************************************************************
 * VCORE_ProcessRequestList : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      - This function can be called in following scenarios.
 *          -#  When a pending request is to be launched (pstReq == NULL)
 *      - When a pending request has to be launched
 *          -#  Get the front node.
 *          -#  Verify its state to be not executing.
 *          -#  Call the corresponding Request-FSM Handler.

 *  - Assumptions:
 *      -#  List manipulation functions provided by UT are thread-safe
 *
 *  \return   
 *  The status returned by this call is not the status of the actual VAPI
 *  request execution. That status will be obtainable by SVapiReq::Status. This
 *  Status inside the SVapiReq will be applicable only when returned Status by 
 *  this call is SUCCESS.\n
 *  Error codes returned by this call are:\n
 *     SUCCESS \n
 *     VAPI_ERR_INVALID_PARAM\n
 *     VAPI_ERR_NOMEM\n
 *     VAPI_ERR_UNDEFINED\n
 *     VAPI_ERR_PENDING\n
 *
 *     negative of MSP error codes ( when MSP returns error )\n
 *
 *  \param pstChnl Channel on which to process request
 */
VSTATUS VCORE_ProcessRequestList(IN SChnl * pstChnl)
{
	SVapiReq *pstReq;
	SLstNode *pstNode;
	VSTATUS Status;

	UT_Log(VCORE, INFO, "VCORE_ProcessRequestList: Entering conn(%u) channel(%u)\n", pstChnl->ConnId,
			pstChnl->usMSPChnlId);

	/*Check whether channel is busy */
	UT_MutexLock(&pstChnl->stReqList.stSem);

	pstNode = __UT_GetFrontNode(&pstChnl->stReqList);
	if (pstNode == NULL)
	{
		/* the list is empty not an issue */
		Status = SUCCESS;
		goto err_unlock;
	}

	pstReq = UT_ContainerOf(pstNode, SVapiReq, stNode);

	if (pstReq->ucInProgress)
	{
		UT_Log(VCORE, PACKET, "VCORE_ProcessRequestList: Req(%x) is already being processed\n", pstReq);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_unlock;
	}

	pstReq->ucInProgress = True;

	UT_MutexUnLock(&pstChnl->stReqList.stSem);

	/*If channlel is not busy Call Req-FSM handler. */
	UT_Log(VCORE, INFO, "VCORE_ProcessRequestList: Launching reqid(%u) state(%u)\n", pstReq->uiReqId, pstReq->usReqState);

	if (!pstReq->pfnReqFSMHdlr)
	{
		UT_ErrorLog(VCORE, "VCORE_ProcessRequestList: pstReq->pfnReqFSMHdlr == NULL");
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_unlock;
	}

	pstReq->pfnReqFSMHdlr(pstChnl, pstReq, FSM_EVENT_START, NULL);

	UT_Log(VCORE, INFO, "VCORE_ProcessRequestList: Exiting status(%d)\n", SUCCESS);

	return SUCCESS;

      err_unlock:
	UT_MutexUnLock(&pstChnl->stReqList.stSem);

	UT_Log(VCORE, INFO, "VCORE_ProcessRequestList: Exiting status(%d)\n", Status);

	return Status;
}

/****************************************************************************
 * VCORE_ProcessFSMhandler : The function does the following things -
  Run fsm-handler
 ***************************************************************************/
void VCORE_ProcessFSMhandler(IN SChnl * pstChnl, U16 Event, IN gtl_msg_t * pstMsg)
{
	SLstNode *pstNode;
	SVapiReq *pstVapiReq;

	UT_TimerStop(pstChnl->pstRespTimer);

	UT_MutexLock(&pstChnl->stAccessFsmHndlr);
	UT_MutexLock(&pstChnl->stReqList.stSem);
	/*Access the request node so that its FSM handler gets called */
	pstNode = __UT_GetFrontNode(&pstChnl->stReqList);
	if (pstNode == NULL)
	{
		UT_MutexUnLock(&pstChnl->stReqList.stSem);
		UT_MutexUnLock(&pstChnl->stAccessFsmHndlr);
		UT_Log(VCORE, WARN, "VCORE_ProcessFSMhandler: Request not found for Chnl %d\n", pstChnl->usMSPChnlId);
		goto out;
	}

	UT_MutexUnLock(&pstChnl->stReqList.stSem);

	pstVapiReq = UT_ContainerOf(pstNode, SVapiReq, stNode);

	/* We get called from ProcessResponse or ProcessEvent or Timeout 
	so we are in VAPI CONTEXT */ 
	pstVapiReq->ucThreadContext = VAPI_CONTEXT;

	/* proceed with the state machine for this request */ 
	pstVapiReq->pfnReqFSMHdlr(pstChnl, pstVapiReq, Event, pstMsg);

	UT_MutexUnLock(&pstChnl->stAccessFsmHndlr);

      out:
	UT_Log(VCORE, INFO, "VCORE_ProcessFSMhandler: Exiting processing Done\n");
	return;
}

/****************************************************************************
 * VCORE_ProcessBootResponse : The function does the following things -
 ***************************************************************************/
/*! 
 *  -  Implementation
 *      -#  Extract channel structure for the supervisor channel DMGR_GetChannel \n
 *      -#  Check for the message index of the received command and compare it
 *          against the expected channel index. 
 *      -#  Increment the message index.
 *      -#  If the message index is OK then call the Request-FSM Handler of 
 *          the head request in SChnl.\n
 *
 *  - Assumptions\n
 *      -#  Message passed is a valid MSP response packet.\n
 *
 *  \return   
 *      None
 *
 *  \param DevId The device from the message has been received
 *  \param pstMsg Message obtained from GTL.
 */
void VCORE_ProcessBootResponse(SDevice *pstDevice, IN gtl_msg_t * pstMsg)
{
	SChnl *pstChnl;
	U8 tmp_index;
	SBootHdr *pstFifo;

	UT_Log(VCORE, INFO, "VCORE_ProcessBootResponse: Entering on Device %u\n", pstDevice->DevId);

	if ((pstDevice->eState == eBOOTING
		     || pstDevice->eState == eUNINITIALIZED
		     || pstDevice->eState == eREADYFORDUMP || pstDevice->eState == eMAAS))
	{
		/*This is a message in response to Boot device commands or during core dump process */
		pstChnl = DMGR_GetChannel(pstDevice, SUPV_CHANNEL);
	}
	else
	{
		UT_ErrorLog(VCORE, "VCORE_ProcessBootResponse: Exiting  dev(%u) in wrong state(%u)\n", pstDevice->DevId, pstDevice->eState);
		return;
	}

	UT_MutexLock(&pstChnl->stSemMsgIndex);

	if (pstDevice->eItfType != ePCI_ITF)
	{
		pstFifo = (SBootHdr *) pstMsg->fifo;
		
		if (pstDevice->eItfType == ePOS_ITF)
			tmp_index = ((U8*)pstMsg->fifo) [1];
		else
			tmp_index = pstFifo->index;

		/* do not check/increment index for ready indication*/
		if ((pstFifo->cmd_class == CMD_CLASS_ETH_BOOT_MSG) && 
			(pstFifo->cmd_type== CMD_TYPE_BRM_READY))
		{
			pstChnl->pstDev->bIsReadyReceived = True;
			goto out;
		}

		if (pstChnl->ucMsgIndex != tmp_index)
		{
			/*This is an invalid (retransmitted) message */
			UT_MutexUnLock(&pstChnl->stSemMsgIndex);
			UT_Log(VCORE, WARN,
				"VCORE_ProcessBootResponse: Exiting retransmitted response %d %d for Chnl %d\n", pstChnl->ucMsgIndex,
					tmp_index, pstMsg->channel);
			return;
		}
	}

	pstChnl->ucMsgIndex++;

out:
	UT_MutexUnLock(&pstChnl->stSemMsgIndex);
	VCORE_ProcessFSMhandler(pstChnl, FSM_EVENT_DEV_MSG, pstMsg);
	UT_Log(VCORE, INFO, "VCORE_ProcessBootResponse: Exiting Processing Done\n");
	return;
}

/****************************************************************************
 * VCORE_ProcessResponse : The function does the following things -
 ***************************************************************************/
/*! 
 *  -  Implementation
 *      -#  Extract the channel id from pstMsg\n
 *      -#  Extract channel structure based on channel ID and device ID using
 *          DMGR_GetChannel \n
 *      -#  If the channel is not found, and the device state is eBOOTING
 *          then, Get the supervisory channel structure.
 *      -#  Check for the message index of the received command and compare it
 *          against the expected channel index. 
 *      -#  In crement the message index.
 *      -#  If the message index is OK then call the Request-FSM Handler of 
 *          the head request in SChnl.\n
 *
 *  - Assumptions\n
 *      -#  Message passed is a valid MSP response packet.\n
 *
 *  \return   
 *      None
 *
 *  \param DevId The device from the message has been received
 *  \param pstMsg Message obtained from GTL.
 */
void VCORE_ProcessResponse(SDevice *pstDevice, IN gtl_msg_t * pstMsg)
{
	SChnl *pstChnl;
	U8 tmp_index;
	SApiHdr *pstFifo;

	UT_Log(VCORE, INFO, "VCORE_ProcessResponse: Entering on Device %u\n", pstDevice->DevId);

	/*Get the channel structure in order to call the request processing function */
	pstChnl = DMGR_GetChannel(pstDevice, pstMsg->channel);

	UT_MutexLock(&pstChnl->stSemMsgIndex);

	if (!(pstDevice->eItfType == ePCI_ITF))
	{

		pstFifo = (SApiHdr *) pstMsg->fifo;
		tmp_index = pstFifo->index;

		if (pstChnl->ucMsgIndex != tmp_index)
		{
			/*This is an invalid (retransmitted) message */
			UT_MutexUnLock(&pstChnl->stSemMsgIndex);
			UT_Log(VCORE, WARN,
				"VCORE_ProcessResponse: Exiting retransmitted response %d %d for Chnl %d\n", pstChnl->ucMsgIndex,
					tmp_index, pstMsg->channel);
			return;
		}
	}

	pstChnl->ucMsgIndex++;
	UT_MutexUnLock(&pstChnl->stSemMsgIndex);
	VCORE_ProcessFSMhandler(pstChnl, FSM_EVENT_DEV_MSG, pstMsg);
	UT_Log(VCORE, INFO, "VCORE_ProcessResponse: Exiting Processing Done\n");
	return;
}

/****************************************************************************
 * VCORE_Close : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Currently there is nothing to do.\n
 *  
 *  \return None 
 */
void VCORE_Close()
{
	pfnEventCallback_g = NULL;
	return;
}

/****************************************************************************
 * VCORE_RegisterEventCallback : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Override the global pfnEventCallback_g\n
 *  
 *  \return None 
 */
VSTATUS VCORE_RegisterEventCallback(IN U32 uiID, IN U8 ucLevel, IN PFNEventCallback pfnEventCallback)
{
	SDevice *pstDev = NULL;
	SChnl *pstChnl = NULL;
	VSTATUS Ret = SUCCESS;
	
	switch(ucLevel)
	{
	case EVENT_LEVEL_CONN:
		if (NULL != (pstChnl = DMGR_GetConnection(uiID)))
		{
			pstChnl->pfnEventCallback = pfnEventCallback;
		}
		else
		{
			Ret = VAPI_ERR_INVALID_CONNID;
		}
		break;
	case EVENT_LEVEL_DEVICE:
		if (NULL != (pstDev = DMGR_GetDevice(uiID)))
		{
			pstDev->pfnEventCallback = pfnEventCallback;
			pstDev->stSupvChnl.pfnEventCallback = pfnEventCallback;
		}
		else
		{
			Ret = VAPI_ERR_INVALID_DEVID;
		}
		break;
	case EVENT_LEVEL_GENERIC:
		pfnEventCallback_g = pfnEventCallback;
		break;
	default:
		UT_ErrorLog(VCORE,"VCORE_RegisterEventCallback: Invalid call level = %d uiID= %d.\n",ucLevel,uiID);
		Ret = VAPI_ERR_INVALID_PARAM;
	}
	return Ret;
}

/****************************************************************************
 * VCORE_SendMSPReq : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Update the message index corresponding to Channel
 *      -#  Start timer for response
 *      -#  Call GTL_Write
 *
 *  - Assumption
 *      -# pstChnl and pstMsg are valid and Non NULL
 *
 *  \return None 
 *
 *  \param pstChnl The channel on which request has to be sent.
 *  \param pstMsg The message to be sent.
 */
VSTATUS VCORE_SendMSPReq(IN SChnl * pstChnl, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;

	/* Flag the message as non boot command */
	switch (pstChnl->pstDev->eItfType)
	{
	case ePCI_ITF:
		pstMsg->type |= PCI; 
		break;

	case eCSM_ITF:
		pstMsg->type |= CSME; 
		break;

	case ePOS_ITF:
		pstMsg->type |= POS;
		break;

	default:
		UT_ErrorLog(GTL, "VCORE_SetupCmdMsg: Unsupported interface type (%u)\n", pstChnl->pstDev->eItfType);
		Status = VAPI_ERR_UNSUPP_FEATURE; 
		goto out;
	}

	/*Set Message index */
	UT_MutexLock(&pstChnl->stSemMsgIndex);
	if (pstMsg->fifo_size)
	{
		if (pstMsg->type & POS)
		{
			pstMsg->channel = pstChnl->usMSPChnlId;
			((SApiHdr *)(pstMsg->fifo))->index = pstChnl->ucMsgIndex;
		}
		else
		{
			if (pstMsg->type & BOOT)
			{
				pstMsg->channel = 0;
				if (pstMsg->type & CSME)
				{
					((SBootHdr *)(pstMsg->fifo))->index = pstChnl->ucMsgIndex;
				}
			}
			else
			{
				pstMsg->channel = pstChnl->usMSPChnlId;
				((SApiHdr *)(pstMsg->fifo))->index = pstChnl->ucMsgIndex;
			}
		}
	}

	UT_MutexUnLock(&pstChnl->stSemMsgIndex);

#ifndef VAPI_NODBG
	GTL_PrintMessage(pstMsg, VCORE, 1);
#endif

	/*Start Timer for response timeout */
	UT_TimerStart(pstChnl->pstRespTimer, stVAPIConfig_g.uiMspRespTimeout, VCORE_RespTimeoutHdlr, (void *)pstChnl);

	/*Call GTL_Write */
	Status = GTL_Write(pstChnl->pstDev->DevId, pstMsg);

	/*Stop Timer if the command is not correctly sent */
	if (Status != SUCCESS)
		UT_TimerStop(pstChnl->pstRespTimer);

	/* Statistics */
	UT_MutexLock(&vapi_stat.mutex_stat);
	vapi_stat.api_frames_sent++;
	UT_MutexUnLock(&vapi_stat.mutex_stat);

out:
	return Status;
}

VSTATUS VCORE_SendIndToMsp(IN SChnl * pstChnl, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;

#ifndef VAPI_NODBG
	GTL_PrintMessage(pstMsg, VCORE, 1);
#endif

	Status = GTL_Write(pstChnl->pstDev->DevId, pstMsg);
	UT_MutexLock(&vapi_stat.mutex_stat);
	vapi_stat.api_frames_sent++;
	UT_MutexUnLock(&vapi_stat.mutex_stat);

	return Status;
}

/****************************************************************************
 * VCORE_RespTimeoutHdlr : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Extract channel identification
 *      -#  Form the special timeout response
 *      -#  Proceed for normal response processing
 *
 *  \return None 
 *
 *  \param pvData Pointer to channel structure on which request was sent.
 */
void VCORE_RespTimeoutHdlr(IN void *pvData)
{
	SChnl *pstChnl = (SChnl *) pvData;

	/*Process this response */
	UT_ErrorLog(VCORE, "VCORE_RespTimeoutHdlr: TIMEOUT: Request on Conn %d Chnl %d\n",
		    pstChnl->ConnId, pstChnl->usMSPChnlId);
	UT_MutexLock(&pstChnl->stSemMsgIndex);
	pstChnl->ucMsgIndex++;
	UT_MutexUnLock(&pstChnl->stSemMsgIndex);

	VCORE_ProcessFSMhandler(pstChnl, FSM_EVENT_TIMEOUT, NULL);
	return;
}


/****************************************************************************
 * VCORE_RecPlayTimeoutHdlr : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#
 *
 *  \return None 
 *
 *  \param pvData Pointer to channel structure on which request was sent.
 */
void VCORE_RecPlayTerminator(IN void *pvData)
{
	SLstNode *pstNode;
	SChnl *pstChnl = (SChnl *) pvData;

	/*Process this response */
	UT_Log(VCORE, INFO, "VCORE_RecPlayTerminator: Request on Conn %d Chnl %d\n",
	       pstChnl->ConnId, pstChnl->usMSPChnlId);

	UT_MutexLock(&pstChnl->stReqList.stSem);
	/*Get Front node */
	pstNode = __UT_GetFrontNode(&pstChnl->stReqList);
	if (pstNode == NULL)
	{
		UT_ErrorLog(VCORE, "VFSM_StartRecording: Request list empty\n");
		UT_MutexUnLock(&pstChnl->stReqList.stSem);
		return;
	}
	UT_MutexUnLock(&pstChnl->stReqList.stSem);

	VCORE_ProcessFSMhandler(pstChnl, FSM_EVENT_REC_PLAY_TERMINATOR, NULL);
}

/****************************************************************************
 * VCORE_ChannelResetTimeoutHdlr : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#
 *
 *  \return None 
 *
 *  \param pvData Pointer to channel structure on which request was sent.
 */
void VCORE_ChannelResetTimeoutHdlr(IN void *pvData)
{
	SLstNode *pstNode;
	SChnl *pstChnl = (SChnl *) pvData;

	/*Process this response */
	UT_Log(VCORE, INFO, "VCORE_ChannelResetTimeoutHdlr: Request on Conn %d Chnl %d\n",
	       pstChnl->ConnId, pstChnl->usMSPChnlId);

	UT_MutexLock(&pstChnl->stReqList.stSem);
	/*Get Front node */
	pstNode = __UT_GetFrontNode(&pstChnl->stReqList);
	if (pstNode == NULL)
	{
		UT_ErrorLog(VCORE, "VCORE_ChannelResetTimeoutHdlr: Request list empty\n");
		UT_MutexUnLock(&pstChnl->stReqList.stSem);
		return;
	}
	UT_MutexUnLock(&pstChnl->stReqList.stSem);

	VCORE_ProcessFSMhandler(pstChnl, FSM_EVENT_TIMEOUT, NULL);
}

/****************************************************************************
 * VCORE_CheckValidity : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Checks the validity of the cms class /type combination
 *      -#
 *
 */
VSTATUS VCORE_CheckCtCcValidity(U16 fc, U8 ucCmdClass, U8 ucCmdType)
{
	VSTATUS Status = SUCCESS;

	switch(ucCmdClass)
	{
		case CMD_CLASS_STAT_CHANNEL:
			if ((ucCmdType >= MIN_CMD_TYPE_STATS) && (ucCmdType <= MAX_CMD_TYPE_STATS))
				Status = SUCCESS;
			else
				Status = VAPI_ERR_ILLEGAL_CT_FOR_CC;
			break;

		case CMD_CLASS_STAT_DEVICE:
			/*No check on statictics */
			Status = SUCCESS;
			break; 

		case CMD_CLASS_CONF_DEVICE:
		case CMD_CLASS_CONF_CHANNEL:
			switch(ucCmdType)
			{
				case CMD_TYPE_CONF_CHANGE:
				case CMD_TYPE_QUERY:
				case CMD_TYPE_INDICATION_RESP:
					Status = SUCCESS;
					break;

				case CMD_TYPE_CONF_RESP:
				case CMD_TYPE_INDICATION:
				case CMD_TYPE_QUERY_RESP:
				default:
					Status = VAPI_ERR_ILLEGAL_CT_FOR_CC;
			}
			break;

		case CMD_CLASS_REF_DIAG:
		case CMD_CLASS_OPEN_DIAG:
			/*No check on diagnostics */
				Status = SUCCESS;
			break;

		case CMD_CLASS_LEGACY_MSG:
			switch(ucCmdType)
			{
			 	case CMD_TYPE_SETTSA:
				case CMD_TYPE_CHANRESET:
				case CMD_TYPE_SETSYNC:
					Status = SUCCESS;
				break;

				default:
					Status =  VAPI_ERR_ILLEGAL_CT_FOR_CC;
			}
		default:
			Status = VAPI_ERR_ILLEGAL_COMMAND;
	}

	if (Status == VAPI_ERR_ILLEGAL_COMMAND)
		UT_ErrorLog(VCORE, "VCORE_CheckCtCcValidity: Unknow Command Class (0x%02x) (FC 0x%04x)\n", ucCmdClass, fc);

	else if (Status == VAPI_ERR_ILLEGAL_CT_FOR_CC)
		UT_ErrorLog(VCORE, "VCORE_CheckCtCcValidity: Illegal Command Type (0x%02x) for Command Class (0x%02x) (FC 0x%04x)\n", ucCmdType, ucCmdClass, fc);

	else if (Status != SUCCESS) 
		UT_ErrorLog(VCORE, "VCORE_CheckCtCcValidity: Unexpected Command Class (0x%02x) / Type (0x%02x) (FC 0x%04x)\n", ucCmdClass, ucCmdType, fc);

	return Status;
}

/****************************************************************************
 * VCORE_CheckCmd : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *	-#  Check the message lenght integrity
 *
 *  \return 
 *      Returns SUCCESS when the message lenght and fifo length are OK 
 *
 *  \param pstMsg Message to be parsed could contain multiple commands.
 */
VSTATUS VCORE_CheckCmd(SDevice *pstDevice, SMsg * pstMsg)
{
	int len;

	/* check if fifo size is big enough for msg header */
	if (pstMsg->uiFifoSize < FIFO_MIN_SIZE)
	{
		UT_ErrorLog(VCORE, "VCORE_CheckCmd: pstMsg->uiFifoSize(%u) < FIFO_MIN_SIZE\n",pstMsg->uiFifoSize);
		return VAPI_ERR_INVALID_PARAM;
	}

	len = ((U8 *) pstMsg->pusFifo)[CIDX_FIFO_LEN];

	/* check is message length is correct */
	if (len < FIFO_MIN_SIZE)
	{
		UT_ErrorLog(VCORE, "VCORE_CheckCmd: len(%u)< FIFO_MIN_SIZE\n", len);
		return VAPI_ERR_INVALID_PARAM;
	}

	/* check if fifo is big enough for this message */
	if (pstMsg->uiFifoSize < len)
	{
		UT_ErrorLog(VCORE, "VCORE_CheckCmd: pstMsg->uiFifoSize < len(%u)\n", len);
		return VAPI_ERR_INVALID_PARAM;
	}

	if (pstMsg->uiFifoSize > pstDevice->MaxMsgSize)
	{
		UT_ErrorLog(VCORE, "VCORE_CheckCmd: pstMsg->uiFifoSize (%d) > Max fifo size (%d)\n", pstMsg->uiFifoSize, pstDevice->MaxMsgSize);
		return  VAPI_ERR_INVALID_PARAM;
	}

	return SUCCESS;
}

/****************************************************************************
 * VCORE_GetNextCmd : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Parse the fifo and modify the fifo pointer to next Command inside
 *          the input SMsg structure.
 *      -#  Modify fifo size accordingly
 *      -#  Return number of remaining bytes in the fifo.
 *
 *  \return 
 *      Returns SUCCESS when there is no next cmd or VAPI_ERR_UNEXPECTED_MSP_RESPONSE 
 *	when there is a issue in the command format.
 *
 *  \param pstMsg Message to be parsed for multiple commands.
 */
int VCORE_GetNextCmd(SMsg * pstMsg)
{
	int len;

	len = ((U8 *) pstMsg->pusFifo)[CIDX_FIFO_LEN];
	len = ALIGN_MESSAGE_LENGTH(len);

	/* pstMsg->pusFifo is a U16 buffer */
	pstMsg->pusFifo += len/2;
	pstMsg->uiFifoSize -= len;

	return pstMsg->uiFifoSize;
}

/****************************************************************************
 * VCORE_OpenConnection : The function does the following things -
 ***************************************************************************/
VSTATUS VCORE_OpenConnection(IN SChnl *pstChnl)
{

	if (UT_MutexInit(&pstChnl->stSemMsgIndex) < 0)
		goto err1;

	if (UT_MutexInit(&pstChnl->stAccessFsmHndlr) < 0)
		goto err2;

	if (UT_MutexInit(&pstChnl->stRecPlayInfo.stSemRecPlay) < 0)
		goto err3;

	if (UT_ListInit(&pstChnl->stReqList) < 0)
		goto err4;

	pstChnl->bIPHdrSet = False;
	pstChnl->pstRespTimer = UT_TimerCreate();
	pstChnl->ucMsgIndex = 1;
	pstChnl->stRecPlayInfo.eRecPlayState = eVop;
	pstChnl->stRecPlayInfo.uiStopPlayCount = 0;
	pstChnl->stRecPlayInfo.uiStopRecCount = 0;
	pstChnl->stRecPlayInfo.eStopAnnType = eDRAIN;

	return SUCCESS;

      err4:
	UT_MutexDestroy(&pstChnl->stRecPlayInfo.stSemRecPlay);
      err3:
        UT_MutexDestroy(&pstChnl->stAccessFsmHndlr);
      err2:
	UT_MutexDestroy(&pstChnl->stSemMsgIndex);
      err1:

	return FAILURE;
}

/****************************************************************************
 * VCORE_CleanParticipant : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Delete this channel form the conference participant array
 *  \return 
 *  None
 *
 *  \param pstChnl the channel to be removed.
 */
void VCORE_CleanParticipant(IN SChnl * pstChnl)
{
	if (((pstChnl->ePartType == eLSP) ||
		(pstChnl->ePartType == eRSP) ||
		(pstChnl->ePartType == eTRANSP) ||
		(pstChnl->ePartType == eLSPWB) ||
		(pstChnl->ePartType == eRSPWB)) &&
		(pstChnl->pstConf != NULL))
	{
		pstChnl->pstConf->apstParticipant[pstChnl->usMSPChnlId] = NULL;
		pstChnl->pstConf->usNumOfParticipant--;
	}
}
/****************************************************************************
 * VCORE_CloseConnection : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Delete all the lists semaphores,timers etc
 *      -#  Free memory allocated to pstChnlParams.
 *      -#  Remove it from Device manager.
 *
 *  \return 
 *  None
 *
 *  \param pstChnl the channel to be removed.
 */
void VCORE_CloseConnection(IN SChnl * pstChnl)
{
	SLstNode *pstNode;
	SVapiReq *pstVapiReq;
	SVapiReq *pstParentReq;
	if (pstChnl == NULL)
		return;


	UT_TimerDelete(pstChnl->pstRespTimer);
	pstChnl->pstRespTimer = NULL;

	/*Cleanup Request List */
	UT_MutexLock(&pstChnl->stReqList.stSem);

	/* Parse all request in the list */
	while ((pstNode = __UT_GetFrontNode(&pstChnl->stReqList)) != NULL)
	{
		pstVapiReq = UT_ContainerOf(pstNode, SVapiReq, stNode);
		pstParentReq = pstVapiReq->pstParentReq;

		/* In case of parent VCORE_DoReqCompletion to proceed with clean up */
		if (pstParentReq == NULL)
		{
			/*Unlock the request during the completion */
			UT_MutexUnLock(&pstChnl->stReqList.stSem);

			/* make sure to release a pending semaphore for SYNC request 
			or call the callback function for ASYNC request */
			VCORE_DoReqCompletion(pstChnl, pstVapiReq, VAPI_REQ_COMPLETED);

			UT_MutexLock(&pstChnl->stReqList.stSem);

		}
		/* In case a child request free it */
		else
		{
			__UT_RemoveNode(&pstChnl->stReqList, &pstVapiReq->stNode);
			/*Free memories of this chid request */
			if (pstVapiReq->pvExecData != NULL)
				UT_FreeMem(pstVapiReq->pvExecData);

			if (pstVapiReq->pvCallbackData != NULL)
				UT_FreeMem(pstVapiReq->pvCallbackData);

			VCORE_FreeRequest(pstVapiReq);
		}
	}

	UT_MutexUnLock(&pstChnl->stReqList.stSem);

	UT_ListDestroy(&pstChnl->stReqList);
	UT_MutexDestroy(&pstChnl->stSemMsgIndex);
	UT_MutexDestroy(&pstChnl->stRecPlayInfo.stSemRecPlay);
	UT_MutexDestroy(&pstChnl->stAccessFsmHndlr);

	if (pstChnl->pstChnlParams != NULL)
	{
		UT_FreeMem(pstChnl->pstChnlParams);
	}

}

 /****************************************************************************
 * VCORE_PurgeDevResources : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# The device entry is removed from the internal datastructures.
 *      -#  It will not wait for any requests to complete.
 *      -#  It will not destroy any existing channels on that device .
 *
 *  \return 
 *  None
 *
 *  \param pstDevice the device to be purged.
 */
int VCORE_PurgeDevResources(SDevice * pstDevice)
{
		VSTATUS Status = SUCCESS;

		UT_Log(APPITF, INFO, "VCORE_PurgeDevResources: Removing Device from GTL\n");
		if ((Status = GTL_Close(pstDevice->DevId)) != SUCCESS)
		{
			UT_ErrorLog(APPITF, "VCORE_PurgeDevResources: GTL_Close Failed %d\n", Status);
		}

		DMGR_RemoveDevice(pstDevice->DevId);
		DMGR_PurgeDevice(pstDevice);

		UT_FreeMem(pstDevice->papstChannels);
		UT_FreeMem(pstDevice->apstConference);

		UT_Log(APPITF, INFO, "VCORE_PurgeDevResources:Removing Device from DMGR\n");

		VCORE_CloseConnection(&pstDevice->stSupvChnl);

		UT_FreeMem(pstDevice);

		return Status;
}

/****************************************************************************
 * VCORE_CheckPCIBootStatus : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Check if the boot message receive is a BRM ACK
  *
 *  \return 
 *  SUCCESS When the message is the expected one \n
 *  VAPI_ERR_UNEXPECTED_MSP_RESPONSE When the message is not the expected one \n
 *
 *  \param pstMsg Response message to be checked
 */
VSTATUS VCORE_CheckPCIBootStatus(U16 Event, IN gtl_msg_t * pstMsg)
{

	U8 ReceivedResp;

	if (Event == FSM_EVENT_TIMEOUT)
		return VAPI_ERR_MSP_NO_RESPONSE;

	ReceivedResp = (pstMsg->mailbox.reg3 >> 8) & 0xff;
	UT_Log(VCORE, INFO, "VCORE_CheckPCIBootMsg: msg_id = 0x%02x for cmd 0x%02x\n", ReceivedResp, pstMsg->mailbox.reg1);

	if (ReceivedResp == CMD_TYPE_BRM_CMD_ACK)
		return SUCCESS;
	else
	{
		UT_ErrorLog(VCORE, "VCORE_CheckPCIBootMsg: Unexpected MSP PCI Boot Response \n");
		UT_ErrorLog(VCORE," Got cmd id :%x \t Expected cmd id : %x \n", ReceivedResp, CMD_TYPE_BRM_CMD_ACK);
		return VAPI_ERR_UNEXPECTED_MSP_RESPONSE;
	}
}

/****************************************************************************
 * VCORE_CheckStatus : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Check whether its a timeout message
 *      -#  Check for expected channel Id
 *      -#  Check for expected Command Class
 *      -#  Check for expected Command Type
 *      -#  Check for expected Function code
 *      -#  Check for the status in FIFO
 *
 *  \return 
 *  SUCCESS When the message is the expected one and response from MSP is OK.\n
 *  VAPI_ERR_MSP_NO_RESPONSE When the message is a timeout message\n
 *  VAPI_ERR_UNEXPECTED_MSP_RESPONSE When the message is not expected\n
 *  -ve of the Status sent by MSP
 *
 *  \param pstMsg Response message to be checked
 *  \param usExpChnlId Expected Channel Id 
 *  \param ucExpCmdCls Expected Command Class
 *  \param ucExpCmdtyp Expected command Type
 *  \param ucExpFCode  Expected Function Code
 */
VSTATUS VCORE_CheckStatus(U16 Event, IN gtl_msg_t * pstMsg, 
			  IN U16 usExpChnlId, IN U8 ucExpCmdCls, IN U8 ucExpCmdTyp, IN U16 usExpFCode)
{
	VSTATUS Status;
	U16 *pausFifo = NULL;
	U8 *paucFifo = NULL;
	U8 CC, CT;
	U16 FC;

	switch (Event)
	{
	case FSM_EVENT_TIMEOUT:
		Status = VAPI_ERR_MSP_NO_RESPONSE;
		break;

	case FSM_EVENT_REC_PLAY_TERMINATOR:
		Status = VAPI_IND_STOP_REC_PLAY;
		break;

	case FSM_EVENT_PLAY_REC_STOP:
		Status = VAPI_IND_STOP_REC_PLAY;
		break;		

	case FSM_EVENT_DEV_MSG:

		if (pstMsg->fifo == NULL)
		{
			Status = VAPI_ERR_UNEXPECTED_MSP_RESPONSE;
			break;
		}

		pausFifo = pstMsg->fifo;
		paucFifo = pstMsg->fifo;

		if (pstMsg->type & BOOT)
		{
			CC = ((SBootHdr *)pausFifo)->cmd_class;
			CT = ((SBootHdr *)pausFifo)->cmd_type;
			FC = UT_LE2CPU16(((SBootHdr *)pausFifo)->func_code);
		}
		else
		{
			CC = ((SApiHdr *)pausFifo)->cmd_class;
			CT = ((SApiHdr *)pausFifo)->cmd_type;
			FC = UT_LE2CPU16(((SApiHdr *)pausFifo)->func_code);
		}

		/*Ensure that this is the expected response else raise alarm */
		if (pstMsg->channel != usExpChnlId
			|| CC != ucExpCmdCls
			|| CT != ucExpCmdTyp
			|| FC != usExpFCode)
		{
			Status = VAPI_ERR_UNEXPECTED_MSP_RESPONSE;
			UT_ErrorLog(VCORE, "Unexpected MSP Response ExpCh:%u ActCh %u Msg %x Fifo = 0x%x\n",
				usExpChnlId, pstMsg->channel, pstMsg, paucFifo);
			UT_ErrorLog(VCORE,
				" Got Class : %x Type : %x, FCode %x\t Expected Class : %x Type : %x, FCode %x\n",
				CC, CT, FC, ucExpCmdCls, ucExpCmdTyp, usExpFCode);
			break;
		}

		/*Check for the returned status */
		if (((CC == CMD_CLASS_CONF_CHANNEL
			|| CC == CMD_CLASS_CONF_DEVICE)
			&& CT == CMD_TYPE_CONF_RESP)
			&& (pausFifo != NULL && pausFifo[SIDX_VSTATUS] != 0))
		{
			Status = -UT_LE2CPU16(pausFifo[SIDX_VSTATUS]);
			UT_ErrorLog(VCORE, "FAILURE Returned by MSP -> 0x%X for FC 0x%04X\n", -Status, FC);
			break;
		}

		Status = SUCCESS;

		break;

	default:

		UT_ErrorLog(VCORE, "VCORE_CheckStatus: Unexpected event received (%u)\n", Event);
		Status = VAPI_ERR_UNDEFINED_EVENT;
		break;

	}

	return Status;
}

/****************************************************************************
 * VCORE_DoReqCompletion : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Call IO ompletion callback if provided
 *      -#  Else Wakeup Request completion semaphore.
 *      -#  If blocking call, free the memory allocated for request
 *
 *  \return 
 *  None
 *
 *  \param pstChnl Channel whose request is getting completed.
 *  \param pstVapiReq The request that has got completed
 *  \param bLaunchNext  Whether the pending request should be launched or not.
 *
 */
void VCORE_DoReqCompletion(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, VSTATUS Status)
{
	UT_Log(VCORE, INFO, "VCORE_DoReqCompletion: Entering conn(%u) req(0x%x) reqid(%d) channel(%u)\n",
	       pstChnl->ConnId, pstVapiReq, pstVapiReq->uiReqId, pstChnl->usMSPChnlId);
	/*Call Io Completion Callback or wakeup semaphore */
	/*Remove the request node from this list */
	UT_MutexLock(&pstChnl->stReqList.stSem);
	__UT_RemoveNode(&pstChnl->stReqList, &pstVapiReq->stNode);
	UT_MutexUnLock(&pstChnl->stReqList.stSem);

	/*Free memory */
	if (pstVapiReq->pvExecData != NULL)
		UT_FreeMem(pstVapiReq->pvExecData);

	/*Statistics */
	UT_MutexLock(&vapi_stat.mutex_stat);
	vapi_stat.vapi_frames_queued--;
	vapi_stat.vapi_frames_completed++;
	UT_MutexUnLock(&vapi_stat.mutex_stat);

	/* save the status to be able to report it to the user thread */
	pstVapiReq->Status = Status;

	if (pstVapiReq->pstParentReq)
	{
		/* It's a child request, notify the parent about our status */

		pstVapiReq->pstParentReq->Status = Status;

		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		if (pstVapiReq->pvUserData != NULL)
			UT_FreeMem(pstVapiReq->pvUserData);

		/*Initiate Pending request on this channel */
		VCORE_ProcessRequestList(pstChnl);

		/* the parent (if any ) heritates the context of the child request */
		pstVapiReq->pstParentReq->ucThreadContext = pstVapiReq->ucThreadContext;

		pstVapiReq->pstParentReq->pfnReqFSMHdlr(pstVapiReq->pstParentReq->pstChnl, pstVapiReq->pstParentReq, FSM_EVENT_START, NULL);

		UT_Log(VCORE, DEBUG, "VCORE_DoReqCompletion: Free child request (0x%X)\n", pstVapiReq);
		UT_FreeMem(pstVapiReq);

	}
	else /* Do request completion on a parent request */
	{
		if (pstVapiReq->bIsSync == False) /* if in async mode */
		{
			/* do not call call_back if DoReqCompletion is being called in the user thread and if there is no response for the command is required */
			if ((pstVapiReq->ucThreadContext == VAPI_CONTEXT) || (pstVapiReq->Status == SUCCESS))
			{
				if (!pstVapiReq->pfnCallback)
				{
					UT_ErrorLog(VCORE, "VCORE_DoReqCompletion: pstVapiReq->pfnCallback == NULL\n");
				}
				else
				{
	#ifdef OLD_RESP_CBK
					pstVapiReq->pfnCallback(pstVapiReq->uiID,
							pstVapiReq->Status, pstVapiReq->pvCallbackData,
							pstVapiReq->uiCallbkDataLen, pstVapiReq->uiReqId);
	#else
					pstVapiReq->pfnCallback(pstVapiReq->uiID, pstVapiReq->ucCmdLevel,
							pstVapiReq->Status, pstVapiReq->pvCallbackData,
							pstVapiReq->uiCallbkDataLen, pstVapiReq->uiReqId);
	#endif
				}
			}
	
			if (pstVapiReq->pvCallbackData != NULL)
				UT_FreeMem(pstVapiReq->pvCallbackData);
	
			UT_Log(VCORE, DEBUG, "VCORE_DoReqCompletion: Free request (0x%X)\n", pstVapiReq);
			VCORE_FreeRequest(pstVapiReq);

			/*Do not process the list if we are currently closing the connection*/
			if(Status != VAPI_REQ_COMPLETED)
				/*Initiate Pending request on this channel */
				VCORE_ProcessRequestList(pstChnl);
		}
		else
		/*Wake up the user thread which is waiting for the req to complete */
		{
			UT_Log(VCORE, INFO, "VCORE_DoReqCompletion: Before wakeup compsem(0x%X) reqid(%d)\n",
			pstVapiReq->pstSemReqComp, pstVapiReq->uiReqId);
	
			/*Do not process the list if we are currently closing the connection*/
			if(pstVapiReq->Status != VAPI_REQ_COMPLETED)
				/*Initiate Pending request on this channel */
				VCORE_ProcessRequestList(pstChnl);

			/* wakeup the user thread waiting in VCORE_ProcessRequest*/
			UT_SemPost(pstVapiReq->pstSemReqComp);
		}
	}
	UT_Log(VCORE, INFO, "VCORE_DoReqCompletion: Exiting conn(%u)\n", pstChnl->ConnId);
	return;
}

/****************************************************************************
 * VCORE_CheckReadyReceived : The function does the following things -
 ***************************************************************************/
/*! 
  */
VSTATUS VCORE_CheckReadyReceived(U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	U16 *pausFifo = NULL;
	U8 *paucFifo = NULL;
	U8 CC, CT;
	U16 FC;

	switch (Event)
	{
	case FSM_EVENT_TIMEOUT:
		Status = VAPI_ERR_MSP_NO_RESPONSE;
		break;

	case FSM_EVENT_DEV_MSG:

		if (pstMsg->fifo == NULL)
		{
			Status = VAPI_ERR_UNEXPECTED_MSP_RESPONSE;
			break;
		}

		pausFifo = pstMsg->fifo;
		paucFifo = pstMsg->fifo;

		if (pstMsg->type & BOOT)
		{
			CC = ((SBootHdr *)pausFifo)->cmd_class;
			CT = ((SBootHdr *)pausFifo)->cmd_type;
			FC = UT_LE2CPU16(((SBootHdr *)pausFifo)->func_code);
		}
		else
		{
			Status = VAPI_ERR_UNEXPECTED_MSP_RESPONSE;
			break;
		}

		if ((CC == CMD_CLASS_ETH_BOOT_MSG) && (CT == CMD_TYPE_BRM_READY))
		{
			Status = SUCCESS;
		}
		else
		{
			Status = VAPI_ERR_UNEXPECTED_MSP_RESPONSE;
			UT_ErrorLog(VCORE, "VCORE_CheckReadyReceived: Unexpected MSP Response");
		}
		break;

	/* do nothing, just wait */
	case FSM_EVENT_START:
		Status = VAPI_ERR_WAIT_READY;
		break;

	default:

		UT_ErrorLog(VCORE, "VCORE_CheckReadyReceived: Unexpected event received (%u)\n", Event);
		Status = VAPI_ERR_UNDEFINED_EVENT;
		break;

	}

	return Status;
}

/***************************************************************************
 * VCORE_ConnectionGenericCheck: The function does the following things -
 ***************************************************************************/
/*!
 *  1. Check if the library is initialised
 *  2. Check if the connection is valid. If so update the pointer to the connection structure
 *  3. Check if the device is in the rigth state
 *
 *  \return
 *  SUCCESS \n
 *  VAPI_ERR_LIB_NOT_INITIALIZED\n
 *  VAPI_ERR_INVALID_CONNID\n
 *  VAPI_ERR_DEV_IS_NOT_UP\n
 *
 *  \param ConnID    It is connaction ID
 *  \param FName     Function name from where we come from
 *                    or connection level.
 *  \param ppstChnl  pointer to the address updated by DMGR_GetConnection
*/
VSTATUS VCORE_ConnectionGenericCheck(IN CONNID ConnId, char *FName, SChnl ** ppstChnl)
{
	VSTATUS Status = SUCCESS;
	SChnl *pstChnl;

	if (!bVapiInitialized_g)
	{
		/*Used printf instead of UT_Log because UT is not yet initialized */
		printf("%s: VAPI library not initialized.\n", FName);
		Status = VAPI_ERR_LIB_NOT_INITIALIZED;
		goto err1;
	}

	pstChnl = DMGR_GetConnection(ConnId);

	if (pstChnl == NULL)
	{
		UT_ErrorLog(APPITF, "%s: Invalid Conn id %u\n", FName, ConnId);
		Status = VAPI_ERR_INVALID_CONNID;
		goto err1;
	}

	*ppstChnl = pstChnl;

	if (pstChnl->pstDev->eState != eUP)
	{
		UT_ErrorLog(APPITF, "%s: Device is not up\n", FName);
		Status = VAPI_ERR_DEV_IS_NOT_UP;
		goto err1;
 	}

	if (!pstChnl->pstDev->bIsDeviceInitialized)
	{
		UT_ErrorLog(APPITF, "%s: Device(%u) not initialized\n", FName,pstChnl->pstDev->DevId);
		Status = VAPI_ERR_DEVICE_NOT_INITIALIZED;
	}

      err1:
	return Status;

}

/***************************************************************************
 * VCORE_DeviceGenericCheck: The function does the following things -
 ***************************************************************************/
/*! 
 *  1. Check if the library is initialised 
 *  2. Check if the device is valid. If so update the pointer to the device structure 
 *  3. Check if the device is in the rigth state
 *
 *  \return
 *  SUCCESS \n
 *  VAPI_ERR_LIB_NOT_INITIALIZED\n
 *  VAPI_ERR_INVALID_DEVID\n
 *  VAPI_ERR_DEV_IS_NOT_UP\n
 *
 *  \param DevID    It is device ID
 *  \param FName    Function name from where  we come from
 *                  or connection level.
 *  \param ppstDevice pointer to the address updated by DMGR_GetDevice
*/
VSTATUS VCORE_DeviceGenericCheck(IN DEVID DevId, char *FName, SDevice ** ppstDevice)
{
	VSTATUS Status = SUCCESS;
	SDevice *pstDevice;

	if (!bVapiInitialized_g)
	{
		/*Used printf instead of UT_Log because UT is not yet initialized */
		printf("%s: VAPI library not initialized.\n", FName);
		Status = VAPI_ERR_LIB_NOT_INITIALIZED;
		goto err1;
	}

	pstDevice = DMGR_GetDevice(DevId);

	if (pstDevice == NULL)
	{
		UT_ErrorLog(APPITF, "Invalid device id\n");
		Status = VAPI_ERR_INVALID_DEVID;
		goto err1;
	}

	*ppstDevice = pstDevice;
	if (pstDevice->eState != eUP)
	{
		UT_ErrorLog(APPITF, "%s: Device is not up.\n", FName);
		Status = VAPI_ERR_DEV_IS_NOT_UP;
	}
      err1:
	return Status;

}

/***************************************************************************
 * VCORE_ConferenceGenericCheck: The function does the following things -
 ***************************************************************************/
/*!
 *  1. Check if the library is initialised
 *  2. Check if the conference is valid. If so update the pointer to the connection structure
 *  3. Check if the device is in the rigth state
 *
 *  \return
 *  SUCCESS \n
 *  VAPI_ERR_LIB_NOT_INITIALIZED\n
 *  VAPI_ERR_INVALID_CONNID\n
 *  VAPI_ERR_DEV_IS_NOT_UP\n
 *
 *  \param Confid    It is conference ID
 *  \param FName     Function name from where we come from
 *                    or connection level.
 *  \param ppstConf  pointer to the address updated by DMGR_GetConnection
*/
VSTATUS VCORE_ConferenceGenericCheck(IN CONFID ConfId, char *FName, SConference ** ppstConf)
{
	VSTATUS Status = SUCCESS;
	SConference *pstConf;

	if (!bVapiInitialized_g)
	{
		/*Used printf instead of UT_Log because UT is not yet initialized */
		printf("%s: VAPI library not initialized.\n", FName);
		Status = VAPI_ERR_LIB_NOT_INITIALIZED;
		goto err1;
	}

	pstConf = DMGR_GetConfById(ConfId);

	if (pstConf == NULL)
	{
		UT_ErrorLog(APPITF, "%s: Invalid Conference id %u\n", FName, ConfId);
		Status = VAPI_ERR_INVALID_CONFID;
		goto err1;
	}

	*ppstConf = pstConf;

	if (pstConf->pstDev->eState != eUP)
	{
		UT_ErrorLog(APPITF, "%s: Device is not up\n", FName);
		Status = VAPI_ERR_DEV_IS_NOT_UP;
	}
      err1:
	return Status;

}

/***************************************************************************
 * VCORE_AllocateRequest: The function does the following things -
 ***************************************************************************/
/*!
 *  1. Allocate memory for the request
 *  2. if ok allocate memory for the UserData (if any)
 *
 *  \b Inputs-Outputs: \n
 *	\li U16 UserDataSize.
 *
 *  \b Returns: \n
 *	\li pointer to allocated Request
*/
SVapiReq *VCORE_AllocateRequest(size_t szUserDataSize)
{
	SVapiReq *pstVapiReq;

	pstVapiReq = UT_Calloc(1, sizeof(SVapiReq));
	if (!pstVapiReq)
	{
		UT_ErrorLog(VCORE, "VCORE_AllocateRequest:No memory available\n");
		goto err0;
	}

	if (!szUserDataSize)
		goto out;

	pstVapiReq->pvUserData = UT_Calloc(1, szUserDataSize);
	if (!pstVapiReq->pvUserData)
	{
		UT_ErrorLog(VCORE, "VCORE_AllocateRequest UserData:No memory available\n");
		goto err1;
	}

out:
	return pstVapiReq;

err1:
	UT_FreeMem(pstVapiReq);

err0:
	return NULL;
}

/***************************************************************************
 * VCORE_FreeRequest: The function does the following things -
 ***************************************************************************/
/*!
 *  1. Free request userdata meory (if any)
 *  2. Free request memory
 *
 *  \b Inputs-Outputs: \n
 *	\li SVapiReq *pstVapiReq pointer to allocated Request
 *
 *  \b Returns: \n
 *	\li none
*/
void VCORE_FreeRequest(SVapiReq *pstVapiReq)
{
	if (pstVapiReq->pvUserData != NULL)
		UT_FreeMem(pstVapiReq->pvUserData);

	UT_FreeMem(pstVapiReq);
}
/***************************************************************************
 * VCORE_SetRequest: The function does the following things -
 ***************************************************************************/
/*!
 *  1. set the callback function if any
 *  2. set the vcore function to call
 *  3. set the data to pass to the vcore function
 *	4. set the connection ID and level
 *  \return
 *  void \n
 *
 *  \param void* function to process
 *  \param void* data to process
 *  \param U32 Connection ID;
*/
void VCORE_SetRequest(SVapiReq * pstVapiReq,
			       SRequest * pstRequest, void *vcore_function, U32 id)
{

	pstVapiReq->usReqState = 0;
	pstVapiReq->ucThreadContext = USER_CONTEXT;
	pstVapiReq->bIsSync = True;

	/* these variables are used by PassThru only */
	pstVapiReq->pvCallbackData = NULL;
	pstVapiReq->uiCallbkDataLen = 0;

	if (pstRequest != NULL)
	{
		pstVapiReq->pfnCallback = pstRequest->pfnIoCompCallback;
		pstVapiReq->uiReqId = pstRequest->uiReqId;

		/* if there is a callback function the request is flaged ASYNC */
		if (pstVapiReq->pfnCallback != NULL)
			pstVapiReq->bIsSync = False;
	}

	pstVapiReq->pfnReqFSMHdlr = vcore_function;
	pstVapiReq->uiID = id;
}


/***************************************************************************
 * VCORE_SetConnectionRequest: The function does the following things -
 ***************************************************************************/
/*!
 *  1. set the callback function if any
 *  2. set the vcore function to call
 *  3. set the data to pass t the vcore function
 *  4. set the connection ID and level
 *  \return
 *  void \n
 *
 *  \param void* function to process
 *  \param void* data to process
 *  \param U32 Connection ID;
*/
void VCORE_SetConnectionRequest(SVapiReq * pstVapiReq, SRequest * pstRequest, void *vcore_function, U32 id)
{

	VCORE_SetRequest(pstVapiReq, pstRequest, vcore_function, id);
	pstVapiReq->ucCmdLevel = CMD_LEVEL_CONN;
}

/***************************************************************************
 * VCORE_SetConferenceRequest: The function does the following things -
 ***************************************************************************/
/*!
 *  1. set the callback function if any
 *  2. set the vcore function to call
 *  3. set the data to pass t the vcore function
 *  4. set the Conference ID and level
 *
 *  \return
 *  void \n
 *
 *  \param void* function to process
 *  \param void* data to process
 *  \param U32 Conference Id;
*/
void VCORE_SetConferenceRequest(SVapiReq * pstVapiReq, SRequest * pstRequest, void *vcore_function, U32 id)
{

	VCORE_SetRequest(pstVapiReq, pstRequest, vcore_function, id);
	pstVapiReq->ucCmdLevel = CMD_LEVEL_CONF;
}

/***************************************************************************
 * VCORE_SetDeviceRequest: The function does the following things -
 ***************************************************************************/
/*!
 *  1. set the callback function if any
 *  2. set the vcore function to call
 *  3. set the data to pass t the vcore function
 *  4. set the device ID and level
 *
 *  \return
 *  void \n
 *
 *  \param void* function to process
 *  \param void* data to process
 *  \param U32 Device Id;
*/
void VCORE_SetDeviceRequest(SVapiReq * pstVapiReq, SRequest * pstRequest, void *vcore_function, U32 id)
{

	VCORE_SetRequest(pstVapiReq, pstRequest, vcore_function, id);
	pstVapiReq->ucCmdLevel = CMD_LEVEL_DEVICE;
}

/****************************************************************************
 * VCORE_ExtractSupervisorParams : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  None
 *
 *  \param pstChnl 		Channel structure
 *  \param pstVoIPParams Extracted params
 *  \param pausFifo Fifo from which to extract.
 */

void VCORE_ExtractSupervisorParams(IN SChnl *pstChnl, IN U16 *pausFifo)
{
	/* retrieve the fifo size of the command sent, minus fifo header size /2 it gives the number of params */ 
	U8 ucNumberOfParams = (((U8) UT_LE2CPU16(pausFifo[CIDX_FIFO_LEN]) - FIFO_MIN_SIZE ) / 2);

	switch(UT_LE2CPU16(pausFifo[SIDX_FUNC_CODE]))
	{
		case FC_IP_SERVICE_CONFIG:
			pstChnl->pstDev->eEthHdrMode = eMODE_IP;
			break;

		case FC_SET_ALT_HDR_DEVICE:
			pstChnl->pstDev->eEthHdrMode = eMODE_ALT;
			break;

		case FC_IP_ADDRESS:
			pstChnl->pstDev->eSrcIpMode = eSINGLE_IP;
			UT_MemCopy(&(pstChnl->pstDev->uiSrcIpAddr),&((U16*)pausFifo)[4],sizeof(U32));
			break;

		case FC_MULTI_IP_ADDR_LIST:
			pstChnl->pstDev->eSrcIpMode = eMULTIPLE_IP;
			break;

		/* An IPv4 or IPv6 header (or both) can be sent using VAPI_PassThru.
		So we consider that a valid ETH header is set in both cases */
		case FC_SET_ETH_HDR:
			pstChnl->pstDev->bEthHdrSet = True;
			break;

		case FC_SPU_FEATURES_CONTROL:
			pstChnl->pstDev->ucSPUSentStatus = SPU_CTRL_SENT;

			pstChnl->pstDev->usSpuFeatureMask = UT_LE2CPU16(pausFifo[4]);

			/*If 2 params */
			if (ucNumberOfParams > 1)
				pstChnl->pstDev->usSpuFeatureMask = UT_LE2CPU16((pausFifo[5] << 16));

			break;

		case FC_CSME_MULTI_CMD:
			pstChnl->pstDev->bMultiCmdEnabled = UT_LE2CPU16(pausFifo[4]);
			break;

		default:
			/* other supervisor commands not tracked */
			break;
	}
}

#define GET_BIT_FIELD_VAL(val, offset, n_bits) (((val) >> (offset)) & ((1 << (n_bits)) - 1))
/****************************************************************************
 *   : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  None
 *
 *  \param pstChnl 		Channel structure
 *  \param pstVoIPParams Extracted params
 *  \param pausFifo Fifo from which to extract.
 */
void VCORE_ExtractVoIPParams(IN SChnl *pstChnl, OUT SVoIPChnlParams *pstVoIPParams, IN U16 *pausFifo)
{
	/* index of the 1st param in the fifo*/
	int j, i;
	U16 usVopenaMode;

	/* retrieve the fifo size of the command sent, minus fifo header size /2 it gives the number of params */ 
	U8 ucNumberOfParams = (((U8) UT_LE2CPU16(pausFifo[CIDX_FIFO_LEN]) - FIFO_MIN_SIZE ) / 2);

	switch(UT_LE2CPU16(pausFifo[SIDX_FUNC_CODE]))
	{
		case FC_VOIP_VOPDIS:
			/*Disable Vopena*/
			pstVoIPParams->stVopena.mode = VOIP_VOPENA_MODE_DISABLE;
			pstChnl->bIsActive = False;
			break;

		case FC_VOIP_VOPENA:
			usVopenaMode = UT_LE2CPU16(pausFifo[4]);

			switch (usVopenaMode)
			{

			case VOIP_VOPENA_MODE_DISABLE:

				pstChnl->bIsActive = False;
				pstVoIPParams->stVopena.mode = usVopenaMode;
				break;

			case VOIP_VOPENA_MODE_ENABLE_RTP: /*Vopena Enable RTP*/
			case VOIP_VOPENA_MODE_ENABLE_T38oRTP:

				pstChnl->bIsActive = True;
				pstVoIPParams->stVopena.mode = usVopenaMode;

				/* parameter [5] bit(s) <15> */
				pstVoIPParams->stVopena.param_5.word = UT_LE2CPU16(pausFifo[5]);

				/*parameters [9:10]*/
				pstVoIPParams->stVopena.rtp_ssrc = (((U32)(UT_BE2CPU16(pausFifo[9])))<<16) |
								(U32)UT_BE2CPU16(pausFifo[10]);

				if(pstVoIPParams->stVopena.param_5.bits.rtp_cc)
				{
					pstVoIPParams->stVopena.rtp_csrc1 = (((U32)(UT_BE2CPU16(pausFifo[11])))<<16) |
								(U32)UT_BE2CPU16(pausFifo[12]);
				}

				/* if the CC bit field is set to 2 , it means that there are
				CSSRC 1  in params 13,14 */
				if(pstVoIPParams->stVopena.param_5.bits.rtp_cc == 2)
				{
					pstVoIPParams->stVopena.rtp_csrc2 = (((U32)(UT_BE2CPU16(pausFifo[13])))<<16) |
								(U32)UT_BE2CPU16(pausFifo[14]);
				}

				break;

			case VOIP_VOPENA_MODE_ENABLE_NORTP:
			case VOIP_VOPENA_MODE_ENABLE_UDPTL:
			case VOIP_VOPENA_MODE_ENABLE_INBAND_TONE:
			case VOIP_VOPENA_MODE_ENABLE_LSP:

				pstChnl->bIsActive = True;
				pstVoIPParams->stVopena.mode = usVopenaMode;
				break;

			default:
				UT_ErrorLog(VCORE, "VCORE_ExtractVoIPParams: Invalid VOPENA mode (%u) for channel type (%u)\n",
							usVopenaMode, pstChnl->usConnType);
				break;
			}
		
			break;

		case FC_VOIP_VCEOPT:
			/* Except the 1st one, all parameters are optional */
			pstVoIPParams->stVoiceOpt.param_4.word = UT_LE2CPU16(pausFifo[4]);

			if (ucNumberOfParams > 1)
				pstVoIPParams->stVoiceOpt.param_5.word = UT_LE2CPU16(pausFifo[5]);

			if (ucNumberOfParams > 2)
				pstVoIPParams->stVoiceOpt.vad_tune = UT_LE2CPU16(pausFifo[6]);

			if (ucNumberOfParams > 3)
				pstVoIPParams->stVoiceOpt.param_7.word = UT_LE2CPU16(pausFifo[7]);

			if (ucNumberOfParams > 4)
				pstVoIPParams->stVoiceOpt.param_8.word = UT_LE2CPU16(pausFifo[8]);

			break;

		case FC_VOIP_DGAIN:
			/* the number of param for dgain is 2 mandatory */
			pstVoIPParams->stDgain.packet_to_pcm_gain = UT_LE2CPU16(pausFifo[4]);
			pstVoIPParams->stDgain.pcm_to_packet_gain = UT_LE2CPU16(pausFifo[5]);
			break;

		case FC_VOIP_DTMFOPT:
			/* the number of param for DTMF is 1 or 2 */
			pstVoIPParams->stDtmfOpt.param_4.word = UT_LE2CPU16(pausFifo[4]);

			/* the DTMF payload is optional */
			if (ucNumberOfParams > 1)
			{
				pstVoIPParams->stDtmfOpt.dtmf_pt = UT_LE2CPU16(pausFifo[5]);
			}

			break;

		case FC_VOIP_ECHOCAN:
			pstVoIPParams->stEchoCan.param_4.bits.ecenb = GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4]),15,1);
			pstVoIPParams->stEchoCan.param_4.bits.ec_dc_removal = GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4]),14,1);
			pstVoIPParams->stEchoCan.param_4.bits.ec_h_reset = GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4]),13,1);
			pstVoIPParams->stEchoCan.param_4.bits.ecinit = GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4]),12,1);
			pstVoIPParams->stEchoCan.param_4.bits.hecfrz = GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4]),11,1);
			pstVoIPParams->stEchoCan.param_4.bits.nlp = GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4]),10,1);
			pstVoIPParams->stEchoCan.param_4.bits.nlptune_hi = GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4]),7,1);
			pstVoIPParams->stEchoCan.param_4.bits.cng = GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4]),6,1);
			pstVoIPParams->stEchoCan.param_4.bits.nlptune_lo = GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4]),4,1);
			pstVoIPParams->stEchoCan.param_4.bits.heclen = GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4]),0,4);

			break;

		case FC_VOIP_TONECTRL:
			pstVoIPParams->stToneCtrl.param_4.word = UT_LE2CPU16(pausFifo[4]);

			if (ucNumberOfParams > 1)
				pstVoIPParams->stToneCtrl.param_5.word = UT_LE2CPU16(pausFifo[5]);

			if (ucNumberOfParams > 2)
				pstVoIPParams->stToneCtrl.param_6.word = UT_LE2CPU16(pausFifo[6]);

			if (ucNumberOfParams > 3)
				pstVoIPParams->stToneCtrl.param_7.word = UT_LE2CPU16(pausFifo[7]);

			if (ucNumberOfParams > 4)
				pstVoIPParams->stToneCtrl.param_8.word = UT_LE2CPU16(pausFifo[8]);
			break;

		case FC_VOIP_TONES_RELAY_OPTION:

			pstVoIPParams->stToneRelay.param_4.word = UT_LE2CPU16(pausFifo[4]);

			if (ucNumberOfParams > 1)
				pstVoIPParams->stToneRelay.param_4.word = UT_LE2CPU16(pausFifo[5]);
			break;

		case FC_VOIP_JBOPT:
			/* Except the 1st one, all parameters are optional */
			pstVoIPParams->stJbopt.delay_min =  UT_LE2CPU16(pausFifo[4]);

			if (ucNumberOfParams > 1)
				pstVoIPParams->stJbopt.delay_max =  UT_LE2CPU16(pausFifo[5]);

			if (ucNumberOfParams > 2)
				pstVoIPParams->stJbopt.delay_init = UT_LE2CPU16(pausFifo[6]);

			if (ucNumberOfParams > 3)
				pstVoIPParams->stJbopt.ap = UT_LE2CPU16(pausFifo[7]);

			if (ucNumberOfParams > 4)
				pstVoIPParams->stJbopt.param_8.word = UT_LE2CPU16(pausFifo[8]);

			if (ucNumberOfParams > 5)
				pstVoIPParams->stJbopt.deletion_threshold = UT_LE2CPU16(pausFifo[9]);

			break;

		case FC_SET_ALT_HDR_CHANNEL:
			pstChnl->bAltHdrSet = True;
			pstChnl->bIPHdrSet = False;
			break;

		case FC_SET_IP_HDR_CHANNEL:
			/* if the IP header is longer than a IPv4 header, it is probably an IPv6 header_length
			So do not update the internal VAPI variables relative to the header in this case */
			if (ucNumberOfParams > (sizeof(struct _SET_IP_HDR_CHANNEL) / 2))
			{
				/* how ever we consider that a valid IP header is set to the channel*/
				pstChnl->bIPHdrSet = True;
				pstChnl->bAltHdrSet = False;
			}
			/* if the IP Header is set with only one parameters, this means we want to reset it
			so we consider the header is not set */
			else if (ucNumberOfParams == 1)
			{
				pstVoIPParams->stIpHdrParams.param_4.word = 0x0000;
				pstChnl->bIPHdrSet = False;
			}
			else
			{
				/* parameter [4] bit(s) <7:0> */
				pstVoIPParams->stIpHdrParams.param_4.word = UT_LE2CPU16(pausFifo[4]);
				pstVoIPParams->stIpHdrParams.param_5.word = UT_LE2CPU16(pausFifo[5]);
				pstVoIPParams->stIpHdrParams.ip_len = UT_LE2CPU16(pausFifo[6]);
				pstVoIPParams->stIpHdrParams.ip_id = UT_LE2CPU16(pausFifo[7]);
				pstVoIPParams->stIpHdrParams.ip_off = UT_LE2CPU16(pausFifo[8]);
				pstVoIPParams->stIpHdrParams.param_9.word = UT_LE2CPU16(pausFifo[9]);
				pstVoIPParams->stIpHdrParams.ip_sum = UT_LE2CPU16(pausFifo[10]);
				pstVoIPParams->stIpHdrParams.uh_ulen = UT_LE2CPU16(pausFifo[17]);
				pstVoIPParams->stIpHdrParams.uh_sum = UT_LE2CPU16(pausFifo[18]);
				pstChnl->bIPHdrSet = True;
				pstChnl->bAltHdrSet = False;
			}

			break;

		case FC_FAXOPT:
			/* parameter [4] bit(s) <15:14> */
			pstVoIPParams->stFaxOpts.param_4.word = UT_LE2CPU16(pausFifo[4]);

			if (ucNumberOfParams > 1)
			{
				pstVoIPParams->stFaxOpts.param_5.word = UT_LE2CPU16(pausFifo[5]);
			}

			break;

		case FC_FAXLVL:
			pstVoIPParams->stFaxTxLevel.faxlvl = UT_LE2CPU16(pausFifo[4]);
			break;

		case FC_VOIP_PTMNG:
			/*update the pt array for this channel*/
			j = 0;
			for (i = 0; i < ucNumberOfParams; i++)
			{
				pstVoIPParams->ucAudioPt[j] = GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4 + i]), 0, 8);
				j++;
				pstVoIPParams->ucAudioPt[j] = GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4 + i]), 8, 8);
				j++;
			}
			break;

		case FC_VIDEO_PTMNG:
			/*update the pt array for this channel*/
			j = 0;
			for (i = 0; i < ucNumberOfParams; i++)
			{
				pstVoIPParams->ucVideoPt[j] = GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4 + i]), 0, 8);
				j++;
				pstVoIPParams->ucVideoPt[j] = GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4 + i]), 8, 8);
				j++;
			}
			break;

		case FC_VOIP_PTSET:
			/*update the pt for this channel*/
			pstVoIPParams->ucAudioPt[GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4]), 8, 8)] = GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4]), 0, 8);
			break;

		case FC_VIDEO_PTSET:
			/*update the pt for this channel*/
			pstVoIPParams->ucVideoPt[GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4]), 8, 8)] = GET_BIT_FIELD_VAL(UT_LE2CPU16(pausFifo[4]), 0, 8);
			break;

		default:
			/* the other commands are not tracked in passthru*/
			break;
		}

	return;
}


/****************************************************************************
 * VCORE_CheckPassThruParams : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Check if the VAPI_PassThru command is a configuration change for device or channel level
 *      -#  If so, check the Parameters accordingly the level
 *
 *  \return 
 *  None
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param gtl_msg  Sent by the application.
 */
void VCORE_CheckPassThruParams(IN SChnl *pstChnl, gtl_msg_t *pstReqMsg)
{
	gtl_msg_t stTempReqMsg;
	gtl_msg_t *pstTempReqMsg = &stTempReqMsg;
	VSTATUS Status;

	/* We want to update variables only if the Command type is configuration change
	on channel or device level */
	if ((((U8*)pstReqMsg->fifo)[CIDX_CMD_TYPE] == CMD_TYPE_CONF_CHANGE) &&
				( (((U8*)pstReqMsg->fifo)[CIDX_CMD_CLS] == CMD_CLASS_CONF_CHANNEL) ||
				  (((U8*)pstReqMsg->fifo)[CIDX_CMD_CLS] == CMD_CLASS_CONF_DEVICE)))
	{

		/* make a copy of the message sent by the application 
		This message can contains multiple commands.
		It will be parsed and modified using the VCORE_GetNextMsg function */
		UT_MemCopy(pstTempReqMsg,pstReqMsg,sizeof(gtl_msg_t));
		/* if it's a voice channel check the VoIP parameters */

		/* if it is the Supervisor channel check some parameters
		which can been also set using VAPI commands such as VAPI_SetDeviceIP */
		if (pstChnl->usMSPChnlId == SUPV_CHANNEL)
		{
			VCORE_ExtractSupervisorParams(pstChnl,pstTempReqMsg->fifo);
		}
		else if (pstChnl->usConnType == eVOIP || pstChnl->usConnType == eFOIP)
		{
			/*Check whether this command is changing any of the voip params*/
			do
			{
				VCORE_ExtractVoIPParams(pstChnl, pstChnl->pstChnlParams, pstTempReqMsg->fifo);
				Status = VCORE_CheckCmd(pstChnl->pstDev, (SMsg*)pstTempReqMsg);
				if(Status != SUCCESS)
					break;
			} while(VCORE_GetNextCmd((SMsg*)pstTempReqMsg) > 0);
		}
	}
}

/****************************************************************************
 * VCORE_ProcessEvent : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# Identify the event from function code
 *      -# Fillup the structure of a particular event if it is identified
 *      -# If not identified then give the whole raw message to the app.
 *
 *  \return 
 *  None
 *
 *  \param DevId Device on which event has been received.
 *  \param pstMsg  Message obtained from MSP.
 */
void VCORE_ProcessEvent(SDevice *pstDevice, gtl_msg_t * pstMsg)
{
	void *pvData = NULL;
	SChnl *pstChnl;
	SUndefinedEventParams *pstUndefined;
	EEventCode eEventCode;
	PFNEventCallback pfnEventCallback;
	SVoIPChnlParams *pstVoIPChannelParams = NULL;
	U16 usLocalChnl = 0;
	SApiHdr *pstFifo;
	U8 CC, CT, ucFifoLen;
	U16 FC;
	U8 *ptrH245MsgTemp;
	SH245IndParams *ptrH245IndFullParams = NULL;
	SH245IndParams *ptrH245IndSegment;

	pstFifo = (SApiHdr *)pstMsg->fifo;
	CC = pstFifo->cmd_class;
	CT = pstFifo->cmd_type;
	FC = UT_LE2CPU16(pstFifo->func_code);
	ucFifoLen = pstFifo->length;

	if (FC_EVT_ALERT_IND == FC)
	{
		if ((CC == CMD_CLASS_LEGACY_MSG) && (CT == CMD_TYPE_SUPV_READY))
		{

			pstChnl = DMGR_GetChannel(pstDevice, SUPV_CHANNEL);
			VCORE_ProcessFSMhandler(pstChnl, FSM_EVENT_DEV_MSG, pstMsg);
			return;
		}

		usLocalChnl = UT_LE2CPU16(((U16 *) pstMsg->fifo)[SIDX_FUNC_CODE + 4]);
		/* if not channel specific,then SUPV chnl. */
		if((usLocalChnl == ALERT_UNKNOWN_CHANNEL) || (usLocalChnl == SUPV_CHANNEL))
			pstChnl = DMGR_GetChannel(pstDevice, SUPV_CHANNEL);

		else
			pstChnl = DMGR_GetChannel(pstDevice, usLocalChnl);
	}
	else
	{
		pstChnl = DMGR_GetChannel(pstDevice, pstMsg->channel);
	}

	if (pstChnl == NULL)
	{
		return;
	}

	if (pstChnl->usConnType == eVOIP && pstChnl->bEnableFaxAutoSwitch)
	{
		pstVoIPChannelParams = (SVoIPChnlParams *) pstChnl->pstChnlParams;
		switch (FC)
		{
		case FC_VOIP_REMDET:
			if (pstChnl->usSwitchoverEventMask & eREM_CNG)
			{
				U16 usRemEvent = UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]);

				if (REMDET_CNG == usRemEvent)
				{
					UT_Log(VCORE, INFO, " VCORE_ProcessEvent: GOT eREM_CNG event\n");
					VCORE_SendProcessFaxEvent(pstChnl, eREM_CNG);
					return;
				}
				break;
			}
			else if (pstChnl->usSwitchoverEventMask & eREM_ANS)
			{
				U16 usRemEvent = UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]);

				if (REMDET_ANS == usRemEvent)
				{
					UT_Log(VCORE, INFO, " VCORE_ProcessEvent: GOT eREM_ANS event\n");
					VCORE_SendProcessFaxEvent(pstChnl, eREM_ANS);
					return;
				}
				break;
			}
			else if (pstChnl->usSwitchoverEventMask & eREM_UDPTL)
			{
				U16 usRemEvent = UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]);

				if (REMDET_UDPTL == usRemEvent)
				{
					UT_Log(VCORE, INFO, " VCORE_ProcessEvent:GOT eREM_UDPTL event\n");
					VCORE_SendProcessFaxEvent(pstChnl, eREM_ANS);
					return;
				}
				break;
			}
			break;
		case FC_VOIP_TONEDET:
			{
				U16 usT38EventSwMask = pstChnl->usSwitchoverEventMask;

				if (usT38EventSwMask)
				{
					U16 usTone = UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]);

					if (0x20 == usTone && (usT38EventSwMask & eLOCAL_CNG))
					{
						UT_Log(VCORE, INFO,
						       " VCORE_ProcessEvent: Got eLOCAL_CNG Flags event\n");
						VCORE_SendProcessFaxEvent(pstChnl, eLOCAL_CNG);
						return;
					}
					else if (0x22 == usTone && (usT38EventSwMask & eLOCAL_V21))
					{
						UT_Log(VCORE, INFO,
						       " VCORE_ProcessEvent: Got eLOCAL_V21 Flags event\n");
						VCORE_SendProcessFaxEvent(pstChnl, eLOCAL_V21);
						return;
					}
					else if (0x10 == usTone && (usT38EventSwMask & eLOCAL_ANS))
					{
						UT_Log(VCORE, INFO, " VCORE_ProcessEvent: Got eLOCAL_ANS EVENT\n");
						VCORE_SendProcessFaxEvent(pstChnl, eLOCAL_ANS);
						return;
					}
					break;
				}
			}

		default:
			break;

		}

	}

	/*Allocate Maximum size of Event Params strucuture, 
	   currently SUndefinedEventParams is largest */
	pvData = UT_AllocMem(sizeof(SUndefinedEventParams));
	if (pvData == NULL)
	{
		UT_ErrorLog(VCORE, "VCORE_ProcessEvent: Allocation of pvData Failed\n");
		return;
	}
	switch (FC)
	{
	case FC_EVT_IND_FROM_SPI_DEVICE:
		{
			SSpiEventParams *pstSpiParams = pvData;

			eEventCode = eVAPI_SPI_EVENT;
			pstSpiParams->DevId = pstDevice->DevId;
			pstSpiParams->uTimestamp = (U32) (UT_LE2CPU16(((U16 *) pstMsg->fifo)[4])) |
			    (U32) (UT_LE2CPU16(((U16 *) pstMsg->fifo)[5]) << 16);
		}
		break;
	case FC_VOIP_TONEDET:
		{
			SToneDetectEventParams *pstToneDet = pvData;

			eEventCode = eVAPI_TONE_DETECT_EVENT;

			pstToneDet->ConId = pstChnl->ConnId;
			pstToneDet->usDetectedTone = UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]);
			pstToneDet->uTimestamp = (U32) (UT_LE2CPU16(((U16 *) pstMsg->fifo)[5])) |
			    (U32) (UT_LE2CPU16(((U16 *) pstMsg->fifo)[6]) << 16);
			pstToneDet->usPpowerLevel = UT_LE2CPU16(((U16 *) pstMsg->fifo)[7]);
		}
		break;
	case FC_VOIP_IPTONEDET:
		{
			SIPToneDetectEventParams *pstIPToneDet = pvData;

			eEventCode = eVAPI_IPTONE_DETECT_EVENT;

			pstIPToneDet->ConId = pstChnl->ConnId;
			pstIPToneDet->usDetectedTone = UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]);
			pstIPToneDet->uTimestamp = (U32) (UT_LE2CPU16(((U16 *) pstMsg->fifo)[5])) |
			    (U32) (UT_LE2CPU16(((U16 *) pstMsg->fifo)[6]) << 16);
			pstIPToneDet->usPpowerLevel = UT_LE2CPU16(((U16 *) pstMsg->fifo)[7]);
		}
		break;
	case FC_VOIP_TCMPLT:
		{
			SToneGenCmpltEventParams *pstTcmplt = pvData;

			eEventCode = eVAPI_TONE_GEN_CMPLT_EVENT;
			pstTcmplt->ConId = pstChnl->ConnId;
			pstTcmplt->usToneTerminateCause = UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]);
		}
		break;
	case FC_VOIP_PTCHNG:
		{
			SPtChangeEventParams *pstPtchng = pvData;

			eEventCode = eVAPI_PT_CHANGE_EVENT;
			pstPtchng->ConId = pstChnl->ConnId;
			pstPtchng->uNewPT = UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]);
		}
		break;
	case FC_VOIP_SSRCCHG:
		{
			SSsrcChangeEventParams *pstSscrChng = pvData;

			eEventCode = eVAPI_SSRC_CHANGE_EVENT;
			pstSscrChng->ConId = pstChnl->ConnId;
			pstSscrChng->uNewSSRC = (U32) UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]) |
			    (U32) (UT_LE2CPU16(((U16 *) pstMsg->fifo)[5]) << 16);
		}
		break;
	case FC_VOIP_SSRCVIOL:
		{
			SSsrcViolationEventParams *pstSsrcViol = pvData;

			eEventCode = eVAPI_SSRC_VIOLATION_EVENT;
			pstSsrcViol->ConId = pstChnl->ConnId;
			pstSsrcViol->uDetectedSSRC = (U32) UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]) |
			    (U32) (UT_LE2CPU16(((U16 *) pstMsg->fifo)[5]) << 16);
		}
		break;
	case FC_VOIP_NTECMPLT:
		{
			SNteTransmitEventParams *pstNteTcmpl = pvData;

			eEventCode = eVAPI_NTE_TRANSMIT_COMPLETE_EVENT;
			pstNteTcmpl->ConId = pstChnl->ConnId;
		}
		break;
	case FC_VOIP_NTERCVD:
		{
			SNteRecvdEventParams *pstNteRcvd = pvData;

			eEventCode = eVAPI_NTE_RECVD_EVENT;
			pstNteRcvd->ConId = pstChnl->ConnId;
			pstNteRcvd->uNtePayload = (U32) (((U16 *) pstMsg->fifo)[4] << 16) |
			    (U32) (((U16 *) pstMsg->fifo)[5]);
			pstNteRcvd->usPayloadTypeOverride = UT_LE2CPU16(((U16 *) pstMsg->fifo)[6]);
			pstNteRcvd->usPayloadType = UT_LE2CPU16(((U16 *) pstMsg->fifo)[7]);
			pstNteRcvd->uSSRC = (U32) (((U16 *) pstMsg->fifo)[8] << 16) | (U32) (((U16 *) pstMsg->fifo)[9]);
		}
		break;
	case FC_CND_DONE:
		{
			SCallerIdCmpltEventParams *stCnddone = pvData;

			eEventCode = eVAPI_CALLER_ID_CMPLT_EVENT;
			stCnddone->ConId = pstChnl->ConnId;
			stCnddone->usCmpltCause = UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]);
		}
		break;

	case FC_VOIP_EVENTDET:
		{
			SEventDetParams *pstEventDet = pvData;

			pstEventDet->ConnId = pstChnl->ConnId;
			pstEventDet->usEventReport = UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]);
			pstEventDet->uTimestamp = (U32) (UT_LE2CPU16(((U16 *) pstMsg->fifo)[5])) |
			    (U32) (UT_LE2CPU16(((U16 *) pstMsg->fifo)[6]) << 16);

			switch (UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]))
			{
			case VOIP_EVENTDET_EVENT_START_G_711:
			case VOIP_EVENTDET_EVENT_START_CONCURRENT_G_711_MULAW:
			case VOIP_EVENTDET_EVENT_START_CONCURRENT_G_711_ALAW:
				eEventCode = eVAPI_G711_CONCURRENT_DECODER_EVENT;
				break;

			case VOIP_EVENTDET_EVENT_START_PASS_THROUGH:
			case VOIP_EVENTDET_EVENT_START_CONCURRENT_PASS_THROUGH_MULAW:
			case VOIP_EVENTDET_EVENT_START_CONCURRENT_PASS_THROUGH_ALAW:
				eEventCode = eVAPI_PASSTHRU_CONCURRENT_DECODER_EVENT;
				break;

			default:
				eEventCode = eVAPI_DECODER_EVENT;
				break;
			}
		}
		break;

	case FC_CNDRXMSG:
		{
			SCIDReceivedMSG *pstCIDReceivedMSG = pvData;

			eEventCode = eVAPI_CALLER_ID_DETECTED_EVENT;

			UT_Log(VCORE, INFO, "Caller Id Detection Event (CNDRXMSG)\n");

			pstCIDReceivedMSG->ConId = pstChnl->ConnId;
			pstCIDReceivedMSG->usMSGStatus = UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]);
			pstCIDReceivedMSG->ucMSGByteCnt = (U8) (((U8 *) pstMsg->fifo)[11]);
			((U8 *) pstMsg->fifo)[11] = ((U8 *) pstMsg->fifo)[10];
			pstCIDReceivedMSG->pucMsg = &(((U8 *) pstMsg->fifo)[11]);

		}
		break;

	case FC_VOIP_REMDET:
		{
			SRemoteDetectEventParams *pstRemoteDetParams = pvData;
			U16 usRemdetParam1 = 0, usRemdetParam2 = 0;

			eEventCode = eVAPI_REMOTE_DETECT_EVENT;

			UT_Log(VCORE, INFO, "Remote Detection Event (REMDET)\n");

			pstRemoteDetParams->ConId = pstChnl->ConnId;
			usRemdetParam1 = UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]);
			/*We are assuming it is padded with zeros even if fifo contains 
			   only param1 */
			usRemdetParam2 = UT_LE2CPU16(((U16 *) pstMsg->fifo)[5]);

			pstRemoteDetParams->ucDetectedEvent = (usRemdetParam1 & 0xFF00) >> 8;
			pstRemoteDetParams->ucRemotePktType = usRemdetParam1 & 0x00FF;
			pstRemoteDetParams->usRemotePktPayload = usRemdetParam2;

		}
		break;

	case FC_EVT_ALERT_IND:
		{

			SAlrtIndParams *pstAlrtIndParams = pvData;

			eEventCode = eVAPI_ALERT_IND;
			pstAlrtIndParams->DevId = pstDevice->DevId;
			pstAlrtIndParams->ConId = usLocalChnl;

			pstAlrtIndParams->usAlrtType = UT_LE2CPU16(((U16 *) pstMsg->fifo)[SIDX_FUNC_CODE + 2]);
			pstAlrtIndParams->ucCPUNo = (U8) ((UT_LE2CPU16(((U16 *) pstMsg->fifo)[SIDX_FUNC_CODE + 3]))
							  & 0x00FF);
			pstAlrtIndParams->usUniqueId = UT_LE2CPU16(((U16 *) pstMsg->fifo)[SIDX_FUNC_CODE + 5]);
			pstAlrtIndParams->uiLnkRegVal = ((UT_LE2CPU16(((U16 *) pstMsg->fifo)[SIDX_FUNC_CODE + 11])) << 16) |
					(UT_LE2CPU16(((U16 *) pstMsg->fifo)[SIDX_FUNC_CODE + 10]));

			pstAlrtIndParams->ucAlertRawDataSize = ucFifoLen - FIFO_MIN_SIZE;	/* fifo len - fifo header */
			UT_MemCopy(pstAlrtIndParams->aucAlertRawData, &(((U16*)pstMsg->fifo)[(U8)(FIFO_MIN_SIZE/2)]), pstAlrtIndParams->ucAlertRawDataSize);
		}
		break;
		
	case FC_SYNCDAT:
		if(pstChnl->stRecPlayInfo.eRecPlayState == eVop)
		{
			SSYNCDATParams *pstSYNCDATParams = pvData;
			eEventCode = eVAPI_SYNCDAT_EVENT;

			pstSYNCDATParams->ConId = pstChnl->ConnId;

			pstSYNCDATParams->uiPayloadLen = pstFifo->length - FIFO_MIN_SIZE;	/* fifo len - fifo header */

			if (pstSYNCDATParams->uiPayloadLen > MAX_SYNCDAT_BUF_SIZE)
				pstSYNCDATParams->uiPayloadLen = MAX_SYNCDAT_BUF_SIZE;
			
			UT_MemCopy(pstSYNCDATParams->aucPayload, &(((U16*)pstMsg->fifo)[(U8)(FIFO_MIN_SIZE/2)]), pstSYNCDATParams->uiPayloadLen);
			
			break;
		}
		else
		{
			VCORE_ProcessFSMhandler(pstChnl, FSM_EVENT_DEV_MSG, pstMsg);
			UT_FreeMem(pvData);
			return;
		}
		
	case FC_SYNCEOF:
		if(pstChnl->stRecPlayInfo.eRecPlayState == eVop)
		{
			SSYNCEOFParams *pstSYNCEOFParams = pvData;
			eEventCode = eVAPI_SYNCEOF_EVENT;

			pstSYNCEOFParams->ConId = pstChnl->ConnId;

			if (pstFifo->length == 0)
			{
				/* extended data transfer */
				pstSYNCEOFParams->uiPayloadLen = ((U16*)pstMsg->fifo)[(U8)(FIFO_MIN_SIZE/2 - 1)];
			}
			else
			{
				pstSYNCEOFParams->uiPayloadLen = pstFifo->length - FIFO_MIN_SIZE;	/* fifo len - fifo header */
			}

			if (pstSYNCEOFParams->uiPayloadLen > MAX_SYNCEOF_BUF_SIZE)
				pstSYNCEOFParams->uiPayloadLen = MAX_SYNCEOF_BUF_SIZE;

			UT_MemCopy(pstSYNCEOFParams->aucPayload, &(((U16*)pstMsg->fifo)[(U8)(FIFO_MIN_SIZE/2)]), pstSYNCEOFParams->uiPayloadLen);
			
			break;
		}
		else
		{
			VCORE_ProcessFSMhandler(pstChnl, FSM_EVENT_DEV_MSG, pstMsg);
			UT_FreeMem(pvData);
			return;
		}
		
	case FC_VOIP_DRAIN_PLAYREC_BUFFER_IND:
		if (pstChnl->stRecPlayInfo.eRecPlayState != eVop)
		{
			VCORE_ProcessFSMhandler(pstChnl, FSM_EVENT_DEV_MSG, pstMsg);
		}
		UT_FreeMem(pvData);
		return;

	case FC_IMSOTDM_TXH245:
		UT_FreeMem(pvData);

		ptrH245IndSegment = ((SH245IndParams *) (((U16 *) pstMsg->fifo) + 4));

		eEventCode = eVAPI_IMSOTDM_H245_EVENT;

		UT_Log(VCORE, INFO, "IMS over TDM H.245 Event\n");

		/* we got the first segment of H.245 indication*/
		if(pstChnl->pstH245MsgIndParam == NULL)
		{
			pstChnl->pstH245MsgIndParam = UT_Calloc(1, sizeof(SH245IndParams));
			((SH245IndParams *) pstChnl->pstH245MsgIndParam)->ptrH245Msg = NULL;
			((SH245IndParams *) pstChnl->pstH245MsgIndParam)->uiH245MsgLen = 0;
		}

		ptrH245MsgTemp = UT_Calloc(ptrH245IndSegment->uiH245MsgLen + 
				((SH245IndParams *) pstChnl->pstH245MsgIndParam)->uiH245MsgLen, 
				sizeof(U8));

		/* copy previous segment if it's existed*/
		if (((SH245IndParams *) pstChnl->pstH245MsgIndParam)->ptrH245Msg != NULL)
		{
			UT_MemCopy(ptrH245MsgTemp,
					((SH245IndParams *) pstChnl->pstH245MsgIndParam)->ptrH245Msg,
					((SH245IndParams *) pstChnl->pstH245MsgIndParam)->uiH245MsgLen);

			UT_FreeMem(((SH245IndParams *) pstChnl->pstH245MsgIndParam)->ptrH245Msg);
		}

		UT_MemCopy(ptrH245MsgTemp + 
				((SH245IndParams *) pstChnl->pstH245MsgIndParam)->uiH245MsgLen, 
				ptrH245IndSegment->ptrH245Msg,
				ptrH245IndSegment->uiH245MsgLen);

		((SH245IndParams *) pstChnl->pstH245MsgIndParam)->uiH245MsgLen += ptrH245IndSegment->uiH245MsgLen;			

		/* check weather the next segemts are to be presented*/
		if (ptrH245IndSegment->uiEndOfMsg == False)
		{	
			((SH245IndParams *) pstChnl->pstH245MsgIndParam)->ptrH245Msg = ptrH245MsgTemp;
			
			return;
		}
		else
		{
			ptrH245IndFullParams = UT_Calloc(1, sizeof(SH245IndParams));

			ptrH245IndFullParams->uiEndOfMsg = True;
			ptrH245IndFullParams->uiH245MsgLen = ((SH245IndParams *) pstChnl->pstH245MsgIndParam)->uiH245MsgLen;
			ptrH245IndFullParams->ptrH245Msg = ptrH245MsgTemp;

			/* it's the last segment of message*/
			UT_FreeMem(pstChnl->pstH245MsgIndParam);
			
			pvData = (void *) ptrH245IndFullParams;			
		}

		break;

	case FC_CHANNELS_RESET_COMPLETE:
		VCORE_ProcessFSMhandler(pstChnl, FSM_EVENT_DEV_MSG, pstMsg);
		UT_FreeMem(pvData);

		return;

	default:
		eEventCode = eVAPI_UNDEFINED_EVENT;
	}

	if (eEventCode == eVAPI_UNDEFINED_EVENT)
	{
		S32 i;

		pstUndefined = pvData;
		if (SUPV_CHANNEL == pstChnl->usMSPChnlId)
		{
			pstUndefined->ucLevel = EVENT_LEVEL_DEVICE;
			pstUndefined->ID.DevId = pstDevice->DevId;
		}
		else
		{
			pstUndefined->ucLevel = EVENT_LEVEL_CONN;
			pstUndefined->ID.ConId = pstChnl->ConnId;
		}
		pstUndefined->ucLength = ((U8*)pstMsg->fifo)[CIDX_FIFO_LEN]; 
		pstUndefined->ucIndex = ((U8 *) pstMsg->fifo)[CIDX_CMD_INDEX];
		pstUndefined->ucCmdClass = ((U8 *) pstMsg->fifo)[CIDX_CMD_CLS];
		pstUndefined->ucCmdType = ((U8 *) pstMsg->fifo)[CIDX_CMD_TYPE];
		pstUndefined->usFnCode = UT_LE2CPU16(((U16 *) pstMsg->fifo)[SIDX_FUNC_CODE]);

		/*Make sure to not overwrite the buffer size */
		if (pstMsg->fifo_size > DEFAULT_FIFO_MAX_SIZE)
		{
			pstMsg->fifo_size = DEFAULT_FIFO_MAX_SIZE;
			UT_Log(VCORE, INFO, "VCORE_ProcessEvent: Received event fifo truncated to %d bytes (out of %d)\n", DEFAULT_FIFO_MAX_SIZE, pstMsg->fifo_size);
		}

		/*If the fifo is not at least 8 bytes there is something wrong */
		if (pstMsg->fifo_size < FIFO_MIN_SIZE)
		{
			UT_ErrorLog(APPITF, "VCORE_ProcessEvent: Warning fifo lenght out of range (%d)\n", pstMsg->fifo_size);
			goto out;
		}
		
		/* The number of parameters is the length of the indication - 8 bytes of API header
		divided by 2 to get it in word, plus 1 (the usual 0x0000 word)*/
		pstUndefined->usNoOfParams = ((pstUndefined->ucLength - sizeof(struct comcerto_api_hdr)) / 2) + 1;

		for (i = 0; i < pstUndefined->usNoOfParams; i++)
		{
			pstUndefined->ausParams[i] = UT_LE2CPU16(((U16 *) pstMsg->fifo)[3 + i]);
		}
	}

	pfnEventCallback = pfnEventCallback_g;
	if (pstChnl->pfnEventCallback != NULL)
	{
		pfnEventCallback = pstChnl->pfnEventCallback;
	}
	else if (pstChnl->pstDev->pfnEventCallback != NULL)
	{
		pfnEventCallback = pstChnl->pstDev->pfnEventCallback;
	}

	if (pfnEventCallback != NULL)
	{
		UT_Log(VCORE, INFO, "VCORE_ProcessEvent Calling event CB Code = %u\n", eEventCode);
		pfnEventCallback(eEventCode, pvData);
	}

out:
	UT_FreeMem(pvData);
	UT_Log(VCORE, INFO, "VCORE_ProcessEvent Exit\n");
	return;
}

/****************************************************************************
 * VCORE_GTLReadCallback : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# Identify whether its a response or an event
 *      -# Call respective handlers.
 *
 *  \return 
 *  None
 *
 *  \param DevId Device on which a message has been received.
 *  \param pstMsg  Message obtained from MSP.
 */
void VCORE_GTLReadCallback(IN S32 DevId, IN gtl_msg_t * pstMsg)
{
	SDevice *pstDevice;
	U8 CC, CT;
	SApiHdr *pstFifo;

	UT_Log(VCORE, INFO, "VCORE_GTLReadCallback: Entering devid(%u) pstMsg(0x%x)\n", DevId, pstMsg);
	if (pstMsg == NULL)
	{
		UT_ErrorLog(VCORE, "VCORE_GTLReadCallback: pstMsg is NULL\n");
		goto out;
	}

	/*we may have a NULL fifo for PCI messages*/
	if((pstMsg->type & CSME) && (pstMsg->fifo == NULL))
	{
		UT_ErrorLog(VCORE, "VCORE_GTLReadCallback: pstMsg NULL fifo for CSME msg type\n");
		goto out;
	}

#ifndef VAPI_NODBG
	GTL_PrintMessage(pstMsg, VCORE, 0);
#endif

	/*Statistics */
	UT_MutexLock(&vapi_stat.mutex_stat);
	vapi_stat.api_frames_received++;
	UT_MutexUnLock(&vapi_stat.mutex_stat);

	pstDevice = DMGR_GetDevice(DevId);
	if (pstDevice == NULL)
	{
		UT_ErrorLog(VCORE, "VCORE_GTLReadCallback: Unknown devid(%u)\n", DevId);
		goto out;
	}

	if (pstMsg->type & BOOT)
		VCORE_ProcessBootResponse(pstDevice, pstMsg);
	else
	{
		pstFifo = (SApiHdr *) pstMsg->fifo;
		CC = pstFifo->cmd_class;
		CT = pstFifo->cmd_type;

		switch (CC)
		{
		case CMD_CLASS_STAT_CHANNEL:
			switch (CT)
			{
			case CMD_TYPE_QUERY:
			case CMD_TYPE_INDICATION:
				VCORE_ProcessEvent(pstDevice, pstMsg);
				break;

			default:
				VCORE_ProcessResponse(pstDevice, pstMsg);
				break;
			}

			break;

		case CMD_CLASS_CONF_CHANNEL:
		case CMD_CLASS_CONF_DEVICE:
			switch (CT)
			{
			case CMD_TYPE_INDICATION:
				VCORE_ProcessEvent(pstDevice, pstMsg);
				break;

			default:
				VCORE_ProcessResponse(pstDevice, pstMsg);
				break;
			}

			break;

		case CMD_CLASS_LEGACY_MSG:
			switch (CT)
			{
			case CMD_TYPE_SUPV_READY:
				VCORE_ProcessEvent(pstDevice, pstMsg);
				break;

			default:
				VCORE_ProcessResponse(pstDevice, pstMsg);
				break;
			}

			break;

		case CMD_CLASS_STAT_DEVICE:
			VCORE_ProcessResponse(pstDevice, pstMsg);
			break;

		case CMD_CLASS_REF_DIAG:
		case CMD_CLASS_OPEN_DIAG:
			switch (CT)
			{
			case CMD_TYPE_DIAG_MON_LIVE_IND:
				VCORE_ProcessEvent(pstDevice, pstMsg);
				break;

			default:
				VCORE_ProcessResponse(pstDevice, pstMsg);
				break;
			}
			break;

		case CMD_CLASS_DIAG_TOOL:
			switch (CT)
			{
			case CMD_TYPE_INDICATION:
				VCORE_ProcessEvent(pstDevice, pstMsg);
				break;
			default:
				VCORE_ProcessResponse(pstDevice, pstMsg);
				break;
			}
	
			break;

		default:
			UT_Log(VCORE, INFO, "VCORE_GTLReadCallback: Unknown command class.\n");
			break;
		}
	}

out:
	UT_Log(VCORE, INFO, "VCORE_GTLReadCallback: Exiting\n");
}

/*! 
 *  - Implementation
 *      -#  Uses Supervisory channel structure to send read commands, 
 *          this is safe because, there will not be any supervisory 
 *          commands until it gets created.
 *      -#  When a CMD_ACK is received, VCORE_ProcessResponse will not get any
 *          device corresponding to channel 0, but it should check the state of
 *          the device and if it is eREADYFORDUMP state then it should pass the 
 *          CMD_ACK to the Supervisory channel.
 *      -#  It does the following
 *          -#  Read device memory.
 *          -#  Do request completion processing any of the listed case:
 *                  -# 1) No further applicable read for this memory 
 *                  -# 2) User supplied buffer is filled
 *                  -# 3) Some error (from MSP)
 *  
 *  \return 
 *  None
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VCORE_ReadDeviceMemory(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	U8 *paucFifo = NULL;
	U16 *pausFifo = NULL;
	VSTATUS Status = SUCCESS;
	Boolean bContinue = True;
	SCoreDumpData *pstCoreDumpData = NULL;
	U32 ulReadSize = 0;

	UT_Log(VCORE, INFO, "Entered VCORE_ReadDeviceMemory\n");

	UT_MutexLock(&pstChnl->stReqList.stSem);
	/*Update Request State */
	pstVapiReq->usReqState++;
	UT_MutexUnLock(&pstChnl->stReqList.stSem);

	UT_Log(VCORE, DEBUG, "VCORE_ReadDeviceMemory: Conn %u Req state = %d\n",
	       pstChnl->ConnId, pstVapiReq->usReqState);

	pstCoreDumpData = (SCoreDumpData *) (pstVapiReq->pvUserData);

	if (pstMsg == NULL)	/* direct (first) call */
	{
		pstVapiReq->pvExecData = UT_AllocMem(sizeof(U32));
		if (pstVapiReq->pvExecData == NULL)
		{
			Status = VAPI_ERR_NOMEM;
		}

		*(pstCoreDumpData->pulDataSize) = 0;
		*(pstCoreDumpData->pbLastRead) = False;
	}
	else			/* This is NOT the direct (first) call */
	{
		pausFifo = (U16 *) pstMsg->fifo;
		paucFifo = (U8 *) pstMsg->fifo;

		if ((pstChnl->pstDev->eItfType == ePCI_ITF) || (pstChnl->pstDev->eItfType == ePOS_ITF))
		{
			Status = VCORE_CheckStatus(Event, pstMsg,
					0, CMD_TYPE_BRM_CMD_ACK, CMD_CLASS_ETH_BOOT_MSG, 0);
		}
		else
		{
			Status = VCORE_CheckStatus(Event, pstMsg,
				0, CMD_CLASS_ETH_BOOT_MSG, CMD_TYPE_BRM_CMD_ACK, 0);
		}
		if (Status == SUCCESS)
		{
			ulReadSize = *((U32 *) pstVapiReq->pvExecData);

			/* Copy the number of bytes which was specified in FIFOREAD 
			 * We will skip first 6 bytes since those will be generic 
			 * encapsulation bytes (index, length, CC, CT, FC) 
			 */
			UT_MemCopy((pstCoreDumpData->puchBuff
				    + *(pstCoreDumpData->pulDataSize)), (paucFifo + 6), ulReadSize);

			/* TODO : Check for POS case */

			*(pstCoreDumpData->pulDataSize) += ulReadSize;

			/* Check if this was the last applicable read */
			if (pstCoreDumpData->ulOffset >= pstCoreDumpData->ulMemSize)
			{
				*(pstCoreDumpData->pbLastRead) = True;
				bContinue = False;
			}

			if (pstChnl->pstDev->eItfType == eCSM_ITF)
			{
				if (((pstCoreDumpData->ulBuffSize - *(pstCoreDumpData->pulDataSize))
				     < VAPI_ETH_DUMP_MAX_BLK)
				    && ((pstCoreDumpData->ulMemSize - pstCoreDumpData->ulOffset) >
					(pstCoreDumpData->ulBuffSize - *(pstCoreDumpData->pulDataSize))))
				{
					/* We have read optimum data let user make a new call */
					bContinue = False;
				}
			}
			else
			{
				if (((pstCoreDumpData->ulBuffSize - *(pstCoreDumpData->pulDataSize))
				     < VAPI_PCI_DUMP_MAX_BLK)
				    && ((pstCoreDumpData->ulMemSize - pstCoreDumpData->ulOffset) >
					(pstCoreDumpData->ulBuffSize - *(pstCoreDumpData->pulDataSize))))
				{
					/* We have read optimum data let user make a new call */
					bContinue = False;
				}

			}
		}

	}			/*else */

	/* Send read command */
	if (Status == SUCCESS && bContinue == True)
	{
		U32 ulTmp1, ulTmp2, ulReadAddr;

		/* Calculate optimum valid mem size for read command */
		ulTmp1 = ((pstCoreDumpData->ulMemSize - pstCoreDumpData->ulOffset)
			  > (pstCoreDumpData->ulBuffSize - *(pstCoreDumpData->pulDataSize)))
		    ? (pstCoreDumpData->ulBuffSize - *(pstCoreDumpData->pulDataSize))
		    : (pstCoreDumpData->ulMemSize - pstCoreDumpData->ulOffset);

		if (pstChnl->pstDev->eItfType == eCSM_ITF)
		{
			ulTmp2 = (ulTmp1 > VAPI_ETH_DUMP_MAX_BLK) ? VAPI_ETH_DUMP_MAX_BLK : ulTmp1;
		}
		else
		{
			ulTmp2 = (ulTmp1 > VAPI_PCI_DUMP_MAX_BLK) ? VAPI_PCI_DUMP_MAX_BLK : ulTmp1;
		}

		/* 
		 *   XXX - we wil read even bytes by truncating to word aligned size 
		 * since BRM_FIFOREAD gives improper data for odd bytes read. 
		 * Till the problem is solved at driver level, we will use this
		 * fix for then case when user gives us an odd bytes sized 
		 * buffer.
		 *   The same problem wil occur when the memory size is odd bytes 
		 * (ulTmp2 == 1) and we may have to use BRM_BYTEREAD in that case. 
		 * This is not taken care of currently.
		 */
		ulTmp2 = ((ulTmp2 != 1) && (ulTmp2 & 1)) ? (ulTmp2 - 1) : ulTmp2;

		*((U32 *) pstVapiReq->pvExecData) = ulTmp2;

		ulReadAddr = pstCoreDumpData->ulMemBase + pstCoreDumpData->ulOffset;
		if (pstChnl->pstDev->eItfType == ePOS_ITF)
			Status = VDEV_POSMemRead(pstChnl, ulReadAddr, *((U32 *) pstVapiReq->pvExecData));
		else
			Status = VDEV_BRMFifoRead(pstChnl, ulReadAddr, *((U32 *) pstVapiReq->pvExecData));

		pstCoreDumpData->ulOffset += ulTmp2;
	}

	if (Status != SUCCESS || bContinue == False)
	{
		UT_Log(VCORE, DEBUG,"VCORE_ReadDeviceMemory: Completing state(%u) status(%d) dev(%u)\n", 
				pstVapiReq->usReqState, Status, pstChnl->pstDev->DevId);


		VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);
	}

	return Status;
}


void VCORE_ProcessAifHeader(struct _AIF_HEADER *pstAifHd)
{
	UT_Log(VCORE, INFO, "Swaping AXF Header\n");

	pstAifHd->BL_DecompressCode = UT_LE2CPU32(pstAifHd->BL_DecompressCode);
	pstAifHd->BL_SelfRelocCode = UT_LE2CPU32(pstAifHd->BL_SelfRelocCode);
	pstAifHd->BL_DbgInitZeroInit = UT_LE2CPU32(pstAifHd->BL_DbgInitZeroInit);
	pstAifHd->EntryPointOffset = UT_LE2CPU32(pstAifHd->EntryPointOffset);
	pstAifHd->ProgramExitInstr = UT_LE2CPU32(pstAifHd->ProgramExitInstr);
	pstAifHd->ImageReadOnlySize = UT_LE2CPU32(pstAifHd->ImageReadOnlySize);
	pstAifHd->ImageReadWriteSize = UT_LE2CPU32(pstAifHd->ImageReadWriteSize);
	pstAifHd->ImageDebugSize = UT_LE2CPU32(pstAifHd->ImageDebugSize);
	pstAifHd->ImageZeroInitSize = UT_LE2CPU32(pstAifHd->ImageZeroInitSize);
	pstAifHd->ImageDebugType = UT_LE2CPU32(pstAifHd->ImageDebugType);
	pstAifHd->ImageBase = UT_LE2CPU32(pstAifHd->ImageBase);
	pstAifHd->WorkSpace = UT_LE2CPU32(pstAifHd->WorkSpace);
	pstAifHd->AddressMode = UT_LE2CPU32(pstAifHd->AddressMode);
	pstAifHd->DataBase = UT_LE2CPU32(pstAifHd->DataBase);
	pstAifHd->FirstFatOffset = UT_LE2CPU32(pstAifHd->FirstFatOffset);
	pstAifHd->Reserved2 = UT_LE2CPU32(pstAifHd->Reserved2);
	pstAifHd->DebugInitInstr = UT_LE2CPU32(pstAifHd->DebugInitInstr);
}

VSTATUS VCORE_HandleFrame(IN gtl_msg_t * pstMsg, IN OUT SFrameInfo * pstFrameInfo)
{
	UT_Log(VCORE, INFO, "VCORE_HandleFrame: Entry\n");
	UT_Log(VCORE, INFO, "fifo size = %d Frame len = %d frame bufflen = %d pstMsg->fifo[0] = %d\n",
		pstMsg->fifo_size, pstFrameInfo->uiFrameLen, sizeof(U8) * (UT_LE2CPU16(*((U16 *) pstMsg->fifo + 9))),
		((U8 *) pstMsg->fifo)[0]);

	if (((U8 *) pstMsg->fifo)[0] <= MIN_FRAME_SIZE)
	{
		UT_ErrorLog(VCORE, "Invalid SYNCDAT/SYNCEOF packet received\n");
		return MALFORMED_PACKET;
	}	

	if (True == pstFrameInfo->ucIsHdrPresent)
	{
		if (FC_SYNCEOF == UT_LE2CPU16(*((U16 *) (pstMsg->fifo) + 2)))
		{
			/* single SYNCEOF packet */
			pstFrameInfo->pucFrameBuffer = (U8 *) pstMsg->fifo + COMMAND_HEADER_LEN + PACKET_HEADER_LEN;
		}
		else
		{
			UT_MemCopy((void *)(pstFrameInfo->pucFrameBuffer + pstFrameInfo->uiFrameLen),
			   	(void *)((U8 *) pstMsg->fifo + COMMAND_HEADER_LEN + PACKET_HEADER_LEN), 
			   	((U8 *) pstMsg->fifo)[0] - COMMAND_HEADER_LEN - PACKET_HEADER_LEN);
		}
		
		pstFrameInfo->uiFrameLen += ((U8 *) pstMsg->fifo)[0] - (COMMAND_HEADER_LEN + PACKET_HEADER_LEN);
	}
	else
	{
		UT_MemCopy((void *)(pstFrameInfo->pucFrameBuffer + pstFrameInfo->uiFrameLen),
			   (void *)((U8 *) pstMsg->fifo + COMMAND_HEADER_LEN), ((U8 *) pstMsg->fifo)[0] - COMMAND_HEADER_LEN);

		pstFrameInfo->uiFrameLen += ((U8 *) pstMsg->fifo)[0] - COMMAND_HEADER_LEN;
	}
	
	if (FC_SYNCDAT == UT_LE2CPU16(*((U16 *) (pstMsg->fifo) + 2)))
	{
		pstFrameInfo->ucIsHdrPresent = False;
		UT_Log(VCORE, INFO, "VCORE_HandleFrame: Exit\n");
		return SYNCDAT_PACKET;
	}
	else if (FC_SYNCEOF == UT_LE2CPU16(*((U16 *) (pstMsg->fifo) + 2)))
	{
		pstFrameInfo->ucIsHdrPresent = True;
		UT_Log(VCORE, INFO, "VCORE_HandleFrame: Exit\n");
		return SYNCEOF_PACKET;		
	}

	return FAILURE;

}

void VCORE_NotifyT38Switchover(U32 uiConId, U8 ucCmdLevel, S32 Result, void *pvRetData, U32 uiDataLen, U32 uiTag)
{
	void *pvData = NULL;
	SChnl *pstChnl;
	SFaxSwEventParams stFaxSwParams;
	EEventCode eEventCode;
	PFNEventCallback pfnEventCallback;

	UT_Log(VCORE, INFO, "VCORE_NotifyT38Switchover Entry Conn = %u\n", uiConId);
	pstChnl = DMGR_GetConnection(uiConId);
	if (pstChnl == NULL)
	{
		UT_ErrorLog(VCORE, "Errors while notifying swithover to user\n");
		return;
	}
	eEventCode = eVAPI_FAX_SWITCH_CMPLT_EVENT;
	if (Result != SUCCESS)
	{
		stFaxSwParams.status = Result;
	}
	else
	{
		stFaxSwParams.status = SUCCESS;
	}
	stFaxSwParams.ConId = uiConId;
	stFaxSwParams.eSwitchEvent = uiTag;
	pvData = &stFaxSwParams;

	pfnEventCallback = pfnEventCallback_g;
	if (pstChnl->pfnEventCallback != NULL)
	{
		pfnEventCallback = pstChnl->pfnEventCallback;
	}
	else if (pstChnl->pstDev->pfnEventCallback != NULL)
	{
		pfnEventCallback = pstChnl->pstDev->pfnEventCallback;
	}

	if (pfnEventCallback != NULL)
	{
		UT_Log(VCORE, INFO, "VCORE_NotifyT38Switchover Calling event CB Code = %u\n", eEventCode);
		pfnEventCallback(eEventCode, pvData);
	}

	UT_Log(VCORE, INFO, "VCORE_NotifyT38Switchover Exit\n");
	return;
}

void VCORE_SendProcessFaxEvent(SChnl * pstChnl, U16 usSwEvent)
{
	SVapiReq *pstVapiReq = NULL;

	pstVapiReq = VCORE_AllocateRequest(sizeof(SFsmFaxOptions));

	if (pstVapiReq == NULL)
	{
		UT_ErrorLog(VCORE, "No memory available\n");
		return;
	}

	UT_MemSet(pstVapiReq, 0, sizeof(SVapiReq));
	
	/* Send VAPI request to VCORE */
	pstVapiReq->usReqState = MODIFY_T38_INIT;
	pstVapiReq->pfnCallback = VCORE_NotifyT38Switchover;
	pstVapiReq->uiReqId = usSwEvent;

	pstVapiReq->pfnReqFSMHdlr = VFSM_SwitchToT38;
	((SFsmFaxOptions *) pstVapiReq->pvUserData)->bUseIt = False;;
	pstVapiReq->uiID = pstChnl->ConnId;
	pstVapiReq->ucCmdLevel = CMD_LEVEL_CONN;
	
	VCORE_ProcessRequest(pstChnl, pstVapiReq);

}

VSTATUS VCORE_CheckStatusMulti(SChnl * pstChnl, gtl_msg_t * pstReqMsg, U16 Event, gtl_msg_t * pstMsg)
{
	gtl_msg_t stTempReqMsg, stTempMsg;
	gtl_msg_t *pstTempReqMsg = &stTempReqMsg;
	gtl_msg_t *pstTempMsg = &stTempMsg;
	U8 ucCmdCls;
	U8 ucCmdType;
	VSTATUS Status = SUCCESS;
	VSTATUS MspResponse = SUCCESS;
	SApiHdr *pstFifo;

	/* parse only 1 command for supervisor */
	if (pstChnl->usMSPChnlId == SUPV_CHANNEL)
	{
		UT_Log(VCORE, INFO, "VCORE_CheckStatusMulti: must not be called for SUPV_CHANNEL\n");
		Status = VAPI_ERR_UNSUPP_FEATURE;
		goto out;
	}

	UT_Log(VCORE, INFO, "VCORE_CheckStatusMulti: Event %u\n", Event);
	switch (Event)
	{
	case FSM_EVENT_TIMEOUT:
		Status = VAPI_ERR_MSP_NO_RESPONSE;
		break;

	case FSM_EVENT_REC_PLAY_TERMINATOR:
		Status = VAPI_IND_STOP_REC_PLAY;
		break;

	case FSM_EVENT_DEV_MSG:

		UT_MemCopy(pstTempReqMsg, pstReqMsg, sizeof(gtl_msg_t));
		UT_MemCopy(pstTempMsg, pstMsg, sizeof(gtl_msg_t));

		/* parse all the commands in the fifo */
		do
		{
			pstFifo = (SApiHdr *)pstTempReqMsg->fifo;
			ucCmdCls = pstFifo->cmd_class;
			ucCmdType = pstFifo->cmd_type;
			if (ucCmdCls == CMD_CLASS_CONF_CHANNEL || ucCmdCls == CMD_CLASS_CONF_DEVICE)
			{
				if (ucCmdType == CMD_TYPE_CONF_CHANGE)
				{
					/* The MSP response status could be a nack */
					Status = VCORE_CheckStatus(Event, pstTempMsg, pstChnl->usMSPChnlId,
								ucCmdCls,
								CMD_TYPE_CONF_RESP,
								UT_LE2CPU16((((U16 *) (pstTempReqMsg->
											fifo))[SIDX_FUNC_CODE])));
					if (Status == SUCCESS)
					{
						/* if we get ack on VOPENA */
						if ((UT_LE2CPU16((((U16 *) (pstTempReqMsg-> fifo))[SIDX_FUNC_CODE])) == FC_VOIP_VOPENA))
						{
							/* on disable, the connection is now inactive */
							if (UT_LE2CPU16((((U16 *) (pstTempReqMsg-> fifo))[SIDX_VSTATUS])) == VOIP_VOPENA_MODE_DISABLE)
								pstChnl->bIsActive = False;
							/* on enable, the connection is now active */
							else
								pstChnl->bIsActive = True;
						}
					}
					else /* we get a nack, save it */
						MspResponse = Status;
				}
			}

			Status = VCORE_CheckCmd(pstChnl->pstDev, (SMsg *)pstTempReqMsg);
			if (Status != SUCCESS)
				break;

			Status = VCORE_CheckCmd(pstChnl->pstDev, (SMsg *)pstTempMsg);
			if (Status != SUCCESS)
				break;

			VCORE_GetNextCmd((SMsg*)pstTempReqMsg);
			VCORE_GetNextCmd((SMsg*)pstTempMsg);

		} while (pstTempReqMsg->fifo_size && pstTempMsg->fifo_size);

		/* all the commands have been parsed */
		/* now report message error format before nacks,
		 but still report message error format if there are no nacks */
		if (Status == SUCCESS)
			Status = MspResponse;

		break;

	default:

		UT_ErrorLog(VCORE, "VCORE_CheckStatusMulti: Unexpected event received (%u)\n", Event);
		Status = VAPI_ERR_UNDEFINED_EVENT;
		break;
	}

out:
	return Status;

}

VSTATUS VCORE_CheckPassThruStatus(SChnl * pstChnl, gtl_msg_t * pstReqMsg, U16 Event, gtl_msg_t * pstMsg)
{
	gtl_msg_t stTempReqMsg, stTempMsg;
	gtl_msg_t *pstTempReqMsg = &stTempReqMsg;
	gtl_msg_t *pstTempMsg = &stTempMsg;
	U8 ucCmdCls;
	U8 ucCmdType;
	VSTATUS Status = SUCCESS;
	VSTATUS MspResponse = SUCCESS;
	SApiHdr *pstFifo;

	UT_Log(VCORE, INFO, "VCORE_CheckPassThruStatus: Event %u\n", Event);
	switch (Event)
	{
	case FSM_EVENT_TIMEOUT:
		Status = VAPI_ERR_MSP_NO_RESPONSE;
		break;

	case FSM_EVENT_REC_PLAY_TERMINATOR:
		Status = VAPI_IND_STOP_REC_PLAY;
		break;

	case FSM_EVENT_DEV_MSG:

		UT_MemCopy(pstTempReqMsg, pstReqMsg, sizeof(gtl_msg_t));
		UT_MemCopy(pstTempMsg, pstMsg, sizeof(gtl_msg_t));

		/* parse all the commands in the fifo */
		do
		{
			pstFifo = (SApiHdr *)pstTempReqMsg->fifo;
			ucCmdCls = pstFifo->cmd_class;
			ucCmdType = pstFifo->cmd_type;
			if (ucCmdCls == CMD_CLASS_CONF_CHANNEL || ucCmdCls == CMD_CLASS_CONF_DEVICE)
			{
	
				if (ucCmdType == CMD_TYPE_CONF_CHANGE)
				{
					/* The MSP response status could be a nack */
					Status = VCORE_CheckStatus(Event, pstTempMsg, pstChnl->usMSPChnlId,
								ucCmdCls,
								CMD_TYPE_CONF_RESP,
								UT_LE2CPU16((((U16 *) (pstTempReqMsg->
											fifo))[SIDX_FUNC_CODE])));
					if (Status == SUCCESS)
						/* we get a ack for this command, check if some internal 
						variables have to be updated */
						VCORE_CheckPassThruParams(pstChnl, pstReqMsg);
					else /* we get a nack, save it */
						MspResponse = Status;
				}
			}

			/* parse only one command in case of supervisor
			multi command not allowed for the supervisor channel */
			if (pstChnl->usMSPChnlId == SUPV_CHANNEL)
				break;


			Status = VCORE_CheckCmd(pstChnl->pstDev, (SMsg *)pstTempReqMsg);
			if (Status != SUCCESS)
				break;

			Status = VCORE_CheckCmd(pstChnl->pstDev, (SMsg *)pstTempMsg);
			if (Status != SUCCESS)
				break;

			VCORE_GetNextCmd((SMsg*)pstTempReqMsg);
			VCORE_GetNextCmd((SMsg*)pstTempMsg);

		} while (pstTempReqMsg->fifo_size && pstTempMsg->fifo_size);

		/* all the commands have been parsed */
		/* now report message error format before nacks,
		 but still report message error format if there are no nacks */
		if (Status == SUCCESS)
			Status = MspResponse;

		break;

	default:
 
		UT_ErrorLog(VCORE, "VCORE_CheckPassThruStatus: Unexpected event received (%u)\n", Event);
		Status = VAPI_ERR_UNDEFINED_EVENT;
		break;
	}

	return Status;
 
}

/***************************************************************************
 * VCORE_SetChildRequest: The function does the following things -
 ***************************************************************************/
/*!
 *  1. Fill the various field of the pre allocated request 
 *
 *	\b Inputs-Outputs: \n
 *		\li IN SVapiReq * pstVapiReq
 *		\li IN SVapiReq * pstChildVapiReq
 *		\li PFNReqFsmHdlr pfnChildReqFSMHdlr
 *		\li U16 usReqStartState
 *		\li void *pvUserData
 *		\li U16 usUserDataSize
 *
 *	\b Returns: \n
 *		\li none
 *
*/
void VCORE_SetChildRequest(IN SVapiReq * pstChildVapiReq, IN SVapiReq * pstVapiReq,
			PFNReqFsmHdlr pfnChildReqFSMHdlr, U16 usReqStartState)
{
	pstChildVapiReq->bIsSync = False;
	pstChildVapiReq->pfnReqFSMHdlr = pfnChildReqFSMHdlr;
	pstChildVapiReq->ucThreadContext = pstVapiReq->ucThreadContext;
	pstChildVapiReq->pstParentReq = pstVapiReq;
	pstChildVapiReq->uiID = pstVapiReq->uiID;
	pstChildVapiReq->uiReqId = pstVapiReq->uiReqId;
	pstChildVapiReq->ucCmdLevel = pstVapiReq->ucCmdLevel;
	pstChildVapiReq->usReqState = usReqStartState;
	pstChildVapiReq->ucInProgress = False;
}

/***************************************************************************
 * VCORE_StartChildRequest: The function does the following things -
 ***************************************************************************/
/*!
 *  1. Add the request to the begining of the list of the current channel 
 *  2. Process the request immediatelly.
 *
 *	\return U16 status \n
 *
 *	\param	IN SChnl * pstChnl \n
 *	\param	IN SVapiReq * pstVapiReq \n
*/
VSTATUS  VCORE_StartChannelChildRequest(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq)
{

	VSTATUS Status;

	UT_Log(VCORE, DEBUG, "VCORE_StartChannelChildRequest: connid (%d) req_state (%d)\n",
	       pstChnl->ConnId, pstVapiReq->usReqState);

	pstVapiReq->pstChnl = pstChnl;
	UT_AddToFront(&pstChnl->stReqList, &pstVapiReq->stNode);

	Status = VCORE_ProcessRequestList(pstChnl);

	return Status;
}

/****************************************************************************
 * VCORE_AddConnection : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Makes an entry in the AVL tree that is based on the connection
 *          ids.
 *
 *  \return 
 *  Return the status \n
 *
 *  \param pstChnl pointer to the structure to be added to the AVL tree.
  */
VSTATUS VCORE_AddConnection(SChnl *pstChnl)
{
	VSTATUS Status;
 
	/*Add the Channel in the AVL tree */
	Status = UT_AddToAVLTree(pstConnTree_g, pstChnl->ConnId, pstChnl);

	return Status;
}

/****************************************************************************
 * VCORE_RemoveConnection : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Removes an entry in the AVL tree that is based on the connection
 *          ids.
 *
 *  \return 
 *  None \n
 *
 *  \param pstChnl pointer to the structure to be removed from the AVL tree.
 */
void VCORE_RemoveConnection(SChnl *pstChnl)
{

	UT_RemFromAVLTree(pstConnTree_g, pstChnl->ConnId);
}

/****************************************************************************
 * VCORE_CheckConnection : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Makes an entry in the AVL tree that is based on the connection
 *          ids.
 *
 *  \return 
 *  Return the status \n
 *
 *  \param pstChnl pointer to the structure to be added to the AVL tree.
  */
VSTATUS VCORE_CheckConnection(SChnl *pstChnl)
{
	VSTATUS Status;
 
	/*Add the Channel in the AVL tree */
	if(UT_SearchInAVLTree(pstConnTree_g, pstChnl->ConnId))
		Status = SUCCESS;
	else
		Status = FAILURE;

	return Status;
}

/****************************************************************************
 * VCORE_AddConference : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Makes an entry in the AVL tree that is based on the conference
 *          ids.
 *
 *  \return 
 *  Status \n
 *
 *  \param pstConf Conference to be added 
 */
VSTATUS VCORE_AddConference(IN SConference *pstConf)
{
	VSTATUS Status;
 
	/*Add the conference in the AVL tree */
	Status = UT_AddToAVLTree(pstConfTree_g, pstConf->ConfId, pstConf);

	if( Status  < 0)
		UT_ErrorLog(VCORE, "VCORE_AddConference: Could not add conference, Invalid ConfId");

	return Status;
}

/****************************************************************************
 * VCORE_RemoveConference : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Removes an entry in the AVL tree that is based on the conference
 *          ids.
 *
 *  \return 
 *  void \n
 *
 *  \param pstConf Conference to be removed 
 */
void VCORE_RemoveConference(IN SConference *pstConf)
{
	/*remove the conference from the AVL tree */
	UT_RemFromAVLTree(pstConfTree_g, pstConf->ConfId);
}

VSTATUS VCORE_ProcessLockRequest(IN SChnl * pstChnl, IN SVapiReq * pstReq)
{
	VSTATUS Status;

	UT_Log(VCORE, INFO, "VCORE_ProcessLockRequest: Entering conn(%u) channel(%u) req(0x%x) reqid(%u) pfnReqHdlr(0x%x)\n",
		pstChnl->ConnId, pstChnl->usMSPChnlId, pstReq, pstReq->uiReqId, pstReq->pfnReqFSMHdlr);

	/*Check whether channel is busy */
	UT_MutexLock(&pstChnl->stReqList.stSem);

	/* Increment the vapi_frames_queued counter only on user call (not on response nor event) */
	if (pstReq->ucThreadContext == USER_CONTEXT)
	{
		UT_MutexLock(&vapi_stat.mutex_stat);
		vapi_stat.vapi_frames_queued++;
		UT_MutexUnLock(&vapi_stat.mutex_stat);
	}

	pstReq->pstChnl = pstChnl;

	if (__UT_IsListEmpty(&pstChnl->stReqList))
	{
		/* Go To the tail of the list of request on the given channel 
		   and add the request */
		__UT_AddToTail(&pstChnl->stReqList, &pstReq->stNode);

		pstReq->ucInProgress = True;

		UT_MutexUnLock(&pstChnl->stReqList.stSem);

		/*If channlel is not busy Call Req-FSM handler. */
		UT_Log(VCORE, INFO, "VCORE_ProcessLockRequest: Launching reqid(%u) state(%u)\n", pstReq->uiReqId, pstReq->usReqState);

		/* proceed with the state mahine on this request */
		Status = pstReq->pfnReqFSMHdlr(pstChnl, pstReq, FSM_EVENT_START, NULL);
	}
	else
	{
		/* Go To the tail of the list of request on the given channel 
		   and add the request */
		__UT_AddToTail(&pstChnl->stReqList, &pstReq->stNode);

		/* In async mode this request has the VAPI_CONTEXT marker from now for all its life */
		if (pstReq->bIsSync == False)
			pstReq->ucThreadContext = VAPI_CONTEXT;

		UT_MutexUnLock(&pstChnl->stReqList.stSem);

		Status = SUCCESS;
	}

	/* retrieve the status set from the GTL thread */ 
	Status = pstReq->Status;

	UT_Log(VCORE, INFO, "VCORE_ProcessLockRequest: Exiting status(%d)\n", Status);

	return Status;
}

/****************************************************************************
 * VCORE_UpdateMR : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Update the MajorRelease variable for the device
 *
 *  \return 
 *  VSTATUS \n
 *
 *  \param pstDev Device to be updated with the MR number 
 *  \param pvFifo fifo buffer containing the result of the FC_ARM_FW_BRANCH query 
 */
VSTATUS VCORE_UpdateMR(struct _SDevice *pstDev, void *pvFifo)
{
	U8 *paucFifo ;
	VSTATUS Status = SUCCESS;

	paucFifo = (U8 *)pvFifo;
	pstDev->MajorRelease = atoi((char *)&paucFifo[17]);
	UT_Log(VCORE, INFO, "VCORE_UpdateMR: Major Release = (%d)\n", pstDev->MajorRelease);

	/* before Major Release 7 the max size is 256 bytes whatever the control interface*/ 
	if (pstDev->MajorRelease < MR7)
		pstDev->MaxMsgSize = DEFAULT_FIFO_MAX_SIZE; /* 256 bytes*/
	else
	{
		switch (pstDev->eItfType)
		{
		case ePCI_ITF:
			pstDev->MaxMsgSize = PCI_FIFO_MAX_SIZE;
			break;
	
		case eCSM_ITF:
			pstDev->MaxMsgSize = ETH_FIFO_MAX_SIZE;
			break;
	
		case ePOS_ITF:
			pstDev->MaxMsgSize = POS_FIFO_MAX_SIZE;
			break;
	
		default:
			UT_ErrorLog(GTL, "VCORE_UpdateMR: Unsupported interface type (%u)\n", pstDev->eItfType);
			Status = VAPI_ERR_UNSUPP_FEATURE; 
			break;
		}
	}

	return Status;
}

/****************************************************************************
 * VCORE_UpdateEth : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Update the Eth structure for the device
 *
 *  \return 
 *  VSTATUS \n
 *
 *  \param pstDev Device to be updated with the Eth header 
 *  \param pvFifo fifo buffer containing the result of the FC_GET_ETH_HDRH query 
 */
VSTATUS VCORE_UpdateEth(struct _SDevice *pstDev, void *pvFifo)
{
	U8 *paucFifo;
	U16 *paucFifoParams;
	int EthLayerHeader;

	paucFifo = (U8 *) pvFifo;
	paucFifoParams = (U16 *) pvFifo;

	EthLayerHeader = 4;

	while(1)
	{
		if (EthLayerHeader*2 >= paucFifo[0])
			return FAILURE;
	
		if (SET_ETH_HDR_PROTOCOL_IP == (UT_LE2CPU16(paucFifoParams[EthLayerHeader]) & 0x0003))
		{
			/* save the Src and Dest MAC addresses as the default for device*/
			UT_Log(VCORE, INFO, "VCORE_UpdateEth: Upadate MACs\n");
			UT_MemCopy(pstDev->stEthMacAddresses.astDestMacAddr, &paucFifo[EthLayerHeader * 2 + 2], 6);
			UT_MemCopy(pstDev->stEthMacAddresses.astSrcMacAddr, &paucFifo[EthLayerHeader * 2 + 8], 6);

			if ((UT_LE2CPU16(paucFifoParams[EthLayerHeader]) & 0xFF00) > 16)
			{
				UT_Log(VCORE, INFO, "VCORE_UpdateEth: Upadate VLAN ID\n");
				pstDev->stEthMacAddresses.usVlanId = UT_LE2CPU32(EthLayerHeader + 7);
			}

			pstDev->bEthHdrSet = True;

			return SUCCESS;
		}

		EthLayerHeader +=  (UT_LE2CPU16(paucFifoParams[EthLayerHeader]) & 0xFF00) /2;
	}

}

/****************************************************************************
 * VCORE_UpdateIP : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Update the Eth structure for the device
 *
 *  \return 
 *  VSTATUS \n
 *
 *  \param pstDev Device to be updated with the IP address 
 *  \param pvFifo fifo buffer containing the result of the FC_IP_ADDRESS) query 
 */
VSTATUS VCORE_UpdateIP(struct _SDevice *pstDev, void *pvFifo)
{
	U32 *paucFifo;
	paucFifo = (U32 *) pvFifo;

	UT_Log(VCORE, INFO, "VCORE_UpdateIP: Upadate IP\n");
	pstDev->uiSrcIpAddr = UT_LE2CPU32(paucFifo[2]);
	
	if (pstDev->uiSrcIpAddr)
	{
		pstDev->eSrcIpMode = eSINGLE_IP;
		pstDev->eEthHdrMode = eMODE_IP;

		return SUCCESS;
	}

	return FAILURE;
}

/****************************************************************************
 * VCORE_UpdateCsmeMulti
 **************************************************************************/
VSTATUS VCORE_UpdateCsmeMulti(struct _SDevice *pstDev, void *pvFifo)
{
	UT_Log(VCORE, INFO, "VCORE_UpdateCsmeMulti: Upadate multi command status\n");

	if (UT_LE2CPU16(((U16 *) pvFifo)[4]) > 1)
	{
		return FAILURE;
	}
	else
	{
		pstDev->bMultiCmdEnabled = UT_LE2CPU16(((U16 *) pvFifo)[4]);
		return SUCCESS;
	}
}

/****************************************************************************
 * VCORE_UpdateSpuFeature
 **************************************************************************/
VSTATUS VCORE_UpdateSpuFeature(struct _SDevice *pstDev, void *pvFifo)
{
	UT_Log(VCORE, INFO, "VCORE_UpdateSpuFeature: Upadate SPU feature mask\n");

	pstDev->usSpuFeatureMask = UT_LE2CPU16(((U16 *) pvFifo)[4]);
	return SUCCESS;
}

/****************************************************************************
 * VCORE_SetupBootCmdMsg : The function does the following things -
 ***************************************************************************/
/*! 
 *  This function fills the various Mailbox or fifo fields for Comcerto boot commands
 *  according the interface type.
 *  It also update the Msg.type field.
 *
 *  \param pstMsg       The boot message that is to be initialized
 *  \param ucCmdType    The MSP Boot Command Type 
 *  \param eItfType     The interface type to control the device (pstChnl->pstDev->eItfType) 
 *
 *  \return SUCCESS/VAPI_ERR_UNSUPP_FEATURE 
*/

VSTATUS VCORE_SetupBootCmdMsg(OUT gtl_msg_t * pstMsg, IN U8 ucCmdType, EDevItfType eItfType)
{
	VSTATUS Status = SUCCESS;

	/* Flag the message as boot command */
	pstMsg->type = BOOT;

	pstMsg->channel = 0;

	switch (eItfType)
	{
	case ePCI_ITF:
		pstMsg->mailbox.reg3 = ucCmdType << 8;
		pstMsg->mailbox.reg2 = 0;
		pstMsg->mailbox.reg1 = 0;
		pstMsg->mailbox.reg0 = 0;

		break;

	case eCSM_ITF:
		*((U16 *) pstMsg->fifo + 0) = 0;
		*((U16 *) pstMsg->fifo + 1) = UT_CPU2BE16(CMD_CLASS_ETH_BOOT_MSG << 8 | ucCmdType);
		*((U16 *) pstMsg->fifo + 2) = 0;

		break;

	case ePOS_ITF:
		*((U16 *) pstMsg->fifo + 0) = 0;
		*((U8 *) pstMsg->fifo + 2) = ucCmdType;
		*((U8 *) pstMsg->fifo + 3) = CMD_CLASS_ETH_BOOT_MSG;
		*((U16 *) pstMsg->fifo + 2) = 0;

		break;

	default:
		UT_ErrorLog(GTL, "VCORE_SetupBootCmdMsg: Unsupported interface type (%u)\n", eItfType);
		Status = VAPI_ERR_UNSUPP_FEATURE; 
		break;
		}

	return Status;
}

/****************************************************************************
 * VCORE_ResetDevice : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Call the registered customer device reset function (if exists)
 *      -   Call the GTL_Reset function (may or may not have been implemented)
 *
 *  \return 
 *  VSTATUS \n
 *
 *  \param pstDev Device structure of the device to be reset 
 */
VSTATUS VCORE_ResetDevice(IN SDevice *pstDev)
{
	VSTATUS Status = SUCCESS;

	/* if customer reset function registered call it */
	if(pstDev->pfnReset != NULL)
	{
		Status = pstDev->pfnReset(pstDev->pvResetData);

		if(Status != SUCCESS) 
			goto out;
	}
	 /* GTL_Reset may or may not have been implemented, so 
	 continue with reset even if GTL_Reset returned error.*/
	GTL_Reset(pstDev->DevId);
 
out:
	return Status;
}

/****************************************************************************
 * VCORE_SetupCmdMsg : The function does the following things -
 ***************************************************************************/
/*! 
 *
 *  \return None 
 *
 *  \param pstMsg       The message that is to be initialized
 *  \param usChnlId     Channel id 
 *  \param ucCmdClass   The MSP Command Class 
 *  \param ucCmdType    The MSP Command Type 
 *  \param usFcode      The MSP function code.
 *  \param ucFifoLen    The actual fifo length (without padding) in bytes 
 */
void VCORE_SetupCmdMsg(OUT gtl_msg_t * pstMsg,
		       IN U16 usChnlId, IN U8 ucCmdClass, IN U8 ucCmdType, IN U16 usFcode, IN U8 ucFifoLen)
{
	/* Flag the message as non boot command */
	pstMsg->type = 0;
	pstMsg->channel = usChnlId;

	/* add potential padding */
	pstMsg->fifo_size = ALIGN_MESSAGE_LENGTH(ucFifoLen);

	*((U16 *) pstMsg->fifo + 0) = ucFifoLen;
	*((U16 *) pstMsg->fifo + 1) = ucCmdClass << 8 | ucCmdType;
	*((U16 *) pstMsg->fifo + 2) = usFcode;

	return;
}

/****************************************************************************
 * VCORE_InitMsg : The function does the following things -
 ***************************************************************************/
/*! 
 *  This function init a allocated message's fields with default values.
 *
 *  \return None 
 *
 *  \param pstMsg	The message that is to be initialized
 *  \param pusFifo	The fifo for this message
 *  \param MaxLen	The max number of bytes the fifo can handle
*/
void VCORE_InitMsg(gtl_msg_t *pstMsg, U8 *pusFifo, U16 usMaxLen)
{
	UT_MemSet(pusFifo, 0, usMaxLen);

	pstMsg->fifo = (void *)pusFifo;
	/* for now we have no bytes in the message*/
	pstMsg->fifo_size = 0;
	pstMsg->max_fifo_size = usMaxLen;
}
/****************************************************************************
 * VCORE_SetupCmdHdrMsg : The function does the following things -
 ***************************************************************************/
/*! 
 *  This function fills the API header of a command and add it to the message fifo.
 *  The message & command fifo length are updated to 8 (size of the API header).
 *  This takes it account that a command could be already present in the fifo.
 *  It can be use to build a single or multiple command.
 *
 *  \return None 
 *
 *  \param pstMsg	The message that is to be filled
 *  \param ucCmdClass	The Comcerto ucCmdClass ID 
 *  \param ucCmdType	The Comcerto ucCmdType ID 
 *  \param usFcode	The Comcerto Function code 
 *			The list of parameters values 
*/
void VCORE_SetupCmdHdrMsg(OUT gtl_msg_t * pstMsg, IN U8 ucCmdClass, IN U8 ucCmdType, IN U16 usFcode)
{
	/* the message may already contains a command.
	points to the beginning of a new command */ 
	SApiHdr *psApiFifo = (SApiHdr *)((U8*)pstMsg->fifo + pstMsg->fifo_size);

	UT_Log(VCORE, INFO, "VCORE_SetupCmdHdrMsg: Entering\n");
	/* Flag the message as non boot command */
	pstMsg->type = 0;

	/* For now set the fifo size to 8 (API header size)*/
	psApiFifo->length = sizeof(SApiHdr);

	psApiFifo->cmd_type = ucCmdType;
	psApiFifo->cmd_class = ucCmdClass;
	psApiFifo->func_code = UT_CPU2LE16(usFcode);

	/* the message has now 8 more bytes in his length*/
	pstMsg->fifo_size += psApiFifo->length;
	UT_Log(VCORE, INFO, "VCORE_SetupCmdHdrMsg: Exiting CC(0x%02x), CT(0x%02x), FC(0x%04x), fifo(%u)\n",
			psApiFifo->cmd_class, psApiFifo->cmd_type, psApiFifo->func_code, pstMsg->fifo_size);

	return;
}


/****************************************************************************
 * VCORE_SetPayloadMsgFromlist : The function does the following things -
 ***************************************************************************/
/*! 
 *  This function fills the fifo parameters from the va list.
 *  The message & command fifo length are updated accordingly the number of parameters.
 *  This takes it account that a command could be already present in the fifo.
 *  It can be use to build a single or multiple command.
 *  The parameters have to be passed in the correct endianness.
 *  This function must be called just after VCORE_SetupCmdHdrMsg
 *  This function is called internally from VCORE_SetPayloadMsg()
 *  or from the user interface
 *
 *  \return None 
 *
 *  \param pstMsg       The message that is to be filled
 *  \param ParamsNum    The number of parameters 
 *  \param ParamList	The list of parameters values 
 */
VSTATUS VCORE_SetPayloadMsgFromList(IN OUT gtl_msg_t * pstMsg,  U16 usParamsNum, va_list ParamList)
{
	SApiHdr *psApiFifo;
	U16 *pusFifoParam;
	int i;

	/* This pointer points to the beginning of the parameters in the fifo */
	pusFifoParam = (U16*) ((U8*)pstMsg->fifo + pstMsg->fifo_size);
	/* this pointer points to the beginning the fifo */
	psApiFifo = (SApiHdr *) ((U8*)pstMsg->fifo + pstMsg->fifo_size - sizeof(SApiHdr));

	for (i = 0; i < usParamsNum; i++)
	{
		pusFifoParam[i] = va_arg (ParamList, int);
	}

	/* Update the fifo length (each param is a U16)*/
	psApiFifo->length += usParamsNum * 2;

	/* update the message length. Take in account the potential padding */
	pstMsg->fifo_size += ALIGN_MESSAGE_LENGTH(usParamsNum * 2);

	if (pstMsg->fifo_size > pstMsg->max_fifo_size)
	{
		UT_ErrorLog(VCORE, "VCORE_SetPayloadMsgFromlist: Wrong fifo size (%u)\n", pstMsg->fifo_size);
		return VAPI_ERR_FIFO_SIZE;
	}
	else
		return SUCCESS;

}

/****************************************************************************
 * VCORE_SetPayloadMsg : The function does the following things -
 ***************************************************************************/
/*! 
 *  This function build the parameters list.
 *  Then it calls VCORE_SetPayloadMsgFromlist which will fill the fifo parameters from the va list.
 *
 *  \return None 
 *
 *  \param pstMsg       The message that is to be filled
 *  \param ParamsNum    The number of parameters 
 *			The list of parameters values 
 */
VSTATUS VCORE_SetPayloadMsg(IN OUT gtl_msg_t * pstMsg, U16 usParamsNum, ...)
{
	VSTATUS Status;
	va_list ParamList;

	UT_va_start(ParamList, usParamsNum);
	Status = VCORE_SetPayloadMsgFromList(pstMsg, usParamsNum, ParamList);
	UT_va_end(ParamList);

	return Status;
}

/****************************************************************************
 * VCORE_PutParamPayloadMsg : The function does the following things -
 ***************************************************************************/
/*! 
 *  This function adds a parameters to a message payload.
 *  The message & command fifo length are updated however it doesn't update the padding.
 *  This takes it account that a command could be already present in the fifo.
 *  It can ONLY BE USED TO BUILD A SINGLE COMMAND.
 *  The parameters have to be passed in the correct endianness.
 *  This functions must be called after VCORE_SetupCmdHdrMsg
 *
 *  \return None 
 *
 *  \param pstMsg       The message that is to be filled
 *  \param Parameter    The parameter to be added 
  */
void VCORE_PutParamPayloadMsg(IN OUT gtl_msg_t * pstMsg, U16 usParameter)
{
	SApiHdr *psApiFifo;
	U16 *pusFifoParam;

	/* This pointer points to the beginning of the parameters in the fifo */
	pusFifoParam = (U16*)((U8*)pstMsg->fifo + pstMsg->fifo_size);
	/* this pointer points to the beginning the fifo */
	psApiFifo = (SApiHdr *)((U8*)pstMsg->fifo);

	*(pusFifoParam) = usParameter;

	/* update the message length. Doesn't take in account the potential padding */
	pstMsg->fifo_size += 2;
	/* Update the command length */
	psApiFifo->length = pstMsg->fifo_size;

}

/****************************************************************************
 * VCORE_SetPayloadMsgFromBuffer : The function does the following things -
 ***************************************************************************/
/*! 
 *  This function fills the fifo parameters from the Parameters buffer.
 *  The message & command fifo length are updated.
 *  This takes it account that a command could be already present in the fifo.
 *  It can be use to build a single or multiple command.
 *  The parameters have to be passed in the correct endianness.
 *  This functions must be called after VCORE_SetupCmdHdrMsg
 *
 *  \return None 
 *
 *  \param pstMsg       The message that is to be filled
 *  \param ParamsNum    The parameter to be added 
 *  \param Parameters    The parameter to be added 
  */
VSTATUS VCORE_SetPayloadMsgFromBuffer(IN OUT gtl_msg_t * pstMsg, U16 usParamsNum, U16 *Parameters)
{
	SApiHdr *psApiFifo;
	U16 *pusFifoParam;

	/* This pointer points to the beginning of the parameters in the fifo */
	pusFifoParam = (U16*) ((U8*)pstMsg->fifo + pstMsg->fifo_size);
	/* this pointer points to the beginning the fifo */
	psApiFifo = (SApiHdr *)((U8*)pstMsg->fifo + pstMsg->fifo_size - sizeof(SApiHdr));

	UT_MemCopy(pusFifoParam, Parameters, 2 * usParamsNum);

	/* Update the fifo length */
	psApiFifo->length += usParamsNum * 2;

	/* update the message length. Take in account the potential padding */
	pstMsg->fifo_size += ALIGN_MESSAGE_LENGTH(usParamsNum * 2);

	if (pstMsg->fifo_size > pstMsg->max_fifo_size)
	{
		UT_ErrorLog(VCORE, "VCORE_SetPayloadMsgFromBuffer: Wrong fifo size (%u)\n", pstMsg->fifo_size);
		return VAPI_ERR_FIFO_SIZE;
	}
	else
		return SUCCESS;

}

/****************************************************************************
 * VCORE_AddPayloadMsgFromBuffer : The function does the following things -
 ***************************************************************************/
/*! 
 *  This function fills the fifo parameters from the Parameters buffer.
 *  The message & command fifo length are updated.
 *  This takes it account that a command could be already present in the fifo.
 *  It can be use to build a SINGLE COMMAND ONLY.
 *  The parameters have to be passed in the correct endianness.
 *  IT DOESN"T ADJUST THE PADDING.
 *  This functions must be called after VCORE_SetupCmdHdrMsg
 *
 *  \return None 
 *
 *  \param pstMsg       The message that is to be filled
 *  \param ParamsNum    The parameter to be added 
 *  \param Parameters    The parameter to be added 
  */
VSTATUS VCORE_AddPayloadMsgFromBuffer(IN OUT gtl_msg_t * pstMsg, U16 usParamsNum, U16 *Parameters)
{
	SApiHdr *psApiFifo;
	U16 *pusFifoParam;

	/* This pointer points to the beginning of the parameters in the fifo */
	pusFifoParam = (U16*) ((U8*)pstMsg->fifo + pstMsg->fifo_size);
	/* this pointer points to the beginning the fifo */
	psApiFifo = (SApiHdr *)((U8*)pstMsg->fifo);

	UT_MemCopy(pusFifoParam, Parameters, 2 * usParamsNum);

	/* Update the fifo length */
	psApiFifo->length += usParamsNum * 2;

	/* update the message length */
	pstMsg->fifo_size += usParamsNum * 2;

	if (pstMsg->fifo_size > pstMsg->max_fifo_size)
	{
		UT_ErrorLog(VCORE, "VCORE_AddPayloadMsgFromBuffer: Wrong fifo size (%u)\n", pstMsg->fifo_size);
		return VAPI_ERR_FIFO_SIZE;
	}
	else
		return SUCCESS;

}
/****************************************************************************
 * VCORE_SetPaddingMsg : The function does the following things -
 ***************************************************************************/
/*! 
 *  This function adjust the pading for the current message.
 *  This functions must be called after VCORE_SetupCmdHdrMsg
 *
 *  \return None 
 *
 *  \param pstMsg       The message that is to be adjusted
  */
VSTATUS VCORE_SetPaddingMsg(IN OUT gtl_msg_t * pstMsg)
{
	/* Add the padding if required */
	pstMsg->fifo_size = ALIGN_MESSAGE_LENGTH(pstMsg->fifo_size);

	if (pstMsg->fifo_size > pstMsg->max_fifo_size)
	{
		UT_ErrorLog(VCORE, "VCORE_SetPaddingMsg: Wrong fifo size (%u)\n", pstMsg->fifo_size);
		return VAPI_ERR_FIFO_SIZE;
	}
	else
		return SUCCESS;
}

/****************************************************************************
 * VCORE_CheckMessage : The function does the following things -
 ***************************************************************************/
/*! 
 *  This function checks the mesage length integrity.
 *
 *  \return VSTATUS 
 *
 *  \param pstDevice	The device to which the message is intended for
 *  \param pstMsg	The message that is to be checked
 *  \param ucLevel	The message level (device or connection level)
 */
VSTATUS VCORE_CheckMessage(SDevice *pstDevice, IN SMsg * pstMsg, IN U8 ucLevel)
{
	VSTATUS Status = SUCCESS;
	U16 usBytesInFifo;

	do
	{
		Status = VCORE_CheckCmd(pstDevice, pstMsg);
		/* check if the full fifo length (including padding) is not out of range */
		if (Status != SUCCESS)
		{
			UT_ErrorLog(APPITF, "VCORE_CheckMessage: Warning fifo lenght out of range (%d)\n", pstMsg->uiFifoSize);
			break;
		}

		usBytesInFifo = VCORE_GetNextCmd((SMsg*)pstMsg);

		if (((ucLevel == CMD_LEVEL_DEVICE) || (ucLevel == CMD_LEVEL_CONF)) && 
				(usBytesInFifo != 0))
		{
			UT_ErrorLog(APPITF, "VCORE_CheckMessage: Only one command allowed for Device or Conference level\n");
			Status = VAPI_ERR_INVALID_PARAM;
			break;
		}

	} while(usBytesInFifo > 0);

	return Status;
	
}

/****************************************************************************
 * VCORE_SetUserFaxOption : The function does the following things -
 ***************************************************************************/
/*! 
 *  This function copies the user fqx configuration to the internal fax structure config.
 *
 *  \return VSTATUS 
 *
 *  \param pstUserFaxOption	Pointer to user fax structure configuration
 *  \param pstFsmFaxOptions	Pointer to inernal fax structure configuration
 */
VSTATUS VCORE_SetUserFaxOption(SFaxConfigOpts *pstUserFaxOption, IN SFsmFaxOptions *pstFsmFaxOptions)
{

	pstFsmFaxOptions->stFaxOpts.param_4.bits.t38_packet_loss_concealment =
				pstUserFaxOption->stFaxOpts.ucT38PktLossConcealment;

	pstFsmFaxOptions->stFaxOpts.param_4.bits.ecm_disable =
			pstUserFaxOption->stFaxOpts.ucECMDisable;

	pstFsmFaxOptions->stFaxOpts.param_4.bits.udptl_redundancy_fax_page =
			pstUserFaxOption->stFaxOpts.ucFaxDataRedundancyCnt;

	pstFsmFaxOptions->stFaxOpts.param_4.bits.udptl_redundancy_t30 =
			pstUserFaxOption->stFaxOpts.ucT30RedundancyCnt;

	pstFsmFaxOptions->stFaxOpts.param_4.bits.speed_limit =
			pstUserFaxOption->stFaxOpts.ucFaxConnSpeedLimit;

	pstFsmFaxOptions->stFaxOpts.param_4.bits.error_recovery_method =
			pstUserFaxOption->stFaxOpts.ucErrRcvyMethod;

	pstFsmFaxOptions->stFaxOpts.param_4.bits.tcf_procedure =
			pstUserFaxOption->stFaxOpts.ucTCFProcedure;

	pstFsmFaxOptions->stFaxLevel.faxlvl = pstUserFaxOption->ucFaxTxLevel;

	pstFsmFaxOptions->bUseIt = True;

	return SUCCESS;
}
