/*! \file vapi.h */

/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of c
 * the Agreement.
 */

#ifndef _VAPI_H
#define _VAPI_H

#ifdef __cplusplus
extern "C" {
#endif

#include "ut.h"
#include "vapi_profile.h"

#ifndef	_UT_H
/** unsigned char */
typedef unsigned char U8;
/** unsigned short */
typedef unsigned short U16;
/** unsigned int */
typedef unsigned int U32;
/** signed char */
typedef signed char S8;
/** signed short */
typedef signed short S16;
/** signed int */
typedef signed int S32;
/** Boolean */
typedef enum {
	False,	/**< 0*/
	True	/**< 1*/
} Boolean;

/** Indicate success of an operation*/
#define SUCCESS 0
/** Indicate failure of an operation*/
#define FAILURE -1
/** Specifying input parameters in function prototypes*/
#define IN
/** Specifying output parameters in function prototypes*/
#define OUT

#endif

#define VAPI_VERSION_MAJOR 2
#define VAPI_VERSION_MINOR 18
#define VAPI_VERSION_PATCH 3
#define VAPI_VERSION_EXTRA ""

/** The VAPI_VERSION defines the vapi implementation release */
#define VAPI_VERSION(a,b,c) (((a) << 16) + ((b) << 8) + (c))
#define VAPI_RELEASE VAPI_VERSION(VAPI_VERSION_MAJOR,VAPI_VERSION_MINOR,VAPI_VERSION_PATCH)


#define VAPI_API_VERSION_MAJOR 8
#define VAPI_API_VERSION_MINOR 0

/** The VAPI_API_VERSION defines the vapi API user interface release \n
A same API release can be share by several VAPI release */
#define VAPI_API_VERSION(a,b) (((a) << 8) + (b))
#define VAPI_API_RELEASE VAPI_API_VERSION(VAPI_API_VERSION_MAJOR,VAPI_API_VERSION_MINOR)

/** int */
typedef int VSTATUS;

typedef U32 CONNID;	/**< Connection ID: used to identify a connection uniquely across all devices.\n A Connection ID used for creating a connection on one device cannot be used for creating a connection on another device*/
typedef U32 DEVID;	/**< Device Id: used to identify a device uniquely. \n The valid value of the Device Id is determined by the device list present in the VAPI source code in gtl_config.c */
typedef U32 CONFID;	/**< Conference ID: used to identify a conference uniquely across all devices.\n Conference IDs may overlap with the values used for Connection IDs (participant IDs may not overlap with Connection IDs).*/
typedef U32 PARTID;	/**<  Participant ID:  used to identify a participant uniquely across all devices.\n Participant ID may be used in the same way as the Connection ID for all API functions.*/

typedef struct _VAPI_Statistics {
	U32 api_frames_received;
	U32 api_error_frames_received;
	U32 vapi_frames_completed;
	U32 api_frames_sent;
	U32 vapi_frames_queued;
	SMutex mutex_stat;
} VAPI_Statistics;

/** Participant State*/
typedef enum {
	eUNHOLD = 0,		/**< UNHOLD */
	eSIMPLEHOLD = 1,	/**< SIMPLE_HOLD */
	eANNHOLD = 2		/**< ANNOUNCEMENT_HOLD */
} EHoldState;


/** Mailbox registers*/
    typedef struct _SMailbox {
	U16 usMsgId;		/**< Bit mapped (see CRM) */
	U16 usParam1;		/**< Number of bytes in FIFO including padding */
	U16 usParam2;		/**< Channel number */
	U16 usParam3;		/**< Reserved */
} SMailbox;

/** Raw message structure used in Pass-thru API*/
typedef struct _SMsg {
	SMailbox sMailbox;	/**< Mailbox register values*/
	U32 uiFifoSize;		/**< Size of Fifo in bytes*/
	U16 *pusFifo;		/**< Fifo contents*/
} SMsg;

/** PCM Coding Standard A-Law*/
#define A_LAW 0
/** PCM Coding Standard U-Law*/
#define U_LAW 1

/** Command Level connection */
#define CMD_LEVEL_CONN      0
/** Command Level device */
#define CMD_LEVEL_DEVICE    1
/** Command Level conference */
#define CMD_LEVEL_CONF      2

/** Event callback connection level*/
#define EVENT_LEVEL_CONN    0
/** Event callback device level*/
#define EVENT_LEVEL_DEVICE  1
/** Event callback generic level*/
#define EVENT_LEVEL_GENERIC 2

/** Min fifo len for control message (API commands)*/
#define FIFO_MIN_SIZE	8

/** Max fifo len for control message (API commands)*/
#define DEFAULT_FIFO_MAX_SIZE 256
#define PCI_FIFO_MAX_SIZE DEFAULT_FIFO_MAX_SIZE
#define ETH_FIFO_MAX_SIZE 1500
#define POS_FIFO_MAX_SIZE 1500

#define MAX_ALERT_DATA_SIZE	248	/* Max lenght of the alert fifo buffer minus header */

/** Number of bytes for padding commands*/
#define PADDING_SIZE	6

/** Maximum memory block to be read at a time during Core Dump
    Currently the maximum values supported are:
        - 224 for CSME Slave and 
        - 256 for PCI Slave */

#define VAPI_PCI_DUMP_MAX_BLK	256

#define VAPI_ETH_DUMP_MAX_BLK	224

/**Debug level: No debug */
#define NO_DEBUG			0
/**Debug level: API level */
#define API_LEVEL			1
/**Debug level: Packect level */
#define API_PACKET_LEVEL		2
/**Debug level: VAPI Information level */
#define VAPI_INFO			3
/**Debug level: GTL Information level */
#define VAPI_GTL_INFO			4
/**Debug level: UT Information level */
#define VAPI_UT				5
/**Debug level: ALL Information level */
#define ALL_INFO			6


/** Severity level INFO to be passed to VAPI_SetDebugLevel() */
#define INFO	1
/** Severity level DEBUG to be passed to VAPI_SetDebugLevel() */
#define DEBUG   2
/** Severity level PACKET to be passed to VAPI_SetDebugLevel() */
#define PACKET  3
/** Severity level WARN to be passed to VAPI_SetDebugLevel() */
#define WARN	4

/**Special severity level*/
#define ALWYS	5		/*Should not be used for setting debug level in array */
#define NODBG   6		/*Should not be used with UT_Log calls */

#define MAX_SRC_IPADDR  10	/*Max possible source IP adresses */

/**Max available buffer size of SYNCDAT/SYNCEOF packets*/
#define MAX_SYNCDAT_BUF_SIZE	240
#define MAX_SYNCEOF_BUF_SIZE	1492

/** Event codes defined for events coming from MSP*/
typedef enum {
	eVAPI_SPI_EVENT,			/**< Event SPI like On-hook/Off-Hook*/
	eVAPI_TONE_DETECT_EVENT,		/**< Event for tone detection (like for DTMF)*/
	eVAPI_TONE_GEN_CMPLT_EVENT,		/**< Event Tone Generation completed */
	eVAPI_PT_CHANGE_EVENT,			/**< Event Payload type change */
	eVAPI_SSRC_CHANGE_EVENT,		/**< Event SSRC Change received */
	eVAPI_SSRC_VIOLATION_EVENT,		/**< Event SSRC Violation received */
	eVAPI_NTE_TRANSMIT_COMPLETE_EVENT,	/**< Event NTE transmit completed */
	eVAPI_NTE_RECVD_EVENT,			/**< Event NTE received */
	eVAPI_CALLER_ID_CMPLT_EVENT,		/**< Event caller ID completed */
	eVAPI_G711_CONCURRENT_DECODER_EVENT,	/**< Event G711 concurrent decoder */
	eVAPI_PASSTHRU_CONCURRENT_DECODER_EVENT, /**< Event Passthru concurrent decoder */
	eVAPI_DECODER_EVENT, 			/**< Event Decoder */
	eVAPI_CALLER_ID_DETECTED_EVENT,		/**< Event Caller ID detection*/
	eVAPI_REMOTE_DETECT_EVENT,		/**< Event for remote detection*/
	eVAPI_FAX_SWITCH_CMPLT_EVENT,		/**< Event VAPI has completed Fax switchover */
	eVAPI_ALERT_IND,			/**< Event ALERT Indication */
	eVAPI_IMSOTDM_H245_EVENT,		/**< IMS over TDM H.245 Indication */
	eVAPI_SYNCDAT_EVENT,			/**< Event SYNCDAT indication*/
	eVAPI_SYNCEOF_EVENT,			/**< Event SYNCEOF indication*/
	eVAPI_IPTONE_DETECT_EVENT,		/**< Event IP side tone detection */
	eVAPI_UNDEFINED_EVENT			/**< Event is not suported by VAPI*/
} EEventCode;

/** Type of connection*/
typedef enum {
	eVOIP = 0x0002,			/**< Voice over IP = 0x02*/
	eFOIP = 0x0007,			/**< Fax over IP = 0x07*/
	eVOATM = 0x0008,		/**< Voice over ATM = 0x08 (using AAL2 or CDMAoAAL5) */
	eIUUPORTP = 0x0009,		/**< IuUP or NbUP over RTP/UDP/IP*/
	eIUUPOAAL2 = 0x000A,		/**< IuUP over AAL2 = 0x0A*/
	eSLIC24 = 0x0010,		/**< Reserved (SLIC24)*/
	eComplexTone = 0x0011,		/**< Complex Tone Generation (TONEGEN)*/
	eToneDet = 0x0012,		/**< Tone Detection (TONEDET)*/
	eWARNING_TONEGEN = 0x0013,	/**< Reserved (warning tone)*/
	eIpRedirect = 0x0014,		/**< IPREDIRECT*/
	eHDL = 0x0015,			/**< Host Data Link (PCI/uP over IPoAAL5)*/
	eHDLCOTDM = 0x0017,		/**< HDLC over TDM*/
	eDSPDIAG = 0x0020,		/**< Reserved (DSPDIAG)*/
	eAAL1CES = 0x0040,		/**< Reserved (AAL1CES)*/
	eIMSOTDM = 0x0064,		/**< IMS over TDM*/
	eCCS = 0x0080,			/**< Common Channel Signaling*/
	eCSEoPSN = 0x0081		/**< Customer Edge Signaling over Packet Switched Network*/	
} EConnType;

/** Channel mode */
typedef enum {
	eNarrowBand = 0,		/**< Narrow band mode*/
	eWideBand = 1,			/**< Wide Band mode*/
} EConnMode;

/** Channel media sub-layer */
typedef enum {
	eAudioSubLayer = 0,	/**< Address to audio sub-layer*/
	eVideoSubLayer = 1	/**< Address to video sub-layer*/
} EMediaSubLayer;

/** Operation mode of connection*/
typedef enum {
	eInactive = 0,		/**< TDM detection Inactive, packetization inactive*/
	eTdmActive = 1,		/**< TDM detection active, packetization inactive.*/
	eActive = 2		/**< TDM detection active, packetization active */
} EConnOpMode;

/** Change operation mode of connection*/
typedef enum {
	eOpModeNone = 0,	/**< No extra configuration */
	eOpModeMac = 1,		/**< Configure MAC address*/
	eOpModeIP = 2,		/**< Configure IP address*/
} EChangeOpMode;

/** Participant Type*/
typedef enum {
	eLSP	= 0x0000,
	eRSP	= 0x0001,
	eTRANSP	= 0x0011,
	eLSPWB	= 0x8000,
	eRSPWB	= 0x8001,
	eNOTPART
} EPartType;

/** Change connection struct */
typedef struct _SChangeConnInfo {
	EConnType eConnType;	
				/**< The channel type */
	EConnMode eConnMode;	
				/**< The conneciton mode Narrow or Wide band mode */
	EPartType ePartType;	
				/**< The participant type weather it's to be attached to 
				conference or not (not participant)*/
	EChangeOpMode eChangeOpMode;	
				/**< Specifies if some extra setting have to be applied to 
				the new created connection. The settings can be MAC 
				and/or IP coming from the original connection. */
} SChangeConnInfo;


/** Type of Loopback on channels*/
typedef enum {
	eTDM = 1,					/**< TDM loopback (no processing)*/
	ePRE_ENCAPS = 2,				/**< Pre encapsulation loopback */
	ePOST_ENCAPS = 3,				/**< Post encapsulation loopback */
	eRTP_DEC = 4,					/**< RTP Decoder loopback */
	eINTER_CHNL_POST_ENCAPS = 5,			/**< Inter channel post encapsulation loopback */
	eINTER_CHNL_TDM = 6,				/**< Mindspeed Reserved */
	eTHC = 7,					/**< TDM Hairpin loopback */
	eREMOVE_CHNL_LOOPBACK = 8,			/**< Stop Loopback on single channel */
	eREMOVE_INTER_CHNL_POSTENCAPS_LOOPBACK = 9,	/**< Stop inter channel post encapsulation loopback */
	eREMOVE_INTER_CHNL_TDM_LOOPBACK = 10,		/**< Stop inter channel TDM loopback */
	eREMOVE_THC_LOOPBACK = 11			/**< Stop hairpin loopback */
} ELoopbackType;

/** Shutdown modes used in VAPI_CloseDevice \n
Currently only ePURGE is supported.*/

typedef enum {
	ePURGE, 	/**< In ePURGE mode the VAPI frees memory allocated to all channels
			and frees memory allocated for all pending requests.
			That is, the application must destroy all created connections using the VAPI_DestroyConnection API.*/
	ePURGE_ALL	/**< The same as ePURGE but it frees all MSP resources as well: 
			disable any created channel (including conference participants and transcoding legs), 
			disable any transcoding session, destroy  any created channel, destroy any created conference*/
} ECloseMode;

/** Payload type direction (asymmetrical option)*/
typedef enum {
	eTx,		/**< Transmit direction */
	eRx,		/**< Receive direction */
	eBoth		/**< Both direction */
} EDirection;

/** Transcoding session options */
typedef struct _STranscodingOption {
	Boolean bAction;		/**< Enable, Disable transcoding */
	U8 ucDSPMode;			/**< DSP Bypass mode */
	U8 ucVirtualTranscoding;	/**< Virtual transcoding latency */
} STranscodingOption;

/** Enhanced conferencing feature flags structure */
typedef struct _SConfParams {
	Boolean bTranscoding;	/**< Transcoding Session: 1-Enable, 0-Disable */
	Boolean bBlockDTMF;	/**< DTMF Tone Block Feature: 1-Enable, 0-Disable */
	Boolean bMixerType;	/**< Type of Conference Mixer: 0 = Narrowband (default), 1 = Wideband */
} SConfParams;

/** Enhanced conferencing dominant talkers parameters structure */
typedef struct _SConfDTSParams {
	Boolean bDTSOnOff;	/**< Dominant talkers enabled (default Off) */
	U16 usAttackConst;	/**< Attack constant (ms) */
	U16 usNTH1;		/**< Noise Threshold 1 (eights of dB) */
	U16 usNTH2;		/**< Noise Threshold 2 (eights of dB) */
	U16 usHoldConst;	/**< Hold constant (ms)  */
	U16 usN1;		/**< Number of participant for group 1*/
	U16 usG1;		/**< Attenuation for group 1 (tenths of dB)*/
	U16 usN2;		/**< Number of participant for group 2*/
	U16 usG2;		/**< Attenuation for group 2 (tenths of dB)*/
	U16 usN3;		/**< Number of participant for group 3*/
	U16 usG3;		/**< Attenuation for group 3 (tenths of dB)*/
} SConfDTSParams;

/** Bits specifying which parameters are set in SHwCfg structure passed to public VAPI functions */
#define VAPI_HWCFG_SDRAM_TIMING_MASK	0x01	/** Both usSdramParamTiming1 and usSdramParamTiming2 are set */
#define VAPI_HWCFG_SDRAM_REFRESH_MASK	0x02	/** usSdramRefresh is set */
#define VAPI_HWCFG_SDRAM_POWERON_MASK	0x04	/** usSdramPoweron is set */
#define VAPI_HWCFG_CS_TIMING_MASK	0x08	/** usCSparamTiming is set */
#define VAPI_HWCFG_REFCLK_MASK		0x10	/** usDeviceRefclk is set */
#define VAPI_HWCFG_SDRAM_PARAMS	0x20	/** uiSdramParam_C300_2 is set */
#define VAPI_HWCFG_M823V2_HIGH_SPEED_MASK	0x40	/** usDeviceAmbaClkM823V2 is set */

/** User data structure for containing Device HW params, please check the M82xxx-SWG-004 documentation for details regarding
various SDRAM and CS parameters */
typedef struct _SHwCfg {
	U16 usSdramParamTiming1;	/**< RAM timings 1 */
	U16 usSdramParamTiming2;	/**< RAM timings 2 */
	U16 usCSparamTiming;		/**< CS timings */
	U16 usSdramRefresh;		/**< RAM refresh period in bus clock cycles */
	U16 usSdramPoweron;		/**< RAM poweron wait in bus clock cycles */
	U16 usDeviceRefclk;		/**< Device reference clock frequency. This parameter is used to define which \n
					dividers should be set for PLL clock. For example C300 EVM may use 10MHz and 25MHz ref. clock */
	U16 usHwParamsMask;		/**< Bits specifying which fields are set in structure, defaults will be used for the other fields. \n
					In order to keep backward compatibility zero value in this field means that all fields are set. */
	U32 *ptrSdramParam_C300_2;
					/**< SDRAM controller initialization sequence, contains the sequence of
					address and value of registers to initialize, used for for M823XX_2 */
	U16 usSdramParamNum;		/**< SDRAM parameter number, actually it's a number of address and value
					sets, used for M823XX_2 */
	U16 usDeviceAmbaClkM823V2;	/**< Device AMBA clock frequency for M823V2 . This parameter is used to define which \n
					dividers should be set for PLL clock. For example C300V2 EVM may use 133MHz and 150MHz AMBA clock */
} SHwCfg; 

/** EVENT structures */
typedef struct _SEventDetParams {
	CONNID ConnId;
	U16 usEventReport;
	U32 uTimestamp;
} SEventDetParams;

/** SPI Event parameters */
typedef struct _SSpiEventParams {
	DEVID DevId;		/**< Device ID */
	U32 uTimestamp;		/**< TimeStamp of the event */
} SSpiEventParams;

/** Tone Detection Event Parameters */
typedef struct _SToneDetectEventParams {
	CONNID ConId;		/**< Connection ID */

	U16 usDetectedTone;
				/**< 0x00: DTMF digit 0 start detected \n
				0x01: DTMF digit 1 start detected \n
				0x02:  DTMF digit 2 start detected \n
				0x03:  DTMF digit 3 start detected \n
				0x04:  DTMF digit 4 start detected \n
				0x05:  DTMF digit 5 start detected \n
				0x06:  DTMF digit 6 start detected \n
				0x07:  DTMF digit 7 start detected \n
				0x08:  DTMF digit 8 start detected \n
				0x09:  DTMF digit 9 start detected \n
				0x0A:  DTMF digit * start detected \n
				0x0B:  DTMF digit # start detected \n
				0x0C:  DTMF digit A start detected \n
				0x0D:  DTMF digit B start detected \n
				0x0E:  DTMF digit C start detected \n
				0x0F:  DTMF digit D start detected \n
				0x10:  V.25 answer tone w/o phase reversals detected \n
				0x11:  V.25 answer tone with phase reversals \n
				0x20:  T.30 CNG start detected (fax calling tone) \n
				0x22:  V.21 flags detected (fax without answer tone) \n
				0x30:  Call Progress Tone detected \n
				0x31:  Inactivity Timeout detected \n
				0x41:  Fax CNG tone with correct cadence detected \n
				0x42:  End of fax CNG sequence \n
				0x43:  Special Information Tone / No Circuit Tone detected \n
				0x50:  SS7/COT tone detected \n
				0x70:  EARLY_CAS_DET \n	
				0x71:  CAS_DET  \n
				0xFF:  End of tone detected */
	U16 usPpowerLevel;
				/**< a value between 0 and 46 dBov in units of 0.1. \n
				1 dBov, giving a parameter range of 0 to 460. */
	U32 uTimestamp;
				/**< Timestamp units increment at the PCM sampling rate of 8 kHz */
} SToneDetectEventParams;

/** IP Tone Detection Event Parameters */
typedef struct _SIPToneDetectEventParams {
	CONNID ConId;		/**< Connection ID */

	U16 usDetectedTone;
				/**< 0x0001 Early Answer tone \n
				0x0003 Confirmed Answer tone \n
				0x0007 2100Hz Answer tone (ANSAM or /ANSAM) \n
				0x000B Phase reversals present in the /ANS Answer tone \n
				0x000F Phase reversals present in the /ANSAM answer tone \n
				0x0010 Bell ANS (2225Hz) or V.22 USB1 (2250Hz) \n
				0x0020 1800Hz Pre-Standard V.32 AA tone \n
				0x0040 IP-side Programmable Tone Detector \n
				0x0080 V.18 Annex A Tone \n
				0x0100 Generic Tone 390Hz \n
				0x0200 Generic Tone 980Hz \n
				0x0400 Generic Tone 1270Hz \n
				0x0800 Generic Tone 1300Hz \n
				0x1000 Generic Tone 1650Hz \n
				0x8000 Generic Tone 1270Hz \n
				0x00FF End of tone */

	U16 usPpowerLevel;	/**<    Comcerto software sends VoIP_IPTONEDET Parameter 3 only for PTD tones.
				The Power Level, range is 0 to -46 dBov in units of -0.1 dBov, giving a parameter range of 0 to 460 (decimal).*/
	U32 uTimestamp;
				/**< Timestamp units increment at the PCM sampling rate of 8 kHz */
} SIPToneDetectEventParams;

/** Tone Genration Complete Event Parameters */
typedef struct _SToneGenCmpltEventParams {
	CONNID ConId;		/**< Connection ID */
	U16 usToneTerminateCause;
				/**< 0x00  Normal termination \n
				0x01: Timestamp has already passed limit value \n
				0x02: Timestamp has not yet reached start value \n
				0x03: Termination due to TONEOFF command \n
				0x04: Termination due to null TONEGEN command \n
				0x05: Termination of active tone by new tone generate request \n
				0x06: Cancellation of waiting tone by new tone generate request \n
				0x07: Termination of protected tone (sent after 50ms off time) \n
				0x08: Protected buffer full  tone dropped \n
				0x09: Stop tone received for buffered unprotected tone  \n
				tone dropped \n
				0x0A: Unprotected tone dropped during protected sequence \n
				0x0B: Protected tone delay more than 5 seconds  tone dropped */
} SToneGenCmpltEventParams;

/** PayLoad Type Change event Parameters */
typedef struct _SPtChangeEventParams {
	CONNID ConId;	/**< Connection ID */
	U32 uNewPT;	/**< New PayLoad Type */
} SPtChangeEventParams;

/** SSRC change Event Parameters */
typedef struct _SSsrcChangeEventParams {
	CONNID ConId;		/**< Connection ID */
	U32 uNewSSRC;		/**< New SSRC */
} SSsrcChangeEventParams;

/** SSRC Voilation Event Parameters */
typedef struct _SSsrcViolationEventParams {
	CONNID ConId;		/**< Connection ID */
	U32 uDetectedSSRC;	/**< Detected SSRC */
} SSsrcViolationEventParams;

typedef struct _SNteTransmitEventParams {
	CONNID ConId;		/**< Connection ID */
} SNteTransmitEventParams;

/** NTE Received Event Parameters */
typedef struct _SNteRecvdEventParams {
	CONNID ConId;			/**< Connection ID */
	U32 uNtePayload;		/**< NTE PayLoad */
	U16 usPayloadTypeOverride;	/**< PayLoad Type Override */
	U32 uSSRC;			/**< SSRC */
	U16 usPayloadType;		/**< PayLoad Type */
} SNteRecvdEventParams;

/** Caller ID Complete Event Parameters */
typedef struct _SCallerIdCmpltEventParams {
	CONNID ConId;		/**< Connection ID */
	U16 usCmpltCause;
				/**< 0: Normal termination of programmed sequence \n
				1: Termination due to callerID stop request \n
				2: Termination due to failure of receiving  \n
				of ACK from CPE in response to CNDOffHookGen \n*/
} SCallerIdCmpltEventParams;

/**G711 Concurrent Decode Event parameters */
typedef struct _SG711ConcurntDecoderEventParams {
	CONNID ConId;		/**< Connection ID */
	U16 usG711EventReport;
				/**< 0x40: Auto-switch to G.711 performed \n
				0x44: Entered G.711 u-law concurrent decode mode \n
				0x45: Entered G.711 A-law concurrent decode mode */
	U32 uTimestamp;		/**< TimeStamp of the event */
} SG711ConcurntDecoderEventParams;

/** Passthrough Concurrent Decoder Event Parameters */
typedef struct _SPassthruConcurntDecoderEventParams {
	CONNID ConId;		/**< Connection ID */
	U16 usPassthruEventReport;
				/**< 0x46: Auto-switch to Pass through performed \n
				0x47: Entered Pass through u-law concurrent decode mode \n
				0x48: Entered Pass through A-law concurrent decode mode */
	U32 uTimestamp; 	/**< TimeStamp of the event */
} SPassthruConcurntDecoderEventParams;

/** Decoder Event Parameters */
typedef struct _SDecoderEventParams {
	CONNID ConId;		/**< Connection ID */
	U16 usEventReport;	/**< Detected Event */
	U32 uTimestamp;		/**< TimeStamp of the event */
} SDecoderEventParams;

/** Tone Detect Event Parameters */
typedef struct _SRemoteDetectEventParams {
	CONNID ConId;		/**< Connection ID */

	U8 ucDetectedEvent;
				/**< 0x00 UDPTL Detection or Unknown Event \n
				0x01 - 0x16: DTMF Digits 0 to D \n
				0x17 - 0x1F: Reserved \n
				0x20 : ANS (RFC 2833 Event 32) \n
				0x21 : /ANS (RFC 2833 Event 33) \n
				0x22 : ANSam (RFC 2833 Event 34) \n
				0x23: /ANSam (RFC 2833 Event 35) \n
				0x24: CNG (RFC 2833 Event 36) \n
				0x25-0xFF Reserved */
	U8 ucRemotePktType;
				/**< 0x00 reserved \n
				0x01  reserved \n
				0x02  unknown UDP packet \n
				0x03  RFC2833 IETF packet \n
				0x04  RFC2833 AAL packet \n
				0x05  UDPTL packet Note for UDPTL packet, the Event field is 0. \n
				0x06-0xFF Reserved */
	U16 usRemotePktPayload;	
				/**< Optional param. Will be set to zero if not received */
} SRemoteDetectEventParams;

/** Caller Id Detection Event*/ 
typedef struct _SCIDReceivedMSG {

	CONNID ConId;		/**< Connection ID */
	U16 usMSGStatus;	/**< Message Status Field */
	U8 ucMSGByteCnt;	/**< Message Byte Count */
	U8 *pucMsg;		/**< Pointer to the actual Message */
} SCIDReceivedMSG;

/** Switchover events*/
typedef enum {
	eLOCAL_CNG = 0x01,	/**< Local Calling tone */
	eREM_CNG = 0x02,	/**< Remote Calling tone */
	eLOCAL_ANS = 0x04,	/**< Local Answer tone */
	eREM_ANS = 0x08,	/**< Remote Answer tone */
	eLOCAL_V21 = 0x10,	/**< Local V21 flags */
	eREM_UDPTL = 0x20	/**< Remote UDPTL */
} EFaxSwitchEvents;


/** Fax Switch Event parameters */
typedef struct _SFaxSwEventParams {
	CONNID ConId;			/**< Connection ID */
	VSTATUS status;			/**< Result of switchover  SUCCESS/ error*/
	EFaxSwitchEvents eSwitchEvent;	/**< Event on which switchover initiated */
} SFaxSwEventParams;

/** UNDEFINED Event Parameters structure */
typedef struct _SUndefinedEventParams {
	U8 ucLevel; 		/**< 0: channel level message \n
				1: device level message */
	union {
		CONNID ConId;	/**< If level = 0  ID will be CONND */
		DEVID DevId;	/**< If level = 1  ID will be DEVID  */
	} ID;
	U8 ucLength;		/**< Comcerto indication payload length. \n
				It includes the payload lenght the API header (8 bytes)  */
	U8 ucIndex;		/**< Comcerto command Index */
	U8 ucCmdClass;		/**< Comcerto command Class */
	U8 ucCmdType;		/**< Comcerto command Type */
	U16 usFnCode;		/**< Comcerto command Function Code */
	U16 usNoOfParams;	/**< Number of parameters in ausParams buffer. \n
				It includes the 0x0000 reserved word*/
	U16 ausParams[DEFAULT_FIFO_MAX_SIZE/2];	/**< Fifo parameter buffer.\n
						\b Note: the 1st U16 in the buffer is the reserved 0x0000 field \n
						The first U16 field of the indication payload is ausParams[1]*/
} SUndefinedEventParams;

/** Country Code */
typedef enum {
	eUSA = 0,	/**< USA */
	eINDIA = 1,	/**< INDIA */
	eMAXCOUNTRY		/* Dummy country code which should be always at last. */
} ECountryCode;

/**Caller Id structure*/
typedef struct _CidInfo {
	U16 MsgLen;		/**<Caller ID Messsage length */
	U8 *pucMsgStr;		/**<Caller ID Messsage String in SDMF or MDMF format \n */
	IN Boolean bIsOnHook;	/**<OnHook / OffHook status of the channel.\n
				If it is TRUE, then ONHOOK caller Id will be generated,\n
				otherwise OFFHOOK caller Id will be generated*/
	IN ECountryCode eCountryCode;	/**<Country Code*/
} CidInfo; 

/**Caller Id structure vor VAPI_PlayCid()*/
typedef struct _SCallerIdInfo {
	U16 MsgLen;		/**<Caller ID Messsage length*/
	U8 *pucMsgStr;		/**<Caller ID Messsage String in SDMF or MDMF format*/
	IN Boolean bIsOnHook;	/**<OnHook / OffHook status of the channel.\n
				If it is TRUE, then ONHOOK caller Id will be generated,\n
				otherwise OFFHOOK caller Id will be generated */
	U8 ucModulType;		/**< 0x00 = BFSK Bel202 standard, 0x01 = CCITT V.23 standard. \n
				Used for onhook caller ID only */
} SCallerIdInfo; 


/** CID info block action */
typedef enum
{
	eRemove = 0,	/**< 0 Remove CID info block */
	eAdd = 1,	/**< 1 Add CID info block */
	eModify = 2,	/**< 2 Modify CID info block */
} EAction;


/** CID generation info block type */
typedef enum
{
	eOnHookXmitInfoType = 0x01,		/**< 0x01 On hook block type*/
	eOffHookXmitInfoType = 0x02,		/**< 0x02 Off hook block type*/
	eCPEAlertToneInfoType = 0x03,		/**< 0x03 CPE alert block type*/
	eCPEAckInfoType = 0x04,			/**< 0x04 CPE ack block type*/
	eSASToneInfoType = 0x05,		/**< 0x05 SAS tone block type*/
	eNTTInfoType = 0x06,			/**< 0x06 NTT block type*/
	eStopBitsInfoType = 0x07,		/**< 0x07 Stop bits block type*/
	ePacketProcessingInfoType = 0x08	/**< 0x08 Packet processing block type*/
} ECidGenInfoType;

/** CID detection info block type */
typedef enum
{
	eDataFormatType = 0x01,		/**< 0x01 Data format block type*/
	eFSKDetectionType = 0x02,	/**< 0x02 FSK detection block type*/
	eOnHookDetType = 0x03,		/**< 0x03 On hook detection block type*/
	eOffHookDetType = 0x04,		/**< 0x04 Off hook detection block type*/
	eAckGenType = 0x05,		/**< 0x05 Ack generation block type*/
	eNTTDetType = 0x06, 		/**< 0x06 NTT detection block type*/
} ECidDetInfoType;

/** CID detection control  */
typedef struct _SCidDetCtrl
{
	U16 usCidEnable:1; 	/**< 1:enable, 0:disable*/
	U16 usHookStatus:2;	/**< 0:on hook, 1:off hook, 2: both*/
} SCidDetCtrl;


# define CFG_MAX_TONES  40
/** Tone Generation Identifier \n
See astDtmfToneGenParams_g structure for DTMF tones \n
See astCountryInfoList_g for other tones  */
typedef enum {
	eDTMFTONE_1 = 0,	/**< DTMF 1 */
	eDTMFTONE_2 = 1,	/**< DTMF 2 */
	eDTMFTONE_3 = 2,	/**< DTMF 3 */
	eDTMFTONE_4 = 3,	/**< DTMF 4 */
	eDTMFTONE_5 = 4,	/**< DTMF 5 */
	eDTMFTONE_6 = 5,	/**< DTMF 6 */
	eDTMFTONE_7 = 6,	/**< DTMF 7 */
	eDTMFTONE_8 = 7,	/**< DTMF 8 */
	eDTMFTONE_9 = 8,	/**< DTMF 9 */
	eDTMFTONE_A = 9,	/**< DTMF A */
	eDTMFTONE_B = 10,	/**< DTMF B */
	eDTMFTONE_C = 11,	/**< DTMF C */
	eDTMFTONE_D = 12,	/**< DTMF D */
	eDTMFTONE_STAR = 13,	/**< DTMF * */
	eDTMFTONE_0 = 14,	/**< DTMF 0 */
	eDTMFTONE_HASH = 15,	/**< DTMF # */
	eNULLTONE = 16,		/**< NULL tone (no tone) */
	eCONGESTIONTONE = 17,	/**< Congestion tone */
	eWARNINGTONE = 18,	/**< Warning tone */
	eWAITINGTONE = 19,	/**< Waiting tone */
	eCONFIRMATIONTONE = 20,	/**< Confirmation tone */
	eDIALTONE = 21,		/**< Dialtone tone */
	eBUSYTONE = 22,		/**< Busy tone */
	eRINGBACKTONE = 23,	/**< Ringback tone */
	eSPECIALTONE = 24,	/**< Special tone */
	eSASTONE = 25,		/**< SAS tone */
	eCUSTOM_1 = 26,		/**< Custom tone 1 */
	eCUSTOM_2 = 27,		/**< Custom tone 2 */
	eCUSTOM_3 = 28,		/**< Custom tone 3 */
	eCUSTOM_4 = 29,		/**< Custom tone 4 */
	eCUSTOM_5 = 30,		/**< Custom tone 5 */
	eCUSTOM_6 = 31,		/**< Custom tone 6 */
	eCUSTOM_7 = 32,		/**< Custom tone 7 */
	eCUSTOM_8 = 33,		/**< Custom tone 8 */
	eCUSTOM_9 = 34,		/**< Custom tone 9 */
	eCUSTOM_10 = 35,	/**< Custom tone 10 */
	eCUSTOM_11 = 36,	/**< Custom tone 11 */
	eCUSTOM_12 = 37,	/**< Custom tone 12 */
	eCUSTOM_13 = 38,	/**< Custom tone 13 */
	eCUSTOM_14 = 39,	/**< Custom tone 14 */
	eCUSTOM_15 = 40,	/**< Custom tone 15 */
	eCUSTOM_16 = 41,	/**< Custom tone 16 */
	eCUSTOM_17 = 42,	/**< Custom tone 17 */
	eCUSTOM_18 = 43,	/**< Custom tone 18 */
	eCUSTOM_19 = 44,	/**< Custom tone 19 */
	eCUSTOM_20 = 45,	/**< Custom tone 20 */
	eCUSTOM_21 = 46,	/**< Custom tone 21 */
	eCUSTOM_22 = 47,	/**< Custom tone 22 */
	eCUSTOM_23 = 48,	/**< Custom tone 23 */
	eCUSTOM_24 = 49,	/**< Custom tone 24 */
	eCUSTOM_25 = 50,	/**< Custom tone 25 */
	eCUSTOM_26 = 51,	/**< Custom tone 26 */
	eCUSTOM_27 = 52,	/**< Custom tone 27 */
	eCUSTOM_28 = 53,	/**< Custom tone 28 */
	eCUSTOM_29 = 54,	/**< Custom tone 29 */
	eCUSTOM_30 = 55,	/**< Custom tone 30 */
	eDUMMYTONE = eDTMFTONE_HASH + CFG_MAX_TONES + 1	/* Dummy Tone which should be always at last */
} EToneId;

/**Tonde direction*/
typedef enum {
	eDirToTDM = 0,
	eDirToIP =1,
	eDirToBoth = 2
} EToneDir;

/**Tone type*/
typedef enum {
	eDualTone = 0,
	eQuadTone = 1
} EToneType;

/**Play tone conrol options*/
typedef struct _SPlayToneOpt
{
        U16 usMixModeOff:1;
        U16 usMixModeOn:1;
        U16 usProtGen:1;
        U16 usTimestampCtrl:2;
} SPlayToneOpt;

/**Codec Type*/
typedef enum {
	eGeneric_Comfort_Noise = 0,	/**< 0: Comfort noise */
	eRedundant_Scheme_for_DTMF = 1,	/**< 1: DTMF redundant scheme*/
	eDTMF_RTP = 2,			/**< 2: DTMF in RTP*/
	eIFP = 3,			/**< 3: Internet Fax Protocol*/
	eG711_ULAW_PCM = 4,		/**< 4: g711 uLaw*/
	eG711_ALAW_PCM = 5,		/**< 5: g711 ALaw*/
	eG723_1 = 6,			/**< 6: g723*/
	eG728_Audio = 7,		/**< 7: g728 audio*/
	eG729_A = 8,			/**< 8: g729 A*/
	eG726_32 = 9,			/**< 9: g726 32*/
	eReserved_Codec0 = 10,		/**< 10: Reserved Codec 0*/
	eGSM_FR_Audio = 11,		/**< 11: GSM FR*/
	eClear_Channel = 12,		/**< 12: Clear Channel*/
	ePassthrough_ULAW = 13,		/**< 13: PassThrough uLaw*/
	ePassthrough_ALAW = 14,		/**< 14: PassThrough uAaw*/
	eG726_16_audio = 15,		/**< 15: g726 16*/
	eG726_24_audio = 16,		/**< 16: g726 24*/
	eG726_40_audio = 17,		/**< 17: g726 40*/
	eGSM_EFR = 18,			/**< 18: GSM EFR*/
	eAMR = 19,			/**< 19: AMR*/
	eEVRC = 20,			/**< 20: EVRC*/
	eCNF_SEC = 21,			/**< 21: Secondary CNF*/
	eSMV = 22,			/**< 22: SMV*/
	eQCELP_13K = 23,		/**< 23: QCELP 13K*/
	eQCELP_8K = 24,			/**< 24: QCELP 8K*/
	eCRBT_ADPCM = 25,		/**< 25: CRBT ADPCM*/
	eG729_EG = 26,			/**< 26: g729 EG*/
	eFEC = 27,			/**< 27: Forward Error Correction*/
	eI_LBC_15K = 28,		/**< 28: I-LBC 15K*/
	eI_LBC_13K = 29,		/**< 29: I-LBC 13K*/
	eRFC2833_EVENT = 30,		/**< 30: RFC2833 Event*/
	eGSM_AMR_BWE = 31,		/**< 31: GSM_AMR_BWE*/
	e4GV = 32,			/**< 32: 4GV*/
	eRFC2833_TONE = 33,		/**< 33: RFC2833 Tone*/
	eAMR_WB = 34,			/**< 34: AMR_WB*/
	eG729_MSPD = 35,		/**< 35: G.729 MSPD*/
	eCESoPSN = 36,			/**< 36: CESoPSN*/
	eGSM_HR = 37,			/**< 37: eGSM_HR*/
	eCSD_BS30T_64k = 38,		/**< 38: CSD_BS30T_64k*/
	eCSD_BS30T_32k = 39,		/**< 39: CSD_BS30T_32k*/
	eCSD_BS30T_33k6 = 40,		/**< 40: CSD_BS30T_33k6*/
	eCSD_BS30T_28k8 = 41,		/**< 41: CSD_BS30T_28k8*/
	eCSD_BS20NT_57k6 = 42,		/**< 42: CSD_BS20NT_57k6*/
	eCSD_BS20NT_28k8 = 43,		/**< 43: CSD_BS20NT_28k8*/
	eCSD_BS20NT_14k4 = 44,		/**< 44: CSD_BS20NT_14k4*/
	eG729_1 = 45,			/**< 45: G.729.1*/
	eAMR_WB_ba = 46,		/**< 46: AMR WB(G.722.2)[byte aligned]*/
	eCCISoIP = 47,			/**< 47: CCISoIP*/
	eG722 = 48,			/**< 48: G.722:*/
	eReserved_Codec1 = 49,		/**< 49: Reserved Codec 1*/
	eReserved_Codec2 = 50,		/**< 50: Reserved Codec 2*/
	eReserved_Codec3 = 51,		/**< 51: Reserved Codec 3*/
	eAMR_IF1 = 52,			/**< 52: AMR IF1*/
	eAMR_IF2 = 53,			/**< 53: AMR IF2*/
	eReserved_Codec4 = 54,		/**< 54: Reserved Codec 4*/
	eMaxCodecVal,
	/*video codecs*/
} ECodecType;
#define NUM_CODECS eMaxCodecVal

/**Codec Index*/
typedef enum {
	eG711_ULAW_ID = 4,		/**< 4: g711 uLaw*/
	eG711_ALAW_ID = 5,		/**< 5: g711 ALaw*/
	eG723_1_ID = 6,			/**< 6: g723*/
	eG728_Audio_ID = 7,		/**< 7: g728 audio*/
	eG729_A_ID = 8,			/**< 8: g729 A*/
	eG726_32_ID = 9,		/**< 9: g726 32*/
	eGSM_FR_Audio_ID = 11,		/**< 11: GSM FR*/
	eClear_Channel_ID = 12,		/**< 12: Clear Channel*/
	ePassthrough_ULAW_ID = 13,	/**< 13: PassThrough uLaw*/
	ePassthrough_ALAW_ID = 14,	/**< 14: PassThrough uAaw*/
	eG726_16_audio_ID = 15,		/**< 15: g726 16*/
	eG726_24_audio_ID = 16,		/**< 16: g726 24*/
	eG726_40_audio_ID = 17,		/**< 17: g726 40*/
	eGSM_EFR_ID = 18,		/**< 18: GSM EFR*/
	eAMR_ID = 19,			/**< 19: AMR*/
	eEVRC_ID = 20,			/**< 20: EVRC*/
	eSMV_ID = 22,			/**< 22: SMV*/
	eQCELP_13K_ID = 23,		/**< 23: QCELP 13K*/
	eQCELP_8K_ID = 24,		/**< 24: QCELP 8K*/
	eCRBT_ADPCM_ID = 25,		/**< 25: CRBT ADPCM*/
	eG729_EG_ID = 26,		/**< 26: g729 EG*/
	eI_LBC_15K_ID = 28,		/**< 28: I-LBC 15K*/
	eI_LBC_13K_ID = 29,		/**< 29: I-LBC 13K*/
	GSM_AMR_BWE_ID = 31,		/**< 31: GSM_AMR_BWE*/
	e4GV_ID = 32,			/**< 32: 4GV*/
	eAMR_WB_ID = 34,		/**< 34: AMR_WB*/
	eG729_MSPD_ID = 35,		/**< 35: G.729 MSPD*/
	eCESoPSN_ID = 36,		/**< 36: CESoPSN*/
	eGSM_HR_ID = 37,		/**< 37: eGSM_HR*/
	eCSD_BS30T_64k_ID = 38,		/**< 38: CSD_BS30T_64k*/
	eCSD_BS30T_32k_ID = 39,		/**< 39: CSD_BS30T_32k*/
	eCSD_BS30T_33k6_ID = 40,	/**< 40: CSD_BS30T_33k6*/
	eCSD_BS30T_28k8_ID = 41,	/**< 41: CSD_BS30T_28k8*/
	eCSD_BS20NT_57k6_ID = 42,	/**< 42: CSD_BS20NT_57k6*/
	eCSD_BS20NT_28k8_ID = 43,	/**< 43: CSD_BS20NT_28k8*/
	eCSD_BS20NT_14k4_ID = 44,	/**< 44: CSD_BS20NT_14k4*/
	eG729_1_ID = 45,		/**< 45: G.729.1*/
	eAMR_WB_ba_ID = 46,		/**< 46: AMR WB(G.722.2)[byte aligned]*/
	eCCISoIP_ID = 47,		/**< 47: CCISoIP*/
	eG722_ID = 48,			/**< 48: G.722*/
	eReserved_Codec1_ID = 49,	/**< 49: Reserved Codec 1*/
	eReserved_Codec2_ID = 50,	/**< 50: Reserved Codec 2*/
	eReserved_Codec3_ID = 51,	/**< 51: Reserved Codec 3*/
	eAMR_IF1_ID = 52,		/**< 52: AMR IF1*/
	eAMR_IF2_ID = 53		/**< 53: AMR IF2*/
} ECodecIndex;

/**Video Codec Index*/
typedef enum {
	eH263_ID = 0,		/**< 0: H.263 (RFC2429/RFC4629)*/
	eRFC2190_ID = 1,	/**< 1: H.263 (RFC2190)*/
	eH264_ID = 2,		/**< 2: H.264*/
	eMPEG4_ID = 3,		/**< 3: MPEG-4*/
	eMaxVideoCodecVal
} EVideoCodecIndex;
#define NUM_VIDEO_CODECS eMaxVideoCodecVal

/** CESOPSN mapping type*/
typedef enum {
	eChannelMap = 0,	/**< Channels are mapped (usage of CESoPSN_MAP_CHANNEL command)*/
	eTSMap = 1		/**< Timeslots are mapped (usage of CESoPSN_SPEC_MAP_CHANNEL command) */
}EMapperType;

/** TDM bus parameters */
typedef struct _STdmBusParams {
	U8 ucBusId;
			/**< Range: 0 only for 1-bus-mode */
			/**< 0-1 for 2-bus-mode */
			/**< 0-3 for 4-bus-mode */
	U8 ucTxOrder;
			/**< 0: LSB is transmitted first */
			/**< 1: MSB is transmitted first */
	U8 ucRxPolarity;
			/**< 0: Data is not inverted (active-high) */
			/**< 1: Data is inverted (active-low) */
	U8 ucTxPolarity;
			/**< 0: Data is not inverted (active-high) */
			/**< 1: Data is inverted (active-low) */
	U8 ucRxClkMode;
			/**< 0: One data bit is received per TDM_CK period */
			/**< 1: One data bit is received every two TDM_CK periods */
	U8 ucTxClkMode;
			/**< 0: One data bit is received per TDM_CK period */
			/**< 1: One data bit is received every two TDM_CK periods */
	U8 ucRxClkEdge;
			/**< 0: Data is received on the falling edge of TDM_CK */
			/**< 1: Data is received on the rising edge of TDM_CK */
	U8 ucTxClkEdge;
			/**< 0: Data is received on the falling edge of TDM_CK */
			/**< 1: Data is received on the rising edge of TDM_CK */
	U8 ucFrameEdge;
			/**< 0: MFS is sampled on falling edge of TDM_CK */
			/**< 1: MFS is sampled on rising edge of TDM_CK */
	U16 usNumOfTS;
			/**< # time slots between 2 consecutive TDM_FS  */
	U8 ucInvertedFrmSig;
			/**< 0: TDM_FS is not inverted (active-high) */
			/**< 1: TDM_FS is inverted (active-low) */
	U8 ucBitOrderRcv;
			/**< 0: LSB is received first */
			/**< 1: MSB is received first */
	U16 usNumTdmClkCyclesDelayTx;
			/**< Number of Cycles Delay, Transmitter */
	U16 usNumTdmClkCyclesDelayRx;
			/**< Number of Cycles Delay, Receiver */
} STdmBusParams;

#define MAX_TDM_BUS 4

/** TDM buses information structure */
typedef struct _STdmSetupParams {
	U16 usMode;
	    /**< 1 : SINGLE_BUS_MODE */
	    /**< 2 : TWO_BUS_MODE */
	    /**< 4 : FOUR_BUS_MODE */
	U8 usNoOfBus;
	    /**< Number of buses */
	STdmBusParams astTdmBusParam[MAX_TDM_BUS];
	    /**< TDM bus parameters array */
} STdmSetupParams;

/** IP address information structure */
typedef struct _SIpAddrInfo {
	U8 ucNumOfSrcIpAddr;	/**< Number of source device IP addr */
	Boolean bIsMultipleMode;/**< False: Single IP Mode*/
				/**< True: Multiple IP Mode*/
	U32 auiDevIPAddress[MAX_SRC_IPADDR];
				/**< Array of source device IP addresses */
	U8 ucChkSumVerification;
				/**< 0: IP checksum on received packets enabled */
				/**< 1: IP checksum on received packets disabled */
	U8 ucEncapsulation;
				/**< 0: VoIP packets are passed to the PCI interface */
				/**< 2: VoIP packets are passed to the Ethernet Layer */
} SIpAddrInfo;

/** FAX options information structure */
typedef struct _SFaxOpts {
	U8 ucT38PktLossConcealment;
				/**<0 = No T.38 packet loss concealment (default) \n
				1 = Packet loss concealment enabled, bad lines in page replaced by white lines \n
				2 = Packet loss concealment enabled, bad lines in page replaced by last good line */
	Boolean ucECMDisable;
				/**<0 = Allow ECM faxes in T.38 (default) \n
				1 = Prevent ECM faxes in T.38 */
	U8 ucFaxDataRedundancyCnt;
				/**<0 - 3 ; Default- 0 */
	U8 ucT30RedundancyCnt;
				/**<0 - 7 ; Default- 3 */
	U8 ucFaxConnSpeedLimit;
				/**<0 = No speed limit (default) \n
				1 = 2400 bps limit \n
				2 = 4800 bps limit \n
				3 = 7200 bps limit \n
				4 = 9600 bps limit \n
				5 = 12000 bps limit \n
				6 = 14400 bps limit */
	U8 ucErrRcvyMethod;
				/**<0 = Redundancy (default) \n
				1 = FEC */
	U8 ucTCFProcedure;
				/**<0 = Procedure 2
				1 = Procedure 1 (default) */
	U8 ucNumFEC;
				/**<Number of FEC packets to send in UDPTL Range 0 (default) to 3 */
	U8 ucNumIFPs;
				/**<Number of IFP used to generate a FEC packet Range 0 (default) to 4 */

	U16 usParam2;		/**< Parameter 2 of fax options */
} SFaxOpts;

/** FAX configuration information structure */
typedef struct _SFaxConfigOpts {
	Boolean bUseExistingOpts;	/**< Use existing options. \n
					   If True, Will not modify FAXOPT & FAXLVL settings
					   That is, stFaxOpts and ucFaxTxLvl 
					   of this struct will not be used and previous
					   settings will be used */
	SFaxOpts stFaxOpts;
					/**< FAX options structure */
	U8 ucFaxTxLevel;
					/**< 0 - 15, in -1dBm steps; Default - 0x0A; */

	Boolean bEnableFaxAutoSwitch;
					/**< Enable VAPI to automatically switch to FOIP mode*/
	U16 usSwitchoverEventMask;
					/**<Bit map to identify events on which switchover is to be performed, the masks are defined in EFaxSwitchEvents */
} SFaxConfigOpts;

/** ALERT Indication Structure */
typedef struct _SAlrtIndParams {
	CONNID ConId;					/**< Id of the connection reporting the alert \n
							- 0xnnnn: Connection ID \n
							- 0xFFFE: ALERT_UNKNOWN_CHANNEL \n
							- 0XFFFF: SUPERVISOR channel
							 */
	DEVID DevId;						/**< Id of the device the alert is reported by */
	U16 usAlrtType;					/**< see the Alert Type table, at the CRM, paragraph \n
							Alert Diagnostic Indication Format */
	U8 ucCPUNo;					/**< CPU Number [0x00 or 0x01]  */
	U16 usUniqueId;					/**< Unique ID Number  */
	U32 uiLnkRegVal;				/**< If level = 1  ID will be DEVID  */	
	U8 aucAlertRawData[MAX_ALERT_DATA_SIZE];	/**< Device Internal State Information */	
	U8 ucAlertRawDataSize;				/**< Device Internal State Information array size*/
} SAlrtIndParams;

/** H.245 Indication Structure */
typedef struct _SH245IndParams {
	U16 uiEndOfMsg;		/**< End of Message flag*/
	U16 uiH245MsgLen;	/**< H.245 Message Length*/
	U8 *ptrH245Msg;		/**< Poiner to H.245 Message Buffer*/
} SH245IndParams;

/** SYNCDAT Indication Structure */
typedef struct _SSYNCDATParams {
	CONNID ConId;				/**< Connection ID */
	U16 uiPayloadLen;			/**< Payload length*/
	U8 aucPayload[MAX_SYNCDAT_BUF_SIZE];	/**< Payload Buffer*/
} SSYNCDATParams;

/** SYNCEOF Indication Structure */
typedef struct _SSYNCEOFParams {
	CONNID ConId;				/**< Connection ID */
	U16 uiPayloadLen;			/**< Payload length*/
	U8 aucPayload[MAX_SYNCEOF_BUF_SIZE];	/**< Payload Buffer*/
} SSYNCEOFParams;

/** IP parameters Structure */
typedef struct _SIpParams {
	IN U32 uiDestIpAddr;	/**< Destination IP address*/
	IN U16 usDestUdpPort;	/**< Destination UDP port */
	IN U32 uiSrcIpAddr;	/**< Source IP address */
	IN U16 usSrcUdpPort;	/**< Source UDP port */
	IN U8 ucIPServiceId;	/**< IP Service identifier (0:RTP, 1:RTCP)*/
} SIpParams;


/** C2C options Structure */
typedef struct _SC2CConnectOption
{
	Boolean bAction;		/**< Enable, Disable */
	U8 ucConnType;			/**< intra (0) / inter (1) Comcerto */
	U8 ucInterMode;			/**< HDLC 1 TS (0) / HDLC 2 TS (1) */
	CONNID ConnA;			/**< Connection ID used for the TrFO connection */ 
	CONNID ConnB;			/**< Connection ID used for the TrFO Intra-Comcerto connection type*/ 
	U8 ucRateControl;		/**< 0 ACK, 1 NACK */
	U8 ucTimerInterval;		/**< Timer interval for RFCI to CMR convertion (in ms) */
	/** RFCI Parameters are present only in the inter-Comcerto mode.*/
	U16 usRFCIModeGroup1;	/**< RFCI number for 5.15, 8.85, 4.75, 6.6 modes */
	U16 usRFCIModeGroup2;	/**< RFCI number for 6.7, 14.25, 5.9, 12.65 modes */
	U16 usRFCIModeGroup3;	/**< RFCI number for 7.95, 18.25, 7.4,	15.85 modes */
	U16 usRFCIModeGroup4;	/**< RFCI number for 12.2, 23.05,10.2,	19.85 modes */
	U16 usRFCIModeGroup5;	/**< RFCI number for 23.85 mode and SID mode*/
	U16 usRFCIModeGroup6;	/**< RFCI number for No Data */   
} SC2CConnectOption;

#include "vapi_config.h"
#include "vapi_device_config.h"

/** A pointer to function used to register an event callback.\n
This function is called when an event is received from a device. \n
Description of EEventCode is given below. \n
The pvData term is a void pointer whose type depends on the value of eEventCode.\n
Event data structures are described as follows. */
typedef void (*PFNEventCallback) (EEventCode eEventCode, void *pvData);

/** Callback registered by user for informing a request completion*/
#ifdef OLD_RESP_CBK
typedef void (*PFNIoCompCallback) (U32 uiId, S32 iStatus, void *pvReturnData, U32 uiReturnDataLen, U32 uiReqId);
#else

/** A pointer to a function used to register request completion callbacks. \n
This function is called when a particular request gets completed. \n
The uiId term contains the Connection or Device Id depending on the request.*/ 
typedef void (*PFNIoCompCallback) (U32 uiId,
				U8 ucCmdLevel,
				/**< The ucCmdLevel term indicates the level of the completed
				request and hence indicates the type of ID uiId is. \n
				It can have the following values:
				- CMD_LEVEL_CONN
				- CMD_LEVEL_DEVICE
				- CMD_LEVEL_CONF.*/
				S32 iStatus,
				/**< The Status term holds the completion status of the request.*/
				void *pvReturnData,
				/**< The pvReturnData term is given via this parameter if there is an output expected*/
				U32 uiReturnDataLen,
				/**< The uiReturnDataLen term is the length of pvReturnData */
				U32 uiReqId);
				/**< The uiReqId term is the ID of the completed request specified with the SRequest parameter while issuing a VAPI call.*/
#endif

/** A data-structure used in API calls to control the Sync/Async nature of the Call. \n
If the pfnIoCompCallback member is NULL then it will be a synchronous call, otherwise it is an asynchronous call.*/
typedef struct _SRequest {
	U32 uiReqId;	
			/**< The uiReqId Member is used to keep track of a request in case of asynchronous calls. \n
			The value contained in this parameter is passed to pfnIoCompCallback
			when the request is completed.\n 
			By using uiReqId, the applications can keep track of issued requests.*/
	PFNIoCompCallback pfnIoCompCallback;
			/**< The callback function pointer */
} SRequest;

/**Announcement Media Host Type*/
typedef enum {
	ePCI = 0,	/**< Over PCI */
	eCSMENCAPS = 1	/**< Over CSM_ENCAPS */
} EMediaHostType;

/**Announcement type*/
typedef enum {
	eHSF = 0,	/**< Host Speech frame */
	ePCM = 1,	/**< PCM */
	eRTP = 2,	/**< RTP */
	eBOTH = 3	/**< Both directions*/
} EAnnounceType;

/**Speech data format*/
typedef enum {
	eHedFormat = 0,		/**< speech data with frame headers*/
	eNoHedFormat = 1	/**< speech data without frame headers*/
} ESpeechDataFormat;

/** Announcement Coding scheme*/
typedef enum {
	eCS_G711_ULAW = 4,		/**< 4: g711 uLaw*/
	eCS_G711_ALAW = 5,		/**< 5: g711 ALaw*/
	eCS_G723_1 = 6,			/**< 6: g723*/
	eCS_G729_A = 8,			/**< 8: g729 A*/
	eCS_G726_32 = 9,		/**< 9: g726 32*/
	eCS_G726_16_audio = 15,		/**< 15: g726 16*/
	eCS_G726_24_audio = 16,		/**< 16: g726 24*/
	eCS_G726_40_audio = 17,		/**< 17: g726 40*/
	eCS_RawPcm = 127		/**< 127: 16 bit signed raw PCM audio*/
} ECodingScheme;

/** Stop announcement type */
typedef enum {
	eDRAIN = 0,	/**< proceed accumulated data, stop*/
	eAS_STOP = 1	/**< clean accumulated data, immediate stop*/
} EStopAnnType;

/** Playback mode */
typedef enum {
	ePlaybackStandard = 2,			/**< standard playback mode*/
	ePlaybackExt = 3,			/**< extended playback mode*/
	ePlaybackSpeedChangeable = 4		/**< speed changeable playback mode*/
} EPlaybackMode ;

/** PlaybackRate */
typedef enum {
	ePlaybackRate_1_1 = 0x00,	/**< normal*/		
	ePlaybackRate_2_1 = 0x01,	/**< fast 2:1*/
	ePlaybackRate_3_2 = 0x02,	/**< fast 3:2*/
	ePlaybackRate_4_3 = 0x03,	/**< fast 4:3*/
	ePlaybackRate_5_4 = 0x04,	/**< fast 5:4*/	
	ePlaybackRate_4_5 = 0x05,	/**< slow 4:5*/
	ePlaybackRate_3_4 = 0x06,	/**< slow 3:4*/
	ePlaybackRate_2_3 = 0x07,	/**< slow 2:3*/
	ePlaybackRate_1_2 = 0x08,	/**< slow 1:2*/	
}EPlaybackRate;

/** Play-related data*/
typedef struct _SPlayData{
	EMediaHostType eMediaHost;
					/**< - 0 for PCI, - 1 for CSM_ENCAPS\n
					Not being used in current implementation.*/
	ECodingScheme eCodingScheme;	
					/**< Source coding schemes for Host Speech Frame\n
					for playing e.g. G.711u-law PCM audio, G.723 audio etc*/
	EAnnounceType eDestinationType;	
					/**< Destination can be one of the following:\n
						- 8-bit PCM
						- RTP
						- Both */
	EPlaybackMode ePlaybackMode;			
					/**< Playback mode.*/

	U32 uiTimeOut;
					/**< Specifies the time (in Sec) for which playback needs to be done.\n
					If its value is 0 ignore this parameter*/
	U8 * pucBuffer;			
					/**< Pointer to the Buffer from which play out is done.\n
					It's used for APIs dealing with buffer (e.g. VAPI_StartPlayback).\n
					It may not be defined for APIs dealing with file (e.g. VAPI_StartPlaybackFile).*/					
	U32 uiBufferLength;		
					/**< Length of the Buffer passed to the API.\n
					It's used for APIs dealing with buffer (e.g. VAPI_StartPlayback).\n
					It may not be defined for APIs dealing with file (e.g. VAPI_StartPlaybackFile).*/					
	char strAnnFileName[256];
					/**< File name from which being played data needs to be read.\n
					It's used for APIs dealing with file (e.g. VAPI_StartPlaybackFile).\n
					It may not be defined for APIs dealing with buffer (e.g. VAPI_StartPlayback).*/
	ESpeechDataFormat uiHostSpeechDataFormat;	
					/**< Format of being played data.*/
}SPlayData;

/** Rec-related data*/
typedef struct _SRecData{
	EMediaHostType eMediaHost;	
					/**< - 0 for PCI, - 1 for CSM_ENCAPS\n
					Not being used in current implementation.*/
	ECodingScheme eCodingScheme;	
					/**< Destination coding scheme for Host Speech.*/
	EAnnounceType eSourceType;	
					/**< Source can be one of the following:\n
						- 8-bit PCM
						- RTP
						- Both */
	U32 uiTimeOut;			
					/**< Specifies the time (in Sec) for which recording needs to be done. \n
					If its value is 0 ignore this parameter.*/
	U8 * pucBuffer;			
					/**< Pointer to the Buffer in which recorded data needs to be stored.\n
					It's used for APIs dealing with buffer (e.g. VAPI_StartRecording).\n
					It may not be defined for APIs dealing with file (e.g. VAPI_StartRecordFile).*/
	U32 uiBufferLength;		
					/**< Length of the Buffer passed to the API.\n
					It's used for APIs dealing with buffer (e.g. VAPI_StartRecording).\n
					It may not be defined for APIs dealing with file (e.g. VAPI_StartRecordFile).*/					
	char strAnnFileName[256];
					/**< File name in which recorded data needs to be stored.\n
					It's used for APIs dealing with file (e.g. VAPI_StartRecordFile).\n
					It may not be defined for APIs dealing with buffer (e.g. VAPI_StartRecord).*/
	
	PFNIoCompCallback pfnFrameHndlr;	
					/**< If not NULL, pfnFrameHndlr is called for 
					each complete frame received from SYNCDAT/SYNEOF*/
	ESpeechDataFormat uiHostSpeechDataFormat;	
					/**< Format of being recorded data.*/
}SRecData;

/** SConvertData data structure*/
typedef struct _SConvertData
{
	U8 * pucSrcBuffer;		
					/**< Pointer to the buffer having host speech frames \n
					which needs to be converted.*/
	U32 uiSrcBufferLength;		
					/**< Buffer length of source buffer.*/
	ECodingScheme ucCurrentCoding;
					/**< Existing coding scheme for the Host Speech Frames.*/
	U8 * pucOutBuffer;
					/**< Pointer to the Buffer in which converted host speech \n
					frames will be stored.*/
	U32 uiOutBufferLength;
					/**< Buffer length of out buffer.*/
	ECodingScheme ucNewCoding;
					/**< New coding of out data.*/
} SConvertData;

/** Modify connection struct */
typedef struct _SModifyConnectionInfo  {
	U16 usFunctionCode;	/**< FunctionCode of the MSP command */
	U16 usParam;		/**< Param number of the MSP command */
	U8 ucfield_shift; 	/**< First bit of the field to be modified */
	U8 ucfield_width; 	/**< Width of the field to be modified */
	U16 usValue;		/**< Value to be applied to the bit field*/
} SModifyConnectionInfo;

/**Resource manager setting action*/
typedef enum _EResAction {
	eResourceDisable = 0,		/**< Disable resource manager */
	eResourceEnable = 1,		/**< Enable resource manager */
	eResourceQuery = 2,		/**< Query available resource */
} EResAction;

/**Resource manager index*/
typedef enum {
	eResG711_20 = 0 ,		/**< 0: g711 u/aLaw 20ms */
	eResG711_10 = 1 ,		/**< 1: g711 u/aLaw 10ms */
	eResG711_5 = 2 ,		/**< 2: g711 u/aLaw 5ms */
	eResG726_20 = 3 ,		/**< 3: g726 20ms */
	eResG726_10 = 4 ,		/**< 4: g726 10ms */
	eResG726_5 = 5 ,		/**< 5: g726 5ms */
	eResG729a_20 = 6 ,		/**< 6: g729a 20ms */
	eResG729a_10 = 7,		/**< 7: g729a 10ms */
	eResG723_1 = 8 ,		/**< 8: g723 */
	eResG728 = 9 ,			/**< 9: g728a */
	eResG729e = 10 ,		/**< 10: g729e */
	eResG722 = 11 ,			/**< 11: g722 */
	eResG722_2_VAD_ON = 12 ,	/**< 12: g722.2 VAD On */
	eResG722_2_VAD_OFF = 13 ,	/**< 13: g722.2 VAD Off */
	eResG729_1_8K = 14 ,		/**< 14: g729.1 8k */
	eResG729_1_12K = 15 ,		/**< 15: g729.1 12k */
	eResG729_1_14K = 16 ,		/**< 16: g729.1 14k */
	eResG729_1_16K = 17 ,		/**< 17: g729.1 16k */
	eResG729_1_18K = 18 ,		/**< 18: g729.1 18k */
	eResG729_1_24K = 19 ,		/**< 19: g729.1 24k */
	eResG729_1_28K = 20 ,		/**< 20: g729.1 28k */
	eResiLBC = 21 ,			/**< 21: iLBC */
	eResGSM_FR = 22 ,		/**< 22: GSM FR */
	eResGSM_HR = 23 ,		/**< 23: GSM HR */
	eResGSM_EFR = 24 ,		/**< 24: GSM EFR */
	eResGSM_AMR = 25 ,		/**< 25: ARM */
	eResCDMA_EVRC = 26 ,		/**< 26: CDMA EVRC */
	eResCDMA_EVRC_B = 27,		/**< 27: CDMA EVRC B */
	eResCDMA_QCELP8 = 28 ,		/**< 28: CDMA QCELP 8 */
	eResCDMA_QCELP13 = 29 ,		/**< 29: CDMA QCELP 13 */
	eResT38 = 30 ,			/**< 30: T.38 */
	eResConferencing = 31 ,		/**< 31: Conferencing */
} EResourceIndex;

/** Size of the buffer intended to hold the resources information*/
#define VAPI_NUM_RESOURCES	eResConferencing + 1

/** A pointer to a function used to register customer specific reset device function. \n
This function is called (if not NULL) by VAPI when a device reset is required for some particular operation (basically bootload or coredump). \n.*/
typedef VSTATUS  (*PFNResetDevice) (void *pvData);

VSTATUS VAPI_RegisterReset(IN DEVID DevId, IN PFNResetDevice pfnReset, void *pvData);

/* Interface Functions */
char *VAPI_GetVersion(void);

VSTATUS VAPI_Init(void *pstDeviceHeaders);

VSTATUS VAPI_Close(void);

VSTATUS VAPI_RegisterEventCallback(IN U32 uiID, IN U8 ucLevel, IN PFNEventCallback pfnEventCallback);

VSTATUS VAPI_OpenDevice(IN DEVID DevId, IN PFNEventCallback pfnEventCallback);

VSTATUS VAPI_CloseDevice(IN DEVID DevId, IN ECloseMode eMode);

VSTATUS VAPI_BootDevice(IN DEVID DevId, IN void *pvImageFile, IN U32 uImageSize, IN SHwCfg * pstSHwCfg, IN SRequest * pstRequest);
VSTATUS VAPI_BootDeviceFile(IN DEVID DevId, IN char *strFileName, IN SHwCfg * pstSHwCfg, IN SRequest * pstRequest);

VSTATUS VAPI_InitDevice(IN DEVID DevId,
				IN U32 ulOperationModeId,
				IN U32 ulProfileId,
				IN U32 *pulUserParams,
				IN SRequest * pstRequest);

VSTATUS VAPI_GetCoreDump(IN DEVID DevId,
			IN U8 uchMemTypeIdx,
			OUT U8 * puchBuff,
			IN U32 ulBuffSize,
			OUT U32 * pulDataSize,
			IN U32 ulOffset,
			OUT Boolean * pbLastRead,
			IN SHwCfg * pstSHwCfg,
			IN SRequest * pstRequest);

VSTATUS VAPI_PassThru(IN U32 uiID,
		      IN U8 ucLevel,
		      IN SMsg * pstMsg, IN SRequest * pstRequest, OUT U8 * pucBuffer, IN OUT U32 * puBufLen);

VSTATUS VAPI_ModifyConnection(IN CONNID ConnId,
				IN U16 usModifyNum,
				IN SModifyConnectionInfo *pstModifyConnectionInfo,
				IN SRequest *pstRequest);

VSTATUS VAPI_SendDeviceMessage(IN U32 uiID, IN SMsg * pstMsg, IN SRequest * pstRequest, 
					OUT U8 * pucBuffer, IN OUT U32 * puBufLen);

VSTATUS VAPI_SendConnectionMessage(IN U32 uiID, IN SMsg * pstMsg, IN SRequest * pstRequest, 
					OUT U8 * pucBuffer, IN OUT U32 * puBufLen);

VSTATUS VAPI_SendConferenceMessage(IN U32 uiID, IN SMsg * pstMsg, IN SRequest * pstRequest, 
					OUT U8 * pucBuffer, IN OUT U32 * puBufLen);

VSTATUS VAPI_CreateConnection(IN DEVID DevId,
			      IN CONNID ConnId,
			      IN EConnType usConnType,
			      IN U16 usTdmTimeSlot,
			      IN U8 ucAU_Law, IN SRequest * pstRequest, IN PFNEventCallback pfnEventCallback);

VSTATUS VAPI_AllocateConnection(IN DEVID DevId,
			      IN CONNID ConnId,
			      IN EConnType eConnType,
			      IN EConnMode eConnMode,
			      IN U8 ucNumOfParams,
			      IN U16 *pusParams, IN SRequest * pstRequest, IN PFNEventCallback pfnEventCallback);

VSTATUS VAPI_DestroyConnection(IN CONNID ConnId, IN SRequest * pstRequest);

VSTATUS VAPI_SetConnectionState(IN CONNID ConnId, IN EConnOpMode eOpMode, IN SRequest * pstRequest);

/** Enable voice on the specified connection \n
This maintains backward compatibility with previous version of VAPI*/
#define VAPI_EnableConnection(x, y) VAPI_SetConnectionState(x, eActive, y)

/** Disable voice on a specified connection \n
This maintains backward compatibility with previous version of VAPI*/
#define VAPI_DisableConnection(x, y) VAPI_SetConnectionState(x, eTdmActive, y)

VSTATUS VAPI_Loopback(IN CONNID Conn1, IN CONNID Conn2, IN ELoopbackType eLoopback, IN SRequest * pstRequest);

VSTATUS VAPI_SetEthMac(IN U32 uiID, IN U8 ucLevel, IN U8 * paucSrcMacAddr, IN U8 * paucDestMacAddr, IN SRequest * pstRequest);

VSTATUS VAPI_SetDeviceVlan(IN DEVID DevId, IN U16 VlanId, IN SRequest * pstRequest);

VSTATUS VAPI_SetDeviceIPAddr(IN DEVID DevId, IN SIpAddrInfo * pstIpAddrInfo, IN SRequest * pstRequest);

VSTATUS VAPI_SetCodecType(IN CONNID ConnId, IN ECodecIndex eCodecIndex, IN SRequest * pstRequest);


VSTATUS VAPI_SetPacketInterval(IN CONNID ConnId, IN U8 uiMilliSec, IN SRequest * pstRequest);

VSTATUS VAPI_SetPayloadType(IN CONNID ConnId,
			    IN ECodecType eCodec, IN U8 ucPlTypVal, IN EDirection eDir, IN SRequest * pstRequest);

VSTATUS VAPI_SetRtpSsrcHeader(IN CONNID ConnId, IN U32 uiSsrcVal, IN SRequest * pstRequest);

VSTATUS VAPI_EchoCancellerReset(IN CONNID ConnId, IN SRequest * pstRequest);

/** Reset the Echo Canceller on a specified connection \n
This maintains backward compatibility with previous version of VAPI */
#define VAPI_EchoCancellorReset(x, y) VAPI_EchoCancellerReset(x, y)

VSTATUS VAPI_SendNteEvent(IN CONNID ConnId,
			  IN U32 uiNtePyLd,
			  IN U16 usOverRideBitField,
			  IN U16 usRedundancyInterval, IN U32 uiSsrc, IN U16 usPyLdType, IN SRequest * pstRequest);


VSTATUS VAPI_MuteParticipant(IN PARTID PartId, IN Boolean bMute, IN SRequest * pstRequest);


VSTATUS VAPI_PutParticipantOnHold(IN PARTID PartId,
				  IN EHoldState eHoldState, IN PARTID SrcPartId, IN SRequest * pstRequest);


VSTATUS VAPI_SetTDMParams(IN DEVID DevId, IN STdmSetupParams * pstTdmSetupParam, IN IN SRequest * pstRequest);


VSTATUS VAPI_StartTone(IN CONNID ConnId,
		       U8 ucDir, EToneId eToneId, ECountryCode eCountryCode, IN SRequest * pstRequest);

VSTATUS VAPI_PlayTone(IN CONNID ConnId, EToneId eToneId, EToneDir eToneDir, SPlayToneOpt *pstPlayToneOpt, 
					U16 usParamNum, IN SRequest * pstRequest, ...);

VSTATUS VAPI_SetTone(EToneId eToneId, EToneType eToneType, U16 usCadenceRpt, U16 usParamNum,...);

VSTATUS VAPI_StopTone(IN CONNID ConnId, IN U16 usToneDuration, IN U16 usSide, IN SRequest * pstRequest);

VSTATUS VAPI_SetCidGenInfo(EAction eAction, ECidGenInfoType eCidGenInfoType, U16 usParamNum, ...);

VSTATUS VAPI_StartCallerId(IN CONNID ConnId, IN CidInfo *pstCidInfo, IN SRequest * pstRequest); 

VSTATUS VAPI_PlayCid(IN CONNID ConnId, IN SCallerIdInfo *pstCidInfo, IN SRequest * pstRequest);

VSTATUS VAPI_StopCallerId(IN CONNID ConnId, IN SRequest * pstRequest);

VSTATUS VAPI_SetCidDetInfo(EAction eAction, ECidDetInfoType eCidDetInfoType, U16 usParamNum, ...);

VSTATUS VAPI_CreateConference(IN DEVID DevId,
			      IN CONFID ConfId,
			      IN SConfParams * pstConfParams,
			      IN Boolean bIsEnhancedConf,
			      IN SConfDTSParams * pstConfDTSParams, IN SRequest * pstRequest);


VSTATUS VAPI_DestroyConference(IN CONFID ConfId, IN SRequest * pstRequest);


VSTATUS VAPI_CreateIpSideParticipant(IN CONFID ConfId,
				     IN PARTID PartId, IN SRequest * pstRequest, IN PFNEventCallback pfnEventCallback);

VSTATUS VAPI_CreateTranscodingParticipant(IN CONFID ConfId,
				     IN PARTID PartId, IN SRequest * pstRequest, IN PFNEventCallback pfnEventCallback);

VSTATUS VAPI_SetParticipantDgain(IN PARTID PartId, IN U16 usGain, IN SRequest *pstRequest);

VSTATUS VAPI_SetParticipantOpt(IN PARTID PartId, IN U16 usRfc2833OnOff, IN SRequest *pstRequest);

VSTATUS VAPI_SetConferenceDgain(IN DEVID DevId, IN U16 usGain, IN SRequest * pstRequest);

VSTATUS VAPI_SetConferenceLatency(IN DEVID DevId, IN U16 usLatency, IN SRequest * pstRequest);

VSTATUS VAPI_CreateTdmSideParticipant(IN CONFID ConfId,
				      IN PARTID PartId,
				      IN U16 usTdmTimeSlot,
				      IN U8 ucAU_Law, IN SRequest * pstRequest, IN PFNEventCallback pfnEventCallback);

VSTATUS VAPI_AllocateParticipant (IN CONFID ConfId,
					IN PARTID PartId,
					IN EPartType ePartType,
					IN U8 ucNumOfParams,
					IN U16 *pusParams,
					IN SRequest * pstRequest, IN PFNEventCallback pfnEventCallback);

VSTATUS VAPI_DestroyParticipant(IN PARTID PartId, IN SRequest * pstRequest);

VSTATUS VAPI_EnableParticipant(IN PARTID PartId, IN SRequest * pstRequest);

VSTATUS VAPI_DisableParticipant(IN PARTID PartId, IN SRequest * pstRequest);


VSTATUS VAPI_EnableCIDDetection(IN CONNID ConnId,
				IN ECountryCode eCountryCode, IN U8 ucEnableType, IN SRequest * pstRequest);

VSTATUS VAPI_SetCidDetection (IN CONNID ConnId, IN SCidDetCtrl *pstCidDetCtrl, IN SRequest * pstRequest);

VSTATUS VAPI_StartRecord(IN CONNID ConnId, IN SRecData * pstRecData, IN SRequest * pstRequest);

VSTATUS VAPI_StopRecord(IN CONNID ConnId, IN EStopAnnType eStopAnnType, IN SRequest * pstRequest);

VSTATUS VAPI_StartPlayback(IN CONNID ConnId, IN SPlayData * pstPlayData, IN SRequest * pstRequest);

VSTATUS VAPI_StopPlayback(IN CONNID ConnId, IN EStopAnnType eStopAnnType, IN SRequest * pstRequest);

VSTATUS VAPI_StartRecordFile(IN CONNID ConnId, IN SRecData * pstRecData, IN SRequest * pstRequest);

VSTATUS VAPI_StartPlaybackFile(IN CONNID ConnId, IN SPlayData * pstPlayData, IN SRequest * pstRequest);

VSTATUS VAPI_StartRecording(IN CONNID ConnId,
				IN EMediaHostType eMediaHost,
				IN ECodingScheme eCodingScheme,
				IN EAnnounceType eSourceType,
				IN U32 uiTimeOut,
				IN U8 * pucBuffer,
				IN U32 uiBufferLength, IN PFNIoCompCallback pfnFrameHndlr, IN SRequest * pstRequest);

VSTATUS VAPI_StopRecording(IN CONNID ConnId, IN SRequest * pstRequest);

VSTATUS VAPI_StartPlaying(IN CONNID ConnId,
				IN EMediaHostType eMediaHost,
				IN ECodingScheme eCodingScheme,
				IN EAnnounceType eDestinationType,
				IN U32 uiTimeOut, IN U8 * pucBuffer, IN U32 uiBufferLength, IN SRequest * pstRequest);

VSTATUS VAPI_StopPlaying(IN CONNID ConnId, IN SRequest * pstRequest);

VSTATUS VAPI_PlaybackSetRate(IN CONNID ConnId, EPlaybackRate ePlaybackRate, IN SRequest * pstRequest);

VSTATUS VAPI_ConvertHostSpeechData(IN CONNID ConnId, IN SConvertData *pstConvertData, IN SRequest * pstRequest);

VSTATUS VAPI_SetDebugLevel(IN U8 ucVapiDebugLevel, IN U8 ucAppDebugLevel);

VSTATUS VAPI_ConfigureT38Options(IN CONNID ConnId, IN SFaxConfigOpts * pstFaxConfigOpts, IN SRequest * pstRequest);
VSTATUS VAPI_SwitchToT38(IN CONNID ConnId, IN SFaxConfigOpts * pstFaxConfigOpts, IN SRequest * pstRequest);
VSTATUS VAPI_AssignBootMAC(IN DEVID DevId, IN SRequest * pstRequest);
VSTATUS VAPI_SetConnIpParams(IN CONNID ConnId, IN SIpParams * pstIpParams, IN SRequest * pstRequest);
VSTATUS VAPI_SetConnVlan(CONNID ConnId, IN U16 VlanId, IN SRequest * pstRequest);
VSTATUS VAPI_SetMediaSubLayer(IN CONNID ConId, IN EMediaSubLayer eMediaSubLayer, IN SRequest * pstRequest);
VSTATUS VAPI_SendH245Message(IN CONNID ConnId, IN U16 uiH245MsgLength, IN U8 *ptrH245Msg, IN SRequest *pstRequest);
STATUS VAPI_TranscodingSession(IN CONNID   ConnId1,
				IN CONNID   ConnId2,
				IN STranscodingOption *pstTranscodingOption,
				IN SRequest *pstRequest);

VSTATUS VAPI_RecoverConnection(IN DEVID DevId,
			      IN CONNID ConnId,
			      IN EConnType eConnType,
			      IN U16 usTdmTimeSlot,
			      IN U16 usRecoverOption,
			      IN SRequest * pstRequest, IN PFNEventCallback pfnEventCallback);

VSTATUS VAPI_CESoPSN_Mapper(IN DEVID DevId,
				IN SRequest * pstRequest,
				IN EMapperType eMapType,
				IN CONNID MasterConnId,
				U16 Action, U16 usConnNum, ...);

VSTATUS VAPI_ChangeConnectionType(IN CONNID ConnId,
				IN SChangeConnInfo *pstChangeConnInfo,
				IN U16 NumParam,
				IN U16 *pusParams,
				IN SRequest *pstRequest);

VSTATUS VAPI_C2CConnectUP(IN DEVID DevId, IN SC2CConnectOption *pstC2CConnectOptions, IN SRequest *pstRequest);

void *VAPI_AllocateMessage(U16 usSize);

void VAPI_InitMessage(void *pvMsg);

VSTATUS VAPI_FreeMessage(void *pvMsg);

VSTATUS VAPI_SetMessage(OUT void *pvMsg , IN U8 ucCmdClass, IN U8 ucCmdType, IN U16 usFcode, 
				IN U16 ucNumParams,... );
VSTATUS VAPI_SetMessageFromBuffer(OUT void *pvMsg , IN U8 ucCmdClass, IN U8 ucCmdType, IN U16 usFcode, 
				IN U16 ucNumParams, IN U16 *pusParameters);

VSTATUS VAPI_SetConnectionResource(IN DEVID DevId, IN CONNID ConnId, IN U8 ucResourceNum, IN U8 *pucRequiredResource);
VSTATUS VAPI_SetResourceManager(IN DEVID DevId, IN EResAction eResAction, OUT U16 *pusQueryInfo);

#ifdef VAPI_OLD_API
#define VAPI_RegisterEventCallback(pfnEventCallback) VAPI_RegisterEventCallback(0,EVENT_LEVEL_GENERIC,pfnEventCallback)
#define VAPI_OpenDevice(DevId) VAPI_OpenDevice(DevId,NULL)
#define VAPI_CreateConnection(DevId,ConnId,eConnType,usTdmTimeSlot,ucAU_Law,pstRequest) \
        VAPI_CreateConnection(DevId,ConnId,eConnType,usTdmTimeSlot,ucAU_Law,pstRequest,NULL)
#endif
/*
 * List of error codes
 */
/** Error base defined for VAPI defined error codes*/
#define VAPI_ERROR_BASE         68000
#define VAPI_ERROR_START        -(VAPI_ERROR_BASE)
/** The specified device ID does not exist*/
#define VAPI_ERR_INVALID_DEVID          -(VAPI_ERROR_BASE + 1)
/** The specified connection does not exist*/
#define VAPI_ERR_INVALID_CONNID         -(VAPI_ERROR_BASE + 2)
/** Operation failed due to insufficient memory*/
#define VAPI_ERR_NOMEM                  -(VAPI_ERROR_BASE + 3)
/** Invalid parameter value was passed*/
#define VAPI_ERR_INVALID_PARAM          -(VAPI_ERROR_BASE + 4)
/** NULL pointer was passed for a variable expected to be not NULL.*/
#define VAPI_ERR_NULL_POINTER_PASSED    -(VAPI_ERROR_BASE + 5)
/** No response was obtained from MSP (Timeout occured)*/
#define VAPI_ERR_MSP_NO_RESPONSE        -(VAPI_ERROR_BASE + 6)
/** This feature is not suported*/
#define VAPI_ERR_UNSUPP_FEATURE         -(VAPI_ERROR_BASE + 7)
/** The device that is being added/opened is already added/opened*/
#define VAPI_ERR_DEVICE_EXISTS          -(VAPI_ERROR_BASE + 8)
/** Some internal resource initialization failed*/
#define VAPI_ERR_RSRC_INIT_FAILED       -(VAPI_ERROR_BASE + 9)
/** There is something wrong with the image that is being downloaded*/
#define VAPI_ERR_INVALID_IMAGE          -(VAPI_ERROR_BASE + 10)
/** Undefined internal error*/
#define VAPI_ERR_UNDEFINED              -(VAPI_ERROR_BASE + 11)
/** Command execution is pending within vapi*/
#define VAPI_ERR_PENDING                -(VAPI_ERROR_BASE + 12)
/** Unexpected response packet obtained from MSP*/
#define VAPI_ERR_UNEXPECTED_MSP_RESPONSE     -(VAPI_ERROR_BASE + 13)
/** The operation is invalid with current settings*/
#define VAPI_ERR_INVALID_OPERATION      -(VAPI_ERROR_BASE + 14)
/** The Library is not yet initialized*/
#define VAPI_ERR_LIB_NOT_INITIALIZED    -(VAPI_ERROR_BASE + 15)
/** Returned by VAPI_BootDevice when device is already up*/
#define VAPI_ERR_DEV_IS_UP              -(VAPI_ERROR_BASE + 16)
/** Device is not up so no requests can be accepted*/
#define VAPI_ERR_DEV_IS_NOT_UP          -(VAPI_ERROR_BASE + 17)
/** Device IP address is not set */
#define VAPI_ERR_DEVIP_NOTSET		-(VAPI_ERROR_BASE + 18)
/** Missing source UDP port */ 
#define VAPI_ERR_INCOMP_SRC_UDP 	-(VAPI_ERROR_BASE + 19)
/** Missing destination UDP port */ 
#define VAPI_ERR_INCOMP_DEST_IPUDP	-(VAPI_ERROR_BASE + 20)

#define VAPI_ERR_ETH_HDR_NOT_SET        -(VAPI_ERROR_BASE + 21)
/** Connection IP header not set */
#define VAPI_ERR_IP_HDR_NOT_SET         -(VAPI_ERROR_BASE + 22)

/** The specified conference does not exist*/
#define VAPI_ERR_INVALID_CONFID         -(VAPI_ERROR_BASE + 23)
/** Device is wrong state for current operation */
#define VAPI_ERR_INVALID_DEV_STATE      -(VAPI_ERROR_BASE + 24)

/** Core Dump API - requested memory does not exist*/
#define VAPI_ERR_INVALID_MEM_IDX        -(VAPI_ERROR_BASE + 25)

/** Core Dump API - Invalid offset in memory*/
#define VAPI_ERR_INVALID_OFFSET         -(VAPI_ERROR_BASE + 26)

#define VAPI_ERR_CONN_DESTROYING	-(VAPI_ERROR_BASE + 27)

/** Invalid conference ID */
#define VAPI_ERR_INVALID_CONF_ID        -(VAPI_ERROR_BASE + 28)

/** SPU control feature not enabled */
#define VAPI_ERR_SPU_FEATURE_NOT_ENABLED        -(VAPI_ERROR_BASE + 29)

#define VAPI_ERR_INVALID_ENABLE_TYPE    -(VAPI_ERROR_BASE + 30)

#define VAPI_ERR_INVALID_FRAME          -(VAPI_ERROR_BASE + 31)

#define VAPI_ERR_STOP_REC_EXIST         -(VAPI_ERROR_BASE + 32)

#define VAPI_ERR_STOP_PLAY_EXIST        -(VAPI_ERROR_BASE + 33)
/** Error returned for BootDevice API*/
#define VAPI_ERR_MAAS_ASSIGN            -(VAPI_ERROR_BASE + 34)
/** Error during Comcerto firmware start */
#define VAPI_ERR_PROG_START             -(VAPI_ERROR_BASE + 35)
/** Supervisor Ready returned an error */
#define VAPI_ERR_SUPVSR_READY		    -(VAPI_ERROR_BASE + 36)
/** Connection in wrong type for current process */
#define VAPI_ERR_INVALID_CONN_TYPE	-(VAPI_ERROR_BASE + 37)

/*#define VAPI_ERR_T38_IPUDP_INFO_MISSING -(VAPI_ERROR_BASE + 38)*/
 /*Removed as now T38 port setting is not done from inside VAPI
    API. Instead the IP params as set as in the case of VoIP channels
    by calling a separate API VAPI_SetConnIPParams */

#define VAPI_ERR_CHECKSUM_FAIL		-(VAPI_ERROR_BASE + 38)
#define VAPI_ERR_DEV_NO_MAAS		-(VAPI_ERROR_BASE + 39)
#define VAPI_ERR_NO_CSM_ITF		-(VAPI_ERROR_BASE + 40)
#define VAPI_ERR_PHYID_ASSIGN           -(VAPI_ERROR_BASE + 41)
#define VAPI_ERR_DEV_ALREADY_OPEN	-(VAPI_ERROR_BASE + 42)
#define VAPI_REQ_COMPLETED		-(VAPI_ERROR_BASE + 43)
/** Timeout on process request */
#define VAPI_ERR_TIMEOUT		-(VAPI_ERROR_BASE + 44)
/** fifo has a wrong size */
#define VAPI_ERR_FIFO_SIZE		-(VAPI_ERROR_BASE + 45)
/** The payload type for the connection is not initialized*/
#define VAPI_ERR_PT_NOT_INITIALIZED	-(VAPI_ERROR_BASE + 46)
/** The device has not been set for multi command mode*/
#define VAPI_ERR_MULTI_CMD_NOT_INITIALIZED	-(VAPI_ERROR_BASE + 47)
/** The device has not been initialized with VAPI_InitDevice()*/
#define VAPI_ERR_DEVICE_NOT_INITIALIZED		-(VAPI_ERROR_BASE + 48)
/** The conenction has no alternate header set*/
#define VAPI_ERR_ALT_HDR_NOT_SET		-(VAPI_ERROR_BASE + 49)
/** Get into a FSM handler with an unhandled event value*/
#define VAPI_ERR_UNDEFINED_EVENT		-(VAPI_ERROR_BASE + 50)
/** Recover connection failed: no Comcerto channel found with required timeslot*/
#define VAPI_ERR_RECOVER_TIMESLOT		-(VAPI_ERROR_BASE + 51)
/** Recover connection failed: Recovered Comcerto channel type doesn't match required connection type*/
#define VAPI_ERR_RECOVER_WRONG_TYPE		-(VAPI_ERROR_BASE + 52)
/** Recover connection failed: Recovered Comcerto channel already in use by another connection*/
#define VAPI_ERR_RECOVER_INVALID		-(VAPI_ERROR_BASE + 53)
/** Create connection failed: the connection or the participant is not created at all*/
#define VAPI_ERR_CREATECONN_FAIL		-(VAPI_ERROR_BASE + 54)
/** Create conference failed: the conference is not created at all*/
#define VAPI_ERR_CREATECONF_FAIL		-(VAPI_ERROR_BASE + 55)
/** The specified device type is not supported*/
#define VAPI_ERR_INVALID_DEV_TYPE          	-(VAPI_ERROR_BASE + 56)
/** Specific to M823XX waiting for ready command from MSP*/
#define VAPI_ERR_WAIT_READY          		-(VAPI_ERROR_BASE + 57)
/** Timeslot is already used by another channel*/
#define VAPI_ERR_TS_IN_USE          		-(VAPI_ERROR_BASE + 58)
/** Wrong combination of Command Type/Class */
#define VAPI_ERR_ILLEGAL_COMMAND		-(VAPI_ERROR_BASE + 59)
#define VAPI_ERR_ILLEGAL_CT_FOR_CC		-(VAPI_ERROR_BASE + 60)

/** Resource manager not enabled */
#define VAPI_ERR_RES_MGR_NOT_ENABLED		-(VAPI_ERROR_BASE + 61)
/** No resource available */
#define VAPI_ERR_NO_RESOURCE			-(VAPI_ERROR_BASE + 62)
/** No resource allocated */
#define VAPI_ERR_NO_RESOURCE_ALLOC		-(VAPI_ERROR_BASE + 63)
/** Allocated resource invalid */
#define VAPI_ERR_INVALID_RESOURCE		-(VAPI_ERROR_BASE + 64)
/** Allocated resource unknow configuration: the configuation is not supported by the resource manager */
#define VAPI_ERR_UNKNOWN_RESOURCE		-(VAPI_ERROR_BASE + 65)
/** Resource manager not supported for this device */
#define VAPI_ERR_RES_MGR_NOT_SUPPORTED		-(VAPI_ERROR_BASE + 66)

#ifdef __cplusplus
};
#endif

#endif
