/*! \file resmgr_itf.c
 @defgroup resource_manager  Resource manager related API functions
 *  @{
 */

/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include "ut.h"
#include "vapi.h"
#include "dmgr.h"
#include "vcore.h"
#include "vcore_voip.h"
#include "appitf.h"
#include "cfg.h"
#include "msp.h"
#include "resmgr.h"

extern Boolean bVapiInitialized_g;
/*Global Definitions*/
extern SVAPIConfig stVAPIConfig_g;
extern SAVLTree *pstResTree_g;

/*COMCERTO resources */
extern U16 M823x9_mix_mode_res[32];
extern U16 M823x8_mix_mode_res[32];
extern U16 M823x6_mix_mode_res[32];
extern U16 M823x4_mix_mode_res[32];
extern U16 M823x3_mix_mode_res[32];
extern U16 M823x2_mix_mode_res[32];
extern U16 M823x1_mix_mode_res[32];
extern U16 M82910_mix_mode_res[32];

/***************************************************************************
 * VAPI_SetResourceManager: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This API is used to enable, disable or query the VAPI resource manager on a device.\n
*	The VAPI resource manager also allows the application to know how many resources is available for different types of codec and packet size. \n
*	The VAPI resource manager operates at device level, this means that the resources are calculated per device. \n
*	The VAPI resource manager is by default disabled. It can't be disabled or enabled when some connections are currently created \n
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">Id of the device to enable the resource manager on</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eResAction</td>
*		<td style="vertical-align: top;">Action to perform</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pucQueryInfo</td>
*		<td style="vertical-align: top;">Pointer to buffer to hold the availability information in case of query.\n
*		A NUM_RESOURCES buffer size must be allocated by the application to store the resources information.
*		The format of the returned information is this one: \n
*		<pre>
*		Resource type		Number of available resource
*		eG711 20			160
*		eG711 10			112
*		eG711 5				66
*		eG726 20			98
*		eG726 10			88
*		eG726 5				60
*		eG729a 20			72
*		eG729a 10			62
*		....				....
*		</pre>
*		Check the ECodecResourceIndex (in vapi.h) for the full list of resources type definition.
*		</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_RES_MGR_NOT_ENABLED
*	\li VAPI_ERR_NULL_POINTER_PASSED
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include resource_manager.c
*
*	\n \b Commands:
*	\li No Comcerto commands sent.
*/
VSTATUS VAPI_SetResourceManager(IN DEVID DevId, IN EResAction eResAction, OUT U16 *pusQueryInfo)
{
	SDevice *pstDevice;
	VSTATUS Status;
	int i;
	U16 *DevResource = NULL;
	U16 DevResSize = 0;

	Status = VCORE_DeviceGenericCheck(DevId, "VAPI_SetResourceManager", &pstDevice);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_SetResourceManager\n");

	/*Point to the right array depending on the model type*/
	switch(pstDevice->usDevModel)
	{
		case M82910:
			DevResource = M82910_mix_mode_res;
			DevResSize = sizeof(M82910_mix_mode_res);
			break;

		case  M82359:
		case  M82349:
			DevResource = M823x9_mix_mode_res;
			DevResSize = sizeof(M823x9_mix_mode_res);
			break;

		case  M82358:
		case  M82348:
		case  M82318:
		case  M82308:
			DevResource = M823x8_mix_mode_res;
			DevResSize = sizeof(M823x8_mix_mode_res);
			break;

		case  M82356:
		case  M82346:
			DevResource = M823x6_mix_mode_res;
			DevResSize = sizeof(M823x6_mix_mode_res);
			break;

		case  M82354:
		case  M82344:
		case  M82334:
		case  M82324:
			DevResource = M823x4_mix_mode_res;
			DevResSize = sizeof(M823x4_mix_mode_res);
			break;

		case  M82353:
		case  M82343:
			DevResource = M823x3_mix_mode_res;
			DevResSize = sizeof(M823x3_mix_mode_res);
			break;

		case  M82352:
		case  M82332:
		case  M82322:
			DevResource = M823x2_mix_mode_res;
			DevResSize = sizeof(M823x2_mix_mode_res);
			break;

		case  M82351:
		case  M82331:
		case  M82321:
		case  M82311:
			DevResource = M823x1_mix_mode_res;
			DevResSize = sizeof(M823x1_mix_mode_res);
			break;

		default:
			UT_ErrorLog(APPITF, "VAPI_SetResourceManager: Doesn't support device model %d \n", pstDevice->usDevModel);
			Status = VAPI_ERR_RES_MGR_NOT_SUPPORTED;
			goto out;
	}

	if((eResAction == eResourceQuery) && (pusQueryInfo == NULL))
	{
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		UT_ErrorLog(APPITF, "VAPI_SetResourceManager: pusQueryInfo NULL\n");
		goto out;
	}

	if((eResAction == eResourceDisable) || (eResAction == eResourceEnable))
	{
		for(i = 0; i < pstDevice->usMaxChnls; i ++)
		{	
			if (pstDevice->papstChannels[i] != NULL)
			{
				UT_ErrorLog(APPITF, "VAPI_SetResourceManager: can't enable/disable VAPI device manager while connections exists \n");
				Status = VAPI_ERR_INVALID_PARAM;
				goto out;
			}
		}
	}

	switch(eResAction)
	{
	case eResourceDisable: /* disable resource manager */
		if (pstDevice->stDevRes.bIsResourceMngr == False)
		{
			Status = VAPI_ERR_RES_MGR_NOT_ENABLED;
			UT_ErrorLog(APPITF, "VAPI_SetResourceManager: not enabled\n");
			goto out;
		}

		pstDevice->stDevRes.bIsResourceMngr = False;
		pstDevice->stDevRes.ResourceCount = 0;
		if (pstDevice->stDevRes.pusGenericResources)
			UT_FreeMem(pstDevice->stDevRes.pusGenericResources);
		if (pstDevice->stDevRes.pusCurrentResources)
			UT_FreeMem(pstDevice->stDevRes.pusCurrentResources);

		if (pstResTree_g)
			UT_DeleteAVLTree(pstResTree_g);

		break; 

	case eResourceEnable: /* enable resource manager */

		pstResTree_g = UT_CreateAVLTree();
		if (pstResTree_g == NULL)
		{
			Status = VAPI_ERR_RSRC_INIT_FAILED;
			goto out;
		}

		pstDevice->stDevRes.bIsResourceMngr = True;
		/* TODO: set the right number accordingly the device model */
		pstDevice->stDevRes.ResourceCount = COMCERTO_RESOURCES;
		pstDevice->stDevRes.pusGenericResources = UT_AllocMem(DevResSize);
		if (pstDevice->stDevRes.pusGenericResources == NULL)
		{
			UT_DeleteAVLTree(pstResTree_g);
			UT_ErrorLog(APPITF, "VAPI_SetResourceManager: No memory available\n");
			Status = VAPI_ERR_NOMEM;
			goto out;
		}

		pstDevice->stDevRes.pusCurrentResources = UT_AllocMem(DevResSize);
		if (pstDevice->stDevRes.pusCurrentResources == NULL)
		{
			UT_FreeMem(pstDevice->stDevRes.pusGenericResources);
			UT_DeleteAVLTree(pstResTree_g);
			UT_ErrorLog(APPITF, "VAPI_SetResourceManager: No memory available\n");
			Status = VAPI_ERR_NOMEM;
			goto out;
		}

		UT_MemCopy(pstDevice->stDevRes.pusGenericResources, DevResource, DevResSize);
		VRES_InitDeviceRes(pstDevice->stDevRes.ResourceCount,
					(DevResSize/sizeof(U16)), DevResource, pstDevice->stDevRes.pusCurrentResources);

		break;

	case eResourceQuery: /* query current resource */
		if (pstDevice->stDevRes.bIsResourceMngr == False)
		{
			Status = VAPI_ERR_RES_MGR_NOT_ENABLED;
			UT_ErrorLog(APPITF, "VAPI_SetResourceManager: not enabled\n");
			goto out;
		}

		VRES_GetDeviceRes(pstDevice->stDevRes.ResourceCount,
					(DevResSize/sizeof(U16)), pstDevice->stDevRes.pusGenericResources, pusQueryInfo);
		Status = SUCCESS;

		break;

	default:
		UT_ErrorLog(APPITF, "VAPI_SetResourceManager: ucAction %d not supported\n", eResAction);
		Status = VAPI_ERR_INVALID_PARAM;
	}

out:
	UT_Log(APPITF,INFO,"VAPI_SetResourceManager: Exiting status(%d), dev(%d)\n", Status, DevId);
	return Status;
}

/***************************************************************************
* VAPI_SetConnectionResource: The function does the following things -
***************************************************************************/
/*!
*	\n \b Description: \n
*	This API allocates the required resources for a connection according to the passed parameters.\n
*	This ensures that resources for the most resource-intense codec are available for this connection for the whole session.\n
*	This includes a potential switch to a more consuming configuration such as a complex codec (e.g. G.729 or T.38). \n
*	This API must be called before VAPI_CreateConnection or VAPI_AllocateConnection APIs.\n
*	The allocated resources are freed when the VAPI_SetConnectionResource API is called for the corresponding connection ID.  \n
*	VAPI_SetConnectionResource can be called several times for a particular connection ID.\n
*	In this case the latest one is taken in account and the resource pool updated accordingly.\n
*	It is not allowed to free a connection resource if the corresponding connection has not been previously destroyed.\n
*	It is not allowed to decrease the resource of a connection if the corresponding connection takes more resource than the required resource.\n
*	(e.g if a Connection is T38, it is not possible to change the resource to G711_20ms).
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">The device on which resource is to be reserved</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection ID.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucResourceNum</td>
*		<td style="vertical-align: top;">Number of required resources.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pucRequiredResource</td>
*		<td style="vertical-align: top;">Buffer containing the list of required resources.</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*	\li VAPI_ERR_NULL_POINTER_PASSED
*	\li VAPI_ERR_RES_MGR_NOT_ENABLED
*	\li VAPI_ERR_NO_RESOURCE_ALLOC
*	\li VAPI_ERR_NO_RESOURCE
*	\li VAPI_ERR_INVALID_RESOURCE
*	\li VAPI_ERR_UNKNOWN_RESOURCE
*
*	\n \b usage
*	\include allocate_resource.c
*
*	\n \b Commands:
*	List of Comcerto commands sent:
*	\li NONE
*/
VSTATUS VAPI_SetConnectionResource(IN DEVID DevId, IN CONNID ConnId, IN U8 ucResourceNum, IN U8 *pucRequiredResource)
{
	VSTATUS Status = SUCCESS;
	SDevice *pstDevice;
	U16 usReqResource;
	SConnResources *pstConRes;
	SChnl *pstChnl = NULL;
	SResRequest stResRequest;

	UT_Log(APPITF, INFO, "Entered VAPI_SetConnectionResource: dev(%u)\n", ConnId);

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_DeviceGenericCheck(DevId, "VAPI_SetConnectionResource", &pstDevice);

	if (Status != SUCCESS)
		goto out;

	if (!pstDevice->bIsDeviceInitialized)
	{
		UT_ErrorLog(APPITF, "VAPI_SetConnectionResource: device(%u) not initialized\n", DevId);
		Status = VAPI_ERR_DEVICE_NOT_INITIALIZED;
		goto out;
	}

	if (pstDevice->stDevRes.bIsResourceMngr == False)
	{
		UT_ErrorLog(APPITF, "VAPI_SetConnectionResource: resource mngr for device(%u) not enabled\n", DevId);
		Status = VAPI_ERR_RES_MGR_NOT_ENABLED;
		goto out;
	}

	if ((ucResourceNum != 0) && (pucRequiredResource == NULL))
	{
		UT_ErrorLog(APPITF, "VAPI_SetConnectionResource: pucRequiredResource NULL\n");
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		goto out;
	}

	/*Free resource for this ConnID*/
	if(ucResourceNum == 0)
	{
		if (DMGR_GetConnection(ConnId) != NULL)
		{
			UT_ErrorLog(APPITF, "VAPI_SetConnectionResource: Connection %u still exists. Destroy the connection First\n", ConnId);
			Status = VAPI_ERR_INVALID_CONNID;
			goto out;
		}

		pstConRes = (SConnResources *) UT_SearchInAVLTree(pstResTree_g, ConnId);
		if (pstConRes == NULL)
		{	/*This ConnId not found*/
			UT_ErrorLog(APPITF, "VAPI_SetConnectionResource: ConnId (%u) not found\n", ConnId);
			Status = VAPI_ERR_NO_RESOURCE_ALLOC;
			goto out;
		}
		/*Put back the resource to the device resource count*/
		pstDevice->stDevRes.ResourceCount += pstConRes->usConnRes;
		UT_RemFromAVLTree(pstResTree_g, ConnId);
		UT_FreeMem(pstConRes);
	}
	/*We want to allocate or update existing resource */
	else
	{
		/*Check if all Resource IDs passed by the user are valid */
		Status = VRES_IsResourceValid (ucResourceNum, pucRequiredResource);
		if (Status != SUCCESS)
		{
			UT_ErrorLog(APPITF, "VAPI_SetConnectionResource: Invalid Resource Id passed\n");
			goto out;
		}

		/*Get the max reservation for the required configs */
		usReqResource = VRES_GetMaxRes(pstDevice, ucResourceNum, pucRequiredResource);
	
		pstConRes = (SConnResources *) UT_SearchInAVLTree(pstResTree_g, ConnId);
	
		/* No resource allocated yet for this ConnId*/
		if (pstConRes == NULL)
		{
			if (pstDevice->stDevRes.ResourceCount - usReqResource <= 0)
			{
				UT_ErrorLog(APPITF, "VAPI_SetConnectionResource: Not enough ressource on device for connid(%u)\n", ConnId);
				Status = VAPI_ERR_NO_RESOURCE;
			}
			else
			{
				pstConRes = UT_AllocMem(sizeof(SConnResources));
				Status = UT_AddToAVLTree(pstResTree_g, ConnId, pstConRes);
				UT_MutexLock(&pstDevice->stDevRes.mutex_res);
				pstDevice->stDevRes.ResourceCount -= usReqResource;
				UT_MutexUnLock(&pstDevice->stDevRes.mutex_res);
		
				pstConRes->ConnId = ConnId;
				pstConRes->usConnRes = usReqResource;
			}
		}
		else
		/* The resource is already allocated the user wants to change it */
		{
			/* check if there is enough resource for the new config */
			if (pstDevice->stDevRes.ResourceCount + pstConRes->usConnRes - usReqResource <= 0)
			{
				UT_ErrorLog(APPITF, "VAPI_SetConnectionResource: Not enough ressource on device for connid(%u)\n", ConnId);
				Status = VAPI_ERR_NO_RESOURCE;
			}
			else
			{
				/*Check if the a connection associated to this resource already exists 
				Do not use the returned status value if the connection doesn't exist it is not an error*/
				pstChnl = DMGR_GetConnection(ConnId);
				/* If a connection already exist for this ID*/
				if (pstChnl != NULL)
				{
					/* Resource manager is only for VoIP or FoIP connection*/
					if ((pstChnl->usConnType != eVOIP) && (pstChnl->usConnType != eFOIP))
					{
						UT_ErrorLog(APPITF, "VAPI_SetConnectionResource: Resource Manager is only supported for VoIP or FoIP connection\n");
						Status = VAPI_ERR_INVALID_PARAM;
						goto out;
					}

					/* if the current connection is FOIP and takes more resource than the new one return error*/
					if((pstConRes->usConnRes > usReqResource) && (pstChnl->usConnType == eFOIP))
					{
						UT_ErrorLog(APPITF, "VAPI_SetConnectionResource: existing connection(%u) uses more resource(%d) than the required one(%d)\n", ConnId, pstConRes->usConnRes, usReqResource);
						Status = VAPI_ERR_INVALID_RESOURCE;
						goto out;
					}

					/*Check required resource against allocated resources */
					stResRequest.eConnType = pstChnl->usConnType;
					stResRequest.ePartType = pstChnl->ePartType;
					stResRequest.eCodecType = pstChnl->pstChnlParams->eCodecVal;
					stResRequest.ucPacketSize = pstChnl->pstChnlParams->stVoiceOpt.param_4.bits.packet_interval;
					stResRequest.ucVAD = pstChnl->pstChnlParams->stVoiceOpt.param_4.bits.vadtype;
					stResRequest.ucG729_1_rate = 0;
		
					Status = VRES_CheckRes(pstChnl, &stResRequest, pstConRes);
					if (Status != SUCCESS)
						goto out;
		
					/*If the connection exists update the global resource count for this device*/
					UT_MutexLock(&pstDevice->stDevRes.mutex_res);
					pstDevice->stDevRes.ResourceCount += pstConRes->usConnRes;
					pstDevice->stDevRes.ResourceCount -= usReqResource;
					UT_MutexUnLock(&pstDevice->stDevRes.mutex_res);
					/*The ressource for this ConnId exists, just update it*/
					pstConRes->ConnId = ConnId;
					pstConRes->usConnRes = usReqResource;
				}
				else
				{
					/*The ressource for this ConnId exists, just update it*/
					pstConRes->ConnId = ConnId;
					pstConRes->usConnRes = usReqResource;
				}
			}
		}
	}
out:
	return Status;
}

/** @} */ /*resource manager*/
