/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include "ut.h"
#include "vapi.h"
#include "dmgr.h"
#include "vcore.h"
#include "msp.h"
#include "vcore_voip.h"
#include "appitf.h"
#include "resmgr.h"

/*For now these IDs are fixed from 0 to 31 but may change in the future */
U8 valid_res_id [VAPI_NUM_RESOURCES] = {
	eResG711_20,
	eResG711_10,
	eResG711_5,
	eResG726_20,
	eResG726_10,
	eResG726_5,
	eResG729a_20,
	eResG729a_10,
	eResG723_1,
	eResG728,
	eResG729e,
	eResG722,
	eResG722_2_VAD_ON,
	eResG722_2_VAD_OFF,
	eResG729_1_8K,
	eResG729_1_12K,
	eResG729_1_14K,
	eResG729_1_16K,
	eResG729_1_18K,
	eResG729_1_24K,
	eResG729_1_28K,
	eResiLBC,
	eResGSM_FR,
	eResGSM_HR,
	eResGSM_EFR,
	eResGSM_AMR,
	eResCDMA_EVRC,
	eResCDMA_EVRC_B,
	eResCDMA_QCELP8,
	eResCDMA_QCELP13,
	eResT38,
	eResConferencing
};

/*M823x9 model channel density */
U16 M823x9_std_mode_res [VAPI_NUM_RESOURCES] = {160,112,66,98,88,60,72,62,58,30,36,46,28,26,32,28,22,20,20,18,16,36,118,24,50,52,48,32,48,48,54,36};
U16 M823x9_mix_mode_res [VAPI_NUM_RESOURCES] = {136,96,66,98,88,60,67,58,58,30,33,46,28,26,32,28,22,20,20,18,16,33,118,24,50,47,48,32,48,48,54,36};
U16 M823x8_mix_mode_res [VAPI_NUM_RESOURCES] = {121,85,59,87,78,53,59,51,52,27,29,41,25,23,28,25,20,18,18,16,14,29,105,21,44,41,43,28,43,43,48,32};
U16 M823x6_mix_mode_res [VAPI_NUM_RESOURCES] = {91,64,44,65,59,40,44,38,39,20,22,31,19,17,21,19,15,13,13,12,11,22,79,16,33,32,32,21,32,32,36,24};
U16 M823x4_mix_mode_res [VAPI_NUM_RESOURCES] = {60,43,29,44,39,27,29,26,26,13,14,20,12,12,14,12,10,9,9,8,7,14,52,11,22,21,21,14,21,21,24,16};
U16 M823x3_mix_mode_res [VAPI_NUM_RESOURCES] = {45,31,22,33,29,20,22,19,19,10,11,15,9,9,11,9,7,7,7,6,5,11,39,8,17,15,16,11,16,16,18,12};
U16 M823x2_mix_mode_res [VAPI_NUM_RESOURCES] = {31,21,15,22,20,13,15,13,13,7,7,10,6,6,7,6,5,4,4,4,4,7,26,5,11,11,11,7,11,11,12,8};
U16 M823x1_mix_mode_res [VAPI_NUM_RESOURCES] = {15,10,7,11,10,7,7,6,6,3,4,5,3,3,4,3,2,2,2,2,2,4,13,3,6,5,5,4,5,5,6,4};


U16 M82910_mix_mode_res [VAPI_NUM_RESOURCES] = {544,286,168,304,258,156,269,232,216,94,115,176,80,80,128,112,88,80,72,72,64,108,170,88,144,173,136,80,152,144,200,192};

/****************************************************************************
 * VRES_GetMaxRes : The function does the following things -
 ***************************************************************************/
/*! 
 *  This function returns the mas weigh resource out of the list passed.
 *
 *  \return VSTATUS 
 *
 *  \param ucNumOfRes		Number of resources
 *  \param pucRequiredRes	Buffer containing the resources
 */
U16 VRES_GetMaxRes(IN SDevice *pstDevice, IN U8 ucNumOfRes, IN U8 *pucRequiredRes) 
{
	int i;
	U16 temp = 0;

	for (i = 0; i < ucNumOfRes; i++)
		temp = UT_MAX(temp, pstDevice->stDevRes.pusCurrentResources[pucRequiredRes[i]]);

	return temp;
}

/****************************************************************************
 * VRES_InitDeviceRes : The function does the following things -
 ***************************************************************************/
/*! 
 *  This function initializes the resource weight for the current device.
 *
 *  \return VSTATUS 
 *
 *  \param usResourceCount	Global ressource count for this device model
 *  \param usNumOfRes		Number of resources
 *  \param pusReferenceRes	Buffer containing the reference resources
 *  \param pusCurrentResources	Buffer containing the current resources for this device
 */
void VRES_InitDeviceRes(int usResourceCount, U16 usNumOfRes, U16 *pusReferenceRes, U16 *pusCurrentResources)
{
	int i;
	for (i = 0; i < usNumOfRes; i ++)
		pusCurrentResources[i] = (usResourceCount/pusReferenceRes[i]);
}

/****************************************************************************
 * VRES_GetDeviceRes : The function does the following things -
 ***************************************************************************/
/*! 
 *  This function calculate the current available channels type to the buffer passed by the host application.
 *
 *  \return VSTATUS 
 *
 *  \param usResourceCount	Global ressource count for this device model
 *  \param usNumOfRes		Number of resources
 *  \param pusReferenceRes	Buffer containing the reference resources for the device
 *  \param pusCurrentResources	Buffer to save the current ressources in
 */
void VRES_GetDeviceRes(int usResourceCount, U16 usNumOfRes, U16 *pusReferenceRes, U16 *pusUserData)
{
	int i;
	for (i = 0; i < usNumOfRes; i ++)
		pusUserData[i] = (usResourceCount * pusReferenceRes[i])/COMCERTO_RESOURCES;
}

/****************************************************************************
 * VRES_IsResourceValid : The function does the following things -
 ***************************************************************************/
/*! 
 *  This function checks if the passed ResourceId are supported.
 *
 *  \return VSTATUS 
 *
 *  \param ucResourceNum	Number of passed resource ID
 *  \param pucRequiredResource	Array of resource ID passed
 */
VSTATUS VRES_IsResourceValid (U8 ucResourceNum, U8 *pucRequiredResource)
{
	int i, j = 0;
	VSTATUS status = VAPI_ERR_INVALID_RESOURCE;

	UT_Log(VCORE, INFO, "VRES_IsResourceValid: Entering n");

	/*For each passed res ID*/
	for (j = 0; j < ucResourceNum; j ++)
	{
		/*Check if one ID matches in the global res ID array */
		for (i = 0; i < sizeof(valid_res_id); i ++)
		{
			if (pucRequiredResource[j] == valid_res_id[i])
			{
				/* Stop ASA a valid res id is find*/
				status = SUCCESS;
				break;
			}
		}
		/* If a inalid res id is find stop */
		if (status == VAPI_ERR_INVALID_RESOURCE)
		{
			UT_ErrorLog(APPITF, "VRES_IsResourceValid: invalid ResId found\n");
			break;
		}
	}
	return status;
}

/****************************************************************************
 * VRES_CheckRes : The function does the following things -
 ***************************************************************************/
/*! 
 *  This function checks if the allocated resources for the connection matches the connection configuration.
 *
 *  \return VSTATUS 
 *
 *  \param pstChnl	Connection on which to check ressource
 *  \param pstConRes	Resource allocated for this connection
 */
VSTATUS VRES_CheckRes(IN SChnl * pstChnl, IN SResRequest *pstResRequest, SConnResources *pstConRes)
{
	U16 usRequiredRes = 0;

	UT_Log(VCORE, INFO, "VRES_CheckRes: Entering conn(%u) \n", pstChnl->ConnId);
	/* If the resource manager is not enable no check required */
	if (pstChnl->pstDev->stDevRes.bIsResourceMngr == False)
		return SUCCESS;

	/*If it is a conference participant */
	if (pstResRequest->ePartType != eNOTPART)
	{
		usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResConferencing]);
	}
	/*If it is a T38 */
	else if (pstResRequest->eConnType == eFOIP)
	{
		usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResT38]);
	}
	else if (pstResRequest->eConnType == eVOIP)
	{
		/*Compute required res for codec / packet size/ VAD option */
		switch(pstResRequest->eCodecType)
		{
			case eG711_ULAW_PCM:
			case eG711_ALAW_PCM:
				switch(pstResRequest->ucPacketSize)
				{
					case 20:
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG711_20]);
						break;
	
					case 10:
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG711_10]);
						break;
	
					case 5:
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG711_5]);
						break;
	
					default: /*For other packet size count it as for 20ms*/
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG711_20]);
					break;
				}
				break;
	
			case eG726_32:
			case eG726_16_audio:
			case eG726_24_audio:
			case eG726_40_audio:
				switch(pstResRequest->ucPacketSize)
				{
					case 20:
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG726_20]);
						break;
	
					case 10:
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG726_10]);
						break;
	
					case 5:
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG726_5]);
						break;
	
					default: /*For other packet size count it as for 20ms*/
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG711_20]);
					break;
				}
				break;
	
			case eG729_A:
				switch(pstResRequest->ucPacketSize)
				{
					case 20:
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG729a_20]);
					break;
	
					case 10:
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG729a_10]);
					break;
	
					default: /*For other packet size count it as for 20ms*/
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG711_20]);
					break;
				}
				break;
	
			case eG723_1:
				usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG723_1]);
				break;
	
			case eG728_Audio:
				usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG728]);
				break;
	
			case eG729_EG:
				usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG729e]);
				break;
	
			case eG722:
				usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG722]);
				break;
	
			case eAMR_WB_ba: /*G.722.2*/
				if (pstResRequest->ucVAD == VOIP_VCEOPT_VADTYPE_DISABLE)
					usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG722_2_VAD_OFF]);
				else
					usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG722_2_VAD_ON]);

			break;
	
			case eG729_1:
				switch(pstResRequest->ucG729_1_rate)
				{
					case G729_WB_MODE_MBS_MBS_8_KBPS:			/* G729_1_8K */
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG729_1_8K]);
						break;
	
					case G729_WB_MODE_MBS_MBS_12_KBPS:			/* G729_1_12K */
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG729_1_12K]);
						break;
	
					case G729_WB_MODE_MBS_MBS_14_KBPS:			/* G729_1_14K */
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG729_1_14K]);
						break;
	
					case G729_WB_MODE_MBS_MBS_16_KBPS:			/* G729_1_16K */
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG729_1_16K]);
						break;
	
					case G729_WB_MODE_MBS_MBS_18_KBPS:			/* This 3 rates cost the same as G729_1_18K */
					case G729_WB_MODE_MBS_MBS_20_KBPS:
					case G729_WB_MODE_MBS_MBS_22_KBPS:
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG729_1_18K]);
						break;
	
					case G729_WB_MODE_MBS_MBS_24_KBPS:			/* This 2 rates cost the same as G729_1_24K */
					case G729_WB_MODE_MBS_MBS_26_KBPS:
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG729_1_24K]);
						break;
	
					case G729_WB_MODE_MBS_MBS_28_KBPS:			/* G729_1_28K*/
					case G729_WB_MODE_MBS_MBS_30_KBPS:
					case G729_WB_MODE_MBS_MBS_32_KBPS:
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG729_1_28K]);
						break;
	
					default:
						usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResG729_1_8K]);
						break;
				}
				break;
	
			case eI_LBC_15K:
			case eI_LBC_13K:
				usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResiLBC]);
				break;
	
			case eGSM_FR_Audio:
				usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResGSM_FR]);
				break;
	
			case eGSM_EFR:
				usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResGSM_EFR]);
				break;
	
			case eGSM_HR:
				usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResGSM_HR]);
				break;
	
			case eGSM_AMR_BWE:
				usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResGSM_AMR]);
				break;
	
			case eEVRC:
				usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResCDMA_EVRC]);
				break;
	
			case e4GV: /* (EVRC-B)*/
				usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResCDMA_EVRC_B]);
				break;
	
			case eQCELP_13K:
				usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResCDMA_QCELP8]);
				break;
	
			case eQCELP_8K:
				usRequiredRes = (COMCERTO_RESOURCES/pstChnl->pstDev->stDevRes.pusGenericResources[eResCDMA_QCELP13]);
				break;
	
			/*Cases for which we do not calculate the required ressource */
			case eClear_Channel:
			case ePassthrough_ULAW:
			case ePassthrough_ALAW:
			case eAMR:
			case eCNF_SEC:
			case eSMV:
			case eCRBT_ADPCM:
			case eAMR_WB:
			case eG729_MSPD:
			case eCESoPSN:
			case eCSD_BS30T_64k:
			case eCSD_BS30T_32k:
			case eCSD_BS30T_33k6:
			case eCSD_BS30T_28k8:
			case eCSD_BS20NT_57k6:
			case eCSD_BS20NT_28k8:
			case eCSD_BS20NT_14k4:
			case eCCISoIP:
			case eAMR_IF1:
			case eAMR_IF2:
			default:
				UT_ErrorLog(APPITF, "VRES_CheckRes: Channel configuration not supported for by resource manager for conn (%u)\n", pstConRes->ConnId);
				return VAPI_ERR_UNKNOWN_RESOURCE;
			break;
		}
	}

	if (pstConRes->usConnRes < usRequiredRes)
	{
		UT_ErrorLog(APPITF, "VRES_CheckRes: not enough ressource allocated for conn (%u)\n", pstConRes->ConnId);
		return VAPI_ERR_INVALID_RESOURCE;
	}
	else
		return SUCCESS;

}
