/*! \file mspcmds.c 
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include <assert.h>

#include "ut.h"
#include "vapi.h"
#include "param_check.h"
#include "vcore.h"
#include "gtl.h"
#include "msp.h"
#include "hwcfg.h"
#include "vcore_voip.h"

extern SToneGeneric astToneGeneric_g[];
extern SCidGen SCidGen_g;
extern SCidDet SCidDet_g;
extern SMutex stMutexCIDGen_g;
extern SMutex stMutexCIDDet_g;

/****************************************************************************
 * VCORE_SetEchoCan : The function does the following things -
 ***************************************************************************/
U8 VCORE_SetEchoCan(IN SChnl * pstChnl)
{
	if (pstChnl->pstDev->usSpuFeatureMask & SPU_FEATURE_DFEC_MASK)
	{
		return VDEV_SendVoipDfeCancel(pstChnl, &((SVoIPChnlParams *) pstChnl->pstChnlParams)->stDfeCan);
	}
        else
	{
		return VDEV_SendVoipEchoCancel(pstChnl, &((SVoIPChnlParams *) pstChnl->pstChnlParams)->stEchoCan);
	}
}

/****************************************************************************
 * VDEV_CreateChannel : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param eConnType Connection type (eg eVOIP)
 *  \param usTDMTimeSlot The TDM timeslot to be requested
 */
VSTATUS VDEV_CreateChannel(IN SChnl * pstChnl, IN SChnl * pstChannelInfo)
{
#define NUM_OF_PARAMS 5
	VSTATUS Status = SUCCESS;
	gtl_msg_t stMsg;
	/* allocate for the max case */
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_SUPVSR_CREATE_CHANNEL);

	/* build and add the API parameters */
	if(pstChannelInfo->usMode == eWideBand)
		Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(pstChannelInfo->usConnType | eWideBand << 15), 
									UT_CPU2LE16(pstChannelInfo->usTimeSlot),
									UT_CPU2LE16(pstChannelInfo->usExtTimeSlot[0]),
									UT_CPU2LE16(pstChannelInfo->usExtTimeSlot[1]),
									UT_CPU2LE16(pstChannelInfo->usExtTimeSlot[2]));
	else
		Status = VCORE_SetPayloadMsg(&stMsg, 2, UT_CPU2LE16(pstChannelInfo->usConnType), UT_CPU2LE16(pstChannelInfo->usTimeSlot));

	if (Status != SUCCESS)
		goto out;

	/* Send the command to the device */
	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_RecoverChannel : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param eConnType Connection type (eg eVOIP)
 *  \param usTDMTimeSlot The TDM timeslot to be requested
 */
VSTATUS VDEV_RecoverChannel(IN SChnl * pstChnl, IN U16 usTDMTimeSlot)
{
#define NUM_OF_PARAMS 1
	VSTATUS Status = SUCCESS;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY, FC_SUPVSR_CREATE_CHANNEL);

	/* build and add the API parameters */
	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(usTDMTimeSlot));
	if (Status != SUCCESS)
		goto out;

	/* Send the command to the device */
	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_DestroyChannel : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param ChnlId The channel id to be destroyed
 */
VSTATUS VDEV_DestroyChannel(IN SChnl * pstChnl, IN CHNLID ChnlId)
{
#define NUM_OF_PARAMS 1
	VSTATUS Status = SUCCESS;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_SUPVSR_DESTROY_CHANNEL);

	/* build and add the API parameters */
	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(ChnlId));
	if (Status != SUCCESS)
		goto out;

	/* Send the command to the device */
	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SetVceopt : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param stMsg the pre allocated message to be filled with VCEOPT parameters
 *  \param pstVceParams Parameters to be sent in VCEOPT command
 */
VSTATUS VDEV_SetVceopt(gtl_msg_t *stMsg, IN struct _VOIP_VCEOPT *pstVceParams)
{
	VSTATUS Status = SUCCESS;

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_VCEOPT);

	/* build and add the API parameters */
	Status = VCORE_SetPayloadMsg(stMsg, 5, UT_CPU2LE16(pstVceParams->param_4.word),
						UT_CPU2LE16(pstVceParams->param_5.word),
						UT_CPU2LE16(pstVceParams->vad_tune),
						UT_CPU2LE16(pstVceParams->param_7.word),
						UT_CPU2LE16(pstVceParams->param_8.word));
	if (Status != SUCCESS)
		goto out;

out:
	return Status;
}


/****************************************************************************
 * VDEV_SendVceopt : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 */
VSTATUS VDEV_SendVceopt(IN SChnl * pstChnl, IN struct _VOIP_VCEOPT *pstVceParams)
{
#define NUM_OF_PARAMS 5

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build the message */
	Status = VDEV_SetVceopt(&stMsg, pstVceParams);
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SetVoipVopena : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status if the parameters are not valid\n
 *
 *  \param stMsg the pre allocated message to be filled with VCEOPT parameters
 *  \param pstVceParams Parameters to be sent in VCEOPT command
 */
VSTATUS VDEV_SetVoipVopena(gtl_msg_t *stMsg, IN struct _VOIP_VOPENA *pstVopenaParams)
{
	VSTATUS Status = SUCCESS;

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_VOPENA);

	/*If RTP Enabled */
	if (pstVopenaParams->mode == VOIP_VOPENA_MODE_ENABLE_RTP)
	{

		switch (pstVopenaParams->param_5.bits.rtp_cc)
		{
		case 1:
			/* build and add the API parameters */
			Status = VCORE_SetPayloadMsg(stMsg, 9,
					UT_CPU2LE16(pstVopenaParams->mode),
					UT_CPU2LE16(pstVopenaParams->param_5.word),
					UT_CPU2LE16(pstVopenaParams->rtp_seq),
					UT_CPU2LE16(pstVopenaParams->rtp_timestamp >> 16),
					UT_CPU2LE16(pstVopenaParams->rtp_timestamp),
					UT_CPU2BE16(pstVopenaParams->rtp_ssrc >> 16),
					UT_CPU2BE16(pstVopenaParams->rtp_ssrc),
					UT_CPU2BE16(pstVopenaParams->rtp_csrc1 >> 16),
					UT_CPU2BE16(pstVopenaParams->rtp_csrc1));

			if (Status != SUCCESS)
				goto out;
			break;

		case 2:
			/* build and add the API parameters */
			Status = VCORE_SetPayloadMsg(stMsg, 11,
					UT_CPU2LE16(pstVopenaParams->mode),
					UT_CPU2LE16(pstVopenaParams->param_5.word),
					UT_CPU2LE16(pstVopenaParams->rtp_seq),
					UT_CPU2LE16(pstVopenaParams->rtp_timestamp >> 16),
					UT_CPU2LE16(pstVopenaParams->rtp_timestamp),
					UT_CPU2BE16(pstVopenaParams->rtp_ssrc >> 16),
					UT_CPU2BE16(pstVopenaParams->rtp_ssrc),
					UT_CPU2BE16(pstVopenaParams->rtp_csrc1 >> 16),
					UT_CPU2BE16(pstVopenaParams->rtp_csrc1),
					UT_CPU2BE16(pstVopenaParams->rtp_csrc2 >> 16),
					UT_CPU2BE16(pstVopenaParams->rtp_csrc2));

			if (Status != SUCCESS)
				goto out;
			break;

		default:
			/* build and add the API parameters */
			Status = VCORE_SetPayloadMsg(stMsg, 7,
					UT_CPU2LE16(pstVopenaParams->mode),
					UT_CPU2LE16(pstVopenaParams->param_5.word),
					UT_CPU2LE16(pstVopenaParams->rtp_seq),
					UT_CPU2LE16(pstVopenaParams->rtp_timestamp >> 16),
					UT_CPU2LE16(pstVopenaParams->rtp_timestamp),
					UT_CPU2BE16(pstVopenaParams->rtp_ssrc >> 16),
					UT_CPU2BE16(pstVopenaParams->rtp_ssrc));

			if (Status != SUCCESS)
				goto out;
		}
	}
	else
		Status = VCORE_SetPayloadMsg(stMsg, 1, UT_CPU2LE16(pstVopenaParams->mode));

out:
	return Status;

}

/****************************************************************************
 * VDEV_SendVoipVopena : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstVopenaParams Parameters to be sent in VOPENA command
 */
VSTATUS VDEV_SendVoipVopena(IN SChnl * pstChnl, IN struct _VOIP_VOPENA *pstVopenaParams)
{
#define NUM_OF_PARAMS 11

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build the message */
	Status = VDEV_SetVoipVopena(&stMsg, pstVopenaParams);
	if (Status != SUCCESS)
		goto err;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

err:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_VopDisable : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param 
 */
VSTATUS VDEV_VopDisable(IN SChnl * pstChnl)
{
#define NUM_OF_PARAMS 1
	gtl_msg_t stMsg;
	VSTATUS Status;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_VOPENA);

	Status = VCORE_SetPayloadMsg(&stMsg, 1, UT_CPU2LE16(VOIP_VOPENA_MODE_DISABLE));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

 
/****************************************************************************
* VDEV_SetVoipDfeCancel : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param stMsg the pre allocated message to be filled with ECHO_CANCEL parameters
 *  \param pstDCanParams Parameters to be sent in ECHO_CANCEL command
 */
VSTATUS VDEV_SetVoipDfeCancel(gtl_msg_t *pstMsg, IN struct _VOIP_DFECAN *pstDCanParams)
{

	VSTATUS Status;

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(pstMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_DFECAN);

	Status = VCORE_SetPayloadMsg(pstMsg, 1, UT_CPU2LE16(pstDCanParams->param_4.word), 0);

	return Status;
}

/****************************************************************************
* VDEV_SendVoipDfeCancel : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstDCanParams Parameters to be sent in ECHO_CANCEL command
 */
VSTATUS VDEV_SendVoipDfeCancel(IN SChnl *pstChnl, IN struct _VOIP_DFECAN *pstDCanParams)
{
#define NUM_OF_PARAMS 2

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build the message */
	Status = VDEV_SetVoipDfeCancel(&stMsg, pstDCanParams);
	if (Status != SUCCESS)
		goto err;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

err:
	return Status;
#undef NUM_OF_PARAMS
}


/****************************************************************************
* VDEV_SetVoipEchoCancel : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param stMsg the pre allocated message to be filled with ECHO_CANCEL parameters
 *  \param pstECanParams Parameters to be sent in ECHO_CANCEL command
 */
VSTATUS VDEV_SetVoipEchoCancel(gtl_msg_t *pstMsg, IN struct _VOIP_ECHOCAN *pstECanParams)
{
	VSTATUS Status;

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(pstMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_ECHOCAN);

	Status = VCORE_SetPayloadMsg(pstMsg, 1, UT_CPU2LE16(pstECanParams->param_4.word), 0);

	return Status;
}

/****************************************************************************
* VDEV_SendVoipEchoCancel : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstECanParams Parameters to be sent in ECHO_CANCEL command
 */
VSTATUS VDEV_SendVoipEchoCancel(IN SChnl *pstChnl, IN struct _VOIP_ECHOCAN *pstECanParams)
{
#define NUM_OF_PARAMS 2

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build the message */
	Status = VDEV_SetVoipEchoCancel(&stMsg, pstECanParams);
	if (Status != SUCCESS)
		goto err;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

err:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
* VDEV_SetVoipJbopt : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param stMsg the pre allocated message to be filled with JBOPT parameters
 *  \param pstJbopt Parameters to be sent in JBOPT command
 */
VSTATUS VDEV_SetVoipJbopt(gtl_msg_t *pstMsg, IN struct _VOIP_JBOPT *pstJbopt)
{
	VSTATUS Status;

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(pstMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_JBOPT);

	Status = VCORE_SetPayloadMsg(pstMsg, 6, UT_CPU2LE16(pstJbopt->delay_min),
					UT_CPU2LE16(pstJbopt->delay_max),
					UT_CPU2LE16(pstJbopt->delay_init),
					UT_CPU2LE16(pstJbopt->ap),
					UT_CPU2LE16(pstJbopt->param_8.word),
					UT_CPU2LE16(pstJbopt->deletion_threshold));

	return Status;
}

/****************************************************************************
* VDEV_SendVoipJbopt : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstJbopt Parameters to be sent in JBOPT command
 */
VSTATUS VDEV_SendVoipJbopt(IN SChnl *pstChnl, IN struct _VOIP_JBOPT *pstJbopt)
{

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + sizeof(struct _VOIP_JBOPT) + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build the message */
	Status = VDEV_SetVoipJbopt(&stMsg, pstJbopt);
	if (Status != SUCCESS)
		goto err;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

err:
	return Status;

}

/****************************************************************************
* VDEV_SetVoipDgain : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param stMsg the pre allocated message to be filled with DGAINOPT parameters
 *  \param pstDgainParams Parameters to be sent in DGAINOPT command
 */
VSTATUS VDEV_SetVoipDgain(gtl_msg_t *pstMsg, IN struct _VOIP_DGAIN *pstDgainParams)
{
	VSTATUS Status;

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(pstMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_DGAIN);

	Status = VCORE_SetPayloadMsg(pstMsg, 2, UT_CPU2LE16(pstDgainParams->packet_to_pcm_gain),
					UT_CPU2LE16(pstDgainParams->pcm_to_packet_gain));

	return Status;
}

/****************************************************************************
* VDEV_SendVoipDgain : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstDgainParams Parameters to be sent in DGAINOPT command
 */
VSTATUS VDEV_SendVoipDgain(IN SChnl *pstChnl, IN struct _VOIP_DGAIN *pstDgainParams)
{
#define NUM_OF_PARAMS 2

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build the message */
	Status = VDEV_SetVoipDgain(&stMsg, pstDgainParams);
	if (Status != SUCCESS)
		goto err;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

err:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
* VDEV_SetVoipDtmfOpt : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param stMsg the pre allocated message to be filled with DTMFOPT parameters
 *  \param pstDtmpOpt Parameters to be sent in DTMFOPT command
 */
VSTATUS VDEV_SetVoipDtmfOpt(gtl_msg_t *pstMsg, IN struct _VOIP_DTMFOPT * pstDtmpOpt)
{
	VSTATUS Status;

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(pstMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_DTMFOPT);

	Status = VCORE_SetPayloadMsg(pstMsg, 2, UT_CPU2LE16(pstDtmpOpt->param_4.word),
					UT_CPU2LE16(pstDtmpOpt->dtmf_pt));

	return Status;
}

/****************************************************************************
* VDEV_SendVoipDtmfOpt : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstDtmpOpt Parameters to be sent in DTMFOPT command
 */
VSTATUS VDEV_SendVoipDtmfOpt(IN SChnl *pstChnl, IN struct _VOIP_DTMFOPT * pstDtmpOpt)
{
#define NUM_OF_PARAMS 2
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build the message */
	Status = VDEV_SetVoipDtmfOpt(&stMsg, pstDtmpOpt);
	if (Status != SUCCESS)
		goto err;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

err:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VCORE_VoipLoopback : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl The channel on which MSP command will be sent
 *  \param usDestChnlId Destination Channel ID
 *  \param eLoobackType Loopback type 
 */
VSTATUS VDEV_VoIPLoopback(IN SChnl * pstChnl, IN U16 usDestChnlId, IN ELoopbackType eLoobackType)
{
#define NUM_OF_PARAMS 2

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];
	U16 p1;

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_LOOPBACK);

	if (eLoobackType == eREMOVE_CHNL_LOOPBACK
	    || eLoobackType == eREMOVE_INTER_CHNL_POSTENCAPS_LOOPBACK
	    || eLoobackType == eREMOVE_INTER_CHNL_TDM_LOOPBACK)
	{
		p1 = 0;
	}
	else
		p1 = eLoobackType;

	if (eLoobackType == eINTER_CHNL_POST_ENCAPS || eLoobackType == eINTER_CHNL_TDM)
		Status = VCORE_SetPayloadMsg(&stMsg, 2, UT_CPU2LE16(p1), UT_CPU2LE16(usDestChnlId));
	else
		Status = VCORE_SetPayloadMsg(&stMsg, 1, UT_CPU2LE16(p1));

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_ThcModeEnable : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl The channel on which MSP command will be sent
 */
VSTATUS VDEV_ThcModeEnable(SChnl * pstChnl)
{
#define NUM_OF_PARAMS 1

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_THC_MODE_ENABLE);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, 0);
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_ThcRedirectRx : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl The channel on which MSP command will be sent
 *  \param usTDMTimeSlot The timeslot that is to be connected by hairpinning
 */
VSTATUS VDEV_ThcRedirectRx(SChnl * pstChnl, U16 usTDMTimeSlot)
{
#define NUM_OF_PARAMS 1

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_THC_REDIRECT_RX);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(usTDMTimeSlot));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SetEthMacAddr
 ***************************************************************************/
VSTATUS VDEV_SetEthMacAddr(gtl_msg_t *pstMsg, IN U16 usMSPChnlId,  IN U8 * pastSrcEthMac, IN U8 * pastDestEthMac, IN U16 usVlanId)
{
/* make sure we have room for VLAN (if any)*/
#define NUM_OF_PARAMS 10 

	U8 *pusFifoParam;
	VSTATUS Status;

	/* for the device simply set the eth header accordingly the parameters*/
	if (usMSPChnlId == SUPV_CHANNEL)
	{
		VCORE_SetupCmdHdrMsg(pstMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_SET_ETH_HDR);

		if (usVlanId == 0)
			Status = VCORE_SetPayloadMsg(pstMsg, 8, UT_CPU2LE16(SET_ETH_HDR_PROTOCOL_IP), 
							0, 0, 0, 0, 0, 0, UT_CPU2BE16(ETH_P_IP));
		else
			Status = VCORE_SetPayloadMsg(pstMsg, 10, UT_CPU2LE16(SET_ETH_HDR_PROTOCOL_IP), 
							0, 0, 0, 0, 0, 0, UT_CPU2BE16(ETH_P_8021Q),
							UT_CPU2BE16(usVlanId), UT_CPU2BE16(ETH_P_IP));
	}
	/* conenction level eth header*/
	else
	{
		VCORE_SetupCmdHdrMsg(pstMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_SET_ETH_HDR_CHAN);

		/* this means the VLAN param was NULL for this connection, so keep the device setting*/
		if (usVlanId == 0)
			Status = VCORE_SetPayloadMsg(pstMsg, 8, UT_CPU2LE16(SET_ETH_HDR_PROTOCOL_IP), 
							0, 0, 0, 0, 0, 0, UT_CPU2BE16(ETH_P_IP));
		else
			Status = VCORE_SetPayloadMsg(pstMsg, 10, UT_CPU2LE16(SET_ETH_HDR_PROTOCOL_IP), 
							0, 0, 0, 0, 0, 0, UT_CPU2BE16(ETH_P_8021Q),
							UT_CPU2BE16(usVlanId), UT_CPU2BE16(ETH_P_IP));
	}

	/* For this particular command the MAC addresses are comming from a buffer in the right endianess
	we fill them fill dummy 0 and overwrite them by a copy */
	if (Status != SUCCESS)
		goto out;

	/* This pointer points to the beginning of the parameters in the fifo */
	pusFifoParam = (U8*) ((U8*)pstMsg->fifo + sizeof(SApiHdr));

	UT_MemCopy(&pusFifoParam[2], pastDestEthMac, 6);
	UT_MemCopy(&pusFifoParam[8], pastSrcEthMac, 6);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SendEthMacAddr : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl          The channel on which MSP command will be sent
 *  \param pastSrcEthMac    Pointer to the source ethernet Mac Address
 *  \param pastDestEthMac   Pointer to the destination ethernet Mac Address
 *
 */
VSTATUS VDEV_SendEthMacAddr(IN SChnl * pstChnl, IN U8 * pastSrcEthMac, IN U8 * pastDestEthMac, IN U16 usVlanId)
{
/* make sure we have room for VLAN (if any)*/
#define NUM_OF_PARAMS 10

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	Status = VDEV_SetEthMacAddr(&stMsg, pstChnl->usMSPChnlId, pastSrcEthMac, pastDestEthMac, usVlanId);

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_GetEthMacAddr : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  Get the MAC address and the payload type of all currently registered Ethernet Layer Protocols.
 *
 */
VSTATUS VDEV_GetEthMacAddr(IN SChnl * pstChnl)
{
#define NUM_OF_PARAMS 0

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg,  CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY, FC_GET_ETH_HDR);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_GetDevIP : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  Get the IP address of the Mindspeed Comcerto Device.
 *
 */
VSTATUS VDEV_GetDevIP(IN SChnl * pstChnl)
{
#define NUM_OF_PARAMS 0

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg,  CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY, FC_IP_ADDRESS);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_GetCsmeMulti
 ***************************************************************************/
VSTATUS VDEV_GetCsmeMulti(IN SChnl * pstChnl)
{
#define NUM_OF_PARAMS 0

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg,  CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY, FC_CSME_MULTI_CMD);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_GetSpuFeature
 ***************************************************************************/
VSTATUS VDEV_GetSpuFeature(IN SChnl * pstChnl)
{
#define NUM_OF_PARAMS 0

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg,  CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY, FC_SPU_FEATURES_CONTROL);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_IpServiceConfig : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl      The channel on which MSP command will be sent
 *  \param ucChksumVer  Check sum verification value.
 *  \param ucEncaps     Encapsulation
 *
 */
VSTATUS VDEV_IpServiceConfig(IN SChnl * pstChnl, IN U8 ucChksumVer, IN U8 ucEncaps)
{
#define NUM_OF_PARAMS 1

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];
	U16 p1;

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_IP_SERVICE_CONFIG);

	p1 = ((U16) (ucChksumVer) | (U16) (ucEncaps << 8));
	p1 = p1 & 0x0301;

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(p1));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SetIpAddr : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl  The channel on which MSP command will be sent
 *  \param uiIpAddr IP Address to be set    
 *
 */
VSTATUS VDEV_SetIpAddr(IN SChnl * pstChnl, IN U32 uiIpAddr)
{
#define NUM_OF_PARAMS 2

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];
	void *vIpAddr  =  &uiIpAddr;

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_IP_ADDRESS);

	/* For this particular command the IP address is comming in the right endianess*/
	Status = VCORE_SetPayloadMsgFromBuffer(&stMsg, NUM_OF_PARAMS,  (U16 *) vIpAddr);
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SetIpAddrList : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl          The channel on which MSP command will be sent
 *  \param pauiIpAddr       Pointer to the array of Src IP Address to be set
 *  \param ucNumOfIpAddr    Number of source IP addresses
 *
 */
VSTATUS VDEV_SetIpAddrList(IN SChnl * pstChnl, IN U32 * pauiIpAddr, IN U8 ucNumOfIpAddr)
{
	VSTATUS Status;
	gtl_msg_t stMsg;
	/* We can have max 10 IP addresses in the list*/
	U8 ausFifo[sizeof(SApiHdr) + (10 * sizeof(U32)) + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_MULTI_IP_ADDR_LIST);

	VCORE_SetPayloadMsgFromBuffer(&stMsg, ucNumOfIpAddr * 2, (U16 *)pauiIpAddr);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

	return Status;
}

/****************************************************************************
 * VDEV_EnableARP : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl  The channel on which MSP command will be sent
 *  \param uiIpAddr IP Address to be set    
 *
 */
VSTATUS VDEV_EnableARP(IN SChnl * pstChnl)
{
#define NUM_OF_PARAMS 1

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_ARP_SERVICE_CONFIG);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(ARP_SERVICE_CONFIG_ARP_REPLY_ENABLE));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_EnableICMP : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl  The channel on which MSP command will be sent
 *  \param uiIpAddr IP Address to be set    
 *
 */
VSTATUS VDEV_EnableICMP(IN SChnl * pstChnl)
{
#define NUM_OF_PARAMS 1

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_ICMP_SERVICE_CONFIG);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS,
				 UT_CPU2LE16(0xff00 | ICMP_SERVICE_CONFIG_ICMP_ECHO_REPLY_REPORT_ENABLE));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SetChannelIpHdrChnl : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl   The channel on which MSP command will be sent
 *
 */
VSTATUS VDEV_SetIpHdrChnl(gtl_msg_t *pstMsg, IN struct _SET_IP_HDR_CHANNEL *pstIpUdpInfo, IN SChnl * pstChnl)
{
#define NUM_OF_PARAMS 15

	VSTATUS Status;
	U16 IpParams [15];

	VCORE_SetupCmdHdrMsg(pstMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_SET_IP_HDR_CHANNEL);

	pstIpUdpInfo->param_4.bits.media_sub_layer = pstChnl->MediaSubLayer;

	IpParams [0] = UT_CPU2LE16(pstIpUdpInfo->param_4.word);
	IpParams [1] = UT_CPU2LE16(pstIpUdpInfo->param_5.word);
	IpParams [2] = UT_CPU2LE16(pstIpUdpInfo->ip_len);
	IpParams [3] = UT_CPU2LE16(pstIpUdpInfo->ip_id);
	IpParams [4] = UT_CPU2LE16(pstIpUdpInfo->ip_off);
	IpParams [5] = UT_CPU2LE16(pstIpUdpInfo->param_9.word);
	IpParams [6] = UT_CPU2LE16(pstIpUdpInfo->ip_sum);

	UT_MemCopy(&IpParams [7], &pstIpUdpInfo->ip_src, sizeof(U32));
	UT_MemCopy(&IpParams [9], &pstIpUdpInfo->ip_dst, sizeof(U32));

	IpParams [11] = pstIpUdpInfo->uh_sport;
	IpParams [12] = pstIpUdpInfo->uh_dport;

	IpParams [13] = UT_CPU2LE16(pstIpUdpInfo->uh_ulen),
	IpParams [14] = UT_CPU2LE16(pstIpUdpInfo->uh_sum);

	Status = VCORE_SetPayloadMsgFromBuffer(pstMsg, NUM_OF_PARAMS, IpParams);

	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SendChannelIpHdrChnl : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl   The channel on which MSP command will be sent
 *
 */
VSTATUS VDEV_SendIpHdrChnl(IN SChnl * pstChnl, IN struct _SET_IP_HDR_CHANNEL *pstIpUdpInfo)
{
#define NUM_OF_PARAMS 15

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	if (pstChnl->usConnType != eVOIP && pstChnl->usConnType != eFOIP && pstChnl->usConnType != eIMSOTDM)
	{
		UT_ErrorLog(VCORE, "VDEV_SetIpHdrChnl: Invalid Channel type(u%)\n", pstChnl->usConnType);
		Status = VAPI_ERR_UNSUPP_FEATURE;
		goto out;
	}

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	Status = VDEV_SetIpHdrChnl (&stMsg, pstIpUdpInfo, pstChnl);

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_PLTypeSet : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl   The channel on which MSP command will be sent
 *  \param eCodec    CODEC value
 *  \param ucPtVal   The payload type value to be set
 *  \param ucTempDir The direction field can be (TX, RX, BOTH) to 
 *                   support asymmetrical payload types.
 */
VSTATUS VDEV_PLTypeSet(IN SChnl * pstChnl, IN U16 usPTIndex, IN U8 ucPtVal, IN U8 ucTempDir)
{
#define NUM_OF_PARAMS 1

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	if (ucTempDir == eRx)
		VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_PTSETRXOVR);
	else
	{
		if (pstChnl->MediaSubLayer == eAudioSubLayer)
			VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_PTSET);
		else
			VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VIDEO_PTSET);
	}

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS,
					UT_CPU2LE16((((U16) usPTIndex << 8) | (U16) (ucPtVal))));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SendNteCmd : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl              The channel on which MSP command will be sent
 *  \param uiNtePyLd            Payload of the NTE packet
 *  \param usOverRideBitField   The host can optionally override the 
 *                              redundancy interval time, disable redundancy,
 *                              override the SSRC, or override the payload 
 *                              type
 *  \param uiSsrc               SSRC value that will override older SSRC 
 *  \param usPyLdType           Payload_type for DTMF in RTP that will 
 *                              override the previous setting.
 *  \param usRedundancyInterval The redundancy interval value that will 
 *                              override the previous value.
 */
VSTATUS VDEV_SendNteCmd(IN SChnl * pstChnl,
			 IN U32 uiNtePyLd,
			 IN U32 uiSsrc, IN U16 usOverRideBitField, IN U16 usPyLdType, IN U16 usRedundancyInterval)
{
#define NUM_OF_PARAMS 7

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_SENDNTE);

	if (usOverRideBitField)
		Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS,
						(U16) UT_CPU2BE16(((uiNtePyLd & 0xffff0000) >> 16)), 
						(U16) UT_CPU2BE16((uiNtePyLd & 0x0000ffff)),
						UT_CPU2LE16(usOverRideBitField),
						UT_CPU2LE16(usRedundancyInterval),
						(U16) UT_CPU2BE16(((uiSsrc & 0xffff0000) >> 16)), 
						(U16) UT_CPU2BE16((uiSsrc & 0x0000ffff)),
						UT_CPU2LE16(usPyLdType));
	else
		Status = VCORE_SetPayloadMsg(&stMsg, 2, (U16) UT_CPU2BE16(((uiNtePyLd & 0xffff0000) >> 16)), 
						(U16) UT_CPU2BE16(uiNtePyLd & 0x0000ffff));

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_HoldParticipant : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl      The channel on which MSP command will be sent
 *  \param EHoldState   Type of hold state
 *  \param usSrcPartId  Source participant id
 */
VSTATUS VDEV_HoldParticipant(IN SChnl * pstChnl,
			      IN EHoldState eHoldState, IN U16 usSrcPartId, IN U16 usMSPConfId, IN U16 usMSPPartId)
{
#define NUM_OF_PARAMS 4

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_CONF_PUT_PARTICIPANT_ON_HOLD);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(usMSPPartId),
							UT_CPU2LE16(usMSPConfId),
							UT_CPU2LE16(eHoldState),
							UT_CPU2LE16(usSrcPartId));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_MuteParticipant : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param usMute if 1 mute the participant else unmute it.
 */
VSTATUS VDEV_MuteParticipant(IN SChnl * pstChnl, IN U16 usMute, IN U16 usMSPConfId, IN U16 usMSPPartId)
{
#define NUM_OF_PARAMS 3

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_CONF_MUTE_PARTICIPANT);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(usMSPPartId),
							UT_CPU2LE16(usMSPConfId),
							UT_CPU2LE16(usMute));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SetPartDgainMsp : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param usDgain the value of gain.
 */
VSTATUS VDEV_SetPartDgain(IN SChnl* pstChnl, IN U16 usGain, IN U16 usMSPConfId,  IN U16 usMSPPartId)
{
#define NUM_OF_PARAMS 3

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_CONF_SET_PARTICIPANT_DGAIN);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(usMSPPartId),
							UT_CPU2LE16(usMSPConfId),
							UT_CPU2LE16(usGain));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SetPartOpt: The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param usRfc2833OnOff enable/disable RFC2833 Processing Feature
 */
VSTATUS VDEV_SetPartOpt(IN SChnl* pstChnl, IN U16 usRfc2833OnOff, IN U16 usMSPConfId,  IN U16 usMSPPartId)
{
#define NUM_OF_PARAMS 3

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_CONF_SET_PARTICIPANT_OPT);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(usMSPPartId),
							UT_CPU2LE16(usMSPConfId),
							UT_CPU2LE16(usRfc2833OnOff));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VCORE_CallerIDSetp : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstParam Parameters to be sent in CNDSETParam command
 */
VSTATUS VDEV_CNDSetup(IN SChnl * pstChnl, IN SCallerIdParams * pstParam)
{
#define FIFO_LEN    0x58
	gtl_msg_t stMsg;
	VSTATUS Status;
	Boolean bValid;
	U16 ausFifo[FIFO_LEN / 2 + 3];
	U8 ucCnt;
	SOnHookParams *pstOnhk = NULL;
	SOffHookParams *pstOffhk = NULL;
	SToneGenParams *pstTonGen = NULL;
	SCdnBlkParams *pstCdn = NULL;
	U8 ucFifoLen = 0;

	UT_MemSet(ausFifo, 0, sizeof(ausFifo));
	stMsg.fifo = (void *)ausFifo;
	VERIFY_CALLER_ID_PARAM(pstParam, bValid);
	if (!bValid)
	{
		return VAPI_ERR_INVALID_PARAM;
	}

	pstOnhk = &pstParam->stOnhookParam;
	VERIFY_CID_ONHOOK_PARAM(pstOnhk, bValid) if (!bValid)
	{
		return VAPI_ERR_INVALID_PARAM;
	}

	pstOffhk = &pstParam->stOffhookParam;
	VERIFY_CID_OFFHOOK_PARAM(pstOffhk, bValid) if (!bValid)
	{
		return VAPI_ERR_INVALID_PARAM;
	}
	if (pstOffhk->bSASToneGen)
	{
		pstTonGen = pstOffhk->pstToneGenParam;
		if (pstTonGen == NULL)
		{
			return VAPI_ERR_INVALID_PARAM;
		}
		VERIFY_START_TONE_PARAM(pstTonGen, bValid);
		if (!bValid)
		{
			return VAPI_ERR_INVALID_PARAM;
		}
	}
	ausFifo[3] = 0;
	/* OnHookXmitInfoType */
	ausFifo[4] = (0x08 << 8) | 0x01;	/* Info Count = 0x08, OnHookXmitInfoType = 0x01 */
	ausFifo[5] = (pstOnhk->ucDataTxPhase << 8) | (pstOnhk->ucFSKTone);
	ausFifo[6] = (pstOnhk->usChnlSeizureBits & 0xFF00) | (pstOnhk->ucAlertInfoByte);
	ausFifo[7] = (pstOnhk->ucChnlSeizureFirstBit << 8) | (pstOnhk->usChnlSeizureBits & 0x00FF);
	ausFifo[8] = UT_SWAP16(pstOnhk->usMarkBits);

	ucFifoLen = 9 * 2;

	/* OffHookXmitInfoType  */
	ausFifo[9] = (0x6 << 8) | 0x02;
	ausFifo[10] = (pstOffhk->ucModFreqByte << 8) | (pstOffhk->ucFSKTone);
	ausFifo[11] = (pstOffhk->usMarkBits & 0xFF00) | (pstOffhk->ucAlertInfoByte);
	ausFifo[12] = pstOffhk->usMarkBits & 0x00FF;

	/* CPEAlertToneInfoType */
	ausFifo[13] = (0xA << 8) | 0x03;
	ausFifo[14] = UT_SWAP16(pstOffhk->usLowNominalFreq);
	ausFifo[15] = UT_SWAP16(pstOffhk->usHighNominalFreq);
	ausFifo[16] = (pstOffhk->ucLevelLowFreq & 0xFF00) | (pstOffhk->ucDuration);
	ausFifo[17] = (pstOffhk->ucLevelHighFreq & 0xFF00) | (pstOffhk->ucLevelLowFreq & 0x00FF);
	ausFifo[18] = pstOffhk->ucLevelHighFreq & 0x00FF;

	/* CPEAckInfoType */
	ausFifo[19] = (0xA << 8) | 0x4;
	ausFifo[20] = (pstOffhk->ucDtmfDigit2 << 8) | pstOffhk->ucDtmfDigit1;
	ausFifo[21] = (pstOffhk->usTimeOutForACK & 0xFF00) | (pstOffhk->ucMinDurationACK);
	ausFifo[22] = (pstOffhk->ucTimeToDataTx << 8) | (pstOffhk->usTimeOutForACK & 0x00FF);
	ausFifo[23] = (pstOffhk->usAckToneMinLevel & 0xFF00) | (pstOffhk->ucTxDataIfNoACK);
	ausFifo[24] = (pstOffhk->usAckToneMinLevel & 0x00FF);

	ucFifoLen = 25 * 2;
	if (pstOffhk->bSASToneGen)
	{
		pstCdn = &pstTonGen->astCdnParam[0];

		/* SASToneInfoType  */
		ausFifo[25] = (0x22 << 8) | 0x5;
		ausFifo[26] = (pstTonGen->ucToneDir) |
		    (pstTonGen->ucTimeStampCntrl << 1) | (pstTonGen->ucProtectedGen << 3) | (pstTonGen->ucCdnRpt << 8);
		ausFifo[27] = (pstTonGen->uiStartTimeStamp & 0xFFFF);
		ausFifo[28] = (pstTonGen->uiStartTimeStamp >> 16);
		ausFifo[29] = (pstTonGen->uiLimitTimeStamp & 0xFFFF);
		ausFifo[30] = (pstTonGen->uiLimitTimeStamp >> 16);
		ausFifo[31] = 0;
		ausFifo[32] = 0;
		ausFifo[33] = 0;
		ausFifo[34] = pstCdn->usPrmToneFreq;
		ausFifo[35] = pstCdn->usPrmToneAmp;
		ausFifo[36] = pstCdn->usSecToneFreq;
		ausFifo[37] = pstCdn->usSecToneAmp;
		ausFifo[38] = pstCdn->usOnTime;
		ausFifo[39] = pstCdn->usOffTime;
		ausFifo[40] = 0;
		ausFifo[41] = 0;
		ausFifo[42] = pstOffhk->ucTxTimeToCas;
		ucFifoLen = 43 * 2;
	}
	/* TBD: NTTInfoType to be added. */
	VCORE_SetupCmdMsg(&stMsg, pstChnl->usMSPChnlId,
			  CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_CND_SET_PARAMS, ucFifoLen);

	for (ucCnt = 0; ucCnt < ucFifoLen / 2; ucCnt++)
		ausFifo[(U32) ucCnt] = UT_CPU2LE16(ausFifo[(U32) ucCnt]);
	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

	return Status;

#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_SetCidParams : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstParam Parameters to be sent in CNDOn/OffhookGen command
 */

VSTATUS VDEV_SetCidParams(IN SChnl * pstChnl)
{
	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 ausFifo[MAX_FIFO_LEN / 2];

	VCORE_InitMsg(&stMsg, (U8*)ausFifo, sizeof(ausFifo));

	/* Add the API header to the msg */
	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_CND_SET_PARAMS);

	UT_MutexLock(&stMutexCIDGen_g);
	/*Add the CID info block if they are configured */
	if (SCidGen_g.SOnHookXmitInfoBlock.bIsConfigured == 1)
		VCORE_AddPayloadMsgFromBuffer(&stMsg, (ON_HOOK_XMIT_INFO_TYPE_CNT + 2)/2,
					SCidGen_g.SOnHookXmitInfoBlock.usOnHookXmitInfoType);

	if (SCidGen_g.SOffHookXmitInfoBlock.bIsConfigured == 1)
		VCORE_AddPayloadMsgFromBuffer(&stMsg, (OFF_HOOK_XMIT_INFO_TYPE_CNT + 2)/2,
					SCidGen_g.SOffHookXmitInfoBlock.usOffHookXmitInfoType);

	if (SCidGen_g.SCPEAlertToneInfoBlock.bIsConfigured == 1)
		VCORE_AddPayloadMsgFromBuffer(&stMsg, (CPE_ALERT_TONE_INFO_TYPE_CNT + 2)/2,
					SCidGen_g.SCPEAlertToneInfoBlock.usCPEAlertToneInfoType);

	if (SCidGen_g.SCPEAckInfoBlock.bIsConfigured == 1)
		VCORE_AddPayloadMsgFromBuffer(&stMsg, (CPE_ACK_INFO_TYPE_CNT + 2)/2,
					SCidGen_g.SCPEAckInfoBlock.usCPEAckInfoType);

	if (SCidGen_g.SSASToneInfoBlock.bIsConfigured == 1)
		VCORE_AddPayloadMsgFromBuffer(&stMsg, SCidGen_g.SSASToneInfoBlock.ucBlockSize + 1, 
					SCidGen_g.SSASToneInfoBlock.usSASToneInfoType); 

	if (SCidGen_g.SNTTInfoBlock.bIsConfigured == 1)
		VCORE_AddPayloadMsgFromBuffer(&stMsg, (NTT_INFO_TYPE_CNT + 2)/2,
					SCidGen_g.SNTTInfoBlock.usNTTInfoType);

	if (SCidGen_g.SStopBitsInfoBlock.bIsConfigured == 1)
		VCORE_AddPayloadMsgFromBuffer(&stMsg, (STOP_BITS_INFO_TYPE_CNT + 2)/2,
					SCidGen_g.SStopBitsInfoBlock.usStopBitsInfoType);

	if (SCidGen_g.SPacketProcessingInfoBlock.bIsConfigured == 1)
		VCORE_AddPayloadMsgFromBuffer(&stMsg, (PACKET_PROCESS_INFO_TYPE_CNT + 2)/2,
				SCidGen_g.SPacketProcessingInfoBlock.usPacketProcessingInfoType);

	UT_MutexUnLock(&stMutexCIDGen_g);

	VCORE_SetPaddingMsg(&stMsg);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);
	
	return Status;
}

/****************************************************************************
 * VDEV_CNDOnGen : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstParam Parameters to be sent in CNDOnhookGen command
 */

VSTATUS VDEV_CNDOnGen(IN SChnl * pstChnl, IN SCallerIdParams * pstParam)
{
	gtl_msg_t stMsg;
	VSTATUS Status;
	U8 numOfParam;
	U16 ausFifo[MAX_FIFO_LEN / 2];

	if (pstParam->ucCIDMsgLen + 1 > MAX_FIFO_LEN)
		return VAPI_ERR_INVALID_PARAM;

	VCORE_InitMsg(&stMsg, (U8*)ausFifo, sizeof(ausFifo));

	/* Add the API header to the msg */
	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE,
				FC_CND_ONHOOK_GEN);

	/* Save the messsage to ausFifo */
	ausFifo[4] = UT_CPU2LE16((pstParam->ucCIDMsg[0] << 8) | pstParam->ucModType);
	UT_MemCopy(&ausFifo[5], &pstParam->ucCIDMsg[1], pstParam->ucCIDMsgLen - 1);

	/* The number of params is the (CID msg length + the Mod type byte + potential padding) / 2 */
	numOfParam = (((pstParam->ucCIDMsgLen + 1) + 1) & ~1) / 2;

	Status = VCORE_SetPayloadMsgFromBuffer(&stMsg, numOfParam, &ausFifo[4]);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);
	
	return Status;
}

/****************************************************************************
 * VDEV_CNDOffGen : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstParam Parameters to be sent in CNDOffHookGen command
 */

VSTATUS VDEV_CNDOffGen(IN SChnl * pstChnl, IN SCallerIdParams * pstParam)
{
	gtl_msg_t stMsg;
	VSTATUS Status;
	U8 numOfParam;
	U16 ausFifo[MAX_FIFO_LEN / 2];

	VCORE_InitMsg(&stMsg, (U8*)ausFifo, sizeof(ausFifo));

	/* Add the API header to the msg */
	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, 
				FC_CND_OFFHOOK_GEN);

	/* Save the messsage to ausFifo */
	UT_MemCopy(&ausFifo[4], pstParam->ucCIDMsg, pstParam->ucCIDMsgLen);

	/* The number of params is the (CID msg length + potential padding) / 2 */
	numOfParam = ((pstParam->ucCIDMsgLen + 1) & ~1) / 2;

	Status = VCORE_SetPayloadMsgFromBuffer(&stMsg, numOfParam, &ausFifo[4]);
	
	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

	return Status;
}

VSTATUS VDEV_CidGen(IN SChnl * pstChnl, IN SVcoreCidInfo * pstParam)
{
	gtl_msg_t stMsg;
	VSTATUS Status;
	U8 numOfParam;
	U16 ausFifo[MAX_FIFO_LEN / 2];

	if (pstParam->MsgLen + 1 > MAX_FIFO_LEN)
		return VAPI_ERR_INVALID_PARAM;

	VCORE_InitMsg(&stMsg, (U8*)ausFifo, sizeof(ausFifo));

	if (pstParam->bIsOnHook == 1)
	{
		/* Add the API header to the msg */
		VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_CND_ONHOOK_GEN);
	
		/* Save the messsage to ausFifo */
		ausFifo[4] = UT_CPU2LE16((pstParam->MsgStr[0] << 8) | pstParam->ucModulType);
		UT_MemCopy(&ausFifo[5], &pstParam->MsgStr[1], pstParam->MsgLen - 1);
	
		/* The number of params is the (CID msg length + the Mod type byte + potential padding) / 2 */
		numOfParam = (((pstParam->MsgLen + 1) + 1) & ~1) / 2;
	}
	else
	{
		/* Add the API header to the msg */
		VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_CND_OFFHOOK_GEN);
	
		/* Save the messsage to ausFifo */
		UT_MemCopy(&ausFifo[4], pstParam->MsgStr, pstParam->MsgLen);
	
		/* The number of params is the (CID msg length + potential padding) / 2 */
		numOfParam = ((pstParam->MsgLen + 1) & ~1) / 2;

	}

	Status = VCORE_SetPayloadMsgFromBuffer(&stMsg, numOfParam, &ausFifo[4]);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);
	
	return Status;
}

/****************************************************************************
 * VDEV_CNDStop : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstParam Parameters to be sent in CNDOnhookGen/CNDOffHookGen command
 */
VSTATUS VDEV_CNDStop(IN SChnl * pstChnl)
{
#define NUM_OF_PARAMS 0

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_CND_STOP);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

	return Status;
#undef NUM_OF_PARAMS
}


/****************************************************************************
 * VDEV_SetCidDetParams : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstParam Parameters to be sent in CNDOn/OffhookGen command
 */

VSTATUS VDEV_SetCidDetParams(IN SChnl * pstChnl)
{
	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 ausFifo[MAX_FIFO_LEN / 2];

	VCORE_InitMsg(&stMsg, (U8*)ausFifo, sizeof(ausFifo));

	/* Add the API header to the msg */
	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_CNDDETSP);

	UT_MutexLock(&stMutexCIDDet_g);
	/*Add the CID info block if they are configured */
	if (SCidDet_g.SDataFormatInfoBlock.bIsConfigured == 1)
		VCORE_AddPayloadMsgFromBuffer(&stMsg, (DATA_FORMAT_INFO_TYPE_CNT + 2)/2,
					SCidDet_g.SDataFormatInfoBlock.usDataFormatInfoType);

	if (SCidDet_g.SFSKDetectionInfoBlock.bIsConfigured == 1)
		VCORE_AddPayloadMsgFromBuffer(&stMsg, (FSK_DETECTION_INFO_TYPE_CNT + 2)/2,
					SCidDet_g.SFSKDetectionInfoBlock.usFSKDetectionInfoType);

	if (SCidDet_g.SOnHookDetInfoBlock.bIsConfigured == 1)
		VCORE_AddPayloadMsgFromBuffer(&stMsg, (ON_HOOK_DET_INFO_TYPE_CNT + 2)/2,
					SCidDet_g.SOnHookDetInfoBlock.usOnHookDetInfoType);

	if (SCidDet_g.SOffHookDetInfoBlock.bIsConfigured == 1)
		VCORE_AddPayloadMsgFromBuffer(&stMsg, (OFF_HOOK_DET_INFO_TYPE_CNT + 2)/2,
					SCidDet_g.SOffHookDetInfoBlock.usOffHookDetInfoType);

	if (SCidDet_g.SAckGenInfoBlock.bIsConfigured == 1)
		VCORE_AddPayloadMsgFromBuffer(&stMsg, (ACK_GEN_INFO_TYPE_CNT + 2)/2, 
					SCidDet_g.SAckGenInfoBlock.usAckGenInfoType); 

	if (SCidDet_g.SNTTDetInfoBlock.bIsConfigured == 1)
		VCORE_AddPayloadMsgFromBuffer(&stMsg, (NTT_DET_INFO_TYPE_CNT + 2)/2,
					SCidDet_g.SNTTDetInfoBlock.usNTTDetInfoType);

	UT_MutexUnLock(&stMutexCIDDet_g);

	VCORE_SetPaddingMsg(&stMsg);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);
	
	return Status;
}

/****************************************************************************
 * VDEV_StartTonGen : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstParam Parameters to be sent in FC_VOIP_TONEGEN command
 */
VSTATUS VDEV_StartTonGen(IN SChnl * pstChnl, IN SToneGenParams * pstParam)
{
#define FIFO_LEN    0x58
	gtl_msg_t stMsg;
	VSTATUS Status;
	Boolean bValid;
	U16 usParamCnt = 0;
	U16 usCdnCount = 0;

	U16 ausFifo[FIFO_LEN / 2 + 3];
	U8 ucCnt;
	SCdnBlkParams *pstCdn = NULL;

	UT_MemSet(ausFifo, 0, sizeof(ausFifo));
	VCORE_InitMsg(&stMsg, (U8*)ausFifo, sizeof(ausFifo));

	ausFifo[3] = 0;

	VERIFY_START_TONE_PARAM(pstParam, bValid);
	if (!bValid)
	{
		return VAPI_ERR_INVALID_PARAM;
	}
	while (bValid && (usCdnCount < pstParam->ucNoOfCdnBlk))
	{
		pstCdn = &pstParam->astCdnParam[usCdnCount++];
		VERIFY_TONE_CADENCE_PARAM(pstCdn, bValid);
	}
	if (!bValid)
	{
		return VAPI_ERR_INVALID_PARAM;
	}
	ausFifo[4] = pstParam->ucToneDir |
	    (pstParam->ucTimeStampCntrl << 1) | (pstParam->ucProtectedGen << 3) | (pstParam->ucCdnRpt << 8);

	ausFifo[5] = pstParam->uiStartTimeStamp & 0x00FF;
	ausFifo[6] = (pstParam->uiStartTimeStamp >> 16);
	ausFifo[7] = pstParam->uiLimitTimeStamp & 0x00FF;
	ausFifo[8] = (pstParam->uiLimitTimeStamp >> 16);
	ausFifo[9] = 0x00;	/* Reserved. Must be set to 0000. */
	ausFifo[10] = 0x00;	/* Reserved. Must be set to 0000. */
	ausFifo[11] = 0x00;	/* Reserved. Must be set to 0000. */
	usParamCnt = 12;
	for (usCdnCount = 0; usCdnCount < pstParam->ucNoOfCdnBlk; usCdnCount++)
	{
		pstCdn = &pstParam->astCdnParam[usCdnCount];
		ausFifo[usParamCnt++] = pstCdn->usPrmToneFreq;
		ausFifo[usParamCnt++] = pstCdn->usPrmToneAmp;
		ausFifo[usParamCnt++] = pstCdn->usSecToneFreq;
		ausFifo[usParamCnt++] = pstCdn->usSecToneAmp;
		ausFifo[usParamCnt++] = pstCdn->usOnTime;
		ausFifo[usParamCnt++] = pstCdn->usOffTime;
		ausFifo[usParamCnt++] = 0x00;	/* Reserved. Must be set to 0000. */
		ausFifo[usParamCnt++] = 0x00;	/* Reserved. Must be set to 0000. */
	}

	VCORE_SetupCmdMsg(&stMsg, pstChnl->usMSPChnlId,
			  CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_TONEGEN, (usParamCnt * 2));
	for (ucCnt = 0; ucCnt < usParamCnt; ucCnt++)
		ausFifo[(U32) ucCnt] = UT_CPU2LE16(ausFifo[(U32) ucCnt]);
	Status = VCORE_SendMSPReq(pstChnl, &stMsg);
	return Status;
#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_PlayDualTone : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstParam Parameters to be sent in FC_VOIP_TONEGEN command
 */
VSTATUS VDEV_PlayDualTone(IN SChnl * pstChnl, IN SPlayToneFlags *pstToneFlags)
{
#define NUM_OF_PARAMS	48
	gtl_msg_t stMsg;
	VSTATUS Status;
	struct _VOIP_TONEGEN stDualTone;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));
	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_TONEGEN);

	UT_MemCopy(&stDualTone, &astToneGeneric_g[pstToneFlags->eToneId].TonePayload.stDualTone, 
			sizeof (struct _VOIP_TONEGEN));

	if (pstToneFlags->eToneDir == eDirToBoth)
		stDualTone.param_4.bits.concurrently = True;
	else
		stDualTone.param_4.bits.tone_dir = pstToneFlags->eToneDir;

	stDualTone.param_4.bits.timestamp_ctrl = pstToneFlags->usTimestampCtrl;
	stDualTone.param_4.bits.prot_gen = pstToneFlags->usProtGen;
	stDualTone.param_4.bits.mix_mode_on = pstToneFlags->usMixModeOn;
	stDualTone.param_4.bits.mix_mode_off = pstToneFlags->usMixModeOff;

	stDualTone.param_4.word = UT_CPU2LE16(stDualTone.param_4.word);

	stDualTone.start_ts_l = UT_CPU2LE16(pstToneFlags->uiStartTimeStamp & 0x00FF);
	stDualTone.start_ts_m = UT_CPU2LE16(pstToneFlags->uiStartTimeStamp >> 16);
	stDualTone.lim_ts_l = UT_CPU2LE16(pstToneFlags->uiLimitTimeStamp & 0x00FF);
	stDualTone.lim_ts_m = UT_CPU2LE16(pstToneFlags->uiLimitTimeStamp >> 16);

	Status = VCORE_SetPayloadMsgFromBuffer(&stMsg, 
					8 + astToneGeneric_g[pstToneFlags->eToneId].ucParamBlockNum * 8, 
					(U16 *) &stDualTone);

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);
out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_PlayQuadTone : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstParam Parameters to be sent in FC_VOIP_TONEGEN command
 */
VSTATUS VDEV_PlayQuadTone(IN SChnl * pstChnl, IN SPlayToneFlags *pstToneFlags)
{
#define NUM_OF_PARAMS   24
	gtl_msg_t stMsg;
	VSTATUS Status;
	struct _VOIP_QUADTONEGEN stQuadTone_1, stQuadTone_2;
	U8 ausFifo[2 * sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	UT_MemCopy(&stQuadTone_1, &astToneGeneric_g[pstToneFlags->eToneId].TonePayload.stQuadTone, 
			sizeof (struct _VOIP_QUADTONEGEN));	

	if(pstToneFlags->eToneDir == eBoth)
	{
		UT_MemCopy(&stQuadTone_2, 
				&astToneGeneric_g[pstToneFlags->eToneId].TonePayload.stQuadTone, 
				sizeof (struct _VOIP_QUADTONEGEN));

		stQuadTone_1.Ctrl = UT_CPU2LE16(eDirToIP);
		stQuadTone_2.Ctrl = UT_CPU2LE16(eDirToTDM);
	}
	else
	{
		stQuadTone_1.Ctrl = UT_CPU2LE16(pstToneFlags->eToneDir);
	}

	VCORE_InitMsg(&stMsg, (U8*)ausFifo, sizeof(ausFifo));
	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_QUADTONEGEN);

	Status = VCORE_SetPayloadMsgFromBuffer(&stMsg, NUM_OF_PARAMS / 2, (U16 *) &stQuadTone_1);

	if(pstToneFlags->eToneDir == eBoth)
	{
		VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_QUADTONEGEN);
		Status = VCORE_SetPayloadMsgFromBuffer(&stMsg, NUM_OF_PARAMS / 2, (U16 *) &stQuadTone_2);
	}
	
	if (Status != SUCCESS)
		goto out;

out:
	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_ToneOff : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstParam Parameters to be sent in FC_VOIP_TONEOFF command
 */
VSTATUS VDEV_ToneOff(IN SChnl * pstChnl, SStopToneData *pstToneOffData)
{

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + 2 * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_TONEOFF);

	/* If the firmware release is below MR12 toneoff has only 1 parameter*/
	if (pstChnl->pstDev->MajorRelease < MR12)
		Status = VCORE_SetPayloadMsg(&stMsg, 1, UT_CPU2LE16(pstToneOffData->usDuration));
	else /*Force to stop the tonegen on both sides*/
		Status = VCORE_SetPayloadMsg(&stMsg, 2, UT_CPU2LE16(pstToneOffData->usDuration), UT_CPU2LE16(pstToneOffData->usSide));

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
}

/****************************************************************************
 * VDEV_SelectTdm : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstParam Parameters to be sent in FC_SUPV_TDM_SELECT_BUS_MODE command
 */
VSTATUS VDEV_SelectTdm(IN SChnl * pstChnl, IN STdmSetupParams * pstParam)
{
#define NUM_OF_PARAMS 1

	VSTATUS Status;
	Boolean bValid;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	VERIFY_SETUP_TDM_PARAM(pstParam, bValid);
	if (!bValid)
	{
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_TDM_SELECT_BUS_MODE);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(pstParam->usMode));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_TdmEnableBus : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstParam Parameters to be sent in FC_SUPV_TDM_ENABLE_BUS command
 */
VSTATUS VDEV_TdmEnableBus(IN SChnl * pstChnl, IN U8 usNoOfBus)
{
#define NUM_OF_PARAMS 1
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_TDM_ENABLE_BUS);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(usNoOfBus));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SetupTdmParam : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstParam Parameters to be sent in FC_SUPV_SETUP_TDM_PARAM command
 */
VSTATUS VDEV_SetupTdmParam(IN SChnl * pstChnl, IN STdmBusParams * pstParam)
{
#define NUM_OF_PARAMS 4
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];
	U16 p1, p2;

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_SUPVSR_SETUP_TDM_PARAMS);

	p1 = (pstParam->ucTxOrder << 15) |
	    (pstParam->ucRxPolarity << 14) |
	    (pstParam->ucTxPolarity << 13) |
	    (pstParam->ucRxClkMode << 12) |
	    (pstParam->ucTxClkMode << 11) |
	    (pstParam->ucRxClkEdge << 10) |
	    (pstParam->ucTxClkEdge << 9) | (pstParam->ucFrameEdge << 8) | (pstParam->usNumOfTS & 0xFF);

	p2 = ((pstParam->usNumTdmClkCyclesDelayTx & 0xC00) << 4) |
	    (pstParam->ucInvertedFrmSig << 11) |
	    (pstParam->ucBitOrderRcv << 10) | (pstParam->usNumTdmClkCyclesDelayTx & 0x3FF);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(p1), UT_CPU2LE16(p2),
						UT_CPU2LE16(pstParam->usNumTdmClkCyclesDelayRx),
						UT_CPU2LE16((pstParam->ucBusId << 2) | (pstParam->usNumOfTS >> 8)));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_CreateConf : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstParam Parameters to be sent in FC_CONF_CREATE_CONFERENCE command
 */
VSTATUS VDEV_CreateConf(IN SChnl * pstChnl, IN SConfParams * pstConfParams)
{
#define NUM_OF_PARAMS 1
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_CONF_CREATE_CONFERENCE);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, 
					UT_CPU2LE16((pstConfParams->bTranscoding | pstConfParams->bBlockDTMF << 1 | pstConfParams->bMixerType << 2)));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_CreatePart : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param ePartType Connection type (eg eLSP)
 *  \param usTDMTimeSlot The TDM timeslot to be requested
 *  \param usConfNo The conference numbe to be requested
 */
VSTATUS VDEV_CreatePart(IN SChnl * pstChnl, IN SChnl *pstChannelInfo)
{
#define NUM_OF_PARAMS 7
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_CONF_CREATE_PARTICIPANT);

	if(pstChannelInfo->ePartType == eLSPWB)
	{
		Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, 
									UT_CPU2LE16(pstChannelInfo->ePartType),
									UT_CPU2LE16(0x0000),
									UT_CPU2LE16(pstChannelInfo->pstConf->usMSPConfId),
									UT_CPU2LE16(pstChannelInfo->usTimeSlot),
									UT_CPU2LE16(pstChannelInfo->usExtTimeSlot[0]),
									UT_CPU2LE16(pstChannelInfo->usExtTimeSlot[1]),
									UT_CPU2LE16(pstChannelInfo->usExtTimeSlot[2]));
	}
	else
	{
		Status = VCORE_SetPayloadMsg(&stMsg, 3, UT_CPU2LE16(pstChannelInfo->ePartType),
									UT_CPU2LE16(pstChannelInfo->usTimeSlot),
									UT_CPU2LE16(pstChannelInfo->pstConf->usMSPConfId));
	}

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_PartAGCSET : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstParam Parameters to be sent in FC_VOIP_AGCSET command
 */
VSTATUS VDEV_PartAGCSET(IN SChnl * pstChnl, IN struct _VOIP_AGCSET * pstParam)
{
#define NUM_OF_PARAMS 7
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_AGCSET);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(pstParam->param_4.word),
						UT_CPU2LE16(pstParam->pkt2pcm_target_level),
						UT_CPU2LE16(pstParam->pcm2pkt_target_level),
						UT_CPU2LE16(pstParam->pkt2pcm_max_gain),
						UT_CPU2LE16(pstParam->pkt2pcm_min_gain),
						UT_CPU2LE16(pstParam->pcm2pkt_max_gain),
						UT_CPU2LE16(pstParam->pcm2pkt_min_gain));

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_ConfSetDTS : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstParam Parameters to be sent in FC_VOIP_AGCSET command
 */
VSTATUS VDEV_ConfSetDTS(IN SChnl * pstChnl, IN U16 usConfNo, IN SConfDTSParams * pstParam)
{
#define NUM_OF_PARAMS 12
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_CONF_SPECIFY_DOMINANT_TALKERS);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(usConfNo),
						UT_CPU2LE16(pstParam->bDTSOnOff),
						UT_CPU2LE16(pstParam->usAttackConst),
						UT_CPU2LE16(pstParam->usNTH1),
						UT_CPU2LE16(pstParam->usNTH2),
						UT_CPU2LE16(pstParam->usHoldConst),
						UT_CPU2LE16(pstParam->usN1),
						UT_CPU2LE16(pstParam->usG1),
						UT_CPU2LE16(pstParam->usN2),
						UT_CPU2LE16(pstParam->usG2),
						UT_CPU2LE16(pstParam->usN3),
						UT_CPU2LE16(pstParam->usG3));

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_DestroyPart : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param ChnlId The participant id to be destroyed
 *  \param usConfNo The conference id of the participant to be destroyed
 */
VSTATUS VDEV_DestroyPart(IN SChnl * pstChnl, IN CHNLID ChnlId, IN U16 usConfId)
{
#define NUM_OF_PARAMS 2
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	UT_Log(VCORE, DEBUG, "VDEV_DestroyParticipant->Dev-%u,Ch-%u\n", pstChnl->pstDev->DevId, ChnlId);
	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_CONF_DESTROY_PARTICIPANT);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(ChnlId), UT_CPU2LE16(usConfId));

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef  NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_DestroyConf : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param usConfNo The conference id of the conference to be destroyed
 */
VSTATUS VDEV_DestroyConf(IN SChnl * pstChnl, IN U16 usConfId)
{
#define NUM_OF_PARAMS 1
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_CONF_DESTROY_CONFERENCE);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(usConfId));

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef  NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SetFlowControl : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param
 */
VSTATUS VDEV_SetFlowControl(IN SChnl * pstChnl)
{
#define NUM_OF_PARAMS 1
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_SET_FLOWCON);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(1));

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef  NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SetMediaHost : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param 
 *  \param 
 */
VSTATUS VDEV_SetMediaHost(IN SChnl * pstChnl, IN U8 ucMediaHost)
{
#define NUM_OF_PARAMS 1
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_SET_MEDIA_HOST);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(ucMediaHost));

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef  NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SetChnlMode : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param 
 *  \param 
 */
VSTATUS VDEV_SetChnlMode(IN SChnl * pstChnl, IN ERecPlayState eRecPlayState, IN SHostSpeechData * pstUserData)
{
#define NUM_OF_PARAMS 2
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];
	U16 p2;

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_SET_CHANNEL_MODE);

	p2 = pstUserData->ucDestCoding << 8 | ((pstUserData->ucDestinationType << 4) | pstUserData->ucSourceType);
	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(pstUserData->ucSrcCoding << 8 | eRecPlayState),
							UT_CPU2LE16(p2));

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef  NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SetDrainPlayRecBuffer : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param 
 *  \param 
 */
VSTATUS VDEV_SetDrainPlayRecBuffer(IN SChnl * pstChnl)
{
#define NUM_OF_PARAMS 1
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_DRAIN_PLAYREC_BUFFER);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

	return Status;
#undef  NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SetAs_Stop : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param 
 *  \param 
 */
VSTATUS VDEV_SetAs_Stop(IN SChnl * pstChnl)
{
#define NUM_OF_PARAMS 1
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_AS_STOP);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

	return Status;
#undef  NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_PlaybackSetRate : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param 
 */
VSTATUS VDEV_PlaybackSetRate(IN SChnl * pstChnl, IN EPlaybackRate ePlaybackRate)
{
#define NUM_OF_PARAMS 1
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_AS_SET_RATE);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(ePlaybackRate));

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef  NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SendData : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param 
 *  \param 
 */
VSTATUS VDEV_SendData(IN SChnl * pstChnl, IN SFrameInfo * pstFrame, IN U16 usFuncCode)
{
	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 *pausFifo;
	U32 uiFifoLen = 0, uiTempLen = 0;
	U32 ucCnt;

	if (usFuncCode == FC_SYNCDAT)
	{
		if (pstFrame->ucIsHdrPresent)
		{
			uiFifoLen = 240 + PACKET_HEADER_LEN + COMMAND_HEADER_LEN;
			uiTempLen = 6;
		}
		else
		{
			uiFifoLen = MAX_SYNCEOF_SIZE + COMMAND_HEADER_LEN;
			uiTempLen = 4;
		}
	}
	else
	{
		if (pstFrame->ucIsHdrPresent)
		{
			uiFifoLen = pstFrame->uiFrameLen + PACKET_HEADER_LEN + COMMAND_HEADER_LEN;
			uiTempLen = 6;
		}
		else
		{
			uiFifoLen = pstFrame->uiFrameLen + COMMAND_HEADER_LEN;
			uiTempLen = 4;
		}
	}

	pausFifo = UT_Calloc(1, ((uiFifoLen + 4 + 4 /*for MultiCmd */ ) * sizeof(U8)));

	if (pausFifo == NULL)
		return VAPI_ERR_NOMEM;

	stMsg.fifo = (void *) pausFifo;

	VCORE_SetupCmdMsg(&stMsg, pstChnl->usMSPChnlId,
			  CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, usFuncCode, uiFifoLen);

	for (ucCnt = 0; ucCnt < uiTempLen; ucCnt++)
	{
		*(pausFifo + ucCnt) = UT_CPU2LE16(*(pausFifo + ucCnt));
	}

	UT_MemCopy(((U8 *) pausFifo) + uiTempLen * 2, pstFrame->pucFrameBuffer, uiFifoLen - uiTempLen * 2);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);
	UT_FreeMem(pausFifo);

	return Status;
}


/****************************************************************************
 * VDEV_SendDataExtended : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param 
 *  \param 
 */
VSTATUS VDEV_SendDataExtended(IN SChnl * pstChnl, IN SFrameInfo * pstFrame)
{
	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 *pausFifo;
	U32 uiFifoLen = 0, uiTempLen = 0;
	U32 ucCnt;

	if (pstFrame->ucIsHdrPresent)
	{
		uiFifoLen = pstFrame->uiFrameLen + PACKET_HEADER_LEN + COMMAND_HEADER_LEN;
		uiTempLen = 6;
	}
	else
	{
		uiFifoLen = pstFrame->uiFrameLen + COMMAND_HEADER_LEN;
		uiTempLen = 4;
	}

	pausFifo = UT_Calloc(1, ((uiFifoLen + 4 + 4 /*for MultiCmd */ ) * sizeof(U8)));

	if (pausFifo == NULL)
		return VAPI_ERR_NOMEM;

	stMsg.fifo = (void *)pausFifo;

	/* setup command message */
	stMsg.type = 0;
	stMsg.channel = pstChnl->usMSPChnlId;

	/* add potential padding */
	stMsg.fifo_size = ALIGN_MESSAGE_LENGTH(uiFifoLen);

	*((U16 *) stMsg.fifo + 0) = 0;
	*((U16 *) stMsg.fifo + 1) = CMD_CLASS_CONF_CHANNEL << 8 | CMD_TYPE_CONF_CHANGE;
	*((U16 *) stMsg.fifo + 2) = FC_SYNCEOF;
	*((U16 *) stMsg.fifo + 3) = ALIGN_MESSAGE_LENGTH(uiFifoLen);

	for (ucCnt = 0; ucCnt < uiTempLen; ucCnt++)
	{
		*(pausFifo + ucCnt) = UT_CPU2LE16(*(pausFifo + ucCnt));
	}

	UT_MemCopy(((U8 *) pausFifo) + uiTempLen * 2, pstFrame->pucFrameBuffer, uiFifoLen - uiTempLen * 2);	

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);
	UT_FreeMem(pausFifo);

	return Status;
}


VSTATUS VDEV_SendDataInd(IN SChnl * pstChnl, IN U16 usFuncCode)
{
#define NUM_OF_PARAMS 2
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_INDICATION_RESP, usFuncCode);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(0x0000), UT_CPU2LE16(0x0000));

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef  NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SendH254 : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param 
 *  \param 
 */
VSTATUS VDEV_SendH254(IN SChnl * pstChnl, IN SH245Packet *pstH245Packet)
{
	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 *pausFifo;
	U32 uiFifoLen;
	U32 ucCnt;

	uiFifoLen = pstH245Packet->uiH245PacketLen + COMMAND_HEADER_LEN + H245_MSG_HDR_LEN;
		
	pausFifo = UT_Calloc(1, uiFifoLen * sizeof(U8));

	if (pausFifo == NULL)
		return VAPI_ERR_NOMEM;

	stMsg.fifo = (void *) pausFifo;

	VCORE_SetupCmdMsg(&stMsg, pstChnl->usMSPChnlId,
			  CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_IMSOTDM_TXH245, uiFifoLen);	

	*((U16 *) stMsg.fifo + 4) = pstH245Packet->uiEndOfMsg;
	*((U16 *) stMsg.fifo + 5) = pstH245Packet->uiH245PacketLen;

	for (ucCnt = 0; ucCnt < (COMMAND_HEADER_LEN + H245_MSG_HDR_LEN) / 2; ucCnt++)
	{
		*(pausFifo + ucCnt) = UT_CPU2LE16(*(pausFifo + ucCnt));
	}

	UT_MemCopy(((U8 *) pausFifo) + COMMAND_HEADER_LEN + H245_MSG_HDR_LEN, 
						pstH245Packet->ptrH245Packet, 
						pstH245Packet->uiH245PacketLen);	

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);
	UT_FreeMem(pausFifo);

	return Status;
}

/****************************************************************************
 * VDEV_SPUFeatureControl : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 * 
 */

VSTATUS VDEV_SPUFeatureControl(IN SChnl * pstChnl)
{
#define NUM_OF_PARAMS 2
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_SPU_FEATURES_CONTROL);
	if (pstChnl->pstDev->MajorRelease < MR19)
		Status = VCORE_SetPayloadMsg(&stMsg, 1, UT_CPU2LE16((U16)(pstChnl->pstDev->usSpuFeatureMask)));
	else
		Status = VCORE_SetPayloadMsg(&stMsg, 2, UT_CPU2LE16((U16)(pstChnl->pstDev->usSpuFeatureMask)), UT_CPU2LE16((U16)(pstChnl->pstDev->usSpuFeatureMask >> 16)));

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef  NUM_OF_PARAMS
}


/****************************************************************************
 * VCORE_CIDDectionSetp : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstCntryInfo gives the parameter to be set for Caller ID detection Feature 
 *
 */
VSTATUS VDEV_CIDDetectionSetup(IN SChnl * pstChnl, IN SCIDDetectionParams * pstCIDDetectionParams)
{
#define FIFO_LEN 0x2C		/* hex equivalent of 44 (21*2+2) */

	VSTATUS Status = SUCCESS;
	gtl_msg_t stMsg;
	U16 ausFifo[FIFO_LEN / 2 + 3];
	U8 ucCnt;

	/*Boolean bValid; */

	U8 ucFifoLen = 0;

	if (pstChnl == NULL)
	{
		return VAPI_ERR_INVALID_PARAM;
	}

	UT_MemSet(ausFifo, 0, sizeof(ausFifo));
	VCORE_InitMsg(&stMsg, (U8*)ausFifo, sizeof(ausFifo));

	ausFifo[3] = 0;		/*Reserved Word */

	/*DataFormatType */

	ausFifo[4] = (0x02 << 8) | 0x01;

	ausFifo[5] = (pstCIDDetectionParams->stDataFormatType.ucCRCChecksumSelection << 8)
	    | (pstCIDDetectionParams->stDataFormatType.ucDataParitySelection);

	ucFifoLen = 6 * 2;

	/*FSKDetectionType */
	ausFifo[6] = (0x06 << 8) | 0x02;
	ausFifo[7] = (pstCIDDetectionParams->stFSKDetectionType.usFSKPullThreshold & 0xFF00)
	    | (pstCIDDetectionParams->stFSKDetectionType.ucFSKDetectorType);
	ausFifo[8] = (pstCIDDetectionParams->stFSKDetectionType.usFSKDropThreshpld & 0xFF00)
	    | (pstCIDDetectionParams->stFSKDetectionType.usFSKPullThreshold & 0x00FF);
	ausFifo[9] = (pstCIDDetectionParams->stFSKDetectionType.usFSKDropThreshpld & 0x00FF);

	ucFifoLen = 10 * 2;

	/*OnHookDetType */
	ausFifo[10] = (0x04 << 8) | 0x03;
	ausFifo[11] = UT_SWAP16(pstCIDDetectionParams->stOnHookDetType.usChnlSeizThreshold);
	ausFifo[12] = UT_SWAP16(pstCIDDetectionParams->stOnHookDetType.usMarkbitThreshold);

	ucFifoLen = 13 * 2;

	/*OffHookDetType */
	ausFifo[13] = (0x02 << 8) | 0x04;
	ausFifo[14] = UT_SWAP16(pstCIDDetectionParams->stOffHookDetType.usMarkbitThreshold);

	ucFifoLen = 15 * 2;

	/*AckGenType */
	ausFifo[15] = (0x06 << 8) | 0x05;
	ausFifo[16] = (pstCIDDetectionParams->stAckGenType.ucDelayCPEAlert << 8)
	    | (pstCIDDetectionParams->stAckGenType.ucDTMFdigits);
	ausFifo[17] = (pstCIDDetectionParams->stAckGenType.usAcktoneTxlevel & 0xFF00)
	    | (pstCIDDetectionParams->stAckGenType.ucAckToneduration);
	ausFifo[18] = (pstCIDDetectionParams->stAckGenType.usAcktoneTxlevel & 0x00FF);

	ucFifoLen = 19 * 2;

	/* NTTDetType */
	ausFifo[19] = (0x02 << 8) | 0x06;
	ausFifo[20] = (pstCIDDetectionParams->stNTTDetType.ucNTTDetector & 0x00FF);

	ucFifoLen = 21 * 2;

	VCORE_SetupCmdMsg(&stMsg, pstChnl->usMSPChnlId, CMD_CLASS_CONF_CHANNEL,
			  CMD_TYPE_CONF_CHANGE, FC_CNDDETSP, ucFifoLen);

	for (ucCnt = 0; ucCnt < FIFO_LEN / 2; ucCnt++)
		ausFifo[(U32) ucCnt] = UT_CPU2LE16(ausFifo[(U32) ucCnt]);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

	return Status;

#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_CNDDETCtrlSend : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param ucOnOffhk gives the type of the Caller ID to be detected
 *
 */

VSTATUS VDEV_CNDDETCtrlSend(IN SChnl * pstChnl, IN U8 ucOnOffhk)
{
#define FIFO_LEN 0x0A
#define CNDDETCTL_ONHK_PARAM  0x0001
#define CNDDETCTL_OFFHK_PARAM 0x0003

	VSTATUS Status = SUCCESS;
	gtl_msg_t stMsg;
	U16 ausFifo[FIFO_LEN / 2 + 3];
	U8 ucCnt;

	if (pstChnl == NULL)
	{
		return VAPI_ERR_INVALID_PARAM;
	}

	UT_MemSet(ausFifo, 0, sizeof(ausFifo));
	VCORE_InitMsg(&stMsg, (U8*)ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdMsg(&stMsg, pstChnl->usMSPChnlId,
			  CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_CNDDETCTL, FIFO_LEN);

	ausFifo[3] = 0;		/*Reserved Word */

	if (!ucOnOffhk)
		ausFifo[4] = CNDDETCTL_ONHK_PARAM /* usCNDDETCTLParamOnhook_g  */ ;
	else
		ausFifo[4] = CNDDETCTL_OFFHK_PARAM /* usCNDDETCTLParamOffhook_g */ ;

	ausFifo[5] = 0;		/*Padding Bytes of Zeros */

	for (ucCnt = 0; ucCnt < FIFO_LEN / 2; ucCnt++)
		ausFifo[(U32) ucCnt] = UT_CPU2LE16(ausFifo[(U32) ucCnt]);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

	return Status;

#undef FIFO_LEN
#undef CNDDETCTL_ONHK_PARAM
#undef CNDDETCTL_OFFHK_PARAM
}

/****************************************************************************
 * VDEV_SetCidCtrl : The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param ucOnOffhk gives the type of the Caller ID to be detected
 *
 */

VSTATUS VDEV_SetCidCtrl(IN SChnl * pstChnl, struct _CNDDETCTL *cid_det_ctrl)
{
#define NUM_OF_PARAMS 2

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_CNDDETCTL);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(cid_det_ctrl->param_4.word), 
								UT_CPU2LE16(0x0000)); /*Reserved set to 0*/

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;

#undef NUM_OF_PARAMS
}

/****************************************************************************
* VDEV_SetFaxOpt : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param stMsg the pre allocated message to be filled with FAXOPT parameters
 *  \param pstFaxOpts Parameters to be sent in FAXOPT command
 */
VSTATUS VDEV_SetFaxOpt(gtl_msg_t *pstMsg, IN struct _FAXOPT *pstFaxOpts)
{
	VSTATUS Status = SUCCESS;

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(pstMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_FAXOPT);

	Status = VCORE_SetPayloadMsg(pstMsg, 3, UT_CPU2LE16(pstFaxOpts->param_4.word),
						UT_CPU2LE16(pstFaxOpts->param_5.word), 
						UT_CPU2LE16(0x0000)); /*Reserved set to 0*/

	return Status;
}

/****************************************************************************
* VDEV_SendFaxOpt : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstFaxOpts Parameters to be sent in FAXOPT command
 */
VSTATUS VDEV_SendFaxOpt(IN SChnl *pstChnl, IN struct _FAXOPT *pstFaxOpts)
{
#define NUM_OF_PARAMS 3

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build the message */
	Status = VDEV_SetFaxOpt(&stMsg, pstFaxOpts);
	if (Status != SUCCESS)
		goto err;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

err:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
* VDEV_SetFaxLevel : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param stMsg the pre allocated message to be filled with FAXLVL parameters
 *  \param ucFaxTxLvl Parameters to be sent in FAXLVL command
 */
VSTATUS VDEV_SetFaxLevel(gtl_msg_t *pstMsg, IN struct _FAXLVL *pstFaxTxLvl)
{
	VSTATUS Status = SUCCESS;

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(pstMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_FAXLVL);

	Status = VCORE_SetPayloadMsg(pstMsg, 1, UT_CPU2LE16(pstFaxTxLvl->faxlvl));

	return Status;
}

/****************************************************************************
* VDEV_SendFaxLevel : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param ucFaxTxLvl Parameters to be sent in FAXLVL command
 */
VSTATUS VDEV_SendFaxLevel(IN SChnl *pstChnl, IN struct _FAXLVL *pstFaxTxLvl)
{
#define NUM_OF_PARAMS 1

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build the message */
	Status = VDEV_SetFaxLevel(&stMsg, pstFaxTxLvl);
	if (Status != SUCCESS)
		goto err;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

err:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
* VDEV_SetToneCtrlOpt : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param stMsg the pre allocated message to be filled with TONECONTROL parameters
 *  \param pstToneCtrl Parameters to be sent in TONECONTROL command
 */
VSTATUS VDEV_SetToneCtrlOpt(gtl_msg_t *pstMsg, IN struct _VOIP_TONECTRL *pstToneCtrl)
{
	VSTATUS Status;

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(pstMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_TONECTRL);

	Status = VCORE_SetPayloadMsg(pstMsg, 5, UT_CPU2LE16(pstToneCtrl->param_4.word),
						UT_CPU2LE16(pstToneCtrl->param_5.word),
						UT_CPU2LE16(pstToneCtrl->param_6.word),
						UT_CPU2LE16(pstToneCtrl->param_7.word),
						UT_CPU2LE16(pstToneCtrl->param_8.word));

	return Status;
}

/****************************************************************************
* VDEV_SendToneCtrlOpt : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param pstToneCtrl Parameters to be sent in TONECONTROL command
 */
VSTATUS VDEV_SendToneCtrlOpt(IN SChnl *pstChnl, IN struct _VOIP_TONECTRL *pstToneCtrl)
{
#define NUM_OF_PARAMS 5

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build the message */
	Status = VDEV_SetToneCtrlOpt(&stMsg, pstToneCtrl);
	if (Status != SUCCESS)
		goto err;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

err:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
* VDEV_SetToneRelayOpt : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param stMsg the pre allocated message to be filled with TONERELAY parameters
 *  \param pstToneRelay Parameters to be sent in TONERELAY command
 */
VSTATUS VDEV_SetToneRelayOpt(gtl_msg_t *pstMsg, IN struct _VOIP_TONES_RELAY_OPTION *pstToneRelay)
{
	VSTATUS Status;

	/* build and add the API header */
	VCORE_SetupCmdHdrMsg(pstMsg, CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_CHANGE, FC_VOIP_TONES_RELAY_OPTION);

	Status = VCORE_SetPayloadMsg(pstMsg, 1, UT_CPU2LE16(pstToneRelay->param_4.word));

	return Status;
}

/****************************************************************************
* VDEV_SendToneRelayOpt : The function does the following things -
***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL \n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param ucFaxTxLvl Parameters to be sent in FAXLVL command
 */
VSTATUS VDEV_SendToneRelayOpt(IN SChnl *pstChnl, IN struct _VOIP_TONES_RELAY_OPTION *pstToneRelay)
{
#define NUM_OF_PARAMS 1

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	/* build the message */
	Status = VDEV_SetToneRelayOpt(&stMsg, pstToneRelay);
	if (Status != SUCCESS)
		goto err;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

err:
	return Status;
#undef NUM_OF_PARAMS
}


VSTATUS VDEV_EnableMultiCmds(IN SChnl * pstChnl, U8 bEnable)
{
#define NUM_OF_PARAMS 1
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_CSME_MULTI_CMD);

	if (bEnable)
		Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(1));
	else
		Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(0));

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef  NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_TransConnect : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
    Send TRANS_CONNECT - enable/disable transcoding
 *
 */
VSTATUS VDEV_TransConnect(IN SChnl *pstChnl, IN STransConnectParams * pstTransConnectParams)
{
#define NUM_OF_PARAMS 5
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_TRANS_CONNECT);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16((pstTransConnectParams->bTranscodingEnable) |
								(pstTransConnectParams->ucDSPMode << 1) |
								(pstTransConnectParams->ucVirtualTranscoding << 3)),
							UT_CPU2LE16(pstTransConnectParams->Channel1Id),
							UT_CPU2LE16(pstTransConnectParams->Channel1PT),
							UT_CPU2LE16(pstTransConnectParams->Channel2Id),
							UT_CPU2LE16(pstTransConnectParams->Channel2PT));

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef  NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_ForceAlert The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
    Send FORCCE ALERT 0x0024 to the device
 *
 */
VSTATUS VDEV_ForceAlert(IN SChnl *pstChnl)
{
#define NUM_OF_PARAMS 1
	VSTATUS Status;
	gtl_msg_t stMsg, *pstMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	pstMsg = &stMsg;
	/* init msg fields to default values*/
	VCORE_InitMsg(pstMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(pstMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_FORCE_ALERT);

	Status = VCORE_SetPayloadMsg(pstMsg, NUM_OF_PARAMS, UT_CPU2LE16(ALERT_UNEX_DATA));

	if (Status != SUCCESS)
		goto out;

	/* Flag the message as non boot command */
	switch (pstChnl->pstDev->eItfType)
	{
	case ePCI_ITF:
		pstMsg->type |= PCI; 
		break;

	case eCSM_ITF:
		pstMsg->type |= CSME; 
		break;

	case ePOS_ITF:
		pstMsg->type |= POS;
		break;

	default:
		UT_ErrorLog(GTL, "VCORE_SetupCmdMsg: Unsupported interface type (%u)\n", pstChnl->pstDev->eItfType);
		Status = VAPI_ERR_UNSUPP_FEATURE; 
		goto out;
	}

	/*Set Message index */
	UT_MutexLock(&pstChnl->stSemMsgIndex);
	if (pstMsg->fifo_size)
	{
		if (pstMsg->type & POS)
		{
			pstMsg->channel = pstChnl->usMSPChnlId;
			((SApiHdr *)(pstMsg->fifo))->index = pstChnl->ucMsgIndex;
		}
		else
		{
			if (pstMsg->type & BOOT)
			{
				pstMsg->channel = 0;
				if (pstMsg->type & CSME)
				{
					((SBootHdr *)(pstMsg->fifo))->index = pstChnl->ucMsgIndex;
				}
			}
			else
			{
				pstMsg->channel = pstChnl->usMSPChnlId;
				((SApiHdr *)(pstMsg->fifo))->index = pstChnl->ucMsgIndex;
			}
		}
	}

	UT_MutexUnLock(&pstChnl->stSemMsgIndex);

#ifndef VAPI_NODBG
	GTL_PrintMessage(pstMsg, VCORE, 1);
#endif

	/*Call GTL_Write */
	Status = GTL_Write(pstChnl->pstDev->DevId, pstMsg);

out:
	return Status;
#undef  NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_GetMR The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
    Get MR version of the device
 *
 */
VSTATUS VDEV_GetMR(IN SChnl * pstChnl)
{
#define NUM_OF_PARAMS 0

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY, FC_ARM_FW_BRANCH);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_GetDevModel The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
    Get device model type
 *
 */
VSTATUS VDEV_GetDevModel(IN SChnl * pstChnl)
{
#define NUM_OF_PARAMS 0

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY, FC_SUPVSR_GET_DEVICE_TYPE);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

	return Status;
#undef NUM_OF_PARAMS
}
/****************************************************************************
 * VDEV_SetQueryCommand
 ***************************************************************************/
VSTATUS VDEV_SetQueryCommand(IN gtl_msg_t *pstMsg, IN U16 *pvComm, IN U16 usNumComm)
{
	VSTATUS Status = SUCCESS;
	int i, j;
	U16 *pvAddedComm;

	if (usNumComm < 1)
	{
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	pvAddedComm = UT_Calloc(usNumComm, sizeof(usNumComm));

	if (pvAddedComm == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	for (i = 0; i < usNumComm; i++)
	{
		for (j = 0; j < usNumComm; j++)
		{
			/* command is already to be queried (added to Msg)*/
			if (pvAddedComm[j] == pvComm[i])
				continue;
		}

		VCORE_SetupCmdHdrMsg(pstMsg,  CMD_CLASS_CONF_CHANNEL, CMD_TYPE_QUERY, pvComm[i]);
		pvAddedComm[i] = pvComm[i];

		if (pstMsg->fifo_size > pstMsg->max_fifo_size)
		{
			UT_ErrorLog(VCORE, "VDEV_SetQueryComand: Wrong fifo size (%u)\n", pstMsg->fifo_size);
			Status = VAPI_ERR_FIFO_SIZE;
			break;
		}
		else
			Status = SUCCESS;
	}

	UT_FreeMem(pvAddedComm);
out:
	return Status;
}

/****************************************************************************
 * VCORE_ModifyQueriedCommand
 ***************************************************************************/
VSTATUS VCORE_ModifyQueriedCommand(IN gtl_msg_t * pstMsg, IN SModifyConnectionInfo *pstModConnInfo, IN U16 usModifyNum)
{
	int i, j;
	U16 usTempParam;
	U16 usValue;
	U16 usMask = 0xFFFF;
	U16 usBytesInFifo = True;
	U16 tempSize;
	void *tempFifo;
	U8 ucSuccessMatches = 0;

	/* take particular array item*/
	for (i = 0; i < usModifyNum; i++)
	{
		tempSize = pstMsg->fifo_size;
		tempFifo = (void *) pstMsg->fifo;

		/* go thru all queried commands for particular array item*/
		for (j = 0; j < usModifyNum || usBytesInFifo > 0; j++)
		{
			/* queried command and FC in array item match*/
			if (UT_LE2CPU16(((U16 *) (pstMsg->fifo))[SIDX_FUNC_CODE]) == pstModConnInfo[i].usFunctionCode)
			{	
				ucSuccessMatches++;
				usTempParam = UT_LE2CPU16(((U16 *)pstMsg->fifo)[3 + pstModConnInfo[i].usParam]);

				usValue = pstModConnInfo[i].usValue;

				usValue = usValue << (16 - pstModConnInfo[i].ucfield_width);
				usValue = usValue >> (16 - pstModConnInfo[i].ucfield_shift - pstModConnInfo[i].ucfield_width);

				usMask = usMask << (16 - pstModConnInfo[i].ucfield_width);
				usMask = usMask >> (16 - pstModConnInfo[i].ucfield_shift - pstModConnInfo[i].ucfield_width);

				usTempParam = usTempParam & ~usMask;
				usTempParam = usTempParam | usValue;

				((U16 *)pstMsg->fifo)[3 + pstModConnInfo[i].usParam] = UT_CPU2LE16(usTempParam);
				((SApiHdr *) pstMsg->fifo)->cmd_class = CMD_CLASS_CONF_CHANNEL;
				((SApiHdr *) pstMsg->fifo)->cmd_type = CMD_TYPE_CONF_CHANGE;
			}
			
			usBytesInFifo = VCORE_GetNextCmd((SMsg*)pstMsg);
		}

		pstMsg->fifo_size = tempSize;
		pstMsg->fifo = tempFifo;
	}

	if(ucSuccessMatches == usModifyNum)
		return SUCCESS;
	else
		return FAILURE;
}

/****************************************************************************
 * VDEV_CESoPSN_Mapper The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *   Send VDEV_CESoPSN_MAP_CHANNELS (FC 0x9800) or
 *   Send VDEV_CESoPSN_SPEC_MAP_CHANNELS (FC 0x9801) to the device
 *
 */
VSTATUS VDEV_CESoPSN_Mapper(IN SChnl *pstSupvsrChnl, 
			SFsmCesoPsnMapChannelInfo *pstSFsmCesoPsnMapChannelInfo)
{
	VSTATUS Status = SUCCESS;
	int i;
	gtl_msg_t stMsg;
	U8 ausFifo[DEFAULT_FIFO_MAX_SIZE];

	if(pstSupvsrChnl == NULL)
	{
		return VAPI_ERR_INVALID_PARAM;
	}

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	if(pstSFsmCesoPsnMapChannelInfo->MapType == eChannelMap)
		VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_CESOPSN_MAP_CHANS);
	else
		VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_CESOPSN_SPEC_MAP_CHANS);

	/* MasterConnId*/
	VCORE_PutParamPayloadMsg(&stMsg, UT_CPU2LE16(pstSFsmCesoPsnMapChannelInfo->MasterChannel));
	/* action map/unmap*/
	VCORE_PutParamPayloadMsg(&stMsg, UT_CPU2LE16(pstSFsmCesoPsnMapChannelInfo->Action));

	for(i = 0; i < pstSFsmCesoPsnMapChannelInfo->NumOfChannels; i++)
		VCORE_PutParamPayloadMsg(&stMsg,
			UT_CPU2LE16(pstSFsmCesoPsnMapChannelInfo->pusLinkedChannels[i]));

	/* update the message length. Take in account the potential padding */
	VCORE_SetPaddingMsg(&stMsg);

	Status = VCORE_SendMSPReq(pstSupvsrChnl,&stMsg);
	
	return Status;
}

/****************************************************************************
 * VDEV_SetConfLatency: The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param usLatency the value of latency.
 */
VSTATUS VDEV_SetConfLatency(IN SChnl* pstChnl, IN U16 usLatency)
{
#define NUM_OF_PARAMS 1

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_CONF_SET_LATENCY);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(usLatency));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_SetConfDgain: The function does the following things -
 ***************************************************************************/
/*!
 *  \return
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 *  \param usGain the attenuation value.
 */
VSTATUS VDEV_SetConfDgain(IN SChnl* pstChnl, IN U16 usGain)
{
#define NUM_OF_PARAMS 1

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_CONF_SET_DGAIN);

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(usGain));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

/****************************************************************************
 * VDEV_C2CConnectUP : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
    Send TRANS_CONNECT - enable/disable transcoding
 *
 */
VSTATUS VDEV_C2CConnectUP(IN SChnl *pstChnl, IN struct _C2C_CONNECT_UMA_UP *pstC2CConnectParams)
{
#define NUM_OF_PARAMS 10
	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_C2C_CONNECT_UMA_UP);

	if (pstC2CConnectParams->param_4.bits.inter_mode == C2C_CONNECT_UMA_UP_CONNECTION_TYPE_INTER)
		Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS,
							UT_CPU2LE16(pstC2CConnectParams->param_4.word),
							UT_CPU2LE16(pstC2CConnectParams->param_5.word),
							UT_CPU2LE16(pstC2CConnectParams->param_6.word),
							UT_CPU2LE16(pstC2CConnectParams->param_7.word),
							UT_CPU2LE16(pstC2CConnectParams->param_8.word),
							UT_CPU2LE16(pstC2CConnectParams->param_9.word),
							UT_CPU2LE16(pstC2CConnectParams->param_10.word),
							UT_CPU2LE16(pstC2CConnectParams->param_11.word),
							UT_CPU2LE16(pstC2CConnectParams->param_12.word),
							UT_CPU2LE16(pstC2CConnectParams->param_13.word));

	else
		Status = VCORE_SetPayloadMsg(&stMsg, 4, 
							UT_CPU2LE16(pstC2CConnectParams->param_4.word),
							UT_CPU2LE16(pstC2CConnectParams->param_5.word),
							UT_CPU2LE16(pstC2CConnectParams->param_6.word),
							UT_CPU2LE16(pstC2CConnectParams->param_7.word));

	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef  NUM_OF_PARAMS
}


/****************************************************************************
 * BOOT commands -
 ***************************************************************************/

void *AllocBootFifo(IN SChnl * pstChnl, U16 usSize, U16 *pusFifoSize)
{
	/* most of the time the fifo size to allocated is the regular size */
	*pusFifoSize = usSize;

	/* for the M823XX/M829XX device in CSME control the boot message size is 512 */
	if ((pstChnl->pstDev->eItfType == eCSM_ITF) && ((pstChnl->pstDev->ucDevType == DEV_TYPE_M823XX) || 
		(pstChnl->pstDev->ucDevType == DEV_TYPE_M829XX) ||
		(pstChnl->pstDev->ucDevType == DEV_TYPE_M823XX_2)) && (usSize < 512)) 
		*pusFifoSize = 512; 
	else 
		*pusFifoSize = usSize;

	return (UT_Calloc(1, *pusFifoSize));
}

/****************************************************************************
 * VDEV_MaasAssign : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 */
VSTATUS VDEV_MaasAssign(IN SChnl * pstChnl)
{
#define FIFO_LEN (14 + sizeof(SBootHdr))

	gtl_msg_t stMsg;
	SGtlCsmeItf *pstCsmItf = NULL;
	U8 *pMac = NULL;
	VSTATUS Status;
	U16 usMsgSize;

	if (pstChnl->pstDev->eItfType == ePCI_ITF)
	{
		UT_ErrorLog(VCORE, "VDEV_MaasAssign: Invalid Interface type(%d)\n", pstChnl->pstDev->eItfType);
		Status =  VAPI_ERR_UNDEFINED;
		goto out;
	}

	pstCsmItf = &pstChnl->pstDev->pvItfInfo->csme;

	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize);
	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_MAAS_ASSIGN, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	stMsg.fifo_size = usMsgSize;

	pMac = pstCsmItf->devMac;
	UT_Log(VCORE, INFO, "VDEV_MaasAssign: psMac->devMac = 0x%x\n", pMac);
	*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16((pMac[0] << 8) | pMac[1]);
	*((U16 *) stMsg.fifo + 4) = UT_CPU2BE16((pMac[2] << 8) | pMac[3]);
	*((U16 *) stMsg.fifo + 5) = UT_CPU2BE16((pMac[4] << 8) | pMac[5]);

	pMac = pstCsmItf->hostMac;
	UT_Log(VCORE, INFO, "VDEV_MaasAssign: psMac->hostMac = 0x%x\n", pMac);
	*((U16 *) stMsg.fifo + 6) = UT_CPU2BE16((pMac[0] << 8) | pMac[1]);
	*((U16 *) stMsg.fifo + 7) = UT_CPU2BE16((pMac[2] << 8) | pMac[3]);
	*((U16 *) stMsg.fifo + 8) = UT_CPU2BE16((pMac[4] << 8) | pMac[5]);

	*((U16 *) stMsg.fifo + 9) = UT_CPU2BE16((U16) 0x889b);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);
out:
	return Status;

#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_QueryReady : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 */
VSTATUS VDEV_QueryReady(IN SChnl * pstChnl)
{
#define FIFO_LEN (8 + sizeof(SBootHdr))

	gtl_msg_t stMsg;
	SGtlCsmeItf *pstCsmItf = NULL;
	U8 *pMac = NULL;
	VSTATUS Status;
	U16 usMsgSize;

	if (pstChnl->pstDev->eItfType == ePCI_ITF)
	{
		UT_ErrorLog(VCORE, "VDEV_QueryReady: Invalid Interface type(%d)\n", pstChnl->pstDev->eItfType);
		Status =  VAPI_ERR_UNDEFINED;
		goto out;
	}

	pstCsmItf = &pstChnl->pstDev->pvItfInfo->csme;

	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize);

	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_QUERY_READY, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	stMsg.fifo_size = usMsgSize;

	pMac = pstCsmItf->hostMac;
	UT_Log(VCORE, INFO, "VDEV_QueryReady: psMac->hostMac = 0x%x\n", pMac);
	*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16((pMac[0] << 8) | pMac[1]);
	*((U16 *) stMsg.fifo + 4) = UT_CPU2BE16((pMac[2] << 8) | pMac[3]);
	*((U16 *) stMsg.fifo + 5) = UT_CPU2BE16((pMac[4] << 8) | pMac[5]);

	*((U16 *) stMsg.fifo + 6) = UT_CPU2BE16((U16) 0x889b);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);

out:
	return Status;

#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_BRMSetClock : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 */
VSTATUS VDEV_BRMSetClock(SChnl * pstChnl)
{
#define FIFO_LEN (2 + sizeof(SBootHdr))

	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 usMsgSize;

	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize);

	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_SETCLOCK, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	switch(pstChnl->pstDev->eItfType)
	{
	case eCSM_ITF:
		stMsg.fifo_size = usMsgSize;
		*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16(BRM_SETCLOCK_PARAM);
		break;

	case ePCI_ITF:
		stMsg.fifo_size = 0;
		stMsg.mailbox.reg2 = BRM_SETCLOCK_PARAM >> 8;
		break;

	default:
		UT_ErrorLog(VCORE, "VDEV_BRMSetClock: Invalid Interface type\n");
		Status =  VAPI_ERR_UNDEFINED;
		goto out_free;
	}

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);

out:
	return Status;

#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_BRMSetPll : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *	BRM_SET_PLL (PCI)
 * This command sets the final frequency for one of three PLLs as follows:
 * Mailbox 0: Bits [15:0] CLKF 
 * Mailbox 1: Bits [15:8] CLKR		Bits [7:0] CLKOD
 * Mailbox 2: Bits[15:0] PLLID where PLLID can be 0 (ARM PLL), 1 (SPU PLL), 2 (AMBA PLL)
 * 
 *	SET_PLL (CSMENCAPS) [ Command ID =0x1F]
 * This command sets the final frequency for one of three PLLs as followes:
 * Parameter 1 (of the CSMENCPAS Payload) Bits [15:0]  PLLID
 *  PllId can be 0 (ARM PLL), 1 (SPU PLL), 2 (AMBA PLL)
 * ucPlld used to identify the PLL number
 * Parameter2: Bits [15:0] CLKF
 * Parameter3: Bits [15:8] CLKR	Bits [7:0] CLKOD
 * Note that ACK to this command from IBR does not return any value.
 * 
 *  \param pstChnl the channel on which MSP command will be sent
 */
VSTATUS VDEV_BRMSetPll(SChnl * pstChnl, IN SHwCfg * pstHwCfg, IN U8 ucPllId)
{
#define FIFO_LEN (6 + sizeof(SBootHdr))

	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 usMsgSize;
	U16 usClkF;
	U16 usClkROd;

	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize);

	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_SETPLL, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	switch (pstChnl->pstDev->ucDevType)
	{
	case DEV_TYPE_M823XX:
		switch (ucPllId)
		{
		case PLL_BUS_ARM:
			switch (pstHwCfg->usDeviceRefclk)
			{
			case DEVICE_REFCLK_10MHZ:
				usClkROd = PLL_REFCLK_M823XX_10MHZ_ARM_NR << 8 | PLL_REFCLK_M823XX_10MHZ_ARM_OD;
				usClkF = PLL_REFCLK_M823XX_10MHZ_ARM_NF;
				break;
			case DEVICE_REFCLK_25MHZ:
				usClkROd = PLL_REFCLK_M823XX_25MHZ_ARM_NR << 8 | PLL_REFCLK_M823XX_25MHZ_ARM_OD;
				usClkF = PLL_REFCLK_M823XX_25MHZ_ARM_NF;
				break;
			default:
				UT_ErrorLog(VCORE, "VDEV_BRMSetPll: Invalid device frequency type %d\n", pstHwCfg->usDeviceRefclk);
				Status =  VAPI_ERR_UNDEFINED;
				goto out_free;
			}
			break;

		case PLL_BUS_SPU:
			usClkROd = PLL_REFCLK_M823XX_SPU_NR << 8 | PLL_REFCLK_M823XX_SPU_OD;
			usClkF = PLL_REFCLK_M823XX_SPU_NF;
			break;

		case PLL_BUS_AMBA:
			switch (pstHwCfg->usDeviceRefclk)
			{
			case DEVICE_REFCLK_10MHZ: 
				usClkROd = PLL_REFCLK_M823XX_10MHZ_AMBA_NR << 8 | PLL_REFCLK_M823XX_10MHZ_AMBA_OD;
				usClkF = PLL_REFCLK_M823XX_10MHZ_AMBA_NF;
				break;
			case DEVICE_REFCLK_25MHZ: 
				usClkROd = PLL_REFCLK_M823XX_25MHZ_AMBA_NR << 8 | PLL_REFCLK_M823XX_25MHZ_AMBA_OD;
				usClkF = PLL_REFCLK_M823XX_25MHZ_AMBA_NF;
				break;
			default:
				UT_ErrorLog(VCORE, "VDEV_BRMSetPll: Invalid device frequency type %d\n", pstHwCfg->usDeviceRefclk);
				Status =  VAPI_ERR_UNDEFINED;
				goto out_free;
			}
			break;

		default:
			UT_ErrorLog(VCORE, "VDEV_BRMSetPll: Invalid PLL ID %d\n", ucPllId);
			Status =  VAPI_ERR_UNDEFINED;
			goto out_free;
		}

		break;

	case DEV_TYPE_M823XX_2:
		switch (ucPllId)
		{
		case PLL_BUS_ARM:
			switch (pstHwCfg->usDeviceAmbaClkM823V2)
			{
			case DEVICE_AMBA_M823V2_150MHZ:
				usClkROd = PLL_REFCLK_M823V2_ARM_550MHZ_NR << 8 | PLL_REFCLK_M823V2_ARM_550MHZ_OD;
				usClkF = PLL_REFCLK_M823V2_ARM_550MHZ_NF;
				break;
			case DEVICE_AMBA_M823V2_137MHZ:
				usClkROd = PLL_REFCLK_M823V2_ARM_450MHZ_NR << 8 | PLL_REFCLK_M823V2_ARM_450MHZ_OD;
				usClkF = PLL_REFCLK_M823V2_ARM_450MHZ_NF;
				break;
			default:
				UT_ErrorLog(VCORE, "VDEV_BRMSetPll: Invalid device frequency type %d\n", pstHwCfg->usDeviceAmbaClkM823V2);
				Status =  VAPI_ERR_UNDEFINED;
				goto out_free;
			}
			break;

		case PLL_BUS_SPU:
			switch (pstHwCfg->usDeviceAmbaClkM823V2)
			{
			case DEVICE_AMBA_M823V2_150MHZ:
				usClkROd = PLL_REFCLK_M823V2_SPU_425MHZ_NR  << 8 | PLL_REFCLK_M823V2_SPU_425MHZ_OD;
				usClkF = PLL_REFCLK_M823V2_SPU_425MHZ_NF;
				break;
			case DEVICE_AMBA_M823V2_137MHZ:
				usClkROd = PLL_REFCLK_M823V2_SPU_350MHZ_NR << 8 | PLL_REFCLK_M823V2_SPU_350MHZ_OD;
				usClkF = PLL_REFCLK_M823V2_SPU_350MHZ_NF;
				break;
			default:
				UT_ErrorLog(VCORE, "VDEV_BRMSetPll: Invalid device frequency type %d\n", pstHwCfg->usDeviceAmbaClkM823V2);
				Status =  VAPI_ERR_UNDEFINED;
				goto out_free;
			}
			break;

		case PLL_BUS_AMBA:
			switch (pstHwCfg->usDeviceAmbaClkM823V2)
			{
			case DEVICE_AMBA_M823V2_150MHZ:
				usClkROd = PLL_REFCLK_M823V2_AMBA_150MHZ_NR  << 8 | PLL_REFCLK_M823V2_AMBA_150MHZ_OD;
				usClkF = PLL_REFCLK_M823V2_AMBA_150MHZ_NF;
				break;
			case DEVICE_AMBA_M823V2_137MHZ:
				usClkROd = PLL_REFCLK_M823V2_AMBA_137MHZ_NR  << 8 | PLL_REFCLK_M823V2_AMBA_137MHZ_OD;
				usClkF = PLL_REFCLK_M823V2_AMBA_137MHZ_NF;
				break;
			default:
				UT_ErrorLog(VCORE, "VDEV_BRMSetPll: Invalid device frequency type %d\n", pstHwCfg->usDeviceAmbaClkM823V2);
				Status =  VAPI_ERR_UNDEFINED;
				goto out_free;
			}
			break;

		default:
			UT_ErrorLog(VCORE, "VDEV_BRMSetPll: Invalid PLL ID %d\n", ucPllId);
			Status =  VAPI_ERR_UNDEFINED;
			goto out_free;
		}

		break;

	default:
		UT_ErrorLog(VCORE, "VDEV_BRMSetPll: Invalid Device type %d\n", pstChnl->pstDev->ucDevType);
		Status =  VAPI_ERR_UNDEFINED;
		goto out_free;
	}

	switch(pstChnl->pstDev->eItfType)
	{
	case eCSM_ITF:
		stMsg.fifo_size = usMsgSize;
		*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16(ucPllId);
		*((U16 *) stMsg.fifo + 4) = UT_CPU2BE16(usClkF);
		*((U16 *) stMsg.fifo + 5) = UT_CPU2BE16(usClkROd);
		break;

	case ePCI_ITF:
		stMsg.fifo_size = 0;
		stMsg.mailbox.reg2 = ucPllId;
		stMsg.mailbox.reg1 = usClkROd;
		stMsg.mailbox.reg0 = usClkF;
		break;

	default:
		UT_ErrorLog(VCORE, "VDEV_BRMSetPll: Invalid Interface type\n");
		Status =  VAPI_ERR_UNDEFINED;
		goto out_free;
	}

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);

out:
	return Status;

#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_BRMArmSetClkMode : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 */
VSTATUS VDEV_BRMArmSetClkMode(SChnl * pstChnl)
{
#define FIFO_LEN (2 + sizeof(SBootHdr))
	gtl_msg_t stMsg;
	VSTATUS Status;

	U16 usMsgSize;

	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize);

	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_SET_ARM_CLKMODE, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	switch(pstChnl->pstDev->eItfType)
	{
	case eCSM_ITF:
		stMsg.fifo_size = usMsgSize;
		*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16(BRM_ARM_CLKMODE_PARAM);
		break;

	case ePCI_ITF:
		stMsg.fifo_size = 0;
		stMsg.mailbox.reg2 = BRM_ARM_CLKMODE_PARAM;
		break;

	default:
		UT_ErrorLog(VCORE, "VDEV_BRMArmSetClkMode: Invalid Interface type\n");
		Status =  VAPI_ERR_UNDEFINED;
		goto out_free;
	}

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);

out:
	return Status;


#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_BRMSetCSParam : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 */
VSTATUS VDEV_BRMSetCSParam(SChnl * pstChnl, IN SHwCfg * pstHwCfg)
{
#define FIFO_LEN (6 + sizeof(SBootHdr))

	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 usCsSize;
	U16 usMsgSize;

	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize);

	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_SET_CS_PARAM, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	if ((pstChnl->pstDev->ucDevType == DEV_TYPE_M829XX) || 
		(pstChnl->pstDev->ucDevType == DEV_TYPE_M823XX) ||
		(pstChnl->pstDev->ucDevType == DEV_TYPE_M823XX_2))
		usCsSize = CS0_PARAM_SIZE_32M;
	else
		usCsSize = CS0_PARAM_SIZE_16M;

	switch(pstChnl->pstDev->eItfType)
	{
	case eCSM_ITF:
		stMsg.fifo_size = usMsgSize;
		*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16(0);	/* chip select 0 */
		*((U16 *) stMsg.fifo + 4) = UT_CPU2BE16(pstHwCfg->usCSparamTiming);
		*((U16 *) stMsg.fifo + 5) = UT_CPU2BE16(usCsSize);
		break;

	case ePCI_ITF:
		stMsg.fifo_size = 0;
		stMsg.mailbox.reg2 = UT_CPU2LE16(0);		/* chip select 0 */
		stMsg.mailbox.reg1 = pstHwCfg->usCSparamTiming;
		stMsg.mailbox.reg0 = usCsSize;
		break;

	default:
		UT_ErrorLog(VCORE, "VDEV_BRMSetCSParam: Invalid Interface type\n");
		Status =  VAPI_ERR_UNDEFINED;
		goto out_free;
	}

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);

out:
	return Status;

#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_BRMSetSdramParam : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 */
VSTATUS VDEV_BRMSetSdramParam(SChnl * pstChnl, IN SHwCfg * pstHwCfg)
{
#define FIFO_LEN (10 + sizeof(SBootHdr))

	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 usMsgSize;
	
	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize);

	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_SET_SDRAM_PARAM, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	switch(pstChnl->pstDev->eItfType)
	{
	case eCSM_ITF:
		stMsg.fifo_size = usMsgSize;
		*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16(pstHwCfg->usSdramParamTiming1);
		*((U16 *) stMsg.fifo + 4) = UT_CPU2BE16(pstHwCfg->usSdramParamTiming2);
		*((U16 *) stMsg.fifo + 5) = UT_CPU2BE16(pstHwCfg->usSdramRefresh);
		*((U16 *) stMsg.fifo + 6) = UT_CPU2BE16(pstHwCfg->usSdramPoweron);
		break;

	case ePCI_ITF:
		stMsg.fifo_size = 0;
		stMsg.mailbox.reg2 = pstHwCfg->usSdramParamTiming1;
		stMsg.mailbox.reg1 = pstHwCfg->usSdramParamTiming2;
		stMsg.mailbox.reg0 = pstHwCfg->usSdramRefresh;
		break;

	default:
		UT_ErrorLog(VCORE, "VDEV_BRMSetSdramParam: Invalid Interface type\n");
		Status =  VAPI_ERR_UNDEFINED;
		goto out_free;
	}

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);

out:
	return Status;

#undef FIFO_LEN
}

VSTATUS VDEV_BRMSetSdramParam2(SChnl *pstChnl, IN SHwCfg *pstHwCfg)
{
#define FIFO_LEN (4 + sizeof(SBootHdr))

	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 usMsgSize; 

	UT_MemSet(&stMsg, 0, sizeof(stMsg));

	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize); 

	if (stMsg.fifo == NULL)
	{ 
		Status = VAPI_ERR_NOMEM; 
		goto out; 	
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_SET_SDRAM_PARAMS_2, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	switch (pstChnl->pstDev->eItfType)
	{
	case eCSM_ITF:
		stMsg.fifo_size = usMsgSize;
		*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16(pstHwCfg->usSdramPoweron);
		break;

	case ePCI_ITF:
		stMsg.fifo_size = 0;
		stMsg.mailbox.reg2 = pstHwCfg->usSdramPoweron;
		stMsg.mailbox.reg1 = 0;
		stMsg.mailbox.reg0 = 0;
		break;

	default:
		UT_ErrorLog(VCORE, "VDEV_BRMSetSdramParam: Invalid Interface type\n");
		Status =  VAPI_ERR_UNDEFINED;
		goto out_free;
	}  

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);	

out:
	return Status;

#undef FIFO_LEN 
}

/****************************************************************************
 * VDEV_CsmDownload : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl The channel on which MSP command will be sent
 *  \param pvBuffer The data to be downloaded
 *  \param usSize The size of buffer
 *  \param uiLoadAddr The Address on which the data should be written
 */
VSTATUS VDEV_CsmDownload(IN SChnl * pstChnl, IN void *pvBuffer, IN U16 usSize, IN U32 uiLoadAddr)
{
	S32 iNoOfParam;
	gtl_msg_t stMsg;
	U16 *pusSrcData = (U16 *) pvBuffer;
	S32 iCount;
	U16 usTemp;
	VSTATUS Status;
	U16 usMsgSize;

	stMsg.fifo = AllocBootFifo(pstChnl, sizeof(SBootHdr) + 6 +  /* boot hdr + 3 params*/
					(usSize * sizeof(U8)) + PADDING_SIZE,  /* + payload + potential padding*/
					&usMsgSize);

	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_FIFOWRITE, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	switch(pstChnl->pstDev->eItfType)
	{
	case eCSM_ITF:
		stMsg.fifo_size = usMsgSize;
		*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16(uiLoadAddr >> 16);
		*((U16 *) stMsg.fifo + 4) = UT_CPU2BE16(uiLoadAddr & 0xFFFF);
		*((U16 *) stMsg.fifo + 5) = UT_CPU2BE16(usSize);

		iNoOfParam = usSize / 2;
		for (iCount = 6; iCount < iNoOfParam + 6; iCount++)
		{
			usTemp = *pusSrcData;
			*((U16 *) stMsg.fifo + iCount) = usTemp;
			pusSrcData++;
		}

		break;

	case ePCI_ITF:
		stMsg.fifo_size = usSize;
		stMsg.mailbox.reg2 = uiLoadAddr & 0xFFFF;
		stMsg.mailbox.reg1 = uiLoadAddr >> 16;
		stMsg.mailbox.reg0 = usSize;

		UT_MemCopy(stMsg.fifo, pvBuffer, usSize);

		break;

	default:
		UT_ErrorLog(VCORE, "VDEV_CsmDownload: Invalid Interface type\n");
		Status =  VAPI_ERR_UNDEFINED;
		goto out_free;
	}

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);

out:
	return Status;
}

/****************************************************************************
 * VDEV_BRMFifoRead : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl The channel on which MSP command will be sent
 *  \param ulReadAddr The Address from which the data should be read 
 *  \param usSize The number of bytes to read 
 */
VSTATUS VDEV_BRMFifoRead(IN SChnl * pstChnl, IN U32 ulReadAddr, IN U16 usSize)
{
#define FIFO_LEN (6 + sizeof(SBootHdr))
	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 usMsgSize;

	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize);

	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_FIFOREAD, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	switch(pstChnl->pstDev->eItfType)
	{
	case eCSM_ITF:
		stMsg.fifo_size = usMsgSize;
		*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16(ulReadAddr >> 16);
		*((U16 *) stMsg.fifo + 4) = UT_CPU2BE16(ulReadAddr & 0xFFFF);
		*((U16 *) stMsg.fifo + 5) = UT_CPU2BE16(usSize);
		break;

	case ePCI_ITF:
		stMsg.fifo_size = 0;
		stMsg.mailbox.reg2 = ulReadAddr & 0xFFFF;
		stMsg.mailbox.reg1 = ulReadAddr >> 16;
		stMsg.mailbox.reg0 = usSize;
		break;

	default:
		UT_ErrorLog(VCORE, "VDEV_BRMFifoRead: Invalid Interface type\n");
		Status =  VAPI_ERR_UNDEFINED;
		goto out_free;
	}

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);

out:
	return Status;

#undef FIFO_LEN

}

/****************************************************************************
 * VDEV_DwordWrite : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl The channel on which MSP command will be sent
 *  \param uAddr The Address to which the data have to be written 
 *  \param uData The data to be written 
 */
VSTATUS VDEV_DwordWrite(SChnl * pstChnl, U32 uAddr, U32 uData)
{
#define FIFO_LEN (10 + sizeof(SBootHdr))

	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 usMsgSize;

	if (pstChnl->pstDev->eItfType == ePCI_ITF)
	{
		UT_ErrorLog(VCORE, "VDEV_DwordWrite: Invalid Interface type(%d)\n", pstChnl->pstDev->eItfType);
		Status =  VAPI_ERR_UNDEFINED;
		goto out;
	}

	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize);

	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_DOUBLEWORDWRITE_ADDR, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	stMsg.fifo_size = usMsgSize;

	*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16(uAddr >> 16);
	*((U16 *) stMsg.fifo + 4) = UT_CPU2BE16(uAddr & 0x0000FFFF);
	*((U16 *) stMsg.fifo + 5) = UT_CPU2BE16(uData >> 16);
	*((U16 *) stMsg.fifo + 6) = UT_CPU2BE16(uData & 0x0000FFFF);
	*((U16 *) stMsg.fifo + 7) = UT_CPU2BE16(0);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);

out:
	return Status;

#undef FIFO_LEN
}

VSTATUS VDEV_BRMDwordRead(SChnl * pstChnl, U32 uAddr)
{
#define FIFO_LEN (6 + sizeof(SBootHdr))

	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 usMsgSize;

	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize);

	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_DOUBLEWORDREAD, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	switch(pstChnl->pstDev->eItfType)
	{
	case eCSM_ITF:
		stMsg.fifo_size = usMsgSize;
		*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16(uAddr >> 16);
		*((U16 *) stMsg.fifo + 4) = UT_CPU2BE16(uAddr & 0x0000FFFF);
		break;

	case ePCI_ITF:
		stMsg.fifo_size = 0;
		stMsg.mailbox.reg2 = uAddr & 0xFFFF;
		stMsg.mailbox.reg1 = uAddr >> 16;
		break;

	default:
		UT_ErrorLog(VCORE, "VDEV_BRMDwordRead: Invalid Interface type\n");
		Status =  VAPI_ERR_UNDEFINED;
		goto out_free;
	}

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);

out:
	return Status;

#undef FIFO_LEN
}

VSTATUS VDEV_BRMDwordWriteAddr(SChnl * pstChnl, U32 uAddr)
{
#define FIFO_LEN (6 + sizeof(SBootHdr))

	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 usMsgSize;

	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize);

	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_DOUBLEWORDWRITE_ADDR, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	switch(pstChnl->pstDev->eItfType)
	{
	case eCSM_ITF:
		stMsg.fifo_size = usMsgSize;
		*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16(uAddr >> 16);
		*((U16 *) stMsg.fifo + 4) = UT_CPU2BE16(uAddr & 0x0000FFFF);
		break;

	case ePCI_ITF:
		stMsg.fifo_size = 0;
		stMsg.mailbox.reg2 = uAddr & 0xFFFF;
		stMsg.mailbox.reg1 = uAddr >> 16;
		break;

	default:
		UT_ErrorLog(VCORE, "VDEV_BRMDwordWriteAddr: Invalid Interface type\n");
		Status =  VAPI_ERR_UNDEFINED;
		goto out_free;
	}

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);

out:
	return Status;

#undef FIFO_LEN
}

VSTATUS VDEV_BRMDwordWriteData(SChnl * pstChnl, U32 uData)
{
#define FIFO_LEN (6 + sizeof(SBootHdr))

	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 usMsgSize;

	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize);

	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_DOUBLEWORDWRITE_DATA, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	switch(pstChnl->pstDev->eItfType)
	{
	case eCSM_ITF:
		stMsg.fifo_size = usMsgSize;
		*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16(uData >> 16);
		*((U16 *) stMsg.fifo + 4) = UT_CPU2BE16(uData & 0x0000FFFF);
		break;

	case ePCI_ITF:
		stMsg.fifo_size = 0;
		stMsg.mailbox.reg2 = uData & 0xFFFF;
		stMsg.mailbox.reg1 = uData >> 16;
		break;

	default:
		UT_ErrorLog(VCORE, "VDEV_BRMDwordWriteAddr: Invalid Interface type\n");
		Status =  VAPI_ERR_UNDEFINED;
		goto out_free;
	}

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);

out:
	return Status;

#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_RunMsp : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl The channel on which MSP command will be sent
 *  \param uiStartAddr 32bit data MSP program start address.
 */
VSTATUS VDEV_RunMsp(IN SChnl * pstChnl, IN U32 uiStartAddr)
{
#define FIFO_LEN (4 + sizeof(SBootHdr))

	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 usMsgSize;

	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize);

	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_PROGSTART, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	switch(pstChnl->pstDev->eItfType)
	{
	case eCSM_ITF:
		stMsg.fifo_size = usMsgSize;
		*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16(uiStartAddr >> 16);
		*((U16 *) stMsg.fifo + 4) = UT_CPU2BE16(uiStartAddr & 0x0000FFFF);
		break;

	case ePCI_ITF:
		stMsg.fifo_size = 0;
		stMsg.mailbox.reg2 = uiStartAddr & 0xFFFF;
		stMsg.mailbox.reg1 = uiStartAddr >> 16;
		break;

	default:
		UT_ErrorLog(VCORE, "VDEV_RunMsp: Invalid Interface type\n");
		Status =  VAPI_ERR_UNDEFINED;
		goto out_free;
	}

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);

out:
	return Status;

#undef FIFO_LEN
}

VSTATUS VDEV_WordWrite(SChnl * pstChnl, U32 uAddr, U16 uData)
{
#define FIFO_LEN (6 + sizeof(SBootHdr))

	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 usMsgSize;

	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize);

	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_WORDWRITE, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	switch(pstChnl->pstDev->eItfType)
	{
	case eCSM_ITF:
		stMsg.fifo_size = usMsgSize;
		*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16(uAddr >> 16);
		*((U16 *) stMsg.fifo + 4) = UT_CPU2BE16(uAddr & 0x0000FFFF);
		*((U16 *) stMsg.fifo + 5) = UT_CPU2BE16(uData);
		break;

	case ePCI_ITF:
		stMsg.fifo_size = 0;
		stMsg.mailbox.reg2 = uAddr & 0xFFFF;
		stMsg.mailbox.reg1 = uAddr >> 16;
		stMsg.mailbox.reg0 = uData;
		break;

	default:
		UT_ErrorLog(VCORE, "VDEV_WordWrite: Invalid Interface type\n");
		Status =  VAPI_ERR_UNDEFINED;
		goto out_free;
	}
	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);

out:
	return Status;

#undef FIFO_LEN
}


VSTATUS VDEV_WordRead(SChnl * pstChnl, U32 uAddr)
{
#define FIFO_LEN (14 + sizeof(SBootHdr))

	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 usMsgSize;

	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize);

	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_WORDREAD, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;

	switch(pstChnl->pstDev->eItfType)
	{
	case eCSM_ITF:
		stMsg.fifo_size = usMsgSize;
		*((U16 *) stMsg.fifo + 3) = UT_CPU2BE16(uAddr >> 16);
		*((U16 *) stMsg.fifo + 4) = UT_CPU2BE16(uAddr & 0x0000FFFF);
		break;
	
	case ePCI_ITF:
		stMsg.fifo_size = 0;
		stMsg.mailbox.reg2 = uAddr & 0xFFFF;
		stMsg.mailbox.reg1 = uAddr >> 16;
		break;

	default:
		UT_ErrorLog(VCORE, "VDEV_WordRead: Invalid Interface type\n");
		Status =  VAPI_ERR_UNDEFINED;
		goto out_free;
	}

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);

out:
	return Status;

#undef FIFO_LEN
}

/****************************************************************************
 * VSTATUS VDEV_RunCheckSum : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 */
VSTATUS VDEV_RunCheckSum(IN SChnl * pstChnl, IN U32 uiLoadAddr)
{
#define FIFO_LEN (4 + sizeof(SBootHdr))

	gtl_msg_t stMsg;
	VSTATUS Status;
	U16 usMsgSize;

	UT_Log(VCORE, INFO, "VDEV_RunCheckSum: addr == 0x%08x\n", uiLoadAddr);

	stMsg.fifo = AllocBootFifo(pstChnl, FIFO_LEN + PADDING_SIZE, &usMsgSize);

	if (stMsg.fifo == NULL)
	{
		Status =  VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_RUN_CHKSUM, pstChnl->pstDev->eItfType);
	if (Status != SUCCESS)
		goto out_free;;

	switch(pstChnl->pstDev->eItfType)
	{
	case eCSM_ITF:
		stMsg.fifo_size = usMsgSize;
		*((U8 *) stMsg.fifo + 6) = uiLoadAddr >> 24;	/*MSB of checksum addr*/
		*((U8 *) stMsg.fifo + 7) = uiLoadAddr >> 16;
		*((U8 *) stMsg.fifo + 8) = uiLoadAddr >> 8;	/*LSB of checksum addr*/
		*((U8 *) stMsg.fifo + 9) = uiLoadAddr;

		break;

	case ePCI_ITF:
		stMsg.fifo_size = 0;
		stMsg.mailbox.reg2 = uiLoadAddr & 0xFFFF;	/*LSB of checksum addr*/
		stMsg.mailbox.reg1 = uiLoadAddr >> 16;		/*MSB of checksum addr*/

		break;

	default:
		UT_ErrorLog(VCORE, "VDEV_RunCheckSum: Invalid Interface type\n");
		Status =  VAPI_ERR_UNDEFINED;
		goto out_free;
	}

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out_free:
	UT_FreeMem(stMsg.fifo);

out:
	return Status;
#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_POSPhyIdAssign : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 */
VSTATUS VDEV_POSPhyIdAssign(IN SChnl * pstChnl)
{
#define FIFO_LEN (8 + (2*4))

	gtl_msg_t stMsg;
	U16 ausFifo[FIFO_LEN / 2 + 1];
	VSTATUS Status;

	U32 uiPhyId;			/* Valid range 0 to 30 */
	SGtlPosItf *pstPosItfData;	/* To ge PHY ID to assign. We are extracting the 
					   POS data here instead on VAPI_BootDevice just 
					   to avoid too much POS specific code. */

	VCORE_InitMsg(&stMsg, (U8*)ausFifo, sizeof(ausFifo));

	if (pstChnl->pstDev->eItfType != ePOS_ITF)
	{
		UT_ErrorLog(VCORE, "VDEV_POSPhyIdAssign: Invalid Interface type\n");
		Status = VAPI_ERR_UNDEFINED;
		goto out;
	}

	pstPosItfData = &pstChnl->pstDev->pvItfInfo->pos;
	uiPhyId = ((PosPathInfo *) pstPosItfData->pvAdptData)->ucPhyId;

	if (uiPhyId > 30)
	{
		UT_ErrorLog(VCORE, "VDEV_POSPhyIdAssign: Invalid Phy Id(%lu). Valid range is 0 to 30\n", uiPhyId);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}


	UT_MemSet(ausFifo, 0, sizeof(ausFifo));
	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_CMD_ACK, pstChnl->pstDev->eItfType);

	stMsg.fifo_size = FIFO_LEN;
	*((U16 *) stMsg.fifo + 3) = 0;	/* To avoid alignment issues */
	*((U32 *) stMsg.fifo + 2) = UT_CPU2LE32(CMD_TYPE_POS_PHYID_ASSIGN);
	*((U32 *) stMsg.fifo + 3) = UT_CPU2LE32(uiPhyId);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;

#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_POSSetPLL : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 */
VSTATUS VDEV_POSSetPLL(IN SChnl * pstChnl)
{
#define FIFO_LEN (8 + (2*4))

	gtl_msg_t stMsg;
	U16 ausFifo[FIFO_LEN / 2 + 1];
	VSTATUS Status;

	VCORE_InitMsg(&stMsg, (U8*)ausFifo, sizeof(ausFifo));

	if (pstChnl->pstDev->eItfType != ePOS_ITF)
	{
		UT_ErrorLog(VCORE, "VDEV_POSSetPLL: Invalid Interface type\n");
		Status = VAPI_ERR_UNDEFINED;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_CMD_ACK, pstChnl->pstDev->eItfType);

	stMsg.fifo_size = FIFO_LEN;
	*((U16 *) stMsg.fifo + 3) = 0;	/* To avoid alignment issues */
	*((U32 *) stMsg.fifo + 2) = UT_CPU2LE32(CMD_TYPE_POS_SET_PLL);
	*((U32 *) stMsg.fifo + 3) = UT_CPU2LE32(25);	/* !@0 BRM_CLOCK_MULTIPLIER is 20, so can't use here */

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;

#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_POSSetSDRAM : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 */
VSTATUS VDEV_POSSetSDRAM(IN SChnl * pstChnl, IN SHwCfg * pstHwCfg)
{
#define FIFO_LEN (8 + (7*4))

	gtl_msg_t stMsg;
	U16 ausFifo[FIFO_LEN / 2 + 1];
	VSTATUS Status;

	VCORE_InitMsg(&stMsg, (U8*)ausFifo, sizeof(ausFifo));

	if (pstChnl->pstDev->eItfType != ePOS_ITF)
	{
		UT_ErrorLog(VCORE, "VDEV_POSSetSDRAM: Invalid Interface type\n");
		Status = VAPI_ERR_UNDEFINED;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_CMD_ACK, pstChnl->pstDev->eItfType);

	stMsg.fifo_size = FIFO_LEN;
	*((U16 *) stMsg.fifo + 3) = 0;	/* To avoid alignment issues */
	*((U32 *) stMsg.fifo + 2) = UT_CPU2LE32(CMD_TYPE_POS_SET_SDRAM);
	*((U32 *) stMsg.fifo + 3) = UT_CPU2LE32(pstHwCfg->usSdramParamTiming1);

	/* !@0 7th Bit is kept 0 in POS boot case. */
	*((U32 *) stMsg.fifo + 4) = UT_CPU2LE32((pstHwCfg->usSdramParamTiming2 & 0x3f));

	*((U32 *) stMsg.fifo + 5) = UT_CPU2LE32(pstHwCfg->usSdramRefresh);
	*((U32 *) stMsg.fifo + 6) = 0;
	*((U32 *) stMsg.fifo + 7) = UT_CPU2LE32(pstHwCfg->usCSparamTiming);

	if (pstChnl->pstDev->ucDevType == DEV_TYPE_M829XX)
		*((U32 *) stMsg.fifo + 8) = UT_CPU2LE32(CS0_PARAM_SIZE_32M);
	else
		*((U32 *) stMsg.fifo + 8) = UT_CPU2LE32(CS0_PARAM_SIZE_16M);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);
out:
	return Status;

#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_POSChksum : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 */
VSTATUS VDEV_POSChksum(IN SChnl * pstChnl, IN U32 uiAddress)
{
#define FIFO_LEN (8 + (2*4))

	gtl_msg_t stMsg;
	U16 ausFifo[FIFO_LEN / 2 + 1];
	VSTATUS Status;

	VCORE_InitMsg(&stMsg, (U8*)ausFifo, sizeof(ausFifo));

	if (pstChnl->pstDev->eItfType != ePOS_ITF)
	{
		UT_ErrorLog(VCORE, "VDEV_POSChksum: Invalid Interface type\n");
		Status = VAPI_ERR_UNDEFINED;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_CMD_ACK, pstChnl->pstDev->eItfType);

	stMsg.fifo_size = FIFO_LEN;
	*((U16 *) stMsg.fifo + 3) = 0;	/* To avoid alignment issues */
	*((U32 *) stMsg.fifo + 2) = UT_CPU2LE32(CMD_TYPE_POS_CHKSUM);
	*((U32 *) stMsg.fifo + 3) = UT_CPU2LE32(uiAddress);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;

#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_POSProgStart : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl the channel on which MSP command will be sent
 */
VSTATUS VDEV_POSProgStart(IN SChnl * pstChnl, IN U32 uiStartAddr)
{
#define FIFO_LEN (8 + (2*4))

	gtl_msg_t stMsg;
	U16 ausFifo[FIFO_LEN / 2 + 1];
	VSTATUS Status;

	VCORE_InitMsg(&stMsg, (U8*)ausFifo, sizeof(ausFifo));

	if (pstChnl->pstDev->eItfType != ePOS_ITF)
	{
		UT_ErrorLog(VCORE, "VDEV_POSProgStart: Invalid Interface type\n");
		Status = VAPI_ERR_UNDEFINED;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_CMD_ACK, pstChnl->pstDev->eItfType);

	stMsg.fifo_size = FIFO_LEN;
	*((U16 *) stMsg.fifo + 3) = 0;	/* To avoid alignment issues */
	*((U32 *) stMsg.fifo + 2) = UT_CPU2LE32(CMD_TYPE_POS_PROGSTART);
	*((U32 *) stMsg.fifo + 3) = UT_CPU2LE32(uiStartAddr);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;

#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_POSDownload : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl The channel on which MSP command will be sent
 *  \param pvBuffer The data to be downloaded
 *  \param usSize The size of buffer
 *  \param uiLoadAddr The Address on which the data should be written
 */
VSTATUS VDEV_POSDownload(IN SChnl * pstChnl, IN void *pvBuffer, IN U16 usSize, IN U32 uiLoadAddr)
{
	gtl_msg_t stMsg;
	U8 *puchSrcData = (U8 *) pvBuffer;
	VSTATUS Status;

	if (pstChnl->pstDev->eItfType != ePOS_ITF)
	{
		UT_ErrorLog(VCORE, "VDEV_POSDownload: Invalid Interface type\n");
		Status = VAPI_ERR_UNDEFINED;
		goto out;
	}

	stMsg.fifo_size = usSize + 8 + 3 * 4;	/* 6 for INDEX, CC, CT FC + alignment fix, 12 for  command */

	stMsg.fifo = (void *)UT_AllocMem(stMsg.fifo_size * sizeof(U8));
	if (stMsg.fifo == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_CMD_ACK, pstChnl->pstDev->eItfType);

	*((U16 *) stMsg.fifo + 3) = 0;	/* To avoid alignment issues */
	*((U32 *) stMsg.fifo + 2) = UT_CPU2LE32(CMD_TYPE_POS_MEMWRITE);
	*((U32 *) stMsg.fifo + 3) = UT_CPU2LE32(uiLoadAddr);
	*((U32 *) stMsg.fifo + 4) = UT_CPU2LE32(usSize);

	UT_MemCopy(((U8 *) stMsg.fifo + 20), puchSrcData, usSize);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);
	UT_FreeMem(stMsg.fifo);

out:
	return Status;
}

/****************************************************************************
 * VDEV_POSMemRead : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return 
 *  SUCCESS\n
 *  or the status given by GTL\n
 *
 *  \param pstChnl The channel on which MSP command will be sent
 *  \param pvBuffer The data to be downloaded
 *  \param usSize The size of buffer
 *  \param uiLoadAddr The Address on which the data should be written
 */
VSTATUS VDEV_POSMemRead(IN SChnl * pstChnl, IN U32 ulReadAddr, IN U16 usSize)
{
#define FIFO_LEN (8 + (3*4))

	gtl_msg_t stMsg;
	U16 ausFifo[FIFO_LEN / 2 + 1];
	VSTATUS Status;

	VCORE_InitMsg(&stMsg, (U8*)ausFifo, sizeof(ausFifo));

	if (pstChnl->pstDev->eItfType != ePOS_ITF)
	{
		UT_ErrorLog(VCORE, "VDEV_POSDownload: Invalid Interface type\n");
		Status = VAPI_ERR_UNDEFINED;
		goto out;
	}

	Status = VCORE_SetupBootCmdMsg(&stMsg, CMD_TYPE_BRM_CMD_ACK, pstChnl->pstDev->eItfType);

	stMsg.fifo_size = FIFO_LEN;
	*((U16 *) stMsg.fifo + 3) = 0;	/* To avoid alignment issues */
	*((U32 *) stMsg.fifo + 2) = UT_CPU2LE32(0x0A);  /*CMD_TYPE_POS_MEMREAD */
	*((U32 *) stMsg.fifo + 3) = UT_CPU2LE32(ulReadAddr);
	*((U32 *) stMsg.fifo + 4) = UT_CPU2LE32(usSize);

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;

#undef FIFO_LEN
}

/****************************************************************************
 * VDEV_ChannelsReset
 ***************************************************************************/
VSTATUS VDEV_ChannelsReset(IN SChnl * pstChnl)
{
#define NUM_OF_PARAMS 1

	VSTATUS Status;
	gtl_msg_t stMsg;
	U8 ausFifo[sizeof(SApiHdr) + NUM_OF_PARAMS * 2 + PADDING_SIZE];

	/* init msg fields to default values*/
	VCORE_InitMsg(&stMsg, ausFifo, sizeof(ausFifo));

	VCORE_SetupCmdHdrMsg(&stMsg, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_CHANNELS_RESET );

	Status = VCORE_SetPayloadMsg(&stMsg, NUM_OF_PARAMS, UT_CPU2LE16(1));
	if (Status != SUCCESS)
		goto out;

	Status = VCORE_SendMSPReq(pstChnl, &stMsg);

out:
	return Status;
#undef NUM_OF_PARAMS
}

