/*! \file misc_itf.c
 @defgroup misc Miscellaneous API functions
 *  @{
 */

/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include "ut.h"
#include "vapi.h"
#include "dmgr.h"
#include "vcore.h"
#include "vcore_voip.h"
#include "appitf.h"
#include "cfg.h"
#include "msp.h"

Boolean bVapiInitialized_g = False;
VAPI_Statistics vapi_stat;

/***************************************************************************
 * VAPI_InitStatisticsCounters: The function inits Statistics counters -
 ***************************************************************************/

void VAPI_InitStatisticsCounters(void)
{

	UT_MutexInit(&vapi_stat.mutex_stat);
	vapi_stat.vapi_frames_completed = 0;
	vapi_stat.vapi_frames_queued = 0;
	vapi_stat.api_frames_received = 0;
	vapi_stat.api_error_frames_received = 0;
	vapi_stat.api_frames_sent = 0;

	return;
}

/***************************************************************************
 * VAPI_PrintStatistics: The function prints statistics -
 ***************************************************************************/

void VAPI_PrintStatistics(void)
{
	UT_MutexLock(&vapi_stat.mutex_stat);
	UT_Log(APPITF, INFO, "vapi_frames_completed=%u\n", vapi_stat.vapi_frames_completed);
	UT_Log(APPITF, INFO, "vapi_frames_queued=%u\n", vapi_stat.vapi_frames_queued);
	UT_Log(APPITF, INFO, "api_frames_received=%u\n", vapi_stat.api_frames_received);
	UT_Log(APPITF, INFO, "api_error_frames_received=%u\n", vapi_stat.api_error_frames_received);
	UT_Log(APPITF, INFO, "api_frames_sent=%u\n", vapi_stat.api_frames_sent);
	UT_MutexUnLock(&vapi_stat.mutex_stat);

	return;
}

/***************************************************************************
 * VAPI_GetVersion: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	Returns version information in a string format. \n
*	The string is similar to "VAPI Library Release 2.4.0, API Version 4.0". \n
*	This includes the Library release version that is implementation dependent \n
*	The API version that depends on API changes, that is parameter types, etc. \n
*	This function uses definitions (in vapi.h):
*	\li VAPI_VERSION_MAJOR
*	\li VAPI_VERSION_MINOR
*	\li VAPI_VERSION_PATCH
*	\li VAPI_API_VERSION_MAJOR
*	\li VAPI_API_VERSION_MINOR 
*	\n Some macros are also available to help on version identification:
*	\li VAPI_VERSION
*	\li VAPI_RELEASE
*	\li VAPI_API_VERSION
*	\li VAPI_API_RELEASE \n
*
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">NONE</td>
*		<td style="vertical-align: top;"></td>
*	</tr>
*	</table>
*
*	\n \b Returns: \n
*		Release string \n
*
*	\b Usage: \n
*	\b NOTE: VAPI_RELEASE didn't exist before VAPI 2.4.x \n
*	\include version.c
*
*	\n \b Commands:
*	\li No command sent to the Comcerto 
*/

char *VAPI_GetVersion(void)
{
	static char strVersion[100];

	UT_Snprintf(strVersion, sizeof(strVersion), "VAPI Library Release %d.%d.%d-%s, API Version %d.%d",
				VAPI_VERSION_MAJOR, VAPI_VERSION_MINOR, VAPI_VERSION_PATCH,VAPI_VERSION_EXTRA,
				VAPI_API_VERSION_MAJOR, VAPI_API_VERSION_MINOR);
	return strVersion;
}

/***************************************************************************
 * VAPI_Init: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This function initializes the VAPI Library resources. \n 
*	Before making any other calls to the library,
*	the user must initialize the library by calling this function. \n 
*	VAPI exports a function pointer pfnUsrQueryVapiDefaultsEnumerate \n 
*	for the user to customize VAPI defaults. \n 
*	The user can override VAPI default configuration by using providing a value to this function pointer. \n 
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstDeviceHeaders</td>
*		<td style="vertical-align: top;">Pointer to list of device configuration structures.</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_UNDEFINED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Commands: \n
*	\li No Comcerto commands sent.
*/
VSTATUS VAPI_Init(void *pstDeviceHeaders)
{

	VSTATUS Status;

	if (bVapiInitialized_g)
	{
		UT_Log(APPITF, INFO, "VAPI_Init: VAPI library already initialized.\n");
		return VAPI_ERR_UNDEFINED;
	}

	/*
	 * Initlize UT Module
	 *
	 */
	if (UT_Init() < 0)
	{
		/*Used printf instead of UT_Log because UT is not yet initialized */
		printf("UT_Init Module Failed to initialize\n");
		Status = FAILURE;
		return Status;
	}

	/*Init counters */
	VAPI_InitStatisticsCounters();
	/*
	 * Initlize GTL Layer
	 */
	if ((Status = GTL_Init(pstDeviceHeaders, VCORE_GTLReadCallback)) != SUCCESS)
	{
		UT_ErrorLog(APPITF, "Failed to initialize GTL Layer\n");
		return Status;
	}

	/*
	 * Initlize device Manager (DMGR) Module
	 */
	if ((Status = DMGR_Init()) != SUCCESS)
	{
		UT_ErrorLog(APPITF, "Failed to initilize DMGR Module\n");
		return Status;

	}
	/*
	 * Initlize VAPI Core (VCORE) Module
	 */
	if ((Status = VCORE_Init()) != SUCCESS)
	{
		UT_ErrorLog(APPITF, "Failed to initilize VCORE Module\n");
		return Status;

	}

	UT_Log(APPITF, INFO, "\n*************************************************************\n");
	UT_Log(APPITF, INFO, "%s\n", VAPI_GetVersion());
	UT_Log(APPITF, INFO, "\n*************************************************************\n");

	bVapiInitialized_g = True;

	/*
	 * Initlize default tone parameters
	 */
	if ((Status = DMGR_ToneInit()) != SUCCESS)
	{
		UT_ErrorLog(APPITF, "Failed to initilize default tone params\n");
		return Status;
	}

	if ((Status = CidInitDefault()) != SUCCESS)
	{
		UT_ErrorLog(APPITF, "Failed to initilize Caller ID default\n");
		return Status;
	}
	
	UT_Log(APPITF, INFO, "VAPI Initilized sucessfully\n");
	return SUCCESS;
}

/***************************************************************************
 * VAPI_Close: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	Releases all resources used for the VAPI library such as memory, timer, etc.\n 
*	The VAPI will clear internal data structures and shuts down the GTL layer.\n 
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">NONE</td>
*		<td style="vertical-align: top;"></td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*
*	\n \b Commands:
*	\li No Comcerto commands sent.
*/
VSTATUS VAPI_Close(void)
{

	VSTATUS Status = SUCCESS;

	if (!bVapiInitialized_g)
	{
		/*Used printf instead of UT_Log because UT is not yet initialized */
		printf("VAPI_Close: VAPI_Close: VAPI library not initialized.\n");
		return VAPI_ERR_LIB_NOT_INITIALIZED;
	}

	UT_Log(APPITF, INFO, "Entered VAPI_Close\n");

	/* Close device Manager (DMGR) Module */
	UT_Log(APPITF, DEBUG, "Closing DMGR...\n");

	DMGR_CloseAllDevices();

	DMGR_Close();

	/* Close VAPI Core (VCORE) Module */
	UT_Log(APPITF, DEBUG, "Closing VCORE...\n");
	VCORE_Close();

	/* Close GTL Layer */
	UT_Log(APPITF, DEBUG, "Closing GTL...\n");
	if ((Status = GTL_Shutdown()) != SUCCESS)
	{
		UT_ErrorLog(APPITF, "Failed to shutdown GTL Layer\n");
	}

	/* Close UT Module */
	UT_Log(APPITF, DEBUG, "Closing UT...\n");
	UT_Close();

	UT_MutexDestroy(&stMutexToneGenArr_g);

	bVapiInitialized_g = False;

	return Status;
}

/***************************************************************************
 * VAPI_RegisterEventCallback: The function does the following things -
 ***************************************************************************/

/*!
*	\n \b Description: \n
*	This function is used to register a callback function for events coming from the MSP devices.\n 
*	Whenever any event is received from a device, the callback routine is invoked with the event code.\n 
*	Other information related to the event is stored in the buffer pointed \n 
*	to by pvData of the callback function.\n 
*	The library allocates memory for the buffer which is freed upon return of control \n 
*	from the user specified callback function.\n 
*	The user must free this thread immediately as this thread is created at the GTL layer\n 
*	and GTL will need it to report further received messages to VAPI.\n 
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">uiID</td>
*		<td style="vertical-align: top;">Device or Connection ID.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucLevel</td>
*		<td style="vertical-align: top;">Level at which the callback is to be registered.\n 
* 			(EVENT_LEVEL_GENERIC,EVENT_LEVEL_DEVICE,EVENT_LEVEL_CONN)</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pfnEventCallback</td>
*		<td style="vertical-align: top;">Callback to be registered. \n 
*			If it is NULL then it will erase the previously registered callback (if any).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*
*	\n \b Usage:
*	\include register_event_callback.c
*
*	\n \b Commands:
*	\li No Comcerto commands sent.
*/
VSTATUS VAPI_RegisterEventCallback(IN U32 uiID, IN U8 ucLevel, IN PFNEventCallback pfnEventCallback)
{
	VSTATUS Status;

	if (!bVapiInitialized_g)
	{
		/*Used printf instead of UT_Log because UT is not yet initialized */
		printf("VAPI_RegisterEventCallback: VAPI library not initialized.\n");
		return VAPI_ERR_LIB_NOT_INITIALIZED;
	}

	UT_Log(APPITF, INFO, "Entered VAPI_RegisterEventCallback %d,%d,%x\n", uiID, ucLevel, pfnEventCallback);
	Status = VCORE_RegisterEventCallback(uiID, ucLevel, pfnEventCallback);
	UT_Log(APPITF, INFO, "VAPI_RegisterEventCallback: Exiting status(%d)\n", Status);
	return Status;
}

/***************************************************************************
 * VAPI_PassThru: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	A pass-thru or proxy API function is provided to allow the user application
*	to send Comcerto commands as defined by the Command Reference Manual directly.
*	This gives user the flexibility to use newly defined MSP APIs that are not 
*	yet used from the VAPI APIs internally. \n
*	Also, pass-through API will be used to modify connection options
*	(DGAIN, ECHOCAN etc) or to do the device related configuration. \n
*	It is not meant to be used for either creating or destroying connections / conferences / participants.
*	Also, the use of this API should be minimized i.e. if there is a VAPI API available
*	for certain functionality then only that API should be used and pass-through must be avoided.
*	For pass-thru commands, the VAPI layer performs minimum processing on each command:
*	- Translate the mapping between Connection ID and MSP channel number for all the channel related
*	commands (CMD_LEVEL_CONN).
*	- Translate the mapping between Device ID and the device for all the device related commands
*	(Supervisor channel is used). (CMD_LEVEL_DEV).
*	- Translate the mapping between user defined Conference ID and MSP assigned conference ID for
*	conference related commands (CMD_LEVEL_CONF).
*	- Parse the function code and its parameters to make sure the specific bit-fields
*	are not changed by a subsequent VAPI function. \n
*	For example, if the user application sends a VCEOPT command using the pass-thru API,
*	this is treated like an override command for VCEOPT. \n
*	Therefore, subsequent VAPI functions that also issue VCEOPT should retain
*	the same parameters as intended by the original pass-thru function call. 
*	Some fields which VAPI has explicitly defined behavior, like packet generation field, 
*	is controlled by VAPI and should not be overridden by the pass through command.
* 
*	<B>User has to take care of the following points while using VAPI_PassThru:</B> \n 
*	- FIFO contents should be 4-byte aligned (include padding)
*	- User can send multiple-command messages using VAPI_PassThru. \n 
*	However, FIFO must have the correct padding (between commands for 4-byte alignment) \n 
*	as defined in the Command Reference Manual. \n 
*	In return VAPI provides complete FIFO containing all the response messages returned by MSP. \n 
*	The status returned is as per the latest failed command (if any) present in multi-command response.
*	- Currently there is a limitation regarding conference commands that only one conference related \n 
*	command is currently supported (CONF_SPECIFY_DOMINANT_TALKERS).\n 
*	This is because the position of participant ID inside the FIFO is dependent on function code. \n
*	Also if mapping for both the Participant ID and the Conference ID is required in a MSP command \n
*	then VAPI_PassThru cannot be used, because uiID	can take only one value that can be either a \n
*	Participant ID (CMD_LEVEL_CONN) or a Conference ID (CMD_LEVEL_CONF). \n 
*	In coming releases this issue will get resolved.
*	- Whether the host is Big-Endian or Little-Endian, it would be the user's responsibility to provide
*	the FIFO words in correct byte order as defined by the Command Reference Manual. 
*	- Currently the following commands are tracked in order to update VAPI stored channel parameters.\n
*	This ensures that these commands shall use the modified values when
*	sent from inside VAPI defined APIs. \n
*	These function codes are:
*	- at connection level 
*		- FC_VOIP_VOPDIS
*		- FC_VOIP_VOPENA
*		- FC_VOIP_VCEOPT
*		- FC_VOIP_DGAIN
*		- FC_VOIP_DTMFOPT
*		- FC_VOIP_ECHOCAN
*		- FC_VOIP_JBOPT
*		- FC_SET_ALT_HDR_CHANNEL
*		- FC_SET_IP_HDR_CHANNEL \n
*	- at connection level 
*		- FC_IP_SERVICE_CONFIG
*		- FC_ALT_HDR_DEVICE
*		- FC_IP_ADDRESS
*		- FC_MULTI_IP_ADDR_LIST
*		- FC_SET_ETH_HDR
*		- FC_SPU_FEATURES_CONTROL \n
*
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">uiID</td>
*		<td style="vertical-align: top;">It is device ID or Connection ID depending upon ucLevel.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucLevel</td>
*		<td style="vertical-align: top;">Indicates whether the command is a device level command
*			or connection level</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstMsg</td>
*		<td style="vertical-align: top;">Message to be sent to MSP. 
*		<b>It is required to only fill the pusFifo and the uiFifoSize fields. The sMailbox is no longer used.</b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pucBuffer</td>
*		<td style="vertical-align: top;">Buffer to hold the response message's fifo.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">puBufLen</td>
*		<td style="vertical-align: top;">This is an IN/OUT parameter, 
*			it should hold the length of input buffer, and when the request
*	 		completes it will hold the length of output buffer</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_NULL_POINTER_PASSED
*	\li VAPI_ERR_INVALID_PARAM
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_INVALID_CONFID
*	\li VAPI_ERR_UNSUPP_FEATURE
*
*	\n \b Usage:
*	\include passthru.c
*
*	\n \b Commands:
*	\li The Comcerto commands sent depends on the psMsg. 
*/
VSTATUS VAPI_PassThru(IN U32 uiID,
		      IN U8 ucLevel,
		      IN SMsg * pstMsg, IN SRequest * pstRequest, OUT U8 * pucBuffer, IN OUT U32 * puBufLen)
{
	SVapiReq *pstVapiReq;
	SConference *pstConf;
	SDevice *pstDevice;
	SChnl *pstChnl;
	VSTATUS Status;
	Boolean bIsSync;
	gtl_msg_t *pstReqMsg;
	int tempSize;
	void *tempFifo;

	if(pstMsg == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_PassThru: Invalid pstMsg\n");
		return VAPI_ERR_NULL_POINTER_PASSED;
	}

	switch (ucLevel)
	{
	case CMD_LEVEL_DEVICE:
		UT_Log(APPITF, INFO, "Entered VAPI_PassThru: dev(%u)\n", uiID);
		Status = VCORE_DeviceGenericCheck(uiID, "VAPI_PassThru", &pstDevice);

		if (Status != SUCCESS)
			goto err_out;

		pstChnl = &(pstDevice->stSupvChnl);

		break;

	case CMD_LEVEL_CONN:
		UT_Log(APPITF, INFO, "Entered VAPI_PassThru: conn(%u)\n", uiID);
		Status = VCORE_ConnectionGenericCheck(uiID, "VAPI_PassThru", &pstChnl);

		if (Status != SUCCESS)
			goto err_out;

		pstDevice = pstChnl->pstDev;
		
		break;

	case CMD_LEVEL_CONF:
		UT_Log(APPITF, INFO, "Entered VAPI_PassThru: conf(%u)\n", uiID);
		Status = VCORE_ConferenceGenericCheck(uiID, "VAPI_PassThru", &pstConf);

		if (Status != SUCCESS)
			goto err_out;

		pstChnl = &(pstConf->pstDev->stSupvChnl);
		pstDevice = pstConf->pstDev;

	default:
		UT_ErrorLog(APPITF, "Entered VAPI_PassThru: Invalid Cmd Level %u\n", ucLevel);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
		break;

	}

	if ((pstMsg->uiFifoSize != 0) && (pstMsg->pusFifo == NULL))
	{
		UT_ErrorLog(APPITF, "VAPI_PassThru: Invalid fifo size = %d fifo = NULL\n", pstMsg->uiFifoSize, pstMsg->pusFifo);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	/*Save the user message before checking if the message is valid (fifo and fifo len are manipulated)*/
	tempSize = pstMsg->uiFifoSize;
	tempFifo = (void *)pstMsg->pusFifo;

	Status = VCORE_CheckMessage(pstDevice, pstMsg, ucLevel);
	if (Status != SUCCESS)
	{
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	pstMsg->uiFifoSize = tempSize;
	pstMsg->pusFifo = tempFifo;

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(gtl_msg_t));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}

	pstReqMsg = pstVapiReq->pvUserData;

	/* allocate buffer to handle the command parameters (+ 4 bytes for multi cmd padding)*/
	pstReqMsg->fifo = UT_Calloc(1, pstMsg->uiFifoSize + 4);
	if (pstReqMsg->fifo == NULL)
	{
		UT_ErrorLog(APPITF, "No memory available\n");
		Status = VAPI_ERR_NOMEM;
		goto err_msg;
	}
	/* init msg fields to default values*/
	VCORE_InitMsg(pstReqMsg, pstReqMsg->fifo, pstMsg->uiFifoSize + 4);

	/* fill the fifo with the user data */
	UT_MemCopy(pstReqMsg->fifo, pstMsg->pusFifo, pstMsg->uiFifoSize);
	pstReqMsg->fifo_size = pstMsg->uiFifoSize;

	/* configure the request */
	VCORE_SetRequest(pstVapiReq, pstRequest, VFSM_PassThru, /* pfnReqFSMHdlr */
						uiID);
	bIsSync = pstVapiReq->bIsSync;
	pstVapiReq->ucCmdLevel = ucLevel;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		Status = pstVapiReq->Status;

		if (pstVapiReq->uiCallbkDataLen > 0
		    && puBufLen != NULL && pstVapiReq->uiCallbkDataLen <= *puBufLen)
		{
			if (pucBuffer != NULL)
			{
				UT_MemCopy(pucBuffer, pstVapiReq->pvCallbackData, pstVapiReq->uiCallbkDataLen);
			}
			*puBufLen = pstVapiReq->uiCallbkDataLen;
		}
		/*in ASYNC mode it is freed in VFSM_PassThru*/
		UT_FreeMem(pstReqMsg->fifo);

		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	switch (ucLevel)
	{
	case CMD_LEVEL_DEVICE:
		UT_Log(APPITF, INFO, "VAPI_PassThru: Exiting status(%d), dev(%u)\n", Status, uiID);
		break;
	case CMD_LEVEL_CONN:
		UT_Log(APPITF, INFO, "VAPI_PassThru: Exiting status(%d), conn(%u)\n", Status, uiID);
		break;
	case CMD_LEVEL_CONF:
		UT_Log(APPITF, INFO, "VAPI_PassThru: Exiting status(%d), conf(%u)\n", Status, uiID);
		break;
	}
/* no error exit */
	return Status;

     err_msg:
	VCORE_FreeRequest(pstVapiReq);
      err_out:
	return Status;
}

/***************************************************************************
 * VAPI_C2CConnectUP: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	Establish or terminate a VOIP TrFO inter or intra Comcerto connection between 2 existing connections. \n
*	This API requires previous extra configurations such as Set_UMA_Params (Function Code 0x8600) or
*	UMA_EventDet (Function Code 0x8601).\n
*	Please check the chapter WCDMA Wireless Support or UP Transcoder Free Operation (TrFO) for more
*	detail.\n
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">Id of the device to send the command to</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">C2CConnectOptions</td>
*		<td style="vertical-align: top;">C2C connection options</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_CONNID
*
*	\n \b Usage:
*	\include c2c_connect_UP.c
*
*	\n \b Commands:
*	- FC_C2C_CONNECT_UP
*/

VSTATUS VAPI_C2CConnectUP(IN DEVID DevId, IN SC2CConnectOption *pstC2CConnectOptions, IN SRequest *pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SC2CConnectOption *pstUserData;
	SDevice *pstDevice;
	VSTATUS Status = SUCCESS;
	Boolean bIsSync;

	Status = VCORE_DeviceGenericCheck(DevId, "VAPI_C2CConnectUP", &pstDevice);

	if (Status != SUCCESS)
		goto out;

	if (pstC2CConnectOptions == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_C2CConnectUP: C2CConnectOptions NULL\n");
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		goto out;
	}
 
	UT_Log(APPITF,INFO,"VAPI_C2CConnectUP: dev(%d)\n", DevId);


	pstVapiReq = VCORE_AllocateRequest(sizeof(SC2CConnectOption));

	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstUserData = pstVapiReq->pvUserData;

	UT_MemCopy(pstUserData, pstC2CConnectOptions, sizeof(SC2CConnectOption));
	/* Fill the request with the regular parameters */
	VCORE_SetDeviceRequest(pstVapiReq, pstRequest, VFSM_C2CConnectUP, DevId);

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(&(pstDevice->stSupvChnl), pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF,INFO,"VAPI_C2CConnectUP: Exiting status(%d), dev(%d)\n", Status, DevId);

      out:
	return Status;
}


/***************************************************************************
 * VAPI_SetDebugLevel: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This API is used to change the VAPI debug levels dynamically. \n
*	Various debug levels are defined to customize the type
*	of debug information being generated from VAPI modules \n
*	To print debug info to console VAPI should be compiled with \n
*	-UVAPI_NODBG key \n
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucVapiDebugLevel</td>
*		<td style="vertical-align: top;">Debug level pertaining to internal VAPI debug information:
*				- NO_DEBUG
*				- ALL_ERRORS
*				- API_LEVEL
*				- API_PACKET_LEVEL
*				- VAPI_INFO
*				- VAPI_GTL_INFO
*				- VAPI_UT
*				- ALL_INFO.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucAppDebugLevel</td>
*		<td style="vertical-align: top;">Debug level pertaining to the user application
*				- INFO
*				- DEBUG
*				- PACKET
*				- WARN
*				- ERRORS</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*
*	\n \b Commands:
*	\li No Comcerto commands sent.
*/
VSTATUS VAPI_SetDebugLevel(U8 ucVapiDebugLevel, U8 ucAppDebugLevel)
{
	if (!bVapiInitialized_g)
	{
		/*Used printf instead of UT_Log because UT is not yet initialized */
		printf("VAPI_SetDebugLevel: VAPI library not initialized.\n");
		return VAPI_ERR_LIB_NOT_INITIALIZED;
	}
	UT_Log(APPITF, INFO, "Entered VAPI_SetDebugLevel\n");
	UT_SetDebugLevel(ucVapiDebugLevel, ucAppDebugLevel);
	UT_Log(APPITF, INFO, "VAPI_SetDebugLevel: Exiting status(%d)\n",SUCCESS);
	return SUCCESS;
}


/** @} */ /*end of misc group*/
