/*! \file fsm_misc.c 
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include"ut.h"
#include"vapi.h"
#include"dmgr.h"
#include"vcore.h"
#include"msp.h"
#include "vcore_voip.h"
#include "appitf.h"

/****************************************************************************
 * VFSM_CreateChannel : The function does the following things -
 ***************************************************************************/
/*! 
 *  \b Implementation: \n
 *      -#  Send create channel to the device
 *
 *  \b Inputs-Outputs: \n
 *	\li pstVapiRep->pvUserData is a SFsmCreateChannelInfo structure containing: 
 *		\li DevChannelType = the new channel type
 *		\li DevChannelTS = the timeslot for the channel
 *		\li *DevChannelId = the address where to store the ID of the new channel.
 *	\li pstVapiReq The processed request.
 *	\li pstMsg  Message obtained from the device.
 *
 *  \b Returns: \n
 *	\li SUCCESS
 *	\li VAPI_ERR_INVALID_PARAM
 *
 */
VSTATUS VFSM_CreateChannel(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status=SUCCESS;
	SChnl *ChannelInfo;
	U16 *pausFifo = NULL;
	SChnl *ppstUserData[1]; 
	DEVID DevId;

	DevId = pstSupvChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_CreateChannel: Entering dev(%u) state(%d)\n", DevId, pstVapiReq->usReqState);

	/* retrieve the connection pointer from the request */
	UT_MemCopy(ppstUserData, pstVapiReq->pvUserData, sizeof(SChnl *));

	ChannelInfo = ppstUserData[0];

	switch (pstVapiReq->usReqState)
	{
	case CREATE_CHANNEL_INIT:

		pstVapiReq->usReqState = CREATE_CHANNEL_FINISHED;

		Status = VDEV_CreateChannel(pstSupvChnl, ChannelInfo);

		if (Status != SUCCESS)
			goto finish;

		break;
 
	case CREATE_CHANNEL_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstSupvChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_SUPVSR_CREATE_CHANNEL);

		if (Status == -ERR_TDMDRV_INVTS)
		{
			Status = VAPI_ERR_TS_IN_USE;
			goto finish;
		}

		if (Status != SUCCESS)
		{
			Status = VAPI_ERR_CREATECONN_FAIL;
			goto finish;
		}

		/* Save the channel ID returned by the device for the current connection VAPI structure*/ 
		pausFifo = (U16 *) pstMsg->fifo;
		ChannelInfo->usMSPChnlId = UT_LE2CPU16(pausFifo[5]);
		Status = DMGR_AddChannel(ChannelInfo->pstDev, ChannelInfo);
		/*FIXME if we can't add the channel ID to the array the error is not correctly handled*/

		goto finish;

		break;

	}

	return SUCCESS;

finish:
	UT_Log(VCORE, DEBUG, "VFSM_CreateChannel: Completing state(%u) status(%d) dev(%u)\n",
		pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_CreateChannel: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

/****************************************************************************
 * VFSM_DestroyChannel : The function does the following things -
 ***************************************************************************/
/*! 
 *  \b Implementation: \n
 *      -#  Send destroy channel to the device
 *
 *  \b Inputs-Outputs: \n
 *	\li pstVapiRep->pvUserData contains the device Channel ID to destroy
 *	\li pstChnl The channel on which the request is going.
 *	\li pstVapiReq The processed request.
 *	\li pstMsg  Message obtained the device.
 *
 *  \b Returns: \n
 *	\li SUCCESS
 *	\li VAPI_ERR_INVALID_PARAM
 *
 */
VSTATUS VFSM_DestroyChannel(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	U16 *ChannelId;
	DEVID DevId;
	DevId = pstSupvChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_DestroyChannel: Entering dev(%u) state(%d)\n", DevId, pstVapiReq->usReqState);

	ChannelId = (U16 *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case DESTROY_CHANNEL_INIT:

		pstVapiReq->usReqState = DESTROY_CHANNEL_FINISHED;

		Status = VDEV_DestroyChannel(pstSupvChnl, *(ChannelId));
		if (Status != SUCCESS)
			goto finish;

		break;
 
	case DESTROY_CHANNEL_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, pstSupvChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_SUPVSR_DESTROY_CHANNEL);

		/* If we are trying to destroy a non existing device channel
		while the connection exists. So let continue with the regular process to remove the 
		VAPI conenction structure*/ 
		if (Status == -ERR_CHANCTRL_NO_SUCH_CHAN)
		{
			UT_ErrorLog(VCORE, "VFSM_DestroyChannel: Device channel doesn't (%u) exist continue\n",*(ChannelId));
			Status = SUCCESS;
		}

		/* remove the channel ID from the device channel array */
		if (Status == SUCCESS)
			DMGR_RemoveChannel(pstSupvChnl->pstDev, *(ChannelId));

		goto finish;
		break;

	}

	return SUCCESS;

finish:
	UT_Log(VCORE, DEBUG,"VFSM_DestroyChannel: Completing req state(%u) status(%d) dev(%u)\n",
	       pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_DestroyChannel: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

/****************************************************************************
 * VFSM_ChangeChannelType : The function does the following things -
 ***************************************************************************/
 /*! 
 *  \b Implementation: \n
 *      -#  Setup and call VFSM_StopConnection child request
 *      -#  Setup and call VFSM_DestroyChannel child request
 *      -#  Setup and call VFSM_CreateChannel child request
 *
 *  \b Inputs-Outputs: \n
 *	\li pstVapiRep->pvUserData is a SFsmCreateChannelInfo structure containing: 
 *		\li DevChannelType = the new channel type
 *		\li DevChannelTS = the timeslot for the channel
 *		\li *DevChannelId = the address where to store the ID of the new channel.
 *	\li pstVapiReq The processed request.
 *	\li pstMsg  Message obtained from the device.
 *
 *  \b Returns: \n
 *	\li SUCCESS
 *	\li VAPI_ERR_INVALID_PARAM
 *
 */
VSTATUS VFSM_ChangeChannelType(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status=SUCCESS;
	U16 *ChannelId = NULL;
	SVapiReq *pstChildVapiReq = NULL;
	IN SChnl * pstSupvChnl;
	CONNID ConnId;
	ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_ChangeChannelType: Entering conn(%u) state(%d)\n", ConnId, pstVapiReq->usReqState);

	/*Take action according to current state of request */
	switch (pstVapiReq->usReqState)
	{

	case CHANGE_TYPE_INIT:

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
			goto err;

		/* initialise the Child request 
		No particular data are required for the child fsm_handler */ 
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */ 
				VFSM_StopConnection,	/* Child Request Handler */
				STOP_CONN_INIT);	/* Child Request handler state */

		pstVapiReq->usReqState = CHANGE_TYPE_STOP_CONN_WAIT_CHILD;
		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case CHANGE_TYPE_STOP_CONN_WAIT_CHILD:

		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		pstSupvChnl = DMGR_GetChannel(pstChnl->pstDev, SUPV_CHANNEL);

		/* This allocate a new request */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(U16));

		if (!pstChildVapiReq)
			goto err;

		/* Initialise the UserData (allocated by VCORE_AllocateRequest)
		The channel id is used by the DestroyChannel command */
		ChannelId = pstChildVapiReq->pvUserData;

		*(ChannelId) = pstChnl->usMSPChnlId;
		/* initialise the Child request 
		The UserData is the channelID used by the VFSM_DestroyChannel handler */
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */ 
				VFSM_DestroyChannel,	/* Child Request Handler */
				DESTROY_CHANNEL_INIT);	/* Child Request handler state */

		pstVapiReq->usReqState = CHANGE_TYPE_DESTROY_WAIT_CHILD;
		/* process request in the supervisor channel */
		VCORE_ProcessRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case CHANGE_TYPE_DESTROY_WAIT_CHILD:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		pstSupvChnl = DMGR_GetChannel(pstChnl->pstDev, SUPV_CHANNEL);

		/* This allocate a new request */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SFsmCreateChannelInfo));

		if (!pstChildVapiReq)
			goto err;

		/* User Data for child request is comming from the parent request */
		UT_MemCopy(pstChildVapiReq->pvUserData, pstVapiReq->pvUserData, sizeof(SFsmCreateChannelInfo));

		/* initialise the Child request 
		The UserData is the channelID used by the VFSM_CreateChannel handler */
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */
				VFSM_CreateChannel,	/* Child Request Handler */
				CREATE_CHANNEL_INIT);	/* Child Request handler state */

		pstVapiReq->usReqState = CHANGE_TYPE_CREATE_FINISHED;
		/* process request in the supervisor channel */
		VCORE_ProcessRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case CHANGE_TYPE_CREATE_FINISHED:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		goto finish;

	}

	return SUCCESS;

err:
	Status = VAPI_ERR_NOMEM;

finish:
	UT_Log(VCORE, DEBUG, "VFSM_ChangeChannelType: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_ChangeChannelType: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;

}

/****************************************************************************
 * VFSM_CreateParticipant : The function does the following things -
 ***************************************************************************/
/*! 
 *  \b Implementation: \n
 *      -#  Send create channel to the device
 *
 *  \b Inputs-Outputs: \n
 *	\li pstVapiRep->pvUserData is a SFsmCreateChannelInfo structure containing: 
 *		\li DevChannelType = the new channel type
 *		\li DevChannelTS = the timeslot for the channel
 *		\li *DevChannelId = the address where to store the ID of the new channel.
 *	\li pstVapiReq The processed request.
 *	\li pstMsg  Message obtained from the device.
 *
 *  \b Returns: \n
 *	\li SUCCESS
 *	\li VAPI_ERR_INVALID_PARAM
 *
 */
VSTATUS VFSM_CreateDeviceParticipant(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status=SUCCESS;
	SChnl *ChannelInfo;
	U16 *pausFifo = NULL;
	SChnl *ppstUserData[1]; 
	DEVID DevId;
	DevId = pstSupvChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_CreateDeviceParticipant: Entering conn(%u) state(%d)\n", DevId, pstVapiReq->usReqState);

	/* retrieve the connection pointer from the request */
	UT_MemCopy(ppstUserData, pstVapiReq->pvUserData, sizeof(SChnl *));

	ChannelInfo = ppstUserData[0];

	switch (pstVapiReq->usReqState)
	{
	case CREATE_PARTICIPANT_INIT:

		pstVapiReq->usReqState = CREATE_PARTICIPANT_FINISHED;
		Status = VDEV_CreatePart(pstSupvChnl, ChannelInfo);

		if (Status != SUCCESS)
			goto finish;

		break;
 
	case CREATE_PARTICIPANT_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstSupvChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_CONF_CREATE_PARTICIPANT);

		if (Status != SUCCESS)
			goto finish;

		/* Save the channel ID returned by the device for the current connection VAPI structure*/ 
		pausFifo = (U16 *) pstMsg->fifo;
		ChannelInfo->usMSPChnlId = UT_LE2CPU16(pausFifo[5]);
		/* add the participant to the array for this device */
		Status = DMGR_AddChannel(pstSupvChnl->pstDev, ChannelInfo);
		if (Status != SUCCESS)
			goto finish;

		goto finish;
		break;

	}

	return SUCCESS;

finish:
	UT_Log(VCORE, DEBUG, "VFSM_CreateDeviceParticipant: Completing req state(%u) status(%d) dev(%u)\n",
	       pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_CreateDeviceParticipant: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

/****************************************************************************
 * VFSM_PassThru : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Get the request from request list
 *      -#  Form a GTL Message from the pvUserData
 *      -#  Send the message
 *      -#  Use the response to fillup the pvCallbackData
 *  
 *  - Assumptions
 *      -# APPITF does the overriding default params job.
 *
 *  \return 
 *  None
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_PassThru(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	gtl_msg_t *pstReqMsg;
	VSTATUS Status;
	U8 ucCmdCls;
	U8 ucCmdType;
	SApiHdr *pstFifo;
	DEVID DevId = pstChnl->pstDev->DevId;
	CONNID ConnId = pstChnl->ConnId;
	U16 Level;


	if (pstChnl->usMSPChnlId == SUPV_CHANNEL)
	{
		Level = CMD_LEVEL_DEVICE;
		UT_Log(VCORE, DEBUG, "VFSM_PassThru: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);
	}
	else
	{
		Level = CMD_LEVEL_CONN;
		UT_Log(VCORE, DEBUG, "VFSM_PassThru: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);
	}

	pstReqMsg = (gtl_msg_t *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case PASSTHRU_INIT:
		/*Send the GTL message */
		pstFifo = (SApiHdr *)pstReqMsg->fifo;
		ucCmdCls = pstFifo->cmd_class;
		ucCmdType = pstFifo->cmd_type;

		pstReqMsg->channel = pstChnl->usMSPChnlId;

		if ((ucCmdCls == CMD_CLASS_CONF_CHANNEL || ucCmdCls == CMD_CLASS_CONF_DEVICE)
		    && (ucCmdType == CMD_TYPE_INDICATION_RESP))
		{
			((U8 *) pstReqMsg->fifo)[7] = 0;
			((U8 *) pstReqMsg->fifo)[6] = 0;

			Status = VCORE_SendIndToMsp(pstChnl, pstReqMsg);
			goto finish;
		}
		else
		{
			pstVapiReq->usReqState = PASSTHRU_FINISHED;
			Status = VCORE_SendMSPReq(pstChnl, pstReqMsg);
			if( Status != SUCCESS)
				goto finish;
		}

		break;

	case PASSTHRU_FINISHED:

		Status = VCORE_CheckPassThruStatus(pstChnl, pstReqMsg, Event, pstMsg);

		/* in case of unexpected error pstMsg = NULL, 
		nothing reported to the application */ 
		if ((Status == VAPI_ERR_MSP_NO_RESPONSE) || 
			(Status == VAPI_IND_STOP_REC_PLAY) || 
			(Status == VAPI_ERR_UNDEFINED_EVENT))
			goto finish;

		pstVapiReq->pvCallbackData = UT_AllocMem(pstMsg->fifo_size);

		if (!pstVapiReq->pvCallbackData)
		{
			Status = VAPI_ERR_NOMEM;
				goto finish;
		}
	
		UT_MemCopy(pstVapiReq->pvCallbackData, pstMsg->fifo, pstMsg->fifo_size);
		pstVapiReq->uiCallbkDataLen = pstMsg->fifo_size;
	
		goto finish;
		break;

	}

	return SUCCESS;

finish:

	/*Do request completion processing */
	if (Level == CMD_LEVEL_DEVICE)
		UT_Log(VCORE, DEBUG, "VFSM_PassThru: Completing state(%u) status(%d) dev(%u)\n",
				pstVapiReq->usReqState, Status, DevId);
	else
		UT_Log(VCORE, DEBUG, "VFSM_PassThru: Completing state(%u) status(%d) conn(%u)\n",
				pstVapiReq->usReqState, Status, ConnId);

	if(Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_PassThru: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
			DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	/*If non blocking call, then clean up the user-data 
	in SYNC mode it is freed in VAPI_PassThru*/
	if (!pstVapiReq->bIsSync)
	{
		UT_FreeMem(pstReqMsg->fifo);
	}

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	/*Do request completion processing */
	if (Level == CMD_LEVEL_DEVICE)
		UT_Log(VCORE, DEBUG, "VFSM_PassThru: Exiting status(%d) dev(%u)\n", Status, DevId);
	else
		UT_Log(VCORE, DEBUG, "VFSM_PassThru: Exiting status(%d) conn(%u)\n", Status, ConnId);


	return Status;
}


/****************************************************************************
 * VFSM_PurgeConference : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Check if there are allocated particpant associated to the onference.
 *      -#  If so destroy the participant recursively
 *
 *  \return 
 *  VAPI_ERR_INVALID_PARAM Invalid parameter supplied
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_PurgeConference(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	U8 *paucFifo = NULL;
	U16 *pausFifo = NULL;
	VSTATUS Status;
	SVapiReq *pstChildVapiReq;
	U16 usPartCnt;
	CONNID	*pulConnId;
	SChnl * pstPartChnl = NULL;
	SConference *pstConf;
	DEVID DevId;
	DevId = pstSupvChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_PurgeConference: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	if (pstMsg != NULL)
	{
		pausFifo = (U16 *) pstMsg->fifo;
		paucFifo = (U8 *) pstMsg->fifo;
	}

	pstConf = DMGR_GetConfById(*(CONFID *) (pstVapiReq->pvUserData));
	if (pstConf == NULL)
	{
		UT_ErrorLog(VCORE, "VFSM_PurgeConference: Conference %u doesn't exists\n", 
			*(CONFID *) (pstVapiReq->pvUserData));
		Status = VAPI_ERR_INVALID_CONNID;
		goto finish;
	}

	switch (pstVapiReq->usReqState)
	{

	/* in this FSM the PURGE_CONFERENCE_INIT is called recursively */
	case PURGE_CONFERENCE_INIT:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		/* Now destroy any participants of the conference */
		for (usPartCnt = 0; usPartCnt < GTL_GetDefaultMaxPartPerConf(pstSupvChnl->pstDev->ucDevType); usPartCnt++)
		{
			if (pstConf->apstParticipant[usPartCnt] != NULL)
			{
				pstPartChnl = pstConf->apstParticipant[usPartCnt];
				break;
			}
		}

		/* if no participant found exit */
		if (pstPartChnl == NULL)
			goto finish;

		/* This allocate a new request (and UserData memory in this new request*/ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(CONNID));
		
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}
		
		/* Initialise the UserData (allocated by VCORE_AllocateRequest)
		It contains the channel and conference Id */
		pulConnId = pstChildVapiReq->pvUserData;
		
		*(pulConnId) = pstPartChnl->ConnId;

		/* remove the connection from the AVL tree 
		this prevent adding new requests to this connection */
		VCORE_RemoveConnection(pstPartChnl);

		/* FIXME: We call VFSM_KillConnection handler istead of VFSM_DestroyConnection
		due to child request limitation (VFSM_DstroyConnection post request to the queue 
		of the supervisor which leads to a locking situation here */ 
		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
					pstVapiReq,		/* Parent Request */
					VFSM_KillConnection,	/* Child Request Handler */
					DESTROY_CONN_INIT);	/* Child Request handler state */
		
		/* keep in the same state */
		pstVapiReq->usReqState = PURGE_CONFERENCE_INIT;

		/*add it at the end of the request list of the supervisor channel
		and call the ProcessRequest function to process this child request first.*/ 
		VCORE_ProcessRequest(pstPartChnl, pstChildVapiReq);

		break;

	case PURGE_CONFERENCE_FINISHED:
		/* retrieve the child status */
		Status = pstVapiReq->Status;
		goto finish;
		break;
	}

	return SUCCESS;

finish:

	UT_Log(VCORE, DEBUG, "VFSM_PurgeConference: Completing req state(%u) status(%d) dev(%u)\n",
		       pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_PurgeConference: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

/****************************************************************************
 * VFSM_KillConnection : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Stop first the connection (VOPENA 0)
 *      -#  The destroy the channel or the participant accordingly
 *      -#  Cleanup the channel ot the Participant from device manager
 *
 *  - Assumptions
 *      -#  This FSM handler can be only called as a child request from a another FSM handler
 *	processing a request on a SUPERVISOR channels
 *
 *  \return 
 *  VAPI_ERR_INVALID_PARAM Invalid parameter supplied
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_KillConnection(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	SChnl *pstSupvChnl;
	SVapiReq *pstChildVapiReq;
	U16 * pusChannelId;
	SFsmParticipantInfo *ParticipantInfo;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_KillConnection: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstSupvChnl = DMGR_GetChannel(pstChnl->pstDev, SUPV_CHANNEL);

	switch (pstVapiReq->usReqState)
	{
	case DESTROY_CONN_INIT:

		/*Wait the child to finish on DESTROY_STOP_CONN_WAIT_CHILD state */
		if (pstChnl->ePartType == eNOTPART)
			pstVapiReq->usReqState = DESTROY_CONN_DESTROY_CHANNEL;
		else
			pstVapiReq->usReqState = DESTROY_CONN_DESTROY_PART;

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* initialise the Child request 
		No particular data are required for the child fsm_handler */ 
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */ 
				VFSM_StopConnection,	/* Child Request Handler */
				STOP_CONN_INIT);	/* Child Request handler state */
		
		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case DESTROY_CONN_DESTROY_CHANNEL:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = DESTROY_CONN_FINISHED;
		/* This allocate a new request */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(U16));

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* Initialise the UserData (allocated by VCORE_AllocateRequest)
		The channel id is used by the DestroyChannel command */
		pusChannelId = pstChildVapiReq->pvUserData;

		*(pusChannelId) = pstChnl->usMSPChnlId;
		/* initialise the Child request 
		The UserData is the channelID used by the VFSM_DestroyChannel handler */
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */
				VFSM_DestroyChannel,	/* Child Request Handler */
				DESTROY_CHANNEL_INIT);	/* Child Request handler state */

		/* process request in the supervisor channel */
		VCORE_StartChannelChildRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case DESTROY_CONN_DESTROY_PART:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = DESTROY_CONN_FINISHED;
		/* This allocate a new request (and UserData memory in this new request*/ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SFsmParticipantInfo));

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* Initialise the UserData (allocated by VCORE_AllocateRequest)
		It contains the channel and conference Id */
		ParticipantInfo = pstChildVapiReq->pvUserData;

		ParticipantInfo->DevPartId = pstChnl->usMSPChnlId;
		ParticipantInfo->DevConfId = pstChnl->pstConf->usMSPConfId;

		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
				pstVapiReq,			/* Parent Request */
				VFSM_DestroyParticipant,	/* Child Request Handler */
				DESTROY_PARTICIPANT_INIT);	/* Child Request handler state */

		/* process request in the supervisor channel */
		VCORE_StartChannelChildRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case DESTROY_CONN_FINISHED:
		/* retrieve the child status */
		Status = pstVapiReq->Status;
		goto finish;
		break;
	}

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_KillConnection: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_KillConnection: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);
	if (Status == SUCCESS)
	{
		/* We finished handling the user request, now clear it's request list ... */
		VCORE_CloseConnection(pstChnl);
		/* free the channel */
		UT_FreeMem(pstChnl);
	}
	else
		/* put back connection to the AVL tree to be able to re-issue commands*/
		VCORE_AddConnection(pstChnl);

	UT_Log(VCORE, INFO, "VFSM_KillConnection: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;

}

/****************************************************************************
 * VFSM_SetTransConnect : The function does the following things -
 ***************************************************************************/
/*! 
 *  \b Implementation: \n
 *      -#  Send trans_connect to the device
 *
 *  \b Inputs-Outputs: \n
 *	\li pstVapiRep->pvUserData is a STranscodingParams structure containing: 
 *		\li TranscodingEnable = Enable/disable session
 *		\li pstChnl1->usMSPChnlId = the channel of conenction 1
 *		\li pstChnl1->usMSPChnlId = the channel of conenction 2
 *	\li pstVapiReq The processed request.
 *	\li pstMsg  Message obtained from the device.
 *
 *  \b Returns: \n
 *	\li SUCCESS
 *	\li VAPI_ERR_INVALID_PARAM
 *
 */
VSTATUS VFSM_SetTransConnect(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	STransConnectParams * pstTransConnectParams; 
	DEVID DevId;
	DevId = pstSupvChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_SetTransConnect: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	switch (pstVapiReq->usReqState)
	{
	case SET_TRANSCONNECT_INIT:
		/* retrieve the transcoding parameters from the request */
		pstTransConnectParams = (STransConnectParams *) pstVapiReq->pvUserData;

		pstVapiReq->usReqState = SET_TRANSCONNECT_FINISHED;
		Status = VDEV_TransConnect(pstSupvChnl, pstTransConnectParams);

		if (Status != SUCCESS)
			goto finish;

		break;
 
	case SET_TRANSCONNECT_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstSupvChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_TRANS_CONNECT);

		goto finish;
		break;

	}

	return SUCCESS;

finish:
	UT_Log(VCORE, DEBUG, "VFSM_SetTransConnect: Completing req state(%u) status(%d) dev(%u)\n",
		pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetTransConnect: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

/****************************************************************************
 * VFSM_InitFaxChannel : The function does the following things -
 ***************************************************************************/
/*! 
 *  \b Implementation: \n
 *      -#  Send several command to the device depending on the OpMode parameter
 *
 *  \b Inputs-Outputs: \n
 *	\li pstVapiRep->pvUserData is a SFsmModifyChannelInfo structure containing: 
 *		\li DevChannelType = the channel type
 *		\li OpMode = the operation mode
 *		\li ProfileId = the profile ID.
 *	\li pstVapiReq The processed request.
 *	\li pstMsg  Message obtained from the device.
 *
 *  \b Returns: \n
 *	\li SUCCESS
 *	\li VAPI_ERR_INVALID_PARAM
 *
 */
VSTATUS VFSM_InitFaxChannel(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status=SUCCESS;
	SInitChannel *pstInitChannel;
	SVoIPChnlParams *pstChnlParams;
	struct _FAXOPT *pstFaxOpts;
	struct _FAXLVL *pstFaxLevel;
	CONNID ConnId;
	ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_InitFaxChannel: Entering conn(%u) state(%d)\n", ConnId, pstVapiReq->usReqState);

	pstChnlParams = pstChnl->pstChnlParams;
	pstInitChannel = (SInitChannel *) pstVapiReq->pvUserData;

	/* FIXME All these command required to be sent in multi mode */
	switch (pstVapiReq->usReqState)
	{
	case INIT_FAX_CHANNEL_INIT:
		pstVapiReq->usReqState = INIT_FAX_CHANNEL_FAXLVL_SEND;

		if (pstInitChannel->stFsmFaxOptions.bUseIt == True)
			pstFaxOpts = &pstInitChannel->stFsmFaxOptions.stFaxOpts;
		else
			pstFaxOpts = &pstChnlParams->stFaxOpts;

		/*Send FAXOPTS */
		Status = VDEV_SendFaxOpt(pstChnl, pstFaxOpts);
		if (Status != SUCCESS)
			goto finish;

		break;
 
	case INIT_FAX_CHANNEL_FAXLVL_SEND:
		/* read the answer to FAXOPTS command */
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_FAXOPT);

		if (Status != SUCCESS)
			goto finish;

		/* udate _FAXOPT parameters if needed*/
		if (pstInitChannel->stFsmFaxOptions.bUseIt == True)
			UT_MemCopy(&pstChnlParams->stFaxOpts, 
						&pstInitChannel->stFsmFaxOptions.stFaxOpts, sizeof (struct _FAXOPT));

		pstVapiReq->usReqState = INIT_FAX_CHANNEL_FINISHED;

		if (pstInitChannel->stFsmFaxOptions.bUseIt == True)
			pstFaxLevel = &pstInitChannel->stFsmFaxOptions.stFaxLevel;
		else
			pstFaxLevel = &pstChnlParams->stFaxTxLevel;

		/*the FAXOPTS is configured now send FAXLVL */
		Status = VDEV_SendFaxLevel(pstChnl, pstFaxLevel);
		if (Status != SUCCESS)
			goto finish;

		break;

	/* This state is not used yet */
	/* It will be used when T38oRTP will be supported */
	case INIT_FAX_CHANNEL_PT_SET_SEND:
		/* read the answer to FAXLVL command */
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_FAXLVL);

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = INIT_FAX_CHANNEL_FINISHED;
		Status = VDEV_PLTypeSet(pstChnl, eIFP, pstChnlParams->stVopena.param_5.bits.rtp_pt, eBoth);
		if (Status != SUCCESS)
			goto finish;

		break;

	case INIT_FAX_CHANNEL_FINISHED:
		/* read the answer to FAXLVL command */
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_FAXLVL);

		if (Status != SUCCESS)
			goto finish;

		/* updated  struct _FAXLVL options if needed*/
		UT_MemCopy(&pstChnlParams->stFaxTxLevel,
						&pstInitChannel->stFsmFaxOptions.stFaxLevel, sizeof (struct _FAXLVL));

		goto finish;
		break;
	}

	return SUCCESS;


finish:
	UT_Log(VCORE, DEBUG, "VFSM_InitFaxChannel: Completing req state(%u) status(%d) conn(%u)\n",
	       pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_InitFaxChannel: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}


/****************************************************************************
 * VFSM_InitChannel : The function does the following things -
 ***************************************************************************/
/*! 
 *  \b Implementation: \n
 *      -#  Call the rigth InitChannel handler depending on Channel type
 *
 *  \b Inputs-Outputs: \n
 *	\li pstVapiRep->pvUserData is a SFsmModifyChannelInfo structure containing: 
 *		\li DevChannelType = the channel type
 *		\li OpMode = the operation mode
 *		\li ProfileId = the profile ID.
 *	\li pstVapiReq The processed request.
 *	\li pstMsg  Message obtained from the device.
 *
 *  \b Returns: \n
 *	\li SUCCESS
 *	\li VAPI_ERR_INVALID_PARAM
 *
 */
VSTATUS VFSM_InitChannel(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	SVapiReq *pstChildVapiReq;
	SInitChannel *pstInitChannel;
	CONNID ConnId;
	ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_InitChannel: Entering conn(%u) state(%d)\n", ConnId, pstVapiReq->usReqState);

	pstInitChannel = (SInitChannel *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{	
	case INIT_CHANNEL_INIT:
		switch (pstInitChannel->stFsmModifyChannelInfo.DevChannelType)
		{
		case eFOIP:
			/* this allocate a new request */ 
			pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SInitChannel));
			if (!pstChildVapiReq)
			{
				Status = VAPI_ERR_NOMEM;
				goto finish;
			}

			pstInitChannel = pstChildVapiReq->pvUserData;

			UT_MemCopy(pstChildVapiReq->pvUserData, pstInitChannel, sizeof(SInitChannel));

			/* initialise the Child request */ 
			VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
					pstVapiReq,		/* Parent Request */
					VFSM_InitFaxChannel,	/* Child Request Handler */
					INIT_FAX_CHANNEL_INIT);	/* Child Request handler state */

			pstVapiReq->usReqState = INIT_CHANNEL_FINISHED;
			/*add it at the front of the request list of the channel
			and call the ProcessRequestList function to process this child request first.*/
			VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

			break;

		case eVOIP:
			break;

		default:
			UT_ErrorLog(APPITF, "VFSM_InitChannel: Channel type %d not supported yet\n");
			Status = VAPI_ERR_INVALID_PARAM;
			goto finish;
			break;
		}

		break;

	case INIT_CHANNEL_FINISHED:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		goto finish;
		break;
	}

	return SUCCESS;

finish:
	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_InitChannel: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_InitChannel: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_InitChannel: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}


/****************************************************************************
 * VFSM_InitChannelTransport : The function does the following things -
 ***************************************************************************/
/*! 
 *  \b Implementation: \n
 *      -#  Send several command to the device depending on the OpMode parameter
 *
 *  \b Inputs-Outputs: \n
 *	\li pstVapiRep->pvUserData is a SFsmModifyChannelInfo structure containing: 
 *		\li DevChannelType = the channel type
 *		\li OpMode = the operation mode
 *		\li ProfileId = the profile ID.
 *	\li pstVapiReq The processed request.
 *	\li pstMsg  Message obtained from the device.
 *
 *  \b Returns: \n
 *	\li SUCCESS
 *	\li VAPI_ERR_INVALID_PARAM
 *
 */
VSTATUS VFSM_InitChannelTransport(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status=SUCCESS;
	SFsmModifyChannelInfo *ChannelInfo;
	SVoIPChnlParams *pstChnlParams;
	CONNID ConnId;
	ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_InitChannelTransport: Entering conn(%u) state(%d)\n", ConnId, pstVapiReq->usReqState);

	pstChnlParams = pstChnl->pstChnlParams;
	ChannelInfo = (SFsmModifyChannelInfo *) pstVapiReq->pvUserData;

	/* FIXME All these command required to be sent in multi mode */
	switch (pstVapiReq->usReqState)
	{
	case INIT_CHANNEL_TRANSPORT_INIT:

		/* configure the eth header */
		pstVapiReq->usReqState = INIT_CHANNEL_TRANSPORT_IP_PARAMS_SEND;
		Status = VDEV_SendEthMacAddr(pstChnl, pstChnl->stEthMacAddresses.astSrcMacAddr,
					pstChnl->stEthMacAddresses.astDestMacAddr,
					pstChnl->stEthMacAddresses.usVlanId);

		if (Status != SUCCESS)
			goto finish;

		break;

	case INIT_CHANNEL_TRANSPORT_IP_PARAMS_SEND:
		/* for the M821xx or M83xxx the SET_ETH_HDR_CHANNEL command has not been sent
		so we do not check it*/
		if((pstChnl->pstDev->ucDevType != DEV_TYPE_M821XX) &&
		   (pstChnl->pstDev->ucDevType != DEV_TYPE_M83XXX))
		{
			/* read the answer to set_eth_channel_mac command */
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
						CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_SET_ETH_HDR_CHAN);

			if (Status != SUCCESS)
				goto finish;
		}

		pstVapiReq->usReqState = INIT_CHANNEL_TRANSPORT_FINISHED;
		Status = VDEV_SendIpHdrChnl(pstChnl, &(pstChnl->pstChnlParams->stIpHdrParams));
		if (Status != SUCCESS)
			goto finish;

		break;

	case INIT_CHANNEL_TRANSPORT_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
						   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_SET_IP_HDR_CHANNEL);

		if (Status != SUCCESS)
			goto finish;

		/* The IP header is now configured */
		pstChnl->bIPHdrSet = True;

		goto finish;
	}

	return SUCCESS;

finish:
	UT_Log(VCORE, DEBUG, "VFSM_InitChannelTransport: Completing req state(%u) status(%d) conn(%u)\n",
	       pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_InitChannelTransport: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_SetNetworkParams
 ***************************************************************************/
VSTATUS VFSM_SetNetworkParams(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	gtl_msg_t *pstMsgMulti;
	SVoIPChnlParams *pstChnlParams;
	CONNID ConnId;
	ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_SetNetworkParams: Entering conn(%u) state(%d)\n", ConnId, pstVapiReq->usReqState);

	pstChnlParams = pstChnl->pstChnlParams;

	/* FIXME All these command required to be sent in multi mode */
	switch (pstVapiReq->usReqState)
	{
	case INIT_CHANGED_CONN_INIT:
		/* configure the eth header */
		pstVapiReq->usReqState = INIT_CHANGED_CONN_FINISHED;

		pstMsgMulti = UT_Calloc(1, sizeof(gtl_msg_t));
		if (pstMsgMulti == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		pstMsgMulti->fifo = UT_Calloc(1, pstChnl->pstDev->MaxMsgSize);	/* alloc the max fifo length   */
		if (pstMsgMulti->fifo == NULL)
		{
			UT_FreeMem(pstMsgMulti);
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* init msg fields to defautl values*/
		VCORE_InitMsg(pstMsgMulti, pstMsgMulti->fifo, pstChnl->pstDev->MaxMsgSize);

		Status = VDEV_SetEthMacAddr(pstMsgMulti, pstChnl->usMSPChnlId,
					pstChnl->stEthMacAddresses.astSrcMacAddr,
					pstChnl->stEthMacAddresses.astDestMacAddr,
					pstChnl->stEthMacAddresses.usVlanId);

		if (Status != SUCCESS)
			goto finish;

		if (*((U8 *) pstVapiReq->pvUserData) == eOpModeIP)
		{
			Status = VDEV_SetIpHdrChnl(pstMsgMulti, &(pstChnl->pstChnlParams->stIpHdrParams), pstChnl);

			if (Status != SUCCESS)
				goto finish;
		}

		pstVapiReq->pvExecData = pstMsgMulti;	/* Save the multilist structure */
		VCORE_SendMSPReq(pstChnl, pstMsgMulti);

		break;

	case INIT_CHANGED_CONN_FINISHED:
		Status = VCORE_CheckStatusMulti(pstChnl, (gtl_msg_t *) (pstVapiReq->pvExecData), Event, pstMsg);
		UT_FreeMem(((gtl_msg_t *) (pstVapiReq->pvExecData))->fifo);
		UT_FreeMem(pstVapiReq->pvExecData);
		pstVapiReq->pvExecData = NULL;

		if (Status != SUCCESS)
			goto finish;

		/* The IP header is now configured */
		pstChnl->bIPHdrSet = True;

		goto finish;
	}

	return SUCCESS;

finish:
	UT_Log(VCORE, DEBUG, "VFSM_SetNetworkParams: Completing req state(%u) status(%d) conn(%u)\n",
	       pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetNetworkParams: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_RecoverChannel : The function does the following things -
 ***************************************************************************/
/*! 
 *  \b Implementation: \n
 *      -#  Send create channel query to the device
 *
 *  \b Inputs-Outputs: \n
 *	\li pstVapiRep->pvUserData is a SFsmCreateChannelInfo structure containing: 
 *		\li DevChannelType = the new channel type
 *		\li DevChannelTS = the timeslot for the channel
 *		\li *DevChannelId = the address where to store the ID of the new channel.
 *	\li pstVapiReq The processed request.
 *	\li pstMsg  Message obtained from the device.
 *
 *  \b Returns: \n
 *	\li SUCCESS
 *	\li VAPI_ERR_INVALID_PARAM
 *
 */
VSTATUS VFSM_RecoverChannel(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	SChnl *ChannelInfo;
	U16 *pausFifo = NULL;
	SChnl *ppstUserData[1]; 
	VSTATUS Status;
	DEVID DevId;
	DevId = pstSupvChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_RecoverChannel: Entering dev(%u) state(%d)\n", DevId, pstVapiReq->usReqState);

	/* retrieve the connection pointer from the request */
	UT_MemCopy(ppstUserData, pstVapiReq->pvUserData, sizeof(SChnl *));

	ChannelInfo = ppstUserData[0];

	switch (pstVapiReq->usReqState)
	{
	case RECOVER_CHANNEL_INIT:

		pstVapiReq->usReqState = CREATE_CHANNEL_FINISHED;
		Status = VDEV_RecoverChannel(pstSupvChnl, ChannelInfo->usTimeSlot);
		if (Status != SUCCESS)
			goto finish;

		break;
 
	case RECOVER_CHANNEL_FINISHED:


		/* if the timeslot not already used by another connection */
		pausFifo = (U16 *) pstMsg->fifo;

		/* Check if the queried timeslot exists */
		if (UT_LE2CPU16(pausFifo[4]) == 0xFFFF)
		{
			Status = VAPI_ERR_RECOVER_TIMESLOT;
			goto finish;
		}

		/* the timeslot exists so recover the channel id */
		ChannelInfo->usMSPChnlId = UT_LE2CPU16(pausFifo[4]);
		Status = DMGR_AddChannel(ChannelInfo->pstDev, ChannelInfo);

		/* check if the queried connection type matches the recoved channel type */
		if (ChannelInfo->usConnType != UT_LE2CPU16(pausFifo[5]))
		{
			Status = VAPI_ERR_RECOVER_WRONG_TYPE;
			goto finish;
		}

		Status = SUCCESS;
		goto finish;

		break;

	}

	return SUCCESS;

finish:

	UT_Log(VCORE, DEBUG, "VFSM_RecoverChannel: Completing state(%u) status(%d) dev(%u)\n",
		pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_RecoverChannel: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

/****************************************************************************
 * VFSM_C2CConnectUp : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# lock 1st channel
 *      -# lock 2ndt channel  (if required) 
 *      -# Send C2C CONNECT UP to supervisor 
 *      -# unlock 1st channel
 *      -# unlock 2ndt channel  (if required) 
 *
 *  \return 
 *  SUCCESS 
 *  VAPI_ERR_UNDEFINED
 *
 *  \param pstChnl Supervisor channel.
 *  \param pstMsg  Message obtained from MSP.
 */

VSTATUS VFSM_C2CConnectUP(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	SC2CConnectOption *pstC2CConnectParams;	/* Params provided by the user */
	struct _C2C_CONNECT_UMA_UP stConnectUMAUP;
	VSTATUS Status;
	SVapiReq *pstChildVapiReq;
	SVapiReq **ppstLockVapiReq;
	DEVID DevId = pstSupvChnl->pstDev->DevId;
	SChnl * pstChnl1 = NULL;
	SChnl * pstChnl2 = NULL;

	UT_Log(VCORE, DEBUG, "VFSM_C2CConnectUP: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);


	pstC2CConnectParams = (SC2CConnectOption *) pstVapiReq->pvUserData;

	/* retrieve the 1st channel involved in the C2CConnectUp session*/
	pstChnl1 = DMGR_GetConnection(pstC2CConnectParams->ConnA);
	if (pstChnl1 == NULL)
	{
		UT_ErrorLog(APPITF, "VFSM_C2CConnectUp: Connection Id(%u) doesn't exist\n",	pstC2CConnectParams->ConnA);
		Status = VAPI_ERR_INVALID_CONNID;
		goto finish;
	}
	/* retrieve the 2nd channel involved in the C2CConnectUp session in case on intra-comcerto type*/
	if (pstC2CConnectParams->ucInterMode == C2C_CONNECT_UMA_UP_CONNECTION_TYPE_INTRA)
	{
		pstChnl2 = DMGR_GetConnection(pstC2CConnectParams->ConnB);
		/* Make sure the 2nd channel is still there */
		if (pstChnl2 == NULL)
		{
			UT_ErrorLog(APPITF, "VFSM_C2CConnectUP: Connection Id(%u) doesn't exist\n",	pstC2CConnectParams->ConnB);
			Status = VAPI_ERR_INVALID_CONNID;
			goto finish;
		}
	}

	switch (pstVapiReq->usReqState)
	{

	case C2C_CONNECT_UP_INIT:

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/*Allocate room for 2 possible requests*/
		pstVapiReq->pvExecData = (U8 *) UT_AllocMem(2 * sizeof(SVapiReq *));
		if (!pstVapiReq->pvExecData)
		{
			UT_FreeMem(pstChildVapiReq);
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* save the pointer of this request in execdata*/
		ppstLockVapiReq = pstVapiReq->pvExecData;
		*ppstLockVapiReq = pstChildVapiReq;

		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
			pstVapiReq,			/* Parent Request */ 
			VFSM_LockConnection,		/* Child Request Handler */
			LOCK_CONN_INIT);		/* Child Request handler state */

		/*If intra-comcerto type lock the second connection*/
		if (pstC2CConnectParams->ucInterMode == C2C_CONNECT_UMA_UP_CONNECTION_TYPE_INTRA)
			pstVapiReq->usReqState = C2C_CONNECT_UP_LOCK_CHANNEL2;
		else
			pstVapiReq->usReqState = C2C_CONNECT_UP_START;

		/*add it to the queue of the current 1st connection for process */
		Status = VCORE_ProcessLockRequest(pstChnl1, pstChildVapiReq);

		break;

	case C2C_CONNECT_UP_LOCK_CHANNEL2:

		/* retrieve the child status */
		Status = pstVapiReq->Status;
		if (Status != SUCCESS)
			goto finish;

		if (Event != FSM_EVENT_LOCKED)
			UT_ErrorLog(APPITF, "VFSM_C2CConnectUP: Something is going wrong in inter Locking\n");

		/* The 1st channel has now a request posted in its queue
		no other request posted from the user application will be processed before 
		we remove the request.
		We can also use the VCORE_StartChannelChildRequest() for this channel */

		/**********************************************************************************/
		/* FROM HERE WE NEED TO MAKE SURE TO UNLOCK THE FIRST CHANNEL IN CASE OF FAILURE */
		/* ONLY CHILD REQUEST CAN BE POSTED TO THE LOCKED CONNECTION			 */
		/* NO DIRECT CALL TO VDEV_xxx functions						 */
		/**********************************************************************************/

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto unlock_finish1;
		}

		/*point to the 2nd pointer*/
		ppstLockVapiReq = pstVapiReq->pvExecData + sizeof(SVapiReq *);
		*ppstLockVapiReq = pstChildVapiReq;

		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
			pstVapiReq,			/* Parent Request */ 
			VFSM_LockConnection,		/* Child Request Handler */
			LOCK_CONN_INIT);		/* Child Request handler state */

		pstVapiReq->usReqState = C2C_CONNECT_UP_START;
		/*add it to the queue of the current connection for process */
		VCORE_StartChannelChildRequest(pstChnl2, pstChildVapiReq);

		break;

	case C2C_CONNECT_UP_START:

		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto unlock_finish2;

		if (Event != FSM_EVENT_LOCKED)
			UT_ErrorLog(APPITF, "VFSM_C2CConnectUP: Something is going wrong in inter Locking\n");

		/* The 2nd channel has been succesfully locked*/
		/* We now want to send the C2C CONNECT UP command to the supervisor */
		/* process the request on supervisor */
		UT_MemSet(&stConnectUMAUP, 0, sizeof(struct _C2C_CONNECT_UMA_UP));

		/* retrieve the C2C connect UP parameters from the request */
		stConnectUMAUP.param_4.bits.connection = pstC2CConnectParams->bAction;
		stConnectUMAUP.param_4.bits.connection_type = pstC2CConnectParams->ucConnType;
		stConnectUMAUP.param_4.bits.inter_mode = pstC2CConnectParams->ucInterMode;

		stConnectUMAUP.param_5.bits.up_channel_a = pstChnl1->usMSPChnlId;

		if (pstC2CConnectParams->ucInterMode == C2C_CONNECT_UMA_UP_CONNECTION_TYPE_INTRA)
			stConnectUMAUP.param_6.bits.up_channel_b = pstChnl2->usMSPChnlId;

		stConnectUMAUP.param_7.word = pstC2CConnectParams->ucTimerInterval | (pstC2CConnectParams->ucRateControl << 10);
		stConnectUMAUP.param_8.word = pstC2CConnectParams->usRFCIModeGroup1;
		stConnectUMAUP.param_9.word = pstC2CConnectParams->usRFCIModeGroup2;
		stConnectUMAUP.param_10.word = pstC2CConnectParams->usRFCIModeGroup3;
		stConnectUMAUP.param_11.word = pstC2CConnectParams->usRFCIModeGroup4;
		stConnectUMAUP.param_12.word = pstC2CConnectParams->usRFCIModeGroup5;
		stConnectUMAUP.param_13.word = pstC2CConnectParams->usRFCIModeGroup6;

		pstVapiReq->usReqState = C2C_CONNECT_UP_FINISHED;
		Status = VDEV_C2CConnectUP(pstSupvChnl, &stConnectUMAUP);

		break;

	case C2C_CONNECT_UP_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstSupvChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_C2C_CONNECT_UMA_UP);

		if (pstC2CConnectParams->ucInterMode == C2C_CONNECT_UMA_UP_CONNECTION_TYPE_INTRA) /*intra-comcerto type unlock both connections*/
			goto unlock_finish2;
		else
			goto unlock_finish1;
			
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_C2CConnectUP: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

unlock_finish2:
		/* we have finished with the 2nd channel
		 so remove the locking request from its queue by calling the FSM lock handler directly
		(stored in pstVapiReq->pvExecData) */

		/*point to the 2nd pointer*/
		ppstLockVapiReq = pstVapiReq->pvExecData + sizeof(SVapiReq *);

		VFSM_LockConnection(pstChnl2, *(ppstLockVapiReq), FSM_EVENT_UNLOCK, NULL);

unlock_finish1:
		/* we have finished with the 1st channel
		 so remove the locking request from its queue by calling the FSM lock handler directly
		(stored in pstVapiReq->pvExecData) */

		ppstLockVapiReq = pstVapiReq->pvExecData;

		VFSM_LockConnection(pstChnl1, *(ppstLockVapiReq), FSM_EVENT_UNLOCK, NULL);

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_C2CConnectUP: Error dev(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n", DevId, pstVapiReq, pstVapiReq->usReqState, pstSupvChnl);

	UT_Log(VCORE, DEBUG, "VFSM_C2CConnectUP: Completing req state(%u) status(%d) dev(%u)\n",
				pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_C2CConnectUP: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}


