/*! \file helper_itf.c
 @defgroup helper_itf VAPI help utilities API functions
 *  @{
 * The VAPI utilities is a set of API intended to help in the Comcerto command construction.\n
 * The constructed command is not linked to a particular device, connection or conference. \n
 * Once constructed the command can be sent using one of the API: \n
 *  - VAPI_SendDeviceMessage()
 *  - VAPI_SendConnectionMessage()
 *  - VAPI_SendConferenceMessage()
 *  - VAPI_PassThru()
 *
 * The same message can be used for several connection. 
 */

/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include "ut.h"
#include "vapi.h"
#include "dmgr.h"
#include "vcore.h"
#include "vcore_voip.h"
#include "appitf.h"
#include "cfg.h"
#include "msp.h"

extern SToneGeneric astToneGeneric_g[];
extern Boolean bVapiInitialized_g;
extern SCidGen SCidGen_g;
extern SCidDet SCidDet_g;

/***************************************************************************
 * VAPI_AllocateMessage The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	Allocate a gtl_msg_t structure message and the associated fifo buffer. \n
*	The gtl_msg_t is an VAPI internal structure and some fields are initialized.\n
*	The passed size is used to internally allocate the fifo buffer
*	which will contain the Comcerto commands. \n
*	This size must be big enougth to contain:
*	- the command(s) header(s) (8 bytes each)
*	- the parameters of each command(s)
*	- the potential inter command(s) padding
*	- the terminator 4 bytes
*
*	It is mandatory to use VAPI_AllocateMessage() to use the VAPI help functions such as: 
*	- VAPI_SetMessage()
*	- VAPI_SetMessageFromBuffer() \n
*
*	The allocated buffers can be freed using VAPI_FreeMessage().\n
*	\n<b>Note 1: The size of the fifo can be Firmware release and control interface dependent.\n
*	It is recommended to take in account to the device fifo size capability </b>
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		 <td style="vertical-align: top;">usSize</td>
*		 <td style="vertical-align: top;">Size of the fifo </td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li Pointer to the message structure
*	\li NULL if the allocation failed
*
*	\n \b Usage:
*	\include allocate_message.c
*
*	\n \b Commands:
*	List of the Comcerto commands: \n
*	- NONE
*/
void *VAPI_AllocateMessage(U16 usSize)
{
	gtl_msg_t *pstMsg;

	/* we take the option to allocate the current max size (1500)
	if the firmware/control interface doesn't suppport this size, 
	the command will get a NACK at sending time*/
	if (usSize > ETH_FIFO_MAX_SIZE)
	{
		UT_ErrorLog(APPITF, "VAPI_AllocateMessage: Size out of range\n");
		goto err;
	}

	pstMsg = UT_Calloc(1, sizeof(gtl_msg_t));
	if (pstMsg == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_AllocateMessage: No memory available\n");
		goto err;
	}

	pstMsg->fifo = UT_Calloc(1, usSize);	/* alloc the max fifo length   */
	if (pstMsg->fifo == NULL)
	{
		UT_FreeMem(pstMsg);
		UT_ErrorLog(APPITF, "VAPI_AllocateMessage: No memory available\n");
		goto err;
	}

	VCORE_InitMsg(pstMsg, (U8 *)pstMsg->fifo, usSize); 

	return pstMsg;
err:
	return NULL;
}

/***************************************************************************
 * VAPI_FreeMessage The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	Frees a previously allocated message structure and the associated fifo buffer \n
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*      </tr>
*      <tr>
*		<td style="vertical-align: top;">pstMsg</td>
*		<td style="vertical-align: top;">Pointer to the message structure to free</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_NULL_POINTER_PASSED
*
*	\n \b Usage:
*	\include free_message.c
*
*	\n \b Commands:
*	List of the Comcerto commands: \n
*	- NONE
*/
VSTATUS VAPI_FreeMessage(void *pvMsg)
{
	VSTATUS status = SUCCESS;
	gtl_msg_t *pstGtlMsg;

	if (pvMsg == NULL)
	{
		status = VAPI_ERR_NULL_POINTER_PASSED;
		goto err;
	}
	pstGtlMsg = (gtl_msg_t *)pvMsg;

	if (pstGtlMsg->fifo != NULL)
		UT_FreeMem(pstGtlMsg->fifo);

	UT_FreeMem(pstGtlMsg);
err:
	return status;

}

/***************************************************************************
 * VAPI_InitMessage The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	Reinitialises all the fields of the gtl_msg_t structure message 
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pvMsg</td>
*		<td style="vertical-align: top;">Pointer to the message structure to initialize</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*
*	\n \b Commands:
*	List of the Comcerto commands: \n
*	- NONE
*/
void VAPI_InitMessage(void *pvMsg)
{
	if(pvMsg != NULL)
	{
		gtl_msg_t *pstMsg = (gtl_msg_t *)pvMsg;

		if(pstMsg->fifo != NULL)
			UT_MemSet(pstMsg->fifo, 0, pstMsg->max_fifo_size);

		pstMsg->fifo_size = 0;
	}
}

/***************************************************************************
* VAPI_SetMessage The function does the following things -
***************************************************************************/
/*!
*      \n \b Description: \n
*	This function add the passed comcerto Command Class, Type, Function code and parameters \n
*	to an existing message structure\n.
*	If the message already contains a command, the new command is added to the message to build \n
*	a multi command message.
*	The padding for the 32 bit alignment is performed by the VAPI_SetMessage().
*	\n<b>Note 1: Multi command is not allowed if the command class is device level.</b>
*	\n<b>Note 2: This command can be used prior to VAPI_Init().</b>
*	\n<b>Note 3: The message structure need to be preallocated and initialized.\n
*		This is done using the VAPI_AllocateMessage() API</b>
*
*	\n
*      <table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*      <tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstMsg</td>
*		<td style="vertical-align: top;">Pointer to the message structure to be updated</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucCmdClass</td>
*		<td style="vertical-align: top;">Comcerto Command Class</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucCmdType</td>
*		<td style="vertical-align: top;"> Comcerto Command type<td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usFcode</td>
*		<td style="vertical-align: top;"> Comcerto Function code<td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucNumParams</td>
*		<td style="vertical-align: top;"> Number of parameters to be added<td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">....</td>
*		<td style="vertical-align: top;"> List of parameters. \n
*					The parameters need to be provided in the correct endianness<td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_NULL_POINTER_PASSED
*	\li VAPI_ERR_FIFO_SIZE
*
*	\n \b Usage:
*	\include set_message.c
*
*	\n \b Commands:
*	List of the Comcerto commands: \n
*	- NONE
*/
VSTATUS VAPI_SetMessage(OUT void *pvMsg , IN U8 ucCmdClass, IN U8 ucCmdType, IN U16 usFcode, 
				IN U16 ucNumParams,... )
{
	VSTATUS Status;
	va_list ParamList;
	gtl_msg_t *pstGtlMsg;

	if(pvMsg == NULL)
	{
		UT_ErrorLog(VCORE, "VAPI_SetMessage: Invalid pstMsg\n");
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		goto out;
	}

	Status = VCORE_CheckCtCcValidity(usFcode, ucCmdClass, ucCmdType);
	if (Status != SUCCESS)
	{
		UT_ErrorLog(VCORE, "VAPI_SetMessage: Invalid commands (fc 0x%04x, cc 0x%02x, ct 0x%02x)\n", usFcode, ucCmdClass, ucCmdType);
		goto out;
	}

	pstGtlMsg = (gtl_msg_t *)pvMsg;

	/* setup the fifo header*/
	VCORE_SetupCmdHdrMsg(pstGtlMsg, ucCmdClass, ucCmdType, usFcode);

	/* setup the fifo payload*/
	UT_va_start(ParamList, ucNumParams);
	Status = VCORE_SetPayloadMsgFromList(pstGtlMsg, ucNumParams, ParamList);
	UT_va_end(ParamList);
out:
	return (Status);
}

/***************************************************************************
* VAPI_SetMessageFromBuffer The function does the following things -
***************************************************************************/
/*!
*      \n \b Description: \n
*	This function add the passed comcerto Command Class, Type, Function code and parameters \n
*	to an existing message structure\n.
*	If the message already contains a command, the new command is added to the message to build \n
*	a multi command message.
*	The padding for the 32 bit alignment is performed by the VAPI_SetMessage().
*	\n<b>Note 1: Multi command is not allowed if the command class is device level.</b>
*	\n<b>Note 2: This command can be used prior to VAPI_Init().</b>
*	\n<b>Note 3: The message structure need to be preallocated and initialized.\n
*		This is done using the VAPI_AllocateMessage() API</b>
*
*	\n
*      <table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*      <tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstMsg</td>
*		<td style="vertical-align: top;">Pointer to the message structure to be updated</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucCmdClass</td>
*		<td style="vertical-align: top;">Comcerto Command Class</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucCmdType</td>
*		<td style="vertical-align: top;"> Comcerto Command type<td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usFcode</td>
*		<td style="vertical-align: top;"> Comcerto Function code<td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucNumParams</td>
*		<td style="vertical-align: top;"> Number of parameters to be added<td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pusParameters</td>
*		<td style="vertical-align: top;"> Buffer containing the parameters of the command. \n
*					The parameters need to be provided in the correct endianness<td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_NULL_POINTER_PASSED
*	\li VAPI_ERR_FIFO_SIZE
*
*	\n \b Usage:
*	\include set_message_from_buffer.c
*
*	\n \b Commands:
*	List of the Comcerto commands: \n
*	- NONE
*/
VSTATUS VAPI_SetMessageFromBuffer(OUT void *pvMsg , IN U8 ucCmdClass, IN U8 ucCmdType, IN U16 usFcode, 
				IN U16 ucNumParams, IN U16 *pusParameters)
{
	VSTATUS Status;
	gtl_msg_t *pstGtlMsg;

	if(pvMsg == NULL)
	{
		UT_ErrorLog(VCORE, "VAPI_SetMessageFromBuffer: Invalid pstMsg\n");
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		goto out;
	}

	Status = VCORE_CheckCtCcValidity(usFcode, ucCmdClass, ucCmdType);
	if (Status != SUCCESS)
	{
		UT_ErrorLog(VCORE, "VAPI_SetMessageFromBuffer: Invalid commands (fc 0x%04x, cc 0x%02x, ct 0x%02x)\n", usFcode, ucCmdClass, ucCmdType);
		goto out;
	}

	pstGtlMsg = (gtl_msg_t *)pvMsg;

	/* setup the fifo header*/
	VCORE_SetupCmdHdrMsg(pstGtlMsg, ucCmdClass, ucCmdType, usFcode);

	Status = VCORE_SetPayloadMsgFromBuffer(pstGtlMsg, ucNumParams, pusParameters);

out:
	return (Status);
}

/***************************************************************************
 * VAPI_SendDeviceMessage The function does the following things -
 ***************************************************************************/
/*!
*      \n \b Description: \n
*	This API is equivalent to VAPI_PassThru() with the level set to CMD_LEVEL_DEVICE \n
*/
VSTATUS VAPI_SendDeviceMessage(IN U32 uiID, IN SMsg * pstMsg, IN SRequest * pstRequest, 
					OUT U8 * pucBuffer, IN OUT U32 * puBufLen)
{
	return (VAPI_PassThru(uiID, CMD_LEVEL_DEVICE, pstMsg, pstRequest, pucBuffer, puBufLen));
}

/***************************************************************************
 * VAPI_SendConnectionMessage The function does the following things -
 ***************************************************************************/
/*!
*      \n \b Description: \n
*	This API is equivalent to VAPI_PassThru() with the level set to CMD_LEVEL_CONN \n
*/
VSTATUS VAPI_SendConnectionMessage(IN U32 uiID, IN SMsg * pstMsg, IN SRequest * pstRequest, 
					OUT U8 * pucBuffer, IN OUT U32 * puBufLen)
{
	return (VAPI_PassThru(uiID, CMD_LEVEL_CONN, pstMsg, pstRequest, pucBuffer, puBufLen));
}

/***************************************************************************
 * VAPI_SendConferenceMessage The function does the following things -
 ***************************************************************************/
/*!
*      \n \b Description: \n
*	This API is equivalent to VAPI_PassThru() with the level set to CMD_LEVEL_CONF \n
*/
VSTATUS VAPI_SendConferenceMessage(IN U32 uiID, IN SMsg * pstMsg, IN SRequest * pstRequest, 
					OUT U8 * pucBuffer, IN OUT U32 * puBufLen)
{
	return (VAPI_PassThru(uiID, CMD_LEVEL_CONF, pstMsg, pstRequest, pucBuffer, puBufLen));
}

/***************************************************************************
 * VAPI_SetTone: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This API set the frequencies, amplitudes, on/off time, cadences for Dual (Function code 0x8009) or 
*	Quad tone (Function code 0x8040).\n
*	With this API the user adds or modifies existing tone definition.\n
*	The tone parameters are stored in a VAPI internal tone array: astToneGeneric_g.\n
*	The parameters such as direction, mixing mode, etc. are handled when the tone is play with VAPI_PlayTone().\n
*	The tone parameters are passed through a list which depends on the type of the tone.\n
*
*	For Dual Tone parameters list could be up to 4 blocks (5 blocks for Comcerto 100 and 1000 devices) with 6 parameters per block as follows: \n
*	{Primary Tone Frequency, Primary Tone Amplitude, Second Tone Frequency, Second Tone Amplitude, On Time, Off Time).\n
*
*	For Quad Tone parameters list is as follows: \n
*	{First Tone Frequency, First Tone Amplitude, Second Tone Frequency, Second Tone Amplitude, \n
*	Third Tone Frequency, Third Tone Amplitude, Fourth Tone Frequency, Fourth Tone Amplitude, On Time, Off Time}\n
*
*	This API can be used only after VAPI_Init() call and before VAPI_InitDevice.\n
*	Actually before channel creation the tone to be played on.\n
*	When VAPI_Init() is called, VAPI_SetTone() is internally issued to configure a set of "default tone".\n
*	Please see the default tone configuration below:\n
*
*	<b>Pre defined DTFM tones.</b>\n
*	<pre>
*	eDTMFTONE_1     697, 100, 1209, 100, 80, 80
*	eDTMFTONE_2     697, 100, 1336, 100, 80, 80
*	eDTMFTONE_3     697, 100, 1477, 100, 80, 80
*	eDTMFTONE_4     770, 100, 1209, 100, 80, 80
*	eDTMFTONE_5     770, 100, 1306, 100, 80, 80
*	eDTMFTONE_6     770, 100, 1477, 100, 80, 80
*	eDTMFTONE_7     852, 100, 1209, 100, 80, 80
*	eDTMFTONE_8     852, 100, 1336, 100, 80, 80
*	eDTMFTONE_9     852, 100, 1477, 100, 80, 80
*	eDTMFTONE_A     697, 100, 1633, 100, 80, 80
*	eDTMFTONE_B     770, 100, 1633, 100, 80, 80
*	eDTMFTONE_C     852, 100, 1633, 100, 80, 80
*	eDTMFTONE_D     941, 100, 1633, 100, 80, 80
*	eDTMFTONE_STAR  941, 100, 1209, 100, 80, 80
*	eDTMFTONE_0     941, 100, 1336, 100, 80, 80
*	eDTMFTONE_HASH  941, 100, 1477, 100, 80, 80
*	</pre>
*
*	<b>Call progess Tones</b>\n
*	<pre>
*                         Repeat Tone frequencies, amplitudes, on/off time definition 
*	eNULLTONE         1,     0, 0, 0, 0, 0, 0
*	eCONGESTIONTONE   0xFF,  480, 100, 620, 100, 250, 250
*	eWARNINGTONE      1,     440, 100, 0, 0, 1500, 8000, 440, 100, 0, 0, 500, 8000
*	eWAITINGTONE      1,     440, 100, 0, 0, 300, 10000
*	eCONFIRMATIONTONE 0      350, 170, 440, 170, 100, 100, 350, 170, 440, 170, 100, 100, 350, 170, 440, 170, 100, 100
*	eDIALTONE         0xFF,  350, 100, 440, 100, 0, 0
*	eBUSYTONE         0xFF,  480, 100, 620, 100, 500, 500
*	eRINGBACKTONE     0xFF,  440, 100, 480, 100, 1000, 3000
*	eSPECIALTONE      1,     950, 100, 0, 0, 330, 0, 1400, 100, 0, 0, 330, 0, 1800, 100, 0, 0, 330, 0
*	eSASTONE          0      440, 0xD4, 00, 00, 0x12C, 0x12C
*	</pre>
*
*	<b>Customizable Tones</b>\n
*	30 customizable tones are avialable (eCUSTOM_1 to eCUSTOM_30)\n
*	The user can add new tones definitions to the existing tones. \n
*	Please check the usage section for more details on tone customization. \n
*
*	After setting parameters for particular eToneId VAPI_PlayTone() may be called \n
*	to play tone to defined direction (please refer to VAPI_PlayTone doc for details).\n
*	<b>NOTE: The current number of tone supported is 35 defined by eDUMMYTONE.\n
*	To increase the number of tones, please add new values to the EToneId enumeration and make\n
*	sure that eDUMMYTONE is correctly updated.\n </b>
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eToneId</td>
*		<td style="vertical-align: top;">The tone index.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eToneType</td>
*		<td style="vertical-align: top;">Specifies tone type dual/quad.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usCadenceRpt</td>
*		<td style="vertical-align: top;">Cadence repeat (only for Dual tone). \n
*			Determines the number of times the cadence pattern \n
*			(defined by the cadence blocks that follow) is repeated (only for Dual Tone)</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usParamNum</td>
*		<td style="vertical-align: top;">Number of parameters to be passed.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">...</td>
*		<td style="vertical-align: top;">List of parameters.</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include set_tone.c
*
*	\n \b Commands:
*	no commands sent
*
*/
VSTATUS VAPI_SetTone(EToneId eToneId, EToneType eToneType, U16 usCadenceRpt, U16 usParamNum,...)
{
	VSTATUS Status = SUCCESS;
	va_list ParamList;
	struct _VOIP_QUADTONEGEN *pstQuad;
	struct _VOIP_TONEGEN *pstDual;
	U32 ausParam[30];
	int i;

	if (!bVapiInitialized_g)
	{
		/*Used printf instead of UT_Log because UT is not yet initialized */
		printf("%s: VAPI library not initialized.\n", "VAPI_SetTone");
		Status = VAPI_ERR_LIB_NOT_INITIALIZED;
		goto out;
	}	

	if (eToneId >= eDUMMYTONE)
	{
		UT_ErrorLog(APPITF, "Invalid Tone ID %d\n", eToneId);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	if (eToneType > eQuadTone)
	{
		UT_ErrorLog(APPITF, "Tone type isn't supported (%d)\n", eToneType);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	if ((eToneType == eDualTone) && 
		((usParamNum < 6) || (usParamNum > 30) || (usParamNum%6)))
	{
		UT_ErrorLog(APPITF, "Invalid Params number %d\n", usParamNum );
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}
	else if ((eToneType == eQuadTone) && (usParamNum != 10))
	{
		UT_ErrorLog(APPITF, "Invalid Params number %d\n", usParamNum);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	/* fill Params array from list*/
	UT_va_start(ParamList, usParamNum);
	for (i = 0; i < usParamNum; i++)
	{
		ausParam[i] = va_arg (ParamList, int);
	}	
	UT_va_end(ParamList);

	/*lock the mutex to lock the access to astToneGeneric_g array*/
	UT_MutexLock(&stMutexToneGenArr_g);

	/* Remove all previous tone parameters for current tone id */
	UT_MemSet(&astToneGeneric_g[eToneId], 0, sizeof (SToneGeneric));
	
	astToneGeneric_g[eToneId].eToneType = eToneType;

	if (eToneType == eDualTone)
	{
		/* dual tone */
		pstDual =  &astToneGeneric_g[eToneId].TonePayload.stDualTone;

		astToneGeneric_g[eToneId].ucParamBlockNum = usParamNum/6;
		pstDual->param_4.bits.cadence_rpt = usCadenceRpt;

		/* parameters for cadence number [1] */
		if (usParamNum >= 6)
		{
			if ((ausParam[0] > TONE_MAX_FREQ)|| \
					(ausParam[1] > TONE_MAX_AMP)|| \
					(ausParam[2] > TONE_MAX_FREQ)||\
					(ausParam[3] > TONE_MAX_AMP))
			{
				UT_ErrorLog(APPITF, "Invalid Dual tone Params for cadence number [1]\n");
				Status = VAPI_ERR_INVALID_PARAM;
				goto unlock_out;
			}
		
			pstDual->prim_tone_freq = UT_CPU2LE16(ausParam[0]);
			pstDual->prim_tone_amp = UT_CPU2LE16(ausParam[1]);
			pstDual->sec_tone_freq = UT_CPU2LE16(ausParam[2]);
			pstDual->sec_tone_amp = UT_CPU2LE16(ausParam[3]);
			pstDual->on_time = UT_CPU2LE16(ausParam[4]);
			pstDual->off_time  = UT_CPU2LE16(ausParam[5]);
		}

		/* parameters for cadence number [2] */
		if (usParamNum >= 12)
		{
			if ((ausParam[6] > TONE_MAX_FREQ)|| \
					(ausParam[7]  > TONE_MAX_AMP)|| \
					(ausParam[8] > TONE_MAX_FREQ)||\
					(ausParam[9] > TONE_MAX_AMP))
			{
				UT_ErrorLog(APPITF, "Invalid Dual tone Params for cadence number [2]\n");
				Status = VAPI_ERR_INVALID_PARAM;
				goto unlock_out;
			}
		
			pstDual->prim_tone_freq2 = UT_CPU2LE16(ausParam[6]);
			pstDual->prim_tone_amp2 = UT_CPU2LE16(ausParam[7]);
			pstDual->sec_tone_freq2 = UT_CPU2LE16(ausParam[8]);
			pstDual->sec_tone_amp2 = UT_CPU2LE16(ausParam[9]);
			pstDual->on_time2 = UT_CPU2LE16(ausParam[10]);
			pstDual->off_time2  = UT_CPU2LE16(ausParam[11]);
		}

		/* parameters for cadence number [3] */
		if (usParamNum >= 18)
		{
			if ((ausParam[12] > TONE_MAX_FREQ)|| \
					(ausParam[13]  > TONE_MAX_AMP)|| \
					(ausParam[14] > TONE_MAX_FREQ)||\
					(ausParam[15] > TONE_MAX_AMP))
			{
				UT_ErrorLog(APPITF, "Invalid Dual tone Params for cadence number [2]\n");
				Status = VAPI_ERR_INVALID_PARAM;
				goto unlock_out;
			}
			
			pstDual->prim_tone_freq3 = UT_CPU2LE16(ausParam[12]);
			pstDual->prim_tone_amp3 = UT_CPU2LE16(ausParam[13]);
			pstDual->sec_tone_freq3 = UT_CPU2LE16(ausParam[14]);
			pstDual->sec_tone_amp3 = UT_CPU2LE16(ausParam[15]);
			pstDual->on_time3 = UT_CPU2LE16(ausParam[16]);
			pstDual->off_time3  = UT_CPU2LE16(ausParam[17]);
		}

		/* parameters for cadence number [4] */
		if (usParamNum >= 24)
		{
			if ((ausParam[18] > TONE_MAX_FREQ)|| \
					(ausParam[19] > TONE_MAX_AMP)|| \
					(ausParam[20] > TONE_MAX_FREQ)||\
					(ausParam[21] > TONE_MAX_AMP))
			{
				UT_ErrorLog(APPITF, "Invalid Dual tone Params for cadence number [4]\n");
				Status = VAPI_ERR_INVALID_PARAM;
				goto unlock_out;
			}
		
			pstDual->prim_tone_freq4 = UT_CPU2LE16(ausParam[18]);
			pstDual->prim_tone_amp4 = UT_CPU2LE16(ausParam[19]);
			pstDual->sec_tone_freq4 = UT_CPU2LE16(ausParam[20]);
			pstDual->sec_tone_amp4 = UT_CPU2LE16(ausParam[21]);
			pstDual->on_time4 = UT_CPU2LE16(ausParam[22]);
			pstDual->off_time4  = UT_CPU2LE16(ausParam[23]);
		}

		/* parameters for cadence number [5] */
		if (usParamNum == 30)
		{
			if ((ausParam[24] > TONE_MAX_FREQ)|| \
					(ausParam[25]  > TONE_MAX_AMP)|| \
					(ausParam[26] > TONE_MAX_FREQ)||\
					(ausParam[27] > TONE_MAX_AMP))
			{
				UT_ErrorLog(APPITF, "Invalid Dual tone Params for cadence number [5]\n");
				Status = VAPI_ERR_INVALID_PARAM;
				goto unlock_out;
			}
		
			pstDual->prim_tone_freq5 = UT_CPU2LE16(ausParam[24]);
			pstDual->prim_tone_amp5 = UT_CPU2LE16(ausParam[25]);
			pstDual->sec_tone_freq5 = UT_CPU2LE16(ausParam[26]);
			pstDual->sec_tone_amp5 = UT_CPU2LE16(ausParam[27]);
			pstDual->on_time5 = UT_CPU2LE16(ausParam[28]);
			pstDual->off_time5  = UT_CPU2LE16(ausParam[29]);
		}
	}
	else
	{
		/* quad tone */
		pstQuad = &astToneGeneric_g[eToneId].TonePayload.stQuadTone;

		if ((ausParam[0] > TONE_MAX_FREQ)|| \
				(ausParam[1] > TONE_MAX_AMP)|| \
				(ausParam[2] > TONE_MAX_FREQ)||\
				(ausParam[3] > TONE_MAX_AMP) ||\
				(ausParam[4] > TONE_MAX_FREQ)||\
				(ausParam[5] > TONE_MAX_AMP)||\
				(ausParam[6] > TONE_MAX_FREQ)||\
				(ausParam[7] > TONE_MAX_AMP))
		{
			UT_ErrorLog(APPITF, "Invalid Quad tone Parameters\n");
			Status = VAPI_ERR_INVALID_PARAM;
			goto unlock_out;
		}

		pstQuad->tone1_freq = UT_CPU2LE16(ausParam[0]);
		pstQuad->tone1_amp = UT_CPU2LE16(ausParam[1]);
		pstQuad->tone2_freq = UT_CPU2LE16(ausParam[2]);
		pstQuad->tone2_amp = UT_CPU2LE16(ausParam[3]);
		pstQuad->tone3_freq = UT_CPU2LE16(ausParam[4]);
		pstQuad->tone3_amp = UT_CPU2LE16(ausParam[5]);
		pstQuad->tone4_freq = UT_CPU2LE16(ausParam[6]);
		pstQuad->tone4_amp = UT_CPU2LE16(ausParam[7]);
		pstQuad->on_time= UT_CPU2LE16(ausParam[8]);
		pstQuad->off_time = UT_CPU2LE16(ausParam[9]);
	}

unlock_out:
	UT_MutexUnLock(&stMutexToneGenArr_g);
out:
	return Status;
}


/***************************************************************************
 * VAPI_SetCidGenInfo The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This API sets the Caller ID information blocks used to generate a Caller ID.\n
*	With this API the user adds, modifies or removes Caller ID information blocks.\n
*	The Caller ID parameters are stored in a VAPI internal array.\n
*	The parameters are applied to a connection when the VAPI_PlayCid is called.\n
*\n
*	The number of passed parameters must match the expected number for each different info blocks, except for the \n
*	SASToneInfoType where the user can pass less than all parameters.
*	If the eAction = eRemove (0) the number of parameters is not taken it account.
*
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Block info type</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b>Number of expected parameters</b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">OnHookXmitInfoType</td>
*		<td style="vertical-align: top;">6</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">OffHookXmitInfoType  </td>
*		<td style="vertical-align: top;">4</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">CPEAlertToneInfoType</td>
*		<td style="vertical-align: top;">5</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">CPEAckInfoType</td>
*		<td style="vertical-align: top;">7</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">SASToneInfoType</td>
*		<td style="vertical-align: top;">8 (1 dual tone block) \n
*						14 (2 dual tone blocks)\n
*						20 (3 dual tone blocks)\n
*						26 (4 dual tone blocks)\n
*						</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">NTTInfoType</td>
*		<td style="vertical-align: top;">17</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">StopBitsInfoType</td>
*		<td style="vertical-align: top;">1</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">PacketProcessingInfoType</td>
*		<td style="vertical-align: top;">1</td>
*	</tr>
*	</table>
*	\n
*	This API can be called any time after VAPI_Init(), but it is convenient to initialize the caller ID during 
*	the VAPI initialization phase (i.e just after VAPI_Init().
*
*	When VAPI_Init() is called, VAPI_SetCidGenInfo() is internally issued to configure a "default Caller ID".\n
*	Please see the default caller ID configuration below:
*
*	<pre>
*	OnHookXmitInfoType: {0xB6, 0x00, 0x01, 0x012C, 0x00, 0x00B4}
*			Where   0xB6:		FSK amplitude = -18.2dbv0
*				0x00:		Data Tx phase during ring
*				0x01:		Alert info byte DT-AS
*				0x012C:		Channel seizure bits: 300 bits
*				0x00:		Channel seizure First Bit Space
*				0x00B4:		Mark bits 180 Bits.
*
*	OffHookXmitInfoType: {0xD4, 0x00, 0x01, 0x0064}
*			Where   0xB6:		FSK ampliture = -18.2dbv0
*				0x00:		Modulation freq Bell 202
*				0x01:		Alert info byte DT-AS
*				0x0064:		Mark Bits 100 bits
*
*	CPEAlertToneInfoType: {0x0852, 0x0ABE, 0x50, 0xD4, 0xD4}
*			Where   0x0852:		Low nominal frequency 2130 Hz 
*				0x0ABE:		High nominal frequency 2750 Hz	
*				0x50:		Duration 80ms
*				0xD4:		Low frequency level -21.2 dbv0
*				0xD4:		High frequency level -21.2 dbv0
*
*	CPEAckInfoType: { 0x41, 0x44, 0x1E, 0xC8, 0xA0, 0x00, 0x012C}
*			Where   0x41:		DTMF Digit 1 'A'
*				0x44:		DTMF Digit 2 'D'
*				0x1E:		Min Duration 30ms
*				0xC8:		Time out for ACK 200ms
*				0xA0:		Time to data transmission 160ms
*				0x00:		Data transmit if no ack: No
*				0x012C		Ack tone min level -30dBv0
*
*	The following block info types are not configured by default (set to 0x00):
*	- SASToneInfoType
*	- NTTInfoType
*	- StopBitsInfoType
*	- PacketProcessingInfoType
*
*	</pre>
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
	<tr>
*		<td style="vertical-align: top;">eAction </td>
*		<td style="vertical-align: top;">CID Info block to modify, add or remove.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eCidGenInfotype</td>
*		<td style="vertical-align: top;">Specifies information block type.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usParamNum</td>
*		<td style="vertical-align: top;">Number of parameters to be passed.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">...</td>
*		<td style="vertical-align: top;">List of parameters (integers of type int).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include set_cid_gen.c
*
*	\n \b Commands:
*	no commands sent
*
*/
VSTATUS VAPI_SetCidGenInfo(EAction eAction, ECidGenInfoType eCidGenInfoType, U16 usParamNum, ...)
{
	VSTATUS Status = SUCCESS;
	va_list ParamList;
	U16 ausParam[30];
	int i, j, n;

	if (!bVapiInitialized_g)
	{
		/*Used printf instead of UT_Log because UT is not yet initialized */
		printf("%s: VAPI library not initialized.\n", "VAPI_SetCidGenInfo");
		Status = VAPI_ERR_LIB_NOT_INITIALIZED;
		goto out;
	}

	if (eAction > eModify)
	{
		UT_ErrorLog(APPITF, "VAPI_SetCidGenInfo: eAction %d out of range\n", eAction);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	if (eCidGenInfoType > ePacketProcessingInfoType)
	{
		UT_ErrorLog(APPITF, "VAPI_SetCidGenInfo: eAction %d out of range\n", eCidGenInfoType);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	UT_va_start(ParamList, usParamNum);
	for (i = 0; i < usParamNum; i++)
	{
		ausParam[i] = va_arg (ParamList, int);
	}
	UT_va_end(ParamList);

	switch (eCidGenInfoType)
	{

		case eOnHookXmitInfoType:

			if (eAction == eRemove)
			{
				SCidGen_g.SOnHookXmitInfoBlock.bIsConfigured = 0;
				goto out;
			}

			if (usParamNum != 6)
			{
				UT_ErrorLog(APPITF, "VAPI_SetCidGenInfo: eOnHookXmitInfoType wrong num of param (%d)\n", usParamNum);
				Status = VAPI_ERR_INVALID_PARAM;
				goto out;
			}
			else
			{
				SCidGen_g.SOnHookXmitInfoBlock.bIsConfigured = 1;
	
				SCidGen_g.SOnHookXmitInfoBlock.usOnHookXmitInfoType[0] = 
					UT_CPU2LE16((ON_HOOK_XMIT_INFO_TYPE_CNT << 8) | eOnHookXmitInfoType);

				/* FSK Tone Amplitude / Data Transmission Phase */
				SCidGen_g.SOnHookXmitInfoBlock.usOnHookXmitInfoType[1] =
							UT_CPU2LE16((ausParam[1] << 8) | ausParam[0]);

				/* Alerting Info / Channel Seizure Bits (MSB) */
				SCidGen_g.SOnHookXmitInfoBlock.usOnHookXmitInfoType[2] =
							UT_CPU2LE16((ausParam[3] & 0xFF00) | ausParam[2]);

				/* Channel Seizure Bits (LSB) / Channel Seizure First Bit */
				SCidGen_g.SOnHookXmitInfoBlock.usOnHookXmitInfoType[3] =
							UT_CPU2LE16((ausParam[4] << 8) | (ausParam[3] & 0x00FF));

				/* Mark Bits (MSB), (LSB) */
				SCidGen_g.SOnHookXmitInfoBlock.usOnHookXmitInfoType[4] =
							UT_CPU2BE16(ausParam[5]);
			}

			break;

		case eOffHookXmitInfoType:

			if (eAction == eRemove)
			{
				SCidGen_g.SOffHookXmitInfoBlock.bIsConfigured = 0;
				goto out;
			}

			if (usParamNum != 4)
			{
				UT_ErrorLog(APPITF, "VAPI_SetCidGenInfo: eOffHookXmitInfoType wrong num of param (%d)\n", usParamNum);
				Status = VAPI_ERR_INVALID_PARAM;
				goto out;
			}
			else
			{
				SCidGen_g.SOffHookXmitInfoBlock.bIsConfigured = 1;
				SCidGen_g.SOffHookXmitInfoBlock.usOffHookXmitInfoType[0] = 
							UT_CPU2LE16((OFF_HOOK_XMIT_INFO_TYPE_CNT << 8) | eOffHookXmitInfoType);

				/* FSK Tone Amplitude / Modulation Frequency Byte */
				SCidGen_g.SOffHookXmitInfoBlock.usOffHookXmitInfoType[1] = 
							UT_CPU2LE16((ausParam[1] << 8) | ausParam[0]);

				/* Alerting Info Byte / Mark Bits (MSB) */
				SCidGen_g.SOffHookXmitInfoBlock.usOffHookXmitInfoType[2] = 
							UT_CPU2LE16((ausParam[3] & 0xFF00) | ausParam[2]);

				/* Mark Bits (LSB) / Reserved */
				SCidGen_g.SOffHookXmitInfoBlock.usOffHookXmitInfoType[3] =
							UT_CPU2LE16(ausParam[3] & 0x00FF);

			}
			break;

		case eCPEAlertToneInfoType:

			if (eAction == eRemove)
			{
				SCidGen_g.SCPEAlertToneInfoBlock.bIsConfigured = 0;
				goto out;
			}

			if (usParamNum != 5)
			{
				UT_ErrorLog(APPITF, "VAPI_SetCidGenInfo: eCPEAlertToneInfoType wrong num of param (%d)\n", usParamNum);
				Status = VAPI_ERR_INVALID_PARAM;
				goto out;
			}
			else
			{
				SCidGen_g.SCPEAlertToneInfoBlock.bIsConfigured = 1;
				SCidGen_g.SCPEAlertToneInfoBlock.usCPEAlertToneInfoType[0] = 
							UT_CPU2LE16((CPE_ALERT_TONE_INFO_TYPE_CNT << 8) | eCPEAlertToneInfoType);

				/* Low Nominal Frequency in Hz (MSB), (LSB) */	
				SCidGen_g.SCPEAlertToneInfoBlock.usCPEAlertToneInfoType[1] = 
							UT_CPU2BE16(ausParam[0]);

				/* High Nominal Frequency in Hz (MSB), (LSB) */
				SCidGen_g.SCPEAlertToneInfoBlock.usCPEAlertToneInfoType[2] = 
							UT_CPU2BE16(ausParam[1]);

				/* Duration in milliseconds / Level Low Freq (MSB) */
				SCidGen_g.SCPEAlertToneInfoBlock.usCPEAlertToneInfoType[3] = 
							UT_CPU2LE16((ausParam[3] & 0xFF00) | ausParam[2]);

				/* Level Low Freq (LSB) / Level High Freq (MSB) */
				SCidGen_g.SCPEAlertToneInfoBlock.usCPEAlertToneInfoType[4] = 
							UT_CPU2LE16(((ausParam[4] & 0xFF00)) | (ausParam[3] & 0x00FF));

				/* Level High Freq (LSB) / Reserved */
				SCidGen_g.SCPEAlertToneInfoBlock.usCPEAlertToneInfoType[5] =
							UT_CPU2LE16(ausParam[4] & 0x00FF);

			}
			break;

		case eCPEAckInfoType:

			if (eAction == eRemove)
			{
				SCidGen_g.SCPEAckInfoBlock.bIsConfigured = 0;
				goto out;
			}

			if (usParamNum != 7)
			{
				UT_ErrorLog(APPITF, "VAPI_SetCidGenInfo: eCPEAckInfoType wrong num of param  (%d)\n", usParamNum);
				Status = VAPI_ERR_INVALID_PARAM;
				goto out;
			}
			else
			{
				SCidGen_g.SCPEAckInfoBlock.bIsConfigured = 1;
				SCidGen_g.SCPEAckInfoBlock.usCPEAckInfoType[0] = 
							UT_CPU2LE16((CPE_ACK_INFO_TYPE_CNT << 8) | eCPEAckInfoType);

				/* DTMF Digit #1 / DTMF Digit #2 */
				SCidGen_g.SCPEAckInfoBlock.usCPEAckInfoType[1] = 
							UT_CPU2LE16((ausParam[1] << 8) | ausParam[0]);

				/* Min Duration ACK / Timeout For ACK (MSB) */
				SCidGen_g.SCPEAckInfoBlock.usCPEAckInfoType[2] = 
							UT_CPU2LE16(((ausParam[3] & 0xFF00)) | ausParam[2]);

				/* Timeout For ACK (LSB) / Time to data transmission */
				SCidGen_g.SCPEAckInfoBlock.usCPEAckInfoType[3] = 
							UT_CPU2LE16((ausParam[4] << 8) | (ausParam[3] & 0x00FF));

				/* Transmit Data If NoACK / Ack Tone MinLevel (MSB) */
				SCidGen_g.SCPEAckInfoBlock.usCPEAckInfoType[4] = 
							UT_CPU2LE16(((ausParam[6] & 0xFF00)) | ausParam[5]);

				/* AckToneMinLevel (LSB) / Reserved */
				SCidGen_g.SCPEAckInfoBlock.usCPEAckInfoType[5] = 
							UT_CPU2LE16(ausParam[6] & 0x00FF);

			}
			break;

		case eSASToneInfoType:

			if (eAction == eRemove)
			{
				SCidGen_g.SSASToneInfoBlock.bIsConfigured = 0;
				goto out;
			}

			if ((usParamNum != 8) && (usParamNum != 14) && (usParamNum != 20) && (usParamNum != 26))
			{
				UT_ErrorLog(APPITF, "VAPI_SetCidGenInfo: eSASToneInfoType wrong num of param (%d)\n", usParamNum);
				Status = VAPI_ERR_INVALID_PARAM;
				goto out;
			}
			else
			{
				UT_MemSet(&SCidGen_g.SSASToneInfoBlock, 0x00, sizeof(SSASToneInfoType));

				SCidGen_g.SSASToneInfoBlock.bIsConfigured = 1;
				SCidGen_g.SSASToneInfoBlock.usSASToneInfoType[0] = 
							UT_CPU2LE16(((usParamNum + 9)* 2 << 8) | eSASToneInfoType);

				/*Tone control */
				SCidGen_g.SSASToneInfoBlock.usSASToneInfoType[1] = UT_CPU2LE16(ausParam[0]);

				/*the time stamps are set to 0 + 3 reserved U16 to 0*/
				i = 9;
				j = 1;

				/* Define dual tone 1, */
				for (n = 0; n < 6; n++, i++, j++)
				{
					SCidGen_g.SSASToneInfoBlock.usSASToneInfoType[i] = UT_CPU2LE16(ausParam[j]);
				}

				if (usParamNum > 8)
				{
					/* Define dual tone 2 (skip 2 reserved U16)*/
					i += 2;
					for (n = 0; n < 6; n++, i++, j++)
					{
						SCidGen_g.SSASToneInfoBlock.usSASToneInfoType[i] = UT_CPU2LE16(ausParam[j]);
					}
				}

				if (usParamNum > 14)
				{
					/* Define dual tone 2 (skip 2 reserved U16)*/
					i += 2;
					for (n = 0; n < 6; n++, i++, j++)
					{
						SCidGen_g.SSASToneInfoBlock.usSASToneInfoType[i] = UT_CPU2LE16(ausParam[j]);
					}
				}

				if (usParamNum > 20)
				{
					/* Define dual tone 3 (skip 2 reserved U16)*/
					i += 2;
					for (n = 0; n < 6; n++, i++, j++)
					{
						SCidGen_g.SSASToneInfoBlock.usSASToneInfoType[i] = UT_CPU2LE16(ausParam[j]);
					}
				}

				/* skip 2 reserved U16 */
				i += 2;
				/*last U16 is Transition Time to CAS*/
				SCidGen_g.SSASToneInfoBlock.usSASToneInfoType[i] = UT_CPU2LE16(ausParam[j]);

				/*the number of parameters is the current number of param + 9 fields set to 0*/
				SCidGen_g.SSASToneInfoBlock.ucBlockSize = (usParamNum + 9);

			}
			break;

		case eNTTInfoType:

			if (eAction == eRemove)
			{
				SCidGen_g.SNTTInfoBlock.bIsConfigured = 0;
				goto out;
			}

			if (usParamNum != 17)
			{
				UT_ErrorLog(APPITF, "VAPI_SetCidGenInfo: eNTTInfoType wrong num of param (%d)\n", usParamNum);
				Status = VAPI_ERR_INVALID_PARAM;
				goto out;
			}
			else
			{
				SCidGen_g.SNTTInfoBlock.bIsConfigured = 1;
				SCidGen_g.SNTTInfoBlock.usNTTInfoType[0] = 
							UT_CPU2LE16((NTT_INFO_TYPE_CNT << 8) | eNTTInfoType);

				/* NTTEnable / IITEnable */
				SCidGen_g.SNTTInfoBlock.usNTTInfoType[1] = 
							UT_CPU2LE16((ausParam[1] << 8) | ausParam[0]);

				/* IIT Tx Level (LSB), (MSB) */
				SCidGen_g.SNTTInfoBlock.usNTTInfoType[2] = 
							UT_CPU2LE16(ausParam[2]);

				/* IITSquelchperiod1 / IITtoCat_time_delay */
				SCidGen_g.SNTTInfoBlock.usNTTInfoType[3] = 
							UT_CPU2LE16((ausParam[4] << 8) | ausParam[3]);

				/* CATenable / CATdigit1 */
				SCidGen_g.SNTTInfoBlock.usNTTInfoType[4] = 
							UT_CPU2LE16((ausParam[6] << 8) | ausParam[5]);

				/* CATdigit2 / CATdigitontime (in ms) */
				SCidGen_g.SNTTInfoBlock.usNTTInfoType[5] = 
							UT_CPU2LE16((ausParam[8] << 8) | ausParam[7]);

				/* CATdigitofftime / CATdigit_High_TxLevel (LSB) */
				SCidGen_g.SNTTInfoBlock.usNTTInfoType[6] = 
							UT_CPU2LE16((ausParam[10] << 8) | ausParam[9]);

				/* CATdigit_High_TxLevel (MSB) / CATdigit_Low_TxLevel (LSB) */
				SCidGen_g.SNTTInfoBlock.usNTTInfoType[7] = 
							UT_CPU2LE16((ausParam[11] << 8) | (ausParam[10] >> 8));

				/* CATdigit_Low_TxLevel (MSB) / CATtoFSK_time_delay */
				SCidGen_g.SNTTInfoBlock.usNTTInfoType[8] = 
							UT_CPU2LE16((ausParam[12] << 8) | (ausParam[11] >> 8));

				/* FSK_to_IIT2_time_delay / IIT2Enable */
				SCidGen_g.SNTTInfoBlock.usNTTInfoType[9] = 
							UT_CPU2LE16((ausParam[14] << 8) | ausParam[13]);

				/* IIT2TxLevel (LSB), (MSB) */
				SCidGen_g.SNTTInfoBlock.usNTTInfoType[10] = 
							UT_CPU2LE16(ausParam[15]);

				/* IIT2SquelchPeriod1 / reserved */
				SCidGen_g.SNTTInfoBlock.usNTTInfoType[11] = 
							UT_CPU2LE16(ausParam[16] & 0x00FF);

			}

			break;

		case eStopBitsInfoType:

			if (eAction == eRemove)
			{
				SCidGen_g.SStopBitsInfoBlock.bIsConfigured = 0;
				goto out;
			}

			if (usParamNum != 1)
			{
				UT_ErrorLog(APPITF, "VAPI_SetCidGenInfo: eStopBitsInfoType wrong num of param (%d)\n", usParamNum);
				Status = VAPI_ERR_INVALID_PARAM;
				goto out;
			}
			else
			{
				SCidGen_g.SStopBitsInfoBlock.bIsConfigured = 1;
				SCidGen_g.SStopBitsInfoBlock.usStopBitsInfoType[0] = 
							UT_CPU2LE16((STOP_BITS_INFO_TYPE_CNT << 8) | eStopBitsInfoType);

				/*NumberofStopbits / Reserved */
				SCidGen_g.SStopBitsInfoBlock.usStopBitsInfoType[1] = 
							UT_CPU2LE16(ausParam[0] & 0x00ff);

			}
			break;

		case ePacketProcessingInfoType:

			if (eAction == eRemove)
			{
				SCidGen_g.SPacketProcessingInfoBlock.bIsConfigured = 0;
				goto out;
			}

			if (usParamNum != 1)
			{
				UT_ErrorLog(APPITF, "VAPI_SetCidGenInfo: ePacketProcessingInfoType wrong num of param (%d)\n", usParamNum);
				Status = VAPI_ERR_INVALID_PARAM;
				goto out;
			}
			else
			{
				SCidGen_g.SPacketProcessingInfoBlock.bIsConfigured = 1;
				SCidGen_g.SPacketProcessingInfoBlock.usPacketProcessingInfoType[0] = 
							UT_CPU2LE16((PACKET_PROCESS_INFO_TYPE_CNT << 8) | ePacketProcessingInfoType);

				/*IP Packet Encoding with Caller ID FSK Tone Generation / Reserved */
				SCidGen_g.SPacketProcessingInfoBlock.usPacketProcessingInfoType[1] = 
							UT_CPU2LE16(ausParam[0] & 0x00ff);

			}
			break;

		default:
			printf("VAPI_SetCidGenInfo: Unknow info block type (%d)\n", eCidGenInfoType);
			UT_ErrorLog(APPITF, "VAPI_SetCidGenInfo: Unknow info block type (%d)\n", eCidGenInfoType);
			Status = VAPI_ERR_INVALID_PARAM;
			goto out;

			break;

	}

out:
	return Status;

}

/***************************************************************************
 * VAPI_SetCidDetInfo The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This API sets the Caller ID information blocks used to detect a Caller ID.\n
*	With this API the user adds, modifies or removes Caller ID detection information blocks.\n
*	The Caller ID parameters are stored in a VAPI internal array.\n
*	The parameters are applied to a connection when the VAPI_SetCidDetection() is called.\n
*\n
*	The number of passed parameters must match the expected number for each different info blocks.\n
*	If the eAction = eRemove (0) the number of parameters is not taken it account.
*
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Block info type</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b>Number of expected parameters</b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DataFormatInfoType</td>
*		<td style="vertical-align: top;">2</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">FSKInfoType</td>
*		<td style="vertical-align: top;">3</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">OnHookDetInfoType</td>
*		<td style="vertical-align: top;">2</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">OffHookDetInfoType</td>
*		<td style="vertical-align: top;">1</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">AckDetInfoType</td>
*		<td style="vertical-align: top;">4</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">NTTDetInfoType</td>
*		<td style="vertical-align: top;">1</td>
*	</tr>
*	</table>
*	\n
*	This API can be called any time after VAPI_Init(), but it is convenient to initialize the caller ID during
*	the VAPI initialization phase (i.e just after VAPI_Init().
*
*	When VAPI_Init() is called, VAPI_SetCidDetInfo() is internally issued to configure a "default Caller ID detection".\n
*	Please see the default caller ID configuration below:
*
*	<pre>
*	DataFormatInfoType: {0, 0}
*			Where	0:		Data bits and parity 8 bits, no parity
*				0:		Checksum used for message integrity
*
*	FSKInfoType: {2, 0, 0}
*			Where	2:		FSK detection type Both B202 and V.23
*				0:		FSK Pull in threshold default
*				0:		FSK Pull out threshold default
*
*	OnHookDetInfoType: {192, 96}
*			Where	192:		Channel Seizure Threshold 192 bits
*				96:		Mark bits Threshold 96 bits
*
*	OffHookDetInfoType: {96}
*			Where	96:		Mark bits Threshold 96 bits
*
*
*	AckGenInfoType: {0x0D, 20, 60,112}
*			Where	0x0D:		DTMF Digit to generate
*				20:		Delay time from CPE Alert Detected 20ms
*				60:		Ack Tone Duration 60ms
*				112:		Ack Tone Tx level -11.2 dBoV
*
*	NTTDetInfoType: {0}
*			Where 0:		NTT Detector disabled
*
*	</pre>
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eAction </td>
*		<td style="vertical-align: top;">CID Info block to modify, add or remove.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eCidDetInfotype</td>
*		<td style="vertical-align: top;">Specifies information block type.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usParamNum</td>
*		<td style="vertical-align: top;">Number of parameters to be passed.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">...</td>
*		<td style="vertical-align: top;">List of parameters (integers of type int).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include set_cid_gen.c
*
*	\n \b Commands:
*	no commands sent
*
*/
VSTATUS VAPI_SetCidDetInfo(EAction eAction, ECidDetInfoType eCidDetInfoType, U16 usParamNum, ...)
{
	VSTATUS Status = SUCCESS;
	va_list ParamList;
	U16 ausParam[30];
	int i;

	if (!bVapiInitialized_g)
	{
		/*Used printf instead of UT_Log because UT is not yet initialized */
		printf("%s: VAPI library not initialized.\n", "VAPI_SetCidDetInfo");
		Status = VAPI_ERR_LIB_NOT_INITIALIZED;
		goto out;
	}

	if (eAction > eModify)
	{
		UT_ErrorLog(APPITF, "VAPI_SetCidDetInfo: eAction %d out of range\n", eAction);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	if (eCidDetInfoType > eNTTDetType)
	{
		UT_ErrorLog(APPITF, "VAPI_SetCidDetInfo: eAction %d out of range\n", eCidDetInfoType);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	UT_va_start(ParamList, usParamNum);
	for (i = 0; i < usParamNum; i++)
	{
		ausParam[i] = va_arg(ParamList, int);
	}
	UT_va_end(ParamList);

	switch (eCidDetInfoType)
	{

		case eDataFormatType:

			if (eAction == eRemove)
			{
				SCidDet_g.SDataFormatInfoBlock.bIsConfigured = 0;
				goto out;
			}

			if (usParamNum != 2)
			{
				UT_ErrorLog(APPITF, "VAPI_SetCidDetInfo: eDataFormatType wrong num of param (%d)\n", usParamNum);
				Status = VAPI_ERR_INVALID_PARAM;
				goto out;
			}
			else
			{
				SCidDet_g.SDataFormatInfoBlock.bIsConfigured = 1;
				SCidDet_g.SDataFormatInfoBlock.usDataFormatInfoType[0] =
					UT_CPU2LE16((DATA_FORMAT_INFO_TYPE_CNT << 8) | eDataFormatType);

				/*Data bits and Parity Selection CRC/Checksum Selection */
				SCidDet_g.SDataFormatInfoBlock.usDataFormatInfoType[1] =
							UT_CPU2LE16((ausParam[1] << 8) | ausParam[0]);

			}

			break;

		case eFSKDetectionType:

			if (eAction == eRemove)
			{
				SCidDet_g.SFSKDetectionInfoBlock.bIsConfigured = 0;
				goto out;
			}

			if (usParamNum != 3)
			{
				UT_ErrorLog(APPITF, "VAPI_SetCidDetInfo: eFSKDetectionType wrong num of param (%d)\n", usParamNum);
				Status = VAPI_ERR_INVALID_PARAM;
				goto out;
			}
			else
			{
				SCidDet_g.SFSKDetectionInfoBlock.bIsConfigured = 1;
				SCidDet_g.SFSKDetectionInfoBlock.usFSKDetectionInfoType[0] =
							UT_CPU2LE16((FSK_DETECTION_INFO_TYPE_CNT << 8) | eFSKDetectionType);

				/* FSK Detector Type / FSK Pull in threshold (MSB) */
				SCidDet_g.SFSKDetectionInfoBlock.usFSKDetectionInfoType[1] =
							UT_CPU2LE16((ausParam[1] & 0xFF00) | ausParam[0]);

				/* FSK Pull in threshold (LSB) / FSK Drop off threshold (MSB) */
				SCidDet_g.SFSKDetectionInfoBlock.usFSKDetectionInfoType[2] =
							UT_CPU2LE16((ausParam[2] & 0xFF00) | (ausParam[1] & 0x00FF));

				/* FSK Drop off threshold (LSB) / Reserved (must be zero) */
				SCidDet_g.SFSKDetectionInfoBlock.usFSKDetectionInfoType[3] =
							UT_CPU2LE16(ausParam[2] & 0x00FF);
			}
			break;

		case eOnHookDetType:

			if (eAction == eRemove)
			{
				SCidDet_g.SOnHookDetInfoBlock.bIsConfigured = 0;
				goto out;
			}

			if (usParamNum != 2)
			{
				UT_ErrorLog(APPITF, "VAPI_SetCidDetInfo: eOnHookDetType wrong num of param (%d)\n", usParamNum);
				Status = VAPI_ERR_INVALID_PARAM;
				goto out;
			}
			else
			{
				SCidDet_g.SOnHookDetInfoBlock.bIsConfigured = 1;

				SCidDet_g.SOnHookDetInfoBlock.usOnHookDetInfoType[0] =
							UT_CPU2LE16((ON_HOOK_DET_INFO_TYPE_CNT << 8) | eOnHookDetType);

				/* Channel Seizure Threshold (MSB), (LSB) */
				SCidDet_g.SOnHookDetInfoBlock.usOnHookDetInfoType[1] =
							UT_CPU2BE16(ausParam[0]);

				/* Mark Bits Threshold (MSB), (LSB) */
				SCidDet_g.SOnHookDetInfoBlock.usOnHookDetInfoType[2] =
							UT_CPU2BE16(ausParam[1]);

			}
			break;

		case eOffHookDetType:

			if (eAction == eRemove)
			{
				SCidDet_g.SOffHookDetInfoBlock.bIsConfigured = 0;
				goto out;
			}

			if (usParamNum != 1)
			{
				UT_ErrorLog(APPITF, "VAPI_SetCidDetInfo: eOffHookDetType wrong num of param  (%d)\n", usParamNum);
				Status = VAPI_ERR_INVALID_PARAM;
				goto out;
			}
			else
			{
				SCidDet_g.SOffHookDetInfoBlock.bIsConfigured = 1;
				SCidDet_g.SOffHookDetInfoBlock.usOffHookDetInfoType[0] =
							UT_CPU2LE16((OFF_HOOK_DET_INFO_TYPE_CNT << 8) | eOffHookDetType);

				/* Mark bits threshold (MSB), (LSB) */
				SCidDet_g.SOffHookDetInfoBlock.usOffHookDetInfoType[1] =
							UT_CPU2BE16(ausParam[1]);
			}
			break;

		case eAckGenType:

			if (eAction == eRemove)
			{
				SCidDet_g.SAckGenInfoBlock.bIsConfigured = 0;
				goto out;
			}

			if (usParamNum != 4)
			{
				UT_ErrorLog(APPITF, "VAPI_SetCidDetInfo: eAckGenType wrong num of param (%d)\n", usParamNum);
				Status = VAPI_ERR_INVALID_PARAM;
				goto out;
			}
			else
			{
				SCidDet_g.SAckGenInfoBlock.bIsConfigured = 1;
				SCidDet_g.SAckGenInfoBlock.usAckGenInfoType[0] =
							UT_CPU2LE16((ACK_GEN_INFO_TYPE_CNT << 8) | eAckGenType);

				/* DTMF Digit to generate / Delay time from CPE Alert Detected */
				SCidDet_g.SAckGenInfoBlock.usAckGenInfoType[1] = 
							UT_CPU2LE16((ausParam[1] << 8) | ausParam[0]);

				/* Ack Tone Duration Ack Tone Tx Level (MSB) */
				SCidDet_g.SAckGenInfoBlock.usAckGenInfoType[2] = 
							UT_CPU2LE16((ausParam[3] & 0xFF00) | ausParam[2]);

				/* Ack Tone Tx Level (LSB) / Reserved must be set to zero */
				SCidDet_g.SAckGenInfoBlock.usAckGenInfoType[3] = 
							UT_CPU2LE16(ausParam[3] & 0x00FF);

			}

			break;

		case eNTTDetType:

			if (eAction == eRemove)
			{
				SCidDet_g.SNTTDetInfoBlock.bIsConfigured = 0;
				goto out;
			}

			if (usParamNum != 1)
			{
				UT_ErrorLog(APPITF, "VAPI_SetCidDetInfo: eNTTDetType wrong num of param (%d)\n", usParamNum);
				Status = VAPI_ERR_INVALID_PARAM;
				goto out;
			}
			else
			{
				SCidDet_g.SNTTDetInfoBlock.bIsConfigured = 1;
				SCidDet_g.SNTTDetInfoBlock.usNTTDetInfoType[0] =
							UT_CPU2LE16((NTT_DET_INFO_TYPE_CNT << 8) | eNTTDetType);

				/* NTT Detector Enable/Disable */
				SCidDet_g.SNTTDetInfoBlock.usNTTDetInfoType[1] =
							UT_CPU2LE16(ausParam[0] & 0x00FF);
			}

			break;

		default:
			printf("VAPI_SetCidDetInfo: Unknow info block type (%d)\n", eCidDetInfoType);
			UT_ErrorLog(APPITF, "VAPI_SetCidDetInfo: Unknow info block type (%d)\n", eCidDetInfoType);
			Status = VAPI_ERR_INVALID_PARAM;
			goto out;

			break;

	}

out:
	return Status;

}

/** @} */ /*end of helper group*/
