/*! \file dmgr.h 
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#ifndef _DMGR_H
#define _DMGR_H

#include "cfg.h"
#include "ut.h"
#include "gtl.h"
#include "vcore_voip.h"
#include "vapi.h"
#include "vcore_types.h"
#include "resmgr.h"

#define SPU_CTRL_NOT_SENT    0
#define SPU_CTRL_SENT        1
#define SPU_CTRL_SENT_ERR    2

#define SPU_FEATURE_ENHEC_MASK		0x0001 /*bit 0*/
#define SPU_FEATURE_SVDET_MASK		0x0002 /*bit 1*/
#define SPU_FEATURE_FSKRX_MASK		0x0004 /*bit 2*/
#define SPU_FEATURE_G723_ANNEX_A_MASK	0x0008 /*bit 3*/
#define SPU_FEATURE_CPDET_MASK		0x0010 /*bit 4*/
#define SPU_FEATURE_IP_DTMF_MASK	0x0020 /*bit 5*/
#define SPU_FEATURE_IP_TONEDET_MASK	0x0040 /*bit 6*/
#define SPU_FEATURE_V18_MASK		0x0080 /*bit 7*/
#define SPU_FEATURE_AGC_MASK		0x0100 /*bit 8*/
#define SPU_FEATURE_SNS_MASK		0x0200 /*bit 9*/
#define SPU_FEATURE_GT_DET_MASK		0x0400 /*bit 10*/
#define SPU_FEATURE_AEC_MASK		0x0800 /*bit 11*/
#define SPU_FEATURE_DFEC_MASK		0x1000 /*bit 12*/

#define SPU_CTRL_FIRMWARE_DEFAULT_MASK	0x0000
#define SPU_CTRL_DEFAULT_MASK 		SPU_FEATURE_FSKRX_MASK 
#define SPU_CTRL_DFECAN_MASK		SPU_FEATURE_FSKRX_MASK | SPU_FEATURE_DFEC_MASK

typedef U16 CHNLID;

typedef enum {
	eMODE_NONE = 0,
	eMODE_IP = 1,
	eMODE_ALT = 2
} EEthHdrMode;

typedef enum {
	eNO_SRC_IP = 0,
	eMULTIPLE_IP = 1,
	eSINGLE_IP = 2
} ESrcIpMode;

typedef enum {
	eVop = 0,
	eRec = 1,
	ePlay = 2,
	ePlayExt = 3,
	ePlaySpeedChangeable = 4
} ERecPlayState;

/** Information of a single conference */
typedef struct _SConference {
	CONFID ConfId;
	U16 usMSPConfId;
	struct _SDevice *pstDev;
		    /**< Device to which this Conference belongs to*/
	struct _SChnl **apstParticipant;
		    /**< Channels indexed on MSPChnlId*/
	U16 usNumOfParticipant;
	SConfParams *pstConfParams;
	Boolean bIsEnhancedConf;
	Boolean bIsTranscoding;
	SConfDTSParams *pstConfDTSParams;
} SConference;

/** Information of the Rec and Play Status on channel */
typedef struct _SRecPlayInfo {
	ERecPlayState eRecPlayState;
	U32 uiStopRecCount;
	U32 uiStopPlayCount;
	U32 uiStartRecCount;
	U32 uiStartPlayCount;
	EStopAnnType eStopAnnType;
	EPlaybackRate ePlaybackRate;
	U8 uiChangeRate;
	SMutex stSemRecPlay;
} SRecPlayInfo;

typedef struct _SSetEthMac {
	U8 astSrcMacAddr[6];
	U8 astDestMacAddr[6];
	U16 usVlanId;
} SSetEthMAc;

/** Information of a single channel*/
typedef struct _SChnl {
	CONNID ConnId;			/**< Unique Connection identifier for the 
					channel. It is specified by the user 
					application at the time of channel creation*/

	struct _SDevice *pstDev;	/**< Device to which this channel belongs to*/
	PFNEventCallback pfnEventCallback;
	U16 usConnType;			/**< Channel type VOIP, AAL2 etc*/
	U16 usMSPChnlId;		/**< Channel Id returned by MSP*/
	U16 usTimeSlot;			/**< TDM Timeslot for this channel*/
	U16 usExtTimeSlot[3];		/**< TDM extra Timeslot for this channel (WB mode)*/
	U16 usMode;			/**< Channel mode narrow or wide band */
	U16 MediaSubLayer;		/**< Channel media sub-layer */
	SVoIPChnlParams *pstChnlParams;	/**< Channel configuration params. Structure depends on Connection type.*/
	SLinkedList stReqList;		/**< List of requests for this channel*/

	U8 *pstH245MsgIndParam;		/**< Pointer to H.245 message indication param*/


	SMutex stSemMsgIndex;		/**< Semaphore to access message index*/

	STimer *pstRespTimer;		/**< Response timer*/
	U8 ucMsgIndex;			/**< Msg Index of the last command sent on this channel*/
	U32 uiDestIp;			/**< Destination IP adress*/
	U16 usDestUdp;			/**< Destination UDP Port*/
	U16 usSrcUdp;			/**< Source UDP Port*/

	SSetEthMAc stEthMacAddresses;	/**< Source and dest mac addres*/
	
	/* conference data */
	EPartType ePartType;
	SConference *pstConf;
	/*Rec Or Play State */
	SRecPlayInfo stRecPlayInfo;
	SMutex stAccessFsmHndlr;

	Boolean bIPHdrSet:1;	    /** Checked during enableConnection (for IP mode only)*/
	Boolean bAltHdrSet:1;	    /** Checked during enableConnection (for ALT mode only)*/
	Boolean bIsActive:1;	    /** reflect the connection state active or not (VOPENA)*/
	Boolean bEnableFaxAutoSwitch:1; /** reflect the user option to use the Fax Auto switch*/

	U16 usSwitchoverEventMask;	/**Bit map to identify events on which switchover is to be performed */

	U16 FsmEvent;
} SChnl;

/** Device State*/
typedef enum {
	eUNINITIALIZED,		/**< The device is not yet initialized*/
	eREADYFORDUMP,		/**< Device is prepared to facilitate Core Dump*/
	eBOOTING,		/**< Device is currently booting up*/
	eUP,			/**< Device is up for accepting commands*/
	eCLOSING,		/**< Device is in the process of closing*/
	eMAAS			/**< Device is in the process of Maas assigning */
} EDevState;

/** Device and its channel related information*/
struct _SDevice {
	DEVID DevId;			/**< Unique device ID*/
	EDevItfType eItfType;		/**< Interface Type (Ethernet/PCI)*/
	GtlInterface *pvItfInfo;	/**< Interface dependent info*/
	PFNEventCallback pfnEventCallback;
	EDevState eState;		/**< State of device (Down/Up/Booting)*/
	EDevMode eDevMode;		/**< DEvice Mode Master/Slave */
	SDevMemInfo stMemInfo;		/**< Device memory info*/

	SArrayNode stArrayNode;

	U32 uMaxFrameSize;		/**< Max. bytes in a frame */
	U16 usMaxChnls;			/**< Max. chnls in the device */
	SChnl **papstChannels;		/**< Channels indexed on MSPChnlId*/
	SChnl stSupvChnl;		/**< Supervisory Channel structure*/
	U32 uiSrcIpAddr;		/**< Ip Addr*/

	SConference **apstConference;	/**< Conference data */
	U8 ucSPUSentStatus; 		/**< Status of device SPU_FEATURE_CONTROL Sent to  the Supervisory channel */
					/**< SPU_CTRL_TO_SEND(0) = SPU_FEATURE_CONTROL to be sent */
					/**< SPU_CTRL_SENT(1) = SPU_FEATURE_CONTROL sent success */
					/**< SPU_CTRL_SENT_ERR(2) = SPU_FEATURE_CONTROL sent failure */

	U8 ucDevType;			/**< Device type */
	U16 usDevModel;			/**< Device model updated by VAPI_InitDevice */
	U8 bMultiCmdEnabled;		/**< True if MultiCommand is enabled*/
	EEthHdrMode eEthHdrMode;	/**< Alternate Hdr or IP Hdr or None*/
	ESrcIpMode eSrcIpMode;		/**< Single IP mode or multiple IP mode*/
	SSetEthMAc stEthMacAddresses;	/**< Source and dest mac addresses and vlanid*/
	U32 usSpuFeatureMask;		/**< Variable to store the SPU_Feature_Control value sent by the user using passthru */
	U16 MajorRelease;		/**< Major Release number of the Comcerto firmware for this device*/
	PFNResetDevice pfnReset;	/**< Device reset function to be registered using VAPI_RegisterReset*/
	void *pvResetData;		/**< Registered data for the reset function*/
	U8 ucEthBootMode;		/**< Boot mode (Backward comaptible, standard or alt for M823xx device*/
	U16 MaxMsgSize;			/**< Max message size supported by the device firmware & the control itf*/

	Boolean bIsDeviceInitialized:1;	/**< True if the device correctly initialized */
	Boolean bEthHdrSet:1;		/**< Is Ethernet Hdr set for Device?*/
	Boolean bIsReadyReceived:1;	/**< Indicate if a READY frame has been received (M823xx only)*/

	SDevResources stDevRes;		/**< Structure to handle resource for this device */
};

typedef struct _SDevice SDevice;

/*Global Function pointer for configuration*/
extern PFNUSRQueryVAPIDefaultsEnumerate pfnUsrQueryVapiDefaultsEnumerate;

/*Interface Functions*/
/** Initialize device manager*/
VSTATUS DMGR_Init(void);

/** Add a device into the device list*/
VSTATUS DMGR_AddDevice(IN SDevice * pstDevice);

/** Remove a device from device list*/
void DMGR_RemoveDevice(IN DEVID DevId);

/** Remove all pending conferences/connections of this device*/
void DMGR_PurgeDevice(IN SDevice * pstDevice);

/** Search a device based on its device ID*/
SDevice *DMGR_GetDevice(IN DEVID DevId);

/** Remove any devices structure*/
void DMGR_CloseAllDevices(void);

/** Search a channel based on device ID and channel ID*/
SChnl *DMGR_GetChannel(SDevice *pstDevice, IN U16 usChnlId);

/** Search a channel based on its connection ID*/
SChnl *DMGR_GetConnection(IN CONNID ConnId);

/** Add a new channel in the channel list of specified device */
VSTATUS DMGR_AddChannel(IN SDevice * pstDevice, SChnl *pstChnl);

/** Remove a specified channel from the channel list of specified device*/
VSTATUS DMGR_RemoveChannel(IN SDevice * pstDevice, IN CHNLID MSPChnlID);

/**Moves the channel structure to new timeslot.Connection Id is retained.*/
VSTATUS DMGR_MoveChnl(IN SDevice * pstDevice, IN CHNLID oldChnlId, IN CHNLID newChnlId);

/** Deallocate Device Manager resources*/
void DMGR_Close(void);

void DMGR_SetVAPIDefaults(SVAPIConfig * pstVAPIConfig);

/** Set Default values of VOIP params (implemented in voip_defaults.c) */
void DMGR_SetVoIPDefaults(SVoIPChnlParams * pstChnlParams);
/** Set the internal VAPI variable to same as default device values when a channel is created */
void DMGR_SetVoIPDevDefaults(SVoIPChnlParams * pstChnlParams);
/** Set Default values of conference params (implemented in voip_defaults.c) */
void DMGR_SetConfDefaults(SConferenceParams *pstConfParams);


/** Set the device type specific information*/
void DMGR_SetVAPIDeviceConfig(SVAPIDeviceConfig * pastDeviceConfig);

/** Get the default configuration params of a channel*/
void *DMGR_GetChnlDefaults(IN U16 usConnType);

/** Get the default conferencing configuration params*/
SConferenceParams *DMGR_GetConfDefaults(void);

/** Get the default country Information */
SCountryInfo *DMGR_GetCountryDefaults(IN ECountryCode eCountryCode);

/** Get the device type specific information*/
SVAPIDeviceConfig *DMGR_GetVAPIDeviceConfig(IN U8 ucDevType);

/** Search a conference based on device ID and MSP conference number*/
SConference *DMGR_GetConfByNo(IN DEVID DevId, IN U16 usConflNo);

/** Search a conference based on its conference ID*/
SConference *DMGR_GetConfById(IN CONFID ConfId);

/** Add a new conference in the conference list of specified device */
SConference *DMGR_AddConf(IN SDevice * pstDevice, IN U16 usConfNo, IN CONFID ConfId);

/** Remove a specified conference from the conference list of specified device*/
void DMGR_RemoveConf(IN SDevice * pstDevice, IN U16 usConfNo);

/** Search a conference based on device ID and MSP conference number*/
SConference *DMGR_GetConfByNo(IN DEVID DevId, IN U16 usConflNo);

/** Search a conference based on its conference ID*/
SConference *DMGR_GetConfById(IN CONFID ConfId);

/**  Stores it at an appropriate location in the conference array that corresponds to the device */
VSTATUS DMGR_AddConference(IN SDevice * pstDevice, SConference *pstConf);

VSTATUS DMGR_AddParticipant(SChnl *pstChnl);

/** Default tone init*/
VSTATUS DMGR_ToneInit(void);

/** Default Called ID init */
VSTATUS CidInitDefault(void);
#endif

