/*! \file dmgr.c 
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */


#include "ut.h"
#include "vapi.h"
#include "dmgr.h"
#include "vcore_voip.h"
#include "vcore.h"
#include "msp.h"
#include "appitf.h"


/** Semaphore to access the device array*/
SMutex stSemDevArr_g;

/** Pointer to Sorted array of devices. This array will contain the pointers
to actual device structures that are present in astDevices*/
SSortArray *pstDeviceArr_g;

/** An avl tree that contains Channel structures and nodes are arranged as per
Connection IDs. The pointer to the channel structure point to the Actual 
structures present in astDevices array */
SAVLTree *pstConnTree_g;

/** An avl tree that contains Channel structures and nodes are arranged as per
Connection IDs. The pointer to the channel structure point to the Actual 
structures present in astDevices array */
SAVLTree *pstConfTree_g;

/** An avl tree that contains resource structures and nodes are arranged as per Connection IDs. */
SAVLTree *pstResTree_g;

/** Hold device type (Chagall,Picaso, Miro etc) specific info*/
SVAPIDeviceConfig astDeviceConfig_g[NUM_DEV_TYPES];

/** Default parameters for a VoIP channel*/
extern SVoIPChnlParams stVoipDefaultParams_g;

extern SVAPIConfig stVAPIConfig_g;
extern SCountryInfo astCountryInfoList_g[];
extern SConferenceParams stConfParams_g;

extern PFNUSRQueryVAPIDefaultsEnumerate pfnUsrQueryVapiDefaultsEnumerate;

/****************************************************************************
 * DMGR_Init : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Initialize Soreted array for devices and creates and AVL tree for 
 *          channels
 *
 *  \return 
 *  SUCCESS Everything is initialized successfully\n
 *  VAPI_ERR_RSRC_INIT_FAILED Some resource initialization failed.
 *  
 */
VSTATUS DMGR_Init()
{
	VSTATUS Status;

	pstConnTree_g = UT_CreateAVLTree();
	if (pstConnTree_g == NULL)
	{
		Status = VAPI_ERR_RSRC_INIT_FAILED;
		goto err0;
	}

	pstConfTree_g = UT_CreateAVLTree();
	if (pstConfTree_g == NULL)
	{
		Status = VAPI_ERR_RSRC_INIT_FAILED;
		goto err1;
	}

	if (UT_MutexInit(&stSemDevArr_g) < 0)
	{
		Status = VAPI_ERR_RSRC_INIT_FAILED;
		goto err2;
	}

	DMGR_SetVAPIDefaults(&stVAPIConfig_g);

	if (pfnUsrQueryVapiDefaultsEnumerate != NULL)
	{
		pfnUsrQueryVapiDefaultsEnumerate(&stVAPIConfig_g);
	}

	DMGR_SetVAPIDeviceConfig(astDeviceConfig_g);


	pstDeviceArr_g = UT_CreateSortedArray(stVAPIConfig_g.uiMaxDevices);
	if (pstDeviceArr_g == NULL)
	{
		Status = VAPI_ERR_RSRC_INIT_FAILED;
		goto err3;
	}

	return SUCCESS;

      err3:
	UT_MutexDestroy(&stSemDevArr_g);

      err2:
	UT_DeleteAVLTree(pstConfTree_g);

      err1:
	UT_DeleteAVLTree(pstConnTree_g);

      err0:
	return Status;
}

/****************************************************************************
 * DMGR_AddDevice : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Add a device to the global device array (copy all data) and add 
 *          its entry in sorted array too.
 *
 *  \return 
 *  SUCCESS Device added successfully\n
 *  VAPI_ERR_DEVICE_EXISTS Device was already present. \n
 *
 *  \param pstDevice Device that needs to be added
 */
VSTATUS DMGR_AddDevice(IN SDevice *pstDevice)
{

	/*Add a node in the sorted array of devices */
	pstDevice->stArrayNode.uiKey = pstDevice->DevId;

	UT_AddElement(pstDeviceArr_g, &pstDevice->stArrayNode);

	return SUCCESS;
}

/****************************************************************************
 * DMGR_CloseAllDevices : The function does the following things -
 ***************************************************************************/
/*!
*/

void DMGR_CloseAllDevices(void)
{
	S32 iIndex;
	S32 iStrtIndx = 0;
	SDevice *pstDevice;
	SArrayNode *pstArrNode;

	UT_Log(VCORE, INFO, "DMGR_CloseAllDevices: Entry\n");

	/*parse all the sorted array of devices */
	for (iIndex = iStrtIndx; iIndex < stVAPIConfig_g.uiMaxDevices; iIndex++)
	{
		pstArrNode = pstDeviceArr_g->pastStart[iIndex];

		if (pstArrNode != NULL)
		{
			/*If there is a device for this node, remove it*/
			UT_MutexLock(&stSemDevArr_g);
			pstDevice = UT_ContainerOf(pstArrNode, SDevice, stArrayNode);
			UT_MutexUnLock(&stSemDevArr_g);

			if (pstDevice != NULL)
			{
				UT_Log(VCORE, INFO, "DMGR_CloseAllDevices: Closing device %u\n", pstDevice->DevId);
				APPITF_CloseDevice(pstDevice, ePURGE);
			}
		}
	}

	UT_Log(VCORE, INFO, "DMGR_CloseAllDevices: Exit\n");

}

/****************************************************************************
 * DMGR_RemoveDevice : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Find the device in the devices array and remove it.
 *      -#  Set the device's allcoated status to False and Remove the node from
 *      -#  This function does not destroy all the channels of this device.
 *      -#  [TBD] Shopuld it deallocate all the channels.
 *
 *  \return 
 *      None
 *
 *  \param DevId The device id if the device to be removed
 */
void DMGR_RemoveDevice(IN DEVID DevId)
{
	SArrayNode *pstArrayNode;
	SDevice *pstDevice;

	pstArrayNode = UT_SearchElement(pstDeviceArr_g, DevId);
	if (pstArrayNode == NULL)
	{
		UT_ErrorLog(VCORE, "DMGR_RemoveDevice: Dev does not exist\n");
		goto out;
	}
	UT_MutexLock(&stSemDevArr_g);
	pstDevice = UT_ContainerOf(pstArrayNode, SDevice, stArrayNode);

	UT_MutexUnLock(&stSemDevArr_g);

	UT_RemElement(pstDeviceArr_g, pstArrayNode);

out:
	return;
}

/****************************************************************************
 * DMGR_GetDevice : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Search for the device id in the sorted array and return the pointer
 *  
 *  \return 
 *  Returns NULL when the device is not found\n
 *  Returns the pointer to the SDevice node, that corresponds to the given
 *  Device id\n
 *
 *  \param DevId The device to be searched
 */
SDevice *DMGR_GetDevice(IN DEVID DevId)
{
	SArrayNode *pstArrayNode;

	pstArrayNode = UT_SearchElement(pstDeviceArr_g, DevId);
	if (pstArrayNode == NULL)
		return NULL;

	return UT_ContainerOf(pstArrayNode, SDevice, stArrayNode);
}

/****************************************************************************
 * DMGR_GetChannel : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation 
 *      -   Get the channel by directly indexing the SDevice::papstChannels
 *
 *  \return 
 *  Returns NULL when either the device or channel are not found.\n
 *
 *  \param pstDevice The device to which this channel belongs
 *  \param usChnlId The MSP channel id of the channel that is to be returned
 */
SChnl *DMGR_GetChannel(SDevice *pstDevice, IN U16 usChnlId)
{
	SChnl *pstChnl;

	if (usChnlId == SUPV_CHANNEL)
	{
		pstChnl = &pstDevice->stSupvChnl;
	}
	else
	{
		if (usChnlId >= pstDevice->usMaxChnls)
			goto err;

		pstChnl = pstDevice->papstChannels[usChnlId];
	}

	return pstChnl;

      err:
	return NULL;
}

/****************************************************************************
 * DMGR_GetConnection : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Search the AVL tree for the given conection id and return the 
 *          corresponding channel structure.\n
 *
 *  \return 
 *  Returns NULL if the specified ConnId is not found.\n
 *
 *  \param ConnId Connection id
 */
SChnl *DMGR_GetConnection(IN CONNID ConnId)
{
	return (SChnl *) UT_SearchInAVLTree(pstConnTree_g, ConnId);
}

/****************************************************************************
 * DMGR_AddChannel : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Stores it at an appropriate location in the channel array that 
 *          corresponds to the device. 
 *
 *  \return 
 *  SUCCESS/ FAILURE \n
 *
 *  \param pstDevice Device which channel will be added.
 *  \param pstChnl Connection structure handling the device channel.
 */
VSTATUS DMGR_AddChannel(IN SDevice * pstDevice, SChnl *pstChnl)
{

	if (pstChnl->usMSPChnlId >= pstDevice->usMaxChnls)
	{
		UT_ErrorLog(VCORE, "DMGR_AddChannel: Invalid channel id(%u)\n",pstChnl->usMSPChnlId);
		return VAPI_ERR_INVALID_PARAM;
	}

	/*If the channel already exists */
	if (pstDevice->papstChannels[pstChnl->usMSPChnlId])
	{
		UT_ErrorLog(VCORE, "DMGR_AddChannel: Channel(%u) already exists on dev(%u)\n", 
				pstChnl->usMSPChnlId, pstDevice->DevId);
		return VAPI_ERR_INVALID_PARAM;
	}
	/*Add the channel in the device array */
	pstDevice->papstChannels[pstChnl->usMSPChnlId] = pstChnl;

	return SUCCESS;
}

/****************************************************************************
 * DMGR_RemoveChannel : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Remove the entry from the Device array \n
 *
 *  \return 
 *  SUCCESS \n
 *
 *  \param pstDevice The device from which this channel hast to be removed.
 *  \param ChnlId The channel id of the channel to be removed.
 */
VSTATUS DMGR_RemoveChannel(IN SDevice * pstDevice, IN CHNLID MSPChnlID)
{

	if (MSPChnlID >= pstDevice->usMaxChnls)
	{
		UT_ErrorLog(VCORE, "DMGR_RemoveChannel: Invalid channel id(%u)\n",MSPChnlID);
		return VAPI_ERR_INVALID_PARAM;
	}

	if (pstDevice->papstChannels[MSPChnlID] == NULL)
	{
		UT_ErrorLog(VCORE, "DMGR_RemoveChannel: Already removed id(%u)\n",MSPChnlID);
		return VAPI_ERR_INVALID_PARAM;
	}

	pstDevice->papstChannels[MSPChnlID] = NULL;

	return SUCCESS;
}

/****************************************************************************
 * DMGR_GetChnlDefaults : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Return channel defaults from the global stVoipDefaultParams_g\n
 *
 *  \return 
 *  Return the appropriate default parms for the specified channel type\n
 *
 *  \param eConnType Type of channel.
 */
void *DMGR_GetChnlDefaults(IN U16 usConnType)
{
	if (usConnType == eVOIP || usConnType == eFOIP || usConnType == eIMSOTDM)
	{
		return stVAPIConfig_g.pstVoIPChnlParams;
	}

	return NULL;
}

/****************************************************************************
 * DMGR_Close : The function does the following things -
 ***************************************************************************/
/*! 
 *  \return None
 */
void DMGR_Close()
{

	if (NULL != pfnUsrDefaultsClose)
	{
		if (SUCCESS != pfnUsrDefaultsClose())
		{
			UT_Log(UT, WARN, " PFNUSRQueryVAPIDefaultsClose returned error\n");
		}
	}
	UT_DeleteSortedArray(pstDeviceArr_g);
	UT_DeleteAVLTree(pstConnTree_g);
	UT_DeleteAVLTree(pstConfTree_g);
	UT_MutexDestroy(&stSemDevArr_g);
}

/****************************************************************************
 * DMGR_GetCountryDefaults : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Return Country Specific defaults info from the global stCountryInfoList\n
 *
 *  \return 
 *  Return the appropriate default Info for the specified Country\n
 *
 *  \param eCountryCode Code of Country.
 */
SCountryInfo *DMGR_GetCountryDefaults(IN ECountryCode eCountryCode)
{

	if (eCountryCode < stVAPIConfig_g.uiNumCountries)
		return &(stVAPIConfig_g.pastCountryInfo[eCountryCode]);
	else
		return NULL;
}

/****************************************************************************
 * DMGR_GetConfByNo : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation 
 *      -   Find the device based on device id from the sorted array.\n
 *      -   Then get the channel by directly indexing the SDevice::apstConference
 *
 *  \return 
 *  Returns NULL when either the device or conference are not found.\n
 *
 *  \param DevId The device to which this channel belongs
 *  \param usConflId The MSP conference number of the channel that is to be returned
 */
SConference *DMGR_GetConfByNo(IN DEVID DevId, IN U16 usConflNo)
{
	SArrayNode *pstArrayNode;
	SDevice *pstDevice;

	if (usConflNo >= CFG_MAX_CONF_PER_DEV)
		goto err;

	pstArrayNode = UT_SearchElement(pstDeviceArr_g, DevId);
	if (pstArrayNode == NULL)
		goto err;

	pstDevice = UT_ContainerOf(pstArrayNode, SDevice, stArrayNode);

	return pstDevice->apstConference[usConflNo];

      err:
	return NULL;
}

/****************************************************************************
 * DMGR_GetConfById : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Search the AVL tree for the given conference id and return the 
 *          corresponding conference structure.\n
 *
 *  \return 
 *  Returns NULL if the specified ConfId is not found.\n
 *
 *  \param ConnId Connection id
 */
SConference *DMGR_GetConfById(IN CONFID ConfId)
{
	return (SConference *) UT_SearchInAVLTree(pstConfTree_g, ConfId);
}

/****************************************************************************
 * DMGR_AddConf : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Allocates a new SConference Structure 
 *      -   Stores it at an appropriate location in the conference array that 
 *          corresponds to the device. 
 *      -   Makes an entry in the AVL tree that is based on the conference
 *          ids.
 *
 *  \return 
 *  Return the newly allocated conference\n
 *
 *  \param pstDevice Device which channel will be added.
 *  \param usConfNo The MSP conference No.
 *  \param ConfId User supplied Connection Id
 */
SConference *DMGR_AddConf(IN SDevice * pstDevice, IN U16 usConfNo, IN CONFID ConfId)
{
	SConference *pstConf = NULL;

	if (pstDevice == NULL)
	{
		return NULL;
	}

	if (usConfNo >= CFG_MAX_CONF_PER_DEV)
	{
		UT_ErrorLog(VCORE, "DMGR_AddConf: Could not add, Invalid conference id\n");
		return NULL;
	}

	/*If the conference already exists */
	if (pstDevice->apstConference[usConfNo])
	{
		UT_ErrorLog(VCORE, "Conference %u already exists on dev. %u\n", usConfNo, pstDevice->DevId);

		return NULL;
	}

	if ((pstConf = UT_AllocMem(sizeof(SConference))) == NULL)
	{
		UT_ErrorLog(VCORE, "DMGR_AddConf: Insufficient memory \n");
		return NULL;
	}
	UT_MemSet(pstConf, 0, sizeof(SConference));

	/*Add the conference in the device array */
	pstDevice->apstConference[usConfNo] = pstConf;
	pstConf->ConfId = ConfId;
	/*Add the conference in the AVL tree */
	if (UT_AddToAVLTree(pstConfTree_g, ConfId, pstConf) < 0)
	{
		pstDevice->apstConference[usConfNo] = NULL;
		UT_FreeMem(pstConf);
		UT_ErrorLog(VCORE, "Could not add conference, Invalid ConfId");
		return NULL;
	}

	return pstConf;
}

/****************************************************************************
 * DMGR_RemoveConf : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Deallocate the memory allocated to the conference and remove its 
 *          entries from the Device array and the AVL tree.\n
 *
 *  \return 
 *  None\n
 *
 *  \param pstDevice The device from which this confernece has to be removed.
 *  \param usConfNo The conference id of the conference to be removed.
 */
void DMGR_RemoveConf(IN SDevice * pstDevice, IN U16 usConfNo)
{
	if (pstDevice == NULL)
	{
		return;
	}
	if (usConfNo >= CFG_MAX_CONF_PER_DEV)
	{
		return;
	}
	if (pstDevice->apstConference[usConfNo] == NULL)
	{
		return;
	}

	UT_RemFromAVLTree(pstConfTree_g, pstDevice->apstConference[usConfNo]->ConfId);
	UT_FreeMem(pstDevice->apstConference[usConfNo]->apstParticipant);
	UT_FreeMem(pstDevice->apstConference[usConfNo]);
	pstDevice->apstConference[usConfNo] = NULL;
	return;
}

/****************************************************************************
 * DMGR_GetConfDefaults : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Return default parametrs of Conference\n
 *
 *  \return 
 *  Return the appropriate default Info for conference\n
 *
 *  
 */
SConferenceParams *DMGR_GetConfDefaults(void)
{
	return stVAPIConfig_g.pstConfParams;
}

/****************************************************************************
 * DMGR_GetVAPIDeviceConfig : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Return device type specific configuration infoirmation\n
 *
 *  \return 
 *  Return the appropriate device config structure\n
 *
 *  
 */
SVAPIDeviceConfig *DMGR_GetVAPIDeviceConfig(IN U8 ucDevType)
{
	if (ucDevType >= NUM_DEV_TYPES)
		return NULL;

	return &(astDeviceConfig_g[(U32) ucDevType]);
}

/****************************************************************************
 * DMGR_MoveChnl : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Moves the channel structure to new timeslot.
 *          Connection Id is retained.
 *
 *  \return 
 *  VSTATUS 
 *
 *  \param pstDevice Device which channel will be added.
 *  \param oldChnlId The old MSP channel ID.
 *  \param newChnlId The new old MSP channel ID.
 */
VSTATUS DMGR_MoveChnl(IN SDevice * pstDevice, IN CHNLID oldChnlId, IN CHNLID newChnlId)
{
	if (pstDevice == NULL)
	{
		return FAILURE;
	}

	if (newChnlId >= pstDevice->usMaxChnls || oldChnlId >= pstDevice->usMaxChnls)
	{
		UT_ErrorLog(VCORE, "DMGR_MoveChnl: Could not move, Invalid channel id\n");
		return FAILURE;
	}

	if (oldChnlId == newChnlId)
		return SUCCESS;
	/*If the channel does not exist */
	if ((pstDevice->papstChannels)[oldChnlId] == NULL)
	{
		UT_ErrorLog(VCORE, "Channel %u does not exist on dev. %u\n", oldChnlId, pstDevice->DevId);
		return FAILURE;
	}
	if (NULL != (pstDevice->papstChannels)[newChnlId])
	{
		UT_ErrorLog(VCORE, "Channel %u  already exists on dev. %u\n", newChnlId, pstDevice->DevId);
		return FAILURE;
	}
	/*Add the channel in the device array */
	(pstDevice->papstChannels)[newChnlId] = (pstDevice->papstChannels)[oldChnlId];
	(pstDevice->papstChannels)[oldChnlId] = NULL;
	return SUCCESS;
}

/****************************************************************************
 * DMGR_AddConference : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Stores it at an appropriate location in the conference array that 
 *          corresponds to the device. 
 *
 *  \return 
 *  status
 *
 *  \param pstDevice Device which channel will be added.
 *  \param pstConf Poiner to confeence structure
 */
VSTATUS DMGR_AddConference(IN SDevice * pstDevice, SConference *pstConf)
{

	if (pstConf->usMSPConfId >= CFG_MAX_CONF_PER_DEV)
	{
		UT_ErrorLog(VCORE, "DMGR_AddConference: Could not add, Invalid conference id\n");
		return VAPI_ERR_INVALID_PARAM;
	}

	/*If the conference already exists */
	if (pstDevice->apstConference[pstConf->usMSPConfId])
	{
		UT_ErrorLog(VCORE, "DMGR_AddConference: Conference %u already exists on dev. %u\n", 
			pstConf->usMSPConfId, pstDevice->DevId);

		return VAPI_ERR_INVALID_PARAM;
	}

	/*Add the conference in the device array */
	pstDevice->apstConference[pstConf->usMSPConfId] = pstConf;
	return SUCCESS;
}

/****************************************************************************
 * DMGR_AddParticipant : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -   Stores it at an appropriate location in the channel array that 
 *          corresponds to the device. 
 *
 *  \return 
 *  SUCCESS/ FAILURE \n
 *
 *  \param pstDevice Device which channel will be added.
 *  \param pstChnl Connection structure handling the device channel.
 */
VSTATUS DMGR_AddParticipant(SChnl *pstChnl)
{

	if (pstChnl->usMSPChnlId >= pstChnl->pstDev->usMaxChnls)
	{
		UT_ErrorLog(VCORE, "DMGR_AddParticipant: Invalid participant id(%u)\n",pstChnl->usMSPChnlId);
		return VAPI_ERR_INVALID_PARAM;
	}

	/*If the channel already exists */
	if (pstChnl->pstConf->apstParticipant[pstChnl->usMSPChnlId])
	{
		UT_ErrorLog(VCORE, "DMGR_AddParticipant: Participant(%u) already exists on dev(%u) %x\n", 
				pstChnl->usMSPChnlId, pstChnl->pstDev->DevId, pstChnl);
		return SUCCESS;
	}

	pstChnl->pstConf->usNumOfParticipant++;

	if (pstChnl->pstConf->usNumOfParticipant >= GTL_GetDefaultMaxPartPerConf(pstChnl->pstDev->ucDevType))
	{
		UT_ErrorLog(VCORE, "DMGR_AddParticipant: Number of participants (%u) exceed max supported per conf\n",pstChnl->pstConf->usNumOfParticipant);
		return VAPI_ERR_INVALID_PARAM;
	}	

	/*Add the channel in the device array */
	pstChnl->pstConf->apstParticipant[pstChnl->usMSPChnlId] = pstChnl;
	return SUCCESS;
}

void DMGR_PurgeDevice(IN SDevice * pstDevice)
{
	U16 uiCount, usPartCnt;
	IN SChnl * pstChnl;

	UT_Log(APPITF, INFO, "DMGR_PurgeDevice: Entering dev(%u)\n", pstDevice->DevId);

	for (uiCount = 0; uiCount < CFG_MAX_CONF_PER_DEV; uiCount++)
	{
		if (pstDevice->apstConference[uiCount] != NULL)
		{
			UT_Log(VCORE, INFO,
				"DMGR_PurgeDevice: Dev(%u) Closing conference(%u)\n", pstDevice->DevId, uiCount);

			/* Remove any participants of the conference */
			for (usPartCnt = 0; usPartCnt < GTL_GetDefaultMaxPartPerConf(pstDevice->ucDevType); usPartCnt++)
			{
				/* the participant structure is freed below as a connection*/
				if (pstDevice->apstConference[uiCount]->apstParticipant[usPartCnt] != NULL)
					VCORE_CleanParticipant (pstDevice->apstConference[uiCount]->apstParticipant[usPartCnt]);
			}
			/* Remove the conference from the AVL tree */
			VCORE_RemoveConference(pstDevice->apstConference[uiCount]);
			/* Remove the conference from device manager */
			/* Free the conference structure */
			DMGR_RemoveConf(pstDevice, uiCount);
		}
	}

	for (uiCount = 0; uiCount < pstDevice->usMaxChnls; uiCount++)
	{
		if ((pstChnl = pstDevice->papstChannels[uiCount]) != NULL)
		{
			UT_Log(VCORE, INFO,
				"DMGR_PurgeDevice: Dev(%u) Closing connection (%u)\n", pstDevice->DevId, uiCount);
			/* Remove the connection from the AVL tree  
			this prevent adding new requests to this connection */
			VCORE_RemoveConnection(pstChnl);

			/* Remove connection indexed from the device array*/
			DMGR_RemoveChannel(pstDevice, uiCount);

			/* Stop timers, remove mutex, semaphores*/
			VCORE_CloseConnection(pstChnl);

			/* free the connection structure*/
			UT_FreeMem(pstChnl);
		}
	}
	UT_Log(APPITF, INFO, "DMGR_PurgeDevice: Exiting\n");

}
