/*! \file device_itf.c
 @defgroup device_level  Device related API functions
 *  @{
 */

/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include "ut.h"
#include "vapi.h"
#include "dmgr.h"
#include "vcore.h"
#include "vcore_voip.h"
#include "appitf.h"
#include "cfg.h"
#include "msp.h"
#include "resmgr.h"

extern Boolean bVapiInitialized_g;
/*Global Definitions*/
extern SVAPIConfig stVAPIConfig_g;

/*COMCERTO resources */
extern SAVLTree *pstResTree_g;
extern U16 M823x9_mix_mode_res[32];
extern U16 M82910_mix_mode_res[32];

/***************************************************************************
 * VAPI_OpenDevice: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This function allocates resources associated with a device.\n
*	The Device ID is specified by the application that uniquely identifies a device.\n
*	The device ID is passed to the GTL layer for device initialization.\n
*	The GTL layer prepares mapping of Device ID to the proper PCI base address (for PCI)
*	or CSME MAC  *address (for CSME) and vice-versa.\n
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">Device to be opened.\n
*		A predefined device identifier corresponding to a particular device. \n 
*		Note: The specified Device ID exists in the device list specified in the \n
*		pstDeviceHeaders parameter of VAPI_Init. \n
*		Several examples code for this device list initialization are provided in Examples section.</td>
*	</tr>
* 
*	<tr>
*		<td style="vertical-align: top;">pfnEventCallback</td>
*		<td style="vertical-align: top;">Device specific Event callback. \n
*		The device level event callback. \n 
*		When events are received on this device this callback is used. \n</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_DEV_ALREADY_OPEN
*	\li VAPI_ERR_NOMEM
*
*	\n \b Commands:
*	No Comcerto commands sent.
*/
VSTATUS VAPI_OpenDevice(IN DEVID DevId, IN PFNEventCallback pfnEventCallback)
{
	SDevice *pstDevice;
	VSTATUS Status;
	gtl_phy_itf_t stPhyItf;
	SVAPIDeviceConfig *pstDevTypeConfig;

	if (!bVapiInitialized_g)
	{
		/*Used printf instead of UT_Log because UT is not yet initialized */
		printf("VAPI_OpenDevice: VAPI library not initialized.\n");
		Status = VAPI_ERR_LIB_NOT_INITIALIZED;
		goto err0;
	}

	UT_Log(APPITF, INFO, "Entered VAPI_OpenDevice: dev(%u), cb(0x%x)\n", DevId, pfnEventCallback);

	if (NULL != DMGR_GetDevice(DevId))
	{
		UT_ErrorLog(APPITF, "Device is already Open\n");
		Status = VAPI_ERR_DEV_ALREADY_OPEN;
		goto err0;
	}

	pstDevice = UT_Calloc(1, sizeof(SDevice));
	if (pstDevice == NULL)
	{
		UT_ErrorLog(APPITF, "GTL Open device Failed\n");
		Status = VAPI_ERR_NOMEM;
		goto err0;
	}

	pstDevice->DevId = DevId;
	pstDevice->pfnEventCallback = pfnEventCallback;

	/* In case of SUPRV channel connID is device id. */
	pstDevice->stSupvChnl.ConnId = DevId;
	pstDevice->stSupvChnl.usMSPChnlId = SUPV_CHANNEL;
	pstDevice->stSupvChnl.pfnEventCallback = pfnEventCallback;

	Status = GTL_QueryInterface(DevId, &stPhyItf);

	if (Status != SUCCESS)
	{
		UT_ErrorLog(APPITF, "VAPI_OpenDevice: GTL_QueryInterface Failed %d\n", Status);
		goto err0;
	}

	pstDevice->pvItfInfo = stPhyItf.pvItfInfo;
	pstDevice->eItfType = stPhyItf.type;
	pstDevice->eDevMode = stPhyItf.eDevMode;
	pstDevice->ucDevType = stPhyItf.ucDevType;
	pstDevice->usMaxChnls = stPhyItf.usMaxChnls;
	pstDevice->eEthHdrMode = eMODE_NONE;
	pstDevice->eSrcIpMode = eNO_SRC_IP;
	pstDevice->bEthHdrSet = False;
	pstDevice->MaxMsgSize = DEFAULT_FIFO_MAX_SIZE;

	/* boot mode for the M823XX device*/
	if ((pstDevice->eItfType == eCSM_ITF ) &&
		((pstDevice->ucDevType== DEV_TYPE_M823XX) || (pstDevice->ucDevType== DEV_TYPE_M823XX_2)))
		pstDevice->ucEthBootMode = stPhyItf.ucEthBootMode;

	/* Deep copy the mem info for the device */
	pstDevTypeConfig = DMGR_GetVAPIDeviceConfig(pstDevice->ucDevType);
	if (pstDevTypeConfig == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_OpenDevice: pstDevTypeConfig == NULL\n");
		Status = VAPI_ERR_INVALID_DEV_TYPE;
		goto err0;
	}

	UT_MemCopy(&pstDevice->stMemInfo, &pstDevTypeConfig->stMemInfo, sizeof(SDevMemInfo));

	pstDevice->papstChannels = UT_Calloc(pstDevice->usMaxChnls, sizeof(SChnl *));
	if (pstDevice->papstChannels == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err0;
	}

	pstDevice->apstConference = UT_Calloc(GTL_GetDefaultMaxConfPerDevice(pstDevice->ucDevType), sizeof(SConference));
	if (pstDevice->apstConference == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err0;
	}	

	if (stPhyItf.type == eCSM_ITF)
	{
		pstDevice->uMaxFrameSize = CFG_MAX_FRM_SIZE_ETH;
	}
	else if (stPhyItf.type == ePOS_ITF)
	{
		pstDevice->uMaxFrameSize = CFG_MAX_FRM_SIZE_POS;
	}
	else
	{
		pstDevice->uMaxFrameSize = CFG_MAX_FRM_SIZE_PCI;
	}

	if (pstDevice->eDevMode == eMASTER)
	{
		pstDevice->eState = eUP;
	}

	pstDevice->ucSPUSentStatus = SPU_CTRL_NOT_SENT;	/*SPU_FEATURE_CONTROL to be sent */

	UT_Log(VCORE, INFO, "VAPI_OpenDevice: dev(%u), ItfType = %d\n", DevId, pstDevice->eItfType);

	/*Initialize supervisory channel */
	pstDevice->stSupvChnl.usMSPChnlId = SUPV_CHANNEL;
	pstDevice->stSupvChnl.pstDev = pstDevice;
	VCORE_OpenConnection(&(pstDevice->stSupvChnl));

	if ((Status = DMGR_AddDevice(pstDevice)) != SUCCESS)
	{
		UT_ErrorLog(APPITF, "DMGR failed to add device\n");
		goto err2;
	}

	if ((Status = GTL_Open(DevId)) < SUCCESS)
	{
		UT_ErrorLog(APPITF, "GTL Open device Failed\n");
		if (Status == GTL_ERR_ALREADY_OPEN)
		{
			Status = VAPI_ERR_DEV_ALREADY_OPEN;
		}
		goto err3;
	}


	UT_Log(APPITF, INFO, "VAPI_OpenDevice: Exiting status(%d), dev(%u)\n", Status, DevId);

	return SUCCESS;

err3:
	DMGR_RemoveDevice(pstDevice->DevId);

err2:
	UT_FreeMem(pstDevice->papstChannels);
	UT_FreeMem(pstDevice->apstConference);

err0:
	return Status;
}

/***************************************************************************
 * VAPI_CloseDevice: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This API closes the devices and does the cleanup based on the mode. 
*	The VAPI removes all pending request for all channels and then calls the GTL_Shutdown API. 
*	In case of ePURGE_ALL if frees all MSP resources (channels, participants, transcoding sessions, conferences).
*	\n
*	<table style="text-align: left;" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">Device to be closed</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eMode</td>
*		<td style="vertical-align: top;">Shutdown Mode</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_NOMEM
*
*	\n \b usage
*	\include close_device.c
*
*	\n \b Commands:
*	- CHANNELS_RESET
*/
VSTATUS VAPI_CloseDevice(IN DEVID DevId, IN ECloseMode eMode)
{
	SDevice *pstDevice;
	VSTATUS Status = SUCCESS;

	if (!bVapiInitialized_g)
	{
		/*Used printf instead of UT_Log because UT is not yet initialized */
		printf("VAPI_CloseDevice: VAPI library not initialized.\n");
		Status = VAPI_ERR_LIB_NOT_INITIALIZED;
		goto err;
	}

	UT_Log(APPITF, INFO, "Entered VAPI_CloseDevice: dev(%u)\n", DevId);

	pstDevice = DMGR_GetDevice(DevId);

	if ((pstDevice == NULL) || (pstDevice->DevId != DevId))
	{
		UT_ErrorLog(APPITF, "Invalid device id\n");
		Status = VAPI_ERR_INVALID_DEVID;
		goto err;
	}

	Status = APPITF_CloseDevice(pstDevice, eMode);

	UT_Log(APPITF, INFO, "VAPI_CloseDevice: Exiting status(%d), dev(%u)\n", Status, DevId);
	return Status;

err:
	return Status;
}

/***************************************************************************
 * APPITF_CloseDevice: The function does the following things -
 ***************************************************************************/
/* 
 *  1. find out the next device structure \n
 *  2. Cleanup according to the mode
 *
 * 
 *  \li SUCCESS or VAPI_ERR_PENDING
 *  VAPI_ERR_INVALID_DEVID\n
 *  VAPI_ERR_UNSUPP_FEATURE\n
 *
 *  DevId Device to be closed.
 *  Shutdown Mode
 */
VSTATUS APPITF_CloseDevice(IN SDevice * pstDevice, IN ECloseMode eMode)
{
	SVapiReq *pstVapiReq;
	VSTATUS Status = SUCCESS;

	UT_Log(APPITF, INFO, "\nAPPITF_CloseDevice:Entry\n");

	pstDevice->eState = eCLOSING;
	switch (eMode)
	{
	case ePURGE:
		Status = VCORE_PurgeDevResources(pstDevice);

		break;

	case ePURGE_ALL:
		/* Allocate memory for the request and its UserData. If error get out */
		pstVapiReq = VCORE_AllocateRequest(0);
		if (pstVapiReq == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			goto err;
		}

		/* Fill the request with the regular parameters */
		VCORE_SetDeviceRequest(pstVapiReq, NULL, VFSM_PurgeMspResources, pstDevice->DevId);

		Status = VCORE_ProcessRequest(&(pstDevice->stSupvChnl), pstVapiReq);

		/* If MSP resources purged OK, purge Device resources */
		if (Status == SUCCESS)
			Status = VCORE_PurgeDevResources(pstDevice);

		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);

		break;

	default:
		UT_ErrorLog(APPITF, "APPITF_CloseDevice: Unsupported mode\n");
		Status = VAPI_ERR_UNSUPP_FEATURE;
		goto err;
	}

	UT_Log(APPITF, INFO, "APPITF_CloseDevice:Exit\n");

	return SUCCESS;

err:
	return Status;
}

/***************************************************************************
 * VAPI_BootDevice: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	The application uses this API to boot a Comcerto device. \n 
*	<b>Note: In Master mode this API is not needed</b>. \n 
*	Both the PCI and Ethernet based devices are booted using this API. \n
*	If the device to be booted is on the Ethernet then VAPI_AssignBootMAC should be \n     
*	called before this API.  This API downloads the AXF image file and starts the image execution. \n
*	The API gets completed after receiving  SUPVSR_READY event from MSP. \n
*	After the completion device is ready to receive commands. \n
*	The boot steps include configuring the clock, SDRAM, chip selects etc 
*	by the using the corresponding MSP commands. \n
*	This API can used to boot more than one slave devices in parallel,
*	provided that for Ethernet interface devices the MAC addresses have been assigned to them using
*	VAPI_AssignBootMAC(). \n
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">The device to be booted.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pvImageFile</td>
*		<td style="vertical-align: top;">The pointer to image buffer.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">uImageSize</td>
*		<td style="vertical-align: top;">The size of image (the length of the buffer provided)</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstSHwCfg</td>
*		<td style="vertical-align: top;">User specified hardware configuration data. \n
*		Use NULL for the default initialization.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_NULL_POINTER_PASSED
*	\li VAPI_ERR_INVALID_OPERATION
*	\li VAPI_ERR_DEV_NO_MAAS
*	\li VAPI_ERR_DEV_IS_UP
*
*	\n \b Commands:
*	The Comcerto commands sent are:
*	- BRM_SET_CLOCK
*	- BRM_SET_ARM_CLKMODE
*	- BRM_SET_CS_PARAMS
*	- BRM_SET_SDRAM_PARAM1
*	- BRM_SET_SDRAM_PARAM2
*	- BRM_FIFOWRITE
*	- BRM_RUN_CHKSUM
*	- BRM_PROGSTART
*/
VSTATUS VAPI_BootDevice(IN DEVID DevId, IN void *pvImageFile, IN U32 uImageSize, IN SHwCfg *pstSHwCfg, IN SRequest *pstRequest)
{
	SVapiReq *pstVapiReq;
	SDevice *pstDevice;
	SBootDevUserData *pstBootDevUserData;
	VSTATUS Status;
	Boolean bIsSync;

	if (!bVapiInitialized_g)
	{
		/*Used printf instead of UT_Log because UT is not yet initialized */
		printf("VAPI_BootDevice: VAPI library not initialized.\n");
		Status = VAPI_ERR_LIB_NOT_INITIALIZED;
		goto err_out;
	}

	UT_Log(APPITF, INFO, "Entered VAPI_BootDevice: dev(%u) (Image Size = %u)\n", DevId, uImageSize);

	if (pvImageFile == NULL)
	{
		UT_ErrorLog(APPITF, "MSP image buffer is NULL\n");
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		goto err_out;
	}

	if (uImageSize == 0)
	{
		UT_ErrorLog(APPITF, "MSP image buffer is NULL\n");
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		goto err_out;
	}
	pstDevice = DMGR_GetDevice(DevId);

	if (pstDevice == NULL)
	{
		UT_ErrorLog(APPITF, "Invalid device id\n");
		Status = VAPI_ERR_INVALID_DEVID;
		goto err_out;
	}

	/* Loading the firmware on Master setup is not allowed.
	   The firmware is loaded by the bootloader */
	if (pstDevice->eDevMode == eMASTER)
	{
		/* this check also avoid to rerun the PLL wa on running device */
		UT_ErrorLog(APPITF, "Device is in Master mode, VAPI_BootDevice not allowed\n");
		Status = VAPI_ERR_INVALID_OPERATION;
		goto err_out;
	}

	if (pstDevice->eState == eCLOSING)
	{
		UT_ErrorLog(APPITF, "VAPI_BootDevice: Device is not up\n");
		Status = VAPI_ERR_DEV_IS_NOT_UP;
		goto err_out;
	}
	if (pstDevice->eItfType == eCSM_ITF && pstDevice->eState != eMAAS)
	{
		UT_ErrorLog(APPITF, "VAPI_BootDevice: MAAS assign not done\n");
		Status = VAPI_ERR_DEV_NO_MAAS;
		goto err_out;
	}
	if (pstDevice->eState == eUP)
	{
		UT_ErrorLog(APPITF, "VAPI_BootDevice: Device already booted\n");
		Status = VAPI_ERR_DEV_IS_UP;
		goto err_out;
	}

	/* Allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SBootDevUserData));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}

	pstBootDevUserData = pstVapiReq->pvUserData;

	pstBootDevUserData->pvImageFile = UT_AllocMem(uImageSize * sizeof(U8));
	if (pstBootDevUserData->pvImageFile == NULL)
	{
		UT_ErrorLog(APPITF, "No memory available\n");
		Status = VAPI_ERR_NOMEM;
		goto err_data;
	}

	UT_MemSet(pstBootDevUserData->pvImageFile, 0, uImageSize * sizeof(U8));
	UT_MemCopy(pstBootDevUserData->pvImageFile, pvImageFile, uImageSize);

	pstBootDevUserData->uImageSize = uImageSize;

	if (pstSHwCfg != NULL)
	{
		pstBootDevUserData->pstSHwCfg = UT_AllocMem(sizeof(SHwCfg));
		UT_MemCopy(pstBootDevUserData->pstSHwCfg, pstSHwCfg, sizeof(SHwCfg));
	}
	else
		pstBootDevUserData->pstSHwCfg = NULL;

	/* Fill the request with the regular parameters */
	VCORE_SetDeviceRequest(pstVapiReq, pstRequest, VFSM_BootDevice, DevId);

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(&(pstDevice->stSupvChnl), pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}


	UT_Log(APPITF, INFO, "VAPI_BootDevice: Exiting status(%d), dev(%u)\n", Status, DevId);

/* no error exit */
	return Status;

      err_data:
	VCORE_FreeRequest(pstVapiReq);
      err_out:
	return Status;
}

/***************************************************************************
 * VAPI_BootDeviceFile: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	The application uses this API to boot a Comcerto device. \n 
*	<b>Note: In Master mode this API is not needed</b>. \n 
*	Both the PCI and Ethernet based devices are booted using this API. \n
*	If the device to be booted is on the Ethernet then VAPI_AssignBootMAC() should be \n     
*	called before this API.  This API downloads the AXF image file and starts the image execution. \n
*	The API gets completed after receiving  SUPVSR_READY event from MSP. \n
*	After the completion device is ready to receive commands. \n
*	The boot steps include configuring the clock, SDRAM, chip selects etc 
*	by the using the corresponding MSP commands. \n
*	This API can used to boot more than one slave devices in parallel,
*	provided that for Ethernet interface devices the MAC addresses have been assigned to them using
*	VAPI_AssignBootMAC(). \n
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">The device to be booted.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">strFileName</td>
*		<td style="vertical-align: top;">The image file name.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstSHwCfg</td>
*		<td style="vertical-align: top;">User specified hardware configuration data. \n
*		Use NULL for the default initialization</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_NULL_POINTER_PASSED
*	\li VAPI_ERR_INVALID_OPERATION
*	\li VAPI_ERR_DEV_NO_MAAS
*	\li VAPI_ERR_DEV_IS_UP
*
*	\n \b Commands:
*	The Comcerto commands sent are:
*	- BRM_SET_CLOCK
*	- BRM_SET_ARM_CLKMODE
*	- BRM_SET_CS_PARAMS
*	- BRM_SET_SDRAM_PARAM1
*	- BRM_SET_SDRAM_PARAM2
*	- BRM_FIFOWRITE
*	- BRM_RUN_CHKSUM
*	- BRM_PROGSTART
*/
VSTATUS VAPI_BootDeviceFile(IN DEVID DevId, IN char *strFileName, IN SHwCfg *pstSHwCfg, IN SRequest *pstRequest)
{
	SVapiReq *pstVapiReq;
	SDevice *pstDevice;
	SBootDevUserData *pstBootDevUserData;
	VSTATUS Status;
	Boolean bIsSync;

	if (!bVapiInitialized_g)
	{
		/* Used printf instead of UT_Log because UT is not yet initialized */
		printf("VAPI_BootDeviceFile: VAPI library not initialized.\n");
		Status = VAPI_ERR_LIB_NOT_INITIALIZED;
		goto err_out;
	}

	if (strFileName == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_BootDeviceFile: MSP file name is NULL\n");
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		goto err_out;
	}

	UT_Log(APPITF, INFO, "Entered VAPI_BootDeviceFile: dev(%u) (Image file name = %s)\n", DevId, strFileName);

	pstDevice = DMGR_GetDevice(DevId);
	if (pstDevice == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_BootDeviceFile: Invalid device id\n");
		Status = VAPI_ERR_INVALID_DEVID;
		goto err_out;
	}

	/* Loading the firmware on Master setup is not allowed.
	   The firmware is loaded by the bootloader */
	if (pstDevice->eDevMode == eMASTER)
	{
		/* this check also avoid to rerun the PLL wa on running device */
		UT_ErrorLog(APPITF, "VAPI_BootDeviceFile: Device is in Master mode, VAPI_BootDevice not allowed\n");
		Status = VAPI_ERR_INVALID_OPERATION;
		goto err_out;
	}

	if (pstDevice->eState == eCLOSING)
	{
		UT_ErrorLog(APPITF, "VAPI_BootDeviceFile: Device is not up\n");
		Status = VAPI_ERR_DEV_IS_NOT_UP;
		goto err_out;
	}
	if (pstDevice->eItfType == eCSM_ITF && pstDevice->eState != eMAAS)
	{
		UT_ErrorLog(APPITF, "VAPI_BootDeviceFile: MAAS assign not done\n");
		Status = VAPI_ERR_DEV_NO_MAAS;
		goto err_out;
	}
	if (pstDevice->eState == eUP)
	{
		UT_ErrorLog(APPITF, "VAPI_BootDeviceFile: Device already booted\n");
		Status = VAPI_ERR_DEV_IS_UP;
		goto err_out;
	}

	/* Allocate memory for the request and its UserData. If error get out. */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SBootDevUserData));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}

	pstBootDevUserData = pstVapiReq->pvUserData;

	UT_MemSet(pstBootDevUserData, 0, sizeof(SBootDevUserData));
	UT_StrCpy((char *)pstBootDevUserData->strFileName, strFileName);
	pstBootDevUserData->pvImageFile = NULL;

	if (pstSHwCfg != NULL)
	{
		pstBootDevUserData->pstSHwCfg = UT_AllocMem(sizeof(SHwCfg));
		UT_MemCopy(pstBootDevUserData->pstSHwCfg, pstSHwCfg, sizeof(SHwCfg));
	}
	else
		pstBootDevUserData->pstSHwCfg = NULL;

	/* Fill the request with the regular parameters */
	VCORE_SetDeviceRequest(pstVapiReq, pstRequest, VFSM_BootDevice, DevId);

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(&(pstDevice->stSupvChnl), pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_BootDeviceFile: Exiting status(%d), dev(%u)\n", Status, DevId);

/* no error exit */
	return Status;

      err_out:
	return Status;
}

/***************************************************************************
 * VAPI_InitDevice: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This command is intended to be sent prior any connections. \n
*	It initializes the operation mode of the device (i.e VoIPoETH, VoAAL2oATM, etc ) \n
*	This API can be called anytime after VAPI_Init(), but prior to VAPI_CreateConnection() \n
*	The parameters for the configuration are retrived using the profile ID \n
*	<b>****** REMARK From the version 2.5.x this API is mandatory ****** \n
*	****** if not sent all subsequent Connection Level APIs ****** \n
*	****** will return the VAPI_ERR_DEVICE_NOT_INITIALIZED error ****** </b> \n
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">The device to initialize.</td>
*	</tr>
* 
*	<tr>
*		<td style="vertical-align: top;">ulOperationModeId</td>
*		<td style="vertical-align: top;">This ID represents the list of command to be sent.\n
*		The list of OperationModeId is available in vapi_profile.h file.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ulProfileId</td>
*		<td style="vertical-align: top;">This ID represents the configuration to be used to initialize the device.\n
*		The list of ProfileId is available in vapi_profile.h file.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pulUserParams</td>
*		<td style="vertical-align: top;">The pointer to a buffer containg parameters provided by the user.\n
*		The user specify here the device specific parameters such as MAC address or IP address.\n
*		The format for these parameters are described in the vapi_profile.h file.\n
*		In case of profile VAPI_DEV_PROF_SPU_CUSTOM the value for SPU Feature mask is passed through the pulUserParams parameters.\n
*		For firmware release below MR19 the SPU Feature Mask is a single param (16 bits). \n
*		From MR19 it could be 2 params (32bits). The LSB is param1 MSB param2 (see Usage section below)\n 
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_UNSUPP_FEATURE
*
*	\n \b Usage:
*	\include init_device.c
*
*	\n \b Commands:
*	For the VAPI release 2.4.x and 2.5x the Comcerto commands sent are:
*	\li SPU_CTRL
*	\li MULTI_CMD
*
*/

VSTATUS VAPI_InitDevice(IN DEVID DevId,
				IN U32 ulOperationModeId,
				IN U32 ulProfileId,
				IN U32 *pulUserParams,
				IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq;
	SDevice *pstDevice;
	U32 *pstUserData;
	VSTATUS Status;
	Boolean bIsSync;
 
	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	if (ulOperationModeId != VAPI_DEV_OPMODE_RECOVER)
	{
		Status = VCORE_DeviceGenericCheck(DevId, "VAPI_InitDevice", &pstDevice);

		if (Status != SUCCESS)
			goto err_out;

		UT_Log(APPITF, INFO, "Entered VAPI_InitDevice: dev(%u), profile_id(%u), operation_mode_id(%u)\n", 
					DevId, ulProfileId, ulOperationModeId);

		/* log the parameters passed if required (define in appitf.h) */
		LOG_DEVICE_INIT_PARAMS;
	}


	switch (ulOperationModeId)
	{
	/* This particular opmode allows the application to set the device in ready to use mode
	it can be used when the device is up ad ready, but the application has been stopped then restarted
	it has to be used prior to VAPI_BootDevice inn this case */
	case VAPI_DEV_OPMODE_RECOVER:
		if (!bVapiInitialized_g)
		{
			/*Used printf instead of UT_Log because UT is not yet initialized */
			printf("%s: VAPI library not initialized.\n", "VAPI_InitDevice");
			Status = VAPI_ERR_LIB_NOT_INITIALIZED;
			goto err_out;
		}	

		pstDevice = DMGR_GetDevice(DevId);
		if (pstDevice == NULL)
		{
			UT_ErrorLog(APPITF, "VAPI_InitDevice: Invalid device id(%d)\n", DevId);
			Status = VAPI_ERR_INVALID_DEVID;
			goto err_out;
		}

		UT_Log(APPITF, INFO, "Entered VAPI_InitDevice: dev(%u), profile_id(%u), operation_mode_id(%u)\n", 
					DevId, ulProfileId, ulOperationModeId);
		/* log the parameters passed if required (define in appitf.h) */
		LOG_DEVICE_INIT_PARAMS;		
		
		Status = SUCCESS;
		pstDevice->eState = eUP;
		pstDevice->bIsDeviceInitialized = True;
		pstDevice->bMultiCmdEnabled = True;

		break;

	/* default operation mode (spu + multicmd) : proceed*/
	case VAPI_DEV_OPMODE_DEFAULT:
		break;

	/* for other values error */
	default:
		Status = VAPI_ERR_UNSUPP_FEATURE;
		goto err_out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(U32));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}

	pstUserData = pstVapiReq->pvUserData;

	/* for now we  have a simple profile handling
	the SPU CTRL is set to DEFAULT or DEFAULT + DFECAN */
	switch (ulProfileId)
	{

	case VAPI_DEV_PROF_SPU_DFECAN_ENABLE:
		*pstUserData = SPU_CTRL_DFECAN_MASK;
		break;

	case VAPI_DEV_PROF_SPU_FIRMWARE_DEFAULT:	
		*pstUserData = SPU_CTRL_FIRMWARE_DEFAULT_MASK;
		break;

	case VAPI_DEV_PROF_DEFAULT:	
		*pstUserData = SPU_CTRL_DEFAULT_MASK;
		break;

	case VAPI_DEV_PROF_SPU_FSK_IP_DETECTION:
		(* pstUserData) = SPU_FEATURE_FSKRX_MASK | SPU_FEATURE_IP_TONEDET_MASK;
		break;
		
	case VAPI_DEV_PROF_SPU_CUSTOM:
		if (pulUserParams == NULL)
		{
			UT_ErrorLog(APPITF, "VAPI_InitDevice:  VAPI_DEV_PROF_SPU_CUSTOM requires pulUserParams (%d)\n");
			Status = VAPI_ERR_INVALID_DEVID;
			goto err_data;
		}
		(* pstUserData) = (* pulUserParams);
		break;

	default:

		Status = VAPI_ERR_UNSUPP_FEATURE;
		goto err_data;
		break;
	}

	/* depending on operation mode recover connection or init device */
	if (ulOperationModeId == VAPI_DEV_OPMODE_RECOVER)
	{
		VCORE_SetDeviceRequest(pstVapiReq, pstRequest, VFSM_InitRecover, DevId);
		pstVapiReq->usReqState = RECOV_GET_MR;
	}	
	else
	{
		VCORE_SetDeviceRequest(pstVapiReq, pstRequest, VFSM_InitDevice, DevId);
		pstVapiReq->usReqState = DEV_INIT_GET_MR;
	}

	bIsSync = pstVapiReq->bIsSync;

	/* process the request */
	Status = VCORE_ProcessRequest(&(pstDevice->stSupvChnl), pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_InitDevice: Exiting status(%d), dev(%u), profile_id(%u), operation_mode_id(%u)\n",
			Status, DevId, ulProfileId, ulOperationModeId);

/* no error exit */
	return Status;

      err_data:
	VCORE_FreeRequest(pstVapiReq);
      err_out:
	return Status;
}


/***************************************************************************
 * VAPI_GetCoreDump: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This API is used to get dump information of the slave device memories, SDRAM, IRAM, etc.\n
*	This API must be called successively to get the complete dump.\n
*	The application supplies a buffer to hold the dump data for a specified memory
*	and writes the data collected by the API to a file for the specific memory.\n 
*	<b>Note 1: This API uses BRM commands that resets the device if it was up,
*	therefore operates only when the device RESET functionality is supported
*	at the adapter level by the implementation of xxx_Reset(),
*	which in turn is called through the GTL_Reset ().</b>\n
*	\n
*	<b>Note 2: This API is not applicable in Master mode, 
*	also this API leaves the slave in a RESET state so the VAPI_BootDevice () 
*	must be called to boot the device for further use.</b>\n
*	\n
*       The VAPI_GetCoreDump() API can also be used to get a RDC (Reduced Size Coredump).\n
*       Please see the rcd.c file in the example section.\n
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">Id of the device to dump memory of.</td>
*	</tr>
* 
*	<tr>
*		<td style="vertical-align: top;">uchMemTypeIdx</td>
*		<td style="vertical-align: top;">An Index to the memory to dump.\n
*			Each index, starting from 0, represents a corresponding memory index.\n
*			The number of memories and their respective sizes are organized by device type in
*			'vapi_device_config.c' file.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">puchBuff</td>
*		<td style="vertical-align: top;">A caller allocated buffer that holds the memory dump data.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ulBuffSize</td>
*		<td style="vertical-align: top;">Size, in bytes, of the allocated memory dump destination buffer.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pulDataSize</td>
*		<td style="vertical-align: top;">A pointer indicating the number of bytes actually read.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ulOffset</td>
*		<td style="vertical-align: top;">Offset into the memory to read.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pbLastRead</td>
*		<td style="vertical-align: top;">A pointer to a Boolean flag indicating if the call to this API was last
*		read for the memory specified by ucMemTypeIdx.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstSHwCfg</td>
*		<td style="vertical-align: top;">User specified hardware configuration data. Use NULL for the default initialization </td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_NULL_POINTER_PASSED
*	\li VAPI_ERR_INVALID_MEM_IDX
*	\li VAPI_ERR_INVALID_DEV_STATE
*	\li VAPI_ERR_INVALID_OFFSET
*	\li VAPI_ERR_UNDEFINED
*
*	\n \b Commands:
*	List of Comcerto commands sent:
*	\li MAAS_ASSIGN
*	\li BRM_SET_CLOCK
*	\li BRM_SET_ARM_CLKMODE
*	\li BRM_SET_CS_PARAMS
*	\li BRM_SET_SDRAM_PARAM1
*	\li BRM_SET_SDRAM_PARAM2
*	\li BRM_FIFOREAD.
*/
VSTATUS VAPI_GetCoreDump(IN DEVID DevId,
			IN U8 uchMemTypeIdx,
			OUT U8 * puchBuff,
			IN U32 ulBuffSize,
			OUT U32 * pulDataSize,
			IN U32 ulOffset,
			OUT Boolean * pbLastRead,
			IN SHwCfg * pstSHwCfg,
			IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq;
	SDevice *pstDevice;
	SCoreDumpData *pstCoreDumpData;
	VSTATUS Status;
	Boolean bIsSync;

	U32 ulMemSize = 0;

	if (!bVapiInitialized_g)
	{
		/*Used printf instead of UT_Log because UT is not yet initialized */
		printf("VAPI_GetCoreDump: VAPI library not initialized.\n");
		Status = VAPI_ERR_LIB_NOT_INITIALIZED;
		goto err;
	}

	UT_Log(APPITF, INFO, "Entered VAPI_GetCoreDump: devid (%d)\n", DevId);

	if ((puchBuff == NULL) || (pulDataSize == NULL) || (pbLastRead == NULL))
	{
		UT_ErrorLog(APPITF, "NULL pointer passed\n");
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		goto err;
	}

	pstDevice = DMGR_GetDevice(DevId);

	if (pstDevice == NULL)
	{
		UT_ErrorLog(APPITF, "Invalid device id\n");
		Status = VAPI_ERR_INVALID_DEVID;
		goto err;
	}

	/* Handle the various device states:
	 * eBOOTING         - return immediately
	 * eCLOSING         - return immediately
	 * eUP              - reset the device & bring it in 
	 *                    eUNINITIALIZED state 
	 * eUNINITIALIZED   - call VFSM_SetDeviceForDump & 
	 *                    bring the device in eREADYFORDUMP state 
	 * eREADYFORDUMP    - continue getting memory dump
	 */

	if (pstDevice->eState == eCLOSING || pstDevice->eState == eBOOTING)
	{
		UT_ErrorLog(APPITF, "VAPI_GetCoreDump: Invalid device state\n");
		Status = VAPI_ERR_INVALID_DEV_STATE;
		goto err;
	}

	if (uchMemTypeIdx > DEV_MAX_MEMORY_INDEX || pstDevice->stMemInfo.aulMemSize[uchMemTypeIdx] == 0)
	{
		UT_ErrorLog(APPITF, "VAPI_GetCoreDump: "
			    "Wrong memory index(%d) it's out of range(0-%d) or device type %d doesn't have such memory\n",
			    (int)uchMemTypeIdx, (int) DEV_MAX_MEMORY_INDEX, (int)(pstDevice->ucDevType));
		Status = VAPI_ERR_INVALID_MEM_IDX;
		goto err;
	}

	ulMemSize = pstDevice->stMemInfo.aulMemSize[(U32) uchMemTypeIdx];
	if (ulOffset >= ulMemSize)
	{
		UT_ErrorLog(APPITF, "VAPI_GetCoreDump: Invalid offset in memory\n"
			    "offset = 0x%x, memory indx = %d\n", ulOffset, (int)uchMemTypeIdx);
		Status = VAPI_ERR_INVALID_OFFSET;
		goto err;
	}

	if (pstDevice->eState == eUP)
	{
		/* remove the device from the global device array
		so no command can be sent to this device*/
		DMGR_RemoveDevice(pstDevice->DevId);

		/*remove all pending conferences/connections of this device*/
		DMGR_PurgeDevice(pstDevice);
		pstDevice->eState = eUNINITIALIZED;

		/* reset all pending requests, timers, semaphore of the supervisor channel */
		VCORE_CloseConnection(&pstDevice->stSupvChnl);
		VCORE_OpenConnection(&(pstDevice->stSupvChnl));

		/*Make the device accessible again*/
		DMGR_AddDevice(pstDevice);

		/* If the device is up send force alert prior reseting the device*/	
		Status = VDEV_ForceAlert(&pstDevice->stSupvChnl);
		if (Status != SUCCESS)
		{
			UT_ErrorLog(APPITF, "VAPI_GetCoreDump: Unable to send Force Alert\n");
			Status = VAPI_ERR_UNDEFINED;	/* #.0 Any proper error code ? */
			goto err;
		}
		UT_Log(APPITF, INFO, "VAPI_GetCoreDump: Force Alert sent devid (%d)\n", DevId);

		/* reset the device (if device reset function registered) and GTL layer too */
		Status = VCORE_ResetDevice(pstDevice);

		if (Status != SUCCESS)
		{
			UT_ErrorLog(APPITF, "VAPI_GetCoreDump: Unable to reset device\n");
			Status = VAPI_ERR_UNDEFINED;	/* #.0 Any proper error code ? */
			goto err;
		}
	}

	pstVapiReq = VCORE_AllocateRequest(sizeof(SCoreDumpData));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err;
	}

	pstCoreDumpData = pstVapiReq->pvUserData;
	pstCoreDumpData->puchBuff = puchBuff;
	pstCoreDumpData->ulBuffSize = ulBuffSize;

	pstCoreDumpData->ulMemBase = pstDevice->stMemInfo.aulMemBase[(U32) uchMemTypeIdx];
	pstCoreDumpData->ulMemSize = ulMemSize;
	pstCoreDumpData->ulOffset = ulOffset;

	/* OUT params */
	pstCoreDumpData->pulDataSize = pulDataSize;
	pstCoreDumpData->pbLastRead = pbLastRead;

	if (pstSHwCfg != NULL)
	{
		pstCoreDumpData->pstSHwCfg = UT_AllocMem(sizeof(SHwCfg));
		UT_MemCopy(pstCoreDumpData->pstSHwCfg, pstSHwCfg, sizeof(SHwCfg));
	}
	else
		pstCoreDumpData->pstSHwCfg = NULL;

	/* Fill the request with the regular parameters */
	VCORE_SetDeviceRequest(pstVapiReq, pstRequest, 
			VFSM_GetCoreDump,	/* pfnReqFSMHdlr */
			DevId);

	bIsSync = pstVapiReq->bIsSync;

	/* if the device is already in the ready to coredump state, skip the init*/
	if (pstDevice->eState == eUNINITIALIZED)
	{
		GTL_Reset(pstDevice->DevId);
		/* in CSME control send first a MAAS assign */
		if (pstDevice->eItfType == eCSM_ITF)
			pstVapiReq->usReqState = COREDUMP_INIT;
		else
			pstVapiReq->usReqState = COREDUMP_SET_HARDWARE;
		
	}
	else
		pstVapiReq->usReqState = COREDUMP_READ;
			

	Status = VCORE_ProcessRequest(&(pstDevice->stSupvChnl), pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_GetCoreDump: Exiting status(%d), dev(%u)\n", Status, DevId);

      err:
	return Status;
}

/***************************************************************************
 * VAPI_SetEthMac: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This API programs the SRC and DEST MAC address of an outgoing IP packet.\n
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">uiID</td>
*		<td style="vertical-align: top;">Device / Connection ID.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucLevel</td>
*		<td style="vertical-align: top;">Level of command (Device/Connection).\n
* 		Valid values are:\n
*  		- CMD_LEVEL_DEVICE
*  		- CMD_LEVEL_CONN
*		</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">paucSrcMacAddr</td>
*		<td style="vertical-align: top;">Source MAC Address in Ethernet frame header (Big-endian).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">paucDestMacAddr</td>
*		<td style="vertical-align: top;">Destination MAC address in Ethernet frame header (Big-endian).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*	\li VAPI_ERR_NULL_POINTER_PASSED
*
*	\n \b usage
*	\include eth_mac.c
*
*	\n \b Commands:
*	List of Comcerto commands sent:
*	\li if CMD_LEVEL_DEVICE:
*		\li  SET_ETH_HDR 
*	\li if CMD_LEVEL_CONN:
*		\li VOPENA
*		\li  SET_ETH_HDR_CHAN 
*/
VSTATUS VAPI_SetEthMac(IN U32 uiID, IN U8 ucLevel, IN U8 * paucSrcMacAddr, IN U8 * paucDestMacAddr, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq;
	SDevice *pstDevice;
	SChnl *pstChnl;
	VSTATUS Status;
	Boolean bIsSync;
	SSetEthMAc *pstMacAddress;

	switch (ucLevel)
	{
	case CMD_LEVEL_DEVICE:
		UT_Log(APPITF, INFO, "Entered VAPI_SetEthMac: dev(%u)\n", uiID);
		Status = VCORE_DeviceGenericCheck(uiID, "VAPI_SetEthMac", &pstDevice);

		if (Status != SUCCESS)
			goto out;

		pstChnl = &(pstDevice->stSupvChnl);

		break;

	case CMD_LEVEL_CONN:
		UT_Log(APPITF, INFO, "Entered VAPI_SetEthMac: conn(%u)\n", uiID);
		Status = VCORE_ConnectionGenericCheck(uiID, "VAPI_SetEthMac", &pstChnl);

		if (Status != SUCCESS)
			goto out;

		/* for the M821xx device we must not issue the SET_ETH_HDR_CHANNEL command*/
		if((pstChnl->pstDev->ucDevType == DEV_TYPE_M821XX) ||
			(pstChnl->pstDev->ucDevType == DEV_TYPE_M83XXX))
		{
			goto finish_M821xx;
		}

		break;
	default:
		UT_ErrorLog(APPITF, "VAPI_SetEthMac: Invalid Cmd Level %u\n", ucLevel);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
		break;
	}

	if (paucSrcMacAddr == NULL || paucDestMacAddr == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_SetEthMac: Invalid Src Mac Addr or Dest Mac Addr\n");
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		goto out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SSetEthMAc));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstMacAddress = pstVapiReq->pvUserData;
	UT_MemCopy(pstMacAddress->astSrcMacAddr, paucSrcMacAddr, 6);
	UT_MemCopy(pstMacAddress->astDestMacAddr, paucDestMacAddr, 6);

	switch (ucLevel)
	{
	case CMD_LEVEL_DEVICE:
		/* Fill the request with the regular parameters */
		VCORE_SetDeviceRequest(pstVapiReq, pstRequest, VFSM_SetDevEthMac,	/* pfnReqFSMHdlr */
				  uiID);
		break;
	case CMD_LEVEL_CONN:
		/* Fill the request with the regular parameters */
		VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_SetConnEthMac,	/* pfnReqFSMHdlr */
				  uiID);
		break;
	}

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	switch (ucLevel)
	{
	case CMD_LEVEL_DEVICE:
		UT_Log(APPITF, INFO, "VAPI_SetEthMac: Exiting status(%d), dev(%u)\n", Status, uiID);
		break;
	case CMD_LEVEL_CONN:
		UT_Log(APPITF, INFO, "VAPI_SetEthMac: Exiting status(%d), conn(%u)\n", Status, uiID);
		break;
	}

      out:
	return Status;

finish_M821xx:
	UT_Log(APPITF, INFO, "VAPI_SetEthMac: Exiting status(%d), conn(%u)\n", Status, uiID);

	/* just ack the API*/
	Status = SUCCESS;

	/* if ASYNC mode call the callback function*/
	if ((pstRequest != NULL) && (pstRequest->pfnIoCompCallback != NULL))
	{
		pstRequest->pfnIoCompCallback(uiID, CMD_LEVEL_CONN, Status, NULL, 0, pstRequest->uiReqId);
		Status = VAPI_ERR_PENDING;
	}

	return Status;
}

/***************************************************************************
 * VAPI_SetDeviceIPAddr: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This function passes the IP address of a Comcerto device to MSP.\n
*	It also configures the IP layer. This function must be called before setting up a connection.\n
*	User can configure the device to be in single IP mode or in Multiple IP mode using SIpAddrInfo::bIsMultipleMode.\n
*	Multiple IP addresses can be provided in the array SIpAddrInfo::auiDevIPAddress.\n
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">Device identifier that uniquely identifies a device.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstIpAddrInfo</td>
*		<td style="vertical-align: top;">IP address information</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\b Returns: \n
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_NULL_POINTER_PASSED
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include set_ip.c
*
*	\n \b Commands:
*	List of Comcerto commands sent:
*	\li IP_SERVICE_CONFIG
*	\li IP_ADDRESS
*	\li ARP_SERVICE_CONFIG
*	\li ICMP_SERVICE_CONFIG
*/
VSTATUS VAPI_SetDeviceIPAddr(IN DEVID DevId, IN SIpAddrInfo * pstIpAddrInfo, IN SRequest * pstRequest)
{
	U32 uiCounter = 0;
	SVapiReq *pstVapiReq;
	SDevice *pstDevice;
	SChnl *pstChnl;
	VSTATUS Status;
	Boolean bIsSync;

	Status = VCORE_DeviceGenericCheck(DevId, "VAPI_SetDeviceIPAddr", &pstDevice);
	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_SetDeviceIPAddr: dev(%u)\n", DevId);

	if (pstIpAddrInfo == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_SetDeviceIPAddr: Invalid Pointer passed as Ip addr info\n");
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		goto out;
	}

	pstChnl = &(pstDevice->stSupvChnl);
	if (pstChnl == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_SetDeviceIPAddr: pointer NULL\n");
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		goto out;
	}

	if ((pstIpAddrInfo->bIsMultipleMode == True &&
	     pstDevice->eSrcIpMode == eSINGLE_IP)
	    || (pstIpAddrInfo->bIsMultipleMode == False && pstDevice->eSrcIpMode == eMULTIPLE_IP))
	{
		UT_ErrorLog(APPITF, "VAPI_SetDeviceIPAddr:Invalid switching bw Single/Multiple mode\n");
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	for (uiCounter = 0; uiCounter < pstIpAddrInfo->ucNumOfSrcIpAddr; uiCounter++)
	{
		if (0 == pstIpAddrInfo->auiDevIPAddress[uiCounter])
		{
			UT_ErrorLog(APPITF, "VAPI_SetDeviceIPAddr:Invalid SrcIpAddr at index %d\n",
				    pstIpAddrInfo->ucNumOfSrcIpAddr);
			Status = VAPI_ERR_INVALID_PARAM;
			goto out;
		}

		if (pstIpAddrInfo->bIsMultipleMode == False)
		{
			break;
		}
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SIpAddrInfo));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	UT_MemCopy(pstVapiReq->pvUserData, pstIpAddrInfo, sizeof(SIpAddrInfo));

	/* Fill the request with the regular parameters */
	VCORE_SetDeviceRequest(pstVapiReq, pstRequest, VFSM_SetDeviceIPAddr,	/* pfnReqFSMHdlr */
			      DevId);

	bIsSync = pstVapiReq->bIsSync;

	/*Process Request */
	Status = VCORE_ProcessRequest(&(pstDevice->stSupvChnl), pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}
	UT_Log(APPITF, INFO, "VAPI_SetDeviceIPAddr: Exiting status(%d), dev(%u)\n", Status, DevId);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_AssignBootMac: The function does the following things -
 ***************************************************************************/
/*!
*	  \n \b Description: \n
*	This function assigns the MAC address to the slave mode Ethernet device. \n
*	It also disables broadcast reception of the device. \n
*	This API is required for Ethernet interface and needs to be called sequentially \n
*	for all slave Ethernet devices (i.e. the slaves should be brought
*	out of reset one by one and then call VAPI_AssignBootMAC()). \n
*	After assigning MAC to all the slave devices they can be booted in parallel using VAPI_BootDevice(). \n
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">Device identifier that uniquely identifies a device.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\b Returns: \n
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_DEV_NO_MAAS
*	\li VAPI_ERR_NO_CSM_ITF
*	\li VAPI_ERR_DEV_IS_UP
*
*	 \n \b Commands:
*	List of Comcerto commands sent:
*	\li MAAS_ASSIGN
*/
VSTATUS VAPI_AssignBootMAC(IN DEVID DevId, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq;
	SDevice *pstDevice;
	VSTATUS Status;
	Boolean bIsSync;

	if (!bVapiInitialized_g)
	{
		/*Used printf instead of UT_Log because UT is not yet initialized */
		printf("VAPI_AssignBootMAC: VAPI library not initialized.\n");
		Status = VAPI_ERR_LIB_NOT_INITIALIZED;
		goto err;
	}

	UT_Log(APPITF, INFO, "Entered VAPI_AssignBootMAC: dev(%u)\n", DevId);

	pstDevice = DMGR_GetDevice(DevId);

	if (pstDevice == NULL)
	{
		UT_ErrorLog(APPITF, "Invalid device id\n");
		Status = VAPI_ERR_INVALID_DEVID;
		goto err;
	}

	if (pstDevice->eItfType != eCSM_ITF)
	{
		UT_ErrorLog(APPITF, "VAPI_AssignMAC: Interface in not CSME\n");
		Status = VAPI_ERR_NO_CSM_ITF;
		goto err;
	}

	if (pstDevice->eState == eCLOSING)
	{
		UT_ErrorLog(APPITF, "VAPI_AssignMAC: Device is not up\n");
		Status = VAPI_ERR_DEV_IS_NOT_UP;
		goto err;
	}

	if (pstDevice->eState == eMAAS)
	{
		UT_ErrorLog(APPITF, "VAPI_AssignMAC: MAAS already done \n");
		Status = VAPI_ERR_DEV_NO_MAAS;	/* New error code   */
		goto err;
	}

	if (pstDevice->eState == eUP)
	{
		UT_ErrorLog(APPITF, "VAPI_AssignMAC: Device already booted\n");
		Status = VAPI_ERR_DEV_IS_UP;
		goto err;
	}

	/* Reset the device and GTL layer before sending MAAS_ASSIGN*/ 
	Status = VCORE_ResetDevice(pstDevice);
	if (Status != SUCCESS)
	{
		UT_ErrorLog(APPITF, "VAPI_AssignMAC: Can't reset the device\n");
		Status = VAPI_ERR_DEV_IS_UP;
		goto err;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(0);
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err;
	}

	/* Fill the request with the regular parameters */
	VCORE_SetDeviceRequest(pstVapiReq, pstRequest, VFSM_AssignBootMAC,	/* pfnReqFSMHdlr */
			      DevId);

	/* for the M823XX device we have 3 boot modes
	so set the right state */
	if ((pstDevice->ucDevType == DEV_TYPE_M823XX) || (pstDevice->ucDevType == DEV_TYPE_M823XX_2))
	{
		switch(pstDevice->ucEthBootMode)
		{
		case BOOT_MODE_STANDARD:
			pstVapiReq->usReqState = BOOT_QUERY_READY;
			break;

		case BOOT_MODE_ALTERNATE:
			/* if a READY frame has not been received yet
			start a timer to wait for it*/
			if (pstDevice->bIsReadyReceived == False)
			{
				pstVapiReq->usReqState = BOOT_WAIT_READY;
				/*Start Timer for ready response timeout */
				UT_TimerStart(pstDevice->stSupvChnl.pstRespTimer, 
					stVAPIConfig_g.uiMspRespTimeout, VCORE_RespTimeoutHdlr,
					(void *)&(pstDevice->stSupvChnl));
			}
			else /* else proceed with the regular MAAS assign*/
			{
				pstVapiReq->usReqState = BOOT_MAAS; /*default*/
			}
			break;

		case BOOT_MODE_LEGACY:
		default:
			pstVapiReq->usReqState = BOOT_MAAS; /*default*/
			break;
		}
	}

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(&(pstDevice->stSupvChnl), pstVapiReq);

	/*If it is a blocking call then free the memory */
	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_AssignBootMAC: Exiting status(%d), dev(%u)\n", Status, DevId);

     err:
	return Status;
}

/***************************************************************************
 * VAPI_SetTDMParams: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This function is used to configure the four TDM buses of the Mindspeed device. \n
*	This function will configure TDM buses as per the input parameters sent in STdmSetupParam structure. \n
*	This function must be called after boot loading the Comcerto device.\n
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">Device identifier that uniquely identifies a device.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstTdmSetupParam</td>
*		<td style="vertical-align: top;">TDM bus parameters</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\b Returns: \n
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include tdm_param.c
*
*	\b Commands: \n
*	List of Comcerto commands sent:
*	\li TDM_SELECT_BUS_MODE
*	\li SUPVSR_SETUP_TDM_PARAMS
*/
VSTATUS VAPI_SetTDMParams(IN DEVID DevId, IN STdmSetupParams * pstTdmSetupParam, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq;
	SDevice *pstDevice;
	VSTATUS Status;
	Boolean bIsSync;

	Status = VCORE_DeviceGenericCheck(DevId, "VAPI_SetTDMParams", &pstDevice);
	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_SetTDMParams: dev(%u)\n", DevId);

	if (pstTdmSetupParam == NULL)
	{
		UT_ErrorLog(APPITF, "Invalid TDM Parameters \n");
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(STdmSetupParams));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	UT_MemCopy(pstVapiReq->pvUserData, pstTdmSetupParam, sizeof(STdmSetupParams));

	/* Fill the request with the regular parameters */
	VCORE_SetDeviceRequest(pstVapiReq, pstRequest, VFSM_SetTDMParams,	/* pfnReqFSMHdlr */
			      DevId);

	bIsSync = pstVapiReq->bIsSync;

	/* Send VAPI request to VCORE */
	Status = VCORE_ProcessRequest(&(pstDevice->stSupvChnl), pstVapiReq);

	/*If it is a blocking call then free the memory */
	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_SetTDMParams: Exiting status(%d), dev(%u)\n", Status, DevId);

      out:
	return Status;
}

/****************************************************************************
 * VAPI_RegisterReset : The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This API stores the customer device reset function and data pointers to the device structure. \n
*	The stored reset function with the associated data pointer is called by VAPI when a device reset is required (boot, coredump)\n
*	This API must be called after the VAPI_OpenDevice() API.
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">ID of the device to which the reset function is intented for</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pfnReset</td>
*		<td style="vertical-align: top;">Customer specific device reset function</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pvResetData</td>
*		<td style="vertical-align: top;">Customer specific data for the reset function</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*
*	\b Usage: \n
*	\include register_reset.c
*
*	\n \b Commands:
*	\li No Comcerto commands sent.
*/
VSTATUS VAPI_RegisterReset(IN DEVID DevId, IN PFNResetDevice pfnReset, void *pvResetData)
{
	SDevice *pstDev;
	VSTATUS Status;

	if (!bVapiInitialized_g)
	{
		/*Used printf instead of UT_Log because UT is not yet initialized */
		printf("VAPI_RegisterReset: VAPI library not initialized.\n");
		Status = VAPI_ERR_LIB_NOT_INITIALIZED;
		goto out;
	}

	UT_Log(APPITF, INFO, "VAPI_RegisterReset: Entering dev(%u)\n", DevId);

	pstDev = DMGR_GetDevice(DevId);

	if (pstDev == NULL)	
	{
		Status = VAPI_ERR_INVALID_DEVID;
		goto out;
	}

	pstDev->pfnReset = pfnReset;
	pstDev->pvResetData = pvResetData;
	Status = SUCCESS;

	UT_Log(APPITF, INFO, "VAPI_RegisterReset: Exting dev(%u) pfnReset(%x), pvResetData(%x)\n", DevId, pstDev->pfnReset, pstDev->pvResetData);

out:
	return Status;
}

/***************************************************************************
* VAPI_SetDeviceVlan: The function does the following things -
***************************************************************************/
/*!
*	\n \b Description: \n
*	This API programs the VLAN ID in the ethernet header of the device.\n
*	This API doesn't send any command to the device. It just set an internal variable.\n
*	It required to issue this API before the VAPI_SetEthMac() to get the VLAN Id taken in account.\n
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">Device ID.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usVlanId </td>
*		<td style="vertical-align: top;">VLAN ID. If VLAN=0, no VLAN set.</td>
*	</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b usage
*	\include vlan.c
*
*	\n \b Commands:
*	List of Comcerto commands sent:
*	\li NONE
*/
VSTATUS VAPI_SetDeviceVlan(IN DEVID DevId, IN U16 usVlanId, IN SRequest * pstRequest)
{
	SDevice *pstDevice;
	VSTATUS Status;

	UT_Log(APPITF, INFO, "Entered VAPI_SetDeviceVlan: dev(%u)\n", DevId);
	Status = VCORE_DeviceGenericCheck(DevId, "VAPI_SetDeviceVlan", &pstDevice);

	if (Status != SUCCESS)
		goto out;

	if(pstDevice->eEthHdrMode != eMODE_NONE)
	{
		UT_ErrorLog(APPITF, "VAPI_SetDeviceVlan: Ethernet Header already set, VLAN Id not set\n");
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	pstDevice->stEthMacAddresses.usVlanId = usVlanId;
	UT_Log(APPITF, INFO, "VAPI_SetDeviceVlan: Exiting status(%d), devn(%u)\n", Status, DevId);

out:
	/* if ASYNC mode call the callback function*/
	if ((pstRequest != NULL) && (pstRequest->pfnIoCompCallback != NULL))
	{
		pstRequest->pfnIoCompCallback(DevId, CMD_LEVEL_DEVICE, Status, NULL, 0, pstRequest->uiReqId);
		Status = VAPI_ERR_PENDING;
	}

	return Status;
}

/****************************************************************************
* VAPI_CESoPSN_Mapper: The function does the following things -
***************************************************************************/
/*!
*	\n \b Description: \n
*	This API maps / unmaps the a master CESOPSN channel to othe CsoPSN channels \n
*	The MasterConnId must has been created previously in CESOPSN type (0x81). \n
*	In case of eMapType = eChannelMap the other connections must have been created previously in CESOPSN type too \n
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">Device ID.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">MasterConnId </td>
*		<td style="vertical-align: top;">Master connection ID.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eMapType </td>
*		<td style="vertical-align: top;">Map channels or timeslots.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">Action </td>
*		<td style="vertical-align: top;">1: Map channels, 0: Unmap channels.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usConnNum </td>
*		<td style="vertical-align: top;">Number of connections to map.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;"> .... </td>
*		<td style="vertical-align: top;">List of connections ID or timeslots to map (depends on eMapType).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b usage
*	\include cesopsn.c
*
*	\n \b Commands:
*	List of Comcerto commands sent:
*	\li  CESoPSN_MAP_CHANS (if eMapType = eChannelMap)
*	\li  CESoPSN_SPEC_MAP_CHANS (if eMapType = eTSMap)
*/

VSTATUS VAPI_CESoPSN_Mapper(IN DEVID DevId,
				IN SRequest * pstRequest,
				IN EMapperType eMapType,
				IN CONNID MasterConnId,
				U16 Action, U16 usConnNum, ...)
{
	SVapiReq *pstVapiReq;
	SDevice *pstDevice;
	VSTATUS Status;
	Boolean bIsSync;
	va_list ParamList;
	SFsmCesoPsnMapChannelInfo *pstSFsmCesoPsnMapChannelInfo;
	int i;
	SChnl *pstTmpChnl = NULL;
	U16 usTmpId;

	Status = VCORE_DeviceGenericCheck(DevId, "VAPI_CESoPSN_Mapper", &pstDevice);
	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_CESoPSN_Mapper: dev(%u)\n", DevId);

	/* cesopsn feature supported from MR13*/
	if(pstDevice->MajorRelease < MR13)
	{
		UT_ErrorLog(APPITF, "VAPI_CESoPSN_Mapper: Feature not supported CESoPSN map \n");
		Status = VAPI_ERR_UNSUPP_FEATURE;
		goto out;
	}

	/* cesopsn spec channels feature supported from MR15*/
	if((eMapType == eTSMap) && (pstDevice->MajorRelease < MR15))
	{
		UT_ErrorLog(APPITF, "VAPI_CESoPSN_Mapper: Feature not supported CESoPSN spec map\n");
		Status = VAPI_ERR_UNSUPP_FEATURE;
		goto out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SFsmCesoPsnMapChannelInfo) + (usConnNum * sizeof(U16)));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstSFsmCesoPsnMapChannelInfo = pstVapiReq->pvUserData;

	Status = VCORE_ConnectionGenericCheck(MasterConnId, "VAPI_CESoPSN_Mapper", &pstTmpChnl);
	if (Status != SUCCESS)
	{
		UT_ErrorLog(APPITF, "VAPI_CESoPSN_Mapper: Invalid Master Connection ID(%u) \n", MasterConnId);
		goto err_req;
	}

	if (usConnNum > MAX_CSEOPSN_CHANNEL)
	{
		UT_ErrorLog(APPITF, "VAPI_CESoPSN_Mapper: Max channel exeeded(%d) \n", usConnNum);
		Status = VAPI_ERR_INVALID_OPERATION;
		goto err_req;
	}

	pstSFsmCesoPsnMapChannelInfo->MasterChannel = pstTmpChnl->usMSPChnlId;
	pstSFsmCesoPsnMapChannelInfo->Action = Action;
	pstSFsmCesoPsnMapChannelInfo->NumOfChannels = usConnNum;
	pstSFsmCesoPsnMapChannelInfo->MapType = eMapType;

	/* fill the LinkedChannels array with the MSP channel ids comming from ConnIDs */
	UT_va_start(ParamList, usConnNum);
	for (i = 0; i < usConnNum; i++)
	{
	
		/* proceed to some basic verification to check if VAPI, the Device, the connection id are in the rigth state and valid */
		usTmpId = va_arg (ParamList, int);

		/* do not check connection for timeslot mapping (not created*/
		if(pstSFsmCesoPsnMapChannelInfo->MapType == eChannelMap)
		{
			Status = VCORE_ConnectionGenericCheck(usTmpId, "VAPI_CESoPSN_Mapper", &pstTmpChnl);
			if (Status != SUCCESS)
			{
				UT_ErrorLog(APPITF, "VAPI_CESoPSN_Mapper: Invalid Connection ID(%u) \n", usTmpId);
				goto err_req;
			}
	
			if (pstTmpChnl->usConnType != eCSEoPSN)
			{
				UT_ErrorLog(APPITF, "VAPI_CESoPSN_Mapper: Invalid Connection type(%u)\n", pstTmpChnl->usConnType);
				Status = VAPI_ERR_INVALID_OPERATION;
				goto err_req;
			}
		}
		/* if the connection is valid save it*/
		/* eTSMap the id is not a connection id but a timeslot provided by the user*/
		pstSFsmCesoPsnMapChannelInfo->pusLinkedChannels[i] = usTmpId;
		
	}
	UT_va_end(ParamList);

	/* Fill the request with the regular parameters */
	VCORE_SetDeviceRequest(pstVapiReq, pstRequest, VFSM_CESoPSN_Mapper, DevId);

	bIsSync = pstVapiReq->bIsSync;

	/* Send VAPI request to VCORE */
	Status = VCORE_ProcessRequest(&(pstDevice->stSupvChnl), pstVapiReq);

	/*If it is a blocking call then free the memory */
	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_CESoPSN_Mapper: Exiting status(%d), dev(%u)\n", Status, DevId);
/* no error exit */
	return Status;

err_req:
	VCORE_FreeRequest(pstVapiReq);

out:
	return Status;
}

/** @} */ /*device_level*/
