/*! \file device_fsm.c 
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */


#include "ut.h"
#include "vapi.h"
#include "dmgr.h"
#include "vcore.h"
#include "msp.h"
#include "vcore_voip.h"
#include "appitf.h"
#include "hwcfg.h"

/*Global Definitions*/
extern SVAPIConfig stVAPIConfig_g;

void DumpElf(Elf32_Ehdr * pstElfHeader)
{
	UT_Log(VCORE, INFO, "ELF_HEADER (%#x)", sizeof(Elf32_Ehdr));

	UT_Log(VCORE, INFO, "File header information (ELF)");
	UT_Log(VCORE, INFO, "Machine class : %d\n", pstElfHeader->e_ident[EI_CLASS]);
	UT_Log(VCORE, INFO, "Data encoding: %d\n", pstElfHeader->e_ident[EI_DATA]);
	UT_Log(VCORE, INFO, "Machine type: %d\n", UT_CPU2LE16(pstElfHeader->e_machine));
	UT_Log(VCORE, INFO, "File type: %d\n", UT_CPU2LE16(pstElfHeader->e_type));
	UT_Log(VCORE, INFO, "Image Entry point: 0x%08X\n", UT_CPU2LE32(pstElfHeader->e_entry));
	UT_Log(VCORE, INFO, "Program header offset: 0x%08X\n", UT_CPU2LE32(pstElfHeader->e_phoff));
	UT_Log(VCORE, INFO, "Section header offset: 0x%08X\n", UT_CPU2LE32(pstElfHeader->e_shoff));
	UT_Log(VCORE, INFO, "Flags: 0x%08X\n", UT_CPU2LE32(pstElfHeader->e_flags));
	UT_Log(VCORE, INFO, "Header size: 0x%04X\n", UT_CPU2LE16(pstElfHeader->e_ehsize));
	UT_Log(VCORE, INFO, "Program header entry size: 0x%04X\n", UT_CPU2LE16(pstElfHeader->e_phentsize));
	UT_Log(VCORE, INFO, "Program header entries: 0x%04X\n", UT_CPU2LE16(pstElfHeader->e_phnum));
	UT_Log(VCORE, INFO, "Section header entry size: 0x%04X\n", UT_CPU2LE16(pstElfHeader->e_shentsize));
	UT_Log(VCORE, INFO, "Section header entries: 0x%04X\n", UT_CPU2LE16(pstElfHeader->e_shnum));
	UT_Log(VCORE, INFO, "String table section index: 0x%04X\n\n", UT_CPU2LE16(pstElfHeader->e_shstrndx));
}

void DumpAxf(struct _AIF_HEADER *pstAifHdr)
{
	UT_Log(VCORE, INFO, "iCodeBytesToLoad = %d\n", pstAifHdr->ImageReadOnlySize);
	UT_Log(VCORE, INFO, "uiCodeLoadAddr = 0x%08x\n", pstAifHdr->ImageBase);
	UT_Log(VCORE, INFO, "uiCodeStartAddr = 0x%08x\n", pstAifHdr->ImageBase + pstAifHdr->EntryPointOffset);

	UT_Log(VCORE, INFO, "iDataBytesToLoad = %d\n", pstAifHdr->ImageReadWriteSize);
	UT_Log(VCORE, INFO, "uiDataLoadAddr = 0x%08x\n", pstAifHdr->ImageBase + pstAifHdr->ImageReadOnlySize);

}

/* check ELF ARM specific header fields*/
int CheckElf(Elf32_Ehdr * pstElfHeader)
{

	if (pstElfHeader->e_ident[EI_MAG0] != ELFMAG0 ||
	    pstElfHeader->e_ident[EI_MAG1] != ELFMAG1 ||
	    pstElfHeader->e_ident[EI_MAG2] != ELFMAG2 ||
	    pstElfHeader->e_ident[EI_MAG3] != ELFMAG3 ||
	    pstElfHeader->e_ident[EI_CLASS] != ELFCLASS32 ||
	    pstElfHeader->e_ident[EI_DATA] != ELFDATA2LSB || pstElfHeader->e_machine != UT_CPU2LE16(EM_ARM))
	{
		return -1;
	}
	else
		return 0;
}

int Check_BootResponse(IN SChnl * pstChnl, U16 Event, IN gtl_msg_t * pstMsg)
{
	int Status = VAPI_ERR_UNDEFINED;

	if (pstChnl->pstDev->eItfType == ePCI_ITF)
	{
		Status = VCORE_CheckPCIBootStatus(Event, pstMsg);
	}
	else if (pstChnl->pstDev->eItfType == ePOS_ITF)
	{
		Status = VCORE_CheckStatus(Event, pstMsg, 0, CMD_TYPE_BRM_CMD_ACK, CMD_CLASS_ETH_BOOT_MSG, 0);
	}
	else
	{
		Status = VCORE_CheckStatus(Event, pstMsg, 0, CMD_CLASS_ETH_BOOT_MSG, CMD_TYPE_BRM_CMD_ACK, 0);
	}
	return Status;
}

static char *device_name[] = {
	"Mxxxxx",	/*0x00 */
	"M82530",	/*0x01 */
	"M82520",	/*0x02 */
	"M82510",	/*0x03 */
	"M82505",	/*0x04 */
	"M82610",	/*0x05 */
	"Reserved",	/*0x06 */
	"Reserved",	/*0x07 */
	"Reserved",	/*0x08 */
	"M82710",	/*0x09 */
	"M82515",	/*0x0A */
	"M82511",	/*0x0B */
	"M82506",	/*0x0C */
	"M82501",	/*0x0D */
	"M82514",	/*0x0E */
	"M82524",	/*0x0F */
	"M82910",	/*0x10 */
	"M82708",	/*0x11 */
	"M82359",	/*0x12 */
	"M82358",	/*0x13 */
	"M82356",	/*0x14 */
	"M82354",	/*0x15 */
	"M82353",	/*0x16 */
	"M82349",	/*0x17 */
	"M82348",	/*0x18 */
	"M82346",	/*0x19 */
	"M82344",	/*0x1A */
	"M82343",	/*0x1B */
	"M82352",	/*0x1C */
	"M82351",	/*0x1D */
	"M82342",	/*0x1E */
	"M82341",	/*0x1F */
	"M82908",	/*0x20 */
	"Reserved",	/*0x21 */
	"Reserved",	/*0x22 */
	"Reserved",	/*0x23 */
	"Reserved",	/*0x24 */
	"Reserved",	/*0x25 */
	"Reserved",	/*0x26 */
	"Reserved",	/*0x27 */
	"Reserved",	/*0x28 */
	"Reserved",	/*0x29 */
	"Reserved",	/*0x2A */
	"Reserved",	/*0x2B */
	"Reserved",	/*0x2C */
	"Reserved",	/*0x2D */
	"Reserved",	/*0x2E */
	"Reserved",	/*0x2F */
	"M82803",	/*0x30 */
	"M82801",	/*0x31 */
	"M82805",	/*0x32 */
	"M82810",	/*0x33 */
	"M82815",	/*0x34 */
	"M82820"	/*0x35 */
};

#define MAX_DEVICE_NAME	(sizeof(device_name) / sizeof(*device_name))

/* The struct to contain buffer data */
typedef struct _SDownloadBuf {
	S32 		size;
	U32		address;
	U32		offset;
} SDownloadBuf;

/** BootDevice AXF FAT section data */
typedef struct _SBootDevAXFFatSegment {
	SDownloadBuf	buf;
	U32 		uiImageEnd;
	U32 		uiFatOffset;
	U32 		uiCheckSumAddr;
} SBootDevAXFFatSegment;

/** BootDevice AXF Execution data */
typedef struct _SBootDevAXFExecData {
	SDownloadBuf		codeseg;
	SDownloadBuf		dataseg;
	SBootDevAXFFatSegment	fatseg;
} SBootDevAXFExecData;

/** BootDevice Execution data */
typedef struct _SBootDevELFExecData {
	U32 			uiCheckSumAddr;
	U32 			uiStartAddr;
	U8 			SectionNumber;
	U8 			PreviousSectionNumber;
	Elf32_Shdr 		*this_section_headers;
	Elf32_Half 		number_of_section;
	Elf32_Off 		string_section_offset;
	Boolean 		bIsElfFormat;
	Boolean 		bIsBootDevice;
	int 			iFileDesc;
	Elf32_Ehdr 		*pstElfHeader;
} SBootDevELFExecData;

/** BootDevice AXF User data */
typedef struct _SBootDevAXFUserData {
	SBootDevAXFExecData 	ExecData;
	SBootDevUserData 	BootData;
} SBootDevAXFUserData;

/** BootDevice ELF User data */
typedef struct _SBootDevELFUserData {
	SBootDevELFExecData 	ExecData;
	SBootDevUserData 	BootData;
} SBootDevELFUserData;

/* The struct to contain buffer data */
typedef struct _SDownloadBufUserData {
	SBootDevAXFFatSegment	DownloadBuf;
	SBootDevUserData  	BootData;
} SDownloadBufUserData;

static void* VCORE_GetImageBuf(IN SBootDevUserData * pvUserData, IN int offset, IN int size)
{
	VSTATUS Status = SUCCESS;
	void *buf = NULL;
	int iFd = -1;

	UT_Log(VCORE, DEBUG, "Entered VCORE_GetImageBuf offset = (%d), size = (%d)\n", offset, size);

	/*Initialize execution data memory */
	if (pvUserData == NULL)
	{
		Status = VAPI_ERR_INVALID_PARAM;
		return NULL;
	}	

	if (!pvUserData->pvImageFile) 
	{
		buf = UT_AllocMem(size);
		if (buf == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			return NULL;
		}		

		/*Initialize execution data */
		iFd = UT_Open(pvUserData->strFileName, O_RDONLY, 0444);

		if (iFd < 0)
		{
			UT_ErrorLog(VCORE, "VCORE_GetImageBuf: MSP image file could not be opened %s\n", pvUserData->strFileName);
			Status = VAPI_ERR_INVALID_PARAM;
			goto err;
		}

		if ((pvUserData->uImageSize = UT_Lseek(iFd, 0L, SEEK_END)) < 0)
		{
			UT_ErrorLog(VCORE, "VCORE_GetImageBuf: File length invalid\n");
			Status = VAPI_ERR_INVALID_PARAM;
			goto err;
		}

		if (pvUserData->uImageSize < offset + size)
		{
			UT_ErrorLog(VCORE, "VCORE_GetImageBuf: code image too small");
			Status = VAPI_ERR_INVALID_IMAGE;
			goto err;
		}

		if ((UT_Lseek(iFd, offset, SEEK_SET)) != offset)
		{
			UT_ErrorLog(VCORE,"VCORE_GetImageBuf: Pointer could not be reset\n");
			Status = VAPI_ERR_INVALID_PARAM;
			goto err;
		}		

		if (UT_Read(iFd, (char *) buf, size) != size)
		{
			UT_ErrorLog(VCORE, "VCORE_GetImageBuf: Error in reading MSP image file\n");
			Status = VAPI_ERR_INVALID_PARAM;
			goto err;
		}

		UT_CloseFd(iFd);	
		UT_Log(VCORE, DEBUG,  "VCORE_GetImageBuf: Exiting status(%d)\n", Status);
		return buf;

err:
		UT_FreeMem(buf);
		UT_CloseFd(iFd);
		UT_ErrorLog(VCORE,"VCORE_GetImageBuf: Exiting status(%d)\n", Status);
		return NULL;		

	}
	else 
	{
		UT_Log(VCORE, DEBUG,  "VCORE_GetImageBuf: Exiting status(%d)\n", Status);
     		return pvUserData->pvImageFile + offset;		
	}	

}

void VCORE_PutImageBuf(IN void *pvUserData, void *buf)
{
	UT_Log(VCORE, DEBUG, "Entered VCORE_PutImageBuf\n");
	if (!((SBootDevUserData *) pvUserData)->pvImageFile)
	{
		if (buf != NULL)
			UT_FreeMem(buf);
	}
	UT_Log(VCORE, DEBUG, "VCORE_PutImageBuf: Exit\n");
}

VSTATUS VFSM_DownloadBuf(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	SDownloadBufUserData *pstUserData;
	SDownloadBuf *pstBuf = NULL;
	U16 usCount = 0;
	char *pstrLoadBuf;
	DEVID DevId = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_DownloadBuf: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	pstUserData = (SDownloadBufUserData *) pstVapiReq->pvUserData;
	pstBuf = (SDownloadBuf *) &pstUserData->DownloadBuf.buf;
	
	switch (pstVapiReq->usReqState)
	{
	case FSM_STATE_BUF_DOWNLOAD:
		/* we keep in this state until we have data to send */
		if (pstBuf->size > 0)
		{
			if (pstBuf->size > pstChnl->pstDev->uMaxFrameSize)
				usCount = pstChnl->pstDev->uMaxFrameSize;
			else
			{
				usCount = pstBuf->size;
				pstVapiReq->usReqState = FSM_STATE_BUF_FINISH_DOWNLOAD;
			}
			
			pstrLoadBuf = VCORE_GetImageBuf(&pstUserData->BootData, pstBuf->offset, usCount);

			if (pstChnl->pstDev->eItfType == ePOS_ITF)
				Status = VDEV_POSDownload(pstChnl, (void *) pstrLoadBuf, usCount, pstBuf->address);

			else
				Status = VDEV_CsmDownload(pstChnl, (void *) pstrLoadBuf, usCount, pstBuf->address);

			if (Status == SUCCESS)
			{
				pstBuf->size -= usCount;
				pstBuf->offset += usCount;
				pstBuf->address += usCount;
			}
			else
			{
				UT_ErrorLog(VCORE, "Issue with BRM Write\n");
				Status = VAPI_ERR_INVALID_IMAGE;
				VCORE_PutImageBuf (&pstUserData->BootData, pstrLoadBuf);
				goto finish;
				
			}

			VCORE_PutImageBuf (&pstUserData->BootData, pstrLoadBuf);
		}
		else
		{
			UT_ErrorLog(VCORE, "Code size 0");
			Status = VAPI_ERR_INVALID_IMAGE;
			goto finish;
		}

		break;

	case FSM_STATE_BUF_FINISH_DOWNLOAD:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_DownloadBuf: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}	

	return SUCCESS;

finish:
	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_DownloadBuf: Error dev(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
			DevId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_DownloadBuf: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_DownloadBuf: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;	
}

VSTATUS VFSM_DownloadCode(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	SVapiReq * pstChildVapiReq;
	SDownloadBufUserData  *pstDownloadCode, *pstDownloadBufUserData;
	DEVID DevId = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_DownloadCode: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	pstDownloadCode = pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case FSM_STATE_CODE_DOWNLOAD:

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SDownloadBufUserData));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* Initialise the UserData (allocated by VCORE_AllocateRequest) */
		pstDownloadBufUserData = pstChildVapiReq->pvUserData;
		UT_MemCopy(pstDownloadBufUserData, pstDownloadCode, sizeof(SDownloadBufUserData));

		/* initialise the Child request */ 
		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
				pstVapiReq,			/* Parent Request */ 
				VFSM_DownloadBuf,		/* Child Request Handler */
				FSM_STATE_BUF_DOWNLOAD);	/* Child Request handler state */

		pstVapiReq->usReqState = FSM_STATE_CODE_FINISH_DOWNLOAD;

		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case FSM_STATE_CODE_FINISH_DOWNLOAD:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_DownloadCode: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:
	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_DownloadCode: Error dev(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
			DevId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_DownloadCode: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_DownloadCode: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

VSTATUS VFSM_DownloadData(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	SVapiReq * pstChildVapiReq;
	SDownloadBufUserData  *pstDownloadData, *pstDownloadBufUserData;
	DEVID DevId;
	DevId = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_DownloadData: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	pstDownloadData = pstVapiReq->pvUserData;
	
	switch (pstVapiReq->usReqState)
	{
	case FSM_STATE_DATA_DOWNLOAD:

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SDownloadBufUserData));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* Initialise the UserData (allocated by VCORE_AllocateRequest) */
		pstDownloadBufUserData = pstChildVapiReq->pvUserData;
		UT_MemCopy(pstDownloadBufUserData, pstDownloadData, sizeof(SDownloadBufUserData));

		/* initialise the Child request */ 
		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
				pstVapiReq,			/* Parent Request */ 
				VFSM_DownloadBuf,		/* Child Request Handler */
				FSM_STATE_BUF_DOWNLOAD);	/* Child Request handler state */

		pstVapiReq->usReqState = FSM_STATE_DATA_FINISH_DOWNLOAD;

		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

 	case FSM_STATE_DATA_FINISH_DOWNLOAD:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_DownloadData: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;
	
finish:
	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_DownloadData: Error dev(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
			pstChnl->pstDev->DevId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_DownloadData: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_DownloadData: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

VSTATUS VFSM_DownloadFAT(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	SDownloadBufUserData *pstDownloadFAT, *pstDownloadBufUserData;
	SBootDevAXFFatSegment *pstExecFATBuf;
	struct _FAT_AIF_HEADER *pstFatHdr = NULL;
	U32 CurrentOffset;
	U32 *pstCheckSumAddr;
	SVapiReq * pstChildVapiReq;
	DEVID DevId = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_DownloadFAT: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	pstDownloadFAT = pstVapiReq->pvUserData;
	pstExecFATBuf = &pstDownloadFAT->DownloadBuf;

	/* At 1st pass this pointer is initialized to FirstFatOffset*/
	CurrentOffset = pstExecFATBuf->uiFatOffset;

	switch (pstVapiReq->usReqState)
	{
	case FSM_STATE_FAT_DOWNLOAD:

		while (CurrentOffset != 0 && pstExecFATBuf->buf.size == 0)
		{
			if (CurrentOffset + sizeof(struct _FAT_AIF_HEADER) > pstExecFATBuf->uiImageEnd)
			{
				Status = VAPI_ERR_INVALID_IMAGE;
				goto finish;
			}

			pstFatHdr  = VCORE_GetImageBuf(&pstDownloadFAT->BootData, CurrentOffset, sizeof(struct _FAT_AIF_HEADER));			

			pstFatHdr->NextFatOffset = UT_LE2CPU32(pstFatHdr->NextFatOffset);
			pstFatHdr->LoadAddress = UT_LE2CPU32(pstFatHdr->LoadAddress);
			pstFatHdr->Size = UT_LE2CPU32(pstFatHdr->Size);
			UT_Log(VCORE, INFO, "Section name %s\n", pstFatHdr->region_name);

			pstExecFATBuf->buf.size = pstFatHdr->Size;
			pstExecFATBuf->buf.address= pstFatHdr->LoadAddress;
			pstExecFATBuf->buf.offset= CurrentOffset + sizeof(*pstFatHdr);

			UT_Log(VCORE, INFO, "CurrentOffset = %d\n", CurrentOffset);
			UT_Log(VCORE, INFO, "iFatBytesToLoad = %d\n", pstExecFATBuf->buf.size);
			UT_Log(VCORE, INFO, "uiFatLoadAddr = %d\n", pstExecFATBuf->buf.address);
			UT_Log(VCORE, INFO, "uiFatImageOffset = %d\n", pstExecFATBuf->buf.offset);

			if (UT_StrCmp((char *)pstFatHdr->region_name, "CHECKSUM") == 0)
			{
				/* Read value from the download file. */
				pstCheckSumAddr = VCORE_GetImageBuf(&pstDownloadFAT->BootData, pstExecFATBuf->buf.offset, sizeof(U32));
				pstExecFATBuf->uiCheckSumAddr = UT_LE2CPU32(*pstCheckSumAddr);
				VCORE_PutImageBuf(&pstDownloadFAT->BootData, pstCheckSumAddr);
				
				UT_Log(VCORE, INFO, "Check Sum Address = 0x%X , Actual = 0x%X size = %d\n",
				pstExecFATBuf->uiCheckSumAddr, pstFatHdr->LoadAddress, pstFatHdr->Size);
				
				((SBootDevExecData *)pstVapiReq->pstParentReq->pstParentReq->pvExecData)->uiCheckSumAddr = pstExecFATBuf->uiCheckSumAddr;
			}
			
			Status = SUCCESS;

			if (pstExecFATBuf->buf.offset + pstExecFATBuf->buf.size > pstExecFATBuf->uiImageEnd)
			{
				Status = VAPI_ERR_INVALID_IMAGE;
				VCORE_PutImageBuf(&pstDownloadFAT->BootData, pstFatHdr);
				goto finish;
			}

			pstExecFATBuf->uiFatOffset = pstFatHdr->NextFatOffset;
			CurrentOffset = pstFatHdr->NextFatOffset;
			VCORE_PutImageBuf(&pstDownloadFAT->BootData, pstFatHdr);
		}

		/* skip section with zero size */
		if (pstExecFATBuf->buf.size != 0)
		{
			/* this allocate a new request (no User Data required) */ 
			pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SDownloadBufUserData));
			if (!pstChildVapiReq)
			{
				Status = VAPI_ERR_NOMEM;
				goto finish;
			}

			/* Initialise the UserData (allocated by VCORE_AllocateRequest) */
			pstDownloadBufUserData = pstChildVapiReq->pvUserData;
			UT_MemCopy(pstDownloadBufUserData, &pstExecFATBuf->buf, sizeof(SDownloadBufUserData));

			/* initialise the Child request */ 
			VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
					pstVapiReq,			/* Parent Request */ 
					VFSM_DownloadBuf,		/* Child Request Handler */
					FSM_STATE_BUF_DOWNLOAD);	/* Child Request handler state */

			/*add it at the front of the request list of the channel
			and call the ProcessRequestList function to process this child request first.*/ 
			VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);
	
			pstExecFATBuf->buf.size = 0;

			/* retrieve the child status */
			Status = pstVapiReq->Status;

			if (Status != SUCCESS)
				goto finish;
		}
		else
		{
			goto finish;
		}

		if (CurrentOffset == 0)
			pstVapiReq->usReqState = FSM_STATE_FAT_DOWNLOAD_FINISH;

		break;

	case FSM_STATE_FAT_DOWNLOAD_FINISH:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_DownloadFAT: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:
	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_DownloadFAT: Error dev(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
			DevId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_DownloadFAT: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_DownloadFAT: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

VSTATUS VFSM_AxfDownload(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{	
	STATUS Status = SUCCESS;
	SVapiReq * pstChildVapiReq;
	struct _AIF_HEADER *pstAifHdr = NULL;
	SBootDevAXFUserData * pvUserData;
	struct _SDownloadBufUserData* pstDownloadBufUserData;
	DEVID DevId = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_AxfDownload: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	pvUserData = (SBootDevAXFUserData *)pstVapiReq->pvUserData;
		
	switch (pstVapiReq->usReqState)
	{
	case FSM_STATE_AXF_CODE_DOWNLOAD:
		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SDownloadBufUserData));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* Initialise the UserData (allocated by VCORE_AllocateRequest) */
		pstDownloadBufUserData = pstChildVapiReq->pvUserData;
		UT_MemCopy(&pstDownloadBufUserData->DownloadBuf, &(pvUserData->ExecData.codeseg), sizeof(SDownloadBuf));
		UT_MemCopy(&pstDownloadBufUserData->BootData, &pvUserData->BootData, sizeof(SBootDevUserData));		
		
		/* initialise the Child request */
		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
				pstVapiReq,			/* Parent Request */ 
				VFSM_DownloadCode,		/* Child Request Handler */
				FSM_STATE_CODE_DOWNLOAD);	/* Child Request handler state */

		/* diag firmware has data_segment set to 0 so skip the data download state */
		pstAifHdr = VCORE_GetImageBuf(&pvUserData->BootData, 0, sizeof(struct _AIF_HEADER));
		if (pstAifHdr->ImageReadWriteSize)
			pstVapiReq->usReqState = FSM_STATE_AXF_DATA_DOWNLOAD;
		else
			pstVapiReq->usReqState  = FSM_STATE_AXF_FAT_DOWNLOAD;

		VCORE_PutImageBuf(&pvUserData->BootData, pstAifHdr);

		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case FSM_STATE_AXF_DATA_DOWNLOAD:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SDownloadBufUserData));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* Initialise the UserData (allocated by VCORE_AllocateRequest) */
		pstDownloadBufUserData = pstChildVapiReq->pvUserData;
		UT_MemCopy(&pstDownloadBufUserData->DownloadBuf, &pvUserData->ExecData.dataseg, sizeof(SDownloadBuf));
		UT_MemCopy(&pstDownloadBufUserData->BootData, &pvUserData->BootData, sizeof(SBootDevUserData));		

		/* initialise the Child request */ 
		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
				pstVapiReq,			/* Parent Request */ 
				VFSM_DownloadData,		/* Child Request Handler */
				FSM_STATE_DATA_DOWNLOAD);	/* Child Request handler state */

		pstVapiReq->usReqState  = FSM_STATE_AXF_FAT_DOWNLOAD;

		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;
		
	case FSM_STATE_AXF_FAT_DOWNLOAD:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;
		
		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SDownloadBufUserData));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* Initialise the UserData (allocated by VCORE_AllocateRequest) */
		pstDownloadBufUserData = (SDownloadBufUserData *) pstChildVapiReq->pvUserData;
		UT_MemCopy(&pstDownloadBufUserData->DownloadBuf, &pvUserData->ExecData.fatseg, sizeof(SBootDevAXFFatSegment));
		UT_MemCopy(&pstDownloadBufUserData->BootData, &pvUserData->BootData, sizeof(SBootDevUserData));				
		
		/* initialise the Child request  */ 
		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
				pstVapiReq,			/* Parent Request */ 
				VFSM_DownloadFAT,		/* Child Request Handler */
				FSM_STATE_FAT_DOWNLOAD);	/* Child Request handler state */

		pstVapiReq->usReqState = FSM_STATE_AXF_FINISH_DOWNLOAD;

		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;
		
	case FSM_STATE_AXF_FINISH_DOWNLOAD:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_AxfDownload: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:
	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_AxfDownload: Error dev(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
			DevId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_AxfDownload: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_AxfDownload: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

VSTATUS VFSM_ElfDownload(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{	
	STATUS Status = SUCCESS;
	Elf32_Ehdr *pstElfHeader=NULL;
	SBootDevELFUserData *pstELFUserData = NULL;
	SBootDevELFExecData *pstELFExecData = NULL;
	char section_name[40];
	char *pstsection_name;
	U32 *pstCheckSumAddr;
	SVapiReq * pstChildVapiReq;
	SDownloadBufUserData *pstDownloadBufUserData;
	int i;
	DEVID DevId = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_ElfDownload: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	switch (pstVapiReq->usReqState)
	{
	case FSM_STATE_ELF_CODE_DOWNLOAD:

		pstELFUserData = pstVapiReq->pvUserData;
		pstELFExecData = &pstELFUserData->ExecData;

		pstElfHeader = VCORE_GetImageBuf(&pstELFUserData ->BootData, 0, sizeof(Elf32_Ehdr));

		/* Figure out how many sections we have in this file */
		pstELFExecData->number_of_section = UT_CPU2LE16(pstElfHeader->e_shnum);
		
		/* Retrieve the pointer to the Section header.  */
		pstELFExecData->this_section_headers = VCORE_GetImageBuf(&pstELFUserData->BootData, UT_CPU2LE32(pstElfHeader->e_shoff), sizeof(Elf32_Shdr) * pstELFExecData->number_of_section);
		pstELFExecData->string_section_offset = UT_CPU2LE32(pstELFExecData->this_section_headers[UT_CPU2LE16(pstElfHeader->e_shstrndx)].sh_offset);
		
		/* save the progstart address */
		pstELFExecData->uiStartAddr = UT_CPU2LE32(pstElfHeader->e_entry);

		VCORE_PutImageBuf(&pstELFUserData ->BootData, pstElfHeader);

		/* Process if the number of section is not 0 else error */
		if (pstELFExecData->number_of_section > 0)
		{
			/* pstELFExecData->SectionNumber has been initialised to 0 in FSM_STATE_BOOT_INIT
			   it is also incremented in this loop */
			/* Start from the latest processed section
			   In this loop we are checking if the section is ok to download
			   If the current section is not to be downloaded, no BRM command is sent to the device.
			   We just jump to the next section */
			/* if we section has data to be downloaded a brm-write is sent to the device.
			   In this case we break the for loop, and get back to this state on response to brm write
			   received from the device */
			for (i = pstELFExecData->SectionNumber; i < pstELFExecData->number_of_section; ++i)
			{
				/* Check if it is a section we want to process 
				   (if not continue with next section) */
				if ((UT_CPU2LE32((pstELFExecData->this_section_headers[pstELFExecData->SectionNumber].sh_flags)) != SHF_MIPS_ADDR)
				    && (UT_CPU2LE32((pstELFExecData->this_section_headers[pstELFExecData->SectionNumber].sh_flags)) != 0))
				{
					pstsection_name = VCORE_GetImageBuf(&pstELFUserData->BootData, (pstELFExecData->string_section_offset +
										UT_CPU2LE32(pstELFExecData->
										this_section_headers[pstELFExecData->SectionNumber].
										sh_name)),40);
					
					UT_MemCopy(section_name, pstsection_name, 40);
					VCORE_PutImageBuf (&pstELFUserData->BootData, pstsection_name);

					/* For Checksum section no download is processed
					   so save the address of the checksum prog and parse the next section */
					if (UT_StrCmp(section_name, "CHECKSUM") == 0)
					{
						pstCheckSumAddr = VCORE_GetImageBuf(&pstELFUserData->BootData, UT_CPU2LE32(pstELFExecData->
										this_section_headers[pstELFExecData->SectionNumber].
										sh_offset), sizeof(U32));
			
						pstELFExecData->uiCheckSumAddr = UT_LE2CPU32(*pstCheckSumAddr);
						VCORE_PutImageBuf(&pstELFUserData->BootData, pstCheckSumAddr);

						UT_Log(VCORE, INFO, "checksum prog at 0x%08x \n",
						       pstELFExecData->uiCheckSumAddr);
						pstELFExecData->PreviousSectionNumber = pstELFExecData->SectionNumber;
						pstELFExecData->SectionNumber++;
						continue;
					}

					/* Check if the section is valid for us */
					if (((UT_CPU2LE32((pstELFExecData->this_section_headers[pstELFExecData->SectionNumber].sh_type)) & 3) == SHT_PROGBITS)
					   && (pstELFExecData->this_section_headers[pstELFExecData->SectionNumber].sh_size != 0))
					{
						/* this allocate a new request */ 
						pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SDownloadBufUserData));
						

						if (!pstChildVapiReq)
						{
							Status = VAPI_ERR_NOMEM;
							goto finish_err;
						}

						/* Initialise the UserData (allocated by VCORE_AllocateRequest) */
						pstDownloadBufUserData = pstChildVapiReq->pvUserData;

						pstDownloadBufUserData->DownloadBuf.buf.size = UT_CPU2LE32(pstELFExecData->this_section_headers[pstELFExecData->SectionNumber].sh_size);
						pstDownloadBufUserData->DownloadBuf.buf.address = UT_CPU2LE32(pstELFExecData->this_section_headers[pstELFExecData->SectionNumber].sh_addr);
						pstDownloadBufUserData->DownloadBuf.buf.offset = UT_CPU2LE32(pstELFExecData->this_section_headers[pstELFExecData->SectionNumber].sh_offset);
						UT_MemCopy(&pstDownloadBufUserData->BootData, &pstELFUserData->BootData, sizeof(SBootDevUserData));	
							
						/* initialise the Child request */ 
						VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
								pstVapiReq,			/* Parent Request */ 
								VFSM_DownloadBuf,		/* Child Request Handler */
								FSM_STATE_BUF_DOWNLOAD);	/* Child Request handler state */

						/*add it at the front of the request list of the channel
						and call the ProcessRequestList function to process this child request first.*/ 
						VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

						/* retrieve the child status */
						Status = pstVapiReq->Status;

						if (Status != SUCCESS)
							goto finish_err;

						pstELFExecData->SectionNumber++;

						break;	/* exit from for loop */
					}
				}

				UT_Log(VCORE, INFO, "Section #%d not processed\n", pstELFExecData->SectionNumber);
				pstELFExecData->SectionNumber++;

				/* if we get here with max number of section, send the appropriate
				   checksun command and go to the next state depending 
				   the control interface */
				if (pstELFExecData->SectionNumber == pstELFExecData->number_of_section)
				{
					((SBootDevExecData *)pstVapiReq->pstParentReq->pvExecData)->uiCheckSumAddr = pstELFExecData->uiCheckSumAddr;
					((SBootDevExecData *)pstVapiReq->pstParentReq->pvExecData)->uiStartAddr = pstELFExecData->uiStartAddr;
					pstVapiReq->usReqState = FSM_STATE_ELF_FINISH_DOWNLOAD;
					goto finish_err;
				}

				continue;
			}
		}
		else
		{
			UT_ErrorLog(VCORE, "Number of section invalid\n");
			Status = VAPI_ERR_INVALID_IMAGE;
			goto finish_err;
		}

		VCORE_PutImageBuf(&pstELFUserData->BootData, pstELFExecData->this_section_headers);

		break;
	
	case FSM_STATE_ELF_FINISH_DOWNLOAD:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		goto finish;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_ElfDownload: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish_err:
	VCORE_PutImageBuf(&pstELFUserData->BootData, pstELFExecData->this_section_headers);
	
finish:
	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_ElfDownload: Error dev(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
			DevId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_ElfDownload: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_ElfDownload: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}
 
/* This is internal VAPI function and should be called from VFSM_BootDevice(), VFSM_BootDeviceFile() and
 * VFSM_CoreDump() functions to copy user HW params for the chip or use default.
 */
void SetVAPIDeviceHwConfig(OUT SHwCfg *pstSHwCfg, IN SHwCfg *pstUserSHwCfg, IN struct _SDevice *pstDev)
{
	SVAPIDeviceConfig *pstDevConfig = DMGR_GetVAPIDeviceConfig(pstDev->ucDevType);
	SHwCfg *pstDefaultHwCfg = &pstDevConfig->stSHwCfg;

	if (pstUserSHwCfg == NULL)
	{
		/* nothing special from user - using defaults */
		UT_MemCopy(pstSHwCfg, pstDefaultHwCfg, sizeof(*pstSHwCfg));
		return;
	}

	/* pass user-supplied parameters to out structure */
	UT_MemCopy(pstSHwCfg, pstUserSHwCfg, sizeof(SHwCfg));

	/* user may pass only the part of configuration, zero value means all fields are passed (backward compatibility) */
	if (pstSHwCfg->usHwParamsMask != 0)
	{
		if ((pstSHwCfg->usHwParamsMask & VAPI_HWCFG_SDRAM_TIMING_MASK) == 0)
		{
			pstSHwCfg->usSdramParamTiming1 = pstDefaultHwCfg->usSdramParamTiming1;
			pstSHwCfg->usSdramParamTiming2 = pstDefaultHwCfg->usSdramParamTiming2;
		}

		if ((pstSHwCfg->usHwParamsMask & VAPI_HWCFG_SDRAM_REFRESH_MASK) == 0)
			pstSHwCfg->usSdramRefresh = pstDefaultHwCfg->usSdramRefresh;

		if ((pstSHwCfg->usHwParamsMask & VAPI_HWCFG_SDRAM_POWERON_MASK) == 0)
			pstSHwCfg->usSdramPoweron = pstDefaultHwCfg->usSdramPoweron;

		if ((pstSHwCfg->usHwParamsMask & VAPI_HWCFG_CS_TIMING_MASK) == 0)
			pstSHwCfg->usCSparamTiming = pstDefaultHwCfg->usCSparamTiming;

		if ((pstSHwCfg->usHwParamsMask & VAPI_HWCFG_REFCLK_MASK) == 0)
			pstSHwCfg->usDeviceRefclk = pstDefaultHwCfg->usDeviceRefclk;

		if ((pstSHwCfg->usHwParamsMask & VAPI_HWCFG_SDRAM_PARAMS) == 0)
		{
			pstSHwCfg->ptrSdramParam_C300_2 = pstDefaultHwCfg->ptrSdramParam_C300_2;
			pstSHwCfg->usSdramParamNum = pstDefaultHwCfg->usSdramParamNum;
		}

		if ((pstSHwCfg->usHwParamsMask & VAPI_HWCFG_M823V2_HIGH_SPEED_MASK) == 0)
			pstSHwCfg->usDeviceAmbaClkM823V2 = pstDefaultHwCfg->usDeviceAmbaClkM823V2;

	}
}

/****************************************************************************
 * VFSM_BootDevice : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  All the commands use Supervisory channel structure, this is safe
 *          because, there will not be any supervisory commands until it gets
 *          created.
 *      -#  When a CMD_ACK is received, VCORE_ProcessResponse will not get any
 *          device corresponding to channel 0, but it should check the state of
 *          the device and if it is booting state then it should pass the 
 *          CMD_ACK to the Supervisory channel.
 *      -#  It does the following
 *          -#  Send a PHYID_ASSIGN if the interfce is POS.
 *          -#  Initialize hardware (Clock,CS,SDRAM etc)
 *          -#  Download various sections of image (Code,Data,FAT)
 *          -#  Send PROGSTART.
 *          -#  Do request completion processing when Supervisor Ready is 
 *              received.
 *  
 *  \return 
 *  None
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_BootDevice(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	struct _AIF_HEADER *pstAifHdr = NULL;
	SBootDevExecData *pstExecData;
	SVapiReq * pstChildVapiReq;
	Elf32_Ehdr *pstElfHeader = NULL;
	U16 device_type;
	SBootDevAXFUserData *pstAXFUserData;
	SBootDevELFUserData *pstElfUserData;
	DEVID DevId = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_BootDevice: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	/*Initialize execution data memory */
	if (pstVapiReq->pvUserData == NULL)
	{
		Status = VAPI_ERR_INVALID_PARAM;
		goto finish;
	}

	switch (pstVapiReq->usReqState)
	{
	case FSM_STATE_BOOT_INIT:
		pstVapiReq->pvExecData = UT_Calloc(1, sizeof(SBootDevExecData));
		if (pstVapiReq->pvExecData == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/*Set Device state to booting */
		pstChnl->pstDev->eState = eBOOTING;

		/*Initialize execution data */
		pstExecData = pstVapiReq->pvExecData;
		
		/* we assume that the buffer contains a ELF format file */
		pstElfHeader  = VCORE_GetImageBuf(pstVapiReq->pvUserData, 0, sizeof(Elf32_Ehdr ));

		/* we check if it is an ELF file for ARM or an AXF */
		if (CheckElf(pstElfHeader) == 0)
		{
			if (((SBootDevUserData *) pstVapiReq->pvUserData)->uImageSize < sizeof(Elf32_Ehdr))
			{
				UT_ErrorLog(VCORE, "code image too small");
				Status = VAPI_ERR_INVALID_IMAGE;
				VCORE_PutImageBuf (pstVapiReq->pvUserData, pstElfHeader);
				goto finish;
			}
			/* Yes it is a ELF file */
			pstExecData->bIsElfFormat = 1;
			pstExecData->SectionNumber = 0;
			pstExecData->PreviousSectionNumber = -1;
			DumpElf(pstElfHeader);
			VCORE_PutImageBuf (pstVapiReq->pvUserData, pstElfHeader);
		}
		else
		{
			pstAifHdr = VCORE_GetImageBuf(pstVapiReq->pvUserData, 0, sizeof(struct _AIF_HEADER));

			pstExecData->bIsElfFormat = 0;

			/* set header according the host endianess */
			VCORE_ProcessAifHeader(pstAifHdr);

			/*Initialize execution data for Code download */
			UT_Log(VCORE, INFO, "VCORE_Boot: Initializing Exec data Code area\n");
			pstExecData->iCodeBytesToLoad = pstAifHdr->ImageReadOnlySize;
			pstExecData->uiCodeLoadAddr = pstAifHdr->ImageBase;
			pstExecData->uiCodeImageOffset = sizeof(*pstAifHdr);

			pstExecData->uiImageEnd = ((SBootDevUserData *) pstVapiReq->pvUserData)->uImageSize;

			pstExecData->uiStartAddr = pstAifHdr->ImageBase + pstAifHdr->EntryPointOffset;

			if (pstExecData->uiCodeImageOffset + pstExecData->iCodeBytesToLoad > pstExecData->uiImageEnd)
			{
				Status = VAPI_ERR_INVALID_IMAGE;
				VCORE_PutImageBuf (pstVapiReq->pvUserData, pstAifHdr);
				goto finish;
			}
			/*Initialize execution data for Data download */
			UT_Log(VCORE, INFO, "VCORE_Boot: Initializing Exec data Data area\n");
			pstExecData->iDataBytesToLoad = pstAifHdr->ImageReadWriteSize;
			pstExecData->uiDataLoadAddr = pstAifHdr->ImageBase + pstAifHdr->ImageReadOnlySize;
			pstExecData->uiDataImageOffset = sizeof(*pstAifHdr) + pstAifHdr->ImageReadOnlySize;

			if (pstExecData->uiDataImageOffset + pstExecData->iDataBytesToLoad > pstExecData->uiImageEnd)
			{
				Status = VAPI_ERR_INVALID_IMAGE;
				goto finish;
			}
			/*Initialize execution data for FAT Download */
			UT_Log(VCORE, INFO, "VCORE_Boot: Initializing Exec data FAT area\n");
			pstExecData->uiFatOffset = pstAifHdr->FirstFatOffset;

			if (pstExecData->uiFatOffset + sizeof(struct _FAT_AIF_HEADER) >
			    pstExecData->uiImageEnd)
			{
				Status = VAPI_ERR_INVALID_IMAGE;
				VCORE_PutImageBuf (pstVapiReq->pvUserData, pstAifHdr);
				goto finish;
			}

			DumpAxf(pstAifHdr);
			VCORE_PutImageBuf (pstVapiReq->pvUserData, pstAifHdr);
		}		/* end of if ELF /AXF */

		/* Reset the device if not in reset 
		for sCSM_ITF the GTL_Reset is done in VAPI_AssignBootMac*/
		if (pstChnl->pstDev->eState != eUNINITIALIZED && pstChnl->pstDev->eItfType != eCSM_ITF)
		{
			Status = VCORE_ResetDevice(pstChnl->pstDev);
			if (Status != SUCCESS)
				goto finish;
		}

		/* this allocate a new request */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SHwCfg));

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		SetVAPIDeviceHwConfig((SHwCfg *) pstChildVapiReq->pvUserData, ((SBootDevUserData *)(pstVapiReq->pvUserData))->pstSHwCfg, pstChnl->pstDev);

		/* initialise the Child request */
		if (pstChnl->pstDev->ucDevType == DEV_TYPE_M823XX || pstChnl->pstDev->ucDevType == DEV_TYPE_M823XX_2)
		{
			VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
					pstVapiReq,			/* Parent Request */ 
					VFSM_SetDeviceHardware300,	/* Child Request Handler */
					DEVICE_HARDWARE_300_INIT);	/* Child Request handler state */
		}
		else if (pstChnl->pstDev->ucDevType == DEV_TYPE_M829XX)
		{
			VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
					pstVapiReq,			/* Parent Request */ 
					VFSM_SetDeviceHardware900,	/* Child Request Handler */
					DEVICE_HARDWARE_900_INIT);	/* Child Request handler state */
		}	
		else
		{
			VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
					pstVapiReq,			/* Parent Request */ 
					VFSM_SetDeviceHardware,		/* Child Request Handler */
					DEVICE_HARDWARE_INIT);		/* Child Request handler state */
		}

		if (pstExecData->bIsElfFormat)
			pstVapiReq->usReqState = FSM_STATE_BOOT_DO_ELF_SECTION_DOWNLOAD;
		else
			pstVapiReq->usReqState = FSM_STATE_BOOT_DO_AXF_SECTION_DOWNLOAD;

		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

		/* we got here from 2 different places
		   1 When we receive the response to VDEV_BRMSetCSParam  
		   2 When we loop in the download phase */

	case FSM_STATE_BOOT_DO_ELF_SECTION_DOWNLOAD:

		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SBootDevELFUserData));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* Initialise the UserData (allocated by VCORE_AllocateRequest) */
		pstElfUserData = pstChildVapiReq->pvUserData;
		UT_MemCopy(&pstElfUserData->ExecData, (U8 *) pstVapiReq->pvExecData + sizeof(SBootDevAXFExecData), sizeof(SBootDevELFExecData));
		UT_MemCopy(&pstElfUserData->BootData, pstVapiReq->pvUserData, sizeof(SBootDevUserData));

		/* initialise the Child request */ 
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */ 
				VFSM_ElfDownload,	/* Child Request Handler */
				FSM_STATE_ELF_CODE_DOWNLOAD);	/* Child Request handler state */

		if (pstChnl->pstDev->eItfType == ePOS_ITF)
			pstVapiReq->usReqState = FSM_STATE_BOOT_DO_RUN_CHECKSUM;
		else
			pstVapiReq->usReqState = FSM_STATE_BOOT_DO_CHECKSUM;

		/* process request in the supervisor channel */
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

		/*The VDEV_BRMSetCSParam has been posted, we got the response from the device
		   so now we want to start the download */
	case FSM_STATE_BOOT_DO_AXF_SECTION_DOWNLOAD:

		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SBootDevAXFUserData));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* Initialise the UserData (allocated by VCORE_AllocateRequest) */
		pstAXFUserData = pstChildVapiReq->pvUserData;
		UT_MemCopy(&pstAXFUserData->ExecData, pstVapiReq->pvExecData, sizeof(SBootDevAXFExecData));
		UT_MemCopy(&pstAXFUserData->BootData, pstVapiReq->pvUserData, sizeof(SBootDevUserData));

		/* initialise the Child request 
		No particular data are required for the child fsm_handler */ 
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */ 
				VFSM_AxfDownload,	/* Child Request Handler */
				FSM_STATE_AXF_CODE_DOWNLOAD);	/* Child Request handler state */

		if (pstChnl->pstDev->eItfType == ePOS_ITF)
			pstVapiReq->usReqState = FSM_STATE_BOOT_DO_RUN_CHECKSUM;
		else
			pstVapiReq->usReqState = FSM_STATE_BOOT_DO_CHECKSUM;

		/* process request in the supervisor channel */
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case FSM_STATE_BOOT_DO_CHECKSUM:
		UT_Log(VCORE, INFO, "VCORE Boot: FSM_STATE_BOOT_DO_CHECKSUM\n");

		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		pstExecData = pstVapiReq->pvExecData;

		pstVapiReq->usReqState = FSM_STATE_BOOT_DO_RUN_CHECKSUM;
		Status = VDEV_RunCheckSum(pstChnl, pstExecData->uiCheckSumAddr);
		if (Status != SUCCESS)
			goto finish;

		break;

		/* We got the response to the latest FAT download command sent.
		   The download phase is finished so now we want to compute the checksum */
	case FSM_STATE_BOOT_DO_RUN_CHECKSUM:
		UT_Log(VCORE, INFO, "VCORE Boot: FSM_STATE_BOOT_DO_RUN_CHECKSUM\n");

		if (pstChnl->pstDev->eItfType == ePOS_ITF)
			Status = pstVapiReq->Status;
		else
			Status = Check_BootResponse(pstChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		pstExecData = pstVapiReq->pvExecData;

		/* Run the checksum if the interface is PCI or POS */
		switch (pstChnl->pstDev->eItfType)
		{
		case ePOS_ITF:
			pstVapiReq->usReqState = FSM_STATE_BOOT_DO_PROGSTART;
			Status = VDEV_POSChksum(pstChnl, pstExecData->uiCheckSumAddr);
			if (Status != SUCCESS)
				goto finish;
			break;
	
		case ePCI_ITF:
			pstVapiReq->usReqState = FSM_STATE_BOOT_DO_WAIT_SUPERVISOR_READY;
			Status = VDEV_RunMsp(pstChnl, pstExecData->uiStartAddr);
			if (Status != SUCCESS)
				goto finish;
			break;
	
		case eCSM_ITF:
		default:
			pstVapiReq->usReqState = FSM_STATE_BOOT_DO_START_TIMER;
			Status = VDEV_RunMsp(pstChnl, pstExecData->uiStartAddr);
			if (Status != SUCCESS)
				goto finish;
			break;
		}

		if (Status != SUCCESS)
			goto finish;
		
		break;

		/* we check here the result of the checksum
		   then run the firmware if the checksum is OK */
	case FSM_STATE_BOOT_DO_PROGSTART:	/*Send PROGSTART */
		UT_Log(VCORE, INFO, "VCORE Boot: FSM_STATE_BOOT_DO_PROGSTART\n");

		Status = Check_BootResponse(pstChnl, Event, pstMsg);
		pstExecData = pstVapiReq->pvExecData;

		switch (pstChnl->pstDev->eItfType)
		{
		case ePOS_ITF:
			/* FIXME: Check checksum issue when POS interface */
			if (Status == SUCCESS)
			{
				Status = VDEV_POSProgStart(pstChnl, pstExecData->uiStartAddr);
			}
			
			break;

		case ePCI_ITF:
			if ((Status == SUCCESS) && (pstMsg->mailbox.reg2 == 0x0000))
			{
				UT_Log(VCORE, INFO, "VFSM_BootDevice: PCI itf FRM Checksum Success status -> %d\n", Status);
				Status = VDEV_RunMsp(pstChnl, pstExecData->uiStartAddr);
			}
			else
			{
				UT_ErrorLog(VCORE, "VFSM_BootDevice: FRM Checksum Failed Status -> %d\n", Status);
				Status = VAPI_ERR_CHECKSUM_FAIL;
			}

			break;

		case eCSM_ITF:
			UT_ErrorLog(VCORE, "VFSM_BootDevice: should not get to this state for device %d\n", pstChnl->pstDev->ucDevType);

			break;
		}

		/* For CSME or POS there is a response to PROGSTART; not for PCI */
		/* So jump to the FSM_STATE_BOOT_DO_START_TIMER to wait the ack to PROGSTART */
		if (pstChnl->pstDev->eItfType != ePCI_ITF)
			pstVapiReq->usReqState = FSM_STATE_BOOT_DO_START_TIMER;
		/* For PCI interface there is no ACK to PROGSTART but the SUPERVISOR_READY indication later on */
		else
		{
			pstVapiReq->usReqState = FSM_STATE_BOOT_DO_WAIT_SUPERVISOR_READY;
		}

		if (Status != SUCCESS)
			goto finish;
		
		break;

		/* we get here for ETH or POS interface only */
		/* we got the ack to PROGSTART, now start a timer to wait for SUPERVISOR READY */
	case FSM_STATE_BOOT_DO_START_TIMER:

		Status = Check_BootResponse(pstChnl, Event, pstMsg);

		if (Status != SUCCESS)
		{
			Status = VAPI_ERR_PROG_START;
			goto finish;
		}
		/*Do nothing but wait for supervisor ready to come 
		   So start a timer and wait for the supervisor_ready indication in the
		   FSM_STATE_BOOT_DO_WAIT_SUPERVISOR_READY state */
		UT_TimerStart(pstChnl->pstRespTimer, stVAPIConfig_g.uiMspRespTimeout,
				VCORE_RespTimeoutHdlr, (void *)pstChnl);

		pstVapiReq->usReqState = FSM_STATE_BOOT_DO_WAIT_SUPERVISOR_READY;

		break;

		/* We got the SUPERVISOR_READY indication */
	case FSM_STATE_BOOT_DO_WAIT_SUPERVISOR_READY:
		UT_Log(VCORE, INFO, "VCORE Boot: FSM_STATE_BOOT_DO_WAIT_SUPERVISOR_READY\n");

		/* Check if the device reports any errors */
		Status = VCORE_CheckStatus(Event, pstMsg, SUPV_CHANNEL, CMD_CLASS_LEGACY_MSG, CMD_TYPE_SUPV_READY, 0);

		if (Status != SUCCESS)
		{
			Status = VAPI_ERR_SUPVSR_READY;
			goto finish;
		}
		else
		{
			if (SUPV_CHANNEL != UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]))
			{
				UT_ErrorLog(VCORE, "Supervisor Ready returned an error %d\n",
					    UT_LE2CPU16(((U16 *) pstMsg->fifo)[4]));
				Status = VAPI_ERR_SUPVSR_READY;
			}
			else
			{
				UT_Log(VCORE, INFO, "Supervisor Ready received OK\n");
				device_type = UT_LE2CPU16(((U16 *) pstMsg->fifo)[5]) & 0x00FF;

				UT_Log(VCORE, INFO, "Device type is 0x%02x\n", device_type);
				if (device_type < MAX_DEVICE_NAME)
				{
					UT_Log(VCORE, INFO, "Device name is %s\n", device_name[device_type]);
				}
			}
		}

		goto finish;
		
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_BootDevice: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:
	if (Status != SUCCESS)
	{
		UT_ErrorLog(VCORE, "VFSM_BootDevice: Error dev(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
			DevId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

		pstChnl->pstDev->eState = eUNINITIALIZED;
	}
	else
	{
		/* Now we consider that the device is ready to be used */
		pstChnl->pstDev->eState = eUP;
	}

	/* free the buffer containing the firmware */
	if (pstVapiReq->pvUserData != NULL)
	{
		UT_FreeMem(((SBootDevUserData *) (pstVapiReq->pvUserData))->pstSHwCfg);	
		if (((SBootDevUserData *) (pstVapiReq->pvUserData))->pvImageFile)
			UT_FreeMem(((SBootDevUserData *) (pstVapiReq->pvUserData))->pvImageFile);
	}

	UT_Log(VCORE, DEBUG, "VFSM_BootDevice: Completing req state(%u) status(%d) dev(%u)\n",
	       pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_BootDevice: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;

}

struct _ADDR_VAL
{
	U32 addr;
	U32 val;
};

struct _ADDR_VAL emac_dma[] = {{0x100DE000, 0x002002F0},
		{0x100DD000, 0x0000001F},
		{0x100DD014, 0x000000F3},
		{0x100DD018, 0x000000F0},
		{0x100DD024, 0x00000013},
		{0x100DD028, 0x00000012},
		{0x100D0018, 0x000000C0},
		{0x100D001C, 0x00000020},
		{0x100D0028, 0x000000E0},
		{0x100D002C, 0x00000040},
		{0x100D0000, 0x0000000C}};

#define EMAC_DMA_MAX_VAL 		10

/****************************************************************************
 * VFSM_SetEmacDma : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  All the commands use Supervisory channel structure, this is safe
 *          because, there will not be any supervisory commands until it gets
 *          created (Device is in eUNINITIALIZED state).
 *      -#  When a CMD_ACK is received, VCORE_ProcessResponse will not get any
 *          device corresponding to channel 0, but it should check the state of
 *          the device and if it is booting state then it should pass the 
 *          CMD_ACK to the Supervisory channel.
 *      -#  It does the following
 *          -#  Initialize hardware (EMAC DMA)
 *  
 *  \return 
 *  None
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_SetEmacDma(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	U32 Addr, Val;
	U16 *pusCounter;
	struct _ADDR_VAL *p_emac_dma;
	U16 usEmacMaxVal;
	DEVID DevId;
	DevId = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_SetEmacDma: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	p_emac_dma = emac_dma;
	usEmacMaxVal = EMAC_DMA_MAX_VAL;

	switch (pstVapiReq->usReqState)
	{
	case SET_EMAC_DMA_INIT:
		pstVapiReq->pvExecData = UT_AllocMem(sizeof(U16));
		if (pstVapiReq->pvExecData == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}
		pusCounter = pstVapiReq->pvExecData;
		*(pusCounter) = 0;
		
		Addr = p_emac_dma[*(pusCounter)].addr;
		Val = p_emac_dma[*(pusCounter)].val;

		pstVapiReq->usReqState = SET_EMAC_DMA_WRITE;
		Status = VDEV_DwordWrite(pstChnl, Addr, Val);

		if (Status != SUCCESS)
			goto finish;

		*(pusCounter) = 1;
		break;

	case SET_EMAC_DMA_WRITE:

		Status = VCORE_CheckStatus(Event, pstMsg, 0, CMD_CLASS_ETH_BOOT_MSG, CMD_TYPE_BRM_CMD_ACK, 0);
		if (Status != SUCCESS)
			goto finish;

		pusCounter = pstVapiReq->pvExecData;
		Addr = p_emac_dma[*(pusCounter)].addr;
		Val = p_emac_dma[*(pusCounter)].val;

		if ( *(pusCounter) == usEmacMaxVal)
			pstVapiReq->usReqState = SET_EMAC_DMA_FINISHED;

		Status = VDEV_DwordWrite(pstChnl, Addr, Val);

		if (Status != SUCCESS)
			goto finish;

		*(pusCounter) += 1;

		break;

	case SET_EMAC_DMA_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, 0, CMD_CLASS_ETH_BOOT_MSG, CMD_TYPE_BRM_CMD_ACK, 0);

		if (Status == SUCCESS)
			pstChnl->pstDev->eState = eREADYFORDUMP;

		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetEmacDma: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}
	return SUCCESS;

finish:

	UT_Log(VCORE, DEBUG, "VFSM_SetEmacDma: Completing state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetEmacDma: Exiting status(%d) dev(%u)\n", Status, DevId);
	return Status;
}


/****************************************************************************
 * VFSM_AssignBootMAC : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  All the commands use Supervisory channel structure, this is safe
 *          because, there will not be any supervisory commands until it gets
 *          created.
 *      -#  When a CMD_ACK is received, VCORE_ProcessResponse will not get any
 *          device corresponding to channel 0, but it should check the state of
 *          the device and if it is booting state then it should pass the 
 *          CMD_ACK to the Supervisory channel.
 *      -#  It does the following
 *          -#  Send a MAAS_ASSIGN if the interfce is CSM_ENCAPS.
 *          -#  Disables promiscuous mode if device is not M829XX
 *
 *
 *  \return 
 *  None
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_AssignBootMAC(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	DEVID DevId;
	DevId  = pstSupvChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_AssignBootMAC: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	switch (pstVapiReq->usReqState)
	{
	case BOOT_MAAS:

		/* Do not reset the promiscuous mode for the M829xx */
		if ((pstSupvChnl->pstDev->ucDevType == DEV_TYPE_M829XX) ||
			(pstSupvChnl->pstDev->ucDevType == DEV_TYPE_M823XX) ||
			(pstSupvChnl->pstDev->ucDevType == DEV_TYPE_M823XX_2))
			pstVapiReq->usReqState = BOOT_INIT_FINISHED;
		else
			pstVapiReq->usReqState = BOOT_PROMISCUOUS_MODE;

		/* Send MAAS_ASSIGN */
		Status = VDEV_MaasAssign(pstSupvChnl);

		if (Status != SUCCESS)
			goto finish;
	
		break;

	/* M823xx only state*/
	case BOOT_QUERY_READY:

		pstVapiReq->usReqState = BOOT_WAIT_QUERY_READY_ACK;

		/* Send Query ready */
		Status = VDEV_QueryReady(pstSupvChnl);

		if (Status != SUCCESS)
			goto finish;
	
		break;

	/* M823xx only state*/
	case BOOT_WAIT_QUERY_READY_ACK:

		Status = VCORE_CheckStatus(Event, pstMsg, 0, CMD_CLASS_ETH_BOOT_MSG, CMD_TYPE_BRM_CMD_ACK, 0);

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = BOOT_INIT_FINISHED;

		/* now send maas assign*/
		Status = VDEV_MaasAssign(pstSupvChnl);

		if (Status != SUCCESS)
			goto finish;
	
		break;

	/* M823xx only state*/
	case BOOT_WAIT_READY:
		Status = VCORE_CheckReadyReceived(Event, pstMsg);

		/* we are waiting in this state until the ready frame is received
		or timeout */
		if (Status == SUCCESS)
		{
			pstVapiReq->usReqState = BOOT_INIT_FINISHED;

			/* now send maas assign*/
			Status = VDEV_MaasAssign(pstSupvChnl);

			/* reset this flag to make sure we are able to handle a
			READY frame on next device reset*/
			pstSupvChnl->pstDev->bIsReadyReceived = False;

			if (Status != SUCCESS)
				goto finish;
		}
		else if (Status == VAPI_ERR_WAIT_READY)
			break;

		else /* != SUCCESS*/
			goto finish;

		break;


	/* Remove the promicuous mode */
	/* removing the promiscuous mode workaround has been kept in the code for old devices revision*/
	case BOOT_PROMISCUOUS_MODE:
		Status = VCORE_CheckStatus(Event, pstMsg, 0, CMD_CLASS_ETH_BOOT_MSG, CMD_TYPE_BRM_CMD_ACK, 0);

		if (Status != SUCCESS)
		{
			Status = VAPI_ERR_MAAS_ASSIGN;
			goto finish;
		}

		pstVapiReq->usReqState = BOOT_INIT_FINISHED;
		Status = VDEV_WordWrite(pstSupvChnl, EMAC_ARCCTRL, ADDRESS_COMPARISON_ENABLE);

		if (Status != SUCCESS)
			goto finish;

		break;

	case BOOT_INIT_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, 0, CMD_CLASS_ETH_BOOT_MSG, CMD_TYPE_BRM_CMD_ACK, 0);

		if (Status == SUCCESS)
			pstSupvChnl->pstDev->eState = eMAAS;

		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_AssignBootMAC: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		

		break;
	}

	return SUCCESS;

finish:

	UT_Log(VCORE, DEBUG, "VFSM_AssignBootMAC: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_AssignBootMAC: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}


/****************************************************************************
 * VFSM_InitDevice : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  4 states
 *      -#  DEV_INIT_SEND_SPU_FEATURE
 *           Send SPU CTRL command to the Comcerto supervisor
 *      -#  DEV_INIT_CHECK_SPU_FEATURE
 *           Wait the response to SPU CTRL
 *           If response OK, enable MULTICMD feature if required
 *      -#  DEV_INIT_CHECK_MULTI_COMMAND
 *           Wait the response to MULTICMD
 *           If response OK, return OK else return ERROR
 *      -#  DEV_INIT_FINISHED (default)
 *           Process of the request is finished -> exit
 *
 *  - Assumptions
 *
 *  \return   SUCCESS or Failure code from Comcerto.
 *  
 *  \param pstChnl The channel on which the request is going.
 *  \param pstVapiReq  VAPI request to handle.
 *  \param pstMsg  Message obtained from Comcerto.
 */
VSTATUS VFSM_InitDevice(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	U32 *pstUserData = NULL;
	DEVID DevId;
	U16 *paucFifo = NULL;

	DevId  = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_InitDevice: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	pstUserData = (U32 *) pstVapiReq->pvUserData;

	/* the state has been initialized to 0 (DEV_INIT_GET_MR) in VAPI_InitDevice()*/
	switch (pstVapiReq->usReqState)
	{
	case DEV_INIT_GET_MR:
		/* save the SPU mask in a internal variable */
		pstChnl->pstDev->usSpuFeatureMask = (* pstUserData );

		/*Get the MR version*/
		pstVapiReq->usReqState = DEV_INIT_GET_MODEL;

		Status = VDEV_GetMR(pstChnl);
		if (Status != SUCCESS)
			goto finish;

		break;

	case DEV_INIT_GET_MODEL:
		/* save the SPU mask in a internal variable */
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				   CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY_RESP, FC_ARM_FW_BRANCH);
		if (Status != SUCCESS)
			goto finish;

		/* save the MR number*/
		VCORE_UpdateMR(pstChnl->pstDev, pstMsg->fifo);
		pstVapiReq->usReqState = DEV_INIT_SPU_INIT;
		/*Get the device model*/
		Status = VDEV_GetDevModel(pstChnl);
		if (Status != SUCCESS)
			goto finish;
		break;
	
	case DEV_INIT_SPU_INIT:
		/* save the SPU mask in a internal variable */
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				   CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY_RESP, FC_SUPVSR_GET_DEVICE_TYPE);

		if (Status != SUCCESS)
			goto finish;

		/*Save the device model for future use */
		paucFifo = (U16 *)(pstMsg->fifo);
		pstChnl->pstDev->usDevModel = paucFifo[4];
		/*For PCI interface init the SPU feature Mask and finish*/
		if (pstChnl->pstDev->eItfType != ePCI_ITF)
			pstVapiReq->usReqState = DEV_INIT_ENABLE_MULTI_CMDS;
		/*For other interfaces init the SPU feature Mask then enable multi cmd mode*/
		else
			pstVapiReq->usReqState = DEV_INIT_FINISHED;

		Status = VDEV_SPUFeatureControl(pstChnl);
		if (Status != SUCCESS)
			goto finish;

		break;

	case DEV_INIT_ENABLE_MULTI_CMDS:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_SPU_FEATURES_CONTROL);

		if (Status != SUCCESS)
			goto finish;

		/* if yes send the command and wait the anwer in DEV_INIT_FINISHED state */
		pstVapiReq->usReqState = DEV_INIT_FINISHED;
		Status = VDEV_EnableMultiCmds(pstChnl, 1);
		if (Status != SUCCESS)
			goto finish;
		break;

	case DEV_INIT_FINISHED:
		if (pstChnl->pstDev->eItfType != ePCI_ITF)
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_CSME_MULTI_CMD);
		else
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_SPU_FEATURES_CONTROL);

		if (Status != SUCCESS)
			pstChnl->pstDev->bMultiCmdEnabled = False;
		else
			pstChnl->pstDev->bMultiCmdEnabled = True;

		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_InitDevice: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:

	if (Status == SUCCESS)
		pstChnl->pstDev->bIsDeviceInitialized = True;
	else
		pstChnl->pstDev->bIsDeviceInitialized = False;

	/* we want to stop the process in case of error on a command 
	or if the process is finished */
	UT_Log(VCORE, DEBUG, "VFSM_InitDevice: Completing req state(%u) status(%d) dev(%u)\n",
		       pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_InitDevice: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

/****************************************************************************
 * VFSM_InitRecover : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  4 states
 *      -#  RECOV_GET_MR
 *           Send FC_ARM_FW_BRANCH query to the Comcerto device to get F/W MR
 *      -#  RECOV_GET_ETH
 *           Send FC_GET_ETH_HDR query to the Comcerto device to get Eth header
 *      -#  RECOV_GET_IP
 *           Send FC_IP_ADDRESS query to the Comcerto device to get device IP
 *      -#  RECOV_FINISH: (default)
 *           Process of the request is finished -> exit
 *
 *  - Assumptions
 *
 *  \return   SUCCESS or Failure code from Comcerto.
 *  
 *  \param pstChnl The channel on which the request is going.
 *  \param pstVapiReq  VAPI request to handle.
 *  \param pstMsg  Message obtained from Comcerto.
 */
VSTATUS VFSM_InitRecover(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	U16 *pstUserData = NULL;
	DEVID DevId;
	DevId  = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_InitRecover: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	pstUserData = (U16 *) pstVapiReq->pvUserData;	

	switch (pstVapiReq->usReqState)
	{
	case RECOV_GET_MR:
		/* get firmware MR number */
		pstVapiReq->usReqState = RECOV_GET_ETH;

		Status = VDEV_GetMR(pstChnl);

		if (Status != SUCCESS)
			goto finish;

		break;

	case RECOV_GET_ETH:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY_RESP, FC_ARM_FW_BRANCH);

		if (Status != SUCCESS)
			goto finish;
		/* save the MR number*/
		VCORE_UpdateMR(pstChnl->pstDev, pstMsg->fifo);

		pstVapiReq->usReqState = RECOV_GET_IP;

		/* get eth MAC address which set on device */
		Status = VDEV_GetEthMacAddr(pstChnl);

		if (Status != SUCCESS)
			goto finish;

		break;

	case RECOV_GET_IP:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY_RESP, FC_GET_ETH_HDR);

		if (Status != SUCCESS)
			goto finish;
		/* save the ETH header to the device structure */
		VCORE_UpdateEth(pstChnl->pstDev, pstMsg->fifo);

		pstVapiReq->usReqState = RECOV_GET_SPU;

		/* get eth IP address which set on device */
		Status = VDEV_GetDevIP(pstChnl);

		if (Status != SUCCESS)
			goto finish;

		break;

	case RECOV_GET_SPU:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY_RESP, FC_IP_ADDRESS);

		if (Status != SUCCESS)
			goto finish;
		
		/* save the IP header to the device structure */
		VCORE_UpdateIP(pstChnl->pstDev, pstMsg->fifo);

		pstVapiReq->usReqState = RECOV_GET_CSME_MULTI;

		/* get SPU feature control */
		Status = VDEV_GetSpuFeature(pstChnl);

		if (Status != SUCCESS)
			goto finish;

		break;

	case RECOV_GET_CSME_MULTI:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY_RESP, FC_SPU_FEATURES_CONTROL);

		if (Status != SUCCESS)
			goto finish;
		
		/* save SPU feature control to the device structure */
		VCORE_UpdateSpuFeature(pstChnl->pstDev, pstMsg->fifo);

		/* multicommand is only for CSME itf type */
		if (pstChnl->pstDev->eItfType != eCSM_ITF)
			goto finish;

		pstVapiReq->usReqState = RECOV_FINISH;

		/* get CSME multi command status */
		Status = VDEV_GetCsmeMulti(pstChnl);

		if (Status != SUCCESS)
			goto finish;

		break;

	case RECOV_FINISH:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY_RESP, FC_CSME_MULTI_CMD);

		if (Status != SUCCESS)
			goto finish;

		/* save CSME multi comman status to the device structure */
		VCORE_UpdateCsmeMulti(pstChnl->pstDev, pstMsg->fifo);

		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_InitRecover: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;
	}

	return SUCCESS;

finish:
	/* we want to stop the process in case of error on a command 
	or if the process is finished */
	UT_Log(VCORE, DEBUG, "VFSM_InitRecover: Completing req state(%u) status(%d) dev(%u)\n",
		       pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_InitRecover: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

/****************************************************************************
 * VFSM_SetTDMParams : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *
 *      -#  Obtain the VAPI request node from the the channel structure.
 *      -#  Check the state of the request usReqState\n
 *      -#  Check the input GTL message (response) for SUCCESS.\n
 *      -#  Send the MSP message according to current state using 
 *          VCORE_SendMSPReq
 *      -#  Update the current state. \n
 *      -#. Following MSP messages will be sent\n
 *          -#  FC_SUPV_TDM_SELECT_BUS_MODE\n
 *          -#  FC_SUPV_SETUP_TDM_PARAM\n
 *      -#  When request is completed give a calback or signal request 
 *          completion semaphore\n
 *
 *  - Assumptions:\n
 *      -#  Default Parameter settings of TDM bus are done by APPITF\n.
 *
 *
 *  \return   SUCCESS or Failure code.
 *
 *  \param pstChnl  This will be a channel on which 
 *                  MSP commands will be sent..
 *  \param pstMsg   Message (response) obtained from GTL. When it is called
 *                  by APPITF this is passed as NULL.
 */
VSTATUS VFSM_SetTDMParams(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	STdmSetupParams *pstTdm = NULL;
	DEVID DevId;
	DevId  = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_SetTDMParams: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	pstTdm = (STdmSetupParams *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case SET_TDM_INIT:
		pstVapiReq->usReqState = SET_TDM_ENABLE;
		Status = VDEV_SelectTdm(pstChnl, pstTdm);
		if (Status != SUCCESS)
			goto finish;
		break;

	case SET_TDM_ENABLE:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_TDM_SELECT_BUS_MODE);
		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = SET_TDM_PARAM_0;
		Status = VDEV_TdmEnableBus(pstChnl, pstTdm->usNoOfBus);
		if (Status != SUCCESS)
			goto finish;

		break;

	case SET_TDM_PARAM_0:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_TDM_ENABLE_BUS);
		if (Status != SUCCESS)
			goto finish;


		if (pstTdm->usNoOfBus == 1)
			pstVapiReq->usReqState = SET_TDM_FINISHED;
		else
			pstVapiReq->usReqState = SET_TDM_PARAM_1;

		Status = VDEV_SetupTdmParam(pstChnl, &pstTdm->astTdmBusParam[0]);
		if (Status != SUCCESS)
			goto finish;

		break;

	case SET_TDM_PARAM_1:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_SUPVSR_SETUP_TDM_PARAMS);
		if (Status != SUCCESS)
			goto finish;

		if (pstTdm->usNoOfBus == 2)
			pstVapiReq->usReqState = SET_TDM_FINISHED;
		else
			pstVapiReq->usReqState = SET_TDM_PARAM_2;

		Status = VDEV_SetupTdmParam(pstChnl, &pstTdm->astTdmBusParam[1]);
		if (Status != SUCCESS)
			goto finish;

		break;

	case SET_TDM_PARAM_2:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_SUPVSR_SETUP_TDM_PARAMS);
		if (Status != SUCCESS)
			goto finish;

		if (pstTdm->usNoOfBus == 3)
			pstVapiReq->usReqState = SET_TDM_FINISHED;
		else
			pstVapiReq->usReqState = SET_TDM_PARAM_3;

		Status = VDEV_SetupTdmParam(pstChnl, &pstTdm->astTdmBusParam[2]);
		if (Status != SUCCESS)
			goto finish;
		break;

	case SET_TDM_PARAM_3:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_SUPVSR_SETUP_TDM_PARAMS);
		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = SET_TDM_FINISHED;
		Status = VDEV_SetupTdmParam(pstChnl, &pstTdm->astTdmBusParam[3]);
		if (Status != SUCCESS)
			goto finish;
		break;

	case SET_TDM_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_SUPVSR_SETUP_TDM_PARAMS);

		goto finish;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetTDMParams: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:

	UT_Log(VCORE, DEBUG, "VFSM_SetTDMParams: Completing req state(%u) status(%d) dev(%u)\n",
		       pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetTDMParams: Exiting status(%d) dev(%u)\n", Status, DevId);
	return Status;
}

/****************************************************************************
 * VFSM_SetEthMac : The function does the following things -
 ***************************************************************************/
/*! 
 *
 */
VSTATUS VFSM_SetDevEthMac(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	SSetEthMAc *pstUserData;
	DEVID DevId;
	DevId  = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_SetDevEthMac: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	pstUserData = (SSetEthMAc *) pstVapiReq->pvUserData;

	/* retrieve the VLAN ID set by VAPI_SetDeviceVlan()*/
	pstUserData->usVlanId = pstChnl->pstDev->stEthMacAddresses.usVlanId;

	switch (pstVapiReq->usReqState)
	{

	case DEV_ETH_MAC_INIT:

		pstVapiReq->usReqState = DEV_ETH_MAC_FINISHED;
		Status = VDEV_SendEthMacAddr(pstChnl, pstUserData->astSrcMacAddr, pstUserData->astDestMacAddr,
			 			pstUserData->usVlanId);

		if (Status != SUCCESS)
			goto finish;
		break;

	case DEV_ETH_MAC_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
						   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_SET_ETH_HDR);

		if (Status == SUCCESS)
		{
			pstChnl->pstDev->bEthHdrSet = True;

			/* save the Src, Dest MAC addresses and VLAN id to connection structure*/
			pstUserData->usVlanId = pstChnl->pstDev->stEthMacAddresses.usVlanId;
			UT_MemCopy( &(pstChnl->pstDev->stEthMacAddresses), pstUserData, sizeof(SSetEthMAc));
		}
		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetDevEthMac: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	};

	return SUCCESS;

finish:

	UT_Log(VCORE, DEBUG, "VFSM_SetDevEthMac: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetDevEthMac: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

/****************************************************************************
 * VFSM_SetDeviceIPAddr : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  
 *      -#  
 *      -#  
 *      -#  
 *
 *  - Assumptions
 *      -#  
 *  
 *  \return 
 *  
 *  
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_SetDeviceIPAddr(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	SIpAddrInfo *pstUserData;
	DEVID DevId;
	DevId  = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_SetDeviceIPAddr: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	pstUserData = (SIpAddrInfo *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{

	case DEV_IP_ADDR_INIT:
		/*pstChnl is the SUPVSR */
		if (pstChnl->pstDev->bEthHdrSet != True)
		{
			UT_ErrorLog(VCORE, "VFSM_SetDeviceIPAddr: Ethernet header not set on Device\n");
			Status = VAPI_ERR_INVALID_OPERATION;

		}
		/* For master mode ARP and ICMP are preformed by the CSP */
		if (pstChnl->pstDev->eDevMode == eMASTER)
			pstVapiReq->usReqState = DEV_IP_ADDR_SET_IP;
		else
			pstVapiReq->usReqState = DEV_IP_ADDR_SET_ARP;

		Status = VDEV_IpServiceConfig(pstChnl, pstUserData->ucChkSumVerification, pstUserData->ucEncapsulation);
		if (Status != SUCCESS)
			goto finish;

		break;

	case DEV_IP_ADDR_SET_ARP:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
						   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_IP_SERVICE_CONFIG);
		if (Status != SUCCESS)
			goto finish;

		/* the device is configure for IPoETH operation */
		pstChnl->pstDev->eEthHdrMode = eMODE_IP;
		pstVapiReq->usReqState = DEV_IP_ADDR_SET_ICMP;
		Status = VDEV_EnableARP(pstChnl);
		if (Status != SUCCESS)
			goto finish;

		break;

	case DEV_IP_ADDR_SET_ICMP:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_ARP_SERVICE_CONFIG);
		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = DEV_IP_ADDR_SET_IP;
		Status = VDEV_EnableICMP(pstChnl);

		if (Status != SUCCESS)
			goto finish;
		break;

	case DEV_IP_ADDR_SET_IP:

		if (pstChnl->pstDev->eDevMode == eMASTER)
		{
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_IP_SERVICE_CONFIG);

			if (Status == SUCCESS)
			/* the device is configure for IPoETH operation */
				pstChnl->pstDev->eEthHdrMode = eMODE_IP;
		}
		else
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_ICMP_SERVICE_CONFIG);

		if (Status != SUCCESS)
			goto finish;

		if (pstUserData->bIsMultipleMode == False)
		{
			pstVapiReq->usReqState = DEV_IP_ADDR_SET_SINGLE;
			Status = VDEV_SetIpAddr(pstChnl, pstUserData->auiDevIPAddress[0]);
			if (Status != SUCCESS)
				goto finish;
		}
		else
		{
			pstVapiReq->usReqState = DEV_IP_ADDR_SET_MULTI;
			Status = VDEV_SetIpAddrList(pstChnl,
						     pstUserData->auiDevIPAddress,
						     pstUserData->ucNumOfSrcIpAddr);
			if (Status != SUCCESS)
				goto finish;
		}
		break;

	case DEV_IP_ADDR_SET_SINGLE:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
						   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_IP_ADDRESS);

		if (Status == SUCCESS)
		{
			pstChnl->pstDev->uiSrcIpAddr = pstUserData->auiDevIPAddress[0];
			pstChnl->pstDev->eSrcIpMode = eSINGLE_IP;
		}
		goto finish;

		break;

	case DEV_IP_ADDR_SET_MULTI:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
						   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_MULTI_IP_ADDR_LIST);

		/* Il Multiple IP mode we store only the fisrt IP as the default one
		The application is responsible to choose the IP addr to use for a channel if it is not the default*/
		if (Status == SUCCESS)
		{
			pstChnl->pstDev->uiSrcIpAddr = pstUserData->auiDevIPAddress[0];
			pstChnl->pstDev->eSrcIpMode = eMULTIPLE_IP;
		}
		goto finish;

		break;
	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetDeviceIPAddr: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	};

	return SUCCESS;

finish:

	UT_Log(VCORE, DEBUG, "VFSM_SetDeviceIPAddr: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetDeviceIPAddr: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}


/****************************************************************************
 * VFSM_GetCoreDump : The function does the following things -
 ***************************************************************************/
VSTATUS VFSM_GetCoreDump(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	SVapiReq *pstChildVapiReq;
	DEVID DevId;
	DevId  = pstSupvChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_GetCoreDump: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	switch (pstVapiReq->usReqState)
	{

	case COREDUMP_INIT:
		/*we get in this state only for CSME interface */

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* initialise the Child request 
		No particular data are required for the child fsm_handler */ 
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */ 
				VFSM_AssignBootMAC,	/* Child Request Handler */
				0);			/* Child Request handler state */
		
		pstVapiReq->usReqState = COREDUMP_SET_HARDWARE;
		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case COREDUMP_SET_HARDWARE:
		/* in case of CSME interface we previously ran the above child request */
		if (pstSupvChnl->pstDev->eItfType == eCSM_ITF)
		{
			/* retrieve the child status */
			Status = pstVapiReq->Status;

			if (Status != SUCCESS)
				goto finish;
		}

		/* this allocate a new request */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SHwCfg));

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		SetVAPIDeviceHwConfig((SHwCfg *) pstChildVapiReq->pvUserData, ((SCoreDumpData *)(pstVapiReq->pvUserData))->pstSHwCfg, pstSupvChnl->pstDev);

		/* initialise the Child request 
		No particular data are required for the child fsm_handler */ 
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */ 
				VFSM_SetDeviceHardware,	/* Child Request Handler */
				DEVICE_HARDWARE_INIT);	/* Child Request handler state */


		/* Initialize the EMAC DMA only for CSME interface and if the device is not M829xx */
		if ((pstSupvChnl->pstDev->ucDevType != DEV_TYPE_M829XX) && 
			(pstSupvChnl->pstDev->ucDevType != DEV_TYPE_M823XX) && 
			(pstSupvChnl->pstDev->eItfType == eCSM_ITF))
		{
			pstVapiReq->usReqState = COREDUMP_SET_EMAC_DMA;
		}
		/* next state ( SET EMAC DMA is not required */
		else
		{
			pstVapiReq->usReqState = COREDUMP_READ;
			pstSupvChnl->pstDev->eState = eREADYFORDUMP;
		}

		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case COREDUMP_SET_EMAC_DMA:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* initialise the Child request 
		No particular data are required for the child fsm_handler */ 
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */ 
				VFSM_SetEmacDma,	/* Child Request Handler */
				SET_EMAC_DMA_INIT);	/* Child Request handler state */

		pstVapiReq->usReqState = COREDUMP_READ;
		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstSupvChnl, pstChildVapiReq);

		break;


	case COREDUMP_READ:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		if (pstSupvChnl->pstDev->eState != eREADYFORDUMP)
			goto finish;

		/* This allocate a new request */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SCoreDumpData));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* Initialise the UserData (allocated by VCORE_AllocateRequest)
		get the user data from the parent request */
		UT_MemCopy(pstChildVapiReq->pvUserData, pstVapiReq->pvUserData, sizeof(SCoreDumpData));

		/* initialise the Child request*/ 
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */
				VCORE_ReadDeviceMemory,	/* Child Request Handler */
				0);			/* Child Request handler state */

		pstVapiReq->usReqState = COREDUMP_FINISHED;
		/* process request in the supervisor channel */
		VCORE_StartChannelChildRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case COREDUMP_FINISHED:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_GetCoreDump: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:
	UT_FreeMem(((SCoreDumpData *) (pstVapiReq->pvUserData))->pstSHwCfg);

	UT_Log(VCORE, DEBUG, "VFSM_GetCoreDump: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_GetCoreDump: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;

}

VSTATUS VFSM_SetDeviceHardware(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	DEVID DevId;
	DevId  = pstSupvChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_SetDeviceHardware: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	switch (pstVapiReq->usReqState)
	{
	case DEVICE_HARDWARE_INIT:

		/* The hardware init is not the same for POS than for PCI/CSME */
		switch (pstSupvChnl->pstDev->eItfType)
		{
		case ePOS_ITF:
			UT_Log(VCORE, INFO, "VFSM_SetDeviceHardware: Sending PHYID assign\n");
			/* for POS interface we send the POSPhyIdAssign command */
			pstVapiReq->usReqState = DEVICE_HARDWARE_POS_SET_PLL;
			Status = VDEV_POSPhyIdAssign(pstSupvChnl);
			if (Status != SUCCESS)
				goto finish;
			break;

		case ePCI_ITF:
			/* make sure that the PLL Workaround is applied.*/
			UT_Log(VCORE, INFO, "VFSM_SetDeviceHardware: PLL Workaround (PCI)\n");
			
			pstVapiReq->usReqState = DEVICE_HARDWARE_SET_PLL_WA_PCI;
			Status = VDEV_BRMDwordWriteAddr(pstSupvChnl, PLL_CONTROL_REGISTER);
			
			if (Status != SUCCESS)
				goto finish;
			break;

		case eCSM_ITF:
			pstVapiReq->usReqState = DEVICE_HARDWARE_SET_ARM_CLOCK;
			Status = VDEV_BRMSetClock(pstSupvChnl);

			if (Status != SUCCESS)
				goto finish;
			break;

		default:
			UT_ErrorLog(VCORE, "Wrong interface type %d", pstSupvChnl->pstDev->eItfType);
			Status = VAPI_ERR_UNSUPP_FEATURE;
			goto finish;
			break;
		}

		break;

	case DEVICE_HARDWARE_SET_PLL_WA_PCI:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = DEVICE_HARDWARE_SET_CLOCK;
		Status = VDEV_BRMDwordWriteData(pstSupvChnl, PLL_WA1_VALUE);

		if (Status != SUCCESS)
			goto finish;
		break;

	case DEVICE_HARDWARE_SET_CLOCK:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);
		if (Status != SUCCESS)
			goto finish;

		/* for PCI or ETH interface we send the VDEV_BRMArmSetClkMode command */
		pstVapiReq->usReqState = DEVICE_HARDWARE_SET_ARM_CLOCK;
		Status = VDEV_BRMSetClock(pstSupvChnl);

		if (Status != SUCCESS)
			goto finish;
		break;

	case DEVICE_HARDWARE_SET_ARM_CLOCK:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		/* for PCI or ETH interface we send the VDEV_BRMArmSetClkMode command */
		pstVapiReq->usReqState = DEVICE_HARDWARE_SET_SDRAM;

		Status = VDEV_BRMArmSetClkMode(pstSupvChnl);

		if (Status != SUCCESS)
			goto finish;
		break;

	case DEVICE_HARDWARE_SET_SDRAM:

		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		if (pstSupvChnl->pstDev->eItfType == ePCI_ITF)
			pstVapiReq->usReqState = DEVICE_HARDWARE_SET_SDRAM_PARAM2;
		else
			/* For CSME interface want to configure the SDRAM chip select */
			pstVapiReq->usReqState = DEVICE_HARDWARE_SET_CS_SDRAM;

		Status = VDEV_BRMSetSdramParam(pstSupvChnl, (SHwCfg *)pstVapiReq->pvUserData);
		if (Status != SUCCESS)
			goto finish;
		break;

	case DEVICE_HARDWARE_SET_SDRAM_PARAM2:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		if (pstSupvChnl->pstDev->eItfType == ePCI_ITF)
			pstVapiReq->usReqState = DEVICE_HARDWARE_SET_CS_BASEADDR;
		else
			pstVapiReq->usReqState = DEVICE_HARDWARE_SET_CS_SDRAM;

		Status = VDEV_BRMSetSdramParam2(pstSupvChnl, (SHwCfg *)pstVapiReq->pvUserData);
		if (Status != SUCCESS)
			goto finish;
		break;

	case DEVICE_HARDWARE_SET_CS_BASEADDR:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		/* for PCI interface we send the VDEV_BRMSetCSParam command state */
		pstVapiReq->usReqState = DEVICE_HARDWARE_SET_CS_SDRAM;
		Status = VDEV_BRMDwordWriteAddr(pstSupvChnl, 0);
		if (Status != SUCCESS)
			goto finish;
		break;

	case DEVICE_HARDWARE_SET_CS_SDRAM:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		/* for PCI interface we send the VDEV_BRMSetCSParam command state */
		pstVapiReq->usReqState = DEVICE_HARDWARE_FINISHED;
		Status = VDEV_BRMSetCSParam(pstSupvChnl, (SHwCfg *)pstVapiReq->pvUserData);
		if (Status != SUCCESS)
			goto finish;
		break;

	case DEVICE_HARDWARE_POS_SET_PLL:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
		{
			Status = VAPI_ERR_PHYID_ASSIGN;
			goto finish;
		}

		/* for POS interface we now send the VDEV_POSSetPLL command */
		pstVapiReq->usReqState = DEVICE_HARDWARE_POS_SET_SDRAM;
		Status = VDEV_POSSetPLL(pstSupvChnl);
		if (Status != SUCCESS)
			goto finish;
		break;

	case DEVICE_HARDWARE_POS_SET_SDRAM:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = DEVICE_HARDWARE_FINISHED;
		Status = VDEV_POSSetSDRAM(pstSupvChnl, (SHwCfg *)pstVapiReq->pvUserData);
		if (Status != SUCCESS)
			goto finish;
		break;

	case DEVICE_HARDWARE_FINISHED:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);
		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetDeviceHardware: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:
	UT_Log(VCORE, DEBUG, "VFSM_SetDeviceHardware: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetDeviceHardware: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

VSTATUS VFSM_SetDeviceHardware900(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	DEVID DevId;
	DevId  = pstSupvChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_SetDeviceHardware: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	switch (pstVapiReq->usReqState)
	{
	case DEVICE_HARDWARE_900_INIT:
		switch (pstSupvChnl->pstDev->eItfType)
		{
		case ePOS_ITF:
			UT_Log(VCORE, INFO, "VFSM_SetDeviceHardware900: Sending PHYID assign\n");
			/* for POS interface we send the POSPhyIdAssign command */
			pstVapiReq->usReqState = DEVICE_HARDWARE_900_POS_SET_PLL;
			Status = VDEV_POSPhyIdAssign(pstSupvChnl);
			if (Status != SUCCESS)
				goto finish;
			break;

		case ePCI_ITF:
			/* make sure that the PLL Workaround is applied. */
			UT_Log(VCORE, INFO, "VFSM_SetDeviceHardware900: PLL Workaround (PCI)\n");

			pstVapiReq->usReqState = DEVICE_HARDWARE_900_SET_PLL_WA_PCI;
			Status = VDEV_BRMDwordWriteAddr(pstSupvChnl, PLL_CONTROL_REGISTER);
			
			if (Status != SUCCESS)
				goto finish;
			break;

		case eCSM_ITF:
			pstVapiReq->usReqState = DEVICE_HARDWARE_900_SET_SDRAM_PARAM2;
			Status = VDEV_BRMSetSdramParam(pstSupvChnl, (SHwCfg *)pstVapiReq->pvUserData);

			if (Status != SUCCESS)
				goto finish;
			break;

		default:
			UT_ErrorLog(VCORE, "Wrong interface type %d", pstSupvChnl->pstDev->eItfType);
			Status = VAPI_ERR_UNSUPP_FEATURE;
			goto finish;

			break;
		}

		break;

	case DEVICE_HARDWARE_900_SET_PLL_WA_PCI:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = DEVICE_HARDWARE_900_SET_SDRAM;
		Status = VDEV_BRMDwordWriteData(pstSupvChnl, PLL_WA1_VALUE_M829XX);

		if (Status != SUCCESS)
			goto finish;
		break;

	case DEVICE_HARDWARE_900_SET_SDRAM:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = DEVICE_HARDWARE_900_SET_SDRAM_PARAM2;
		Status = VDEV_BRMSetSdramParam(pstSupvChnl, (SHwCfg *)pstVapiReq->pvUserData);

		if (Status != SUCCESS)
			goto finish;
		break;

	case DEVICE_HARDWARE_900_SET_SDRAM_PARAM2:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		if (pstSupvChnl->pstDev->eItfType == ePCI_ITF)
			pstVapiReq->usReqState = DEVICE_HARDWARE_900_SET_CS_BASEADDR;
		else
			pstVapiReq->usReqState = DEVICE_HARDWARE_900_SET_CS_SDRAM;

		Status = VDEV_BRMSetSdramParam2(pstSupvChnl, (SHwCfg *)pstVapiReq->pvUserData);
		if (Status != SUCCESS)
			goto finish;

		break;

	case DEVICE_HARDWARE_900_SET_CS_BASEADDR:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		/* for PCI interface we send the VDEV_BRMSetCSParam command state */
		pstVapiReq->usReqState = DEVICE_HARDWARE_900_SET_CS_SDRAM;
		Status = VDEV_BRMDwordWriteAddr(pstSupvChnl, 0);
		if (Status != SUCCESS)
			goto finish;
		break;

	case DEVICE_HARDWARE_900_SET_CS_SDRAM:
		/* don't check boot response for M829xx */
		pstVapiReq->usReqState = DEVICE_HARDWARE_900_FINISHED;
		Status = VDEV_BRMSetCSParam(pstSupvChnl, (SHwCfg *)pstVapiReq->pvUserData);

		if (Status != SUCCESS)
			goto finish;
		break;

	case DEVICE_HARDWARE_900_POS_SET_PLL:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
		{
			Status = VAPI_ERR_PHYID_ASSIGN;
			goto finish;
		}

		/* for POS interface we now send the VDEV_POSSetPLL command */
		pstVapiReq->usReqState = DEVICE_HARDWARE_900_POS_SET_SDRAM;
		Status = VDEV_POSSetPLL(pstSupvChnl);
		if (Status != SUCCESS)
			goto finish;
		break;

	case DEVICE_HARDWARE_900_POS_SET_SDRAM:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = DEVICE_HARDWARE_900_FINISHED;
		Status = VDEV_POSSetSDRAM(pstSupvChnl, (SHwCfg *)pstVapiReq->pvUserData);

		if (Status != SUCCESS)
			goto finish;
		break;

	case DEVICE_HARDWARE_900_FINISHED:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);
		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetDeviceHardware900: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:
	UT_Log(VCORE, DEBUG, "VFSM_SetDeviceHardware900: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetDeviceHardware900: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

VSTATUS VFSM_SetDeviceHardware300(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	DEVID DevId;
	SVapiReq * pstChildVapiReq;
	DevId  = pstSupvChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_SetDeviceHardware300: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	switch (pstVapiReq->usReqState)
	{
	case DEVICE_HARDWARE_300_INIT:

		/* The hardware init is not the same for PCI/CSME */
		switch (pstSupvChnl->pstDev->eItfType)
		{
		case ePCI_ITF:
			pstVapiReq->usReqState = DEVICE_HARDWARE_300_SET_BUS_ARM;
			break;

		case eCSM_ITF:
			pstVapiReq->usReqState = DEVICE_HARDWARE_300_SET_SDRAM;
			break;

		default:
			UT_ErrorLog(VCORE, "Wrong interface type %d", pstSupvChnl->pstDev->eItfType);
			Status = VAPI_ERR_UNSUPP_FEATURE;
			goto finish;

			break;
		}

		Status = VDEV_BRMSetPll(pstSupvChnl, (SHwCfg *)pstVapiReq->pvUserData, PLL_BUS_AMBA);

		if (Status != SUCCESS)
			goto finish;

		break;

	case DEVICE_HARDWARE_300_SET_BUS_ARM:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		/* for PCI */
		pstVapiReq->usReqState = DEVICE_HARDWARE_300_SET_SDRAM;
		Status = VDEV_BRMSetPll(pstSupvChnl,(SHwCfg *)pstVapiReq->pvUserData, PLL_BUS_ARM);
	
		if (Status != SUCCESS)
			goto finish;

		break;

	case DEVICE_HARDWARE_300_SET_SDRAM:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		/* this allocate a new request */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SHwCfg));

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		SetVAPIDeviceHwConfig((SHwCfg *) pstChildVapiReq->pvUserData, pstVapiReq->pvUserData, pstSupvChnl->pstDev);

		/* initialise the Child request */
		if (pstSupvChnl->pstDev->ucDevType == DEV_TYPE_M823XX)
		{
			VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
					pstVapiReq,		/* Parent Request */ 
					VFSM_SetSdram_300_1,	/* Child Request Handler */
					SET_SDRAM_300_1_INIT);	/* Child Request handler state */
		}
		else
		{
			VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
					pstVapiReq,		/* Parent Request */ 
					VFSM_SetSdram_300_2, 	/* Child Request Handler */
					SET_SDRAM_300_2_INIT);	/* Child Request handler state */
		}

		if (pstSupvChnl->pstDev->eItfType == ePCI_ITF)
			pstVapiReq->usReqState = DEVICE_HARDWARE_300_SET_CS_BASEADDR;
		else
			pstVapiReq->usReqState = DEVICE_HARDWARE_300_SET_CS_SDRAM;

		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case DEVICE_HARDWARE_300_SET_CS_BASEADDR:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = DEVICE_HARDWARE_300_SET_CS_SDRAM;
		Status = VDEV_BRMDwordWriteAddr(pstSupvChnl, 0);
		if (Status != SUCCESS)
			goto finish;

		break;

	case DEVICE_HARDWARE_300_SET_CS_SDRAM:
		if (pstSupvChnl->pstDev->eItfType == ePCI_ITF)
			Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);
		else
			/* retrieve the child status */
			Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;
		
		/* for PCI interface we send the VDEV_BRMSetCSParam command state */
		pstVapiReq->usReqState = DEVICE_HARDWARE_300_FINISHED;
		Status = VDEV_BRMSetCSParam(pstSupvChnl, (SHwCfg *)pstVapiReq->pvUserData);
		if (Status != SUCCESS)
			goto finish;

		break;

	case DEVICE_HARDWARE_300_FINISHED:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);
		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetDeviceHardware300: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;
	}

	return SUCCESS;

finish:
	UT_Log(VCORE, DEBUG, "VFSM_SetDeviceHardware300: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetDeviceHardware300: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

VSTATUS VFSM_SetSdram_300_1(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	DEVID DevId;
	DevId  = pstSupvChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_SetSdram_300_1: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	switch (pstVapiReq->usReqState)
	{
	case SET_SDRAM_300_1_INIT:
		if (pstSupvChnl->pstDev->eItfType == ePCI_ITF)
			pstVapiReq->usReqState = SET_SDRAM_300_1_PARAM2;
		else
			/* For CSME interface configure only SRDAM_PARAM_1*/
			pstVapiReq->usReqState = SET_SDRAM_300_1_FINISHED;
	
		Status = VDEV_BRMSetSdramParam(pstSupvChnl, (SHwCfg *)pstVapiReq->pvUserData);
		if (Status != SUCCESS)
			goto finish;
		
		break;

		/* Only for PCI interface  */
	case SET_SDRAM_300_1_PARAM2:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);
		
		if (Status != SUCCESS)
			goto finish;
	
		pstVapiReq->usReqState = SET_SDRAM_300_1_FINISHED;
		Status = VDEV_BRMSetSdramParam2(pstSupvChnl, (SHwCfg *)pstVapiReq->pvUserData);
		if (Status != SUCCESS)
			goto finish;

		break;

	case SET_SDRAM_300_1_FINISHED:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);
		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetSdram_300_1: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;
	}

	return SUCCESS;

finish:
	UT_Log(VCORE, DEBUG, "VFSM_SetSdram_300_1: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetSdram_300_1: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

VSTATUS VFSM_SetSdram_300_2(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	DEVID DevId;
	U8 ucParamNum;
	U32 SysConfig, *ptrBufferType;
	U32 *ptrSdramParam_C300_2;
	DevId  = pstSupvChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_SetSdram_300_2: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	ptrSdramParam_C300_2 = ((SHwCfg *)pstVapiReq->pvUserData)->ptrSdramParam_C300_2;

	/* first entry */
	if (pstVapiReq->pvExecData == NULL)
		pstVapiReq->pvExecData = UT_Calloc(2, sizeof(U32));

	if (pstVapiReq->pvExecData == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto finish;
	}	

	ucParamNum = *((U8 *) pstVapiReq->pvExecData);
	ptrBufferType = &((U32*)pstVapiReq->pvExecData)[1];

	switch (pstVapiReq->usReqState)
	{
	case SET_SDRAM_300_2_INIT:
		Status = VDEV_BRMDwordRead(pstSupvChnl, 0x1007001C);

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = SET_SDRAM_300_2_SET_BUFFER_TYPE;

		break;

	case SET_SDRAM_300_2_SET_BUFFER_TYPE:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		if (pstSupvChnl->pstDev->eItfType == ePCI_ITF)
		{
			SysConfig = pstMsg->mailbox.reg0 << 16;
			SysConfig |= pstMsg->mailbox.reg2;
		}
		else
		{
			SysConfig = UT_BE2CPU16(((U16 *) pstMsg->fifo)[3]) << 16;
			SysConfig |= UT_BE2CPU16(((U16 *) pstMsg->fifo)[4]);
		}

		if (SysConfig & 0x0F00)
			*ptrBufferType = 0x588;
		else
			*ptrBufferType = 0x421;

		if (pstSupvChnl->pstDev->eItfType == ePCI_ITF)
		{
			Status = VDEV_BRMDwordWriteAddr(pstSupvChnl, 0x10070034);
			pstVapiReq->usReqState = SET_SDRAM_300_2_SET_BUFFER_TYPE_PCI;
		}
		else
		{
			Status = VDEV_DwordWrite(pstSupvChnl, 0x10070034, *ptrBufferType);
			pstVapiReq->usReqState = SET_SDRAM_300_2_INIT_SDC;
		}

		if (Status != SUCCESS)
			goto finish;

		break;

	case SET_SDRAM_300_2_SET_BUFFER_TYPE_PCI:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

		if (Status != SUCCESS)
			goto finish;

		Status = VDEV_BRMDwordWriteData(pstSupvChnl, *ptrBufferType);
		
		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = SET_SDRAM_300_2_INIT_SDC;

		break;

	case SET_SDRAM_300_2_INIT_SDC:
		/* first entry */
		if (ucParamNum != 0)
		{
			Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);

			if (Status != SUCCESS)
				goto finish;
		}

		if (pstSupvChnl->pstDev->eItfType == ePCI_ITF)
		{
			pstVapiReq->usReqState = SET_SDRAM_300_2_WRITE_DATA;
			Status =  VDEV_BRMDwordWriteAddr(pstSupvChnl, ptrSdramParam_C300_2[2*ucParamNum + 0]);
		}
		else
		{
			if ((ucParamNum + 1) < ((SHwCfg *)pstVapiReq->pvUserData)->usSdramParamNum)
				pstVapiReq->usReqState = SET_SDRAM_300_2_INIT_SDC;
			else
				pstVapiReq->usReqState = SET_SDRAM_300_2_FINISHED;

			if (ptrSdramParam_C300_2[2*ucParamNum] != 0x1008008C)
			{
				Status =  VDEV_DwordWrite(pstSupvChnl, ptrSdramParam_C300_2[2*ucParamNum + 0],
								ptrSdramParam_C300_2[2*ucParamNum + 1]);
			}
			else
			{	/* EXPCLK DLL is fixed to 39 for the -12 devices, auto-calibrated value is used for -11 */
				Status =  VDEV_DwordWrite(pstSupvChnl, 0x1008008C, *ptrBufferType & 1 ? 0 : 0xA7);
			}

			*((U8 *) pstVapiReq->pvExecData) += 1;
		}
			
		if (Status != SUCCESS)
			goto finish;
		
		break;
		
	case SET_SDRAM_300_2_WRITE_DATA:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);
		
		if (Status != SUCCESS)
			goto finish;

		if ((ucParamNum + 1) < ((SHwCfg *)pstVapiReq->pvUserData)->usSdramParamNum)
			pstVapiReq->usReqState = SET_SDRAM_300_2_INIT_SDC;
		else
			pstVapiReq->usReqState = SET_SDRAM_300_2_FINISHED;

		if (ptrSdramParam_C300_2[2*ucParamNum] != 0x1008008C)
		{
			Status = VDEV_BRMDwordWriteData(pstSupvChnl, ptrSdramParam_C300_2[2*ucParamNum + 1]);
		}
		else
		{	/* EXPCLK DLL is fixed to 39 for the -12 devices, auto-calibrated value is used for -11 */
			Status = VDEV_BRMDwordWriteData(pstSupvChnl, *ptrBufferType & 1 ? 0 : 0xA7);
		}
		
		if (Status != SUCCESS)
			goto finish;

		*((U8 *) pstVapiReq->pvExecData) += 1;

		break;

	case SET_SDRAM_300_2_FINISHED:
		Status = Check_BootResponse(pstSupvChnl, Event, pstMsg);
		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetSdram_300_2: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;
	}

	return SUCCESS;

finish:
	UT_Log(VCORE, DEBUG, "VFSM_SetSdram_300_2: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetSdram_300_2: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}


/****************************************************************************
 * VFSM_CESoPSN_Mapper : The function does the following things -
 ***************************************************************************/
VSTATUS VFSM_CESoPSN_Mapper(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	SFsmCesoPsnMapChannelInfo *pstSFsmCesoPsnMapChannelInfo = NULL;
	int i;
	IN SChnl * pstTempChnl = NULL;
	DEVID DevId;
	DevId  = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_CESoPSN_Mapper: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	pstSFsmCesoPsnMapChannelInfo = (SFsmCesoPsnMapChannelInfo *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case SET_CSE_MAP_CHANNEL_INIT:

		/* Check if the master connection is still there*/
		pstTempChnl = DMGR_GetConnection(pstSFsmCesoPsnMapChannelInfo->MasterChannel);
		if(pstTempChnl == NULL)
		{
			UT_ErrorLog(APPITF, "VFSM_CESoPSN_Mapper: Invalid Master Conn id %u\n", 
					pstSFsmCesoPsnMapChannelInfo->MasterChannel);
			Status = VAPI_ERR_INVALID_CONNID;
			goto finish;
		}

		/* do not check connection for timeslot mapping (not created*/
		if(pstSFsmCesoPsnMapChannelInfo->MapType == eChannelMap)
		{
			/* Check if all connections to be maped are still there*/
			for(i = 0; i < pstSFsmCesoPsnMapChannelInfo->NumOfChannels; i++)
			{
				pstTempChnl = DMGR_GetConnection(pstSFsmCesoPsnMapChannelInfo->pusLinkedChannels[i]);
				if(pstTempChnl == NULL)
				{
					UT_ErrorLog(APPITF, "VFSM_CESoPSN_Mapper: Invalid Mapped Conn id %u\n", 
					pstSFsmCesoPsnMapChannelInfo->pusLinkedChannels[i]);
					Status = VAPI_ERR_INVALID_CONNID;
					goto finish;
				}
				/* save the associated MSP channel in the array*/
				pstSFsmCesoPsnMapChannelInfo->pusLinkedChannels[i] = pstTempChnl->usMSPChnlId;
			}
		}

		pstVapiReq->usReqState = SET_CSE_MAP_CHANNEL_FINISHED;
		Status = VDEV_CESoPSN_Mapper(pstChnl, pstSFsmCesoPsnMapChannelInfo);
		if (Status != SUCCESS)
			goto finish;
		break;


	case SET_CSE_MAP_CHANNEL_FINISHED:
		if(pstSFsmCesoPsnMapChannelInfo->MapType == eChannelMap)
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_CESOPSN_MAP_CHANS);
		else
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_CESOPSN_SPEC_MAP_CHANS);

		goto finish;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_CESoPSN_Mapper: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:

	UT_Log(VCORE, DEBUG, "VFSM_CESoPSN_Mapper: Completing req state(%u) status(%d) dev(%u)\n",
		       pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_CESoPSN_Mapper: Exiting status(%d) dev(%u)\n", Status, DevId);
	return Status;
}

VSTATUS VFSM_PurgeMspResources(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	DEVID DevId= pstSupvChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_PurgeMspResources: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	switch (pstVapiReq->usReqState)
	{
	case PURGE_MSP_RES_INIT:
		if (!pstSupvChnl->pstDev->bIsDeviceInitialized)
		{
			UT_ErrorLog(APPITF, "VFSM_PurgeMspResources: device(%u) not initialized\n", DevId);
			Status = VAPI_ERR_DEVICE_NOT_INITIALIZED;
			goto finish;
		}

		if (pstSupvChnl->pstDev->MajorRelease < MR23)
		{
			UT_ErrorLog(VCORE, "VFSM_PurgeMspResources: CHANNELS_RESET feature is not supported\n");
			Status = VAPI_ERR_UNDEFINED;
			goto finish;
		}

		pstVapiReq->usReqState = PURGE_MSP_RES_WAIT_IND;

		Status = VDEV_ChannelsReset(pstSupvChnl);
		if (Status != SUCCESS)
			goto finish;

		break;

	case PURGE_MSP_RES_WAIT_IND:
		Status = VCORE_CheckStatus(Event, pstMsg, pstSupvChnl->usMSPChnlId,
						CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_CHANNELS_RESET);
		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = PURGE_MSP_RES_FINISHED;

		pstVapiReq->pvExecData = (void *) UT_TimerCreate();

		/* start timer to wait indication from MSP*/
		UT_TimerStart(pstVapiReq->pvExecData, stVAPIConfig_g.uiMspRespTimeout,
						VCORE_ChannelResetTimeoutHdlr, (void *) pstSupvChnl);

		break;

	case PURGE_MSP_RES_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, pstSupvChnl->usMSPChnlId,
						CMD_CLASS_CONF_DEVICE, CMD_TYPE_INDICATION, FC_CHANNELS_RESET_COMPLETE);

		if (Status == SUCCESS)
			UT_TimerStop((STimer *) pstVapiReq->pvExecData);

		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_PurgeMspResources: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;
	}

	return SUCCESS;

finish:
	UT_Log(VCORE, DEBUG, "VFSM_PurgeMspResources: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	if (pstVapiReq->pvExecData)
		UT_TimerDelete((STimer *) pstVapiReq->pvExecData);

	pstVapiReq->pvExecData = NULL;

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_PurgeMspResources: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;

}

