/*! \file connection_itf.c
 @defgroup connection_level Connection related API functions
 *  @{
 */

/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */


#include "ut.h"
#include "vapi.h"
#include "dmgr.h"
#include "vcore.h"
#include "vcore_voip.h"
#include "appitf.h"
#include "cfg.h"
#include "msp.h"

extern SToneGenParams astDtmfToneGenParams_g[];
extern SAVLTree *pstResTree_g;

/***************************************************************************
 * VAPI_CreateConnection: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This API creates a Comcerto device channel. \n
*	This channel is set up as g.711 codec, and packets disabled if the channel type is VoIP. \n
*	This channel is set up as UDPTL, and packets disabled if the channel type is FoIP. \n
*	The channel is in the state to be ready to generate and detect tones on the TDM side (if VoIP).  \n
*	This is known "as signaling" state or TDM Active Operation Mode. \n
*	VCEOPT Packet generation will be disabled and VOPENA enabled. \n
*	The default parameters are defined by VAPI library. \n
*	If user wants to override them, the VAPI_PassThru() can be used to issue the commands
*	with required parameters. \n
*	The user cannot control the set of MSP commands to be issued at the time of connection creation. \n
*
*	\n \b Note 1:
*	<b>Some configurations command  have to be sent when the channel is Inactive (i.e SRTP, RTCP configuration).\n
*	In this case the command VAPI_SetConnectionState() has to be called before proceeding
*	with these kind of commands.</b>\n
*
*	\n \b Note 2:
*	<b>When VAPI_CreateConnection() is called in ASYNC mode, the internal VAPI resources for the connection \n
*	are allocated only after getting a successful response to SUPVSR_CREATE_CHNL Comcerto command.\n 
*	The application must not send any other command on that connection till the callback returns success for that
*	connection id. \n
*	If the application sends a command before completion callback of VAPI_CreateConnection(), it will get an error
*	response (Invalid Connection Id) as no resources exist for that particular connection ID in VAPI. \n
*	So, user should consider waiting for the completion callback to VAPI_CreateConnection before sending other
*	commands on that Connection. </b>\n
*
*	\n \b Note 3:
*	<b>The VAPI_CreateConnection() is divided in 2 main steps:</b>\n
*	- <b>The device channel creation (SUPVSR_CREATE_CHNL Comcerto command).</b>\n 
*	- <b>The device channel initialization (VCEOPT, ECHCAN, JBOPT, etc Comcerto commands).</b>\n
*	<b>If an error occurs during the step 1 the connection doesn't exist at all 
*	and the returned error is VAPI_ERR_CREATECONN_FAIL.\n
*	If an error occurs during the step 2 the connection exists but not initialized as expected.\n
*	In this case the user has the opportunity to send some configuration commands
*	(i.e VAPI_SetCodecType(), VAPI_SetPacketInterval(), etc ) or VAPI_DestroyConnection() 
*	to remove the connection.</b> \n
*
*	\n \b Note 4:
*	<b>In case that the VAPI_InitDevice() function has not been previously issued, \n
*	VAPI_CreateConnection() returns VAPI_ERR_DEVICE_NOT_INITIALIZED</b> \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">The device on which connection is to be created.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">The connection ID is specified by the user application.  \n
*		The connection ID is assumed unique across all devices and channels.  \n
*		The VAPI library keeps track of the connection resource mapped by the user-specified connection ID. \n
*		The VAPI library looks up the device ID mapped to the connection ID and passes that down to GTL layer</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eConnType</td>
*		<td style="vertical-align: top;">Specifies the type of channel (VoIP, VoATM, T38, etc). \n
*		It can have following values:
*		- eVOIP
*		- eFOIP
*		- eVOATM
*		- eIUUPOAAL2</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usTdmTimeSlot</td>
*		<td style="vertical-align: top;">TDM timeslot for the new channel.\n
*		The TS number must be in a valid range according the TDM parameters.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucAU_Law</td>
*		<td style="vertical-align: top;">TDM side coding law to be used (A_LAW=0 or U_LAW=1).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pfnEventCallback</td>
*		<td style="vertical-align: top;">Connection Level event callback. \n
*		When events will be received for this connection this event callback will be called</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_UNDEFINED
*	\li VAPI_ERR_CREATECONN_FAIL
*	\li VAPI_ERR_TS_IN_USE
*	\li MSP error code
* 
*	\n \b Usage: \n
*	\include create_connection.c
*
*	\n \b Commands:
*	Comcerto commands sent depends on channel type required: \n
*	- SUPVSR_CREATE_CHANNEL
*	VAPI will send several Comcerto APIs (with default parameters) depending on channel type: \n
*	- VoIP:
*		- VCEOPT
*		- ECHOCAN or DFECAN depending on SPU_CTRL mask
*		- JBOPT
*		- DGAIN
*		- TONECTRL
*		- DTMFOPT
*		- TONERELAY
*		- VOPENA - is sent by default \n
*		It depends on the field stVAPIConfig_g.pstVoIPChnlParams->stVopena.uVoiceOperation \n
*		in the global structure SVAPIConfig stVAPIConfig_g.\n
*		If it is set to VOIP_VOPENA_MODE_DISABLE (0x00) then VOPENA is ignored and not issued \n
*		else if uVoiceOperation=VOIP_VOPENA_MODE_ENABLE_RTP, VOPENA command is issued. \n
*		An example of non-issuing VOPENA is available in the \include set_default_config.c
*
*	- FoIP:
*		- FAXOPT
*		- FAXLVL
*	- Other type:
*		- No command is sent \n
*
*/
VSTATUS VAPI_CreateConnection(IN DEVID DevId,
			      IN CONNID ConnId,
			      IN EConnType eConnType,
			      IN U16 usTdmTimeSlot,
			      IN U8 ucAU_Law, IN SRequest * pstRequest, IN PFNEventCallback pfnEventCallback)
{
	SVapiReq *pstVapiReq;
	SDevice *pstDevice;
	SChnl *pstChnl;
	void *pvChnDef;
	VSTATUS Status;
	Boolean bIsSync;
	SChnl **ppstUserData;
	SConnResources *pstConRes = NULL;
	SResRequest stResRequest;

	if (eConnType != eVOIP && eConnType != eFOIP && eConnType != eVOATM && 
		eConnType != eIUUPORTP && eConnType != eIUUPOAAL2 && 
		eConnType != eComplexTone && eConnType != eToneDet && 
		eConnType != eIpRedirect && eConnType != eHDL && 
		eConnType != eHDLCOTDM && eConnType != eCCS && 
		eConnType != eCSEoPSN && eConnType != eSLIC24  &&
		eConnType != eWARNING_TONEGEN && eConnType != eDSPDIAG && eConnType != eAAL1CES)
	{
		UT_ErrorLog(APPITF, "VAPI_CreateConnection: Connection type(%d) is invalid. \n", eConnType);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	/* the coding law can be on 0 (ulaw) or 1 (Alaw) */
	if (ucAU_Law != 1 && ucAU_Law != 0)
	{
		UT_ErrorLog(APPITF, "VAPI_CreateConnection: Invalid coding law(%d).\n", ucAU_Law);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_DeviceGenericCheck(DevId, "VAPI_CreateConnection", &pstDevice);

	if (Status != SUCCESS)
		goto err_out;

	if (!pstDevice->bIsDeviceInitialized)
	{
		UT_ErrorLog(APPITF, "VAPI_CreateConnection: device(%u) not initialized\n", DevId);
		Status = VAPI_ERR_DEVICE_NOT_INITIALIZED;
		goto err_out;
	}

	/* If the ressource manager is enabled check if resources have been allocated */
	if (pstDevice->stDevRes.bIsResourceMngr == True)
	{
		/*Get the ressource for this connection */
		pstConRes = (SConnResources *) UT_SearchInAVLTree(pstResTree_g, ConnId);
		if (pstConRes == NULL)
		{
			UT_ErrorLog(APPITF, "VAPI_CreateConnection: no resource allocated for connection (%u) \n", ConnId);
			Status = VAPI_ERR_NO_RESOURCE_ALLOC;
			goto err_out;
		}
	}

	UT_Log(APPITF, INFO, "Entered VAPI_CreateConnection: devid(%d), connid(%d)\n", DevId, ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_CREATE_CONNECTION_PARAMS;

	if (DMGR_GetConnection(ConnId) != NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_CreateConnection: Connection %u already exists\n", ConnId);
		Status = VAPI_ERR_INVALID_CONNID;
		goto err_out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SChnl *));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}

	/* Create the connection structure to handle the channel
	This structure is freed by DestroyConnection or by the VFSM_CreateConnection
	in case the device create channel fails */
	pstChnl = UT_Calloc(1, sizeof(SChnl));
	if (pstChnl == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_req;
	}

	/* initialise the connection structure */
	pstChnl->ConnId = ConnId;
	pstChnl->pstDev = pstDevice;
	pstChnl->pfnEventCallback = pfnEventCallback;
	pstChnl->usConnType = eConnType;
	pstChnl->usTimeSlot = usTdmTimeSlot;
	pstChnl->ePartType = eNOTPART;
	/* the connection is not active */
	pstChnl->bIsActive = False;
	pstChnl->bEnableFaxAutoSwitch = False;
	pstChnl->usSwitchoverEventMask = eLOCAL_CNG | eREM_CNG;

	/* pass the connection structure pointer to the request*/ 
	ppstUserData = pstVapiReq->pvUserData;
	*ppstUserData = pstChnl;

	/* retrieve the default configuration for this type of channel (VoIP or FoIP)
	   if the channel type is not VoIP or FoIP there are no defualt parameters */
	pvChnDef = DMGR_GetChnlDefaults(pstChnl->usConnType);

	if (pstChnl->usConnType == eVOIP || pstChnl->usConnType == eFOIP)
	{
		pstChnl->pstChnlParams = UT_AllocMem(sizeof(SVoIPChnlParams));
		if (pstChnl->pstChnlParams == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			goto err_chnl;
		}

		if (pvChnDef != NULL)
		{
			UT_MemCopy(pstChnl->pstChnlParams, pvChnDef, sizeof(SVoIPChnlParams));
		}
		else
		{
			Status = VAPI_ERR_UNDEFINED;
			goto err_params;
		}

		/*If the connection is FOIP to set the voice operation accordingly */
		if (pstChnl->usConnType == eFOIP)
			pstChnl->pstChnlParams->stVopena.mode = VOIP_VOPENA_MODE_ENABLE_UDPTL;

		/* Overwrite the default value by the one requested by the user */
		pstChnl->pstChnlParams->stVoiceOpt.param_4.bits.compander = ucAU_Law;

	}

	/* Initialize few variables, timer,and mutexes */
	Status = VCORE_OpenConnection(pstChnl);
	if (Status != SUCCESS)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_params;
	}

	/* Resource manager is only for VoIP or FoIP connection*/
	if ((pstChnl->usConnType == eVOIP) || (pstChnl->usConnType == eFOIP))
	{
		/*Check required resource against allocated resources */
		stResRequest.eConnType = pstChnl->usConnType;
		stResRequest.ePartType = pstChnl->ePartType;
		/* If connection type is VoIP or FoIP use configured values*/
		stResRequest.eCodecType = pstChnl->pstChnlParams->eCodecVal;
		stResRequest.ucPacketSize = pstChnl->pstChnlParams->stVoiceOpt.param_4.bits.packet_interval;
		stResRequest.ucVAD = pstChnl->pstChnlParams->stVoiceOpt.param_4.bits.vadtype;
		stResRequest.ucG729_1_rate = 0;
	
		Status = VRES_CheckRes(pstChnl, &stResRequest, pstConRes);
		if (Status != SUCCESS)
			goto err_params;
	}

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_CreateConnection, ConnId);

	bIsSync = pstVapiReq->bIsSync;

	/* process the request */
	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_CreateConnection: Exiting status(%d), dev(%u), conn(%u)\n", Status, DevId, ConnId);
/* no error exit */
	return Status;

      err_params:
	UT_FreeMem(pstChnl->pstChnlParams);
      err_chnl:
	UT_FreeMem(pstChnl);
      err_req:
	VCORE_FreeRequest(pstVapiReq);
      err_out:
	return Status;
}

/***************************************************************************
 * VAPI_AllocateConnection: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This API creates a Comcerto device channel. \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">The device on which connection is to be created.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">The connection ID is specified by the user application.  \n
*		The connection ID is assumed unique across all devices and channels.  \n
*		The VAPI library keeps track of the connection resource mapped by the user-specified connection ID. \n
*		The VAPI library looks up the device ID mapped to the connection ID and passes that down to GTL layer</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eConnType</td>
*		<td style="vertical-align: top;">Specifies the type of channel (VoIP, VoATM, T38, etc). \n
*		It can have following values:
*		- eVOIP
*		- eFOIP
*		- eVOATM
*		- eIUUPOAAL2, etc ..</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eConnMode</td>
*		<td style="vertical-align: top;">Channel mode Narrow or Wide band.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucNumOfParam</td>
*		<td style="vertical-align: top;">Number of parameters contained in the usParams buffer.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usParams</td>
*		<td style="vertical-align: top;">Ponter to a buffer containing the channel parameters \n
*						This buffer contains mainly timeslot number.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pfnEventCallback</td>
*		<td style="vertical-align: top;">Connection Level event callback. \n
*		When events will be received for this connection this event callback will be called</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_UNDEFINED
*	\li MSP error code
* 
*	\n \b Usage: \n
*	\include allocate_connection.c
*
*	\n \b Commands:
*	Comcerto commands sent depends on channel type required: \n
*	- SUPVSR_CREATE_CHANNEL
*
*/
VSTATUS VAPI_AllocateConnection(IN DEVID DevId,
			      IN CONNID ConnId,
			      IN EConnType eConnType,
			      IN EConnMode eConnMode,
			      IN U8 ucNumOfParams,
			      IN U16 *pusParams, IN SRequest * pstRequest, IN PFNEventCallback pfnEventCallback)
{
	SVapiReq *pstVapiReq;
	SDevice *pstDevice;
	SChnl *pstChnl;
	VSTATUS Status;
	Boolean bIsSync;
	SChnl **ppstUserData;
	int i;

	if ((eConnMode != eWideBand)  && (eConnMode != eNarrowBand))
	{
		UT_ErrorLog(APPITF, "VAPI_AllocateConnection: Connection mode(%d) is invalid. \n", eConnMode);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_DeviceGenericCheck(DevId, "VAPI_AllocateConnection", &pstDevice);

	if (Status != SUCCESS)
		goto err_out;

	if (!pstDevice->bIsDeviceInitialized)
	{
		UT_ErrorLog(APPITF, "VAPI_AllocateConnection: device(%u) not initialized\n", DevId);
		Status = VAPI_ERR_DEVICE_NOT_INITIALIZED;
		goto err_out;
	}

	UT_Log(APPITF, INFO, "Entered VAPI_AllocateConnection: devid(%d), connid(%d)\n", DevId, ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_ALLOCATE_CONNECTION_PARAMS;

	if (NULL != DMGR_GetConnection(ConnId))
	{
		UT_ErrorLog(APPITF, "VAPI_AllocateConnection: Connection %u already exists\n", ConnId);
		Status = VAPI_ERR_INVALID_CONNID;
		goto err_out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SChnl *));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}

	/* Create the connection structure to handle the channel
	This structure is freed by DestroyConnection or by the VFSM_CreateConnection
	in case the device create channel fails */
	pstChnl = UT_Calloc(1, sizeof(SChnl));
	if (pstChnl == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_req;
	}

	/* initialise the connection structure */
	pstChnl->ConnId = ConnId;
	pstChnl->pstDev = pstDevice;
	pstChnl->pfnEventCallback = pfnEventCallback;
	pstChnl->usConnType = eConnType;
	pstChnl->usTimeSlot = pusParams[0];
	pstChnl->usMode = eConnMode;

	if (pstChnl->usMode == eWideBand)
	{
		/*limit the number of TS to 4 (in case of user mistake)*/
		if (ucNumOfParams > MAX_WB_TS)
			ucNumOfParams = MAX_WB_TS;

		for (i = 0; i < ucNumOfParams-1; i ++)
			pstChnl->usExtTimeSlot[i] = pusParams[i + 1];
	}

	pstChnl->ePartType = eNOTPART;
	/* the connection is not active */
	pstChnl->bIsActive = False;

	/* pass the connection structure pointer to the request*/ 
	ppstUserData = pstVapiReq->pvUserData;
	*ppstUserData = pstChnl;

	pstChnl->pstChnlParams = UT_AllocMem(sizeof(SVoIPChnlParams));
	if (pstChnl->pstChnlParams == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_chnl;
	}

	/* Init default internal variables to match the MSP default values 
	in case of VoIP or FoIP channel type*/
	if (pstChnl->usConnType == eVOIP || pstChnl->usConnType == eFOIP)
	{
		DMGR_SetVoIPDevDefaults(pstChnl->pstChnlParams);
	}

	/* Initialize few variables, timer,and mutexes */
	Status = VCORE_OpenConnection(pstChnl);
	if (Status != SUCCESS)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_params;
	}

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_AllocateConnection, ConnId);

	bIsSync = pstVapiReq->bIsSync;

	/* process the request */
	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_AllocateConnection: Exiting status(%d), dev(%u), conn(%u)\n", Status, DevId, ConnId);
/* no error exit */
	return Status;

      err_params:
	UT_FreeMem(pstChnl->pstChnlParams);
      err_chnl:
	UT_FreeMem(pstChnl);
      err_req:
	VCORE_FreeRequest(pstVapiReq);
      err_out:
	return Status;
}

/***************************************************************************
 * VAPI_DestroyConnection: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	Stops voice processing and destroys the channel associated with the connection ID \n
*	It also removes all the resources associated with the channel & the channel from the channel list. \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">The identifier of connection that is to be destroyedr</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include destroy_connection.c
*
*	\n \b Commands:
*	- VoIP_VOPENA
*	- SUPVSR_DESTROY_CHANNEL
*
*/
VSTATUS VAPI_DestroyConnection(IN CONNID ConnId, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq;
	SChnl *pstChnl;
	VSTATUS Status;
	Boolean bIsSync;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_DestroyConnection", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_DestroyConnection: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_DESTROY_CONNECTION_PARAMS;

	/*allocate memory for the request and its UserData (if any). If error get out */
	pstVapiReq = VCORE_AllocateRequest(0);
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	if (pstChnl->stRecPlayInfo.uiStopPlayCount < pstChnl->stRecPlayInfo.uiStartPlayCount)
	{
		UT_MutexLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
		pstChnl->stRecPlayInfo.uiStopPlayCount = pstChnl->stRecPlayInfo.uiStartPlayCount;
		UT_MutexUnLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
	}

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest,
				VFSM_DestroyConnection,	/* pfnReqFSMHdlr */
				ConnId);

	bIsSync = pstVapiReq->bIsSync;

	pstVapiReq->usReqState = DESTROY_CONN_INIT;

	/* remove the connection from the AVL tree 
	this prevent adding new requests to this connection */
	VCORE_RemoveConnection(pstChnl);

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		UT_FreeMem(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_DestroyConnection: Exiting status(%d), conn(%u)\n", Status, ConnId);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_ChangeConnectionType
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This API modifies a Comcerto device channel type. \n
*	Depending on the type the VAPI_ChangeConnectionType() handler:
*	- parses the parameters
*	- checks their validity
*	- destroys the current MSP channel associated to the connection
*	- creates the MSP channel or participant with the appropriated MSP command.
*	A connection with the specified ID must have been previously created with \n
*	VAPI_CreateConnection() / VAPI_AllocateConnection() or VAPI_CrearteParticipant(). \n
*	The VAPI library keeps track of the connection resource mapped by the user-specified\n
*	connection ID. \n
*	<b> The parameters must be MSP create channel/participant API relevant.</b>\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection to be switched in T38 mode.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstChangeConnectionInfo</td>
*		<td style="vertical-align: top;">Pointer to structure that specifies the modify connection settings.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">NumParam</td>
*		<td style="vertical-align: top;">Number of parameters.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pusParams</td>
*		<td style="vertical-align: top;">Parameters for the new connection (i.e. timeslots).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_CREATECONN_FAIL
*
*	\n \b Usage:
*	\include change_connection_type.c
*
*	\n \b Commands:
*	- SUPVSR_DESTROY_CHANNEL
*	- SUPVSR_CREATE_CHANNEL 
*
*/
VSTATUS VAPI_ChangeConnectionType(IN CONNID ConnId,
				IN SChangeConnInfo *pstChangeConnInfo, 
				IN U16 NumParam,
				IN U16 *pusParams,
				IN SRequest *pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl *pstChnl = NULL;
	VSTATUS Status;
	Boolean bIsSync;
	SFullChangeConnInfo *pstUserData = NULL;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_ChangeConnectionType ", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_ChangeConnectionType : conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_CHANGE_CONNECTION_TYPE;

	if ((pstChangeConnInfo->eConnMode != eWideBand)  && (pstChangeConnInfo->eConnMode != eNarrowBand))
	{
		UT_ErrorLog(APPITF, "VAPI_ChangeConnectionType : Connection mode(%d) is invalid. \n", pstChangeConnInfo->eConnMode);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	if (pstChangeConnInfo->eChangeOpMode > eOpModeIP)
	{
		UT_ErrorLog(APPITF, "VAPI_ChangeConnectionType : Operation mode(%d) is invalid. \n", pstChangeConnInfo->eChangeOpMode);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	pstVapiReq = VCORE_AllocateRequest(sizeof(SFullChangeConnInfo));

	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstUserData = pstVapiReq->pvUserData;

	UT_MemCopy(&pstUserData->stChangeConnInfo, pstChangeConnInfo, sizeof(SChangeConnInfo));
	pstUserData->NumParam = NumParam;
	pstUserData->pusParams = pusParams;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest,
				VFSM_ChangeConnection,	/* pfnReqFSMHdlr */
				ConnId);		/*uiID */

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}
	UT_Log(APPITF, INFO, "VAPI_ChangeConnectionType : Exiting status(%d), conn(%u)\n", Status, ConnId);

      out:
	return Status;
}
/***************************************************************************
 * VAPI_SetConnectionState: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	Changes the connection operation mode accordingly the eOpMode parameter (this API is applicable to conference participant too). \n
*	It could be: \n
*	\li  Inactive:    TDM detection inactive, packetization inactive. \n
*	\li  TDM Active:  TDM detection active, packetization inactive. \n
*	\li  Active:      TDM detection active, packetization active. \n
* 
*	By setting the connection to Inactive:\n
*	The stream of packets from the TDM bus is not be decoded \n
*	No packet is sent to the transport interface. \n
*	\n
*	By setting the connection to TDM Active:\n
*	The TDM tone detection is active and tones can be generated to the TDM \n 
*	\n
*	By setting the connection to Active:\n
*	The stream of packets from the TDM bus will be decoded into PCM signals on the line,\n
*	and PCM signals from the line will be encoded into a stream of packets. \n 
*	Internally the API issues VoIP_VCEOPT (with packet generation enabled) and VoIP_VOPENA.\n
*	If the user application did not configure the connection using VAPI_SetCodecType(), VAPI_SetPacketInterval(), \n 
*	then the default 20ms g.711 packet type is assumed.\n
*	Like for the TDM active mode, the tone detection / generation is active..\n
*	The IP header must be set before setting the connection to Active (VAPI_SetConnIpParams()).
* 
*	<table style="text-align: left; width: 640px" border="1" cellpadding="2" cellspacing="2">
*	<tr>
*	<td style="background-color: rgb(213, 225, 232);"><b>Current mode</b></td>
*	<td style="background-color: rgb(213, 225, 232);"><b>New mode</b></td>
*	<td style="background-color: rgb(213, 225, 232);"><b>Commands Sent</b></td>
*	</tr>
*	<tr><td>Inactive</td><td>TdmActive</td><td>VCEOPT with Bit PacketGeneration disable = 1<br>VOPENA</td></tr>
*	<tr><td>Inactive</td><td>Active</td><td>VCEOPT with Bit PacketGeneration disable = 0<br>VOPENA</td></tr>
*	<tr><td>TDM Active</td><td>Active</td><td>VOPENA</td></tr>
*	<tr><td>Active</td><td>TDM Active</td><td>VCEOPT with Bit PacketGeneration disable = 1</td></tr>
*	<tr><td>Active</td><td>Inactive</td><td>VOPENA Disable</td></tr>
*	<tr><td>TDM Active</td><td>Inactive</td><td>VOPENA Disable</td></tr>
*	</table>
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">The identifier of connection to which that 
*		the processing has to be changed</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eOpMode</td>
*		<td style="vertical-align: top;">The required Operation Mode.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include connection_state.c
*
*	\n \b Commands:
*	- VoIP_VCEOPT
*	- VoIP_VOPENA
*	- VoIP_VOPDIS
*/
VSTATUS VAPI_SetConnectionState(IN CONNID ConnId, IN EConnOpMode eOpMode, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq;
	SChnl *pstChnl;
	VSTATUS Status;
	Boolean bIsSync;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_SetConnectionState", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_SetConnectionState: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_SET_CONNECTION_OPMODE_PARAMS;

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(0);
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	switch(eOpMode)
	{
	case eInactive:
		/* Fill the request with the regular parameters */
		VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_StopConnection,	/* pfnReqFSMHdlr */
				ConnId);	/*uiID */
		break;

	case eTdmActive:
		/* Fill the request with the regular parameters */
		VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_StartConnectionTDM,	/* pfnReqFSMHdlr */
				ConnId);	/*uiID */
		break;

	case eActive:
		/* Fill the request with the regular parameters */
		VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_StartConnectionPacket,	/* pfnReqFSMHdlr */
				ConnId);	/*uiID */
		break;

	default:
		UT_ErrorLog(APPITF, "VAPI_SetConnectionState: Operation Mode %u\n", eOpMode);
		Status = VAPI_ERR_INVALID_OPERATION;
		goto out;
		break;
	}

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_SetConnectionState: Exiting status(%d), conn(%u)\n", Status, ConnId);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_Loopback: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This API creates and removes single channel and inter-channel loop-backs. \n
*	Loopback created of a particular type should be removed by using removal \n
*	eLoopbackType (eREMOVE_XXX) of for the same type.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId1</td>
*		<td style="vertical-align: top;">Connection id on which loopback will be done.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId2</td>
*		<td style="vertical-align: top;">Second connection in case this is an inter-connection loopback 
*		(else not relevant).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eLoopback</td>
*		<td style="vertical-align: top;">Type of loopback requested.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_OPERATION
* 
*	\n \b Usage:
*	\include loopback.c
*
*	\n \b Commands:
*	- For VoIP Loopback
*		- VOIP_LOOPBACK
*	- For THC Loopback 
*		- THC_MODE_ENABLE
*		- THC_REDIRECT_RX
*
*/
VSTATUS VAPI_Loopback(IN CONNID ConnId1, IN CONNID ConnId2, IN ELoopbackType eLoopback, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq;
	SChnl *pstChnl1 = NULL;
	SChnl *pstChnl2 = NULL;
	VSTATUS Status;
	Boolean bIsSync;
	SLpbackUserData *pstUserData;

	if ((ConnId1 == ConnId2) && (eINTER_CHNL_POST_ENCAPS ||eINTER_CHNL_TDM || eTHC))
	{
  		UT_ErrorLog(APPITF, "VAPI_Loopback: For this Loopback type ConnId1 can not be the same as ConnId2\n");
  		Status = VAPI_ERR_INVALID_OPERATION;
  		goto out;
	}

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId1, "VAPI_LoopBack", &pstChnl1);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_Loopback: conn1(%d), conn2(%d), type(%d)\n", ConnId1, ConnId2,
	       eLoopback);

	if ( (eLoopback > eREMOVE_THC_LOOPBACK) || (eLoopback < eTDM))
	{
		UT_ErrorLog(APPITF, "VAPI_Loopback: Unsupported loopback type %u\n", eLoopback);
		Status = VAPI_ERR_INVALID_OPERATION;
		goto out;
	}

	/* log the parameters passed if required (define in appitf.h) */
	LOG_LOOPBACK_PARAMS;

	/* loopback allowed on VoIP channel only */

	if (pstChnl1->usConnType != eVOIP)
	{
		UT_ErrorLog(APPITF, "VAPI_Loopback: Connection %u type is not VOIP\n", ConnId1);
		Status = VAPI_ERR_INVALID_OPERATION;
		goto out;
	}

	/* For loopback involving 2 channels check the state of the second channel */
	if (eLoopback == eTHC || eLoopback == eINTER_CHNL_POST_ENCAPS
	    || eLoopback == eINTER_CHNL_TDM || eLoopback == eREMOVE_INTER_CHNL_POSTENCAPS_LOOPBACK
	    || eLoopback == eREMOVE_INTER_CHNL_TDM_LOOPBACK || eLoopback == eREMOVE_THC_LOOPBACK)
	{
		Status = VCORE_ConnectionGenericCheck(ConnId2, "VAPI_LoopBack", &pstChnl2);

		if (Status != SUCCESS)
			goto out;

		if (pstChnl2->usConnType != eVOIP)
		{
			UT_ErrorLog(APPITF, "VAPI_Loopback: Connection %u type is not VOIP\n", ConnId2);
			Status = VAPI_ERR_INVALID_OPERATION;
			goto out;
		}
		/* we must have both channels on the same device to proceed to loopback */
		if (pstChnl1->pstDev->DevId != pstChnl2->pstDev->DevId)
		{
			UT_ErrorLog(APPITF, "VAPI_Loopback: Connection %u & %u: not on the same device\n", ConnId1, ConnId2);
			Status = VAPI_ERR_INVALID_OPERATION;
			goto out;
		}
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SLpbackUserData));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstUserData = pstVapiReq->pvUserData;
	/*Fill userdata */
	pstUserData->ConnId2 = ConnId2;
	pstUserData->eLoopback = eLoopback;

	switch (eLoopback)
	{
	case eTHC:
	case eREMOVE_THC_LOOPBACK:
		/* Fill the request with the regular parameters */
		VCORE_SetConnectionRequest(pstVapiReq, pstRequest,
				VFSM_LoopbackTHC,	/* pfnReqFSMHdlr */
				ConnId1);	/*uiID */

		/* If we want to stop the THC loopback start the FSM in the rigth state */
		if (eLoopback == eREMOVE_THC_LOOPBACK)
			pstVapiReq->usReqState = THC_DISABLE_LOOPBACK_INIT;

		break;

	case eINTER_CHNL_POST_ENCAPS:
	case eINTER_CHNL_TDM:
	case eREMOVE_INTER_CHNL_POSTENCAPS_LOOPBACK:
	case eREMOVE_INTER_CHNL_TDM_LOOPBACK:
		/* Fill the request with the regular parameters */
		VCORE_SetConnectionRequest(pstVapiReq, pstRequest,
				VFSM_LoopbackInter,	/* pfnReqFSMHdlr */
				ConnId1);	/*uiID */
		break;

	case eTDM:
	case ePRE_ENCAPS:
	case ePOST_ENCAPS: 
	case eRTP_DEC:
	case eREMOVE_CHNL_LOOPBACK:
		/* Fill the request with the regular parameters */
		VCORE_SetConnectionRequest(pstVapiReq, pstRequest,
				VFSM_LoopbackSingle,	/* pfnReqFSMHdlr */
				ConnId1);	/*uiID */

		break;
	}

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl1, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_Loopback: Exiting status(%d), conn1(%d), conn2(%d)\n", Status, ConnId1, ConnId2);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_StartTone: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	Start to generate a tone to the TDM interface for the specified connection. \n
*	The connection may be in either signaling state or packet enabled state. \n
*	Repeated invocations of this function will cease the previous tone and start a new tone. \n
*	<b>NOTE: \n
*	There supported tones are enumerated in EToneId.
*	If user wants to add more tones the user should modify CFG_MAX_TONES in vapi.h.\n
*	Correspondingly user will need to add elements in SCountryInfo::stToneGenParams for newly defined tones.\n
*	Also, user may have to add enumerations for new tones in the EToneId structure.\n
*	The new enumerations should be added after eDTMFTONE_HASH and before eDUMMYTONE.\n
*	The parameter eToneId is used to calculate index to array SCountryInfo:: stToneGenParams.\n
*	The calculation is done as: INDEX = eToneId - eDTMFTONE_HASH - 1.\n
*	Alternatively user can completely override the country specific settings done using \n
*	pfnUsrQueryVapiDefaultsEnumerate API. With this method information for more countries can be added.\n</b>
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection id on which the tone has to be started.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucDir</td>
*		<td style="vertical-align: top;">Specifies direction, whether to generate tone to TDM (0) or packet interface (1).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eToneId</td>
*		<td style="vertical-align: top;">Specifies the type of tone.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eCountryCode</td>
*		<td style="vertical-align: top;">The country code parameter specifies the localized tone setting.  \n
*		The VAPI library by default only supports a small set of country codes. \n
*		The country defaults can be by overrided by assigning their own function pointer to \n
*		 pfnUsrQueryVapiDefaultsEnumerate which is exported by VAPI.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include starttone.c
*
*	\n \b Commands:
*	- VOIP_TONEGEN
*
*/
VSTATUS VAPI_StartTone(IN CONNID ConnId, U8 ucDir, EToneId eToneId, ECountryCode eCountryCode, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl *pstChnl = NULL;
	SCountryInfo *pstCntryInfo = NULL;
	SToneGenParams *pstTone = NULL;
	VSTATUS Status;
	Boolean bIsSync;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_StartTone", &pstChnl);

	if (Status != SUCCESS)
		goto err_out;

	UT_Log(APPITF, INFO, "Entered VAPI_StartTone: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_STARTTONE_PARAMS;

	if (eToneId >= eDUMMYTONE)
	{
		UT_ErrorLog(APPITF, "Invalid Tone ID %d\n", eToneId);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	if (ucDir > 1)
	{
		UT_ErrorLog(APPITF, "Invalid Direction \n");
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SToneGenParams));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}

	pstTone = pstVapiReq->pvUserData;

	pstCntryInfo = DMGR_GetCountryDefaults(eCountryCode);
	if (pstCntryInfo == NULL)
	{
		UT_ErrorLog(APPITF, "Invalid Country Code \n");
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_request;
	}

	if ((eToneId >= eDTMFTONE_1) && (eToneId <= eDTMFTONE_HASH))
	{
		UT_MemCopy(pstTone, &astDtmfToneGenParams_g[eToneId], sizeof(SToneGenParams));
	}
	else
	{
		UT_MemCopy(pstTone, &pstCntryInfo->stToneGenParams[eToneId - eDTMFTONE_HASH - 1],
			   sizeof(SToneGenParams));
	}
	pstTone->ucToneDir = ucDir;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_StartTone,	/* pfnReqFSMHdlr */
				  ConnId);	/*uiID */

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_StartTone: Exiting status(%d), conn(%u)\n", Status, ConnId);

/* no error exit */
	return Status;

      err_request:
	VCORE_FreeRequest(pstVapiReq);
      err_out:
	return Status;
}

/***************************************************************************
 * VAPI_PlayTone: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	Start to generate a tone for the specified connection.\n
*	The tone could be either dual or quad frequency.\n
*	usParamNum defines the number of additional parameters. \n
*	It could be 2 - pass start and limit timestamps parameters as follows:\n
*	{Start Timestamp, Limit Timestamp} or 0 - no timestamps.\n
*	Repeated invocations of this function will cease the previous tone and start a new tone.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection id on which the tone has to be played.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eToneId</td>
*		<td style="vertical-align: top;">Specifies the type of tone.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eToneDir</td>
*		<td style="vertical-align: top;">Specifies the direction to play the tone to (TDM, IP, Both).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstPlayToneOpt</td>
*		<td style="vertical-align: top;">Tone option (only for Dual tone). If it's NULL apply\n 
*					the MSP API default values:\n
*							usMixModeOff = 0x0;\n
*							usMixModeOn = 0x0;\n
*							usProtGen = 0x0;\n
*							usTimestampCtrl = 0x0;\n
*					Else use passed Play Tone Options.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usParamNum</td>
*		<td style="vertical-align: top;">The additional parameters number (only for dual tone).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">...</td>
*		<td style="vertical-align: top;">Additional parameters.</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include play_tone.c
*
*	\n \b Commands:
*	- VOIP_TONEGEN
*	- VoIP_QUADTONEGEN
*
*/
VSTATUS VAPI_PlayTone(IN CONNID ConnId, EToneId eToneId, EToneDir eToneDir, SPlayToneOpt *pstPlayToneOpt, U16 usParamNum, IN SRequest * pstRequest, ...)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl *pstChnl = NULL;
	SPlayToneFlags *pstTone = NULL;
	va_list ParamList;
	U32 ausParam[2];
	int i;
	VSTATUS Status;
	Boolean bIsSync;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_PlayTone", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_PlayTone: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_PLAYTONE_PARAMS;

	if (eToneId >= eDUMMYTONE)
	{
		UT_ErrorLog(APPITF, "Invalid Tone ID %d\n", eToneId);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	if (eToneDir > eDirToBoth)
	{
		UT_ErrorLog(APPITF, "Invalid Direction \n");
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SPlayToneFlags));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstTone = pstVapiReq->pvUserData;

	pstTone->eToneId = eToneId;
	pstTone->eToneDir = eToneDir;

	/* fill user's tone params*/
	if ((usParamNum != 2) && (usParamNum != 0))
	{
		UT_ErrorLog(APPITF, "Invalid param number(%d)\n", usParamNum);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	UT_MemSet(&ausParam[0], 0, sizeof(ausParam));

	/* fill Params array from list*/
	UT_va_start(ParamList, pstRequest);
	for (i = 0; i < usParamNum; i++)
	{
		ausParam[i] = va_arg(ParamList, int);
	}	
	UT_va_end(ParamList);

	pstTone->uiStartTimeStamp = ausParam[0];
	pstTone->uiLimitTimeStamp = ausParam[1];

	if (pstPlayToneOpt != NULL)
	{
		pstTone->usTimestampCtrl = pstPlayToneOpt->usTimestampCtrl;
		pstTone->usProtGen = pstPlayToneOpt->usProtGen;
		pstTone->usMixModeOn = pstPlayToneOpt->usMixModeOn;
		pstTone->usMixModeOff = pstPlayToneOpt->usMixModeOff;
	}
	else
	{
		pstTone->usTimestampCtrl = VOIP_TONEGEN_TIMESTAMP_CTRL_DEFAULT;
		pstTone->usProtGen = VOIP_TONEGEN_PROT_GEN_DEFAULT;
		pstTone->usMixModeOn = VOIP_TONEGEN_MIX_MODE_ON_DEFAULT;
		pstTone->usMixModeOff = VOIP_TONEGEN_MIX_MODE_OFF_DEFAULT;
	}

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, 
				VFSM_PlayTone,	/* pfnReqFSMHdlr */
				ConnId);	/*uiID */

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_PlayTone: Exiting status(%d), conn(%u)\n", Status, ConnId);

out:
	return Status;
}

/***************************************************************************
 * VAPI_StopTone: The function does the following things -
 ***************************************************************************/
/*! \b Description: \n
*	Stops the last tone generated. \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection id on which the tone has to be stopped.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usToneDuration</td>
*		<td style="vertical-align: top;">Turn off the tone after usToneDuration time (0 to 65535 ms).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usSide</td>
*		<td style="vertical-align: top;">Side on which to stop the tone generation (0=Both, 1=TDM, 2=IP).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include starttone.c
*
*	\n \b Commands:
*	- VOIP_TONEOFF
*
*/
VSTATUS VAPI_StopTone(IN CONNID ConnId, IN U16 usToneDuration, IN U16 usSide, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl *pstChnl = NULL;
	VSTATUS Status;
	Boolean bIsSync;
	SStopToneData *pStopToneData  = NULL;;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_StopTone", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_StopTone: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_STOPTONE_PARAMS;

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SStopToneData));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pStopToneData = pstVapiReq->pvUserData;
	pStopToneData->usDuration = usToneDuration;
	pStopToneData->usSide = usSide;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_StopTone,	/* pfnReqFSMHdlr */
				  ConnId);	/*uiID */

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}
	UT_Log(APPITF, INFO, "VAPI_StopTone: Exiting status(%d), conn(%u)\n", Status, ConnId);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_StartCallerId: The function does the following things -
 ***************************************************************************/
/*! 
*	<b> THIS API IS OBSOLETE. \n
*	IT IS KEPT FOR NOW FOR BACKWARD COMPATIBILITY BUT MAY BE REMOVED IN A FUTURE VAPI VERSION</b> \n
*
*	\n \b Description: \n
*	Sends the Caller ID message string for the specified connection. \n
* 	The VAPI library configures the proper Caller ID parameters according to country code settings.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection id on which the CID has to be generated.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstCidInfo</td>
*		<td style="vertical-align: top;">Pointer to Caller ID structure.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include startcid.c
*
*	\n \b Commands:
*	- CND_ONHOOK_GEN
*	- CND_OFFHOOK_GEN
*	- CND_SET_PARAMS
*
*/

VSTATUS VAPI_StartCallerId(IN CONNID ConnId, IN CidInfo *pstCidInfo, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl *pstChnl = NULL;
	SCountryInfo *pstCntryInfo = NULL;
	SCallerIdParams *pstCIDParams = NULL;
	SToneGenParams *pstTone = NULL;
	VSTATUS Status;
	Boolean bIsSync;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_StartCallerId", &pstChnl);

	if (Status != SUCCESS)
		goto err_out;

	UT_Log(APPITF, INFO, "Entered VAPI_StartCallerId: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_START_CALLER_ID_PARAMS;

	if (pstCidInfo->pucMsgStr == NULL)
	{
		UT_ErrorLog(APPITF, "Invalid Caller Id Message String \n");
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	if (pstCidInfo->bIsOnHook!= True && pstCidInfo->bIsOnHook!= False)
	{
		UT_ErrorLog(APPITF, "VAPI_StartCallerId: Invalid bIsOnHook parameter (%d)\n", pstCidInfo->bIsOnHook);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	if (pstCidInfo->eCountryCode >= eMAXCOUNTRY)
	{
		UT_ErrorLog(APPITF, "VAPI_StartCallerId: Invalid eCountryCode parameter (%d)\n", pstCidInfo->eCountryCode);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SCallerIdParams) + sizeof(SToneGenParams));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}

	pstCIDParams = pstVapiReq->pvUserData;

	pstCntryInfo = DMGR_GetCountryDefaults(pstCidInfo->eCountryCode); 
	if (pstCntryInfo == NULL)
	{
		UT_ErrorLog(APPITF, "Invalid Country Code \n");
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_request;
	}

	UT_MemCopy(pstCIDParams, &pstCntryInfo->stCallerIdParams, sizeof(SCallerIdParams));

	if (pstCIDParams->stOffhookParam.bSASToneGen)
	{
		pstTone = (SToneGenParams *) (pstCIDParams + 1);
		UT_MemSet(pstTone, 0, sizeof(SToneGenParams));
		pstCIDParams->stOffhookParam.pstToneGenParam = pstTone;
		UT_MemCopy(pstTone, &pstCntryInfo->stToneGenParams[eSASTONE], sizeof(SToneGenParams));
	}

	/* Copy Caller ID string */
	pstCIDParams->ucCIDMsgLen = pstCidInfo->MsgLen; 

	/* make sure the data can fits into the fifo */
	if (pstCIDParams->ucCIDMsgLen > MAX_CALLER_ID_DATA)
	{
		Status = VAPI_ERR_INVALID_PARAM; 
		goto err_request;
	}
		
	UT_MemCopy(pstCIDParams->ucCIDMsg, pstCidInfo->pucMsgStr, pstCIDParams->ucCIDMsgLen); 

	pstCIDParams->ucOnhookOffhookSelect = pstCidInfo->bIsOnHook; 

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_StartCallerId,	/* pfnReqFSMHdlr */
				  ConnId);	/*uiID */

	bIsSync = pstVapiReq->bIsSync;

	/* Send VAPI request to VCORE */
	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}
	UT_Log(APPITF, INFO, "VAPI_StartCallerId: Exiting status(%d), conn(%u)\n", Status, ConnId);

/* no error exit */
	return Status;

      err_request:
	VCORE_FreeRequest(pstVapiReq);
      err_out:
	return Status;

}


/***************************************************************************
 * VAPI_PlayCid: The function does the following things -
 ***************************************************************************/
/*! 
*	<b> This API replaces the VAPI_StartCallerId() API</b>\n
*
*	\n \b Description: \n
*	Sends the Caller ID message string for the specified connection. \n
* 	The VAPI library configures the proper Caller ID parameters according the settings previously
*	configured by the VAPI_SetCidGenInfo() API.\n
*	The caller ID can transmit call messages or notification messages such as:
*	\li SDMF - Single Data Message Format (Call message type 0x04)
*	\li MDMF - Multiple Data Message Format (Call message type 0x80)
*	\li VMWI - Visual Message Waiting Indicator (Notification message type 0x82) 
*
*	\n The SDMF provides the caller's telephone number, the date and time of the call. \n 
*	The MDMF in addition to the information provided by SDMF format, can also provide the directory listed name for the particular number \n
*	The pucMsgStr buffer of the pstCidInfo structure must be filled by the application with a valid SDMF or MDMF caller id parameters. \n
*
*	The main Caller ID parameters for MDMF format are described below:\n
*	<pre>
*	<b> Parameter Types </b>
*	Type Value	Length	Name 
*	0x01		8	Date and Time 
*	0x02		20 max	Calling Line Identity (CLI) 
*	0x04		1	Reason for absence of CLI 
*	0x07		20 max	Calling Party Name (CPN) 
*	0x08		1	Reason for absence of CPN 
*	0x11		1	Call type 
*	0x12		20	First Called Line Identity (in case of forwarded call) 
*	0x15		1	Type of Forwarded call (in case of forwarded call) 
*	
*	<b> Date and Time Parameter Format
*	Byte Num Contents </b>
*	1	0x01 Type Date 
*	2	0x08 Length 
*	3,4	Month
*	5,6	Day
*	7,8	Hours
*	9,10	Minutes
*	
*	<b> CLI Parameter Format 
*	Byte Num Contents </b>
*	1	0x02 Type CLI 
*	2	n length 
*	3	Digit 1 
*	4	Digit 2 
*	... ... 
*	n+2	Digit n 
*	
*	<b> Reason of Absence of CLI Parameter Format 
*	Byte Num Contents </b>
*	1	0x04 Type Reason for absence of CLI 
*	2	0x01 Length 
*	3	'O' (0x4f) Number unavailable
*		'P' (0x50)Private 
*	
*	<b> Calling Party Name Format
*	Byte Num Contents </b>
*	1	0x07 Type Calling Party Name (CPN)
*	2	n    Length 
*	3	Char 1 
*	4	Char 2 
*	... ...
*	n+2	Char n 
*	
*	<b> Reason of Absence of Calling Party Name Format 
*	Byte Num Contents </b>
*	1	0x08 Type Reason for absence of CPN 
*	2	0x01 Length 
*	3	'O' (0x4f) Number unavailable
*		'P' (0x50)Private 
*	
*	<b> Call Type Parameter Format 
*	Byte Num Contents </b>
*	1	0x11 Type Call type 
*	2	0x01 Length 
*	3	0x01 Voice Call
*		0x02 CLI Ring Back when free call
*		0x03 Calling Name Delivery 
*		0x81 Message Waiting Call 
*	
*	<b> First Called Line Identity Parameter Format
*	Byte Num Contents </b>
*	1	0x12 Type First Called Line Identity 
*	2	n Length 
*	3	Digit 1 
*	4	Digit 2 
*	... ... 
*	n+2	Digit n 
*	
*	<b> Type of Forwarded Call Parameter Format
*	Byte Num Contents </b>
*	1	0x15 Type of Forwarded call 
*	2	0x01 Length 
*	3	0x00 Unavailable or unknown call type 
*		0x01 Forwarded call on busy 
*		0x02 Forwarded call on no reply 
*		0x03 Unconditional forwarded call 
*		0x04 Deflected call (after alerting) 
*		0x05 Deflected call (immediate) 
*		0x06 Forwarded call on inability to reach mobile subscriber 
*	</pre>
*	\n 
*
*	The main Caller ID parameters for Notification messages format are described below:\n
*	<pre>
*	<b> Parameter Types </b>
*	Type Value	Length	Name 
*	0x01		8	       Date and Time 
*	0x02		18 max	Calling Party Number
*	0x07		20 max	Calling Party Name 
*	0x0b		1	       Action (VMWI)
*
*	<b> VMWI Type Parameter Format 
*	Byte Num Contents </b>
*	1	0x11 Type Call type 
*	2	0x01 Length 
*	3	0x00 Disable
*		0xFF Enable
*	</pre> 
*	\n
*
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection id on which the CID has to be generated.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstCidInfo</td>
*		<td style="vertical-align: top;">Pointer to Caller ID structure.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include playcid.c
*
*	\n \b Commands:
*	- CND_SET_PARAMS
*	- CND_ONHOOK_GEN or CND_OFFHOOK_GEN depending hook option
*
*/
VSTATUS VAPI_PlayCid(IN CONNID ConnId, IN SCallerIdInfo *pstCidInfo, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl *pstChnl = NULL;
	SVcoreCidInfo *pstCIDParams = NULL;
	VSTATUS Status;
	Boolean bIsSync;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_PlayCid", &pstChnl);

	if (Status != SUCCESS)
		goto err_out;

	UT_Log(APPITF, INFO, "Entered VAPI_PlayCid: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_PLAY_CALLER_ID_PARAMS;

	if (pstCidInfo == NULL)
	{
		UT_ErrorLog(APPITF, "Invalid Caller Id info \n");
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	if (pstCidInfo->pucMsgStr == NULL)
	{
		UT_ErrorLog(APPITF, "Invalid Caller Id Message String \n");
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SVcoreCidInfo));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}

	pstCIDParams = pstVapiReq->pvUserData;

	/* make sure the data can fits into the fifo */
	if (pstCIDParams->MsgLen > MAX_CALLER_ID_DATA)
	{
		Status = VAPI_ERR_INVALID_PARAM; 
		goto err_request;
	}
		
	pstCIDParams->MsgLen = pstCidInfo->MsgLen;
	pstCIDParams->bIsOnHook = pstCidInfo->bIsOnHook;
	pstCIDParams->ucModulType = pstCidInfo->ucModulType;

	UT_MemCopy(pstCIDParams->MsgStr, pstCidInfo->pucMsgStr, pstCidInfo->MsgLen);

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_PlayCid, ConnId);

	bIsSync = pstVapiReq->bIsSync;

	/* Send VAPI request to VCORE */
	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}
	UT_Log(APPITF, INFO, "VAPI_PlayCid: Exiting status(%d), conn(%u)\n", Status, ConnId);

/* no error exit */
	return Status;

      err_request:
	VCORE_FreeRequest(pstVapiReq);
      err_out:
	return Status;

}

/***************************************************************************
 * VAPI_StopCallerId: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	Stops the Caller ID generation for the specified connection. \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection id on which the CID has to be stoped.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include startcid.c
*
*	\n \b Commands:
*	- CND_STOP
*
*/
VSTATUS VAPI_StopCallerId(IN CONNID ConnId, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl *pstChnl = NULL;
	VSTATUS Status;
	Boolean bIsSync;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_StopCallerId", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_StopCallerId: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_STOP_CALLER_ID_PARAMS;

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(0);
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_StopCallerId,	/* pfnReqFSMHdlr */
				  ConnId);	/*uiID */

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_StopCallerId: Exiting status(%d), conn(%u)\n", Status, ConnId);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_EnableCIDDetection: The function does the following things -
 ***************************************************************************/
/*! 
*	<b> THIS API IS OBSOLETE. \n
*	IT IS KEPT FOR NOW FOR BACKWARD COMPATIBILITY BUT MAY BE REMOVED IN A FUTURE VAPI VERSION</b>\n
*
*	\n \b Description: \n
*	Enables the detection of Type I and Type II Caller ID signals from the CO over voice channel. \n
*	The VAPI library configures the Caller ID Detection parameters according to country code settings. \n
*	<b>Note:
*	This function will return VAPI_ERR_SPU_FEATURE_NOT_ENABLED if SPU feature is not enabled. \n
*	This feature is enabled through the VAPI_InitDevice() function.</b>\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which CallerID Detection is to be done.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eCountryCode</td>
*		<td style="vertical-align: top;">Country code.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucEnableType</td>
*		<td style="vertical-align: top;">Type of Caller ID to be detected:
*			- 0 Enable On Hook CallerID
*			- 1 Enable Off Hook CallerID
*			- 2 Enable Both</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_SPU_FEATURE_NOT_ENABLED
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include enablecid.c 
*
*	\n \b Commands:
*	- CNDDETCTL
*	- CNDDETSP
*
*/
VSTATUS VAPI_EnableCIDDetection(IN CONNID ConnId, IN ECountryCode eCountryCode, IN U8 ucEnableType, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl *pstChnl = NULL;
	SCountryInfo *pstCntryInfo = NULL;
	SCIDDetectionParams *pstCIDDetectionParams = NULL;
	VSTATUS Status;
	Boolean bIsSync;

	if (ucEnableType > 2)
	{
		UT_ErrorLog(APPITF, "VAPI_EnableCIDDetection: Invalid enable type (%d)\n", ucEnableType);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_EnableCIDDetection", &pstChnl);

	if (Status != SUCCESS)
		goto err_out;

	UT_Log(APPITF, INFO, "Entered VAPI_EnableCIDDetection: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_ENABLE_CID_DETECTION_PARAMS;

	if ((pstChnl->pstDev->usSpuFeatureMask & SPU_FEATURE_FSKRX_MASK) == 0)
	{
		UT_ErrorLog(APPITF, "VAPI_EnableCIDDetection: SPU FSKRx Feature not Enabled\n");
		Status = VAPI_ERR_SPU_FEATURE_NOT_ENABLED;
		goto err_out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SCIDDetectionParams));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}
	pstCIDDetectionParams = pstVapiReq->pvUserData;

	pstCntryInfo = DMGR_GetCountryDefaults(eCountryCode);
	if (pstCntryInfo == NULL)
	{
		UT_ErrorLog(APPITF, "Invalid Country Code \n");
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_request;
	}

	UT_MemCopy(pstCIDDetectionParams, &pstCntryInfo->stCIDDetectionParams, sizeof(SCIDDetectionParams));

	/* set CID Detection Enable type according to the user preference */
	pstCIDDetectionParams->ucCIDDEnableType = ucEnableType;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_EnableCIDDetection,	/* pfnReqFSMHdlr */
				  ConnId);	/*uiID */

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_EnableCIDDetection: Exiting status(%d), conn(%u)\n", Status, ConnId);

/* no error exit */
	return Status;

      err_request:
	VCORE_FreeRequest(pstVapiReq);
      err_out:
	return Status;
}


/***************************************************************************
 * VAPI_SetCidDetection: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	<b> This API replaces the VAPI_EnableCIDDetection() API</b>\n
*
*	Enable or disable the detection of Type I or Type II Caller ID signals from the CO over voice channel. \n
*	The VAPI library configures the Caller ID Detection parameters according the configuration previously set
*	by the VAPI_SetCidDetInfo() function. \n
*	<b>Note:
*	This function will return VAPI_ERR_SPU_FEATURE_NOT_ENABLED if SPU feature is not enabled. \n
*	This feature is enabled through the VAPI_InitDevice() function.</b>\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which CallerID Detection is to be done.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstCidDetCtrl</td>
*		<td style="vertical-align: top;">Pointer to Caller Id detection control structure.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_NULL_POINTER_PASSED
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_SPU_FEATURE_NOT_ENABLED
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include set_cid_det.c 
*
*	\n \b Commands:
*	- CNDDETCTL
*	- CNDDETSP
*
*/
VSTATUS VAPI_SetCidDetection (IN CONNID ConnId, IN SCidDetCtrl *pstCidDetCtrl, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl *pstChnl = NULL;
	SCidDetCtrl *pstUserCidDetCtrl = NULL;
	VSTATUS Status;
	Boolean bIsSync;

	if (pstCidDetCtrl == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_SetCidDetection: pstCidDetCtrl is Null\n");
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		goto err_out;
	}

	if (pstCidDetCtrl->usHookStatus > 2)
	{
		UT_ErrorLog(APPITF, "VAPI_SetCidDetection: invalid Hook Status type (%d)\n", pstCidDetCtrl->usHookStatus);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	if (pstCidDetCtrl->usCidEnable > 1)
	{
		UT_ErrorLog(APPITF, "VAPI_SetCidDetection: invalid enabled type (%d)\n", pstCidDetCtrl->usCidEnable);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_SetCidDetection", &pstChnl);

	if (Status != SUCCESS)
		goto err_out;

	UT_Log(APPITF, INFO, "Entered VAPI_SetCidDetection: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_ENABLE_SET_CID_DETECTION_PARAMS;

	if ((pstChnl->pstDev->usSpuFeatureMask & SPU_FEATURE_FSKRX_MASK) == 0)
	{
		UT_ErrorLog(APPITF, "VAPI_SetCidDetection: SPU FSKRx Feature not Enabled\n");
		Status = VAPI_ERR_SPU_FEATURE_NOT_ENABLED;
		goto err_out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SCidDetCtrl));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}
	pstUserCidDetCtrl = pstVapiReq->pvUserData;

	/* Pass CID Detection user preference to FSM*/
	UT_MemCopy(pstUserCidDetCtrl, pstCidDetCtrl, sizeof(SCidDetCtrl));	

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_SetCIDDetection, ConnId);

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_SetCidDetection: Exiting status(%d), conn(%u)\n", Status, ConnId);

/* no error exit */
	return Status;

	VCORE_FreeRequest(pstVapiReq);
      err_out:
	return Status;
}

/***************************************************************************
 * VAPI_SetCodecType: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	Sets the codec type for the specified connection. \n
*	The codec parameter is not same as Payload type (PT) value. \n
*	The VAPI library defines a set of acceptable codec types as an enumerated value. \n
*	VAPI internally maintains the codec types valid for a particular device type. \n
*	If the codec is not valid for that device type and error is returned to the user.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which the Codec has to be set.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eCodecIndex</td>
*		<td style="vertical-align: top;">Codec index.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_PT_NOT_INITIALIZED
*
*	\n \b Usage:
*	\include codectype.c
*
*	\n \b Commands:
*	- VoIP_VOPENA
*
*/
VSTATUS VAPI_SetCodecType(IN CONNID ConnId, IN ECodecIndex eCodecIndex, IN SRequest * pstRequest)
{
	SChnl *pstChnl = NULL;
	SVapiReq *pstVapiReq = NULL;
	VSTATUS Status;
	Boolean bIsSync;
	ECodecType *peUserData;
	SVAPIDeviceConfig *pstDevConfig;
	SConnResources *pstConRes = NULL;
	SResRequest stResRequest;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_SetCodecType", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_SetCodecType: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_SET_CODEC_TYPE_PARAMS;

	if (eCodecIndex >= NUM_CODECS)
	{
		UT_ErrorLog(APPITF, "VAPI_SetCodecType: Invalid codec value\n");
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	pstDevConfig = DMGR_GetVAPIDeviceConfig(pstChnl->pstDev->ucDevType);
	if (pstDevConfig == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_SetCodecType: Device type for this conn(%u) is invalid \n", ConnId);
		Status = VAPI_ERR_INVALID_DEV_TYPE;
		goto out;
	}

	if (pstDevConfig->astCodecInfo[eCodecIndex].bIsSupported != True)
	{
		UT_ErrorLog(APPITF, "VAPI_SetCodecType: "
			    "Codec %u not supported on device %u of type %u\n",
			    eCodecIndex, pstChnl->pstDev->DevId, pstChnl->pstDev->ucDevType);

		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	/* Resource manager is only for VoIP or FoIP connection*/
	if ((pstChnl->usConnType == eVOIP) || (pstChnl->usConnType == eFOIP))
	{
		/* If the ressource manager is enabled check if resources have been allocated */
		if (pstChnl->pstDev->stDevRes.bIsResourceMngr == True)
		{
			/*Get the ressource for this connection */
			pstConRes = (SConnResources *) UT_SearchInAVLTree(pstResTree_g, ConnId);
			if (pstConRes == NULL)
			{
				UT_ErrorLog(APPITF, "VAPI_SetCodecType: no resource allocated for connection (%u) \n", ConnId);
				Status = VAPI_ERR_NO_RESOURCE_ALLOC;
				goto out;
			}
			stResRequest.eConnType = pstChnl->usConnType;
			stResRequest.ePartType = pstChnl->ePartType;
			stResRequest.eCodecType = eCodecIndex;
			stResRequest.ucPacketSize = pstChnl->pstChnlParams->stVoiceOpt.param_4.bits.packet_interval;
			stResRequest.ucVAD = pstChnl->pstChnlParams->stVoiceOpt.param_4.bits.vadtype;
			stResRequest.ucG729_1_rate = 0;
			Status = VRES_CheckRes(pstChnl, &stResRequest, pstConRes);
			if (Status != SUCCESS)
				goto out;
		}
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(eCodecIndex));

	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	peUserData = pstVapiReq->pvUserData;
	*peUserData = eCodecIndex;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_SetCodecType, ConnId);

	bIsSync = pstVapiReq->bIsSync;

	pstVapiReq->usReqState = SET_CODEC_INIT;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_SetCodecType: Exiting status(%d), conn(%u)\n", Status, ConnId);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_SetPacketInterval: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	Sets the packet interval in ms for the connection. \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which the packet size has to be set.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucMilliSec</td>
*		<td style="vertical-align: top;">Packet interval in milliseconds. \n
*		The library checks against incompatible codec and packet interval combinations \n
*		(e.g. 20ms for g.723)</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include packetinterval.c
*
*	\n \b Commands:
*	- VoIP_VOPENA
*	- VoIP_VCEOPT
*/
VSTATUS VAPI_SetPacketInterval(IN CONNID ConnId, IN U8 ucMilliSec, IN SRequest * pstRequest)
{
	SChnl *pstChnl = NULL;
	SVapiReq *pstVapiReq = NULL;
	VSTATUS Status;
	Boolean bIsSync;
	U8 *pucUserData;
	SVAPIDeviceConfig *pstDevConfig;
	SConnResources *pstConRes = NULL;
	SResRequest stResRequest;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_SetPacketInterval", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_SetPacketInterval: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_SET_PACKET_INTERVAL_PARAMS;

	pstDevConfig = DMGR_GetVAPIDeviceConfig(pstChnl->pstDev->ucDevType);
	if (pstDevConfig == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_SetPacketInterval: Device type for this conn(%u) is invalid \n", ConnId);
		Status = VAPI_ERR_INVALID_DEV_TYPE;
		goto out;
	}

	/* Resource manager is only for VoIP or FoIP connection*/
	if ((pstChnl->usConnType == eVOIP) || (pstChnl->usConnType == eFOIP))
	{
		/* If the ressource manager is enabled check if resources have been allocated */
		if (pstChnl->pstDev->stDevRes.bIsResourceMngr == True)
		{
			/*Get the ressource for this connection */
			pstConRes = (SConnResources *) UT_SearchInAVLTree(pstResTree_g, ConnId);
			if (pstConRes == NULL)
			{
				UT_ErrorLog(APPITF, "VAPI_SetPacketInterval: no resource allocated for connection (%u) \n", ConnId);
				Status = VAPI_ERR_NO_RESOURCE_ALLOC;
				goto out;
			}
			stResRequest.eConnType = pstChnl->usConnType;
			stResRequest.ePartType = pstChnl->ePartType;
			stResRequest.ucPacketSize = ucMilliSec;
			stResRequest.eCodecType = pstChnl->pstChnlParams->eCodecVal;
			stResRequest.ucVAD = pstChnl->pstChnlParams->stVoiceOpt.param_4.bits.vadtype;
			stResRequest.ucG729_1_rate = 0;
	
			Status = VRES_CheckRes(pstChnl, &stResRequest, pstConRes);
			if (Status != SUCCESS)
				goto out;
		}
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(U8));

	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pucUserData = pstVapiReq->pvUserData;
	*pucUserData = ucMilliSec;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_SetPacketInterval, ConnId);

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_SetPacketInterval: Exiting status(%d), conn(%u)\n", Status, ConnId);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_SetPayloadType: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	Configures the payload type for each codec. \n
*	Each codec has a default PT value that can be overridden with this function. \n
*	Repeated invocations of this function are required to override more than one payload type. \n
*	VAPI internally maintains the codec types valid for a particular device type. \n
*	If the codec is not valid for that device type and error is returned to the user.\n
*	For Video Media sub layer use EVideoCodecIndex for codec ID.
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which the payload type has to be set.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucPlTypVal</td>
*		<td style="vertical-align: top;">The payload type value to be set.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eCodec</td>
*		<td style="vertical-align: top;">codec ID</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eDir</td>
*		<td style="vertical-align: top;">The direction field can be (eTx, eRx, eBoth) 
*			to support asymmetrical payload types.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include payloadtype.c
*
*	\n \b Commands:
*	- VoIP_PTSET
*	- VoIP_PTSETRXOVR
*/
VSTATUS VAPI_SetPayloadType(IN CONNID ConnId,
			    IN ECodecType eCodec, IN U8 ucPlTypVal, IN EDirection eDir, IN SRequest * pstRequest)
{
	SChnl *pstChnl = NULL;
	SVapiReq *pstVapiReq = NULL;
	VSTATUS Status;
	Boolean bIsSync;
	SPtmngUserData *pstUserData;
	SVAPIDeviceConfig *pstDevConfig;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_SetPayloadType", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_SetPayloadType: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_SET_PAYLOAD_TYPE_PARAMS;

	if (eCodec >= NUM_CODECS)
	{
		UT_ErrorLog(APPITF, "VAPI_SetPayloadType: " "Invalid Codec type, Codec = %d\n", eCodec);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	if ((eDir != eTx) && (eDir != eRx) && (eDir != eBoth))
	{
		UT_ErrorLog(APPITF, "VAPI_SetPayloadType: " "Invalid Dir,eDir = %d\n", eDir);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	pstDevConfig = DMGR_GetVAPIDeviceConfig(pstChnl->pstDev->ucDevType);
	if (pstDevConfig == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_SetPayloadType: Device type for this conn(%u) is invalid \n", ConnId);
		Status = VAPI_ERR_INVALID_DEV_TYPE;
		goto out;
	}

	if (pstDevConfig->astCodecInfo[eCodec].bIsSupported != True)
	{
		UT_ErrorLog(APPITF, "VAPI_SetPayloadType: "
			    "Codec %u not supported on device %u of type %u\n",
			    eCodec, pstChnl->pstDev->DevId, pstChnl->pstDev->ucDevType);

		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	pstVapiReq = VCORE_AllocateRequest(sizeof(SPtmngUserData));

	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstUserData = pstVapiReq->pvUserData;

	pstUserData->eCodec = eCodec;
	pstUserData->usPTIndex = pstDevConfig->astCodecInfo[eCodec].usPTIndex;
	pstUserData->ucPlTypVal = ucPlTypVal;
	pstUserData->ucDir = (U8) eDir;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_SetPayloadType,	/* pfnReqFSMHdlr */
				  ConnId);	/*uiID */

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_SetPayloadType: Exiting status(%d), conn(%u)\n", Status, ConnId);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_EchoCancellerReset: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	Re-initializes the device's Echo Canceller. \n
*	This is to allow the user application to tell the device to retrain the EC \n 
*	if the same connection is used to connect to another far-end party.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which the Echo Canceller has to be reset.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include echocan.c
*
*	\n \b Commands:
*	- VOIP_ECHOCAN or VOIP_DFECAN depending on Echo Canceller used (see VAPI_InitDevice() function)
*/
VSTATUS VAPI_EchoCancellerReset(IN CONNID ConnId, IN SRequest * pstRequest)
{
	SChnl *pstChnl = NULL;
	SVapiReq *pstVapiReq = NULL;
	VSTATUS Status = SUCCESS;
	Boolean bIsSync;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_EchoCancellerReset", &pstChnl);
	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_EchoCancellerReset: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_SET_ECHOCANCELLOR_RESET_PARAMS;

	pstVapiReq = VCORE_AllocateRequest(0);
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest,
				VFSM_EchoCancellerReset,	/* pfnReqFSMHdlr */
				ConnId);	/*uiID */

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_EchoCancellerReset: Exiting status(%d), conn(%u)\n", Status, ConnId);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_SendNteEvent: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	Sends a non-DTMF NTE RTP packet. \n
*	The usOverRideBitField sets the override options for SSRC, redundancy interval, and payload type.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which the NTE packet has to be sent.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">uiNtePyLd</td>
*		<td style="vertical-align: top;">Payload of the NTE packet.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usOverRideBitField</td>
*		<td style="vertical-align: top;">The host can optionally override the redundancy interval time,
*			 disable redundancy, override the SSRC, or override the payload type.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">uiSsrc</td>
*		<td style="vertical-align: top;">SSRC value that will override older SSRC.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usPyLdType</td>
*		<td style="vertical-align: top;">Payload_type for DTMF in RTP that will override the previous setting.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usRedundancyInterval</td>
*		<td style="vertical-align: top;">The redundancy interval value that will override the previous value.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include sendnte.c
*
*	\n \b Commands:
*	- VoIP_SENDNTE
*
*/
VSTATUS VAPI_SendNteEvent(IN CONNID ConnId,
			  IN U32 uiNtePyLd,
			  IN U16 usOverRideBitField,
			  IN U16 usRedundancyInterval, IN U32 uiSsrc, IN U16 usPyLdType, IN SRequest * pstRequest)
{
	SChnl *pstChnl = NULL;
	SVapiReq *pstVapiReq = NULL;
	VSTATUS Status;
	Boolean bIsSync;
	SNteEventUserData *pstUserData = NULL;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_SendNteEvent", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_SendNteEvent: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_SEND_NTE_EVENT_PARAMS;

	pstVapiReq = VCORE_AllocateRequest(sizeof(SNteEventUserData));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstUserData = pstVapiReq->pvUserData;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_SendNte,	/* pfnReqFSMHdlr */
				  ConnId);	/*uiID */

	bIsSync = pstVapiReq->bIsSync;

	pstUserData->uiNtePyLd = uiNtePyLd;
	pstUserData->uiSsrc = uiSsrc;
	pstUserData->usOverRideBitField = usOverRideBitField;
	pstUserData->usRedundancyInterval = usRedundancyInterval;
	pstUserData->usPyLdType = usPyLdType;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_SendNteEvent: Exiting status(%d), conn(%u)\n", Status, ConnId);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_SetRtpSsrcHeader: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	Sets the RTP SSRC header field. This API is only valid for RTP-encoded connections. \n
*	If the connection is not active the SSRC value is stored in the VAPI 
*	variable and used on subsequent VOPENA command. \n
*	If the connection is active the SSRC value is take in account immediately. \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which the SSRC has to be changed.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">uiSsrcVal</td>
*		<td style="vertical-align: top;">New SSRC value.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include setssrc.c
*
*	\n \b Commands:
*	- VoIP_VOPENA
*/
VSTATUS VAPI_SetRtpSsrcHeader(IN CONNID ConnId, IN U32 uiSsrcVal, IN SRequest * pstRequest)
{
	SChnl *pstChnl = NULL;
	SVapiReq *pstVapiReq = NULL;
	VSTATUS Status;
	Boolean bIsSync;
	U32 *puiUserData;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_SetRtpSsrcHeader", &pstChnl);
	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_SetRtpSsrcHeader: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_SET_RTP_SSRC_HDR_PARAMS;

	pstVapiReq = VCORE_AllocateRequest(sizeof(U32));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	puiUserData = pstVapiReq->pvUserData;
	*puiUserData = uiSsrcVal;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest,
				VFSM_SetRtpSsrcHeader,	/* pfnReqFSMHdlr */
				ConnId);	/*uiID */

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_SetRtpSsrcHeader: Exiting status(%d), conn(%u)\n", Status, ConnId);

      out:
	return Status;
}


/***************************************************************************
 * VAPI_ConfigureT38Options: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This API configures the Fax options to be used while switching over to T38 mode. \n
*	It stores the options in the channel parameters structure associated with the connection if the chnnael is VoIP. \n
*	The options will be used later on if the channel is switched to FoIP (see VAPI_SwitchToT38()). \n
*	If the channel is already in FoIP mode, the MSP FAX commands are sent to configure the channel. \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection to which the T38 options have to be changed.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstFaxConfigOpts</td>
*		<td style="vertical-align: top;">Pointer to fax options structure which gives the fax switchover options
*					to be used while switching to FoIP mode</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage: \n
*	\include configuret38.c
*
*	\n \b Commands:
*	List of Comcerto commands sent (if FoIP channel):
*	- FAXOPT 
*	- FAXLVL
*/
VSTATUS VAPI_ConfigureT38Options(IN U32 ConnId, IN SFaxConfigOpts * pstFaxConfigOpts, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl *pstChnl = NULL;
	VSTATUS Status;
	Boolean bIsSync;
	SFsmFaxOptions *pstUserData = NULL;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_ConfigureT38Options", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_ConfigureT38Options: conn(%u)\n", ConnId);

	if (pstFaxConfigOpts == NULL)
	{
		UT_ErrorLog(APPITF, "Invalid parameter\n");
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		goto out;
	}

	/* log the parameters passed if required (define in appitf.h) */
	LOG_CONFIGURE_T38_OPTIONS_PARAMS;

	if((pstChnl->ePartType == eLSP) 
		|| (pstChnl->ePartType == eRSP)
		|| (pstChnl->ePartType == eTRANSP)
		|| (pstChnl->ePartType == eLSPWB)
		|| (pstChnl->ePartType == eRSPWB))
	{
		UT_ErrorLog(APPITF, "VAPI_ConfigureT38Options: Not valid on participants\n");
		Status = VAPI_ERR_INVALID_CONNID;
		goto out;
	}

	pstVapiReq = VCORE_AllocateRequest(sizeof(SFsmFaxOptions));

	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstUserData = pstVapiReq->pvUserData;

	/* save the user passed options to the FSM internal structure*/
	VCORE_SetUserFaxOption(pstFaxConfigOpts, pstUserData);
	pstChnl->bEnableFaxAutoSwitch = pstFaxConfigOpts->bEnableFaxAutoSwitch;
	pstChnl->usSwitchoverEventMask = pstFaxConfigOpts->usSwitchoverEventMask;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_SetT38Options, ConnId);

	bIsSync = pstVapiReq->bIsSync;

	/* If the user wants to keep the existing options
	we start the FSM handler in T38_OPTIONS_FAX_SKIP directly */
	if (pstFaxConfigOpts->bUseExistingOpts)
		pstVapiReq->usReqState = T38_OPTIONS_FAX_SKIP;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_ConfigureT38Options: Exiting status(%d), conn(%u)\n", Status, ConnId);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_SwitchToT38: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This API is used to switch the channel to FoIP mode under the control of the application. \n
*	Alternatively the application can set the fax options to enable an automatic switchover from VAPI.  \n 
*	The transport parameters (MAC address, IP/UDP) are the same as the previous VoIP connection
*	The application could use the VAPI_SetConnIpParams() and VAPI_SetEthMac() to reconfigure them if required. \n
*	the T.38 port info and then call VAPI_EnableConnection() to enable packet generation. \n
*	<b>NOTE: The connection is not enabled after the switch.\n
*	This gives the opportunity to send other configuration commands before re-enabling the connection. </b>\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection to be switched in T38 mode.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstFaxConfigOpts</td>
*		<td style="vertical-align: top;">Pointer to fax options structure which gives the fax switchover options \n
*					to be used while switching to FoIP mode.
*					If NULL VAPI uses the fax options already configured on the connection \n 
*					or the default values if no options have been explicitly configured.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include switchT38.c
*
*	\n \b Commands:
*	List of Comcerto commands sent: (if any).
*	- VOPENA (Disable)
*	- SUPVSR_DESTROY_CHANNEL
*	- SUPVSR_CREATE_CHANNEL
*	- FAXOPT
*	- FAXLVL
*	- SET_ETH_HDR_CHAN
*	- SET_IP_HDR_CHANNEL
*/
VSTATUS VAPI_SwitchToT38(IN CONNID ConnId, IN SFaxConfigOpts * pstFaxConfigOpts, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl *pstChnl = NULL;
	VSTATUS Status;
	Boolean bIsSync;
	SFsmFaxOptions *pstUserData = NULL;
	SConnResources *pstConRes = NULL;
	SResRequest stResRequest;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_SwitchToT38", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	/* If the ressource manager is enabled check if resources have been allocated */
	if (pstChnl->pstDev->stDevRes.bIsResourceMngr == True)
	{
		/*Get the ressource for this connection */
		pstConRes = (SConnResources *) UT_SearchInAVLTree(pstResTree_g, ConnId);
		if (pstConRes == NULL)
		{
			UT_ErrorLog(APPITF, "VAPI_CreateConnection: no resource allocated for connection (%u) \n", ConnId);
			Status = VAPI_ERR_NO_RESOURCE_ALLOC;
			goto out;
		}
		/* Resource manager is only for VoIP or FoIP connection*/
		if ((pstChnl->usConnType == eVOIP) || (pstChnl->usConnType == eFOIP))
		{
			/*Check required resource against allocated resources */
			stResRequest.eConnType = eFOIP;
			stResRequest.ePartType = eNOTPART;
			/* Connection FoIP so use eIFP */
			stResRequest.eCodecType = eIFP;
			stResRequest.ucPacketSize = pstChnl->pstChnlParams->stVoiceOpt.param_4.bits.packet_interval;
			stResRequest.ucVAD = pstChnl->pstChnlParams->stVoiceOpt.param_4.bits.vadtype;
			stResRequest.ucG729_1_rate = 0;
		
			Status = VRES_CheckRes(pstChnl, &stResRequest, pstConRes);
			if (Status != SUCCESS)
				goto out;
		}
	}

	UT_Log(APPITF, INFO, "Entered VAPI_SwitchToT38: conn(%u)\n", ConnId);

	pstVapiReq = VCORE_AllocateRequest(sizeof(SFsmFaxOptions));

	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstUserData = pstVapiReq->pvUserData;

	/*Copy the User fax options to the internal VAPI structure/variables (if provided)*/
	if (pstFaxConfigOpts != NULL)
	{
		/*If the user wants to use passed option, save them to the internal variables*/
		if (pstFaxConfigOpts->bUseExistingOpts == False)
		{
			VCORE_SetUserFaxOption(pstFaxConfigOpts, pstUserData);
			pstChnl->bEnableFaxAutoSwitch = pstFaxConfigOpts->bEnableFaxAutoSwitch;
			pstChnl->usSwitchoverEventMask = pstFaxConfigOpts->usSwitchoverEventMask;
		}
		/* log the parameters passed if required (define in appitf.h) */
		LOG_SWITCH_TO_T38_PARAMS;
	}

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_SwitchToT38, ConnId);

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}
	UT_Log(APPITF, INFO, "VAPI_SwitchToT38: Exiting status(%d), conn(%u)\n", Status, ConnId);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_SetConnIpParams: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	Sets the source and destination IP addresses and UDP ports \n
*	required to generate complete RTP Encapsulated packets for the specified connection ID. \n
*	The source IP address is same as from VAPI_SetDeviceIPAddr().\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which the IP parameters have to be changed.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstIpParams</td>
*		<td style="vertical-align: top;">Structure for destination and source side IP Addresses and UDP ports. \n
*				<b>Please note that all the addresses and ports must
*				be provided in Network byte order (Big Endian).</b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage: \n
*	\include set_ipparams.c
*
*	\n \b Commands:
*	- SET_IP_HDR_CHANNEL
*/
VSTATUS VAPI_SetConnIpParams(IN CONNID ConnId, IN SIpParams * pstIpParams, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl *pstChnl = NULL;
	VSTATUS Status;
	Boolean bIsSync;
	struct _SET_IP_HDR_CHANNEL *pstUserData = NULL;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_SetConnIpParams", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_SetConnIpParams: conn(%u)\n", ConnId);

	if (pstIpParams == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_SetConnIpParams: Invalid IP params passed\n");
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	/* log the parameters passed if required (define in appitf.h) */
	LOG_SET_CONN_IP_PARAMS;

	pstVapiReq = VCORE_AllocateRequest(sizeof(struct _SET_IP_HDR_CHANNEL));

	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstUserData = pstVapiReq->pvUserData;

	/* get the default IP header config from the connection structure */
	UT_MemCopy(pstUserData, &(pstChnl->pstChnlParams)->stIpHdrParams, sizeof(struct _SET_IP_HDR_CHANNEL ));

	/*overwrite some fields with user parameters */
	pstUserData->param_4.bits.serviceid = pstIpParams->ucIPServiceId;
	pstUserData->ip_src = pstIpParams->uiSrcIpAddr;
	pstUserData->ip_dst = pstIpParams->uiDestIpAddr;
	pstUserData->uh_sport = pstIpParams->usSrcUdpPort;
	pstUserData->uh_dport = pstIpParams->usDestUdpPort;

	/*If the device is set in single IP address, use it as IP src for the connection */
	/*In multimode (limited or unlimited) use the user one.*/
	if (pstChnl->pstDev->eSrcIpMode == eSINGLE_IP)
		pstUserData->ip_src = pstChnl->pstDev->uiSrcIpAddr;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_SetConnectionIPParams, ConnId);

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_SetConnIpParams: Exiting status(%d), conn(%u)\n", Status, ConnId);

      out:
	return Status;
}


/***************************************************************************
 * VAPI_TranscodingSession: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	Sets a transcoding session between 2 existing connections. \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId1</td>
*		<td style="vertical-align: top;">Id of the first connection involved in the transcoding session</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId2</td>
*		<td style="vertical-align: top;">Id of the second connection involved in the transcoding session</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">TranscodingOption</td>
*		<td style="vertical-align: top;">Start / Stop transcoding, set DSP bypass mode</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_CONNID
*
*	\n \b Usage:
*	\include transcoding.c
*
*	\n \b Commands:
*	- VoIP_VOPDIS
*	- FC_SUPV_TRANS_CONNECT
*	- VoIP_VOPENA
*/
VSTATUS VAPI_TranscodingSession(IN CONNID ConnId1, IN CONNID ConnId2, IN STranscodingOption  *pstTranscodingOption, IN SRequest *pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	STranscodingParams *pstUserData;
	SChnl* pstChnl1 = NULL;
	SChnl* pstChnl2 = NULL;
	VSTATUS Status = SUCCESS;
	Boolean bIsSync;

	if (pstTranscodingOption == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_TranscodingSession: transcoding options NULL\n");
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		goto out;
	}

	/* we first ed to make sure that the 2 required connections exist */
	Status = VCORE_ConnectionGenericCheck(ConnId1,"VAPI_TranscodingSession", &pstChnl1);
	if (Status != SUCCESS )
		goto out;
 
	Status = VCORE_ConnectionGenericCheck(ConnId2,"VAPI_TranscodingSession", &pstChnl2);
	if (Status != SUCCESS )
		goto out;

	if (pstChnl1->pstDev->DevId != pstChnl2->pstDev->DevId)
	{
		UT_ErrorLog(APPITF, "VAPI_TranscodingSession: transcoding only allowed on the same device\n");
		Status = VAPI_ERR_INVALID_CONNID;
		goto out;
	}

	UT_Log(APPITF,INFO,"VAPI_TranscodingSession: conn1(%d), conn2(%d)\n", ConnId1, ConnId2);

	/* FIXME: Need to implement LOG_TRANSCODING_SESSION */

	pstVapiReq = VCORE_AllocateRequest(sizeof(STranscodingParams));

	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstUserData = pstVapiReq->pvUserData;

	pstUserData->Connection2Id = pstChnl2->ConnId;
	pstUserData->TranscodingEnable = pstTranscodingOption->bAction;
	pstUserData->ucDSPMode = pstTranscodingOption->ucDSPMode;
	pstUserData->ucVirtualTranscoding = pstTranscodingOption->ucVirtualTranscoding;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_TranscodingSession, ConnId1);

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl1, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF,INFO,"VAPI_TranscodingSession: Exiting status(%d), conn1(%d), conn2(%d)\n", Status, ConnId1, ConnId2);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_ModifyConnection: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This API changes the current connection configuration for the passed function code with the passed parameters.\n
*	Depending on the number of parameters the VAPI_ModifyConnection() handler must: \n
*	 - Parse the parameters to get the different function codes
*	 - Run a query (or a multi query) to get the current configuration
*	 - Build the appropriated MSP (multi) command with the required parameters
*	 - Apply the command to the connection
*	 - Update the appropriated internal VAPI variables
*
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">The connection ID is specified by the user application.\n
*			A connection with the specified ID must have been previously created with 
*			VAPI_CreateConnection() or VAPI_AllocateConnection().\n
*			The VAPI library keeps track of the connection resource mapped by the user-specified 
*			connection ID. 
*			The VAPI library looks up the device ID mapped to the connection ID and passes that 
*			down to GTL layer .</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usModifyNum</td>
*		<td style="vertical-align: top;">Specifies the number of the parameters to be modified \n
*			(number of element in the  pstModifyConnectionInfo array) </td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstModifyConnectionInfo</td>
*		<td style="vertical-align: top;">Array of SModifyConnectionInfo.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_NULL_POINTER_PASSED
*	\li VAPI_ERR_INVALID_PARAM
*	\li VAPI_ERR_INVALID_CONNID
*
*	\n \b Usage:
*	\include modify_connection.c
*
*	\n \b Commands:
*	\li The Comcerto commands sent depends on usFC parameter
*/
VSTATUS VAPI_ModifyConnection(IN CONNID ConnId,
				IN U16 usModifyNum,
				IN SModifyConnectionInfo *pstModifyConnectionInfo,
				IN SRequest *pstRequest)
{
	SVapiReq *pstVapiReq;
	SFullModConnInfo *pstUserData;
	SModifyConnectionInfo *pstUserModiConnInfo;
	SChnl *pstChnl;
	VSTATUS Status;
	Boolean bIsSync;
	int i, j;
	int MaxValue;

	if (pstModifyConnectionInfo == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_ModifyConnection: pstModifyConnectionInfo is NULL\n");
		return VAPI_ERR_NULL_POINTER_PASSED;
	}

	if (usModifyNum == 0)
	{
		UT_ErrorLog(APPITF, "VAPI_ModifyConnection: zero usModifyNum\n");
		return VAPI_ERR_INVALID_PARAM;
	}
	
	for (i = 0; i < usModifyNum; i++)
	{
		for (j = 0, MaxValue = 1; j < pstModifyConnectionInfo[i].ucfield_width; j++)
			MaxValue = MaxValue * 2;
		
		MaxValue -= 1;

		if (pstModifyConnectionInfo[i].usValue > MaxValue)
		{
			UT_ErrorLog(APPITF, "VAPI_ModifyConnection: usValue (%d) is to big for current width(%d) in (%d) array item\n",\
						pstModifyConnectionInfo[i].usValue, \
						pstModifyConnectionInfo[i].ucfield_width, i);
			return VAPI_ERR_INVALID_PARAM;
		}
	}

	UT_Log(APPITF, INFO, "Entered VAPI_ModifyConnection: conn(%u)\n", ConnId);
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_ModifyConnection", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	pstUserModiConnInfo = UT_Calloc(usModifyNum, sizeof(SModifyConnectionInfo));
	if (pstUserModiConnInfo == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}	

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SModifyConnectionInfo));

	if (pstVapiReq == NULL)
	{
		UT_FreeMem(pstUserModiConnInfo);
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstUserData = pstVapiReq->pvUserData;

	UT_MemCopy(pstUserModiConnInfo, pstModifyConnectionInfo, usModifyNum * sizeof(SModifyConnectionInfo));

	pstUserData->pstModConnInfo = pstUserModiConnInfo;
	pstUserData->usModifyNum = usModifyNum;

	/* configure the request */
	VCORE_SetConnectionRequest(pstVapiReq,
					pstRequest,
					VFSM_ModConn, /* pfnReqFSMHdlr */
					ConnId);
	
	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		/*in ASYNC mode it is freed in VFSM_ModConn*/
		UT_FreeMem(pstUserModiConnInfo);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_ModifyConnection: Exiting status(%d), conn(%u)\n", Status, ConnId);

out:
	return Status;
}

/***************************************************************************
* VAPI_SetConnVlan: The function does the following things -
***************************************************************************/
/*!
*	\n \b Description: \n
*	This API programs the VLAN ID in the ethernet header of the connection.\n
*	If this API is not issued to the connection, it heritates the VLAN tag of the device (if any)\n
*	This API doesn't send any command to the device. It just set an internal variable.\n
*	It required to issue this API before the VAPI_SetEthMac() to get the VLAN Id taken in account.\n
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection ID.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usVlanId </td>
*		<td style="vertical-align: top;">VLAN ID. If VLAN=0, no VLAN set.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b usage
*	\include vlan.c
*
*	\n \b Commands:
*	List of Comcerto commands sent:
*	\li NONE
*/
VSTATUS VAPI_SetConnVlan(IN CONNID ConnId, IN U16 usVlanId, IN SRequest * pstRequest)
{
	VSTATUS Status;
	SChnl* pstChnl = NULL;

	UT_Log(APPITF, INFO, "Entered VAPI_SetConnVlan: dev(%u)\n", ConnId);
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_SetConnVlan", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	pstChnl->stEthMacAddresses.usVlanId = usVlanId;
	UT_Log(APPITF, INFO, "VAPI_SetConnVlan: Exiting status(%d), devn(%u)\n", Status, ConnId);

out:
	/* if ASYNC mode call the callback function*/
	if ((pstRequest != NULL) && (pstRequest->pfnIoCompCallback != NULL))
	{
		pstRequest->pfnIoCompCallback(ConnId, CMD_LEVEL_CONN, Status, NULL, 0, pstRequest->uiReqId);
		Status = VAPI_ERR_PENDING;
	}

	return Status;
}

/***************************************************************************
 * VAPI_RecoverConnection: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This API is intended to recover a Comcerto device channel when the system is in a unsynchronized status. \n
*	Typically it may happen when Comcerto channels are up and running and the host applications stops.\n
*	In this case to be able to restart the application and have a chance to retrieve the previous status, \n
*	the application can use this API. \n
*	The reference numbers on the host application side are the Connection ID and the timeslot to use for this 
*	connection. \n
*	On VAPI side the reference numbers are the Connection ID and the Comcerto channel ID. \n
*	When VAPI_CreateConnection is used it makes a mapping between : \n
*	the Timeslot ID <-> Connection ID <-> Channel ID. \n
*	On erronenous situation this mapping can be lost. Therefore this API can be used to rebuild it mapping \n
*	VAPI_RecoverConnection() will return:
*	\li  SUCCESS if the required timeslot is used by a device channel.
*	\li  VAPI_ERR_RECOVER_TIMESLOT if no Comcerto channel found with required timeslot.
*	\li  VAPI_ERR_RECOVER_WRONG_TYPE if the recovered Comcerto channel type doesn't match
*	required connection type. In that case the connection exists, 
*	but the host application has to take the proper actions.
*
* 	\n<b>Note 1: Once the Connection is recovered, it is the responsability 
*		of the application to reconfigure the connection.</b> 
*
* 	\n<b>Note 2:<span style="color: rgb(255, 0, 0);"> Notes: This API requires the 
*		firmware to support the SUPERVISOR_TIMESLOT query command </span></b> \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">The device on which connection is to be recovered</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">The connection ID is specified by the user application.  \n
*		The connection ID is assumed unique across all devices and channels.  \n
*		The VAPI library keeps track of the connection resource mapped by the user-specified connection ID. \n
*		The VAPI library looks up the device ID mapped to the connection ID and passes that down to GTL layer</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eConnType</td>
*		<td style="vertical-align: top;">Specifies the type of channel (VoIP, VoATM, T38, etc). \n
*				It can have following values:
*				- eVOIP
*				- eFOIP
*				- eVOATM
*				- eIUUPOAAL2.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">The connection ID is specified by the user application.  \n
*		The connection ID is assumed unique across all devices and channels.  \n
*		The VAPI library keeps track of the connection resource mapped by the user-specified connection ID. \n
*		The VAPI library looks up the device ID mapped to the connection ID and passes that down to GTL layer</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usTdmTimeSlot</td>
*		<td style="vertical-align: top;">TDM timeslot of the channel to be recovered.\n
*			The TS number must be in a valid range according the TDM parameters..</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usRecoverOption</td>
*		<td style="vertical-align: top;">this parameter is intended for future use.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pfnEventCallback</td>
*		<td style="vertical-align: top;">Connection Level event callback. \n
*		When events will be received for this connection this event callback will be called</td>
*	</tr>
*	</table>
*
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_UNSUPP_FEATURE
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_UNDEFINED
*	\li VAPI_ERR_RECOVER_TIMESLOT
*	\li VAPI_ERR_RECOVER_WRONG_TYPE
*
*	\n \b Usage:
*	\include recover_connection.c
*
*	\n \b Commands:
*   	Comcerto commands sent depends on channel type required: \n
*	- NOT defined yet
*/
VSTATUS VAPI_RecoverConnection(IN DEVID DevId,
			      IN CONNID ConnId,
			      IN EConnType eConnType,
			      IN U16 usTdmTimeSlot,
			      IN U16 usRecoverOption,
			      IN SRequest * pstRequest, IN PFNEventCallback pfnEventCallback)
{
	SVapiReq *pstVapiReq;
	SDevice *pstDevice;
	SChnl *pstChnl;
	void *pvChnDef;
	VSTATUS Status;
	Boolean bIsSync;
	SChnl **ppstUserData;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_DeviceGenericCheck(DevId, "VAPI_RecoverConnection", &pstDevice);

	if (Status != SUCCESS)
		goto err_out;

	if (!pstDevice->bIsDeviceInitialized)
	{
		UT_ErrorLog(APPITF, "VAPI_RecoverConnection: device(%u) not initialized\n", DevId);
		Status = VAPI_ERR_DEVICE_NOT_INITIALIZED;
		goto err_out;
	}

	UT_Log(APPITF, INFO, "Entered VAPI_RecoverConnection: devid(%d), connid(%d)\n", DevId, ConnId);

	if(pstDevice->MajorRelease < MR12)
	{
		UT_ErrorLog(APPITF, "VAPI_RecoverConnection: feature not supported by MR:%u\n", pstDevice->MajorRelease);
		Status = VAPI_ERR_UNSUPP_FEATURE;
		goto err_out;
	}

	/* log the parameters passed if required (define in appitf.h) */
	LOG_RECOVER_CONNECTION_PARAMS;

	if (DMGR_GetConnection(ConnId) != NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_RecoverConnection: Connection %u already exists\n", ConnId);
		Status = VAPI_ERR_INVALID_CONNID;
		goto err_out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SChnl *));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}

	/* Create the connection structure to handle the channel */
	pstChnl = UT_Calloc(1, sizeof(SChnl));
	if (pstChnl == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_req;
	}

	/* initialise the connection structure */
	pstChnl->ConnId = ConnId;
	pstChnl->pstDev = pstDevice;
	pstChnl->pfnEventCallback = pfnEventCallback;
	pstChnl->usConnType = eConnType;
	pstChnl->usTimeSlot = usTdmTimeSlot;
	pstChnl->ePartType = eNOTPART;

	/* pass the connection structure pointer to the request*/ 
	ppstUserData = pstVapiReq->pvUserData;
	*ppstUserData = pstChnl;

	/* retrieve the default configuration for this type of channel (VoIP or FoIP)
	   if the channel type is not VoIP or FoIP there are no defualt parameters */
	pvChnDef = DMGR_GetChnlDefaults(pstChnl->usConnType);

	if (pstChnl->usConnType == eVOIP || pstChnl->usConnType == eFOIP)
	{

		pstChnl->pstChnlParams = UT_AllocMem(sizeof(SVoIPChnlParams));
		if (pstChnl->pstChnlParams == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			goto err_chnl;
		}

		if (pvChnDef != NULL)
		{
			UT_MemCopy(pstChnl->pstChnlParams, pvChnDef, sizeof(SVoIPChnlParams));
		}
		else
		{
			Status = VAPI_ERR_UNDEFINED;
			goto err_params;
		}

	}

	/* Initialize few variables, timer,and mutexes */
	Status = VCORE_OpenConnection(pstChnl);
	if (Status != SUCCESS)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_params;
	}

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest,
				VFSM_RecoverConnection,	/* pfnReqFSMHdlr */
				ConnId);

	bIsSync = pstVapiReq->bIsSync;

	/* process the request */
	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (Status != SUCCESS)
		{
			UT_FreeMem(pstChnl->pstChnlParams);
		}
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_RecoverConnection: Exiting status(%d), dev(%u), conn(%u)\n", Status, DevId, ConnId);
/* no error exit */
	return Status;

      err_params:
	UT_FreeMem(pstChnl->pstChnlParams);
      err_chnl:
	UT_FreeMem(pstChnl);
      err_req:
	VCORE_FreeRequest(pstVapiReq);
      err_out:
	return Status;
}


/** @} */ /*connection_level*/
